#include <sp/spDefs.h>
#include <sp/spBase.h>
#include <sp/spMemory.h>
#include <sp/spThread.h>

#include <sp/fftplugin.h>
#include <sp/spPluginMain.h>

#include <fftw3.h>

typedef struct _spPluginInstanceFFTW
{
    void *mutex;
} *spPluginInstanceFFTW;

typedef struct _spFFTRecFFTW
{
    long fftl;
    
    fftw_plan plan_fw;
    fftw_plan plan_bw;
    fftw_plan plan_r2c;
    fftw_plan plan_c2r;

    fftw_complex *cplx_data;
    
    fftwf_plan planf_fw;
    fftwf_plan planf_bw;
    fftwf_plan planf_r2c;
    fftwf_plan planf_c2r;
    
    fftwf_complex *cplx_dataf;
    
} *spFFTRecFFTW;

static spBool spInitFFTW(const char *lang)
{
    return SP_TRUE;
}

static spBool spFreeFFTW(void)
{
    fftwf_cleanup();
    fftw_cleanup();
    
    return SP_TRUE;
}

static void *spInitInstanceFFTW(const char *lang)
{
    spPluginInstanceFFTW pinstance;

    pinstance = xalloc(1, struct _spPluginInstanceFFTW);
    memset(pinstance, 0, sizeof(struct _spPluginInstanceFFTW));
    pinstance->mutex = spCreateMutex("sp_fftw_mutex");
    
    return (void *)pinstance;
}

static spBool spFreeInstanceFFTW(void *instance)
{
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;

    spDestroyMutex(pinstance->mutex);
    
    xfree(instance);
    
    return SP_TRUE;
}

static spBool spIsPrecisionSupportedFFTW(void *instance, spFFTPrecision precision, spFFTSpeed *speed)
{
    if (precision == SP_FFT_DOUBLE_PRECISION) {
	if (speed != NULL) *speed = SP_FFT_SPEED_FASTER;
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

static void *spInitFFTFFTW(void *instance, long order, long batch, spFFTPrecision precision)
{
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec;

    fftrec = xalloc(1, struct _spFFTRecFFTW);
    memset(fftrec, 0, sizeof(struct _spFFTRecFFTW));
    fftrec->fftl = POW2(order);
    spDebug(50, "spInitFFTFFTW", "order = %ld, batch = %ld, fftl = %ld\n", order, batch, fftrec->fftl);
    
    spLockMutex(pinstance->mutex);
    fftrec->cplx_data = (fftw_complex*)fftw_malloc(sizeof(fftw_complex) * fftrec->fftl);
    fftrec->cplx_dataf = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fftrec->fftl);

    fftrec->plan_fw = fftw_plan_dft_1d(fftrec->fftl, fftrec->cplx_data, fftrec->cplx_data, FFTW_FORWARD, FFTW_ESTIMATE);
    fftrec->plan_bw = fftw_plan_dft_1d(fftrec->fftl, fftrec->cplx_data, fftrec->cplx_data, FFTW_BACKWARD, FFTW_ESTIMATE);
    fftrec->plan_r2c = fftw_plan_dft_r2c_1d(fftrec->fftl, (double *)fftrec->cplx_data, fftrec->cplx_data, FFTW_ESTIMATE);
    fftrec->plan_c2r = fftw_plan_dft_c2r_1d(fftrec->fftl, fftrec->cplx_data, (double *)fftrec->cplx_data, FFTW_ESTIMATE);
    
    fftrec->planf_fw = fftwf_plan_dft_1d(fftrec->fftl, fftrec->cplx_dataf, fftrec->cplx_dataf, FFTW_FORWARD, FFTW_ESTIMATE);
    fftrec->planf_bw = fftwf_plan_dft_1d(fftrec->fftl, fftrec->cplx_dataf, fftrec->cplx_dataf, FFTW_BACKWARD, FFTW_ESTIMATE);
    fftrec->planf_r2c = fftwf_plan_dft_r2c_1d(fftrec->fftl, (float *)fftrec->cplx_dataf, fftrec->cplx_dataf, FFTW_ESTIMATE);
    fftrec->planf_c2r = fftwf_plan_dft_c2r_1d(fftrec->fftl, fftrec->cplx_dataf, (float *)fftrec->cplx_dataf, FFTW_ESTIMATE);
    spUnlockMutex(pinstance->mutex);
    
    spDebug(80, "spInitFFTFFTW", "done\n");
    
    return fftrec;
}

static spBool spFreeFFTFFTW(void *instance, void *ifftrec)
{
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec = (spFFTRecFFTW)ifftrec;

    spLockMutex(pinstance->mutex);
    
    if (fftrec->plan_fw != NULL) {
	fftw_destroy_plan(fftrec->plan_fw);
    }
    if (fftrec->plan_bw != NULL) {
	fftw_destroy_plan(fftrec->plan_bw);
    }
    if (fftrec->plan_r2c != NULL) {
	fftw_destroy_plan(fftrec->plan_r2c);
    }
    if (fftrec->plan_c2r != NULL) {
	fftw_destroy_plan(fftrec->plan_c2r);
    }
    if (fftrec->cplx_data != NULL) {
	fftw_free(fftrec->cplx_data);
    }

    if (fftrec->planf_fw != NULL) {
	fftwf_destroy_plan(fftrec->planf_fw);
    }
    if (fftrec->planf_bw != NULL) {
	fftwf_destroy_plan(fftrec->planf_bw);
    }
    if (fftrec->planf_r2c != NULL) {
	fftwf_destroy_plan(fftrec->planf_r2c);
    }
    if (fftrec->planf_c2r != NULL) {
	fftwf_destroy_plan(fftrec->planf_c2r);
    }
    if (fftrec->cplx_dataf != NULL) {
	fftwf_free(fftrec->cplx_dataf);
    }
    
    spUnlockMutex(pinstance->mutex);
    
    xfree(fftrec);
    
    return SP_TRUE;
}

static spBool spExecFFTFFFTW(void *instance, void *ifftrec, float *real, float *imag, int inv)
{
    long k;
    float dfftl;
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec = (spFFTRecFFTW)ifftrec;

    for (k = 0; k < fftrec->fftl; k++) {
	fftrec->cplx_dataf[k][0] = real[k];
	fftrec->cplx_dataf[k][1] = imag[k];
    }
    
    if (inv) {
#if 0
	if (fftrec->planf_bw == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->planf_bw = fftwf_plan_dft_1d(fftrec->fftl, fftrec->cplx_dataf, fftrec->cplx_dataf, FFTW_BACKWARD, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
    
	fftwf_execute(fftrec->planf_bw);

	dfftl = (float)fftrec->fftl;
	for (k = 0; k < fftrec->fftl; k++) {
	    real[k] = fftrec->cplx_dataf[k][0] / dfftl;
	    imag[k] = fftrec->cplx_dataf[k][1] / dfftl;
	}
    } else {
#if 0
	if (fftrec->planf_fw == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->planf_fw = fftwf_plan_dft_1d(fftrec->fftl, fftrec->cplx_dataf, fftrec->cplx_dataf, FFTW_FORWARD, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
    
	fftwf_execute(fftrec->planf_fw);
	
	for (k = 0; k < fftrec->fftl; k++) {
	    real[k] = fftrec->cplx_dataf[k][0];
	    imag[k] = fftrec->cplx_dataf[k][1];
	}
    }
    
    return SP_TRUE;
}

static spBool spExecFFTFFTW(void *instance, void *ifftrec, double *real, double *imag, int inv)
{
    long k;
    double dfftl;
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec = (spFFTRecFFTW)ifftrec;

    for (k = 0; k < fftrec->fftl; k++) {
	fftrec->cplx_data[k][0] = real[k];
	fftrec->cplx_data[k][1] = imag[k];
    }
    
    if (inv) {
#if 0
	if (fftrec->plan_bw == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->plan_bw = fftw_plan_dft_1d(fftrec->fftl, fftrec->cplx_data, fftrec->cplx_data, FFTW_BACKWARD, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
    
	fftw_execute(fftrec->plan_bw);

	dfftl = (double)fftrec->fftl;
	for (k = 0; k < fftrec->fftl; k++) {
	    real[k] = fftrec->cplx_data[k][0] / dfftl;
	    imag[k] = fftrec->cplx_data[k][1] / dfftl;
	}
    } else {
#if 0
	if (fftrec->plan_fw == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->plan_fw = fftw_plan_dft_1d(fftrec->fftl, fftrec->cplx_data, fftrec->cplx_data, FFTW_FORWARD, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
    
	fftw_execute(fftrec->plan_fw);
	
	for (k = 0; k < fftrec->fftl; k++) {
	    real[k] = fftrec->cplx_data[k][0];
	    imag[k] = fftrec->cplx_data[k][1];
	}
    }
    
    return SP_TRUE;
}

static spBool spExecRealFFTFFFTW(void *instance, void *ifftrec, float *data, int inv)
{
    long k, fftl2;
    float dfftl;
    float *dptr;
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec = (spFFTRecFFTW)ifftrec;

    fftl2 = fftrec->fftl / 2;
    dptr = (float *)fftrec->cplx_dataf;
    
    if (inv) {
	for (k = 0; k < fftl2; k++) {
	    fftrec->cplx_dataf[k][0] = data[k * 2];
	    fftrec->cplx_dataf[k][1] = data[k * 2 + 1];
	}
	fftrec->cplx_dataf[k][0] = data[1];
	fftrec->cplx_dataf[k][1] = 0.0;

#if 0
	if (fftrec->planf_c2r == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->planf_c2r = fftwf_plan_dft_c2r_1d(fftrec->fftl, fftrec->cplx_dataf, dptr, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
	
	fftwf_execute(fftrec->planf_c2r);

	dfftl = (float)fftrec->fftl;
	for (k = 0; k < fftrec->fftl; k++) {
	    data[k] = dptr[k] / dfftl;
	}
	
    } else {
	for (k = 0; k < fftrec->fftl; k++) {
	    dptr[k] = data[k];
	}

#if 0
	if (fftrec->planf_r2c == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->planf_r2c = fftwf_plan_dft_r2c_1d(fftrec->fftl, dptr, fftrec->cplx_dataf, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
	
	fftwf_execute(fftrec->planf_r2c);

	for (k = 0; k < fftl2; k++) {
	    data[k * 2] = fftrec->cplx_dataf[k][0];
	    data[k * 2 + 1] = fftrec->cplx_dataf[k][1];
	}
	data[1] = fftrec->cplx_dataf[k][0];
    }
    
    return SP_TRUE;
}

static spBool spExecRealFFTFFTW(void *instance, void *ifftrec, double *data, int inv)
{
    long k, fftl2;
    double dfftl;
    double *dptr;
    spPluginInstanceFFTW pinstance = (spPluginInstanceFFTW)instance;
    spFFTRecFFTW fftrec = (spFFTRecFFTW)ifftrec;

    fftl2 = fftrec->fftl / 2;
    dptr = (double *)fftrec->cplx_data;
    
    if (inv) {
	for (k = 0; k < fftl2; k++) {
	    fftrec->cplx_data[k][0] = data[k * 2];
	    fftrec->cplx_data[k][1] = data[k * 2 + 1];
	}
	fftrec->cplx_data[k][0] = data[1];
	fftrec->cplx_data[k][1] = 0.0;

#if 0
	if (fftrec->plan_c2r == NULL) {
	    spLockMutex(pinstance->mutex);
	    fftrec->plan_c2r = fftw_plan_dft_c2r_1d(fftrec->fftl, fftrec->cplx_data, dptr, FFTW_ESTIMATE);
	    spUnlockMutex(pinstance->mutex);
	}
#endif
	
	fftw_execute(fftrec->plan_c2r);

	dfftl = (double)fftrec->fftl;
	for (k = 0; k < fftrec->fftl; k++) {
	    data[k] = dptr[k] / dfftl;
	}
	
    } else {
	for (k = 0; k < fftrec->fftl; k++) {
	    dptr[k] = data[k];
	}

#if 0
	if (fftrec->plan_r2c == NULL) {
	    spDebug(80, "spExecRealFFTFFTW", "before spLockMutex\n");
	    spLockMutex(pinstance->mutex);
	    fftrec->plan_r2c = fftw_plan_dft_r2c_1d(fftrec->fftl, dptr, fftrec->cplx_data, FFTW_ESTIMATE);
	    spDebug(80, "spExecRealFFTFFTW", "fftrec->plan_r2c = %ld\n", (long)fftrec->plan_r2c);
	    spUnlockMutex(pinstance->mutex);
	    spDebug(80, "spExecRealFFTFFTW", "after spUnlockMutex\n");
	}
#endif
	
	fftw_execute(fftrec->plan_r2c);

	for (k = 0; k < fftl2; k++) {
	    data[k * 2] = fftrec->cplx_data[k][0];
	    data[k * 2 + 1] = fftrec->cplx_data[k][1];
	}
	data[1] = fftrec->cplx_data[k][0];
    }
    
    return SP_TRUE;
}

static spFFTPluginRec sp_fft_plugin_fftw = {
    NULL,
    NULL,

    SP_PLUGIN_FFT,
    "FFTW",
    1,
    SP_PLUGIN_PRIORITY_MIDDLE,
    SP_PLUGIN_CAPS_THREAD_SAFE,	/* caps */
    spInitFFTW,
    spFreeFFTW,
    "FFTW",
    "FFTW Plugin  Version 0.1",
    
    spInitInstanceFFTW,
    spFreeInstanceFFTW,
    NULL,
    NULL,

    NULL,
    NULL,
    NULL,
	
    spIsPrecisionSupportedFFTW,

    NULL,
    NULL,
	
    spInitFFTFFTW,
    spFreeFFTFFTW,
    NULL,
	
    spExecFFTFFFTW,
    spExecFFTFFTW,
	
    spExecRealFFTFFFTW,
    spExecRealFFTFFTW,
	
    NULL,
    NULL,
};

spPluginExport spPluginRec *spGetPluginRec(void)
{
    return (spPluginRec *)&sp_fft_plugin_fftw;
}
