#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

/*
**** original cqt.m and icqt.m license ****
% Copyright (C) 2013 Christian Schﾃｶrkhuber.
% 
% This work is licensed under the Creative Commons 
% Attribution-NonCommercial-ShareAlike 3.0 Unported 
% License. To view a copy of this license, visit 
% http://creativecommons.org/licenses/by-nc-sa/3.0/ 
% or send a letter to 
% Creative Commons, 444 Castro Street, Suite 900, 
% Mountain View, California, 94041, USA.
%
% Authors: Christian Schﾃｶrkhuber
% Date: 20.09.13

**** original nsgcqwin.m license ****
% Copyright (C) 2013 Nicki Holighaus, Christian Schﾃｶrkhuber.
% This work is licensed under the Creative Commons 
% Attribution-NonCommercial-ShareAlike 3.0 Unported 
% License. To view a copy of this license, visit 
% http://creativecommons.org/licenses/by-nc-sa/3.0/ 
% or send a letter to 
% Creative Commons, 444 Castro Street, Suite 900, 
% Mountain View, California, 94041, USA.

**** original nsgtf_real.m and nsigtf_real.m license ****
% Copyright (C) 2013 Nicki Holighaus.
% This file is part of NSGToolbox version 0.1.0
% 
% This work is licensed under the Creative Commons 
% Attribution-NonCommercial-ShareAlike 3.0 Unported 
% License. To view a copy of this license, visit 
% http://creativecommons.org/licenses/by-nc-sa/3.0/ 
% or send a letter to 
% Creative Commons, 444 Castro Street, Suite 900, 
% Mountain View, California, 94041, USA.
%
% Author: Nicki Holighaus, Gino Velasco
% Date: 23.04.13
% Edited by Christian Schﾃｶrkhuber, 25.09.2013

**** original nsdual.m license ****
% Copyright (C) 2013 Nicki Holighaus.
% This file is part of NSGToolbox version 0.1.0
% 
% This work is licensed under the Creative Commons 
% Attribution-NonCommercial-ShareAlike 3.0 Unported 
% License. To view a copy of this license, visit 
% http://creativecommons.org/licenses/by-nc-sa/3.0/ 
% or send a letter to 
% Creative Commons, 444 Castro Street, Suite 900, 
% Mountain View, California, 94041, USA.

% Author: Nicki Holighaus, Gino Velasco
% Date: 23.04.13
 */

#include <sp/spLib.h>
#include <sp/spThread.h>

#undef SP_CQT_FFT_SIG_CENTER
#define SP_CQT_BLOCK_SPECTRUM_POWER_SUM

#define SP_CQT_PROGRESS_PERCENT_STOPPED (-2.0)

struct spCQTThreadRec {
    spCQTRec cqtrec;
    
    void *thread;
    void *event;

    long index;
    long channel_start;
    long channel_end;

    spBool forward_finished;
};

struct spCQTRec {
    double samp_freq;
    double min_freq;
    long bins_per_octave;
    long whole_input_length;

    spCQTCallbackFunc callback_func;
    void *user_data;
#if defined(SP_CQT_USE_THREAD)
    volatile
#endif
    double progress_percent;

    spPlugin *fft_plugin;
    spFFTRec fftrec;
    
    long block_length;
    long fftl;
    double fftl_factor;
    double length_crop_factor;
    spBool phasemode_global;
    spDVectors g;
    spLVector shift;
    spDVector M;
    long maxM;

    long total_freq_len;
    spLVector posit;
    
    long total_nbins;
    
    spDVector win;

    spDVectors gd;

#if defined(SP_CQT_USE_THREAD)    
    /* thread related members */
    long num_thread;
    spDVector Mcumsum;
    volatile spDVector current_f;
    volatile spDVectors current_c;
    volatile spDVector current_fr;
    void *mutex;
    void *event_for_process;
    spCQTThreadRec *thread_recs;
    
    volatile spBool proceed_flag;
    volatile spBool terminate_thread;
    volatile long num_computing;
    volatile double start_progress_percent;
    volatile double target_progress_percent;
#endif
};

struct spCQTBlock {
    long process_count;
    long pushed_length;
    
    spDVector x;
    spDVector wx;
    spDVector f;
    spDVectors c;
    
    spDVectors prev_c;
};

static long cqt_current_freq_index = -1;
static long cqt_current_block_index = -1;

static spDVector xdvcalcnormlizevec(spDVectors g, spDVector M, long fft_length, spCQTOptionMask normalizeMethod,
                                    spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long bins;
    spDVector normFacVecTurn;

    bins = M->length / 2 - 1;

    if ((normalizeMethod & SP_CQT_OPTION_NORMALIZE_METHOD_SINE) || (normalizeMethod & SP_CQT_OPTION_NORMALIZE_METHOD_IMPULSE)) {
        spDVector normFacVec;
        
        normFacVec = xdvcut(M, 0, bins + 2);
        if (normalizeMethod & SP_CQT_OPTION_NORMALIZE_METHOD_SINE) {
            dvscoper(normFacVec, "*", 2.0 / (double)fft_length);
        } else {
            long k;
            for (k = 0; k < normFacVec->length; k++) {
                normFacVec->data[k] = 2.0 * M->data[k] / (double)g->vector[k]->length;
            }
        }
        normFacVecTurn = xdvfftturn(normFacVec, M->length);
        xdvfree(normFacVec);
    } else {
        normFacVecTurn = xdvones(M->length);
    }
    
    return normFacVecTurn;
}

static void cqtapplynormlizevec(spDVector normFacVec, spDVectors io_g,
                                spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long k;

    for (k = 0; k < io_g->num_vector; k++) {
        dvscoper(io_g->vector[k], "*", normFacVec->data[k]);
        if (callback_func != NULL) {
            callback_func(cqtrec, SP_CQT_PREPARE_CHANNEL_FILTER_NORMALIZED_CALLBACK, &k, io_g->vector[k], user_data);
        }
    }

    return;
}

/* in cqt.m after nsgcqwin function */
static spDVector xdvcalccqtfbasfromshift(spLVector shift, double fs, long fft_length, long length_M)
{
    long k;
    long len;
    long cumsumshift;
    spDVector fbas;

    len = length_M / 2 - 1;
    len = MIN(len, shift->length - 1);
    fbas = xdvalloc(len);

    cumsumshift = 0;
    
    for (k = 0; k < fbas->length; k++) {
        cumsumshift += shift->data[k + 1];

        fbas->data[k] = fs * (double)cumsumshift / (double)fft_length;
    }

    return fbas;
}

static spDVector xdvcalccqtfbas(double fmin, double end_freq/* 0.0: ignore */, long bins_per_octave,
                                long length, spBool start_from_0Hz)
{
    double b;
    double invbins;
    double start;
    spDVector fbas;

    invbins = 1.0 / (double)bins_per_octave;
    spDebug(80, "xdvcalccqtfbas", "fmin = %f, end_freq = %f, bins_per_octave = %ld, length = %ld, invbins = %f\n",
            fmin, end_freq, bins_per_octave, length, invbins);

    if (start_from_0Hz) {
        start = -invbins;
        b = (double)(length - 2);
    } else {
        start = 0.0;
        b = (double)(length - 1);
    }

    // fbas = fmin .* 2.^((0:b).'./bins);
    fbas = xdvinit(start, invbins, b * invbins);
    dvscoper(fbas, "!^", 2.0);
    dvscoper(fbas, "*", fmin);
    if (start_from_0Hz) {
        fbas->data[0] = 0.0;
    }
    if (end_freq > 0.0) {
        fbas->data[fbas->length - 1] = end_freq;
    }

    return fbas;
}

static double cqtwinvalue_hann(double x)
{
    // g = .5 + .5*cos(2*pi*x);
    return 0.5 + 0.5 * cos(2.0 * PI * x);
}

static spDVector xdvcqtwintukey(spCQTWindowValueFunc func, long N, long L, long tukeytotal)
{
    long n, n2;
    long n2_offset;
    double x, x2;
    double incr;
    double x2_offset;
    spDVector win;
    spBool even_flag;
    long Nhalf;
    long tukeyoffset;

    if (L == 0) L = N;
    if (L < N) {
        spWarning("Output length L must be larger than or equal to N\n");
        return NODATA;
    }

    if (tukeytotal > N) {
        tukeyoffset = (tukeytotal / 2) - (N / 2);
        spDebug(100, "xdvcqtwintukey", "tukeytotal = %ld, N = %ld, L = %ld, tukeyoffset = %ld\n",
                tukeytotal, N, L, tukeyoffset);
        win = xdvzeros(tukeytotal);
    } else {
        tukeyoffset = 0;
        win = xdvzeros(L);
    }

    if (N % 2 == 0) {
        even_flag = SP_TRUE;
    } else {
        even_flag = SP_FALSE;
    }
    
    Nhalf = N / 2;
    incr = 1.0 / (double)N;
    n2_offset = Nhalf + L - N;
    x2_offset = -0.5;
    if (even_flag == SP_FALSE) {
        ++n2_offset;
        x2_offset += 0.5 * incr;
    }

    for (n = 0; n < tukeyoffset; n++) {
        win->data[n] = 1.0;
    }

    for (n = 0; n < Nhalf; n++) {
        n2 = n2_offset + n;
        
        x = (double)n * incr;
        x2 = x2_offset + x;

        win->data[tukeyoffset + n] = func(x);
        win->data[tukeyoffset + n2] = func(x2);
    }
    if (even_flag == SP_FALSE) {
        x = 0.5 - 0.5 * incr;
        win->data[tukeyoffset + n] = func(x);
    }
    for (n = tukeyoffset + n2_offset + Nhalf; n < win->length; n++) {
        win->data[n] = 1.0;
    }
     
    return win;
}

static spDVector xdvcqtwin(spCQTWindowValueFunc func, long N, long L)
{
    return xdvcqtwintukey(func, N, L, 0);
}

static spBool xnsgcqwin(double fmin, double fmax, long bins, double gamma, double fs, long fft_length,
                        spCQTWindowValueFunc winfunc, spDVectors *xo_g, spLVector *xo_shift, spDVector *xo_M,
                        spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    int percent;
    double progress_incr;
    double progress_percent;
    double callbacked_percent;
    long i, j;
    double b;
    double invbins;
    double nf;
    double fftres;
    double Q;
    long start, end;
    long Lfbas, Lfbas1;
    double Morig;
    double Mceil, Mceilnext;
    double posit_l, posit_r;
    double prev_posit_l, prev_posit_r;
    double diff_posit_l, diff_posit_r;
    spDVector fbas = NODATA;
    spDVector fbasfull = NODATA;
    spDVector cqtbw = NODATA;
    spDVector bw = NODATA;
    spLVector shift = NODATA;
    spDVectors g = NODATA;
    spDVector M = NODATA;
    long bwfac = 1;
    double min_win = 4.0;
    spBool proceed_flag = SP_TRUE;

    if (winfunc == NULL) winfunc = cqtwinvalue_hann;

    nf = fs / 2.0;
    
    if (fmax <= fmin || fmax > nf) {
        fmax = nf;
    }

    spDebug(30, "xnsgcqwin", "fmin = %f, fmax = %f, bins = %ld, gamma = %f, fs = %f, fft_length = %ld\n",
            fmin, fmax, bins, gamma, fs, fft_length);
    
    fftres = fs / (double)fft_length;
    b = floor(bins * log10(fmax / fmin) / log10(2.0));
    invbins = 1.0 / (double)bins;
    spDebug(50, "xnsgcqwin", "fftres = %f, b = %f, invbins = %f\n", fftres, b, invbins);
    
    // fbas = fmin .* 2.^((0:b).'./bins);
    fbas = xdvinit(0.0, invbins, b * invbins);
    dvscoper(fbas, "!^", 2.0);
    dvscoper(fbas, "*", fmin);
    //dvdump(fbas); exit(0);

    // Q = 2^(1/bins) - 2^(-1/bins);
    Q = pow(2.0, invbins) - pow(2.0, -invbins);
    spDebug(50, "xnsgcqwin", "Q = %f\n", Q);
    
    // cqtbw = Q*fbas + gamma;
    cqtbw = xdvscoper(fbas, "*", Q);
    dvscoper(cqtbw, "+", gamma);

    start = end = -1;
    for (i = 0, j = fbas->length - 1; i < fbas->length; i++, j--) {
        //%make sure the support of highest filter won't exceed nf
        if (end < 0 && fbas->data[i] + cqtbw->data[i] / 2.0 > nf) {
            end = i - 1;
        }
        //%make sure the support of the lowest filter won't exceed DC
        if (start < 0 && fbas->data[j] - cqtbw->data[j] / 2.0 < 0) {
            start = j + 1;
        }
        if (start >= 0 && end >= 0) {
            break;
        }
    }
    if (end < 0) end = fbas->length - 1;
    if (start < 0) start = 0;
    Lfbas = end - start + 1;
    Lfbas1 = Lfbas + 1;
    spDebug(50, "xnsgcqwin", "start = %ld, end = %ld, Lfbas = %ld\n", start, end, Lfbas);
    
    //dvdump(cqtbw); exit(0);
    
    if (callback_func != NULL) {
        percent = 2;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
    }
    
    fbasfull = xdvzeros(Lfbas1 * 2);
    bw = xdvzeros(2 * Lfbas1);

    bw->data[0] = 2.0 * fmin / fftres;
    for (i = 1; i < Lfbas1; i++) {
        fbasfull->data[i] = fbas->data[start + i - 1] / fftres;
        fbasfull->data[Lfbas1 + i] = (fs - fbas->data[start + Lfbas - i]) / fftres;
        bw->data[i] = cqtbw->data[start + i - 1] / fftres;
        bw->data[2 * Lfbas1 - i] = bw->data[i];
    }
    fbasfull->data[Lfbas1] = nf / fftres;
    bw->data[Lfbas1] = (fbasfull->data[Lfbas1 + 1] - fbasfull->data[Lfbas1 - 1]);
    
    //dvdump(fbasfull); exit(0);
    //dvdump(bw); exit(0);
    
    if (callback_func != NULL) {
        percent = 3;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
    }
    
    shift = xlvzeros(bw->length);

    prev_posit_l = floor(fbasfull->data[0]);
    prev_posit_r = ceil(fbasfull->data[Lfbas1 + 1]);
    for (i = 1; i <= Lfbas1; i++) {
        posit_l = floor(fbasfull->data[i]);
        diff_posit_l = posit_l - prev_posit_l;
        shift->data[i] = (long)spRound(diff_posit_l);
        prev_posit_l = posit_l;

        if (i < Lfbas) {
            posit_r = ceil(fbasfull->data[Lfbas1 + 1 + i]);
            diff_posit_r = posit_r - prev_posit_r;
            shift->data[Lfbas1 + 1 + i] = (long)spRound(diff_posit_r);
            prev_posit_r = posit_r;
        }
    }
    shift->data[0] = (long)spRound(spDMod(-ceil(fbasfull->data[fbasfull->length - 1]), (double)fft_length));
    posit_l = ceil(fbasfull->data[Lfbas1 + 1]);
    diff_posit_l = posit_l - prev_posit_l;
    shift->data[Lfbas1 + 1] = (long)spRound(diff_posit_l);
    //lvdump(shift); exit(0);
    if (callback_func != NULL) {
        percent = 4;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_PREPARE_CENTER_FREQUENCIES_OBTAINED_CALLBACK, shift, NULL, user_data);
    }

    M = xdvzeros(bw->length);
    for (i = 0; i < bw->length; i++) {
        M->data[i] = spRound(bw->data[i]);
        if (M->data[i] < min_win) {
            M->data[i] = min_win;
        }
    }

    progress_percent = 4.0;
    progress_incr = 90.5 / (double)M->length;
    callbacked_percent = 4.0;

    g = xdvsalloc(M->length);
    Mceil = 0.0;
    for (i = g->num_vector - 1; i >= 0; i--) {
        Morig = M->data[i];
        Mceilnext = Mceil;
        if (bwfac != 1) {
            // M = bwfac*ceil(M/bwfac);
            Mceil = bwfac * ceil(M->data[i] / (double)bwfac);
        } else {
            Mceil = M->data[i];
        }
        spDebug(100, "xnsgcqwin", "i = %ld, Morig = %f, Mceil = %f, Mceilnext = %f\n",
                i, Morig, Mceil, Mceilnext);
        
        if ((i == 0 || i == Lfbas1) && Mceil > Mceilnext) { /* after ceil */
            //% Setup Tukey window for 0- and Nyquist-frequency
            g->vector[i] = xdvcqtwintukey(/*winfunc*/cqtwinvalue_hann, (long)spRound(Mceilnext), 0, (long)spRound(Mceil));
            dvscoper(g->vector[i], "/", sqrt(Mceil));
        } else {
            g->vector[i] = xdvcqtwin(winfunc, (long)spRound(Morig), 0);
        }
        if (callback_func != NULL) {
            if (progress_percent - callbacked_percent >= 1.0) {
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                    goto bail;
                }
                callbacked_percent = percent;
            }
        
            callback_func(cqtrec, SP_CQT_PREPARE_CHANNEL_FILTER_OBTAINED_CALLBACK, &i, g->vector[i], user_data);
        }
        
        if (bwfac != 1 && i < g->num_vector - 1) {
            /* now, update M of i+1 (don't update M->data[i]) */
            M->data[i+1] = Mceilnext;
        }

        progress_percent += progress_incr;
    }
    if (bwfac != 1) {
        M->data[0] = Mceil;
    }
    if (io_progress_percent != NULL) *io_progress_percent = 95.0;
    
    if (callback_func != NULL) {
        percent = 95;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_PREPARE_BANDWIDTHS_OBTAINED_CALLBACK, M, NULL, user_data);
    }

    //dvdump(M); exit(0);
    //dvdump(g->vector[0]); exit(0);
    //dvdump(g->vector[1]); exit(0);
    //dvdump(g->vector[2]); exit(0);
    //dvdump(g->vector[Lfbas1]); exit(0);

  bail:
    if (fbas != NODATA) xdvfree(fbas);
    if (fbasfull != NODATA) xdvfree(fbasfull);
    if (cqtbw != NODATA) xdvfree(cqtbw);
    if (bw != NODATA) xdvfree(bw);

    if (proceed_flag == SP_FALSE) {
        if (g != NODATA) xdvsfree(g);
        if (shift != NODATA) xlvfree(shift);
        if (M != NODATA) xdvfree(M);
        *xo_g = NODATA;
        *xo_shift = NODATA;
        *xo_M = NODATA;
    } else {
        *xo_g = g;
        *xo_shift = shift;
        *xo_M = M;
    }
    
    return proceed_flag;
}

static spBool dvcqtupdateM(spDVector M, double fmin, double fmax, long refsize, spCQTOptionMask rasterizeMethod,
                           long *o_maxM, double *io_stride)
{
    long k;
    long n;
    long bins;
    long bins_nyquist;
    long orig_refsize;
    double stride;
    double mtemp;
    double maxM;
    spBool stride_status = SP_TRUE;

    orig_refsize = refsize;
    bins = M->length / 2 - 1;

    /* (bins + 2) in MATLAB */
    bins_nyquist = bins + 1;

    maxM = MAX(M->data[bins_nyquist], M->data[bins]);
    spDebug(50, "dvcqtupdateM", "orig_refsize = %ld, bins = %ld, maxM = %f\n", orig_refsize, bins, maxM);
    //dvdump(M); exit(0);
    
    stride = 0.0;
    
    if (io_stride != NULL) {
        spDebug(50, "dvcqtupdateM", "original stride = %f\n", *io_stride);
        if (*io_stride <= 0.0 || refsize <= 0) {
            if (*io_stride == SP_CQT_REFERENCE_STRIDE_MAX) {
                stride = (double)orig_refsize / maxM;
            }
        } else {
            stride = *io_stride;
            refsize = (long)floor((double)refsize / stride);
            if (maxM > (double)refsize) {
                stride = (double)orig_refsize / maxM;
                stride = MAX(stride, 1.0);
                refsize = (long)floor((double)orig_refsize / stride);
                stride_status = SP_FALSE;
            }
        }
        spDebug(50, "dvcqtupdateM", "maxM = %f, refsize = %ld, orig_refsize = %ld, stride = %f, stride_status = %d\n",
                maxM, refsize, orig_refsize, stride, stride_status);
        *io_stride = stride;
    }
    
    if (rasterizeMethod & SP_CQT_OPTION_RASTERIZE_METHOD_FULL) {
        if (rasterizeMethod & SP_CQT_OPTION_RASTERIZE_METHOD_POWER_OF_2) {
            mtemp = (double)spPow2(spNextPow2((long)spRound(maxM)));
            if (stride >= 1.0 && refsize > 0) {
                if (rasterizeMethod & SP_CQT_OPTION_STRICT_REFERENCE_STRIDE) {
                    mtemp = 1.0;
                }
                n = spPow2(spNextPow2((long)spRound((double)refsize / mtemp)) - 1);
                mtemp = ceil((double)refsize / (double)n);
            }
            M->data[bins] = mtemp;
            M->data[bins_nyquist] = M->data[bins];
            M->data[0] = M->data[bins];
        } else if (stride >= 1.0 && refsize > 0) {
            if (rasterizeMethod & SP_CQT_OPTION_STRICT_REFERENCE_STRIDE) {
                n = 1;
            } else {
                n = spPow2(spNextPow2((long)spRound((double)refsize / (double)M->data[bins])) - 1);
            }
            
            M->data[bins] = ceil((double)refsize / (double)n);
            
            if (rasterizeMethod & SP_CQT_OPTION_STRICT_REFERENCE_STRIDE) {
                M->data[bins_nyquist] = M->data[bins];
                M->data[0] = M->data[bins];
            }
        }
        for (k = 1; k < bins; k++) {
            M->data[k] = M->data[bins];
            M->data[M->length - k] = M->data[k];
        }
    } else if (rasterizeMethod & SP_CQT_OPTION_RASTERIZE_METHOD_PIECEWISE) {
        double temp;
        double octs;
        double oct_lin_count;

        temp = M->data[bins];
        //octs = ceil(log2(fmax / fmin));
        octs = ceil(log10(fmax / fmin) / log10(2.0));
        //%make sure that the number of coefficients in the highest octave is
        //%dividable by 2 at least octs-times
        oct_lin_count = pow(2.0, octs);
        temp = ceil(temp / oct_lin_count) * oct_lin_count;
        
        for (k = 1; k < M->length; k++) {
            if (k == bins_nyquist) {
                //mtemp(bins+2) = M(bins+2); %don't rasterize Nyquist bin
            } else {
                mtemp = temp / M->data[k];
                mtemp = pow(2.0,  ceil(log10(mtemp) / log10(2.0)) - 1.0);
                M->data[k] = temp / mtemp;
            }
        }
        //mtemp(1) = M(1); %don't rasterize DC bin
    } else if (rasterizeMethod & SP_CQT_OPTION_RASTERIZE_METHOD_POWER_OF_2) {
        long Mround, Mnew;
        
        for (k = 0; k < bins_nyquist; k++) {
            Mround = (long)spRound(M->data[k]);
            Mnew = POW2(spNextPow2(Mround));
            
            M->data[k] = Mnew;
            
            if (k > 0) {
                M->data[M->length - k] = M->data[k];
            }
        }
    } else {
        if (stride >= 1.0 && refsize > 0) {
            for (k = 0; k < bins_nyquist; k++) {
                if (k == bins && (rasterizeMethod & SP_CQT_OPTION_STRICT_REFERENCE_STRIDE)) {
                    n = 1;
                } else {
                    mtemp = M->data[k];
                    n = spPow2(spNextPow2((long)spRound((double)refsize / mtemp)) - 1);
                }
                M->data[k] = ceil((double)refsize / (double)n);
            
                if (k > 0) {
                    M->data[M->length - k] = M->data[k];
                }
            }
        }
    }
    
    maxM = MAX(M->data[bins_nyquist], M->data[bins]);
    
    if (io_stride != NULL) {
        stride = (double)orig_refsize / maxM;
        spDebug(50, "dvcqtupdateM", "final maxM = %f, stride = %f, orig_refsize = %ld\n", maxM, stride, orig_refsize);

        *io_stride = stride;
    }
    if (o_maxM != NULL) *o_maxM = (long)spRound(maxM);
    
    return stride_status;
}

static void dvcqtcorrectM(spDVector M, long refsize, long fft_length, long minM, long *io_maxM, double *io_stride)
{
    long k;
    long oddflag;
    long fftl_factor;
    long Mvalue;
    long bins_nyquist;

    dvscmax(M, (double)minM);

    oddflag = refsize % 2;
    bins_nyquist = M->length / 2;
    
    fftl_factor = (long)spRound((double)fft_length / (double)refsize);
    spDebug(80, "dvcqtcorrectM", "oddflag = %ld, fftl_factor = %ld\n", oddflag, fftl_factor);

    if (fftl_factor != 1) {
        long rem;
        long fftl_factor2;
        
        fftl_factor2 = fftl_factor * 2;

        if (oddflag) {
            rem = fftl_factor;
        } else {
            rem = 0;
        }
        
        for (k = 0; k < M->length; k++) {
            Mvalue = (long)spRound(M->data[k]);
            if (Mvalue % fftl_factor2 != rem) {
                if (oddflag) {
                    M->data[k] = fftl_factor + fftl_factor2 * spRound(M->data[k] / (double)fftl_factor2);
                } else {
                    M->data[k] = fftl_factor2 * ceil(M->data[k] / (double)fftl_factor2);
                }
                if (io_maxM != NULL && (k == bins_nyquist || k == bins_nyquist - 1)) {
                    if (M->data[k] > *io_maxM) {
                        *io_maxM = M->data[k];
                        if (io_stride != NULL) {
                            *io_stride = (double)refsize / M->data[k];
                        }
                    }
                }
            }
        }
    } else {
        for (k = 0; k < M->length; k++) {
            Mvalue = (long)spRound(M->data[k]);
            if (Mvalue % 2 != oddflag) {
                Mvalue += 1;
                spDebug(100, "dvcqtcorrectM", "k = %ld, Mvalue = %ld, oddflag = %ld\n", k, Mvalue, oddflag);
                M->data[k] = (double)Mvalue;
                if (io_maxM != NULL && (k == bins_nyquist || k == bins_nyquist - 1)) {
                    if (M->data[k] > *io_maxM) {
                        *io_maxM = M->data[k];
                        if (io_stride != NULL) {
                            *io_stride = (double)refsize / M->data[k];
                        }
                    }
                }
            }
        }
    }

    return;
}

static spLVector xlvcalcwinrange(long Lghalf_floor, long Lghalf_ceil, long Ls, long posit)
{
    long k;
    long len;
    spLVector win_range;
    
    //win_range = mod(posit(ii)+(-floor(Lg(ii)/2):ceil(Lg(ii)/2)-1),Ls+fill)+1;
    len = Lghalf_floor + Lghalf_ceil;
    win_range = xlvalloc(len);
    for (k = 0; k < len; k++) {
        win_range->data[k] = spLMod(posit - Lghalf_floor + k, Ls);
    }

    return win_range;
}

static void dvcqtglobalmapfreq(spDVector spc, long fkBins, long fsNewBins, spBool inv)
{
    long displace;
        
    //%apply frequency mapping function (see cqt)
    //fsNewBins = M;
    //fkBins = posit;
    displace = fkBins - (fkBins / fsNewBins) * fsNewBins;
    spDebug(50, "dvcqtglobalmapfreq", "displace = %ld\n", displace);
    dvcircshift(spc, inv ? -displace : displace);
    
    return;
}

static void nsgtf_real_each(spFFTRec fftrec, spDVector f, long channel, spDVector g, double normFac,
                            long posit, spBool phasemode_global, spDVector o_c,
                            spCQTCallbackFunc callback_func, void *cqtrec, void *user_data)
{
    long k;
    long M;
    long Lg;
    long Lghalf_ceil;
    long Lghalf_floor;
    double gvalue;
    spLVector idxl, idxr;
    spLVector win_range;
    spDVector temp;

    M = o_c->length;
    
    spDebug(50, "nsgtf_real_each", "normFac = %f, M = %ld, posit = %ld\n", normFac, M, posit);
    
    Lg = g->length;
    Lghalf_floor = Lg / 2;
    
    if (posit - Lghalf_floor > f->length / 2) {
        return;
    }

    Lghalf_ceil = (long)ceil((double)Lg / 2.0);
    spDebug(50, "nsgtf_real_each", "Lg = %ld, Lghalf_floor = %ld, Lghalf_ceil = %ld\n",
            Lg, Lghalf_floor, Lghalf_ceil);
    
    //idx = [ceil(Lg(ii)/2)+1:Lg(ii),1:ceil(Lg(ii)/2)];
    idxl = xlvinit(Lghalf_ceil, 1, Lg - 1);
    idxr = xlvinit(0, 1, Lghalf_ceil - 1);

    //win_range = mod(posit(ii)+(-floor(Lg(ii)/2):ceil(Lg(ii)/2)-1),Ls+fill)+1;
    win_range = xlvcalcwinrange(Lghalf_floor, Lghalf_ceil, f->length, posit);
#if 0
    if (cqt_current_freq_index == 1 && cqt_current_block_index == 2) {
        lvdump(win_range); exit(0);
    }
#endif
    
    temp = o_c;
    dvrizeros(temp, temp->length);
    for (k = 0; k < Lghalf_floor; k++) {
        gvalue = g->data[idxl->data[k]] * normFac;
        temp->data[M - Lghalf_floor + k] = f->data[win_range->data[k]] * gvalue;
        temp->imag[M - Lghalf_floor + k] = f->imag[win_range->data[k]] * gvalue;
    }
    for (k = 0; k < Lghalf_ceil; k++) {
        gvalue = g->data[idxr->data[k]] * normFac;
        temp->data[k] = f->data[win_range->data[Lghalf_floor + k]] * gvalue;
        temp->imag[k] = f->imag[win_range->data[Lghalf_floor + k]] * gvalue;
    }
#if 0
    if (cqt_current_freq_index == 1 && cqt_current_block_index == 2) {
        dvdump(temp); exit(0);
    }
#endif
    if (callback_func != NULL) {
        callback_func(cqtrec, SP_CQT_FORWARD_CHANNEL_CONVOLUTION_FINISHED_CALLBACK, &channel, temp, user_data);
    }

    if (phasemode_global == SP_TRUE) {
        dvcqtglobalmapfreq(temp, posit, M, SP_FALSE);
#if 0
        if (cqt_current_freq_index == 1 && cqt_current_block_index == 2) {
            dvdump(temp); exit(0);
        }
#endif
        if (callback_func != NULL) {
            callback_func(cqtrec, SP_CQT_FORWARD_CHANNEL_PHASE_GLOBALIZE_FINISHED_CALLBACK, &channel, temp, user_data);
        }
    }

    dvifft(temp);
#if 0
    if (cqt_current_freq_index == 1 && cqt_current_block_index == 2) {
        dvdump(temp); exit(0);
    }
#endif
    if (callback_func != NULL) {
        callback_func(cqtrec, SP_CQT_FORWARD_CHANNEL_IFFT_FINISHED_CALLBACK, &channel, temp, user_data);
    }

    xlvfree(idxl);
    xlvfree(idxr);
    xlvfree(win_range);

    return;
}

static long get_total_freq_len_from_posit(spLVector shift, spLVector posit)
{
    return posit->data[posit->length - 1] + shift->data[0];
}

static spLVector xlvcalcposit(spLVector shift, long *o_total_freq_len)
{
    spLVector posit;

    // posit = cumsum(shift)-shift(1); % Calculate positions from shift vector
    posit = xlvcumsum(shift);
    if (o_total_freq_len != NULL) {
        *o_total_freq_len = posit->data[posit->length - 1];
    }
    lvscoper(posit, "-", shift->data[0]);

    return posit;
}

static void nsgtf_windowing_and_fft_input(spFFTRec fftrec, spDVector x, spDVector win, spDVector o_wx, spDVector o_f)
{
    long offset;

    if (o_wx->length != x->length) {
#if defined(SP_CQT_FFT_SIG_CENTER)
        offset = (o_wx->length - x->length) / 2;
#else
        offset = 0;
#endif
        dvzeros(o_wx, o_wx->length);
        dvadd(o_wx, offset, x, 0, x->length, SP_FALSE);
    } else {
        dvcopy(o_wx, x);
    }
    if (win != NODATA) {
        dvoper(o_wx, "*", win);
    }
    dvizeros(o_f, o_f->length);
    dvrcopyr(o_f, o_wx);
    dvfftex(fftrec, o_f);

    return;
}

static spDVector xdvcqt_exec_fft_input(spFFTRec fftrec, spDVector x, long offset_or_center, long length, spDVector win, spBool center_flag, spBool real_fft_flag)
{
    long fftl;
    long offset;
    spDVector f;

    fftl = spGetFFTLengthSpecified(fftrec);
    spDebug(80, "xdvcqt_exec_fft_input", "fftl = %ld, length = %ld\n", fftl, length);
    f = xdvzeros(fftl);
    if (win != NODATA && length <= 0) {
        length = win->length;
        if (center_flag) {
            offset = offset_or_center - length / 2;
        } else {
            offset = offset_or_center;
        }
    } else if (length <= 0) {
        if (center_flag) {
            length = fftl;
            offset = offset_or_center - fftl / 2;
        } else {
            offset = offset_or_center;
            length = x->length - offset;
        }
    } else {
        if (center_flag) {
            offset = offset_or_center - length / 2;
        } else {
            offset = offset_or_center;
        }
    }
    spDebug(80, "xdvcqt_exec_fft_input", "offset = %ld, length = %ld, fftl = %ld\n", offset, length, fftl);
    if (win != NODATA && offset != 0) {
        spDVector x2;
        x2 = xdvcut(x, offset, length);
        dvoper(x2, "*", win);
        dvadd(f, 0, x2, 0, length, SP_FALSE);
        xdvfree(x2);
    } else {
        dvadd(f, 0, x, offset, length, SP_FALSE);
        if (win != NODATA) {
            dvoper(f, "*", win);
        }
    }

    if (real_fft_flag) {
        dvrfftex(fftrec, f);
    } else {
        dvfftex(fftrec, f);
    }
    
    return f;
}

static spDVector xdvsnsgtf_real_fft_input(spFFTRec fftrec, spDVector x, long offset, long length, spDVector win, long total_freq_len,
                                          double *o_length_crop_factor)
{
    long Ls;
    long fill;
    spDVector f;

    f = xdvcqt_exec_fft_input(fftrec, x, offset, length, win, SP_FALSE, SP_FALSE);

    //Ls = x->length;
    Ls = f->length;

    //% A small amount of zero-padding might be needed (e.g. for scale frames)
    //fill = sum(shift)-Ls;
    fill = total_freq_len - Ls;
    if (fill > 0) {
        spDVector ffill;
        //f = [f;zeros(fill,CH)];
        ffill = xdvrizeros(f->length + fill);
        dvcopy(ffill, f);
        xdvfree(f);
        f = ffill;
    }

    if (o_length_crop_factor != NULL) *o_length_crop_factor = (double)length / (double)Ls;
    
    return f;
}

static spDVector xdvsnsgtf_real_get_window_for_block(long fft_length, long blocksize, double transition_factor)
{
    long i, n;
    long N;
    long windowstartpos;
    long transitionsize;
    long endpos;
    long halfblocksize;
    long leftstartpos;
    long rightstartpos;
    spDVector win;

    if (fft_length <= 0) fft_length = blocksize;

    transition_factor = MAX(transition_factor, 2.0);
    
    win = xdvzeros(fft_length);

    if (fft_length > blocksize) {
#if defined(SP_CQT_FFT_SIG_CENTER)
        windowstartpos = (fft_length - blocksize) / 2;
#else
        windowstartpos = 0;
#endif
    } else {
        windowstartpos = 0;
    }
    transitionsize = 2 * (long)spRound(0.5 * (double)blocksize / transition_factor);
    halfblocksize = blocksize / 2;
    leftstartpos = windowstartpos + halfblocksize / 2 - transitionsize / 2;
    rightstartpos = windowstartpos + blocksize - halfblocksize / 2 - transitionsize / 2;
    N = transitionsize * 2;

    endpos = leftstartpos + transitionsize - 1;
    for (i = leftstartpos; i < endpos; i++) {
        n = i - leftstartpos + 1;
        win->data[i] = 0.5 * (1.0 - cos(2.0 * PI * n / (double)N));
    }
    endpos = rightstartpos;
    for (; i < endpos; i++) {
        win->data[i] = 1.0;
    }
    endpos = rightstartpos + transitionsize - 1;
    for (; i < endpos; i++) {
        n = i - rightstartpos + transitionsize + 1;
        win->data[i] = 0.5 * (1.0 - cos(2.0 * PI * n / (double)N));
    }
    
    return win;
}

static spDVector xdvsnsgtf_real_get_block(spDVector x, long blocksize, long center_pos)
{
    spDVector xc;

    xc = xdvcut(x, center_pos - blocksize / 2, blocksize);
    
    return xc;
}

static long nsgtf_real_calc_total_nbins(spDVectors g, spLVector posit, long fftl)
{
    long ii;
    
    for (ii = 0;; ii++) {
        if (posit->data[ii] - g->vector[ii]->length / 2 > fftl / 2) {
            break;
        }
    }

    return ii;
}

static spDVectors xdvsnsgtf_real_alloc_vectors(spDVector M, long total_nbins)
{
    long ii;
    spDVectors c;
    
    c = xdvsalloc(total_nbins);
    
    for (ii = 0; ii < total_nbins; ii++) {
        c->vector[ii] = xdvrizeros((long)spRound(M->data[ii]));
    }

    return c;
}

static void nsgtf_real_clear_vectors(spDVectors c)
{
    long ii;
    
    for (ii = 0; ii < c->num_vector; ii++) {
        dvrizeros(c->vector[ii], c->vector[ii]->length);
    }

    return;
}

static spBool nsgtf_real_mainloop(spFFTRec fftrec, spDVectors g, spLVector shift,
                                  spBool phasemode_global, spLVector posit, spDVector f, spDVectors io_c,
                                  spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long ii;
    long total_nbins;
    int percent;
    double progress_incr;
    double progress_percent;
    double callbacked_percent;
    spBool proceed_flag = SP_TRUE;

    total_nbins = io_c->num_vector;
    spDebug(50, "nsgtf_real_mainloop", "total_nbins = %ld\n", total_nbins);
    
    progress_percent = io_progress_percent != NULL ? *io_progress_percent : 30.0;
    progress_incr = (99.5 - progress_percent) / (double)total_nbins;
    callbacked_percent = progress_percent;

    for (ii = 0; ii < total_nbins; ii++) {
        cqt_current_freq_index = ii;
        
        spDebug(80, "nsgtf_real_mainloop", "ii = %ld / %ld\n", ii, total_nbins);
        nsgtf_real_each(fftrec, f, ii, g->vector[ii], 1.0, posit->data[ii], phasemode_global, io_c->vector[ii],
                        callback_func, cqtrec, user_data);
#if 0
        if (ii == 1 && cqt_current_block_index == /*2*/103) {
            spDebug(80, "nsgtf_real_mainloop", "io_c->vector[%ld]->length = %ld\n", ii, io_c->vector[ii]->length);
            dvdump(io_c->vector[ii]); exit(0);
            //dvdump(g->vector[ii]); exit(0);
        }
#endif
        if (callback_func != NULL) {
            if (progress_percent - callbacked_percent >= 1.0) {
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                    break;
                }
                callbacked_percent = percent;
            }
            callback_func(cqtrec, SP_CQT_FORWARD_CHANNEL_FINISHED_CALLBACK, &ii, io_c->vector[ii], user_data);
        }
        
        progress_percent += progress_incr;
    }

    return proceed_flag;
}

static spBool xcqtprepare(spFFTRec fftrec, double fmin, double fmax, long bins, double gamma,
                          long refsize, double stride, long minM, double fs, spCQTWindowValueFunc winfunc,
                          spCQTOptionMask options, spDVectors *xo_g, spLVector *xo_shift, spDVector *xo_M, long *o_maxM, double *o_stride,
                          spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    int percent;
    spBool proceed_flag = SP_TRUE;
    long fft_length;
    double orig_stride;
    spDVectors g;
    spLVector shift;
    spDVector M;
    spDVector normFacVec;
    spBool stride_status;
    spBool flag;

    if (fmax <= fmin) {
        fmax = fs / 2.0;
    }

    fft_length = spGetFFTLengthSpecified(fftrec);
    
    proceed_flag = xnsgcqwin(fmin, fmax, bins, gamma, fs, fft_length, winfunc, &g, &shift, &M,
                             callback_func, cqtrec, user_data, io_progress_percent);
    //dvdump(M); exit(0);
    //dvdump(shift); exit(0);
    if (proceed_flag == SP_FALSE) {
        flag = SP_FALSE;
        goto bail;
    }
    
    orig_stride = stride;
    stride_status = dvcqtupdateM(M, fmin, fmax, refsize, options, o_maxM, &stride);
    
    if (stride_status ==  SP_FALSE && (options & SP_CQT_OPTION_ERROR_STOP_BY_WRONG_REFERENCE_STRIDE)) {
        xdvsfree(g); xlvfree(shift); xdvfree(M);
        flag = SP_FALSE;
    } else {
        spDebug(50, "xcqtprepare", "refsize = %ld, stride = %f\n", refsize, stride);
        if (refsize > 0) {
            dvcqtcorrectM(M, refsize, fft_length, minM, o_maxM, &stride);
        }
        //dvdump(M); exit(0);
        
        if (callback_func != NULL) {
            callback_func(cqtrec, SP_CQT_PREPARE_BANDWIDTHS_REFINED_CALLBACK, M, NULL, user_data);
        }
        
        normFacVec = xdvcalcnormlizevec(g, M, fft_length, options, callback_func, cqtrec, user_data, io_progress_percent);
        //dvdump(normFacVec); exit(0);
        if (callback_func != NULL && proceed_flag == SP_TRUE) {
            percent = 96;
            if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                goto bail2;
            }
            callback_func(cqtrec, SP_CQT_PREPARE_NORMALIZE_VECTOR_OBTAINED_CALLBACK, normFacVec, NULL, user_data);
        }

        cqtapplynormlizevec(normFacVec, g, callback_func, cqtrec, user_data, io_progress_percent);
        //dvdump(g->vector[0]); exit(0);
        //dvdump(g->vector[2]); exit(0);
        //dvdump(g->vector[g->num_vector/2-1]); exit(0);
        //dvdump(g->vector[g->num_vector/2]); exit(0);
        if (callback_func != NULL) {
            percent = 97;
            if ((proceed_flag = callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                goto bail2;
            }
            callback_func(cqtrec, SP_CQT_PREPARE_ANALYSIS_FILTERS_OBTAINED_CALLBACK, g, NULL, user_data);
        }

      bail2:
        if (xo_g != NULL && proceed_flag == SP_TRUE) {
            *xo_g = g;
        } else {
            xdvsfree(g);
        }
        if (xo_shift != NULL && proceed_flag == SP_TRUE) {
            *xo_shift = shift;
        } else {
            xlvfree(shift);
        }
        if (xo_M != NULL && proceed_flag == SP_TRUE) {
            *xo_M = M;
        } else {
            xdvfree(M);
        }
    
        xdvfree(normFacVec);
        flag = proceed_flag;
    }

  bail:
    if (io_progress_percent != NULL) *io_progress_percent = 98.0;

    if (o_stride != NULL) *o_stride = stride;
    
    return flag;
}

spDVectors xdvscqtwholesigtest(spFFTRec fftrec, spDVector x, double fmin, double fmax, long bins,
                               double gamma, long refsize, double stride, long minM, double fs, 
                               spCQTWindowValueFunc winfunc, spCQTOptionMask options, 
                               spDVectors *xo_g, spLVector *xo_shift, spDVector *xo_M, long *o_maxM, double *o_stride)
{
    long total_freq_len;
    spDVectors g;
    spLVector shift;
    spDVector M;
    spDVector f;
    spLVector posit;
    spDVectors c;

    if (xcqtprepare(fftrec, fmin, fmax, bins, gamma, refsize, stride, minM, fs,
                    winfunc, options, &g, &shift, &M, o_maxM, o_stride, NULL, NULL, NULL, NULL) == SP_FALSE) {
        return NODATA;
    }

    posit = xlvcalcposit(shift, &total_freq_len);
    spDebug(50, "xdvscqtwholesigtest", "total_freq_len = %ld\n", total_freq_len);
    //lvdump(posit); exit(0);
    f = xdvsnsgtf_real_fft_input(fftrec, x, 0, x->length, NODATA, total_freq_len, NULL);
    //dvdump(f); exit(0);

    c = xdvsnsgtf_real_alloc_vectors(M, nsgtf_real_calc_total_nbins(g, posit, f->length));
    
    nsgtf_real_mainloop(fftrec, g, shift, (options & SP_CQT_OPTION_PHASE_MODE_GLOBAL) ? SP_TRUE : SP_FALSE, posit, f, c,
                        NULL, NULL, NULL, NULL);

    if (xo_g != NULL) {
        *xo_g = g;
    } else {
        xdvsfree(g);
    }
    if (xo_shift != NULL) {
        *xo_shift = shift;
    } else {
        xlvfree(shift);
    }
    if (xo_M != NULL) {
        *xo_M = M;
    } else {
        xdvfree(M);
    }
    
    xlvfree(posit);
    xdvfree(f);

    return c;
}

long xcqtblocktest(spFFTRec fftrec, spDVector x, double fmin, double fmax, long bins, double gamma,
                   long blocksize, double stride, long minM, double transition_factor, double fs,
                   spCQTWindowValueFunc winfunc, spCQTOptionMask options, 
                   spDVectors *xo_g, spLVector *xo_shift, spDVector *xo_M, spDVectors **xo_c_list, long *o_maxM, double *o_stride)
{
    long m;
    long nblock;
    long total_freq_len;
    long center_pos;
    spDVectors g;
    spLVector shift;
    spDVector M;
    spLVector posit;
    spDVectors c;
    spDVectors *c_list;
    spDVector win;
    spDVector f;
    spDVector xc;

    if (xcqtprepare(fftrec, fmin, fmax, bins, gamma, blocksize, stride, minM, fs,
                    winfunc, options, &g, &shift, &M, o_maxM, o_stride, NULL, NULL, NULL, NULL) == SP_FALSE) {
        return -1;
    }

    posit = xlvcalcposit(shift, &total_freq_len);
    //lvdump(posit); exit(0);

    win = xdvsnsgtf_real_get_window_for_block(spGetFFTLengthSpecified(fftrec), blocksize, transition_factor);
    //dvdump(win); exit(0);

    nblock = (long)ceil((double)x->length / (double)(blocksize / 2));
    spDebug(50, "xcqtblocktest", "total_freq_len = %ld, nblock = %ld\n", total_freq_len, nblock);
    
    c_list = xspAlloc(nblock, spDVectors);
    center_pos = 0;

    for (m = 0; m < nblock; m++) {
        cqt_current_block_index = m;
        spDebug(80, "xcqtblocktest", "m = %ld / %ld\n", m, nblock);
        xc = xdvsnsgtf_real_get_block(x, blocksize, center_pos);
        f = xdvsnsgtf_real_fft_input(fftrec, xc, 0, xc->length, win, total_freq_len, NULL);
        
        c = xdvsnsgtf_real_alloc_vectors(M, nsgtf_real_calc_total_nbins(g, posit, f->length));
        
        nsgtf_real_mainloop(fftrec, g, shift, (options & SP_CQT_OPTION_PHASE_MODE_GLOBAL) ? SP_TRUE : SP_FALSE, posit, f, c,
                            NULL, NULL, NULL, NULL);
        c_list[m] = c;

        center_pos += win->length / 2;

        xdvfree(f);
        xdvfree(xc);
    }

    if (xo_g != NULL) {
        *xo_g = g;
    } else {
        xdvsfree(g);
    }
    if (xo_shift != NULL) {
        *xo_shift = shift;
    } else {
        xlvfree(shift);
    }
    if (xo_M != NULL) {
        *xo_M = M;
    } else {
        xdvfree(M);
    }
    if (xo_c_list != NULL) {
        *xo_c_list = c_list;
    } else {
        for (m = 0; m < nblock; m++) {
            xdvsfree(c_list[m]);
        }
        xspFree(c_list);
    }
    
    xlvfree(posit);
    xdvfree(win);

    return nblock;
}

spLVector xlvnsdual_update_diagonal_each(spDVector g, long Ls, double M, long posit, spDVector io_diagonal)
{
    long k;
    long Lg;
    long Lghalf_ceil;
    long Lghalf_floor;
    spLVector win_range;
    spDVector g2;

    Lg = g->length;
    Lghalf_floor = Lg / 2;
    Lghalf_ceil = (long)ceil((double)Lg / 2.0);
    win_range = xlvcalcwinrange(Lghalf_floor, Lghalf_ceil, Ls, posit);

    g2 = xdvfftshift(g, g->length);
    dvscoper(g2, "^", 2.0);
    dvscoper(g2, "*", M);

    for (k = 0; k < win_range->length; k++) {
        io_diagonal->data[win_range->data[k]] += g2->data[k];
    }

    xdvfree(g2);
    
    return win_range;
}

spDVector xdvnsdual_calcgd(spDVector g, spLVector win_range, spDVector diagonal)
{
    long k;
    spDVector gd;

    gd = xdvfftshift(g, g->length);

    for (k = 0; k < win_range->length; k++) {
        gd->data[k] /= diagonal->data[win_range->data[k]];
    }

    dvifftshift(gd);

    return gd;
}

spDVectors xdvsnsdual(spDVectors g, spLVector shift, spLVector posit, spDVector M, long Ls,
                      spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long ii;
    long N;
    spDVector diagonal;
    spLVectors win_ranges;
    spDVectors gd = NODATA;
    int percent;
    double progress_incr;
    double progress_percent;
    double callbacked_percent;
    spBool proceed_flag = SP_TRUE;

    //% Setup the necessary parameters
    N = shift->length;

    diagonal = xdvzeros(Ls);
    win_ranges = xlvsalloc(N);

    //% Construct the diagonal of the frame operator matrix explicitly
    
    for (ii = 0; ii < N; ii++) {
        cqt_current_freq_index = ii;
        win_ranges->vector[ii] = xlvnsdual_update_diagonal_each(g->vector[ii], Ls, M->data[ii], posit->data[ii], diagonal);
    }
    if (callback_func != NULL) {
        percent = 3;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_INVERSE_FRAME_OPERATOR_DIAGONAL_OBTAINED_CALLBACK, diagonal, win_ranges, user_data);
    }
            
    //% Using the frame operator and the original window sequence, compute
    //% the dual window sequence
    
    gd = xdvsalloc(N);
    
    progress_percent = 3.0;
    progress_incr = (19.5 - progress_percent) / (double)N;
    callbacked_percent = progress_percent;

    for (ii = 0; ii < N; ii++) {
        spDebug(100, "xdvsnsdual", "ii = %ld / %ld\n", ii, N);
        cqt_current_freq_index = ii;
        gd->vector[ii] = xdvnsdual_calcgd(g->vector[ii], win_ranges->vector[ii], diagonal);
#if 0
        if (ii == /*0*//*1*/2) {
            //dvdump(g->vector[ii]); exit(0);
            dvdump(gd->vector[ii]); exit(0);
        }
#endif
        if (callback_func != NULL) {
            if (progress_percent - callbacked_percent >= 1.0) {
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                    goto bail;
                }
                callbacked_percent = percent;
            }
            callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_SYNTHESIS_FILTER_OBTAINED_CALLBACK, &ii, gd->vector[ii], user_data);
        }
        
        progress_percent += progress_incr;
    }

    if (callback_func != NULL) {
        percent = 20;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_INVERSE_SYNTHESIS_FILTERS_OBTAINED_CALLBACK, gd, NULL, user_data);
    }

  bail:
    xdvfree(diagonal);
    xlvsfree(win_ranges);
    if (proceed_flag == SP_FALSE) {
        if (io_progress_percent != NULL) *io_progress_percent = SP_CQT_PROGRESS_PERCENT_STOPPED;
        xdvsfree(gd);
        return NODATA;
    } else {
        if (io_progress_percent != NULL) *io_progress_percent = 20.0;
        return gd;
    }
}

static spDVector xdvnsigtf_real_each(long channel, spDVector c, spDVector g, long posit, spBool phasemode_global,
                                     spCQTCallbackFunc callback_func, void *cqtrec, void *user_data)
{
    spDVector temp;
    
    //temp = fft(c{ii},[],1)*length(c{ii});
    temp = xdvfft(c, c->length);
    dvscoper(temp, "*", (double)c->length);
    if (callback_func != NULL) {
        callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_FFT_FINISHED_CALLBACK, &channel, temp, user_data);
    }

    if (phasemode_global == SP_TRUE) {
        dvcqtglobalmapfreq(temp, posit, c->length, SP_TRUE);
        if (callback_func != NULL) {
            callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_PHASE_UNGLOBALIZE_FINISHED_CALLBACK, &channel, temp, user_data);
        }
    }

    return temp;
}

static void nsigtf_real_update_fr(long channel, spDVector g, spDVector spc, long posit, spDVector io_fr,
                                  spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, void *mutex)
{
    long k;
    long NN;
    long Lg;
    long Lghalf_ceil;
    long Lghalf_floor;
    long idxl, idxr;
    double g_value;
    spLVector win_range;
    spDVector buffer;
    
    NN = io_fr->length;
    Lg = g->length;
    Lghalf_floor = Lg / 2;
    Lghalf_ceil = (long)ceil((double)Lg / 2.0);

    //win_range = mod(posit(ii)+(-floor(Lg/2):ceil(Lg/2)-1),NN)+1;
    win_range = xlvcalcwinrange(Lghalf_floor, Lghalf_ceil, NN, posit);

    buffer = xdvrialloc(win_range->length);

    for (k = 0; k < Lghalf_floor; k++) {
        idxl = spLMod(spc->length - Lghalf_floor + k, spc->length);
        buffer->data[k] = spc->data[idxl];
        buffer->imag[k] = spc->imag[idxl];
    }
    for (k = 0; k < Lghalf_ceil; k++) {
        idxr = spLMod(k, spc->length);
        buffer->data[k + Lghalf_floor] = spc->data[idxr];
        buffer->imag[k + Lghalf_floor] = spc->imag[idxr];
    }
#if 0
    if (cqt_current_freq_index == 3) {
        //dvdump(spc); exit(0);
        dvdump(buffer); exit(0);
    }
#endif
    if (callback_func != NULL) {
        callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_SPECTRUM_OBTAINED_CALLBACK, &channel, buffer, user_data);
    }

    if (mutex != NULL) spLockMutex(mutex);
    for (k = 0; k < win_range->length; k++) {
        if (k < Lghalf_floor) {
            g_value = g->data[Lg - Lghalf_floor + k];
        } else {
            g_value = g->data[k - Lghalf_floor];
        }
        io_fr->data[win_range->data[k]] += g_value * buffer->data[k];
        io_fr->imag[win_range->data[k]] += g_value * buffer->imag[k];
    }
    if (callback_func != NULL) {
        callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_CONVOLUTION_FINISHED_CALLBACK, &channel, io_fr, user_data);
    }
    if (mutex != NULL) spUnlockMutex(mutex);

    xlvfree(win_range);
    xdvfree(buffer);

    return;
}

static spBool nsigtf_real_ifft_fr(spFFTRec fftrec, spDVector io_fr, spBool block_flag,
                                  spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    int percent;
    spBool proceed_flag = SP_TRUE;
    
    dvfftturn(io_fr);
    //dvdump(io_fr); exit(0);
    
    dvifftex(fftrec, io_fr);
    if (callback_func != NULL) {
        double target_percent;

        target_percent = block_flag ? 99.0 : 100.0;
        percent = io_progress_percent != NULL ? (int)spRound(*io_progress_percent) : (int)target_percent;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_INVERSE_FINAL_IFFT_FINISHED_CALLBACK, io_fr, NULL, user_data);
        if (io_progress_percent != NULL) *io_progress_percent = target_percent;
    }
    
  bail:
    return proceed_flag;
}

static spBool nsigtf_real_mainloop(spFFTRec fftrec, spDVectors c, spDVectors g, spLVector posit, spBool phasemode_global, spBool block_flag,
                                   spDVector o_fr, spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long ii;
    long N;
    spDVector temp;
    int percent;
    double target_percent;
    double progress_incr;
    double progress_percent;
    double callbacked_percent;
    spBool proceed_flag = SP_TRUE;

    N = c->num_vector;

    dvrizeros(o_fr, o_fr->length);
    
    progress_percent = io_progress_percent != NULL ? *io_progress_percent : 20.0;
    target_percent = 90.0;
    progress_incr = (target_percent - progress_percent) / (double)N;
    callbacked_percent = progress_percent;

    for (ii = 0; ii < N; ii++) {
        spDebug(100, "nsigtf_real_mainloop", "ii = %ld / %ld\n", ii, N);
        cqt_current_freq_index = ii;
        temp = xdvnsigtf_real_each(ii, c->vector[ii], g->vector[ii], posit->data[ii], phasemode_global,
                                   callback_func, cqtrec, user_data);
        
        nsigtf_real_update_fr(ii, g->vector[ii], temp, posit->data[ii], o_fr, callback_func, cqtrec, user_data, NULL);
#if 0
        if (ii == /*0*//*1*//*2*/3) {
            //dvdump(temp); exit(0);
            dvdump(o_fr); exit(0);
        }
#endif
        if (callback_func != NULL) {
            if (progress_percent - callbacked_percent >= 1.0) {
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
                    goto bail;
                }
                callbacked_percent = percent;
                if (io_progress_percent != NULL) *io_progress_percent = callbacked_percent;
            }
            callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_FINISHED_CALLBACK, &ii, o_fr, user_data);
        }
        
        xdvfree(temp);
        
        progress_percent += progress_incr;
    }

    //dvdump(o_fr); exit(0);
    
    proceed_flag = nsigtf_real_ifft_fr(fftrec, o_fr, block_flag, callback_func, cqtrec, user_data, io_progress_percent);
    
  bail:
    return proceed_flag;
}

static spDVector xdvnsigtf_real(spFFTRec fftrec, spDVectors c, spDVectors g, spLVector shift, spLVector posit, long NN,
                                spBool phasemode_global, spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    spDVector fr;

    fr = xdvrizeros(NN);

    nsigtf_real_mainloop(fftrec, c, g, posit, phasemode_global, SP_FALSE, fr, callback_func, cqtrec, user_data, io_progress_percent);
    
    dvreal(fr);
    //dvdump(fr); exit(0);

    return fr;
}

spDVector xdvicqttest(spFFTRec fftrec, spDVectors c, spDVectors g, spLVector shift, spLVector posit, spDVector M, long Ls,
                      spBool phasemode_global)
{
    spDVectors gd;
    spDVector x;

    gd = xdvsnsdual(g, shift, posit, M, Ls, NULL, NULL, NULL, NULL);

    x = xdvnsigtf_real(fftrec, c, gd, shift, posit, Ls, phasemode_global, NULL, NULL, NULL, NULL);

    xdvsfree(gd);

    return x;
}

static long icqtolablock(spFFTRec fftrec, long blocksize, long offset, spDVector fr_buffer, spDVector io_x,
                         spCQTCallbackFunc callback_func, void *cqtrec, void *user_data)
{
    long fftl;
    long fr_offset;
    long fr_length;
    spBool proceed_flag = SP_TRUE;
    
    fftl = spGetFFTLengthSpecified(fftrec);
    
    if (blocksize != fftl) {
#if defined(SP_CQT_FFT_SIG_CENTER)
        fr_offset = (fftl - blocksize) / 2;
#else
        fr_offset = 0;
#endif
        fr_length = blocksize;
    } else {
        fr_offset = 0;
        fr_length = MIN(fr_buffer->length, fftl);
    }
    dvadd(io_x, offset, fr_buffer, fr_offset, fr_length, SP_TRUE);
    if (callback_func != NULL) {
        int percent = 100;
        if ((proceed_flag = callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK, &percent, NULL, user_data)) == SP_FALSE) {
            goto bail;
        }
        callback_func(cqtrec, SP_CQT_INVERSE_BLOCK_OVERLAP_ADD_FINISHED_CALLBACK, io_x, &offset, user_data);
    }

  bail:
    return proceed_flag;
}

static long icqtprocessblock(spFFTRec fftrec, spDVectors c, spDVectors gd, spLVector posit, spBool phasemode_global,
                             long blocksize, long offset, spDVector fr_buffer, spDVector io_x,
                             spCQTCallbackFunc callback_func, void *cqtrec, void *user_data, double *io_progress_percent)
{
    long fftl;
    long fr_offset;
    long fr_length;
    spBool proceed_flag = SP_TRUE;
    
    if ((proceed_flag = nsigtf_real_mainloop(fftrec, c, gd, posit, phasemode_global, SP_TRUE, fr_buffer,
                                             callback_func, cqtrec, user_data, io_progress_percent)) == SP_FALSE) {
        goto bail;
    }

    if ((proceed_flag = icqtolablock(fftrec, blocksize, offset, fr_buffer, io_x,
                                     callback_func, cqtrec, user_data)) == SP_FALSE) {
        goto bail;
    }
    if (io_progress_percent != NULL) *io_progress_percent = 100.0;

    offset += blocksize - (blocksize / 2);

  bail:
    if (proceed_flag == SP_FALSE) {
        if (io_progress_percent != NULL) *io_progress_percent = SP_CQT_PROGRESS_PERCENT_STOPPED;
    }
    return offset;
}

spDVector xdvicqtblocktest(spFFTRec fftrec, spDVectors *c_list, long num_c_list, spDVectors g,
                           spLVector shift, spLVector posit, spDVector M, long Ls, long blocksize, spBool phasemode_global)
{
    long NN;
    long m;
    long nblock;
    long offset;
    spDVectors gd;
    spDVector fr;
    spDVector x;

    nblock = num_c_list;
    
    gd = xdvsnsdual(g, shift, posit, M, Ls, NULL, NULL, NULL, NULL);

    posit = xlvcalcposit(shift, &NN);
    fr = xdvrizeros(NN);
    
    x = xdvzeros(nblock * (blocksize / 2));
    offset = -blocksize / 2;

    for (m = 0; m < nblock; m++) {
        cqt_current_block_index = m;
        spDebug(80, "xdvicqtblocktest", "m = %ld / %ld\n", m, nblock);
        offset = icqtprocessblock(fftrec, c_list[m], gd, posit, phasemode_global, blocksize, offset, fr, x,
                                  NULL, NULL, NULL, NULL);
    }

    xdvfree(fr);
    xdvsfree(gd);

    return x;
}

#if 0
#endif

#if defined(SP_CQT_USE_THREAD)
static spBool nsgtf_real_partial(spCQTRec cqtrec, long channel_start, long channel_end, spDVectors g, spLVector shift,
                                 spBool phasemode_global, spLVector posit, spDVector f, spDVectors io_c)
{
    long ii;
    long channels;
    int percent;
    double progress_incr;
    double progress_percent;
    spBool proceed_flag = SP_TRUE;

    channels = channel_end - channel_start + 1;

    progress_percent = cqtrec->start_progress_percent;
    progress_incr = (cqtrec->target_progress_percent - progress_percent) / (double)cqtrec->num_thread / (double)channels;
    spDebug(80, "nsgtf_real_partial", "progress_percent = %f, progress_incr = %f\n", progress_percent, progress_incr);
    
    for (ii = channel_start; ii <= channel_end; ii++) {
        if (cqtrec->proceed_flag == SP_FALSE) {
            proceed_flag = SP_FALSE;
            break;
        }
        
        spDebug(80, "nsgtf_real_partial", "ii = %ld\n", ii);
        nsgtf_real_each(cqtrec->fftrec, f, ii, g->vector[ii], 1.0, posit->data[ii], phasemode_global, io_c->vector[ii],
                        cqtrec->callback_func, cqtrec, cqtrec->user_data);
        
        if (cqtrec->callback_func != NULL) {
            spLockMutex(cqtrec->mutex);
            if (progress_percent - cqtrec->progress_percent >= 1.0) {
                spDebug(80, "nsgtf_real_partial", "ii = %ld, progress_percent = %f, cqtrec->progress_percent = %f\n",
                        ii, progress_percent, cqtrec->progress_percent);
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK,
                                                          &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
                    goto bail;
                }
                cqtrec->progress_percent = percent;
                progress_percent = cqtrec->progress_percent;
            }
            spUnlockMutex(cqtrec->mutex);
            
            cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_CHANNEL_FINISHED_CALLBACK, &ii, io_c->vector[ii], cqtrec->user_data);
        }
        
        progress_percent += progress_incr;
    }

  bail:
    return proceed_flag;
}

static spBool nsigtf_real_partial(spCQTRec cqtrec, long channel_start, long channel_end,
                                  spDVectors c, spDVectors g, spLVector posit, spBool phasemode_global, spDVector o_fr)
{
    long ii;
    long channels;
    spDVector temp;
    int percent;
    double progress_incr;
    double progress_percent;
    spBool proceed_flag = SP_TRUE;

    channels = channel_end - channel_start + 1;

    progress_percent = cqtrec->start_progress_percent;
    progress_incr = (cqtrec->target_progress_percent - progress_percent) / (double)cqtrec->num_thread / (double)channels;
    spDebug(80, "nsigtf_real_partial", "progress_percent = %f, progress_incr = %f\n", progress_percent, progress_incr);
    
    for (ii = channel_start; ii <= channel_end; ii++) {
        if (cqtrec->proceed_flag == SP_FALSE) {
            proceed_flag = SP_FALSE;
            break;
        }
        spDebug(100, "nsigtf_real_partial", "ii = %ld\n", ii);
        temp = xdvnsigtf_real_each(ii, c->vector[ii], g->vector[ii], posit->data[ii], phasemode_global,
                                   cqtrec->callback_func, cqtrec, cqtrec->user_data);
        
        nsigtf_real_update_fr(ii, g->vector[ii], temp, posit->data[ii], o_fr,
                              cqtrec->callback_func, cqtrec, cqtrec->user_data, cqtrec->mutex);
        
        if (cqtrec->callback_func != NULL) {
            spLockMutex(cqtrec->mutex);
            if (progress_percent - cqtrec->progress_percent >= 1.0) {
                spDebug(80, "nsigtf_real_partial", "ii = %ld, progress_percent = %ld cqtrec->progress_percent = %ld\n",
                        ii, progress_percent, cqtrec->progress_percent);
                percent = (int)spRound(progress_percent);
            
                if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_INVERSE_PROGRESS_CALLBACK,
                                                          &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
                    goto bail;
                }
                cqtrec->progress_percent = percent;
                progress_percent = cqtrec->progress_percent;
            }
            spUnlockMutex(cqtrec->mutex);
            
            cqtrec->callback_func(cqtrec, SP_CQT_INVERSE_CHANNEL_FINISHED_CALLBACK, &ii, o_fr, cqtrec->user_data);
        }
        
        xdvfree(temp);
    }

  bail:
    return proceed_flag;
}

spThreadReturn spCQTThread(void *data)
{
    spBool proceed_flag = SP_TRUE;
    spCQTRec cqtrec;
    spCQTThreadRec *thread_rec = (spCQTThreadRec *)data;

    cqtrec = thread_rec->cqtrec;
    spDebug(100, "spCQTThread", "thread generated: thread_rec->index = %ld\n", thread_rec->index);
    
    while (cqtrec->terminate_thread == SP_FALSE) {
        spDebug(100, "spCQTThread", "%ld: before spWaitEvent(thread_rec->event)\n", thread_rec->index);
        spWaitEvent(thread_rec->event);
        spDebug(100, "spCQTThread", "%ld: after spWaitEvent(thread_rec->event)\n", thread_rec->index);
        
        if (cqtrec->terminate_thread == SP_TRUE) {
            break;
        }

        if (thread_rec->forward_finished == SP_FALSE) {
            proceed_flag = nsgtf_real_partial(cqtrec, thread_rec->channel_start, thread_rec->channel_end, cqtrec->g, cqtrec->shift,
                                              cqtrec->phasemode_global, cqtrec->posit, cqtrec->current_f, cqtrec->current_c);
            thread_rec->forward_finished = SP_TRUE;
        } else {
            proceed_flag = nsigtf_real_partial(cqtrec, thread_rec->channel_start, thread_rec->channel_end,
                                               cqtrec->current_c, cqtrec->gd, cqtrec->posit, cqtrec->phasemode_global, cqtrec->current_fr);
        }

        spLockMutex(cqtrec->mutex);
	spDebug(100, "spCQTThread", "%ld: original num_computing = %ld\n", thread_rec->index, cqtrec->num_computing);
        cqtrec->proceed_flag = proceed_flag;
        --cqtrec->num_computing;
        if (cqtrec->num_computing <= 0) {
	    spDebug(80, "spCQTThread", "%ld: spSetEvent for event\n", thread_rec->index);
	    spSetEvent(cqtrec->event_for_process);  
        }
        spUnlockMutex(cqtrec->mutex);        
    }

    return SP_THREAD_RETURN_SUCCESS;
}

static spBool spCQTThreadInitialize(spCQTRec cqtrec)
{
    spBool status = SP_TRUE;
    
    spDebug(80, "spCQTThreadInitialize", "cqtrec->num_thread = %ld\n", cqtrec->num_thread);
    
    if (cqtrec->num_thread <= 0) {
        return SP_TRUE;
    }
    
    cqtrec->terminate_thread = SP_FALSE;
    
    if ((cqtrec->mutex = spCreateMutex(NULL)) != NULL
        && (cqtrec->event_for_process = spCreateEvent(SP_FALSE, SP_FALSE)) != NULL) {
        long k, m;
        long nch;
        long ch_nyquist;
        double total_cumM;
        double cumM_per_thread;
        double current_cumM;
        double prev_cumM;

        cqtrec->Mcumsum = xdvcumsum(cqtrec->M);
        ch_nyquist = /*cqtrec->M->length / 2*/cqtrec->total_nbins - 1;
        total_cumM = cqtrec->Mcumsum->data[ch_nyquist];
        
        cumM_per_thread = total_cumM / (double)cqtrec->num_thread;
        spDebug(80, "spCQTThreadInitialize", "cqtrec->num_thread = %ld, ch_nyquist = %ld, total_cumM = %f, cumM_per_thread = %f\n",
                cqtrec->num_thread, ch_nyquist, total_cumM, cumM_per_thread);
        
        cqtrec->thread_recs = xalloc(cqtrec->num_thread, struct spCQTThreadRec);
        memset(cqtrec->thread_recs, 0, sizeof(struct spCQTThreadRec) * cqtrec->num_thread);

        current_cumM = total_cumM;
        
        for (k = 0, nch = ch_nyquist; k < cqtrec->num_thread; k++) {
            if ((cqtrec->thread_recs[k].event = spCreateEvent(SP_FALSE, SP_FALSE)) == NULL) {
                status = SP_FALSE;
                break;
            }
            
            cqtrec->thread_recs[k].cqtrec = cqtrec;
            cqtrec->thread_recs[k].index = k;
            cqtrec->thread_recs[k].channel_end = nch;
            
            for (m = nch, prev_cumM = current_cumM;; m--) {
                current_cumM -= cqtrec->M->data[m];
                if (m == 0 || prev_cumM - current_cumM >= cumM_per_thread) {
                    break;
                }
            }

            cqtrec->thread_recs[k].channel_start = m;
            spDebug(80, "spCQTThreadInitialize", "Thread %ld: %ld-%ld\n",
                    k, cqtrec->thread_recs[k].channel_start, cqtrec->thread_recs[k].channel_end);

            if ((cqtrec->thread_recs[k].thread = spCreateThread(0, 0, spCQTThread, &cqtrec->thread_recs[k])) == NULL) {
                status = SP_FALSE;
                break;
            }

            if (m == 0) {
                cqtrec->num_thread = k + 1;
                spDebug(50, "spCQTThreadInitialize", "k = %ld, final cqtrec->num_thread = %ld\n",
                        k, cqtrec->num_thread);
                break;
            }
            nch = m - 1;
            
            current_cumM -= cumM_per_thread;
        }
    } else {
        status = SP_FALSE;
    }
    
    return status;
}

static spBool spCQTThreadFree(spCQTRec cqtrec)
{
    long nth;

    if (cqtrec->num_thread <= 0) return SP_FALSE;

    cqtrec->terminate_thread = SP_TRUE;
    
    if (cqtrec->event_for_process != NULL) {
        spSetEvent(cqtrec->event_for_process);
    }
        
    for (nth = 0; nth < cqtrec->num_thread; nth++) {
        if (cqtrec->thread_recs[nth].event != NULL) {
            spSetEvent(cqtrec->thread_recs[nth].event);
        }
        
        if (cqtrec->thread_recs[nth].thread != NULL) {
            spDebug(100, "spCQTThreadFree", "thread %ld waiting\n", nth); 
            spWaitThread(cqtrec->thread_recs[nth].thread);
            spDebug(100, "spCQTThreadFree", "thread %ld done\n", nth); 
            spDestroyThread(cqtrec->thread_recs[nth].thread);
        }
        
        if (cqtrec->thread_recs[nth].event != NULL) {
            spDestroyEvent(cqtrec->thread_recs[nth].event);
        }
        cqtrec->thread_recs[nth].cqtrec = NULL;
    }

    if (cqtrec->event_for_process != NULL) {
        spDestroyEvent(cqtrec->event_for_process);
        cqtrec->event_for_process = NULL;
    }
    if (cqtrec->mutex != NULL) {
        spDestroyMutex(cqtrec->mutex);
        cqtrec->mutex = NULL;
    }
        
    if (cqtrec->Mcumsum != NODATA) {
        xdvfree(cqtrec->Mcumsum);
        cqtrec->Mcumsum = NODATA;
    }
    cqtrec->current_f = NODATA;
    cqtrec->current_c = NODATA;
    cqtrec->current_fr = NODATA;
    
    xfree(cqtrec->thread_recs);
    cqtrec->thread_recs = NULL;
    cqtrec->terminate_thread = SP_FALSE;
    cqtrec->num_thread = 0;
    
    return SP_TRUE;
}
#endif

spBool spInitCQTConfig_(spCQTConfig *config, unsigned long version_id)
{
    if (config == NULL || version_id > SP_CQT_VERSION_ID) return SP_FALSE;
    
    memset(config, 0, sizeof(spCQTConfig));
    config->version_id = version_id;

    config->samp_freq = SP_CQT_DEFAULT_SAMP_FREQ;
    config->min_freq = SP_CQT_DEFAULT_MIN_FREQ;

    config->bins_per_octave = SP_CQT_DEFAULT_BINS_PER_OCTAVE;
    config->gamma = SP_CQT_DEFAULT_GAMMA;

    config->options = SP_CQT_OPTIONS_DEFAULT;
    
    config->reference_stride = SP_CQT_REFERENCE_STRIDE_DEFAULT;

    config->block_length_ms = SP_CQT_DEFAULT_BLOCK_LENGTH_MS;
    config->block_transition_factor = SP_CQT_DEFAULT_BLOCK_TRANSITION_FACTOR;
    config->min_sample_length_for_block = SP_CQT_DEFAULT_MIN_SAMPLE_LENGTH_FOR_BLOCK;

    config->num_thread = 0;
    
    return SP_TRUE;
}

spCQTOptionMask spConvCQTRasterizeMethodFromString(char *rasterize_method_string)
{
    if (strnone(rasterize_method_string)) return SP_CQT_OPTION_RASTERIZE_METHOD_NONE;

    if (rasterize_method_string[0] == '2' || strcaseeq(rasterize_method_string, "powerof2")
        || strcaseeq(rasterize_method_string, "power_of_2")) {
        return SP_CQT_OPTION_RASTERIZE_METHOD_POWER_OF_2;
    } else if (rasterize_method_string[0] == 'p' || rasterize_method_string[0] == 'P') {
        return SP_CQT_OPTION_RASTERIZE_METHOD_PIECEWISE;
    } else if (((rasterize_method_string[0] == 'f' || rasterize_method_string[0] == 'F') && rasterize_method_string[1] == '2')
               || strcaseeq(rasterize_method_string, "fullpowerof2")
               || strcaseeq(rasterize_method_string, "full_power_of_2")) {
        return SP_CQT_OPTION_RASTERIZE_METHOD_FULL_POWER_OF_2;
    } else if (rasterize_method_string[0] == 'f' || rasterize_method_string[0] == 'F') {
        return SP_CQT_OPTION_RASTERIZE_METHOD_FULL;
    } else {
        return SP_CQT_OPTION_RASTERIZE_METHOD_NONE;
    }
}

spCQTOptionMask spConvCQTNormalizeMethodFromString(char *normalize_method_string)
{
    if (strnone(normalize_method_string)) return SP_CQT_OPTION_NORMALIZE_METHOD_NONE;

    if (normalize_method_string[0] == 's' || normalize_method_string[0] == 'S') {
        return SP_CQT_OPTION_NORMALIZE_METHOD_SINE;
    } else if (normalize_method_string[0] == 'i' || normalize_method_string[0] == 'I') {
        return SP_CQT_OPTION_NORMALIZE_METHOD_IMPULSE;
    } else {
        return SP_CQT_OPTION_NORMALIZE_METHOD_NONE;
    }
}

double spCalcCQTGammaOfERB(long bins_per_octave /* 0: default */)
{
    double invbins;
    double Q;
    double gamma;

    if (bins_per_octave <= 0) {
        bins_per_octave = SP_CQT_DEFAULT_BINS_PER_OCTAVE;
    }

    invbins = 1.0 / (double)bins_per_octave;

    // Q = 2^(1/bins) - 2^(-1/bins);
    Q = pow(2.0, invbins) - pow(2.0, -invbins);

    // gamma = Q * 24.7 / 0.108; 0.108 = 24.7 * 4.37 / 1000;
    gamma = Q / (4.37 / 1000.0);

    return gamma;
}

long spCalcCQTMinFFTLength(double min_freq /* -1: default */, long bins_per_octave /* 0: default */, double samp_freq, spBool pow2flag)
{
    double invbins;
    double min_interval;
    long fftl;

    if (samp_freq <= 0.0) return -1;

    if (min_freq < 0.0) {
        min_freq = SP_CQT_DEFAULT_MIN_FREQ;
    }
    if (bins_per_octave <= 0) {
        bins_per_octave = SP_CQT_DEFAULT_BINS_PER_OCTAVE;
    }
    
    invbins = 1.0 / (double)bins_per_octave;

    // min_interval = min_freq * ((2^(1/bins_per_octave)) - 1);
    min_interval = min_freq * (pow(2.0, invbins) - 1.0);

    fftl = (long)ceil(samp_freq / min_interval);

    if (pow2flag) {
        return spPow2(spNextPow2(fftl));
    } else {
        return fftl;
    }
}

long spCalcCQTMinBlockLength(double min_freq /* -1: default */, long bins_per_octave /* 0: default */, double gamma,
                             double margin_factor /* 0.0: default=1.0 */, double samp_freq, spBool pow2flag, spBool min_freq_based_on_musical_note)
{
    double invbins;
    double fbas;
    double Q;
    double cqtbw;
    double length_d;
    long length;

    if (samp_freq <= 0.0) return -1;
    
    if (min_freq < 0.0) {
        min_freq = SP_CQT_DEFAULT_MIN_FREQ;
    } else if (min_freq_based_on_musical_note) {
        char note_name[4];
        int octave_index;
        double note_freq;
        if (spFreqToNoteName(min_freq, 3, note_name, &octave_index, NULL) == SP_TRUE
            && spNoteNameToFreq(note_name, octave_index, 0.0, 3, &note_freq) == SP_TRUE) {
            spDebug(50, "spCalcCQTMinBlockLength", "min_freq, note_freq = %f\n", min_freq, note_freq);
            min_freq = note_freq;
        } else {
            min_freq = SP_CQT_DEFAULT_MIN_FREQ;
        }
    }
    if (bins_per_octave <= 0) {
        bins_per_octave = SP_CQT_DEFAULT_BINS_PER_OCTAVE;
    }
    if (margin_factor <= 0.0) margin_factor = 1.0;

    invbins = 1.0 / (double)bins_per_octave;
    fbas = min_freq;
    
    // Q = 2^(1/bins) - 2^(-1/bins);
    Q = pow(2.0, invbins) - pow(2.0, -invbins);
    cqtbw = Q * fbas + gamma;

    length_d = samp_freq / cqtbw;
    length = (long)ceil(margin_factor * length_d);
    spDebug(50, "spCalcCQTMinBlockLength", "length_d = %f, length = %ld, margin_factor = %f\n", length_d, length, margin_factor);
    
    if (pow2flag) {
        return spPow2(spNextPow2(length));
    } else {
        return length;
    }
}

spCQTRec spInitCQT(spCQTConfig *config)
{
    int percent;
    long fftl = 0;
    long block_length = 0;
    long ref_length;
    double fftl_factor = 1.0;
    spCQTRec cqtrec = NULL;
    spPlugin *fft_plugin = NULL;
    spFFTRec fftrec = NULL;
    spBool proceed_flag = SP_TRUE;
    
    if (config == NULL) return NULL;

    if (config->samp_freq <= 0.0 || config->bins_per_octave <= 0 || config->block_transition_factor < 2.0) {
        spWarning("Wrong config parameters.\n");
        return NULL;
    }

    if (config->max_freq <= 0.0) {
        config->max_freq = config->samp_freq / 2.0;
    } else {
        config->max_freq = MIN(config->max_freq, config->samp_freq / 2.0);
    }

    if (!strnone(config->fft_plugin_name)) {
        if ((fft_plugin = spLoadFFTPlugin(config->fft_plugin_name)) == NULL) {
            spWarning("Can't open FFT plugin: %s\n", config->fft_plugin_name);
        } else {
            spDebug(10, "spInitCQT", "FFT Plugin: %s\n", config->fft_plugin_name);
        }
    }
    
    spDebug(50, "spInitCQT", "samp_freq = %f, config->whole_input_length = %ld, config->block_length_ms = %f, config->fft_length = %ld\n",
            config->samp_freq, config->whole_input_length, config->block_length_ms, config->fft_length);
    
    if (config->whole_input_length > 0) {
        if (config->fft_length >= config->whole_input_length) {
            fftl = config->fft_length;
        } else {
            if (config->reference_stride >= 2.0) {
                fftl = (long)ceil(config->reference_stride * (long)ceil((double)config->whole_input_length / config->reference_stride));
            } else {
                fftl = config->whole_input_length;
            }
            spDebug(50, "spInitCQT", "config->whole_input_length = %ld, fftl = %ld\n", config->whole_input_length, fftl);
            config->fft_length = 0;
        }
        if (config->options & SP_CQT_OPTION_POWER_OF_2_FFT_LENGTH) {
            fftl = spPow2(spNextPow2(fftl));
            config->fft_length = fftl;
        }
        fftl_factor = (double)fftl / (double)config->whole_input_length;
        config->block_length = block_length = 0;
        config->block_length_ms = 0.0;
    } else if (config->block_length_ms > 0.0 || config->block_length > 0) {
        long blockorder = 0;

        if (config->block_length_ms > 0.0) {
            long block_length2;
            block_length2 = (long)spRound(config->samp_freq * config->block_length_ms / 1000.0);
            if (config->fft_length < block_length2) {
                blockorder = spNextPow2(block_length2);
                config->fft_length = 0;
            } else {
                block_length = block_length2;
            }
        }
        if (config->block_length > 0) {
            if (config->fft_length < config->block_length) {
                long blockorder2;
                blockorder2 = spNextPow2(config->block_length);
                blockorder = MAX(blockorder, blockorder2);
                config->fft_length = 0;
            } else {
                block_length = config->block_length;
            }
        }

        if (blockorder > 0) {
            blockorder = MAX(blockorder, config->min_fft_order);
            block_length = spPow2(blockorder);
            fftl = block_length;
        } else {
            if (block_length > 0 && config->fft_length > block_length) {
                fftl = config->fft_length;
                fftl_factor = (double)config->fft_length / (double)block_length;
            } else {
                block_length = MAX(block_length, config->fft_length);
                config->fft_length = fftl;
                fftl = block_length;
            }
        }
        
        config->whole_input_length = MIN(config->whole_input_length, block_length);
    }
    spDebug(50, "spInitCQT", "config->whole_input_length = %ld, block_length = %ld, fftl = %ld, fftl_factor = %f\n",
            config->whole_input_length, block_length, fftl, fftl_factor);

    if (config->whole_input_length <= 0 && (block_length == 0 || fftl == 0)) {
        spWarning("config->block_length_ms, config->total_input_length, or config->fft_length must >be more than 0.\n");
        return NULL;
    }
    
    if ((fftrec = spInitFFTByPlugin(fft_plugin, -fftl, SP_FFT_DEFAULT_PRECISION)) == NULL) {
        spWarning("Can't initialize FFT engine\n");
        return NULL;
    }

    if (config->whole_input_length > 0) {
        ref_length = config->whole_input_length;
    } else {
        ref_length = block_length;
    }
    
    cqtrec = xalloc(1, struct spCQTRec);
    memset(cqtrec, 0, sizeof(struct spCQTRec));
    cqtrec->samp_freq = config->samp_freq;
    if (config->min_freq <= 0.0) {
        config->min_freq = SP_CQT_DEFAULT_MIN_FREQ;
    }
    if (config->options & SP_CQT_OPTION_MIN_FREQ_BASED_ON_MUSICAL_NOTE) {
        char note_name[4];
        int octave_index;
        double note_freq;
        if (spFreqToNoteName(config->min_freq, 3, note_name, &octave_index, NULL) == SP_TRUE
            && spNoteNameToFreq(note_name, octave_index, 0.0, 3, &note_freq) == SP_TRUE) {
            spDebug(50, "spInitCQT", "cqtrec->min_freq, note_freq = %f\n", cqtrec->min_freq, note_freq);
            cqtrec->min_freq = note_freq;
            config->min_freq = note_freq;
        } else {
            cqtrec->min_freq = config->min_freq;
        }
    } else {
        cqtrec->min_freq = config->min_freq;
    }
    cqtrec->bins_per_octave = config->bins_per_octave;
    cqtrec->whole_input_length = config->whole_input_length;
    cqtrec->callback_func = config->callback_func;
    cqtrec->user_data = config->user_data;
    cqtrec->progress_percent = 0.0;
    cqtrec->fft_plugin = fft_plugin;
    cqtrec->fftrec = fftrec;
    cqtrec->block_length = block_length;
    cqtrec->fftl = spGetFFTLengthSpecified(fftrec);
    cqtrec->fftl_factor = fftl_factor;
    cqtrec->length_crop_factor = 1.0;

    if (config->options & SP_CQT_OPTION_PHASE_MODE_GLOBAL) {
        cqtrec->phasemode_global = SP_TRUE; /* global phase mode (default) */
    } else {
        cqtrec->phasemode_global = SP_FALSE; /* local phase mode */
    }
    spDebug(50, "spInitCQT", "cqtrec->block_length = %ld, cqtrec->fftl = %ld, stride = %f\n",
            cqtrec->block_length, cqtrec->fftl, config->reference_stride);

    if (cqtrec->callback_func != NULL) {
        percent = 1;
        proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data);
    }

    if (proceed_flag == SP_TRUE) {
        if (xcqtprepare(cqtrec->fftrec, config->min_freq, config->max_freq, config->bins_per_octave, config->gamma,
                        ref_length, config->reference_stride, config->min_sample_length_for_block, config->samp_freq,
                        config->window_func, config->options, &cqtrec->g, &cqtrec->shift,
                        &cqtrec->M, &cqtrec->maxM, &config->reference_stride, cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent) == SP_FALSE) {
            spFreeCQT(cqtrec);
            return NULL;
        }
        spDebug(50, "spInitCQT", "stride = %f\n", config->reference_stride);
        if (cqtrec->callback_func != NULL) {
            percent = 98;
            if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
                goto bail;
            }
        }
        
        cqtrec->posit = xlvcalcposit(cqtrec->shift, &cqtrec->total_freq_len);
        if (cqtrec->block_length > 0) {
            spDebug(50, "spInitCQT", "call xdvsnsgtf_real_get_window_for_block, cqtrec->block_length = %ld, config->block_transition_factor = %f\n",
                    cqtrec->block_length, config->block_transition_factor);
            cqtrec->win = xdvsnsgtf_real_get_window_for_block(cqtrec->fftl, cqtrec->block_length, config->block_transition_factor);
            if (cqtrec->callback_func != NULL) {
                percent = 99;
                if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
                    goto bail;
                }
                cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_BLOCK_WINDOW_OBTAINED_CALLBACK, cqtrec->win, NULL, cqtrec->user_data);
            }
        } else {
            cqtrec->win = NODATA;
        }
    
        cqtrec->total_nbins = nsgtf_real_calc_total_nbins(cqtrec->g, cqtrec->posit, cqtrec->fftl);
        spDebug(50, "spInitCQT", "cqtrec->total_nbins = %ld, M->length = %ld\n", cqtrec->total_nbins, cqtrec->M->length);
    
#if defined(SP_CQT_USE_THREAD)
        cqtrec->num_thread = config->num_thread;
        if (spCQTThreadInitialize(cqtrec) == SP_FALSE) {
            spFreeCQT(cqtrec);
            return NULL;
        }
#endif
    
        if (cqtrec->callback_func != NULL) {
            percent = 100;
            if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
                goto bail;
            }
            cqtrec->callback_func(cqtrec, SP_CQT_PREPARE_FINISHED_CALLBACK, &cqtrec->total_nbins, NULL, cqtrec->user_data);
        }
    }
    
  bail:
    if (proceed_flag == SP_FALSE) {
        spFreeCQT(cqtrec);
        return NULL;
    }
    
    cqtrec->progress_percent = 0.0;
    
    return cqtrec;
}

spBool spFreeCQT(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return SP_FALSE;

#if defined(SP_CQT_USE_THREAD)
    spCQTThreadFree(cqtrec);
#endif
    
    if (cqtrec->gd != NODATA) xdvsfree(cqtrec->gd);
    
    if (cqtrec->win != NODATA) xdvfree(cqtrec->win);
    if (cqtrec->posit != NODATA) xlvfree(cqtrec->posit);
    if (cqtrec->M != NODATA) xdvfree(cqtrec->M);
    if (cqtrec->shift != NODATA) xlvfree(cqtrec->shift);
    if (cqtrec->g != NODATA) xdvsfree(cqtrec->g);
    
    if (cqtrec->fftrec != NULL) spFreeFFT(cqtrec->fftrec);
    
    if (cqtrec->fft_plugin != NULL) {
	spFreeFFTPlugin(cqtrec->fft_plugin);
    }

    xfree(cqtrec);
    
    return SP_TRUE;
}

long spGetCQTFFTLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->fftl;
}

long spGetCQTFreqCount(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->total_nbins;
}

long spGetCQTTimeCount(spCQTRec cqtrec, long freq_index, spBool no_crop)
{
    long count;
    
    if (cqtrec == NULL || cqtrec->M == NODATA
        || freq_index < 0 || freq_index >= cqtrec->M->length) return -1;
    
    if (cqtrec->fftl_factor != 1.0 && no_crop == SP_FALSE) {
        count = (long)spRound(cqtrec->M->data[freq_index] / cqtrec->fftl_factor);
    } else {
        count = (long)spRound(cqtrec->M->data[freq_index]);
    }

    return count;
}

double spConvCQTIndexToFreq(spCQTRec cqtrec, long index)
{
    double fbas;
    
    if (cqtrec == NULL) return -1.0;

    if (index == 0) return 0;

    fbas = cqtrec->min_freq * pow(2.0, (double)(index - 1) / (double)cqtrec->bins_per_octave);

    return fbas;
}

spDVector xspGetCQTFreqAxis(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return NODATA;

    spDebug(80, "xspGetCQTFreqAxis", "min_freq = %f, samp_freq = %f, bins_per_octave = %ld, total_nbins = %ld\n",
            cqtrec->min_freq, cqtrec->samp_freq, cqtrec->bins_per_octave, cqtrec->total_nbins);
    return xdvcalccqtfbas(cqtrec->min_freq, cqtrec->samp_freq / 2.0,
                          cqtrec->bins_per_octave, cqtrec->total_nbins, SP_TRUE);
}

long spGetCQTWholeInputLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->whole_input_length;
}

long spGetCQTValidResultLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    if (cqtrec->block_length > 0) {
        return cqtrec->maxM / 2;
    } else {
        return cqtrec->maxM;
    }
}

static void dvcqtspec2powerspec(spDVector spectrum, double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */)
{
    if (exponent == 0.5) {
        dvabs(spectrum);
    } else if (exponent != 0.0) {
        dvsquare(spectrum);
        if (exponent != 1.0) {
            dvscpower(spectrum, exponent, SP_FALSE);
        }
    }
    
    return;
}

static long spCopyCQTSpectrumAtCore(spCQTRec cqtrec, spDVectors c, long pos, spDVector o_spectrum)
{
    long k;
    long start_k;
    long end_k;
    long pos_k;
    double rate;

    rate = (double)pos / (double)(cqtrec->maxM - 1);
    spDebug(80, "spCopyCQTSpectrumAtCore", "pos = %ld, rate = %f\n", pos, rate);
        
    for (k = 0; k < c->num_vector; k++) {
        if (k >= o_spectrum->length) {
            break;
        }
        
#if defined(SP_CQT_FFT_SIG_CENTER)
        if (cqtrec->fftl_factor != 1.0) {
            long length;
            length = (long)spRound((double)c->vector[k]->length / cqtrec->fftl_factor);
            start_k = (c->vector[k]->length - length) / 2;
            end_k = start_k + length;
        } else {
            start_k = 0;
            end_k = c->vector[k]->length - 1;
        }
#else
        start_k = 0;
        if (cqtrec->fftl_factor != 1.0) {
            end_k = (long)spRound((double)c->vector[k]->length / cqtrec->fftl_factor) - 1;
        } else {
            end_k = c->vector[k]->length - 1;
        }
#endif
        pos_k = start_k + (long)spRound(rate * (end_k - start_k));
        spDebug(100, "spCopyCQTSpectrumAtCore", "k = %ld, pos_k = %ld, start_k = %ld, end_k = %ld\n", k, pos_k, start_k, end_k);

        if (o_spectrum->imag != NULL) {
            o_spectrum->data[k] = c->vector[k]->data[pos_k];
            o_spectrum->imag[k] = c->vector[k]->imag[pos_k];
        } else {
            o_spectrum->data[k] = sqrt(c->vector[k]->data[pos_k] * c->vector[k]->data[pos_k] + c->vector[k]->imag[pos_k] * c->vector[k]->imag[pos_k]);
        }
    }

    return k;
}

static long spCopyCQTCenterSpectrumCore(spCQTRec cqtrec, spDVectors c, spDVector o_center_spectrum)
{
    long k;
    long center;
        
    for (k = 0; k < c->num_vector; k++) {
        if (k >= o_center_spectrum->length) {
            break;
        }
        
#if defined(SP_CQT_FFT_SIG_CENTER)
        center = (c->vector[k]->length - 1) / 2;
#else
        if (cqtrec->fftl_factor != 1.0) {
            center = (long)spRound((double)(c->vector[k]->length - 1) / cqtrec->fftl_factor) / 2;
        } else {
            center = (c->vector[k]->length - 1) / 2;
        }
#endif
        /*center = MAX(center, 0);*/
        
        if (o_center_spectrum->imag != NULL) {
            o_center_spectrum->data[k] = c->vector[k]->data[center];
            o_center_spectrum->imag[k] = c->vector[k]->imag[center];
        } else {
            o_center_spectrum->data[k] = sqrt(c->vector[k]->data[center] * c->vector[k]->data[center] + c->vector[k]->imag[center] * c->vector[k]->imag[center]);
        }
    }

    return k;
}

long spCopyCQTSpectrumAt(spCQTRec cqtrec, spDVectors c, long pos, spDVector o_spectrum)
{
    if (cqtrec == NULL || c == NODATA || o_spectrum == NODATA || o_spectrum->length <= 0) return -1;

    pos = MAX(pos, 0);
    pos = MIN(pos, cqtrec->maxM - 1);
    
    return spCopyCQTSpectrumAtCore(cqtrec, c, pos, o_spectrum);
}

spDVector xspCopyCQTPowerSpectrumAt(spCQTRec cqtrec, spDVectors c, long pos,
                                    double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */)
{
    spDVector spectrum;

    if (cqtrec == NULL || c == NODATA) return SP_FALSE;
    
    spectrum = xdvrizeros(spGetCQTFreqCount(cqtrec));
    
    spCopyCQTSpectrumAtCore(cqtrec, c, pos, spectrum);
    dvcqtspec2powerspec(spectrum, exponent);

    return spectrum;
}

long spCopyCQTCenterSpectrum(spCQTRec cqtrec, spDVectors c, spDVector o_center_spectrum)
{
    if (cqtrec == NULL || c == NODATA || o_center_spectrum == NODATA || o_center_spectrum->length <= 0) return -1;
    
    return spCopyCQTCenterSpectrumCore(cqtrec, c, o_center_spectrum);
}

spDVector xspCopyCQTCenterPowerSpectrum(spCQTRec cqtrec, spDVectors c, 
                                        double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */)
{
    spDVector spectrum;

    if (cqtrec == NULL || c == NODATA) return SP_FALSE;
    
    spectrum = xdvrizeros(spGetCQTFreqCount(cqtrec));
    
    spCopyCQTCenterSpectrumCore(cqtrec, c, spectrum);
    dvcqtspec2powerspec(spectrum, exponent);

    return spectrum;
}

static spDVectors xspExecCQTWholeInputCore(spCQTRec cqtrec, spDVector sig, long offset, spDVector o_center_spectrum)
{
    int percent;
    spBool proceed_flag = SP_TRUE;
    spDVectors c = NODATA;
    spDVector f = NODATA;

    cqtrec->progress_percent = 0.0;
    
    f = xdvsnsgtf_real_fft_input(cqtrec->fftrec, sig, offset, cqtrec->whole_input_length, NODATA, cqtrec->total_freq_len, &cqtrec->length_crop_factor);
    //dvdump(f); exit(0);
    if (cqtrec->callback_func != NULL) {
        percent = 30;
        if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
            goto bail;
        }
        cqtrec->progress_percent = 30.0;
        cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_INPUT_FFT_FINISHED_CALLBACK, f, NULL, cqtrec->user_data);
    }
        
    c = xdvsnsgtf_real_alloc_vectors(cqtrec->M, cqtrec->total_nbins);

#if defined(SP_CQT_USE_THREAD)
    if (cqtrec->num_thread > 0) {
        long nth;
        
        spLockMutex(cqtrec->mutex);
        cqtrec->start_progress_percent = cqtrec->progress_percent;
        cqtrec->target_progress_percent = 99.5;
        cqtrec->proceed_flag = SP_TRUE;
        cqtrec->num_computing = cqtrec->num_thread;
        cqtrec->current_f = f;
        cqtrec->current_c = c;
        spUnlockMutex(cqtrec->mutex);        

        for (nth = 0; nth < cqtrec->num_thread; nth++) {
            cqtrec->thread_recs[nth].forward_finished = SP_FALSE;
            spDebug(100, "xspExecCQTWholeInputCore", "call spSetEvent for thread %ld / %ld\n",
                    nth, cqtrec->num_thread); 
            spSetEvent(cqtrec->thread_recs[nth].event);
        }

        spWaitEvent(cqtrec->event_for_process);
        
        spLockMutex(cqtrec->mutex);
        proceed_flag = cqtrec->proceed_flag;
        cqtrec->current_f = NODATA;
        cqtrec->current_c = NODATA;
        spUnlockMutex(cqtrec->mutex);
    } else
#endif
    if ((proceed_flag = nsgtf_real_mainloop(cqtrec->fftrec, cqtrec->g, cqtrec->shift, cqtrec->phasemode_global, cqtrec->posit, f, c,
                                            cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent)) == SP_FALSE) {
        goto bail;
    }

    if (o_center_spectrum != NODATA) {
        spCopyCQTCenterSpectrumCore(cqtrec, c, o_center_spectrum);
    }

    if (cqtrec->callback_func != NULL) {
        percent = 100;
        if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
            goto bail;
        }
        cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_FINISHED_CALLBACK, c, NULL, cqtrec->user_data);
    }

  bail:
    if (f != NODATA) xdvfree(f);

    if (proceed_flag == SP_FALSE) {
        xdvsfree(c);
        return NODATA;
    } else {
        return c;
    }
}

spDVectors xspExecCQTWholeInput(spCQTRec cqtrec, spDVector sig, long offset)
{
    if (cqtrec == NULL || cqtrec->whole_input_length <= 0) return NODATA;

    return xspExecCQTWholeInputCore(cqtrec, sig, offset, NODATA);
}

spDVector xspExecICQTWholeInput(spCQTRec cqtrec, spDVectors c)
{
    spDVectors gd;
    spDVector x = NODATA;
    spBool proceed_flag = SP_TRUE;

    if (cqtrec == NULL || cqtrec->whole_input_length <= 0 || c == NODATA) return NODATA;

    cqtrec->progress_percent = 0.0;
    
    gd = xdvsnsdual(cqtrec->g, cqtrec->shift, cqtrec->posit, cqtrec->M, cqtrec->total_freq_len,
                    cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent);
    if (gd == NODATA) {
        return NODATA;
    } else {
#if defined(SP_CQT_USE_THREAD)
        if (cqtrec->num_thread > 0) {
            long nth;
        
            spLockMutex(cqtrec->mutex);
            cqtrec->start_progress_percent = cqtrec->progress_percent;
            cqtrec->target_progress_percent = 99.5;
            cqtrec->proceed_flag = SP_TRUE;
            cqtrec->num_computing = cqtrec->num_thread;
            cqtrec->gd = gd;
            cqtrec->current_c = c;
            x = xdvrizeros(cqtrec->total_freq_len);
            cqtrec->current_fr = x;
            spUnlockMutex(cqtrec->mutex);        

            for (nth = 0; nth < cqtrec->num_thread; nth++) {
                cqtrec->thread_recs[nth].forward_finished = SP_TRUE;
                spDebug(100, "xspExecICQTWholeInput", "call spSetEvent for thread %ld / %ld\n",
                        nth, cqtrec->num_thread); 
                spSetEvent(cqtrec->thread_recs[nth].event);
            }

            spWaitEvent(cqtrec->event_for_process);
            
            spLockMutex(cqtrec->mutex);
            cqtrec->gd = NODATA;
            cqtrec->current_c = NODATA;
            cqtrec->current_fr = NODATA;
            proceed_flag = cqtrec->proceed_flag;
            spUnlockMutex(cqtrec->mutex);
            
            if (proceed_flag == SP_FALSE) {
                goto bail;
            }
            if ((proceed_flag = nsigtf_real_ifft_fr(cqtrec->fftrec, x, SP_FALSE,
                                                    cqtrec->callback_func, cqtrec, cqtrec->user_data, NULL)) == SP_FALSE) {
                goto bail;
            }
            dvreal(x);
        } else
#endif
        {
            x = xdvnsigtf_real(cqtrec->fftrec, c, gd, cqtrec->shift, cqtrec->posit, cqtrec->total_freq_len, cqtrec->phasemode_global,
                               cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent);
            if (cqtrec->progress_percent == SP_CQT_PROGRESS_PERCENT_STOPPED) {
                proceed_flag = SP_FALSE;
                goto bail;
            }
        }

        if (cqtrec->callback_func != NULL) {
            /* 100%: in nsigtf_real_ifft_fr */
            cqtrec->callback_func(cqtrec, SP_CQT_INVERSE_FINISHED_CALLBACK, x, &x->length, cqtrec->user_data);
        }
    
        xdvsfree(gd);
    }

  bail:
    if (proceed_flag == SP_FALSE) {
        if (x != NODATA) xdvfree(x);
        return NODATA;
    }
    return x;
}

spBool spExtractCQTWholeInputSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector o_center_spectrum, spDVectors *xo_c /* can be NULL */)
{
    long offset;
    spDVectors c;

    if (cqtrec == NULL || cqtrec->whole_input_length <= 0 || o_center_spectrum == NODATA) return SP_FALSE;

    offset = sig_center_pos - cqtrec->whole_input_length / 2;

    c = xspExecCQTWholeInputCore(cqtrec, sig, offset, o_center_spectrum);

    if (xo_c != NULL) {
        *xo_c = c;
    } else {
        xdvsfree(c);
    }
    
    return SP_TRUE;
}

spDVector xspExtractCQTWholeInputPowerSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos,
                                               double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */, spDVectors *xo_c /* can be NULL */)
{
    spDVector spectrum;

    spectrum = xdvrizeros(spGetCQTFreqCount(cqtrec));
    
    if (spExtractCQTWholeInputSpectrum(cqtrec, sig, sig_center_pos, spectrum, xo_c) == SP_FALSE) {
        xdvfree(spectrum);
        return NODATA;
    }

    dvcqtspec2powerspec(spectrum, exponent);
    
    return spectrum;
}

spDVector xspExecCQTCompatibleFFT(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win)
{
    if (cqtrec == NULL || cqtrec->fftrec == NULL || sig == NODATA) return NODATA;
    
    return xdvcqt_exec_fft_input(cqtrec->fftrec, sig, sig_center_pos, 0, win, SP_TRUE, SP_FALSE);
}

spDVector xspExecCQTCompatibleRealFFT(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win)
{
    if (cqtrec == NULL || cqtrec->fftrec == NULL || sig == NODATA) return NODATA;
    
    return xdvcqt_exec_fft_input(cqtrec->fftrec, sig, sig_center_pos, 0, win, SP_TRUE, SP_TRUE);
}

spDVector xspExtractCQTCompatibleFFTPowerSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win, double exponent /* 1: power spectrum, 0.5: amplitude spectrum */)
{
    spDVector spectrum;

    if ((spectrum = xspExecCQTCompatibleRealFFT(cqtrec, sig, sig_center_pos, win)) == NODATA) {
        return NODATA;
    }

    dvrffttopowerex(cqtrec->fftrec, spectrum, exponent);
    xdvrealloc(spectrum, cqtrec->fftl / 2 + 1);
    
    return spectrum;
}

spDVector xspGetCQTCompatibleFFTFreqAxis(spCQTRec cqtrec, spBool half_len_flag)
{
    spDVector faxis;

    if (cqtrec == NULL || cqtrec->fftrec == NULL) return NODATA;

    if (half_len_flag) {
        faxis = xdvinit(0.0, 1.0, cqtrec->fftl / 2);
    } else {
        faxis = xdvinit(0.0, 1.0, cqtrec->fftl - 1);
    }
    dvscoper(faxis, "*", cqtrec->samp_freq / (double)cqtrec->fftl);

    return faxis;
}

long spGetCQTBlockLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->block_length;
}

long spGetCQTBlockPushLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->block_length / 2;
}

long spGetCQTBlockValidResultLength(spCQTRec cqtrec)
{
    if (cqtrec == NULL) return -1;

    return cqtrec->maxM / 2;
}

long spGetCQTBlockCurrentValidResultLength(spCQTRec cqtrec, spCQTBlock block)
{
    long length;
    
    if (cqtrec == NULL) return -1;

    length = cqtrec->maxM / 2;

    if (block != NULL && cqtrec->block_length / 2 > block->pushed_length) {
        double rate;

        rate = (double)block->pushed_length / (double)(cqtrec->block_length / 2);
        spDebug(80, "spGetCQTBlockCurrentValidResultLength", "rate = %f, original length = %ld\n", rate, length);

        if (rate < 1.0) {
            length = (long)floor(rate * (double)length);
            spDebug(80, "spGetCQTBlockCurrentValidResultLength", "rate = %f, updated length = %ld\n", rate, length);
        }
    }

    return length;
}

long spCalcCQTBlockCount(spCQTRec cqtrec, long sig_length)
{
    long  nblock;
    
    if (cqtrec == NULL) return -1;
    
    nblock = (long)ceil((double)(sig_length + cqtrec->block_length / 4) / (double)(cqtrec->block_length / 2));

    return nblock;
}

spCQTBlock spInitCQTBlock(spCQTRec cqtrec, long *o_push_length, long *o_delay)
{
    spCQTBlock block;
    
    if (cqtrec == NULL || cqtrec->block_length <= 0) return NULL;
    
    block = xalloc(1, struct spCQTBlock);
    memset(block, 0, sizeof(struct spCQTBlock));
    block->x = xdvzeros(cqtrec->block_length);
    block->wx = xdvzeros(cqtrec->fftl);
    block->f = xdvrizeros(cqtrec->fftl);

    block->c = xdvsnsgtf_real_alloc_vectors(cqtrec->M, cqtrec->total_nbins);
    block->prev_c = xdvsnsgtf_real_alloc_vectors(cqtrec->M, cqtrec->total_nbins);

    block->pushed_length = block->x->length / 2;
    
    if (o_push_length != NULL) *o_push_length = cqtrec->block_length / 2;
    if (o_delay != NULL) *o_delay = cqtrec->block_length / 2;
    
    return block;
}

spBool spFreeCQTBlock(spCQTBlock block)
{
    if (block == NULL) return SP_FALSE;

    if (block->x != NODATA) xdvfree(block->x);
    if (block->wx != NODATA) xdvfree(block->wx);
    if (block->f != NODATA) xdvfree(block->f);
    if (block->c != NODATA) xdvsfree(block->c);
    if (block->prev_c != NODATA) xdvsfree(block->prev_c);

    xfree(block);
    
    return SP_TRUE;
}

spBool spResetCQTBlock(spCQTBlock block)
{
    if (block == NULL) return SP_FALSE;

    dvzeros(block->x, block->x->length);
    nsgtf_real_clear_vectors(block->c);
    nsgtf_real_clear_vectors(block->prev_c);
    block->process_count = 0;
    block->pushed_length = block->x->length / 2;
    
    return SP_TRUE;
}

long spGetCQTBlockResultDelay(spCQTRec cqtrec, long freq_index)
{
    long M;

    if (cqtrec == NULL || freq_index < 0 || freq_index >= cqtrec->M->length) return -1;

    M = (long)spRound(cqtrec->M->data[freq_index]);

    return M / 2;
}

long spCopyCQTBlockResult(spCQTRec cqtrec, spCQTBlock block, long freq_index, spBool no_crop, spDVector o_result, long *io_result_offset)
{
    long halfsize;
    long halfsize_rem;
    long result_offset;
    long c_skip_offset;
    long c_length;
    long addlen;
    spDVector c;
    spDVector prev_c;
    
    if (block == NULL || o_result == NODATA || freq_index < 0 || freq_index >= block->c->num_vector) return -1;

    c = block->c->vector[freq_index];
    prev_c = block->prev_c->vector[freq_index];
    
    if (no_crop == SP_FALSE && block->f->length != block->x->length) {
        double fftl_factor;
        
        fftl_factor = (double)block->f->length / (double)block->x->length;
        c_length = (long)spRound((double)c->length /fftl_factor);
#if defined(SP_CQT_FFT_SIG_CENTER)
        c_skip_offset = (c->length - c_length) / 2;
#else
        c_skip_offset = 0;
#endif
    } else {
        c_skip_offset = 0;
        c_length = c->length;
    }
    halfsize = c_length / 2;
    halfsize_rem = c_length - halfsize;

    result_offset = io_result_offset != NULL ? *io_result_offset : 0;

    spDebug(80, "spCopyCQTBlockResult", "freq_index = %ld, result_offset = %ld, process_count = %ld\n",
            freq_index, result_offset, block->process_count);
    
    if (block->process_count >= 2) {
        dvadd(o_result, result_offset, prev_c, c_skip_offset + halfsize_rem, halfsize, SP_FALSE);
    }
    addlen = dvadd(o_result, result_offset, c, c_skip_offset, halfsize, block->process_count >= 2 ? SP_TRUE : SP_FALSE);
    result_offset += addlen;

    if (io_result_offset != NULL) *io_result_offset = result_offset;

#if 0
    if (cqt_current_block_index == /*2*/103) {
        //dvdump(c); exit(0);
        //dvdump(prev_c); exit(0);
        dvdump(o_result); exit(0);
    }
#endif

    return halfsize;
}

static long spCopyCQTBlockSpectrumAtCore(spCQTRec cqtrec, spCQTBlock block, long pos, spDVector o_spectrum)
{
    long k;
    long c_length;
    long halfsize;
    long halfsize_rem;
    long start_k;
    long end_k;
    long pos_k;
    long prev_pos_k;
    double rate;
    double fftl_factor;
    double real, imag;
    spDVectors c;
    spDVectors prev_c;
    
    c = block->c;
    prev_c = block->prev_c;

    rate = (double)pos / (double)(cqtrec->maxM - 1);
    
    if (cqtrec->block_length / 2 > block->pushed_length) {
        double pushed_rate;
        
        pushed_rate = (double)block->pushed_length / (double)cqtrec->block_length;
        spDebug(80, "spCopyCQTBlockSpectrumAtCore", "pushed_rate = %f, rate = %f, block->pushed_length = %ld / %ld\n",
                pushed_rate, rate, block->pushed_length, cqtrec->block_length);
        if (rate >= pushed_rate) {
            return 0;
        }
    }
    
    spDebug(100, "spCopyCQTBlockSpectrumAtCore", "pos = %ld, maxM = %ld, rate = %f, block->pushed_length = %ld / %ld\n",
            pos, cqtrec->maxM, rate, block->pushed_length, cqtrec->block_length);
    
    fftl_factor = (double)block->f->length / (double)block->x->length;
    spDebug(80, "spCopyCQTBlockSpectrumAtCore", "rate = %f, fftl_factor = %f, num_vector = %ld\n", rate, fftl_factor, c->num_vector); 

    for (k = 0; k < c->num_vector; k++) {
#if defined(SP_CQT_FFT_SIG_CENTER)
        if (fftl_factor != 1.0) {
            c_length = (long)spRound((double)c->vector[k]->length / fftl_factor);
            start_k = (c->vector[k]->length - c_length) / 2;
            end_k = start_k + length;
        } else {
            c_length = c->vector[k]->length;
            start_k = 0;
            end_k = c_length - 1;
        }
#else
        start_k = 0;
        if (fftl_factor != 1.0) {
            c_length = (long)spRound((double)c->vector[k]->length / fftl_factor);
            end_k = c_length - 1;
        } else {
            c_length = c->vector[k]->length;
            end_k = c_length - 1;
        }
#endif
        pos_k = start_k + (long)spRound(rate * (end_k - start_k));
        halfsize = c_length / 2;
        halfsize_rem = c_length - halfsize;
        spDebug(100, "spCopyCQTBlockSpectrumAtCore", "%ld: pos_k = %ld, start_k = %ld, end_k = %ld, halfsize = %ld, halfsize_rem = %ld\n",
                k, pos_k, start_k, end_k, halfsize, halfsize_rem); 

        real = imag = 0.0;
        
        if (block->process_count >= 2) {
            prev_pos_k = halfsize_rem + pos_k;
            if (prev_pos_k < prev_c->vector[k]->length) {
#if defined(SP_CQT_BLOCK_SPECTRUM_POWER_SUM)
                if (o_spectrum->imag == NULL) {
                    real = prev_c->vector[k]->data[prev_pos_k] * prev_c->vector[k]->data[prev_pos_k]
                        + prev_c->vector[k]->imag[prev_pos_k] * prev_c->vector[k]->imag[prev_pos_k];
                    imag = 0.0;
                } else
#endif
                {
                    real = prev_c->vector[k]->data[prev_pos_k];
                    imag = prev_c->vector[k]->imag[prev_pos_k];
                }
            }
        }
#if defined(SP_CQT_BLOCK_SPECTRUM_POWER_SUM)
        if (o_spectrum->imag == NULL) {
            real += (c->vector[k]->data[pos_k] * c->vector[k]->data[pos_k] + c->vector[k]->imag[pos_k] * c->vector[k]->imag[pos_k]);
            imag = 0.0;
        } else
#endif
        {
            real += c->vector[k]->data[pos_k];
            imag += c->vector[k]->data[pos_k];
        }
        
        if (o_spectrum->imag != NULL) {
            o_spectrum->data[k] = real;
            o_spectrum->imag[k] = imag;
        } else {
#if defined(SP_CQT_BLOCK_SPECTRUM_POWER_SUM)
            o_spectrum->data[k] = sqrt(real);
#else
            o_spectrum->data[k] = sqrt(real * real + imag * imag);
#endif
        }
    }

    return k;
}

long spCopyCQTBlockSpectrumAt(spCQTRec cqtrec, spCQTBlock block, long pos, spDVector o_spectrum)
{
    if (cqtrec == NULL || block == NULL || o_spectrum == NODATA || o_spectrum->length <= 0
        || pos < 0 || 2 * pos >= cqtrec->maxM) return -1;

    return spCopyCQTBlockSpectrumAtCore(cqtrec, block, pos, o_spectrum);
}

spDVector xspCopyCQTBlockPowerSpectrumAt(spCQTRec cqtrec, spCQTBlock block, long pos,
                                         double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */)
{
    spDVector spectrum;

    if (cqtrec == NULL || block == NULL || pos < 0 || 2 * pos >= cqtrec->maxM) return NODATA;

    spectrum = xdvrizeros(spGetCQTFreqCount(cqtrec));
    
    spCopyCQTBlockSpectrumAtCore(cqtrec, block, pos, spectrum);
    
    dvcqtspec2powerspec(spectrum, exponent);

    return spectrum;
}

long spProcessCQTBlock(spCQTRec cqtrec, spDVector sigpush, long sigpush_offset, long limit_length /* 0: default */, spCQTBlock io_block)
{
    int percent;
    long push_length;
    long remain_length;
    spBool proceed_flag = SP_TRUE;
    
    if (cqtrec == NULL || io_block == NULL) return -1;

    cqtrec->progress_percent = 0.0;
    
    cqt_current_block_index = io_block->process_count;

    push_length = cqtrec->block_length / 2;
    if (sigpush != NODATA) {
        spDebug(80, "spProcessCQTBlock", "sigpush->length = %ld sigpush_offset = %ld\n",
                sigpush->length, sigpush_offset);
        remain_length = sigpush->length - sigpush_offset;
        remain_length = MIN(remain_length, push_length);
        remain_length = MAX(remain_length, 0);
        if (limit_length > 0 && limit_length < remain_length) {
            spDebug(50, "spProcessCQTBlock", "**** new remain_length = %ld, original = %ld\n",
                    limit_length, remain_length);
            remain_length = limit_length;
        }
    } else {
        remain_length = 0;
    }
    spDebug(80, "spProcessCQTBlock", "push_length = %ld, remain_length = %ld, pushed_length = %ld, process_count = %ld\n",
            push_length, remain_length, io_block->pushed_length, io_block->process_count);

    if (io_block->process_count >= 1) {
        push_length = MIN(push_length, io_block->pushed_length);
        dvscopy(io_block->prev_c, io_block->c);
        dvdatashift(io_block->x, -push_length);
        io_block->pushed_length -= push_length;
        push_length = MIN(push_length, io_block->pushed_length);
        spDebug(80, "spProcessCQTBlock", "updated pushed_length = %ld, push_length = %ld\n",
                io_block->pushed_length, push_length);
    }

#if 0
    if (io_block->pushed_length < cqtrec->block_length / 4 && remain_length <= 0) {
        spDebug(80, "spProcessCQTBlock", "final frame: remain_length = %ld, pushed_length = %ld, process_count = %ld\n",
                remain_length, io_block->pushed_length, io_block->process_count);
        nsgtf_real_clear_vectors(io_block->c);
        return -1;
    }
#endif

    if (sigpush != NODATA && remain_length > 0) {
        dvadd(io_block->x, push_length, sigpush, sigpush_offset, remain_length, SP_FALSE);
    } else if (sigpush == NODATA) {
        dverase(io_block->x, push_length, io_block->x->length - push_length, SP_FALSE);
    }
    io_block->pushed_length += remain_length;

    nsgtf_windowing_and_fft_input(cqtrec->fftrec, io_block->x, cqtrec->win, io_block->wx, io_block->f);
#if 0
    if (io_block->process_count == 2) {
        dvdump(io_block->f); exit(0);
    }
#endif
    if (cqtrec->callback_func != NULL) {
        percent = 30;
        if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
            goto bail;
        }
        cqtrec->progress_percent = 30.0;
        cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_INPUT_FFT_FINISHED_CALLBACK, io_block->f, NULL, cqtrec->user_data);
    }

    
#if defined(SP_CQT_USE_THREAD)
    if (cqtrec->num_thread > 0) {
        long nth;
        
        spLockMutex(cqtrec->mutex);
        cqtrec->start_progress_percent = cqtrec->progress_percent;
        cqtrec->target_progress_percent = 99.5;
        cqtrec->proceed_flag = SP_TRUE;
        cqtrec->num_computing = cqtrec->num_thread;
        cqtrec->current_f = io_block->f;
        cqtrec->current_c = io_block->c;
        spUnlockMutex(cqtrec->mutex);        

        for (nth = 0; nth < cqtrec->num_thread; nth++) {
            cqtrec->thread_recs[nth].forward_finished = SP_FALSE;
            spDebug(100, "spProcessCQTBlock", "call spSetEvent for thread %ld / %ld\n",
                    nth, cqtrec->num_thread); 
            spSetEvent(cqtrec->thread_recs[nth].event);
        }

        spWaitEvent(cqtrec->event_for_process);
        
        spLockMutex(cqtrec->mutex);
        proceed_flag = cqtrec->proceed_flag;
        cqtrec->current_f = NODATA;
        cqtrec->current_c = NODATA;
        spUnlockMutex(cqtrec->mutex);
        
        if (proceed_flag == SP_FALSE) {
            goto bail;
        }
    } else
#endif
    if ((proceed_flag = nsgtf_real_mainloop(cqtrec->fftrec, cqtrec->g, cqtrec->shift, cqtrec->phasemode_global, cqtrec->posit, io_block->f, io_block->c,
                                            cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent)) == SP_FALSE) {
        goto bail;
    }
    
    //dvdump(io_block->c->vector[1]); exit(0);
    
    if (cqtrec->callback_func != NULL) {
        percent = 100;
        if ((proceed_flag = cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_PROGRESS_CALLBACK, &percent, NULL, cqtrec->user_data)) == SP_FALSE) {
            goto bail;
        }
        cqtrec->callback_func(cqtrec, SP_CQT_FORWARD_FINISHED_CALLBACK, io_block->c, NULL, cqtrec->user_data);
    }
    io_block->process_count++;

  bail:
    spDebug(80, "spProcessCQTBlock", "done: push_length = %ld, pushed_length = %ld, process_count = %ld\n",
            push_length, io_block->pushed_length, io_block->process_count);
    
    return push_length;
}

long spProcessCQTBlockOneShort(spCQTRec cqtrec, spDVector sigpush, long sigpush_offset, long limit_length /* 0: default */, spCQTBlock io_block)
{
    if (spResetCQTBlock(io_block) == SP_FALSE) return -1;

    return spProcessCQTBlock(cqtrec, sigpush, sigpush_offset, limit_length, io_block);
}

long spProcessFinalCQTBlock(spCQTRec cqtrec, spCQTBlock io_block)
{
    return spProcessCQTBlock(cqtrec, NODATA, 0, 0, io_block);
}

long spInvertCQTBlock(spCQTRec cqtrec, spCQTBlock block, spDVector io_syn, long *io_synpos)
{
    long offset;
    long new_offset;
    
    if (cqtrec == NULL || block == NULL || io_syn == NODATA) return -1;

    cqtrec->progress_percent = 0.0;
    
    if (io_synpos != NULL) {
        offset = *io_synpos;
        if (offset >= io_syn->length) {
            return -1;
        }
    } else {
        offset = 0;
    }
    
    if (cqtrec->gd == NODATA) {
        cqtrec->gd = xdvsnsdual(cqtrec->g, cqtrec->shift, cqtrec->posit, cqtrec->M, cqtrec->total_freq_len,
                                cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent);
    }

#if defined(SP_CQT_USE_THREAD)
    if (cqtrec->num_thread > 0) {
        long nth;
        spBool proceed_flag = SP_TRUE;
        
        new_offset = 0;
        
        spLockMutex(cqtrec->mutex);
        dvrizeros(block->f, block->f->length);
        
        cqtrec->start_progress_percent = cqtrec->progress_percent;
        cqtrec->target_progress_percent = 90.0;
        cqtrec->proceed_flag = SP_TRUE;
        cqtrec->num_computing = cqtrec->num_thread;
        cqtrec->current_c = block->c;
        cqtrec->current_fr = block->f;
        spUnlockMutex(cqtrec->mutex);        

        for (nth = 0; nth < cqtrec->num_thread; nth++) {
            cqtrec->thread_recs[nth].forward_finished = SP_TRUE;
            spDebug(100, "spInvertCQTBlock", "call spSetEvent for thread %ld / %ld\n",
                    nth, cqtrec->num_thread); 
            spSetEvent(cqtrec->thread_recs[nth].event);
        }

        spWaitEvent(cqtrec->event_for_process);
            
        spLockMutex(cqtrec->mutex);
        cqtrec->current_c = NODATA;
        cqtrec->current_fr = NODATA;
        proceed_flag = cqtrec->proceed_flag;
        spUnlockMutex(cqtrec->mutex);
            
        if (proceed_flag == SP_FALSE) {
            goto bail;
        }
        //dvdump(block->f); exit(0);
        if ((proceed_flag = nsigtf_real_ifft_fr(cqtrec->fftrec, block->f, SP_TRUE,
                                                cqtrec->callback_func, cqtrec, cqtrec->user_data, NULL)) == SP_FALSE) {
            goto bail;
        }
        if ((proceed_flag = icqtolablock(cqtrec->fftrec, cqtrec->block_length, offset, block->f, io_syn,
                                         cqtrec->callback_func, cqtrec, cqtrec->user_data)) == SP_FALSE) {
            goto bail;
        }
        new_offset = offset + (cqtrec->block_length - cqtrec->block_length / 2);
        
      bail:
        if (proceed_flag == SP_FALSE) {
            cqtrec->progress_percent = SP_CQT_PROGRESS_PERCENT_STOPPED;
        }
    } else
#endif
    {
        new_offset = icqtprocessblock(cqtrec->fftrec, block->c, cqtrec->gd, cqtrec->posit, cqtrec->phasemode_global,
                                      cqtrec->block_length, offset, block->f, io_syn,
                                      cqtrec->callback_func, cqtrec, cqtrec->user_data, &cqtrec->progress_percent);
    }

    if (cqtrec->progress_percent == SP_CQT_PROGRESS_PERCENT_STOPPED) {
        return -1;
    }
    
    if (cqtrec->callback_func != NULL) {
        /* 100%: in icqtolablock */
        cqtrec->callback_func(cqtrec, SP_CQT_INVERSE_FINISHED_CALLBACK, io_syn, &new_offset, cqtrec->user_data);
    }
    
    if (io_synpos != NULL) *io_synpos = new_offset;

    return new_offset - offset;
}
