/** @file sp/cqt.h
 */

#ifndef SPLIB_CQT_H
#define SPLIB_CQT_H

#include <sp/vector.h>
#include <sp/fft.h>

#ifdef __cplusplus
extern "C" {
#endif

/** @~english @defgroup fftGroup <sp/cqt.h>: CQT (Constant-Q transform)
@code
#include <sp/cqt.h>
@endcode
 */

#define SP_CQT_VERSION_STRING      "0.3.0"
#define SP_CQT_VERSION             0
#define SP_CQT_REVISION            3
#define SP_CQT_UPDATE_LEVEL        0
#define SP_CQT_VERSION_ID          (SP_CQT_VERSION * 1000 + SP_CQT_REVISION)

#define SP_CQT_USE_THREAD

#define SP_CQT_DEFAULT_SAMP_FREQ 44100.0
#define SP_CQT_DEFAULT_MIN_FREQ 27.5
#define SP_CQT_DEFAULT_BINS_PER_OCTAVE 48
#define SP_CQT_DEFAULT_GAMMA 0.0/* 20.0: */

#define SP_CQT_DEFAULT_BLOCK_LENGTH_MS 180.0 /* 44.1 kHz sampling: 7938 points --> 8192 points */
#define SP_CQT_DEFAULT_BLOCK_TRANSITION_FACTOR /*64.0*/8.0
#define SP_CQT_DEFAULT_MIN_SAMPLE_LENGTH_FOR_BLOCK 32

typedef struct spCQTRec *spCQTRec;
typedef struct spCQTBlock *spCQTBlock;
typedef unsigned long spCQTCallbackType;
typedef struct spCQTThreadRec spCQTThreadRec;

typedef spBool (*spCQTCallbackFunc)(spCQTRec cqtrec, spCQTCallbackType callback_type,
                                    void *data1, void *data2, void *user_data);
typedef double (*spCQTWindowValueFunc)(double);

#define SP_CQT_NO_CALLBACK 0
#define SP_CQT_PREPARE_PROGRESS_CALLBACK (1<<0)  /* data1: int *percent */
#define SP_CQT_PREPARE_CENTER_FREQUENCIES_OBTAINED_CALLBACK (1<<1)  /* data1: LVector shift */
#define SP_CQT_PREPARE_CHANNEL_FILTER_OBTAINED_CALLBACK (1<<2)  /* data1: long *channel, data2: DVector channel_filter */
#define SP_CQT_PREPARE_BANDWIDTHS_OBTAINED_CALLBACK (1<<3)  /* data1: DVector M */
#define SP_CQT_PREPARE_NORMALIZE_VECTOR_OBTAINED_CALLBACK (1<<4)  /* data1: DVector normFacVec */
#define SP_CQT_PREPARE_BANDWIDTHS_REFINED_CALLBACK (1<<5)  /* data1: DVector M */
#define SP_CQT_PREPARE_CHANNEL_FILTER_NORMALIZED_CALLBACK (1<<6)  /* data1: long *channel, data2: DVector channel_filter */
#define SP_CQT_PREPARE_ANALYSIS_FILTERS_OBTAINED_CALLBACK (1<<7)  /* data1: DVectors g */
#define SP_CQT_PREPARE_BLOCK_WINDOW_OBTAINED_CALLBACK (1<<8)  /* data1: DVector window (block version only) */
#define SP_CQT_PREPARE_FINISHED_CALLBACK (1<<9)  /* data1: long *total_nbins */
#define SP_CQT_FORWARD_PROGRESS_CALLBACK (1<<12)  /* data1: int *percent */
#define SP_CQT_FORWARD_INPUT_FFT_FINISHED_CALLBACK (1<<13)  /* data1: DVector spec */
#define SP_CQT_FORWARD_CHANNEL_CONVOLUTION_FINISHED_CALLBACK (1<<14)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_FORWARD_CHANNEL_PHASE_GLOBALIZE_FINISHED_CALLBACK (1<<15)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_FORWARD_CHANNEL_IFFT_FINISHED_CALLBACK (1<<17)  /* data1: long *channel, data2: DVector channel_sig */
#define SP_CQT_FORWARD_CHANNEL_FINISHED_CALLBACK (1<<18)  /* data1: long *channel, data2: DVector channel_sig */
#define SP_CQT_FORWARD_FINISHED_CALLBACK (1<<19)  /* data1: DVectors all_channel_sigs */
#define SP_CQT_INVERSE_PROGRESS_CALLBACK (1<<20)  /* data1: int *percent */
#define SP_CQT_INVERSE_FRAME_OPERATOR_DIAGONAL_OBTAINED_CALLBACK (1<<21)  /* data1: DVector diagonal, data2: LVectors win_ranges */
#define SP_CQT_INVERSE_CHANNEL_SYNTHESIS_FILTER_OBTAINED_CALLBACK (1<<22)  /* data1: long *channel, data2: DVector channel_filter */
#define SP_CQT_INVERSE_SYNTHESIS_FILTERS_OBTAINED_CALLBACK (1<<23)  /* data1: DVectors gd */
#define SP_CQT_INVERSE_CHANNEL_FFT_FINISHED_CALLBACK (1<<24)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_INVERSE_CHANNEL_PHASE_UNGLOBALIZE_FINISHED_CALLBACK (1<<25)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_INVERSE_CHANNEL_SPECTRUM_OBTAINED_CALLBACK (1<<26)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_INVERSE_CHANNEL_CONVOLUTION_FINISHED_CALLBACK (1<<27)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_INVERSE_CHANNEL_FINISHED_CALLBACK (1<<28)  /* data1: long *channel, data2: DVector channel_spec */
#define SP_CQT_INVERSE_FINAL_IFFT_FINISHED_CALLBACK (1<<29)  /* data1: DVector syn_frame */
#define SP_CQT_INVERSE_BLOCK_OVERLAP_ADD_FINISHED_CALLBACK (1<<30)  /* data1: DVector sig, data2: long *ola_offset (block version only) */
#define SP_CQT_INVERSE_FINISHED_CALLBACK (1<<31)  /* data1: DVector syn_sig, data2: long *fixed_length */

typedef unsigned long spCQTOptionMask;
#define SP_CQT_OPTION_RASTERIZE_METHOD_NONE (1<<0)
#define SP_CQT_OPTION_RASTERIZE_METHOD_PIECEWISE (1<<1)
#define SP_CQT_OPTION_RASTERIZE_METHOD_FULL (1<<2)
#define SP_CQT_OPTION_RASTERIZE_METHOD_POWER_OF_2 (1<<3)
#define SP_CQT_OPTION_RASTERIZE_METHOD_FULL_POWER_OF_2 (SP_CQT_OPTION_RASTERIZE_METHOD_FULL|SP_CQT_OPTION_RASTERIZE_METHOD_POWER_OF_2)
#define SP_CQT_OPTION_NORMALIZE_METHOD_NONE (1<<6)
#define SP_CQT_OPTION_NORMALIZE_METHOD_SINE (1<<7)
#define SP_CQT_OPTION_NORMALIZE_METHOD_IMPULSE (1<<8)
#define SP_CQT_OPTION_PHASE_MODE_GLOBAL (1<<12)
#define SP_CQT_OPTION_PHASE_MODE_LOCAL (1<<13)
#define SP_CQT_OPTION_RASTERIZE_METHOD_MASK (0x3f)
#define SP_CQT_OPTION_NORMALIZE_METHOD_MASK (0xfc0)
#define SP_CQT_OPTION_PHASE_MODE_MASK (0x7000)

#define SP_CQT_OPTION_POWER_OF_2_FFT_LENGTH (1<<15)
#define SP_CQT_OPTION_STRICT_REFERENCE_STRIDE (1<<16)
#define SP_CQT_OPTION_ERROR_STOP_BY_WRONG_REFERENCE_STRIDE (1<<17)
#define SP_CQT_OPTION_MIN_FREQ_BASED_ON_MUSICAL_NOTE (1<<18)

#define SP_CQT_OPTIONS_DEFAULT (SP_CQT_OPTION_RASTERIZE_METHOD_NONE|SP_CQT_OPTION_NORMALIZE_METHOD_SINE|SP_CQT_OPTION_PHASE_MODE_GLOBAL)

#define SP_CQT_REFERENCE_STRIDE_DEFAULT (0)
#define SP_CQT_REFERENCE_STRIDE_MAX (-1)

typedef struct spCQTConfig {
    unsigned long version_id;	/* this member is automatically initialized by spInitCQTConfig */
    
    double samp_freq;		/* sampling frequency in Hz. default=44100Hz */
    double min_freq;	        /* minimum frequency in Hz. */
    double max_freq;    	/* maximum frequency in Hz. 0(default): Nyquist frequency. */
    
    long bins_per_octave;
    double gamma;

    spCQTOptionMask options;

    double reference_stride;
    long whole_input_length;	/* if whole_input_length > 0, non-block processing will be done */
    long block_length;		/* the value becomes the next power of 2 internally */
    double block_length_ms;	/* the value becomes the next power of 2 internally */
    double block_transition_factor; /* transition factor for slicing window (transition length: window_length / factor) */
    long fft_length;
    long min_fft_order;
    long min_sample_length_for_block;

    char fft_plugin_name[SP_MAX_PATHNAME];
    long num_thread;

    spCQTWindowValueFunc window_func;
    spCQTCallbackFunc callback_func;
    void *user_data;
} spCQTConfig;

#if defined(MACOS)
#pragma import on
#endif

#define spInitCQTConfig(conf) spInitCQTConfig_(conf, SP_CQT_VERSION_ID)
extern spBool spInitCQTConfig_(spCQTConfig *config, unsigned long version_id);
extern spCQTOptionMask spConvCQTRasterizeMethodFromString(char *rasterize_method_string);
extern spCQTOptionMask spConvCQTNormalizeMethodFromString(char *normalize_method_string);
extern double spCalcCQTGammaOfERB(long bins_per_octave /* 0: default */);
extern long spCalcCQTMinFFTLength(double min_freq /* -1: default */, long bins_per_octave /* 0: default */, double samp_freq, spBool pow2flag);
extern long spCalcCQTMinBlockLength(double min_freq /* -1: default */, long bins_per_octave /* 0: default */, double gamma,
                                    double margin_factor /* 0.0: default=1.0 */, double samp_freq, spBool pow2flag, spBool min_freq_based_on_musical_note);

extern spCQTRec spInitCQT(spCQTConfig *config);
extern spBool spFreeCQT(spCQTRec cqtrec);

extern long spGetCQTFFTLength(spCQTRec cqtrec);
extern long spGetCQTFreqCount(spCQTRec cqtrec);
extern long spGetCQTTimeCount(spCQTRec cqtrec, long freq_index, spBool no_crop);

extern double spConvCQTIndexToFreq(spCQTRec cqtrec, long index);
extern spDVector xspGetCQTFreqAxis(spCQTRec cqtrec);

extern long spGetCQTWholeInputLength(spCQTRec cqtrec);
extern long spGetCQTValidResultLength(spCQTRec cqtrec);

extern long spCopyCQTSpectrumAt(spCQTRec cqtrec, spDVectors c, long pos, spDVector o_spectrum);
extern spDVector xspCopyCQTPowerSpectrumAt(spCQTRec cqtrec, spDVectors c, long pos,
                                           double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */);
extern long spCopyCQTCenterSpectrum(spCQTRec cqtrec, spDVectors c, spDVector o_center_spectrum);
extern spDVector xspCopyCQTCenterPowerSpectrum(spCQTRec cqtrec, spDVectors c, 
                                               double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */);

extern spDVectors xspExecCQTWholeInput(spCQTRec cqtrec, spDVector sig, long offset);
extern spDVector xspExecICQTWholeInput(spCQTRec cqtrec, spDVectors c);

extern spBool spExtractCQTWholeInputSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector o_center_spectrum, spDVectors *xo_c /* can be NULL */);
extern spDVector xspExtractCQTWholeInputPowerSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos,
                                                      double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */, spDVectors *xo_c /* can be NULL */);

extern spDVector xspExecCQTCompatibleFFT(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win /* can be NODATA (rectanble window) */);
extern spDVector xspExecCQTCompatibleRealFFT(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win /* can be NODATA (rectanble window) */);
/* output: real (non-complex) spectrum of half length (length=fftl/2+1) */
extern spDVector xspExtractCQTCompatibleFFTPowerSpectrum(spCQTRec cqtrec, spDVector sig, long sig_center_pos, spDVector win, double exponent /* 1: power spectrum, 0.5: amplitude spectrum */);
extern spDVector xspGetCQTCompatibleFFTFreqAxis(spCQTRec cqtrec, spBool half_len_flag);

extern long spGetCQTBlockLength(spCQTRec cqtrec);
extern long spGetCQTBlockPushLength(spCQTRec cqtrec);
extern long spGetCQTBlockValidResultLength(spCQTRec cqtrec);
extern long spGetCQTBlockCurrentValidResultLength(spCQTRec cqtrec, spCQTBlock block);
extern long spCalcCQTBlockCount(spCQTRec cqtrec, long sig_length);
extern spCQTBlock spInitCQTBlock(spCQTRec cqtrec, long *o_push_length, long *o_delay);
extern spBool spFreeCQTBlock(spCQTBlock block);
extern spBool spResetCQTBlock(spCQTBlock block);

extern long spGetCQTBlockResultDelay(spCQTRec cqtrec, long freq_index);
extern long spCopyCQTBlockResult(spCQTRec cqtrec, spCQTBlock block, long freq_index, spBool no_crop, spDVector o_result, long *io_result_offset);
extern long spCopyCQTBlockSpectrumAt(spCQTRec cqtrec, spCQTBlock block, long pos, spDVector o_spectrum);
extern spDVector xspCopyCQTBlockPowerSpectrumAt(spCQTRec cqtrec, spCQTBlock block, long pos,
                                                double exponent /* 0: complex spectrum, 1: power spectrum, 0.5: amplitude spectrum */);

extern long spProcessCQTBlock(spCQTRec cqtrec, spDVector sigpush, long sigpush_offset, long limit_length /* 0: default */, spCQTBlock io_block);
extern long spProcessFinalCQTBlock(spCQTRec cqtrec, spCQTBlock io_block);
extern long spInvertCQTBlock(spCQTRec cqtrec, spCQTBlock block, spDVector io_syn, long *io_synpos);

#if defined(MACOS)
#pragma import off
#endif

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_CQT_H */
