#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <sp/spLib.h>
#include <sp/spMain.h>

static spBool quiet_flag = SP_FALSE;

static spBool callbackFunc(spCQTRec cqtrec, spCQTCallbackType callback_type,
                           void *data1, void *data2, void *user_data)
{
    long *pchannel;
    spDVector vector1;
    
    if (callback_type == SP_CQT_PREPARE_PROGRESS_CALLBACK
        || callback_type == SP_CQT_FORWARD_PROGRESS_CALLBACK
        || callback_type == SP_CQT_INVERSE_PROGRESS_CALLBACK) {
        static int prev_percent = 0;
        int *percent = (int *)data1;
#if 1
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: time = %f, percent = %d, prev_percent = %d%s\n",
                      callback_type, (double)clock()*1000.0, *percent, prev_percent, (*percent - prev_percent <= 0 ? "********" : ""));
        }
#endif
        prev_percent = *percent;
    } else if (callback_type == SP_CQT_PREPARE_FINISHED_CALLBACK) {
        long *total_nbins = (long *)data1;
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: total_nbins = %ld\n", callback_type, *total_nbins);
        }
    } else if (callback_type == SP_CQT_PREPARE_CENTER_FREQUENCIES_OBTAINED_CALLBACK
               || callback_type == SP_CQT_PREPARE_BANDWIDTHS_OBTAINED_CALLBACK
               || callback_type == SP_CQT_PREPARE_NORMALIZE_VECTOR_OBTAINED_CALLBACK
               || callback_type == SP_CQT_PREPARE_BANDWIDTHS_REFINED_CALLBACK
               || callback_type == SP_CQT_FORWARD_INPUT_FFT_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_FINAL_IFFT_FINISHED_CALLBACK) {
        vector1 = (spDVector)data1;
#if 0
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: vector1->length = %ld\n", callback_type, vector1->length);
        }
#endif
    } else if (callback_type == SP_CQT_PREPARE_CHANNEL_FILTER_OBTAINED_CALLBACK
               || callback_type == SP_CQT_PREPARE_CHANNEL_FILTER_NORMALIZED_CALLBACK
               || callback_type == SP_CQT_FORWARD_CHANNEL_CONVOLUTION_FINISHED_CALLBACK
               || callback_type == SP_CQT_FORWARD_CHANNEL_PHASE_GLOBALIZE_FINISHED_CALLBACK
               || callback_type == SP_CQT_FORWARD_CHANNEL_IFFT_FINISHED_CALLBACK
               || callback_type == SP_CQT_FORWARD_CHANNEL_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_SYNTHESIS_FILTER_OBTAINED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_FFT_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_PHASE_UNGLOBALIZE_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_SPECTRUM_OBTAINED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_CONVOLUTION_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_CHANNEL_FINISHED_CALLBACK) {
        pchannel = (long *)data1;
        vector1 = (spDVector)data2;
#if 0
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: channel = %ld, vector1->length = %ld\n", callback_type, *pchannel, vector1->length);
        }
#endif
    } else if (callback_type == SP_CQT_INVERSE_BLOCK_OVERLAP_ADD_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_FINISHED_CALLBACK) {
        long *length = (long *)data2;
        vector1 = (spDVector)data1;
#if 1
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: vector1->length = %ld, length = %ld\n", callback_type, vector1->length, *length);
        }
#endif
    } else if (callback_type == SP_CQT_PREPARE_ANALYSIS_FILTERS_OBTAINED_CALLBACK
               || callback_type == SP_CQT_FORWARD_FINISHED_CALLBACK
               || callback_type == SP_CQT_INVERSE_SYNTHESIS_FILTERS_OBTAINED_CALLBACK) {
        spDVectors vectors;
        vectors = (spDVectors)data1;
#if 1
        if (quiet_flag == SP_FALSE) {
            spMessage("%lx: vectors->num_vector = %ld\n", callback_type, vectors->num_vector);
        }
#endif
    }

    return SP_TRUE;
}

static double frame_shift = -1.0;

static double analysis_pos = -1.0;
static spBool exec_fft_for_pos = SP_FALSE;
static double analysis_length_for_efp = -1.0;

static spCQTConfig cqt_config;

static spBool cqt_erb_flag;
static spBool snr_flag;
static char rasterize_method_string[SP_MAX_PATHNAME] = "";
static char normalize_method_string[SP_MAX_PATHNAME] = "";
static char o_freq_axis_file[SP_MAX_PATHNAME] = "";
static char o_result_mat_file[SP_MAX_PATHNAME] = "";
static spBool o_result_mat_swap = SP_FALSE;
static char fft_plugin_name[SP_MAX_PATHNAME] = "";
static char i_param_file[SP_MAX_PATHNAME] = "";
static char o_param_file[SP_MAX_PATHNAME] = "";
static int debug_level = -1;
static spBool help_flag;

static long o_freq_index = -1;

static spOption option[] = {
    {"-shift", NULL, "frame shift [ms]", "frame_shift",
	 SP_TYPE_DOUBLE, &frame_shift, NULL},
    {"-bl", "-blocklength", "block length", "block_length",
	 SP_TYPE_LONG, &cqt_config.block_length, NULL},
    {"-fftl", "-fftlength", "FFT length", "fft_length",
	 SP_TYPE_LONG, &cqt_config.fft_length, NULL},
    {"-tf", "-transition", "transition factor for slicing window (transition length: window_length / factor)", "transition_factor",
	 SP_TYPE_DOUBLE, &cqt_config.block_transition_factor, "4"},
    {"-r", "-raster", "rasterize method ('n': none, 'p': piecewise, 'f': full, '2': power of 2)", "rasterize method",
	 SP_TYPE_STRING_A, rasterize_method_string, "n"},
    {"-n", "-normalize", "normalize method ('n': none, 's': sine, 'i': impulse)", "normalize method",
	 SP_TYPE_STRING_A, normalize_method_string, "s"},
    {"-pos", NULL, "analysis center position time [s]", "analysis_pos",
	 SP_TYPE_DOUBLE, &analysis_pos, NULL},
    {"-efp", NULL, "execute FFT insted of CQT for '-pos' option", "exec_fft_for_pos",
	 SP_TYPE_BOOLEAN, &exec_fft_for_pos, SP_FALSE_STRING},
    {"-efpl", "-efplen", "analysis length for '-efp' option", "analysis_length_for_efp",
	 SP_TYPE_DOUBLE, &analysis_length_for_efp, "30.0"},
    {"-fmin", NULL, "CQT minimum frequency", "cqt_fmin",
	 SP_TYPE_DOUBLE, &cqt_config.min_freq, "27.5"},
    {"-fmax", NULL, "CQT maximum frequency", "cqt_fmax",
         SP_TYPE_DOUBLE, &cqt_config.max_freq, /*"20000.0"*/NULL},
    {"-bins", NULL, "CQT bins per octave", "cqt_bins_per_octave",
	 SP_TYPE_LONG, &cqt_config.bins_per_octave, "48"},
    {"-gamma", NULL, "CQT frequency warpinig parameter value", "cqt_gamma",
	 SP_TYPE_DOUBLE, &cqt_config.gamma, "0.0"},
    {"-erb", NULL, "CQT frequency warpinig parameter is set to simulate the ERB", "cqt_erb",
	 SP_TYPE_BOOLEAN, &cqt_erb_flag, SP_FALSE_STRING},
    {"-ofi", "-ofreqidx", "output frequency index", "o_freq_index",
	 SP_TYPE_LONG, &o_freq_index, NULL},
    {"-ofa", "-ofaf", "output frequency axis file", "o_freq_axis_file",
	 SP_TYPE_STRING_A, o_freq_axis_file, ""},
    {"-omat", NULL, "output result matrix binary file", "o_result_mat_file",
	 SP_TYPE_STRING_A, o_result_mat_file, ""},
    {"-omatswap", NULL, "enable swap format of output result matrix binary file", "o_result_mat_swap",
	 SP_TYPE_BOOLEAN, &o_result_mat_swap, SP_FALSE_STRING},
    {"-nth", "-nthread", "number of threads (0: without thread)", "num_thread",
	 SP_TYPE_LONG, &cqt_config.num_thread, "4"},
    {"-fftplugin", NULL, "FFT plugin name", "fft_plugin", 
	 SP_TYPE_STRING_A, cqt_config.fft_plugin_name, NULL},
    {"-sn", "-snr", "display SNR", NULL,
	 SP_TYPE_BOOLEAN, &snr_flag, SP_FALSE_STRING},
    {"-iparam", NULL, "input parameter file", NULL,
	 SP_TYPE_STRING_A, i_param_file, NULL},
    {"-oparam", NULL, "output parameter file", NULL,
	 SP_TYPE_STRING_A, o_param_file, NULL},
    {"-debug", NULL, "debug level", NULL,
	 SP_TYPE_INT, &debug_level, NULL},
    {"-q", "-quiet", "enable quiet mode", NULL,
	 SP_TYPE_BOOLEAN, &quiet_flag, SP_FALSE_STRING},
    {"-h", "-help", "display this message", NULL,
	 SP_TYPE_BOOLEAN, &help_flag, SP_FALSE_STRING},
};

static char *filelabel[] = {
    "<input file>",
    "<output file>",
};

int spMain(int argc, char **argv)
{
    const char *i_filename;
    const char *o_filename;
    spOptions options;
    int samp_bit, num_channel;
    double fs;
    long m;
    spDVector sig;
    spDVector syn;
    spDVector freq_axis_data = NODATA;
    spDVector freq_result = NODATA;
    spDVector spec_result = NODATA;
    spCQTRec cqtrec;
    FILE *fp;

    if (spInitCQTConfig(&cqt_config) == SP_FALSE) {
        spError(1, "spInitCQTConfig failed.\n");
    }
    
    spSetHelpMessage(&help_flag, "CQT test  version 0.1.0");
    options = spGetOptions(argc, argv, option, filelabel);
    spGetOptionsValue(argc, argv, options);
    spSetDebugLevel(debug_level);

    if (!strnone(i_param_file)) {
	spMessage("Input Parameter File: %s\n", i_param_file);
	spReadSetup(i_param_file, options);
    }

    i_filename = spGetFile(options);
    o_filename = spGetFile(options);
    spCheckNumFile(options);
    
    sig = xdvreadaudiofile(i_filename, NULL, &fs, &samp_bit, &num_channel, 1.0);
    if (sig == NODATA) {
        spError(1, "Cannot open file: %s\n", i_filename);
    } else if (num_channel != 1) {
        spError(1, "Stereo signal is not supported.\n");
    }

    if (cqt_erb_flag == SP_TRUE) {
        cqt_config.gamma = spCalcCQTGammaOfERB(cqt_config.bins_per_octave);
    }

    spDebug(-50, "spMain", "fs = %f, frame_shift = %f, fmin = %f, bins = %ld, gamma = %f, block_transition_factor = %f\n",
            fs, frame_shift, cqt_config.min_freq, cqt_config.bins_per_octave, cqt_config.gamma, cqt_config.block_transition_factor);

    cqt_config.samp_freq = fs;
    
    if (analysis_pos < 0.0 && (frame_shift > 0.0 || cqt_config.block_length > 0)) {
        if (cqt_config.block_length <= 0) {
            cqt_config.block_length_ms = 2.0 * frame_shift;
        }
    } else {
        if (analysis_pos >= 0.0) {
            cqt_config.whole_input_length = spCalcCQTMinFFTLength(cqt_config.min_freq, cqt_config.bins_per_octave,
                                                                  cqt_config.samp_freq, SP_FALSE);
        } else {
            cqt_config.whole_input_length = sig->length;
        }
    }

    cqt_config.callback_func = callbackFunc;
    cqt_config.options = spConvCQTNormalizeMethodFromString(normalize_method_string);
    cqt_config.options |= spConvCQTRasterizeMethodFromString(rasterize_method_string);
    
    syn = NODATA;
    cqtrec = spInitCQT(&cqt_config);
    spDebug(-50, "spMain", "cqt_config.reference_stride = %f\n", cqt_config.reference_stride);
    
    if (spGetCQTBlockLength(cqtrec) <= 0) {
        spDVectors c;
        
        if (analysis_pos >= 0.0) {
            long analysis_posl;

            analysis_posl = (long)spRound(analysis_pos * cqt_config.samp_freq);
            if (exec_fft_for_pos == SP_TRUE) {
                long analysis_lengthl_for_efp;
                spDVector win;

                if (analysis_length_for_efp <= 0.0) {
                    analysis_length_for_efp = 30.0;
                }
                analysis_lengthl_for_efp = (long)spRound(cqt_config.samp_freq * analysis_length_for_efp / 1000.0);
                win = xdvnhamming(analysis_lengthl_for_efp);
                spec_result = xspExtractCQTCompatibleFFTPowerSpectrum(cqtrec, sig, analysis_posl, win, 0.5);
                if (cqt_config.options & SP_CQT_OPTION_NORMALIZE_METHOD_SINE) {
                    /* makes spectral density */
                    dvscoper(spec_result, "/", (double)win->length);
                }
                xdvfree(win);
                if (!strnone(o_freq_axis_file)) {
                    freq_axis_data = xspGetCQTCompatibleFFTFreqAxis(cqtrec, SP_TRUE);
                }
            } else {
                spec_result = xdvzeros(spGetCQTFreqCount(cqtrec));
            
                spExtractCQTWholeInputSpectrum(cqtrec, sig, analysis_posl, spec_result, NULL);
            }
        } else {
            if ((c = xspExecCQTWholeInput(cqtrec, sig, 0)) != NODATA) {
                //dvdump(c->vector[1]); exit(0);
                syn = xspExecICQTWholeInput(cqtrec, c);

                if (o_freq_index >= 0) {
                    if (o_freq_index < c->num_vector) {
                        freq_result = xdvclone(c->vector[o_freq_index]);
                    } else {
                        spWarning("Frequency index %ld exceeds max index %ld\n", o_freq_index, c->num_vector);
                        o_freq_index = -1;
                    }
                }

                if (!strnone(o_result_mat_file)) {
                    if ((fp = spOpenFile(o_result_mat_file, "wb")) == NULL) {
                        spWarning("Can't open output text file: %s\n", o_result_mat_file);
                    } else {
                        long result_length;
                        
                        result_length = spGetCQTValidResultLength(cqtrec);
                        spec_result = xdvzeros(spGetCQTFreqCount(cqtrec));
                        spDebug(-50, "spMain", "o_result_mat_file = %s, result_length = %ld, spec_result->length = %ld\n",
                                o_result_mat_file, result_length, spec_result->length);
                        
                        for (m = 0; m < result_length; m++) {
                            spDebug(100, "spMain", "m = %ld, call spCopyCQTSpectrumAt\n", m);
                            spCopyCQTSpectrumAt(cqtrec, c, m, spec_result);
                            dvfwritedvector(fp, spec_result, 0, o_result_mat_swap);
                        }

                        spMessage("Output Binary File of Analysis Result (%ld x %ld): %s\n",
                                  result_length, spec_result->length, o_result_mat_file);

                        xdvfree(spec_result); spec_result = NODATA;
                        spCloseFile(fp);
                    }
                }
            
                xdvsfree(c);
                
                if (syn == NODATA) {
                    spError(1, "xspExecICQTWholeInput failed.\n");
                }
            }
        }
    } else {
        long k;
        long offset;
        long delay;
        long syn_pos;
        long nprocess;
        long ninvert;
        long nblock;
        spCQTBlock block;
        long freq_result_offset = 0;
        long valid_result_len = 0;
        long result_frames = 0;

        syn = xdvzeros(sig->length);

        block = spInitCQTBlock(cqtrec, NULL, &delay);
        syn_pos = -delay;

        nblock = spCalcCQTBlockCount(cqtrec, sig->length);

        if (o_freq_index >= 0) {
            long freq_count;
            long time_count;

            freq_count = spGetCQTFreqCount(cqtrec);
            
            if (o_freq_index >= freq_count) {
                spWarning("Frequency index %ld exceeds max index %ld\n", o_freq_index, freq_count);
                o_freq_index = -1;
            } else {
                time_count = spGetCQTTimeCount(cqtrec, o_freq_index, SP_FALSE);
                freq_result = xdvrizeros((time_count / 2) * nblock);
                freq_result_offset = -spGetCQTBlockResultDelay(cqtrec, o_freq_index);
            }
        }

        fp = NULL;
        if (!strnone(o_result_mat_file)) {
            if ((fp = spOpenFile(o_result_mat_file, "wb")) == NULL) {
                spWarning("Can't open output text file: %s\n", o_result_mat_file);
            } else {
                spec_result = xdvzeros(spGetCQTFreqCount(cqtrec));
            }
        }
        
        for (m = 0, offset = 0;; m++) {
            nprocess = spProcessCQTBlock(cqtrec, offset >= sig->length ? NODATA : sig, offset, sig->length - offset, block);
            spDebug(80, "spMain", "m = %ld, nprocess = %ld, offset = %ld / %ld\n", m, nprocess, offset, sig->length);

            if (o_freq_index >= 0) {
                spCopyCQTBlockResult(cqtrec, block, o_freq_index, SP_FALSE, freq_result, &freq_result_offset);
            }
            if (fp != NULL && m >= 1) {
                valid_result_len = spGetCQTBlockCurrentValidResultLength(cqtrec, block);
                for (k = 0; k < valid_result_len; k++) {
                    spCopyCQTBlockSpectrumAt(cqtrec, block, k, spec_result);
                    dvfwritedvector(fp, spec_result, 0, o_result_mat_swap);
                    ++result_frames;
                }
            }
            
            ninvert = spInvertCQTBlock(cqtrec, block, syn, &syn_pos);
            spDebug(80, "spMain", "m = %ld, ninvert = %ld, syn_pos = %ld\n", m, ninvert, syn_pos);
            
            if (nprocess <= 0) {
                break;
            }

            offset += nprocess;
        }
        spDebug(-50, "spMain", "loop end: m = %ld, offset = %ld, syn_pos = %ld, original blocks = %ld\n",
                m, offset, syn_pos, nblock);

        if (fp != NULL) {
            spMessage("Output Binary File of Analysis Result (%ld x %ld): %s\n",
                      result_frames, spec_result->length, o_result_mat_file);
            xdvfree(spec_result); spec_result = NODATA;
            spCloseFile(fp); fp = NULL;
        }

        if (freq_result != NODATA) {
            spDebug(-50, "spMain", "loop end: freq_result_offset = %ld / %ld\n", freq_result_offset, freq_result->length);
        }
        
        spFreeCQTBlock(block);
    }
    
    if (syn != NODATA) {
        spDebug(50, "spMain", "syn->length = %ld\n", syn->length);

        if (analysis_pos < 0.0 && snr_flag) {
            spDVector noise;
            double sig_power;
            double noise_power;
            double snr;

            noise = xdvoper(sig, "-", syn);
            sig_power = dvsqsum(sig);
            noise_power = dvsqsum(noise);

            if (noise_power == 0.0) {
                snr = 300.0;
            } else {
                snr = spdBpow(sig_power / noise_power);
                spDebug(-50, "spMain", "original snr = %f\n", snr);
                snr = MIN(snr, 300.0);
            }
            spMessage("SNR: %.2f [dB]\n", snr);

            xdvfree(noise);
        }
    
        //dvdump(syn); exit(0);
    }

    if (!strnone(o_freq_axis_file)) {
        if (freq_axis_data == NODATA) {
            freq_axis_data = xspGetCQTFreqAxis(cqtrec);
        }
        spMessage("Output Text File of Frequency Axis: %s\n", o_freq_axis_file);
        if ((fp = spOpenFile(o_freq_axis_file, "wt")) == NULL) {
            spWarning("Can't open output text file: %s\n", o_freq_axis_file);
        } else {
            dvfdump(freq_axis_data, fp);
            spCloseFile(fp);
        }
    }
    
    if (analysis_pos >= 0.0 && spec_result != NODATA) {
        spMessage("Output Text File of Spectrum of position %f [s]: %s\n", analysis_pos, o_filename);
        if ((fp = spOpenFile(o_filename, "wt")) == NULL) {
            spWarning("Can't open output text file: %s\n", o_filename);
        } else {
            dvfdump(spec_result, fp);
            spCloseFile(fp);
        }
    } else if (freq_result != NODATA && (spEqSuffix(o_filename, ".csv") || spEqSuffix(o_filename, ".txt"))) {
        spMessage("Output Text File of Frequency Index %ld: %s\n", o_freq_index, o_filename);
        if ((fp = spOpenFile(o_filename, "wt")) == NULL) {
            spWarning("Can't open output text file: %s\n", o_filename);
        } else {
            dvfdump(freq_result, fp);
            spCloseFile(fp);
        }
    } else {
        if (freq_result != NODATA) {
            dvdump(freq_result);
        }
        spMessage("Output WAV File: %s\n", o_filename);
        dvwriteaudiofile(syn, o_filename, NULL, fs, samp_bit, num_channel, 1.0);
    }

    if (freq_result != NODATA) xdvfree(freq_result);
    if (spec_result != NODATA) xdvfree(spec_result);
    if (syn != NODATA) xdvfree(syn);
    xdvfree(sig);

    spFreeCQT(cqtrec);
    
    if (!strnone(o_param_file)) {
        spMessage("Output Parameter File: %s\n", o_param_file);
        spWriteSetup(o_param_file, options);
    }
    
    return 0;
}
