#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <sp/spLib.h>
#include <sp/spMain.h>

#if defined(IPHONE) || defined(MACOSX)
#define VECTOR_PLUGIN "vectorAccelerate"
#elif defined(USE_CUDA)
#define VECTOR_PLUGIN "vector_cuda"
#else
#undef VECTOR_PLUGIN 
#endif

int unlock_test = 1;

void inittest(void)
{
    long k;
    spSVector sx, sx2;
    spLVector lx, lx2;
    spFVector fx, fx2;
    spDVector dx, dx2;
    char op[4];

    printf("-------- init test --------\n");
    
    sx = xsvalloc(8);
    lx = xlvalloc(8);
    fx = xfvalloc(8);
    dx = xdvalloc(8);
    sx2 = xsvalloc(8);
    lx2 = xlvalloc(8);
    fx2 = xfvalloc(8);
    dx2 = xdvalloc(8);
    strcpy(op, "/");
    
    for (k = 1; k < /*5*/8; k++) {
        if (unlock_test) {
            svunlock(sx);
            lvunlock(lx);
            fvunlock(fx);
            dvunlock(dx);
            svunlock(sx2);
            lvunlock(lx2);
            fvunlock(fx2);
            dvunlock(dx2);
        }

        svriinit(sx, k, k + 1, 7 * (k + 1) + k);
        lvriinit(lx, k, k + 1, 7 * (k + 1) + k);
        fvriinit(fx, k, k + 1, 7 * (k + 1) + k);
        dvriinit(dx, k, k + 1, 7 * (k + 1) + k);

#if 1
        svriinit(sx2, k, 1, k + 7);
        lvriinit(lx2, k, 1, k + 7);
        fvriinit(fx2, k, 1, k + 7);
        dvriinit(dx2, k, 1, k + 7);
#else
        svinit(sx2, k, 1, k + 7);
        lvinit(lx2, k, 1, k + 7);
        fvinit(fx2, k, 1, k + 7);
        dvinit(dx2, k, 1, k + 7);
#endif

        printf("sum of fx = %f, sum of dx = %f\n", fvsum(fx), dvsum(dx));
        printf("sum of fx2 = %f, sum of dx2 = %f\n", fvsum(fx2), dvsum(dx2));
        printf("sqsum of fx = %f, sqsum of dx = %f\n", fvsqsum(fx), dvsqsum(dx));
        printf("sqsum of fx2 = %f, sqsum of dx2 = %f\n", fvsqsum(fx2), dvsqsum(dx2));
        printf("abssum of fx = %f, abssum of dx = %f\n", fvabssum(fx), dvabssum(dx));
        printf("abssum of fx2 = %f, abssum of dx2 = %f\n", fvabssum(fx2), dvabssum(dx2));

        printf("min of fx = %f\n", fvmin(fx, NULL));
        printf("min of dx = %f\n", dvmin(dx, NULL));
        printf("max of fx = %f, max of dx = %f\n", fvmax(fx, NULL), dvmax(dx, NULL));
        
        svoper(sx, op, sx2);
        lvoper(lx, op, lx2);
        fvoper(fx, op, fx2);
        dvoper(dx, op, dx2);

#if 0
        svcumsum(sx2);
        lvcumsum(lx2);
        fvcumsum(fx2);
        dvcumsum(dx2);
#else
        svcumprod(sx2);
        lvcumprod(lx2);
        fvcumprod(fx2);
        dvcumprod(dx2);
#endif
        
        if (unlock_test) {
            svlock(sx);
            lvlock(lx);
            fvlock(fx);
            dvlock(dx);
            svlock(sx2);
            lvlock(lx2);
            fvlock(fx2);
            dvlock(dx2);
        }
        
        svdump(sx);
        lvdump(lx);
        fvdump(fx);
        dvdump(dx);
        
        svdump(sx2);
        lvdump(lx2);
        fvdump(fx2);
        dvdump(dx2);
    }

    xsvfree(sx);
    xlvfree(lx);
    xfvfree(fx);
    xdvfree(dx);
    xsvfree(sx2);
    xlvfree(lx2);
    xfvfree(fx2);
    xdvfree(dx2);
}

void mathtest(void)
{
    long k;
    spFVector fx;
    spDVector dx;
    
    printf("-------- math test --------\n");
    
    fx = xfvalloc(8);
    dx = xdvalloc(8);
    
    for (k = 0; k < 3; k++) {
        if (unlock_test) {
            fvunlock(fx);
            dvunlock(dx);
        }

        fvrizeros(fx, fx->length);
        dvrizeros(dx, dx->length);
        
        fvinit(fx, 0, PI / 3.0, 2.0 * PI);
        dvinit(dx, 0, PI / 3.0, 2.0 * PI);

        switch (k) {
          case 0:
            fvcos(fx);
            dvcos(dx);
            break;
          case 1:
            fvsin(fx);
            dvsin(dx);
            break;
          case 2:
            fvtan(fx);
            dvtan(dx);
            break;
        }
    
        if (unlock_test) {
            fvlock(fx);
            dvlock(dx);
        }
    
        fvdump(fx);
        dvdump(dx);
    }

    xfvfree(fx);
    xdvfree(dx);
}

void evaltest(void)
{
    spDVector dx, dx2, dx3;
    
    printf("-------- eval test --------\n");
    
    dx = xdvinit(1.0, 1.0, 5);
    dx2 = xdvnums(5, 5.0);
    dvdump(dx);
    dvdump(dx2);
        
    dx3 = xdvalloc(dx->length);
        
    dvevalf(dx3, "%f * %v + %v * %f", 3.0, dx, dx2, 2.0);
    dvdump(dx3);
        
    dvevalf(dx3, "%f + %f * %f / (%f - %f) ^ %f ^ %f", 3.0, 4.0, 2.0, 1.0, 5.0, 2.0, 3.0);
    dvdump(dx3);
        
    xdvfree(dx3);
    xdvfree(dx2);
    xdvfree(dx);
}

void randtest(void)
{
    long k;
    spLVector idxv;
    spFVector fx;
    spDVector dx, dx2, dx3;

    printf("-------- rand test --------\n");
    
#if 0
    fx = xfvalloc(16);
    dx = xdvalloc(16);
#elif 1
    fx = xfvalloc(22);
    dx = xdvalloc(22);
#else
    fx = xfvalloc(126);
    dx = xdvalloc(126);
#endif
    
    for (k = 0; k <= 2; k++) {
        if (unlock_test) {
            fvunlock(fx);
            dvunlock(dx);
        }
    
        switch (k) {
          case 0:
            fvrandun(fx);
            dvrandun(dx);
            break;
            
          case 1:
            fvgauss(fx, 0, 3);
            dvgauss(dx, 0, 3);
            break;
            
          case 2:
            fvrandn(fx);
            dvrandn(dx);
            break;
        }

#if 0
        fvsort(fx);
        dvsort(dx);
#elif 1
        dvlock(dx);
        dvdump(dx);
        dx3 = xdvclone(dx);
        idxv = xdvsortidx(dx);
        lvdump(idxv);
        dx2 = xdvremap(dx3, idxv);
        dvdump(dx2);
        xdvfree(dx2);
        xdvfree(dx3);
        xlvfree(idxv);
#else
        //printf("min of dx = %f\n", dvmin(dx, NULL));
        //printf("max of dx = %f\n", dvmax(dx, NULL));
        
        idxv = xdvnmin(dx, 5, 0, 0);
        lvlock(idxv);
        lvdump(idxv);
        
        dx2 = xdvremap(dx, idxv);
        dvlock(dx2);
        dvdump(dx2);
        
        xdvfree(dx2);
        xlvfree(idxv);

        dx2 = xdvzerosul(NULL, 8, SP_TRUE);
        dvsetval(dx2, 0, -0.5 * (double)(-k));
        dvsetval(dx2, 3, 1.0 * (double)(-k));
        dvsetval(dx2, 5, -1.0 * (double)(-k));
        dvsetval(dx2, 7, 0.5 * (double)(-k));
        
        idxv = xdvnmin(dx2, 5, 0, SP_NMINMAX_OPTION_PEAK_ONLY);
        lvlock(idxv);
        lvdump(idxv);
        xlvfree(idxv);
        idxv = xdvnmax(dx2, 5, 0, SP_NMINMAX_OPTION_PEAK_ONLY);
        lvlock(idxv);
        lvdump(idxv);
        xlvfree(idxv);
        
        dvlock(dx2);
        dvdump(dx2);
        xdvfree(dx2);
#endif
        
        if (unlock_test) {
            fvlock(fx);
            dvlock(dx);
        }

        fvdump(fx);
        dvdump(dx);
    }
    
    xfvfree(fx);
    xdvfree(dx);
}

int spMain(int argc, char **argv)
{
    /*spSetDebugLevel(100);*/
    
#ifdef VECTOR_PLUGIN
    if (spSetDefaultVectorPlugin(VECTOR_PLUGIN) == NULL) {
        spError(1, "Cannot load vector plugin: %s\n", VECTOR_PLUGIN);
    }
    
    {
        int i;
        int unit;
        int num_unit;

        if (spGetDefaultVectorPluginNumUnit(&num_unit)) {
            spDebug(-20, "main", "num_unit = %d\n", num_unit);
            for (i = 0; i < num_unit; i++) {
                printf("Unit %d: %s\n", i, spGetDefaultVectorPluginUnitName(i));
            }
                
            if (argc >= 2 && (unit = atoi(argv[1])) >= 0 && unit < num_unit) {
                if (spSelectDefaultVectorPluginUnit(unit)) {
                    printf("%s selected.\n", spGetDefaultVectorPluginUnitName(unit));
                } else {
                    printf("%s selection error.\n", spGetDefaultVectorPluginUnitName(unit));
                }
            }
        }
    }
    spSetDefaultVectorPluginFastMode(SP_TRUE);
#endif

    inittest();

    mathtest();
    
    randtest();
    
    return 0;
}
