/*
 *	fft.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#if defined(MACOSX)
#if !defined(IPHONE)
#include <CoreServices/CoreServices.h>
#endif
#if defined(IPHONE) || (defined(MAC_OS_X_VERSION_MIN_REQUIRED) && (MAC_OS_X_VERSION_MIN_REQUIRED >= 1060) \
			|| defined(__IPHONE_OS_VERSION_MIN_REQUIRED) && (__IPHONE_OS_VERSION_MIN_REQUIRED >= 50000))
#include <Accelerate/Accelerate.h>
#else
#include <vecLib/vDSP.h>
#endif
#elif defined(USE_SPL)
#define HAVE_QUICK_FFT 1
#define HAVE_ACCURATE_QUICK_FFT 1

#define nsp_UsesTransform
#define nsp_UsesConversion
#include <nsp.h>
#endif

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/fft.h>
#include <sp/vector.h>
#include <sp/voperate.h>
#include <sp/filter.h>
#include <sp/fileio.h>

#include <sp/spThread.h>

#include <sp/fftP.h>
#include <sp/fftplugin.h>

static char sp_default_fft_plugin_name[SP_MAX_PATHNAME] = "";

spBool sp_use_quick_fft = SP_FALSE;

static spBool execFFTF(spFFTRec fftrec, float *real, float *imag, int inv);
static spBool execFFT(spFFTRec fftrec, double *real, double *imag, int inv);
static spBool execRealFFTF(spFFTRec fftrec, float *data, int inv);
static spBool execRealFFT(spFFTRec fftrec, double *data, int inv);

static void spfft_normal(double *real, double *imag, long p, long n, int inv);

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
static DVector xdvbsexparg(long fftl_specified);
static DVector xdvbscalcw(DVector exparg);
static DVector xdvbscalcwcirc(DVector w, LVector revidx, long fftl, long batch);

static void spfftf_bs_normal(float *real, float *imag, long n, int inv, int real_flag)
{
    long i;
    long order;
    long fftl;
    LVector revidx;
    DVector exparg;
    DVector w, wc;
    DVector W;
    DVector buffer;

    order = nextpow2(n * 2 - 1);
    fftl = POW2(order);

    /* exparg = PI * n^2 / N */
    exparg = xdvbsexparg(n);
    /* w = exp(j * PI * n^2 / N) */
    w = xdvbscalcw(exparg);
    
    /* wc = conj(w) */
    wc = xdvconj(w);

    revidx = xlvinit(n - 1, -1, 1);

    if (inv) {
        W = xdvbscalcwcirc(wc, revidx, fftl, 1);
        dvscoper(W, "/", (double)n);
    } else {
        W = xdvbscalcwcirc(w, revidx, fftl, 1);
    }
    spfft_normal(W->data, W->imag, order, fftl, 0);

    buffer = xdvrizeros(W->length);

    for (i = 0; i < n; i++) {
        buffer->data[i] = (double)real[i];
        if (!real_flag) {
            buffer->imag[i] = (double)imag[i];
        }
    }

    if (real_flag && inv) {
        rffttofftip(buffer->data, buffer->imag, n);
    }
    
    if (inv) {
        dvoper(buffer, "*", w);
    } else {
        dvoper(buffer, "*", wc);
    }
    
    spfft_normal(buffer->data, buffer->imag, order, fftl, 0);

    dvoper(buffer, "*", W);

    spfft_normal(buffer->data, buffer->imag, order, fftl, 1);

    if (inv) {
        dvoper(buffer, "*", w);
    } else {
        dvoper(buffer, "*", wc);
    }

    if (real_flag && !inv) {
        ffttorfftip(buffer->data, buffer->imag, n);
    }
    
    for (i = 0; i < n; i++) {
        real[i] = (float)buffer->data[i];
        if (!real_flag) {
            imag[i] = (float)buffer->imag[i];
        }
    }

    xlvfree(revidx);
    xdvfree(exparg);
    xdvfree(w);
    xdvfree(wc);
    xdvfree(W);
    xdvfree(buffer);
    
    return;
}

static void spfft_bs_normal(double *real, double *imag, long n, int inv, int real_flag)
{
    long i;
    long order;
    long fftl;
    LVector revidx;
    DVector exparg;
    DVector w, wc;
    DVector W;
    DVector buffer;

    order = nextpow2(n * 2 - 1);
    fftl = POW2(order);

    /* exparg = PI * n^2 / N */
    exparg = xdvbsexparg(n);
    /* w = exp(j * PI * n^2 / N) */
    w = xdvbscalcw(exparg);
    
    /* wc = conj(w) */
    wc = xdvconj(w);

    revidx = xlvinit(n - 1, -1, 1);

    if (inv) {
        W = xdvbscalcwcirc(wc, revidx, fftl, 1);
        dvscoper(W, "/", (double)n);
    } else {
        W = xdvbscalcwcirc(w, revidx, fftl, 1);
    }
    spfft_normal(W->data, W->imag, order, fftl, 0);

    buffer = xdvrizeros(W->length);

    if (real_flag) {
        if (inv) {
            rffttofft(real, buffer->data, buffer->imag, n);
        } else {
            memcpy(buffer->data, real, sizeof(double) * n);
        }
    } else {
        memcpy(buffer->data, real, sizeof(double) * n);
        memcpy(buffer->imag, imag, sizeof(double) * n);
    }
    
    if (inv) {
        dvoper(buffer, "*", w);
    } else {
        dvoper(buffer, "*", wc);
    }
    
    spfft_normal(buffer->data, buffer->imag, order, fftl, 0);

    dvoper(buffer, "*", W);

    spfft_normal(buffer->data, buffer->imag, order, fftl, 1);

    if (inv) {
        dvoper(buffer, "*", w);
    } else {
        dvoper(buffer, "*", wc);
    }

    if (real_flag) {
        if (inv) {
            memcpy(real, buffer->data, sizeof(double) * n);
        } else {
            ffttorfft(buffer->data, buffer->imag, real, n);
        }
    } else {
        memcpy(real, buffer->data, sizeof(double) * n);
        memcpy(imag, buffer->imag, sizeof(double) * n);
    }

    xlvfree(revidx);
    xdvfree(exparg);
    xdvfree(w);
    xdvfree(wc);
    xdvfree(W);
    xdvfree(buffer);
    
    return;
}
#endif

/*
 *	calculate FFT for float data 
 */
static void spfftf_normal(float *real, float *imag, long p, long n, int inv)
{
    long i, ip, j, k, m, me, meh, nh;
    float uRe, uIm, vRe, vIm, wRe, wIm, tRe, tIm;

    nh = n / 2;

    if (inv) {
	for (i = 0; i < n; i++) {
	    imag[i] = -imag[i];
	}
    }

    /* bit reversion */
    for (i = 0, j = 0; i < n - 1; i++) {
	if (j > i) {
	    tRe = real[j];	tIm = imag[j];
	    real[j] = real[i];	imag[j] = imag[i];
	    real[i] = tRe;	imag[i] = tIm;
	}
	for (k = nh; k > (j ^= k); k >>= 1);
    }

    /* butterfly numeration */
    for (m = 1; m <= p; m++) {
	me = POW2(m);		meh = me / 2;
	uRe = 1.0;		uIm = 0.0;
	wRe = cosf((float)PI / (float)meh);
	wIm = -sinf((float)PI / (float)meh);
	for (j = 0; j < meh; j++) { /* loop for each butterfly pairs */
	    for (i = j; i < n; i += me) {
		ip = i + meh;	/* butterfly pair: i and ip */
		tRe = real[ip] * uRe - imag[ip] * uIm;
		tIm = real[ip] * uIm + imag[ip] * uRe;
		real[ip] = real[i] - tRe;
		imag[ip] = imag[i] - tIm;
		real[i] += tRe;
		imag[i] += tIm;
	    }
	    vRe = uRe * wRe - uIm * wIm;
	    vIm = uRe * wIm + uIm * wRe;
	    uRe = vRe;
	    uIm = vIm;
	}
    }

    if (inv) {
	for (i = 0; i < n; i++) {
	    real[i] /= (float)n;
	    imag[i] /= (float)(-n);
	}
    }

    return;
}

void sprifftf_normal(float *data, long p, long n, int inv)
{
    long i, ir, ii, ip, ipr, ipi, j, jr, ji, k, m, me, meh, nh;
    float uRe, uIm, vRe, vIm, wRe, wIm, tRe, tIm;

    nh = n / 2;

    if (inv) {
	for (i = 0; i < n; i++) {
	    ii = 2 * i + 1;
	    data[ii] = -data[ii];
	}
    }

    /* bit reversion */
    for (i = 0, j = 0; i < n - 1; i++) {
	if (j > i) {
	    ir = 2 * i; ii = ir + 1;
	    jr = 2 * j; ji = jr + 1;
	    tRe = data[jr];		tIm = data[ji];
	    data[jr] = data[ir];	data[ji] = data[ii];
	    data[ir] = tRe;		data[ii] = tIm;
	}
	for (k = nh; k > (j ^= k); k >>= 1);
    }

    /* butterfly numeration */
    for (m = 1; m <= p; m++) {
	me = POW2(m);		meh = me / 2;
	uRe = 1.0;		uIm = 0.0;
	wRe = cosf((float)PI / (float)meh);
	wIm = -sinf((float)PI / (float)meh);
	for (j = 0; j < meh; j++) { /* loop for each butterfly pairs */
	    for (i = j; i < n; i += me) {
		ip = i + meh;	/* butterfly pair: i and ip */
		
		ir = 2 * i; 	ii = ir + 1;
		ipr = 2 * ip; 	ipi = ipr + 1;
		
		tRe = data[ipr] * uRe - data[ipi] * uIm;
		tIm = data[ipr] * uIm + data[ipi] * uRe;
		data[ipr] = data[ir] - tRe;
		data[ipi] = data[ii] - tIm;
		data[ir] += tRe;
		data[ii] += tIm;
	    }
	    vRe = uRe * wRe - uIm * wIm;
	    vIm = uRe * wIm + uIm * wRe;
	    uRe = vRe;
	    uIm = vIm;
	}
    }

    if (inv) {
	for (i = 0; i < n; i++) {
	    ir = 2 * i;	ii = ir + 1;
	    data[ir] /= (float)n;
	    data[ii] /= (float)(-n);
	}
    }

    return;
}

void sprfftf_normal(float *data, long p, long n, int inv)
{
    long i, i1, i2, i3, i4, nh, nq;
    float c1, h1r, h1i, h2r, h2i;
    float uRe, uIm, wRe, wIm, vRe, theta;

    nh = n / 2;
    
    theta = (float)(PI / (double)nh);
    if (inv) {
	c1 = 0.5;
	theta = -theta;
    } else {
	c1 = -0.5;
	sprifftf_normal(data, p - 1, nh, 0);
    }
    wRe = cosf(theta);
    wIm = -sinf(theta);
    uRe = wRe;
    uIm = wIm;
    nq = n / 4;

    for (i = 2; i <= nq; i++) {
	i1 = 2 * i - 2;		/* 3, 5; 2* i - 1*/
	i2 = i1 + 1;		/* 4, 6 */
	i3 = n + 1 - i2;	/* 7, 5;  11 - 4 = 7,  n = 8 */
	i4 = i3 + 1;		/* 8, 6 */

	h1r = 0.5f * (data[i1] + data[i3]);
	h1i = 0.5f * (data[i2] - data[i4]);
	h2r = -c1 * (data[i2] + data[i4]);
	h2i = c1 * (data[i1] - data[i3]);
	data[i1] = h1r + uRe * h2r - uIm * h2i;
	data[i2] = h1i + uRe * h2i + uIm * h2r;
	data[i3] = h1r - uRe * h2r + uIm * h2i;
	data[i4] = -h1i + uRe * h2i + uIm * h2r;

	vRe = uRe;
	uRe = uRe * wRe - uIm * wIm;
	uIm = vRe * wIm + uIm * wRe;
    }

    if (inv) {
	h1r = data[0];
	data[0] = 0.5f * (h1r + data[1]);
	data[1] = 0.5f * (h1r - data[1]);
	sprifftf_normal(data, p - 1, n / 2, 1);
    } else {
	h1r = data[0];
	data[0] = h1r + data[1];
	data[1] = h1r - data[1];
    }
    
    return;
}

int spfftf(float *real, float *imag, long fftl, int inv)
{
    long p;
    long n;
    
    p = nextpow2(fftl);
    n = POW2(p);
    if (n != fftl) {
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
        spfftf_bs_normal(real, imag, fftl, inv, 0);
        return SP_SUCCESS;
#else
	fprintf(stderr, "fft error: fft point must be a power of 2\n");
	return SP_FAILURE;
#endif
    }

    spfftf_normal(real, imag, p, n, inv);
    return SP_SUCCESS;
}

int sprfftf(float *data, long fftl, int inv)
{
    long p;
    long n;
    
    p = nextpow2(fftl);
    n = POW2(p);
    if (n != fftl) {
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
        spfftf_bs_normal(data, NULL, fftl, inv, 1);
        return SP_SUCCESS;
#else
	fprintf(stderr, "fft error: fft point must be a power of 2\n");
	return SP_FAILURE;
#endif
    }

    sprfftf_normal(data, p, n, inv);
    return SP_SUCCESS;
}

/*
 *	calculate FFT 
 */
static void spfft_normal(double *real, double *imag, long p, long n, int inv)
{
    long i, ip, j, k, m, me, meh, nh;
    double uRe, uIm, vRe, vIm, wRe, wIm, tRe, tIm;

    nh = n / 2;

    if (inv) {
	for (i = 0; i < n; i++) {
	    imag[i] = -imag[i];
	}
    }

    /* bit reversion */
    for (i = 0, j = 0; i < n - 1; i++) {
	if (j > i) {
	    tRe = real[j];	tIm = imag[j];
	    real[j] = real[i];	imag[j] = imag[i];
	    real[i] = tRe;	imag[i] = tIm;
	}
	for (k = nh; k > (j ^= k); k >>= 1);
    }

    /* butterfly numeration */
    for (m = 1; m <= p; m++) {
	me = POW2(m);		meh = me / 2;
	uRe = 1.0;		uIm = 0.0;
	wRe = cos(PI / (double)meh);
	wIm = -sin(PI / (double)meh);
	for (j = 0; j < meh; j++) { /* loop for each butterfly pairs */
	    for (i = j; i < n; i += me) {
		ip = i + meh;	/* butterfly pair: i and ip */
		tRe = real[ip] * uRe - imag[ip] * uIm;
		tIm = real[ip] * uIm + imag[ip] * uRe;
		real[ip] = real[i] - tRe;
		imag[ip] = imag[i] - tIm;
		real[i] += tRe;
		imag[i] += tIm;
	    }
	    vRe = uRe * wRe - uIm * wIm;
	    vIm = uRe * wIm + uIm * wRe;
	    uRe = vRe;
	    uIm = vIm;
	}
    }

    if (inv) {
	for (i = 0; i < n; i++) {
	    real[i] /= (double)n;
	    imag[i] /= (double)(-n);
	}
    }

    return;
}

void sprifft_normal(double *data, long p, long n, int inv)
{
    long i, ir, ii, ip, ipr, ipi, j, jr, ji, k, m, me, meh, nh;
    double uRe, uIm, vRe, vIm, wRe, wIm, tRe, tIm;

    nh = n / 2;

    if (inv) {
	for (i = 0; i < n; i++) {
	    ii = 2 * i + 1;
	    data[ii] = -data[ii];
	}
    }

    /* bit reversion */
    for (i = 0, j = 0; i < n - 1; i++) {
	if (j > i) {
	    ir = 2 * i; ii = ir + 1;
	    jr = 2 * j; ji = jr + 1;
	    tRe = data[jr];		tIm = data[ji];
	    data[jr] = data[ir];	data[ji] = data[ii];
	    data[ir] = tRe;		data[ii] = tIm;
	}
	for (k = nh; k > (j ^= k); k >>= 1);
    }

    /* butterfly numeration */
    for (m = 1; m <= p; m++) {
	me = POW2(m);		meh = me / 2;
	uRe = 1.0;		uIm = 0.0;
	wRe = cos(PI / (double)meh);
	wIm = -sin(PI / (double)meh);
	for (j = 0; j < meh; j++) { /* loop for each butterfly pairs */
	    for (i = j; i < n; i += me) {
		ip = i + meh;	/* butterfly pair: i and ip */
		
		ir = 2 * i; 	ii = ir + 1;
		ipr = 2 * ip; 	ipi = ipr + 1;
		
		tRe = data[ipr] * uRe - data[ipi] * uIm;
		tIm = data[ipr] * uIm + data[ipi] * uRe;
		data[ipr] = data[ir] - tRe;
		data[ipi] = data[ii] - tIm;
		data[ir] += tRe;
		data[ii] += tIm;
	    }
	    vRe = uRe * wRe - uIm * wIm;
	    vIm = uRe * wIm + uIm * wRe;
	    uRe = vRe;
	    uIm = vIm;
	}
    }

    if (inv) {
	for (i = 0; i < n; i++) {
	    ir = 2 * i;	ii = ir + 1;
	    data[ir] /= (double)n;
	    data[ii] /= (double)(-n);
	}
    }

    return;
}

void sprfft_normal(double *data, long p, long n, int inv)
{
    long i, i1, i2, i3, i4, nh, nq;
    double c1, h1r, h1i, h2r, h2i;
    double uRe, uIm, wRe, wIm, vRe, theta;

    nh = n / 2;
    
    theta = PI / (double)nh;
    if (inv) {
	c1 = 0.5;
	theta = -theta;
    } else {
	c1 = -0.5;
	sprifft_normal(data, p - 1, nh, 0);
    }
    wRe = cos(theta);
    wIm = -sin(theta);
    uRe = wRe;
    uIm = wIm;
    nq = n / 4;

    for (i = 2; i <= nq; i++) {
	i1 = 2 * i - 2;		/* 3, 5; 2* i - 1*/
	i2 = i1 + 1;		/* 4, 6 */
	i3 = n + 1 - i2;	/* 7, 5;  11 - 4 = 7,  n = 8 */
	i4 = i3 + 1;		/* 8, 6 */

	h1r = 0.5 * (data[i1] + data[i3]);
	h1i = 0.5 * (data[i2] - data[i4]);
	h2r = -c1 * (data[i2] + data[i4]);
	h2i = c1 * (data[i1] - data[i3]);
	data[i1] = h1r + uRe * h2r - uIm * h2i;
	data[i2] = h1i + uRe * h2i + uIm * h2r;
	data[i3] = h1r - uRe * h2r + uIm * h2i;
	data[i4] = -h1i + uRe * h2i + uIm * h2r;

	vRe = uRe;
	uRe = uRe * wRe - uIm * wIm;
	uIm = vRe * wIm + uIm * wRe;
    }

    if (inv) {
	h1r = data[0];
	data[0] = 0.5 * (h1r + data[1]);
	data[1] = 0.5 * (h1r - data[1]);
	sprifft_normal(data, p - 1, n / 2, 1);
    } else {
	h1r = data[0];
	data[0] = h1r + data[1];
	data[1] = h1r - data[1];
    }
    
    return;
}

int spfft(double *real, double *imag, long fftl, int inv)
{
    long p;
    long n;
    
    p = nextpow2(fftl);
    n = POW2(p);
    if (n != fftl) {
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
        spfft_bs_normal(real, imag, fftl, inv, 0);
        return SP_SUCCESS;
#else
	fprintf(stderr, "fft error: fft point must be a power of 2\n");
	return SP_FAILURE;
#endif
    }

    spfft_normal(real, imag, p, n, inv);
    return SP_SUCCESS;
}

int sprfft(double *data, long fftl, int inv)
{
    long p;
    long n;
    
    p = nextpow2(fftl);
    n = POW2(p);
    if (n != fftl) {
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
        spfft_bs_normal(data, NULL, fftl, inv, 1);
        return SP_SUCCESS;
#else
	fprintf(stderr, "fft error: fft point must be a power of 2\n");
	return SP_FAILURE;
#endif
    }

    sprfft_normal(data, p, n, inv);
    return SP_SUCCESS;
}

void datashifts(short *data, long length, long shift)
{
    long k;
    long len;

    if (shift == 0) {
	return;
    } else if (shift > 0) {
	for (k = length - 1; k >= 0; k--) {
	    if (k >= shift) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0;
	    }
	}
    } else {
	len = length + shift;
	for (k = 0; k < length; k++) {
	    if (k < len) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0;
	    }
	}
    }
    
    return;
}

void datashiftl(long *data, long length, long shift)
{
    long k;
    long len;

    if (shift == 0) {
	return;
    } else if (shift > 0) {
	for (k = length - 1; k >= 0; k--) {
	    if (k >= shift) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0;
	    }
	}
    } else {
	len = length + shift;
	for (k = 0; k < length; k++) {
	    if (k < len) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0;
	    }
	}
    }
    
    return;
}

void datashiftf(float *data, long length, long shift)
{
    long k;
    long len;

    if (shift == 0) {
	return;
    } else if (shift > 0) {
	for (k = length - 1; k >= 0; k--) {
	    if (k >= shift) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0.0;
	    }
	}
    } else {
	len = length + shift;
	for (k = 0; k < length; k++) {
	    if (k < len) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0.0;
	    }
	}
    }
    
    return;
}

void datashift(double *data, long length, long shift)
{
    long k;
    long len;

    if (shift == 0) {
	return;
    } else if (shift > 0) {
	for (k = length - 1; k >= 0; k--) {
	    if (k >= shift) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0.0;
	    }
	}
    } else {
	len = length + shift;
	for (k = 0; k < length; k++) {
	    if (k < len) {
		data[k] = data[k - shift];
	    } else {
		data[k] = 0.0;
	    }
	}
    }
    
    return;
}

void ridatashifts(short *real, short *imag, long length, long shift)
{
    if (length <= 0 || shift == 0) return;

    if (real != NULL) {
	datashifts(real, length, shift);
    }

    if (imag != NULL) {
	datashifts(imag, length, shift);
    }
    
    return;
}

void ridatashiftl(long *real, long *imag, long length, long shift)
{
    if (length <= 0 || shift == 0) return;

    if (real != NULL) {
	datashiftl(real, length, shift);
    }

    if (imag != NULL) {
	datashiftl(imag, length, shift);
    }
    
    return;
}

void ridatashiftf(float *real, float *imag, long length, long shift)
{
    if (length <= 0 || shift == 0) return;

    if (real != NULL) {
	datashiftf(real, length, shift);
    }

    if (imag != NULL) {
	datashiftf(imag, length, shift);
    }
    
    return;
}

void ridatashift(double *real, double *imag, long length, long shift)
{
    if (length <= 0 || shift == 0) return;

    if (real != NULL) {
	datashift(real, length, shift);
    }

    if (imag != NULL) {
	datashift(imag, length, shift);
    }
    
    return;
}

static void circshiftcores(short *data, long length, long shift, int flag)
{
    long init;
    long i, k, l;
    short value, value2;

    k = 0;
    init = k;
    value = data[k];
    for (i = 0; i < length; i++) {
	l = k + shift;
	if (l < 0) l += length;
	else if (l >= length) l -= length;

	value2 = data[l];
	data[l] = value;
	value = value2;
	k = l;
	    
	if (k == init && flag) {
	    k++;
	    init = k;
	    value = data[k];
	}
    }

    return;
}

static void circshiftcorel(long *data, long length, long shift, int flag)
{
    long init;
    long i, k, l;
    long value, value2;

    k = 0;
    init = k;
    value = data[k];
    for (i = 0; i < length; i++) {
	l = k + shift;
	if (l < 0) l += length;
	else if (l >= length) l -= length;

	value2 = data[l];
	data[l] = value;
	value = value2;
	k = l;
	    
	if (k == init && flag) {
	    k++;
	    init = k;
	    value = data[k];
	}
    }

    return;
}

static void circshiftcoref(float *data, long length, long shift, int flag)
{
    long init;
    long i, k, l;
    float value, value2;

    k = 0;
    init = k;
    value = data[k];
    for (i = 0; i < length; i++) {
	l = k + shift;
	if (l < 0) l += length;
	else if (l >= length) l -= length;

	value2 = data[l];
	data[l] = value;
	value = value2;
	k = l;
	    
	if (k == init && flag) {
	    k++;
	    init = k;
	    value = data[k];
	}
    }

    return;
}

static void circshiftcore(double *data, long length, long shift, int flag)
{
    long init;
    long i, k, l;
    double value, value2;

    k = 0;
    init = k;
    value = data[k];
    for (i = 0; i < length; i++) {
	l = k + shift;
	if (l < 0) l += length;
	else if (l >= length) l -= length;

	value2 = data[l];
	data[l] = value;
	value = value2;
	k = l;
	    
	if (k == init && flag) {
	    k++;
	    init = k;
	    value = data[k];
	}
    }

    return;
}

void circshifts(short *data, long length, long shift)
{
    int flag = 0;

    if (data == NULL || length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    circshiftcores(data, length, shift, flag);

    return;
}

void circshiftl(long *data, long length, long shift)
{
    int flag = 0;

    if (data == NULL || length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    circshiftcorel(data, length, shift, flag);

    return;
}

void circshiftf(float *data, long length, long shift)
{
    int flag = 0;

    if (data == NULL || length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    circshiftcoref(data, length, shift, flag);

    return;
}

void circshift(double *data, long length, long shift)
{
    int flag = 0;

    if (data == NULL || length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    circshiftcore(data, length, shift, flag);

    return;
}

void ricircshiftf(float *real, float *imag, long length, long shift)
{
    int flag = 0;

    if (length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    if (real != NULL) {
	circshiftcoref(real, length, shift, flag);
    }

    if (imag != NULL) {
	circshiftcoref(imag, length, shift, flag);
    }

    return;
}

void ricircshifts(short *real, short *imag, long length, long shift)
{
    int flag = 0;

    if (length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    if (real != NULL) {
	circshiftcores(real, length, shift, flag);
    }

    if (imag != NULL) {
	circshiftcores(imag, length, shift, flag);
    }

    return;
}

void ricircshiftl(long *real, long *imag, long length, long shift)
{
    int flag = 0;

    if (length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    if (real != NULL) {
	circshiftcorel(real, length, shift, flag);
    }

    if (imag != NULL) {
	circshiftcorel(imag, length, shift, flag);
    }

    return;
}

void ricircshift(double *real, double *imag, long length, long shift)
{
    int flag = 0;

    if (length <= 0 || shift == 0 || ((shift = shift % length) == 0)) return;

    if (ABS(gcd(length, shift)) != 1) flag = 1;

    if (real != NULL) {
	circshiftcore(real, length, shift, flag);
    }

    if (imag != NULL) {
	circshiftcore(imag, length, shift, flag);
    }

    return;
}

/*
 *	fft turn for float data 
 */
void fftturnf(float *real, float *imag, long fftl)
{
    long i;
    long hfftl;

    hfftl = fftl - (fftl / 2);

    if (real != NULL) {
	/* real part */
	for (i = 1; i < hfftl; i++) {
	    real[fftl - i] = real[i];
	}
    }
    if (imag != NULL) {
	/* imaginary part */
	for (i = 1; i < hfftl; i++) {
	    imag[fftl - i] = -imag[i];
	}
    }

    return;
} 

/*
 *	fft turn 
 */
void fftturn(double *real, double *imag, long fftl)
{
    long i;
    long hfftl;

    hfftl = fftl - (fftl / 2);

    if (real != NULL) {
	/* real part */
	for (i = 1; i < hfftl; i++) {
	    real[fftl - i] = real[i];
	}
    }
    if (imag != NULL) {
	/* imaginary part */
	for (i = 1; i < hfftl; i++) {
	    imag[fftl - i] = -imag[i];
	}
    }

    return;
} 

/*
 *	fft shift for float data
 */
void fftshiftf(float *real, float *imag, long fftl)
{
    long i, j;
    long hfftl, hfftl2;
    float value, value2;

    hfftl = fftl / 2;
    hfftl2 = fftl - hfftl;

    if (real != NULL) {
	/* real part */
	value2 = real[hfftl2 - 1];
	real[hfftl2 - 1] = real[0];	/* if fft point is odd */
	for (i = fftl - 1; i >= hfftl2; i--) {
            value = real[i];
	    real[i] = value2;

	    j = i - hfftl - 1;
            if (j >= 0) {
                value2 = real[j];
            }
            real[i - hfftl2] = value;
	}
    }
    if (imag != NULL) {
	/* imaginaly part */
	value2 = imag[hfftl2 - 1];
	imag[hfftl2 - 1] = imag[0];	/* if fft point is odd */
	for (i = fftl - 1; i >= hfftl2; i--) {
	    value = imag[i];
	    imag[i] = value2;

	    j = i - hfftl - 1;
	    if (j >= 0) {
		value2 = imag[j];
	    }
	    imag[i - hfftl2] = value;
	}
    }

    return;
} 

void ifftshiftf(float *real, float *imag, long fftl)
{
    long i, j;
    long hfftl, hfftl2;
    float value, value2;

    hfftl = fftl / 2;
    hfftl2 = fftl - hfftl;

    if (real != NULL) {
	/* real part */
	value2 = real[hfftl];
	real[hfftl] = real[fftl - 1];	/* if fft point is odd */
	for (i = 0; i < hfftl; i++) {
	    value = real[i];
	    real[i] = value2;

	    j = i + hfftl + 1;
	    if (j < fftl) {
		value2 = real[j];
	    }
	    real[i + hfftl2] = value;
	}
    }
    if (imag != NULL) {
	/* imaginaly part */
	value2 = imag[hfftl];
	imag[hfftl] = imag[fftl - 1];	/* if fft point is odd */
	for (i = 0; i < hfftl; i++) {
	    value = imag[i];
	    imag[i] = value2;

	    j = i + hfftl + 1;
	    if (j < fftl) {
		value2 = imag[j];
	    }
	    imag[i + hfftl2] = value;
	}
    }

    return;
} 

/*
 *	fft shift 
 */
void fftshift(double *real, double *imag, long fftl)
{
    long i, j;
    long hfftl, hfftl2;
    double value, value2;

    hfftl = fftl / 2;
    hfftl2 = fftl - hfftl;

    if (real != NULL) {
	/* real part */
	value2 = real[hfftl2 - 1];
	real[hfftl2 - 1] = real[0];	/* if fft point is odd */
	for (i = fftl - 1; i >= hfftl2; i--) {
            value = real[i];
	    real[i] = value2;

	    j = i - hfftl - 1;
            if (j >= 0) {
                value2 = real[j];
            }
            real[i - hfftl2] = value;
	}
    }
    if (imag != NULL) {
	/* imaginaly part */
	value2 = imag[hfftl2 - 1];
	imag[hfftl2 - 1] = imag[0];	/* if fft point is odd */
	for (i = fftl - 1; i >= hfftl2; i--) {
	    value = imag[i];
	    imag[i] = value2;

	    j = i - hfftl - 1;
	    if (j >= 0) {
		value2 = imag[j];
	    }
	    imag[i - hfftl2] = value;
	}
    }

    return;
} 

void ifftshift(double *real, double *imag, long fftl)
{
    long i, j;
    long hfftl, hfftl2;
    double value, value2;

    hfftl = fftl / 2;
    hfftl2 = fftl - hfftl;

    if (real != NULL) {
	/* real part */
	value2 = real[hfftl];
	real[hfftl] = real[fftl - 1];	/* if fft point is odd */
	for (i = 0; i < hfftl; i++) {
	    value = real[i];
	    real[i] = value2;

	    j = i + hfftl + 1;
	    if (j < fftl) {
		value2 = real[j];
	    }
	    real[i + hfftl2] = value;
	}
    }
    if (imag != NULL) {
	/* imaginaly part */
	value2 = imag[hfftl];
	imag[hfftl] = imag[fftl - 1];	/* if fft point is odd */
	for (i = 0; i < hfftl; i++) {
	    value = imag[i];
	    imag[i] = value2;

	    j = i + hfftl + 1;
	    if (j < fftl) {
		value2 = imag[j];
	    }
	    imag[i + hfftl2] = value;
	}
    }

    return;
}

/* support in-place (can be real == rspec) and NULL-imag (version 0.9.1+) */
void ffttorfftf(float *real, float *imag, float *rspec, long fftl)
{
    long k;
    long remain;
    long fftl2;
    long tidx;
    float real_fftl2;

    remain = fftl % 2;
    fftl2 = fftl / 2;

    if (remain == 1) {
        for (k = fftl2; k >= 1; k--) {
            tidx = k * 2 - 1;
            rspec[tidx] = real[k];
            if (imag == NULL) {
                rspec[tidx + 1] = 0.0;
            } else {
                rspec[tidx + 1] = imag[k];
            }
        }
    } else {
        real_fftl2 = real[fftl2];
        for (k = fftl2 - 1; k >= 1; k--) {
            tidx = k * 2;
            rspec[tidx] = real[k];
            if (imag == NULL) {
                rspec[tidx + 1] = 0.0;
            } else {
                rspec[tidx + 1] = imag[k];
            }
        }
        rspec[1] = real_fftl2;
    }
    rspec[0] = real[0];

    return;
}

void ffttorfft(double *real, double *imag, double *rspec, long fftl)
{
    long k;
    long remain;
    long fftl2;
    long tidx;
    double real_fftl2;

    remain = fftl % 2;
    fftl2 = fftl / 2;

    if (remain == 1) {
        for (k = fftl2; k >= 1; k--) {
            tidx = k * 2 - 1;
            rspec[tidx] = real[k];
            if (imag == NULL) {
                rspec[tidx + 1] = 0.0;
            } else {
                rspec[tidx + 1] = imag[k];
            }
        }
    } else {
        real_fftl2 = real[fftl2];
        for (k = fftl2 - 1; k >= 1; k--) {
            tidx = k * 2;
            rspec[tidx] = real[k];
            if (imag == NULL) {
                rspec[tidx + 1] = 0.0;
            } else {
                rspec[tidx + 1] = imag[k];
            }
        }
        rspec[1] = real_fftl2;
    }
    rspec[0] = real[0];

    return;
}

void ffttorfftipf(float *real, float *imag, long fftl)
{
    ffttorfftf(real, imag, real, fftl);
    return;
}

void ffttorfftip(double *real, double *imag, long fftl)
{
    ffttorfft(real, imag, real, fftl);
    return;
}

void rffttofftf(float *rspec, float *real, float *imag, long fftl)
{
    long k;
    long remain;
    long fftl2;
    long oidx;
    float rspec_1;

    remain = fftl % 2;
    fftl2 = fftl / 2;

    real[0] = rspec[0];

    if (remain == 1) {
        for (k = 1; k <= fftl2; k++) {
            oidx = k * 2 - 1;
            real[k] = rspec[oidx];
            if (imag != NULL) {
                imag[k] = rspec[oidx + 1];
            }
        }
        if (imag != NULL) {
            imag[0] = 0.0f;
        }
        ++fftl2;
    } else {
        rspec_1 = rspec[1];
        for (k = 1; k < fftl2; k++) {
            oidx = k * 2;
            real[k] = rspec[oidx];
            if (imag != NULL) {
                imag[k] = rspec[oidx + 1];
            }
        }
        real[fftl2] = rspec_1;
        if (imag != NULL) {
            imag[0] = imag[fftl2] = 0.0f;
        }
    }

    for (k = 1; k < fftl2; k++) {
        real[fftl - k] = real[k];
        if (imag != NULL) {
            imag[fftl - k] = -imag[k];
        }
    }

    return;
}

void rffttofft(double *rspec, double *real, double *imag, long fftl)
{
    long k;
    long remain;
    long fftl2;
    long oidx;
    double rspec_1;

    remain = fftl % 2;
    fftl2 = fftl / 2;

    real[0] = rspec[0];

    if (remain == 1) {
        for (k = 1; k <= fftl2; k++) {
            oidx = k * 2 - 1;
            real[k] = rspec[oidx];
            if (imag != NULL) {
                imag[k] = rspec[oidx + 1];
            }
        }
        if (imag != NULL) {
            imag[0] = 0.0f;
        }
        ++fftl2;
    } else {
        rspec_1 = rspec[1];
        for (k = 1; k < fftl2; k++) {
            oidx = k * 2;
            real[k] = rspec[oidx];
            if (imag != NULL) {
                imag[k] = rspec[oidx + 1];
            }
        }
        real[fftl2] = rspec_1;
        if (imag != NULL) {
            imag[0] = imag[fftl2] = 0.0f;
        }
    }

    for (k = 1; k < fftl2; k++) {
        real[fftl - k] = real[k];
        if (imag != NULL) {
            imag[fftl - k] = -imag[k];
        }
    }

    return;
}

void rffttofftipf(float *real, float *imag, long fftl)
{
    rffttofftf(real, real, imag, fftl);
    return;
}

void rffttofftip(double *real, double *imag, long fftl)
{
    rffttofft(real, real, imag, fftl);
    return;
}

void multrfftf(float *rspec1, float *rspec2, long fftl)
{
    long k;
    float xr, xi;
    
    rspec1[0] = rspec1[0] * rspec2[0];

    if (fftl % 2 == 1) {
        k = 1;
    } else {
        rspec1[1] = rspec1[1] * rspec2[1];
        k = 2;
    }
    
    for (; k < fftl - 1; k += 2) {
	xr = rspec1[k] * rspec2[k] - rspec1[k+1] * rspec2[k+1];
	xi = rspec1[k] * rspec2[k+1] + rspec1[k+1] * rspec2[k];
	rspec1[k] = xr;
	rspec1[k+1] = xi;
    }

    return;
}

void multrfft(double *rspec1, double *rspec2, long fftl)
{
    long k;
    double xr, xi;

    rspec1[0] = rspec1[0] * rspec2[0];
    
    if (fftl % 2 == 1) {
        k = 1;
    } else {
        rspec1[1] = rspec1[1] * rspec2[1];
        k = 2;
    }
    
    for (; k < fftl - 1; k += 2) {
	xr = rspec1[k] * rspec2[k] - rspec1[k+1] * rspec2[k+1];
	xi = rspec1[k] * rspec2[k+1] + rspec1[k+1] * rspec2[k];
	rspec1[k] = xr;
	rspec1[k+1] = xi;
    }

    return;
}

void rffttopowerf(float *rspec, float exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    float xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    xIm0 = 0.0f;
    
    if (exponent == 1.0f) {
	xRe0 = rspec[0] * rspec[0];
        if (remain == 0) xIm0 = rspec[1] * rspec[1];

	for (k = 1; k < fftl2; k++) {
	    rspec[k] = rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain];
	}
    } else if (exponent == 0.5f) {
	xRe0 = fabsf(rspec[0]);
        if (remain == 0) xIm0 = fabsf(rspec[1]);
	
	for (k = 1; k < fftl2; k++) {
	    rspec[k] = sqrtf(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain]);
	}
    } else {
	xRe0 = powf(rspec[0] * rspec[0], exponent);
        if (remain == 0) xIm0 = powf(rspec[1] * rspec[1], exponent);
	
	for (k = 1; k < fftl2; k++) {
	    rspec[k] = powf(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain], exponent);
	}
    }
    rspec[0] = xRe0;
    if (remain == 0) {
        rspec[fftl2] = xIm0;
    }
    
    fftturnf(rspec, NULL, fftl);

    return;
}

void rffttopower(double *rspec, double exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    double xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    xIm0 = 0.0;
    
    if (exponent == 1.0) {
	xRe0 = rspec[0] * rspec[0];
        if (remain == 0) xIm0 = rspec[1] * rspec[1];

	for (k = 1; k < fftl2; k++) {
	    rspec[k] = rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain];
	}
    } else if (exponent == 0.5) {
	xRe0 = fabs(rspec[0]);
        if (remain == 0) xIm0 = fabs(rspec[1]);
	
	for (k = 1; k < fftl2; k++) {
	    rspec[k] = sqrt(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain]);
	}
    } else {
	xRe0 = pow(rspec[0] * rspec[0], exponent);
        if (remain == 0) xIm0 = pow(rspec[1] * rspec[1], exponent);
	
	for (k = 1; k < fftl2; k++) {
	    rspec[k] = pow(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain], exponent);
	}
    }
    rspec[0] = xRe0;
    if (remain == 0) {
        rspec[fftl2] = xIm0;
    }
    
    fftturn(rspec, NULL, fftl);

    return;
}

void powertorfftf(float *powspec, float orig_exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    float value_fftl2;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    if (remain == 0) {
        value_fftl2 = powspec[fftl2];
    } else {
        value_fftl2 = 0.0f;
    }
    
    if (orig_exponent == 1.0f) {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = sqrtf(powspec[k]);
	    powspec[2 * k + 1 - remain] = 0.0f;
	}
        if (remain == 0) powspec[1] = sqrtf(value_fftl2);
    } else if (orig_exponent == 0.5f) {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = powspec[k];
	    powspec[2 * k + 1 - remain] = 0.0f;
	}
        if (remain == 0) powspec[1] = value_fftl2;
    } else {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = powf(powspec[k], 0.5f / orig_exponent);
	    powspec[2 * k + 1 - remain] = 0.0f;
	}
        if (remain == 0) powf(value_fftl2, 0.5f / orig_exponent);
    }

    return;
}

void powertorfft(double *powspec, double orig_exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    double value_fftl2;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    if (remain == 0) {
        value_fftl2 = powspec[fftl2];
    } else {
        value_fftl2 = 0.0;
    }
    
    if (orig_exponent == 1.0) {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = sqrt(powspec[k]);
	    powspec[2 * k + 1 - remain] = 0.0;
	}
        if (remain == 0) powspec[1] = sqrt(value_fftl2);
    } else if (orig_exponent == 0.5) {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = powspec[k];
	    powspec[2 * k + 1 - remain] = 0.0;
	}
        if (remain == 0) powspec[1] = value_fftl2;
    } else {
	for (k = fftl2 - 1; k >= 1; k--) {
	    powspec[2 * k - remain] = pow(powspec[k], 0.5 / orig_exponent);
	    powspec[2 * k + 1 - remain] = 0.0;
	}
        if (remain == 0) pow(value_fftl2, 0.5 / orig_exponent);
    }

    return;
}

void rffttohpowerf(float *rspec, float *ohpowspec, float exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    float xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    xIm0 = 0.0f;
    
    if (exponent == 1.0f) {
	xRe0 = rspec[0] * rspec[0];
        if (remain == 0) {
            xIm0 = rspec[1] * rspec[1];
        }

	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain];
	}
    } else if (exponent == 0.5f) {
	xRe0 = fabsf(rspec[0]);
        if (remain == 0) {
            xIm0 = fabsf(rspec[1]);
        }
	
	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = sqrtf(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain]);
	}
    } else {
	xRe0 = powf(rspec[0] * rspec[0], exponent);
        if (remain == 0) {
            xIm0 = powf(rspec[1] * rspec[1], exponent);
        }
	
	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = powf(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain], exponent);
	}
    }
    ohpowspec[0] = xRe0;
    if (remain == 0) {
        ohpowspec[fftl2] = xIm0;
    }
    
    return;
}

void rffttohpower(double *rspec, double *ohpowspec, double exponent, long fftl)
{
    long k;
    long remain;
    long fftl2;
    double xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    xIm0 = 0.0;
    
    if (exponent == 1.0) {
	xRe0 = rspec[0] * rspec[0];
        if (remain == 0) {
            xIm0 = rspec[1] * rspec[1];
        }

	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain];
	}
    } else if (exponent == 0.5) {
	xRe0 = fabs(rspec[0]);
        if (remain == 0) {
            xIm0 = fabs(rspec[1]);
        }
	
	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = sqrt(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain]);
	}
    } else {
	xRe0 = pow(rspec[0] * rspec[0], exponent);
        if (remain == 0) {
            xIm0 = pow(rspec[1] * rspec[1], exponent);
        }
	
	for (k = 1; k < fftl2; k++) {
	    ohpowspec[k] = pow(rspec[2*k-remain] * rspec[2*k-remain] + rspec[2*k+1-remain] * rspec[2*k+1-remain], exponent);
	}
    }
    ohpowspec[0] = xRe0;
    if (remain == 0) {
        ohpowspec[fftl2] = xIm0;
    }
    
    return;
}

void rffttoanglef(float *rspec, long fftl)
{
    long k;
    long remain;
    long fftl2;
    float xRe, xIm;
    float xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    
    xRe = rspec[0];
    xRe0 = (xRe == 0.0f ? 0.0f : atan2f(0.0f, xRe));
    if (remain == 0) {
        xRe = rspec[1];
        xIm0 = (xRe == 0.0f ? 0.0f : atan2f(0.0f, xRe));
    } else {
        xIm0 = 0.0f;
    }
	
    for (k = 1; k < fftl2; k++) {
	xRe = rspec[2*k-remain];
	xIm = rspec[2*k-remain+1];
	if (xRe == 0.0f && xIm == 0.0f) {
	    rspec[k] = 0.0f;
	} else {
	    rspec[k] = atan2f(xIm, xRe);
	}
    }
    rspec[0] = xRe0;
    if (remain == 0) {
        rspec[fftl2] = xIm0;
    }
	
    fftturnf(NULL, rspec, fftl);
    
    return;
}

void rffttoangle(double *rspec, long fftl)
{
    long k;
    long remain;
    long fftl2;
    double xRe, xIm;
    double xRe0, xIm0;
    
    remain = fftl % 2;
    fftl2 = fftl / 2 + remain;
    
    xRe = rspec[0];
    xRe0 = (xRe == 0.0 ? 0.0 : atan2(0.0, xRe));
    if (remain == 0) {
        xRe = rspec[1];
        xIm0 = (xRe == 0.0 ? 0.0 : atan2(0.0, xRe));
    } else {
        xIm0 = 0.0;
    }
	
    for (k = 1; k < fftl2; k++) {
	xRe = rspec[2*k-remain];
	xIm = rspec[2*k-remain+1];
	if (xRe == 0.0 && xIm == 0.0) {
	    rspec[k] = 0.0;
	} else {
	    rspec[k] = atan2(xIm, xRe);
	}
    }
    rspec[0] = xRe0;
    if (remain == 0) {
        rspec[fftl2] = xIm0;
    }
	
    fftturn(NULL, rspec, fftl);
    
    return;
}

void rffttologf(float *rspec, long fftl)
{
    long k;
    long remain;
    float xRe;
    
    remain = fftl % 2;

    for (k = 0; k <= 1; k++) {
        if (remain == 1) {
            break;
        }
        xRe = rspec[k] * rspec[k];
	if (xRe <= 0.0f) {
	    xRe = (float)SP_TINY_NUMBER;
	}
	xRe = 0.5f * logf(xRe);
        rspec[k] = xRe;
    }
    
    for (; k < fftl - 1; k += 2) {
	xRe = rspec[k] * rspec[k] + rspec[k + 1] * rspec[k + 1];
	if (xRe <= 0.0f) {
	    xRe = (float)SP_TINY_NUMBER;
	}
	xRe = 0.5f * logf(xRe);
	rspec[k] = xRe;
	rspec[k + 1] = 0.0f;
    }
    
    return;
}

void rffttolog(double *rspec, long fftl)
{
    long k;
    long remain;
    double xRe;
    
    remain = fftl % 2;

    for (k = 0; k <= 1; k++) {
        if (remain == 1) {
            break;
        }
        xRe = rspec[k] * rspec[k];
	if (xRe <= 0.0) {
	    xRe = SP_TINY_NUMBER;
	}
	xRe = 0.5 * log(xRe);
        rspec[k] = xRe;
    }
    
    for (; k < fftl - 1; k += 2) {
	xRe = rspec[k] * rspec[k] + rspec[k + 1] * rspec[k + 1];
	if (xRe <= 0.0) {
	    xRe = SP_TINY_NUMBER;
	}
	xRe = 0.5 * log(xRe);
	rspec[k] = xRe;
	rspec[k + 1] = 0.0;
    }
    
    return;
}

void rffttoexpf(float *rspec, long fftl)
{
    long k;
    long remain;
    float amp, theta;
    
    remain = fftl % 2;

    rspec[0] = expf(rspec[0]);
    
    if (remain == 1) {
        k = 1;
    } else {
        rspec[1] = expf(rspec[1]);
        k = 2;
    }
    for (; k < fftl - 1; k += 2) {
	amp = expf(rspec[k]);
	theta = rspec[k + 1];
	
	rspec[k] = amp * cosf(theta);
	rspec[k + 1] = amp * sinf(theta);
    }
    
    return;
}

void rffttoexp(double *rspec, long fftl)
{
    long k;
    long remain;
    double amp, theta;
    
    remain = fftl % 2;

    rspec[0] = exp(rspec[0]);
    
    if (remain == 1) {
        k = 1;
    } else {
        rspec[1] = exp(rspec[1]);
        k = 2;
    }
    for (; k < fftl - 1; k += 2) {
	amp = exp(rspec[k]);
	theta = rspec[k + 1];
	
	rspec[k] = amp * cos(theta);
	rspec[k + 1] = amp * sin(theta);
    }
    
    return;
}

int rfftcep(double *x, double *pw, long fftl)
{
    long k;
    long remain;
    double value;

    remain = fftl % 2;
    
    sprfft(x, fftl, 0);

    for (k = 0; k < fftl;) {
	if (k == 0) {
	    value = x[k] * x[k];
	} else {
            value = x[k] * x[k] + x[k+1] * x[k+1];
	}

	if (pw != NULL) {
            if (remain == 1 && k > 0) {
                pw[(k + 1) / 2] = value;
            } else {
                pw[k / 2] = value;
            }
	}
	
	if (value <= 0.0) {
	    value = SP_TINY_NUMBER;
	}
	x[k] = 0.5 * log(value);

	if (k == 0) {
            if (remain != 1) {
                value = x[k+1] * x[k+1];
                x[k+1] = 0.5 * log(value);
	    
                if (pw != NULL) {
                    pw[fftl / 2] = value;
                }
            }
	} else {
            x[k+1] = 0.0;
	}

        if (remain == 1 && k == 0) {
            k += 1;
        } else {
            k += 2;
        }
    }
    
    if (pw != NULL) {
	fftturn(pw, NULL, fftl);
    }
    
    sprfft(x, fftl, 1);

    return SP_SUCCESS;
}

/* in-place */
int spectocepip(double *real, double *imag, long fftl)
{
    long k;

    for (k = 0; k < fftl; k++) {
	real[k] = real[k] * real[k] + imag[k] * imag[k];
	if (real[k] <= 0.0) {
	    real[k] = SP_TINY_NUMBER;
	}
	
	real[k] = 0.5 * log(real[k]);
	imag[k] = 0.0;
    }

    fft(real, imag, fftl, 1);

    return SP_SUCCESS;
}

int spectocep(double *real, double *imag, double *cep, long fftl)
{
    long k;
    long idx;
    long remain;
    long fftl2;
    double value;

    fftl2 = fftl / 2;
    remain = fftl % 2;

    for (k = 0; k < fftl2; k++) {
        idx = k + remain;
	value = real[idx] * real[idx] + imag[idx] * imag[idx];
	if (value <= 0.0) {
	    value = SP_TINY_NUMBER;
	}
	
	cep[2 * k + remain] = 0.5 * log(value);
	cep[2 * k + 1 + remain] = 0.0;
    }

    if (remain == 1) {
        value = real[0] * real[0] + imag[0] * imag[0];
        if (value <= 0.0) {
            value = SP_TINY_NUMBER;
        }
        cep[0] = 0.5 * log(value);
    } else {
        value = real[fftl2] * real[fftl2] + imag[fftl2] * imag[fftl2];
        if (value <= 0.0) {
            value = SP_TINY_NUMBER;
        }
        cep[1] = 0.5 * log(value);
    }

    sprfft(cep, fftl, 1);
    
    return SP_SUCCESS;
}

/* in-place */
int ceptospecip(double *real, double *imag, long fftl)
{
    long k;
    
    for (k = 0; k < fftl; k++) {
	imag[k] = 0.0;
    }

    fft(real, imag, fftl, 0);

    for (k = 0; k < fftl; k++) {
	cexp(&real[k], &imag[k]);
    }

    return SP_SUCCESS;
}

int ceptospec(double *cep, double *real, double *imag, long fftl)
{
    long k;
    
    for (k = 0; k < fftl; k++) {
	real[k] = cep[k];
	imag[k] = 0.0;
    }

    ceptospecip(real, imag, fftl);

    return SP_SUCCESS;
}

int ceptompc(double *cep, long fftl)
{
    long k;
    long fftl2;

    fftl2 = fftl / 2 + fftl % 2;
    
    for (k = 0; k < fftl; k++) {
	if (k == 0) {
	    /*cep[k] = cep[k];*/
	} else if (k < fftl2) {
	    cep[k] = 2.0 * cep[k];
	} else {
	    cep[k] = 0.0;
	}
    }
    
    return SP_SUCCESS;
}

int ceplif(double *cep, long fftl, long lif)
{
    long k;
    long lif2;

    if (lif >= 0) {
	lif2 = fftl - lif;
	for (k = 0; k < fftl; k++) {
	    if (k > lif && k < lif2) {
		cep[k] = 0.0;
	    }
	}
    } else {
	lif *= -1;
	lif2 = fftl - lif;
	for (k = 0; k < fftl; k++) {
	    if (k <= lif || k >= lif2) {
		cep[k] = 0.0;
	    }
	}
    }

    return SP_SUCCESS;
}

int cepwlif(double *cep, long fftl, long period, long width)
{
    long k, l;
    long hfftl;
    long lif, lif2;

    if (period <= 0) {
	return SP_SUCCESS;
    }

    width = spMax(width, 0);
    hfftl = fftl / 2 + 1;

    for (l = 1;; l++) {
	lif = l * period - width / 2;
	lif2 = fftl - lif;

	for (k = 0; k < width; k++) {
	    if (lif + k < hfftl) {
		cep[lif + k] = 0.0;
		cep[lif2 - k] = 0.0;
	    }
	}
	if (lif + k >= hfftl) {
	    break;
	}
    }

    return SP_SUCCESS;
}

static spBool spSearchFFTPluginCallback(spPlugin *plugin, void *call_data)
{
    return spIsFFTPlugin(plugin);
}

int spGetNumFFTPlugin(void)
{
    return spGetNumSpecificPlugin(spSearchFFTPluginCallback, NULL);
}

char *xspSearchFFTPluginFile(int *index)
{
    return xspSearchSpecificPluginFile(index, spSearchFFTPluginCallback, NULL);
}

spBool spSetDefaultFFTPlugin(const char *name)
{
    spPlugin *plugin;

    if (strnone(name) || strcaseeq(name, "none") || strcaseeq(name, "default")) {
	sp_default_fft_plugin_name[0] = NUL;
	spDebug(20, "spSetDefaultFFTPlugin", "use default FFT\n");
	return SP_TRUE;
    }
    
    if ((plugin = spLoadFFTPlugin(name)) != NULL) {
	spDebug(20, "spSetDefaultFFTPlugin", "use %s\n", name);
	spStrCopy(sp_default_fft_plugin_name, sizeof(sp_default_fft_plugin_name), name);
	
	spFreeFFTPlugin(plugin);
	return SP_TRUE;
    } else {
	spDebug(20, "spSetDefaultFFTPlugin", "spLoadFFTPlugin failed: name = %s\n", name);
	return SP_FALSE;
    }
}

spBool spIsFFTPlugin(spPlugin *plugin)
{
    if (plugin == NULL) return SP_FALSE;

    if (spEqPluginType(plugin, SP_PLUGIN_FFT) == SP_TRUE) {
	spDebug(100, "spIsFFTPlugin", "plugin %s: SP_PLUGIN_FFT\n", spGetPluginName(plugin));
	return SP_TRUE;
    }
    
    spDebug(100, "spIsFFTPlugin", "plugin %s: NOT SP_PLUGIN_FFT\n", spGetPluginName(plugin));
    
    return SP_FALSE;
}

spPlugin *spLoadFFTPlugin(const char *name)
{
    spPlugin *plugin;
    
    if ((plugin = spLoadPlugin(name)) == NULL) {
	spDebug(20, "spLoadFFTPlugin", "spLoadPlugin failed\n");
	return NULL;
    }
    if (spIsFFTPlugin(plugin) == SP_FALSE
	|| spInitPluginInstance(plugin) == SP_FALSE) {
	spDebug(20, "spLoadFFTPlugin", "spIsFFTPlugin or spInitPluginInstance failed\n");
	spFreePlugin(plugin);
	return NULL;
    }

    return plugin;
}

spBool spFreeFFTPlugin(spPlugin *plugin)
{
    if (plugin == NULL) return SP_FALSE;
    
    spFreePluginInstance(plugin);
    
    return spFreePlugin(plugin);
}

spFFTSpeed spGetFFTSpeed(spFFTPrecision precision)
{
#if defined(IPHONE)
    if (vDSP_create_fftsetupD != NULL) {
	return SP_FFT_SPEED_VERY_FAST;
    }
#elif defined(MACOSX)
    if (spIsAltiVecDoubleSupportedMac() == SP_TRUE) {
	return SP_FFT_SPEED_VERY_FAST;
    }
    if (spIsAltiVecSupportedMac() == SP_TRUE
	&& precision <= SP_FFT_FLOAT_PRECISION) {
	return SP_FFT_SPEED_VERY_FAST;
    }
#elif defined(USE_SPL)
    {
	return SP_FFT_SPEED_VERY_FAST;
    }
#endif
    
    return SP_FFT_SPEED_NORMAL;
}

static void spFreeFFTArch(spFFTRec fftrec)
{
#if defined(MACOSX)
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
	    if (fftrec->setupD != NULL) {
		vDSP_destroy_fftsetupD(fftrec->setupD);
	    }
	
	    if (fftrec->scbufD.realp != NULL) {
		xfree(fftrec->scbufD.realp);
		fftrec->scbufD.realp = NULL;
	    }
	    if (fftrec->scbufD.imagp != NULL) {
		xfree(fftrec->scbufD.imagp);
		fftrec->scbufD.imagp = NULL;
	    }
	} else {
	    if (fftrec->setup != NULL) {
		vDSP_destroy_fftsetup(fftrec->setup);
	    }
	
	    if (fftrec->scbuf.realp != NULL) {
		xfree(fftrec->scbuf.realp);
		fftrec->scbuf.realp = NULL;
	    }
	    if (fftrec->scbuf.imagp != NULL) {
		xfree(fftrec->scbuf.imagp);
		fftrec->scbuf.imagp = NULL;
	    }
	}

	return;
    }
#elif defined(USE_SPL)
    {
	if (fftrec->precision == SP_FFT_FLOAT_PRECISION) {
	    nspcrFft(NULL, NULL, fftrec->order, NSP_Free);
	    nspsRealFftl(NULL, fftrec->order, NSP_Free);
	    nspsCcsFftl(NULL, fftrec->order, NSP_Free);
	
	    if (fftrec->freal != NULL) xfree(fftrec->freal);
	    if (fftrec->fimag != NULL) xfree(fftrec->fimag);
	} else {
	    nspzrFft(NULL, NULL, fftrec->order, NSP_Free);
	    nspdRealFftl(NULL, fftrec->order, NSP_Free);
	    nspdCcsFftl(NULL, fftrec->order, NSP_Free);
    
	    if (fftrec->data != NULL) xfree(fftrec->data);
	}
	nspRedirectError(NULL);
	
	return;
    }
#endif
    
    return;
}

#if defined(USE_SPL)
int __STDCALL spErrorSPL(NSPStatus status, const char *func,
			 const char *context, const char *file, int line)
{
    fprintf(stderr, "SP Library error: %s, ", nspErrorStr(status));
    fprintf(stderr, "function %s, ", func ? func : "<unknown>");
    if (line > 0) fprintf(stderr, "line %d, ", line);
    if (file != NULL) fprintf(stderr, "file %s, ", file);
    if (context) fprintf(stderr, "context %s\n", context);
    nspSetErrStatus(status);
    exit(1);
}
#endif

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
static spBool _dvfftex(spFFTRec fftrec, DVector x, int inv, int real_flag, spBool public_call);

static spBool checkBluesteinRequired(long *fftl_specified)
{
    long order;
    long fftl;

    if (*fftl_specified > 0) {
        return SP_FALSE;
    }
    
    order = nextpow2(-(*fftl_specified));
    fftl = POW2(order);
    spDebug(20, "checkBluesteinRequired", "fftl_specified = %ld, order = %ld, fftl = %ld\n",
            *fftl_specified, order, fftl);
    
    if (fftl == -(*fftl_specified)) {
        *fftl_specified = order;
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

static void spUpdateBluesteinFFTOrder(spFFTRec fftrec, long fftl_specified)
{
    if (fftl_specified < 0) {
        fftl_specified *= -1;
    }
    fftrec->fftl_specified = fftl_specified;
    fftrec->order = nextpow2(2 * fftl_specified - 1);
    fftrec->fftl = POW2(fftrec->order);
    spDebug(20, "spUpdateBluesteinFFTOrder", "fftrec->fftl_specified = %ld, fftrec->order = %ld, fftrec->fftl = %ld\n",
            fftrec->fftl_specified, fftrec->order, fftrec->fftl);

    return;
}

static DVector xdvbsexparg(long fftl_specified)
{
    DVector exparg;
    
    /* bs_exparg = PI * n^2 / N */
    exparg = xdvinit(0.0, 1.0, (double)(fftl_specified - 1));
    dvscoper(exparg, "^", 2.0);
    dvscoper(exparg, "*", PI / (double)fftl_specified);

    return exparg;
}

static DVector xdvbscalcw(DVector exparg)
{
    DVector w;
    
    w = xdvrizeros(exparg->length);
    dvrcopyi(w, exparg);
    dvexp(w);

    return w;
}

static DVector xdvbscalcwcirc(DVector w, LVector revidx, long fftl, long batch)
{
    DVector wcirc;
    DVector w_rev;

    w_rev = xdvremap(w, revidx);

    wcirc = xdvrizeros(fftl * batch);
    dvadd(wcirc, 0, w, 0, w->length, SP_FALSE);
    dvadd(wcirc, fftl - w_rev->length, w_rev, 0, w_rev->length, SP_FALSE);

    xdvfree(w_rev);
    
    return wcirc;
}

static void spInitFFTBluestein(spFFTRec fftrec)
{
    long m;
    long fftl_specified;
    LVector revidx;

    fftl_specified = fftrec->fftl_specified;

    /* bs_exparg = PI * n^2 / N */
    fftrec->bs_exparg = xdvbsexparg(fftl_specified);
    /* bs_w = exp(j * PI * n^2 / N) */
    fftrec->bs_w = xdvbscalcw(fftrec->bs_exparg);
    //dvdump(fftrec->bs_w); exit(0);
    
    /* bs_wc = conj(bs_w) */
    fftrec->bs_wc = xdvconj(fftrec->bs_w);
    //dvdump(fftrec->bs_wc); exit(0);

    revidx = xlvinit(fftl_specified - 1, -1, 1);
    
    /* bs_W = fft(modified bs_w, fftl) */
    fftrec->bs_W = xdvbscalcwcirc(fftrec->bs_w, revidx, fftrec->fftl, fftrec->batch);
    //dvdump(fftrec->bs_W); exit(0);

    /* bs_W_for_inv = fft(modified bs_wc / fftl_specified, fftl) */
    fftrec->bs_W_for_inv = xdvbscalcwcirc(fftrec->bs_wc, revidx, fftrec->fftl, fftrec->batch);
    dvscoper(fftrec->bs_W_for_inv, "/", (double)fftl_specified);

    if (fftrec->batch >= 2) {
        fftrec->bs_w = xdvrealloc(fftrec->bs_w, fftrec->bs_W->length);
        fftrec->bs_wc = xdvrealloc(fftrec->bs_wc, fftrec->bs_W->length);
    }
    
    for (m = 1; m < fftrec->batch; m++) {
        dvadd(fftrec->bs_W, m * fftrec->fftl, fftrec->bs_W, 0, fftrec->fftl, SP_FALSE);
        dvadd(fftrec->bs_W_for_inv, m * fftrec->fftl, fftrec->bs_W_for_inv, 0, fftrec->fftl, SP_FALSE);
        
        dvadd(fftrec->bs_w, m * fftrec->fftl, fftrec->bs_w, 0, fftrec->bs_w->length, SP_FALSE);
        dvadd(fftrec->bs_wc, m * fftrec->fftl, fftrec->bs_wc, 0, fftrec->bs_wc->length, SP_FALSE);
    }
    _dvfftex(fftrec, fftrec->bs_W, 0, 0, SP_FALSE);
    _dvfftex(fftrec, fftrec->bs_W_for_inv, 0, 0, SP_FALSE);
    //dvdump(fftrec->bs_W); exit(0);
    
    fftrec->bs_buffer = xdvrizeros(fftrec->bs_W->length);

    xlvfree(revidx);

    return;
}

static void spFreeFFTBluestein(spFFTRec fftrec)
{
    if (fftrec->bs_exparg != NODATA) xdvfree(fftrec->bs_exparg);
    if (fftrec->bs_w != NODATA) xdvfree(fftrec->bs_w);
    if (fftrec->bs_wc != NODATA) xdvfree(fftrec->bs_wc);
    if (fftrec->bs_W != NODATA) xdvfree(fftrec->bs_W);
    if (fftrec->bs_W_for_inv != NODATA) xdvfree(fftrec->bs_W_for_inv);
    if (fftrec->bs_buffer != NODATA) xdvfree(fftrec->bs_buffer);

    fftrec->bs_exparg = NODATA;
    fftrec->bs_w = NODATA;
    fftrec->bs_wc = NODATA;
    fftrec->bs_W = NODATA;
    fftrec->bs_W_for_inv = NODATA;
    fftrec->bs_buffer = NODATA;
    
    return;
}

static void execFFTBluesteinCore(spFFTRec fftrec, int inv)
{
    if (inv) {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_w);
    } else {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_wc);
    }
    
    _dvfftex(fftrec, fftrec->bs_buffer, 0, 0, SP_FALSE);

    if (inv) {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_W_for_inv);
    } else {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_W);
    }

    _dvfftex(fftrec, fftrec->bs_buffer, 1, 0, SP_FALSE);

    if (inv) {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_w);
    } else {
        dvoper(fftrec->bs_buffer, "*", fftrec->bs_wc);
    }
    //dvdump(fftrec->bs_buffer); exit(0);

    return;
}

static void execFFTFBluestein(spFFTRec fftrec, float *real, float *imag, int inv, int real_flag)
{
    long n;
    long m;
    long offset;

    dvrizeros(fftrec->bs_buffer, fftrec->bs_buffer->length);
    for (m = 0; m < fftrec->batch; m++) {
        offset = m * fftrec->fftl;
        for (n = 0; n < fftrec->fftl_specified; n++) {
            fftrec->bs_buffer->data[offset + n] = (double)real[offset + n];
            if (!real_flag) {
                fftrec->bs_buffer->imag[offset + n] = (double)imag[offset + n];
            }
        }
    }
    if (real_flag && inv) {
        dvrffttofftex(fftrec, fftrec->bs_buffer);
    }

    execFFTBluesteinCore(fftrec, inv);
    
    if (real_flag) {
        dvffttorfftex(fftrec, fftrec->bs_buffer);
    }
    for (m = 0; m < fftrec->batch; m++) {
        offset = m * fftrec->fftl;
        for (n = 0; n < fftrec->fftl_specified; n++) {
            real[offset + n] = (float)fftrec->bs_buffer->data[offset + n];
            if (!real_flag) {
                imag[offset + n] = (float)fftrec->bs_buffer->imag[offset + n];
            }
        }
#if 1
        if (fftrec->batch >= 2) {
            for (; n < fftrec->fftl; n++) {
                real[offset + n] = 0.0;
                if (!real_flag) {
                    imag[offset + n] = 0.0;
                }
            }
        }
#endif
    }

    return;
}

static void execFFTBluestein(spFFTRec fftrec, double *real, double *imag, int inv, int real_flag)
{
    long n;
    long m;
    long offset;

    dvrizeros(fftrec->bs_buffer, fftrec->bs_buffer->length);
    for (m = 0; m < fftrec->batch; m++) {
        offset = m * fftrec->fftl;
        for (n = 0; n < fftrec->fftl_specified; n++) {
            fftrec->bs_buffer->data[offset + n] = real[offset + n];
            if (!real_flag) {
                fftrec->bs_buffer->imag[offset + n] = imag[offset + n];
            }
        }
    }
    if (real_flag && inv) {
        dvrffttofftex(fftrec, fftrec->bs_buffer);
    }

    execFFTBluesteinCore(fftrec, inv);
    
    if (real_flag) {
        dvffttorfftex(fftrec, fftrec->bs_buffer);
    }
    for (m = 0; m < fftrec->batch; m++) {
        offset = m * fftrec->fftl;
        for (n = 0; n < fftrec->fftl_specified; n++) {
            real[offset + n] = fftrec->bs_buffer->data[offset + n];
            if (!real_flag) {
                imag[offset + n] = fftrec->bs_buffer->imag[offset + n];
            }
        }
#if 1
        if (fftrec->batch >= 2) {
            for (; n < fftrec->fftl; n++) {
                real[offset + n] = 0.0;
                if (!real_flag) {
                    imag[offset + n] = 0.0;
                }
            }
        }
#endif
    }

    return;
}

static void _dvfftexBluestein(spFFTRec fftrec, DVector x, int inv, int real_flag)
{
    spDebug(80, "_dvfftexBluestein", "in: x->length = %ld\n", x->length);
    dvrizeros(fftrec->bs_buffer, fftrec->bs_buffer->length);
    dvcopy(fftrec->bs_buffer, x);
    if (real_flag && inv) {
        dvrffttofftex(fftrec, fftrec->bs_buffer);
    }

    execFFTBluesteinCore(fftrec, inv);
    spDebug(80, "_dvfftexBluestein", "after core: x->length = %ld\n", x->length);

    if (fftrec->batch >= 2 && x->length < fftrec->bs_buffer->length) {
        x = xdvrealloc(x, fftrec->bs_buffer->length);
    }

    if (real_flag) {
        dvffttorfftex(fftrec, fftrec->bs_buffer);
        dvcopy(x, fftrec->bs_buffer);
        if (x->imag != NULL) {
            dvizeros(x, x->length);
        }
    } else {
        if (x->imag == NULL) {
            dvizeros(x, x->length);
        }
        dvcopy(x, fftrec->bs_buffer);
        //dvdump(fftrec->bs_buffer); exit(0);
        //dvdump(x); exit(0);
    }
    spDebug(80, "_dvfftexBluestein", "done: x->length = %ld\n", x->length);

    return;
}
#endif

static spBool spInitFFTArch(spFFTRec fftrec)
{
    spDebug(20, "spInitFFTArch", "precision = %d, fftl = %ld\n",
	    fftrec->precision, fftrec->fftl);
    
#if defined(MACOSX)
    fftrec->setup = NULL;
    fftrec->scbuf.realp = NULL;
    fftrec->scbuf.imagp = NULL;
    
    fftrec->setupD = NULL;
    fftrec->scbufD.realp = NULL;
    fftrec->scbufD.imagp = NULL;
    
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
	    if ((fftrec->setupD = vDSP_create_fftsetupD(fftrec->order, FFT_RADIX2)) == NULL) {
		spWarning("Can't create FFTsetupD\n");
		return SP_FALSE;
	    }
    
	    fftrec->scbufD.realp = xalloc(fftrec->fftl, double);
	    fftrec->scbufD.imagp = xalloc(fftrec->fftl, double);
	} else {
	    if ((fftrec->setup = vDSP_create_fftsetup(fftrec->order, FFT_RADIX2)) == NULL) {
		spWarning("Can't create FFTsetup\n");
		return SP_FALSE;
	    }
    
	    fftrec->scbuf.realp = xalloc(fftrec->fftl, float);
	    fftrec->scbuf.imagp = xalloc(fftrec->fftl, float);
	}
        spDebug(20, "spInitFFTArch", "vDSP based FFT is used\n");
	return SP_TRUE;
    }
#elif defined(USE_SPL)
    {
	nspSetErrMode(NSP_ErrModeParent);
	nspRedirectError(spErrorSPL);
	
	if (fftrec->precision == SP_FFT_FLOAT_PRECISION) {
	    nspcrFft(NULL, NULL, fftrec->order, NSP_Init);
	    nspsRealFftl(NULL, fftrec->order, NSP_Init);
	    nspsCcsFftl(NULL, fftrec->order, NSP_Init);
    
	    fftrec->freal = xalloc(fftrec->fftl, float);
	    fftrec->fimag = xalloc(fftrec->fftl, float);
	} else {
	    nspzrFft(NULL, NULL, fftrec->order, NSP_Init);
	    nspdRealFftl(NULL, fftrec->order, NSP_Init);
	    nspdCcsFftl(NULL, fftrec->order, NSP_Init);
    
	    fftrec->data = xalloc(fftrec->fftl, double);
	}
	return SP_TRUE;
    }
#endif

    /* no internal buffer is used */
    fftrec->thread_safe = SP_TRUE;
    spDebug(20, "spInitFFTArch", "normal internal FFT is used\n");
    
    return SP_TRUE;
}

spThreadReturn batchedFFTThread(void *data)
{
    spFFTThreadRec *thread_rec = (spFFTThreadRec *)data;
    spFFTRec fftrec;

    spDebug(20, "batchedFFTThread", "in\n");
    
    fftrec = thread_rec->fftrec;

    while (fftrec->terminate_thread == SP_FALSE) {
	/*spDebug(-100, "batchedFFTThread", "before spWaitEvent\n");*/
	spWaitEvent(thread_rec->event_for_data);
	/*spDebug(-100, "batchedFFTThread", "after spWaitEvent: fft_type = %d\n", thread_rec->fft_type);*/

	if (thread_rec->data != NULL) {
	    if (thread_rec->fft_type == SP_FFT_TYPE_FLOAT_COMPLEX) {
		execFFTF(fftrec, thread_rec->data, thread_rec->imag, thread_rec->inv);    
	    } else if (thread_rec->fft_type == SP_FFT_TYPE_DOUBLE_COMPLEX) {
		execFFT(fftrec, thread_rec->data, thread_rec->imag, thread_rec->inv);    
	    } else if (thread_rec->fft_type == SP_FFT_TYPE_FLOAT_REAL) {
		execRealFFTF(fftrec, thread_rec->data, thread_rec->inv);
	    } else if (thread_rec->fft_type == SP_FFT_TYPE_DOUBLE_REAL) {
		execRealFFT(fftrec, thread_rec->data, thread_rec->inv);
	    }
	}

	spLockMutex(thread_rec->parent->mutex);
	spDebug(20, "batchedFFTThread", "original thread_rec->parent->num_computing = %ld\n", thread_rec->parent->num_computing);
	--thread_rec->parent->num_computing;
	if (thread_rec->parent->num_computing <= 0) {
	    spDebug(20, "batchedFFTThread", "spSetEvent for thread_rec->parent->event_for_proc\n");
	    spSetEvent(thread_rec->parent->event_for_proc);
	}
	spUnlockMutex(thread_rec->parent->mutex);
    }
    
    spDebug(20, "batchedFFTThread", "done\n");
    
    return SP_THREAD_RETURN_SUCCESS;
}

static spBool destroyBatchedFFTThread(spFFTRec fftrec)
{
    long k;

    if (fftrec->num_thread <= 0) return SP_FALSE;
    
    spDebug(20, "destroyBatchedFFTThread", "in\n");
    
    fftrec->terminate_thread = SP_TRUE;

    if (fftrec->event_for_proc != NULL) {
	spSetEvent(fftrec->event_for_proc);
    }
    
    for (k = 0; k < fftrec->num_thread; k++) {
	if (fftrec->thread_safe == SP_FALSE) {
	    fftrec->thread_recs[k].fftrec->terminate_thread = SP_TRUE;
	}

	fftrec->thread_recs[k].fft_type = SP_FFT_TYPE_UNKNOWN;
	fftrec->thread_recs[k].data = NULL;
	fftrec->thread_recs[k].imag = NULL;

	if (fftrec->thread_recs[k].event_for_data != NULL) {
	    spSetEvent(fftrec->thread_recs[k].event_for_data);
	}
	
	if (fftrec->thread_recs[k].thread != NULL) {
	    spWaitThread(fftrec->thread_recs[k].thread);
	    spDestroyThread(fftrec->thread_recs[k].thread);
	}
	
	if (fftrec->thread_recs[k].event_for_data != NULL) {
	    spDestroyEvent(fftrec->thread_recs[k].event_for_data);
	}

	if (fftrec->thread_safe == SP_FALSE) {
	    spFreeFFT(fftrec->thread_recs[k].fftrec);
	}
	fftrec->thread_recs[k].fftrec = NULL;
    }

    if (fftrec->event_for_proc != NULL) {
	spDestroyEvent(fftrec->event_for_proc);
    }
    if (fftrec->mutex != NULL) {
	spDestroyMutex(fftrec->mutex);
    }
    
    xfree(fftrec->thread_recs);
    
    fftrec->thread_recs = NULL;
    fftrec->terminate_thread = SP_FALSE;

    spDebug(20, "destroyBatchedFFTThread", "done\n");
    
    return SP_TRUE;
}

static long createBatchedFFTThread(spFFTRec fftrec)
{
    spDebug(20, "createBatchedFFTThread", "in: batch = %ld, support_batch = %d\n",
	    fftrec->batch, fftrec->support_batch);
    
    fftrec->num_thread = 0;
    fftrec->terminate_thread = SP_FALSE;
    
    if (fftrec->batch > 1 && fftrec->support_batch == SP_FALSE) {
	long k;

	fftrec->num_thread = fftrec->batch - 1;
	fftrec->thread_recs = xalloc(fftrec->num_thread, struct _spFFTThreadRec);
	memset(fftrec->thread_recs, 0, sizeof(struct _spFFTThreadRec) * fftrec->num_thread);

	if ((fftrec->mutex = spCreateMutex(NULL)) == NULL
	    || (fftrec->event_for_proc = spCreateEvent(SP_FALSE, SP_TRUE)) == NULL) {
	    fftrec->num_thread = 0;
	    return -1;
	}
	
	for (k = 0; k < fftrec->num_thread; k++) {
	    fftrec->thread_recs[k].parent = fftrec;
	    if (fftrec->thread_safe == SP_TRUE) {
		fftrec->thread_recs[k].fftrec = fftrec;
	    } else {
		fftrec->thread_recs[k].fftrec = spInitFFTByPlugin(fftrec->plugin, fftrec->order,
								  fftrec->precision);
	    }
	    
	    if ((fftrec->thread_recs[k].event_for_data = spCreateEvent(SP_FALSE, SP_FALSE)) == NULL
		|| (fftrec->thread_recs[k].thread = spCreateThread(0, 0, batchedFFTThread,
								    &fftrec->thread_recs[k])) == NULL) {
		spDebug(20, "createBatchedFFTThread", "thread error: k = %ld\n", k);
		fftrec->num_thread = k + 1;
		return -1;
	    }
	    fftrec->thread_recs[k].index = k;
	    fftrec->thread_recs[k].fft_type = SP_FFT_TYPE_UNKNOWN;
	    fftrec->thread_recs[k].data = NULL;
	    fftrec->thread_recs[k].imag = NULL;
	    fftrec->thread_recs[k].inv = 0;
	}
	spDebug(20, "createBatchedFFTThread", "thread started: num_thread = %ld\n", fftrec->num_thread);
    }

    return fftrec->num_thread;
}

spFFTRec spInitBatchedFFTByPlugin(spPlugin *plugin, long order, long batch, spFFTPrecision precision)
{
    spFFTRec fftrec;

    spDebug(20, "spInitBatchedFFTByPlugin", "in: order = %ld, batch = %ld\n", order, batch);
    
#ifndef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (order <= 0) {
        return NULL;
    }
#endif
    fftrec = xalloc(1, struct _spFFTRec);
    memset(fftrec, 0, sizeof(struct _spFFTRec));
    fftrec->batch = MAX(batch, 1);
    fftrec->precision = precision;
    fftrec->speed = spGetFFTSpeed(fftrec->precision);
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (checkBluesteinRequired(&order) == SP_TRUE) {
        spUpdateBluesteinFFTOrder(fftrec, -order);
    } else
#endif
    {
        fftrec->order = order;
        fftrec->fftl = POW2(fftrec->order);
        fftrec->fftl_specified = fftrec->fftl;
    }
    spDebug(20, "spInitBatchedFFTByPlugin", "fftl_specified = %ld, order = %ld, fftl = %ld\n",
            fftrec->fftl_specified, fftrec->order, fftrec->fftl);

    if (plugin == NULL && !strnone(sp_default_fft_plugin_name)) {
	plugin = spLoadFFTPlugin(sp_default_fft_plugin_name);
    }

    if (plugin != NULL) {
	fftrec->plugin = plugin;
	if (spInitFFT_Plugin(fftrec) == SP_FALSE) {
	    spDebug(20, "spInitBatchedFFTByPlugin", "spInitFFT_Plugin failed\n");
	    /* don't call spFreeFFT_Plugin */
	    /*spFreeFFT_Plugin(fftrec);*/
#if 0
	    spFreeFFTPlugin(fftrec->plugin);
#endif
	    xfree(fftrec);
	    return NULL;
	}
	fftrec->support_batch = spIsPluginCapable(fftrec->plugin, SP_FFT_PLUGIN_CAPS_SUPPORT_BATCH);
	spDebug(20, "spInitBatchedFFTByPlugin", "%s: support_batch = %d\n",
		spGetPluginId(fftrec->plugin), fftrec->support_batch);
    } else {
	if (spInitFFTArch(fftrec) == SP_FALSE) {
	    spFreeFFTArch(fftrec);
	    xfree(fftrec);
	    return NULL;
	}
    }

    if (createBatchedFFTThread(fftrec) < 0) {
	spFreeFFT(fftrec);
	return NULL;
    }

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (fftrec->fftl_specified != fftrec->fftl) {
        spInitFFTBluestein(fftrec);
    }
#endif
    
    spDebug(20, "spInitBatchedFFTByPlugin", "fftrec->precision = %d, fftrec->fftl = %ld\n",
	    fftrec->precision, fftrec->fftl);
    
    return fftrec;
}

spFFTRec spInitBatchedFFT(long order, long batch, spFFTPrecision precision)
{
    return spInitBatchedFFTByPlugin(NULL, order, batch, precision);
}

spFFTRec spInitFFTByPlugin(spPlugin *plugin, long order, spFFTPrecision precision)
{
    return spInitBatchedFFTByPlugin(plugin, order, 1, precision);
}

spFFTRec spInitFFT(long order, spFFTPrecision precision)
{
    return spInitBatchedFFTByPlugin(NULL, order, 1, precision);
}

spBool spFreeFFT(spFFTRec fftrec)
{
    if (fftrec == NULL) return SP_FALSE;
    
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    spFreeFFTBluestein(fftrec);
#endif
    
    destroyBatchedFFTThread(fftrec);
    
    if (fftrec->plugin != NULL) {
	spFreeFFT_Plugin(fftrec);
#if 0
	spFreeFFTPlugin(fftrec->plugin);
#endif
    } else {
	spFreeFFTArch(fftrec);
    }

    xfree(fftrec);
    
    return SP_TRUE;
}

spBool spResetFFTOrder(spFFTRec fftrec, long order)
{
    if (order == fftrec->order) {
        return SP_TRUE;
    }
    
    if (fftrec->plugin == NULL) {
	spFreeFFTArch(fftrec);
    }
    
#ifndef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (order <= 0) {
        return SP_FALSE;
    }
#else
    spFreeFFTBluestein(fftrec);
    if (checkBluesteinRequired(&order) == SP_TRUE) {
        spUpdateBluesteinFFTOrder(fftrec, -order);
    } else
#endif
    {
        fftrec->order = order;
        fftrec->fftl = POW2(fftrec->order);
        fftrec->fftl_specified = fftrec->fftl;
    }

    if (fftrec->plugin != NULL) {
	spResetFFTOrder_Plugin(fftrec);
    } else {
	spInitFFTArch(fftrec);
    }
    
#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (fftrec->fftl_specified != fftrec->fftl) {
        spInitFFTBluestein(fftrec);
    }
#endif
    
    return SP_TRUE;
}

long spGetFFTOrder(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;
    
    return fftrec->order;
}

long spGetFFTLength(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;
    
    return fftrec->fftl;
}

long spGetFFTLengthSpecified(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;
    
    return fftrec->fftl_specified;
}

long spGetFFTBatch(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;
    
    return fftrec->batch;
}

static spBool setFFTEventForData(spFFTRec fftrec, spFFTType fft_type,
				 void *real, void *imag, int inv)
{
    long k;
    spBool float_flag;
    double *reald = NULL;
    double *imagd = NULL;
    float *realf = NULL;
    float *imagf = NULL;
    
    if (fftrec->num_thread > 0) {
	spLockMutex(fftrec->mutex);
	fftrec->num_computing = fftrec->num_thread;
	spUnlockMutex(fftrec->mutex);
    
	spResetEvent(fftrec->event_for_proc);
	
	if (fft_type == SP_FFT_TYPE_FLOAT_COMPLEX
	    || fft_type == SP_FFT_TYPE_FLOAT_REAL) {
	    float_flag = SP_TRUE;
	    realf = real;
	    imagf = imag;
	} else {
	    float_flag = SP_FALSE;
	    reald = real;
	    imagd = imag;
	}
	
	for (k = 0; k < fftrec->num_thread; k++) {
	    fftrec->thread_recs[k].fft_type = fft_type;
	    if (float_flag == SP_TRUE) {
		fftrec->thread_recs[k].data = realf + (k + 1) * fftrec->fftl;
		if (imagf != NULL) {
		    fftrec->thread_recs[k].imag = imagf + (k + 1) * fftrec->fftl;
		} else {
		    fftrec->thread_recs[k].imag = NULL;
		}
	    } else {
		fftrec->thread_recs[k].data = reald + (k + 1) * fftrec->fftl;
		if (imagd != NULL) {
		    fftrec->thread_recs[k].imag = imagd + (k + 1) * fftrec->fftl;
		} else {
		    fftrec->thread_recs[k].imag = NULL;
		}
	    }
	    fftrec->thread_recs[k].inv = inv;
	    spSetEvent(fftrec->thread_recs[k].event_for_data);
	}

	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

static spBool waitFFTEventForProc(spFFTRec fftrec)
{
    long k;
    
    if (fftrec->num_thread > 0) {
	spDebug(100, "waitFFTEventForProc", "before spWaitEvent for fftrec->event_for_proc\n");
	spWaitEvent(fftrec->event_for_proc);
	spDebug(100, "waitFFTEventForProc", "after spWaitEvent for fftrec->event_for_proc\n");
	
	for (k = 0; k < fftrec->num_thread; k++) {
	    fftrec->thread_recs[k].fft_type = SP_FFT_TYPE_UNKNOWN;
	    fftrec->thread_recs[k].data = NULL;
	    fftrec->thread_recs[k].imag = NULL;
	    fftrec->thread_recs[k].inv = 0;
	}
	
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

static spBool execFFTF(spFFTRec fftrec, float *real, float *imag, int inv)
{
    spDebug(80, "spExecFFTF", "in\n");
    
    if (fftrec->plugin != NULL) {
	return spExecFFTF_Plugin(fftrec, real, imag, inv);
    }
    
#if defined(MACOSX)
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	long k;
    
	for (k = 0; k < fftrec->fftl; k++) {
	    fftrec->scbuf.realp[k] = real[k];
	    fftrec->scbuf.imagp[k] = imag[k];
	}
    
	vDSP_fft_zip(fftrec->setup, &fftrec->scbuf, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));

	if (inv) {
	    float factor;

	    factor = fftrec->fftl;
	    
	    for (k = 0; k < fftrec->fftl; k++) {
		real[k] = fftrec->scbuf.realp[k] / factor;
		imag[k] = fftrec->scbuf.imagp[k] / factor;
	    }
	} else {
	    for (k = 0; k < fftrec->fftl; k++) {
		real[k] = fftrec->scbuf.realp[k];
		imag[k] = fftrec->scbuf.imagp[k];
	    }
	}

	return SP_TRUE;
    }
#elif defined(USE_SPL)
    {
	nspcrFft(real, imag, fftrec->order, inv ? NSP_Inv : NSP_Forw);
	
	return SP_TRUE;
    }
#endif
    
    spfftf_normal(real, imag, fftrec->order, fftrec->fftl, inv);
    
    return SP_TRUE;
}

spBool spExecFFTF(spFFTRec fftrec, float *real, float *imag, int inv)
{
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (fftrec->fftl_specified != fftrec->fftl) {
        execFFTFBluestein(fftrec, real, imag, inv, 0);
        return SP_TRUE;
    }
#endif
    
    setFFTEventForData(fftrec, SP_FFT_TYPE_FLOAT_COMPLEX, real, imag, inv);

    ret = execFFTF(fftrec, real, imag, inv);

    waitFFTEventForProc(fftrec);
    
    return ret;
}

static spBool execFFT(spFFTRec fftrec, double *real, double *imag, int inv)
{
    spDebug(80, "spExecFFT", "in\n");
    
    if (fftrec->plugin != NULL) {
	return spExecFFT_Plugin(fftrec, real, imag, inv);
    }
    
#if defined(MACOSX)
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	long k;
    
	if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
	    for (k = 0; k < fftrec->fftl; k++) {
		fftrec->scbufD.realp[k] = real[k];
		fftrec->scbufD.imagp[k] = imag[k];
	    }
	    spDebug(80, "spExecFFT", "before vDSP_fft_zipD\n");
    
	    vDSP_fft_zipD(fftrec->setupD, &fftrec->scbufD, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));
	    spDebug(80, "spExecFFT", "after vDSP_fft_zipD\n");
	} else {
	    for (k = 0; k < fftrec->fftl; k++) {
		fftrec->scbuf.realp[k] = (float)real[k];
		fftrec->scbuf.imagp[k] = (float)imag[k];
	    }
    
	    vDSP_fft_zip(fftrec->setup, &fftrec->scbuf, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));
	}

	if (inv) {
	    double factor;

	    factor = fftrec->fftl;
	    spDebug(80, "spExecFFT", "factor = %f, fftl = %ld\n", factor, fftrec->fftl);
	    
	    if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
		for (k = 0; k < fftrec->fftl; k++) {
		    real[k] = fftrec->scbufD.realp[k] / factor;
		    imag[k] = fftrec->scbufD.imagp[k] / factor;
		}
	    } else {
		for (k = 0; k < fftrec->fftl; k++) {
		    real[k] = (double)fftrec->scbuf.realp[k] / factor;
		    imag[k] = (double)fftrec->scbuf.imagp[k] / factor;
		}
	    }
	} else {
	    if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
		for (k = 0; k < fftrec->fftl; k++) {
		    real[k] = fftrec->scbufD.realp[k];
		    imag[k] = fftrec->scbufD.imagp[k];
		}
	    } else {
		for (k = 0; k < fftrec->fftl; k++) {
		    real[k] = (double)fftrec->scbuf.realp[k];
		    imag[k] = (double)fftrec->scbuf.imagp[k];
		}
	    }
	}

	spDebug(80, "spExecFFT", "done\n");
	return SP_TRUE;
    }
#elif defined(USE_SPL)
    {
	if (fftrec->precision == SP_FFT_FLOAT_PRECISION) {
	    long k;

	    for (k = 0; k < fftrec->fftl; k++) {
		fftrec->freal[k] = (float)real[k];
		fftrec->fimag[k] = (float)imag[k];
	    }
	    
	    nspcrFft(fftrec->freal, fftrec->fimag, fftrec->order, inv ? NSP_Inv : NSP_Forw);
	    
	    for (k = 0; k < fftrec->fftl; k++) {
		real[k] = fftrec->freal[k];
		imag[k] = fftrec->fimag[k];
	    }
	} else {
	    nspzrFft(real, imag, fftrec->order, inv ? NSP_Inv : NSP_Forw);
	}
	
	return SP_TRUE;
    }
#endif
    
    spfft_normal(real, imag, fftrec->order, fftrec->fftl, inv);
    
    return SP_TRUE;
}

spBool spExecFFTInternal(spFFTRec fftrec, double *real, double *imag, int inv, spBool public_call)
{
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        execFFTBluestein(fftrec, real, imag, inv, 0);
        return SP_TRUE;
    }
#endif
    
    setFFTEventForData(fftrec, SP_FFT_TYPE_DOUBLE_COMPLEX, real, imag, inv);

    ret = execFFT(fftrec, real, imag, inv);

    waitFFTEventForProc(fftrec);
    
    return ret;
}

spBool spExecFFT(spFFTRec fftrec, double *real, double *imag, int inv)
{
    return spExecFFTInternal(fftrec, real, imag, inv, SP_TRUE);
}

static spBool execRealFFTF(spFFTRec fftrec, float *data, int inv)
{
    spDebug(80, "spExecRealFFTF", "in\n");
    
    if (fftrec->plugin != NULL) {
	return spExecRealFFTF_Plugin(fftrec, data, inv);
    }
    
#if defined(MACOSX)
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	long k;
	long fftl2;
	float factor;

	fftl2 = fftrec->fftl / 2;

	if (inv) {
	    factor = (float)fftrec->fftl;
	} else {
	    factor = 2.0;
	}
	
	for (k = 0; k < fftl2; k++) {
	    fftrec->scbuf.realp[k] = data[k*2];
	    fftrec->scbuf.imagp[k] = data[k*2+1];
	}
    
        spDebug(80, "execRealFFTF", "call vDSP_fft_zrip, fftl2 = %ld\n", fftl2);
	vDSP_fft_zrip(fftrec->setup, &fftrec->scbuf, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));

	for (k = 0; k < fftl2; k++) {
	    data[k*2] = fftrec->scbuf.realp[k] / factor;
	    data[k*2+1] = fftrec->scbuf.imagp[k] / factor;
	}
	
	return SP_TRUE;
    }
#elif defined(USE_SPL)
    {
	long k;
	long range;
	
	range = fftrec->fftl - 1;
	    
	if (fftrec->precision == SP_FFT_FLOAT_PRECISION) {
	    if (inv) {
		fftrec->freal[0] = data[0];
		fftrec->freal[range] = data[1];
		for (k = 1; k < range; k++) {
		    fftrec->freal[k] = data[k + 1];
		}
		
		spDebug(80, "spExecRealFFT", "call nspsCcsFftl ...: order = %ld\n", fftrec->order);
		nspsCcsFftl(fftrec->freal, fftrec->order, NSP_Inv | NSP_InRCPack);
		spDebug(80, "spExecRealFFT", "call nspsCcsFftl done\n");
		
		for (k = 0; k < fftrec->fftl; k++) {
		    data[k] = fftrec->freal[k];
		}
	    } else {
		for (k = 0; k < fftrec->fftl; k++) {
		    fftrec->freal[k] = data[k];
		}
		
		spDebug(80, "spExecRealFFT", "call nspsRealFftl ...: order = %ld\n", fftrec->order);
		nspsRealFftl(fftrec->freal, fftrec->order, NSP_Forw | NSP_OutRCPack);
		spDebug(80, "spExecRealFFT", "call nspsRealFftl done\n");

		data[0] = fftrec->freal[0];
		data[1] = fftrec->freal[range];
		for (k = 1; k < range; k++) {
		    data[k + 1] = fftrec->freal[k];
		}
	    }
	} else {
	    if (inv) {
		fftrec->data[0] = (double)data[0];
		fftrec->data[range] = (double)data[1];
		for (k = 1; k < range; k++) {
		    fftrec->data[k] = (double)data[k + 1];
		}
	
		nspdCcsFftl(fftrec->data, fftrec->order, NSP_Inv | NSP_InRCPack);
		
		for (k = 0; k < fftrec->fftl; k++) {
		    data[k] = (float)fftrec->data[k];
		}
	    } else {
		for (k = 0; k < fftrec->fftl; k++) {
		    fftrec->data[k] = (double)data[k];
		}
		
		nspdRealFftl(fftrec->data, fftrec->order, NSP_Forw | NSP_OutRCPack);
		
		data[0] = (float)fftrec->data[0];
		data[1] = (float)fftrec->data[fftrec->fftl - 1];
		for (k = 1; k < range; k++) {
		    data[k + 1] = (float)fftrec->data[k];
		}
	    }
	}
	
	return SP_TRUE;
    }
#endif
    
    sprfftf_normal(data, fftrec->order, fftrec->fftl, inv);
    
    return SP_TRUE;
}

spBool spExecRealFFTFInternal(spFFTRec fftrec, float *data, int inv, spBool public_call)
{
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        execFFTFBluestein(fftrec, data, NULL, inv, 1);
        return SP_TRUE;
    }
#endif
    
    setFFTEventForData(fftrec, SP_FFT_TYPE_FLOAT_REAL, data, NULL, inv);

    ret = execRealFFTF(fftrec, data, inv);

    waitFFTEventForProc(fftrec);
    
    return ret;
}

spBool spExecRealFFTF(spFFTRec fftrec, float *data, int inv)
{
    return spExecRealFFTFInternal(fftrec, data, inv, SP_TRUE);
}

static spBool execRealFFT(spFFTRec fftrec, double *data, int inv)
{
    spDebug(80, "execRealFFT", "in\n");
    
    if (fftrec->plugin != NULL) {
	return spExecRealFFT_Plugin(fftrec, data, inv);
    }
    
#if defined(MACOSX)
    if (fftrec->speed >= SP_FFT_SPEED_VERY_FAST) {
	long k;
	long fftl2;
	double factor;

	fftl2 = fftrec->fftl / 2;

	if (inv) {
	    factor = (double)fftrec->fftl;
	} else {
	    factor = 2.0;
	}
	
	if (fftrec->precision == SP_FFT_DOUBLE_PRECISION) {
	    for (k = 0; k < fftl2; k++) {
		fftrec->scbufD.realp[k] = data[k*2];
		fftrec->scbufD.imagp[k] = data[k*2+1];
	    }
    
            spDebug(80, "execRealFFT", "call vDSP_fft_zripD, fftl2 = %ld\n", fftl2);
	    vDSP_fft_zripD(fftrec->setupD, &fftrec->scbufD, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));

	    for (k = 0; k < fftl2; k++) {
		data[k*2] = fftrec->scbufD.realp[k] / factor;
		data[k*2+1] = fftrec->scbufD.imagp[k] / factor;
	    }
	} else {
	    for (k = 0; k < fftl2; k++) {
		fftrec->scbuf.realp[k] = (float)data[k*2];
		fftrec->scbuf.imagp[k] = (float)data[k*2+1];
	    }
    
            spDebug(80, "execRealFFT", "call vDSP_fft_zrip, fftl2 = %ld\n", fftl2);
	    vDSP_fft_zrip(fftrec->setup, &fftrec->scbuf, 1, fftrec->order, (inv ? FFT_INVERSE : FFT_FORWARD));

	    for (k = 0; k < fftl2; k++) {
		data[k*2] = (double)fftrec->scbuf.realp[k] / factor;
		data[k*2+1] = (double)fftrec->scbuf.imagp[k] / factor;
	    }
	}
	
	return SP_TRUE;
    }
#elif defined(USE_SPL)
    {
	if (fftrec->precision == SP_FFT_FLOAT_PRECISION) {
	    long k;
	    long range;

	    range = fftrec->fftl - 1;
	    
	    if (inv) {
		fftrec->freal[0] = (float)data[0];
		fftrec->freal[range] = (float)data[1];
		for (k = 1; k < range; k++) {
		    fftrec->freal[k] = (float)data[k + 1];
		}
		
		spDebug(80, "spExecRealFFT", "call nspsCcsFftl ...: order = %ld\n", fftrec->order);
		nspsCcsFftl(fftrec->freal, fftrec->order, NSP_Inv | NSP_InRCPack);
		spDebug(80, "spExecRealFFT", "call nspsCcsFftl done\n");
		
		for (k = 0; k < fftrec->fftl; k++) {
		    data[k] = fftrec->freal[k];
		}
	    } else {
		for (k = 0; k < fftrec->fftl; k++) {
		    fftrec->freal[k] = (float)data[k];
		}
		
		spDebug(80, "execRealFFT", "call nspsRealFftl ...: order = %ld\n", fftrec->order);
		nspsRealFftl(fftrec->freal, fftrec->order, NSP_Forw | NSP_OutRCPack);
		spDebug(80, "execRealFFT", "call nspsRealFftl done\n");

		data[0] = fftrec->freal[0];
		data[1] = fftrec->freal[range];
		for (k = 1; k < range; k++) {
		    data[k + 1] = fftrec->freal[k];
		}
	    }
	} else {
	    if (inv) {
		memcpy(&fftrec->data[1], &data[2], sizeof(double) * (fftrec->fftl - 2));
		fftrec->data[0] = data[0];
		fftrec->data[fftrec->fftl - 1] = data[1];
	
		nspdCcsFftl(fftrec->data, fftrec->order, NSP_Inv | NSP_InRCPack);
		memcpy(data, fftrec->data, sizeof(double) * fftrec->fftl);
	    } else {
		memcpy(fftrec->data, data, sizeof(double) * fftrec->fftl);
		nspdRealFftl(fftrec->data, fftrec->order, NSP_Forw | NSP_OutRCPack);
		memcpy(&data[2], &fftrec->data[1], sizeof(double) * (fftrec->fftl - 2));
		data[0] = fftrec->data[0];
		data[1] = fftrec->data[fftrec->fftl - 1];
	    }
	}
	
	return SP_TRUE;
    }
#endif
    
    sprfft_normal(data, fftrec->order, fftrec->fftl, inv);
    
    return SP_TRUE;
}

spBool spExecRealFFTInternal(spFFTRec fftrec, double *data, int inv, spBool public_call)
{
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        execFFTBluestein(fftrec, data, NULL, inv, 1);
        return SP_TRUE;
    }
#endif

    setFFTEventForData(fftrec, SP_FFT_TYPE_DOUBLE_REAL, data, NULL, inv);

    ret = execRealFFT(fftrec, data, inv);

    waitFFTEventForProc(fftrec);
    
    return ret;
}

spBool spExecRealFFT(spFFTRec fftrec, double *data, int inv)
{
    return spExecRealFFTInternal(fftrec, data, inv, SP_TRUE);
}

spBool spExecFFTPowerFInternal(spFFTRec fftrec, float *data, float exponent, spBool public_call)
{
    long k;
    spBool real_fft_executed;
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        execFFTFBluestein(fftrec, data, NULL, 0, 1);
        ret = SP_TRUE;
        real_fft_executed = SP_TRUE;
    } else
#endif
    {
        setFFTEventForData(fftrec, SP_FFT_TYPE_FLOAT_REAL, data, NULL, 0);

        if (fftrec->plugin != NULL && SpFFTRecPluginRec(fftrec)->exec_fft_powerf != NULL) {
            ret = spExecFFTPowerF_Plugin(fftrec, data, exponent);
            real_fft_executed = SP_FALSE;
        } else {
            ret = execRealFFTF(fftrec, data, 0);
            real_fft_executed = SP_TRUE;
        }

        waitFFTEventForProc(fftrec);
    }
    
    if (real_fft_executed == SP_TRUE) {
	/*for (k = 0; k <= fftrec->num_thread; k++) {*/
	for (k = 0; k < fftrec->batch; k++) {
	    rffttopowerf(data + k * fftrec->fftl, exponent, fftrec->fftl);
	}
    }
    
    return ret;
}

spBool spExecFFTPowerF(spFFTRec fftrec, float *data, float exponent)
{
    return spExecFFTPowerFInternal(fftrec, data, exponent, SP_TRUE);
}

spBool spExecFFTPowerInternal(spFFTRec fftrec, double *data, double exponent, spBool public_call)
{
    long k;
    spBool real_fft_executed;
    spBool ret;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        execFFTBluestein(fftrec, data, NULL, 0, 1);
        ret = SP_TRUE;
        real_fft_executed = SP_TRUE;
    } else
#endif
    {
        setFFTEventForData(fftrec, SP_FFT_TYPE_DOUBLE_REAL, data, NULL, 0);

        if (fftrec->plugin != NULL && SpFFTRecPluginRec(fftrec)->exec_fft_power != NULL) {
            ret = spExecFFTPower_Plugin(fftrec, data, exponent);
            real_fft_executed = SP_FALSE;
        } else {
            ret = execRealFFT(fftrec, data, 0);
            real_fft_executed = SP_TRUE;
        }

        waitFFTEventForProc(fftrec);
    }
    
    if (real_fft_executed == SP_TRUE) {
	/*for (k = 0; k <= fftrec->num_thread; k++) {*/
	for (k = 0; k < fftrec->batch; k++) {
	    rffttopower(data + k * fftrec->fftl, exponent, fftrec->fftl);
	}
    }
    
    return ret;
}

spBool spExecFFTPower(spFFTRec fftrec, double *data, double exponent)
{
    return spExecFFTPowerInternal(fftrec, data, exponent, SP_TRUE);
}

spBool spTurnFFTSpectrumF(spFFTRec fftrec, float *real, float *imag)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    fftturnf(real, imag, fftrec->fftl);

    if (fftrec->batch >= 2) {
	long m;
	
	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		fftturnf(real + m * fftrec->fftl, NULL, fftrec->fftl);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		fftturnf(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
	    }
	}
    }

    return SP_TRUE;
}

spBool spTurnFFTSpectrum(spFFTRec fftrec, double *real, double *imag)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    fftturn(real, imag, fftrec->fftl);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		fftturn(real + m * fftrec->fftl, NULL, fftrec->fftl);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		fftturn(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
	    }
	}
    }

    return SP_TRUE;
}

spBool spExecFFTShiftF(spFFTRec fftrec, float *real, float *imag)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    fftshiftf(real, imag, fftrec->fftl);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		fftshiftf(real + m * fftrec->fftl, NULL, fftrec->fftl);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		fftshiftf(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
	    }
	}
    }

    return SP_TRUE;
}

spBool spExecFFTShift(spFFTRec fftrec, double *real, double *imag)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    fftshift(real, imag, fftrec->fftl);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		fftshift(real + m * fftrec->fftl, NULL, fftrec->fftl);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		fftshift(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
	    }
	}
    }

    return SP_TRUE;
}

#if 1
spBool spExecDataShiftF(spFFTRec fftrec, float *real, float *imag, long shift)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    ridatashiftf(real, imag, fftrec->fftl, shift);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		ridatashiftf(real + m * fftrec->fftl, NULL, fftrec->fftl, shift);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		ridatashiftf(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl, shift);
	    }
	}
    }

    return SP_TRUE;
}

spBool spExecDataShift(spFFTRec fftrec, double *real, double *imag, long shift)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    ridatashift(real, imag, fftrec->fftl, shift);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		ridatashift(real + m * fftrec->fftl, NULL, fftrec->fftl, shift);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		ridatashift(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl, shift);
	    }
	}
    }

    return SP_TRUE;
}

spBool spExecCircShiftF(spFFTRec fftrec, float *real, float *imag, long shift)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    ricircshiftf(real, imag, fftrec->fftl, shift);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		ricircshiftf(real + m * fftrec->fftl, NULL, fftrec->fftl, shift);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		ricircshiftf(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl, shift);
	    }
	}
    }

    return SP_TRUE;
}

spBool spExecCircShift(spFFTRec fftrec, double *real, double *imag, long shift)
{
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    ricircshift(real, imag, fftrec->fftl, shift);

    if (fftrec->batch >= 2) {
	long m;

	if (imag == NULL) {
	    for (m = 1; m < fftrec->batch; m++) {
		ricircshift(real + m * fftrec->fftl, NULL, fftrec->fftl, shift);
	    }
	} else {
	    for (m = 1; m < fftrec->batch; m++) {
		ricircshift(real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl, shift);
	    }
	}
    }

    return SP_TRUE;
}
#endif

spBool spGetSpectrumForRealFFTF(spFFTRec fftrec, float *real, float *imag, float *rspec)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	ffttorfftf(real + m * fftrec->fftl, imag + m * fftrec->fftl,
		   rspec + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

spBool spGetSpectrumForRealFFT(spFFTRec fftrec, double *real, double *imag, double *rspec)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	ffttorfft(real + m * fftrec->fftl, imag + m * fftrec->fftl,
		  rspec + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

spBool spGetComplexFromRealFFTF(spFFTRec fftrec, float *rspec, float *real, float *imag)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	rffttofftf(rspec + m * fftrec->fftl,
		   real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

spBool spGetComplexFromRealFFT(spFFTRec fftrec, double *rspec, double *real, double *imag)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	rffttofft(rspec + m * fftrec->fftl,
		  real + m * fftrec->fftl, imag + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

spBool spMultiplySpectrumOfRealFFTF(spFFTRec fftrec, float *rspec1, float *rspec2)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	multrfftf(rspec1 + m * fftrec->fftl, rspec2 + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

spBool spMultiplySpectrumOfRealFFT(spFFTRec fftrec, double *rspec1, double *rspec2)
{
    long m;
    
    if (fftrec == NULL || fftrec->fftl <= 0) return SP_FALSE;
	
    for (m = 0; m < fftrec->batch; m++) {
	multrfft(rspec1 + m * fftrec->fftl, rspec2 + m * fftrec->fftl, fftrec->fftl);
    }

    return SP_TRUE;
}

void fvfft(FVector x)
{
    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }

    fftf(x->data, x->imag, x->length, 0);

    return;
}

void dvfft(DVector x)
{
    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }

    fft(x->data, x->imag, x->length, 0);

    return;
}

void fvifft(FVector x)
{
    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }

    fftf(x->data, x->imag, x->length, 1);

    return;
}

void dvifft(DVector x)
{
    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }

    fft(x->data, x->imag, x->length, 1);

    return;
}

FVector xfvfft(FVector x, long length)
{
    long fftl;
    FVector y;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    fftl = length;
#else
    fftl = POW2(nextpow2(length));
#endif

    y = xfvrizeros(fftl);
    fvcopy(y, x);

    fftf(y->data, y->imag, fftl, 0);

    return y;
}

DVector xdvfft(DVector x, long length)
{
    long fftl;
    DVector y;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    fftl = length;
#else
    fftl = POW2(nextpow2(length));
#endif

    y = xdvrizeros(fftl);
    dvcopy(y, x);

    fft(y->data, y->imag, fftl, 0);

    return y;
}

FVector xfvifft(FVector x, long length)
{
    long fftl;
    FVector y;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    fftl = length;
#else
    fftl = POW2(nextpow2(length));
#endif

    y = xfvrizeros(fftl);
    fvcopy(y, x);

    fftf(y->data, y->imag, fftl, 1);

    return y;
}

DVector xdvifft(DVector x, long length)
{
    long fftl;
    DVector y;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    fftl = length;
#else
    fftl = POW2(nextpow2(length));
#endif

    y = xdvrizeros(fftl);
    dvcopy(y, x);

    fft(y->data, y->imag, fftl, 1);

    return y;
}

void svdatashift(SVector x, long shift)
{
    ridatashifts(x->data, x->imag, x->length, shift);

    return;
}

void lvdatashift(LVector x, long shift)
{
    ridatashiftl(x->data, x->imag, x->length, shift);

    return;
}

void fvdatashift(FVector x, long shift)
{
    ridatashiftf(x->data, x->imag, x->length, shift);

    return;
}

void dvdatashift(DVector x, long shift)
{
    ridatashift(x->data, x->imag, x->length, shift);

    return;
}

void fvdatashiftbatched(FVector x, long length, long stride, long batch, long shift)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        ridatashiftf(x->data + m * stride, x->imag + m * stride, length, shift);
    }

    return;
}

void dvdatashiftbatched(DVector x, long length, long stride, long batch, long shift)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        ridatashift(x->data + m * stride, x->imag + m * stride, length, shift);
    }

    return;
}

void svcircshift(SVector x, long shift)
{
    ricircshifts(x->data, x->imag, x->length, shift);

    return;
}

void lvcircshift(LVector x, long shift)
{
    ricircshiftl(x->data, x->imag, x->length, shift);

    return;
}

void fvcircshift(FVector x, long shift)
{
    ricircshiftf(x->data, x->imag, x->length, shift);

    return;
}

void dvcircshift(DVector x, long shift)
{
    ricircshift(x->data, x->imag, x->length, shift);

    return;
}

void fvcircshiftbatched(FVector x, long length, long stride, long batch, long shift)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        ricircshiftf(x->data + m * stride, x->imag + m * stride, length, shift);
    }

    return;
}

void dvcircshiftbatched(DVector x, long length, long stride, long batch, long shift)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        ricircshift(x->data + m * stride, x->imag + m * stride, length, shift);
    }

    return;
}

void fvfftshift(FVector x)
{
    fftshiftf(x->data, x->imag, x->length);

    return;
}

void dvfftshift(DVector x)
{
    fftshift(x->data, x->imag, x->length);

    return;
}

void fvfftshiftbatched(FVector x, long length, long stride, long batch)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        fftshiftf(x->data + m * stride, x->imag + m * stride, length);
    }

    return;
}

void dvfftshiftbatched(DVector x, long length, long stride, long batch)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        fftshift(x->data + m * stride, x->imag + m * stride, length);
    }

    return;
}

void fvifftshift(FVector x)
{
    ifftshiftf(x->data, x->imag, x->length);

    return;
}

void dvifftshift(DVector x)
{
    ifftshift(x->data, x->imag, x->length);

    return;
}

void dvifftshiftbatched(DVector x, long length, long stride, long batch)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        ifftshift(x->data + m * stride, x->imag + m * stride, length);
    }

    return;
}

FVector xfvfftshift(FVector x, long length)
{
    FVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(length);
    } else {
	y = xfvzeros(length);
    }
    fvcopy(y, x);
    
    fftshiftf(y->data, y->imag, y->length);
    
    return y;
}

DVector xdvfftshift(DVector x, long length)
{
    DVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(length);
    } else {
	y = xdvzeros(length);
    }
    dvcopy(y, x);
    
    fftshift(y->data, y->imag, y->length);
    
    return y;
}

FVector xfvfftshiftbatched(FVector x, long length, long stride, long batch)
{
    long m;
    FVector y;

    batch = MAX(batch, 1);
    
    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(stride * batch);
    } else {
	y = xfvzeros(stride * batch);
    }
    fvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        fftshiftf(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

DVector xdvfftshiftbatched(DVector x, long length, long stride, long batch)
{
    long m;
    DVector y;

    batch = MAX(batch, 1);
    
    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(stride * batch);
    } else {
	y = xdvzeros(stride * batch);
    }
    dvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        fftshift(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

FVector xfvifftshift(FVector x, long length)
{
    FVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(length);
    } else {
	y = xfvzeros(length);
    }
    fvcopy(y, x);
    
    ifftshiftf(y->data, y->imag, y->length);
    
    return y;
}

DVector xdvifftshift(DVector x, long length)
{
    DVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(length);
    } else {
	y = xdvzeros(length);
    }
    dvcopy(y, x);
    
    ifftshift(y->data, y->imag, y->length);
    
    return y;
}

FVector xfvifftshiftbatched(FVector x, long length, long stride, long batch)
{
    long m;
    FVector y;

    batch = MAX(batch, 1);
    
    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(stride * batch);
    } else {
	y = xfvzeros(stride * batch);
    }
    fvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        ifftshiftf(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

DVector xdvifftshiftbatched(DVector x, long length, long stride, long batch)
{
    long m;
    DVector y;

    batch = MAX(batch, 1);
    
    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(stride * batch);
    } else {
	y = xdvzeros(stride * batch);
    }
    dvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        ifftshift(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

void fvfftturn(FVector x)
{
    fftturnf(x->data, x->imag, x->length);

    return;
}

void dvfftturn(DVector x)
{
    fftturn(x->data, x->imag, x->length);

    return;
}

void fvfftturnbatched(FVector x, long length, long stride, long batch)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        fftturnf(x->data + m * stride, x->imag + m * stride, length);
    }

    return;
}

void dvfftturnbatched(DVector x, long length, long stride, long batch)
{
    long m;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    for (m = 0; m < batch; m++) {
        fftturn(x->data + m * stride, x->imag + m * stride, length);
    }

    return;
}

FVector xfvfftturn(FVector x, long length)
{
    FVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(length);
    } else {
	y = xfvzeros(length);
    }
    fvcopy(y, x);
    
    fftturnf(y->data, y->imag, y->length);
    
    return y;
}

DVector xdvfftturn(DVector x, long length)
{
    DVector y;

    if (length <= 0) {
        length = x->length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(length);
    } else {
	y = xdvzeros(length);
    }
    dvcopy(y, x);
    
    fftturn(y->data, y->imag, y->length);
    
    return y;
}

FVector xfvfftturnbatched(FVector x, long length, long stride, long batch)
{
    long m;
    FVector y;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xfvrizeros(stride * batch);
    } else {
	y = xfvzeros(stride * batch);
    }
    fvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        fftturnf(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

DVector xdvfftturnbatched(DVector x, long length, long stride, long batch)
{
    long m;
    DVector y;

    batch = MAX(batch, 1);

    if (length <= 0) {
        if (stride > 0) {
            length = stride;
        } else {
            length = x->length / batch;
        }
    }
    if (stride <= 0) {
        stride = length;
    }

    if (x->imag != NULL) {
	y = xdvrizeros(stride * batch);
    } else {
	y = xdvzeros(stride * batch);
    }
    dvcopy(y, x);
    
    for (m = 0; m < batch; m++) {
        fftturn(y->data + m * stride, y->imag + m * stride, length);
    }
    
    return y;
}

/*
 *	convolution using fft
 */
FVector xfvfftconv(FVector a, FVector b, long fftl)
{
    FVector x;
    FVector ca, cb;

    /* fft of a */
    ca = xfvfft(a, fftl);
    
    /* fft of b */
    cb = xfvfft(b, fftl);

    /* convolution */
    fvoper(ca, "*", cb);

    /* ifft */
    x = xfvifft(ca, fftl);
    fvreal(x);

    /* memory free */
    xfvfree(ca);
    xfvfree(cb);

    return x;
}

DVector xdvfftconv(DVector a, DVector b, long fftl)
{
    DVector x;
    DVector ca, cb;

    /* fft of a */
    ca = xdvfft(a, fftl);
    
    /* fft of b */
    cb = xdvfft(b, fftl);

    /* convolution */
    dvoper(ca, "*", cb);

    /* ifft */
    x = xdvifft(ca, fftl);
    dvreal(x);

    /* memory free */
    xdvfree(ca);
    xdvfree(cb);

    return x;
}

FVector xfvfftpower(FVector x, long fftl)
{
    FVector y;

    /* fft */
    y = xfvfft(x, fftl);

    /* square of complex */
    fvsquare(y);

    return y;
}

DVector xdvfftpower(DVector x, long fftl)
{
    DVector y;

    /* fft */
    y = xdvfft(x, fftl);

    /* square of complex */
    dvsquare(y);

    return y;
}

FVector xfvfftabs(FVector x, long fftl)
{
    FVector y;

    /* fft */
    y = xfvfft(x, fftl);

    /* abs of complex */
    fvabs(y);

    return y;
}

DVector xdvfftabs(DVector x, long fftl)
{
    DVector y;

    /* fft */
    y = xdvfft(x, fftl);

    /* abs of complex */
    dvabs(y);

    return y;
}

FVector xfvfftangle(FVector x, long fftl)
{
    FVector y;

    /* fft */
    y = xfvfft(x, fftl);

    /* phase angle */
    fvangle(y);

    return y;
}

DVector xdvfftangle(DVector x, long fftl)
{
    DVector y;

    /* fft */
    y = xdvfft(x, fftl);

    /* phase angle */
    dvangle(y);

    return y;
}

FVector xfvfftgrpdly(FVector x, long fftl)
{
    long k;
    float value;
    FVector fx;
    FVector dfx;
    FVector gd;

    /* fft of input signal */
    fx = xfvfft(x, fftl);

    /* calculate frequency diff spectrum */
    dfx = xfvrizeros(fx->length);
    for (k = 0; k < x->length; k++) {
	if (k >= dfx->length) break;
	
	dfx->imag[k] = -(float)k * x->data[k];
    }
    fvfft(dfx);

    /* calculate group delay */
    gd = xfvalloc(fx->length);
    for (k = 0; k < fx->length; k++) {
	value = SQUARE(fx->data[k]) + SQUARE(fx->imag[k]);
	if (value == 0.0f) {
	    gd->data[k] = 0.0f;
	} else {
	    gd->data[k] = -(dfx->imag[k] * fx->data[k] -
			    fx->imag[k] * dfx->data[k]) / value;
	}
    }

    /* memory free */
    xfvfree(fx);
    xfvfree(dfx);

    return gd;
}

DVector xdvfftgrpdly(DVector x, long fftl)
{
    long k;
    double value;
    DVector fx;
    DVector dfx;
    DVector gd;

    /* fft of input signal */
    fx = xdvfft(x, fftl);

    /* calculate frequency diff spectrum */
    dfx = xdvrizeros(fx->length);
    for (k = 0; k < x->length; k++) {
	if (k >= dfx->length) break;
	
	dfx->imag[k] = -(double)k * x->data[k];
    }
    dvfft(dfx);

    /* calculate group delay */
    gd = xdvalloc(fx->length);
    for (k = 0; k < fx->length; k++) {
	value = SQUARE(fx->data[k]) + SQUARE(fx->imag[k]);
	if (value == 0.0) {
	    gd->data[k] = 0.0;
	} else {
	    gd->data[k] = -(dfx->imag[k] * fx->data[k] -
			    fx->imag[k] * dfx->data[k]) / value;
	}
    }

    /* memory free */
    xdvfree(fx);
    xdvfree(dfx);

    return gd;
}

FVector xfvgdtophase(FVector gd, long fftl, int reverse)
{
    long k;
    float rv;
    float value;
    float twopi;
    FVector phs;

    twopi = (float)(2.0 * PI);
    
    /* integrate group delay */
    phs = xfvscoper(gd, "*", -twopi / (float)(fftl));
    
    /* if necessary, reverse signal */
    if (reverse == 1) {
	phs->data[0] = (float)PI;
    } else {
	phs->data[0] = 0.0f;
    }
    fvcumsum(phs);

    rv = phs->data[fftl-1] + phs->data[1] - 2.0f * phs->data[0];
    value = -rv + twopi * (float)round(rv / twopi);
    
    /* make phase continuous */
    for (k = 0; k < fftl; k++) {
	phs->data[k] = phs->data[k] + (float)k * value / (float)fftl;
    }

    return phs;
}

DVector xdvgdtophase(DVector gd, long fftl, int reverse)
{
    long k;
    double rv;
    double value;
    double twopi;
    DVector phs;

    twopi = 2.0 * PI;

    /* integrate group delay */
    phs = xdvscoper(gd, "*", -twopi / (double)(fftl));
    
    /* if necessary, reverse signal */
    if (reverse == 1) {
	phs->data[0] = PI;
    } else {
	phs->data[0] = 0.0;
    }
    dvcumsum(phs);

    rv = phs->data[fftl-1] + phs->data[1] - 2.0 * phs->data[0];
    value = -rv + twopi * round(rv / twopi);
    
    /* make phase continuous */
    for (k = 0; k < fftl; k++) {
	phs->data[k] = phs->data[k] + (double)k * value / (double)fftl;
    }

    return phs;
}

FVector xfvzerophase(FVector x, long fftl)
{
    FVector y;

    /* fft */
    y = xfvfft(x, fftl);

    /* abs of complex */
    fvabs(y);

    /* ifft */
    fvifft(y);

    /* fft shift */
    fvfftshift(y);

    return y;
}

DVector xdvzerophase(DVector x, long fftl)
{
    DVector y;

    /* fft */
    y = xdvfft(x, fftl);

    /* abs of complex */
    dvabs(y);

    /* ifft */
    dvifft(y);

    /* fft shift */
    dvfftshift(y);

    return y;
}

FVector xfvcspec(FVector mag, FVector phs)
{
    FVector spc;
    FVector phe;

    if (mag == NODATA && phs == NODATA) {
	return NODATA;
    } else if (phs == NODATA) {
	phe = xfvabs(mag);
    } else {
	/* exponential of phase */
	phe = xfvcplx(NODATA, phs);
	fvexp(phe);

	if (mag != NODATA) {
	    /* multiply phase */
	    spc = xfvabs(mag);
	    fvoper(phe, "*", spc);

	    /* memory free */
	    xfvfree(spc);
	}
    }

    return phe;
}

DVector xdvcspec(DVector mag, DVector phs)
{
    DVector spc;
    DVector phe;

    if (mag == NODATA && phs == NODATA) {
	return NODATA;
    } else if (phs == NODATA) {
	phe = xdvabs(mag);
    } else {
	/* exponential of phase */
	phe = xdvcplx(NODATA, phs);
	dvexp(phe);

	if (mag != NODATA) {
	    /* multiply phase */
	    spc = xdvabs(mag);
	    dvoper(phe, "*", spc);

	    /* memory free */
	    xdvfree(spc);
	}
    }

    return phe;
}

void dvspectocep(DVector x)
{
    long k;

    dvabs(x);
#if 0
    for (k = 0; k < x->length; k++) {
	if (x->data[k] > 0.0) {
	    x->data[k] = log(x->data[k]);
	} else {
	    x->data[k] = log(SP_TINY_NUMBER);
	}
    }
#else
    dvlog(x);
#endif

    dvifft(x);
    dvreal(x);

    return;
}

void dvceptospec(DVector x)
{
    /* convert cepstrum to spectrum */
    dvfft(x);
    dvexp(x);

    return;
}

DVector xdvrcep(DVector x, long fftl)
{
    DVector cep;

    cep = xdvfftabs(x, fftl);
    dvspectocep(cep);

    return cep;
}

void dvrcep(DVector x)
{
    rfftcep(x->data, NULL, x->length);
    
    return;
}

void dvceptompc(DVector cep)
{
    ceptompc(cep->data, cep->length);
    return;
}

DVector xdvmpcep(DVector x, long fftl)
{
    DVector cep;

    cep = xdvrcep(x, fftl);
    dvceptompc(cep);

    return cep;
}

void dvmpcep(DVector x)
{
    dvrcep(x);
    dvceptompc(x);
    
    return;
}

/*
 *	liftering function
 *	If lif < 0, high quefrency components are extracted.
 */
void dvlif(DVector cep, long fftl, long lif)
{
    long k;
    long lif2;

    if (lif >= 0) {
	lif2 = fftl - lif;
	for (k = 0; k < cep->length; k++) {
	    if (k > lif && k < lif2) {
		cep->data[k] = 0.0;
		if (cep->imag != NULL) {
		    cep->imag[k] = 0.0;
		}
	    }
	}
    } else {
	lif *= -1;
	lif2 = fftl - lif;
	for (k = 0; k < cep->length; k++) {
	    if (k <= lif || k >= lif2) {
		cep->data[k] = 0.0;
		if (cep->imag != NULL) {
		    cep->imag[k] = 0.0;
		}
	    }
	}
    }

    return;
}

#if 1
DVector xdvmelwarp(DVector cep, long order, long mel_order, double alpha)
{
    long i, k;
    double old_value, new_value;
    DVector wcep;

    wcep = xdvzeros(mel_order + 1);
    
    for (i = order; i >= 0; i--) {
	new_value = cep->data[i] + alpha * wcep->data[0];
	old_value = wcep->data[0];
	wcep->data[0] = new_value;
	
	new_value = (1.0 - SQUARE(alpha)) * old_value + alpha * wcep->data[1];
	old_value = wcep->data[1];
	wcep->data[1] = new_value;
	
	for (k = 2; k <= mel_order; k++) {
	    new_value = old_value + alpha * (wcep->data[k] - wcep->data[k - 1]);
	    old_value = wcep->data[k];
	    wcep->data[k] = new_value;
	}
    }
    
    return wcep;
}
#endif

DVector xdvtsp(double amp, long m, long n, int inverse)
{
    DVector tsp;
    spFFTRec fftrec;

#if 0
    tsp = xdvftsp(amp, m, n, inverse);
    fftrec = spInitFFT(nextpow2(tsp->length), SP_FFT_DOUBLE_PRECISION);
    spExecFFT(fftrec, tsp->data, tsp->imag, 1);
    xdvreal(tsp);
    spFreeFFT(fftrec);
#else
    tsp = xdvrftsp(amp, m, n, inverse);
    fftrec = spInitFFT(nextpow2(tsp->length), SP_FFT_DOUBLE_PRECISION);
    spExecRealFFT(fftrec, tsp->data, 1);
    spFreeFFT(fftrec);
#endif

    return tsp;
}

DVector xdvftsp(double amp, long m, long n, int inverse)
{
    long k;
    long fftl, hfftl;
    double a0, theta;
    DVector ftsp;

    fftl = POW2(nextpow2(n));
    hfftl = fftl / 2;
    a0 = 4.0 * PI * (double)m / ((double)fftl * (double)fftl);
    if (inverse) a0 *= -1.0;

    ftsp = xdvrialloc(fftl);

    ftsp->data[0] = amp;
    ftsp->imag[0] = 0.0;
    for (k = 1; k <= hfftl; k++) {
	theta = a0 * (double)k * (double)k;
	ftsp->data[k] = amp * cos(theta);
	ftsp->imag[k] = amp * sin(theta);
	ftsp->data[fftl - k] = ftsp->data[k];
	ftsp->imag[fftl - k] = -ftsp->imag[k];
    }

    return ftsp;
}

DVector xdvrftsp(double amp, long m, long n, int inverse)
{
    long k;
    long fftl, hfftl;
    double a0, theta;
    DVector rftsp;

    fftl = POW2(nextpow2(n));
    hfftl = fftl / 2;
    a0 = 4.0 * PI * (double)m / ((double)fftl * (double)fftl);
    if (inverse) a0 *= -1.0;

    rftsp = xdvalloc(fftl);

    rftsp->data[0] = amp;
    for (k = 1; k < hfftl; k++) {
	theta = a0 * (double)k * (double)k;
	rftsp->data[k * 2] = amp * cos(theta);
	rftsp->data[k * 2 + 1] = amp * sin(theta);
    }
    theta = a0 * (double)k * (double)k;
    rftsp->data[1] = amp * cos(theta);

    return rftsp;
}

DVector xdvgtsp(double amp, double alpha, long n, int inverse)
{
    long m;

    m = (long)round((double)n / (2.0 * alpha));

    return xdvtsp(amp, m, n, inverse);
}

DVector xdvfgtsp(double amp, double alpha, long n, int inverse)
{
    long m;

    m = (long)round((double)n / (2.0 * alpha));

    return xdvftsp(amp, m, n, inverse);
}

DVector xdvrfgtsp(double amp, double alpha, long n, int inverse)
{
    long m;

    m = (long)round((double)n / (2.0 * alpha));

    return xdvrftsp(amp, m, n, inverse);
}

void dvtspshift(DVector tsp, long m, int inverse)
{
    long num_shift;
    
    if (inverse) {
	num_shift = tsp->length / 2 + m;
    } else {
	num_shift = tsp->length / 2 - m;
    }

    dvcircshift(tsp, -num_shift);
    
    return;
}

void dvgtspshift(DVector gtsp, double alpha, int inverse)
{
    long m;

    m = (long)round((double)gtsp->length / (2.0 * alpha));

    dvtspshift(gtsp, m, inverse);
    
    return;
}

spFFTRec dvinitfft(long order, spFFTPrecision precision)
{
    return spInitFFT(order, precision);
}

spBool dvfreefft(spFFTRec fftrec)
{
    return spFreeFFT(fftrec);
}

spBool dvresetfftorder(spFFTRec fftrec, long order)
{
    return spResetFFTOrder(fftrec, order);
}

long dvgetfftorder(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;

    return fftrec->order;
}

long dvgetfftlength(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;

    return fftrec->fftl;
}

long dvgetfftlengthspecified(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;

    return fftrec->fftl_specified;
}

long dvgetfftbatch(spFFTRec fftrec)
{
    if (fftrec == NULL) return 0;

    return fftrec->batch;
}

spBool dvexecfftf(spFFTRec fftrec, float *real, float *imag, int inv)
{
    return spExecFFTF(fftrec, real, imag, inv);
}

spBool dvexecfft(spFFTRec fftrec, double *real, double *imag, int inv)
{
    return spExecFFT(fftrec, real, imag, inv);
}

spBool dvexecrfftf(spFFTRec fftrec, float *data, int inv)
{
    return spExecRealFFTF(fftrec, data, inv);
}

spBool dvexecrfft(spFFTRec fftrec, double *data, int inv)
{
    return spExecRealFFT(fftrec, data, inv);
}

static spBool _dvfftex(spFFTRec fftrec, DVector x, int inv, int real_flag, spBool public_call)
{
    if (fftrec == NULL || x == NODATA || x->data == NULL) return SP_FALSE;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    spDebug(80, "_dvfftex", "public_call = %d, x->length = %ld, fftrec->fftl_specified = %ld, fftrec->fftl = %ld, fftrec->batch = %ld\n",
            public_call, x->length, fftrec->fftl_specified, fftrec->fftl, fftrec->batch);
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        _dvfftexBluestein(fftrec, x, inv, real_flag);
        return SP_TRUE;
    }
#endif
    
    if ((fftrec->fftl == fftrec->fftl_specified || fftrec->batch >= 2)
        && x->length < fftrec->fftl * fftrec->batch) {
	x = xdvrealloc(x, fftrec->fftl * fftrec->batch);
    }

    spDebug(80, "_dvfftex", "x->length = %ld\n", x->length);
    
    if (real_flag) {
	return spExecRealFFTInternal(fftrec, x->data, inv, SP_FALSE);
    } else {
	if (x->imag == NULL) dvizeros(x, x->length);
	
	return spExecFFTInternal(fftrec, x->data, x->imag, inv, SP_FALSE);
    }
}

spBool dvfftex(spFFTRec fftrec, DVector x)
{
    return _dvfftex(fftrec, x, 0, 0, SP_TRUE);
}

spBool dvifftex(spFFTRec fftrec, DVector x)
{
    return _dvfftex(fftrec, x, 1, 0, SP_TRUE);
}

spBool dvrfftex(spFFTRec fftrec, DVector x)
{
    return _dvfftex(fftrec, x, 0, 1, SP_TRUE);
}

spBool dvirfftex(spFFTRec fftrec, DVector x)
{
    return _dvfftex(fftrec, x, 1, 1, SP_TRUE);
}

static DVector _xdvfftex(spFFTRec fftrec, DVector x, int inv, int real_flag, spBool public_call)
{
    DVector y;
    spBool flag;
    
    if (fftrec == NULL || x == NODATA || x->data == NULL) return NODATA;

#ifdef SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM
    if (public_call == SP_TRUE && fftrec->fftl_specified != fftrec->fftl) {
        y = xdvclone(x);
        _dvfftexBluestein(fftrec, y, inv, real_flag);
        return y;
    }
#endif
    
    if (!real_flag) {
	y = xdvrizeros(fftrec->fftl * fftrec->batch);
    } else {
	y = xdvzeros(fftrec->fftl * fftrec->batch);
    }
    dvcopy(y, x);
    
    if (real_flag) {
	flag = spExecRealFFTInternal(fftrec, y->data, inv, public_call);
    } else {
	flag = spExecFFTInternal(fftrec, y->data, y->imag, inv, public_call);
    }

    if (flag == SP_FALSE) {
	xdvfree(y);
	y = NODATA;
    } else {
        if (public_call == SP_TRUE && fftrec->batch <= 1 && fftrec->fftl_specified != y->length) {
            DVector y2;
            
            if (real_flag) {
                y2 = xdvzeros(fftrec->fftl_specified);
            } else {
                y2 = xdvrizeros(fftrec->fftl_specified);
            }
            dvcopy(y2, y);
            xdvfree(y);
            y = y2;
        }
    }

    return y;
}

DVector xdvfftex(spFFTRec fftrec, DVector x)
{
    return _xdvfftex(fftrec, x, 0, 0, SP_TRUE);
}

DVector xdvifftex(spFFTRec fftrec, DVector x)
{
    return _xdvfftex(fftrec, x, 1, 0, SP_TRUE);
}

DVector xdvrfftex(spFFTRec fftrec, DVector x)
{
    return _xdvfftex(fftrec, x, 0, 1, SP_TRUE);
}

DVector xdvirfftex(spFFTRec fftrec, DVector x)
{
    return _xdvfftex(fftrec, x, 1, 1, SP_TRUE);
}


spBool dvfftturnex(spFFTRec fftrec, DVector x)
{
    if (x == NODATA || x->data == NULL) return SP_FALSE;

    return spTurnFFTSpectrum(fftrec, x->data, x->imag);
}

DVector xdvfftturnex(spFFTRec fftrec, DVector x)
{
    DVector y;

    y = xdvclone(x);

    if (dvfftturnex(fftrec, y) == SP_FALSE) {
	xdvfree(y);
	return NODATA;
    }

    return y;
}

spBool dvfftshiftex(spFFTRec fftrec, DVector x)
{
    if (x == NODATA || x->data == NULL) return SP_FALSE;

    return spExecFFTShift(fftrec, x->data, x->imag);
}

DVector xdvfftshiftex(spFFTRec fftrec, DVector x)
{
    DVector y;

    y = xdvclone(x);

    if (dvfftshiftex(fftrec, y) == SP_FALSE) {
	xdvfree(y);
	return NODATA;
    }

    return y;
}

spBool dvmultrfftex(spFFTRec fftrec, DVector a, DVector b)
{
    if (a == NODATA || b == NODATA || a->data == NULL || b->data == NULL) return SP_FALSE;
    
    return spMultiplySpectrumOfRealFFT(fftrec, a->data, b->data);
}

void dvrffttofftex(spFFTRec fftrec, DVector x)
{
    long m;
    double *data;
    double *imag;
    
    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;
	
	if (x->imag == NULL) {
	    rffttofftip(data, NULL, fftrec->fftl_specified);
	} else {
	    imag = x->imag + m * fftrec->fftl;
	    rffttofftip(data, imag, fftrec->fftl_specified);
	}
    }
    
    return;
}

void dvffttorfftex(spFFTRec fftrec, DVector x)
{
    long m;
    double *data;
    double *imag;
    
    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;
	
	if (x->imag == NULL) {
	    ffttorfftip(data, NULL, fftrec->fftl_specified);
	} else {
	    imag = x->imag + m * fftrec->fftl;
	    ffttorfftip(data, imag, fftrec->fftl_specified);
	}
    }
    
    return;
}

void dvrffttopowerex(spFFTRec fftrec, DVector x, double exponent)
{
    long m;
    double *data;
    
    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;
	
	rffttopower(data, exponent, fftrec->fftl_specified);
    }

    return;
}

void dvpowertorfftex(spFFTRec fftrec, DVector x, double orig_exponent)
{
    long m;
    double *data;
    
    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;
	
	powertorfft(data, orig_exponent, fftrec->fftl_specified);
    }

    return;
}

spBool _dvgetfftspecex(spFFTRec fftrec, DVector x, int spec_type)
{
    long m;
    double *data;
    
    if (dvrfftex(fftrec, x) == SP_FALSE) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;
	
	if (spec_type == 0) {	/* power spectrum */
	    rffttopower(data, 1.0, fftrec->fftl_specified);
	} else if (spec_type == 1) { /* absolute spectrum */
	    rffttopower(data, 0.5, fftrec->fftl_specified);
	} else if (spec_type == 2) { /* angle */
	    rffttoangle(data, fftrec->fftl_specified);
	}
    }
    
    return SP_TRUE;
}

spBool dvfftpowerex(spFFTRec fftrec, DVector x)
{
    return _dvgetfftspecex(fftrec, x, 0);
}

spBool dvfftabsex(spFFTRec fftrec, DVector x)
{
    return _dvgetfftspecex(fftrec, x, 1);
}

spBool dvifftabsex(spFFTRec fftrec, DVector x)
{
    long k, m;
    long remain;
    double value;
    double *data;
    
    if (fftrec == NULL || x == NODATA || x->data == NULL) return SP_FALSE;

    for (m = 0; m < fftrec->batch; m++) {
	data = x->data + m * fftrec->fftl;

        remain = fftrec->fftl % 2;

        if (remain == 0) {
            value = data[fftrec->fftl/2];
        } else {
            value = 0.0;
        }
	for (k = fftrec->fftl - 2; k >= 0; k -= 2) {
	    data[k] = data[k/2];
	    data[k + 1] = 0.0;
	}
        if (remain == 0) {
            data[1] = value;
        }
    }
    
    return dvirfftex(fftrec, x);
}

spBool dvfftangleex(spFFTRec fftrec, DVector x)
{
    return _dvgetfftspecex(fftrec, x, 2);
}

spBool dvfftgrpdlyex(spFFTRec fftrec, DVector x)
{
    long k, m;
    double value;
    DVector fx, dfx;
    spBool flag = SP_FALSE;
    
    /* calculate frequency diff spectrum */
    fx = xdvrizeros(fftrec->fftl * fftrec->batch);
    dfx = xdvrizeros(fftrec->fftl * fftrec->batch);
    for (k = 0; k < x->length; k++) {
	if (k < fx->length) {
	    fx->data[k] = x->data[k];
	}
	if (k < dfx->length) {
	    if (fftrec->batch >= 2) {
		m = k % fftrec->fftl;
		dfx->imag[k] = -(double)m * x->data[k];
	    } else {
		dfx->imag[k] = -(double)k * x->data[k];
	    }
	}
    }
    if (dvfftex(fftrec, fx) == SP_TRUE
	&& dvfftex(fftrec, dfx) == SP_TRUE) {

	if (x->length != fx->length) {
	    x = xdvrealloc(x, fx->length);
	}
    
	/* calculate group delay */
	for (k = 0; k < fx->length; k++) {
	    value = SQUARE(fx->data[k]) + SQUARE(fx->imag[k]);
	    if (value == 0.0) {
		x->data[k] = 0.0;
	    } else {
		x->data[k] = -(dfx->imag[k] * fx->data[k] -
			       fx->imag[k] * dfx->data[k]) / value;
	    }
	}

	flag = SP_TRUE;
    }

    /* memory free */
    xdvfree(fx);
    xdvfree(dfx);

    return flag;
}

DVector xdvfftgrpdlyex(spFFTRec fftrec, DVector x)
{
    DVector gd;
    
    gd = xdvclone(x);

    if (dvfftgrpdlyex(fftrec, gd) == SP_FALSE) {
	xdvfree(gd);
	return NODATA;
    }

    return gd;
}

spBool dvfftconvex(spFFTRec fftrec, DVector a, DVector b)
{
    DVector cb;
    spBool flag = SP_FALSE;
    
    if (dvrfftex(fftrec, a) == SP_TRUE) {
	/*cb = xdvrizeros(fftrec->fftl);*/
	cb = xdvzeros(fftrec->fftl * fftrec->batch);
	dvcopy(cb, b);
	
	if (dvrfftex(fftrec, cb) == SP_TRUE) {
	    /* convolution */
	    spMultiplySpectrumOfRealFFT(fftrec, a->data, cb->data);
	    
	    if (dvirfftex(fftrec, a) == SP_TRUE) {
		flag = SP_TRUE;
	    }
	}

	xdvfree(cb);
	
	return flag;
    }

    return SP_FALSE;
}

DVector xdvfftconvex(spFFTRec fftrec, DVector a, DVector b)
{
    DVector y;

    y = xdvclone(a);

    if (dvfftconvex(fftrec, y, b) == SP_FALSE) {
	xdvfree(y);
	return NODATA;
    }

    return y;
}

spBool dvspectocepex(spFFTRec fftrec, DVector x)
{
    dvffttorfftex(fftrec, x);
    return dvrffttocepex(fftrec, x);
}

spBool dvspectompcex(spFFTRec fftrec, DVector x)
{
    if (dvspectocepex(fftrec, x) == SP_TRUE) {
	dvceptompcex(fftrec, x);
	return SP_TRUE;
    }
    
    return SP_FALSE;
}

spBool dvrffttocepex(spFFTRec fftrec, DVector x)
{
    long m;
	    
    for (m = 0; m < fftrec->batch; m++) {
	rffttolog(x->data + m * fftrec->fftl, fftrec->fftl_specified);
    }
	
    return dvirfftex(fftrec, x);
}

spBool dvrffttompcex(spFFTRec fftrec, DVector x)
{
    if (dvrffttocepex(fftrec, x) == SP_TRUE) {
	dvceptompcex(fftrec, x);
	return SP_TRUE;
    }
    
    return SP_FALSE;
}

spBool dvceptospecex(spFFTRec fftrec, DVector x)
{
    if (dvfftex(fftrec, x) == SP_TRUE) {
	dvexp(x);
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool dvceptorfftex(spFFTRec fftrec, DVector x)
{
    if (dvrfftex(fftrec, x) == SP_TRUE) {
	long m;
	
	for (m = 0; m < fftrec->batch; m++) {
	    rffttoexp(x->data + m * fftrec->fftl, fftrec->fftl);
	}
	
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

void dvceptompcex(spFFTRec fftrec, DVector x)
{
    long m;
	    
    for (m = 0; m < fftrec->batch; m++) {
	ceptompc(x->data + m * fftrec->fftl, fftrec->fftl);
    }
	
    return;
}

spBool dvrcepex(spFFTRec fftrec, DVector x)
{
    if (dvrfftex(fftrec, x) == SP_TRUE) {
	return dvrffttocepex(fftrec, x);
    }
    
    return SP_FALSE;
}

spBool dvmpcepex(spFFTRec fftrec, DVector x)
{
    if (dvrcepex(fftrec, x) == SP_TRUE) {
	dvceptompcex(fftrec, x);
	
	return SP_TRUE;
    }

    return SP_FALSE;
}
