/*
 *	fileio.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/vector.h>
#include <sp/voperate.h>
#include <sp/matrix.h>
#include <sp/fft.h>
#include <sp/fileio.h>

#define SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "7"
#define SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "15"

static spDVector sp_prev_vec = NODATA;

long dvreadfirst(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp)
{
    long ndata = 0;
    long k;
    long len;
    short value;
    double dvalue;

    if (vector == NODATA) {
	return 0;
    }
    
    if (length <= 0 || vector->length < length) {
	length = vector->length;
    }
    
    if (shift < 0) {
	len = -shift;
	if (len >= vector->length) {
	    fprintf(stderr, "-shift must be less than input length\n");
	}
	
	for (k = 0; k < len; k++) {
	    vector->data[k] = 0.0;
	}
    } else {
	if (double_flag) {
	    for (k = 0; k < shift; k++) {
		freaddouble(&dvalue, 1, 0, fp);
	    }
	} else {
	    for (k = 0; k < shift; k++) {
		freadshort(&value, 1, 0, fp);
	    }
	}
	len = 0;
    }

    /* read data */
    if (double_flag) {
	ndata = freaddouble(vector->data + len, length - len, swap, fp);
    } else {
	ndata = freadshorttod(vector->data + len, length - len, swap, fp);
    }
    for (k = len + ndata; k < vector->length; k++) {
	vector->data[k] = 0.0;
    }
    
    if (sp_prev_vec != NODATA) {
	xdvfree(sp_prev_vec);
    }
    sp_prev_vec = xdvcut(vector, 0, len + ndata);
    
    return ndata;
}

long dvreadsfirst(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvreadfirst(vector, shift, length, swap, 0, fp);
}

long dvreaddfirst(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvreadfirst(vector, shift, length, swap, 1, fp);
}

long dvreadframe(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp)
{
    long ndata = 0;
    long k;
    long len;
    short value;
    double dvalue;

    if (vector == NODATA) {
	return 0;
    }
    
    if (length <= 0 || vector->length < length) {
	length = vector->length;
    }
    
    if (shift < 0) {
	fprintf(stderr, "shift must be 0 or more\n");
	return 0;
    }

    if (sp_prev_vec != NODATA) {
	if (shift < sp_prev_vec->length) {
	    len = sp_prev_vec->length - shift;
	    dvadd(vector, 0, sp_prev_vec, shift, len, 0);
	} else {
	    len = shift - sp_prev_vec->length;
	    if (double_flag) {
		for (k = 0; k < len; k++) {
		    freaddouble(&dvalue, 1, 0, fp);
		}
	    } else {
		for (k = 0; k < len; k++) {
		    freadshort(&value, 1, 0, fp);
		}
	    }
	    len = 0;
	}
    } else {
	len = 0;
    }

    /* read data */
    if (double_flag) {
	ndata = freaddouble(vector->data + len, length - len, swap, fp);
    } else {
	ndata = freadshorttod(vector->data + len, length - len, swap, fp);
    }
    for (k = len + ndata; k < vector->length; k++) {
	vector->data[k] = 0.0;
    }
    
    if (sp_prev_vec != NODATA) {
	xdvfree(sp_prev_vec);
    }
    sp_prev_vec = xdvcut(vector, 0, len + ndata);
    
    return ndata;
}

long dvreadsframe(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvreadframe(vector, shift, length, swap, 0, fp);
}

long dvreaddframe(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvreadframe(vector, shift, length, swap, 1, fp);
}

long dvwriteframe(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp)
{
    long ndata = 0;

    if (vector == NODATA) {
	return 0;
    }
    
    if (length <= 0 || vector->length < length) {
	length = vector->length;
    }

    if (double_flag) {
	ndata = fwritedouble(vector->data, length, swap, fp);
    } else {
	ndata = fwritedoubletos(vector->data, length, swap, fp);
    }
    dvdatashift(vector, -shift);
    
    return ndata;
}

long dvwritesframe(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvwriteframe(vector, shift, length, swap, 0, fp);
}

long dvwritedframe(spDVector vector, long shift, long length, int swap, FILE *fp)
{
    return dvwriteframe(vector, shift, length, swap, 1, fp);
}

long svfwritesvector(FILE *fp, spSVector vector, long length, int swap)
{
    /* write data */
    return fwriteshort(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long lvfwritelvector(FILE *fp, spLVector vector, long length, int swap)
{
    /* write data */
    /*return fwritelong(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);*/
    return fwritelong32(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long fvfwritefvector(FILE *fp, spFVector vector, long length, int swap)
{
    /* write data */
    return fwritefloat(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long dvfwritedvector(FILE *fp, spDVector vector, long length, int swap)
{
    /* write data */
    return fwritedouble(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long dvfwritesvector(FILE *fp, spDVector vector, long length, int swap)
{
    /* write data */
    return fwritedoubletos(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long dvfwritelvector(FILE *fp, spDVector vector, long length, int swap)
{
    /* write data */
    /* return fwritedoubletol(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);*/
    return fwritedoubletol32(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

long dvfwritefvector(FILE *fp, spDVector vector, long length, int swap)
{
    /* write data */
    return fwritedoubletof(vector->data, length <= 0 ? vector->length : MIN(vector->length, length), swap, fp);
}

/*
 *	read short signal
 */
spSVector xsvreadssignal(const char *filename, long headlen, int swap)
{
    long length;
    spSVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, short)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xsvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xsvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, headlen, 0);

        /* read data */
        if (freadshort(vector->data, vector->length, swap, fp) != vector->length) {
            xsvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

/*
 *	read long signal
 */
spLVector xlvreadlsignal(const char *filename, long headlen, int swap)
{
    long length;
    spLVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, /*long*/spLong32)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xlvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xlvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, headlen, 0);

        /* read data */
        /*freadlong(vector->data, vector->length, swap, fp);*/
        if (freadlong32(vector->data, vector->length, swap, fp) != vector->length) {
            xlvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

/*
 *	read float signal
 */
spFVector xfvreadfsignal(const char *filename, long headlen, int swap)
{
    long length;
    spFVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, /*double*/float)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xfvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xfvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, headlen, 0);

        /* read data */
        if (freadfloat(vector->data, vector->length, swap, fp) != vector->length) {
            xfvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

/*
 *	read double signal
 */
spDVector xdvreaddsignal(const char *filename, long headlen, int swap)
{
    long length;
    spDVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, double)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xdvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, headlen, 0);

        /* read data */
        if (freaddouble(vector->data, vector->length, swap, fp) != vector->length) {
            xdvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

spDVector xdvreadssignal(const char *filename, long headlen, int swap)
{
    long length;
    spDVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, short)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xdvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, (long)headlen, 0);

        /* read data */
        if (freadshorttod(vector->data, vector->length, swap, fp) != vector->length) {
            xdvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

spDVector xdvreadlsignal(const char *filename, long headlen, int swap)
{
    long length;
    spDVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, /*long*/spLong32)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xdvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, (long)headlen, 0);

        /* read data */
        /*freadlongtod(vector->data, vector->length, swap, fp);*/
        if (freadlong32tod(vector->data, vector->length, swap, fp) != vector->length) {
            xdvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

spDVector xdvreadfsignal(const char *filename, long headlen, int swap)
{
    long length;
    spDVector vector;
    FILE *fp;

    /* get signal length */
    if ((length = (long)getsiglen(filename, headlen, float)) <= 0) {
	return NODATA;
    }

    /* memory allocate */
    vector = xdvalloc(length);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdvfree(vector);
        vector = NODATA;
    } else {
        /* skip header */
        if (headlen > 0)
            fseek(fp, (long)headlen, 0);

        /* read data */
        if (freadfloattod(vector->data, vector->length, swap, fp) != vector->length) {
            xdvfree(vector);
            vector = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return vector;
}

/*
 *	write short signal
 */
spBool svwritessignal(const char *filename, spSVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = svfwritesvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}
/*
 *	write long signal
 */
spBool lvwritelsignal(const char *filename, spLVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = lvfwritelvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

/*
 *	write float signal
 */
spBool fvwritefsignal(const char *filename, spFVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fvfwritefvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

/*
 *	write double signal
 */
spBool dvwritedsignal(const char *filename, spDVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = dvfwritedvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dvwritessignal(const char *filename, spDVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = dvfwritesvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dvwritelsignal(const char *filename, spDVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = dvfwritelvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dvwritefsignal(const char *filename, spDVector vector, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = dvfwritefvector(fp, vector, 0, swap);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == vector->length) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
    
}

spSVectors xsvsreadsvectors(const char *filename, long num_vector, int swap)
{
    long n;
    long length;
    long total_length;
    spSVectors vecs;
    FILE *fp;

    /* get data length */
    if (num_vector <= 0 || (total_length = (long)getsiglen(filename, 0, short)) <= 0) {
	return NODATA;
    }
    if (total_length % num_vector != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }

    vecs = NODATA;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "rb"))) {
        /* memory allocate */
        vecs = xsvsalloc(num_vector);
        length = total_length / num_vector;
        
        for (n = 0; n < vecs->num_vector; n++) {
            vecs->vector[n] = xsvalloc(length);
            
            /* read data */
            if (freadshort(vecs->vector[n]->data, length, swap, fp) != length) {
                xsvsfree(vecs);
                vecs = NODATA;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    return vecs;
}

spLVectors xlvsreadlvectors(const char *filename, long num_vector, int swap)
{
    long n;
    long length;
    long total_length;
    spLVectors vecs;
    FILE *fp;

    /* get data length */
    if (num_vector <= 0 || (total_length = (long)getsiglen(filename, 0, long)) <= 0) {
	return NODATA;
    }
    if (total_length % num_vector != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }

    vecs = NODATA;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "rb"))) {
        /* memory allocate */
        vecs = xlvsalloc(num_vector);
        length = total_length / num_vector;
        
        for (n = 0; n < vecs->num_vector; n++) {
            vecs->vector[n] = xlvalloc(length);
            
            /* read data */
            if (freadlong(vecs->vector[n]->data, length, swap, fp) != length) {
                xlvsfree(vecs);
                vecs = NODATA;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    return vecs;
}

spFVectors xfvsreadfvectors(const char *filename, long num_vector, int swap)
{
    long n;
    long length;
    long total_length;
    spFVectors vecs;
    FILE *fp;

    /* get data length */
    if (num_vector <= 0 || (total_length = (long)getsiglen(filename, 0, float)) <= 0) {
	return NODATA;
    }
    if (total_length % num_vector != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }

    vecs = NODATA;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "rb"))) {
        /* memory allocate */
        vecs = xfvsalloc(num_vector);
        length = total_length / num_vector;
        
        for (n = 0; n < vecs->num_vector; n++) {
            vecs->vector[n] = xfvalloc(length);
            
            /* read data */
            if (freadfloat(vecs->vector[n]->data, length, swap, fp) != length) {
                xfvsfree(vecs);
                vecs = NODATA;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    return vecs;
}

spDVectors xdvsreaddvectors(const char *filename, long num_vector, int swap)
{
    long n;
    long length;
    long total_length;
    spDVectors vecs;
    FILE *fp;

    /* get data length */
    if (num_vector <= 0 || (total_length = (long)getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    if (total_length % num_vector != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }

    vecs = NODATA;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "rb"))) {
        /* memory allocate */
        vecs = xdvsalloc(num_vector);
        length = total_length / num_vector;
        
        for (n = 0; n < vecs->num_vector; n++) {
            vecs->vector[n] = xdvalloc(length);
            
            /* read data */
            if (freaddouble(vecs->vector[n]->data, length, swap, fp) != length) {
                xdvsfree(vecs);
                vecs = NODATA;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    return vecs;
}

long svswritesvectors(const char *filename, spSVectors vecs, int swap)
{

    FILE *fp;
    long n, k;
    long max_length;
    long writelen;
    long nremain;
    long nwrite = -1;
    static short buf[8] = {0, 0, 0, 0, 0, 0, 0, 0};

    if (vecs->num_vector <= 0) return 0;

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "wb"))) {
        return -1;
    }
    
    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
        
    for (n = 0; n < vecs->num_vector; n++) {
        /* write data */
        nwrite = fwriteshort(vecs->vector[n]->data, vecs->vector[n]->length, swap, fp);
        if (nwrite != vecs->vector[n]->length) {
            nwrite = -1;
        } else {
            if (vecs->vector[n]->length < max_length) {
                nremain = max_length - vecs->vector[n]->length;
                while (nremain > 0) {
                    writelen = MIN(nremain, 8);
                    nwrite = fwriteshort(buf, writelen, swap, fp);
                    if (nwrite != writelen) {
                        nwrite = -1;
                        break;
                    }
                    nremain -= nwrite;
                }
            }
        }

        if (nwrite < 0) {
            break;
        }
    }

    /* close file */
    spCloseFile(fp);

    if (nwrite <= 0) {
        return nwrite;
    } else {
        return max_length;
    }
}

long lvswritelvectors(const char *filename, spLVectors vecs, int swap)
{

    FILE *fp;
    long n, k;
    long max_length;
    long writelen;
    long nremain;
    long nwrite = -1;
    static long buf[8] = {0, 0, 0, 0, 0, 0, 0, 0};

    if (vecs->num_vector <= 0) return 0;

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "wb"))) {
        return -1;
    }
    
    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
        
    for (n = 0; n < vecs->num_vector; n++) {
        /* write data */
        nwrite = fwritelong(vecs->vector[n]->data, vecs->vector[n]->length, swap, fp);
        if (nwrite != vecs->vector[n]->length) {
            nwrite = -1;
        } else {
            if (vecs->vector[n]->length < max_length) {
                nremain = max_length - vecs->vector[n]->length;
                while (nremain > 0) {
                    writelen = MIN(nremain, 8);
                    nwrite = fwritelong(buf, writelen, swap, fp);
                    if (nwrite != writelen) {
                        nwrite = -1;
                        break;
                    }
                    nremain -= nwrite;
                }
            }
        }

        if (nwrite < 0) {
            break;
        }
    }

    /* close file */
    spCloseFile(fp);

    if (nwrite <= 0) {
        return nwrite;
    } else {
        return max_length;
    }
}

long fvswritefvectors(const char *filename, spFVectors vecs, int swap)
{

    FILE *fp;
    long n, k;
    long max_length;
    long writelen;
    long nremain;
    long nwrite = -1;
    static float buf[8] = {0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f};

    if (vecs->num_vector <= 0) return 0;

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "wb"))) {
        return -1;
    }
    
    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
        
    for (n = 0; n < vecs->num_vector; n++) {
        /* write data */
        nwrite = fwritefloat(vecs->vector[n]->data, vecs->vector[n]->length, swap, fp);
        if (nwrite != vecs->vector[n]->length) {
            nwrite = -1;
        } else {
            if (vecs->vector[n]->length < max_length) {
                nremain = max_length - vecs->vector[n]->length;
                while (nremain > 0) {
                    writelen = MIN(nremain, 8);
                    nwrite = fwritefloat(buf, writelen, swap, fp);
                    if (nwrite != writelen) {
                        nwrite = -1;
                        break;
                    }
                    nremain -= nwrite;
                }
            }
        }

        if (nwrite < 0) {
            break;
        }
    }

    /* close file */
    spCloseFile(fp);

    if (nwrite <= 0) {
        return nwrite;
    } else {
        return max_length;
    }
}

long dvswritedvectors(const char *filename, spDVectors vecs, int swap)
{

    FILE *fp;
    long n, k;
    long max_length;
    long writelen;
    long nremain;
    long nwrite = -1;
    static double buf[8] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

    if (vecs->num_vector <= 0) return 0;

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "wb"))) {
        return -1;
    }
    
    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
        
    for (n = 0; n < vecs->num_vector; n++) {
        /* write data */
        nwrite = fwritedouble(vecs->vector[n]->data, vecs->vector[n]->length, swap, fp);
        if (nwrite != vecs->vector[n]->length) {
            nwrite = -1;
        } else {
            if (vecs->vector[n]->length < max_length) {
                nremain = max_length - vecs->vector[n]->length;
                while (nremain > 0) {
                    writelen = MIN(nremain, 8);
                    nwrite = fwritedouble(buf, writelen, swap, fp);
                    if (nwrite != writelen) {
                        nwrite = -1;
                        break;
                    }
                    nremain -= nwrite;
                }
            }
        }

        if (nwrite < 0) {
            break;
        }
    }

    /* close file */
    spCloseFile(fp);

    if (nwrite <= 0) {
        return nwrite;
    } else {
        return max_length;
    }
}

spSMatrix xsmreadsmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spSMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, short)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xsmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xsmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freadshort(mat->data[0], length, swap, fp) != length) {
            xsmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spLMatrix xlmreadlmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spLMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, /*long*/spLong32)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xlmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xlmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        /*freadlong(mat->data[0], mat->row * mat->col, swap, fp);*/
        if (freadlong32(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xlmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spFMatrix xfmreadfmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spFMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, float)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xfmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xfmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freadfloat(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xfmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spDMatrix xdmreaddmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spDMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xdmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freaddouble(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xdmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spDMatrix xdmreadsmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spDMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, short)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xdmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freadshorttod(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xdmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spDMatrix xdmreadlmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spDMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, /*long*/spLong32)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xdmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freadlong32tod(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xdmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spDMatrix xdmreadfmatrix(const char *filename, long ncol, int swap)
{
    long nrow;
    long length;
    spDMatrix mat;
    FILE *fp;

    /* get data length */
    if (ncol <= 0 || (length = (long)getsiglen(filename, 0, float)) <= 0) {
	return NODATA;
    }
    if (length % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    nrow = length / ncol;

    /* memory allocate */
    mat = xdmalloc(nrow, ncol);

    /* open file */
    if (NULL == (fp = spOpenFile(filename, "rb"))) {
        xdmfree(mat);
	mat = NODATA;
    } else {
        /* read data */
        if (freadfloattod(mat->data[0], mat->row * mat->col, swap, fp) != length) {
            xdmfree(mat);
            mat = NODATA;
        }

        /* close file */
        spCloseFile(fp);
    }

    return mat;
}

spBool smwritesmatrix(const char *filename, spSMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwriteshort(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool lmwritelmatrix(const char *filename, spLMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        /*nwrite = fwritelong(mat->data[0], mat->row * mat->col, swap, fp);*/
        nwrite = fwritelong32(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool fmwritefmatrix(const char *filename, spFMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwritefloat(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dmwritedmatrix(const char *filename, spDMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwritedouble(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dmwritesmatrix(const char *filename, spDMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwritedoubletos(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dmwritelmatrix(const char *filename, spDMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwritedoubletol32(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dmwritefmatrix(const char *filename, spDMatrix mat, int swap)
{
    FILE *fp;
    long nwrite = -1;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        /* write data */
        nwrite = fwritedoubletof(mat->data[0], mat->row * mat->col, swap, fp);

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == mat->row * mat->col) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

#if 1
spSMatrices xsmsreadsmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spSMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xsmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xsmalloc(nrow, ncol);
            /* read data */
            if (freadshort(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xsmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spLMatrices xlmsreadlmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spLMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xlmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xlmalloc(nrow, ncol);
            /* read data */
            if (freadlong32(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xlmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spFMatrices xfmsreadfmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spFMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xfmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xfmalloc(nrow, ncol);
            /* read data */
            if (freadfloat(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xfmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spDMatrices xdmsreaddmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spDMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, double)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xdmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xdmalloc(nrow, ncol);
            /* read data */
            if (freaddouble(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xdmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spDMatrices xdmsreadsmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spDMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, short)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xdmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xdmalloc(nrow, ncol);
            /* read data */
            if (freadshorttod(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xdmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spDMatrices xdmsreadlmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spDMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, spLong32)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xdmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xdmalloc(nrow, ncol);
            /* read data */
            if (freadlong32tod(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xdmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

spDMatrices xdmsreadfmatrices(const char *filename, long nmat, long ncol, int swap)
{
    spFileOffset length;
    long matlen;
    long nrow;
    long k;
    FILE *fp;
    spDMatrices xs = NODATA;

    /* get data length */
    if (nmat <= 0 || ncol <= 0 || (length = getsiglen(filename, 0, float)) <= 0) {
	return NODATA;
    }
    matlen = (long)(length / (spFileOffset)nmat);
    if (length % nmat != 0 || matlen % ncol != 0) {
	fprintf(stderr, "Wrong data format: %s\n", filename);
	return NODATA;
    }
    
    /* open file */
    if ((fp = spOpenFile(filename, "rb")) != NULL) {
        nrow = matlen / ncol;
        xs = xdmsalloc(nmat);

        for (k = 0; k < nmat; k++) {
            xs->matrix[k] = xdmalloc(nrow, ncol);
            /* read data */
            if (freadfloattod(xs->matrix[k]->data[0], matlen, swap, fp) != matlen) {
                xdmsfree(xs);
                xs = NODATA;
                break;
            }
        }
        
        /* close file */
        spCloseFile(fp);
    }

    return xs;
}

/* the sizes of all matrices must be the same. */
spBool smswritesmatrices(const char *filename, spSMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwriteshort(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool lmswritelmatrices(const char *filename, spLMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritelong32(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool fmswritefmatrices(const char *filename, spFMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritefloat(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool dmswritedmatrices(const char *filename, spDMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritedouble(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool dmswritesmatrices(const char *filename, spDMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritedoubletos(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool dmswritelmatrices(const char *filename, spDMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritedoubletol32(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

/* the sizes of all matrices must be the same. */
spBool dmswritefmatrices(const char *filename, spDMatrices xs, int swap)
{
    FILE *fp;
    long k;
    long matlen, prev_matlen;
    long nwrite = -1;

    matlen = prev_matlen = -1;
    for (k = 0; k < xs->num_matrix; k++) {
        if (xs->matrix[k] == NODATA) {
            matlen = -1;
            break;
        }
        matlen = xs->matrix[k]->row * xs->matrix[k]->col;
        if (matlen != prev_matlen) {
            matlen = -1;
            break;
        }
        prev_matlen = matlen;
    }

    if (matlen < 0) return SP_FALSE;

    /* open file */
    if (NULL != (fp = spOpenFile(filename, "wb"))) {
        for  (k = 0; k < xs->num_matrix; k++) {
            /* write data */
            nwrite = fwritedoubletof(xs->matrix[k]->data[0], matlen, swap, fp);
            if (nwrite != matlen) {
                nwrite = -1;
                break;
            }
        }

        /* close file */
        spCloseFile(fp);
    }

    if (nwrite == -1) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}
#endif

spBool svreadsvector_txt(const char *filename, spSVector vector)
{
    int nscan;
    long lc;
    int value;
    char line[SP_MAX_PATHNAME];
    char string[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    lc = 0;
    while (fgetline(line, fp) != EOF) {
	if (lc >= vector->length) {
	    break;
	}

#if defined(HAVE_SSCANF_S)
        sscanf_s(line, "%s", string, (unsigned int)sizeof(string));
        nscan = sscanf_s(string, "%d", &value);
#else
        sscanf(line, "%s", string);
        nscan = sscanf(string, "%d", &value);
#endif
        if (nscan == 1) {
	    vector->data[lc] = (short)value;
	    lc++;
        }
    }

    for (; lc < vector->length; lc++) {
	vector->data[lc] = 0;
    }

    spCloseFile(fp);

    return SP_TRUE;
}

spBool lvreadlvector_txt(const char *filename, spLVector vector)
{
    int nscan;
    long lc;
    long value;
    char line[SP_MAX_PATHNAME];
    char string[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    lc = 0;
    while (fgetline(line, fp) != EOF) {
	if (lc >= vector->length) {
	    break;
	}

#if defined(HAVE_SSCANF_S)
        sscanf_s(line, "%s", string, (unsigned int)sizeof(string));
        nscan = sscanf_s(string, "%ld", &value);
#else
        sscanf(line, "%s", string);
        nscan = sscanf(string, "%ld", &value);
#endif
        if (nscan == 1) {
	    vector->data[lc] = value;
	    lc++;
        }
    }

    for (; lc < vector->length; lc++) {
	vector->data[lc] = 0;
    }

    spCloseFile(fp);

    return SP_TRUE;
}

spBool fvreadfvector_txt(const char *filename, spFVector vector)
{
    int nscan;
    long lc;
    float value;
    char line[SP_MAX_PATHNAME];
    char string[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    lc = 0;
    while (fgetline(line, fp) != EOF) {
	if (lc >= vector->length) {
	    break;
	}

#if defined(HAVE_SSCANF_S)
        sscanf_s(line, "%s", string, (unsigned int)sizeof(string));
        nscan = sscanf_s(string, "%f", &value);
#else
        sscanf(line, "%s", string);
        nscan = sscanf(string, "%f", &value);
#endif
        if (nscan == 1) {
	    vector->data[lc] = value;
	    lc++;
        }
    }

    for (; lc < vector->length; lc++) {
	vector->data[lc] = 0.0f;
    }

    spCloseFile(fp);

    return SP_TRUE;
}

spBool dvreaddvector_txt(const char *filename, spDVector vector)
{
    int nscan;
    long lc;
    double value;
    char line[SP_MAX_PATHNAME];
    char string[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    lc = 0;
    while (fgetline(line, fp) != EOF) {
	if (lc >= vector->length) {
	    break;
	}

#if defined(HAVE_SSCANF_S)
        sscanf_s(line, "%s", string, (unsigned int)sizeof(string));
        nscan = sscanf_s(string, "%lf", &value);
#else
        sscanf(line, "%s", string);
        nscan = sscanf(string, "%lf", &value);
#endif
            
        if (nscan == 1) {
	    vector->data[lc] = value;
	    lc++;
        }
    }

    for (; lc < vector->length; lc++) {
	vector->data[lc] = 0.0;
    }

    spCloseFile(fp);

    return SP_TRUE;
}

spSVector xsvreadsvector_txt(const char *filename)
{
    long length;
    spSVector vector;

    if ((length = getfilesize_txt(filename)) == SP_FAILURE) {
	return NODATA;
    }

    vector = xsvalloc(length);

    if (svreadsvector_txt(filename, vector) == SP_FALSE) {
	xsvfree(vector);
	return NODATA;
    }

    return vector;
}

spLVector xlvreadlvector_txt(const char *filename)
{
    long length;
    spLVector vector;

    if ((length = getfilesize_txt(filename)) == SP_FAILURE) {
	return NODATA;
    }

    vector = xlvalloc(length);

    if (lvreadlvector_txt(filename, vector) == SP_FALSE) {
	xlvfree(vector);
	return NODATA;
    }

    return vector;
}

spFVector xfvreadfvector_txt(const char *filename)
{
    long length;
    spFVector vector;

    if ((length = getfilesize_txt(filename)) == SP_FAILURE) {
	return NODATA;
    }

    vector = xfvalloc(length);

    if (fvreadfvector_txt(filename, vector) == SP_FALSE) {
	xfvfree(vector);
	return NODATA;
    }

    return vector;
}

spDVector xdvreaddvector_txt(const char *filename)
{
    long length;
    spDVector vector;

    if ((length = getfilesize_txt(filename)) == SP_FAILURE) {
	return NODATA;
    }

    vector = xdvalloc(length);

    if (dvreaddvector_txt(filename, vector) == SP_FALSE) {
	xdvfree(vector);
	return NODATA;
    }

    return vector;
}

spBool svreadcol_txt(const char *filename, int col, spSVector vector)
{
    long k;
    int value;
    char buf[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    for (k = 0; k < vector->length; k++) {
	if (fgetcol(buf, col, fp) == EOF) {
	    break;
	}
#if defined(HAVE_SSCANF_S)
	sscanf_s(buf, "%d", &value);
#else
	sscanf(buf, "%d", &value);
#endif
        vector->data[k] = (short)value;
    }

    spCloseFile(fp);
    
    return SP_TRUE;
}

spBool lvreadcol_txt(const char *filename, int col, spLVector vector)
{
    long k;
    char buf[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    for (k = 0; k < vector->length; k++) {
	if (fgetcol(buf, col, fp) == EOF) {
	    break;
	}
#if defined(HAVE_SSCANF_S)
	sscanf_s(buf, "%ld", &vector->data[k]);
#else
	sscanf(buf, "%ld", &vector->data[k]);
#endif
    }

    spCloseFile(fp);
    
    return SP_TRUE;
}

spBool fvreadcol_txt(const char *filename, int col, spFVector vector)
{
    long k;
    char buf[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    for (k = 0; k < vector->length; k++) {
	if (fgetcol(buf, col, fp) == EOF) {
	    break;
	}
#if defined(HAVE_SSCANF_S)
	sscanf_s(buf, "%f", &vector->data[k]);
#else
	sscanf(buf, "%f", &vector->data[k]);
#endif
    }

    spCloseFile(fp);
    
    return SP_TRUE;
}

spBool dvreadcol_txt(const char *filename, int col, spDVector vector)
{
    long k;
    char buf[SP_MAX_LINE];
    FILE *fp;

    if (NULL == (fp = spOpenFile(filename, "r"))) {
        fprintf(stderr, "Can't open file: %s\n", filename);
	return SP_FALSE;
    }

    for (k = 0; k < vector->length; k++) {
	if (fgetcol(buf, col, fp) == EOF) {
	    break;
	}
#if defined(HAVE_SSCANF_S)
	sscanf_s(buf, "%lf", &vector->data[k]);
#else
	sscanf(buf, "%lf", &vector->data[k]);
#endif
    }

    spCloseFile(fp);
    
    return SP_TRUE;
}

spSVector xsvreadcol_txt(const char *filename, int col)
{
    long length;
    spSVector vector;

    length = (long)getnumrow_txt(filename);

    if (length <= 0) {
	return NODATA;
    } else {
	vector = xsvzeros(length);
	svreadcol_txt(filename, col, vector);
    }

    return vector;
}

spLVector xlvreadcol_txt(const char *filename, int col)
{
    long length;
    spLVector vector;

    length = (long)getnumrow_txt(filename);

    if (length <= 0) {
	return NODATA;
    } else {
	vector = xlvzeros(length);
	lvreadcol_txt(filename, col, vector);
    }

    return vector;
}

spFVector xfvreadcol_txt(const char *filename, int col)
{
    long length;
    spFVector vector;

    length = (long)getnumrow_txt(filename);

    if (length <= 0) {
	return NODATA;
    } else {
	vector = xfvzeros(length);
	fvreadcol_txt(filename, col, vector);
    }

    return vector;
}

spDVector xdvreadcol_txt(const char *filename, int col)
{
    long length;
    spDVector vector;

    length = (long)getnumrow_txt(filename);

    if (length <= 0) {
	return NODATA;
    } else {
	vector = xdvzeros(length);
	dvreadcol_txt(filename, col, vector);
    }

    return vector;
}

spBool svwritesvector_txt(const char *filename, spSVector vector)
{
    long lc = -1;
    FILE *fp;

    if (NULL != (fp = spOpenFile(filename, "w"))) {
        for (lc = 0; lc < vector->length; lc++) {
            fprintf(fp, "%d\n", vector->data[lc]);
        }

        /* close file */
        spCloseFile(fp);
    }

    if (lc > 0) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool lvwritelvector_txt(const char *filename, spLVector vector)
{
    long lc = -1;
    FILE *fp;

    if (NULL != (fp = spOpenFile(filename, "w"))) {
        for (lc = 0; lc < vector->length; lc++) {
            fprintf(fp, "%ld\n", vector->data[lc]);
        }

        /* close file */
        spCloseFile(fp);
    }

    if (lc > 0) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool fvwritefvector_txt(const char *filename, spFVector vector)
{
    long lc = -1;
    FILE *fp;

    if (NULL != (fp = spOpenFile(filename, "w"))) {
        for (lc = 0; lc < vector->length; lc++) {
            fprintf(fp, "%f\n", vector->data[lc]);
        }

        /* close file */
        spCloseFile(fp);
    }

    if (lc > 0) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spBool dvwritedvector_txt(const char *filename, spDVector vector)
{
    long lc = -1;
    FILE *fp;

    if (NULL != (fp = spOpenFile(filename, "w"))) {
        for (lc = 0; lc < vector->length; lc++) {
            fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f\n", vector->data[lc]);
        }

        /* close file */
        spCloseFile(fp);
    }

    if (lc > 0) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

float fvmaxamp(spFVector vec)
{
    float value;
    
    value = MAX(FABS(fvmax(vec, NULL)), FABS(fvmin(vec, NULL)));
    
    return value;
}

double dvmaxamp(spDVector vec)
{
    double value;
    
    value = MAX(FABS(dvmax(vec, NULL)), FABS(dvmin(vec, NULL)));
    
    return value;
}

float fvadjustamp(spFVector vec, float amp)
{
    float value;

    value = fvmaxamp(vec);
    if (value > 0.0) {
	/*fvscoper(vec, "*", amp / value);*/
	fvscoper(vec, "/", value / amp);
    }
    
    return value;
}

double dvadjustamp(spDVector vec, double amp)
{
    double value;

    value = dvmaxamp(vec);
    if (value > 0.0) {
	/*dvscoper(vec, "*", amp / value);*/
	dvscoper(vec, "/", value / amp);
    }

    return value;
}

float fvlimitamp(spFVector vec, float amp)
{
    float value;

    value = fvmaxamp(vec);
    if (value > 0.0 && value > amp) {
	spWarning("power is too big: %f\n", value);
	spWarning("execute normalization\n");
	fvscoper(vec, "*", amp / value);
    }

    return value;
}

double dvlimitamp(spDVector vec, double amp)
{
    double value;

    value = dvmaxamp(vec);
    if (value > 0.0 && value > amp) {
	spWarning("power is too big: %f\n", value);
	spWarning("execute normalization\n");
	dvscoper(vec, "*", amp / value);
    }

    return value;
}

spFVector xfvextractchannel(spFVector x, int channel, int num_channel)
{
    long k;
    spFVector y;

    if (x == NODATA || num_channel <= 0) return NODATA;
    
    if (channel < 0 || channel >= num_channel) {
	channel = 0;
    }
    
    y = xfvalloc(x->length / (long)num_channel);
    
    for (k = 0; k < y->length; k++) {
	y->data[k] = x->data[k * num_channel + channel];
    }

    return y;
}

spDVector xdvextractchannel(spDVector x, int channel, int num_channel)
{
    long k;
    spDVector y;

    if (x == NODATA || num_channel <= 0) return NODATA;
    
    if (channel < 0 || channel >= num_channel) {
	channel = 0;
    }
    
    y = xdvalloc(x->length / (long)num_channel);
    
    for (k = 0; k < y->length; k++) {
	y->data[k] = x->data[k * num_channel + channel];
    }

    return y;
}

/*
 *	dump data
 */
void svdump(spSVector vec)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
	    printf("%d\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0) {
		printf("%d - %di\n", vec->data[k], -vec->imag[k]);
	    } else {
		printf("%d + %di\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    printf("\n");

    return;
}

void lvdump(spLVector vec)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
	    printf("%ld\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0) {
		printf("%ld - %ldi\n", vec->data[k], -vec->imag[k]);
	    } else {
		printf("%ld + %ldi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    printf("\n");

    return;
}

void fvdump(spFVector vec)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0.0*/) {
	    printf("%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0.0) {
		printf("%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "fi\n", vec->data[k], -vec->imag[k]);
	    } else {
		printf("%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "fi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    printf("\n");

    return;
}

void dvdump(spDVector vec)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0.0*/) {
	    printf("%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0.0) {
		printf("%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi\n", vec->data[k], -vec->imag[k]);
	    } else {
		printf("%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    printf("\n");

    return;
}

void svfdump(spSVector vec, FILE *fp)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
	    fprintf(fp, "%d\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0) {
		fprintf(fp, "%d - %di\n", vec->data[k], -vec->imag[k]);
	    } else {
		fprintf(fp, "%d + %di\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    fprintf(fp, "\n");

    return;
}

void lvfdump(spLVector vec, FILE *fp)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
	    fprintf(fp, "%ld\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0) {
		fprintf(fp, "%ld - %ldi\n", vec->data[k], -vec->imag[k]);
	    } else {
		fprintf(fp, "%ld + %ldi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    fprintf(fp, "\n");

    return;
}

void fvfdump(spFVector vec, FILE *fp)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0.0*/) {
	    fprintf(fp, "%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0.0) {
		fprintf(fp, "%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "fi\n", vec->data[k], -vec->imag[k]);
	    } else {
		fprintf(fp, "%." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_FLOAT_DECIMAL_POINTS "fi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    fprintf(fp, "\n");

    return;
}

void dvfdump(spDVector vec, FILE *fp)
{
    long k;

    if (vec == NODATA) return;

    for (k = 0; k < vec->length; k++) {
	if (vec->imag == NULL /*|| vec->imag[k] == 0.0*/) {
	    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f\n", vec->data[k]);
	} else {
	    if (vec->imag[k] < 0.0) {
		fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi\n", vec->data[k], -vec->imag[k]);
	    } else {
		fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi\n", vec->data[k], vec->imag[k]);
	    }
	}
    }
    fprintf(fp, "\n");

    return;
}

void svnfdump(FILE *fp, /*spSVector vec,*/ ...)
{
    int nrow, ncol;
    int end;
    char buf[SP_MAX_LINE];
    char line[SP_MAX_MESSAGE];
    spSVector p;
    va_list argp;

    end = 0;
    nrow = 0;
    while (1) {
	line[0] = NUL;

	va_start(argp, /*vec*/fp);

	ncol = 0;
	while ((p = va_arg(argp, spSVector)) != NULL) {
	    if (nrow >= p->length) {
		end = 1;
		break;
	    } else {
#if defined(HAVE_SPRINTF_S)
		sprintf_s(buf, sizeof(buf), "%d ", p->data[nrow]);
#elif defined(HAVE_SNPRINTF)
		snprintf(buf, sizeof(buf), "%d ", p->data[nrow]);
#else
		sprintf(buf, "%d ", p->data[nrow]);
#endif
		spStrCat(line, sizeof(line), buf);
	    }
	    ncol++;
	}

	va_end(argp);

	if (end == 1) {
	    break;
	} else {
	    fprintf(fp, "%s\n", line);
	}
	nrow++;
    }

    return;
}

void lvnfdump(FILE *fp, /*spLVector vec,*/ ...)
{
    int nrow, ncol;
    int end;
    char buf[SP_MAX_LINE];
    char line[SP_MAX_MESSAGE];
    spLVector p;
    va_list argp;

    end = 0;
    nrow = 0;
    while (1) {
	line[0] = NUL;

	va_start(argp, /*vec*/fp);

	ncol = 0;
	while ((p = va_arg(argp, spLVector)) != NULL) {
	    if (nrow >= p->length) {
		end = 1;
		break;
	    } else {
#if defined(HAVE_SPRINTF_S)
		sprintf_s(buf, sizeof(buf), "%ld ", p->data[nrow]);
#elif defined(HAVE_SNPRINTF)
		snprintf(buf, sizeof(buf), "%ld ", p->data[nrow]);
#else
		sprintf(buf, "%ld ", p->data[nrow]);
#endif
		spStrCat(line, sizeof(line), buf);
	    }
	    ncol++;
	}

	va_end(argp);

	if (end == 1) {
	    break;
	} else {
	    fprintf(fp, "%s\n", line);
	}
	nrow++;
    }

    return;
}

void fvnfdump(FILE *fp, /*spFVector vec,*/ ...)
{
    int nrow, ncol;
    int end;
    char buf[SP_MAX_LINE];
    char line[SP_MAX_MESSAGE];
    spFVector p;
    va_list argp;

    end = 0;
    nrow = 0;
    while (1) {
	line[0] = NUL;

	va_start(argp, /*vec*/fp);

	ncol = 0;
	while ((p = va_arg(argp, spFVector)) != NULL) {
	    if (nrow >= p->length) {
		end = 1;
		break;
	    } else {
#if defined(HAVE_SPRINTF_S)
		sprintf_s(buf, sizeof(buf), "%f ", p->data[nrow]);
#elif defined(HAVE_SNPRINTF)
		snprintf(buf, sizeof(buf), "%f ", p->data[nrow]);
#else
		sprintf(buf, "%f ", p->data[nrow]);
#endif
		spStrCat(line, sizeof(line), buf);
	    }
	    ncol++;
	}

	va_end(argp);

	if (end == 1) {
	    break;
	} else {
	    fprintf(fp, "%s\n", line);
	}
	nrow++;
    }

    return;
}

void dvnfdump(FILE *fp, /*spDVector vec,*/ ...)
{
    int nrow, ncol;
    int end;
    char buf[SP_MAX_LINE];
    char line[SP_MAX_MESSAGE];
    spDVector p;
    va_list argp;

    end = 0;
    nrow = 0;
    while (1) {
	line[0] = NUL;

	va_start(argp, /*vec*/fp);

	ncol = 0;
	while ((p = va_arg(argp, spDVector)) != NULL) {
	    if (nrow >= p->length) {
		end = 1;
		break;
	    } else {
#if defined(HAVE_SPRINTF_S)
		sprintf_s(buf, sizeof(buf), "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f ", p->data[nrow]);
#elif defined(HAVE_SNPRINTF)
		snprintf(buf, sizeof(buf), "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f ", p->data[nrow]);
#else
		sprintf(buf, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f ", p->data[nrow]);
#endif
		spStrCat(line, sizeof(line), buf);
	    }
	    ncol++;
	}

	va_end(argp);

	if (end == 1) {
	    break;
	} else {
	    fprintf(fp, "%s\n", line);
	}
	nrow++;
    }

    return;
}

void svsfdump(spSVectors vecs, FILE *fp)
{
    long n, k;
    long max_length;
    spSVector vec;

    if (vecs == NODATA || vecs->num_vector <= 0) return;

    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
            
    for (k = 0; k < max_length; k++) {
        for (n = 0; n < vecs->num_vector; n++) {
            vec = vecs->vector[n];
            if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
                if (k < vec->length) {
                    fprintf(fp, "%d  ", vec->data[k]);
                } else {
                    fprintf(fp, "%d  ", 0);
                }
            } else {
                if (k < vec->length) {
                    if (vec->imag[k] < 0) {
                        fprintf(fp, "%d - %di  ", vec->data[k], -vec->imag[k]);
                    } else {
                        fprintf(fp, "%d + %di  ", vec->data[k], vec->imag[k]);
                    }
                } else {
                    fprintf(fp, "%d + %di  ", 0, 0);
                }
            }
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void lvsfdump(spLVectors vecs, FILE *fp)
{
    long n, k;
    long max_length;
    spLVector vec;

    if (vecs == NODATA || vecs->num_vector <= 0) return;

    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
            
    for (k = 0; k < max_length; k++) {
        for (n = 0; n < vecs->num_vector; n++) {
            vec = vecs->vector[n];
            if (vec->imag == NULL /*|| vec->imag[k] == 0*/) {
                if (k < vec->length) {
                    fprintf(fp, "%ld  ", vec->data[k]);
                } else {
                    fprintf(fp, "%ld  ", 0L);
                }
            } else {
                if (k < vec->length) {
                    if (vec->imag[k] < 0) {
                        fprintf(fp, "%ld - %ldi  ", vec->data[k], -vec->imag[k]);
                    } else {
                        fprintf(fp, "%ld + %ldi  ", vec->data[k], vec->imag[k]);
                    }
                } else {
                    fprintf(fp, "%ld + %ldi  ", 0L, 0L);
                }
            }
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void fvsfdump(spFVectors vecs, FILE *fp)
{
    long n, k;
    long max_length;
    spFVector vec;

    if (vecs == NODATA || vecs->num_vector <= 0) return;

    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
            
    for (k = 0; k < max_length; k++) {
        for (n = 0; n < vecs->num_vector; n++) {
            vec = vecs->vector[n];
            if (vec->imag == NULL /*|| vec->imag[k] == 0.0f*/) {
                if (k < vec->length) {
                    fprintf(fp, "%f  ", vec->data[k]);
                } else {
                    fprintf(fp, "%f  ", 0.0f);
                }
            } else {
                if (k < vec->length) {
                    if (vec->imag[k] < 0.0f) {
                        fprintf(fp, "%f - %fi  ", vec->data[k], -vec->imag[k]);
                    } else {
                        fprintf(fp, "%f + %fi  ", vec->data[k], vec->imag[k]);
                    }
                } else {
                    fprintf(fp, "%f + %fi  ", 0.0f, 0.0f);
                }
            }
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void dvsfdump(spDVectors vecs, FILE *fp)
{
    long n, k;
    long max_length;
    spDVector vec;

    if (vecs == NODATA || vecs->num_vector <= 0) return;

    max_length = vecs->vector[0]->length;
    for (n = 1; n < vecs->num_vector; n++) {
        if (max_length < vecs->vector[n]->length) {
            max_length = vecs->vector[n]->length;
        }
    }
            
    for (k = 0; k < max_length; k++) {
        for (n = 0; n < vecs->num_vector; n++) {
            vec = vecs->vector[n];
            if (vec->imag == NULL /*|| vec->imag[k] == 0.0*/) {
                if (k < vec->length) {
                    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f  ", vec->data[k]);
                } else {
                    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f  ", 0.0);
                }
            } else {
                if (k < vec->length) {
                    if (vec->imag[k] < 0.0) {
                        fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi  ", vec->data[k], -vec->imag[k]);
                    } else {
                        fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi  ", vec->data[k], vec->imag[k]);
                    }
                } else {
                    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi  ", 0.0, 0.0);
                }
            }
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void smfdump(spSMatrix mat, FILE *fp)
{
    long k, l;

    if (mat == NODATA) return;

    for (k = 0; k < mat->row; k++) {
	for (l = 0; l < mat->col; l++) {
	    if (mat->imag == NULL /*|| mat->imag[k][l] == 0*/) {
		fprintf(fp, "%d  ", mat->data[k][l]);
	    } else {
		if (mat->imag[k][l] < 0) {
		    fprintf(fp, "%d - %di  ", mat->data[k][l], -mat->imag[k][l]);
		} else {
		    fprintf(fp, "%d + %di  ", mat->data[k][l], mat->imag[k][l]);
		}
	    }
	}
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void lmfdump(spLMatrix mat, FILE *fp)
{
    long k, l;

    if (mat == NODATA) return;

    for (k = 0; k < mat->row; k++) {
	for (l = 0; l < mat->col; l++) {
	    if (mat->imag == NULL /*|| mat->imag[k][l] == 0*/) {
		fprintf(fp, "%ld  ", mat->data[k][l]);
	    } else {
		if (mat->imag[k][l] < 0) {
		    fprintf(fp, "%ld - %ldi  ", mat->data[k][l], -mat->imag[k][l]);
		} else {
		    fprintf(fp, "%ld + %ldi  ", mat->data[k][l], mat->imag[k][l]);
		}
	    }
	}
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void fmfdump(spFMatrix mat, FILE *fp)
{
    long k, l;

    if (mat == NODATA) return;

    for (k = 0; k < mat->row; k++) {
	for (l = 0; l < mat->col; l++) {
	    if (mat->imag == NULL /*|| mat->imag[k][l] == 0.0*/) {
		fprintf(fp, "%f  ", mat->data[k][l]);
	    } else {
		if (mat->imag[k][l] < 0.0f) {
		    fprintf(fp, "%f - %fi  ", mat->data[k][l], -mat->imag[k][l]);
		} else {
		    fprintf(fp, "%f + %fi  ", mat->data[k][l], mat->imag[k][l]);
		}
	    }
	}
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void dmfdump(spDMatrix mat, FILE *fp)
{
    long k, l;

    if (mat == NODATA) return;

    for (k = 0; k < mat->row; k++) {
	for (l = 0; l < mat->col; l++) {
	    if (mat->imag == NULL /*|| mat->imag[k][l] == 0.0*/) {
		fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f  ", mat->data[k][l]);
	    } else {
		if (mat->imag[k][l] < 0.0) {
		    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f - %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi  ", mat->data[k][l], -mat->imag[k][l]);
		} else {
		    fprintf(fp, "%." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "f + %." SP_TEXT_FORMAT_DOUBLE_DECIMAL_POINTS "fi  ", mat->data[k][l], mat->imag[k][l]);
		}
	    }
	}
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

spBool dmcsvwrite(char *filename, spDMatrix A, int decimal_point, spBool transpose)
{
    long i, j;
    char format[SP_MAX_LINE];
    FILE *fp;

    if ((fp = spOpenFile(filename, "wt")) == NULL) {
        return SP_FALSE;
    }

    sprintf(format, "%%.%df", decimal_point);

    if (transpose) {
        for (i = 0; i < A->col; i++) {
            for (j = 0;; j++) {
                fprintf(fp, format, A->data[j][i]);
                if (j == A->row - 1) {
                    fprintf(fp, "\n");
                    break;
                } else {
                    fprintf(fp, ",");
                }
            }
        }
    } else {
        for (i = 0; i < A->row; i++) {
            for (j = 0;; j++) {
                fprintf(fp, format, A->data[i][j]);
                if (j == A->col - 1) {
                    fprintf(fp, "\n");
                    break;
                } else {
                    fprintf(fp, ",");
                }
            }
        }
    }

    return SP_TRUE;
}

#if 1
#include <sp/spWave.h>
#include <sp/spInputPlugin.h>

#define SP_AUDIO_FILE_BUFFER_LENGTH 8192

spBool dvsupportaudiofile(const char *filename, const char *plugin_name)
{
    spPlugin *i_plugin;
    spWaveInfo wave_info;
    
    spInitWaveInfo(&wave_info);
    
    if ((i_plugin = spOpenFilePluginArg(NULL, filename, "r",
					SP_PLUGIN_DEVICE_FILE,
					&wave_info, NULL, 0, NULL, NULL)) == NULL) {
	if (strnone(plugin_name)
	    || (i_plugin = spOpenFilePluginArg(plugin_name, filename, "r",
					       SP_PLUGIN_DEVICE_FILE,
					       &wave_info, NULL, 0, NULL, NULL)) == NULL) {
	    return SP_FALSE;
	}
    }

    spCloseFilePlugin(i_plugin);

    return SP_TRUE;
}

spDVector xdvreadaudiofileex(const char *filename, const char *plugin_name,
                           double *samp_freq, int *samp_bit, int *num_channel, double weight,
                           spDVectorCallbackFunc func, void *data)
{
    spPlugin *i_plugin;
    spWaveInfo wave_info;
    spDVector x;
    long length;
    long nread;
    long total_read;
    long total_length;
    
    spInitWaveInfo(&wave_info);
    if (samp_freq != NULL) wave_info.samp_rate = *samp_freq;
    if (samp_bit != NULL) wave_info.samp_bit = *samp_bit;
    if (num_channel != NULL) wave_info.num_channel = *num_channel;
    
    if ((i_plugin = spOpenFilePluginArg(NULL, filename, "r",
					SP_PLUGIN_DEVICE_FILE,
					&wave_info, NULL, 0, NULL, NULL)) == NULL) {
	if (strnone(plugin_name)
	    || (i_plugin = spOpenFilePluginArg(plugin_name, filename, "r",
					       SP_PLUGIN_DEVICE_FILE,
					       &wave_info, NULL, 0, NULL, NULL)) == NULL) {
	    return NODATA;
	}
    }

    total_length = (long)wave_info.length * (long)wave_info.num_channel;
    length = SP_AUDIO_FILE_BUFFER_LENGTH;
    x = xdvalloc(total_length > 0 ? total_length : length);

    if (func != NULL) {
        if (func(x, SP_DVECTOR_READ_STARTED_CALLBACK, 0, data) == SP_FALSE) {
            return x;
        }
    }

    for (total_read = 0;;) {
	if ((nread = spReadPluginDoubleWeighted(i_plugin, x->data + total_read, MIN(length, x->length - total_read), weight)) <= 0) {
	    break;
	}
	total_read += nread;

        if (func != NULL) {
            if (func(x, SP_DVECTOR_READ_PROGRESS_CALLBACK, (void *)total_read, data) == SP_FALSE) {
                return x;
            }
        }
        
	if (total_read + length > x->length) {
	    xdvrealloc(x, x->length + length);
	}
    }

    x->length = total_read;
    
    spCloseFilePlugin(i_plugin);

    if (x->length <= 0) {
	xdvfree(x);
	x = NODATA;
    } else {
        if (func != NULL) {
            func(x, SP_DVECTOR_READ_FINISHED_CALLBACK, (void *)x->length, data);
        }
        
	if (samp_freq != NULL) *samp_freq = wave_info.samp_rate;
	if (samp_bit != NULL) *samp_bit = wave_info.samp_bit;
	if (num_channel != NULL) *num_channel = wave_info.num_channel;
    }

    return x;
}

spDVector xdvreadaudiofile(const char *filename, const char *plugin_name,
			 double *samp_freq, int *samp_bit, int *num_channel, double weight)
{
    return xdvreadaudiofileex(filename, plugin_name, samp_freq, samp_bit, num_channel, weight, NULL, NULL);
}

long dvwriteaudiofileex(spDVector x, const char *filename, const char *plugin_name,
                        double samp_freq, int samp_bit, int num_channel, double weight,
                        spDVectorCallbackFunc func, void *data)
{
    spPlugin *o_plugin;
    spWaveInfo wave_info;
    long nwrite;
    long total_write;
    
    spInitWaveInfo(&wave_info);
    wave_info.samp_rate = samp_freq;
    wave_info.samp_bit = samp_bit;
    wave_info.num_channel = num_channel;
    wave_info.length = x->length / MAX(num_channel, 1);
    
    if ((o_plugin = spOpenFilePluginArg(plugin_name, filename, "w",
					SP_PLUGIN_DEVICE_FILE,
					&wave_info, NULL, 0, NULL, NULL)) == NULL) {
	return -1;
    }

    if (func != NULL) {
        if (func(x, SP_DVECTOR_WRITE_STARTED_CALLBACK, 0, data) == SP_FALSE) {
            return 0;
        }
    }

    for (total_write = 0; total_write < x->length;) {
        if ((nwrite = spWritePluginDoubleWeighted(o_plugin, x->data + total_write, MIN(x->length - total_write, SP_AUDIO_FILE_BUFFER_LENGTH), weight)) < 0) {
            break;
        }
        total_write += nwrite;
        
        if (func != NULL) {
            if (func(x, SP_DVECTOR_WRITE_PROGRESS_CALLBACK, (void *)total_write, data) == SP_FALSE) {
                return total_write;
            }
        }
    }
    
    spCloseFilePlugin(o_plugin);
    
    if (func != NULL) {
        func(x, SP_DVECTOR_WRITE_FINISHED_CALLBACK, (void *)total_write, data);
    }
        
    return total_write;
}

long dvwriteaudiofile(spDVector x, const char *filename, const char *plugin_name,
		      double samp_freq, int samp_bit, int num_channel, double weight)
{
    return dvwriteaudiofileex(x, filename, plugin_name, samp_freq, samp_bit, num_channel, weight, NULL, NULL);
}
#endif
