/*
 *	filter.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/voperate.h>
#include <sp/fft.h>
#include <sp/matrix.h>
#include <sp/kaiser.h>
#include <sp/window.h>
#include <sp/fileio.h>
#include <sp/filter.h>

void fvunwrap(spFVector phs, float cutoff)
{
    long k;
    float a;
    float twopi;
    spFVector f;
    
    if (cutoff <= 0.0f) {
	cutoff = (float)PI;
    }

    twopi = (float)(2.0 * PI);
    a = fvmin(phs, NULL);
    for (k = 0; k < phs->length; k++) {
	phs->data[k] = (float)rem(phs->data[k] - a, twopi) + a;
    }

    f = xfvzeros(phs->length);
    for (k = 1; k < f->length; k++) {
	a = phs->data[k] - phs->data[k - 1];
	if (a > cutoff) {
	    f->data[k] = -twopi;
	} else if (a < -cutoff) {
	    f->data[k] = twopi;
	}
    }

    fvcumsum(f);
    fvoper(phs, "+", f);

    xfvfree(f);

    return;
}

void dvunwrap(spDVector phs, double cutoff)
{
    long k;
    double a;
    double twopi;
    spDVector f;
    
    if (cutoff <= 0.0) {
	cutoff = PI;
    }

    twopi = 2.0 * PI;
    a = dvmin(phs, NULL);
    for (k = 0; k < phs->length; k++) {
	phs->data[k] = rem(phs->data[k] - a, twopi) + a;
    }

    f = xdvzeros(phs->length);
    for (k = 1; k < f->length; k++) {
	a = phs->data[k] - phs->data[k - 1];
	if (a > cutoff) {
	    f->data[k] = -twopi;
	} else if (a < -cutoff) {
	    f->data[k] = twopi;
	}
    }

    dvcumsum(f);
    dvoper(phs, "+", f);

    xdvfree(f);

    return;
}

float sincf(float x)
{
    float a;

    if (x == 0.0f) {
	a = 1.0f;
    } else {
	a = sinf(x) / x;
    }
    return a;
}

double sinc(double x)
{
    double a;

    if (x == 0.0) {
	a = 1.0;
    } else {
	a = sin(x) / x;
    }
    return a;
}

float sinccf(float x, float c)
{
    float a;

    if (x == 0.0f) {
	a = c;
    } else {
	a = (c * sinf(x)) / x;
    }
    return a;
}

double sincc(double x, double c)
{
    double a;

    if (x == 0.0) {
	a = c;
    } else {
	a = (c * sin(x)) / x;
    }
    return a;
}

spDVector xdvlowhighpass(double hp_cutoff, double lp_cutoff, double sidelobe, double trans, double gain, int bandstop)
{
    long k;
    long half, length;
    double beta;
    double lp_value, hp_value;
    spDVector filter;

    /* get kaiser parameter */
    getkaiserparam(sidelobe, trans, &beta, &length);

    /* make sure length is odd */
    length = (long)(length / 2) * 2 + 1;
    half = (length - 1) / 2;

    spDebug(10, "xdvbandpass", "lp_cutoff = %f, hp_cutoff = %f, gain = %f, beta = %f, length = %ld\n",
	    lp_cutoff, hp_cutoff, gain, beta, length);

    /* memory allocate */
    filter = xdvalloc(length);

    /* get kaiser window */
    if (kaiser(filter->data, filter->length, beta) == SP_FAILURE) {
	xdvfree(filter);
	return NULL;
    }

    if (hp_cutoff >= 0.0) {
	for (k = 0; k < length; k++) {
	    /* calculate highpass filter */
	    if ((bandstop || lp_cutoff < 0.0) && /*lp_cutoff < hp_cutoff &&*/ k == half) {
		hp_value = 1.0 - hp_cutoff;
	    } else {
		hp_value = -sincc(PI * hp_cutoff * (double)(k - half), hp_cutoff);
	    }
	    
	    if (lp_cutoff < 0.0) {
		lp_value = 0.0;
	    } else {
		/* calculate lowpass filter */
		lp_value = sincc(PI * lp_cutoff * (double)(k - half), lp_cutoff);
	    }
	    filter->data[k] *= (hp_value + lp_value) * gain;
	    spDebug(100, "xdvbandpass", "bandpass: filter->data[%ld] = %f\n", k, filter->data[k]);
	}
    } else {
	/* calculate lowpass filter */
	for (k = 0; k < length; k++) {
	    lp_value = sincc(PI * lp_cutoff * (double)(k - half), lp_cutoff);
	    filter->data[k] *= lp_value * gain;
	    spDebug(100, "xdvbandpass", "lowpass: filter->data[%ld] = %f\n", k, filter->data[k]);
	}
    }

    return filter;
}

spDVector xdvbandstop(double lp_cutoff, double hp_cutoff, double sidelobe, double trans, double gain)
{
    return xdvlowhighpass(hp_cutoff, lp_cutoff, sidelobe, trans, gain, 1);
}

spDVector xdvbandpass(double hp_cutoff, double lp_cutoff, double sidelobe, double trans, double gain)
{
    return xdvlowhighpass(hp_cutoff, lp_cutoff, sidelobe, trans, gain, 0);
}

spDVector xdvlowpass(double cutoff, double sidelobe, double trans, double gain)
{
    return xdvbandpass(-1.0, cutoff, sidelobe, trans, gain);
}

spDVector xdvhighpass(double cutoff, double sidelobe, double trans, double gain)
{
    return xdvbandpass(cutoff, -1.0, sidelobe, trans, gain);
}

#define SP_FFT_FILT_MIN_BLOCK_LENGTH 128

#define getnumloop(x,y) MAX((int)(((x) + (y) - 1) / (y)), 1)

spDVector _xdvfftfiltex(spFFTRec fftrec, spDVector b, spDVector x, int truncation_flag)
{
    long i, nloop;
    long k;
    long pos = 0;
    long fftl;
    long length, block;
    long olength;
    double real, imag;
    double *hr, *hi;
    double *xr, *xi;
    spDVector xo;

    fftl = dvgetfftlength(fftrec);
    
    /* initialize */
    length = b->length / 2 - 1;
    block = fftl - b->length + 1;
    if (block <= 0) {
	spDebug(10, "_xdvfftfiltex", "FFT length (%ld) must be larger than filter length (%ld)\n",
		fftl, b->length);
	return NODATA;
    }
    nloop = getnumloop(x->length, block);
    spDebug(50, "_xdvfftfiltex", "fftl = %ld, b->length = %ld, length = %ld, block = %ld, nloop = %ld\n",
	    fftl, b->length, length, block, nloop);

    /* memory allocate */
    hr = xalloc(fftl, double);
    hi = xalloc(fftl, double);
    xr = xalloc(fftl, double);
    xi = xalloc(fftl, double);

    if (truncation_flag) {
	olength = x->length;
    } else {
	olength = x->length + b->length - 1;
    }
    
    if (x->imag != NULL || b->imag != NULL) {
	xo = xdvrizeros(olength);
    } else {
	xo = xdvzeros(olength);
    }

    /* fft of the filter */
    for (k = 0; k < fftl; k++) {
	if (k < b->length) {
	    hr[k] = b->data[k];
	    if (b->imag != NULL) {
		hi[k] = b->imag[k];
	    } else {
		hi[k] = 0.0;
	    }
	} else {
	    hr[k] = 0.0;
	    hi[k] = 0.0;
	}
    }
    dvexecfft(fftrec, hr, hi, 0);

    /* loop for every block of data */
    for (i = 0; i < nloop; i++) {
	/* fft of input speech data */
	for (k = 0; k < fftl; k++) {
	    if (k < block && (pos = i * block + k) < x->length) {
		xr[k] = x->data[pos];
		if (x->imag != NULL) {
		    xi[k] = x->imag[pos];
		} else {
		    xi[k] = 0.0;
		}
	    } else {
		xr[k] = 0.0;
		xi[k] = 0.0;
	    }
	}
	dvexecfft(fftrec, xr, xi, 0);

	/* multiplication in frequency domain */
	for (k = 0; k < fftl; k++) {  
	    real = xr[k] * hr[k] - xi[k] * hi[k];
	    imag = xr[k] * hi[k] + xi[k] * hr[k];
	    xr[k] = real;
	    xi[k] = imag;
	}

	/* ifft */
	dvexecfft(fftrec, xr, xi, 1);

	/* overlap adding */
	for (k = 0; k < fftl && (pos = i * block + k) < xo->length; k++) {
	    if (pos >= 0) {
		xo->data[pos] += xr[k];
		if (xo->imag != NULL) {
		    xo->imag[pos] += xi[k];
		}
	    }
	}
    }

    /* memory free */
    free(hr);
    free(hi);
    free(xr);
    free(xi);

    return xo;
}

spDVector xdvfftfiltex(spFFTRec fftrec, spDVector b, spDVector x)
{
    /*if (dvgetfftlength(fftrec) >= 2 * b->length) {*/
    if (dvgetfftlength(fftrec) >= SP_FFT_FILT_MIN_BLOCK_LENGTH + b->length - 1) {
	return _xdvfftfiltex(fftrec, b, x, 0);
    } else {
	return xdvfftfilt(b, x, dvgetfftlength(fftrec));
    }
}

spDVector xdvfftfiltmex(spFFTRec fftrec, spDVector b, spDVector x)
{
    /*if (dvgetfftlength(fftrec) >= 2 * b->length) {*/
    if (dvgetfftlength(fftrec) >= SP_FFT_FILT_MIN_BLOCK_LENGTH + b->length - 1) {
	return _xdvfftfiltex(fftrec, b, x, 1);
    } else {
	return xdvfftfiltm(b, x, dvgetfftlength(fftrec));
    }
}

spDVector _xdvfftfilt(spDVector b, spDVector x, long fftl, int truncation_flag)
{
    long fftl_p;
    long length;
    spFFTRec fftrec;
    spDVector y = NODATA;
    
    /* get fft point */
    if (x->length < b->length) {
	length = b->length + x->length - 1;
    } else {
	length = SP_FFT_FILT_MIN_BLOCK_LENGTH + b->length - 1;
    }
    /*fftl_p = nextpow2(MAX(fftl, 2 * length));*/
    fftl_p = nextpow2(MAX(fftl, length));
    fftl = POW2(fftl_p);
	
    if ((fftrec = dvinitfft(fftl_p, SP_FFT_DOUBLE_PRECISION)) != NULL) {
	y = _xdvfftfiltex(fftrec, b, x, truncation_flag);
	dvfreefft(fftrec);
    }

    return y;
}

spDVector xdvfftfilt(spDVector b, spDVector x, long fftl)
{
    return _xdvfftfilt(b, x, fftl, 0);
}

spDVector xdvfftfiltm(spDVector b, spDVector x, long fftl)
{
    return _xdvfftfilt(b, x, fftl, 1);
}

spDVector xdvconv(spDVector a, spDVector b)
{
    long i, j, pos;
    double ar, ai;
    double br, bi;
    double sum, sumi;
    spDVector c;

    if (a->imag != NULL || b->imag != NULL) {
	c = xdvrizeros(a->length + b->length - 1);
    } else {
	c = xdvzeros(a->length + b->length - 1);
    }

    for (i = 0; i < c->length; i++) {
	if (c->imag != NULL) {
	    for (j = 0, sum = sumi = 0.0; j < a->length; j++) {
		pos = i - j;
		if (pos >= 0 && pos < b->length) {
		    ar = a->data[j];
		    br = b->data[pos];
		    if (a->imag == NULL) {
			ai = 0.0;
		    } else {
			ai = a->imag[j];
		    }
		    if (b->imag == NULL) {
			bi = 0.0;
		    } else {
			bi = b->imag[pos];
		    }
		    sum += ar * br - ai * bi;
		    sumi += ar * bi + ai * br;
		}
	    }
	    c->data[i] = sum;
	    c->imag[i] = sumi;
	} else {
	    for (j = 0, sum = 0.0; j < a->length; j++) {
		pos = i - j;
		if (pos >= 0 && pos < b->length) {
		    sum += a->data[j] * b->data[pos];
		}
	    }
	    c->data[i] = sum;
	}
    }

    return c;
}

spDVector xdvinitfilterstates(spDVector b, spDVector a)
{
    if (b == NODATA || a == NODATA) return NODATA;
    
    return xdvzeros(MAX(b->length, a->length) - 1);
}

/* length of z must be larger than MAX(length of b, length of a)-1 */
void dvfiltersamplecplx(spDVector b, spDVector a, spDVector z, double *re, double *im)
{
    long n, i, j;
    double xr, xi;
    double yr, yi;

    if (b == NODATA || a == NODATA || z == NODATA || z->imag == NULL
	|| re == NULL || im == NULL) return;

    n = MAX(b->length, a->length) - 1;
    if (z->length < n) return;
    
    xr = *re;
    xi = *im;

    if (b->imag == NULL) {
	yr = b->data[0] * xr + z->data[0];
	yi = z->imag[0];
    } else {
	yr = b->data[0] * xr - b->imag[0] * xi + z->data[0];
	yi = b->data[0] * xi + b->imag[0] * xr + z->imag[0];
    }

    *re = yr;
    *im = yi;
    
    for (i = 0; i < n; i++) {
	j = i + 1;
	if (j < n) {
	    z->data[i] = z->data[j];
	    z->imag[i] = z->imag[j];
	} else {
	    z->data[i] = 0.0;
	    z->imag[i] = 0.0;
	}
	if (j < b->length) {
	    if (b->imag == NULL) {
		z->data[i] += b->data[j] * xr;
	    } else {
		z->data[i] += (b->data[j] * xr - b->imag[j] * xi);
		z->imag[i] += (b->data[j] * xi + b->imag[j] * xr);
	    }
	}
	if (j < a->length) {
	    if (a->imag == NULL) {
		z->data[i] -= a->data[j] * yr;
	    } else {
		z->data[i] -= (a->data[j] * yr - a->imag[j] * yi);
		z->imag[i] -= (a->data[j] * yi + a->imag[j] * yr);
	    }
	}
    }
    for (; i < z->length; i++) {
	z->data[i] = 0.0;
	z->imag[i] = 0.0;
    }

    return;
}

double filtersample(double *b, long blen, double *a, long alen, double *z, long zlen, double x)
{
    long n, i, j;
    double y;

    n = MAX(blen, alen) - 1;
    if (zlen < n) return 0.0;
    
    y = b[0] * x + z[0];

    for (i = 0; i < n; i++) {
	j = i + 1;
	if (j < n) {
	    z[i] = z[j];
	} else {
	    z[i] = 0.0;
	}
	if (j < blen) {
	    z[i] += b[j] * x;
	}
	if (j < alen) {
	    z[i] -= a[j] * y;
	}
    }
    for (; i < zlen; i++) {
	z[i] = 0.0;
    }
    
    return y;
}

double dvfiltersample(spDVector b, spDVector a, spDVector z, double x)
{
    if (b == NODATA || a == NODATA || z == NODATA) return 0.0;
    
    return filtersample(b->data, b->length, a->data, a->length, z->data, z->length, x);
}

spDVector _xdvfilterex(spDVector b, spDVector a, spDVector z, spDVector x, spBool cplx_flag)
{
    long n;
    spDVector xo;
    
    /* normalize filter-coefficient */
    if (a->data[0] != 1.0) {
	spWarning("normalize filter-coefficient\n");
	dvscoper(b, "/", a->data[0]);
	dvscoper(a, "/", a->data[0]);
    }
    
    if (cplx_flag) {
	xo = xdvrizeros(x->length);
	
	for (n = 0; n < xo->length; n++) {
	    xo->data[n] = x->data[n];
	    if (x->imag == NULL) {
		xo->imag[n] = 0.0;
	    } else {
		xo->imag[n] = x->imag[n];
	    }
	    dvfiltersamplecplx(b, a, z, &xo->data[n], &xo->imag[n]);
	}
    } else {
	xo = xdvzeros(x->length);
	
	for (n = 0; n < xo->length; n++) {
	    xo->data[n] = dvfiltersample(b, a, z, x->data[n]);
	}
    }

    return xo;
}

spDVector xdvfilterex(spDVector b, spDVector a, spDVector z, spDVector x)
{
    spBool cplx_flag = SP_FALSE;
    
    if (b == NODATA || a == NODATA || z == NODATA || x == NODATA) return NODATA;

    if (a->imag != NULL || b->imag != NULL || x->imag != NULL) {
	cplx_flag = SP_TRUE;
    }

    return _xdvfilterex(b, a, z, x, cplx_flag);
}

spDVector xdvfilter(spDVector b, spDVector a, spDVector x)
{
    long len;
    spDVector z;
    spDVector xo;
    spBool cplx_flag = SP_FALSE;
    
    if (b == NODATA || a == NODATA || x == NODATA) return NODATA;

    len = MAX(b->length, a->length) - 1;

    if (a->imag != NULL || b->imag != NULL || x->imag != NULL) {
	cplx_flag = SP_TRUE;
	z = xdvrizeros(len);
    } else {
	z = xdvzeros(len);
    }

    xo = _xdvfilterex(b, a, z, x, cplx_flag);

    xdvfree(z);
    
    return xo;
}

spDVector xdvfreqzex(spFFTRec fftrec, spDVector b /* can be NODATA */, spDVector a /* can be NODATA */)
{
    spDVector B, A;

    if (b == NODATA && a == NODATA) {
        return NODATA;
    }

    if (a == NODATA) {
        return xdvfftex(fftrec, b);
    } else if (b == NODATA) {
        A = NODATA;
        if ((A = xdvfftex(fftrec, a)) != NODATA) {
            dvscoper(A, "!/", 1.0);
        }
        return A;
    } else {
        A = NODATA;
        if ((B = xdvfftex(fftrec, b)) != NODATA) {
            if ((A = xdvfftex(fftrec, a)) != NODATA) {
                dvoper(A, "!/", B);
            }
            xdvfree(B);
        }
        return A;
    }
}

spDVector xdvfreqz(spDVector b /* can be NODATA */, spDVector a /* can be NODATA */, long fftl /* can be 0 (auto select) */)
{
    long length;
    long fftl_p;
    spFFTRec fftrec;
    spDVector H = NODATA;

    if (b == NODATA && a == NODATA) {
        return NODATA;
    }

    if (fftl <= 0) {
        if (b == NODATA) {
            length = a->length;
        } else if (a == NODATA) {
            length = b->length;
        } else {
            length = MAX(b->length, a->length);
        }
        fftl = 2 * length;
    }
    fftl_p = nextpow2(fftl);
    
    if ((fftrec = dvinitfft(fftl_p, SP_FFT_DOUBLE_PRECISION)) != NULL) {
        H = xdvfreqzex(fftrec, b, a);
	dvfreefft(fftrec);
    }

    return H;
}

#if 1
int convfile(const char *a_filename, const char *b_filename, int i_swap, const char *o_filename, int o_swap,
	     double coef)
{
    long k, l;
    FILE *a_fp, *b_fp, *o_fp;
    int nwrite;
    long pos;
    long num_skip;
    long a_length, b_length, c_length;
    double a, b, c;

    a_length = (long)getsiglen(a_filename, 0, short);
    b_length = (long)getsiglen(b_filename, 0, short);
    
    /* open file */
    if (NULL == (a_fp = spOpenFile(a_filename, "rb"))) {
	return SP_FAILURE;
    }

    /* open file */
    if (NULL == (b_fp = spOpenFile(b_filename, "rb"))) {
	return SP_FAILURE;
    }

    /* open file */
    if (NULL == (o_fp = spOpenFile(o_filename, "wb"))) {
	return SP_FAILURE;
    }

    c_length = a_length + b_length - 1;
    
    for (k = 0; k < c_length; k++) {
	/*printf("k = %ld\n", k);*/
	
	if (k > 0) {
	    fseek(a_fp, 0, 0);
	    fseek(b_fp, 0, 0);
	    num_skip = fskipdata(MIN(k, b_length), b_fp, short);
	}

	c = 0.0;
	for (l = 0; l < a_length; l++) {
	    freadshorttod(&a, 1, i_swap, a_fp);
	    
	    pos = k - l;
	    if (pos >= 0 && pos < b_length) {
		fseek(b_fp, 0, 0);
		num_skip = fskipdata(pos, b_fp, short);
		freadshorttod(&b, 1, i_swap, b_fp);
		c += a * b;
		/*printf("c = %f, a = %f, b = %f\n", c, a, b);*/
	    }
	}

	if (coef != 0.0) c *= coef;

	nwrite = fwritedoubletos(&c, 1, o_swap, o_fp);
    }
    
    spCloseFile(a_fp);
    spCloseFile(b_fp);
    spCloseFile(o_fp);

    return SP_SUCCESS;
}
#endif

/*
 * 
 * functions implemented by Tomoki Toda
 * 
 */
static void bisearch(double *x,	/* x must be monotonically increasing */
		     long fidx,	/* first index */
		     long lidx,	/* last of x */
		     double a,
		     long *len,
		     long *ansidx)
{
    long idx;

    *ansidx = fidx;
    *len = lidx - fidx + 1;
    if (*len < 2) {
	spWarning("Error: binary search\n");
	return;
    }
    if (*len > 2) {
	idx = *len / 2 + fidx;
	if (x[idx] > a) {
	    bisearch(x, fidx, idx, a, len, ansidx);
	} else if (x[idx] < a) {
	    bisearch(x, idx, lidx, a, len, ansidx);
	} else {
	    *ansidx = idx;
	}
    }
}

/* linear interpolation (X must be monotonically increasing) */
spBool dvinterp1l(spDVector x, spDVector y, spDVector xi, spDVector yi)
{
    long ki, kx, len;
    long x_length;
    long xi_length;

    x_length = MIN(x->length, y->length);
    xi_length = MIN(xi->length, yi->length);

    for (ki = 0, kx = 0; ki < xi_length; ki++) {
	len = x_length;
	bisearch(x->data, 0, x_length - 1, xi->data[ki], &len, &kx);
	spDebug(100, "dvinterp1l",
		"binary search: len = %ld, kx = %ld, x_length = %ld\n",
		len, kx, x_length);
	if (kx >= x_length - 1) {
	    spWarning("binary search: kx (%ld) >= x_length - 1 (%ld)\n",
		      kx, x_length - 1);
	    kx = x_length - 2;
	}
	/* interpolation */
	if (x->data[kx + 1] != x->data[kx]) {
	    spDebug(100, "dvinterp1l", "xi->data[%ld] = %f, x->data[%ld] = %f, y->data[%ld] = %f, y->data[%ld] = %f\n",
		    ki, xi->data[ki], kx + 1, x->data[kx + 1], kx, y->data[kx], kx + 1, y->data[kx + 1]);
	    yi->data[ki] = (x->data[kx + 1] - xi->data[ki]) * y->data[kx] + (xi->data[ki] - x->data[kx]) * y->data[kx + 1];
	    yi->data[ki] /= x->data[kx + 1] - x->data[kx];
	    spDebug(100, "dvinterp1l", "yi->data[%ld] = %f\n", ki, yi->data[ki]);
	} else {
	    yi->data[ki] = y->data[kx];
	}
    }
    
    return SP_TRUE;
}

spDVector xdvinterp1l(spDVector x, spDVector y, spDVector xi)
{
    spDVector yi;
    
    yi = xdvalloc(xi->length);

    dvinterp1l(x, y, xi, yi);

    return yi;
}

/* nearest neighbor interpolation (X must be monotonically increasing) */
spBool dvinterp1n(spDVector x, spDVector y, spDVector xi, spDVector yi)
{
    long ki, kx, len;
    long x_length;
    long xi_length;

    x_length = MIN(x->length, y->length);
    xi_length = MIN(xi->length, yi->length);

    for (ki = 0, kx = 0; ki < xi->length; ki++) {
	len = x_length;
	bisearch(x->data, 0, x_length - 1, xi->data[ki], &len, &kx);
	spDebug(100, "dvinterp1n",
		"binary search: len = %ld, kx = %ld, x_length = %ld\n",
		len, kx, x_length);
	if (kx >= x_length - 1) {
	    spWarning("binary search: kx (%ld) >= x_length - 1 (%ld)\n",
		      kx, x_length - 1);
	}
	/* interpolation */
	if (x->data[kx + 1] != x->data[kx]) {
	    if (fabs(x->data[kx + 1] - xi->data[ki]) < fabs(xi->data[ki] - x->data[kx])) {
		yi->data[ki] = y->data[kx + 1];
	    } else {
		yi->data[ki] = y->data[kx];
	    }
	} else {
	    yi->data[ki] = y->data[kx];
	}
    }
    
    return SP_TRUE;
}

spDVector xdvinterp1n(spDVector x, spDVector y, spDVector xi)
{
    spDVector yi;
    
    yi = xdvalloc(xi->length);

    dvinterp1n(x, y, xi, yi);

    return yi;
}

/* cubic spline interpolation on not-a-knot condition (X must be monotonically increasing) */
spBool dvinterp1s(spDVector x, spDVector y, spDVector xi, spDVector yi)
{
    return dvspline(x, y, xi, yi, SP_TRUE);
}

spDVector xdvinterp1s(spDVector x, spDVector y, spDVector xi)
{
    return xdvspline(x, y, xi, SP_TRUE);
}

/* Resample data at a higher rate using lowpass interpolation */
spDVector xdvinterpex(spDVector x, long r, long l, double a)
{
    long k;
    long bias;
    spDVector izx = NODATA;
    spDVector lpf = NODATA;
    spDVector usxb = NODATA;
    spDVector usx = NODATA;

    izx = xdvzeros(x->length * r);
    for (k = 0; k < x->length; k++) izx->data[k * r] = x->data[k];
    lpf = xdvfir1(2 * l * r, a);
    usxb = xdvfftfilt(lpf, izx, lpf->length * 2);
    usx = xdvalloc(izx->length);
    bias = l * r;
    for (k = 0; k < izx->length; k++)
	usx->data[k] = usxb->data[k + bias] * (double)r;

    /* memory free */
    xdvfree(izx);
    xdvfree(lpf);
    xdvfree(usxb);

    return usx;
}

spDVector xdvinterp(spDVector x, long r)
{
    return xdvinterpex(x, r, 4, 0.5);
}

/*
 * splinesub
 *   reference: ``C gengo ni yoru saishin algorithm jiten'' [in Japanese]
 *   see http://oku.edu.mie-u.ac.jp/~okumura/algo/
 */
static double splinesub(double t, double *x, double *y, double *z, long N)
{
    long i, j, k;
    double d, h;

    i = 0;  j = N - 1;
    while (i < j) {
        k = (i + j) / 2;
        if (x[k] < t) i = k + 1;  else j = k;
    }
    if (i > 0) i--;
    h = x[i + 1] - x[i];  d = t - x[i];
    return (((z[i + 1] - z[i]) * d / h + z[i] * 3) * d
        + ((y[i + 1] - y[i]) / h
        - (z[i] * 2 + z[i + 1]) * h)) * d + y[i];
}

static spBool _dvspline(spDVector x, spDVector y, spDVector xi, spDVector yi,
			spBool clamped, double leftslope, double rightslope, spBool notaknot)
{
    long i;
    long xi_length;
    spDVector h, d, b, z;
    spDMatrix A;

    if (x == NODATA || y == NODATA || xi == NODATA) return SP_FALSE;

    if (y->length < x->length) {
	spWarning("y->length must be equal or larger than x->length.\n");
	return SP_FALSE;
    }

    h = xdvalloc(x->length - 1);
    d = xdvalloc(x->length - 1);

    for (i = 0; i < h->length; i++) {
	h->data[i] = x->data[i + 1] - x->data[i];
	d->data[i] = (y->data[i + 1] - y->data[i]) / h->data[i];
    }

    b = xdvalloc(x->length);
    if (clamped) {
	b->data[0] = d->data[0] - leftslope;
	b->data[b->length - 1] = rightslope - d->data[d->length - 1];
    } else {
	b->data[0] = 0;
	b->data[b->length - 1] = 0;
    }
    for (i = 1; i < h->length; i++) {
	b->data[i] = d->data[i] - d->data[i - 1];
    }
    
    A = xdmzeros(x->length, x->length);
    if (clamped) {
	A->data[0][0] = 2.0 * h->data[0];
	A->data[0][1] = h->data[1];
	A->data[h->length][h->length] = 2.0 * h->data[h->length - 1];
	A->data[h->length][h->length - 1] = h->data[h->length - 1];
    } else if (notaknot) {
	A->data[0][0] = h->data[1];
	A->data[0][1] = -h->data[0] - h->data[1];
	A->data[0][2] = h->data[0];
	A->data[h->length][h->length] = h->data[h->length - 1];
	A->data[h->length][h->length - 1] = -h->data[h->length - 2] - h->data[h->length - 1];
	A->data[h->length][h->length - 2] = h->data[h->length - 2];
    } else {
	A->data[0][0] = 1.0;
	A->data[h->length][h->length] = 1.0;
    }
    for (i = 1; i < h->length; i++) {
	A->data[i][i - 1] = h->data[i - 1];
	A->data[i][i] = 2.0 * (h->data[i - 1] + h->data[i]);
	A->data[i][i + 1] = h->data[i];
    }

    z = xdvmldivide(A, b);

    xi_length = MIN(xi->length, yi->length);
    for (i = 0; i < xi_length; i++) {
	yi->data[i] = splinesub(xi->data[i], x->data, y->data, z->data, x->length);
    }

    xdvfree(h);
    xdvfree(d);
    xdvfree(b);
    xdvfree(z);
    xdmfree(A);

    return SP_TRUE;
}

spBool dvsplinecl(spDVector x, spDVector y, spDVector xi, spDVector yi, double leftslope, double rightslope)
{
    return _dvspline(x, y, xi, yi, SP_TRUE, leftslope, rightslope, SP_FALSE);
}

spDVector xdvsplinecl(spDVector x, spDVector y, spDVector xi, double leftslope, double rightslope)
{
    spDVector yi;
    
    yi = xdvalloc(xi->length);
    
    if (dvsplinecl(x, y, xi, yi, leftslope, rightslope) == SP_FALSE) {
	xdvfree(yi);
	return NODATA;
    }
    
    return yi;
}

spBool dvspline(spDVector x, spDVector y, spDVector xi, spDVector yi, spBool notaknot)
{
    return _dvspline(x, y, xi, yi, SP_FALSE, 0.0, 0.0, notaknot);
}

spDVector xdvspline(spDVector x, spDVector y, spDVector xi, spBool notaknot)
{
    spDVector yi;
    
    yi = xdvalloc(xi->length);

    if (dvspline(x, y, xi, yi, notaknot) == SP_FALSE) {
	xdvfree(yi);
	return NODATA;
    }

    return yi;
}

/* Lowpass FIR digital filter by using specified window */
void dvfir1win(spDVector filter,	/* (i/o) input is window, output is filter */
	       double cutoff)	/* cut-off frequency */
{
    long k, length;
    long order;
    double half;
    double value, a;

    /* 0 <= cufoff <= 1 */
    cutoff = MIN(cutoff, 1.0);
    cutoff = MAX(cutoff, 0.0);

    order = filter->length - 1;
    half = (double)order / 2.0;
    length = order + 1;

    /* calculate lowpass filter */
    for (k = 0; k < length; k++) {
	a = PI * ((double)k - half);
	if (a == 0.0) {
	    value = cutoff;
	} else {
	    value = sin(cutoff * a) / a;
	}
	filter->data[k] *= value;
    }

    return;
}

/* Lowpass FIR digital filter by using specified window */
spDVector xdvfir1win(spDVector win,
		   double cutoff)	/* cut-off frequency */
{
    spDVector filter;

    filter = xdvclone(win);
    dvfir1win(filter, cutoff);

    return filter;
}

/* Lowpass FIR digital filter by using Hamming window */
spDVector xdvfir1(long order,	/* return [length + 1] */
		double cutoff)	/* cut-off frequency */
{
    spDVector filter;

    filter = xdvhamming(order + 1);
    dvfir1win(filter, cutoff);

    return filter;
}

/* Bandpass FIR Filter (startf-endf) by using specified window */
spBool dvfir1bpwin(spDVector filter, /* (i/o) input is window, output is filter */
		   double startf,
		   double endf)
{
    long k;
    long order;
    double half, cutoff;
    double fmf;

    if (startf * endf < 0.0 || startf * endf > 1.0) {
	spWarning("0 <= freq <= 1, freq = %f, %f\n", startf, endf);
	return SP_FALSE;
    } else if (startf > endf) {
	spWarning("start-freq(%f) < end-freq(%f)\n", startf, endf);
	return SP_FALSE;
    }

    order = filter->length - 1;
    cutoff = endf - startf;
    cutoff /= 2.0;
    /* lowpass filter */
    dvfir1win(filter, cutoff);
    /* FM */
    fmf = startf + cutoff;
    half = (double)order / 2.0;
    for (k = 0; k < filter->length; k++) {
	filter->data[k] *= 2.0 * sin(PI * fmf * ((double)k - half + 1.0 / 2.0 / fmf));
    }

    return SP_TRUE;
}

/* Bandpass FIR Filter (startf-endf) by using specified window */
spDVector xdvfir1bpwin(spDVector win,
		     double startf,
		     double endf)
{
    spDVector filter;

    filter = xdvclone(win);
    if (dvfir1bpwin(filter, startf, endf) == SP_FALSE) {
	xdvfree(filter);
	return NODATA;
    }

    return filter;
}

/* Bandpass FIR Filter (startf-endf) by using Hamming window */
spDVector xdvfir1bp(long order,
		  double startf,
		  double endf)
{
    spDVector filter;

    filter = xdvhamming(order + 1);
    if (dvfir1bpwin(filter, startf, endf) == SP_FALSE) {
	xdvfree(filter);
	return NODATA;
    }

    return filter;
}

/* Bandstop FIR Filter (startf-endf) by using specified window */
spBool dvfir1bswin(spDVector filter, /* (i/o) input is window, output is filter */
		   double startf,
		   double endf)
{
    long order;
    long half;

    order = filter->length - 1;

    /* bandpass filter */
    if (dvfir1bpwin(filter, startf, endf) == SP_FALSE) {
	return SP_FALSE;
    }
    
    half = order / 2;
    
    dvscoper(filter, "*", -1.0);
    filter->data[half] = 1.0 + filter->data[half]; /* since filter->data[half] multiplied by -1.0 */

    return SP_TRUE;
}

/* Bandstop FIR Filter (startf-endf) by using specified window */
spDVector xdvfir1bswin(spDVector win,
		     double startf,
		     double endf)
{
    spDVector filter;

    filter = xdvclone(win);
    if (dvfir1bswin(filter, startf, endf) == SP_FALSE) {
	xdvfree(filter);
	return NODATA;
    }

    return filter;
}

/* Bandstop FIR Filter (startf-endf) by using Hamming window */
spDVector xdvfir1bs(long order,		/* return [odd length] */
		  double startf,
		  double endf)
{
    spDVector filter;

    order = ((order + 1) / 2) * 2;	/* make even */

    filter = xdvhamming(order + 1);
    if (dvfir1bswin(filter, startf, endf) == SP_FALSE) {
	xdvfree(filter);
	return NODATA;
    }

    return filter;
}

/* Highpass FIR digital filter by using specified window */
void dvfir1hpwin(spDVector filter,	/* (i/o) input is window, output is filter */
		 double cutoff)		/* cut-off frequency */
{
    long order;
    long half;

    /* lowpass filter */
    dvfir1win(filter, cutoff);
    
    order = filter->length - 1;
    half = order / 2;
    
    dvscoper(filter, "*", -1.0);
    filter->data[half] = 1.0 + filter->data[half]; /* since filter->data[half] multiplied by -1.0 */

    return;
}

/* Highpass FIR digital filter by using specified window */
spDVector xdvfir1hpwin(spDVector win,
		     double cutoff)	/* cut-off frequency */
{
    spDVector filter;

    filter = xdvclone(win);
    dvfir1hpwin(filter, cutoff);

    return filter;
}

/* Highpass FIR digital filter by using Hamming window */
spDVector xdvfir1hp(long order,		/* return [odd length] */
		  double cutoff)	/* cut-off frequency */
{
    spDVector filter;

    order = ((order + 1) / 2) * 2;	/* make even */

    filter = xdvhamming(order + 1);
    dvfir1hpwin(filter, cutoff);

    return filter;
}

/* decrease the sampling rate for a sequence (decimation) */
spDVector xdvdecimateex(spDVector x,
		      long r,
		      long order, /* 0: use default (chebyshev: 8, FIR: 30) */
		      unsigned long options)
{
    long k;
    long delay;
    long pos;
    long x_length;
    spDVector tx;
    spDVector ttx;
    spDVector dsx;
    spDVector firfilter;
    spDVectors filter;

    if (x == NODATA || r <= 0) return NODATA;

    if (r == 1) return xdvclone(x);
    
    spDebug(100, "xdvdecimateex", "r = %ld, order = %ld, options = %ld\n",
	    r, order, options);
    
    x_length = x->length;
    
    if (options & SP_DECIMATE_OPTION_USE_FIR) {	/* use FIR filter */
	if (order <= 0) order = 30;

        if (options & SP_DECIMATE_OPTION_KEEP_EDGE) {
            x_length += order;
        }
    } else {
	if (order <= 0) order = 8;
    }
    
    /* memory allocation */
    dsx = xdvalloc((long)ceil((double)x_length / (double)r));
    spDebug(100, "xdvdecimateex", "x->length = %ld, dsx->length = %ld\n", x->length, dsx->length);
    
    /* decimation */
    if (options & SP_DECIMATE_OPTION_USE_FIR) {	/* use FIR filter */
        long start_margin, end_margin;
        
	/* FIR lowpass filter */
	firfilter = xdvfir1(order, 1.0 / (double)r);
        dvscoper(firfilter, "*", 1.0 / dvsum(firfilter));

        if (options & SP_DECIMATE_OPTION_KEEP_EDGE) {
            ttx = xdvfftfilt(firfilter, x, MAX(firfilter->length * 2, 128));
            start_margin = end_margin = 0;
            delay = 0;
        } else {
            start_margin = MIN(order + 1, x->length - 1);
            end_margin = MIN(2 * (order + 1), x->length - 1);

            /* filtering */
            tx = xdvalloc(x->length + start_margin + end_margin);
            dvpaste(tx, x, start_margin, x->length, 0);
            for (k = 0; k < start_margin; k++) {
                tx->data[k] = 2.0 * x->data[0] - x->data[start_margin - k];
            }
            for (k = 0; k < end_margin; k++) {
                tx->data[start_margin + x->length + k] = 2.0 * x->data[x->length - 1] - x->data[x->length - 2 - k];
            }
            ttx = xdvfftfilt(firfilter, tx, MAX(firfilter->length * 2, 128));
            xdvfree(tx);
            
            delay = order / 2 + start_margin;
        }

	/* down-sampling */
	for (k = 0; k < dsx->length; k++) {
	    pos = k * r + delay;
	    if (pos < ttx->length) {
		dsx->data[k] = ttx->data[pos];
	    } else {
		dsx->data[k] = 0.0;
	    }
	}

	/* memory free */
	xdvfree(ttx);
	xdvfree(firfilter);
    } else {    /* use chebyshev filter */
	/* lowpass chebyshev filter */
	if ((filter = xdvscheby1(order, 0.05, 0.8 / (double)r, SP_FALSE)) == NODATA) {
	    spWarning("Failed: Chebyshev Filter\n");
	    return NODATA;
	}

	/* filtering */
        if (options & SP_DECIMATE_OPTION_NO_FILTFILT) {
            tx = xdvfilter(filter->vector[0], filter->vector[1], x);
            delay = 0;
        } else {
            tx = xdvfiltfiltex(filter->vector[0], filter->vector[1], x, SP_TRUE);
            delay = MAX(r - (r * dsx->length - x->length) - 1, 0);
        }

	/* down-sampling */
	spDebug(100, "xdvdecimateex", "delay = %ld, dsx->length = %ld, tx->length = %ld\n", delay, dsx->length, tx->length);
	for (k = 0; k < dsx->length; k++) {
	    pos = k * r + delay;
	    if (pos < tx->length) {
		dsx->data[k] = tx->data[pos];
	    } else {
		dsx->data[k] = 0.0;
	    }
	}

	/* memory free */
	xdvfree(tx);
	xdvsfree(filter);
    }

    spDebug(100, "xdvdecimateex", "done\n");
    
    return dsx;
}

spDVector xdvdecimate(spDVector x, long r)
{
    return xdvdecimateex(x, r, 0, 0L);
}

#if 1
struct _spDecimateRec {
    long r;
    long max_input_buf_length;
    unsigned long options;
    
    spFFTRec fftrec;
    spDVector firfilter;
    spDVector firfilterspec;
    spDVector spec;
    
    spDVectors filter;
    spDVector states;

    spDVector buffer;
    
    long offset;
    spULong frame_count;
    spULong total_input;
    spULong total_output;
};

spDecimateRec spDecimateOpen(long r,
                             long order, /* 0: use default (chebyshev: 8, FIR: 30) */
                             long max_input_buf_length,
                             unsigned long options,
                             long *max_output_buf_length,
                             long *default_delay) /* *default_delay = -1: cannot calculate delay */
{
    long fftorder;
    long delay;
    spDVector firfilter = NODATA;
    spDVector firfilterspec = NODATA;
    spDVectors filter = NODATA;
    spDVector states = NODATA;
    spDVector buffer = NODATA;
    spFFTRec fftrec = NULL;
    spDecimateRec decimate;

    if (max_output_buf_length == NULL) return NULL;

    spDebug(100, "spDecimateOpen", "r = %ld, order = %ld, max_input_buf_length = %ld, options = %lx\n",
	    r, order, max_input_buf_length, options);
    
    if (options & SP_DECIMATE_OPTION_USE_FIR) {	/* use FIR filter */
	if (order <= 0) order = 30;

        /* FIR lowpass filter */
        if ((firfilter = xdvfir1(order, 1.0 / (double)r)) == NODATA) {
            spWarning("spDecimateOpen: xdvfir1 failed\n");
            return NULL;
        }
        dvscoper(firfilter, "*", 1.0 / dvsum(firfilter));

        /* 2 * firfilter->length for margin of first and last frame */
        fftorder = spNextPow2(2 * firfilter->length + max_input_buf_length + order);
        spDebug(100, "spDecimateOpen", "order = %ld, firfilter->length = %ld, fftorder = %ld\n",
                order, firfilter->length, fftorder);
        
        if ((fftrec = spInitFFT(fftorder, /*SP_FFT_DEFAULT_PRECISION*/SP_FFT_DOUBLE_PRECISION)) == NULL) {
            xdvfree(firfilter);
            spDebug(80, "spDecimateOpen", "spInitFFT failed\n");
            return NULL;
        }

        firfilterspec = xdvzeros(spGetFFTLength(fftrec));
        dvcopy(firfilterspec, firfilter);
        spExecRealFFT(fftrec, firfilterspec->data, 0);
        buffer = xdvzeros(spGetFFTLength(fftrec));

        delay = order / 2;
    } else {    /* use chebyshev filter */
	if (order <= 0) order = 8;

	/* lowpass chebyshev filter */
	if ((filter = xdvscheby1(order, 0.05, 0.8 / (double)r, SP_FALSE)) == NODATA) {
	    spWarning("spDecimateOpen: xdvscheby1 failed\n");
	    return NULL;
	}
        delay = -1;
        buffer = xdvzeros(max_input_buf_length);
        states = xdvinitfilterstates(filter->vector[0], filter->vector[1]);
    }

    spDebug(100, "spDecimateOpen", "updated order = %ld, delay = %ld, buffer->length = %ld\n",
            order, delay, buffer->length);
    
    decimate = xalloc(1, struct _spDecimateRec);
    memset(decimate, 0, sizeof(struct _spDecimateRec));

    decimate->r = r;
    decimate->max_input_buf_length = max_input_buf_length;
    decimate->options = options;
    
    if (firfilterspec != NODATA && (decimate->options & SP_DECIMATE_OPTION_KEEP_EDGE)) {
        decimate->offset = 0;
        *max_output_buf_length = (long)ceil((double)(max_input_buf_length + order) / (double)r);
    } else {
        decimate->offset = MAX(delay, 0);
        *max_output_buf_length = (long)ceil((double)(max_input_buf_length + decimate->offset) / (double)r);
    }
    
    spDebug(100, "spDecimateOpen", "offset = %ld, max_output_buf_length = %ld\n",
            decimate->offset, *max_output_buf_length);
    
    decimate->fftrec = fftrec;
    decimate->firfilter = firfilter;
    decimate->firfilterspec = firfilterspec;
    decimate->filter = filter;
    decimate->states = states;
    decimate->buffer = buffer;

    if (decimate->firfilterspec != NODATA) {
        decimate->spec = xdvzeros(decimate->firfilterspec->length);
    }

    if (default_delay != NULL) {
        if (delay > 0) {
            *default_delay = (long)ceil((double)delay / (double)r);
        } else {
            *default_delay = delay;
        }
    }

    return decimate;
}

spBool spDecimateClose(spDecimateRec decimate)
{
    if (decimate != NULL) {
	if (decimate->fftrec != NULL) spFreeFFT(decimate->fftrec);
	if (decimate->firfilter != NODATA) xdvfree(decimate->firfilter);
	if (decimate->firfilterspec != NODATA) xdvfree(decimate->firfilterspec);
	if (decimate->spec != NODATA) xdvfree(decimate->spec);
        
	if (decimate->filter != NODATA) xdvsfree(decimate->filter);
	if (decimate->states != NODATA) xdvfree(decimate->states);
        
	if (decimate->buffer != NODATA) xdvfree(decimate->buffer);

	xfree(decimate);

	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

long spDecimateProcess(spDecimateRec decimate, double *input_buf, long input_buf_length, double *output_buf,
                       spBool last_frame, long *current_delay)
{
    long k, m;
    long pos;
    long offset;
    long filter_delay;
    long output_length;
    long prepend_length = 0;
    long append_length = 0;
    
    if (decimate == NULL
        || (decimate->filter == NODATA && decimate->firfilter == NODATA)) return -1;

    spDebug(80, "spDecimateProcess", "input_buf_length = %ld, last_frame = %d, frame_count = %ld\n",
            input_buf_length, last_frame, (long)decimate->frame_count);
    
    if (decimate->firfilterspec != NODATA) {
        if (decimate->frame_count == 0 && !(decimate->options & SP_DECIMATE_OPTION_KEEP_EDGE)) {
            prepend_length = MIN(decimate->firfilter->length, input_buf_length - 1);
            for (k = 0; k < prepend_length; k++) {
                decimate->spec->data[k] = 2.0 * input_buf[0] - input_buf[prepend_length - k];
            }
            for (; k < prepend_length + input_buf_length; k++) {
                decimate->spec->data[k] = input_buf[k - prepend_length];
            }
        } else {
            for (k = 0; k < input_buf_length; k++) {
                decimate->spec->data[k] = input_buf[k];
            }
        }

        if (last_frame && !(decimate->options & SP_DECIMATE_OPTION_KEEP_EDGE)) {
            append_length = MIN(decimate->firfilter->length, input_buf_length - 1);
            for (m = 0; m < append_length; m++) {
                decimate->spec->data[k] = 2.0 * input_buf[input_buf_length - 1] - input_buf[input_buf_length - 2 - m];
                k++;
            }
        }

        for (; k < decimate->spec->length; k++) {
            decimate->spec->data[k] = 0.0;
        }
        
        spExecRealFFT(decimate->fftrec, decimate->spec->data, 0);
        spMultiplySpectrumOfRealFFT(decimate->fftrec, decimate->spec->data, decimate->firfilterspec->data);
        spExecRealFFT(decimate->fftrec, decimate->spec->data, 1);
        
        dvadd(decimate->buffer, 0, decimate->spec, 0, decimate->spec->length, SP_TRUE);

        filter_delay = (decimate->firfilter->length - 1) / 2;
        if (current_delay != NULL) {
            *current_delay = (long)ceil((double)(filter_delay - decimate->offset) / (double)decimate->r);
        }
    } else {
        for (k = 0; k < input_buf_length; k++) {
            decimate->buffer->data[k] = dvfiltersample(decimate->filter->vector[0], decimate->filter->vector[1],
                                                       decimate->states, input_buf[k]);
        }
        for (; k < decimate->buffer->length; k++) {
            decimate->buffer->data[k] = 0.0;
        }

        filter_delay = 0;
        if (current_delay != NULL) {
            *current_delay = -1;
        }
    }
    
    offset = decimate->offset;
    if (last_frame) {
        if ((decimate->options & SP_DECIMATE_OPTION_KEEP_EDGE) && decimate->firfilter != NODATA) {
            output_length = (long)ceil((double)(input_buf_length - offset + decimate->firfilter->length - 1) / (double)decimate->r);
        } else {
            output_length = (long)ceil((double)(input_buf_length - offset + filter_delay) / (double)decimate->r);
        }
    } else {
        output_length = (long)ceil((double)(input_buf_length - offset) / (double)decimate->r);
    }
    output_length = MAX(output_length, 0);
    spDebug(80, "spDecimateProcess", "offset = %ld, input_buf_length = %ld, output_length = %ld\n",
            offset, input_buf_length, output_length);

    for (k = 0; k < output_length; k++) {
        pos = k * decimate->r + offset + prepend_length;
        if (pos >= 0) {
            output_buf[k] = decimate->buffer->data[pos];
        } else {
            output_buf[k] = 0.0;
        }
    }
    
    decimate->offset = output_length * decimate->r + offset;
    decimate->total_input += (spULong)input_buf_length;
    decimate->total_output += (spULong)output_length;
    decimate->offset -= input_buf_length;
    spDebug(80, "spDecimateProcess", "updated offset = %ld\n", decimate->offset);
    
    if (last_frame == SP_FALSE) {
        spExecDataShift(decimate->fftrec, decimate->buffer->data, NULL, -(prepend_length + input_buf_length));
    }
    
    decimate->frame_count++;

    return output_length;
}
#endif

static spDVector xdvinitfiltfiltstates(spDVector b, spDVector a)
{
    long i;
    long nfilt;
    long n;
    spDVector c;
    spDVector zi;
    spDMatrix mat;

    nfilt = MAX(b->length, a->length);
    n = nfilt - 1;
    mat = xdmzeros(n, n);

    for (i = 0; i < n; i++) {
	mat->data[i][0] = a->data[i + 1];
    }
    mat->data[0][0] += 1.0;
    
    for (i = 1; i < n; i++) {
	mat->data[i][i] = 1.0;
    }
    for (i = 1; i < n; i++) {
	mat->data[i - 1][i] = -1.0;
    }

    c = xdvcut(a, 1, n);
    dvscoper(c, "*", -b->data[0]);
    dvadd(c, 0, b, 1, n, SP_TRUE);
    
    zi = xdvmldivide(mat, c);

    xdvfree(c);
    xdmfree(mat);
    
    return zi;
}

/*  Zero-phase forward and reverse digital filtering */
spDVector xdvfiltfiltex(spDVector b,	/* filter-coefficient (numerator) */
		      spDVector a,	/* filter-coefficient (denominator) */
		      spDVector x,	/* input data */
		      spBool use_optimum_state) /* SP_TRUE if optimum initial states are used */
{
    long n;
    long len;
    long offset;
    long nfact;
    spDVector zi, z;
    spDVector y, y2;
    spBool cplx_flag = SP_FALSE;
    
    if (b == NODATA || a == NODATA || x == NODATA) return NODATA;

    if (a->imag != NULL || b->imag != NULL || x->imag != NULL) {
	cplx_flag = SP_TRUE;
    }

    if (use_optimum_state) {
	zi = xdvinitfiltfiltstates(b, a);
    } else {
	len = MAX(b->length, a->length) - 1;
	zi = xdvzeros(len);
    }
    if (cplx_flag) {
	dvizeros(zi, zi->length);
    }
    spDebug(100, "xdvfiltfiltex", "a->length = %ld, b->length = %ld, zi->length = %ld, cplx_flag = %d\n",
	    a->length, b->length, zi->length, cplx_flag);

    nfact = zi->length * 3;

    if (cplx_flag) {
	y = xdvrialloc(x->length + 2 * nfact);
    } else {
	y = xdvalloc(x->length + 2 * nfact);
    }
    spDebug(100, "xdvfiltfiltex", "nfact = %ld, x->length = %ld, y->length = %ld\n", nfact, x->length, y->length);
    
    for (n = 0; n < nfact; n++) {
	if (nfact - n < x->length) {
	    y->data[n] = 2.0 * x->data[0] - x->data[nfact - n];
	} else {
	    y->data[n] = 2.0 * x->data[0];
	}
	if (y->imag != NULL) {
	    if (x->imag != NULL) {
		if (nfact - n < x->length) {
		    y->imag[n] = 2.0 * x->imag[0] - x->imag[nfact - n];
		} else {
		    y->imag[n] = 2.0 * x->imag[0];
		}
	    } else {
		y->imag[n] = 0.0;
	    }
	}
    }
    dvpaste(y, x, nfact, x->length, 0);
    offset = nfact + x->length;
    for (n = 0; n < nfact; n++) {
	if (x->length - n - 2 < x->length) {
	    y->data[offset + n] = 2.0 * x->data[x->length - 1] - x->data[x->length - n - 2];
	} else {
	    y->data[offset + n] = 2.0 * x->data[x->length - 1];
	}
	if (y->imag != NULL) {
	    if (x->imag != NULL) {
		if (x->length - n - 2 < x->length) {
		    y->imag[offset + n] = 2.0 * x->imag[x->length - 1] - x->imag[x->length - n - 2];
		} else {
		    y->imag[offset + n] = 2.0 * x->imag[x->length - 1];
		}
	    } else {
		y->imag[offset + n] = 0.0;
	    }
	}
    }

    if (use_optimum_state) {
	if (cplx_flag) {
	    z = xdvrialloc(zi->length);
	} else {
	    z = xdvalloc(zi->length);
	}
	for (n = 0; n < zi->length; n++) {
	    if (y->imag != NULL) {
		z->data[n] = zi->data[n] * y->data[0] - zi->imag[n] * y->imag[0];
		z->imag[n] = zi->data[n] * y->imag[0] + zi->imag[n] * y->data[0];
	    } else {
		z->data[n] = zi->data[n] * y->data[0];
		if (z->imag != NULL) {
		    z->imag[n] = 0.0;
		}
	    }
	}
    } else {
	z = xdvclone(zi);
    }
    
    /* forward filtering */
    for (n = 0; n < y->length; n++) {
	if (cplx_flag) {
	    dvfiltersamplecplx(b, a, z, &y->data[n], &y->imag[n]);
	} else {
	    y->data[n] = dvfiltersample(b, a, z, y->data[n]);
	}
    }

    xdvfree(z);
    
    if (use_optimum_state) {
	if (cplx_flag) {
	    z = xdvrialloc(zi->length);
	} else {
	    z = xdvalloc(zi->length);
	}
	for (n = 0; n < zi->length; n++) {
	    if (y->imag != NULL) {
		z->data[n] = zi->data[n] * y->data[y->length - 1] - zi->imag[n] * y->imag[y->length - 1];
		z->imag[n] = zi->data[n] * y->imag[y->length - 1] + zi->imag[n] * y->data[y->length - 1];
	    } else {
		z->data[n] = zi->data[n] * y->data[y->length - 1];
		if (z->imag != NULL) {
		    z->imag[n] = 0.0;
		}
	    }
	}
    } else {
	z = xdvclone(zi);
    }
    
    /* backward filtering */
    for (n = y->length - 1; n >= 0; n--) {
	if (cplx_flag) {
	    dvfiltersamplecplx(b, a, z, &y->data[n], &y->imag[n]);
	} else {
	    y->data[n] = dvfiltersample(b, a, z, y->data[n]);
	}
    }

    xdvfree(z);
    
    y2 = xdvcut(y, nfact, x->length);
	
    /* memory free */
    xdvfree(zi);
    xdvfree(y);

    return y2;
}

spDVector xdvfiltfilt(spDVector b,	/* filter-coefficient (numerator) */
		    spDVector a,	/* filter-coefficient (denominator) */
		    spDVector x)	/* input data */
{
    return xdvfiltfiltex(b, a, x, SP_FALSE);
}

/* Bilinear transformation for filter (analog -> digital) */
spDVectors xdvsbilinear(spDVector pv,	/* pole */
		      double gain,	/* gain */
		      double cutoff,	/* cutoff frequency */
		      spBool hp_flag)	/* highpass filter */
{
    long n, polydnum, l, l2, rn, cn, k;
    double r2, rp, ip, mapcoef, lphpcoef;
    spDVector vec1d;
    spDVector vec2d;
    spDVector vec1n;
    spDVector vec2n;
    spDMatrix polyd;
    spDMatrix polyn;
    spDMatrices calmatd;
    spDMatrices calmatn;
    spDVectors coef;

    /* order */
    n = pv->length;

    mapcoef = PI / 2.0;
    /* match frequency */
    if (cutoff != 0.0) mapcoef = tan(cutoff * mapcoef) / cutoff;

    /* gain */
    if (hp_flag == SP_FALSE) {	/* lowpass filter */
	gain *= pow(mapcoef, (double)n);
	lphpcoef = 1.0;
    } else {			/* highpass filter */
	lphpcoef = -1.0;
    }
    
    /* polynomial z^-2 */
    polydnum = (long)pow(2.0, (double)(nextpow2(n) - 1));
    spDebug(100, "xdvsbilinear", "polydnum = %ld, n = %ld\n", polydnum, n);
    polyd = xdmzeros(polydnum, 3);
    polyn = xdmzeros(polydnum, 3);
    for (l = 0; l < n / 2; l++) {	/* real + imag, real - imag */
	l2 = l * 2;
	r2 = pv->data[l2] * 2.0;
	rp = pv->data[l2] * pv->data[l2];
	ip = pv->imag[l2] * pv->imag[l2];
	/* z^0 */
	polyd->data[l][0] = 1.0 - mapcoef * r2 + pow(mapcoef, 2.0) * (rp + ip);
	polyn->data[l][0] = 1.0;
	/* z^-1 */
	polyd->data[l][1] = -2.0 + 2.0 * pow(mapcoef, 2.0) * (rp + ip);
	polyn->data[l][1] = lphpcoef * 2.0;
	/* z^-2 */
	polyd->data[l][2] = 1.0 + mapcoef * r2 + pow(mapcoef, 2.0) * (rp + ip);
	polyn->data[l][2] = 1.0;
    }
    if ((l2 = l * 2) < n) {	/* imag = 0 */
	/* z^0 */
	polyd->data[l][0] = 1.0 - mapcoef * pv->data[l2];
	polyn->data[l][0] = 1.0;
	/* z^-1 */
	polyd->data[l][1] = -1.0 - mapcoef * pv->data[l2];
	polyn->data[l][1] = lphpcoef;

	l++;
    }
    for (; l < polydnum; l++) {
	polyd->data[l][0] = 1.0;
	polyn->data[l][0] = 1.0;
    }

    for (k = 1; k < nextpow2(n); k++) {
	spDebug(100, "xdvsbilinear", "%ld: current polyd->row = %ld, polyd->col = %ld, polyn->row = %ld, polyn->col = %ld\n",
		k, polyd->row, polyd->col, polyn->row, polyn->col);
	/* calculate matrices */
	calmatd = xdmsalloc(polyd->row / 2);
	calmatn = xdmsalloc(calmatd->num_matrix);
	for (l = 0; l < calmatd->num_matrix; l++) {
	    spDebug(100, "xdvsbilinear", "k = %ld, l = %ld / %ld\n", k, l, calmatd->num_matrix);
	    calmatd->matrix[l] = xdmalloc(polyd->col, polyd->col);
	    calmatn->matrix[l] = xdmalloc(polyn->col, polyn->col);
	    l2 = l * 2;
	    vec1d = xdmextractrow(polyd, l2);
	    vec2d = xdmextractrow(polyd, l2 + 1);
	    vec1n = xdmextractrow(polyn, l2);
	    vec2n = xdmextractrow(polyn, l2 + 1);
	    for (rn = 0; rn < polyd->col; rn++) {
		for (cn = 0; cn < polyd->col; cn++) {
		    calmatd->matrix[l]->data[rn][cn]
			= vec1d->data[rn] * vec2d->data[cn];
		    calmatn->matrix[l]->data[rn][cn]
			= vec1n->data[rn] * vec2n->data[cn];
		}
	    }
	    xdvfree(vec1d);
	    xdvfree(vec2d);
	    xdvfree(vec1n);
	    xdvfree(vec2n);
	}
	xdmfree(polyd);
	xdmfree(polyn);

	/* calculate polynomial */
	polyd = xdmzeros(l, (long)pow(2.0, (double)(k + 1)) + 1);
	spDebug(100, "xdvsbilinear", "%ld: new polyd->row = %ld, polyd->col = %ld\n", k, polyd->row, polyd->col);
	polyn = xdmzeros(l, polyd->col);
	for (l = 0; l < polyd->row; l++) {
	    for (rn = 0; rn < calmatd->matrix[l]->row; rn++) {
		for (cn = 0; cn < calmatd->matrix[l]->col; cn++) {
		    polyd->data[l][rn + cn] += calmatd->matrix[l]->data[rn][cn];
		    polyn->data[l][rn + cn] += calmatn->matrix[l]->data[rn][cn];
		}
	    }
	}
	xdmsfree(calmatd);
	xdmsfree(calmatn);
    }

    /* Filter-coefficient */
    coef = xdvsalloc(2);
    coef->vector[0] = xdvalloc(n + 1);
    coef->vector[1] = xdvalloc(n + 1);
    spDebug(100, "xdvsbilinear", "final: n = %ld, polyd->row = %ld, polyd->col = %ld\n", n, polyd->row, polyd->col);
    for (l = 0; l < n + 1; l++) {
	coef->vector[0]->data[l]
	    = polyn->data[0][l] * gain / polyd->data[0][0];
	coef->vector[1]->data[l] = polyd->data[0][l] / polyd->data[0][0];
    }

    /* memory free */
    xdmfree(polyd);
    xdmfree(polyn);

    return coef;
}

static double dvconvertpole(spDVector pv, long n, double cutoff, spBool hp_flag)
{
    long k;
    double gain = 1.0;

    if (hp_flag == SP_TRUE) {
	/* highpass filter */
	for (k = 0; k + 1 < n; k += 2) {
	    gain *= pow(pv->data[k], 2.0) + pow(pv->imag[k], 2.0);
	}
	if (k < n) gain *= -1.0 * pv->data[k];
	gain = 1.0 / gain;
	dvscoper(pv, "^", -1.0);
    } else {
	gain *= pow(cutoff, (double)pv->length);
    }
    dvscoper(pv, "*", cutoff);
    
    return gain;
}

/* Butterworth analog lowpass filter prototype (gain = 1.0) */
spDVector xdvbuttap(long n)	/* order */
{
    long l, l2;
    double w;
    spDVector pv = NODATA;
    
    /* calculate pole of butterworth filter */
    pv = xdvrialloc(n);
    for (l = 0; l < n / 2; l++) {
	w = ((double)n + 1.0 + 2.0 * (double)l) / (2.0 * (double)n) * PI;
	l2 = l * 2;
	pv->data[l2] = cos(w);
	pv->imag[l2] = sin(w);
	pv->data[l2 + 1] = pv->data[l2];
	pv->imag[l2 + 1] = -1.0 * pv->imag[l2];
    }
    if ((l2 = l * 2) < n) {
	w = ((double)n + 1.0 + 2.0 * (double)l) / (2.0 * (double)n) * PI;
	pv->data[l2] = cos(w);
	pv->imag[l2] = 0.0;
    }

    return pv;
}

/* Butterworth digital and analog high-pass filter design */
spDVectors xdvsbutter(long n,		/* order */
		    double cutoff,	/* cut-off frequency */
		    spBool hp_flag)     /* highpass filter */

{
    double gain = 1.0;
    spDVector pv = NODATA;
    spDVectors filter = NODATA;

    /* 0 <= cufoff <= 1 */
    cutoff = MIN(cutoff, 1.0);
    cutoff = MAX(cutoff, 0.0);

    /* 2 <= n <= 16 */
    if (n < 2 || n > 16) {
	fprintf(stderr, "2 <= Order <= 16\n");
	return NODATA;
    }

    /* calculate gain and pole of butterworth filter */
    pv = xdvbuttap(n);

    /* convert filter gain and poles */
    gain = dvconvertpole(pv, n, cutoff, hp_flag);
   
    /* convert analog into digital through bilinear transformation */
    filter = xdvsbilinear(pv, gain, cutoff, hp_flag);	/* highpass filter */

    /* memory free */
    xdvfree(pv);

    return filter;
}

/* Chebyshev analog lowpass filter prototype */
spDVector xdvcheb1ap(long n,		/* order */
		   double rp,		/* passband ripple [dB] */
		   double *gain)	/* gain */
{
    long l, l2;
    long nhalf;
    double e, e2, alpha, alpha1, alpha2, a, b, w;
    spDVector pv;
    
    spDebug(100, "xdvcheb1ap", "n = %ld, rp = %f\n", n, rp);
    
    /* calculate gain and pole of chebyshev filter */
    e2 = pow(10.0, rp / 10.0) - 1.0;
    e = sqrt(e2);
    alpha = 1.0 / e + sqrt(1.0 + 1.0 / e2);
    
    if (gain != NULL) {
	/* gain */
	*gain = 1.0 / e * pow(0.5, (double)(n - 1));
	spDebug(100, "xdvcheb1ap", "gain = %f\n", *gain);
    }
    
    /* pole */
    alpha1 = pow(alpha, 1.0 / (double)n);
    alpha2 = pow(alpha, -1.0 / (double)n);
    a = (alpha1 - alpha2) / 2.0;
    b = (alpha1 + alpha2) / 2.0;
    pv = xdvrialloc(n);
    nhalf = n / 2;
    for (l = 0; l < nhalf; l++) {
	w = ((double)n + 1.0 + 2.0 * (double)l) / (2.0 * (double)n) * PI;
	l2 = l * 2;
	spDebug(100, "xdvcheb1ap", "l = %ld, l2 = %ld, n = %ld\n", l, l2, n);
	pv->data[l2] = a * cos(w);
	pv->imag[l2] = b * sin(w);
	pv->data[l2 + 1] = pv->data[l2];
	pv->imag[l2 + 1] = -1.0 * pv->imag[l2];
    }
    if ((l2 = l * 2) < n) {
	spDebug(100, "xdvcheb1ap", "last element l2 = %ld, n = %ld\n", l2, n);
	w = ((double)n + 1.0 + 2.0 * (double)l) / (2.0 * (double)n) * PI;
	pv->data[l2] = a * cos(w);
	pv->imag[l2] = 0.0;
    }
    
    return pv;
}

/* Chebyshev digital and analog filter design */
spDVectors xdvscheby1(long n,		/* order */
		    double rp,		/* passband ripple [dB] */
		    double cutoff,	/* cut-off frequency */
		    spBool hp_flag)	/* highpass filter */
{
    double gain;
    spDVector pv;
    spDVectors filter;

    /* 0 <= cufoff <= 1 */
    cutoff = MIN(cutoff, 1.0);
    cutoff = MAX(cutoff, 0.0);

    spDebug(100, "xdvscheby1", "n = %ld, rp = %f, cutoff = %f, hp_flag = %d\n",
	    n, rp, cutoff, hp_flag);

    /* 2 <= n <= 16 */
    if (n < 2 || n > 16) {
	spWarning("2 <= Order <= 16\n");
	return NODATA;
    }

    /* calculate gain and pole of chebyshev filter */
    pv = xdvcheb1ap(n, rp, &gain);

    /* convert filter gain and pole */
    gain *= dvconvertpole(pv, n, cutoff, hp_flag);
    
    /* convert analog into digital through bilinear transformation */
    filter = xdvsbilinear(pv, gain, cutoff, hp_flag);

    /* memory free */
    xdvfree(pv);

    return filter;
}
