/** @file sp/filter.h
 */

#ifndef SPLIB_FILTER_H
#define SPLIB_FILTER_H

#include <sp/vector.h>
#include <sp/fft.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(MACOS)
#pragma import on
#endif

/** @~english @defgroup filterGroup <sp/filter.h>: Filtering
@code
#include <sp/filter.h>
@endcode
 */

/** @addtogroup filterGroup
 *  @{ */  /*----@addtogroup filterGroup----*/

/** @~english @name Utility Functions for Filtering (<sp/filter.h>)
 *  @{ */

extern void fvunwrap(spFVector phs, float cutoff);
extern void dvunwrap(spDVector phs, double cutoff);
extern float sincf(float x);
extern double sinc(double x);
extern float sinccf(float x, float c);
extern double sincc(double x, double c);
    
/** @~english @} */

/** @~english @name FIR Filter (<sp/filter.h>)
 *  @{ */

extern spDVector xdvbandstop(double lp_cutoff, double hp_cutoff, double sidelobe, double trans, double gain);
extern spDVector xdvbandpass(double hp_cutoff, double lp_cutoff, double sidelobe, double trans, double gain);
extern spDVector xdvlowpass(double cutoff, double sidelobe, double trans, double gain);
extern spDVector xdvhighpass(double cutoff, double sidelobe, double trans, double gain);

/** @~english @} */

/** @~english @name Filtering (<sp/filter.h>)
 *  @{ */

extern spDVector xdvfftfiltex(spFFTRec fftrec, spDVector b, spDVector x);
extern spDVector xdvfftfiltmex(spFFTRec fftrec, spDVector b, spDVector x);
extern spDVector xdvfftfilt(spDVector b, spDVector x, long fftl);
extern spDVector xdvfftfiltm(spDVector b, spDVector x, long fftl);
extern spDVector xdvconv(spDVector a, spDVector b);

/* length of z must be larger than MAX(length of b, length of a)-1 */
extern spDVector xdvinitfilterstates(spDVector b, spDVector a); /* 0.9.1+ */
extern void dvfiltersamplecplx(spDVector b, spDVector a, spDVector z, double *re, double *im);
extern double filtersample(double *b, long blen, double *a, long alen, double *z, long zlen, double x);
extern double dvfiltersample(spDVector b, spDVector a, spDVector z, double x);
extern spDVector xdvfilterex(spDVector b, spDVector a, spDVector z, spDVector x);
extern spDVector xdvfilter(spDVector b, spDVector a, spDVector x);

extern spDVector xdvfreqzex(spFFTRec fftrec, spDVector b /* can be NODATA */, spDVector a /* can be NODATA */);
extern spDVector xdvfreqz(spDVector b /* can be NODATA */, spDVector a /* can be NODATA */, long fftl /* can be 0 (auto select) */);

/** @~english @} */

/** @~english @name Interpolation and Decimation (<sp/filter.h>)
 *  @{ */

/* linear interpolation (X must be monotonically increasing) */
extern spBool dvinterp1l(spDVector x, spDVector y, spDVector xi, spDVector yi);
extern spDVector xdvinterp1l(spDVector x, spDVector y, spDVector xi);

/* nearest neighbor interpolation (X must be monotonically increasing) */
extern spBool dvinterp1n(spDVector x, spDVector y, spDVector xi, spDVector yi);
extern spDVector xdvinterp1n(spDVector x, spDVector y, spDVector xi);
    
/* cubic spline interpolation on not-a-knot condition (X must be monotonically increasing) */
extern spBool dvinterp1s(spDVector x, spDVector y, spDVector xi, spDVector yi);
extern spDVector xdvinterp1s(spDVector x, spDVector y, spDVector xi);
    
/* Resample data at a higher rate using lowpass interpolation */
extern spDVector xdvinterpex(spDVector x, long r, long l, double a);
extern spDVector xdvinterp(spDVector x, long r); /* l = 4, a = 0.5 */

/* clamped cubic spline interpolation */
extern spBool dvsplinecl(spDVector x, spDVector y, spDVector xi, spDVector yi, double leftslope, double rightslope);
extern spDVector xdvsplinecl(spDVector x, spDVector y, spDVector xi, double leftslope, double rightslope);
/* cubic spline.
 * if notaknot == SP_TRUE, not-a-knot condition (same as MATLAB), otherwise natural condition. */
extern spBool dvspline(spDVector x, spDVector y, spDVector xi, spDVector yi, spBool notaknot);
extern spDVector xdvspline(spDVector x, spDVector y, spDVector xi, spBool notaknot);

#define SP_DECIMATE_OPTION_USE_FIR (1<<0)
#define SP_DECIMATE_OPTION_KEEP_EDGE (2<<0)
#define SP_DECIMATE_OPTION_NO_FILTFILT (3<<0)

/* decrease the sampling rate for a sequence (decimation) */
extern spDVector xdvdecimateex(spDVector x,
			     long r,
			     long order, /* 0: use default (chebyshev: 8, FIR: 30) */
			     unsigned long options);
extern spDVector xdvdecimate(spDVector x, long r); /* use 8-order chebyshev IIR */

typedef struct _spDecimateRec *spDecimateRec;
extern spDecimateRec spDecimateOpen(long r,
                                    long order, /* 0: use default (chebyshev: 8, FIR: 30) */
                                    long max_input_buf_length,
                                    unsigned long options,
                                    long *max_output_buf_length,
                                    long *default_delay); /* *default_delay = -1: cannot calculate delay */
extern spBool spDecimateClose(spDecimateRec decimate);
extern long spDecimateProcess(spDecimateRec decimate, double *input_buf, long input_buf_length, double *output_buf,
                              spBool last_frame, long *current_delay); /* *current_delay = -1: cannot calculate delay */

/** @~english @} */

/** @~english @name FIR Filter (<sp/filter.h>)
 *  @{ */

/* Lowpass FIR digital filter by using specified window */
extern void dvfir1win(spDVector filter,	/* (i/o) input is window, output is filter */
		      double cutoff);	/* cut-off frequency */
extern spDVector xdvfir1win(spDVector win,
			  double cutoff);	/* cut-off frequency */
    
/* Lowpass FIR digital filter by using Hamming window */
extern spDVector xdvfir1(long order,		/* return [order + 1] */
                         double cutoff);	/* cut-off frequency */
    
/* Bandpass FIR Filter (startf-endf) by using specified window */
extern spBool dvfir1bpwin(spDVector filter, /* (i/o) input is window, output is filter */
			  double startf,
			  double endf);
extern spDVector xdvfir1bpwin(spDVector win,
			    double startf,
			    double endf);

/* Bandpass FIR Filter (startf-endf) by using Hamming window */
extern spDVector xdvfir1bp(long order,
			 double startf,
			 double endf);
    
/* Bandstop FIR Filter (startf-endf) by using specified window */
extern spBool dvfir1bswin(spDVector filter, /* (i/o) input is window, output is filter */
			  double startf,
			  double endf);
extern spDVector xdvfir1bswin(spDVector win,
			    double startf,
			    double endf);
    
/* Bandstop FIR Filter (startf-endf) by using Hamming window */
extern spDVector xdvfir1bs(long order,	/* return [odd length] */
			 double startf,
			 double endf);
    
/* Highpass FIR digital filter by using specified window */
extern void dvfir1hpwin(spDVector filter,	/* (i/o) input is window, output is filter */
		 double cutoff);	/* cut-off frequency */
extern spDVector xdvfir1hpwin(spDVector win,
			    double cutoff);	/* cut-off frequency */
    
/* Highpass FIR digital filter by using Hamming window */
extern spDVector xdvfir1hp(long order,		/* return [odd length] */
			 double cutoff);	/* cut-off frequency */
    
/** @~english @} */

/** @~english @name Filtering (<sp/filter.h>)
 *  @{ */

/*  Zero-phase forward and reverse digital filtering */
extern spDVector xdvfiltfiltex(spDVector b,	/* filter-coefficient (numerator) */
			     spDVector a,	/* filter-coefficient (denominator) */
			     spDVector x,	/* input data */
			     spBool use_optimum_state); /* SP_TRUE if optimum initial states are used */
extern spDVector xdvfiltfilt(spDVector b,	/* filter-coefficient (numerator) */
			   spDVector a,	/* filter-coefficient (denominator) */
			   spDVector x);	/* input data */
    
/** @~english @} */

/** @~english @name IIR Filter (<sp/filter.h>)
 *  @{ */

/* Bilinear transformation for filter (analog -> digital) */
extern spDVectors xdvsbilinear(spDVector pv,	/* pole */
			     double gain,	/* gain */
			     double cutoff,	/* cutoff frequency */
			     spBool hp_flag);	/* highpass filter */
    
/* Chebyshev analog lowpass filter prototype */
extern spDVector xdvcheb1ap(long n,	/* order */
			  double rp,	/* passband ripple [dB] */
			  double *gain);/* gain */
    
/* Chebyshev digital and analog filter design */
extern spDVectors xdvscheby1(long n,		/* order */
			   double rp,		/* passband ripple [dB] */
			   double cutoff,	/* cut-off frequency */
			   spBool hp_flag);	/* highpass filter */
    
/* Butterworth analog lowpass filter prototype (gain = 1.0) */
extern spDVector xdvbuttap(long n);	/* order */
extern spDVectors xdvsbutter(long n,		/* order */
			   double cutoff,	/* cut-off frequency */
			   spBool hp_flag); 	/* highpass filter */

/** @~english @} */

/** @} */  /*----@addtogroup windowGroup----*/

#if defined(MACOS)
#pragma import off
#endif

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_FILTER_H */
