/*
 *	kaiser.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <sp/sp.h>
#include <sp/kaiser.h>

#ifdef SP_USE_VECTOR_ENGINE
#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>
#endif

#define SP_KAISER_MIN_TRANS 0.0001

float getkaiserparamexf(float sidelobe, float trans, long max_length, float *beta, long *length)
{
    float value;

    if (trans <= 0.0f) trans = (float)SP_KAISER_MIN_TRANS;

    value = (2.285f * (float)PI * trans);
    *length = (long)ceil(((double)sidelobe - /*8.0*/7.95) / (double)value) + 1L;
    if (max_length > 0 && *length > max_length) {
	sidelobe = /*8.0f*/7.95f + (float)(max_length - 1L) * value;
	*length = max_length;
    }
    
    if (sidelobe < 21.0f) {
	*beta = 0;
    } else if (sidelobe > 50.0f) {
	*beta = 0.1102f * ((float)sidelobe - 8.7f);
    } else {
	value = (float)sidelobe - 21.0f;
	*beta = 0.5842f * (float)pow((double)value, 0.4) + 0.07886f * value;
    }

    return sidelobe;
}

double getkaiserparamex(double sidelobe, double trans, long max_length, double *beta, long *length)
{
    double value;

    if (trans <= 0.0) trans = SP_KAISER_MIN_TRANS;

    value = (2.285 * PI * trans);
    *length = (long)ceil(((double)sidelobe - /*8.0*/7.95) / value) + 1L;
    if (max_length > 0 && *length > max_length) {
	sidelobe = /*8.0*/7.95 + (double)(max_length - 1L) * value;
	*length = max_length;
    }
    
    if (sidelobe < 21.0) {
	*beta = 0;
    } else if (sidelobe > 50.0) {
	*beta = 0.1102 * ((double)sidelobe - 8.7);
    } else {
	value = (double)sidelobe - 21.0;
	*beta = 0.5842 * pow(value, 0.4) + 0.07886 * value;
    }
    spDebug(100, "getkaiserparamex", "beta = %f, sidelobe = %f, length = %ld\n", *beta, sidelobe, *length);

    return sidelobe;
}

void getkaiserparamf(float sidelobe, float trans, float *beta, long *length)
{
    getkaiserparamexf(sidelobe, trans, 0, beta, length);
    return;
}

void getkaiserparam(double sidelobe, double trans, double *beta, long *length)
{
    getkaiserparamex(sidelobe, trans, 0, beta, length);
    return;
}

float kaiserordexf(float cutlow, float cuthigh, float devlow, float devhigh, spBool hp_flag,
		   long max_order, long *order, float *cutoff, float *beta)
{
    float trans;
    float sidelobe;
    float dev;

    spDebug(100, "kaiserordexf", "cutlow = %f, cuthigh = %f, devlow = %f, devhigh = %f\n",
	    cutlow, cuthigh, devlow, devhigh);
    
    if (cutoff != NULL) {
	*cutoff = (cutlow + cuthigh) / 2.0f;
    }
    
    trans = cuthigh - cutlow;
    dev = MIN(devlow, devhigh);
    
    if (dev == 0.0f) {
	sidelobe = 200.0f;
    } else {
	sidelobe = -20.0f * log10f(dev);
    }
    
    spDebug(100, "kaiserordexf", "trans = %f, dev = %f, sidelobe = %f\n", trans, dev, sidelobe);
    
    if (max_order <= 0) {
	max_order = -1;
    }
    sidelobe = getkaiserparamexf(sidelobe, trans, max_order + 1, beta, order);

    if (hp_flag && *order % 2 == 0) {
	/* highpass case: when length is even, make order even */
    } else {
	*order -= 1;
    }
    spDebug(100, "kaiserordexf", "result: order = %ld, sidelobe = %f\n", *order, sidelobe);

    return sidelobe;
}

double kaiserordex(double cutlow, double cuthigh, double devlow, double devhigh, spBool hp_flag,
		   long max_order, long *order, double *cutoff, double *beta)
{
    double trans;
    double sidelobe;
    double dev;

    spDebug(100, "kaiserordex", "cutlow = %f, cuthigh = %f, devlow = %f, devhigh = %f\n",
	    cutlow, cuthigh, devlow, devhigh);
    
    if (cutoff != NULL) {
	*cutoff = (cutlow + cuthigh) / 2.0;
    }
    
    trans = cuthigh - cutlow;
    dev = MIN(devlow, devhigh);
    
    if (dev == 0.0) {
	sidelobe = 200.0;
    } else {
	sidelobe = -20.0 * log10(dev);
    }
    
    spDebug(100, "kaiserordex", "trans = %f, dev = %f, sidelobe = %f\n", trans, dev, sidelobe);
    
    if (max_order <= 0) {
	max_order = -1;
    }
    sidelobe = getkaiserparamex(sidelobe, trans, max_order + 1, beta, order);

    if (hp_flag && *order % 2 == 0) {
	/* highpass case: when length is even, make order even */
    } else {
	*order -= 1;
    }
    spDebug(100, "kaiserordex", "result: order = %ld, sidelobe = %f\n", *order, sidelobe);

    return sidelobe;
}

void kaiserordf(float cutlow, float cuthigh, float devlow, float devhigh, spBool hp_flag,
                long *order, float *cutoff, float *beta)
{
    kaiserordexf(cutlow, cuthigh, devlow, devhigh, hp_flag, 0, order, cutoff, beta);
    return;
}

void kaiserord(double cutlow, double cuthigh, double devlow, double devhigh, spBool hp_flag,
	       long *order, double *cutoff, double *beta)
{
    kaiserordex(cutlow, cuthigh, devlow, devhigh, hp_flag, 0, order, cutoff, beta);
    return;
}

float ai0f(float x)
{
    float d, ds, s;

    ds = 1.0f;
    d = 2.0f;
    s = ds;
    ds = (x * x) / 4.0f;
    while (ds >= 0.2e-8f * s) {
	d += 2.0f;
	s += ds;
	ds *= (x * x) / (d * d);
    }
    
    return s;
}

double ai0(double x)
{
    double d, ds, s;

    ds = 1.0;
    d = 2.0;
    s = ds;
    ds = (x * x) / 4.0;
    while (ds >= 0.2e-8 * s) {
	d += 2.0;
	s += ds;
	ds *= (x * x) / (d * d);
    }
    
    return s;
}

int kaisercoref(float *w, long n, float beta, float *rms)
{
    float an1, t, ai0beta;
    long i;

    if (n <= 1) 
	return SP_FAILURE;

    if (rms != NULL) {
	*rms = 0.0f;
    }
    
    an1 = 1.0f / (float)(n - 1);
    ai0beta = ai0f(beta);
    for(i = 0; i < n; i++) {
	t = ((float)(2 * i - n + 1)) * an1;
	w[i] = ai0f(beta * sqrtf(1.0f - (t * t)));
	w[i] /= ai0beta;

	if (rms != NULL) {
	    *rms += w[i] * w[i];
	}
    }

    if (rms != NULL) {
	*rms = sqrtf(*rms / (float)n);
    }
    
    return SP_SUCCESS;
}

int kaisercore(double *w, long n, double beta, double *rms)
{
    double an1, t, ai0beta;
    long i;

    if (n <= 1) 
	return SP_FAILURE;

    if (rms != NULL) {
	*rms = 0.0;
    }
    
    an1 = 1.0 / (double)(n - 1);
    ai0beta = ai0(beta);
    for(i = 0; i < n; i++) {
	t = ((double)(2 * i - n + 1)) * an1;
	w[i] = ai0(beta * sqrt(1.0 - t * t));
	w[i] /= ai0beta;

	if (rms != NULL) {
	    *rms += w[i] * w[i];
	}
    }

    if (rms != NULL) {
	*rms = sqrt(*rms / (double)n);
    }
    
    return SP_SUCCESS;
}

int kaiserf(float *w, long n, float beta)
{
    return kaisercoref(w, n, beta, NULL);
}

int kaiser(double *w, long n, double beta)
{
    return kaisercore(w, n, beta, NULL);
}

int nkaiserf(float *w, long n, float beta)
{
    long k;
    float rms = 0.0;
    
    if (kaisercoref(w, n, beta, &rms) == SP_SUCCESS) {
	for (k = 0; k < n; k++) {
	    w[k] /= rms;
	}

	return SP_SUCCESS;
    } else {
	return SP_FAILURE;
    }
}

int nkaiser(double *w, long n, double beta)
{
    long k;
    double rms = 0.0;
    
    if (kaisercore(w, n, beta, &rms) == SP_SUCCESS) {
	for (k = 0; k < n; k++) {
	    w[k] /= rms;
	}

	return SP_SUCCESS;
    } else {
	return SP_FAILURE;
    }
}

void fvkaiser(spFVector vec, float beta)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvislocked(vec) == SP_FALSE && vec->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->kaiser != NULL) {
	flist->kaiser(vec->instance, vec->length, beta);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    kaiserf(vec->data, vec->length, beta);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvkaiser(spDVector vec, double beta)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvislocked(vec) == SP_FALSE && vec->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->kaiser != NULL) {
        flist->kaiser(vec->instance, vec->length, beta);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    kaiser(vec->data, vec->length, beta);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvkaiserul(spPlugin *plugin, long length, float beta, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvkaiser(vec, beta);

    return vec;
}

spDVector xdvkaiserul(spPlugin *plugin, long length, double beta, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvkaiser(vec, beta);

    return vec;
}

spFVector xfvkaiser(long length, float beta)
{
    return xfvkaiserul(NULL, length, beta, SP_FALSE);
}

spDVector xdvkaiser(long length, double beta)
{
    return xdvkaiserul(NULL, length, beta, SP_FALSE);
}

void fvnkaiser(spFVector vec, float beta)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvislocked(vec) == SP_FALSE && vec->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nkaiser != NULL) {
	flist->nkaiser(vec->instance, vec->length, beta);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nkaiserf(vec->data, vec->length, beta);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnkaiser(spDVector vec, double beta)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvislocked(vec) == SP_FALSE && vec->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nkaiser != NULL) {
        flist->nkaiser(vec->instance, vec->length, beta);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nkaiser(vec->data, vec->length, beta);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnkaiserul(spPlugin *plugin, long length, float beta, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvnkaiser(vec, beta);

    return vec;
}

spDVector xdvnkaiserul(spPlugin *plugin, long length, double beta, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnkaiser(vec, beta);

    return vec;
}

spFVector xfvnkaiser(long length, float beta)
{
    return xfvnkaiserul(NULL, length, beta, SP_FALSE);
}

spDVector xdvnkaiser(long length, double beta)
{
    return xdvnkaiserul(NULL, length, beta, SP_FALSE);
}
