/*
 * lpc.c : LPC-based analysis/synthesis routines
 *
 * Some codes are based on LPC library in C developed at Itakura laboratory (Nagoya University)
 * from 1984 to 1990, and the LPC library was ported from fortran-based LPC library developed at NTT.
 * We have got permission from Prof. Itakura to release our routines based on these libraries.
 */

#include <stdio.h>
#include <math.h>

#include <sp/spBase.h>
#include <sp/spMemory.h>
#include <sp/lpc.h>

long _SigToCor(double *sig, long siglen, double *cor, long corlen, double *cor0)
{
    long m;
    long t;
    long len;
    double sumsig;
    double factor;

    if (sig == NULL || siglen <= 0 || cor == NULL || corlen <= 0) return -1;

    if (cor0 != NULL) {
        len = MIN(siglen, corlen + 1);
    } else {
        len = MIN(siglen, corlen);
    }

    factor = (double)siglen;
    
    for (m = 0; m < len; m++) {
	sumsig = 0.0;
	for (t = 0; t < siglen - m; t++) {
	    sumsig += sig[t] * sig[t + m];
	}
	if (cor0 != NULL) {
            if (m == 0) {
                factor = *cor0 = sumsig;
            } else {
                cor[m - 1] = sumsig / factor;
            }
	} else {
            cor[m] = sumsig / factor;
        }
    }
    if (cor0 != NULL && m - 1 < corlen) {
        cor[m - 1] = 0.0;
    }
    for (; m < corlen; m++) {
	cor[m] = 0.0;
    }

    return corlen;
}

long spSigToCor(double *sig, long siglen, double *cor, long corlen)
{
    return _SigToCor(sig, siglen, cor, corlen, NULL);
}

long spSigToNCor(double *sig, long siglen, double *ncor, long ncorlen, double *cor0 /* can be NULL*/)
{
    double cor0t = 0.0;
    
    if (cor0 == NULL) {
        cor0 = &cor0t;
    }
    return _SigToCor(sig, siglen, ncor, ncorlen, cor0);
}

long _SigToCorFFT(spFFTRec fftrec, double *sig, long siglen, double *cor, long corlen, double *cor0)
{
    long k;
    long N;
    double *data;
    double factor;

    if (fftrec == NULL || sig == NULL || siglen <= 0
	|| cor == NULL || corlen <= 0) return -1;

    N = spGetFFTLength(fftrec);
    data = xalloc(N, double);

    for (k = 0; k < siglen; k++) {
	data[k] = sig[k];
	if (k >= N) {
	    break;
	}
    }
    for (; k < N; k++) {
	data[k] = 0.0;
    }
    
    if (spExecRealFFT(fftrec, data, 0) == SP_TRUE) {
	data[0] *= data[0];
	data[1] *= data[1];
	for (k = 2; k < N; k += 2) {
	    data[k] = (data[k] * data[k]) + (data[k+1] * data[k+1]);
	    data[k+1] = 0.0;
	}
	if (spExecRealFFT(fftrec, data, 1) == SP_TRUE) {
	    if (cor0 != NULL) {
		factor = *cor0 = data[0];
                for (k = 1; k <= corlen; k++) {
                    cor[k - 1] = data[k] / factor;
                }
	    } else {
		factor = (double)siglen;
                for (k = 0; k < corlen; k++) {
                    cor[k] = data[k] / factor;
                }
	    }
	} else {
	    corlen = -1;
	}
    } else {
	corlen = -1;
    }

    xfree(data);

    return corlen;
}

long spSigToCorFFT(spFFTRec fftrec, double *sig, long siglen, double *cor, long corlen)
{
    return _SigToCorFFT(fftrec, sig, siglen, cor, corlen, NULL);
}

long spSigToNCorFFT(spFFTRec fftrec, double *sig, long siglen, double *ncor, long ncorlen, double *cor0 /* can be NULL*/)
{
    double cor0t = 0.0;
    
    if (cor0 == NULL) {
        cor0 = &cor0t;
    }
    return _SigToCorFFT(fftrec, sig, siglen, ncor, ncorlen, cor0);
}

/* speclen must be larger than half of fft length */
long _SpecToCorFFT(spFFTRec fftrec, double *spec, long speclen, double *cor, long corlen, double preemph_db, double *cor0)
{
    long k;
    long N;
    long fftl2, hfftl;
    double *data;
    double factor;
    double specval;
    double log_e2 = 1.0;
    double preemph_weight = 1.0;
    
    if (fftrec == NULL || spec == NULL || speclen <= 0
	|| cor == NULL || corlen <= 0) return -1;

    N = spGetFFTLength(fftrec);
    fftl2 = N / 2;
    if (speclen < fftl2) {
	return -1;
    }
    hfftl = fftl2 + 1;
    data = xalloc(N, double);

    if (preemph_db != 0.0) {
	preemph_weight = preemph_db / 20.0;
	log_e2 = log(2.0);
    }
    
    data[0] = spec[0] * spec[0];
    if (speclen >= hfftl) {
	specval = spec[fftl2];
	if (preemph_db != 0.0) {
	    specval *= pow(10.0, preemph_weight * log((double)(fftl2 + 1)) / log_e2);
	}
	
	data[1] = specval * specval;
    } else {
	data[1] = 0.0;
    }
    for (k = 1; k < fftl2; k++) {
	specval = spec[k];
	if (preemph_db != 0.0) {
	    specval *= pow(10.0, preemph_weight * log((double)(k + 1)) / log_e2);
	}
	
	data[k * 2] = (specval * specval);
	data[k * 2 + 1] = 0.0;
    }
    if (spExecRealFFT(fftrec, data, 1) == SP_TRUE) {
	if (cor0 != NULL) {
	    factor = data[0];
            for (k = 1; k < corlen; k++) {
                cor[k - 1] = data[k] / factor;
            }
	} else {
            for (k = 0; k < corlen; k++) {
                cor[k] = data[k];
            }
	}
    } else {
	corlen = -1;
    }

    xfree(data);

    return corlen;
}

long spSpecToCorFFT(spFFTRec fftrec, double *spec, long speclen, double *cor, long corlen, double preemph_db)
{
    return _SpecToCorFFT(fftrec, spec, speclen, cor, corlen, preemph_db, NULL);
}

long spSpecToNCorFFT(spFFTRec fftrec, double *spec, long speclen, double *ncor, long ncorlen, double preemph_db, double *cor0 /* can be NULL */)
{
    double cor0t = 0.0;
    
    if (cor0 == NULL) {
        cor0 = &cor0t;
    }
    return _SpecToCorFFT(fftrec, spec, speclen, ncor, ncorlen, preemph_db, cor0);
}

long _spCorToSpecFFT(spFFTRec fftrec, double *cor, long corlen, double *cor0,
                     double *spec, long speclen, double deemph_db)
{
    long k;
    long N, Nleft;
    long fftl2, hfftl;
    double *data;
    double log_e2 = 1.0;
    double deemph_weight = 1.0;
    
    if (fftrec == NULL || spec == NULL || speclen <= 0
	|| cor == NULL || corlen <= 0) return -1;
    
    N = spGetFFTLength(fftrec);
    fftl2 = N / 2;
    hfftl = fftl2 + 1;

    data = xalloc(N, double);

    if (cor0 != NULL) {
        data[0] = *cor0;
        for (k = 1; k < corlen; k++) {
            data[k] = cor[k - 1];
            if (k >= fftl2) {
                break;
            }
            data[N - k] = data[k];
        }
    } else {
        data[0] = cor[0];
        for (k = 1; k < corlen; k++) {
            data[k] = cor[k];
            if (k >= fftl2) {
                break;
            }
            data[N - k] = data[k];
        }
    }
    Nleft = N - corlen + 1;
    for (; k < Nleft; k++) {
	data[k] = 0.0;
    }

    if (spExecRealFFT(fftrec, data, 0) == SP_TRUE) {
	if (deemph_db != 0.0) {
	    deemph_weight = deemph_db / 20.0;
	    log_e2 = log(2.0);
	}
    
	spec[0] = sqrt(MAX(data[0], 0.0));
	
	for (k = 1; k < speclen; k++) {
	    if (k >= fftl2) {
		spec[fftl2] = sqrt(MAX(data[1], 0.0));
		if (deemph_db != 0.0) {
		    spec[fftl2] /= pow(10.0, deemph_weight * log((double)(fftl2 + 1)) / log_e2);
		}
		speclen = fftl2 + 1;
		break;
	    }
	    spec[k] = sqrt(MAX(data[k * 2], 0.0));
	    if (deemph_db != 0.0) {
		spec[k] /= pow(10.0, deemph_weight * log((double)(k + 1)) / log_e2);
	    }
	    
	    if (speclen > fftl2) {
		spec[N - k] = spec[k];
	    }
	}
    } else {
	speclen = -1;
    }
    
    xfree(data);

    return speclen;
}

long spCorToSpecFFT(spFFTRec fftrec, double *cor, long corlen, double *spec, long speclen, double deemph_db)
{
    return _spCorToSpecFFT(fftrec, cor, corlen, NULL, spec, speclen, deemph_db);
}

long spNCorToSpecFFT(spFFTRec fftrec, double *ncor, long ncorlen, double cor0,
                     double *spec, long speclen, double deemph_db)
{
    return _spCorToSpecFFT(fftrec, ncor, ncorlen, &cor0, spec, speclen, deemph_db);
}

long _spCorToParcor(double *cor, double *alpha, double *parcor, long order, double *resid, spBool ncor_flag)
{
    long m, i;
    double wm, um;
    long m_minus_1;
    long m_minus_i;
    double parcor_m;
    double alpha_i;
    double alpha_m_minus_i;
    
    if (cor == NULL || alpha == NULL || order <= 0) return -1;

    if (ncor_flag) {
        /* initialize */
        wm = cor[0];
        um = 1.0;

        /* m = 1 */
        parcor_m = wm;
        cor--;
    } else {
        /* initialize */
        wm = cor[1];
        um = cor[0];

        /* m = 1 */
        parcor_m = wm / um;
    }
    um *= (1.0 - (parcor_m * parcor_m));
    alpha[0] = -parcor_m;
    if (parcor != NULL) parcor[0] = parcor_m;
	
    for (m = 2; m <= order; m++) {
	m_minus_1 = m - 1;
	
	wm = cor[m];
	for (i = 1; i <= m_minus_1; i++) {
	    wm += alpha[i - 1] * cor[m - i];
	}

	parcor_m = wm / um;
	um *= (1.0 - (parcor_m * parcor_m));

	if (parcor != NULL) parcor[m_minus_1] = parcor_m;
	alpha[m_minus_1] = -parcor_m;

	for (i = 1;; i++) {
	    m_minus_i = m - i;
	    if (i > m_minus_i) {
		break;
	    }

            alpha_i = alpha[i - 1];
	    alpha_m_minus_i = alpha[m_minus_i - 1];
	    if (i < m_minus_i) {
		alpha[m_minus_i - 1] = alpha_m_minus_i - parcor_m * alpha_i;
	    }
	    alpha[i - 1] = alpha_i - parcor_m * alpha_m_minus_i;
	}
    }

    if (resid != NULL) *resid = um;
    
    return order;
}

long spCorToParcor(double *cor, double *alpha, double *parcor, long order, double *resid)
{
    return _spCorToParcor(cor, alpha, parcor, order, resid, SP_FALSE);
}

long spNCorToParcor(double *ncor, double *alpha, double *parcor, long order, double *resid)
{
    return _spCorToParcor(ncor, alpha, parcor, order, resid, SP_TRUE);
}

long spCorToAlpha(double *cor, double *alpha, long order, double *resid)
{
    return spCorToParcor(cor, alpha, NULL, order, resid);
}

long spNCorToAlpha(double *cor, double *alpha, long order, double *resid)
{
    return spNCorToParcor(cor, alpha, NULL, order, resid);
}

long spParcorToAlpha(double *parcor, double *alpha, long order)
{
    int m, i, mi;
    double parcor_m, prev_alpha_i;

    if (order <= 0) return -1;
	
    alpha[0] = -parcor[0];
    
    if (order <= 1) return 1;
    
    for (m = 1; m < order; m++) {
	parcor_m = -parcor[m];
	alpha[m] = parcor_m;
	i = 0;
	do {
	    mi = m - i - 1;
	    prev_alpha_i = alpha[i];
	    alpha[i] = prev_alpha_i + alpha[mi] * parcor_m;
	    if (i >= mi)
		break;
	    
	    alpha[mi] = alpha[mi] + prev_alpha_i * parcor_m;
	    i++;
	} while (i < mi);
    }
    
    return order;
}

long spAlphaToParcor(double *alpha, double *parcor, long order, double *resid, spBool *is_stable)
{
    int    m, j, k;
    double  parcor_m, parcor_j, res_m;

    if (resid != NULL) *resid = 1.0;
    if (is_stable != NULL) *is_stable = SP_TRUE;

    if (order <= 0) return -1;

    for (j = 0; j < order; j++)
	parcor[j] = alpha[j];
    
    m = order - 1;
    do {
	parcor_m = -parcor[m];
	res_m = 1.0 - parcor_m * parcor_m;
	if( res_m <= 0.0 ) {
	    if (is_stable != NULL) *is_stable = SP_FALSE;
	}
	
	if (resid != NULL) *resid *= res_m;
	
	for (j = m - 1, k = 0; j >= k; j--, k++) {
	    parcor_j = parcor[j];
	    parcor[j] = ( parcor_j + parcor_m * parcor[k] ) / res_m;
	    if( j != k )
		parcor[k] = ( parcor[k] + parcor_m * parcor_j ) / res_m;
	    else
		break;
	}
	parcor[m] = parcor_m;
	m--;
    } while ( m > -1 );

    return order;
}

long spAlphaToPSD(double *alpha, long order, double resid, double *psd, long Nhalf)
{
    long i;
    long j;
    long k;
    double A;
    double omega, domega;
    double weight;

    if (alpha == NULL || psd == NULL || order <= 0 || Nhalf < 2) return -1;
    
    domega = PI / (double)(Nhalf - 1);
    
    for (i = 0; i <= order; i++) {
	if (i == 0) {
            A = 1.0;
        } else {
            A = alpha[i - 1];
        }
	for (j = 0; j < order - i; j++) {
	    A += alpha[j] * alpha[j + i];
	}
	if (i == 0) {
	    for (k = 0; k < Nhalf; k++) {
		psd[k] = A;
	    }
	} else {
	    omega = 0.0;
	    A *= 2.0;
	    for (k = 0; k < Nhalf; k++) {
		psd[k] += A * cos((double)i * omega);
		omega += domega;
	    }
	}
    }

    weight = resid/* / (2.0 * PI)*/;
    for (k = 0; k < Nhalf; k++) {
	psd[k] = weight / psd[k];
    }

    return Nhalf;
}

long spAlphaToPSDFFT(double *alpha, long order, double resid, double *psd, long fftl)
{
    long i;
    long fftl2;
    double re0, im0;

    if (alpha == NULL || psd == NULL || order <= 0 || fftl < 4) return -1;

    memset(psd, 0, fftl * sizeof(double));
    psd[0] = 1.0;
    for (i = 1; i <= order; i++) {
        psd[i] = alpha[i - 1];
    }
    
    sprfft(psd, fftl, 0);

    fftl2 = fftl / 2;
    re0 = resid / SQUARE(psd[0]);
    im0 = resid / SQUARE(psd[1]);
    for (i = 1; i < fftl2; i++) {
        psd[i] = resid / (SQUARE(psd[2 * i]) + SQUARE(psd[2 * i + 1]));
    }
    psd[0] = re0;
    psd[fftl2] = im0;

    fftturn(psd, NULL, fftl);

    return fftl;
}

long spParcorToVTAreaRatio(double *parcor, double *vtar, long order, spBool linear)
{
    long i;
    
    if (parcor == NULL || vtar == NULL || order <= 0) return -1;

    for (i = 0; i < order; i++) {
	vtar[i] = (1.0 - parcor[i]) / (1.0 + parcor[i]);
	if (!linear) {
	    vtar[i] = log(vtar[i]);
	}
    }
    
    return order;
}

long spVTAreaRatioToParcor(double *vtar, double *parcor, long order, spBool linear)
{
    long i;
    double vtar_i;
    
    if (parcor == NULL || vtar == NULL || order <= 0) return -1;

    for (i = 0; i < order; i++) {
	if (!linear) {
	    vtar_i = exp(vtar[i]);
	} else {
	    vtar_i = vtar[i];
	}
	parcor[i] = -(vtar_i - 1.0) / (vtar_i + 1.0);
    }
    
    return order;
}

long spVTRatioToArea(double *vtar, double *vta, long order, double ref_area, spBool linear)
{
    long i;
    double vtar_i;
    double prev_vta;
    
    if (vtar == NULL || vta == NULL || order <= 0 || ref_area == 0.0) return -1;

    prev_vta = ref_area;
    
    for (i = order - 1; i >= 0; i--) {
	if (!linear) {
	    vtar_i = exp(vtar[i]);
	} else {
	    vtar_i = vtar[i];
	}
	vta[i] = prev_vta / vtar_i;
	prev_vta = vta[i];
    }

    return order;
}

long spVTAreaToRatio(double *vta, double *vtar, long order, double ref_area, spBool linear)
{
    long i;
    double vtar_i;
    
    if (vtar == NULL || vta == NULL || order <= 0 || ref_area == 0.0) return -1;
    
    for (i = 0; i < order; i++) {
	if (i == order - 1) {
	    vtar_i = ref_area / vta[i];
	} else {
	    vtar_i = vta[i + 1] / vta[i];
	}

	if (!linear) {
	    vtar[i] = log(vtar_i);
	} else {
	    vtar[i] = vtar_i;
	}
    }

    return order;
}

long spParcorToVTArea(double *parcor, double *vta, long order, double ref_area)
{
    long i;
    double prev_vta;
    
    if (parcor == NULL || vta == NULL || order <= 0 || ref_area == 0.0) return -1;

    prev_vta = ref_area;
    for (i = order - 1; i >= 0; i--) {
	vta[i] = (1.0 + parcor[i]) / (1.0 - parcor[i]) * prev_vta;
        prev_vta = vta[i];
    }
    
    return order;
}

long spVTAreaToParcor(double *vta, double *parcor, long order, double ref_area)
{
    long i;
    double current_vta, prev_vta;
    
    if (parcor == NULL || vta == NULL || order <= 0 || ref_area == 0.0) return -1;

    prev_vta = ref_area;
    for (i = order - 1; i >= 0; i--) {
        current_vta = vta[i];
	parcor[i] = (current_vta - prev_vta) / (current_vta + prev_vta);
        prev_vta = current_vta;
    }
    
    return order;
}

#if 0
long spAdaptiveInverseFilterStageFFT(spFFTRec fftrec, double *spec, double *ospec, double *aispec,
				     long speclen, int stage) /* stage: 1-5 */
{
    spSpecToCorFFT(fftrec, spec, speclen, ospec, speclen);
    
}
#endif

double spAlphaSynFiltering(double *alpha, double *state, long order, double e_t)
{
    long t;
    double x_t = 0.0;

    if (alpha == NULL || state == NULL || order <= 0) return 0.0;
    
    for (t = order; t >= 1; t--) {
	x_t -= alpha[t] * state[t - 1];
	if (t < order) {
	    state[t] = state[t - 1];
	}
    }
    x_t += e_t;
    state[0] = x_t;
    
    return x_t;
}

double spParcorSynFiltering(double *parcor, double *state, long order, double e_t)
{
    long m;
    double x_t = 0.0;
    double out_m = 0.0;
    double out_m_plus_1;
    double state_m_plus_1;
    
    if (parcor == NULL || state == NULL || order <= 0) return 0.0;
    
    out_m_plus_1 = e_t;

    for (m = order - 1; m >= 0; m--) {
	out_m = out_m_plus_1 + (parcor[m] * state[m]);
	state_m_plus_1 = state[m] - (parcor[m] * out_m);
	
	if (m <= order - 2) {
	    state[m + 1] = state_m_plus_1;
	}
	
	out_m_plus_1 = out_m;
    }
    x_t = out_m;
    state[0] = x_t;
    
    return x_t;
}

/********************************************************************
*      The following functions are modified for spLib by H.Banno    *
*********************************************************************/

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*								    *
*               CODED BY S.SAGAYAMA,                 5/5/1982       *
*	Trans. From F77 to C by H.Nakaiwa 	      11/7/1986	    *
*	Last Modified by M.Ikeda      		    Jul/15/1989     *
*								    *
*********************************************************************

  Description:
    * Expansion of linear combination of Tchebycheff polynomials
 
	T(n,x)+a[0]*T(n-1,x)+a[1]*T(n-2,x)+ ... +a[n-1]*T(0,x)
 
      into a polynomial of x ** n.
        n           n-1          n-2
 	x  + b[0] * x   + b[1] * x   + ... + b[n-1]
 
    * This routine uses following recursion
 	T(0, x) = 1
 	T(1, x) = x
 	   ...
 	T(n, x) = 2 * x  * T(n-1, x) - T(n-2, x)
 
  ---------------
  excheb(n, a, b)
  ---------------
  Arguments
    long   n   : order of polynominal
    double a[] : input Tchebyshev polynominal coefficients
 		 its 0-th coefficient is implicitly assumed to be 1.0
    double b[] : output argebric  polynominal coefficients
 		 its 0-th coefficient is implicitly assumed to be 1.0
  Note
    return algebric polynominal is normalized to be 0-th coefficient 1.0
    (or normalized by 2**(n-1))

 */
static long excheb(long n, double *a, double *b)
{
    long i, j, nm1, nm2;
    double x, t, *fw;	/* fw : work for expansion */

    if (n <= 0) return -1;

    fw = xalloc(n + 1, double);
    
    nm1 = n - 1;
    nm2 = n - 2;

    fw[n] = 0.0;
    fw[nm1] = 1.0;
    fw[nm2] = 1.0;

    b[nm1] = a[nm1];
    b[nm2] = a[nm2];

    for (i = n - 3; i >= 0; --i) {
	t     = a[i];
	b[i]  = 0.0;
	fw[i] = 0.0;
	for (j = i; j < n; j += 2) {
	    x  =  2.0 * fw[j + 1] - fw[j];
	    b[j] +=  t * x;
	    fw[j] = x;
	}
    }
    /* last step (a[-1] = 0th expansion coefficent = 1.0) */
    for (j = 1; j < n; j += 2) {
	x = 2.0 * fw[j+1] - fw[j];
	b[j] += x;
	fw[j] = x;
    }
    t =  0.5 / fw[0];
    for (i = 0; i < n; ++i) b[i] *= t;

    xfree(fw);
    
    return n;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  K.Yoshida                 Nov./1986    *
*                                                                   *
*********************************************************************
---  nrstep  ---
   description:
     * apply a single step of newton-raphson iteration to an
       algebraic equation and divide it by ( x - root ).
     * the necesary and sufficient condition for existence of
       the solution is that all the roots of polynomial a(z) lie
       inside the unit circle.

   calling sequence:
          ------------------
           nrstep(coef,n,&x)
          ------------------
   coef[.] : input/output. real array : dimension=n.
             coefficients of the n-th order polynomial (input).
             coefficients of new (n-1)-th order polynomial (output).
             new polynomial(x) = given polynomial(x)/(x-root).
             coef(0) is implicitly assumed to be 1.0.
   n       : input.        integer.
             the order of the polynomial (n>0).
   x       : input/output. real.
             initial value of the iteration (input),
             the root (output).

   note: * originally coded by f.itakura (original name was "newton"),
           renamed and revised by s.sagayama,  1981.11.14. 
         * effective dimension of "coef" changes into (n-1) after
           calling this subroutine.  coef(n) is meaningless.
*/
static long nrstep(double *coef, long n, double *x)
{
    static double  eps = 1.0e-6;
    long           i;
    double         bb, aa, dx;

    if ( n <= 0 ) return -1;

    if ( n < 2 ) {
	*x = -coef[0];
	return n;
    }

    do {
	bb = 1.0;
	aa = *x + coef[0];
	for ( i = 2 ; i <= n ; ++i ) {     
	    bb = bb * *x + aa;
	    aa = aa * *x + coef[i - 1];
	}
	dx = aa / bb;
	*x = *x - dx;
    } while ( dx > eps );
      
    coef[0] = coef[0] + *x;
    if ( n != 2 )
	for ( i = 2 ; i <= n - 1 ; ++i )
	    coef[i - 1] = coef[i - 1] + *x * coef[i - 2];

    return n;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  T.Umezaki                 Oct./1986    *
*                                                                   *
*********************************************************************

---  spAlphaToLSP  ---

     Description:
       * Conversion of "alpha" into "lsp".
       * Computation of line spectrum pair frequencies from linear
         prediction coefficients.
       * Computation of roots of p(x) and q(x):
                                ip
             p(x) = z * a(z) - z   * a(1/z)
                                ip
             q(x) = z * a(z) + z   * a(1/z)
         where
                     ip           ip-1
             a(z) = z   + a(1) * z     + .... + a(ip).
         In case ip=even, the roots of p(x) are cosine of pi times of:
            0, lsp(2), lsp(4), ... , lsp(ip);
         and the roots of p(x) are cosine of pi times of:
            lsp(1), lsp(3), ... ,lsp(ip-1), 1.
       * The necesary and sufficient condition for existence of
         the solution is that all the roots of polynomial a(z) lie
         inside the unit circle.

     Calling sequence:
            ------------------------
            long spAlphaToLSP(double alpha[], double lsp[], long order)
            ------------------------
     ip      : Input.        Integer.     2 =< ip =< 14.
               The order of analysis; the number of poles in LPC;
     alpha[ ]  : Input.        Real array : dimension=ip.
               Linear prediction coefficients; AR parameters.
               alpha(0) is implicitly assumed to be 1.0.
     lsp[ ] : Output.       Real array : dimension=ip.
               LSP frequencies, ranging between 0 and 1;
               CSM frequencies under two diferrent conditions,
                 ip=even:   lsp(0)=0,order=n / lsp(n+1)=1,order=n
                 ip=odd:    order=n / lsp(0)=0,lsp(n+1)=1,order=n-1,
               where n=((ip+1)/2).
               Increasingly ordered.  fq(1)=<fq(2)=<.....

     Note: (1) ip must not greater than 20. (limit of "excheb")
           (2) subroutine call: "excheb", "nrstep".
----------------------------------------------------------------------*/
long spAlphaToLSP(double *alpha, double *lsp, long order)
{
    double  *opm, *opp, b, x;
    int    nm, np, km, kp, i, j;

    if (order <= 0) return -1;
    
    np = order / 2;
    nm = order - np;

    opm = xalloc(nm, double);
    opp = xalloc(nm, double);

    if (nm == np) {
        /*........... In case of order = even ...........*/
        opp[0] = alpha[0] - alpha[order-1] + 1.0;
        opm[0] = alpha[0] + alpha[order-1] - 1.0;
        if (nm > 1) {
            for (i = 1; i <= nm - 1; i++) {
                b = alpha[order-i-1];
                opp[i] = alpha[i] - b + opp[i-1];
                opm[i] = alpha[i] + b - opm[i-1];
            }
        }
        opp[np-1] *= .5; 
        excheb(np, opp, opp);
    } else {
        /*........... In case of order = odd  ...........*/
        opm[0] = alpha[0] + alpha[order-1];
        if (nm > 1) {
            opp[0] = alpha[0] - alpha[order-1];
            opm[1] = alpha[1] + alpha[order-2];
            if (nm > 2) {
                opp[1] = alpha[1] - alpha[order-2] + 1.0;
                for (i = 2; i <= nm - 1; i++)
                    opm[i] = alpha[i] + alpha[order-i-1];
                if (np > 2) {
                    for ( i = 2; i <= np - 1; i++)
                        opp[i] = alpha[i] - alpha[order-i-1] + opp[i-2];
                }
            }
            opp[np-1] *= .5;  
            excheb(np, opp, opp);
        }
    }
    opm[nm-1] *= .5;  
    excheb(nm, opm, opm);
    if (order != 1) {
        if (order != 2) {
            kp = np;
            km = nm;
            /*............... Find roots of the polynomials ..........*/
            j = 2;
            /*                Initial value for roots.                */
            x = 1.0;

            for (i = 0; i <= order - 3; i++) {
                /*                    Index. If j=1, minus;j=2, plus. */
                j = 3 - j;
                if (j != 2) {   /*  Newton-raphson step for real zero */
                    nrstep(opm, km, &x);
                    km--;
                } else {        /*  Newton-raphson step for real zero */
                    nrstep(opp, kp, &x);
                    kp--;
                }
                lsp[i] = x;
            }

            if (j == 1) {
                lsp[order-2] = -opp[0];    /* order = odd.  */
                lsp[order-1] = -opm[0];    /* order = 1.    */
            } else {
                lsp[order-2] = -opm[0];    /* order = even. */
                lsp[order-1] = -opp[0];
            }

        } else {
            lsp[order-2] = -opm[0];        /* order = even. */
            lsp[order-1] = -opp[0];
        }

    } else 
	lsp[order-1] = -opm[0];

    /*        Arccosine conversion for real frequencies         */
    for (i = 0; i <= order - 1; i++) 
        lsp[i] = acos(lsp[i]) / PI;

    xfree(opm);
    xfree(opp);
    
    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*								    *
*               coded by S.Sagayama,                 5/6/1982       *
*               transrated into C  by K.Yoshida     11/5/1986       *
*                                                                   *
********************************************************************/
/*
 * --- MPQUAD.C ---
 *
 * Description:
 *   * multiply a reciprocal polynomial:
 *                n        n-1                           n-1     n
 *        s(x) = z + p(1)*z   + ... + p(n) + ... + p(1)/z   + 1/z
 *
 *     by a reciprocal quadratic factor:
 *
 *        q(z) = z + a + 1/z.
 *
 * Calling sequence:
 *        ---------------
 *        mpquad(p,&n,a)
 *        ---------------
 * Arguments
 *   p[.]    : input/output. real array : dimension=n.
 *             coefficients of s(x). p(0)=1.0 implicitly.
 *             p(1),...,p(n) are given.
 *   n       : input/output. integer.     n <-- n+1.
 *             dimension of the array "p", which is automatically
 *             incremented by 1 after calling this routine.
 *   a       : input.        real.
 *             the constant term in quadratic factor.
 */
static long mpquad(double *p, long *n, double a)
{
    register long i;

    if (*n < 0) return -1;

    if ( *n < 1 ) {
	*n = 1;
	p[0] = a;
	return *n;
    }
    if ( *n < 2 ) {
	*n = 2;
	p[1] = 2.0 + a * p[0];
	p[0] += a;
	return *n;
    }
    ++(*n);
    p[*n-1] = 2.0 * p[*n-3] + a * p[*n-2];
    if ( *n > 3 )
	for ( i= *n - 2; i >= 2; --i)
	    p[i] += a * p[i-1] + p[i-2];
    p[1] += a * p[0] + 1.0;
    p[0] += a;

    return *n;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*								    *
*               coded by S.Sagayama,                  5/5/1982      *
*               transrated into C  by  K.Yoshida,    11/5/1986      *
*                                                                   *
********************************************************************/
/* --- LSPALPHA.C ---
 * 
 * description:
 *  * conversion of "lsp" into "alpha".
 *  * retrieval of a polynomial a(z) giving normalized arguments of
 *    roots of polynomials of x:
 *
 *    in case order=even,
 *                               order                            (order+1)
 *        qm(x) = (  z * a(z) + z   * a(1/z)  ) / ( z + 1 ) / 2
 *                               order                            (order+1)
 *        qp(x) = (  z * a(z) - z   * a(1/z)  ) / ( z - 1 ) / 2
 *
 *    in case order=odd,
 *                               order                (order+2)
 *        qm(x) = (  z * a(z) + z   * a(1/z)  ) / 2
 *                               order                  2          order
 *        qp(x) = (  z * a(z) - z   * a(1/z)  ) / ( z  - 1 ) / 2
 *
 *    where
 *                order           order-1
 *        a(z) = z   + a(1) * z     + .... + a(order)
 *               2
 *        x = ( z  + 1 ) / 2.
 * -------------------------
 * long spLSPToAlpha(double lsp[], double alpha[], long order)
 * -------------------------
 * Arguments
 *   order     : input.        integer.
 *            the order of analysis; the number of poles in lpc;
 *   lsp[] : input.        real array : dimension=order.
 *            lsp frequencies ranging between 0.0 and 1.0,
 *            increasingly ordered, i.e.,lsp(0)<lsp(1)<..<lsp(lp-1).
 *   alpha[]  : output.       real array : dimension=order.
 *            linear prediction coefficients; ar parameters.
 *            alpha[-1] is implicitly assumed to be 1.0.
 * Notes
 *   function call: "mpquad"(multiply quadratic factors)
 */
long spLSPToAlpha(double *lsp, double *alpha, long order)
{
    long   np, nm, i, m;
    double  *opp, *opm, c, zm, zp;
    
    if ( order <= 0 ) return -1;

    opp = xalloc(order, double);
    
    m = order / 2;
    opm = opp + m;

/* ---- make products of quadratic factors ---- */

    np = nm = 0;
    for ( i = 0; i < order ; i++ ) {
	c = -2.0 * cos(lsp[i] * PI);
	if ( nm > np )
	    mpquad(opp, &np, c);  /*  i=2,4,6,...  */
	else
	    mpquad(opm, &nm, c);  /*  i=1,3,5,...  */
    }

    if ( nm == np ) {     /* --- in case of order=even --- */
	alpha[order-1] = 2.0 + opm[0] - opp[0];
	alpha[0] = opm[0] + opp[0];
	for ( i = 1; i < nm ; ++i) {
	    zm = opm[i] + opm[i-1];
	    zp = opp[i] - opp[i-1];
	    alpha[i] = zm + zp;
	    alpha[order-i-1] = zm-zp;
	}
    }else{               /* ---- in case of order=odd ---- */
	alpha[nm-1] = opm[nm-1];
	if ( nm > 1 ) {
	    alpha[0] = opm[0]+opp[0];
	    alpha[order-1] = opm[0]-opp[0];
	}
	if ( nm > 2 ) {
	    zm = opm[1];
	    zp = opp[1]-1.0;
	    alpha[1] = zm + zp;
	    alpha[order-2] = zm - zp;
	}
	if ( nm > 3 )
	    for ( i = 2 ; i < nm-1 ; i++ ) {
		zm = opm[i];
		zp = opp[i] - opp[i-2];
		alpha[i] = zm + zp;
		alpha[order-i-1] = zm - zp;
	    }
    }

    /* ---- normalization ---- */
    while(order--) *alpha++ *= 0.5;

    xfree(opp);
    
    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Sep./1976    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  H.Hoshino                 Nov./1986    *
*                                                                   *
*********************************************************************
 .   description:
 .     * conversion of "parcor" into "cor".
 .       "alpha" is simultaneously obtained.
 .     * computation of autocorrelation coefficients "cor"
 .       of an ar-process specified by "parcor" 
 .
 .   calling sequence:
 .          -------------------------------
 .          long _spParcorToCor(double *parcor, double *alpha, long order, double cor0,
 .                              double *cor, long n, double *resid)
 .          -------------------------------
 .   parcor[.]  : input.        real array : dimension=order.
 .             parcor coefficients; reflection coefficients.
 .             all of parcor[.] range between -1 and 1.
 .   alpha[.]  : output.       real array : dimension=order.
 .             linear prediction coefficients; ar parameters.
 .             The 1st, 2nd, ..., order-th order LPC coefficients
 .             correspond to alpha[0], alpha[1], ..., alpha[order-1],
 .             respectively.
 .             The 0th order LPC coefficient is implicitly
 .             assumed to be 1.0.
 .   order      : input.        integer.
 .             the order of analysis; the number of poles in lpc;
 .             the degree of freedom of the model - 1.
 .   cor0    : The 0th order autocorrelation coefficient.
 .   cor[.]  : output.       real array : dimension=order
 .             autocorrelation coefficients.
 .             The 0th, 1st, 2nd, ..., n-th order autocorrelation
 .             coefficients correspond to cor[0], cor[1], ...,
 .             cor[n-1], respectively.
 .   n       : input.        integer.
 .             the number of required points of autocorrelation.
 .             note that the degree of freedom remains order.
 .   resid   : output.       real.
 .             linear prediction / parcor residual power;
 .             reciprocal of power gain of parcor/lpc/lsp all-pole
 .             filter.
 .
 */
long _spParcorToCor(double *parcor, double *alpha, long order, double cor0, double *cor, long corlen, double *resid, spBool ncor_flag)
{
    double parcori, cori, prev_alpha;
    double resid1;
    long len;
    long i, im1, j, imj, order1;

    if (corlen <= 0) return -1;

    if (order <= 0) {
	for (i = 0; i < corlen; i++) cor[i] = 0.0;
	return -1;
    }

    resid1 = cor0;

    parcori = parcor[0];
    if (ncor_flag) {
        cor[0] = parcori;
        --cor;
        len = MIN(order, corlen);
    } else {
        cor[0] = cor0;
        cor[1] = parcori * cor0;
        len = MIN(order, corlen - 1);
    }
    alpha[0] = -parcori;
    resid1 *= (1.0 - parcori * parcori);
    if (order != 1) {
	for (i = 2; i <= len; i++) {
	    parcori = parcor[i - 1];
	    cori = parcori * resid1;
	    im1 = i - 1;
	    for (j = 1; j <= im1; j++) {
		imj = i - j;
		cori -= alpha[j - 1] * cor[imj];
	    }
	    cor[i] = cori;
	    alpha[i - 1] = -parcori;

	    j = 1;
	    do {
		prev_alpha = alpha[j - 1];
		imj = i - j;
		alpha[j - 1] = prev_alpha - parcori * alpha[imj - 1];
		if (j >= imj)
		    break;
		alpha[imj - 1] -= parcori * prev_alpha;
		j++;
	    } while (j < imj);
	    resid1 *= (1.0 - parcori * parcori);
	}
    }

    if (ncor_flag) {
        len = corlen + 1;
    } else {
        len = corlen;
    }
        
    if (len > order) {
	order1 = order + 1;
	for (i = order1; i < len; i++) {
	    cori = 0.0;
	    for (j = 1; j <= order; j++) {
		imj = i - j;
		cori -= alpha[j - 1] * cor[imj];
	    }
	    cor[i] = cori;
	}
    }

    if (resid != NULL) *resid = resid1;
    
    return corlen;
}

long spParcorToCor(double *parcor, double *alpha, long order, double cor0, double *cor, long corlen, double *resid)
{
    return _spParcorToCor(parcor, alpha, order, cor0, cor, corlen, resid, SP_FALSE);
}

long spParcorToNCor(double *parcor, double *alpha, long order, double *ncor, long ncorlen, double *resid)
{
    return _spParcorToCor(parcor, alpha, order, 1.0, ncor, ncorlen, resid, SP_TRUE);
}

/* signal & alpha --> residual signal */
long spSigAlphaToRessig(double *sig, long siglen, double *alpha, long order, double *ressig /* output */)
{
    double value;
    long i, j, k;

    if (sig == NULL || alpha == NULL || ressig == NULL
        || order <= 0 || siglen <= 0) return -1;

    for (i = 0; i < siglen; i++) {
	value = 0.0;
	for (j = 0; j < order; j++) {
	    k = i - j - 1;
	    if (k >= 0) {
		value -= alpha[j] * sig[k];
	    }
	}
	ressig[i] = sig[i] - value;
    }

    return siglen;
}

/* residual signal & alpha --> signal */
long spRessigAlphaToSig(double *ressig, long siglen, double *alpha, long order, double *sig /* output */)
{
    long i, j;
    double presig;

    if (ressig == NULL || alpha == NULL || sig == NULL
        || order <= 0 || siglen <= 0) return -1;

    for (i = 0; i < siglen; i++) {
        presig = 0.0;
        for (j = 0; j < order; j++) {
            if (i - j - 1>= 0) {
                presig -= alpha[j] * sig[i - j - 1];
            }
        }
        sig[i] = ressig[i] + presig;
    }

    return siglen;
}

/********************************************************************
*      The following functions are modified for spLibEx by H.Banno    *
*********************************************************************/

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  N.Morita                  Nov./1986    *
*                                                                   *
*********************************************************************

---  lagwdw  ---
 *
 *   description:
 *     * lag window (pascal) data generation.
 *
 *   calling sequence:
 *          -------------------
 *          spLagWindow(wdata,length,hb)
 *          -------------------
 *
 *   wdata[.]: output.       real array : dimension=length.
 *             linear prediction coefficients; ar parameters.
 *             alf(0) is implicitly assumed to be 1.0.
 *   length      : input.        integer.
 *             dimension of wdata(.).
 *   hb      : input.        real.
 *              0.0 < hb < 1.0 .    If hb=0, wdata(i)=1.0 for all i.
 *             window half value band width to sampling frequency ratio.
 *             example: lag window half value band width = 100 hz, and
 *             sampling frequency = 8 khz, then hb = 100/8k = 1/80 =0.0125
 *   note:
 *	lag window w(n) = conbination(2*K,K+n) / conbination(2*K,K)
 *	      Z{ w(n) } = ( 2*cos(omega/2) )**K / conbination(2*K,K)
 *
 */

long spLagWindow(double *wdata, long length, double hb)
{
    int	  i;
    double a, b, w;

    if (wdata == NULL || length  <= 0) return -1;

    if (hb <= 0.0) {
	for (i = 0; i < length ; i++) 
	    *wdata++ = 1.0;
	return length;
    }

    a = log(0.5) * 0.5 / log(cos(0.5 * PI * hb));
    a = (double)((int) a);
    w = 1.0;
    b = a;
    for (i = 0 ; i < length ; ++i) {
	b	+= 1.0;
	w 	*= a / b;
	*wdata++ = w;
	a 	-= 1.0;
    }
    
    return length;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  T.Umezaki                 Oct./1986    *
*                                                                   *
*********************************************************************

---  spAlphaToCep  ---

     Description:
       * Conversion of "alpha  into "cep".
       * Computation of cepstrum of AR-process specified by "alpha".
       * Computation of sum of m-th power of LPC poles, m=1,..,n,
         since:         1   order               m
               cep(m)= --- * sum ( (lpc pole(i))  ).
                        m    i=1
     Calling sequence:
            -------------------------
              spAlphaToCep(alpha, order, cep, ceplen);
            -------------------------
     order   : Input.        Integer.
               The order of analysis; the number of poles in LPC;
               the degree of freedom of the model - 1.
     alpha[ ]: Input.        Real array : dimension=order.
               Linear prediction coefficients; AR parameters.
               alpha(0) is implicitly assumed to be 1.0.
     cep[ ]  : Output.       Real array : dimension=order.
               LPC (all-pole modeled) cepstrum.
               cep(0) is implicitly assumed to be alog(resid/pi),
               where "resid" is residual power of LPC/PARCOR.
     ceplen  : Input.        Integer.
               The number of required points of LPC-cepstrum.
               Note that the degree of freedom remains order.
----------------------------------------------------------------------*/

long spAlphaToCep(double *alpha, long order, double *cep, long ceplen)
{
    double ss;
    long i, j, m;

    if (alpha == NULL || order <= 0 || cep == NULL || ceplen <= 0) return -1;

    m = -1;
    while (m + 1 < ceplen) {
	m++; ss = 0.0;
	for (i = 0; i < order; i++) {
	    if (i >= m) {
		ss -= alpha[m] * (double)(m + 1);
		break;
	    } else {
		j = m - i - 1;
		ss -= alpha[i] * cep[j];
	    }
	}
	cep[m] = ss;
    }
    if (ceplen > 1)
        for (m = 1; m < ceplen; m++)
            cep[m] /= (double)(m + 1);

    return ceplen;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  Kuo-jian LI               Oct./1986    *
*                                                                   *
*********************************************************************

---  spCepToAlpha  ---

     Description:
       * Conversion of "cep" into "alpha".
       * Computation of ar-coefficients "alpha" from its cepstrum.
       * Computation of polynomial coefficients from m-th power of
         its zeros (roots), for m=1,...,order, since:
       * Computation of LPC cepstrum from LPC poles:
                        1     order                m
               cep[m]= --- * sum ( (LPC pole(i))  ).
                        m    i=1
         Conversion from the sum of m-th power of complex numbers
         for m=1,...,order into their symmetric homogeneous
         polynomials, i.e.,
         given are                  m     m         m
                     m * cep[m-1] = z1  + z2  +...+ zp  .
         Computed are  alpha[0] = z1+z2+z3+.................+zp
                       alpha[1] = z1*z2+z1*z3+z1*z4+..+z(p-1)*zp
                                   ............
                       alpha[order]= z1*z2*z3*...*zp. 
       * Including file:
               math.h

     Subroutine Calling sequence:
               ------------------
               spCepToAlpha(cep, ceplen, alpha, order)
               ------------------

     order      : Input.       long
               The order of analysis; the number of poles in LPC;
               The degree of freedom of the model - 1.
     cep[ ]  : Input.       double array : Dimension=order.
               LPC (all-pole modeled) cepstrum.
               0th order LPC cepstrum is implicitly assumed to be
	       alog(resid/PI),
               where "resid" is residual power of LPC/PARCOR.
     alpha[ ]  : Output.      double array : Dimension=order.
               Linear prediction coefficients; AR parameters.
               0th order linear prediction coefficients is implicitly
	       assumed to be 1.

----------------------------------------------------------------------*/

long spCepToAlpha(double *cep, long ceplen, double *alpha, long order)
{
    double fj, aa, jm1;
    long i, j, k;

    if (cep == NULL || ceplen <= 0 || alpha == NULL || order <= 0) return -1;

    alpha[0] = -cep[0];
    if (order == 1)
        return 1;
    
    fj = -1.0;
    for (j = 1; j < order; j++)
    {
        fj -= 1.0;
        if (j >= ceplen) {
            aa = 0.0;
        } else {
            aa = cep[j] * (double)(j + 1);
        }
        jm1 = j;
        for (i = 0; i < jm1; i++)
        {
            k = j - i - 1;
            if (k < ceplen) {
                aa += alpha[i] * cep[k] * (double)(k + 1);
            }
        }
        alpha[j] = aa / fj;
    }

    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  T.Umezaki                 Oct./1986    *
*                                                                   *
*********************************************************************

---  spAlphaToPole (original name: bairst)  ---

     Description:
       * Solve real-coeeficient algebraic equation by Bairstow
         iterative algorithm.
       * This subroutine is a modification of FACOM270/20 SSL routine.
       * Including file:
               stdio.h

     Subroutine Calling sequence:
               ---------------------
               spAlphaToPole(alpha, polere, poleim, order, eps)
               ---------------------

     order   : Input.       long
               Order of the algebraic equation.
     alpha[ ]: Input.       double array : Dimension=n.
               Coefficients.  The polynomial is given as:
                  x**n + a[0]*x**(n-1) +....+ a[n-1]
               a[0] is implicitly assumed to be 1.
     polere[ ]: Output.      double array : Dimension=n.
               Real parts of the roots(solution).
     poleim[ ]: Output.      double array : Dimension=n.
               Imaginary parts of the roots (solution).
     eps     : Input.       double
               Convergence criterion in Bairstow iterative algorithm.

     Note: * The contents of array "a" are not saved.
----------------------------------------------------------------------*/

#define SP_LPC_ITTMAX  1000

long spAlphaToPole(double *alpha, double *polere, double *poleim, long order, double eps)
{
    double   d, m, x, y, p, q, w, a0, dy, delt;
    double   cj = 0.0, bj = 0.0, cj1 = 0.0, bj1 = 0.0, cj2 = 0.0, bj2 = 0.0, cj3 = 0.0;
    double   *a;
    long     i, j, k, n1, np, i1, itt;
    spBool   b, sw, test; /* ----- logical ----- */

    if (order <= 0) return -1;

    a = xalloc(order, double);

    for (i = 0; i < order; i++)
	a[i] = alpha[i];

    /*  n1 = order + 1;  */
    p = q = a0 = 1;
    np = order;
    x = 0.0;
    if (2 * (order / 2) != order)
        /* ----- newton iteration ----- */
        /* ----- for factorization of odd order polynonoal -----*/
    {
        for (j = 1; j <= 2; j++)
        {
            x = (double)j * 2 - 3;
            b = SP_TRUE;
            for (itt = 1; itt <= SP_LPC_ITTMAX; itt++)
            {
                m = 0;
                y = a0;
                dy = y;
                for (i = 0; i < order; i++)
                {
                    y *= x;
                    if (fabs(y) > m)
                        m = fabs(y);
                    y += a[i];
                    if (i != order - 1)
                        dy = dy * x + y;
                }
                if (dy == 0)
                    x += 1;
                else
                {
                    x -= y / dy;
                    if (fabs(y) <= eps * m) {
                        if (b == SP_TRUE) {
                            b = SP_FALSE;
                        } else {
                            goto lbl40;
                        }
                    }
                }
            }
            xfree(a);
	    return 0;
        }
      lbl40:
        n1 = order - 1;
        a[0] += x * a0;
        if (n1 > 1)
        {
            for (i = 1; i < n1; i++)
                a[i] += x * a[i - 1];
        }
        polere[order - 1] = x;
        poleim[order - 1] = 0;
        np = order - 1;
    }

    /* ----- Bairstow method ----- */
    /* ----- for factorization of even order polynomial -----*/
    if (np >= 4)
    {
        for (k = 4; k <= np; k += 2)
        {
            i = np - k + 3;
            sw = SP_TRUE;
            itt = 0;
          loop1:
            delt = bj2 = cj2 = w = 0;
            i1 = i + 2;
            bj1 = cj1 = a0;
            if (i == 2)
                cj3 = a0;
            if (i1 > 0)
            {
                for (j = 1; j < i1; j++)
                {
                    w = p * bj1 + q * bj2;
                    bj = a[j-1] - w;
                    cj = bj - p * cj1 - q * cj2;
                    if (j == i - 2)
                        cj3 = cj;
                    if (j != i + 1)
                    {
                        bj2 = bj1;
                        bj1 = bj;
                        cj2 = cj1;
                        cj1 = cj;
                    }
                    if (fabs(w) > delt)
                        delt = fabs(w);
                }
            }
            if (delt * eps < fabs(bj1) + fabs(bj + p * bj1))
                test = SP_TRUE;
            else
                test = SP_FALSE;
            cj1 = - p * cj2 - q * cj3;
            delt = cj2 * cj2 - cj1 * cj3;
            if ( delt == 0 )
            {
                p += 1;
                goto loop1;
            }
            p += (bj1 * cj2 - bj * cj3) / delt;
            q += (bj * cj2 - bj1 * cj1) / delt;
            itt++;
            if (itt <= SP_LPC_ITTMAX)
            {
                if (sw == SP_TRUE && test == SP_TRUE)
                    goto loop1;
                if (sw == SP_TRUE)
                {
                    sw = SP_FALSE;
                    goto loop1;
                }
            }

            /* ----- divide by quadratic factor ----- */
            bj1 = a0;
            bj2 = 0;
            i1 = i;
            if (i1 > 0)
                for (j = 1; j < i1; j++)
                {
                    bj = a[j-1] - p * bj1 - q * bj2;
                    a[j-1] = bj;
                    bj2 = bj1;
                    bj1 = bj;
                }
            polere[i - 1] = p;
            polere[i] = q;
        }
    }
    polere[0] = a[0] / a0;
    polere[1] = a[1] / a0;

    /* ----- solve each quadratic equation ----- */
    for (i = 0; i < np; i += 2)
    {
        w = - 0.5 * polere[i];
        d = w * w - polere[i + 1];
        if (d < 0)
        {
            polere[i] = polere[i + 1] = w;
            poleim[i] = sqrt(-d);
            poleim[i + 1] = -poleim[i];
        }
        else
        {
            d = sqrt(d) + fabs(w);
            w = polere[i + 1];
            poleim[i] = poleim[i + 1] = 0;
            if ( polere[i] > 0 )
            {
                polere[i] = - w / d;
                polere[i + 1] = - d;
            }
            else
            {
                polere[i] = d;
                polere[i + 1] = w / d;
            }
        }
    }
    
    xfree(a);
    
    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  T.Umezaki                 Oct./1986    *
*                                                                   *
*********************************************************************

---  spPoleToFreq (original name: alfpol)  ---

     Description:
       * Conversion of "alpha" into
         pole" (combination of "freq" and "bw").
       * Solve algebraic equation and obtain pole frequencies and
         band widths.

     Calling sequence:
            ----------------------------------
            spPoleToFreq(polere, poleim, freq, bw, order fs);
            ----------------------------------
     order    : Input.        long
               The order of analysis; the number of ploes in LPC;
               The degree of freedom of the model - 1.
     polere[ ]: Input.      double array : Dimension=order.
               Real parts of the roots(solution).
     poleim[ ]: Input.      double array : Dimension=order.
               Imaginary parts of the roots (solution).
     freq[ ] : Output.       double array : dimension=order.
               Pole frequencies. Ranging between 0 and fs.
               Ordered by magnitude. (increasing)
     bw[ ]   : Output.       double array : dimension=order.
               Pole band widths.
               bw[i] corresponds to freq[i].
     fs      : Input.        double
               Sampling frequency; or frequency normalization constant.
               ( 0.0   = or <   freq[ ]   = or <  fs )
---------------------------------------------------------------------*/

long spPoleToFreq(double *polere, double *poleim, double *freq /* output */, double *bw /* output, can be NULL */, long order, double fs, spBool sort_flag)
{
    double  bb, ff, x, y;
    long    i, j, k, ipm1, i1;

    if (polere == NULL || poleim == NULL || freq == NULL
        || order <= 0 || fs <= 0.0)  return -1;

    for (i = 0; i < order; i++) {
        x = polere[i];
        y = poleim[i];
        if (y != 0.0) {
            if (fabs(x) > 0.1e-10) {
                ff = fs * atan(y / x) / (PI * 2.0);
                if (x < 0.0)
                    ff += fs / 2.0;
                if (x > 0.0 && y < 0.0)
                    ff += fs;
            }
            else {
                ff = fs / 4.0;
                if (y < 0.0)
                    ff += fs / 2.0;
            }
            freq[i] = ff;
            if (bw != NULL) {
                bb = -fs * log(x * x + y * y) / (PI * 2.0);
                bw[i] = bb;
            }
        }
        else {
            if (x >= 0.0)
                freq[i] = 0.0;
            else
                freq[i] = fs / 2.0;

            if (bw != NULL) {
                bw[i] = -fs * log(x * x) / (PI * 2.0);
            }
        }
    }

    /*----- reordering ----------*/
    if (sort_flag && order > 1) {
        ipm1 = order - 1;
        for (i = 0;i <= ipm1 - 1;i++) {
            ff = freq[i];
            k = i;
            i1 = i + 2;
            for (j = i1 - 1; j <= order - 1;j++) {
                if (ff > freq[j]) {
                    k = j;
                    ff = freq[j];
                }
            }
            if (k != i) {
                ff = freq[i];
                freq[i] = freq[k];
                freq[k] = ff;
                if (bw != NULL) {
                    bb = bw[i];
                    bw[i] = bw[k];
                    bw[k] = bb;
                }
            }
        }
    }
    
    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded  by  S.Sagayama (NTT)                   Aut./1975    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  M.Katoh                   Oct./1986    *
*                                                                   *
*********************************************************************

---  spFreqToPole, spPoleToAlpha  ---
----------------------------------------------------------------
    * Conversion of "pole" into "alpha".
      "pole" is a combination of "freq" and "bw".
    * Computation of linear prediction coefficients from lpc
      poles represented by their frequencies and band widths.
----------------------------------------------------------------
  order    : Input.        integer.
          the order of analysis; the number of poles in lpc;
  freq[]: Input.        real array : dimension=order.
          pole frequencies. ranging between 0 and fs.
          increasingly ordered.  freq[0]=<freq[0]=<....
  bw[]  : Input.        real array : dimension=order.
          pole band widths.
          bw[i] corresponds to freq[i].
  polere[ ]: Output/Input.      double array : Dimension=order.
          Real parts of the roots(solution).
  poleim[ ]: Output/Input.      double array : Dimension=order.
          Imaginary parts of the roots (solution).
  fs    : Input.        real.
          sampling frequency; or frequency normalization constant.
          ( 0.0   =or<   freq[]   =or<  fs )
  alpha[] : Output.       real array : dimension=order.
          linear prediction coefficients; ar parameters.
          alpha[0] is implicitly assumed to be 1.0.

  note: * alpha[]  is assumed to be real.  So, total degree of
          "freq" and "bw" should be order.  That means only real
          poles and half of complex poles whose frequencis are
          between 0 and fs/2 are needed.  Conjugate poles whose
          frequencies lay above fs/2 are not needed to be given.
--------------------------------------------------------------------*/

long spFreqToPole(double *freq, double *bw, double *polere, double *poleim, long order, double fs)
{
    long i;
    double weight;
    double r, theta;
    
    if (freq == NULL || bw == NULL || polere == NULL || poleim == NULL
        || order <= 0 || fs <= 0.0) return -1;

    weight = PI / fs;

    for (i = 0; i < order; i++) {
        r = exp(-weight * bw[i]);
        theta = 2.0 * weight * freq[i];
        polere[i] = r * cos(theta);
        poleim[i] = r * sin(theta);
    }
    
    return order;
}

long spPoleToAlpha(double *polere, double *poleim, double *alpha, long order)
{
    long    i, j, k, j1, l;
    double  p, q, r;

    if (polere == NULL || poleim == NULL || alpha == NULL
        || order <= 0 ) return -1;

    for (i = 0; i < order; i++) alpha[i] = 0.0;
    j = 0;
    for (i = 0; i < order; i++) {
        r = sqrt(polere[i] * polere[i] + poleim[i] * poleim[i]);
        if (poleim[i] == 0.0) {
            if (polere[i] > 0.0)
                r *= -1;
 	    if (j > 0) {
                j1 = j + 1;
	        for (k = 1; k <= j; ++k) {
      		    l = j1 - k;
   		    alpha[l] += (alpha[l-1] * r);
		}
	    }
       	    alpha[0] += r;
      	    j += 1;
	} else {
            p = -2.0 * polere[i];
      	    q = r * r;
            if (j > 0) {
      	        j1 = j + 1;
	        for (k = 1; k <= j; ++k) {
      		    l = j1 - k;
      		    alpha[l + 1] += (alpha[l] * p + alpha[l-1] * q);
		}
	    }
	    alpha[1] += (alpha[0] * p + q);
      	    alpha[0] += p;
      	    j += 2;
	}
	if (j >= order) break;
    }
    
    return order;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
*        coded    by S.Sagayama (NTT)                  Apr./1981    *
*        modified by S.Sagayama (NTT)                  Mar./1982    *
*        converted to C  at Itakura Lab.(Nagoya Univ.)              *
*                        by  Kuo-jian LI               Nov./1986    *
*                                                                   *
*********************************************************************

---  spCepSynFiltering  ---

     Description:
       * Conversion of "cep" into "sig".
       * Speech synthesizer filter whose coefficients are LPC-cepstra.
       * Including file:
               math.h

     Function Calling sequence:
               ------------------------
               spCepSynFiltering(cep, state2, order, insig)
               ------------------------

     spCepSynFiltering : Output.       double function.
               Sample output of LPC-cepstrum all-pole filter.
     order   : Input.        long
               The order of analysis; the number of poles in LPC;
               The degree of freedom of the model - 1.
     cep[ ]  : Input.        double array : Dimension=order*2.
               LPC (all-pole modeled) cepstrum.
               0th order LPC cepstrum is implicitly assumed to be
	       alog(resid/PI),
               where "resid" is residual power of LPC/PARCOR.
     state2[ ]: Input/Output. double array : Dimension=order*2.
               Input/Output zone: [0,1,...,order-1],
               Working zone: [order,order+1,...,order*2-1].
               Inner state in the cepstrum filter.
               Let be all 0 for initialization.
     insig   : Input.       double
               Filter Input sample.

----------------------------------------------------------------------*/

double spCepSynFiltering(double *cep, double *state2, long order, double insig)
{
    double   x, result;
    long     i, j, k;

    /* ----- output of the filter ----- */
    result = insig;
    if (order <= 0)
        return result;
    result -= state2[0];

    /* ----- clear the working area ----- */
    for (i = 0; i < order; i++)
        state2[i + order] = 0;

    /* ----- filter network ----- */
    x = result;
    if (order > 1)
        for (i = 0; i < order - 1; i++) {
            for (j = i; j < order; j++) {
                k = j - i;
                state2[j + order] -= (double)(k + 1) / (double)(j + 1) * cep[k] * x;
            }
            x = state2[i + order];
            state2[i] = state2[i + 1] + state2[i + order];
        }
    state2[order - 1] = state2[order + order - 1]- 1.0 / (double)order * cep[0] * x;
    
    return result;
}

/********************************************************************
*                                                                   *
*      Speech  Signal  Processing                                   *
*              Analysis / Synthesis   Library          Ver 0.1      *
*                                                                   *
********************************************************************/

/*-------------------------------------------------------------------
 * LSP synthesis filer (IIR filter)
 * ------------------------
 * spLSPSynFiltering(lsp, state2plus1, order, insig)
 * ------------------------
 * Arguments
 *   long   order   : order of LSP
 *   double lsp[]: lsp frequencies
 *   double state2plus1[] : digital delay (need over order * 2 + 1)
 *   double insig  : filter input
 * Return value
 *   Synthesizd signal (= filter output)
 *-------------------------------------------------------------------
 */

double spLSPSynFiltering(double *lsp, double *state2plus1, long order, double insig)
{
    long i, m , mp1;
    double u, d, out, w, *d0;

    if (order <= 0) return insig;
    
    m = order / 2;
    mp1 = order - m;
    d0 = state2plus1;
    u = -0.5 * *d0;
    d = u;
    out = 0.0;

    for (i = 1; i <= m; ++i) {
		/*--- odd ---*/
	w     = *++state2plus1 - u * 2.0 * cos(PI * *lsp++);
	*state2plus1   = u;
	out  += w;
	u    += *++state2plus1;
	*state2plus1 = w;
		/*--- even ---*/
    	w     = *++state2plus1 - d * 2.0 * cos(PI * *lsp++);
	*state2plus1   = d;
	out  += w;
	d    += *++state2plus1;
	*state2plus1   = w;
    }
		/*--- last step for the case of order = odd ---*/
    if (mp1 != m) {
	w     = *++state2plus1 - u * 2.0 * cos(PI * *lsp);
	*state2plus1 = u;
	u     = w;
	w     = *++state2plus1;
	*state2plus1   = d;
	d     = w;
    }
    out += u - d + insig;
    *d0 = out;
    
    return out;
}

