/** @file sp/matrix.h
 */

#ifndef SPLIB_MATRIX_H
#define SPLIB_MATRIX_H

#include <sp/vector.h>

#ifdef __cplusplus
extern "C" {
#endif

#if (defined(SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE) && !defined(SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE))
#define SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE
#endif

/** @~english @defgroup matrixGroup <sp/matrix.h>: Matrix
@code
#include <sp/matrix.h>
@endcode

Introduction
------------

The matrix type is a type which is used as an argument or a return value of matrix-related functions of this library.
Types of spSMatrix for short type, spLMatrix for long type, spFMatrix for float type and spDMatrix for double type are provided.

The memory of spSMatrix , spLMatrix , spFMatrix or spDMatrix can be allocated by the functions which have a prefix xsm, xlm, xfm or xdm respectively.
If you don't use the matrix anymore,
you have to deallocate the memory by calling xsmfree() , xlmfree() , xfmfree() or xdmfree() .
*/

/** @addtogroup matrixGroup
 *  @{ */  /*----@addtogroup matrixGroup----*/

typedef struct _spSMatrix  *spSMatrix;
typedef struct _spLMatrix  *spLMatrix;
typedef struct _spFMatrix  *spFMatrix;
typedef struct _spDMatrix  *spDMatrix;

/** @class spSMatrix matrix.h <sp/matrix.h>
 *  @~english
 *  @brief Matrix type that contains the elements of short type.
 *  @details Actually, This is a typedefed type by `typedef struct _spSMatrix *spSMatrix;`.
 *  To allocate memory for spSMatrix, call a function with prefix xsm such as xsmalloc() , xsmzeros() , etc.
 *  To deallocate memory, call xsmfree() . */
struct _spSMatrix {
    /** @~english The number of rows of spSMatrix.
     *  You can access the number of elements as \p x->row for the memory allocated \p x of spSMatrix. */
    long row;
    /** @~english The number of columns of spSMatrix.
     *  You can access the number of elements as \p x->col for the memory allocated \p x of spSMatrix. */
    long col;
    /** @~english Data of the real part of spSMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->data[0][0]
     *  for the memory allocated \p x of spSMatrix. */
    short **data;
    /** @~english Data of the imaginary part of spSMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->imag[0][0]
     *  for the memory allocated \p x of spSMatrix.
     *  If no imaginary part exists, this member can be NULL. */
    short **imag;
    
#ifdef SP_USE_MATRIX_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spLMatrix matrix.h <sp/matrix.h>
 *  @~english
 *  @brief Matrix type that contains the elements of long type.
 *  @details Actually, This is a typedefed type by `typedef struct _spLMatrix *spLMatrix;`.
 *  To allocate memory for spLMatrix, call a function with prefix xlm such as xlmalloc() , xlmzeros() , etc.
 *  To deallocate memory, call xlmfree() . */
struct _spLMatrix {
    /** @~english The number of rows of spLMatrix.
     *  You can access the number of elements as \p x->row for the memory allocated \p x of spLMatrix. */
    long row;
    /** @~english The number of columns of spLMatrix.
     *  You can access the number of elements as \p x->col for the memory allocated \p x of spLMatrix. */
    long col;
    /** @~english Data of the real part of spLMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->data[0][0]
     *  for the memory allocated \p x of spLMatrix. */
    long **data;
    /** @~english Data of the imaginary part of spLMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->imag[0][0]
     *  for the memory allocated \p x of spLMatrix.
     *  If no imaginary part exists, this member can be NULL. */
    long **imag;
    
#ifdef SP_USE_MATRIX_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spFMatrix matrix.h <sp/matrix.h>
 *  @~english
 *  @brief Matrix type that contains the elements of float type.
 *  @details Actually, This is a typedefed type by `typedef struct _spFMatrix *spFMatrix;`.
 *  To allocate memory for spFMatrix, call a function with prefix xfm such as xfmalloc() , xfmzeros() , etc.
 *  To deallocate memory, call xfmfree() . */
struct _spFMatrix {
    /** @~english The number of rows of spFMatrix.
     *  You can access the number of elements as \p x->row for the memory allocated \p x of spFMatrix. */
    long row;
    /** @~english The number of columns of spFMatrix.
     *  You can access the number of elements as \p x->col for the memory allocated \p x of spFMatrix. */
    long col;
    /** @~english Data of the real part of spFMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->data[0][0]
     *  for the memory allocated \p x of spFMatrix. */
    float **data;
    /** @~english Data of the imaginary part of spFMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->imag[0][0]
     *  for the memory allocated \p x of spFMatrix.
     *  If no imaginary part exists, this member can be NULL. */
    float **imag;
    
#ifdef SP_USE_MATRIX_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spDMatrix matrix.h <sp/matrix.h>
 *  @~english
 *  @brief Matrix type that contains the elements of double type.
 *  @details Actually, This is a typedefed type by `typedef struct _spDMatrix *spDMatrix;`.
 *  To allocate memory for spDMatrix, call a function with prefix xdm such as xdmalloc() , xdmzeros() , etc.
 *  To deallocate memory, call xdmfree() . */
struct _spDMatrix {
    /** @~english The number of rows of spDMatrix.
     *  You can access the number of elements as \p x->row for the memory allocated \p x of spDMatrix. */
    long row;
    /** @~english The number of columns of spDMatrix.
     *  You can access the number of elements as \p x->col for the memory allocated \p x of spDMatrix. */
    long col;
    /** @~english Data of the real part of spDMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->data[0][0]
     *  for the memory allocated \p x of spDMatrix. */
    double **data;
    /** @~english Data of the imaginary part of spDMatrix.
     *  You can access the element of the data at row 0, column 0 as \p x->imag[0][0]
     *  for the memory allocated \p x of spDMatrix.
     *  If no imaginary part exists, this member can be NULL. */
    double **imag;
    
#ifdef SP_USE_MATRIX_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_MATRIX_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

typedef struct _spSMatrices {
    long num_matrix;
    spSMatrix *matrix;
} *spSMatrices;

typedef struct _spLMatrices {
    long num_matrix;
    spLMatrix *matrix;
} *spLMatrices;

typedef struct _spFMatrices {
    long num_matrix;
    spFMatrix *matrix;
} *spFMatrices;

typedef struct _spDMatrices {
    long num_matrix;
    spDMatrix *matrix;
} *spDMatrices;

/** @} */  /*----@addtogroup matrixGroup----*/

#define SMatrix spSMatrix
#define LMatrix spLMatrix
#define FMatrix spFMatrix
#define DMatrix spDMatrix
#define _SMatrix _spSMatrix
#define _LMatrix _spLMatrix
#define _FMatrix _spFMatrix
#define _DMatrix _spDMatrix

#define SMatrices spSMatrices
#define LMatrices spLMatrices
#define FMatrices spFMatrices
#define DMatrices spDMatrices
#define _SMatrices _spSMatrices
#define _LMatrices _spLMatrices
#define _FMatrices _spFMatrices
#define _DMatrices _spDMatrices

#if defined(MACOS)
#pragma import on
#endif

extern spSMatrix xsmalloc(long row, long col);
extern spLMatrix xlmalloc(long row, long col);
extern spFMatrix xfmalloc(long row, long col);
extern spDMatrix xdmalloc(long row, long col);
extern void xsmfree(spSMatrix matrix);
extern void xlmfree(spLMatrix matrix);
extern void xfmfree(spFMatrix matrix);
extern void xdmfree(spDMatrix matrix);

extern void smialloc(spSMatrix x);
extern void lmialloc(spLMatrix x);
extern void fmialloc(spFMatrix x);
extern void dmialloc(spDMatrix x);
extern void smifree(spSMatrix x);
extern void lmifree(spLMatrix x);
extern void fmifree(spFMatrix x);
extern void dmifree(spDMatrix x);

extern spSMatrix xsmrialloc(long row, long col);
extern spLMatrix xlmrialloc(long row, long col);
extern spFMatrix xfmrialloc(long row, long col);
extern spDMatrix xdmrialloc(long row, long col);

extern spSMatrices xsmsalloc(long num);
extern spLMatrices xlmsalloc(long num);
extern spFMatrices xfmsalloc(long num);
extern spDMatrices xdmsalloc(long num);
extern void xsmsfree(spSMatrices xs);
extern void xlmsfree(spLMatrices xs);
extern void xfmsfree(spFMatrices xs);
extern void xdmsfree(spDMatrices xs);

extern void smset(spSMatrix matrix, short **data, long row, long col);
extern void lmset(spLMatrix matrix, long **data, long row, long col);
extern void fmset(spFMatrix matrix, float **data, long row, long col);
extern void dmset(spDMatrix matrix, double **data, long row, long col);

extern void smreal(spSMatrix x);
extern void lmreal(spLMatrix x);
extern void fmreal(spFMatrix x);
extern void dmreal(spDMatrix x);
extern void smimag(spSMatrix x);
extern void lmimag(spLMatrix x);
extern void fmimag(spFMatrix x);
extern void dmimag(spDMatrix x);
extern void smconj(spSMatrix x);
extern void lmconj(spLMatrix x);
extern void fmconj(spFMatrix x);
extern void dmconj(spDMatrix x);
extern void smriswap(spSMatrix x);
extern void lmriswap(spLMatrix x);
extern void fmriswap(spFMatrix x);
extern void dmriswap(spDMatrix x);

extern void smcopy(spSMatrix y, spSMatrix x);
extern void lmcopy(spLMatrix y, spLMatrix x);
extern void fmcopy(spFMatrix y, spFMatrix x);
extern void dmcopy(spDMatrix y, spDMatrix x);
extern spSMatrix xsmclone(spSMatrix x);
extern spLMatrix xlmclone(spLMatrix x);
extern spFMatrix xfmclone(spFMatrix x);
extern spDMatrix xdmclone(spDMatrix x);
    
extern spSMatrix xsmnums(long row, long col, short value);
extern spLMatrix xlmnums(long row, long col, long value);
extern spFMatrix xfmnums(long row, long col, float value);
extern spDMatrix xdmnums(long row, long col, double value);
extern void smnums(spSMatrix mat, long row, long col, short value);
extern void lmnums(spLMatrix mat, long row, long col, long value);
extern void fmnums(spFMatrix mat, long row, long col, float value);
extern void dmnums(spDMatrix mat, long row, long col, double value);
extern void sminums(spSMatrix mat, long row, long col, short value);
extern void lminums(spLMatrix mat, long row, long col, long value);
extern void fminums(spFMatrix mat, long row, long col, float value);
extern void dminums(spDMatrix mat, long row, long col, double value);
extern spSMatrix xsmrinums(long row, long col, short value);
extern spLMatrix xlmrinums(long row, long col, long value);
extern spFMatrix xfmrinums(long row, long col, float value);
extern spDMatrix xdmrinums(long row, long col, double value);

extern void smeye(spSMatrix mat);
extern void lmeye(spLMatrix mat);
extern void fmeye(spFMatrix mat);
extern void dmeye(spDMatrix mat);
extern spSMatrix xsmeye(long row, long col);
extern spLMatrix xlmeye(long row, long col);
extern spFMatrix xfmeye(long row, long col);
extern spDMatrix xdmeye(long row, long col);

extern spFMatrix xfmpascal(long n);
extern spDMatrix xdmpascal(long n);
    
extern spSMatrix xsminitrow(long nrow, long j, long incr, long n);
extern spLMatrix xlminitrow(long nrow, long j, long incr, long n);
extern spFMatrix xfminitrow(long nrow, float j, float incr, float n);
extern spDMatrix xdminitrow(long nrow, double j, double incr, double n);
extern spSMatrix xsminitcol(long ncol, long j, long incr, long n);
extern spLMatrix xlminitcol(long ncol, long j, long incr, long n);
extern spFMatrix xfminitcol(long ncol, float j, float incr, float n);
extern spDMatrix xdminitcol(long ncol, double j, double incr, double n);

extern void smeraserow(spSMatrix mat, long row, long offset, long length, spBool inv);
extern void lmeraserow(spLMatrix mat, long row, long offset, long length, spBool inv);
extern void fmeraserow(spFMatrix mat, long row, long offset, long length, spBool inv);
extern void dmeraserow(spDMatrix mat, long row, long offset, long length, spBool inv);
extern spSMatrix xsmeraserow(spSMatrix mat, long row, long offset, long length, spBool inv);
extern spLMatrix xlmeraserow(spLMatrix mat, long row, long offset, long length, spBool inv);
extern spFMatrix xfmeraserow(spFMatrix mat, long row, long offset, long length, spBool inv);
extern spDMatrix xdmeraserow(spDMatrix mat, long row, long offset, long length, spBool inv);
extern void smerasecol(spSMatrix mat, long col, long offset, long length, spBool inv);
extern void lmerasecol(spLMatrix mat, long col, long offset, long length, spBool inv);
extern void fmerasecol(spFMatrix mat, long col, long offset, long length, spBool inv);
extern void dmerasecol(spDMatrix mat, long col, long offset, long length, spBool inv);
extern spSMatrix xsmerasecol(spSMatrix mat, long col, long offset, long length, spBool inv);
extern spLMatrix xlmerasecol(spLMatrix mat, long col, long offset, long length, spBool inv);
extern spFMatrix xfmerasecol(spFMatrix mat, long col, long offset, long length, spBool inv);
extern spDMatrix xdmerasecol(spDMatrix mat, long col, long offset, long length, spBool inv);
    
extern long smcutrowto(spSMatrix mat, long row, long offset, long length, spSVector vec, long vec_offset, int overlap);
extern long smcutrow(spSMatrix mat, long row, long offset, long length, spSVector vec);
extern spSVector xsmcutrow(spSMatrix mat, long row, long offset, long length);
extern long lmcutrowto(spLMatrix mat, long row, long offset, long length, spLVector vec, long vec_offset, int overlap);
extern long lmcutrow(spLMatrix mat, long row, long offset, long length, spLVector vec);
extern spLVector xlmcutrow(spLMatrix mat, long row, long offset, long length);
extern long fmcutrowto(spFMatrix mat, long row, long offset, long length, spFVector vec, long vec_offset, int overlap);
extern long fmcutrow(spFMatrix mat, long row, long offset, long length, spFVector vec);
extern spFVector xfmcutrow(spFMatrix mat, long row, long offset, long length);
extern long dmcutrowto(spDMatrix mat, long row, long offset, long length, spDVector vec, long vec_offset, int overlap);
extern long dmcutrow(spDMatrix mat, long row, long offset, long length, spDVector vec);
extern spDVector xdmcutrow(spDMatrix mat, long row, long offset, long length);
extern long smcutcolto(spSMatrix mat, long col, long offset, long length, spSVector vec, long vec_offset, int overlap);
extern long smcutcol(spSMatrix mat, long col, long offset, long length, spSVector vec);
extern spSVector xsmcutcol(spSMatrix mat, long col, long offset, long length);
extern long lmcutcolto(spLMatrix mat, long col, long offset, long length, spLVector vec, long vec_offset, int overlap);
extern long lmcutcol(spLMatrix mat, long col, long offset, long length, spLVector vec);
extern spLVector xlmcutcol(spLMatrix mat, long col, long offset, long length);
extern long fmcutcolto(spFMatrix mat, long col, long offset, long length, spFVector vec, long vec_offset, int overlap);
extern long fmcutcol(spFMatrix mat, long col, long offset, long length, spFVector vec);
extern spFVector xfmcutcol(spFMatrix mat, long col, long offset, long length);
extern long dmcutcolto(spDMatrix mat, long col, long offset, long length, spDVector vec, long vec_offset, int overlap);
extern long dmcutcol(spDMatrix mat, long col, long offset, long length, spDVector vec);
extern spDVector xdmcutcol(spDMatrix mat, long col, long offset, long length);

extern long smpasterowfrom(spSMatrix mat, long row, long offset, long length,
                           spSVector vec, long vec_offset, int overlap);
extern long smpasterow(spSMatrix mat, long row, spSVector vec, long offset, long length, int overlap);
extern long lmpasterowfrom(spLMatrix mat, long row, long offset, long length,
                           spLVector vec, long vec_offset, int overlap);
extern long lmpasterow(spLMatrix mat, long row, spLVector vec, long offset, long length, int overlap);
extern long fmpasterowfrom(spFMatrix mat, long row, long offset, long length,
                           spFVector vec, long vec_offset, int overlap);
extern long fmpasterow(spFMatrix mat, long row, spFVector vec, long offset, long length, int overlap);
extern long dmpasterowfrom(spDMatrix mat, long row, long offset, long length,
                           spDVector vec, long vec_offset, int overlap);
extern long dmpasterow(spDMatrix mat, long row, spDVector vec, long offset, long length, int overlap);
extern long smpastecolfrom(spSMatrix mat, long col, long offset, long length,
                           spSVector vec, long vec_offset, int overlap);
extern long smpastecol(spSMatrix mat, long col, spSVector vec, long offset, long length, int overlap);
extern long lmpastecolfrom(spLMatrix mat, long col, long offset, long length,
                           spLVector vec, long vec_offset, int overlap);
extern long lmpastecol(spLMatrix mat, long col, spLVector vec, long offset, long length, int overlap);
extern long fmpastecolfrom(spFMatrix mat, long col, long offset, long length,
                           spFVector vec, long vec_offset, int overlap);
extern long fmpastecol(spFMatrix mat, long col, spFVector vec, long offset, long length, int overlap);
extern long dmpastecolfrom(spDMatrix mat, long col, long offset, long length,
                           spDVector vec, long vec_offset, int overlap);
extern long dmpastecol(spDMatrix mat, long col, spDVector vec, long offset, long length, int overlap);

extern long smpastematfrom(spSMatrix matto, long rowto, long colto, long rowlength, long collength,
                           spSMatrix matfrom, long rowfrom, long colfrom, int overlap);
extern long smpastemat(spSMatrix matto, long rowto, long colto, spSMatrix matfrom,
                       long rowlength, long collength, int overlap);
extern long lmpastematfrom(spLMatrix matto, long rowto, long colto, long rowlength, long collength,
                           spLMatrix matfrom, long rowfrom, long colfrom, int overlap);
extern long lmpastemat(spLMatrix matto, long rowto, long colto, spLMatrix matfrom,
                       long rowlength, long collength, int overlap);
extern long fmpastematfrom(spFMatrix matto, long rowto, long colto, long rowlength, long collength,
                           spFMatrix matfrom, long rowfrom, long colfrom, int overlap);
extern long fmpastemat(spFMatrix matto, long rowto, long colto, spFMatrix matfrom,
                       long rowlength, long collength, int overlap);
extern long dmpastematfrom(spDMatrix matto, long rowto, long colto, long rowlength, long collength,
                           spDMatrix matfrom, long rowfrom, long colfrom, int overlap);
extern long dmpastemat(spDMatrix matto, long rowto, long colto, spDMatrix matfrom,
                       long rowlength, long collength, int overlap);

extern long smpastematrowfrom(spSMatrix matto, long rowto, long offset, long length,
                              spSMatrix matfrom, long rowfrom, long offsetfrom, int overlap);
extern long smpastematrow(spSMatrix matto, long rowto, spSMatrix matfrom, long rowfrom,
                          long offset, long length, int overlap);
extern long smpastematcolfrom(spSMatrix matto, long colto, long offset, long length,
                              spSMatrix matfrom, long colfrom, long offsetfrom, int overlap);
extern long smpastematcol(spSMatrix matto, long colto, spSMatrix matfrom, long colfrom,
                          long offset, long length, int overlap);
extern long lmpastematrowfrom(spLMatrix matto, long rowto, long offset, long length,
                              spLMatrix matfrom, long rowfrom, long offsetfrom, int overlap);
extern long lmpastematrow(spLMatrix matto, long rowto, spLMatrix matfrom, long rowfrom,
                          long offset, long length, int overlap);
extern long lmpastematcolfrom(spLMatrix matto, long colto, long offset, long length,
                              spLMatrix matfrom, long colfrom, long offsetfrom, int overlap);
extern long lmpastematcol(spLMatrix matto, long colto, spLMatrix matfrom, long colfrom,
                          long offset, long length, int overlap);
extern long fmpastematrowfrom(spFMatrix matto, long rowto, long offset, long length,
                              spFMatrix matfrom, long rowfrom, long offsetfrom, int overlap);
extern long fmpastematrow(spFMatrix matto, long rowto, spFMatrix matfrom, long rowfrom,
                          long offset, long length, int overlap);
extern long fmpastematcolfrom(spFMatrix matto, long colto, long offset, long length,
                              spFMatrix matfrom, long colfrom, long offsetfrom, int overlap);
extern long fmpastematcol(spFMatrix matto, long colto, spFMatrix matfrom, long colfrom,
                          long offset, long length, int overlap);
extern long dmpastematrowfrom(spDMatrix matto, long rowto, long offset, long length,
                              spDMatrix matfrom, long rowfrom, long offsetfrom, int overlap);
extern long dmpastematrow(spDMatrix matto, long rowto, spDMatrix matfrom, long rowfrom,
                          long offset, long length, int overlap);
extern long dmpastematcolfrom(spDMatrix matto, long colto, long offset, long length,
                              spDMatrix matfrom, long colfrom, long offsetfrom, int overlap);
extern long dmpastematcol(spDMatrix matto, long colto, spDMatrix matfrom, long colfrom,
                          long offset, long length, int overlap);

/* extract vector in row direction */
extern spSVector xsmrextract(spSMatrix mat, spLVector colidx);
extern spLVector xlmrextract(spLMatrix mat, spLVector colidx);
extern spFVector xfmrextract(spFMatrix mat, spLVector colidx);
extern spDVector xdmrextract(spDMatrix mat, spLVector colidx);
/* extract vector in column direction */
extern spSVector xsmcextract(spSMatrix mat, spLVector rowidx);
extern spLVector xlmcextract(spLMatrix mat, spLVector rowidx);
extern spFVector xfmcextract(spFMatrix mat, spLVector rowidx);
extern spDVector xdmcextract(spDMatrix mat, spLVector rowidx);

extern spSMatrix xsmextractrows(spSMatrix mat, spLVector rowidx);
extern spLMatrix xlmextractrows(spLMatrix mat, spLVector rowidx);
extern spFMatrix xfmextractrows(spFMatrix mat, spLVector rowidx);
extern spDMatrix xdmextractrows(spDMatrix mat, spLVector rowidx);
extern spSMatrix xsmextractcols(spSMatrix mat, spLVector colidx);
extern spLMatrix xlmextractcols(spLMatrix mat, spLVector colidx);
extern spFMatrix xfmextractcols(spFMatrix mat, spLVector colidx);
extern spDMatrix xdmextractcols(spDMatrix mat, spLVector colidx);

extern spSMatrix xsmextractrowscols(spSMatrix mat, spLVector rowidx, spLVector colidx);
extern spLMatrix xlmextractrowscols(spLMatrix mat, spLVector rowidx, spLVector colidx);
extern spFMatrix xfmextractrowscols(spFMatrix mat, spLVector rowidx, spLVector colidx);
extern spDMatrix xdmextractrowscols(spDMatrix mat, spLVector rowidx, spLVector colidx);

extern void smabs(spSMatrix mat);
extern void lmabs(spLMatrix mat);
extern void fmabs(spFMatrix mat);
extern void dmabs(spDMatrix mat);
extern spSMatrix xsmabs(spSMatrix mat);
extern spLMatrix xlmabs(spLMatrix mat);
extern spFMatrix xfmabs(spFMatrix mat);
extern spDMatrix xdmabs(spDMatrix mat);

extern spLVector xsmrmax(spSMatrix mat);
extern spLVector xlmrmax(spLMatrix mat);
extern spLVector xfmrmax(spFMatrix mat);
extern spLVector xdmrmax(spDMatrix mat);
extern spLVector xsmrmin(spSMatrix mat);
extern spLVector xlmrmin(spLMatrix mat);
extern spLVector xfmrmin(spFMatrix mat);
extern spLVector xdmrmin(spDMatrix mat);
extern spLVector xsmcmax(spSMatrix mat);
extern spLVector xlmcmax(spLMatrix mat);
extern spLVector xfmcmax(spFMatrix mat);
extern spLVector xdmcmax(spDMatrix mat);
extern spLVector xsmcmin(spSMatrix mat);
extern spLVector xlmcmin(spLMatrix mat);
extern spLVector xfmcmin(spFMatrix mat);
extern spLVector xdmcmin(spDMatrix mat);
extern short smmin(spSMatrix mat, long *row, long *col);
extern long lmmin(spLMatrix mat, long *row, long *col);
extern float fmmin(spFMatrix mat, long *row, long *col);
extern double dmmin(spDMatrix mat, long *row, long *col);
extern short smmax(spSMatrix mat, long *row, long *col);
extern long lmmax(spLMatrix mat, long *row, long *col);
extern float fmmax(spFMatrix mat, long *row, long *col);
extern double dmmax(spDMatrix mat, long *row, long *col);

extern spSMatrix xsmcrop(spSMatrix mat, long row_offset, long row_length, long col_offset, long col_length);
extern spLMatrix xlmcrop(spLMatrix mat, long row_offset, long row_length, long col_offset, long col_length);
extern spFMatrix xfmcrop(spFMatrix mat, long row_offset, long row_length, long col_offset, long col_length);
extern spDMatrix xdmcrop(spDMatrix mat, long row_offset, long row_length, long col_offset, long col_length);

extern spSMatrix xsmhcat(spSMatrix a, spSMatrix b);
extern spSMatrix xsmvcat(spSMatrix a, spSMatrix b);
extern spLMatrix xlmhcat(spLMatrix a, spLMatrix b);
extern spLMatrix xlmvcat(spLMatrix a, spLMatrix b);
extern spFMatrix xfmhcat(spFMatrix a, spFMatrix b);
extern spFMatrix xfmvcat(spFMatrix a, spFMatrix b);
extern spDMatrix xdmhcat(spDMatrix a, spDMatrix b);
extern spDMatrix xdmvcat(spDMatrix a, spDMatrix b);

extern void smshiftrow(spSMatrix x, long shift);
extern void smshiftcol(spSMatrix x, long shift);
extern void lmshiftrow(spLMatrix x, long shift);
extern void lmshiftcol(spLMatrix x, long shift);
extern void fmshiftrow(spFMatrix x, long shift);
extern void fmshiftcol(spFMatrix x, long shift);
extern void dmshiftrow(spDMatrix x, long shift);
extern void dmshiftcol(spDMatrix x, long shift);
    
extern spSVector xsvdiag(spSMatrix A);
extern spLVector xlvdiag(spLMatrix A);
extern spFVector xfvdiag(spFMatrix A);
extern spDVector xdvdiag(spDMatrix A);
/* supdiag: superdiagonal (shift > 0), subdialognal (shift < 0), or diagonal (shift = 0) */
extern spSVector xsvsupdiag(spSMatrix A, long shift);
extern spLVector xlvsupdiag(spLMatrix A, long shift);
extern spFVector xfvsupdiag(spFMatrix A, long shift);
extern spDVector xdvsupdiag(spDMatrix A, long shift);
extern void smdiag(spSMatrix A);
extern void lmdiag(spLMatrix A);
extern void fmdiag(spFMatrix A);
extern void dmdiag(spDMatrix A);
extern void smsupdiag(spSMatrix A, long shift);
extern void lmsupdiag(spLMatrix A, long shift);
extern void fmsupdiag(spFMatrix A, long shift);
extern void dmsupdiag(spDMatrix A, long shift);
extern spSMatrix xsmdiag(spSMatrix A);
extern spLMatrix xlmdiag(spLMatrix A);
extern spFMatrix xfmdiag(spFMatrix A);
extern spDMatrix xdmdiag(spDMatrix A);
extern spSMatrix xsmsupdiag(spSMatrix A, long shift);
extern spLMatrix xlmsupdiag(spLMatrix A, long shift);
extern spFMatrix xfmsupdiag(spFMatrix A, long shift);
extern spDMatrix xdmsupdiag(spDMatrix A, long shift);
extern spSMatrix xsmmakediag(spSVector d);
extern spLMatrix xlmmakediag(spLVector d);
extern spFMatrix xfmmakediag(spFVector d);
extern spDMatrix xdmmakediag(spDVector d);
extern spSMatrix xsmmakesupdiag(spSVector d, long shift);
extern spLMatrix xlmmakesupdiag(spLVector d, long shift);
extern spFMatrix xfmmakesupdiag(spFVector d, long shift);
extern spDMatrix xdmmakesupdiag(spDVector d, long shift);
/* make permutation matrix */
extern spSMatrix xsmmakeperm(spLVector index);
extern spLMatrix xlmmakeperm(spLVector index);
extern spFMatrix xfmmakeperm(spLVector index);
extern spDMatrix xdmmakeperm(spLVector index);

extern spBool smtranspose(spSMatrix A);
extern spSMatrix xsmtranspose(spSMatrix A);
extern spBool smconjtranspose(spSMatrix A);
extern spSMatrix xsmconjtranspose(spSMatrix A);
extern spBool lmtranspose(spLMatrix A);
extern spLMatrix xlmtranspose(spLMatrix A);
extern spBool lmconjtranspose(spLMatrix A);
extern spLMatrix xlmconjtranspose(spLMatrix A);
extern spBool fmtranspose(spFMatrix A);
extern spFMatrix xfmtranspose(spFMatrix A);
extern spBool fmconjtranspose(spFMatrix A);
extern spFMatrix xfmconjtranspose(spFMatrix A);
extern spBool dmtranspose(spDMatrix A);
extern spDMatrix xdmtranspose(spDMatrix A);
extern spBool dmconjtranspose(spDMatrix A);
extern spDMatrix xdmconjtranspose(spDMatrix A);

extern void dmscoper(spDMatrix x, const char *op, double t);
extern spDMatrix xdmscoper(spDMatrix x, const char *op, double t);
    
extern spBool dmbalance(spDMatrix A /* input/output */, spDMatrix D /* outputA=inv(D)*A*D */);
extern spDMatrix xdmbalance(spDMatrix A, spDMatrix *xoD /* outputA=inv(D)*A*D */);

/*
 * functions for householder matrix
 * x --> v (normalized householder vector)
 * H = eye(n) - 2 * v * v'; (H: householder matrix)
 * H * x --> [x(0)*x0weight, 0, 0, ...];
 * dvhouse return: x0weight
 * complex vector/matrix is supported 
 */
extern double dvhouse(spDVector x);
extern spDVector xdvhouse(spDVector x,
                        double *x0weight /* output weight for x(0) (conversion result of x(0)) */);
extern spBool dmhouse(spDMatrix H /* output householder matrix */,
                      spDVector v /* input hyperplane normal vector */);
extern spDMatrix xdmhouse(/* output householder matrix */
                        spDVector v /* input hyperplane normal vector */);

extern spBool dmhess(spDMatrix A /* input/output */, spDMatrix P /* inputA=P*outputA*P' */);
extern spDMatrix xdmhess(spDMatrix A, spDMatrix *xoP /* A=P*output*P' */);

extern spBool dmbidiag(spDMatrix A /* input/output */, spDMatrix U, spDMatrix V); /* inputA = U*outputA*V'. U and V can be NODATA */
extern spDMatrix xdmbidiag(spDMatrix A, spDMatrix *xoU, spDMatrix *xoV); /* inputA = U*outputA*V'. xoU and xoV can be NULL, must be freed */

/* functions for QR decomposition */
/* complex matrix is now supported (0.9.4+) */
extern spBool dmqr(spDMatrix A, spDMatrix Q, spDMatrix R);
/* double shift QR algorithm. accepts complex matrix */
extern spDVector xdmeigdsqr(spDMatrix A, long max_iteration /* 0: no limit */, double tolerance,
                          spDVectors *xoeigvecs /* can be NULL, must be freed */);
    
extern spBool dmsvdbidiag(spDMatrix B /* input: bidiagonal matrix, output: SVD diagonal matrix. size = [M, N] */,
                          long max_iteration, double tolerance,
                          spDMatrix oU /* size = [M, M] */, spDMatrix oV /* size = [N, N] */); /* inputB = oU * outputB * oV' */
extern spBool dmsvd(spDMatrix A /* output: SVD diagonal matrix. size = [M, N] */, long max_iteration, double tolerance,
                    spDMatrix oU /* size = [M, M] */, spDMatrix oV /* size = [N, N] */); /* inputA = oU * outputA * oV' */
extern spDMatrix xdmsvd(spDMatrix A /* size = [M, N] */, long max_iteration, double tolerance,
                      spDMatrix *xoU /* size = [M, M] */, spDMatrix *xoV /* size = [N, N] */); /* A = xoU * diag(output) * xoV' */

/* functions for Cholesky decomposition */
/* complex matrix is supported */
extern spBool dmchol(spDMatrix C/* size: (A->row, A->row) */, spDMatrix A, spBool lower);
extern spDMatrix xdmchol(spDMatrix A, spBool lower);
extern spBool dmcholinv(spDMatrix Ainv, spDMatrix C, spBool lower);
extern spDMatrix xdmcholinv(spDMatrix C, spBool lower);
extern spBool dvcholsolve(spDVector x, spDMatrix C, spDVector b, spBool lower);
extern spDVector xdvcholsolve(spDMatrix C, spDVector b, spBool lower);
    
/* functions for LU decomposition */
/* complex matrix is mostly supported (use dmlucplx or xdmlucplx for complex input matrix A) */
extern spBool dmlu(spDMatrix A, spLVector *xoindex /* can be NULL, must be freed */, double *det); /* det (determinant) can be NULL */
/* real LU decomposition with buffer (no memory allocation in this function) */
extern spBool dmluwb(spDMatrix A, spDVector weightbuf/* length: A->row (= A->col) */,
                     spLVector index/* length: A->row (= A->col) */, double *det); /* det (determinant) can be NULL */
extern spDMatrix xdmlu(spDMatrix A, spLVector *xoindex /* can be NULL, must be freed */, double *det); /* det (determinant) can be NULL */
extern spBool dmlucplx(spDMatrix A, spLVector *xoindex /* can be NULL, must be freed */, double *detr, double *deti); /* detr and deti (determinant) can be NULL */
/* complex LU decomposition with buffer (no memory allocation in this function) */
extern spBool dmlucplxwb(spDMatrix A, spDVector weightbuf/* length: A->row (= A->col) */,
                         spLVector index/* length: A->row (= A->col) */,
                         double *detr, double *deti); /* detr and deti (determinant) can be NULL */
extern spDMatrix xdmlucplx(spDMatrix A, spLVector *xoindex /* can be NULL, must be freed */, double *detr, double *deti); /* detr and deti (determinant) can be NULL */
extern spBool dmluupper(spDMatrix U /* output */, spDMatrix LU, spLVector index);
extern spBool dmlulower(spDMatrix L /* output */, spDMatrix LU, spLVector index);
extern spBool dmlupermlower(spDMatrix L1 /* output */, spDMatrix LU, spLVector index);
extern spDMatrix xdmluupper(spDMatrix LU, spLVector index);
extern spDMatrix xdmlulower(spDMatrix LU, spLVector index);
extern spDMatrix xdmlupermlower(spDMatrix LU, spLVector index);
extern spDMatrix xdmludiag(spDMatrix LU, spLVector index);
extern spBool dmluinv(spDMatrix Ainv, spDMatrix LU, spLVector index);
extern spDMatrix xdmluinv(spDMatrix LU, spLVector index);
extern spBool dvlusolve(spDVector x, spDMatrix LU, spLVector index, spDVector b);
extern spDVector xdvlusolve(spDMatrix LU, spLVector index, spDVector b);

extern spDMatrix xdminv(spDMatrix A);
extern spDVector xdvmldivide(spDMatrix A, spDVector b);

extern spBool dmtimes(spDMatrix C/* size: (A->row, A->col) */, spDMatrix A, spDMatrix B);
extern spDMatrix xdmtimes(spDMatrix A, spDMatrix B);
extern spBool dvmvtimes(spDVector y, spDMatrix A, spDVector x/* column vector */);
extern spDVector xdvmvtimes(spDMatrix A, spDVector x/* column vector */);
extern spBool dvvmtimes(spDVector y/* row vector */, spDVector x/* row vector */, spDMatrix A);
extern spDVector xdvvmtimes(spDVector x/* row vector */, spDMatrix A);
extern spBool dmvvtimes(spDMatrix C/* size: (a->length, b->length) */, spDVector a/* column vector */, spDVector b/* row vector */);
extern spDMatrix xdmvvtimes(spDVector a/* column vector */, spDVector b/* row vector */);

extern double dmfrobnorm(spDMatrix A);
extern double dminftynorm(spDMatrix A);
extern double dmonenorm(spDMatrix A);
extern double dmtwonorm(spDMatrix A);
/* p == -1: infinity norm, p == 0: frobenius norm, p == 1: 1-norm, p == 2: 2-norm */
extern double dmnorm(spDMatrix A, long p);

extern double dmcondp(spDMatrix A, long p);
extern double dmcond(spDMatrix A);

extern double dmrank(spDMatrix A, double tolerance);

extern spDMatrix xdmcompan(spDVector u, double tolerance,
                         long *onum_leading_zero /* can be NULL */, long *onum_trailing_zero /* can be NULL */);
extern spDVector xdvrootscompan(spDMatrix C /* must be companion matrix */, long num_leading_zero, long num_trailing_zero);
extern spDVector xdvroots(spDVector u, double tolerance);
    
#if defined(MACOS)
#pragma import off
#endif

#define smextractrow(mat, k, vec) smcutrow(mat, (long)(k), 0, mat->col, vec)
#define lmextractrow(mat, k, vec) lmcutrow(mat, (long)(k), 0, mat->col, vec)
#define fmextractrow(mat, k, vec) fmcutrow(mat, (long)(k), 0, mat->col, vec)
#define dmextractrow(mat, k, vec) dmcutrow(mat, (long)(k), 0, mat->col, vec)
#define smextractcol(mat, k, vec) smcutcol(mat, (long)(k), 0, mat->row, vec)
#define lmextractcol(mat, k, vec) lmcutcol(mat, (long)(k), 0, mat->row, vec)
#define fmextractcol(mat, k, vec) fmcutcol(mat, (long)(k), 0, mat->row, vec)
#define dmextractcol(mat, k, vec) dmcutcol(mat, (long)(k), 0, mat->row, vec)
    
#define xsmextractrow(mat, k) xsmcutrow(mat, (long)(k), 0, mat->col)
#define xlmextractrow(mat, k) xlmcutrow(mat, (long)(k), 0, mat->col)
#define xfmextractrow(mat, k) xfmcutrow(mat, (long)(k), 0, mat->col)
#define xdmextractrow(mat, k) xdmcutrow(mat, (long)(k), 0, mat->col)
#define xsmextractcol(mat, k) xsmcutcol(mat, (long)(k), 0, mat->row)
#define xlmextractcol(mat, k) xlmcutcol(mat, (long)(k), 0, mat->row)
#define xfmextractcol(mat, k) xfmcutcol(mat, (long)(k), 0, mat->row)
#define xdmextractcol(mat, k) xdmcutcol(mat, (long)(k), 0, mat->row)

/* copy vector to matrix row */
#define smcopyrow(mat, k, vec) smpasterowfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define lmcopyrow(mat, k, vec) lmpasterowfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define fmcopyrow(mat, k, vec) fmpasterowfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define dmcopyrow(mat, k, vec) dmpasterowfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
/* copy vector to matrix column */
#define smcopycol(mat, k, vec) smpastecolfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define lmcopycol(mat, k, vec) lmpastecolfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define fmcopycol(mat, k, vec) fmpastecolfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)
#define dmcopycol(mat, k, vec) dmpastecolfrom(mat, (long)(k), 0, vec->length, vec, 0, 0)

/* copy matrix to particular position of matrix */
#define smcopyto(matto, rowto, colto, matfrom) smpastematfrom(matto, rowto, colto, 0, 0, matfrom, 0, 0, 0)
#define lmcopyto(matto, rowto, colto, matfrom) lmpastematfrom(matto, rowto, colto, 0, 0, matfrom, 0, 0, 0)
#define fmcopyto(matto, rowto, colto, matfrom) fmpastematfrom(matto, rowto, colto, 0, 0, matfrom, 0, 0, 0)
#define dmcopyto(matto, rowto, colto, matfrom) dmpastematfrom(matto, rowto, colto, 0, 0, matfrom, 0, 0, 0)

#define smcopymatrow(matto, rowto, matfrom, rowfrom) smpastematrowfrom(matto, rowto, 0, matfrom->col, matfrom, rowfrom, 0, 0)
#define smcopymatcol(matto, colto, matfrom, colfrom) smpastematcolfrom(matto, colto, 0, matfrom->row, matfrom, colfrom, 0, 0)
#define lmcopymatrow(matto, rowto, matfrom, rowfrom) lmpastematrowfrom(matto, rowto, 0, matfrom->col, matfrom, rowfrom, 0, 0)
#define lmcopymatcol(matto, colto, matfrom, colfrom) lmpastematcolfrom(matto, colto, 0, matfrom->row, matfrom, colfrom, 0, 0)
#define fmcopymatrow(matto, rowto, matfrom, rowfrom) fmpastematrowfrom(matto, rowto, 0, matfrom->col, matfrom, rowfrom, 0, 0)
#define fmcopymatcol(matto, colto, matfrom, colfrom) fmpastematcolfrom(matto, colto, 0, matfrom->row, matfrom, colfrom, 0, 0)
#define dmcopymatrow(matto, rowto, matfrom, rowfrom) dmpastematrowfrom(matto, rowto, 0, matfrom->col, matfrom, rowfrom, 0, 0)
#define dmcopymatcol(matto, colto, matfrom, colfrom) dmpastematcolfrom(matto, colto, 0, matfrom->row, matfrom, colfrom, 0, 0)
    
#define xsmzeros(row, col) xsmnums(row, col, 0)
#define xlmzeros(row, col) xlmnums(row, col, 0)
#define xfmzeros(row, col) xfmnums(row, col, 0.0f)
#define xdmzeros(row, col) xdmnums(row, col, 0.0)
#define xsmones(row, col) xsmnums(row, col, 1)
#define xlmones(row, col) xlmnums(row, col, 1)
#define xfmones(row, col) xlmnums(row, col, 1.0f)
#define xdmones(row, col) xdmnums(row, col, 1.0)

#define smzeros(mat, row, col) smnums(mat, row, col, 0)
#define lmzeros(mat, row, col) lmnums(mat, row, col, 0)
#define fmzeros(mat, row, col) fmnums(mat, row, col, 0.0f)
#define dmzeros(mat, row, col) dmnums(mat, row, col, 0.0)
#define smones(mat, row, col) smnums(mat, row, col, 1)
#define lmones(mat, row, col) lmnums(mat, row, col, 1)
#define fmones(mat, row, col) fmnums(mat, row, col, 1.0f)
#define dmones(mat, row, col) dmnums(mat, row, col, 1.0)

#define smizeros(mat, row, col) sminums(mat, row, col, 0)
#define lmizeros(mat, row, col) lminums(mat, row, col, 0)
#define fmizeros(mat, row, col) fminums(mat, row, col, 0.0f)
#define dmizeros(mat, row, col) dminums(mat, row, col, 0.0)
#define smiones(mat, row, col) sminums(mat, row, col, 1)
#define lmiones(mat, row, col) lminums(mat, row, col, 1)
#define fmiones(mat, row, col) fminums(mat, row, col, 1.0f)
#define dmiones(mat, row, col) dminums(mat, row, col, 1.0)

#define xsmrizeros(row, col) xsmrinums(row, col, 0)
#define xlmrizeros(row, col) xlmrinums(row, col, 0)
#define xfmrizeros(row, col) xfmrinums(row, col, 0.0f)
#define xdmrizeros(row, col) xdmrinums(row, col, 0.0)
#define xsmriones(row, col) xsmrinums(row, col, 1)
#define xlmriones(row, col) xlmrinums(row, col, 1)
#define xfmriones(row, col) xfmrinums(row, col, 1.0f)
#define xdmriones(row, col) xdmrinums(row, col, 1.0)

#define xsmnull() xsmalloc(0, 0)
#define xlmnull() xlmalloc(0, 0)
#define xfmnull() xfmalloc(0, 0)
#define xdmnull() xdmalloc(0, 0)

#define msset(xs, index, x) {(xs)->matrix[index]=(x);}
#define smsset msset
#define lmsset msset
#define fmsset msset
#define dmsset msset

/* for backwards compatibility */
typedef spSMatrix SMATRIX;
typedef spLMatrix LMATRIX;
typedef spFMatrix FMATRIX;
typedef spDMatrix DMATRIX;

typedef spSMatrices SMATRICES;
typedef spLMatrices LMATRICES;
typedef spFMatrices FMATRICES;
typedef spDMatrices DMATRICES;

#define xdmmin dmmin
#define xdmmax dmmax

#define xdvmtimes xdvmvtimes

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_MATRIX_H */
