/*
 *	vector.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <sp/sp.h>
#include <sp/memory.h>
#include <sp/vector.h>

#ifdef SP_USE_VECTOR_ENGINE
#if 1
#define SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK
#else
#undef SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK
#endif

#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>

spPlugin *sp_default_vector_plugin = NULL;

#ifdef SP_USE_VECTOR_ENGINE
#define xfreevecdata(x, data) (x->plugin != NULL && SpVectorPluginRec(x)->free_host != NULL ? SpVectorPluginRec(x)->free_host(x->plugin->instance, data) : _xspFree((char *)(data)))
#else
#define xfreevecdata(x, data) _xspFree((char *)(data))
#endif

#define SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(name) { #name , spOffsetOf(spVectorPluginInternalFuncList, name )}
spVectorPluginInternalFuncListOffsetTable sp_vector_plugin_func_list_offset_table[] = {
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rset),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rget),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(iset),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(iget),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(conj),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rcopy),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(icopy),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rcopyi),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(icopyr),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(radd),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(iadd),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rerase),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(ierase),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rinit),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(iinit),

    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(ridxnums),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(iidxnums),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rshift),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(ishift),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(plus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(minus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(times),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(divide),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(power),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(logiand),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(logior),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(logixor),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scplus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scminus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sctimes),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scdivide),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scpower),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(linplus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(linminus),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(lintimes),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(lindivide),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(linpower),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(morph),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(find),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(remap),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevalle),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevallt),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevalge),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevalgt),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevaleq),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scevalne),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sum),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(isum),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sqsum),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(abssum),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(prod),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(riprod),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(minfunc),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(maxfunc),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scmin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(scmax),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(elmin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(elmax),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nmin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nmax),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(abs),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sqrt),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(square),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sign),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(cumsum),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(cumprod),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(codiff),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sort),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sortidx),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(median),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(zerocross),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(angle),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(unwrap),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(hamming),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nhamming),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(hanning),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nhanning),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(hann),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nhann),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(blackman),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nblackman),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(gausswin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(ngausswin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(kaiser),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nkaiser),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(nboxcar),

    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(decibela),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(decibelp),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(undecibela),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(undecibelp),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(initfft),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(resetfft),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(freefft),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(fft),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rfft),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rffttopower),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rffttoreal),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rffttoimag),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(rffttoangle),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(fftturn),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(fftshift),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(fftgrpdly),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(ceptompc),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(srand),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(randun),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(gauss),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(dot),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(cos),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(tan),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(cosh),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(sinh),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(tanh),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(acos),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(asin),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(atan),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(atan2),
    
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(exp),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(log),
    SP_DECLARE_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_OFFSET_ENTRY(log10),
    
    {NULL, 0},
};

spBool spCreateVectorPluginInternalFuncListTable(spPlugin *plugin)
{
    int i, j;
    spVectorPluginRec *vprec;
    spVectorPluginInternalFuncList *table = NULL;
    spVectorPluginInternalFuncList *list;
    void *func;
    spBool flag;
    char type;
    static const char *typelist = "slfd";

    spDebug(100, "spCreateVectorPluginInternalFuncListTable", "in\n");
    
    vprec = (spVectorPluginRec *)plugin->rec;
    if (vprec->get_func == NULL) {
        spDebug(30, "spCreateVectorPluginInternalFuncListTable", "get_func is NULL\n");
        return SP_FALSE;
    }
    
    if ((flag = spAllocPluginSpecificData(plugin, 8 * sizeof(struct _spVectorPluginInternalFuncList))) != SP_FALSE
        && (table = (spVectorPluginInternalFuncList *)spGetPluginSpecificData(plugin)) != NULL) {
        if (table[0].version_id != SP_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_VERSION_ID) {
            for (i = 0; sp_vector_plugin_func_list_offset_table[i].name != NULL; i++) {
                for (j = 0; j < 4; j++) {
                    type = typelist[j];
                
                    list = &table[j * 2];
                    list->version_id = SP_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_VERSION_ID;
                    func = vprec->get_func(plugin->instance, type, SP_FALSE, sp_vector_plugin_func_list_offset_table[i].name);
                    if (func != NULL) {
                        *(void **)(((char *)list) + sp_vector_plugin_func_list_offset_table[i].offset) = func;
                    }

                    list = &table[j * 2 + 1];
                    list->version_id = SP_VECTOR_PLUGIN_INTERNAL_FUNC_LIST_VERSION_ID;
                    func = vprec->get_func(plugin->instance, type, SP_TRUE, sp_vector_plugin_func_list_offset_table[i].name);
                    if (func != NULL) {
                        *(void **)(((char *)list) + sp_vector_plugin_func_list_offset_table[i].offset) = func;
                    }
                }
            }
        }
    }

    spDebug(100, "spCreateVectorPluginInternalFuncListTable", "flag = %d\n", flag);
    
    if (flag == SP_FALSE || table == NULL) {
        spAllocPluginSpecificData(plugin, 0);
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

spVectorPluginInternalFuncList *spGetVectorPluginInternalFuncListTable(spPlugin *plugin)
{
    return (spVectorPluginInternalFuncList *)spGetPluginSpecificData(plugin);
}

spVectorPluginInternalFuncList *spGetVectorPluginInternalFuncList(spPlugin *plugin, int index, int cplx_flag)
{
    spVectorPluginInternalFuncList *table;

    if ((table = spGetVectorPluginInternalFuncListTable(plugin)) == NULL) return NULL;
    
    spDebug(100, "spGetVectorPluginInternalFuncList", "index = %d, cplx_flag = %d\n", index, cplx_flag);
    return &table[2 * index + cplx_flag];
}

int spGetVectorPluginFuncIndexFromList(spVectorPluginFuncList *list, const char *func_name)
{
    int i;
    int index = -1;
    
    for (i = 0; list[i].name != NULL; i++) {
        if (streq(list[i].name, func_name)) {
            index = i;
        }
    }

    return index;
}

spVectorPluginFuncList *spGetVectorPluginFuncFromList(spVectorPluginFuncList *list, const char *func_name)
{
    int index;
    
    index = spGetVectorPluginFuncIndexFromList(list, func_name);

    if (index >= 0) {
        return &list[index];
    } else {
        return NULL;
    }
}

void *spFindVectorPluginFuncFromList(spVectorPluginFuncList *list, char type, spBool cplx_flag, const char *func_name)
{
    int index;
    void *func = NULL;

    index = spGetVectorPluginFuncIndexFromList(list, func_name);
    
    if (index < 0 || (cplx_flag && (list[index].mask & SP_VECTOR_PLUGIN_FUNC_MASK_COMPLEX) == 0L)) return NULL;
    
    switch (type) {
      case 's':
        if (list[index].mask & SP_VECTOR_PLUGIN_FUNC_MASK_TYPE_SVECTOR) {
            func = list[index].func;
        }
        break;
      case 'l':
        if (list[index].mask & SP_VECTOR_PLUGIN_FUNC_MASK_TYPE_LVECTOR) {
            func = list[index].func;
        }
        break;
      case 'f':
        if (list[index].mask & SP_VECTOR_PLUGIN_FUNC_MASK_TYPE_FVECTOR) {
            func = list[index].func;
        }
        break;
      case 'd':
        if (list[index].mask & SP_VECTOR_PLUGIN_FUNC_MASK_TYPE_DVECTOR) {
            func = list[index].func;
        }
        break;
    }

    return func;
}

static spBool spSearchVectorPluginCallback(spPlugin *plugin, void *call_data)
{
    return spIsVectorPlugin(plugin);
}

int spGetNumVectorPlugin(void)
{
    return spGetNumSpecificPlugin(spSearchVectorPluginCallback, NULL);
}

char *xspSearchVectorPluginFile(int *index)
{
    return xspSearchSpecificPluginFile(index, spSearchVectorPluginCallback, NULL);
}

spPlugin *spSetDefaultVectorPlugin(const char *name)
{
    spPlugin *plugin;

    if (sp_default_vector_plugin != NULL) {
	spFreeVectorPlugin(sp_default_vector_plugin);
	sp_default_vector_plugin = NULL;
    }
    
    if (strnone(name) || strcaseeq(name, "none") || strcaseeq(name, "default")) {
    } else {
	if ((plugin = spLoadVectorPlugin(name)) != NULL) {
	    sp_default_vector_plugin = plugin;
	} else {
	    sp_default_vector_plugin = NULL;
	}
    }

    return sp_default_vector_plugin;
}

spBool spIsVectorPlugin(spPlugin *plugin)
{
    if (plugin == NULL) return SP_FALSE;

    if (spEqPluginType(plugin, SP_PLUGIN_VECTOR) == SP_TRUE) {
	return SP_TRUE;
    }
    
    return SP_FALSE;
}

spPlugin *spLoadVectorPlugin(const char *name)
{
    spPlugin *plugin;
    
    if ((plugin = spLoadPlugin(name)) == NULL) {
	spDebug(20, "spLoadVectorPlugin", "spLoadPlugin failed\n");
	return NULL;
    }
    if (spIsVectorPlugin(plugin) == SP_FALSE
	|| spInitPluginInstance(plugin) == SP_FALSE
        || spCreateVectorPluginInternalFuncListTable(plugin) == SP_FALSE) {
	spDebug(20, "spLoadVectorPlugin", "spIsVectorPlugin or spInitPluginInstance failed\n");
	spFreePlugin(plugin);
	return NULL;
    }

    spDebug(100, "spLoadVectorPlugin", "done\n");
    
    return plugin;
}

spBool spFreeVectorPlugin(spPlugin *plugin)
{
    if (plugin == NULL) return SP_FALSE;
    
    return spFreePlugin(plugin);
}

spPlugin *spGetDefaultVectorPlugin(void)
{
    return sp_default_vector_plugin;
}

spBool spFreeDefaultVectorPlugin(void)
{
    spBool flag;
    
    flag = spFreeVectorPlugin(sp_default_vector_plugin);
    sp_default_vector_plugin = NULL;
    
    return flag;
}

spBool spGetVectorPluginNumUnit(spPlugin *plugin, int *num_unit)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	*num_unit = 1;
	return SP_TRUE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->get_num_unit == NULL) {
	return SP_FALSE;
    } else {
        return vprec->get_num_unit(plugin->instance, num_unit);
    }
}

const char *spGetVectorPluginUnitName(spPlugin *plugin, int index)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return "default";
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->get_unit_name == NULL) {
	return NULL;
    } else {
        return vprec->get_unit_name(plugin->instance, index);
    }
}

spBool spSelectVectorPluginUnit(spPlugin *plugin, int index)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	if (index == 0) {
	    return SP_TRUE;
	} else {
	    return SP_FALSE;
	}
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->select_unit == NULL) {
	return SP_FALSE;
    } else {
        return vprec->select_unit(plugin->instance, index);
    }
}

spBool spGetDefaultVectorPluginNumUnit(int *num_unit)
{
    return spGetVectorPluginNumUnit(sp_default_vector_plugin, num_unit);
}

const char *spGetDefaultVectorPluginUnitName(int index)
{
    return spGetVectorPluginUnitName(sp_default_vector_plugin, index);
}

spBool spSelectDefaultVectorPluginUnit(int index)
{
    return spSelectVectorPluginUnit(sp_default_vector_plugin, index);
}

spBool spIsVectorPluginPrecisionSupported(spPlugin *plugin, spPrecision precision, spComputeSpeed *speed)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->is_precision_supported == NULL) {
	return SP_FALSE;
    } else {
	return vprec->is_precision_supported(plugin->instance, precision, speed);
    }
}

spBool spSetVectorPluginPrecision(spPlugin *plugin, spPrecision precision)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;

    if (vprec->set_precision == NULL) {
	return SP_FALSE;
    } else {
	return vprec->set_precision(plugin->instance, precision);
    }
}

spBool spGetVectorPluginPrecision(spPlugin *plugin, spPrecision *precision)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->get_precision == NULL) {
	return SP_FALSE;
    } else {
	return vprec->get_precision(plugin->instance, precision);
    }
}

spBool spIsVectorPluginFastModeSupported(spPlugin *plugin, spBool fast_flag)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->is_fast_mode_supported == NULL) {
	return SP_FALSE;
    } else {
	return vprec->is_fast_mode_supported(plugin->instance, fast_flag);
    }
}

spBool spSetVectorPluginFastMode(spPlugin *plugin, spBool fast_flag)
{
    spVectorPluginRec *vprec;
    
    spDebug(100, "spSetVectorPluginFastMode", "fast_flag = %d\n", fast_flag);
    
    if (plugin == NULL || plugin->instance == NULL) {
        spDebug(30, "spSetVectorPluginFastMode", "no plugin\n");
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;

    if (vprec->set_fast_mode == NULL) {
        spDebug(30, "spSetVectorPluginFastMode", "set_fast_mode == NULL\n");
	return SP_FALSE;
    } else {
        spDebug(80, "spSetVectorPluginFastMode", "call set_fast_mode\n");
	return vprec->set_fast_mode(plugin->instance, fast_flag);
    }
}

spBool spGetVectorPluginFastMode(spPlugin *plugin, spBool *fast_flag)
{
    spVectorPluginRec *vprec;
    
    if (plugin == NULL || plugin->instance == NULL) {
	return SP_FALSE;
    }
    
    vprec = (spVectorPluginRec *)plugin->rec;
    
    if (vprec->get_fast_mode == NULL) {
	return SP_FALSE;
    } else {
	return vprec->get_fast_mode(plugin->instance, fast_flag);
    }
}

spBool spSetDefaultVectorPluginFastMode(spBool fast_flag)
{
    return spSetVectorPluginFastMode(sp_default_vector_plugin, fast_flag);
}

spBool spGetDefaultVectorPluginFastMode(spBool *fast_flag)
{
    return spGetVectorPluginFastMode(sp_default_vector_plugin, fast_flag);
}

spBool svlockcore(spSVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->rlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->rlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->ilock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->ilock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool lvlockcore(spLVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->rlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->rlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->ilock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->ilock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool fvlockcore(spFVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->rlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->rlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->ilock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->ilock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool dvlockcore(spDVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->rlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->rlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->ilock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->ilock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool svunlockcore(spSVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->runlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->runlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->iunlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->iunlock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool lvunlockcore(spLVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->runlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->runlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->iunlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->iunlock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool fvunlockcore(spFVector x, int lock_switch)
{
    if (SpVectorPluginRec(x)->runlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->runlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->iunlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->iunlock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}

spBool dvunlockcore(spDVector x, int lock_switch)
{
    spDebug(100, "dvunlockcore", "in\n");
    if (SpVectorPluginRec(x)->runlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
        if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
            spDebug(100, "dvunlockcore", "call first ralloc\n");
            SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
        }
#endif
        SpVectorPluginRec(x)->runlock(x->instance, x->data, x->length, lock_switch);
    }
    
    if (x->imag != NULL) {
        if (SpVectorPluginRec(x)->iunlock != NULL) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
                spDebug(100, "dvunlockcore", "call first ialloc\n");
                SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
            }
#endif
            SpVectorPluginRec(x)->iunlock(x->instance, x->imag, x->length, lock_switch);
        }
    }

    return SP_TRUE;
}
#endif /* SP_USE_VECTOR_ENGINE */

long spsizeof(char type)
{
    spDebug(100, "spsizeof", "type = %c\n", type);
    
    if (type == 's') {
	return sizeof(short);
    } else if (type == 'l') {
	return sizeof(long);
    } else if (type == 'f') {
	return sizeof(float);
    } else if (type == 'd') {
	return sizeof(double);
    } else {
	spDebug(10, "spsizeof", "unknown type %c\n", type);
	return 0;
    }
}

static spBool svgetvinstance(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL) {
	spDebug(100, "xsvalloc", "call get_vinstance\n");
	if ((x->instance = SpVectorPluginRec(x)->get_vinstance(x->plugin->instance, 's')) != NULL) {
	    spDebug(100, "xsvalloc", "get_vinstance done\n");
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->runlock == NULL)
#endif
            {
                SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
            }
	    
	    return SP_TRUE;
	}
    }
#endif

    return SP_FALSE;
}

static spBool lvgetvinstance(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL) {
	spDebug(100, "xlvalloc", "call get_vinstance\n");
	if ((x->instance = SpVectorPluginRec(x)->get_vinstance(x->plugin->instance, 'l')) != NULL) {
	    spDebug(100, "xlvalloc", "get_vinstance done\n");
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->runlock == NULL)
#endif
            {
                SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
            }
	    
	    return SP_TRUE;
	}
    }
#endif

    return SP_FALSE;
}

static spBool fvgetvinstance(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL) {
	spDebug(100, "xfvalloc", "call get_vinstance\n");
	if ((x->instance = SpVectorPluginRec(x)->get_vinstance(x->plugin->instance, 'f')) != NULL) {
	    spDebug(100, "xfvalloc", "get_vinstance done\n");
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->runlock == NULL)
#endif
            {
                SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
            }
	    
	    return SP_TRUE;
	}
    }
#endif

    return SP_FALSE;
}

static spBool dvgetvinstance(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL) {
	spDebug(100, "xdvalloc", "call get_vinstance\n");
	if ((x->instance = SpVectorPluginRec(x)->get_vinstance(x->plugin->instance, 'd')) != NULL) {
	    spDebug(100, "xdvalloc", "get_vinstance done\n");
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
            if (SpVectorPluginRec(x)->runlock == NULL)
#endif
            {
                SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
            }
	    
	    return SP_TRUE;
	}
    }
#endif

    return SP_FALSE;
}

spPlugin *svgetplugin(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return NULL;
    return x->plugin;
#else
    return NULL;
#endif
}

spPlugin *lvgetplugin(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return NULL;
    return x->plugin;
#else
    return NULL;
#endif
}

spPlugin *fvgetplugin(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return NULL;
    return x->plugin;
#else
    return NULL;
#endif
}

spPlugin *dvgetplugin(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return NULL;
    return x->plugin;
#else
    return NULL;
#endif
}

#ifdef SP_USE_VECTOR_ENGINE
#define xreplaceplugin(x, plugin, type) \
{ \
    if (x->plugin != NULL) { \
        void *data = NULL; \
        void *imag = NULL; \
        spVectorPluginRec *new_vprec = NULL; \
	if (x->instance != NULL) { \
            if (SpVectorPluginRec(x)->getrptr(x->instance) != NULL) { \
                SpVectorPluginRec(x)->rfree(x->instance); \
            } \
            if (SpVectorPluginRec(x)->getiptr(x->instance) != NULL) { \
                SpVectorPluginRec(x)->ifree(x->instance); \
            } \
        } \
        if (plugin != NULL) { \
            new_vprec = (spVectorPluginRec *)plugin->rec; \
            if (new_vprec->alloc_host != NULL) { \
                if (x->data != NULL) { \
                    data = new_vprec->alloc_host(plugin->instance, x->length * sizeof(type)); \
                } \
                if (x->imag != NULL) { \
                    imag = new_vprec->alloc_host(plugin->instance, x->length * sizeof(type)); \
                } \
            } \
        } else if (SpVectorPluginRec(x)->alloc_host != NULL || SpVectorPluginRec(x)->free_host != NULL) { \
            if (x->data != NULL) { \
                data = xalloc(x->length, type); /* to replace old allocated data */ \
            } \
            if (x->imag != NULL) { \
                imag = xalloc(x->length, type); /* to replace old allocated imag */ \
            } \
        } \
        if (data != NULL) { \
            memcpy(data, x->data, x->length * sizeof(type)); \
        } \
        if (imag != NULL) { \
            memcpy(imag, x->imag, x->length * sizeof(type)); \
        } \
        if (SpVectorPluginRec(x)->free_host != NULL) { \
            if (x->data != NULL) { \
                SpVectorPluginRec(x)->free_host(x->plugin->instance, x->data); \
                x->data = NULL; \
            } \
            if (x->imag != NULL) { \
                SpVectorPluginRec(x)->free_host(x->plugin->instance, x->imag); \
                x->imag = NULL; \
            } \
        } \
	if (x->instance != NULL) { \
	    SpVectorPluginRec(x)->free_vinstance(x->plugin->instance, x->instance); \
	} \
        if (data != NULL) { \
            x->data = data; \
        } \
        if (imag != NULL) { \
            x->imag = imag; \
        } \
    } \
    x->plugin = plugin; \
}
#endif

spBool svsetplugin(spSVector x, spPlugin *plugin)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return SP_FALSE;
    if (x->plugin == plugin) return SP_TRUE;
    
    xreplaceplugin(x, plugin, short);
    
    if (x->plugin != NULL) {
	return svgetvinstance(x);
    } else {
	return SP_TRUE;
    }
#else
    return SP_FALSE;
#endif
}

spBool lvsetplugin(spLVector x, spPlugin *plugin)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return SP_FALSE;
    if (x->plugin == plugin) return SP_TRUE;
    
    xreplaceplugin(x, plugin, long);
    
    if (x->plugin != NULL) {
	return lvgetvinstance(x);
    } else {
	return SP_TRUE;
    }
#else
    return SP_FALSE;
#endif
}

spBool fvsetplugin(spFVector x, spPlugin *plugin)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return SP_FALSE;
    if (x->plugin == plugin) return SP_TRUE;
    
    xreplaceplugin(x, plugin, float);
    
    if (x->plugin != NULL) {
	return fvgetvinstance(x);
    } else {
	return SP_TRUE;
    }
#else
    return SP_FALSE;
#endif
}

spBool dvsetplugin(spDVector x, spPlugin *plugin)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA) return SP_FALSE;
    if (x->plugin == plugin) return SP_TRUE;

    xreplaceplugin(x, plugin, double);
    
    if (x->plugin != NULL) {
	return dvgetvinstance(x);
    } else {
	return SP_TRUE;
    }
#else
    return SP_FALSE;
#endif
}

/*
 *	allocate and free memory
 */
short *xallocsdata(spSVector x, long length)
{
    short *data = NULL;
    
    if (length >= 1) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	    data = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(short));
	} else {
	    data = xalloc(length, short);
	}
#else
	data = xalloc(length, short);
#endif
    }

    return data;
}

spSVector xsvallocve(spPlugin *plugin, long length)
{
    spSVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spSVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
#endif

    x->data = xallocsdata(x, length);
    x->imag = NULL;
    x->length = length;

    svgetvinstance(x);

    return x;
}

long *xallocldata(spLVector x, long length)
{
    long *data = NULL;
    
    if (length >= 1) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	    data = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(long));
	} else {
	    data = xalloc(length, long);
	}
#else
	data = xalloc(length, long);
#endif
    }

    return data;
}

spLVector xlvallocve(spPlugin *plugin, long length)
{
    spLVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spLVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
#endif
    
    x->data = xallocldata(x, length);
    x->imag = NULL;
    x->length = length;
    
    lvgetvinstance(x);

    return x;
}

float *xallocfdata(spFVector x, long length)
{
    float *data = NULL;
    
    if (length >= 1) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	    data = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(float));
	} else {
	    data = xalloc(length, float);
	}
#else
	data = xalloc(length, float);
#endif
    }

    return data;
}

spFVector xfvallocve(spPlugin *plugin, long length)
{
    spFVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spFVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
#endif
    
    x->data = xallocfdata(x, length);
    x->imag = NULL;
    x->length = length;
    
    fvgetvinstance(x);

    return x;
}

double *xallocddata(spDVector x, long length)
{
    double *data = NULL;
    
    if (length >= 1) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	    data = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(double));
	} else {
	    data = xalloc(length, double);
	}
#else
	data = xalloc(length, double);
#endif
    }

    return data;
}

spDVector xdvallocve(spPlugin *plugin, long length)
{
    spDVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spDVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
#endif
    
    x->data = xallocddata(x, length);
    x->imag = NULL;
    x->length = length;
    
    dvgetvinstance(x);

    return x;
}

spSVector xsvalloc(long length)
{
    return xsvallocve(NULL, length);
}

spLVector xlvalloc(long length)
{
    return xlvallocve(NULL, length);
}

spFVector xfvalloc(long length)
{
    return xfvallocve(NULL, length);
}

spDVector xdvalloc(long length)
{
    return xdvallocve(NULL, length);
}

void xsvfree(spSVector x)
{
    if (x != NULL) {
	svrfree(x);
	svifree(x);
	
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL) {
	    SpVectorPluginRec(x)->free_vinstance(x->plugin->instance, x->instance);
	}
#endif
	xfree(x);
    }

    return;
}

void xlvfree(spLVector x)
{
    if (x != NULL) {
	lvrfree(x);
	lvifree(x);
	
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL) {
	    SpVectorPluginRec(x)->free_vinstance(x->plugin->instance, x->instance);
	}
#endif
	xfree(x);
    }

    return;
}

void xfvfree(spFVector x)
{
    if (x != NULL) {
	fvrfree(x);
	fvifree(x);
	
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL) {
	    SpVectorPluginRec(x)->free_vinstance(x->plugin->instance, x->instance);
	}
#endif
	xfree(x);
    }

    return;
}

void xdvfree(spDVector x)
{
    if (x != NULL) {
	dvrfree(x);
	dvifree(x);
	
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL) {
	    SpVectorPluginRec(x)->free_vinstance(x->plugin->instance, x->instance);
	}
#endif
	xfree(x);
    }

    return;
}

spSVector xsvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag)
{
    spSVector x;
    
    x = xalloc(1, struct _spSVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = instance;
    x->locked = SP_FALSE;
#endif
    
    x->length = MAX(length, 0);
    x->data = NULL;
    x->imag = NULL;
    
    if (x->length >= 1) {
	x->data = xallocsdata(x, x->length);
	if (imag_flag) {
	    x->imag = xallocsdata(x, x->length);
	}
    }
    
    return x;
}

spLVector xlvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag)
{
    spLVector x;
    
    x = xalloc(1, struct _spLVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = instance;
    x->locked = SP_FALSE;
#endif
    
    x->length = MAX(length, 0);
    x->data = NULL;
    x->imag = NULL;
    
    if (x->length >= 1) {
	x->data = xallocldata(x, x->length);
	if (imag_flag) {
	    x->imag = xallocldata(x, x->length);
	}
    }
    
    return x;
}

spFVector xfvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag)
{
    spFVector x;
    
    x = xalloc(1, struct _spFVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = instance;
    x->locked = SP_FALSE;
#endif
    
    x->length = MAX(length, 0);
    x->data = NULL;
    x->imag = NULL;
    
    if (x->length >= 1) {
	x->data = xallocfdata(x, x->length);
	if (imag_flag) {
	    x->imag = xallocfdata(x, x->length);
	}
    }
    
    return x;
}

spDVector xdvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag)
{
    spDVector x;
    
    x = xalloc(1, struct _spDVector);
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = instance;
    x->locked = SP_FALSE;
#endif
    
    x->length = MAX(length, 0);
    x->data = NULL;
    x->imag = NULL;
    
    if (x->length >= 1) {
	x->data = xallocddata(x, x->length);
	if (imag_flag) {
	    x->imag = xallocddata(x, x->length);
	}
    }
    
    return x;
}

spSVector xsvallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spSVector x;
    
    x = xsvallocve(plugin, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag == SP_TRUE && x->instance != NULL) {
	svunlockcore(x, SP_VECTOR_LOCK_SWITCH_NOSYNC);
	x->locked = SP_FALSE;
    }
#endif
    
    return x;
}

spLVector xlvallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spLVector x;
    
    x = xlvallocve(plugin, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag == SP_TRUE && x->instance != NULL) {
	lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NOSYNC);
	x->locked = SP_FALSE;
    }
#endif
    
    return x;
}

spFVector xfvallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector x;
    
    x = xfvallocve(plugin, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag == SP_TRUE && x->instance != NULL) {
	fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NOSYNC);
	x->locked = SP_FALSE;
    }
#endif
    
    return x;
}

spDVector xdvallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector x;
    
    x = xdvallocve(plugin, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag == SP_TRUE && x->instance != NULL) {
	dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NOSYNC);
	x->locked = SP_FALSE;
    }
#endif
    
    return x;
}

spSVector xsvalloculfromv(spSVector x, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    return xsvallocul(x->plugin, length, spIsFalse(svislocked(x)));
#else
    return xsvalloc(length);
#endif
}

spLVector xlvalloculfromv(spLVector x, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    return xlvallocul(x->plugin, length, spIsFalse(lvislocked(x)));
#else
    return xlvalloc(length);
#endif
}

spFVector xfvalloculfromv(spFVector x, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    return xfvallocul(x->plugin, length, spIsFalse(fvislocked(x)));
#else
    return xfvalloc(length);
#endif
}

spDVector xdvalloculfromv(spDVector x, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    return xdvallocul(x->plugin, length, spIsFalse(dvislocked(x)));
#else
    return xdvalloc(length);
#endif
}

void svrfree(spSVector x)
{
    if (x->data != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->rfree(x->instance);
	}
	xfreevecdata(x, x->data);
#else
	xfree(x->data);
#endif
	x->data = NULL;
    }
    
    return;
}

void lvrfree(spLVector x)
{
    if (x->data != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->rfree(x->instance);
	}
	xfreevecdata(x, x->data);
#else
	xfree(x->data);
#endif
	x->data = NULL;
    }
    
    return;
}

void fvrfree(spFVector x)
{
    if (x->data != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->rfree(x->instance);
	}
	xfreevecdata(x, x->data);
#else
	xfree(x->data);
#endif
	x->data = NULL;
    }
    
    return;
}

void dvrfree(spDVector x)
{
    if (x->data != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->rfree(x->instance);
	}
	xfreevecdata(x, x->data);
#else
	xfree(x->data);
#endif
	x->data = NULL;
    }
    
    return;
}

void svialloc(spSVector x)
{
    svifree(x);
    x->imag = xallocsdata(x, x->length);
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	&& (svislocked(x) == SP_FALSE || SpVectorPluginRec(x)->iunlock == NULL)
#endif
	) {
	SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
    }
#endif

    return;
}

void lvialloc(spLVector x)
{
    lvifree(x);
    x->imag = xallocldata(x, x->length);
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	&& (lvislocked(x) == SP_FALSE || SpVectorPluginRec(x)->iunlock == NULL)
#endif
	) {
	SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
    }
#endif

    return;
}

void fvialloc(spFVector x)
{
    fvifree(x);
    x->imag = xallocfdata(x, x->length);
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	&& (fvislocked(x) == SP_FALSE || SpVectorPluginRec(x)->iunlock == NULL)
#endif
	) {
	SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
    }
#endif

    return;
}

void dvialloc(spDVector x)
{
    dvifree(x);
    x->imag = xallocddata(x, x->length);
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	&& (dvislocked(x) == SP_FALSE || SpVectorPluginRec(x)->iunlock == NULL)
#endif
	) {
	SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
    }
#endif

    return;
}

void svifree(spSVector x)
{
    if (x->imag != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getiptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->ifree(x->instance);
	}
	xfreevecdata(x, x->imag);
#else
	xfree(x->imag);
#endif
	x->imag = NULL;
    }
    
    return;
}

void lvifree(spLVector x)
{
    if (x->imag != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getiptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->ifree(x->instance);
	}
	xfreevecdata(x, x->imag);
#else
	xfree(x->imag);
#endif
	x->imag = NULL;
    }
    
    return;
}

void fvifree(spFVector x)
{
    if (x->imag != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getiptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->ifree(x->instance);
	}
	xfreevecdata(x, x->imag);
#else
	xfree(x->imag);
#endif
	x->imag = NULL;
    }
    
    return;
}

void dvifree(spDVector x)
{
    if (x->imag != NULL) {
#ifdef SP_USE_VECTOR_ENGINE
	if (x->instance != NULL && SpVectorPluginRec(x)->getiptr(x->instance) != NULL) {
	    SpVectorPluginRec(x)->ifree(x->instance);
	}
	xfreevecdata(x, x->imag);
#else
	xfree(x->imag);
#endif
	x->imag = NULL;
    }
    
    return;
}

spSVector xsvriallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spSVector x;

    x = xsvallocul(plugin, length, unlock_flag);
    svialloc(x);

    return x;
}

spLVector xlvriallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spLVector x;

    x = xlvallocul(plugin, length, unlock_flag);
    lvialloc(x);

    return x;
}

spFVector xfvriallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector x;

    x = xfvallocul(plugin, length, unlock_flag);
    fvialloc(x);

    return x;
}

spDVector xdvriallocul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector x;

    x = xdvallocul(plugin, length, unlock_flag);
    dvialloc(x);

    return x;
}

spSVector xsvrialloc(long length)
{
    return xsvriallocul(NULL, length, SP_FALSE);
}

spLVector xlvrialloc(long length)
{
    return xlvriallocul(NULL, length, SP_FALSE);
}

spFVector xfvrialloc(long length)
{
    return xfvriallocul(NULL, length, SP_FALSE);
}

spDVector xdvrialloc(long length)
{
    return xdvriallocul(NULL, length, SP_FALSE);
}

short *xreallocsdata(spSVector x, short *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL && SpVectorPluginRec(x)->realloc_host != NULL) {
	data = SpVectorPluginRec(x)->realloc_host(x->plugin->instance, data, length * sizeof(short));
    } else if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	short *data2;
	
	data2 = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(short));
	memcpy(data2, data, sizeof(short) * x->length);
	xfreevecdata(x, data);
	
	data = data2;
    } else {
	data = xrealloc(data, length, short);
    }
#else
    data = xrealloc(data, length, short);
#endif

    return data;
}

spSVector xsvrealloc(spSVector x, long length)
{
    long k;
    
    if (x == NODATA) {
	x = xsvzeros(length);
    } else {
	if (length > x->length) {
	    x->data = xreallocsdata(x, x->data, length);
	    for (k = x->length; k < length; k++) {
		x->data[k] = 0;
	    }
	    if (x->imag != NULL) {
		x->imag = xreallocsdata(x, x->imag, length);
		for (k = x->length; k < length; k++) {
		    x->imag[k] = 0;
		}
	    }
#ifdef SP_USE_VECTOR_ENGINE
	    if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
		/* (*(SpVectorPluginRec(x)->realloc)) is required */
		(*(SpVectorPluginRec(x)->realloc))(x->instance, x->data, x->imag, x->length, length);
	    }
#endif
	}

	x->length = length;
    }

    return x;
}

long *xreallocldata(spLVector x, long *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL && SpVectorPluginRec(x)->realloc_host != NULL) {
	data = SpVectorPluginRec(x)->realloc_host(x->plugin->instance, data, length * sizeof(long));
    } else if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	long *data2;
	
	data2 = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(long));
	memcpy(data2, data, sizeof(long) * x->length);
	xfreevecdata(x, data);
	
	data = data2;
    } else {
	data = xrealloc(data, length, long);
    }
#else
    data = xrealloc(data, length, long);
#endif

    return data;
}

spLVector xlvrealloc(spLVector x, long length)
{
    long k;
    
    if (x == NODATA) {
	x = xlvzeros(length);
    } else {
	if (length > x->length) {
	    x->data = xreallocldata(x, x->data, length);
	    for (k = x->length; k < length; k++) {
		x->data[k] = 0;
	    }
	    if (x->imag != NULL) {
		x->imag = xreallocldata(x, x->imag, length);
		for (k = x->length; k < length; k++) {
		    x->imag[k] = 0;
		}
	    }
#ifdef SP_USE_VECTOR_ENGINE
	    if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
		/* (*(SpVectorPluginRec(x)->realloc)) is required */
		(*(SpVectorPluginRec(x)->realloc))(x->instance, x->data, x->imag, x->length, length);
	    }
#endif
	}
	x->length = length;
    }

    return x;
}

float *xreallocfdata(spFVector x, float *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL && SpVectorPluginRec(x)->realloc_host != NULL) {
	data = SpVectorPluginRec(x)->realloc_host(x->plugin->instance, data, length * sizeof(float));
    } else if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	float *data2;
	
	data2 = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(float));
	memcpy(data2, data, sizeof(float) * x->length);
	xfreevecdata(x, data);
	
	data = data2;
    } else {
	data = xrealloc(data, length, float);
    }
#else
    data = xrealloc(data, length, float);
#endif

    return data;
}

spFVector xfvrealloc(spFVector x, long length)
{
    long k;
    
    if (x == NODATA) {
	x = xfvzeros(length);
    } else {
	if (length > x->length) {
	    x->data = xreallocfdata(x, x->data, length);
	    for (k = x->length; k < length; k++) {
		x->data[k] = 0.0;
	    }
	    if (x->imag != NULL) {
		x->imag = xreallocfdata(x, x->imag, length);
		for (k = x->length; k < length; k++) {
		    x->imag[k] = 0.0;
		}
	    }
#ifdef SP_USE_VECTOR_ENGINE
	    if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
		/* (*(SpVectorPluginRec(x)->realloc)) is required */
		(*(SpVectorPluginRec(x)->realloc))(x->instance, x->data, x->imag, x->length, length);
	    }
#endif
	}
	x->length = length;
    }

    return x;
}

double *xreallocddata(spDVector x, double *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x->plugin != NULL && SpVectorPluginRec(x)->realloc_host != NULL) {
	data = SpVectorPluginRec(x)->realloc_host(x->plugin->instance, data, length * sizeof(double));
    } else if (x->plugin != NULL && SpVectorPluginRec(x)->alloc_host != NULL) {
	double *data2;
	
	data2 = SpVectorPluginRec(x)->alloc_host(x->plugin->instance, length * sizeof(double));
	memcpy(data2, data, sizeof(double) * x->length);
	xfreevecdata(x, data);
	
	data = data2;
    } else {
	data = xrealloc(data, length, double);
    }
#else
    data = xrealloc(data, length, double);
#endif

    return data;
}

spDVector xdvrealloc(spDVector x, long length)
{
    long k;
    
    if (x == NODATA) {
	x = xdvzeros(length);
    } else {
	if (length > x->length) {
	    x->data = xreallocddata(x, x->data, length);
	    for (k = x->length; k < length; k++) {
		x->data[k] = 0.0;
	    }
	    if (x->imag != NULL) {
		x->imag = xreallocddata(x, x->imag, length);
		for (k = x->length; k < length; k++) {
		    x->imag[k] = 0.0;
		}
	    }
#ifdef SP_USE_VECTOR_ENGINE
	    if (x->instance != NULL && SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
		/* (*(SpVectorPluginRec(x)->realloc)) is required */
		(*(SpVectorPluginRec(x)->realloc))(x->instance, x->data, x->imag, x->length, length);
	    }
#endif
	}
	x->length = length;
    }

    return x;
}

spBool svisplugincomputable(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->instance != NULL
        && (x->locked == SP_FALSE || SpVectorPluginRec(x)->rlock == NULL)) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_FALSE;
#endif
}

spBool lvisplugincomputable(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->instance != NULL
        && (x->locked == SP_FALSE || SpVectorPluginRec(x)->rlock == NULL)) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_FALSE;
#endif
}

spBool fvisplugincomputable(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->instance != NULL
        && (x->locked == SP_FALSE || SpVectorPluginRec(x)->rlock == NULL)) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_FALSE;
#endif
}

spBool dvisplugincomputable(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->instance != NULL
        && (x->locked == SP_FALSE || SpVectorPluginRec(x)->rlock == NULL)) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_FALSE;
#endif
}

spBool svislocked(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->locked) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_TRUE;
#endif
}

spBool lvislocked(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->locked) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_TRUE;
#endif
}

spBool fvislocked(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->locked) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_TRUE;
#endif
}

spBool dvislocked(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA && x->locked) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
#else
    return SP_TRUE;
#endif
}

spBool svlock(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_TRUE) return SP_FALSE;
    
    if (x->instance != NULL) {
	svlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_TRUE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool svunlock(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_FALSE) return SP_FALSE;
    
    if (x->instance != NULL) {
	svunlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_FALSE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool lvlock(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_TRUE) return SP_FALSE;
    
    if (x->instance != NULL) {
	lvlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_TRUE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool lvunlock(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_FALSE) return SP_FALSE;
    
    if (x->instance != NULL) {
	lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_FALSE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool fvlock(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_TRUE) return SP_FALSE;
    
    if (x->instance != NULL) {
	fvlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_TRUE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool fvunlock(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_FALSE) return SP_FALSE;
    
    if (x->instance != NULL) {
	fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_FALSE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool dvlock(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_TRUE) return SP_FALSE;
    
    if (x->instance != NULL) {
	dvlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_TRUE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool dvunlock(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->locked == SP_FALSE) return SP_FALSE;
    
    if (x->instance != NULL) {
	dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_NORMAL);
	x->locked = SP_FALSE;
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool svsetlock(spSVector x, spBool lock)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	if (lock == SP_FALSE && x->instance != NULL) {
	    if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
		SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
	    }
	    if (x->imag != NULL) {
		if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
		    SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
		}
	    }
	}
#endif
	x->locked = lock;
	return SP_TRUE;
    }
#endif
    return SP_FALSE;
}

spBool lvsetlock(spLVector x, spBool lock)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	if (lock == SP_FALSE && x->instance != NULL) {
	    if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
		SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
	    }
	    if (x->imag != NULL) {
		if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
		    SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
		}
	    }
	}
#endif
	x->locked = lock;
	return SP_TRUE;
    }
#endif
    return SP_FALSE;
}

spBool fvsetlock(spFVector x, spBool lock)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	if (lock == SP_FALSE && x->instance != NULL) {
	    if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
		SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
	    }
	    if (x->imag != NULL) {
		if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
		    SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
		}
	    }
	}
#endif
	x->locked = lock;
	return SP_TRUE;
    }
#endif
    return SP_FALSE;
}

spBool dvsetlock(spDVector x, spBool lock)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x != NODATA) {
#if defined(SP_VECTOR_ENGINE_ALLOC_IN_UNLOCK)
	if (lock == SP_FALSE && x->instance != NULL) {
	    if (SpVectorPluginRec(x)->getrptr(x->instance) == NULL) {
		SpVectorPluginRec(x)->ralloc(x->instance, x->data, x->length);
	    }
	    if (x->imag != NULL) {
		if (SpVectorPluginRec(x)->getiptr(x->instance) == NULL) {
		    SpVectorPluginRec(x)->ialloc(x->instance, x->imag, x->length);
		}
	    }
	}
#endif
	x->locked = lock;
	return SP_TRUE;
    }
#endif
    return SP_FALSE;
}

spBool svsync(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->instance == NULL) return SP_FALSE;
    
    if (x->locked == SP_TRUE) {
	return svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    } else {
	return svlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#else
    return SP_FALSE;
#endif
}

spBool lvsync(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->instance == NULL) return SP_FALSE;
    
    if (x->locked == SP_TRUE) {
	return lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    } else {
	return lvlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#else
    return SP_FALSE;
#endif
}

spBool fvsync(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->instance == NULL) return SP_FALSE;
    
    if (x->locked == SP_TRUE) {
	return fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    } else {
	return fvlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#else
    return SP_FALSE;
#endif
}

spBool dvsync(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (x == NODATA || x->instance == NULL) return SP_FALSE;
    
    if (x->locked == SP_TRUE) {
	return dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    } else {
	return dvlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#else
    return SP_FALSE;
#endif
}

void *svgetrptr(spSVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    spDebug(100, "svgetrptr", "locked = %d\n", x->locked);
    
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getrptr(x->instance);
    }
#endif
    
    return x->data;
}

void *lvgetrptr(spLVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    spDebug(100, "lvgetrptr", "locked = %d\n", x->locked);
    
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getrptr(x->instance);
    }
#endif
    
    return x->data;
}

void *fvgetrptr(spFVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    spDebug(100, "fvgetrptr", "locked = %d\n", x->locked);
    
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getrptr(x->instance);
    }
#endif
    
    return x->data;
}

void *dvgetrptr(spDVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    spDebug(100, "dvgetrptr", "locked = %d\n", x->locked);
    
    if (x->instance != NULL && x->locked == SP_FALSE) {
	void *ptr;
	ptr = SpVectorPluginRec(x)->getrptr(x->instance);
	spDebug(100, "dvgetrptr", "ptr = %ld\n", (long)ptr);
	return ptr;
    }
#endif
    
    return x->data;
}

void *svgetiptr(spSVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getiptr(x->instance);
    }
#endif
    
    return x->imag;
}

void *lvgetiptr(spLVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getiptr(x->instance);
    }
#endif
    
    return x->imag;
}

void *fvgetiptr(spFVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getiptr(x->instance);
    }
#endif
    
    return x->imag;
}

void *dvgetiptr(spDVector x)
{
    if (x == NODATA) return NULL;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL && x->locked == SP_FALSE) {
	return SpVectorPluginRec(x)->getiptr(x->instance);
    }
#endif
    
    return x->imag;
}

spSVectors xsvsalloc(long num)
{
    long k;
    spSVectors xs;

    xs = xalloc(1, struct _spSVectors);
    xs->vector = xalloc(MAX(num, 1), spSVector);
    xs->num_vector = num;
    
    for (k = 0; k < xs->num_vector; k++) {
	xs->vector[k] = NODATA;
    }

    return xs;
}

spLVectors xlvsalloc(long num)
{
    long k;
    spLVectors xs;

    xs = xalloc(1, struct _spLVectors);
    xs->vector = xalloc(MAX(num, 1), spLVector);
    xs->num_vector = num;
    
    for (k = 0; k < xs->num_vector; k++) {
	xs->vector[k] = NODATA;
    }

    return xs;
}

spFVectors xfvsalloc(long num)
{
    long k;
    spFVectors xs;

    xs = xalloc(1, struct _spFVectors);
    xs->vector = xalloc(MAX(num, 1), spFVector);
    xs->num_vector = num;
    
    for (k = 0; k < xs->num_vector; k++) {
	xs->vector[k] = NODATA;
    }

    return xs;
}

spDVectors xdvsalloc(long num)
{
    long k;
    spDVectors xs;

    xs = xalloc(1, struct _spDVectors);
    xs->vector = xalloc(MAX(num, 1), spDVector);
    xs->num_vector = num;
    
    for (k = 0; k < xs->num_vector; k++) {
	xs->vector[k] = NODATA;
    }

    return xs;
}

spSVectors xsvsrealloc(spSVectors xs, long num)
{
    long k;

    if (xs == NULL) return xsvsalloc(num);

    if (num > xs->num_vector) {
	xs->vector = xrealloc(xs->vector, num, spSVector);
    
	for (k = xs->num_vector; k < num; k++) {
	    xs->vector[k] = NODATA;
	}
	xs->num_vector = num;
    }

    return xs;
}

spLVectors xlvsrealloc(spLVectors xs, long num)
{
    long k;

    if (xs == NULL) return xlvsalloc(num);

    if (num > xs->num_vector) {
	xs->vector = xrealloc(xs->vector, num, spLVector);
    
	for (k = xs->num_vector; k < num; k++) {
	    xs->vector[k] = NODATA;
	}
	xs->num_vector = num;
    }

    return xs;
}

spFVectors xfvsrealloc(spFVectors xs, long num)
{
    long k;

    if (xs == NULL) return xfvsalloc(num);

    if (num > xs->num_vector) {
	xs->vector = xrealloc(xs->vector, num, spFVector);
    
	for (k = xs->num_vector; k < num; k++) {
	    xs->vector[k] = NODATA;
	}
	xs->num_vector = num;
    }

    return xs;
}

spDVectors xdvsrealloc(spDVectors xs, long num)
{
    long k;

    if (xs == NULL) return xdvsalloc(num);

    if (num > xs->num_vector) {
	xs->vector = xrealloc(xs->vector, num, spDVector);
    
	for (k = xs->num_vector; k < num; k++) {
	    xs->vector[k] = NODATA;
	}
	xs->num_vector = num;
    }

    return xs;
}

void xsvsfree(spSVectors xs)
{
    long k;

    if (xs != NULL) {
	if (xs->vector != NULL) {
	    for (k = 0; k < xs->num_vector; k++) {
		if (xs->vector[k] != NODATA) {
		    xsvfree(xs->vector[k]);
		}
	    }
	    xfree(xs->vector);
	}
	xfree(xs);
    }

    return;
}

void xlvsfree(spLVectors xs)
{
    long k;

    if (xs != NULL) {
	if (xs->vector != NULL) {
	    for (k = 0; k < xs->num_vector; k++) {
		if (xs->vector[k] != NODATA) {
		    xlvfree(xs->vector[k]);
		}
	    }
	    xfree(xs->vector);
	}
	xfree(xs);
    }

    return;
}

void xfvsfree(spFVectors xs)
{
    long k;

    if (xs != NULL) {
	if (xs->vector != NULL) {
	    for (k = 0; k < xs->num_vector; k++) {
		if (xs->vector[k] != NODATA) {
		    xfvfree(xs->vector[k]);
		}
	    }
	    xfree(xs->vector);
	}
	xfree(xs);
    }

    return;
}

void xdvsfree(spDVectors xs)
{
    long k;

    if (xs != NULL) {
	if (xs->vector != NULL) {
	    for (k = 0; k < xs->num_vector; k++) {
		if (xs->vector[k] != NODATA) {
		    xdvfree(xs->vector[k]);
		}
	    }
	    xfree(xs->vector);
	}
	xfree(xs);
    }

    return;
}

void svscopy(spSVectors ys, spSVectors xs)
{
    long k;
    long num_vector;

    num_vector = MIN(ys->num_vector, xs->num_vector);

    for (k = 0; k < num_vector; k++) {
        svcopy(ys->vector[k], xs->vector[k]);
    }

    return;
}

void lvscopy(spLVectors ys, spLVectors xs)
{
    long k;
    long num_vector;

    num_vector = MIN(ys->num_vector, xs->num_vector);

    for (k = 0; k < num_vector; k++) {
        lvcopy(ys->vector[k], xs->vector[k]);
    }

    return;
}

void fvscopy(spFVectors ys, spFVectors xs)
{
    long k;
    long num_vector;

    num_vector = MIN(ys->num_vector, xs->num_vector);

    for (k = 0; k < num_vector; k++) {
        fvcopy(ys->vector[k], xs->vector[k]);
    }

    return;
}

void dvscopy(spDVectors ys, spDVectors xs)
{
    long k;
    long num_vector;

    num_vector = MIN(ys->num_vector, xs->num_vector);

    for (k = 0; k < num_vector; k++) {
        dvcopy(ys->vector[k], xs->vector[k]);
    }

    return;
}

spSVectors xsvsclone(spSVectors xs)
{
    long k;
    spSVectors ys;

    ys = xsvsalloc(xs->num_vector);

    for (k = 0; k < xs->num_vector; k++) {
        ys->vector[k] = xsvclone(xs->vector[k]);
    }

    return ys;
}

spLVectors xlvsclone(spLVectors xs)
{
    long k;
    spLVectors ys;

    ys = xlvsalloc(xs->num_vector);

    for (k = 0; k < xs->num_vector; k++) {
        ys->vector[k] = xlvclone(xs->vector[k]);
    }

    return ys;
}

spFVectors xfvsclone(spFVectors xs)
{
    long k;
    spFVectors ys;

    ys = xfvsalloc(xs->num_vector);

    for (k = 0; k < xs->num_vector; k++) {
        ys->vector[k] = xfvclone(xs->vector[k]);
    }

    return ys;
}

spDVectors xdvsclone(spDVectors xs)
{
    long k;
    spDVectors ys;

    ys = xdvsalloc(xs->num_vector);

    for (k = 0; k < xs->num_vector; k++) {
        ys->vector[k] = xdvclone(xs->vector[k]);
    }

    return ys;
}

void svsshift(spSVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    if (xs->vector[k] != NODATA) {
		xsvfree(xs->vector[k]);
	    }
	    if (k >= shift) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    if (xs->vector[k] != NODATA) {
		xsvfree(xs->vector[k]);
	    }
	    if (k < len) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    }
    
    return;
}

void lvsshift(spLVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    if (xs->vector[k] != NODATA) {
		xlvfree(xs->vector[k]);
	    }
	    if (k >= shift) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    if (xs->vector[k] != NODATA) {
		xlvfree(xs->vector[k]);
	    }
	    if (k < len) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    }
    
    return;
}

void fvsshift(spFVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    if (xs->vector[k] != NODATA) {
		xfvfree(xs->vector[k]);
	    }
	    if (k >= shift) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    if (xs->vector[k] != NODATA) {
		xfvfree(xs->vector[k]);
	    }
	    if (k < len) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    }
    
    return;
}

void dvsshift(spDVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    if (xs->vector[k] != NODATA) {
		xdvfree(xs->vector[k]);
	    }
	    if (k >= shift) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    if (xs->vector[k] != NODATA) {
		xdvfree(xs->vector[k]);
	    }
	    if (k < len) {
		xs->vector[k] = xs->vector[k - shift];
		xs->vector[k - shift] = NODATA;
	    } else {
		xs->vector[k] = NODATA;
	    }
	}
    }
    
    return;
}

void svsmemshift(spSVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    svzeros(xs->vector[k], xs->vector[k]->length);
	    if (k >= shift) {
		svcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    svzeros(xs->vector[k], xs->vector[k]->length);
	    if (k < len) {
		svcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    }
    
    return;
}

void lvsmemshift(spLVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    lvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k >= shift) {
		lvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    lvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k < len) {
		lvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    }
    
    return;
}

void fvsmemshift(spFVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    fvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k >= shift) {
		fvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    fvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k < len) {
		fvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    }
    
    return;
}

void dvsmemshift(spDVectors xs, long shift)
{
    long k, len;
    
    if (shift > 0) {
	for (k = xs->num_vector - 1; k >= 0; k--) {
	    dvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k >= shift) {
		dvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    } else {
	len = xs->num_vector + shift;
	for (k = 0; k < xs->num_vector; k++) {
	    dvzeros(xs->vector[k], xs->vector[k]->length);
	    if (k < len) {
		dvcopy(xs->vector[k], xs->vector[k - shift]);
	    }
	}
    }
    
    return;
}

spSVector xsvcplx(spSVector xr, spSVector xi)
{
    long k;
    spSVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (xr != NODATA && xi != NODATA) {
	z = xsvrialloc(MIN(xr->length, xi->length));
    } else if (xr != NODATA) {
	z = xsvrialloc(xr->length);
    } else if (xi != NODATA) {
	z = xsvrialloc(xi->length);
    } else {
#if 0
	z = xsvnull();
	return z;
#else
	return NODATA;
#endif
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(xr) == SP_FALSE && svislocked(xi) == SP_FALSE) {
        svsetlock(z, SP_FALSE);
    }
    
    if (z->instance != NULL && (flist = SpGetSVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->rcopyi != NULL) {
        if ((xr != NODATA && svisplugincomputable(xr) == SP_TRUE)
            || (xi != NODATA && svisplugincomputable(xi) == SP_TRUE)) {
            if (svislocked(z) == SP_TRUE) {
                if (svislocked(xr) == SP_TRUE) {
                    svsync(xr);
                }
                if (svislocked(xi) == SP_TRUE) {
                    svsync(xi);
                }
            }
	
            if (xr != NODATA) flist->rcopy(z->instance, xr->instance, z->length);
            if (xi != NODATA) flist->rcopyi(z->instance, xi->instance, z->length);
	
            if (svisplugincomputable(z) == SP_TRUE) {
                return z;
            }
        }
    }

    if (xr != NODATA && svislocked(xr) == SP_FALSE) {
        svsync(xr);
    }
    if (xi != NODATA && svislocked(xi) == SP_FALSE) {
        svsync(xi);
    }
#endif
    
#if 0
    for (k = 0; k < z->length; k++) {
	if (xr != NODATA) {
	    z->data[k] = xr->data[k];
	} else {
	    z->data[k] = 0;
	}
	if (xi != NODATA) {
	    z->imag[k] = xi->data[k];
	} else {
	    z->imag[k] = 0;
	}
    }
#else
    if (xr != NODATA) {
        memcpy(z->data, xr->data, sizeof(short) * z->length);
    } else {
        memset(z->data, 0, sizeof(short) * z->length);
    }
    if (xi != NODATA) {
        memcpy(z->imag, xi->data, sizeof(short) * z->length);
    } else {
        memset(z->imag, 0, sizeof(short) * z->length);
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(z) == SP_FALSE) {
        svunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

spLVector xlvcplx(spLVector xr, spLVector xi)
{
    long k;
    spLVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (xr != NODATA && xi != NODATA) {
	z = xlvrialloc(MIN(xr->length, xi->length));
    } else if (xr != NODATA) {
	z = xlvrialloc(xr->length);
    } else if (xi != NODATA) {
	z = xlvrialloc(xi->length);
    } else {
#if 0
	z = xlvnull();
	return z;
#else
	return NODATA;
#endif
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(xr) == SP_FALSE && lvislocked(xi) == SP_FALSE) {
        lvsetlock(z, SP_FALSE);
    }
    
    if (z->instance != NULL && (flist = SpGetLVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->rcopyi != NULL) {
        if ((xr != NODATA && lvisplugincomputable(xr) == SP_TRUE)
            || (xi != NODATA && lvisplugincomputable(xi) == SP_TRUE)) {
            if (lvislocked(z) == SP_TRUE) {
                if (lvislocked(xr) == SP_TRUE) {
                    lvsync(xr);
                }
                if (lvislocked(xi) == SP_TRUE) {
                    lvsync(xi);
                }
            }
	
            if (xr != NODATA) flist->rcopy(z->instance, xr->instance, z->length);
            if (xi != NODATA) flist->rcopyi(z->instance, xi->instance, z->length);
	
            if (lvisplugincomputable(z) == SP_TRUE) {
                return z;
            }
        }
    }

    if (xr != NODATA && lvislocked(xr) == SP_FALSE) {
        lvsync(xr);
    }
    if (xi != NODATA && lvislocked(xi) == SP_FALSE) {
        lvsync(xi);
    }
#endif
    
#if 0
    for (k = 0; k < z->length; k++) {
	if (xr != NODATA) {
	    z->data[k] = xr->data[k];
	} else {
	    z->data[k] = 0;
	}
	if (xi != NODATA) {
	    z->imag[k] = xi->data[k];
	} else {
	    z->imag[k] = 0;
	}
    }
#else
    if (xr != NODATA) {
        memcpy(z->data, xr->data, sizeof(long) * z->length);
    } else {
        memset(z->data, 0, sizeof(long) * z->length);
    }
    if (xi != NODATA) {
        memcpy(z->imag, xi->data, sizeof(long) * z->length);
    } else {
        memset(z->imag, 0, sizeof(long) * z->length);
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(z) == SP_FALSE) {
        lvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

spFVector xfvcplx(spFVector xr, spFVector xi)
{
    long k;
    spFVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (xr != NODATA && xi != NODATA) {
	z = xfvrialloc(MIN(xr->length, xi->length));
    } else if (xr != NODATA) {
	z = xfvrialloc(xr->length);
    } else if (xi != NODATA) {
	z = xfvrialloc(xi->length);
    } else {
#if 0
	z = xfvnull();
	return z;
#else
	return NODATA;
#endif
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(xr) == SP_FALSE && fvislocked(xi) == SP_FALSE) {
        fvsetlock(z, SP_FALSE);
    }
    
    if (z->instance != NULL && (flist = SpGetFVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->rcopyi != NULL) {
        if ((xr != NODATA && fvisplugincomputable(xr) == SP_TRUE)
            || (xi != NODATA && fvisplugincomputable(xi) == SP_TRUE)) {
            if (fvislocked(z) == SP_TRUE) {
                if (fvislocked(xr) == SP_TRUE) {
                    fvsync(xr);
                }
                if (fvislocked(xi) == SP_TRUE) {
                    fvsync(xi);
                }
            }
	
            if (xr != NODATA) flist->rcopy(z->instance, xr->instance, z->length);
            if (xi != NODATA) flist->rcopyi(z->instance, xi->instance, z->length);
	
            if (fvisplugincomputable(z) == SP_TRUE) {
                return z;
            }
        }
    }

    if (xr != NODATA && fvislocked(xr) == SP_FALSE) {
        fvsync(xr);
    }
    if (xi != NODATA && fvislocked(xi) == SP_FALSE) {
        fvsync(xi);
    }
#endif

#if 0
    for (k = 0; k < z->length; k++) {
	if (xr != NODATA) {
	    z->data[k] = xr->data[k];
	} else {
	    z->data[k] = 0.0;
	}
	if (xi != NODATA) {
	    z->imag[k] = xi->data[k];
	} else {
	    z->imag[k] = 0.0;
	}
    }
#else
    if (xr != NODATA) {
        memcpy(z->data, xr->data, sizeof(float) * z->length);
    } else {
        memset(z->data, 0, sizeof(float) * z->length);
    }
    if (xi != NODATA) {
        memcpy(z->imag, xi->data, sizeof(float) * z->length);
    } else {
        memset(z->imag, 0, sizeof(float) * z->length);
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(z) == SP_FALSE) {
        fvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

spDVector xdvcplx(spDVector xr, spDVector xi)
{
    long k;
    spDVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (xr != NODATA && xi != NODATA) {
	z = xdvrialloc(MIN(xr->length, xi->length));
    } else if (xr != NODATA) {
	z = xdvrialloc(xr->length);
    } else if (xi != NODATA) {
	z = xdvrialloc(xi->length);
    } else {
#if 0
	z = xdvnull();
	return z;
#else
	return NODATA;
#endif
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(xr) == SP_FALSE && dvislocked(xi) == SP_FALSE) {
        dvsetlock(z, SP_FALSE);
    }
    
    if (z->instance != NULL && (flist = SpGetDVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->rcopyi != NULL) {
        if ((xr != NODATA && dvisplugincomputable(xr) == SP_TRUE)
            || (xi != NODATA && dvisplugincomputable(xi) == SP_TRUE)) {
            if (dvislocked(z) == SP_TRUE) {
                if (dvislocked(xr) == SP_TRUE) {
                    dvsync(xr);
                }
                if (dvislocked(xi) == SP_TRUE) {
                    dvsync(xi);
                }
            }
	
            if (xr != NODATA) flist->rcopy(z->instance, xr->instance, z->length);
            if (xi != NODATA) flist->rcopyi(z->instance, xi->instance, z->length);
	
            if (dvisplugincomputable(z) == SP_TRUE) {
                return z;
            }
        }
    }

    if (xr != NODATA && dvislocked(xr) == SP_FALSE) {
        dvsync(xr);
    }
    if (xi != NODATA && dvislocked(xi) == SP_FALSE) {
        dvsync(xi);
    }
#endif

#if 0
    for (k = 0; k < z->length; k++) {
	if (xr != NODATA) {
	    z->data[k] = xr->data[k];
	} else {
	    z->data[k] = 0.0;
	}
	if (xi != NODATA) {
	    z->imag[k] = xi->data[k];
	} else {
	    z->imag[k] = 0.0;
	}
    }
#else
    if (xr != NODATA) {
        memcpy(z->data, xr->data, sizeof(double) * z->length);
    } else {
        memset(z->data, 0, sizeof(double) * z->length);
    }
    if (xi != NODATA) {
        memcpy(z->imag, xi->data, sizeof(double) * z->length);
    } else {
        memset(z->imag, 0, sizeof(double) * z->length);
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(z) == SP_FALSE) {
        dvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

void svreal(spSVector x)
{
    if (x->imag != NULL) {
	svifree(x);
    }

    return;
} 

void lvreal(spLVector x)
{
    if (x->imag != NULL) {
	lvifree(x);
    }

    return;
} 

void fvreal(spFVector x)
{
    if (x->imag != NULL) {
	fvifree(x);
    }

    return;
} 

void dvreal(spDVector x)
{
    if (x->imag != NULL) {
	dvifree(x);
    }

    return;
} 

void svimag(spSVector x)
{
    if (x->imag == NULL) {
	svzeros(x, x->length);
    } else {
        xfreevecdata(x, x->data);
        x->data = x->imag;
        x->imag = NULL;
        
#ifdef SP_USE_VECTOR_ENGINE
        if (x->instance != NULL) {
            SpVectorPluginRec(x)->imag(x->instance, x->data, x->length);
        }
#endif
    }

    return;
} 

void lvimag(spLVector x)
{
    if (x->imag == NULL) {
	lvzeros(x, x->length);
    } else {
        xfreevecdata(x, x->data);
        x->data = x->imag;
        x->imag = NULL;
        
#ifdef SP_USE_VECTOR_ENGINE
        if (x->instance != NULL) {
            SpVectorPluginRec(x)->imag(x->instance, x->data, x->length);
        }
#endif
    }

    return;
} 

void fvimag(spFVector x)
{
    if (x->imag == NULL) {
	fvzeros(x, x->length);
    } else {
        xfreevecdata(x, x->data);
        x->data = x->imag;
        x->imag = NULL;
        
#ifdef SP_USE_VECTOR_ENGINE
        if (x->instance != NULL) {
            SpVectorPluginRec(x)->imag(x->instance, x->data, x->length);
        }
#endif
    }

    return;
} 

void dvimag(spDVector x)
{
    if (x->imag == NULL) {
	dvzeros(x, x->length);
    } else {
        xfreevecdata(x, x->data);
        x->data = x->imag;
        x->imag = NULL;

#ifdef SP_USE_VECTOR_ENGINE
        if (x->instance != NULL) {
            SpVectorPluginRec(x)->imag(x->instance, x->data, x->length);
        }
#endif
    }
    
    return;
} 

spSVector xsvreal(spSVector x)
{
    spSVector y;

    y = xsvalloculfromv(x, x->length);

    /* copy data */
    svcopy(y, x);

    return y;
} 

spLVector xlvreal(spLVector x)
{
    spLVector y;

    y = xlvalloculfromv(x, x->length);

    /* copy data */
    lvcopy(y, x);

    return y;
} 

spFVector xfvreal(spFVector x)
{
    spFVector y;

    y = xfvalloculfromv(x, x->length);

    /* copy data */
    fvcopy(y, x);

    return y;
} 

spDVector xdvreal(spDVector x)
{
    spDVector y;

    y = xdvalloculfromv(x, x->length);

    /* copy data */
    dvcopy(y, x);

    return y;
} 

spSVector xsvimag(spSVector x)
{
    spSVector y;

    if (x->imag == NULL) {
	y = xsvzeros(x->length);

	return y;
    }
    y = xsvalloculfromv(x, x->length);

    /* copy data */
    svicopyr(y, x);

    return y;
} 

spLVector xlvimag(spLVector x)
{
    spLVector y;

    if (x->imag == NULL) {
	y = xlvzeros(x->length);

	return y;
    }
    y = xlvalloculfromv(x, x->length);

    /* copy data */
    lvicopyr(y, x);

    return y;
} 

spFVector xfvimag(spFVector x)
{
    spFVector y;

    if (x->imag == NULL) {
	y = xfvzeros(x->length);

	return y;
    }
    y = xfvalloculfromv(x, x->length);

    /* copy data */
    fvicopyr(y, x);

    return y;
} 

spDVector xdvimag(spDVector x)
{
    spDVector y;

    if (x->imag == NULL) {
	y = xdvzeros(x->length);

	return y;
    }
    y = xdvalloculfromv(x, x->length);

    /* copy data */
    dvicopyr(y, x);

    return y;
} 

void svconj(spSVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->conj != NULL) {
	flist->conj(x->instance, x->length);
	return;
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->imag[k] = -x->imag[k];
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvconj(spLVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->conj != NULL) {
	flist->conj(x->instance, x->length);
	return;
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->imag[k] = -x->imag[k];
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvconj(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->conj != NULL) {
	flist->conj(x->instance, x->length);
	return;
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->imag[k] = -x->imag[k];
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvconj(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->conj != NULL) {
	flist->conj(x->instance, x->length);
	return;
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->imag[k] = -x->imag[k];
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spSVector xsvconj(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svconj(y);

    return y;
}

spLVector xlvconj(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvconj(y);

    return y;
}

spFVector xfvconj(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvconj(y);

    return y;
}

spDVector xdvconj(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvconj(y);

    return y;
}

void svriswap(spSVector x)
{
    short *p;

    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL) {
        SpVectorPluginRec(x)->riswap(x->instance, x->imag, x->data, x->length);
    }
#endif

    p = x->data;
    x->data = x->imag;
    x->imag = p;

    return;
}

void lvriswap(spLVector x)
{
    long *p;

    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL) {
         SpVectorPluginRec(x)->riswap(x->instance, x->imag, x->data, x->length);
    }
#endif

    p = x->data;
    x->data = x->imag;
    x->imag = p;

    return;
}

void fvriswap(spFVector x)
{
    float *p;

    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL) {
         SpVectorPluginRec(x)->riswap(x->instance, x->imag, x->data, x->length);
    }
#endif

    p = x->data;
    x->data = x->imag;
    x->imag = p;

    return;
}

void dvriswap(spDVector x)
{
    double *p;

    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
#ifdef SP_USE_VECTOR_ENGINE
    if (x->instance != NULL) {
         SpVectorPluginRec(x)->riswap(x->instance, x->imag, x->data, x->length);
    }
#endif

    p = x->data;
    x->data = x->imag;
    x->imag = p;

    return;
}

spSVector xsvriswap(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svriswap(y);

    return y;
}

spLVector xlvriswap(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvriswap(y);

    return y;
}

spFVector xfvriswap(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvriswap(y);

    return y;
}

spDVector xdvriswap(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvriswap(y);

    return y;
}

/*
 *	copy data of x into y
 */
void svcopy(spSVector y, spSVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL && flist->icopy != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (x->imag != NULL && y->imag != NULL) {
	    flist->icopy(y->instance, x->instance, length);
	}
	if (svisplugincomputable(y) == SP_TRUE && svisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    /* copy data */
    memcpy(y->data, x->data, sizeof(short) * length);
    
    if (x->imag != NULL && y->imag != NULL) {
	memcpy(y->imag, x->imag, sizeof(short) * length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvcopy(spLVector y, spLVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL && flist->icopy != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (x->imag != NULL && y->imag != NULL) {
	    flist->icopy(y->instance, x->instance, length);
	}
	if (lvisplugincomputable(y) == SP_TRUE && lvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(y->data, x->data, sizeof(long) * length);
    
    if (x->imag != NULL && y->imag != NULL) {
	memcpy(y->imag, x->imag, sizeof(long) * length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvcopy(spFVector y, spFVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL && flist->icopy != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (x->imag != NULL && y->imag != NULL) {
	    flist->icopy(y->instance, x->instance, length);
	}
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(y->data, x->data, sizeof(float) * length);
    
    if (x->imag != NULL && y->imag != NULL) {
	memcpy(y->imag, x->imag, sizeof(float) * length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvcopy(spDVector y, spDVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL && flist->icopy != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (x->imag != NULL && y->imag != NULL) {
	    flist->icopy(y->instance, x->instance, length);
	}
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(y->data, x->data, sizeof(double) * length);
    
    if (x->imag != NULL && y->imag != NULL) {
	memcpy(y->imag, x->imag, sizeof(double) * length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svrcopyi(spSVector y, spSVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (y->imag == NULL) {
	svialloc(y);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rcopyi != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	flist->rcopyi(y->instance, x->instance, length);
	if (svisplugincomputable(y) == SP_TRUE && svisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->data, sizeof(short) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvrcopyi(spLVector y, spLVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (y->imag == NULL) {
	lvialloc(y);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rcopyi != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	flist->rcopyi(y->instance, x->instance, length);
	if (lvisplugincomputable(y) == SP_TRUE && lvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->data, sizeof(long) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvrcopyi(spFVector y, spFVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (y->imag == NULL) {
	fvialloc(y);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rcopyi != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	flist->rcopyi(y->instance, x->instance, length);
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->data, sizeof(float) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvrcopyi(spDVector y, spDVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (y->imag == NULL) {
	dvialloc(y);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rcopyi != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	flist->rcopyi(y->instance, x->instance, length);
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->data, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svicopyr(spSVector y, spSVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	svzeros(y, length);
	return;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->icopyr != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	flist->icopyr(y->instance, x->instance, length);
	if (svisplugincomputable(y) == SP_TRUE && svisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->imag, sizeof(short) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvicopyr(spLVector y, spLVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	lvzeros(y, length);
	return;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->icopyr != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	flist->icopyr(y->instance, x->instance, length);
	if (lvisplugincomputable(y) == SP_TRUE && lvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->imag, sizeof(long) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvicopyr(spFVector y, spFVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	fvzeros(y, length);
	return;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->icopyr != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	flist->icopyr(y->instance, x->instance, length);
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->imag, sizeof(float) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvicopyr(spDVector y, spDVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	dvzeros(y, length);
	return;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->icopyr != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	flist->icopyr(y->instance, x->instance, length);
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->imag, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svrcopyr(spSVector y, spSVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (svisplugincomputable(y) == SP_TRUE && svisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->data, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvrcopyr(spLVector y, spLVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (lvisplugincomputable(y) == SP_TRUE && lvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->data, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvrcopyr(spFVector y, spFVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->data, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvrcopyr(spDVector y, spDVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rcopy != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	flist->rcopy(y->instance, x->instance, length);
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->data, x->data, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svicopyi(spSVector y, spSVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	svizeros(y, length);
        return;
    } else if (y->imag == NULL) {
        svizeros(y, length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->icopy != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	flist->icopy(y->instance, x->instance, length);
	if (svisplugincomputable(y) == SP_TRUE && svisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->imag, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvicopyi(spLVector y, spLVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	lvizeros(y, length);
        return;
    } else if (y->imag == NULL) {
        lvizeros(y, length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->icopy != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	flist->icopy(y->instance, x->instance, length);
	if (lvisplugincomputable(y) == SP_TRUE && lvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->imag, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvicopyi(spFVector y, spFVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	fvizeros(y, length);
        return;
    } else if (y->imag == NULL) {
        fvizeros(y, length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->icopy != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	flist->icopy(y->instance, x->instance, length);
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->imag, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvicopyi(spDVector y, spDVector x)
{
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    length = MIN(y->length, x->length);

    if (x->imag == NULL) {
	dvizeros(y, length);
        return;
    } else if (y->imag == NULL) {
        dvizeros(y, length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->icopy != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	flist->icopy(y->instance, x->instance, length);
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    /* copy data */
    memcpy(y->imag, x->imag, sizeof(double) * length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

long svcopyfromarray(spSVector y, long y_offset, short *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = svislocked(y);
    
    if (locked == SP_FALSE) {
        svsync(y);
    }
#endif
    
    /* copy data */
    memcpy(y->data + y_offset, data, sizeof(short) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long lvcopyfromarray(spLVector y, long y_offset, long *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = lvislocked(y);
    
    if (locked == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    /* copy data */
    memcpy(y->data + y_offset, data, sizeof(long) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long fvcopyfromarray(spFVector y, long y_offset, float *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = fvislocked(y);
    
    if (locked == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    /* copy data */
    memcpy(y->data + y_offset, data, sizeof(float) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long dvcopyfromarray(spDVector y, long y_offset, double *data, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = dvislocked(y);
    
    if (locked == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    /* copy data */
    memcpy(y->data + y_offset, data, sizeof(double) * length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long svcopyfromriarray(spSVector y, long y_offset, short *data, short *imag, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = svislocked(y);
    
    if (locked == SP_FALSE) {
        svsync(y);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(y->data + y_offset, data, sizeof(short) * length);
    }
    if (imag != NULL) {
        if (y->imag == NULL) {
            svialloc(y);
        }
        memcpy(y->imag + y_offset, imag, sizeof(short) * length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long lvcopyfromriarray(spLVector y, long y_offset, long *data, long *imag, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = lvislocked(y);
    
    if (locked == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(y->data + y_offset, data, sizeof(long) * length);
    }
    if (imag != NULL) {
        if (y->imag == NULL) {
            lvialloc(y);
        }
        memcpy(y->imag + y_offset, imag, sizeof(long) * length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long fvcopyfromriarray(spFVector y, long y_offset, float *data, float *imag, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = fvislocked(y);
    
    if (locked == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(y->data + y_offset, data, sizeof(float) * length);
    }
    if (imag != NULL) {
        if (y->imag == NULL) {
            fvialloc(y);
        }
        memcpy(y->imag + y_offset, imag, sizeof(float) * length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long dvcopyfromriarray(spDVector y, long y_offset, double *data, double *imag, long length)
{
#ifdef SP_USE_VECTOR_ENGINE
    spBool locked;
#endif

    length = MIN(y->length - y_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    locked = dvislocked(y);
    
    if (locked == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(y->data + y_offset, data, sizeof(double) * length);
    }
    if (imag != NULL) {
        if (y->imag == NULL) {
            dvialloc(y);
        }
        memcpy(y->imag + y_offset, imag, sizeof(double) * length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (locked == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return length;
}

long svcopytoarray(spSVector x, long x_offset, short *data, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    /* copy data */
    memcpy(data, x->data + x_offset, sizeof(short) * length);
    
    return length;
}

long lvcopytoarray(spLVector x, long x_offset, long *data, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(data, x->data + x_offset, sizeof(long) * length);
    
    return length;
}

long fvcopytoarray(spFVector x, long x_offset, float *data, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(data, x->data + x_offset, sizeof(float) * length);
    
    return length;
}

long dvcopytoarray(spDVector x, long x_offset, double *data, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    /* copy data */
    memcpy(data, x->data + x_offset, sizeof(double) * length);
    
    return length;
}

long svcopytoriarray(spSVector x, long x_offset, short *data, short *imag, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(data, x->data + x_offset, sizeof(short) * length);
    }
    if (imag != NULL) {
        if (x->imag == NULL) {
            memset(data, 0, sizeof(short) * length);
        } else {
            memcpy(imag, x->imag + x_offset, sizeof(short) * length);
        }
    }
    
    return length;
}

long lvcopytoriarray(spLVector x, long x_offset, long *data, long *imag, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(data, x->data + x_offset, sizeof(long) * length);
    }
    if (imag != NULL) {
        if (x->imag == NULL) {
            memset(data, 0, sizeof(long) * length);
        } else {
            memcpy(imag, x->imag + x_offset, sizeof(long) * length);
        }
    }
    
    return length;
}

long fvcopytoriarray(spFVector x, long x_offset, float *data, float *imag, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(data, x->data + x_offset, sizeof(float) * length);
    }
    if (imag != NULL) {
        if (x->imag == NULL) {
            memset(data, 0, sizeof(float) * length);
        } else {
            memcpy(imag, x->imag + x_offset, sizeof(float) * length);
        }
    }
    
    return length;
}

long dvcopytoriarray(spDVector x, long x_offset, double *data, double *imag, long length)
{
    length = MIN(x->length - x_offset, length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    /* copy data */
    if (data != NULL) {
        memcpy(data, x->data + x_offset, sizeof(double) * length);
    }
    if (imag != NULL) {
        if (x->imag == NULL) {
            memset(data, 0, sizeof(double) * length);
        } else {
            memcpy(imag, x->imag + x_offset, sizeof(double) * length);
        }
    }
    
    return length;
}

/*
 *	convert type of vector
 */
spLVector xsvtol(spSVector x)
{
    long k;
    spLVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	svsync(x);
    }
#endif
    
    y = xlvallocul(svgetplugin(x), x->length, spIsFalse(svislocked(x)));
    if (x->imag != NULL) {
	lvialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (long)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (long)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	lvunlock(y);
    }
#endif
    
    return y;
}

spFVector xsvtof(spSVector x)
{
    long k;
    spFVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	svsync(x);
    }
#endif
    
    y = xfvallocul(svgetplugin(x), x->length, spIsFalse(svislocked(x)));
    if (x->imag != NULL) {
	fvialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (float)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (float)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	fvunlock(y);
    }
#endif
    
    return y;
}

spDVector xsvtod(spSVector x)
{
    long k;
    spDVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	svsync(x);
    }
#endif
    
    y = xdvallocul(svgetplugin(x), x->length, spIsFalse(svislocked(x)));
    if (x->imag != NULL) {
	dvialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (double)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (double)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	dvunlock(y);
    }
#endif
    
    return y;
}

spSVector xdvtos(spDVector x)
{
    long k;
    spSVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	dvsync(x);
    }
#endif
    
    y = xsvallocul(dvgetplugin(x), x->length, spIsFalse(dvislocked(x)));
    if (x->imag != NULL) {
	svialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (short)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (short)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	svunlock(y);
    }
#endif
    
    return y;
}

spLVector xdvtol(spDVector x)
{
    long k;
    spLVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	dvsync(x);
    }
#endif

    y = xlvallocul(dvgetplugin(x), x->length, spIsFalse(dvislocked(x)));
    if (x->imag != NULL) {
	lvialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (long)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (long)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	lvunlock(y);
    }
#endif
    
    return y;
}

spFVector xdvtof(spDVector x)
{
    long k;
    spFVector y;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	dvsync(x);
    }
#endif
    
    y = xfvallocul(dvgetplugin(x), x->length, spIsFalse(dvislocked(x)));
    if (x->imag != NULL) {
	fvialloc(y);
    }

    for (k = 0; k < y->length; k++) {
	y->data[k] = (float)x->data[k];
    }
    if (y->imag != NULL) {
	for (k = 0; k < y->length; k++) {
	    y->imag[k] = (float)x->imag[k];
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	fvunlock(y);
    }
#endif
    
    return y;
}

/*
 *	make clone of input vector
 */
spSVector xsvclone(spSVector x)
{
    spSVector y;

    if (x == NODATA) return NODATA;

    /* memory allocate */
    y = xsvalloculfromv(x, x->length);
    if (x->imag != NULL) {
	svialloc(y);
    }
    
    /* copy data */
    svcopy(y, x);

    return y;
}

spLVector xlvclone(spLVector x)
{
    spLVector y;

    if (x == NODATA) return NODATA;
    
    /* memory allocate */
    y = xlvalloculfromv(x, x->length);
    if (x->imag != NULL) {
	lvialloc(y);
    }
    
    /* copy data */
    lvcopy(y, x);

    return y;
}

spFVector xfvclone(spFVector x)
{
    spFVector y;

    if (x == NODATA) return NODATA;
    
    /* memory allocate */
    y = xfvalloculfromv(x, x->length);
    if (x->imag != NULL) {
	fvialloc(y);
    }
    
    /* copy data */
    fvcopy(y, x);

    return y;
}

spDVector xdvclone(spDVector x)
{
    spDVector y;

    if (x == NODATA) return NODATA;
    
    /* memory allocate */
    y = xdvalloculfromv(x, x->length);
    if (x->imag != NULL) {
	dvialloc(y);
    }
    
    /* copy data */
    dvcopy(y, x);

    return y;
}

/*
 *	concatenate vector
 */
spSVector xsvcat(spSVector x, spSVector y)
{
    long k;
    spSVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    /* memory allocate */
    z = xsvalloculfromv(x, x->length + y->length);
    if (x->imag != NULL || y->imag != NULL) {
	svialloc(z);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE && svislocked(y) == SP_FALSE) {
	svsetlock(z, SP_FALSE);
    }
    
    if (svislocked(x) == SP_FALSE && z->instance != NULL && x->instance != NULL && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->radd != NULL && flist->icopy != NULL&& flist->iadd != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}
	flist->rcopy(z->instance, x->instance, x->length);
	flist->radd(z->instance, x->length, y->instance, 0, y->length, 0);
	if (z->imag != NULL) {
	    flist->icopy(z->instance, x->instance, x->length);
	    flist->iadd(z->instance, x->length, y->instance, 0, y->length, 0);
	}
	
	if (svisplugincomputable(z) == SP_TRUE) {
	    return z;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    /* concatenate data */
    for (k = 0; k < z->length; k++) {
	if (k < x->length) {
	    z->data[k] = x->data[k];
	} else {
	    z->data[k] = y->data[k - x->length];
	}
    }
    if (z->imag != NULL) {
	for (k = 0; k < z->length; k++) {
	    if (k < x->length) {
		if (x->imag != NULL) {
		    z->imag[k] = x->imag[k];
		} else {
		    z->imag[k] = 0;
		}
	    } else {
		if (y->imag != NULL) {
		    z->imag[k] = y->imag[k - x->length];
		} else {
		    z->imag[k] = 0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(z) == SP_FALSE) {
        svunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

spLVector xlvcat(spLVector x, spLVector y)
{
    long k;
    spLVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    /* memory allocate */
    z = xlvalloculfromv(x, x->length + y->length);
    if (x->imag != NULL || y->imag != NULL) {
	lvialloc(z);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE && lvislocked(x) == SP_FALSE && lvislocked(y) == SP_FALSE) {
	lvsetlock(z, SP_FALSE);
    }
    
    if (lvislocked(x) == SP_FALSE && z->instance != NULL && x->instance != NULL && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->radd != NULL && flist->icopy != NULL&& flist->iadd != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}
	flist->rcopy(z->instance, x->instance, x->length);
	flist->radd(z->instance, x->length, y->instance, 0, y->length, 0);
	if (z->imag != NULL) {
	    flist->icopy(z->instance, x->instance, x->length);
	    flist->iadd(z->instance, x->length, y->instance, 0, y->length, 0);
	}
	
	if (lvisplugincomputable(z) == SP_TRUE) {
	    return z;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    /* concatenate data */
    for (k = 0; k < z->length; k++) {
	if (k < x->length) {
	    z->data[k] = x->data[k];
	} else {
	    z->data[k] = y->data[k - x->length];
	}
    }
    if (z->imag != NULL) {
	for (k = 0; k < z->length; k++) {
	    if (k < x->length) {
		if (x->imag != NULL) {
		    z->imag[k] = x->imag[k];
		} else {
		    z->imag[k] = 0;
		}
	    } else {
		if (y->imag != NULL) {
		    z->imag[k] = y->imag[k - x->length];
		} else {
		    z->imag[k] = 0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(z) == SP_FALSE) {
        lvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return z;
}

spFVector xfvcat(spFVector x, spFVector y)
{
    long k;
    spFVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    /* memory allocate */
    z = xfvalloculfromv(x, x->length + y->length);
    if (x->imag != NULL || y->imag != NULL) {
	fvialloc(z);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_FALSE) {
	fvsetlock(z, SP_FALSE);
    }
    
    if (fvislocked(x) == SP_FALSE && z->instance != NULL && x->instance != NULL && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->radd != NULL && flist->icopy != NULL&& flist->iadd != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}
	flist->rcopy(z->instance, x->instance, x->length);
	flist->radd(z->instance, x->length, y->instance, 0, y->length, 0);
	if (z->imag != NULL) {
	    flist->icopy(z->instance, x->instance, x->length);
	    flist->iadd(z->instance, x->length, y->instance, 0, y->length, 0);
	}
	
	if (fvisplugincomputable(z) == SP_TRUE) {
	    return z;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    /* concatenate data */
    for (k = 0; k < z->length; k++) {
	if (k < x->length) {
	    z->data[k] = x->data[k];
	} else {
	    z->data[k] = y->data[k - x->length];
	}
    }
    if (z->imag != NULL) {
	for (k = 0; k < z->length; k++) {
	    if (k < x->length) {
		if (x->imag != NULL) {
		    z->imag[k] = x->imag[k];
		} else {
		    z->imag[k] = 0.0;
		}
	    } else {
		if (y->imag != NULL) {
		    z->imag[k] = y->imag[k - x->length];
		} else {
		    z->imag[k] = 0.0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(z) == SP_FALSE) {
        fvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

spDVector xdvcat(spDVector x, spDVector y)
{
    long k;
    spDVector z;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    /* memory allocate */
    z = xdvalloculfromv(x, x->length + y->length);
    if (x->imag != NULL || y->imag != NULL) {
	dvialloc(z);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_FALSE) {
	dvsetlock(z, SP_FALSE);
    }
    
    if (dvislocked(x) == SP_FALSE && z->instance != NULL && x->instance != NULL && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(z)) != NULL
        && flist->rcopy != NULL && flist->radd != NULL && flist->icopy != NULL&& flist->iadd != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}
	flist->rcopy(z->instance, x->instance, x->length);
	flist->radd(z->instance, x->length, y->instance, 0, y->length, 0);
	if (z->imag != NULL) {
	    flist->icopy(z->instance, x->instance, x->length);
	    flist->iadd(z->instance, x->length, y->instance, 0, y->length, 0);
	}
	
	if (dvisplugincomputable(z) == SP_TRUE) {
	    return z;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    /* concatenate data */
    for (k = 0; k < z->length; k++) {
	if (k < x->length) {
	    z->data[k] = x->data[k];
	} else {
	    z->data[k] = y->data[k - x->length];
	}
    }
    if (z->imag != NULL) {
	for (k = 0; k < z->length; k++) {
	    if (k < x->length) {
		if (x->imag != NULL) {
		    z->imag[k] = x->imag[k];
		} else {
		    z->imag[k] = 0.0;
		}
	    } else {
		if (y->imag != NULL) {
		    z->imag[k] = y->imag[k - x->length];
		} else {
		    z->imag[k] = 0.0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(z) == SP_FALSE) {
        dvunlockcore(z, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return z;
}

void svcat(spSVector x, spSVector y)
{
    long k;
    long old_x_length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    old_x_length = x->length;
    
    /* memory allocate */
    xsvrealloc(x, x->length + y->length);
    if (x->imag == NULL && y->imag != NULL) {
	svialloc(x);
	svizeros(x, old_x_length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}
	flist->radd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	if (y->imag != NULL) {
	    flist->iadd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	}
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif

    /* concatenate data */
    for (k = old_x_length; k < x->length; k++) {
	x->data[k] = y->data[k - old_x_length];
    }
    if (x->imag != NULL) {
	for (k = old_x_length; k < x->length; k++) {
	    if (y->imag != NULL) {
		x->imag[k] = y->imag[k - old_x_length];
	    } else {
		x->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvcat(spLVector x, spLVector y)
{
    long k;
    long old_x_length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    old_x_length = x->length;
    
    /* memory allocate */
    xlvrealloc(x, x->length + y->length);
    if (x->imag == NULL && y->imag != NULL) {
	lvialloc(x);
	lvizeros(x, old_x_length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}
	flist->radd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	if (y->imag != NULL) {
	    flist->iadd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	}
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif

    /* concatenate data */
    for (k = old_x_length; k < x->length; k++) {
	x->data[k] = y->data[k - old_x_length];
    }
    if (x->imag != NULL) {
	for (k = old_x_length; k < x->length; k++) {
	    if (y->imag != NULL) {
		x->imag[k] = y->imag[k - old_x_length];
	    } else {
		x->imag[k] = 0L;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvcat(spFVector x, spFVector y)
{
    long k;
    long old_x_length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    old_x_length = x->length;
    
    /* memory allocate */
    xfvrealloc(x, x->length + y->length);
    if (x->imag == NULL && y->imag != NULL) {
	fvialloc(x);
	fvizeros(x, old_x_length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}
	flist->radd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	if (y->imag != NULL) {
	    flist->iadd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	}
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
	
    /* concatenate data */
    for (k = old_x_length; k < x->length; k++) {
	x->data[k] = y->data[k - old_x_length];
    }
    if (x->imag != NULL) {
	for (k = old_x_length; k < x->length; k++) {
	    if (y->imag != NULL) {
		x->imag[k] = y->imag[k - old_x_length];
	    } else {
		x->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvcat(spDVector x, spDVector y)
{
    long k;
    long old_x_length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    old_x_length = x->length;
    
    /* memory allocate */
    xdvrealloc(x, x->length + y->length);
    if (x->imag == NULL && y->imag != NULL) {
	dvialloc(x);
	dvizeros(x, old_x_length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}
	flist->radd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	if (y->imag != NULL) {
	    flist->iadd(x->instance, old_x_length, y->instance, 0, y->length, 0);
	}
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif

    /* concatenate data */
    for (k = old_x_length; k < x->length; k++) {
	x->data[k] = y->data[k - old_x_length];
    }
    if (x->imag != NULL) {
	for (k = old_x_length; k < x->length; k++) {
	    if (y->imag != NULL) {
		x->imag[k] = y->imag[k - old_x_length];
	    } else {
		x->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

/*
 *	get initialized vector
 */
long lvgetinitcount(long x_length, long m, long incr, long n)
{
    long num;
    
    if ((incr > 0 && m > n) || (incr < 0 && m < n)) {
	fprintf(stderr, "bad increment value\n");
	return -1;
    }
    if (incr == 0) {
	num = n;
	if (num <= 0) {
	    num = x_length;
	}
    } else {
	num = labs((n - m) / incr) + 1;
    }

    return num;
}

static spBool svinitcore(spSVector x, long m, long incr, long n,
			 spBool real_flag, spBool imag_flag)
{
    long k;
    long num;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if ((num = lvgetinitcount(x->length, m, incr, n)) < 0L) {
	return SP_FALSE;
    }

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	svialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL && flist->iinit != NULL) {
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->rinit(x->instance, x->length, m, incr, num) == SP_TRUE
		&& flist->iinit(x->instance, x->length, m, incr, num) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->rinit(x->instance, x->length, m, incr, num);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iinit(x->instance, x->length, m, incr, num);
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    /* initailize data */
    for (k = 0; k < num; k++) {
	if (k >= x->length) {
	    break;
	}
	if (real_flag == SP_TRUE) {
	    x->data[k] = (short)(m + (k * incr));
	}
	if (imag_flag == SP_TRUE) {
	    x->imag[k] = (short)(m + (k * incr));
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool svinit(spSVector x, long m, long incr, long n)
{
    return svinitcore(x, m, incr, n, SP_TRUE, SP_FALSE);
}

static spBool lvinitcore(spLVector x, long m, long incr, long n,
			 spBool real_flag, spBool imag_flag)
{
    long k;
    long num;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if ((num = lvgetinitcount(x->length, m, incr, n)) < 0L) {
	return SP_FALSE;
    }

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	lvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL && flist->iinit != NULL) {
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->rinit(x->instance, x->length, m, incr, num) == SP_TRUE
		&& flist->iinit(x->instance, x->length, m, incr, num) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->rinit(x->instance, x->length, m, incr, num);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iinit(x->instance, x->length, m, incr, num);
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    /* initailize data */
    for (k = 0; k < num; k++) {
	if (k >= x->length) {
	    break;
	}
	if (real_flag == SP_TRUE) {
	    x->data[k] = m + (k * incr);
	}
	if (imag_flag == SP_TRUE) {
	    x->imag[k] = m + (k * incr);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool lvinit(spLVector x, long m, long incr, long n)
{
    return lvinitcore(x, m, incr, n, SP_TRUE, SP_FALSE);
}

long fvgetinitcount(long x_length, float m, float incr, float n)
{
    long num;
    
    if ((incr > 0.0F && m > n) || (incr < 0.0F && m < n)) {
	fprintf(stderr, "bad increment value\n");
	return -1;
    }
    if (incr == 0.0F) {
	num = (long)(1.0e-10F + n);
	if (num <= 0) {
	    num = x_length;
	}
    } else {
        num = ((long)(1.0e-10F + (n - m) / incr)) + 1;
    }

    return num;
}

static spBool fvinitcore(spFVector x, float m, float incr, float n,
			 spBool real_flag, spBool imag_flag)
{
    long k;
    long num;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if ((num = fvgetinitcount(x->length, m, incr, n)) < 0L) {
	return SP_FALSE;
    }

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	fvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL && flist->iinit != NULL) {
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->rinit(x->instance, x->length, m, incr, num) == SP_TRUE
		&& flist->iinit(x->instance, x->length, m, incr, num) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->rinit(x->instance, x->length, m, incr, num);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iinit(x->instance, x->length, m, incr, num);
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    /* initailize data */
    for (k = 0; k < num; k++) {
	if (k >= x->length) {
	    break;
	}
	if (real_flag == SP_TRUE) {
	    x->data[k] = m + (k * incr);
	}
	if (imag_flag == SP_TRUE) {
	    x->imag[k] = m + (k * incr);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool fvinit(spFVector x, float m, float incr, float n)
{
    return fvinitcore(x, m, incr, n, SP_TRUE, SP_FALSE);
}

long dvgetinitcount(long x_length, double m, double incr, double n)
{
    long num;
    
    if ((incr > 0.0 && m > n) || (incr < 0.0 && m < n)) {
	fprintf(stderr, "bad increment value\n");
	return SP_FALSE;
    }
    if (incr == 0.0) {
	num = (long)(1.0e-10 + n);
	if (num <= 0) {
	    num = x_length;
	}
    } else {
        num = ((long)(1.0e-10 + (n - m) / incr)) + 1;
    }

    return num;
}

static spBool dvinitcore(spDVector x, double m, double incr, double n,
			 spBool real_flag, spBool imag_flag)
{
    long k;
    long num;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if ((num = dvgetinitcount(x->length, m, incr, n)) < 0L) {
	return SP_FALSE;
    }

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	dvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL && flist->iinit != NULL) {
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->rinit(x->instance, x->length, m, incr, num) == SP_TRUE
		&& flist->iinit(x->instance, x->length, m, incr, num) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->rinit(x->instance, x->length, m, incr, num);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iinit(x->instance, x->length, m, incr, num);
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    /* initailize data */
    for (k = 0; k < num; k++) {
	if (k >= x->length) {
	    break;
	}
	if (real_flag == SP_TRUE) {
	    x->data[k] = m + (k * incr);
	}
	if (imag_flag == SP_TRUE) {
	    x->imag[k] = m + (k * incr);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool dvinit(spDVector x, double m, double incr, double n)
{
    return dvinitcore(x, m, incr, n, SP_TRUE, SP_FALSE);
}


spSVector xsvinit(long m, long incr, long n)
{
    return xsvinitul(NULL, m, incr, n, SP_FALSE);
}

spLVector xlvinit(long m, long incr, long n)
{
    return xlvinitul(NULL, m, incr, n, SP_FALSE);
}

spFVector xfvinit(float m, float incr, float n)
{
    return xfvinitul(NULL, m, incr, n, SP_FALSE);
}

spDVector xdvinit(double m, double incr, double n)
{
    return xdvinitul(NULL, m, incr, n, SP_FALSE);
}

spSVector xsvinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag)
{
    long k;
    long num;
    spSVector x;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "xsvinitul", "unlock_flag = %d\n", unlock_flag);
    
    if ((num = lvgetinitcount(0L, m, incr, n)) <= 0L) {
	if (num <= 0L) {
	    fprintf(stderr, "wrong value\n");
	}
	return xsvnullul(plugin, unlock_flag);
    }
    
    /* memory allocate */
    x = xsvallocul(plugin, num, unlock_flag);

#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag) {
	if (x->instance != NULL && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL) {
	    flist->rinit(x->instance, x->length, m, incr, num);
	    return x;
	}
    }
#endif
    
    /* initailize data */
    for (k = 0; k < x->length; k++) {
	x->data[k] = (short)(m + (k * incr));
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return x;
}

spLVector xlvinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag)
{
    long k;
    long num;
    spLVector x;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "xlvinitul", "unlock_flag = %d\n", unlock_flag);
    
    if ((num = lvgetinitcount(0L, m, incr, n)) <= 0L) {
	if (num <= 0L) {
	    fprintf(stderr, "wrong value\n");
	}
	return xlvnullul(plugin, unlock_flag);
    }
    
    /* memory allocate */
    x = xlvallocul(plugin, num, unlock_flag);

#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag) {
	if (x->instance != NULL && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL) {
	    flist->rinit(x->instance, x->length, m, incr, num);
	    return x;
	}
    }
#endif
    
    /* initailize data */
    for (k = 0; k < x->length; k++) {
	x->data[k] = m + (k * incr);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return x;
}

spFVector xfvinitul(spPlugin *plugin, float m, float incr, float n, spBool unlock_flag)
{
    long k;
    long num;
    spFVector x;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "xfvinitul", "unlock_flag = %d\n", unlock_flag);
    
    if ((num = fvgetinitcount(0L, m, incr, n)) <= 0L) {
	if (num <= 0L) {
	    fprintf(stderr, "wrong value\n");
	}
	return xfvnullul(plugin, unlock_flag);
    }
    
    /* memory allocate */
    x = xfvallocul(plugin, num, unlock_flag);

#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag) {
	if (x->instance != NULL && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL) {
	    flist->rinit(x->instance, x->length, m, incr, num);
	    return x;
	}
    }
#endif
    
    /* initailize data */
    for (k = 0; k < x->length; k++) {
	x->data[k] = m + (k * incr);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return x;
}

spDVector xdvinitul(spPlugin *plugin, double m, double incr, double n, spBool unlock_flag)
{
    long k;
    long num;
    spDVector x;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "xdvinitul", "unlock_flag = %d\n", unlock_flag);
    
    if ((num = dvgetinitcount(0L, m, incr, n)) <= 0L) {
	if (num <= 0L) {
	    fprintf(stderr, "wrong value\n");
	}
	return xdvnullul(plugin, unlock_flag);
    }
    
    /* memory allocate */
    x = xdvallocul(plugin, num, unlock_flag);
	
#ifdef SP_USE_VECTOR_ENGINE
    if (unlock_flag) {
	if (x->instance != NULL && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rinit != NULL) {
	    flist->rinit(x->instance, x->length, m, incr, num);
	    return x;
	}
    }
#endif

    /* initailize data */
    for (k = 0; k < x->length; k++) {
	x->data[k] = m + (k * incr);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return x;
}

spBool sviinit(spSVector x, long m, long incr, long n)
{
    return svinitcore(x, m, incr, n, SP_FALSE, SP_TRUE);
}

spBool lviinit(spLVector x, long m, long incr, long n)
{
    return lvinitcore(x, m, incr, n, SP_FALSE, SP_TRUE);
}

spBool fviinit(spFVector x, float m, float incr, float n)
{
    return fvinitcore(x, m, incr, n, SP_FALSE, SP_TRUE);
}

spBool dviinit(spDVector x, double m, double incr, double n)
{
    return dvinitcore(x, m, incr, n, SP_FALSE, SP_TRUE);
}

spBool svriinit(spSVector x, long m, long incr, long n)
{
    return svinitcore(x, m, incr, n, SP_TRUE, SP_TRUE);
}

spBool lvriinit(spLVector x, long m, long incr, long n)
{
    return lvinitcore(x, m, incr, n, SP_TRUE, SP_TRUE);
}

spBool fvriinit(spFVector x, float m, float incr, float n)
{
    return fvinitcore(x, m, incr, n, SP_TRUE, SP_TRUE);
}

spBool dvriinit(spDVector x, double m, double incr, double n)
{
    return dvinitcore(x, m, incr, n, SP_TRUE, SP_TRUE);
}

spSVector xsvriinit(long m, long incr, long n)
{
    return xsvriinitul(NULL, m, incr, n, SP_FALSE);
}

spLVector xlvriinit(long m, long incr, long n)
{
    return xlvriinitul(NULL, m, incr, n, SP_FALSE);
}

spFVector xfvriinit(float m, float incr, float n)
{
    return xfvriinitul(NULL, m, incr, n, SP_FALSE);
}

spDVector xdvriinit(double m, double incr, double n)
{
    return xdvriinitul(NULL, m, incr, n, SP_FALSE);
}

spSVector xsvriinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag)
{
    spSVector x;

    x = xsvinitul(plugin, m, incr, n, unlock_flag);
    svialloc(x);
    sviinit(x, m, incr, n);

    return x;
}

spLVector xlvriinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag)
{
    spLVector x;

    x = xlvinitul(plugin, m, incr, n, unlock_flag);
    lvialloc(x);
    lviinit(x, m, incr, n);

    return x;
}

spFVector xfvriinitul(spPlugin *plugin, float m, float incr, float n, spBool unlock_flag)
{
    spFVector x;

    x = xfvinitul(plugin, m, incr, n, unlock_flag);
    fvialloc(x);
    fviinit(x, m, incr, n);

    return x;
}

spDVector xdvriinitul(spPlugin *plugin, double m, double incr, double n, spBool unlock_flag)
{
    spDVector x;

    x = xdvinitul(plugin, m, incr, n, unlock_flag);
    dvialloc(x);
    dviinit(x, m, incr, n);

    return x;
}

#if 1
static spBool svidxnumscore(spSVector x, spLVector idx, short value, spBool real_flag, spBool imag_flag)
{
    long k, l;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	svialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && idx->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->ridxnums != NULL && flist->iidxnums != NULL) {
	if (svislocked(x) == SP_FALSE && lvislocked(idx) == SP_TRUE) {
	    lvsync(idx);
	}
        
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE
		&& flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value);
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (lvislocked(idx) == SP_FALSE) {
        lvsync(idx);
    }
#endif
    
    for (k = 0; k < idx->length; k++) {
        l = idx->data[k];
        if (l >= 0 && l < x->length) {
            if (real_flag == SP_TRUE) {
                x->data[l] = value;
            }
            if (imag_flag == SP_TRUE) {
                x->imag[l] = value;
            }
        }
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool svidxnums(spSVector x, spLVector idx, short value)
{
    return svidxnumscore(x, idx, value, SP_TRUE, SP_FALSE);
}

spBool sviidxnums(spSVector x, spLVector idx, short value)
{
    return svidxnumscore(x, idx, value, SP_FALSE, SP_TRUE);
}

spBool svriidxnums(spSVector x, spLVector idx, short r, short i)
{
    if (svidxnumscore(x, idx, r, SP_TRUE, SP_FALSE) == SP_TRUE
        && svidxnumscore(x, idx, i, SP_FALSE, SP_TRUE) == SP_TRUE) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spSVector xsvidxnums(spSVector x, spLVector idx, short value)
{
    spSVector y;

    y = xsvclone(x);
    svidxnums(y, idx, value);

    return y;
}

spSVector xsviidxnums(spSVector x, spLVector idx, short value)
{
    spSVector y;

    y = xsvclone(x);
    sviidxnums(y, idx, value);

    return y;
}

spSVector xsvriidxnums(spSVector x, spLVector idx, short r, short i)
{
    spSVector y;

    y = xsvclone(x);
    svriidxnums(y, idx, r, i);

    return y;
}

static spBool lvidxnumscore(spLVector x, spLVector idx, long value, spBool real_flag, spBool imag_flag)
{
    long k, l;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	lvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && idx->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->ridxnums != NULL && flist->iidxnums != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(idx) == SP_TRUE) {
	    lvsync(idx);
	}
        
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE
		&& flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value);
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(idx) == SP_FALSE) {
        lvsync(idx);
    }
#endif
    
    for (k = 0; k < idx->length; k++) {
        l = idx->data[k];
        if (l >= 0 && l < x->length) {
            if (real_flag == SP_TRUE) {
                x->data[l] = value;
            }
            if (imag_flag == SP_TRUE) {
                x->imag[l] = value;
            }
        }
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool lvidxnums(spLVector x, spLVector idx, long value)
{
    return lvidxnumscore(x, idx, value, SP_TRUE, SP_FALSE);
}

spBool lviidxnums(spLVector x, spLVector idx, long value)
{
    return lvidxnumscore(x, idx, value, SP_FALSE, SP_TRUE);
}

spBool lvriidxnums(spLVector x, spLVector idx, long r, long i)
{
    if (lvidxnumscore(x, idx, r, SP_TRUE, SP_FALSE) == SP_TRUE
        && lvidxnumscore(x, idx, i, SP_FALSE, SP_TRUE) == SP_TRUE) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spLVector xlvidxnums(spLVector x, spLVector idx, long value)
{
    spLVector y;

    y = xlvclone(x);
    lvidxnums(y, idx, value);

    return y;
}

spLVector xlviidxnums(spLVector x, spLVector idx, long value)
{
    spLVector y;

    y = xlvclone(x);
    lviidxnums(y, idx, value);

    return y;
}

spLVector xlvriidxnums(spLVector x, spLVector idx, long r, long i)
{
    spLVector y;

    y = xlvclone(x);
    lvriidxnums(y, idx, r, i);

    return y;
}

static spBool fvidxnumscore(spFVector x, spLVector idx, float value, spBool real_flag, spBool imag_flag)
{
    long k, l;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	fvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && idx->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->ridxnums != NULL && flist->iidxnums != NULL) {
	if (fvislocked(x) == SP_FALSE && lvislocked(idx) == SP_TRUE) {
	    lvsync(idx);
	}
        
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE
		&& flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value);
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (lvislocked(idx) == SP_FALSE) {
        lvsync(idx);
    }
#endif
    
    for (k = 0; k < idx->length; k++) {
        l = idx->data[k];
        if (l >= 0 && l < x->length) {
            if (real_flag == SP_TRUE) {
                x->data[l] = value;
            }
            if (imag_flag == SP_TRUE) {
                x->imag[l] = value;
            }
        }
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool fvidxnums(spFVector x, spLVector idx, float value)
{
    return fvidxnumscore(x, idx, value, SP_TRUE, SP_FALSE);
}

spBool fviidxnums(spFVector x, spLVector idx, float value)
{
    return fvidxnumscore(x, idx, value, SP_FALSE, SP_TRUE);
}

spBool fvriidxnums(spFVector x, spLVector idx, float r, float i)
{
    if (fvidxnumscore(x, idx, r, SP_TRUE, SP_FALSE) == SP_TRUE
        && fvidxnumscore(x, idx, i, SP_FALSE, SP_TRUE) == SP_TRUE) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spFVector xfvidxnums(spFVector x, spLVector idx, float value)
{
    spFVector y;

    y = xfvclone(x);
    fvidxnums(y, idx, value);

    return y;
}

spFVector xfviidxnums(spFVector x, spLVector idx, float value)
{
    spFVector y;

    y = xfvclone(x);
    fviidxnums(y, idx, value);

    return y;
}

spFVector xfvriidxnums(spFVector x, spLVector idx, float r, float i)
{
    spFVector y;

    y = xfvclone(x);
    fvriidxnums(y, idx, r, i);

    return y;
}

static spBool dvidxnumscore(spDVector x, spLVector idx, double value, spBool real_flag, spBool imag_flag)
{
    long k, l;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (imag_flag == SP_TRUE && x->imag == NULL) {
	dvialloc(x);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && idx->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->ridxnums != NULL && flist->iidxnums != NULL) {
	if (dvislocked(x) == SP_FALSE && lvislocked(idx) == SP_TRUE) {
	    lvsync(idx);
	}
        
	if (real_flag == SP_TRUE && imag_flag == SP_TRUE) {
	    if (flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE
		&& flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value) == SP_TRUE) {
		return SP_TRUE;
	    } else {
		return SP_FALSE;
	    }
	} else if (real_flag == SP_TRUE) {
	    return flist->ridxnums(x->instance, x->length, idx->instance, idx->length, value);
	} else if (imag_flag == SP_TRUE) {
	    return flist->iidxnums(x->instance, x->length, idx->instance, idx->length, value);
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (lvislocked(idx) == SP_FALSE) {
        lvsync(idx);
    }
#endif
    
    for (k = 0; k < idx->length; k++) {
        l = idx->data[k];
        if (l >= 0 && l < x->length) {
            if (real_flag == SP_TRUE) {
                x->data[l] = value;
            }
            if (imag_flag == SP_TRUE) {
                x->imag[l] = value;
            }
        }
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool dvidxnums(spDVector x, spLVector idx, double value)
{
    return dvidxnumscore(x, idx, value, SP_TRUE, SP_FALSE);
}

spBool dviidxnums(spDVector x, spLVector idx, double value)
{
    return dvidxnumscore(x, idx, value, SP_FALSE, SP_TRUE);
}

spBool dvriidxnums(spDVector x, spLVector idx, double r, double i)
{
    if (dvidxnumscore(x, idx, r, SP_TRUE, SP_FALSE) == SP_TRUE
        && dvidxnumscore(x, idx, i, SP_FALSE, SP_TRUE) == SP_TRUE) {
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
}

spDVector xdvidxnums(spDVector x, spLVector idx, double value)
{
    spDVector y;

    y = xdvclone(x);
    dvidxnums(y, idx, value);

    return y;
}

spDVector xdviidxnums(spDVector x, spLVector idx, double value)
{
    spDVector y;

    y = xdvclone(x);
    dviidxnums(y, idx, value);

    return y;
}

spDVector xdvriidxnums(spDVector x, spLVector idx, double r, double i)
{
    spDVector y;

    y = xdvclone(x);
    dvriidxnums(y, idx, r, i);

    return y;
}
#endif

/*
 *	cut vector
 */
spSVector xsvcut(spSVector x, long offset, long length)
{
    long k;
    long pos;
    spSVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    y = xsvalloculfromv(x, length);
    if (x->imag != NULL) {
	svialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
	svsetlock(y, SP_FALSE);
    }
    
    if (x->instance != NULL && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	k = MIN(x->length - offset, y->length);
	flist->radd(y->instance, 0, x->instance, offset, k, 0);
	if (y->imag != NULL) {
	    flist->iadd(y->instance, 0, x->instance, offset, k, 0);
	}
	if (svisplugincomputable(y) == SP_TRUE) {
	    return y;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < y->length; k++) {
	pos = k + offset;
	if (pos >= 0 && pos < x->length) {
	    y->data[k] = x->data[pos];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[pos];
	    }
	} else {
	    y->data[k] = 0;
	    if (y->imag != NULL) {
		y->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return y;
}

spLVector xlvcut(spLVector x, long offset, long length)
{
    long k;
    long pos;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    y = xlvalloculfromv(x, length);
    if (x->imag != NULL) {
	lvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
	lvsetlock(y, SP_FALSE);
    }
    
    if (x->instance != NULL && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	k = MIN(x->length - offset, y->length);
	flist->radd(y->instance, 0, x->instance, offset, k, 0);
	if (y->imag != NULL) {
	    flist->iadd(y->instance, 0, x->instance, offset, k, 0);
	}
	if (lvisplugincomputable(y) == SP_TRUE) {
	    return y;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < y->length; k++) {
	pos = k + offset;
	if (pos >= 0 && pos < x->length) {
	    y->data[k] = x->data[pos];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[pos];
	    }
	} else {
	    y->data[k] = 0;
	    if (y->imag != NULL) {
		y->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return y;
}

spFVector xfvcut(spFVector x, long offset, long length)
{
    long k;
    long pos;
    spFVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    y = xfvalloculfromv(x, length);
    if (x->imag != NULL) {
	fvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
	fvsetlock(y, SP_FALSE);
    }
    
    if (x->instance != NULL && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	k = MIN(x->length - offset, y->length);
	flist->radd(y->instance, 0, x->instance, offset, k, 0);
	if (y->imag != NULL) {
	    flist->iadd(y->instance, 0, x->instance, offset, k, 0);
	}
	if (fvisplugincomputable(y) == SP_TRUE) {
	    return y;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < y->length; k++) {
	pos = k + offset;
	if (pos >= 0 && pos < x->length) {
	    y->data[k] = x->data[pos];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[pos];
	    }
	} else {
	    y->data[k] = 0.0;
	    if (y->imag != NULL) {
		y->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return y;
}

spDVector xdvcut(spDVector x, long offset, long length)
{
    long k;
    long pos;
    spDVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    y = xdvalloculfromv(x, length);
    if (x->imag != NULL) {
	dvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
	dvsetlock(y, SP_FALSE);
    }
    
    if (x->instance != NULL && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(x)->getrptr(x->instance) != NULL) {
	k = MIN(x->length - offset, y->length);
	flist->radd(y->instance, 0, x->instance, offset, k, 0);
	if (y->imag != NULL) {
	    flist->iadd(y->instance, 0, x->instance, offset, k, 0);
	}
	if (dvisplugincomputable(y) == SP_TRUE) {
	    return y;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < y->length; k++) {
	pos = k + offset;
	if (pos >= 0 && pos < x->length) {
	    y->data[k] = x->data[pos];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[pos];
	    }
	} else {
	    y->data[k] = 0.0;
	    if (y->imag != NULL) {
		y->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return y;
}

/*
 *	paste data of x in y
 *	if length equal to 0, length is set length of x
 */
long svpaste(spSVector y, spSVector x, long y_offset, long x_length, spBool overlap)
{
    long k = 0;
    long pos;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x_length <= 0 || x_length > x->length) {
	x_length = x->length;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	k = MIN(x_length, y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, 0, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, 0, k, overlap);
	}
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif

    if (overlap) {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] = x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] = x->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k;
}

long lvpaste(spLVector y, spLVector x, long y_offset, long x_length, spBool overlap)
{
    long k = 0;
    long pos;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x_length <= 0 || x_length > x->length) {
	x_length = x->length;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	k = MIN(x_length, y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, 0, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, 0, k, overlap);
	}
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif

    if (overlap) {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] = x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] = x->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k;
}

long fvpaste(spFVector y, spFVector x, long y_offset, long x_length, spBool overlap)
{
    long k = 0;
    long pos;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x_length <= 0 || x_length > x->length) {
	x_length = x->length;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	k = MIN(x_length, y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, 0, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, 0, k, overlap);
	}
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif

    if (overlap) {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] = x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] = x->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k;
}

long dvpaste(spDVector y, spDVector x, long y_offset, long x_length, spBool overlap)
{
    long k = 0;
    long pos;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x_length <= 0 || x_length > x->length) {
	x_length = x->length;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	k = MIN(x_length, y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, 0, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, 0, k, overlap);
	}
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif

    if (overlap) {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x_length; k++) {
	    pos = k + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] = x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] = x->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k;
}

long svadd(spSVector y, long y_offset, spSVector x, long x_offset, long x_length, spBool overlap)
{
    long k;
    long pos;
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    length = MIN(x_offset + x_length, x->length);
#if 0
    if (length <= x_offset) {
	length = x->length;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (svislocked(y) == SP_FALSE && svislocked(x) == SP_TRUE) {
	    svsync(x);
	}
	k = MIN(MIN(x->length - x_offset, x_length), y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	}
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif

    if (overlap) {
	for (k = MAX(x_offset, 0); k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = x_offset; k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
                if (k < 0) {
                    y->data[pos] = 0;
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = 0;
                    }
                } else {
                    y->data[pos] = x->data[k];
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = x->imag[k];
                    }
                }
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(y) == SP_FALSE) {
        svunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k - x_offset;
}

long lvadd(spLVector y, long y_offset, spLVector x, long x_offset, long x_length, spBool overlap)
{
    long k;
    long pos;
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    length = MIN(x_offset + x_length, x->length);
#if 0
    if (length <= x_offset) {
	length = x->length;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (lvislocked(y) == SP_FALSE && lvislocked(x) == SP_TRUE) {
	    lvsync(x);
	}
	k = MIN(MIN(x->length - x_offset, x_length), y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	}
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif

    if (overlap) {
	for (k = MAX(x_offset, 0); k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = x_offset; k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
                if (k < 0) {
                    y->data[pos] = 0;
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = 0;
                    }
                } else {
                    y->data[pos] = x->data[k];
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = x->imag[k];
                    }
                }
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(y) == SP_FALSE) {
        lvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k - x_offset;
}

long fvadd(spFVector y, long y_offset, spFVector x, long x_offset, long x_length, spBool overlap)
{
    long k;
    long pos;
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    length = MIN(x_offset + x_length, x->length);
#if 0
    if (length <= x_offset) {
	length = x->length;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}
	k = MIN(MIN(x->length - x_offset, x_length), y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	}
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif

    if (overlap) {
	for (k = MAX(x_offset, 0); k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = x_offset; k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
                if (k < 0) {
                    y->data[pos] = 0.0f;
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = 0.0f;
                    }
                } else {
                    y->data[pos] = x->data[k];
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = x->imag[k];
                    }
                }
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k - x_offset;
}

long dvadd(spDVector y, long y_offset, spDVector x, long x_offset, long x_length, spBool overlap)
{
    long k;
    long pos;
    long length;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    length = MIN(x_offset + x_length, x->length);
#if 0
    if (length <= x_offset) {
	length = x->length;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->radd != NULL && flist->iadd != NULL
	&& SpVectorPluginRec(y)->getrptr(y->instance) != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}
	k = MIN(MIN(x->length - x_offset, x_length), y->length - y_offset);
	flist->radd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	if (y->imag != NULL && x->imag != NULL) {
	    flist->iadd(y->instance, y_offset, x->instance, x_offset, k, overlap);
	}
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return k;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    if (overlap) {
	for (k = MAX(x_offset, 0); k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
		y->data[pos] += x->data[k];
		if (x->imag != NULL && y->imag != NULL) {
		    y->imag[pos] += x->imag[k];
		}
	    }
	}
    } else {
	for (k = x_offset; k < length; k++) {
	    pos = k - x_offset + y_offset;
	    if (pos >= y->length) {
		break;
	    }
	    if (pos >= 0) {
                if (k < 0) {
                    y->data[pos] = 0.0;
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = 0.0;
                    }
                } else {
                    y->data[pos] = x->data[k];
                    if (x->imag != NULL && y->imag != NULL) {
                        y->imag[pos] = x->imag[k];
                    }
                }
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return k - x_offset;
}

void sverase(spSVector x, long offset, long length, spBool inv)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    offset = MAX(offset, 0);
    length = MIN(x->length - offset, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rerase != NULL && flist->ierase != NULL) {
	flist->rerase(x->instance, x->length, offset, length, inv);

	if (x->imag != NULL) {
	    flist->ierase(x->instance, x->length, offset, length, inv);
	}

	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    if (inv) {
	long start, remain;
	if (offset > 0) {
	    memset(x->data, 0, sizeof(short) * offset);
	    if (x->imag != NULL) {
		memset(x->imag, 0, sizeof(short) * offset);
	    }
	}
	start = offset + length;
	remain = x->length - start;

	if (remain > 0) {
	    memset(x->data + start, 0, sizeof(short) * remain);
	    if (x->imag != NULL) {
		memset(x->imag + start, 0, sizeof(short) * remain);
	    }
	}
    } else {
	memset(x->data + offset, 0, sizeof(short) * length);
	if (x->imag != NULL) {
	    memset(x->imag + offset, 0, sizeof(short) * length);
	}
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lverase(spLVector x, long offset, long length, spBool inv)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    offset = MAX(offset, 0);
    length = MIN(x->length - offset, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rerase != NULL && flist->ierase != NULL) {
	flist->rerase(x->instance, x->length, offset, length, inv);

	if (x->imag != NULL) {
	    flist->ierase(x->instance, x->length, offset, length, inv);
	}

	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    if (inv) {
	long start, remain;
	if (offset > 0) {
	    memset(x->data, 0, sizeof(long) * offset);
	    if (x->imag != NULL) {
		memset(x->imag, 0, sizeof(long) * offset);
	    }
	}
	start = offset + length;
	remain = x->length - start;

	if (remain > 0) {
	    memset(x->data + start, 0, sizeof(long) * remain);
	    if (x->imag != NULL) {
		memset(x->imag + start, 0, sizeof(long) * remain);
	    }
	}
    } else {
	memset(x->data + offset, 0, sizeof(long) * length);
	if (x->imag != NULL) {
	    memset(x->imag + offset, 0, sizeof(long) * length);
	}
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fverase(spFVector x, long offset, long length, spBool inv)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    offset = MAX(offset, 0);
    length = MIN(x->length - offset, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rerase != NULL && flist->ierase != NULL) {
	flist->rerase(x->instance, x->length, offset, length, inv);

	if (x->imag != NULL) {
	    flist->ierase(x->instance, x->length, offset, length, inv);
	}

	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    if (inv) {
	long start, remain;
	if (offset > 0) {
	    memset(x->data, 0, sizeof(float) * offset);
	    if (x->imag != NULL) {
		memset(x->imag, 0, sizeof(float) * offset);
	    }
	}
	start = offset + length;
	remain = x->length - start;

	if (remain > 0) {
	    memset(x->data + start, 0, sizeof(float) * remain);
	    if (x->imag != NULL) {
		memset(x->imag + start, 0, sizeof(float) * remain);
	    }
	}
    } else {
	memset(x->data + offset, 0, sizeof(float) * length);
	if (x->imag != NULL) {
	    memset(x->imag + offset, 0, sizeof(float) * length);
	}
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dverase(spDVector x, long offset, long length, spBool inv)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    offset = MAX(offset, 0);
    length = MIN(x->length - offset, length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rerase != NULL && flist->ierase != NULL) {
	flist->rerase(x->instance, x->length, offset, length, inv);

	if (x->imag != NULL) {
	    flist->ierase(x->instance, x->length, offset, length, inv);
	}

	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    if (inv) {
	long start, remain;
	if (offset > 0) {
	    memset(x->data, 0, sizeof(double) * offset);
	    if (x->imag != NULL) {
		memset(x->imag, 0, sizeof(double) * offset);
	    }
	}
	start = offset + length;
	remain = x->length - start;

	if (remain > 0) {
	    memset(x->data + start, 0, sizeof(double) * remain);
	    if (x->imag != NULL) {
		memset(x->imag + start, 0, sizeof(double) * remain);
	    }
	}
    } else {
	memset(x->data + offset, 0, sizeof(double) * length);
	if (x->imag != NULL) {
	    memset(x->imag + offset, 0, sizeof(double) * length);
	}
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svset(spSVector x, short *data, long length)
{
    if (x == NODATA) return;
    
    if (data != NULL) svrfree(x);
    
    length = MAX(length, 0);
    x->data = data;
    x->length = length;
    
    svsetlock(x, SP_TRUE);
    
    return;
}

void lvset(spLVector x, long *data, long length)
{
    if (x == NODATA) return;

    if (data != NULL) lvrfree(x);
    
    length = MAX(length, 0);
    x->data = data;
    x->length = length;
    
    lvsetlock(x, SP_TRUE);
    
    return;
}

void fvset(spFVector x, float *data, long length)
{
    if (x == NODATA) return;
    
    if (data != NULL) fvrfree(x);
    
    length = MAX(length, 0);
    x->data = data;
    x->length = length;
    
    fvsetlock(x, SP_TRUE);
    
    return;
}

void dvset(spDVector x, double *data, long length)
{
    if (x == NODATA) return;

    if (data != NULL) dvrfree(x);
    
    length = MAX(length, 0);
    x->data = data;
    x->length = length;

    dvsetlock(x, SP_TRUE);
    
    return;
}

void sviset(spSVector x, short *imag, long length)
{
    if (x == NODATA) return;
    
    svifree(x);
    
    length = MAX(length, 0);
    x->imag = imag;
    x->length = length;
    
    svsetlock(x, SP_TRUE);
    
    return;
}

void lviset(spLVector x, long *imag, long length)
{
    if (x == NODATA) return;
    
    lvifree(x);
    
    length = MAX(length, 0);
    x->imag = imag;
    x->length = length;
    
    lvsetlock(x, SP_TRUE);
    
    return;
}

void fviset(spFVector x, float *imag, long length)
{
    if (x == NODATA) return;
    
    fvifree(x);
    
    length = MAX(length, 0);
    x->imag = imag;
    x->length = length;
    
    fvsetlock(x, SP_TRUE);
    
    return;
}

void dviset(spDVector x, double *imag, long length)
{
    if (x == NODATA) return;

    dvifree(x);
    
    length = MAX(length, 0);
    x->imag = imag;
    x->length = length;

    dvsetlock(x, SP_TRUE);
    
    return;
}

spSVector xsvsetve(spPlugin *plugin, short *data, long length)
{
    spSVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spSVector);
    x->data = data;
    x->imag = NULL;
    x->length = length;
    
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
    
    svgetvinstance(x);
#endif
    
    return x;
}

spLVector xlvsetve(spPlugin *plugin, long *data, long length)
{
    spLVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spLVector);
    x->data = data;
    x->imag = NULL;
    x->length = length;
    
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
    
    lvgetvinstance(x);
#endif
    
    return x;
}

spFVector xfvsetve(spPlugin *plugin, float *data, long length)
{
    spFVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spFVector);
    x->data = data;
    x->imag = NULL;
    x->length = length;
    
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;
    
    fvgetvinstance(x);
#endif
    
    return x;
}

spDVector xdvsetve(spPlugin *plugin, double *data, long length)
{
    spDVector x;

    length = MAX(length, 0);
    x = xalloc(1, struct _spDVector);
    x->data = data;
    x->imag = NULL;
    x->length = length;
    
#ifdef SP_USE_VECTOR_ENGINE
    x->plugin = (plugin != NULL ? plugin : sp_default_vector_plugin);
    x->instance = NULL;
    x->locked = SP_TRUE;

    dvgetvinstance(x);
#endif
    
    return x;
}

spSVector xsvset(short *data, long length)
{
    return xsvsetve(NULL, data, length);
}

spLVector xlvset(long *data, long length)
{
    return xlvsetve(NULL, data, length);
}

spFVector xfvset(float *data, long length)
{
    return xfvsetve(NULL, data, length);
}

spDVector xdvset(double *data, long length)
{
    return xdvsetve(NULL, data, length);
}

spSVector xsvsetnewve(spPlugin *plugin, short *data, long length)
{
    spSVector x;

    x = xsvallocve(plugin, length);
    memcpy(x->data, data, sizeof(short) * length);
    
    return x;
}

spLVector xlvsetnewve(spPlugin *plugin, long *data, long length)
{
    spLVector x;

    x = xlvallocve(plugin, length);
    memcpy(x->data, data, sizeof(long) * length);
    
    return x;
}

spFVector xfvsetnewve(spPlugin *plugin, float *data, long length)
{
    spFVector x;

    x = xfvallocve(plugin, length);
    memcpy(x->data, data, sizeof(float) * length);
    
    return x;
}

spDVector xdvsetnewve(spPlugin *plugin, double *data, long length)
{
    spDVector x;

    x = xdvallocve(plugin, length);
    memcpy(x->data, data, sizeof(double) * length);
    
    return x;
}

spSVector xsvsetnew(short *data, long length)
{
    return xsvsetnewve(NULL, data, length);
}

spLVector xlvsetnew(long *data, long length)
{
    return xlvsetnewve(NULL, data, length);
}

spFVector xfvsetnew(float *data, long length)
{
    return xfvsetnewve(NULL, data, length);
}

spDVector xdvsetnew(double *data, long length)
{
    return xdvsetnewve(NULL, data, length);
}

spBool svsetval(spSVector x, long index, short value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->rset != NULL) {
	return flist->rset(x->instance, index, (char *)&value);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    x->data[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool lvsetval(spLVector x, long index, long value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->rset != NULL) {
	return flist->rset(x->instance, index, (char *)&value);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    x->data[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool fvsetval(spFVector x, long index, float value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->rset != NULL) {
	return flist->rset(x->instance, index, (char *)&value);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    x->data[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool dvsetval(spDVector x, long index, double value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->rset != NULL) {
	return flist->rset(x->instance, index, (char *)&value);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    x->data[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool svsetival(spSVector x, long index, short value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) return SP_FALSE;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->iset != NULL) {
	return flist->iset(x->instance, index, (char *)&value);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    x->imag[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool lvsetival(spLVector x, long index, long value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) return SP_FALSE;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->iset != NULL) {
	return flist->iset(x->instance, index, (char *)&value);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    x->imag[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool fvsetival(spFVector x, long index, float value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) return SP_FALSE;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->iset != NULL) {
	return flist->iset(x->instance, index, (char *)&value);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    x->imag[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}

spBool dvsetival(spDVector x, long index, double value)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) return SP_FALSE;
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->iset != NULL) {
	return flist->iset(x->instance, index, (char *)&value);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    x->imag[index] = value;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return SP_TRUE;
}
