/*
 *	vmath.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifdef USE_SPL

#define nsp_UsesVector
#include <nsp.h>
#endif

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/vector.h>
#include <sp/voperate.h>

#ifdef SP_USE_VECTOR_ENGINE
#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>
#endif

void svabs(spSVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->abs != NULL) {
	flist->abs(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspwbAbs1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = ABS(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = (short)CABS(x->data[k], x->imag[k]);
	}
	svifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvabs(spLVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->abs != NULL) {
	flist->abs(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = ABS(x->data[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = (long)CABS(x->data[k], x->imag[k]);
	}
	lvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvabs(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->abs != NULL) {
	flist->abs(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspsbAbs1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = FABS(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = (float)CABS(x->data[k], x->imag[k]);
	}
	fvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvabs(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->abs != NULL) {
	flist->abs(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspdbAbs1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = FABS(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = CABS(x->data[k], x->imag[k]);
	}
	dvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvabs(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svabs(y);

    return y;
}

spLVector xlvabs(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvabs(y);

    return y;
}

spFVector xfvabs(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvabs(y);

    return y;
}

spDVector xdvabs(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvabs(y);

    return y;
}

void svsqrt(spSVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sqrt != NULL) {
	flist->sqrt(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspwbSqrt1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = (short)sqrt((double)x->data[k]);
	}
#endif
    } else {
	double r, xr;
	
	for (k = 0; k < x->length; k++) {
	    r = CABS(x->data[k], x->imag[k]);
	    xr = x->data[k];
	    x->data[k] = (short)sqrt((r + xr) / 2.0);
	    x->imag[k] = (short)sqrt((r - xr) / 2.0);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvsqrt(spLVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sqrt != NULL) {
	flist->sqrt(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = (short)sqrt((double)x->data[k]);
	}
    } else {
	double r, xr;
	
	for (k = 0; k < x->length; k++) {
	    r = CABS(x->data[k], x->imag[k]);
	    xr = x->data[k];
	    x->data[k] = (long)sqrt((r + xr) / 2.0);
	    x->imag[k] = (long)sqrt((r - xr) / 2.0);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvsqrt(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sqrt != NULL) {
	flist->sqrt(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspsbSqrt1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sqrtf(x->data[k]);
	}
#endif
    } else {
	float r, xr;
	
	for (k = 0; k < x->length; k++) {
	    r = (float)CABS(x->data[k], x->imag[k]);
	    xr = x->data[k];
	    x->data[k] = sqrtf((r + xr) / 2.0f);
            if (x->imag[k] < 0.0f) {
                x->imag[k] = -sqrtf((r - xr) / 2.0f);
            } else {
                x->imag[k] = sqrtf((r - xr) / 2.0f);
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsqrt(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sqrt != NULL) {
	flist->sqrt(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspdbSqrt1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sqrt(x->data[k]);
	}
#endif
    } else {
	double r, xr;
	
	for (k = 0; k < x->length; k++) {
	    r = CABS(x->data[k], x->imag[k]);
	    xr = x->data[k];
	    x->data[k] = sqrt((r + xr) / 2.0);
            if (x->imag[k] < 0.0) {
                x->imag[k] = -sqrt((r - xr) / 2.0);
            } else {
                x->imag[k] = sqrt((r - xr) / 2.0);
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvsqrt(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svsqrt(y);

    return y;
}

spLVector xlvsqrt(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvsqrt(y);

    return y;
}

spFVector xfvsqrt(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsqrt(y);

    return y;
}

spDVector xdvsqrt(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsqrt(y);

    return y;
}

void svsquare(spSVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->square != NULL) {
	flist->square(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = SQUARE(x->data[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = CSQUARE(x->data[k], x->imag[k]);
	}
	svifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvsquare(spLVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->square != NULL) {
	flist->square(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = SQUARE(x->data[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = CSQUARE(x->data[k], x->imag[k]);
	}
	lvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvsquare(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->square != NULL) {
	flist->square(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspsbSqr1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = SQUARE(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = CSQUARE(x->data[k], x->imag[k]);
	}
	fvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsquare(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->square != NULL) {
	flist->square(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspdbSqr1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = SQUARE(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = CSQUARE(x->data[k], x->imag[k]);
	}
	dvifree(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvsquare(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svsquare(y);

    return y;
}

spLVector xlvsquare(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvsquare(y);

    return y;
}

spFVector xfvsquare(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsquare(y);

    return y;
}

spDVector xdvsquare(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsquare(y);

    return y;
}

void svsign(spSVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sign != NULL) {
	flist->sign(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] > 0) {
		x->data[k] = 1;
	    } else if (x->data[k] == 0) {
		x->data[k] = 0;
	    } else {
		x->data[k] = -1;
	    }
	}
    } else {
	double value;
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] != 0 || x->imag[k] != 0) {
		value = CABS(x->data[k], x->imag[k]);
		x->data[k] = (short)((double)x->data[k] / value);
		x->imag[k] = (short)((double)x->imag[k] / value);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvsign(spLVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sign != NULL) {
	flist->sign(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] > 0) {
		x->data[k] = 1;
	    } else if (x->data[k] == 0) {
		x->data[k] = 0;
	    } else {
		x->data[k] = -1;
	    }
	}
    } else {
	double value;
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] != 0 || x->imag[k] != 0) {
		value = CABS(x->data[k], x->imag[k]);
		x->data[k] = (long)((double)x->data[k] / value);
		x->imag[k] = (long)((double)x->imag[k] / value);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvsign(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sign != NULL) {
	flist->sign(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] > 0.0) {
		x->data[k] = 1.0;
	    } else if (x->data[k] == 0.0) {
		x->data[k] = 0.0;
	    } else {
		x->data[k] = -1.0;
	    }
	}
    } else {
	double value;
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] != 0.0 || x->imag[k] != 0.0) {
		value = CABS(x->data[k], x->imag[k]);
		x->data[k] = (float)((double)x->data[k] / value);
		x->imag[k] = (float)((double)x->imag[k] / value);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsign(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sign != NULL) {
	flist->sign(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] > 0.0) {
		x->data[k] = 1.0;
	    } else if (x->data[k] == 0.0) {
		x->data[k] = 0.0;
	    } else {
		x->data[k] = -1.0;
	    }
	}
    } else {
	double value;
	for (k = 0; k < x->length; k++) {
	    if (x->data[k] != 0.0 || x->imag[k] != 0.0) {
		value = CABS(x->data[k], x->imag[k]);
		x->data[k] = (double)x->data[k] / value;
		x->imag[k] = (double)x->imag[k] / value;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvsign(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svsign(y);

    return y;
}

spLVector xlvsign(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvsign(y);

    return y;
}

spFVector xfvsign(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsign(y);

    return y;
}

spDVector xdvsign(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsign(y);

    return y;
}

void svcumsum(spSVector x)
{
    long k;
    long sum;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cumsum != NULL) {
	flist->cumsum(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0, sum = 0; k < x->length; k++) {
	sum += x->data[k];
	x->data[k] = (short)sum;
    }
    if (x->imag != NULL) {
	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += x->imag[k];
	    x->imag[k] = (short)sum;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvcumsum(spLVector x)
{
    long k;
    long sum;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cumsum != NULL) {
	flist->cumsum(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0, sum = 0; k < x->length; k++) {
	sum += x->data[k];
	x->data[k] = sum;
    }
    if (x->imag != NULL) {
	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += x->imag[k];
	    x->imag[k] = sum;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvcumsum(spFVector x)
{
    long k;
    float sum;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cumsum != NULL) {
	flist->cumsum(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0, sum = 0.0; k < x->length; k++) {
	sum += x->data[k];
	x->data[k] = sum;
    }
    if (x->imag != NULL) {
	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->imag[k];
	    x->imag[k] = sum;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcumsum(spDVector x)
{
    long k;
    double sum;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cumsum != NULL) {
	flist->cumsum(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0, sum = 0.0; k < x->length; k++) {
	sum += x->data[k];
	x->data[k] = sum;
    }
    if (x->imag != NULL) {
	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->imag[k];
	    x->imag[k] = sum;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvcumsum(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svcumsum(y);

    return y;
}

spLVector xlvcumsum(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvcumsum(y);

    return y;
}

spFVector xfvcumsum(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvcumsum(y);

    return y;
}

spDVector xdvcumsum(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvcumsum(y);

    return y;
}

void svcumprod(spSVector x)
{
    long k;
    
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cumprod != NULL) {
	flist->cumprod(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    if (x->imag != NULL) {
	long xr, xi;
	
	for (k = 1; k < x->length; k++) {
	    xr = x->data[k] * x->data[k-1] - x->imag[k] * x->imag[k-1];
	    xi = x->data[k] * x->imag[k-1] + x->imag[k] * x->data[k-1];
	    x->data[k] = (short)xr;
	    x->imag[k] = (short)xi;
	}
    } else {
	long prod;

	for (k = 1, prod = x->data[0]; k < x->length; k++) {
	    prod *= x->data[k];
	    x->data[k] = (short)prod;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvcumprod(spLVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cumprod != NULL) {
	flist->cumprod(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    if (x->imag != NULL) {
	long xr, xi;
	
	for (k = 1; k < x->length; k++) {
	    xr = x->data[k] * x->data[k-1] - x->imag[k] * x->imag[k-1];
	    xi = x->data[k] * x->imag[k-1] + x->imag[k] * x->data[k-1];
	    x->data[k] = xr;
	    x->imag[k] = xi;
	}
    } else {
	long prod;
	
	for (k = 1, prod = x->data[0]; k < x->length; k++) {
	    prod *= x->data[k];
	    x->data[k] = prod;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvcumprod(spFVector x)
{
    long k;
    
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cumprod != NULL) {
	flist->cumprod(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag != NULL) {
	float xr, xi;
	
	for (k = 1; k < x->length; k++) {
	    xr = x->data[k] * x->data[k-1] - x->imag[k] * x->imag[k-1];
	    xi = x->data[k] * x->imag[k-1] + x->imag[k] * x->data[k-1];
	    x->data[k] = xr;
	    x->imag[k] = xi;
	}
    } else {
	float prod;

	for (k = 1, prod = x->data[0]; k < x->length; k++) {
	    prod *= x->data[k];
	    x->data[k] = prod;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcumprod(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cumprod != NULL) {
	flist->cumprod(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag != NULL) {
	double xr, xi;
	
	for (k = 1; k < x->length; k++) {
	    xr = x->data[k] * x->data[k-1] - x->imag[k] * x->imag[k-1];
	    xi = x->data[k] * x->imag[k-1] + x->imag[k] * x->data[k-1];
	    x->data[k] = xr;
	    x->imag[k] = xi;
	}
    } else {
	double prod;

	for (k = 1, prod = x->data[0]; k < x->length; k++) {
	    prod *= x->data[k];
	    x->data[k] = prod;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvcumprod(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svcumprod(y);

    return y;
}

spLVector xlvcumprod(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvcumprod(y);

    return y;
}

spFVector xfvcumprod(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvcumprod(y);

    return y;
}

spDVector xdvcumprod(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvcumprod(y);

    return y;
}

void svcodiff(spSVector x, double coef)
{
    long k;
    long len;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->codiff != NULL) {
	flist->codiff(x->instance, x->length, coef);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    len = x->length - 1;

    for (k = 0; k < len; k++) {
	x->data[k] = x->data[k + 1] - (short)spRound(coef * (double)x->data[k]);
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k + 1] - (short)spRound(coef * (double)x->imag[k]);
	}
    }
    
    x->data[len] = -(short)spRound(coef * (double)x->data[len]);
    if (x->imag != NULL) {
	x->imag[len] = -(short)spRound(coef * (double)x->imag[len]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvcodiff(spLVector x, double coef)
{
    long k;
    long len;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->codiff != NULL) {
	flist->codiff(x->instance, x->length, coef);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    len = x->length - 1;

    for (k = 0; k < len; k++) {
	x->data[k] = x->data[k + 1] - (long)spRound(coef * (double)x->data[k]);
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k + 1] - (long)spRound(coef * (double)x->imag[k]);
	}
    }
    
    x->data[len] = -(long)spRound(coef * (double)x->data[len]);
    if (x->imag != NULL) {
	x->imag[len] = -(long)spRound(coef * (double)x->imag[len]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvcodiff(spFVector x, double coef)
{
    long k;
    long len;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->codiff != NULL) {
	flist->codiff(x->instance, x->length, coef);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    len = x->length - 1;

    for (k = 0; k < len; k++) {
	x->data[k] = x->data[k + 1] - (float)(coef * (double)x->data[k]);
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k + 1] - (float)(coef * (double)x->imag[k]);
	}
    }
    
    x->data[len] = -(float)(coef * (double)x->data[k]);
    if (x->imag != NULL) {
	x->imag[len] = -(float)(coef * (double)x->imag[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcodiff(spDVector x, double coef)
{
    long k;
    long len;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->codiff != NULL) {
	flist->codiff(x->instance, x->length, coef);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    len = x->length - 1;

    for (k = 0; k < len; k++) {
	x->data[k] = x->data[k + 1] - coef * x->data[k];
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k + 1] - coef * x->imag[k];
	}
    }
    
    x->data[len] *= -coef;
    if (x->imag != NULL) {
	x->imag[len] *= -coef;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvcodiff(spSVector x, double coef)
{
    spSVector y;

    if (x->length <= 1) {
	y = xsvnullul(svgetplugin(x), spIsFalse(svislocked(x)));
	return y;
    }
    
    y = xsvclone(x);
    svcodiff(y, coef);
    y->length = x->length - 1;

    return y;
}

spLVector xlvcodiff(spLVector x, double coef)
{
    spLVector y;

    if (x->length <= 1) {
	y = xlvnullul(lvgetplugin(x), spIsFalse(lvislocked(x)));
	return y;
    }
    
    y = xlvclone(x);
    lvcodiff(y, coef);
    y->length = x->length - 1;

    return y;
}

spFVector xfvcodiff(spFVector x, double coef)
{
    spFVector y;

    if (x->length <= 1) {
	y = xfvnullul(fvgetplugin(x), spIsFalse(fvislocked(x)));
	return y;
    }
    
    y = xfvclone(x);
    fvcodiff(y, coef);
    y->length = x->length - 1;

    return y;
}

spDVector xdvcodiff(spDVector x, double coef)
{
    spDVector y;

    if (x->length <= 1) {
	y = xdvnullul(dvgetplugin(x), spIsFalse(dvislocked(x)));
	return y;
    }
    
    y = xdvclone(x);
    dvcodiff(y, coef);
    y->length = x->length - 1;

    return y;
}

long svsum(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sum != NULL) {
	return (long)flist->sum(x->instance, x->length);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += (long)x->data[k];
	}

	return sum;
    }
}

long lvsum(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sum != NULL) {
	return (long)flist->sum(x->instance, x->length);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += x->data[k];
	}

	return sum;
    }
}

float fvsum(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sum != NULL) {
	return (float)flist->sum(x->instance, x->length);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

#ifdef USE_SPL
    return nspsSum(x->data, x->length);
#else
    {
	long k;
	float sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->data[k];
	}

	return sum;
    }
#endif
}

double dvsum(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sum != NULL) {
	return flist->sum(x->instance, x->length);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

#ifdef USE_SPL
    return nspdSum(x->data, x->length);
#else
    {
	long k;
	double sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->data[k];
	}

	return sum;
    }
#endif
}

long svisum(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->isum != NULL) {
	return (long)flist->isum(x->instance, x->length);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += (long)x->imag[k];
	}

	return sum;
    }
}

long lvisum(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->isum != NULL) {
	return (long)flist->isum(x->instance, x->length);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += x->imag[k];
	}

	return sum;
    }
}

float fvisum(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->isum != NULL) {
	return (float)flist->isum(x->instance, x->length);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

#ifdef USE_SPL
    return nspsSum(x->imag, x->length);
#else
    {
	long k;
	float sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->imag[k];
	}

	return sum;
    }
#endif
}

double dvisum(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->isum != NULL) {
	return flist->isum(x->instance, x->length);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

#ifdef USE_SPL
    return nspdSum(x->imag, x->length);
#else
    {
	long k;
	double sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += x->imag[k];
	}

	return sum;
    }
#endif
}

long svsqsum(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sqsum != NULL) {
	return (long)flist->sqsum(x->instance, x->length);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += SQUARE((long)x->data[k]);
	}
    
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		sum += SQUARE((long)x->imag[k]);
	    }
	}

	return sum;
    }
}

long lvsqsum(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sqsum != NULL) {
	return (long)flist->sqsum(x->instance, x->length);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    {
	long k;
	long sum;

	for (k = 0, sum = 0; k < x->length; k++) {
	    sum += SQUARE(x->data[k]);
	}

	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		sum += SQUARE(x->imag[k]);
	    }
	}

	return sum;
    }
}

float fvsqsum(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sqsum != NULL) {
	return (float)flist->sqsum(x->instance, x->length);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    {
	long k;
	float sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += SQUARE(x->data[k]);
	}

	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		sum += SQUARE(x->imag[k]);
	    }
	}

	return sum;
    }
}

double dvsqsum(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sqsum != NULL) {
	return flist->sqsum(x->instance, x->length);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    {
	long k;
	double sum;

	for (k = 0, sum = 0.0; k < x->length; k++) {
	    sum += SQUARE(x->data[k]);
	}

	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		sum += SQUARE(x->imag[k]);
	    }
	}

	return sum;
    }
}

long svabssum(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->abssum != NULL) {
	return (long)flist->abssum(x->instance, x->length);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    {
	long k;
	long sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0; k < x->length; k++) {
		sum += (long)CABS(x->data[k], x->imag[k]);
	    }
	} else {
	    for (k = 0, sum = 0; k < x->length; k++) {
		sum += FABS((long)x->data[k]);
	    }
	}

	return sum;
    }
}

long lvabssum(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->abssum != NULL) {
	return (long)flist->abssum(x->instance, x->length);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    {
	long k;
	long sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0; k < x->length; k++) {
		sum += (long)CABS(x->data[k], x->imag[k]);
	    }
	} else {
	    for (k = 0, sum = 0; k < x->length; k++) {
		sum += FABS(x->data[k]);
	    }
	}

	return sum;
    }
}

float fvabssum(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->abssum != NULL) {
	return (float)flist->abssum(x->instance, x->length);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    {
	long k;
	float sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0f; k < x->length; k++) {
		sum += (float)CABS(x->data[k], x->imag[k]);
	    }
	} else {
	    for (k = 0, sum = 0.0f; k < x->length; k++) {
		sum += FABS(x->data[k]);
	    }
	}

	return sum;
    }
}

double dvabssum(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->abssum != NULL) {
	return flist->abssum(x->instance, x->length);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    {
	long k;
	double sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
		sum += CABS(x->data[k], x->imag[k]);
	    }
	} else {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
		sum += FABS(x->data[k]);
	    }
	}

	return sum;
    }
}

long svprod(spSVector x)
{
    long k;
    long prod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->prod != NULL) {
	return (long)flist->prod(x->instance, x->length);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 1, prod = x->data[0]; k < x->length; k++) {
        prod *= x->data[k];
    }

    return prod;
}

long lvprod(spLVector x)
{
    long k;
    long prod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->prod != NULL) {
	return (long)flist->prod(x->instance, x->length);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 1, prod = x->data[0]; k < x->length; k++) {
        prod *= x->data[k];
    }

    return prod;
}

float fvprod(spFVector x)
{
    long k;
    float prod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->prod != NULL) {
	return (float)flist->prod(x->instance, x->length);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 1, prod = x->data[0]; k < x->length; k++) {
        prod *= x->data[k];
    }

    return prod;
}

double dvprod(spDVector x)
{
    long k;
    double prod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->prod != NULL) {
	return flist->prod(x->instance, x->length);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 1, prod = x->data[0]; k < x->length; k++) {
        prod *= x->data[k];
    }

    return prod;
}

long svriprod(spSVector x, long *oiprod)
{
    long k;
    long r, i;
    long rprod, iprod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
        rprod = svprod(x);
        if (oiprod != NULL) *oiprod = 0;
        return rprod;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->riprod != NULL) {
        double diprod = 0.0;
	rprod = (long)flist->riprod(x->instance, x->length, &diprod);
        if (oiprod != NULL) *oiprod = (long)diprod;
        return rprod;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    rprod = x->data[0]; iprod = x->imag[0];
    
    for (k = 1; k < x->length; k++) {
        r = rprod * x->data[k] - iprod * x->imag[k];
        i = rprod * x->imag[k] + iprod * x->data[k];
        rprod = r; iprod = i;
    }

    if (oiprod != NULL) *oiprod = iprod;
    
    return rprod;
}

long lvriprod(spLVector x, long *oiprod)
{
    long k;
    long r, i;
    long rprod, iprod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
        rprod = lvprod(x);
        if (oiprod != NULL) *oiprod = 0;
        return rprod;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->riprod != NULL) {
        double diprod = 0.0;
	rprod = (long)flist->riprod(x->instance, x->length, &diprod);
        if (oiprod != NULL) *oiprod = (long)diprod;
        return rprod;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    rprod = x->data[0]; iprod = x->imag[0];
    
    for (k = 1; k < x->length; k++) {
        r = rprod * x->data[k] - iprod * x->imag[k];
        i = rprod * x->imag[k] + iprod * x->data[k];
        rprod = r; iprod = i;
    }

    if (oiprod != NULL) *oiprod = iprod;
    
    return rprod;
}

float fvriprod(spFVector x, float *oiprod)
{
    long k;
    float r, i;
    float rprod, iprod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
        rprod = fvprod(x);
        if (oiprod != NULL) *oiprod = 0.0f;
        return rprod;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->riprod != NULL) {
        double diprod = 0.0;
	rprod = (float)flist->riprod(x->instance, x->length, &diprod);
        if (oiprod != NULL) *oiprod = (float)diprod;
        return rprod;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    rprod = x->data[0]; iprod = x->imag[0];
    
    for (k = 1; k < x->length; k++) {
        r = rprod * x->data[k] - iprod * x->imag[k];
        i = rprod * x->imag[k] + iprod * x->data[k];
        rprod = r; iprod = i;
    }

    if (oiprod != NULL) *oiprod = iprod;
    
    return rprod;
}

double dvriprod(spDVector x, double *oiprod)
{
    long k;
    double r, i;
    double rprod, iprod;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
        rprod = dvprod(x);
        if (oiprod != NULL) *oiprod = 0.0;
        return rprod;
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->riprod != NULL) {
	rprod = flist->riprod(x->instance, x->length, &iprod);
        if (oiprod != NULL) *oiprod = iprod;
        return rprod;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    rprod = x->data[0]; iprod = x->imag[0];
    
    for (k = 1; k < x->length; k++) {
        r = rprod * x->data[k] - iprod * x->imag[k];
        i = rprod * x->imag[k] + iprod * x->data[k];
        rprod = r; iprod = i;
    }

    if (oiprod != NULL) *oiprod = iprod;
    
    return rprod;
}

/* p = 0: default euclid norm (p = 2), p = -1: infinity norm */
double svnorm(spSVector x, long p)
{
    if (p < -1) return -1.0;
    
    if (p == 1) {
        return svabssum(x);
    } else if (p == 0 || p == 2) {
        return sqrt(svsqsum(x));
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    {
        spVectorPluginInternalFuncList *flist;
    
        if (svisplugincomputable(x) == SP_TRUE
            && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->norm != NULL) {
            return flist->norm(x->instance, x->length, p);
        }

        if (svislocked(x) == SP_FALSE) {
            svsync(x);
        }
    }
#endif

    if (p == -1) {
        spSVector ax;
        short value;

        ax = xsvabs(x);
        value = svmax(ax, NULL);
        xsvfree(ax);
        return value;
    } else {
	long k;
        double re, im;
	double sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k]; im = x->imag[k];
                sum += pow(re * re + im * im, (double)p / 2.0);
	    }
	} else {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k];
		sum += pow(re * re, (double)p / 2.0);
	    }
	}

	return pow(sum, 1.0 / (double)p);
    }
}

double lvnorm(spLVector x, long p)
{
    if (p < -1) return -1.0;
    
    if (p == 1) {
        return lvabssum(x);
    } else if (p == 0 || p == 2) {
        return sqrt(lvsqsum(x));
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    {
        spVectorPluginInternalFuncList *flist;
    
        if (lvisplugincomputable(x) == SP_TRUE
            && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->norm != NULL) {
            return flist->norm(x->instance, x->length, p);
        }

        if (lvislocked(x) == SP_FALSE) {
            lvsync(x);
        }
    }
#endif

    if (p == -1) {
        spLVector ax;
        long value;

        ax = xlvabs(x);
        value = lvmax(ax, NULL);
        xlvfree(ax);
        return value;
    } else {
	long k;
        double re, im;
	double sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k]; im = x->imag[k];
                sum += pow(re * re + im * im, (double)p / 2.0);
	    }
	} else {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k];
		sum += pow(re * re, (double)p / 2.0);
	    }
	}

	return pow(sum, 1.0 / (double)p);
    }
}

float fvnorm(spFVector x, long p)
{
    if (p < -1) return -1.0;
    
    if (p == 1) {
        return fvabssum(x);
    } else if (p == 0 || p == 2) {
        return sqrtf(fvsqsum(x));
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    {
        spVectorPluginInternalFuncList *flist;
    
        if (fvisplugincomputable(x) == SP_TRUE
            && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->norm != NULL) {
            return (float)flist->norm(x->instance, x->length, p);
        }

        if (fvislocked(x) == SP_FALSE) {
            fvsync(x);
        }
    }
#endif

    if (p == -1) {
        spFVector ax;
        float value;

        ax = xfvabs(x);
        value = fvmax(ax, NULL);
        xfvfree(ax);
        return value;
    } else {
	long k;
        float re, im;
	float sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0f; k < x->length; k++) {
                re = (float)x->data[k]; im = (float)x->imag[k];
                sum += powf(re * re + im * im, (float)p / 2.0f);
	    }
	} else {
	    for (k = 0, sum = 0.0f; k < x->length; k++) {
                re = (float)x->data[k];
		sum += powf(re * re, (float)p / 2.0f);
	    }
	}

	return powf(sum, 1.0f / (float)p);
    }
}

double dvnorm(spDVector x, long p)
{
    if (p < -1) return -1.0;
    
    if (p == 1) {
        return dvabssum(x);
    } else if (p == 0 || p == 2) {
        return sqrt(dvsqsum(x));
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    {
        spVectorPluginInternalFuncList *flist;
    
        if (dvisplugincomputable(x) == SP_TRUE
            && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->norm != NULL) {
            return flist->norm(x->instance, x->length, p);
        }

        if (dvislocked(x) == SP_FALSE) {
            dvsync(x);
        }
    }
#endif

    if (p == -1) {
        spDVector ax;
        double value;

        ax = xdvabs(x);
        value = dvmax(ax, NULL);
        xdvfree(ax);
        return value;
    } else {
	long k;
        double re, im;
	double sum;

	if (x->imag != NULL) {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k]; im = x->imag[k];
                sum += pow(re * re + im * im, (double)p / 2.0);
	    }
	} else {
	    for (k = 0, sum = 0.0; k < x->length; k++) {
                re = x->data[k];
		sum += pow(re * re, (double)p / 2.0);
	    }
	}

	return pow(sum, 1.0 / (double)p);
    }
}

long svdot(spSVector x, spSVector y)
{
    long k;
    long a = 0;

    if (x == NODATA || y == NODATA) {
	a = 0;
    } else if (x->length != y->length) {
	fprintf(stderr, "svdot: vector length must agree\n");
	exit(1);
    } else {
#ifdef SP_USE_VECTOR_ENGINE
        spVectorPluginInternalFuncList *flist;
        
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE
            && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->dot != NULL) {
	    return (long)flist->dot(x->instance, y->instance, x->length);
	}

        if (svislocked(x) == SP_FALSE) {
            svsync(x);
        }
#endif
    
	for (k = 0; k < x->length; k++) {
	    a += x->data[k] * y->data[k];
	}
    }
    
    return a;
}

long lvdot(spLVector x, spLVector y)
{
    long k;
    long a = 0;

    if (x == NODATA || y == NODATA) {
	a = 0;
    } else if (x->length != y->length) {
	fprintf(stderr, "lvdot: vector length must agree\n");
	exit(1);
    } else {
#ifdef SP_USE_VECTOR_ENGINE
        spVectorPluginInternalFuncList *flist;
        
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE
            && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->dot != NULL) {
	    return (long)flist->dot(x->instance, y->instance, x->length);
	}

        if (lvislocked(x) == SP_FALSE) {
            lvsync(x);
        }
#endif
    
	for (k = 0; k < x->length; k++) {
	    a += x->data[k] * y->data[k];
	}
    }
    
    return a;
}

float fvdot(spFVector x, spFVector y)
{
    long k;
    float a = 0.0;

    if (x == NODATA || y == NODATA) {
	a = 0.0;
    } else if (x->length != y->length) {
	fprintf(stderr, "fvdot: vector length must agree\n");
	exit(1);
    } else {
#ifdef SP_USE_VECTOR_ENGINE
        spVectorPluginInternalFuncList *flist;
        
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE
            && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->dot != NULL) {
	    return (float)flist->dot(x->instance, y->instance, x->length);
	}

        if (fvislocked(x) == SP_FALSE) {
            fvsync(x);
        }
#endif
    
#ifdef USE_SPL
	a = nspsDotProd(x->data, y->data, x->length);
#else	
	for (k = 0; k < x->length; k++) {
	    a += x->data[k] * y->data[k];
	}
#endif
    }
    
    return a;
}

double dvdot(spDVector x, spDVector y)
{
    long k;
    double a = 0.0;

    if (x == NODATA || y == NODATA) {
	a = 0.0;
    } else if (x->length != y->length) {
	fprintf(stderr, "dvdot: vector length must agree\n");
	exit(1);
    } else {
#ifdef SP_USE_VECTOR_ENGINE
        spVectorPluginInternalFuncList *flist;
        
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE
            && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->dot != NULL) {
	    return flist->dot(x->instance, y->instance, x->length);
	}

        if (dvislocked(x) == SP_FALSE) {
            dvsync(x);
        }
#endif
    
#ifdef USE_SPL
	a = nspdDotProd(x->data, y->data, x->length);
#else	
	for (k = 0; k < x->length; k++) {
	    a += x->data[k] * y->data[k];
	}
#endif
    }
    
    return a;
}

void svcplxdot(spSVector x, spSVector y, long *re, long *im)
{
    long k;

    if (x == NODATA || y == NODATA) {
	*re = *im = 0;
    } else if (x->length != y->length) {
	fprintf(stderr, "svcplxdot: vector length must agree\n");
	exit(1);
    } else {
        if (x->imag == NULL && y->imag == NULL) {
            *re = svdot(x, y);
            *im = 0;
            return;
        }
#ifdef SP_USE_VECTOR_ENGINE
        {
            spVectorPluginInternalFuncList *flist;
        
            if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE
                && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxdot != NULL) {
                double dre, dim;
                flist->cplxdot(x->instance, y->instance, x->length, &dre, &dim);
                *re = (long)dre;
                *im = (long)dim;
                return;
            }

            if (svislocked(x) == SP_FALSE) {
                svsync(x);
            }
        }
#endif

        *re = *im = 0;
	for (k = 0; k < x->length; k++) {
            if (x->imag != NULL && y->imag != NULL) {
                *re += x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
                *im += -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
            } else if (x->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += x->imag[k] * y->data[k];
            } else if (y->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += -x->data[k] * y->imag[k];
            }
	}
    }
    
    return;
}

void lvcplxdot(spLVector x, spLVector y, long *re, long *im)
{
    long k;

    if (x == NODATA || y == NODATA) {
	*re = *im = 0;
    } else if (x->length != y->length) {
	fprintf(stderr, "lvcplxdot: vector length must agree\n");
	exit(1);
    } else {
        if (x->imag == NULL && y->imag == NULL) {
            *re = lvdot(x, y);
            *im = 0;
            return;
        }
#ifdef SP_USE_VECTOR_ENGINE
        {
            spVectorPluginInternalFuncList *flist;
        
            if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE
                && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxdot != NULL) {
                double dre, dim;
                flist->cplxdot(x->instance, y->instance, x->length, &dre, &dim);
                *re = (long)dre;
                *im = (long)dim;
                return;
            }

            if (lvislocked(x) == SP_FALSE) {
                lvsync(x);
            }
        }
#endif

        *re = *im = 0;
	for (k = 0; k < x->length; k++) {
            if (x->imag != NULL && y->imag != NULL) {
                *re += x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
                *im += -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
            } else if (x->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += x->imag[k] * y->data[k];
            } else if (y->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += -x->data[k] * y->imag[k];
            }
	}
    }
    
    return;
}

void fvcplxdot(spFVector x, spFVector y, float *re, float *im)
{
    long k;

    if (x == NODATA || y == NODATA) {
	*re = *im = 0.0f;
    } else if (x->length != y->length) {
	fprintf(stderr, "fvcplxdot: vector length must agree\n");
	exit(1);
    } else {
        if (x->imag == NULL && y->imag == NULL) {
            *re = fvdot(x, y);
            *im = 0.0f;
            return;
        }
#ifdef SP_USE_VECTOR_ENGINE
        {
            spVectorPluginInternalFuncList *flist;
            double red, imd;
        
            if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE
                && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxdot != NULL) {
                flist->cplxdot(x->instance, y->instance, x->length, &red, &imd);
                *re = (float)red;
                *im = (float)imd;
                return;
            }

            if (fvislocked(x) == SP_FALSE) {
                fvsync(x);
            }
        }
#endif

        *re = *im = 0.0f;
	for (k = 0; k < x->length; k++) {
            if (x->imag != NULL && y->imag != NULL) {
                *re += x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
                *im += -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
            } else if (x->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += x->imag[k] * y->data[k];
            } else if (y->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += -x->data[k] * y->imag[k];
            }
	}
    }
    
    return;
}

void dvcplxdot(spDVector x, spDVector y, double *re, double *im)
{
    long k;

    if (x == NODATA || y == NODATA) {
	*re = *im = 0.0;
    } else if (x->length != y->length) {
	fprintf(stderr, "dvcplxdot: vector length must agree\n");
	exit(1);
    } else {
        if (x->imag == NULL && y->imag == NULL) {
            *re = dvdot(x, y);
            *im = 0.0;
            return;
        }
#ifdef SP_USE_VECTOR_ENGINE
        {
            spVectorPluginInternalFuncList *flist;
        
            if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE
                && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxdot != NULL) {
                flist->cplxdot(x->instance, y->instance, x->length, re, im);
                return;
            }

            if (dvislocked(x) == SP_FALSE) {
                dvsync(x);
            }
        }
#endif

        *re = *im = 0.0;
	for (k = 0; k < x->length; k++) {
            if (x->imag != NULL && y->imag != NULL) {
                *re += x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
                *im += -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
            } else if (x->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += x->imag[k] * y->data[k];
            } else if (y->imag != NULL) {
                *re += x->data[k] * y->data[k];
                *im += -x->data[k] * y->imag[k];
            }
	}
    }
    
    return;
}

double svvar(spSVector x, double *mean)
{
    long k;
    double m, var;
    double value;

    if (x->length <= 1) return 0.0;

    m = svmean(x);
    if (mean != NULL) *mean = m;

#if 0
    for (k = 0, var = 0.0; k < x->length; k++) {
	value = (double)x->data[k] - m;
	var += SQUARE(value);
    }
    var /= (double)(x->length - 1);
#else
    var = ((double)svsqsum(x) - SQUARE(m) * (double)x->length) / (double)(x->length - 1);
#endif

    return var;
}

double lvvar(spLVector x, double *mean)
{
    long k;
    double m, var;
    double value;

    if (x->length <= 1) return 0.0;

    m = lvmean(x);
    if (mean != NULL) *mean = m;

#if 0
    for (k = 0, var = 0.0; k < x->length; k++) {
	value = (double)x->data[k] - m;
	var += SQUARE(value);
    }
    var /= (double)(x->length - 1);
#else
    var = ((double)lvsqsum(x) - SQUARE(m) * (double)x->length) / (double)(x->length - 1);
#endif

    return var;
}

float fvvar(spFVector x, float *mean)
{
    long k;
    float m, var;
    float value;

    if (x->length <= 1) return 0.0f;

    m = fvmean(x);
    if (mean != NULL) *mean = m;
    
#if 0
    for (k = 0, var = 0.0; k < x->length; k++) {
	value = x->data[k] - m;
	var += SQUARE(value);
    }
    var /= (float)(x->length - 1);
#else
    var = (fvsqsum(x) - SQUARE(m) * (float)x->length) / (float)(x->length - 1);
#endif

    return var;
}

double dvvar(spDVector x, double *mean)
{
    long k;
    double m, var;
    double value;

    if (x->length <= 1) return 0.0;

    m = dvmean(x);
    if (mean != NULL) *mean = m;

#if 0
    for (k = 0, var = 0.0; k < x->length; k++) {
	value = x->data[k] - m;
	var += SQUARE(value);
    }
    var /= (double)(x->length - 1);
#else
    var = (dvsqsum(x) - SQUARE(m) * (double)x->length) / (double)(x->length - 1);
#endif

    return var;
}

double svstd(spSVector x, double *mean)
{
    return sqrt(svvar(x, mean));
}

double lvstd(spLVector x, double *mean)
{
    return sqrt(lvvar(x, mean));
}

float fvstd(spFVector x, float *mean)
{
    return (float)sqrt(fvvar(x, mean));
}

double dvstd(spDVector x, double *mean)
{
    return sqrt(dvvar(x, mean));
}

short svmin(spSVector x, long *index)
{
    long k;
    long ind;
    short min;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->minfunc != NULL) {
	return (short)flist->minfunc(x->instance, x->length, index);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	min = nspwMinExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    min = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (min > x->data[k]) {
	    ind = k;
	    min = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return min;
}

long lvmin(spLVector x, long *index)
{
    long k;
    long ind;
    long min;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->minfunc != NULL) {
	return (long)flist->minfunc(x->instance, x->length, index);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    ind = 0;
    min = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (min > x->data[k]) {
	    ind = k;
	    min = x->data[k];
	}
    }

    if (index != NULL) {
	*index = ind;
    }

    return min;
}

float fvmin(spFVector x, long *index)
{
    long k;
    long ind;
    float min;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "fvmin", "in\n");
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->minfunc != NULL) {
	return (float)flist->minfunc(x->instance, x->length, index);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	min = nspsMinExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    min = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (min > x->data[k]) {
	    ind = k;
	    min = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return min;
}

double dvmin(spDVector x, long *index)
{
    long k;
    long ind;
    double min;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    spDebug(100, "dvmin", "in\n");
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->minfunc != NULL) {
	return flist->minfunc(x->instance, x->length, index);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	min = nspdMinExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    min = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (min > x->data[k]) {
	    ind = k;
	    min = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return min;
}

short svmax(spSVector x, long *index)
{
    long k;
    long ind;
    short max;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->maxfunc != NULL) {
	return (short)flist->maxfunc(x->instance, x->length, index);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	max = nspwMaxExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    max = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (max < x->data[k]) {
	    ind = k;
	    max = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return max;
}

long lvmax(spLVector x, long *index)
{
    long k;
    long ind;
    long max;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->maxfunc != NULL) {
	return (long)flist->maxfunc(x->instance, x->length, index);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    ind = 0;
    max = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (max < x->data[k]) {
	    ind = k;
	    max = x->data[k];
	}
    }

    if (index != NULL) {
	*index = ind;
    }

    return max;
}

float fvmax(spFVector x, long *index)
{
    long k;
    long ind;
    float max;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->maxfunc != NULL) {
	return (float)flist->maxfunc(x->instance, x->length, index);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	max = nspsMaxExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    max = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (max < x->data[k]) {
	    ind = k;
	    max = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return max;
}

double dvmax(spDVector x, long *index)
{
    long k;
    long ind;
    double max;
    
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->maxfunc != NULL) {
	return flist->maxfunc(x->instance, x->length, index);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

#ifdef USE_SPL
    {
	int iind;
	if (x->length <= 0) {
	    spwarning("warning: dvmax: length of x is zero\n");
	}
    
	max = nspdMaxExt(x->data, x->length, &iind);
	ind = iind;
    }
#else
    ind = 0;
    max = x->data[ind];
    for (k = 1; k < x->length; k++) {
	if (max < x->data[k]) {
	    ind = k;
	    max = x->data[k];
	}
    }
#endif

    if (index != NULL) {
	*index = ind;
    }

    return max;
}

void svscmin(spSVector x, short a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scmin != NULL) {
	flist->scmin(x->instance, x->length, a);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MIN(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscmin(spLVector x, long a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scmin != NULL) {
	flist->scmin(x->instance, x->length, a);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MIN(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscmin(spFVector x, float a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scmin != NULL) {
	flist->scmin(x->instance, x->length, a);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MIN(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscmin(spDVector x, double a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scmin != NULL) {
	flist->scmin(x->instance, x->length, a);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MIN(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscmax(spSVector x, short a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scmax != NULL) {
	flist->scmax(x->instance, x->length, a);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MAX(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscmax(spLVector x, long a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scmax != NULL) {
	flist->scmax(x->instance, x->length, a);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MAX(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscmax(spFVector x, float a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scmax != NULL) {
	flist->scmax(x->instance, x->length, a);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MAX(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscmax(spDVector x, double a)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scmax != NULL) {
	flist->scmax(x->instance, x->length, a);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = MAX(x->data[k], a);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

#if 1
/* calculate min at each element */
void svelmin(spSVector x, spSVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->elmin != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}
	
	flist->elmin(x->instance, y->instance, len);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MIN(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvelmin(spLVector x, spLVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->elmin != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}
	
	flist->elmin(x->instance, y->instance, len);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MIN(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvelmin(spFVector x, spFVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->elmin != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}
	
	flist->elmin(x->instance, y->instance, len);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MIN(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvelmin(spDVector x, spDVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->elmin != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}
	
	flist->elmin(x->instance, y->instance, len);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MIN(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

/* calculate max at each element */
void svelmax(spSVector x, spSVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->elmax != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}
	
	flist->elmax(x->instance, y->instance, len);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MAX(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvelmax(spLVector x, spLVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->elmax != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}
	
	flist->elmax(x->instance, y->instance, len);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MAX(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvelmax(spFVector x, spFVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->elmax != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}
	
	flist->elmax(x->instance, y->instance, len);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MAX(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvelmax(spDVector x, spDVector y)
{
    long k;
    long len;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    len = MIN(x->length, y->length);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->elmax != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}
	
	flist->elmax(x->instance, y->instance, len);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < len; k++) {
	x->data[k] = MAX(x->data[k], y->data[k]);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}
#endif

spLVector xsvnmin(spSVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    short min;
    short pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->nmin != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmin(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    idx = xlvallocul(svgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	min = 0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;
	    
	    if (idx->data[i] < 0 || min > x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff < 0 && ndiff > 0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0 && ndiff > 0 && j == 0)
				|| (pdiff < 0 && ndiff == 0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    min = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xlvnmin(spLVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    long min;
    long pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->nmin != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmin(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    idx = xlvallocul(lvgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	min = 0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;
	    
	    if (idx->data[i] < 0 || min > x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff < 0 && ndiff > 0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0 && ndiff > 0 && j == 0)
				|| (pdiff < 0 && ndiff == 0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    min = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xfvnmin(spFVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    float min;
    float pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->nmin != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmin(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    idx = xlvallocul(fvgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	min = 0.0f;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;
	    
	    if (idx->data[i] < 0 || min > x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0.0f;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0.0f;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff < 0.0f && ndiff > 0.0f)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0.0f && ndiff > 0.0f && j == 0)
				|| (pdiff < 0.0f && ndiff == 0.0f && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    min = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xdvnmin(spDVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    double min;
    double pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->nmin != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmin(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    idx = xlvallocul(dvgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	min = 0.0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;
	    
	    if (idx->data[i] < 0 || min > x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0.0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0.0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff < 0.0 && ndiff > 0.0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0.0 && ndiff > 0.0 && j == 0)
				|| (pdiff < 0.0 && ndiff == 0.0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    min = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xsvnmax(spSVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    short max;
    short pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->nmax != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmax(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    idx = xlvallocul(svgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	max = 0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;

	    if (idx->data[i] < 0 || max < x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff > 0 && ndiff < 0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0 && ndiff < 0 && j == 0)
				|| (pdiff > 0 && ndiff == 0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    max = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xlvnmax(spLVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    long max;
    long pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->nmax != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmax(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    idx = xlvallocul(lvgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	max = 0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;

	    if (idx->data[i] < 0 || max < x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff > 0 && ndiff < 0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0 && ndiff < 0 && j == 0)
				|| (pdiff > 0 && ndiff == 0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    max = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

spLVector xfvnmax(spFVector x, long n, long margin, unsigned long options)
{
    int skip, peak;
    long i, m;
    long j;
    float max;
    float pdiff, ndiff;
    spLVector idx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->nmax != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmax(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    idx = xlvallocul(fvgetplugin(x), n, SP_FALSE);

    for (i = 0; i < n; i++) {
	max = 0.0f;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;

	    if (idx->data[i] < 0 || max < x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0.0f;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0.0f;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff > 0.0f && ndiff < 0.0f)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0.0f && ndiff < 0.0f && j == 0)
				|| (pdiff > 0.0f && ndiff == 0.0f && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    max = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    idx->length = i;
	    break;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

long dvnmax(spLVector idx, spDVector x, long margin, unsigned long options)
{
    int skip, peak;
    long n;
    long i, m;
    long j;
    long nfound;
    double max;
    double pdiff, ndiff;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    n = idx->length;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(idx) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->nmax != NULL) {
	return flist->nmax(x->instance, x->length, n, margin, options, idx->instance);
    }
    
    if (dvislocked(x) == SP_FALSE) {
	dvsync(x);
    }
#endif

    nfound = n;
    
    for (i = 0; i < n; i++) {
	max = 0.0;
	idx->data[i] = -1;
	for (j = 0; j < x->length; j++) {
	    skip = 0;
	    for (m = 0; m < i; m++) {
		if (j >= idx->data[m] - margin && j <= idx->data[m] + margin) {
		    skip = 1;
		    break;
		}
	    }
	    if (skip) continue;

	    if (idx->data[i] < 0 || max < x->data[j]) {
		peak = 1;
		if (options & SP_NMINMAX_OPTION_PEAK_ONLY) {
		    if (j < 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    pdiff = x->data[j];
			} else {
			    pdiff = 0.0;
			}
		    } else {
			pdiff = x->data[j] - x->data[j - 1];
		    }
		    if (j >= x->length - 1) {
			if (options & SP_NMINMAX_OPTION_ZERO_EDGES) {
			    ndiff = -x->data[j];
			} else {
			    ndiff = 0.0;
			}
		    } else {
			ndiff = x->data[j + 1] - x->data[j];
		    }

		    if ((pdiff > 0.0 && ndiff < 0.0)
			|| (!(options & SP_NMINMAX_OPTION_ZERO_EDGES)
			    && ((pdiff == 0.0 && ndiff < 0.0 && j == 0)
				|| (pdiff > 0.0 && ndiff == 0.0 && j == x->length - 1)))) {
			peak = 1;
		    } else {
			peak = 0;
		    }
		}

		if (peak) {
		    idx->data[i] = j;
		    max = x->data[j];
		}
	    }
	}
	
	if (idx->data[i] < 0) {
	    spDebug(100, "dvnmax", "peak not found: i = %ld\n", i);
	    nfound = i;
	    break;
	}
    }

    return nfound;
}

spLVector xdvnmax(spDVector x, long n, long margin, unsigned long options)
{
    spLVector idx;
    
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->nmax != NULL) {
	idx = xlvallocul(x->plugin, n, SP_TRUE);
	idx->length = flist->nmax(x->instance, x->length, n, margin, options, idx->instance);
	return idx;
    }
#endif

    idx = xlvallocul(dvgetplugin(x), n, SP_FALSE);

    idx->length = dvnmax(idx, x, margin, options);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        lvunlock(idx);
    }
#endif

    return idx;
}

#if defined(__STDC__) || defined(_WIN32) || defined(MACOS)
static int qsort_snumcmp(const void *x, const void *y)
#else
static int qsort_snumcmp(char *x, char *y)
#endif
{
    short tx, ty;

    tx = *(short *)x;
    ty = *(short *)y;

    if (tx < ty) return (-1);
    if (tx > ty) return (1);
    return (0);
}

void svsort(spSVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 1)
	return;

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
	flist->sort(x->instance, x->length);
	return;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    qsort(x->data, (unsigned)x->length, sizeof(short), qsort_snumcmp);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

#if defined(__STDC__) || defined(_WIN32) || defined(MACOS)
static int qsort_lnumcmp(const void *x, const void *y)
#else
static int qsort_lnumcmp(char *x, char *y)
#endif
{
    long tx, ty;

    tx = *(long *)x;
    ty = *(long *)y;

    if (tx < ty) return (-1);
    if (tx > ty) return (1);
    return (0);
}

void lvsort(spLVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 1)
	return;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
	flist->sort(x->instance, x->length);
	return;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    qsort(x->data, (unsigned)x->length, sizeof(long), qsort_lnumcmp);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

#if defined(__STDC__) || defined(_WIN32) || defined(MACOS)
static int qsort_fnumcmp(const void *x, const void *y)
#else
static int qsort_fnumcmp(char *x, char *y)
#endif
{
    float tx, ty;

    tx = *(float *)x;
    ty = *(float *)y;

    if (tx < ty) return (-1);
    if (tx > ty) return (1);
    return (0);
}

void fvsort(spFVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 1)
	return;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
	flist->sort(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    qsort(x->data, (unsigned)x->length, sizeof(float), qsort_fnumcmp);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

#if defined(__STDC__) || defined(_WIN32) || defined(MACOS)
static int qsort_dnumcmp(const void *x, const void *y)
#else
static int qsort_dnumcmp(char *x, char *y)
#endif
{
    double tx, ty;

    tx = *(double *)x;
    ty = *(double *)y;

    if (tx < ty) return (-1);
    if (tx > ty) return (1);
    return (0);
}

void dvsort(spDVector x)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 1)
	return;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
	flist->sort(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    qsort(x->data, (unsigned)x->length, sizeof(double), qsort_dnumcmp);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spSVector xsvsort(spSVector x)
{
    spSVector y;

    y = xsvclone(x);
    svsort(y);

    return y;
}

spLVector xlvsort(spLVector x)
{
    spLVector y;

    y = xlvclone(x);
    lvsort(y);

    return y;
}

spFVector xfvsort(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsort(y);

    return y;
}

spDVector xdvsort(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsort(y);

    return y;
}

#if 1
static int spheapsort_snumcmp(void *keysptr, long i, long j)
{
    short *data = (short *)keysptr;

    if (data[i] < data[j]) {
	return -1;
    } else if (data[i] > data[j]) {
	return 1;
    } else {
	return 0;
    }
}

static void spheapsort_snumswap(void *keysptr, void *valuesptr, long i, long j)
{
    short key;
    long index;
    short *keys = (short *)keysptr;
    long *indices = (long *)valuesptr;

    key = keys[i];
    keys[i] = keys[j];
    keys[j] = key;

    index = indices[i];
    indices[i] = indices[j];
    indices[j] = index;

    return;
}

static int spheapsort_lnumcmp(void *keysptr, long i, long j)
{
    short *data = (short *)keysptr;

    if (data[i] < data[j]) {
	return -1;
    } else if (data[i] > data[j]) {
	return 1;
    } else {
	return 0;
    }
}

static void spheapsort_lnumswap(void *keysptr, void *valuesptr, long i, long j)
{
    short key;
    long index;
    short *keys = (short *)keysptr;
    long *indices = (long *)valuesptr;

    key = keys[i];
    keys[i] = keys[j];
    keys[j] = key;

    index = indices[i];
    indices[i] = indices[j];
    indices[j] = index;

    return;
}

static int spheapsort_fnumcmp(void *keysptr, long i, long j)
{
    float *data = (float *)keysptr;

    if (data[i] < data[j]) {
	return -1;
    } else if (data[i] > data[j]) {
	return 1;
    } else {
	return 0;
    }
}

static void spheapsort_fnumswap(void *keysptr, void *valuesptr, long i, long j)
{
    float key;
    long index;
    float *keys = (float *)keysptr;
    long *indices = (long *)valuesptr;

    key = keys[i];
    keys[i] = keys[j];
    keys[j] = key;

    index = indices[i];
    indices[i] = indices[j];
    indices[j] = index;

    return;
}

static int spheapsort_dnumcmp(void *keysptr, long i, long j)
{
    double *data = (double *)keysptr;

    if (data[i] < data[j]) {
	return -1;
    } else if (data[i] > data[j]) {
	return 1;
    } else {
	return 0;
    }
}

static void spheapsort_dnumswap(void *keysptr, void *valuesptr, long i, long j)
{
    double key;
    long index;
    double *keys = (double *)keysptr;
    long *indices = (long *)valuesptr;

    key = keys[i];
    keys[i] = keys[j];
    keys[j] = key;

    index = indices[i];
    indices[i] = indices[j];
    indices[j] = index;

    return;
}

static void spheapsortsub(void *keysptr, void *valuesptr, long length, long i, 
			  int (*compare_func)(void *keysptr, long i, long j),
			  void (*swap_func)(void *keysptr, void *valuesptr, long i, long j))
{
    long j;
    
    while ((j = 2 * i) <= length) {
	if (j < length && (*compare_func)(keysptr, j - 1, j) < 0) j++;
	if ((*compare_func)(keysptr, i - 1, j - 1) >= 0) break;

	(*swap_func)(keysptr, valuesptr, i - 1, j - 1);
	i = j;
    }
    
    return;
}

void spheapsort(void *keysptr, void *valuesptr, long length,
		int (*compare_func)(void *keysptr, long i, long j),
		void (*swap_func)(void *keysptr, void *valuesptr, long i, long j))
{
    long i, k;

    for (k = length / 2; k >= 1; k--) {
	i = k;
	spheapsortsub(keysptr, valuesptr, length, i, compare_func, swap_func);
    }
    while (length > 1) {
	(*swap_func)(keysptr, valuesptr, length - 1, 0);
	length--;
	i = 1;
	spheapsortsub(keysptr, valuesptr, length, i, compare_func, swap_func);
    }
}

spLVector xsvsortidx(spSVector x)
{
    spLVector idxv;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    
    if (x == NODATA || x->length <= 1)
	return NODATA;

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sortidx != NULL) {
	idxv = xlvallocul(x->plugin, x->length, SP_TRUE);
	lvinit(idxv, 0, 1, x->length - 1);
	
	flist->sortidx(x->instance, idxv->instance, x->length);

	return idxv;
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    idxv = xlvinit(0, 1, x->length - 1);
    
    spheapsort(x->data, idxv->data, x->length, spheapsort_snumcmp, spheapsort_snumswap);

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        lvunlock(idxv);
    }
#endif

    return idxv;
}

spLVector xlvsortidx(spLVector x)
{
    spLVector idxv;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    
    if (x == NODATA || x->length <= 1)
	return NODATA;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sortidx != NULL) {
	idxv = xlvallocul(x->plugin, x->length, SP_TRUE);
	lvinit(idxv, 0, 1, x->length - 1);
	
	flist->sortidx(x->instance, idxv->instance, x->length);

	return idxv;
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    idxv = xlvinit(0, 1, x->length - 1);
    
    spheapsort(x->data, idxv->data, x->length, spheapsort_lnumcmp, spheapsort_lnumswap);

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(idxv);
    }
#endif

    return idxv;
}

spLVector xfvsortidx(spFVector x)
{
    spLVector idxv;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    
    if (x == NODATA || x->length <= 1)
	return NODATA;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sortidx != NULL) {
	idxv = xlvallocul(x->plugin, x->length, SP_TRUE);
	lvinit(idxv, 0, 1, x->length - 1);
	
	flist->sortidx(x->instance, idxv->instance, x->length);

	return idxv;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    idxv = xlvinit(0, 1, x->length - 1);
    
    spheapsort(x->data, idxv->data, x->length, spheapsort_fnumcmp, spheapsort_fnumswap);

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        lvunlock(idxv);
    }
#endif

    return idxv;
}

spLVector xdvsortidx(spDVector x)
{
    spLVector idxv;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    
    if (x == NODATA || x->length <= 1)
	return NODATA;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sortidx != NULL) {
	idxv = xlvallocul(x->plugin, x->length, SP_TRUE);
	lvinit(idxv, 0, 1, x->length - 1);
	
	flist->sortidx(x->instance, idxv->instance, x->length);

	return idxv;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    idxv = xlvinit(0, 1, x->length - 1);
    
    spheapsort(x->data, idxv->data, x->length, spheapsort_dnumcmp, spheapsort_dnumswap);

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        lvunlock(idxv);
    }
#endif

    return idxv;
}

#endif

/* x will be sorted */
double svmedian(spSVector x)
{
    long hlen;
    double median;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 0) return 0.0;

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->median != NULL) {
	return flist->median(x->instance, x->length);
    }
#endif
    
    svsort(x);
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
        svsync(x);
    }
#endif
    hlen = x->length / 2;

    if (hlen * 2 == x->length) {
	median = (double)(x->data[hlen - 1] + x->data[hlen]) / 2.0;
    } else {
	median = x->data[hlen];
    }

    return median;
}

double lvmedian(spLVector x)
{
    long hlen;
    double median;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 0) return 0.0;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->median != NULL) {
	return flist->median(x->instance, x->length);
    }
#endif
    
    lvsort(x);
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
        lvsync(x);
    }
#endif
    hlen = x->length / 2;

    if (hlen * 2 == x->length) {
	median = (double)(x->data[hlen - 1] + x->data[hlen]) / 2.0;
    } else {
	median = x->data[hlen];
    }

    return median;
}

float fvmedian(spFVector x)
{
    long hlen;
    float median;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 0) return 0.0f;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->median != NULL) {
	return (float)flist->median(x->instance, x->length);
    }
#endif
    
    fvsort(x);
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
        fvsync(x);
    }
#endif
    hlen = x->length / 2;

    if (hlen * 2 == x->length) {
	median = (x->data[hlen - 1] + x->data[hlen]) / 2.0f;
    } else {
	median = x->data[hlen];
    }

    return median;
}

double dvmedian(spDVector x)
{
    long hlen;
    double median;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA || x->length <= 0) return 0.0;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->median != NULL) {
	return flist->median(x->instance, x->length);
    }
#endif
    
    dvsort(x);
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sort != NULL) {
        dvsync(x);
    }
#endif
    hlen = x->length / 2;

    if (hlen * 2 == x->length) {
	median = (x->data[hlen - 1] + x->data[hlen]) / 2.0;
    } else {
	median = x->data[hlen];
    }

    return median;
}

spSVector xsvmedian(spSVector x, double *median)
{
    spSVector xs;

    if (median == NULL) return NODATA;

    xs = xsvclone(x);
    
    *median = svmedian(xs);
    
    return xs;
}

spLVector xlvmedian(spLVector x, double *median)
{
    spLVector xs;

    if (median == NULL) return NODATA;

    xs = xlvclone(x);
    
    *median = lvmedian(xs);
    
    return xs;
}

spFVector xfvmedian(spFVector x, float *median)
{
    spFVector xs;

    if (median == NULL) return NODATA;

    xs = xfvclone(x);
    
    *median = fvmedian(xs);
    
    return xs;
}

spDVector xdvmedian(spDVector x, double *median)
{
    spDVector xs;

    if (median == NULL) return NODATA;

    xs = xdvclone(x);
    
    *median = dvmedian(xs);
    
    return xs;
}

/* slope_sign: 0=both, 1=positive only, -1=negative only */
long svzerocross(spSVector x, int slope_sign) 
{
    long k;
    long count;
    short mx, dx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->zerocross != NULL) {
	return flist->zerocross(x->instance, x->length, slope_sign);
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    count = 0;
    for (k = 1; k < x->length; k++) {
	mx = x->data[k] * x->data[k - 1];
	if (mx < 0) {
	    if (slope_sign == 0) {
		++count;
	    } else {
		dx = x->data[k] - x->data[k - 1];
	    
		if (dx > 0 && slope_sign > 0) {
		    ++count;
		} else if (dx < 0 && slope_sign < 0) {
		    ++count;
		}
	    }
	}
    }

    return count;
}

long lvzerocross(spLVector x, int slope_sign) 
{
    long k;
    long count;
    long mx, dx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->zerocross != NULL) {
	return flist->zerocross(x->instance, x->length, slope_sign);
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    count = 0;
    for (k = 1; k < x->length; k++) {
	mx = x->data[k] * x->data[k - 1];
	if (mx < 0) {
	    if (slope_sign == 0) {
		++count;
	    } else {
		dx = x->data[k] - x->data[k - 1];
	    
		if (dx > 0 && slope_sign > 0) {
		    ++count;
		} else if (dx < 0 && slope_sign < 0) {
		    ++count;
		}
	    }
	}
    }

    return count;
}

long fvzerocross(spFVector x, int slope_sign) 
{
    long k;
    long count;
    float mx, dx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->zerocross != NULL) {
	return flist->zerocross(x->instance, x->length, slope_sign);
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    count = 0;
    for (k = 1; k < x->length; k++) {
	mx = x->data[k] * x->data[k - 1];
	if (mx < 0.0f) {
	    if (slope_sign == 0) {
		++count;
	    } else {
		dx = x->data[k] - x->data[k - 1];
	    
		if (dx > 0.0f && slope_sign > 0) {
		    ++count;
		} else if (dx < 0.0f && slope_sign < 0) {
		    ++count;
		}
	    }
	}
    }

    return count;
}

long dvzerocross(spDVector x, int slope_sign) 
{
    long k;
    long count;
    double mx, dx;

#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->zerocross != NULL) {
	return flist->zerocross(x->instance, x->length, slope_sign);
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    count = 0;
    for (k = 1; k < x->length; k++) {
	mx = x->data[k] * x->data[k - 1];
	if (mx < 0.0) {
	    if (slope_sign == 0) {
		++count;
	    } else {
		dx = x->data[k] - x->data[k - 1];
	    
		if (dx > 0.0 && slope_sign > 0) {
		    ++count;
		} else if (dx < 0.0 && slope_sign < 0) {
		    ++count;
		}
	    }
	}
    }

    return count;
}

/*-- the following functions are supported for float and double only --*/
void fvangle(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	fvzeros(x, x->length);
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->angle != NULL) {
	flist->angle(x->instance, x->length);
	fvreal(x);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
	    x->data[k] = 0.0f;
	} else {
	    x->data[k] = atan2f(x->imag[k], x->data[k]);
	}
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    fvreal(x);

    return;
}

spFVector xfvangle(spFVector x)
{
    spFVector phs;

    if (x->imag == NULL) {
	phs = xfvzeros(x->length);
	return phs;
    }

    phs = xfvclone(x);
    fvangle(phs);

    return phs;
}

void dvangle(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	dvzeros(x, x->length);
	return;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->angle != NULL) {
	flist->angle(x->instance, x->length);
	dvreal(x);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
	    x->data[k] = 0.0;
	} else {
	    x->data[k] = atan2(x->imag[k], x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    dvreal(x);

    return;
}

spDVector xdvangle(spDVector x)
{
    spDVector phs;

    if (x->imag == NULL) {
	phs = xdvzeros(x->length);
	return phs;
    }

    phs = xdvclone(x);
    dvangle(phs);

    return phs;
}

void fvcos(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cos != NULL) {
	flist->cos(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = cosf(x->data[k]);
	}
    } else {
	float re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = cosf(x->data[k]) * coshf(x->imag[k]);
	    im = -sinf(x->data[k]) * sinhf(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcos(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cos != NULL) {
	flist->cos(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = cos(x->data[k]);
	}
    } else {
	double re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = cos(x->data[k]) * cosh(x->imag[k]);
	    im = -sin(x->data[k]) * sinh(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvcos(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvcos(y);

    return y;
}

spDVector xdvcos(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvcos(y);

    return y;
}

void fvsin(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sin != NULL) {
	flist->sin(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sinf(x->data[k]);
	}
    } else {
	float re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = sinf(x->data[k]) * coshf(x->imag[k]);
	    im = cosf(x->data[k]) * sinhf(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsin(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sin != NULL) {
	flist->sin(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sin(x->data[k]);
	}
    } else {
	double re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = sin(x->data[k]) * cosh(x->imag[k]);
	    im = cos(x->data[k]) * sinh(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvsin(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsin(y);

    return y;
}

spDVector xdvsin(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsin(y);

    return y;
}

void fvtan(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->tan != NULL) {
	flist->tan(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = tanf(x->data[k]);
	}
    } else {
	float re, im, tan_re, tanh_im, den;
	
	for (k = 0; k < x->length; k++) {
	    tanh_im = tanhf(x->imag[k]);
	    re = tanf(x->data[k] * (1.0f - tanh_im * tanh_im));
	    tan_re = tanf(x->data[k]);
	    im = tanhf(x->imag[k]) * (tan_re * tan_re + 1.0f);
	    den = tan_re * tanh_im;
	    den = 1.0f + den * den;
	    x->data[k] = re / den;
	    x->imag[k] = im / den;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvtan(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->tan != NULL) {
	flist->tan(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = tan(x->data[k]);
	}
    } else {
	double re, im, tan_re, tanh_im, den;
	
	for (k = 0; k < x->length; k++) {
	    tanh_im = tanh(x->imag[k]);
	    re = tan(x->data[k] * (1.0 - tanh_im * tanh_im));
	    tan_re = tan(x->data[k]);
	    im = tanh(x->imag[k]) * (tan_re * tan_re + 1.0);
	    den = tan_re * tanh_im;
	    den = 1.0 + den * den;
	    x->data[k] = re / den;
	    x->imag[k] = im / den;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvtan(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvtan(y);

    return y;
}

spDVector xdvtan(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvtan(y);

    return y;
}

void fvcosh(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cosh != NULL) {
	flist->cosh(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = coshf(x->data[k]);
	}
    } else {
	float re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = coshf(x->data[k]) * cosf(x->imag[k]);
	    im = sinhf(x->data[k]) * sinf(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcosh(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cosh != NULL) {
	flist->cosh(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = cosh(x->data[k]);
	}
    } else {
	double re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = cosh(x->data[k]) * cos(x->imag[k]);
	    im = sinh(x->data[k]) * sin(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvcosh(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvcosh(y);

    return y;
}

spDVector xdvcosh(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvcosh(y);

    return y;
}

void fvsinh(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sinh != NULL) {
	flist->sinh(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sinhf(x->data[k]);
	}
    } else {
	float re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = sinhf(x->data[k]) * cosf(x->imag[k]);
	    im = coshf(x->data[k]) * sinf(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsinh(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sinh != NULL) {
	flist->sinh(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = sinh(x->data[k]);
	}
    } else {
	double re, im;
	
	for (k = 0; k < x->length; k++) {
	    re = sinh(x->data[k]) * cos(x->imag[k]);
	    im = cosh(x->data[k]) * sin(x->imag[k]);
	    x->data[k] = re;
	    x->imag[k] = im;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvsinh(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvsinh(y);

    return y;
}

spDVector xdvsinh(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvsinh(y);

    return y;
}

void fvtanh(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->tanh != NULL) {
	flist->tanh(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = tanhf(x->data[k]);
	}
    } else {
	float re, im, tanh_re, tan_im, den;
	
	for (k = 0; k < x->length; k++) {
	    tan_im = tanf(x->imag[k]);
	    re = tanhf(x->data[k] * (1.0f + tan_im * tan_im));
	    tanh_re = tanhf(x->data[k]);
	    im = tanf(x->imag[k]) * (1.0f - tanh_re * tanh_re);
	    den = tanh_re * tan_im;
	    den = 1.0f + den * den;
	    x->data[k] = re / den;
	    x->imag[k] = im / den;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvtanh(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->tanh != NULL) {
	flist->tanh(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = tanh(x->data[k]);
	}
    } else {
	double re, im, tanh_re, tan_im, den;
	
	for (k = 0; k < x->length; k++) {
	    tan_im = tan(x->imag[k]);
	    re = tanh(x->data[k] * (1.0 + tan_im * tan_im));
	    tanh_re = tanh(x->data[k]);
	    im = tan(x->imag[k]) * (1.0 - tanh_re * tanh_re);
	    den = tanh_re * tan_im;
	    den = 1.0 + den * den;
	    x->data[k] = re / den;
	    x->imag[k] = im / den;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvtanh(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvtanh(y);

    return y;
}

spDVector xdvtanh(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvtanh(y);

    return y;
}

void fvacos(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->acos != NULL) {
	flist->acos(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = acosf(x->data[k]);
	}
    } else {
	float re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* sqrt(1 - x^2) */
	    re = 1.0f - x->data[k] * x->data[k] + x->imag[k] * x->imag[k];
	    im = -2.0f * x->data[k] * x->imag[k];
	    clogf(&re, &im);
	    re *= 0.5f;
	    im *= 0.5f;
	    cexpf(&re, &im);

	    /* log(x + i * sqrt(1 - x^2)) */
	    re2 = x->data[k] - im;
	    im2 = x->imag[k] + re;
	    clogf(&re2, &im2);

	    /* -i * log(x + i * sqrt(1 - x^2)) */
	    x->data[k] = im2;
	    x->imag[k] = -re2;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvacos(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->acos != NULL) {
	flist->acos(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = acos(x->data[k]);
	}
    } else {
	double re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* sqrt(1 - x^2) */
	    re = 1.0 - x->data[k] * x->data[k] + x->imag[k] * x->imag[k];
	    im = -2.0 * x->data[k] * x->imag[k];
	    clog(&re, &im);
	    re *= 0.5;
	    im *= 0.5;
	    cexp(&re, &im);

	    /* log(x + i * sqrt(1 - x^2)) */
	    re2 = x->data[k] - im;
	    im2 = x->imag[k] + re;
	    clog(&re2, &im2);

	    /* -i * log(x + i * sqrt(1 - x^2)) */
	    x->data[k] = im2;
	    x->imag[k] = -re2;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvacos(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvacos(y);

    return y;
}

spDVector xdvacos(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvacos(y);

    return y;
}

void fvasin(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->asin != NULL) {
	flist->asin(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = asinf(x->data[k]);
	}
    } else {
	float re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* sqrt(1 - x^2) */
	    re = 1.0f - x->data[k] * x->data[k] + x->imag[k] * x->imag[k];
	    im = -2.0f * x->data[k] * x->imag[k];
	    clogf(&re, &im);
	    re *= 0.5f;
	    im *= 0.5f;
	    cexpf(&re, &im);

	    /* log(i * x + sqrt(1 - x^2)) */
	    re2 = -x->imag[k] + re;
	    im2 = x->data[k] + im;
	    clogf(&re2, &im2);

	    /* i * log(i * x + sqrt(1 - x^2)) */
	    x->data[k] = -im2;
	    x->imag[k] = re2;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvasin(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->asin != NULL) {
	flist->asin(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = asin(x->data[k]);
	}
    } else {
	double re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* sqrt(1 - x^2) */
	    re = 1.0 - x->data[k] * x->data[k] + x->imag[k] * x->imag[k];
	    im = -2.0 * x->data[k] * x->imag[k];
	    clog(&re, &im);
	    re *= 0.5f;
	    im *= 0.5f;
	    cexp(&re, &im);

	    /* log(i * x + sqrt(1 - x^2)) */
	    re2 = -x->imag[k] + re;
	    im2 = x->data[k] + im;
	    clog(&re2, &im2);

	    /* i * log(i * x + sqrt(1 - x^2)) */
	    x->data[k] = -im2;
	    x->imag[k] = re2;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvasin(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvasin(y);

    return y;
}

spDVector xdvasin(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvasin(y);

    return y;
}

void fvatan(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->atan != NULL) {
	flist->atan(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = atanf(x->data[k]);
	}
    } else {
	float re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* log(i + x) */
	    re = x->data[k];
	    im = x->imag[k] + 1.0f;
	    clogf(&re, &im);

	    /* log(i - x) */
	    re2 = -x->data[k];
	    im2 = -x->imag[k] + 1.0f;
	    clogf(&re2, &im2);
	    
	    /* i * (log((i + x) / (i - x)) / 2 */
	    x->data[k] = -(im - im2) / 2.0f;
	    x->imag[k] = (re - re2) / 2.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvatan(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->atan != NULL) {
	flist->atan(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
	for (k = 0; k < x->length; k++) {
	    x->data[k] = atan(x->data[k]);
	}
    } else {
	double re, im, re2, im2;
	
	for (k = 0; k < x->length; k++) {
	    /* log(i + x) */
	    re = x->data[k];
	    im = x->imag[k] + 1.0;
	    clog(&re, &im);

	    /* log(i - x) */
	    re2 = -x->data[k];
	    im2 = -x->imag[k] + 1.0;
	    clog(&re2, &im2);
	    
	    /* i * (log((i + x) / (i - x)) / 2 */
	    x->data[k] = -(im - im2) / 2.0;
	    x->imag[k] = (re - re2) / 2.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvatan(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvatan(y);

    return y;
}

spDVector xdvatan(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvatan(y);

    return y;
}

/* imaginary part is ignored. */
void fvatan2(spFVector y, spFVector x, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif


#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(y)) != NULL && flist->atan2 != NULL) {
	if (fvislocked(y) == SP_FALSE && fvislocked(x) == SP_TRUE) {
	    fvsync(x);
	}

	flist->atan2(y->instance, x->instance, MIN(y->length, x->length), reverse);
	
	if (fvisplugincomputable(y) == SP_TRUE && fvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    y->data[k] = atan2f(x->data[k], y->data[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    y->data[k] = atan2f(y->data[k], x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(y) == SP_FALSE) {
        fvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvatan2(spDVector y, spDVector x, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(y) == SP_TRUE && x->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(y)) != NULL && flist->atan2 != NULL) {
	if (dvislocked(y) == SP_FALSE && dvislocked(x) == SP_TRUE) {
	    dvsync(x);
	}

	flist->atan2(y->instance, x->instance, MIN(y->length, x->length), reverse);
	
	if (dvisplugincomputable(y) == SP_TRUE && dvisplugincomputable(x) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    y->data[k] = atan2(x->data[k], y->data[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    y->data[k] = atan2(y->data[k], x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(y) == SP_FALSE) {
        dvunlockcore(y, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvatan2(spFVector y, spFVector x)
{
    spFVector z;

    z = xfvclone(y);
    fvatan2(z, x, 0);
    
    return z;
}

spDVector xdvatan2(spDVector y, spDVector x)
{
    spDVector z;

    z = xdvclone(y);
    dvatan2(z, x, 0);
    
    return z;
}

void fvexp(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->exp != NULL) {
	flist->exp(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspsbExp1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = expf(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    cexpf(&x->data[k], &x->imag[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvexp(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->exp != NULL) {
	flist->exp(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    if (x->imag == NULL) {
#ifdef USE_SPL
	nspdbExp1(x->data, x->length);
#else
	for (k = 0; k < x->length; k++) {
	    x->data[k] = exp(x->data[k]);
	}
#endif
    } else {
	for (k = 0; k < x->length; k++) {
	    cexp(&x->data[k], &x->imag[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvexp(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvexp(y);

    return y;
}

spDVector xdvexp(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvexp(y);

    return y;
}

void fvlog(spFVector x)
{
    int flag = 0;
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->log != NULL) {
	flist->log(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
#if 0
    for (k = 0; k < x->length; k++) {
	if (x->imag != NULL || x->data[k] < 0.0) {
	    flag = 1;
	    break;
	}
    }
#else
    if (x->imag != NULL) flag = 1;
#endif

    if (flag) {
	if (x->imag == NULL) {
	    fvizeros(x, x->length);
	}
	for (k = 0; k < x->length; k++) {
	    clogf(&x->data[k], &x->imag[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    clogf(&x->data[k], NULL);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvlog(spDVector x)
{
    int flag = 0;
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->log != NULL) {
	flist->log(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
#if 0
    for (k = 0; k < x->length; k++) {
	if (x->imag != NULL || x->data[k] < 0.0) {
	    flag = 1;
	    break;
	}
    }
#else
    if (x->imag != NULL) flag = 1;
#endif

    if (flag) {
	if (x->imag == NULL) {
	    dvizeros(x, x->length);
	}
	for (k = 0; k < x->length; k++) {
	    clog(&x->data[k], &x->imag[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    clog(&x->data[k], NULL);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvlog(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvlog(y);

    return y;
}

spDVector xdvlog(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvlog(y);

    return y;
}

void fvlog10(spFVector x)
{
    int flag = 0;
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->log10 != NULL) {
	flist->log10(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
#if 0
    for (k = 0; k < x->length; k++) {
	if (x->imag != NULL || x->data[k] < 0.0) {
	    flag = 1;
	    break;
	}
    }
#else
    if (x->imag != NULL) flag = 1;
#endif

    if (flag) {
	if (x->imag == NULL) {
	    fvizeros(x, x->length);
	}
	for (k = 0; k < x->length; k++) {
	    clog10f(&x->data[k], &x->imag[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    clog10f(&x->data[k], NULL);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvlog10(spDVector x)
{
    int flag = 0;
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->log10 != NULL) {
	flist->log10(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
#if 0
    for (k = 0; k < x->length; k++) {
	if (x->imag != NULL || x->data[k] < 0.0) {
	    flag = 1;
	    break;
	}
    }
#else
    if (x->imag != NULL) flag = 1;
#endif

    if (flag) {
	if (x->imag == NULL) {
	    dvizeros(x, x->length);
	}
	for (k = 0; k < x->length; k++) {
	    clog10(&x->data[k], &x->imag[k]);
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    clog10(&x->data[k], NULL);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvlog10(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvlog10(y);

    return y;
}

spDVector xdvlog10(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvlog10(y);

    return y;
}

void fvdecibela(spFVector x)
{
    long k;
#if !(defined(IPHONE) || defined(ANDROID))
    spBool warn_flag = SP_FALSE;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->decibela != NULL) {
            flist->decibela(x->instance, x->length);
        } else {
            fvlog10(x);
            fvscoper(x, "*", 20.0f);
        }
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= 0.0f) {
#if !(defined(IPHONE) || defined(ANDROID))
            if (warn_flag == SP_FALSE) {
                spwarning("warning: fvdecibela: log of zero\n");
                warn_flag = SP_TRUE;
            }
#endif

	    x->data[k] = 20.0f * log10f((float)SP_TINY_NUMBER);
	} else {
	    x->data[k] = 20.0f * log10f(x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvdecibela(spDVector x)
{
    long k;
#if !(defined(IPHONE) || defined(ANDROID))
    spBool warn_flag = SP_FALSE;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->decibela != NULL) {
            flist->decibela(x->instance, x->length);
        } else {
            dvlog10(x);
            dvscoper(x, "*", 20.0);
        }
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= 0.0) {
#if !(defined(IPHONE) || defined(ANDROID))
            if (warn_flag == SP_FALSE) {
                spwarning("warning: dvdecibela: log of zero\n");
                warn_flag = SP_TRUE;
            }
#endif

	    x->data[k] = 20.0 * log10(SP_TINY_NUMBER);
	} else {
	    x->data[k] = 20.0 * log10(x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvdecibela(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvdecibela(y);

    return y;
}

spDVector xdvdecibela(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvdecibela(y);

    return y;
}

void fvdecibelp(spFVector x)
{
    long k;
#if !(defined(IPHONE) || defined(ANDROID))
    spBool warn_flag = SP_FALSE;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->decibelp != NULL) {
            flist->decibelp(x->instance, x->length);
        } else {
            fvlog10(x);
            fvscoper(x, "*", 10.0f);
        }
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= 0.0f) {
#if !(defined(IPHONE) || defined(ANDROID))
            if (warn_flag == SP_FALSE) {
                spwarning("warning: fvdecibelp: log of zero\n");
                warn_flag = SP_TRUE;
            }
#endif

	    x->data[k] = 10.0f * log10f((float)SP_TINY_NUMBER);
	} else {
	    x->data[k] = 10.0f * log10f(x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvdecibelp(spDVector x)
{
    long k;
#if !(defined(IPHONE) || defined(ANDROID))
    spBool warn_flag = SP_FALSE;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->decibelp != NULL) {
            flist->decibelp(x->instance, x->length);
        } else {
            dvlog10(x);
            dvscoper(x, "*", 10.0);
        }
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= 0.0) {
#if !(defined(IPHONE) || defined(ANDROID))
            if (warn_flag == SP_FALSE) {
                spwarning("warning: dvdecibelp: log of zero\n");
                warn_flag = SP_TRUE;
            }
#endif

	    x->data[k] = 10.0 * log10(SP_TINY_NUMBER);
	} else {
	    x->data[k] = 10.0 * log10(x->data[k]);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvdecibelp(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvdecibelp(y);

    return y;
}

spDVector xdvdecibelp(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvdecibelp(y);

    return y;
}

void fvundecibelp(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->undecibelp != NULL) {
            flist->undecibelp(x->instance, x->length);
        } else {
            fvscoper(x, "/", 10.0f);
            fvscoper(x, "!^", 10.0f);
        }
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = powf(10.0f, x->data[k] / 10.0f);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvundecibelp(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->undecibelp != NULL) {
            flist->undecibelp(x->instance, x->length);
        } else {
            dvscoper(x, "/", 10.0);
            dvscoper(x, "!^", 10.0);
        }
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = pow(10.0, x->data[k] / 10.0);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvundecibelp(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvundecibelp(y);

    return y;
}

spDVector xdvundecibelp(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvundecibelp(y);

    return y;
}

void fvdecibel(spFVector x)
{
    /* |x|^2 */
    fvsquare(x);

    fvdecibelp(x);

    return;
}

void dvdecibel(spDVector x)
{
    /* |x|^2 */
    dvsquare(x);
    
    dvdecibelp(x);

    return;
}

spFVector xfvdecibel(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvdecibel(y);

    return y;
}

spDVector xdvdecibel(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvdecibel(y);

    return y;
}

void fvundecibel(spFVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->undecibela != NULL) {
            flist->undecibela(x->instance, x->length);
        } else {
            fvscoper(x, "/", 20.0f);
            fvscoper(x, "!^", 10.0f);
        }
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = powf(10.0f, x->data[k] / 20.0f);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvundecibel(spDVector x)
{
    long k;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE) {
        spVectorPluginInternalFuncList *flist;
        if ((flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->undecibela != NULL) {
            flist->undecibela(x->instance, x->length);
        } else {
            dvscoper(x, "/", 20.0);
            dvscoper(x, "!^", 10.0);
        }
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = pow(10.0, x->data[k] / 20.0);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvundecibel(spFVector x)
{
    spFVector y;

    y = xfvclone(x);
    fvundecibel(y);

    return y;
}

spDVector xdvundecibel(spDVector x)
{
    spDVector y;

    y = xdvclone(x);
    dvundecibel(y);

    return y;
}

void fvrandun(spFVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->randun != NULL) {
	flist->randun(x->instance, x->length);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = (float)randun();
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvrandun(spDVector x)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->randun != NULL) {
	flist->randun(x->instance, x->length);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = randun();
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvrandunul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector x;

    x = xfvallocul(plugin, length, unlock_flag);
    fvrandun(x);

    return x;
}

spDVector xdvrandunul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector x;

    x = xdvallocul(plugin, length, unlock_flag);
    dvrandun(x);

    return x;
}

spFVector xfvrandun(long length)
{
    return xfvrandunul(NULL, length, SP_FALSE);
}

spDVector xdvrandun(long length)
{
    return xdvrandunul(NULL, length, SP_FALSE);
}

void fvgauss(spFVector x, float mu, float sigma)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->gauss != NULL) {
	flist->gauss(x->instance, x->length, mu, sigma);
	return;
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = spGaussf(mu, sigma);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvgauss(spDVector x, double mu, double sigma)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->gauss != NULL) {
	flist->gauss(x->instance, x->length, mu, sigma);
	return;
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = spGauss(mu, sigma);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

spFVector xfvgaussul(spPlugin *plugin, long length, float mu, float sigma, spBool unlock_flag)
{
    spFVector x;

    x = xfvallocul(plugin, length, unlock_flag);
    fvgauss(x, mu, sigma);

    return x;
}

spDVector xdvgaussul(spPlugin *plugin, long length, double mu, double sigma, spBool unlock_flag)
{
    spDVector x;

    x = xdvallocul(plugin, length, unlock_flag);
    dvgauss(x, mu, sigma);

    return x;
}

spFVector xfvgauss(long length, float mu, float sigma)
{
    return xfvgaussul(NULL, length, mu, sigma, SP_FALSE);
}

spDVector xdvgauss(long length, double mu, double sigma)
{
    return xdvgaussul(NULL, length, mu, sigma, SP_FALSE);
}

void fvrandn(spFVector x)
{
    fvgauss(x, 0.0f, 1.0f);
    return;
}

void dvrandn(spDVector x)
{
    dvgauss(x, 0.0, 1.0);
    return;
}

spFVector xfvrandnul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector x;

    x = xfvallocul(plugin, length, unlock_flag);
    fvrandn(x);

    return x;
}

spDVector xdvrandnul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector x;

    x = xdvallocul(plugin, length, unlock_flag);
    dvrandn(x);

    return x;
}

spFVector xfvrandn(long length)
{
    return xfvrandnul(NULL, length, SP_FALSE);
}

spDVector xdvrandn(long length)
{
    return xdvrandnul(NULL, length, SP_FALSE);
}

double dvscpolyval(spDVector p, double x)
{
    long n, k;
    double y;

    y = 0.0;
    for (k = 0; k < p->length; k++) {
        n = p->length - k - 1;
        if (n >= 2) {
            y += p->data[k] * pow(x, (double)n);
        } else if (n == 1) {
            y += p->data[k] * x;
        } else {
            y += p->data[k];
        }
    }

    return y;
}

void dvcplxpolyval(spDVector p, double xre, double xim, double *oyre, double *oyim)
{
    long n, k, m;
    double xnre, xnim;
    double yre, yim;
    double tre, tim;

    yre = yim = 0.0;
    for (k = 0; k < p->length; k++) {
        n = p->length - k - 1;
        xnre = p->data[k];
        if (p->imag != NULL) {
            xnim = p->imag[k];
        } else {
            xnim = 0.0;
        }
        for (m = 0; m < n; m++) {
            tre = xnre * xre - xnim * xim;
            tim = xnre * xim + xnim * xre;
            xnre = tre;
            xnim = tim;
        }
        yre += xnre;
        yim += xnim;
    }

    *oyre = yre;
    *oyim = yim;

    return;
}

void dvpolyval(spDVector iox, spDVector p)
{
    long m;
    
    if (iox->imag == NULL && p->imag == NULL) {
        for (m = 0; m < iox->length; m++) {
            iox->data[m] = dvscpolyval(p, iox->data[m]);
        }
    } else {
        if (iox->imag == NULL) {
            dvizeros(iox, iox->length);
        }
        for (m = 0; m < iox->length; m++) {
            dvcplxpolyval(p, iox->data[m], iox->imag[m], &iox->data[m], &iox->imag[m]);
        }
    }

    return;
}

spDVector xdvpolyval(spDVector x, spDVector p)
{
    spDVector y;

    y = xdvclone(x);
    dvpolyval(y, p);

    return y;
}
