/** @file sp/vmath.h
 */

#ifndef SPLIB_VMATH_H
#define SPLIB_VMATH_H

#include <sp/vector.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(MACOS)
#pragma import on
#endif

/** @~english @defgroup vmathGroup <sp/vmath.h>: Math Functions for Vector
@code
#include <sp/vmath.h>
@endcode
 */

/** @addtogroup vmathGroup
 *  @{ */  /*----@addtogroup vmathGroup----*/

/** @~english @name Absolute Value (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvabs()
 *  @public @memberof spSVector */
extern void svabs(spSVector x);
/** @copydoc spDVector::dvabs()
 *  @public @memberof spLVector */
extern void lvabs(spLVector x);
/** @copydoc spDVector::dvabs()
 *  @public @memberof spFVector */
extern void fvabs(spFVector x);
/** @~english Calculates the absolute value of each element of \p x .
 *  If the imaginary part of \p x exists, it will be disposed.
 *  @public @memberof spDVector */
extern void dvabs(spDVector x);
/** @copydoc spDVector::xdvabs()
 *  @public @memberof spSVector */
extern spSVector xsvabs(spSVector x);
/** @copydoc spDVector::xdvabs()
 *  @public @memberof spLVector */
extern spLVector xlvabs(spLVector x);
/** @copydoc spDVector::xdvabs()
 *  @public @memberof spFVector */
extern spFVector xfvabs(spFVector x);
/** @~english Creates a new vector whose each element is the absolute value of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvabs(spDVector x);

/** @~english @} */

/** @~english @name Square Root (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvsqrt
 *  @public @memberof spSVector */
extern void svsqrt(spSVector x);
/** @copydoc spDVector::dvsqrt
 *  @public @memberof spLVector */
extern void lvsqrt(spLVector x);
/** @copydoc spDVector::dvsqrt
 *  @public @memberof spFVector */
extern void fvsqrt(spFVector x);
/** @~english Calculates the square root of each element of \p x .
 *  @public @memberof spDVector */
extern void dvsqrt(spDVector x);
/** @copydoc spDVector::xdvsqrt
 *  @public @memberof spSVector */
extern spSVector xsvsqrt(spSVector x);
/** @copydoc spDVector::xdvsqrt
 *  @public @memberof spLVector */
extern spLVector xlvsqrt(spLVector x);
/** @copydoc spDVector::xdvsqrt
 *  @public @memberof spFVector */
extern spFVector xfvsqrt(spFVector x);
/** @~english Creates a new vector whose each element is the square root of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvsqrt(spDVector x);

/** @~english @} */

/** @~english @name Square (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvsquare
 *  @public @memberof spSVector */
extern void svsquare(spSVector x);
/** @copydoc spDVector::dvsquare
 *  @public @memberof spLVector */
extern void lvsquare(spLVector x);
/** @copydoc spDVector::dvsquare
 *  @public @memberof spFVector */
extern void fvsquare(spFVector x);
/** @~english Calculates `|x|^2` of each element of \p x .
 *  @public @memberof spDVector */
extern void dvsquare(spDVector x);
/** @copydoc spDVector::xdvsquare
 *  @public @memberof spSVector */
extern spSVector xsvsquare(spSVector x);
/** @copydoc spDVector::xdvsquare
 *  @public @memberof spLVector */
extern spLVector xlvsquare(spLVector x);
/** @copydoc spDVector::xdvsquare
 *  @public @memberof spFVector */
extern spFVector xfvsquare(spFVector x);
/** @~english Creates a new vector whose each element is `|x|^2` of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvsquare(spDVector x);

/** @~english @} */

/** @~english @name Sign Function (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvsign
 *  @public @memberof spSVector */
extern void svsign(spSVector x);
/** @copydoc spDVector::dvsign
 *  @public @memberof spLVector */
extern void lvsign(spLVector x);
/** @copydoc spDVector::dvsign
 *  @public @memberof spFVector */
extern void fvsign(spFVector x);
/** @~english Calculates the sign function of each element of \p x .
 *  The sign function returns 1 if the element's value is greater than 0, 0 if that is equal to 0, or -1 if that is less than 0.
 *  @public @memberof spDVector */
extern void dvsign(spDVector x);
/** @copydoc spDVector::xdvsign
 *  @public @memberof spSVector */
extern spSVector xsvsign(spSVector x);
/** @copydoc spDVector::xdvsign
 *  @public @memberof spLVector */
extern spLVector xlvsign(spLVector x);
/** @copydoc spDVector::xdvsign
 *  @public @memberof spFVector */
extern spFVector xfvsign(spFVector x);
/** @~english Creates a new vector whose each element is output of the sign function of the corresponding element of \p x .
 *  The sign function returns 1 if the element's value is greater than 0, 0 if that is equal to 0, or -1 if that is less than 0.
 *  @public @memberof spDVector */
extern spDVector xdvsign(spDVector x);

/** @~english @} */

/** @~english @name Cumulative Sum (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvcumsum
 *  @public @memberof spSVector */
extern void svcumsum(spSVector x);
/** @copydoc spDVector::dvcumsum
 *  @public @memberof spLVector */
extern void lvcumsum(spLVector x);
/** @copydoc spDVector::dvcumsum
 *  @public @memberof spFVector */
extern void fvcumsum(spFVector x);
/** @~english Calculates the cumulative sum of the elements of \p x .
 *  @public @memberof spDVector */
extern void dvcumsum(spDVector x);
/** @copydoc spDVector::xdvcumsum
 *  @public @memberof spSVector */
extern spSVector xsvcumsum(spSVector x);
/** @copydoc spDVector::xdvcumsum
 *  @public @memberof spLVector */
extern spLVector xlvcumsum(spLVector x);
/** @copydoc spDVector::xdvcumsum
 *  @public @memberof spFVector */
extern spFVector xfvcumsum(spFVector x);
/** @~english Creates a new vector containing the cumulative sum of the elements of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvcumsum(spDVector x);

/** @~english @} */

/** @~english @name Cumulative Product (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvcumprod
 *  @public @memberof spSVector */
extern void svcumprod(spSVector x);
/** @copydoc spDVector::dvcumprod
 *  @public @memberof spLVector */
extern void lvcumprod(spLVector x);
/** @copydoc spDVector::dvcumprod
 *  @public @memberof spFVector */
extern void fvcumprod(spFVector x);
/** @~english Calculates the cumulative product of the elements of \p x .
 *  @public @memberof spDVector */
extern void dvcumprod(spDVector x);
/** @copydoc spDVector::xdvcumprod
 *  @public @memberof spSVector */
extern spSVector xsvcumprod(spSVector x);
/** @copydoc spDVector::xdvcumprod
 *  @public @memberof spLVector */
extern spLVector xlvcumprod(spLVector x);
/** @copydoc spDVector::xdvcumprod
 *  @public @memberof spFVector */
extern spFVector xfvcumprod(spFVector x);
/** @~english Creates a new vector containing the cumulative product of the elements of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvcumprod(spDVector x);

/** @~english @} */

/** @~english @name Differences (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvcodiff
 *  @public @memberof spSVector */
extern void svcodiff(spSVector x, double coef);
/** @copydoc spDVector::dvcodiff
 *  @public @memberof spLVector */
extern void lvcodiff(spLVector x, double coef);
/** @copydoc spDVector::dvcodiff
 *  @public @memberof spFVector */
extern void fvcodiff(spFVector x, double coef);
/** @~english Calculates the differences between the elements of \p x using \p coef as a weighting factor.
 *  For example, the `k`-th element of the differences is `x->data[k + 1] - coef * x->data[k]`.
 *  The final element is calculated by regarding `x->data[k + 1] = 0`.
 *  @public @memberof spDVector */
extern void dvcodiff(spDVector x, double coef);
/** @copydoc spDVector::xdvcodiff
 *  @public @memberof spSVector */
extern spSVector xsvcodiff(spSVector x, double coef);
/** @copydoc spDVector::xdvcodiff
 *  @public @memberof spLVector */
extern spLVector xlvcodiff(spLVector x, double coef);
/** @copydoc spDVector::xdvcodiff
 *  @public @memberof spFVector */
extern spFVector xfvcodiff(spFVector x, double coef);
/** @~english Creates a new vector containing the differences between the elements of \p x using \p coef as a weighting factor.
 *  For example, the `k`-th element of the new vector is `x->data[k + 1] - coef * x->data[k]`.
 *  The number of elements of the new vector is `x->length - 1`.
 *  @public @memberof spDVector */
extern spDVector xdvcodiff(spDVector x, double coef);

/** @copydoc spDVector::dvdiff
 *  @public @memberof spSVector */
#define svdiff(x) svcodiff(x, 1.0);
/** @copydoc spDVector::dvdiff
 *  @public @memberof spLVector */
#define lvdiff(x) lvcodiff(x, 1.0);
/** @copydoc spDVector::dvdiff
 *  @public @memberof spFVector */
#define fvdiff(x) fvcodiff(x, 1.0);
/** @~english Calculates the differences between the elements of \p x .
 *  For example, the `k`-th element of the differences is `x->data[k + 1] - x->data[k]`.
 *  The final element is calculated by regarding `x->data[k + 1] = 0`.
 *  @public @memberof spDVector */
#define dvdiff(x) dvcodiff(x, 1.0);
/** @copydoc spDVector::xdvdiff
 *  @public @memberof spSVector */
#define xsvdiff(x) xsvcodiff(x, 1.0);
/** @copydoc spDVector::xdvdiff
 *  @public @memberof spLVector */
#define xlvdiff(x) xlvcodiff(x, 1.0);
/** @copydoc spDVector::xdvdiff
 *  @public @memberof spFVector */
#define xfvdiff(x) xfvcodiff(x, 1.0);
/** @~english Creates a new vector containing the differences between the elements of \p x .
 *  For example, the `k`-th element of the new vector is `x->data[k + 1] - x->data[k]`.
 *  The number of elements of the new vector is `x->length - 1`.
 *  @public @memberof spDVector */
#define xdvdiff(x) xdvcodiff(x, 1.0);
    
/** @~english @} */

/** @~english @name Sum (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvsum
 *  @public @memberof spSVector */
extern long svsum(spSVector x);
/** @copydoc spDVector::dvsum
 *  @public @memberof spLVector */
extern long lvsum(spLVector x);
/** @copydoc spDVector::dvsum
 *  @public @memberof spFVector */
extern float fvsum(spFVector x);
/** @~english Calculates the sum of the real elements of \p x .
 *  @public @memberof spDVector */
extern double dvsum(spDVector x);

/** @copydoc spDVector::dvisum
 *  @public @memberof spSVector */
extern long svisum(spSVector x);
/** @copydoc spDVector::dvisum
 *  @public @memberof spLVector */
extern long lvisum(spLVector x);
/** @copydoc spDVector::dvisum
 *  @public @memberof spFVector */
extern float fvisum(spFVector x);
/** @~english Calculates the sum of the imaginary elements of \p x .
 *  @public @memberof spDVector */
extern double dvisum(spDVector x);

/** @copydoc spDVector::dvsqsum
 *  @public @memberof spSVector */
extern long svsqsum(spSVector x);
/** @copydoc spDVector::dvsqsum
 *  @public @memberof spLVector */
extern long lvsqsum(spLVector x);
/** @copydoc spDVector::dvsqsum
 *  @public @memberof spFVector */
extern float fvsqsum(spFVector x);
/** @~english Calculates the sum of the absolute square elements of \p x .
 *  @public @memberof spDVector */
extern double dvsqsum(spDVector x);

/** @copydoc spDVector::dvabssum
 *  @public @memberof spSVector */
extern long svabssum(spSVector x);
/** @copydoc spDVector::dvabssum
 *  @public @memberof spLVector */
extern long lvabssum(spLVector x);
/** @copydoc spDVector::dvabssum
 *  @public @memberof spFVector */
extern float fvabssum(spFVector x);
/** @~english Calculates the sum of the absolute elements of \p x .
 *  @public @memberof spDVector */
extern double dvabssum(spDVector x);

/** @~english @} */

/** @~english @name Product (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvprod
 *  @public @memberof spSVector */
extern long svprod(spSVector x);
/** @copydoc spDVector::dvprod
 *  @public @memberof spLVector */
extern long lvprod(spLVector x);
/** @copydoc spDVector::dvprod
 *  @public @memberof spFVector */
extern float fvprod(spFVector x);
/** @~english Calculates the product of the real elements of \p x .
 *  @public @memberof spDVector */
extern double dvprod(spDVector x);

/** @copydoc spDVector::dvriprod
 *  @public @memberof spSVector */
extern long svriprod(spSVector x, long *oiprod);
/** @copydoc spDVector::dvriprod
 *  @public @memberof spLVector */
extern long lvriprod(spLVector x, long *oiprod);
/** @copydoc spDVector::dvriprod
 *  @public @memberof spFVector */
extern float fvriprod(spFVector x, float *oiprod);
/** @~english Calculates the product of the real and imaginary elements of \p x .
 *  @public @memberof spDVector */
extern double dvriprod(spDVector x, double *oiprod);

/** @~english @} */

/** @~english @name Norm (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvnorm
 *  @public @memberof spSVector */
extern double svnorm(spSVector x, long p);
/** @copydoc spDVector::dvnorm
 *  @public @memberof spLVector */
extern double lvnorm(spLVector x, long p);
/** @copydoc spDVector::dvnorm
 *  @public @memberof spFVector */
extern float fvnorm(spFVector x, long p);
/** @~english Calculates the norm of the input vector \p x . The value of \p p must be greater than or equal to 1
    except that `p = 0`: the default Euclidean norm which corresponds to `p = 2`, and `p = -1`: the infinity norm.
 *  @param[in] x The input vector.
 *  @param[in] p The number which is greater than or equal to -1 corresponding to the norm type.
 *  @return The calculated norm.
 *  @public @memberof spDVector */
extern double dvnorm(spDVector x, long p);

/** @~english @} */

/** @~english @name Dot Product (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvdot
 *  @public @memberof spSVector */
extern long svdot(spSVector x, spSVector y);
/** @copydoc spDVector::dvdot
 *  @public @memberof spLVector */
extern long lvdot(spLVector x, spLVector y);
/** @copydoc spDVector::dvdot
 *  @public @memberof spFVector */
extern float fvdot(spFVector x, spFVector y);
/** @~english Calculates the scalar dot product of \p x and \p y for the real part.
 *  @public @memberof spDVector */
extern double dvdot(spDVector x, spDVector y);

/** @copydoc spDVector::dvcplxdot
 *  @public @memberof spSVector */
extern void svcplxdot(spSVector x, spSVector y, long *re, long *im);
/** @copydoc spDVector::dvcplxdot
 *  @public @memberof spLVector */
extern void lvcplxdot(spLVector x, spLVector y, long *re, long *im);
/** @copydoc spDVector::dvcplxdot
 *  @public @memberof spFVector */
extern void fvcplxdot(spFVector x, spFVector y, float *re, float *im);
/** @~english Calculates the scalar dot product of \p x and \p y for complex data.
 *  @public @memberof spDVector */
extern void dvcplxdot(spDVector x, spDVector y, double *re, double *im);

/** @~english @} */

/** @~english @name Mean and Variance (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvvar
 *  @public @memberof spSVector */
extern double svvar(spSVector x, double *mean);
/** @copydoc spDVector::dvvar
 *  @public @memberof spLVector */
extern double lvvar(spLVector x, double *mean);
/** @copydoc spDVector::dvvar
 *  @public @memberof spFVector */
extern float fvvar(spFVector x, float *mean);
/** @~english Calculates the variance of the real elements of \p x .
 *  @param[in] x The input vector.
 *  @param[out] mean The address of a variable to receive the mean of real elements of \p x. This can be NULL.
 *  @return Returns the value of the variance.
 *  @public @memberof spDVector */
extern double dvvar(spDVector x, double *mean);

/** @copydoc spDVector::dvstd
 *  @public @memberof spSVector */
extern double svstd(spSVector x, double *mean);
/** @copydoc spDVector::dvstd
 *  @public @memberof spLVector */
extern double lvstd(spLVector x, double *mean);
/** @copydoc spDVector::dvstd
 *  @public @memberof spFVector */
extern float fvstd(spFVector x, float *mean);
/** @~english Calculates the standard deviation of the real elements of \p x .
 *  @param[in] x The input vector.
 *  @param[out] mean The address of a variable to receive the mean of real elements of \p x. This can be NULL.
 *  @return Returns the value of the standard deviation.
 *  @public @memberof spDVector */
extern double dvstd(spDVector x, double *mean);

/** @copydoc spDVector::dvmean
 *  @public @memberof spSVector */
#define svmean(x) ((double)svsum(x) / (double)x->length)
/** @copydoc spDVector::dvmean
 *  @public @memberof spLVector */
#define lvmean(x) ((double)lvsum(x) / (double)x->length)
/** @copydoc spDVector::dvmean
 *  @public @memberof spFVector */
#define fvmean(x) (fvsum(x) / (float)x->length)
/** @~english Calculates the mean of the real elements of \p x .
 *  @public @memberof spDVector */
#define dvmean(x) (dvsum(x) / (double)x->length)

/** @~english @} */

/** @~english @name Minimum and Maximum (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvmin
 *  @public @memberof spSVector */
extern short svmin(spSVector x, long *index);
/** @copydoc spDVector::dvmin
 *  @public @memberof spLVector */
extern long lvmin(spLVector x, long *index);
/** @copydoc spDVector::dvmin
 *  @public @memberof spFVector */
extern float fvmin(spFVector x, long *index);
/** @~english Finds the smallest real elements of \p x .
 *  @param[in] x The input vector.
 *  @param[in] index The address of a variable to receive the index of the smallest element. This can be NULL.
 *  @return Returns the smallest value.
 *  @public @memberof spDVector */
extern double dvmin(spDVector x, long *index);

/** @copydoc spDVector::dvmax
 *  @public @memberof spSVector */
extern short svmax(spSVector x, long *index);
/** @copydoc spDVector::dvmax
 *  @public @memberof spLVector */
extern long lvmax(spLVector x, long *index);
/** @copydoc spDVector::dvmax
 *  @public @memberof spFVector */
extern float fvmax(spFVector x, long *index);
/** @~english Finds the largest real elements of \p x .
 *  @param[in] x The input vector.
 *  @param[in] index The address of a variable to receive the index of the largest element. This can be NULL.
 *  @return Returns the largest value.
 *  @public @memberof spDVector */
extern double dvmax(spDVector x, long *index);

/** @copydoc spDVector::dvscmin
 *  @public @memberof spSVector */
extern void svscmin(spSVector x, short a);
/** @copydoc spDVector::dvscmin
 *  @public @memberof spLVector */
extern void lvscmin(spLVector x, long a);
/** @copydoc spDVector::dvscmin
 *  @public @memberof spFVector */
extern void fvscmin(spFVector x, float a);
/** @~english Takes the smallest value between each element of \p x and \p a .
 *  @public @memberof spDVector */
extern void dvscmin(spDVector x, double a);

/** @copydoc spDVector::dvscmax
 *  @public @memberof spSVector */
extern void svscmax(spSVector x, short a);
/** @copydoc spDVector::dvscmax
 *  @public @memberof spLVector */
extern void lvscmax(spLVector x, long a);
/** @copydoc spDVector::dvscmax
 *  @public @memberof spFVector */
extern void fvscmax(spFVector x, float a);
/** @~english Takes the largest value between each element of \p x and \p a .
 *  @public @memberof spDVector */
extern void dvscmax(spDVector x, double a);

/** @copydoc spDVector::dvelmin
 *  @public @memberof spSVector */
extern void svelmin(spSVector x, spSVector y);
/** @copydoc spDVector::dvelmin
 *  @public @memberof spLVector */
extern void lvelmin(spLVector x, spLVector y);
/** @copydoc spDVector::dvelmin
 *  @public @memberof spFVector */
extern void fvelmin(spFVector x, spFVector y);
/** @~english Takes the smallest value between each element of \p x and that of \p y .
 *  @public @memberof spDVector */
extern void dvelmin(spDVector x, spDVector y);

/** @copydoc spDVector::dvelmax
 *  @public @memberof spSVector */
extern void svelmax(spSVector x, spSVector y);
/** @copydoc spDVector::dvelmax
 *  @public @memberof spLVector */
extern void lvelmax(spLVector x, spLVector y);
/** @copydoc spDVector::dvelmax
 *  @public @memberof spFVector */
extern void fvelmax(spFVector x, spFVector y);
/** @~english Takes the largest value between each element of \p x and that of \p y .
 *  @public @memberof spDVector */
extern void dvelmax(spDVector x, spDVector y);

/** @~english @} */

/** @internal */
#define SP_NMINMAX_OPTION_PEAK_ONLY 1
#define SP_NMINMAX_OPTION_ZERO_EDGES (1<<1)
extern spLVector xsvnmin(spSVector x, long n, long margin, unsigned long options);
extern spLVector xlvnmin(spLVector x, long n, long margin, unsigned long options);
extern spLVector xfvnmin(spFVector x, long n, long margin, unsigned long options);
extern spLVector xdvnmin(spDVector x, long n, long margin, unsigned long options);
extern spLVector xsvnmax(spSVector x, long n, long margin, unsigned long options);
extern spLVector xlvnmax(spLVector x, long n, long margin, unsigned long options);
extern spLVector xfvnmax(spFVector x, long n, long margin, unsigned long options);
extern long dvnmax(spLVector idx, spDVector x, long margin, unsigned long options);
extern spLVector xdvnmax(spDVector x, long n, long margin, unsigned long options);
    
/** @~english @name Sort and Median (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvsort
 *  @public @memberof spSVector */
extern void svsort(spSVector x);
/** @copydoc spDVector::dvsort
 *  @public @memberof spLVector */
extern void lvsort(spLVector x);
/** @copydoc spDVector::dvsort
 *  @public @memberof spFVector */
extern void fvsort(spFVector x);
/** @~english Sorts the real elements of \p x in ascending order.
 *  @public @memberof spDVector */
extern void dvsort(spDVector x);
/** @copydoc spDVector::xdvsort
 *  @public @memberof spSVector */
extern spSVector xsvsort(spSVector x);
/** @copydoc spDVector::xdvsort
 *  @public @memberof spLVector */
extern spLVector xlvsort(spLVector x);
/** @copydoc spDVector::xdvsort
 *  @public @memberof spFVector */
extern spFVector xfvsort(spFVector x);
/** @~english Creates a new vector containing the sorted real elements \p x in ascending order.
 *  @public @memberof spDVector */
extern spDVector xdvsort(spDVector x);

/** @internal */
extern void spheapsort(void *keysptr, void *valuesptr, long length,
		       int (*compare_func)(void *keysptr, long i, long j),
		       void (*swap_func)(void *keysptr, void *valuesptr, long i, long j));
    
/** @copydoc spDVector::xdvsortidx
 *  @public @memberof spSVector */
extern spLVector xsvsortidx(spSVector x);
/** @copydoc spDVector::xdvsortidx
 *  @public @memberof spLVector */
extern spLVector xlvsortidx(spLVector x);
/** @copydoc spDVector::xdvsortidx
 *  @public @memberof spFVector */
extern spLVector xfvsortidx(spFVector x);
/** @~english Sorts the real elements of \p x in ascending order and returns the index vector which describes the rearrangement of th elements in sorting.
 *  @param[in,out] x The input vector. After calling this function, the elements are sorted in ascending order.
 *  @return The memory-allocated index vector of spLVector.
 *  @public @memberof spDVector */
extern spLVector xdvsortidx(spDVector x);

/** @copydoc spDVector::dvmedian
 *  @public @memberof spSVector */
extern double svmedian(spSVector x);
/** @copydoc spDVector::dvmedian
 *  @public @memberof spLVector */
extern double lvmedian(spLVector x);
/** @copydoc spDVector::dvmedian
 *  @public @memberof spFVector */
extern float fvmedian(spFVector x);
/** @~english Sorts the real elements of \p x and calculates the median value.
 *  @param[in,out] x The input vector. After calling this function, the elements are sorted in ascending order.
 *  @return The median value.
 *  @public @memberof spDVector */
extern double dvmedian(spDVector x);
/** @copydoc spDVector::xdvmedian
 *  @public @memberof spSVector */
extern spSVector xsvmedian(spSVector x, double *median);
/** @copydoc spDVector::xdvmedian
 *  @public @memberof spLVector */
extern spLVector xlvmedian(spLVector x, double *median);
/** @copydoc spDVector::xdvmedian
 *  @public @memberof spFVector */
extern spFVector xfvmedian(spFVector x, float *median);
/** @~english Sorts the real elements of \p x and calculates the median value.
 *  @param[in] x The input vector. 
 *  @param[out] median The address of a variable to receive the median value.
 *  @return The memory-allocated vector sorted in ascending order.
 *  @public @memberof spDVector */
extern spDVector xdvmedian(spDVector x, double *median);

/** @~english @} */

/** @~english @name Zero-Crossing Count (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvzerocross
 *  @public @memberof spSVector */
extern long svzerocross(spSVector x, int slope_sign);
/** @copydoc spDVector::dvzerocross
 *  @public @memberof spLVector */
extern long lvzerocross(spLVector x, int slope_sign);
/** @copydoc spDVector::dvzerocross
 *  @public @memberof spFVector */
extern long fvzerocross(spFVector x, int slope_sign);
/** @~english Counts the number of zero-crossings of the input vector \p x .
 *  @param[in] x The input vector. 
 *  @param[in] slope_sign The switch to determine which slope direction is used to calculate the zero-crossings;
 *  0: both directions, 1: positive direction only, and -1: negative direction only.
 *  @return The number of zero-crossings.
 *  @public @memberof spDVector */
extern long dvzerocross(spDVector x, int slope_sign);

/** @~english @} */

/*-- the following functions are supported for float and double only --*/

/** @~english @name Phase Angle (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvangle
 *  @public @memberof spFVector */
extern void fvangle(spFVector x);
/** @copydoc spDVector::xdvangle
 *  @public @memberof spFVector */
extern spFVector xfvangle(spFVector x);
/** @~english Calculates the phase angle of each element of \p x .
 *  @public @memberof spDVector */
extern void dvangle(spDVector x);
/** @~english Creates a new vector whose each element is the phase angle of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvangle(spDVector x);

/** @~english @} */

/** @~english @name Trigonometric functions (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvcos
 *  @public @memberof spFVector */
extern void fvcos(spFVector x);
/** @~english Calculates \p cos of each element of \p x .
 *  @public @memberof spDVector */
extern void dvcos(spDVector x);
/** @copydoc spDVector::xdvcos
 *  @public @memberof spFVector */
extern spFVector xfvcos(spFVector x);
/** @~english Creates a new vector whose each element is \p cos of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvcos(spDVector x);

/** @copydoc spDVector::dvsin
 *  @public @memberof spFVector */
extern void fvsin(spFVector x);
/** @~english Calculates \p sin of each element of \p x .
 *  @public @memberof spDVector */
extern void dvsin(spDVector x);
/** @copydoc spDVector::xdvsin
 *  @public @memberof spFVector */
extern spFVector xfvsin(spFVector x);
/** @~english Creates a new vector whose each element is \p sin of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvsin(spDVector x);

/** @copydoc spDVector::dvtan
 *  @public @memberof spFVector */
extern void fvtan(spFVector x);
/** @~english Calculates \p tan of each element of \p x .
 *  @public @memberof spDVector */
extern void dvtan(spDVector x);
/** @copydoc spDVector::xdvtan
 *  @public @memberof spFVector */
extern spFVector xfvtan(spFVector x);
/** @~english Creates a new vector whose each element is \p tan of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvtan(spDVector x);

/** @copydoc spDVector::dvcosh
 *  @public @memberof spFVector */
extern void fvcosh(spFVector x);
/** @~english Calculates \p cosh of each element of \p x .
 *  @public @memberof spDVector */
extern void dvcosh(spDVector x);
/** @copydoc spDVector::xdvcosh
 *  @public @memberof spFVector */
extern spFVector xfvcosh(spFVector x);
/** @~english Creates a new vector whose each element is \p cosh of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvcosh(spDVector x);

/** @copydoc spDVector::dvsinh
 *  @public @memberof spFVector */
extern void fvsinh(spFVector x);
/** @~english Calculates \p sinh of each element of \p x .
 *  @public @memberof spDVector */
extern void dvsinh(spDVector x);
/** @copydoc spDVector::xdvsinh
 *  @public @memberof spFVector */
extern spFVector xfvsinh(spFVector x);
/** @~english Creates a new vector whose each element is \p sinh of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvsinh(spDVector x);

/** @copydoc spDVector::dvtanh
 *  @public @memberof spFVector */
extern void fvtanh(spFVector x);
/** @~english Calculates \p tanh of each element of \p x .
 *  @public @memberof spDVector */
extern void dvtanh(spDVector x);
/** @copydoc spDVector::xdvtanh
 *  @public @memberof spFVector */
extern spFVector xfvtanh(spFVector x);
/** @~english Creates a new vector whose each element is \p tanh of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvtanh(spDVector x);

/** @copydoc spDVector::dvacos
 *  @public @memberof spFVector */
extern void fvacos(spFVector x);
/** @~english Calculates \p acos of each element of \p x .
 *  @public @memberof spDVector */
extern void dvacos(spDVector x);
/** @copydoc spDVector::xdvacos
 *  @public @memberof spFVector */
extern spFVector xfvacos(spFVector x);
/** @~english Creates a new vector whose each element is \p acos of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvacos(spDVector x);

/** @copydoc spDVector::dvasin
 *  @public @memberof spFVector */
extern void fvasin(spFVector x);
/** @~english Calculates \p asin of each element of \p x .
 *  @public @memberof spDVector */
extern void dvasin(spDVector x);
/** @copydoc spDVector::xdvasin
 *  @public @memberof spFVector */
extern spFVector xfvasin(spFVector x);
/** @~english Creates a new vector whose each element is \p asin of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvasin(spDVector x);

/** @copydoc spDVector::dvatan
 *  @public @memberof spFVector */
extern void fvatan(spFVector x);
/** @~english Calculates \p atan of each element of \p x .
 *  @public @memberof spDVector */
extern void dvatan(spDVector x);
/** @copydoc spDVector::xdvatan
 *  @public @memberof spFVector */
extern spFVector xfvatan(spFVector x);
/** @~english Creates a new vector whose each element is \p atan of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvatan(spDVector x);

/** @copydoc spDVector::dvatan2
 *  @public @memberof spFVector */
extern void fvatan2(spFVector y, spFVector x, spBool reverse);
/** @~english Calculates `atan2(y, x)` for each element and overwrites the result to \p y .
 *  If reverse is SP_TRUE , this function calculates `atan2(x, y)` for each element and overwrites the result to \p y .
 *  The imaginary parts of \p y and \p x are not used.
 *  @public @memberof spDVector */
extern void dvatan2(spDVector y, spDVector x, spBool reverse);
/** @copydoc spDVector::xdvatan2
 *  @public @memberof spFVector */
extern spFVector xfvatan2(spFVector y, spFVector x);
/** @~english Creates a new vector whose each element is \p atan2 of the corresponding element of \p x .
 *  The imaginary parts of \p y and \p x are not used.
 *  @public @memberof spDVector */
extern spDVector xdvatan2(spDVector y, spDVector x);

/** @~english @} */

/** @~english @name Exponents and Logarithms (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvexp
 *  @public @memberof spFVector */
extern void fvexp(spFVector x);
/** @~english Calculates \p exp of each element of \p x .
 *  @public @memberof spDVector */
extern void dvexp(spDVector x);
/** @copydoc spDVector::xdvexp
 *  @public @memberof spFVector */
extern spFVector xfvexp(spFVector x);
/** @~english Creates a new vector whose each element is \p exp of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvexp(spDVector x);

/** @copydoc spDVector::dvlog
 *  @public @memberof spFVector */
extern void fvlog(spFVector x);
/** @~english Calculates \p log of each element of \p x .
 *  @public @memberof spDVector */
extern void dvlog(spDVector x);
/** @copydoc spDVector::xdvlog
 *  @public @memberof spFVector */
extern spFVector xfvlog(spFVector x);
/** @~english Creates a new vector whose each element is \p log of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvlog(spDVector x);

/** @copydoc spDVector::dvlog10
 *  @public @memberof spFVector */
extern void fvlog10(spFVector x);
/** @~english Calculates \p log10 of each element of \p x .
 *  @public @memberof spDVector */
extern void dvlog10(spDVector x);
/** @copydoc spDVector::xdvlog10
 *  @public @memberof spFVector */
extern spFVector xfvlog10(spFVector x);
/** @~english Creates a new vector whose each element is \p log10 of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvlog10(spDVector x);    

/** @copydoc spDVector::dvdecibela
 *  @public @memberof spFVector */
extern void fvdecibela(spFVector x);
/** @~english Calculates `20*log10(x)` of each element of \p x .
 *  If \p x includes negative values, the result is undefined.
 *  @public @memberof spDVector */
extern void dvdecibela(spDVector x);
/** @copydoc spDVector::xdvdecibela
 *  @public @memberof spFVector */
extern spFVector xfvdecibela(spFVector x);
/** @~english Creates a new vector whose each element is `20*log10(x)` of the corresponding element of \p x .
 *  If \p x includes negative values, the result is undefined.
 *  @public @memberof spDVector */
extern spDVector xdvdecibela(spDVector x);
    
/** @copydoc spDVector::dvdecibelp
 *  @public @memberof spFVector */
extern void fvdecibelp(spFVector x);
/** @~english Calculates `10*log10(x)` of each element of \p x .
 *  If \p x includes negative values, the result is undefined.
 *  @public @memberof spDVector */
extern void dvdecibelp(spDVector x);
/** @copydoc spDVector::xdvdecibelp
 *  @public @memberof spFVector */
extern spFVector xfvdecibelp(spFVector x);
/** @~english Creates a new vector whose each element is `10*log10(x)` of the corresponding element of \p x .
 *  If \p x includes negative values, the result is undefined.
 *  @public @memberof spDVector */
extern spDVector xdvdecibelp(spDVector x);

/** @copydoc spDVector::dvundecibelp
 *  @public @memberof spFVector */
extern void fvundecibelp(spFVector x);
/** @~english Calculates `10^(x/10)` of each element of \p x .
 *  @public @memberof spDVector */
extern void dvundecibelp(spDVector x);
/** @copydoc spDVector::xdvundecibelp
 *  @public @memberof spFVector */
extern spFVector xfvundecibelp(spFVector x);
/** @~english Creates a new vector whose each element is `10^(x/10)` of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvundecibelp(spDVector x);

/** @copydoc spDVector::dvdecibel
 *  @public @memberof spFVector */
extern void fvdecibel(spFVector x);
/** @~english Calculates `10*log10(|x|^2)` of each element of \p x .
 *  @public @memberof spDVector */
extern void dvdecibel(spDVector x);
/** @copydoc spDVector::xdvdecibel
 *  @public @memberof spFVector */
extern spFVector xfvdecibel(spFVector x);
/** @~english Creates a new vector whose each element is `10*log10(|x|^2)` of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvdecibel(spDVector x);

/** @copydoc spDVector::dvundecibel
 *  @public @memberof spFVector */
extern void fvundecibel(spFVector x);
/** @~english Calculates `10^(x/20)` of each element of \p x .
 *  @public @memberof spDVector */
extern void dvundecibel(spDVector x);
/** @copydoc spDVector::xdvundecibel
 *  @public @memberof spFVector */
extern spFVector xfvundecibel(spFVector x);
/** @~english Creates a new vector whose each element is `10^(x/20)` of the corresponding element of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvundecibel(spDVector x);
    
/** @~english @} */

/** @~english @name Random Number (<sp/vmath.h>)
 *  @{ */

/** @copydoc spDVector::dvrandun
 *  @public @memberof spFVector */
extern void fvrandun(spFVector x);
/** @~english Calculates a uniformly distributed random number in the interval (0,1) for each element of \p x .
 *  @public @memberof spDVector */
extern void dvrandun(spDVector x);

/** @copydoc spDVector::xdvrandunul
 *  @public @memberof spFVector */
extern spFVector xfvrandunul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Creates a new plugin-based vector whose each element is a uniformly distributed random number in the interval (0,1).
 *  @public @memberof spDVector */
extern spDVector xdvrandunul(spPlugin *plugin, long length, spBool unlock_flag);

/** @copydoc spDVector::xdvrandun
 *  @public @memberof spFVector */
extern spFVector xfvrandun(long length);
/** @~english Creates a new vector whose each element is a uniformly distributed random number in the interval (0,1).
 *  @public @memberof spDVector */
extern spDVector xdvrandun(long length);

/** @copydoc spDVector::dvgauss
 *  @public @memberof spFVector */
extern void fvgauss(spFVector x, float mu, float sigma);
/** @~english Calculates a Gaussian random number with the mean of \p mu and the standard dev. of \p sigma for each element of \p x .
 *  @public @memberof spDVector */
extern void dvgauss(spDVector x, double mu, double sigma);

/** @copydoc spDVector::xdvgaussul
 *  @public @memberof spFVector */
extern spFVector xfvgaussul(spPlugin *plugin, long length, float mu, float sigma, spBool unlock_flag);
/** @~english Creates a new plugin-based vector whose each element is a Gaussian random number with the mean of \p mu and the standard dev. of \p sigma .
 *  @public @memberof spDVector */
extern spDVector xdvgaussul(spPlugin *plugin, long length, double mu, double sigma, spBool unlock_flag);

/** @copydoc spDVector::xdvgauss
 *  @public @memberof spFVector */
extern spFVector xfvgauss(long length, float mu, float sigma);
/** @~english Creates a new vector whose each element is a Gaussian random number with the mean of \p mu and the standard dev. of \p sigma .
 *  @public @memberof spDVector */
extern spDVector xdvgauss(long length, double mu, double sigma);

/** @copydoc spDVector::dvrandn
 *  @public @memberof spFVector */
extern void fvrandn(spFVector x);
/** @~english Calculates a normally distributed random number for each element of \p x .
 *  @public @memberof spDVector */
extern void dvrandn(spDVector x);

/** @copydoc spDVector::xdvrandnul
 *  @public @memberof spFVector */
extern spFVector xfvrandnul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Creates a new plugin-based vector whose each element is a normally distributed random number.
 *  @public @memberof spDVector */
extern spDVector xdvrandnul(spPlugin *plugin, long length, spBool unlock_flag);

/** @copydoc spDVector::xdvrandn
 *  @public @memberof spFVector */
extern spFVector xfvrandn(long length);
/** @~english Creates a new vector whose each element is a normally distributed random number.
 *  @public @memberof spDVector */
extern spDVector xdvrandn(long length);

/** @~english @} */

/** @} */  /*----@addtogroup vmathGroup----*/

/** @internal */
extern double dvscpolyval(spDVector p, double x);
extern void dvcplxpolyval(spDVector p, double xre, double xim, double *oyre, double *oyim);
extern void dvpolyval(spDVector iox, spDVector p);
extern spDVector xdvpolyval(spDVector x, spDVector p);

#if defined(MACOS)
#pragma import off
#endif

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_VMATH_H */
