/*
 *	base.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <stdarg.h>

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>

#ifdef SP_USE_VECTOR_ENGINE
#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>

extern spPlugin *sp_default_vector_plugin;
#endif

static int sp_randun_init = 0;

spBool spSrand(unsigned int seed)
{
#ifdef SP_USE_VECTOR_ENGINE
    if (sp_default_vector_plugin != NULL) {
        spVectorPluginInternalFuncList *flist;

        if ((flist = spGetVectorPluginInternalFuncList(sp_default_vector_plugin, 2, 0)) != NULL
             && flist->srand != NULL) {
             flist->srand(seed);
        }
        if ((flist = spGetVectorPluginInternalFuncList(sp_default_vector_plugin, 3, 0)) != NULL
             && flist->srand != NULL) {
             flist->srand(seed);
        }
    }
#endif
    
    srand(seed);
    sp_randun_init = 1;
    
    return SP_TRUE;
}

void spRandunNoInit(void)
{
    sp_randun_init = 1;
    return;
}

float spRandunf(void)
{
    float x;
    
    /* reset random function */
    if (!sp_randun_init) {
	srand((unsigned int)spGetProcessId());
	sp_randun_init = 1;
    }

    x = (float)rand() / ((float)RAND_MAX + 1.0f);

    return x;
}

float spRandun1f(void)
{
    return spRandunf();
}

double spRandun(void)
{
    double x;
    
    /* reset random function */
    if (!sp_randun_init) {
	srand((unsigned int)spGetProcessId());
	sp_randun_init = 1;
    }

    x = (double)rand() / ((double)RAND_MAX + 1.0);

    return x;
}

double spRandun1(void)
{
    return spRandun();
}

static int sp_gauss_use_rand = 0;

void spGaussUseRand(int flag)
{
    sp_gauss_use_rand = (flag ? 1 : 0);
    return;
}

/*
 *	mu : mean  sigma : standard deviation
 */
float spGaussf(float mu, float sigma)
{
    int i;
    float a, x;

    if (sp_gauss_use_rand) {
	for (i = 0, a = 0.0; i < 12; i++) {
	    a += (float)spRandun();
	}
    } else {
	for (i = 0, a = 0.0; i < 12; i++) {
	    a += (float)spRandun1();
	}
    }

    x = (a - 6.0f) * sigma + mu;

    return x;
}

double spGauss(double mu, double sigma)
{
    int i;
    double a, x;

    if (sp_gauss_use_rand) {
	for (i = 0, a = 0.0; i < 12; i++) {
	    a += spRandun();
	}
    } else {
	for (i = 0, a = 0.0; i < 12; i++) {
	    a += spRandun1();
	}
    }

    x = (a - 6.0) * sigma + mu;

    return x;
}

long spGcd(long x, long y)
{
    long a;

    while (y != 0) {
	a = x % y;
	x = y;
	y = a;
    }

    return x;
}

void spCExpf(float *xr, float *xi)
{
    float a;

    if (xr == NULL) {
	return;
    } else if (*xr == 0.0f) {
	*xr = cosf(*xi);
	*xi = sinf(*xi);
    } else if (xi != NULL && *xi != 0.0f) {
	a = expf(*xr);
	*xr = a * cosf(*xi);
	*xi = a * sinf(*xi);
    } else {
	*xr = expf(*xr);
    }

    return;
}

void spCExp(double *xr, double *xi)
{
    double a;

    if (xr == NULL) {
	return;
    } else if (*xr == 0.0) {
	*xr = cos(*xi);
	*xi = sin(*xi);
    } else if (xi != NULL && *xi != 0.0) {
	a = exp(*xr);
	*xr = a * cos(*xi);
	*xi = a * sin(*xi);
    } else {
	*xr = exp(*xr);
    }

    return;
}

void spCLogf(float *xr, float *xi)
{
    float a;

    if (xi != NULL && (*xr < 0.0f || *xi != 0.0f)) {
	a = (float)CABS(*xr, *xi);
	*xi = atan2f(*xi, *xr);
	*xr = logf(a);
    } else {
	if (*xr <= 0.0f) {
	    spwarning("warning: clogf: log of zero\n");
	    
	    *xr = logf((float)SP_TINY_NUMBER);
	} else {
	    *xr = logf(*xr);
	}
    }

    return;
}

void spCLog(double *xr, double *xi)
{
    double a;

    if (xi != NULL && (*xr < 0.0 || *xi != 0.0)) {
	a = CABS(*xr, *xi);
	*xi = atan2(*xi, *xr);
	*xr = log(a);
    } else {
	if (*xr <= 0.0) {
	    spwarning("warning: clog: log of zero\n");
	    
	    *xr = log(SP_TINY_NUMBER);
	} else {
	    *xr = log(*xr);
	}
    }

    return;
}

void spCLog10f(float *xr, float *xi)
{
    float a;

    if (xi != NULL && (*xr < 0.0f || *xi != 0.0f)) {
	a = (float)CABS(*xr, *xi);
	*xi = atan2f(*xi, *xr) / (float)SP_LN10;
	*xr = log10f(a);
    } else {
	if (*xr <= 0.0f) {
	    spwarning("warning: clog: log of zero\n");
	    
	    *xr = log10f((float)SP_TINY_NUMBER);
	} else {
	    *xr = log10f(*xr);
	}
    }

    return;
}

void spCLog10(double *xr, double *xi)
{
    double a;

    if (xi != NULL && (*xr < 0.0 || *xi != 0.0)) {
	a = CABS(*xr, *xi);
	*xi = atan2(*xi, *xr) / SP_LN10;
	*xr = log10(a);
    } else {
	if (*xr <= 0.0) {
	    spwarning("warning: clog: log of zero\n");
	    
	    *xr = log10(SP_TINY_NUMBER);
	} else {
	    *xr = log10(*xr);
	}
    }

    return;
}

typedef struct _spRandSortData
{
    int index;
    char *data;
    double randnum;
} spRandSortData;

#if defined(__STDC__) || defined(_WIN32) || defined(MACOS)
static int spRandSortNumCmp(const void *x, const void *y)
#else
static int spRandSortNumCmp(void *x, void *y)
#endif
{
    spRandSortData *xdata, *ydata;
    
    xdata = (spRandSortData *)x; ydata = (spRandSortData *)y;
    
    if (xdata->randnum < ydata->randnum) {
	return (-1);
    } else if (xdata->randnum > ydata->randnum) {
	return (1);
    } else {
	return (0);
    }
}

void spRandSort(void *data, int num, int size)
{
    int i;
    char *ptr;
    spRandSortData *data_list;

    data_list = xalloc(num, spRandSortData);

    ptr = data;
    for (i = 0; i < num; i++) {
	data_list[i].index = i;
	data_list[i].data = xalloc(size, char);
	memmove(data_list[i].data, ptr, size);
	
	data_list[i].randnum = randun1();
	spDebug(100, "spRandSort", "data_list[%d].randnum = %f\n", i, data_list[i].randnum);

	ptr = ptr + size;
    }
    
    qsort(data_list, (unsigned)num, (unsigned)sizeof(spRandSortData), spRandSortNumCmp);

    ptr = data;
    for (i = 0; i < num; i++) {
	memmove(ptr, data_list[i].data, size);
	xfree(data_list[i].data);
	
	ptr = ptr + size;
    }
    
    xfree(data_list);

    return;
}

void spDecibel(double *x, long length)
{
    long k;

    for (k = 0; k < length; k++) {
	x[k] = x[k] * x[k];
	if (x[k] <= 0.0) {
	    spwarning("warning: decibel: log of zero\n");
	    
	    x[k] = 10.0 * log10(SP_TINY_NUMBER);
	} else {
	    x[k] = 10.0 * log10(x[k]);
	}
    }

    return;
}

void spDecibelp(double *x, long length)
{
    long k;

    for (k = 0; k < length; k++) {
	if (x[k] <= 0.0) {
	    spwarning("warning: decibelp: log of zero\n");
	    
	    x[k] = 10.0 * log10(SP_TINY_NUMBER);
	} else {
	    x[k] = 10.0 * log10(x[k]);
	}
    }

    return;
}

static int sp_nearest_white_key_indexes_upper[] = {
    0/*C*/, 1/*C#*/, 1/*D*/, 2/*D#*/, 2/*E*/, 3/*F*/, 4/*F#*/, 4/*G*/, 5/*G#*/, 5/*A*/, 6/*A#*/, 6/*B*/
};
static int sp_nearest_white_key_indexes_lower[] = {
    0/*C*/, 0/*C#*/, 1/*D*/, 1/*D#*/, 2/*E*/, 3/*F*/, 3/*F#*/, 4/*G*/, 4/*G#*/, 5/*A*/, 5/*A#*/, 6/*B*/
};
static const char *sp_note_name_list[12] = {
    "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"
};

static spBool sp_black_key_flag[12] = {
    SP_FALSE/*C*/, SP_TRUE/*C#*/, SP_FALSE/*D*/, SP_TRUE/*D#*/, SP_FALSE/*E*/, SP_FALSE/*F*/, SP_TRUE/*F#*/,
    SP_FALSE/*G*/, SP_TRUE/*G#*/, SP_FALSE/*A*/, SP_TRUE/*A#*/, SP_FALSE/*B*/
};

static spBool sp_white_key_flag[12] = {
    SP_TRUE/*C*/, SP_FALSE/*C#*/, SP_TRUE/*D*/, SP_FALSE/*D#*/, SP_TRUE/*E*/, SP_TRUE/*F*/, SP_FALSE/*F#*/,
    SP_TRUE/*G*/, SP_FALSE/*G#*/, SP_TRUE/*A*/, SP_FALSE/*A#*/, SP_TRUE/*B*/
};

spBool spIsIndexBlackKey(int key)
{
    if (key < 0 || key >= 12) return SP_FALSE;
    
    return sp_black_key_flag[key];
}

spBool spIsIndexWhiteKey(int key)
{
    if (key < 0 || key >= 12) return SP_FALSE;
    
    return sp_white_key_flag[key];
}

const char *spGetNoteNameFromIndex(int key)
{
    if (key < 0 || key >= 12) return NULL;
    
    return sp_note_name_list[key];
}

int spGetKeyIndex(char *name)
{
    int i;
    int key_index;
    
    key_index = -1;
    for (i = 0; i < 12; i++) {
        if (strcaseeq(name, sp_note_name_list[i])) {
            key_index = i;
            break;
        }
    }

    return key_index;
}

int spGetNearestWhiteKeyIndexFromIndex(int key_index, spBool upper)
{
    if (key_index < 0 || key_index >= 12) {
        spDebug(1, "spGetNearestWhiteKeyIndexFromIndex", "invalid note name index: %d\n", key_index);
        return -1;
    }

    if (upper) {
        return sp_nearest_white_key_indexes_upper[key_index];
    } else {
        return sp_nearest_white_key_indexes_lower[key_index];
    }
}

int spGetNearestWhiteKeyIndex(char *name, spBool upper)
{
    int key_index;
    
    key_index = spGetKeyIndex(name);
    if (key_index < 0) {
        spDebug(1, "spGetNearestWhiteKeyIndex", "%s is not note name.\n", name);
        return -1;
    }

    return spGetNearestWhiteKeyIndexFromIndex(key_index, upper);
}

double spGetNearestWhiteKeyBoundaryCent(int key_index, spBool upper)
{
    int white_key_index;
    double cent;
    
    white_key_index = spGetNearestWhiteKeyIndexFromIndex(key_index, SP_TRUE);
    if (upper == SP_FALSE) {
        --white_key_index;
        if (white_key_index < 0) {
            return -50.0;
        }
    }
    
    if (white_key_index >= 3) {
        /* F-B: 700/4 cent each */
        cent = 450.0 + (double)(white_key_index - 2) * 700.0 / 4.0;
    } else {
        /* C-E: 500/3 cent each */
        cent = -50.0 + (double)(white_key_index + 1) * 500.0 / 3.0;
    }

    return cent;
}

spBool spIncrementNoteName(char *io_name, int *io_octave_index, int incr)
{
    int key_index;
    int new_index;
    int octave_incr;
    const char *name_ptr;

    if (io_name == NULL) return SP_FALSE;
    
    key_index = spGetKeyIndex(io_name);
    if (key_index < 0) {
        spDebug(1, "spIncrementNoteName", "%s is not note name.\n", io_name);
        return SP_FALSE;
    }

    new_index = key_index + incr;
    octave_incr = (int)floor((double)new_index / 12.0);
    new_index = new_index - 12 * octave_incr;

    if (io_octave_index != NULL) {
        *io_octave_index += octave_incr;
    }
    
    name_ptr = sp_note_name_list[new_index];
    strcpy(io_name, name_ptr);
    
    return SP_TRUE;
}

spBool spFreqToNoteName(double freq, int mid_C_octave_index/* 3: SPN/IPN/Roland, 4: YAMAHA/Apple */, char *o_name, int *o_octave_index, double *o_cent)
{
    int ref_octave_index;
    int semitone_int_diff_from_C;
    double semitone_diff_from_mid_C, semitone_diff_from_C;
    double semitone_diff_from_nearest;
    const char *name_ptr;

    if (freq <= 0.0) {
        if (o_name != NULL) {
            o_name[0] = NUL;
        }
        return SP_FALSE;
    }

    semitone_diff_from_mid_C = 12.0 * log10(freq / 440.0) / log10(2.0) + 9.0;
    ref_octave_index = (int)floor((semitone_diff_from_mid_C + 0.5) / 12.0);
    semitone_diff_from_C = semitone_diff_from_mid_C - (double)ref_octave_index * 12.0;
    semitone_int_diff_from_C = (int)spRound(semitone_diff_from_C);
    
    if (o_name != NULL) {
        name_ptr = sp_note_name_list[semitone_int_diff_from_C];
        strcpy(o_name, name_ptr);
    }
    if (o_cent != NULL) {
        semitone_diff_from_nearest = semitone_diff_from_C - (double)semitone_int_diff_from_C;
        *o_cent = semitone_diff_from_nearest * 100.0;
    }
    if (o_octave_index != NULL) {
        *o_octave_index = ref_octave_index + mid_C_octave_index;
    }

    return SP_TRUE;
}

spBool spNoteNameToFreq(char *name, int octave_index, double cent, int mid_C_octave_index/* 3: SPN/IPN/Roland, 4: YAMAHA/Apple */, double *o_freq)
{
    int i;
    int ref_octave_index;
    int semitone_int_diff_from_C;
    double semitone_diff_from_mid_C, semitone_diff_from_C;
    double semitone_diff_from_nearest;

    if (o_freq == NULL || name == NULL || name[0] == NUL) return SP_FALSE;
    
    ref_octave_index = octave_index - mid_C_octave_index;
    semitone_diff_from_nearest = cent / 100.0;

    semitone_int_diff_from_C = -1;
    for (i = 0; i < 12; i++) {
        if (strcaseeq(name, sp_note_name_list[i])) {
            semitone_int_diff_from_C = i;
            break;
        }
    }
    if (semitone_int_diff_from_C < 0) {
        spDebug(1, "spNoteNameToFreq", "%s is not note name.\n", name);
        return SP_FALSE;
    }
    semitone_diff_from_C = semitone_diff_from_nearest + semitone_int_diff_from_C;
    semitone_diff_from_mid_C = semitone_diff_from_C + (double)(ref_octave_index * 12);

    *o_freq = 440.0 * pow(2.0, (semitone_diff_from_mid_C - 9.0) / 12.0);

    return SP_TRUE;
}
