/*
 *        picolatest.c
 *
 *        Last modified: <2018-12-13 14:51:56 hideki>
 */

#include <stdio.h>

#include <sp/spBaseLib.h>
#include <sp/spInputPlugin.h>

#include <sp/spPicola.h>

#include <sp/spMain.h>

spPicolaConfig picola_config;

static double rate;

static double gain;
static char output_file[SP_MAX_PATHNAME] = "";
static char *plugin_search_path;
static char o_plugin_name[SP_MAX_LINE] = "";

static spBool help_flag;
static int debug_level = -1;

static spOptions options;
static spOption option[] = {
    {"-r", "-rate", "speech rate (speaking speed)", "rate",
         SP_TYPE_DOUBLE, &rate, "1.0"},
    {"-g", "-gain", "gain for output data", "gain",
         SP_TYPE_DOUBLE, &gain, "1.0"},
    {"-min", NULL, "minimum pitch period [ms]", "minimum_period",
         SP_TYPE_DOUBLE, &picola_config.min_period_ms, NULL},
    {"-max", NULL, "maximum pitch period [ms]", "maximum_period",
         SP_TYPE_DOUBLE, &picola_config.max_period_ms, NULL},
    {"-O", "-output", "output file (default format: wav)", NULL,
         SP_TYPE_STRING_A, output_file, NULL},
    {"-op", "-oplugin", "plugin name for output", "output_plugin",
         SP_TYPE_STRING_A, o_plugin_name, NULL},
    {"-path", NULL, "plugin search path", "plugin_path",
         SP_TYPE_STRING, &plugin_search_path, NULL},
    {"-debug", NULL, "debug level", NULL,
         SP_TYPE_INT, &debug_level, NULL},
    {"-h", "-help", "display this message", NULL,
         SP_TYPE_BOOLEAN, &help_flag, SP_FALSE_STRING},
};

static char *filelabel[] = {
    "<input file>",
};
    
int spMain(int argc, char *argv[])
{
    const char *input_file;
    spPlugin *i_plugin;
    spPlugin *o_plugin;
    spWaveInfo wave_info;
    spPluginDeviceType device_type;
            
    spPicolaInitConfig(&picola_config);
    
    /* get command line options */
    spSetHelpMessage(&help_flag, "picola command-line version");
    options = spGetOptions(argc, argv, option, filelabel);
    spGetOptionsValue(argc, argv, options);
    spSetDebugLevel(debug_level);
    spCheckNumFile(options);

    input_file = spGetFile(options);
    spDebug(-1, "main", "input_file = %s\n", input_file);

    spInitWaveInfo(&wave_info);

    if ((i_plugin = spOpenFilePluginArg(NULL, input_file, "r",
                                      SP_PLUGIN_DEVICE_FILE,
                                      &wave_info, NULL, 0, NULL, NULL)) != NULL) {
        if (spStrNone(output_file)) {
            device_type = SP_PLUGIN_DEVICE_AUDIO;
            strcpy(o_plugin_name, "output_audio");
        } else {
            device_type = SP_PLUGIN_DEVICE_FILE;
            spDebug(-1, "main", "output_file = %s\n", output_file);
        }
        spDebug(-1, "main", "o_plugin_name = %s\n", o_plugin_name);

        strcpy(wave_info.file_type, "");
            
        if ((o_plugin = spOpenFilePluginArg(o_plugin_name, output_file, "w",
                                            device_type,
                                            &wave_info, NULL, 0, NULL, NULL)) != NULL) {
            spPicola picola;
            long max_output_buf_length;
            long orig_input_buf_length;
            long read_length;
            long input_buf_length, output_buf_length;
            long input_delay;
            double *input_buf, *output_buf;

            spDebug(-1, "main", "samp_rate = %f\n", wave_info.samp_rate);
            picola_config.samp_freq = wave_info.samp_rate;

            if ((picola = spPicolaOpen(&picola_config)) != NULL) {
                if (spPicolaSetRate(picola, rate, &orig_input_buf_length, &max_output_buf_length) == SP_TRUE) {
                    spDebug(-1, "main", "orig_input_buf_length = %ld, max_output_buf_length = %ld\n",
                            orig_input_buf_length, max_output_buf_length);

                    input_buf = xspAlloc(orig_input_buf_length, double);
                    output_buf = xspAlloc(max_output_buf_length, double);

                    while (1) {
                        read_length = spReadPluginDouble(i_plugin, input_buf, orig_input_buf_length);
                        spDebug(-1, "main", "read_length = %ld\n", read_length);
                        
                        output_buf_length = max_output_buf_length;

                        if (read_length <= 0) {
                            spPicolaFlush(picola, output_buf, &output_buf_length);
                            spDebug(-1, "main", "picola flushed: output_buf_length = %ld\n", output_buf_length);
                        } else {
                            input_buf_length = read_length;

                            spPicolaProcess(picola, 
                                            input_buf, &input_buf_length, output_buf, &output_buf_length,
                                            &input_delay);
                            spDebug(1, "main", "input_buf_length = %ld, output_buf_length = %ld\n",
                                    input_buf_length, output_buf_length);
                        }

                        if (output_buf_length > 0) {
                            spWritePluginDoubleWeighted(o_plugin, output_buf, output_buf_length, gain);
                        }

                        if (read_length <= 0) {
                            break;
                        }
                    }

                    xspFree(input_buf);
                    xspFree(output_buf);
                }

                spPicolaClose(picola);
                spDebug(-1, "main", "picola finished\n");
            }
            
            spCloseFilePlugin(o_plugin);
        }
        spCloseFilePlugin(i_plugin);
    }

    return 0;
}
