/** @file sp/fft.h
 */

#ifndef SPLIB_FFT_H
#define SPLIB_FFT_H

#include <sp/spPlugin.h>
#include <sp/vector.h>

#ifdef __cplusplus
extern "C" {
#endif

#define SP_FFT_SUPPORT_BLUESTEIN_ALGORITHM

/** @~english @defgroup fftGroup <sp/fft.h>: FFT
@code
#include <sp/fft.h>
@endcode
 */

/** @addtogroup fftGroup
 *  @{ */  /*----@addtogroup fftGroup----*/

#define spFFTPrecision spPrecision /** @~english Indicates precision of FFT computation. */
#define spFFTSpeed spComputeSpeed /**< @~english Indicates speed of FFT computation. */

#define SP_FFT_HIGHEST_PRECISION SP_HIGHEST_PRECISION /**< @~english Highest precision. */
#define SP_FFT_HIGH_DOUBLE_PRECISION SP_HIGH_DOUBLE_PRECISION /**< @~english Double type precision higher than normal double type. */
#define SP_FFT_DOUBLE_PRECISION SP_DOUBLE_PRECISION /**< @~english double type precision.  */
#define SP_FFT_LOW_DOUBLE_PRECISION SP_LOW_DOUBLE_PRECISION /**< @~english double type precision lower than normal double type  */
#define SP_FFT_HIGH_FLOAT_PRECISION SP_HIGH_FLOAT_PRECISION /**< @~english float type precision higher than normal double type.  */
#define SP_FFT_FLOAT_PRECISION SP_FLOAT_PRECISION /**< @~english float type precision.  */
#define SP_FFT_LOW_FLOAT_PRECISION SP_LOW_FLOAT_PRECISION /**< @~english float type precision lower than normal float type  */
#define SP_FFT_32BIT_PRECISION SP_32BIT_PRECISION /**< @~english 32-bit integer precision.  */
#define SP_FFT_24BIT_PRECISION SP_24BIT_PRECISION /**< @~english 24-bit integer precision.  */
#define SP_FFT_16BIT_PRECISION SP_16BIT_PRECISION /**< @~english 16-bit integer precision.  */
#define SP_FFT_IGNORE_PRECISION SP_IGNORE_PRECISION /**< @~english Indicates that precision is ignored.  */
#define SP_FFT_UNKNOWN_PRECISION SP_UNKNOWN_PRECISION /**< @~english Unknown precision.  */
#define SP_FFT_DEFAULT_PRECISION SP_FFT_FLOAT_PRECISION /**< @~english Default precision.  */

#define SP_FFT_SPEED_EXTREMELY_FAST SP_COMPUTE_SPEED_EXTREMELY_FAST /**< @~english Extremely fast speed. */
#define SP_FFT_SPEED_VERY_FAST SP_COMPUTE_SPEED_VERY_FAST /**< @~english Very fast speed.  */
#define SP_FFT_SPEED_FASTER SP_COMPUTE_SPEED_FASTER /**< @~english Faster speed.  */
#define SP_FFT_SPEED_FAST SP_COMPUTE_SPEED_FAST /**< @~english Fast speed.  */
#define SP_FFT_SPEED_NORMAL SP_COMPUTE_SPEED_NORMAL /**< @~english Normal speed.  */
#define SP_FFT_SPEED_SLOW SP_COMPUTE_SPEED_SLOW /**< @~english Slow speed.  */
#define SP_FFT_SPEED_SLOWER SP_COMPUTE_SPEED_SLOWER /**< @~english Slower speed.  */
#define SP_FFT_SPEED_VERY_SLOW SP_COMPUTE_SPEED_VERY_SLOW /**< @~english Very slow speed.  */
#define SP_FFT_SPEED_EXTREMELY_SLOW SP_COMPUTE_SPEED_EXTREMELY_SLOW /**< @~english Extremely slow speed.  */
#define SP_FFT_SPEED_UNKNOWN SP_COMPUTE_SPEED_UNKNOWN /**< @~english Unknown speed.  */
    
/** @~english
 *  @class spFFTRec fft.h <sp/fft.h>
 *  @brief A class to handle FFT execution.
@details Actually, spFFTRec is not a class, but a typedefed type by `typedef struct _spFFTRec *spFFTRec;`.
You can use this class after initialization by spInitFFT() , spInitFFTByPlugin() , spInitBatchedFFT() , spInitBatchedFFTByPlugin() , should free resources of this class
after use by spFreeFFT() .
 *  @sa spInitFFT() , spInitFFTByPlugin() , spInitBatchedFFT() , spInitBatchedFFTByPlugin() , spFreeFFT() 
 */
typedef struct _spFFTRec *spFFTRec;  /**< @~english Definition of spFFTRec. */
    
#if defined(MACOS)
#pragma import on
#endif

/** @~english @name FFT Execution (<sp/fft.h>)
 *  @{ */

/** @copydoc spfft */
extern int spfftf(float *xRe, float *xIm, long fftl, int inv);
/** @~english Executes FFT of input data.
 *  @param[in] xRe The real part buffer of the data. The size of this buffer must be larger than \p fftl .
 *  @param[in] xIm The imaginary part buffer of the data. The size of this buffer must be larger than \p fftl .
 *  @param[in] fftl The number of FFT points.
 *  @param[in] inv Nonzero value enables execution of the inverse FFT.
 *  @retval SP_SUCCESS Success  @retval SP_FAILURE Failure   */
extern int spfft(double *xRe, double *xIm, long fftl, int inv);
/* spectrum: data[0]=Re(X[0]), data[1]=Re(X[fftl/2]), data[2*n]=Re(X[n]), data[2*n+1]=Im(X[n]) (n > 0, n < fftl/2) */
/** @copydoc sprfft */
extern int sprfftf(float *data, long fftl, int inv);
/** @~english Executes FFT of input real data.
 *  @param[in] data The buffer of the data. The size of this buffer must be larger than \p fftl .
 *  After FFT execution, \p data holds `data[0]=Re(X[0])`, `data[1]=Re(X[fftl/2])`, `data[2*n]=Re(X[n])`, and `data[2*n+1]=Im(X[n])` (n > 0, n < fftl/2), where `X[n]` is the complex spectrum.
 *  In other words, even elements except the index 0  and odd elements except the index 1 hold the real part and the imaginary part of the complex spectrum, respectively.
 *  @param[in] fftl The number of FFT points.
 *  @param[in] inv Nonzero value enables execution of the inverse FFT.
 *  @retval SP_SUCCESS Success  @retval SP_FAILURE Failure   */
extern int sprfft(double *data, long fftl, int inv);

/** @~english @} */

/** @~english @name Data Shift for FFT (<sp/fft.h>)
 *  @{ */

/** @copydoc datashift */
extern void datashifts(short *data, long length, long shift);
/** @copydoc datashift */
extern void datashiftl(long *data, long length, long shift);
/** @copydoc datashift */
extern void datashiftf(float *data, long length, long shift);
/** @~english Shifts input data. The pushed out data by shifting is discarded.
 *  @param[in] data The buffer containing input data. The input data is replaced with shifted data.
 *  @param[in] length The size of \p data .
 *  @param[in] shift The shift length. Positive and negative values indicate shift to left and right respectively. */
extern void datashift(double *data, long length, long shift);
/** @copydoc ridatashift */
extern void ridatashifts(short *real, short *imag, long length, long shift);
/** @copydoc ridatashift */
extern void ridatashiftl(long *real, long *imag, long length, long shift);
/** @copydoc ridatashift */
extern void ridatashiftf(float *real, float *imag, long length, long shift);
/** @~english Shifts input complex data. The pushed out data by shifting is discarded.
 *  @param[in] real The buffer containing input real data. The input data is replaced with shifted data.
 *  @param[in] imag The buffer containing input imaginary data. The input data is replaced with shifted data.
 *  @param[in] length The size of input data.
 *  @param[in] shift The shift length. Positive and negative values indicate shift to left and right respectively. */
extern void ridatashift(double *real, double *imag, long length, long shift);
/** @copydoc circshift */
extern void circshifts(short *data, long length, long shift);
/** @copydoc circshift */
extern void circshiftl(long *data, long length, long shift);
/** @copydoc circshift */
extern void circshiftf(float *data, long length, long shift);
/** @~english Circular-shifts input data. 
 *  @param[in] data The buffer containing input data. The input data is replaced with shifted data.
 *  @param[in] length The size of input data.
 *  @param[in] shift The shift length. Positive and negative values indicate shift to left and right respectively. */
extern void circshift(double *data, long length, long shift);
/** @copydoc ricircshift */
extern void ricircshifts(short *real, short *imag, long length, long shift);
/** @copydoc ricircshift */
extern void ricircshiftl(long *real, long *imag, long length, long shift);
/** @copydoc ricircshift */
extern void ricircshiftf(float *real, float *imag, long length, long shift);
/** @~english Circular-shifts input complex data. 
 *  @param[in] real The buffer containing input real data. The input data is replaced with shifted data.
 *  @param[in] imag The buffer containing input imaginary data. The input data is replaced with shifted data.
 *  @param[in] length The size of input data.
 *  @param[in] shift The shift length. Positive and negative values indicate shift to left and right respectively. */
extern void ricircshift(double *real, double *imag, long length, long shift);

/** @copydoc fftturn */
extern void fftturnf(float *xRe, float *xIm, long fftl);
/** @~english Makes the real part of the spectrum reflection symmetric and the imaginary part rotational symmetric with respect to the index `fftl/2`. The right half data is not used for computation.
 *  @param[in] xRe The buffer of the real part of input data. The right half data is overwritten.
 *  @param[in] xIm The buffer of the imaginary part of input data. The right half data is overwritten. If you don't use the immaginary part, NULL can be specified.
 *  @param[in] fftl The number of FFT points. The buffer size must be larger than this value. */
extern void fftturn(double *xRe, double *xIm, long fftl);
/** @copydoc fftshift */
extern void fftshiftf(float *xRe, float *xIm, long fftl);
/** @~english Circular-shifts input data so that the index `fftl/2` of the buffer moves to the index `0` .
 *  @param[in] xRe The buffer of the real part of input data. The input data is replaced with shifted data.
 *  @param[in] xIm The buffer of the imaginary part of input data. The input data is replaced with shifted data.
 *  @param[in] fftl The number of FFT points. The buffer size must be larger than this value. */
extern void fftshift(double *xRe, double *xIm, long fftl);
    
/** @~english @} */

/** @~english @name Data Conversion for FFT (<sp/fft.h>)
 *  @{ */

/* support in-place (can be real == rspec) and NULL-imag (version 0.9.1+) */
extern void ffttorfftf(float *real, float *imag, float *rspec, long fftl);
extern void ffttorfft(double *real, double *imag, double *rspec, long fftl);
extern void ffttorfftipf(float *real, float *imag, long fftl);
extern void ffttorfftip(double *real, double *imag, long fftl);
extern void rffttofftf(float *rspec, float *real, float *imag, long fftl);
extern void rffttofft(double *rspec, double *real, double *imag, long fftl);
extern void rffttofftipf(float *real, float *imag, long fftl);
extern void rffttofftip(double *real, double *imag, long fftl);
    
extern void multrfftf(float *rspec1, float *rspec2, long fftl);
extern void multrfft(double *rspec1, double *rspec2, long fftl);
extern void rffttopowerf(float *rspec, float exponent, long fftl); /* 1.0=power, 0.5=abs */
extern void rffttopower(double *rspec, double exponent, long fftl); /* 1.0=power, 0.5=abs */
extern void powertorfftf(float *powspec, float orig_exponent, long fftl); /* 1.0=power, 0.5=abs */
extern void powertorfft(double *powspec, double orig_exponent, long fftl); /* 1.0=power, 0.5=abs */
extern void rffttohpowerf(float *rspec, float *ohpowspec, float exponent, long fftl);
extern void rffttohpower(double *rspec, double *ohpowspec, double exponent, long fftl);
extern void rffttoanglef(float *rspec, long fftl);
extern void rffttoangle(double *rspec, long fftl);
extern void rffttologf(float *rspec, long fftl);
extern void rffttolog(double *rspec, long fftl);
extern void rffttoexpf(float *rspec, long fftl);
extern void rffttoexp(double *rspec, long fftl);

/** @~english @} */

/** @~english @name Cepstrum Analysis (<sp/fft.h>)
 *  @{ */

extern int rfftcep(double *x, double *pw, long fftl);
extern int spectocepip(double *real, double *imag, long fftl);
extern int spectocep(double *real, double *imag, double *cep, long fftl);
extern int ceptospecip(double *real, double *imag, long fftl);
extern int ceptospec(double *cep, double *real, double *imag, long fftl);
extern int ceptompc(double *cep, long fftl);
extern int ceplif(double *cep, long fftl, long lif);
extern int cepwlif(double *cep, long fftl, long period, long width);

/** @~english @} */

/** @~english @name FFT Execution with FFT Plugin (<sp/fft.h>)
 *  @{ */

/** @~english Gets the number of FFT plugins available.
 *  @return The number of FFT plugins available. */
extern int spGetNumFFTPlugin(void);
/** @~english Finds the name of an FFT plugin registered.
 *  @param[in] index The address of a variable which holds the index to find. The index of the first call must be 0. After the function call, the value of the variable will be the index of the FFT plugin obtained.
 *  @return A null-terminated string indicating the file name of the FFT plugin.
 You have to deallocate the memory by calling xspFree() . NULL will be returned if error occurs or no plugin is found anymore. */
extern char *xspSearchFFTPluginFile(int *index);
/** @~english Sets the specified FFT plugin default.
 *  @param[in] name A null-terminated string of the name of an FFT plugin.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure */
extern spBool spSetDefaultFFTPlugin(const char *name);
/** @~english Checks if the plugin is an FFT plugin.
 *  @param[in] plugin spPlugin type pointer to check.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure */
extern spBool spIsFFTPlugin(spPlugin *plugin);
/** @~english Loads an FFT plugin.
 *  @param[in] name The name of the FFT plugin.
 *  @return The loaded FFT plugin if successful or NULL if failure.   */
extern spPlugin *spLoadFFTPlugin(const char *name);
/** @~english Unloads the FFT plugin.
 *  @param[in] plugin The loaded FFT plugin obtained by spLoadFFTPlugin() .
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure */
extern spBool spFreeFFTPlugin(spPlugin *plugin);

/** @~english Gets speed of the default FFT engine. This function doesn't get speed of the plugin-based FFT engine if the plugin-based FFT is set to default.
 *  @param[in] precision The precision of the FFT engine to get speed in spFFTPrecision type.
 *  @return The speed of the FFT engine in spFFTSpeed type.  */
extern spFFTSpeed spGetFFTSpeed(spFFTPrecision precision);
/** @~english Initializes resources for plugin-based FFT inside spFFTRec.
 *  The return value can be used for FFT-related functions such as spExecFFT() and spExecRealFFT() .
 *  The difference from spInitFFTByPlugin() is that a single FFT call executes multiple FFTs of the batch count in parallel.
 *  The buffer for spExecFFT() or spExecRealFFT() contains a series of vectors of the batch count.
 *  @param[in] plugin The FFT plugin used as FFT engine.
 *  @param[in] order The order (the power of 2) of FFT points. For example, 8 for the FFT points of 256.
 *  @param[in] batch The batch count.
 *  @param[in] precision The precision of FFT engine such as #SP_FFT_DEFAULT_PRECISION , #SP_FFT_DOUBLE_PRECISION , #SP_FFT_FLOAT_PRECISION and so on.
Normally, #SP_FFT_DEFAULT_PRECISION is recommended. Higher precision can make FFT execution slow on some environments.
 *  @return The memory-allocated spFFTRec type variable including resources for FFT execution if successful or NULL if failure.
 *  You have to call spFreeFFT() to free the resources.  */
extern spFFTRec spInitBatchedFFTByPlugin(spPlugin *plugin, long order, long batch, spFFTPrecision precision);
/** @~english Initializes resources for FFT inside spFFTRec.
 *  The return value can be used for FFT-related functions such as spExecFFT() and spExecRealFFT() .
 *  The difference from spInitFFT() is that a single FFT call executes multiple FFTs of the batch count in parallel.
 *  The buffer for spExecFFT() or spExecRealFFT() contains a series of vectors of the batch count.
 *  @param[in] order The order (the power of 2) of FFT points. For example, 8 for the FFT points of 256.
 *  @param[in] batch The batch count.
 *  @param[in] precision The precision of FFT engine such as #SP_FFT_DEFAULT_PRECISION , #SP_FFT_DOUBLE_PRECISION , #SP_FFT_FLOAT_PRECISION and so on.
Normally, #SP_FFT_DEFAULT_PRECISION is recommended. Higher precision can make FFT execution slow on some environments.
 *  @return The memory-allocated spFFTRec type variable including resources for FFT execution if successful or NULL if failure.
 *  You have to call spFreeFFT() to free the resources.  */
extern spFFTRec spInitBatchedFFT(long order, long batch, spFFTPrecision precision);
/** @~english Initializes resources for plugin-based FFT inside spFFTRec.
 *  The return value can be used for FFT-related functions such as spExecFFT() and spExecRealFFT() .
 *  @param[in] plugin The FFT plugin used as FFT engine.
 *  @param[in] order The order (the power of 2) of FFT points. For example, 8 for the FFT points of 256.
 *  @param[in] precision The precision of FFT engine such as #SP_FFT_DEFAULT_PRECISION , #SP_FFT_DOUBLE_PRECISION , #SP_FFT_FLOAT_PRECISION and so on.
Normally, #SP_FFT_DEFAULT_PRECISION is recommended. Higher precision can make FFT execution slow on some environments.
 *  @return The memory-allocated spFFTRec type variable including resources for FFT execution if successful or NULL if failure.
 *  You have to call spFreeFFT() to free the resources.  */
extern spFFTRec spInitFFTByPlugin(spPlugin *plugin, long order, spFFTPrecision precision);
/** @~english Initializes resources for FFT inside spFFTRec.
 *  The return value can be used for FFT-related functions such as spExecFFT() and spExecRealFFT() .
 *  @param[in] order The order (the power of 2) of FFT points. For example, 8 for the FFT points of 256.
 *  @param[in] precision The precision of FFT engine such as #SP_FFT_DEFAULT_PRECISION , #SP_FFT_DOUBLE_PRECISION , #SP_FFT_FLOAT_PRECISION and so on.
Normally, #SP_FFT_DEFAULT_PRECISION is recommended. Higher precision can make FFT execution slow on some environments.
 *  @return The memory-allocated spFFTRec type variable including resources for FFT execution if successful or NULL if failure.
 *  You have to call spFreeFFT() to free the resources.  */
extern spFFTRec spInitFFT(long order, spFFTPrecision precision);
/** @~english Frees the resources inside spFFTRec for FFT execution.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure */
extern spBool spFreeFFT(spFFTRec fftrec);
/** @~english Resets the order of FFT. The buffer length must be larger than the length of 2 to the order-th power.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @param[in] order The order (the power of 2) of FFT points. For example, 8 for the FFT points of 256.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure */
extern spBool spResetFFTOrder(spFFTRec fftrec, long order);
/** @~english Gets the order (the power of 2) of FFT points.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @return The order of FFT points.  */
extern long spGetFFTOrder(spFFTRec fftrec);
/** @~english Gets the number of FFT points.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @return The number of FFT points.  */
extern long spGetFFTLength(spFFTRec fftrec);
/** @~english Gets the number of FFT points specified in the call of spInitFFT() or similar functions (spLib version 0.9.5+).
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @return The number of FFT points specified in the function call.  */
extern long spGetFFTLengthSpecified(spFFTRec fftrec);
/** @~english Gets the batch size.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitBatchedFFT() or spInitBatchedFFTByPlugin() .
 *  @return The batch size.  */
extern long spGetFFTBatch(spFFTRec fftrec);
/** @copydoc spExecFFT */
extern spBool spExecFFTF(spFFTRec fftrec, float *real, float *imag, int inv);
/** @~english Executes FFT of input data with an FFT plugin.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @param[in] real The real part buffer of the data. The size of this buffer must be larger than the number of FFT points.
 *  @param[in] imag The imaginary part buffer of the data. The size of this buffer must be larger than the number of FFT points.
 *  @param[in] inv Nonzero value enables execution of the inverse FFT.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure   */
extern spBool spExecFFT(spFFTRec fftrec, double *real, double *imag, int inv);
/** @copydoc spExecRealFFT */
extern spBool spExecRealFFTF(spFFTRec fftrec, float *data, int inv);
/** @~english Executes FFT of input real data with an FFT plugin.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @param[in] data The buffer of the data. The size of this buffer must be larger than the number of FFT points.
 *  After FFT execution, \p data holds `data[0]=Re(X[0])`, `data[1]=Re(X[fftl/2])`, `data[2*n]=Re(X[n])`, and `data[2*n+1]=Im(X[n])` (n > 0, n < fftl/2), where `X[n]` is the complex spectrum.
 *  In other words, even elements except the index 0  and odd elements except the index 1 hold the real part and the imaginary part of the complex spectrum, respectively.
 *  @param[in] inv Nonzero value enables execution of the inverse FFT.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure   */
extern spBool spExecRealFFT(spFFTRec fftrec, double *data, int inv);
/** @copydoc spExecFFTPower */
extern spBool spExecFFTPowerF(spFFTRec fftrec, float *data, float exponent);
/** @~english Calculates a power spectrum of input data with an FFT plugin.
 *  @param[in] fftrec spFFTRec type variable obtained by spInitFFT() or similar functions. 
 *  @param[in] data The buffer containing input data. The size of this buffer must be larger than the number of FFT points. After the function call, the buffer is replaced by the power spectrum calculated.
 *  @param[in] exponent The exponent value to a power spectrum. For example, 1.0 and 0.5 correspond to the power spectrum and the amplitude spectrum, respectively.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure   */
extern spBool spExecFFTPower(spFFTRec fftrec, double *data, double exponent);

/** @~english @} */

/** @~english @name Data Shift for FFT with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool spTurnFFTSpectrumF(spFFTRec fftrec, float *real, float *imag);
extern spBool spTurnFFTSpectrum(spFFTRec fftrec, double *real, double *imag);
extern spBool spExecFFTShiftF(spFFTRec fftrec, float *real, float *imag);
extern spBool spExecFFTShift(spFFTRec fftrec, double *real, double *imag);
extern spBool spExecDataShiftF(spFFTRec fftrec, float *real, float *imag, long shift);
extern spBool spExecDataShift(spFFTRec fftrec, double *real, double *imag, long shift);
extern spBool spExecCircShiftF(spFFTRec fftrec, float *real, float *imag, long shift);
extern spBool spExecCircShift(spFFTRec fftrec, double *real, double *imag, long shift);

/** @~english @} */

/** @~english @name Data Conversion for FFT with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool spGetSpectrumForRealFFTF(spFFTRec fftrec, float *real, float *imag, float *rspec);
extern spBool spGetSpectrumForRealFFT(spFFTRec fftrec, double *real, double *imag, double *rspec);
extern spBool spGetComplexFromRealFFTF(spFFTRec fftrec, float *rspec, float *real, float *imag);
extern spBool spGetComplexFromRealFFT(spFFTRec fftrec, double *rspec, double *real, double *imag);
extern spBool spMultiplySpectrumOfRealFFTF(spFFTRec fftrec, float *rspec1, float *rspec2);
extern spBool spMultiplySpectrumOfRealFFT(spFFTRec fftrec, double *rspec1, double *rspec2);

/** @~english @} */

/** @~english @name FFT Execution for Vector (<sp/fft.h>)
 *  @{ */

extern void fvfft(FVector x);
extern void dvfft(DVector x);
extern void fvifft(FVector x);
extern void dvifft(DVector x);
extern FVector xfvfft(FVector x, long length);
extern DVector xdvfft(DVector x, long length);
extern FVector xfvifft(FVector x, long length);
extern DVector xdvifft(DVector x, long length);

/** @~english @} */

/** @~english @name Data Shift of Vector for FFT (<sp/fft.h>)
 *  @{ */

extern void svdatashift(SVector x, long shift);
extern void lvdatashift(LVector x, long shift);
extern void fvdatashift(FVector x, long shift);
extern void dvdatashift(DVector x, long shift);
extern void fvdatashiftbatched(FVector x, long length, long stride, long batch, long shift);
extern void dvdatashiftbatched(DVector x, long length, long stride, long batch, long shift);
extern void svcircshift(SVector x, long shift);
extern void lvcircshift(LVector x, long shift);
extern void fvcircshift(FVector x, long shift);
extern void dvcircshift(DVector x, long shift);
extern void fvcircshiftbatched(FVector x, long length, long stride, long batch, long shift);
extern void dvcircshiftbatched(DVector x, long length, long stride, long batch, long shift);
extern void fvfftshift(FVector x);
extern void dvfftshift(DVector x);
extern void fvfftshiftbatched(FVector x, long length, long stride, long batch);
extern void dvfftshiftbatched(DVector x, long length, long stride, long batch);
extern void fvifftshift(FVector x);
extern void dvifftshift(DVector x);
extern void fvifftshiftbatched(FVector x, long length, long stride, long batch);
extern void dvifftshiftbatched(DVector x, long length, long stride, long batch);
extern FVector xfvfftshift(FVector x, long length);
extern DVector xdvfftshift(DVector x, long length);
extern FVector xfvfftshiftbatched(FVector x, long length, long stride, long batch);
extern DVector xdvfftshiftbatched(DVector x, long length, long stride, long batch);
extern FVector xfvifftshift(FVector x, long length);
extern DVector xdvifftshift(DVector x, long length);
extern FVector xfvifftshiftbatched(FVector x, long length, long stride, long batch);
extern DVector xdvifftshiftbatched(DVector x, long length, long stride, long batch);
extern void fvfftturn(FVector x);
extern void dvfftturn(DVector x);
extern void fvfftturnbatched(FVector x, long length, long stride, long batch);
extern void dvfftturnbatched(DVector x, long length, long stride, long batch);
extern FVector xfvfftturn(FVector x, long length);
extern DVector xdvfftturn(DVector x, long length);
extern FVector xfvfftturnbatched(FVector x, long length, long stride, long batch);
extern DVector xdvfftturnbatched(DVector x, long length, long stride, long batch);

/** @~english @} */

/** @~english @name FFT-Related Computation of Vector (<sp/fft.h>)
 *  @{ */

extern FVector xfvfftconv(FVector a, FVector b, long fftl);
extern DVector xdvfftconv(DVector a, DVector b, long fftl);

extern FVector xfvfftpower(FVector x, long fftl);
extern DVector xdvfftpower(DVector x, long fftl);
extern FVector xfvfftabs(FVector x, long fftl);
extern DVector xdvfftabs(DVector x, long fftl);

extern FVector xfvfftangle(FVector x, long fftl);
extern DVector xdvfftangle(DVector x, long fftl);
extern FVector xfvfftgrpdly(FVector x, long fftl);
extern DVector xdvfftgrpdly(DVector x, long fftl);
extern FVector xfvgdtophase(FVector gd, long fftl, int reverse);
extern DVector xdvgdtophase(DVector gd, long fftl, int reverse);
extern FVector xfvzerophase(FVector x, long fftl);
extern DVector xdvzerophase(DVector x, long fftl);
extern FVector xfvcspec(FVector mag, FVector phs);
extern DVector xdvcspec(DVector mag, DVector phs);

/** @~english @} */

/** @~english @name Cepstrum Analysis of Vector (<sp/fft.h>)
 *  @{ */

extern void dvspectocep(DVector x);
extern void dvceptospec(DVector x);
extern DVector xdvrcep(DVector x, long fftl);
extern void dvrcep(DVector x);
extern void dvceptompc(DVector cep);
extern DVector xdvmpcep(DVector x, long fftl);
extern void dvmpcep(DVector x);
extern void dvlif(DVector cep, long fftl, long lif);
extern DVector xdvmelwarp(DVector cep, long order, long mel_order, double alpha);

/** @~english @} */

/** @~english @name Functions for Time-Stretched Pulses (<sp/fft.h>)
 *  @{ */

extern DVector xdvtsp(double amp, long m, long n, int inverse);
extern DVector xdvftsp(double amp, long m, long n, int inverse); /* tsp in frequency domain */
extern DVector xdvrftsp(double amp, long m, long n, int inverse); /* tsp in frequency domain for real FFT */

/* alpha: duty factor associated with the duration (>= 2 is recommended).
 *        Bigger value makes the duration short. */
extern DVector xdvgtsp(double amp, double alpha, long n, int inverse);
extern DVector xdvfgtsp(double amp, double alpha, long n, int inverse);	/* gtsp in frequency domain */
extern DVector xdvrfgtsp(double amp, double alpha, long n, int inverse); /* gtsp in frequency domain for real FFT */

extern void dvtspshift(DVector tsp, long m, int inverse);
extern void dvgtspshift(DVector gtsp, double alpha, int inverse);

/** @~english @} */

/** @internal */
extern spFFTRec dvinitfft(long order, spFFTPrecision precision);
extern spBool dvfreefft(spFFTRec fftrec);
extern spBool dvresetfftorder(spFFTRec fftrec, long order);
extern long dvgetfftorder(spFFTRec fftrec);
extern long dvgetfftlength(spFFTRec fftrec);
extern long dvgetfftlengthspecified(spFFTRec fftrec);
extern long dvgetfftbatch(spFFTRec fftrec);

extern spBool dvexecfftf(spFFTRec fftrec, float *real, float *imag, int inv);
extern spBool dvexecfft(spFFTRec fftrec, double *real, double *imag, int inv);
extern spBool dvexecrfftf(spFFTRec fftrec, float *data, int inv);
extern spBool dvexecrfft(spFFTRec fftrec, double *data, int inv);
    
/** @~english @name FFT Execution with FFT Plugin for Vector (<sp/fft.h>)
 *  @{ */

extern spBool dvfftex(spFFTRec fftrec, DVector x);
extern spBool dvifftex(spFFTRec fftrec, DVector x);
extern spBool dvrfftex(spFFTRec fftrec, DVector x);
extern spBool dvirfftex(spFFTRec fftrec, DVector x);
extern DVector xdvfftex(spFFTRec fftrec, DVector x);
extern DVector xdvifftex(spFFTRec fftrec, DVector x);
extern DVector xdvrfftex(spFFTRec fftrec, DVector x);
extern DVector xdvirfftex(spFFTRec fftrec, DVector x);

/** @~english @} */

/** @~english @name Data Shift of Vector for FFT with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool dvfftturnex(spFFTRec fftrec, DVector x);
extern DVector xdvfftturnex(spFFTRec fftrec, DVector x);
extern spBool dvfftshiftex(spFFTRec fftrec, DVector x);
extern DVector xdvfftshiftex(spFFTRec fftrec, DVector x);
    
/** @~english @} */

/** @~english @name FFT-Related Conversion of Vector with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool dvmultrfftex(spFFTRec fftrec, DVector a, DVector b);
extern void dvrffttofftex(spFFTRec fftrec, DVector x);
extern void dvffttorfftex(spFFTRec fftrec, DVector x);
extern void dvrffttopowerex(spFFTRec fftrec, DVector x, double exponent /* 1: power spectrum, 0.5: amplitude spectrum */);
extern void dvpowertorfftex(spFFTRec fftrec, DVector x, double orig_exponent);
    
/** @~english @} */

/** @~english @name FFT-Related Computation of Vector with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool dvfftpowerex(spFFTRec fftrec, DVector x);
extern spBool dvfftabsex(spFFTRec fftrec, DVector x);
extern spBool dvifftabsex(spFFTRec fftrec, DVector x);
extern spBool dvfftangleex(spFFTRec fftrec, DVector x);
extern spBool dvfftgrpdlyex(spFFTRec fftrec, DVector x);
extern DVector xdvfftgrpdlyex(spFFTRec fftrec, DVector x);
extern spBool dvfftconvex(spFFTRec fftrec, DVector a, DVector b);
extern DVector xdvfftconvex(spFFTRec fftrec, DVector a, DVector b);

/** @~english @} */

/** @~english @name Cepstrum Analysis of Vector with FFT Plugin (<sp/fft.h>)
 *  @{ */

extern spBool dvspectocepex(spFFTRec fftrec, DVector x);
extern spBool dvspectompcex(spFFTRec fftrec, DVector x);
extern spBool dvrffttocepex(spFFTRec fftrec, DVector x);
extern spBool dvrffttompcex(spFFTRec fftrec, DVector x);
extern spBool dvceptospecex(spFFTRec fftrec, DVector x);
extern spBool dvceptorfftex(spFFTRec fftrec, DVector x);
extern void dvceptompcex(spFFTRec fftrec, DVector x);
extern spBool dvrcepex(spFFTRec fftrec, DVector x);
extern spBool dvmpcepex(spFFTRec fftrec, DVector x);

/** @~english @} */

#if defined(MACOS)
#pragma import off
#endif

/** @} */  /*----@addtogroup fftGroup----*/
    
#define fftf(xRe, xIm, fftl, inv) spfftf(xRe, xIm, fftl, inv);
#define fft(xRe, xIm, fftl, inv) spfft(xRe, xIm, fftl, inv);

#define xdvrceps(x, fftl) xdvrcep(x, fftl)
#define dvrceps(x) dvrcep(x)
#define xdvmpceps(x, fftl) xdvmpcep(x, fftl)
#define dvmpceps(x) dvmpcep(x)

#define spectocep_ow spectocepip
#define ceptospec_ow ceptospecip

#define fftpowtorfftf(powspec, orig_exponent, fftl) powertorfftf(powspec, orig_exponent, fftl)
#define fftpowtorfft(powspec, orig_exponent, fftl) powertorfft(powspec, orig_exponent, fftl)
    
#define DVFFTRec spFFTRec

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_FFT_H */
