/** @file sp/fileio.h
 */

#ifndef SPLIB_FILEIO_H
#define SPLIB_FILEIO_H

#include <sp/spFile.h>

#include <sp/sp.h>
#include <sp/vector.h>
#include <sp/matrix.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(MACOS)
#pragma import on
#endif

/** @~english @defgroup fileioGroup <sp/fileio.h>: File I/O
@code
#include <sp/fileio.h>
@endcode
 */

/** @addtogroup fileioGroup
 *  @{ */  /*----@addtogroup fileioGroup----*/

extern long dvreadfirst(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp);
extern long dvreadsfirst(spDVector vector, long shift, long length, int swap, FILE *fp);
extern long dvreaddfirst(spDVector vector, long shift, long length, int swap, FILE *fp);
extern long dvreadframe(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp);
extern long dvreadsframe(spDVector vector, long shift, long length, int swap, FILE *fp);
extern long dvreaddframe(spDVector vector, long shift, long length, int swap, FILE *fp);
extern long dvwriteframe(spDVector vector, long shift, long length, int swap, int double_flag, FILE *fp);
extern long dvwritesframe(spDVector vector, long shift, long length, int swap, FILE *fp);
extern long dvwritedframe(spDVector vector, long shift, long length, int swap, FILE *fp);

extern long svfwritesvector(FILE *fp, spSVector vector, long length, int swap);
extern long lvfwritelvector(FILE *fp, spLVector vector, long length, int swap);
extern long fvfwritefvector(FILE *fp, spFVector vector, long length, int swap);
extern long dvfwritedvector(FILE *fp, spDVector vector, long length, int swap);
extern long dvfwritesvector(FILE *fp, spDVector vector, long length, int swap);
extern long dvfwritelvector(FILE *fp, spDVector vector, long length, int swap);
extern long dvfwritefvector(FILE *fp, spDVector vector, long length, int swap);

/** @~english @name File I/O for Vector (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDVector::xdvreaddsignal
 *  @public @memberof spSVector */
extern spSVector xsvreadssignal(const char *filename, long headlen, int swap);
/** @copydoc spDVector::xdvreaddsignal
 *  @public @memberof spLVector */
extern spLVector xlvreadlsignal(const char *filename, long headlen, int swap);
/** @copydoc spDVector::xdvreaddsignal
 *  @public @memberof spFVector */
extern spFVector xfvreadfsignal(const char *filename, long headlen, int swap);
/** @~english Loads a binary signal into a new vector.
 *  @param[in] filename The name of the input signal file.
 *  @param[in] headlen The header size in bytes of the file. The beginning of the file of this size will be skipped.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreaddsignal(const char *filename, long headlen, int swap);

/** @~english Loads a binary signal of short type into a new vector of double type.
 *  @param[in] filename The name of the input signal file containing data of short type.
 *  @param[in] headlen The header size in bytes of the file. The beginning of the file of this size will be skipped.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreadssignal(const char *filename, long headlen, int swap);
/** @~english Loads a binary signal of long type into a new vector of double type.
 *  @param[in] filename The name of the input signal file containing data of long type.
 *  @param[in] headlen The header size in bytes of the file. The beginning of the file of this size will be skipped.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreadlsignal(const char *filename, long headlen, int swap);
/** @~english Loads a binary signal of float type into a new vector of double type.
 *  @param[in] filename The name of the input signal file containing data of float type.
 *  @param[in] headlen The header size in bytes of the file. The beginning of the file of this size will be skipped.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreadfsignal(const char *filename, long headlen, int swap);

/** @copydoc spDVector::dvwritedsignal
 *  @public @memberof spSVector */
extern spBool svwritessignal(const char *filename, spSVector vector, int swap);
/** @copydoc spDVector::dvwritedsignal
 *  @public @memberof spLVector */
extern spBool lvwritelsignal(const char *filename, spLVector vector, int swap);
/** @copydoc spDVector::dvwritedsignal
 *  @public @memberof spFVector */
extern spBool fvwritefsignal(const char *filename, spFVector vector, int swap);
/** @~english Saves the vector as a binary signal file.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvwritedsignal(const char *filename, spDVector vector, int swap);
/** @~english Saves the vector of double type as a binary signal file of short type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */

extern spBool dvwritessignal(const char *filename, spDVector vector, int swap);
/** @~english Saves the vector of double type as a binary signal file of long type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvwritelsignal(const char *filename, spDVector vector, int swap);
/** @~english Saves the vector of double type as a binary signal file of float type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvwritefsignal(const char *filename, spDVector vector, int swap);

/** @copydoc spDVector::xdvreaddvector
 *  @relatesalso spSVector */
#define xsvreadsvector(filename, swap) xsvreadssignal((filename), 0, (swap))
/** @copydoc spDVector::xdvreaddvector
 *  @relatesalso spLVector */
#define xlvreadlvector(filename, swap) xlvreadlsignal((filename), 0, (swap))
/** @copydoc spDVector::xdvreaddvector
 *  @relatesalso spFVector */
#define xfvreadfvector(filename, swap) xfvreadfsignal((filename), 0, (swap))
/** @~english Loads a binary vector into a new vector.
 *  @param[in] filename The name of the input vector file.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @relatesalso spDVector */
#define xdvreaddvector(filename, swap) xdvreaddsignal((filename), 0, (swap))
/** @~english Loads a binary vector of short type into a new vector of double type.
 *  @param[in] filename The name of the input vector file containing data of short type.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @relatesalso spDVector */
#define xdvreadsvector(filename, swap) xdvreadssignal((filename), 0, (swap))
/** @~english Loads a binary vector of long type into a new vector of double type.
 *  @param[in] filename The name of the input vector file containing data of long type.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @relatesalso spDVector */
#define xdvreadlvector(filename, swap) xdvreadlsignal((filename), 0, (swap))
/** @~english Loads a binary vector of float type into a new vector of double type.
 *  @param[in] filename The name of the input vector file containing data of float type.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated vector.
 *  @relatesalso spDVector */
#define xdvreadfvector(filename, swap) xdvreadfsignal((filename), 0, (swap))

/** @copydoc spDVector::dvwritedvector
 *  @relatesalso spSVector */
#define svwritesvector(filename, vector, swap) svwritessignal(filename, vector, swap)
/** @copydoc spDVector::dvwritedvector
 *  @relatesalso spLVector */
#define lvwritelvector(filename, vector, swap) lvwritelsignal(filename, vector, swap)
/** @copydoc spDVector::dvwritedvector
 *  @relatesalso spFVector */
#define fvwritefvector(filename, vector, swap) fvwritefsignal(filename, vector, swap)
/** @~english Saves the vector as a binary vector file.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @relatesalso spDVector */
#define dvwritedvector(filename, vector, swap) dvwritedsignal(filename, vector, swap)
/** @~english Saves the vector of double type as a binary vector file of short type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @relatesalso spDVector */
#define dvwritesvector(filename, vector, swap) dvwritessignal(filename, vector, swap)
/** @~english Saves the vector of double type as a binary vector file of long type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @relatesalso spDVector */
#define dvwritelvector(filename, vector, swap) dvwritelsignal(filename, vector, swap)
/** @~english Saves the vector of double type as a binary vector file of float type.
 *  @param[in] filename The name of the file.
 *  @param[in] vector The vector to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @relatesalso spDVector */
#define dvwritefvector(filename, vector, swap) dvwritefsignal(filename, vector, swap)

/** @~english @} */
/** @internal */

extern spSVectors xsvsreadsvectors(const char *filename, long num_vector, int swap);
extern spLVectors xlvsreadlvectors(const char *filename, long num_vector, int swap);
extern spFVectors xfvsreadfvectors(const char *filename, long num_vector, int swap);
extern spDVectors xdvsreaddvectors(const char *filename, long num_vector, int swap);
/* return: length of one vector */
extern long svswritesvectors(const char *filename, spSVectors vecs, int swap);
extern long lvswritelvectors(const char *filename, spLVectors vecs, int swap);
extern long fvswritefvectors(const char *filename, spFVectors vecs, int swap);
extern long dvswritedvectors(const char *filename, spDVectors vecs, int swap);

/** @~english @name File I/O for Matrix (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDMatrix::xdmreaddmatrix
 *  @public @memberof spSMatrix */
extern spSMatrix xsmreadsmatrix(const char *filename, long ncol, int swap);
/** @copydoc spDMatrix::xdmreaddmatrix
 *  @public @memberof spLMatrix */
extern spLMatrix xlmreadlmatrix(const char *filename, long ncol, int swap);
/** @copydoc spDMatrix::xdmreaddmatrix
 *  @public @memberof spFMatrix */
extern spFMatrix xfmreadfmatrix(const char *filename, long ncol, int swap);
/** @~english Loads a binary matrix into a new matrix.
 *  @param[in] filename The name of the input matrix file.
 *  @param[in] ncol The number of columns of the input matrix file.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated matrix.
 *  @public @memberof spDMatrix */
extern spDMatrix xdmreaddmatrix(const char *filename, long ncol, int swap);
/** @~english Loads a binary matrix of short type into a new matrix of double type.
 *  @param[in] filename The name of the input matrix file containing data of short type.
 *  @param[in] ncol The number of columns of the input matrix file.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated matrix.
 *  @public @memberof spDMatrix */
extern spDMatrix xdmreadsmatrix(const char *filename, long ncol, int swap);
/** @~english Loads a binary matrix of long type into a new matrix of double type.
 *  @param[in] filename The name of the input matrix file containing data of long type.
 *  @param[in] ncol The number of columns of the input matrix file.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated matrix.
 *  @public @memberof spDMatrix */
extern spDMatrix xdmreadlmatrix(const char *filename, long ncol, int swap);
/** @~english Loads a binary matrix of float type into a new matrix of double type.
 *  @param[in] filename The name of the input matrix file containing data of float type.
 *  @param[in] ncol The number of columns of the input matrix file.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @return The memory-allocated matrix.
 *  @public @memberof spDMatrix */
extern spDMatrix xdmreadfmatrix(const char *filename, long ncol, int swap);
/** @copydoc spDMatrix::dmwritedmatrix
 *  @public @memberof spSMatrix */
extern spBool smwritesmatrix(const char *filename, spSMatrix mat, int swap);
/** @copydoc spDMatrix::dmwritedmatrix
 *  @public @memberof spLMatrix */
extern spBool lmwritelmatrix(const char *filename, spLMatrix mat, int swap);
/** @copydoc spDMatrix::dmwritedmatrix
 *  @public @memberof spFMatrix */
extern spBool fmwritefmatrix(const char *filename, spFMatrix mat, int swap);
/** @~english Saves the matrix as a binary matrix file.
 *  @param[in] filename The name of the file.
 *  @param[in] mat The matrix to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDMatrix */
extern spBool dmwritedmatrix(const char *filename, spDMatrix mat, int swap);
/** @~english Saves the matrix of double type as a binary matrix file of short type.
 *  @param[in] filename The name of the file.
 *  @param[in] mat The matrix to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDMatrix */
extern spBool dmwritesmatrix(const char *filename, spDMatrix mat, int swap);
/** @~english Saves the matrix of double type as a binary matrix file of long type.
 *  @param[in] filename The name of the file.
 *  @param[in] mat The matrix to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDMatrix */
extern spBool dmwritelmatrix(const char *filename, spDMatrix mat, int swap);
/** @~english Saves the matrix of double type as a binary matrix file of float type.
 *  @param[in] filename The name of the file.
 *  @param[in] mat The matrix to save.
 *  @param[in] swap Nonzero value enables the endian (byte-order) conversion.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDMatrix */
extern spBool dmwritefmatrix(const char *filename, spDMatrix mat, int swap);

/** @~english @} */
/** @internal */

extern spSMatrices xsmsreadsmatrices(const char *filename, long nmat, long ncol, int swap);
extern spLMatrices xlmsreadlmatrices(const char *filename, long nmat, long ncol, int swap);
extern spFMatrices xfmsreadfmatrices(const char *filename, long nmat, long ncol, int swap);
extern spDMatrices xdmsreaddmatrices(const char *filename, long nmat, long ncol, int swap);
extern spDMatrices xdmsreadsmatrices(const char *filename, long nmat, long ncol, int swap);
extern spDMatrices xdmsreadlmatrices(const char *filename, long nmat, long ncol, int swap);
extern spDMatrices xdmsreadfmatrices(const char *filename, long nmat, long ncol, int swap);
/* the sizes of all matrices must be the same. */
extern spBool smswritesmatrices(const char *filename, spSMatrices xs, int swap);
extern spBool lmswritelmatrices(const char *filename, spLMatrices xs, int swap);
extern spBool fmswritefmatrices(const char *filename, spFMatrices xs, int swap);
extern spBool dmswritedmatrices(const char *filename, spDMatrices xs, int swap);
extern spBool dmswritesmatrices(const char *filename, spDMatrices xs, int swap);
extern spBool dmswritelmatrices(const char *filename, spDMatrices xs, int swap);
extern spBool dmswritefmatrices(const char *filename, spDMatrices xs, int swap);

/** @~english @name Text I/O for Vector (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDVector::dvreaddvector_txt
 *  @public @memberof spSVector */
extern spBool svreadsvector_txt(const char *filename, spSVector vector);
/** @copydoc spDVector::dvreaddvector_txt
 *  @public @memberof spLVector */
extern spBool lvreadlvector_txt(const char *filename, spLVector vector);
/** @copydoc spDVector::dvreaddvector_txt
 *  @public @memberof spFVector */
extern spBool fvreadfvector_txt(const char *filename, spFVector vector);
/** @~english Loads a text vector into the specified vector. If the number of lines of the file is longer than the length of the specified vector, the lines whose index exceeds the length will be ignored. If the number of lines of the file is shorter than the length of the specified vector, the trailing data of the vector will be zero-padded.
 *  @param[in] filename The name of the input text file. Each line of the text file corresponds to an element of the vector.
 *  @param[in,out] vector The vector into which the data of the file loads.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvreaddvector_txt(const char *filename, spDVector vector);
/** @copydoc spDVector::xdvreaddvector_txt
 *  @public @memberof spSVector */
extern spSVector xsvreadsvector_txt(const char *filename);
/** @copydoc spDVector::xdvreaddvector_txt
 *  @public @memberof spLVector */
extern spLVector xlvreadlvector_txt(const char *filename);
/** @copydoc spDVector::xdvreaddvector_txt
 *  @public @memberof spFVector */
extern spFVector xfvreadfvector_txt(const char *filename);
/** @~english Loads a text vector into a new vector.
 *  @param[in] filename The name of the input text file. Each line of the text file corresponds to an element of the vector.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreaddvector_txt(const char *filename);

/** @copydoc spDVector::dvreadcol_txt
 *  @public @memberof spSVector */
extern spBool svreadcol_txt(const char *filename, int col, spSVector vector);
/** @copydoc spDVector::dvreadcol_txt
 *  @public @memberof spLVector */
extern spBool lvreadcol_txt(const char *filename, int col, spLVector vector);
/** @copydoc spDVector::dvreadcol_txt
 *  @public @memberof spFVector */
extern spBool fvreadcol_txt(const char *filename, int col, spFVector vector);
/** @~english Loads a text vector containing multiple data columns into the specified vector. If the number of lines of the file is longer than the length of the specified vector, the lines whose index exceeds the length will be ignored. If the number of lines of the file is shorter than the length of the specified vector, the trailing data of the vector will be zero-padded.
 *  @param[in] filename The name of the input text file. Each line of the text file contains multiples numbers separated by a whitespace.
 *  @param[in] col A 0-based column index.
 *  @param[in,out] vector The vector into which the data of the file loads.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvreadcol_txt(const char *filename, int col, spDVector vector);
/** @copydoc spDVector::xdvreadcol_txt
 *  @public @memberof spSVector */
extern spSVector xsvreadcol_txt(const char *filename, int col);
/** @copydoc spDVector::xdvreadcol_txt
 *  @public @memberof spLVector */
extern spLVector xlvreadcol_txt(const char *filename, int col);
/** @copydoc spDVector::xdvreadcol_txt
 *  @public @memberof spFVector */
extern spFVector xfvreadcol_txt(const char *filename, int col);
/** @~english Loads a text vector containing multiple data columns into a new vector.
 *  @param[in] filename The name of the input text file. Each line of the text file contains multiples numbers separated by a whitespace.
 *  @param[in] col The zero-based column number to load.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvreadcol_txt(const char *filename, int col);

/** @copydoc spDVector::dvwritedvector_txt
 *  @public @memberof spSVector */
extern spBool svwritesvector_txt(const char *filename, spSVector vector);
/** @copydoc spDVector::dvwritedvector_txt
 *  @public @memberof spLVector */
extern spBool lvwritelvector_txt(const char *filename, spLVector vector);
/** @copydoc spDVector::dvwritedvector_txt
 *  @public @memberof spFVector */
extern spBool fvwritefvector_txt(const char *filename, spFVector vector);
/** @~english Saves the vector as a text vector file.
 *  @param[in] filename The name of the file. Each line of the text file corresponds to an element of the vector.
 *  @param[in] vector The vector to save.
 *  @retval SP_TRUE Success  @retval SP_FALSE Failure 
 *  @public @memberof spDVector */
extern spBool dvwritedvector_txt(const char *filename, spDVector vector);

/** @~english @} */

/** @~english @name Vector Modification for I/O (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDVector::dvmaxamp
 *  @public @memberof spFVector */
extern float fvmaxamp(spFVector vec);
/** @~english Gets the maximal absolute amplitude of the vector.
 *  @public @memberof spDVector */
extern double dvmaxamp(spDVector vec);
/** @copydoc spDVector::dvadjustamp
 *  @public @memberof spFVector */
extern float fvadjustamp(spFVector vec, float amp);
/** @~english Amplifies the specified vector so that the maximal absolute amplitude of the vector is \p amp .
 *  @public @memberof spDVector */
extern double dvadjustamp(spDVector vec, double amp);
/** @copydoc spDVector::dvlimitamp
 *  @public @memberof spFVector */
extern float fvlimitamp(spFVector vec, float amp);
/** @~english Amplifies the specified vector so that the maximal absolute amplitude of the vector is \p amp only if the maximal absolute amplitude exceeds \p amp .
 *  @public @memberof spDVector */
extern double dvlimitamp(spDVector vec, double amp);

/** @copydoc spDVector::xdvextractchannel
 *  @public @memberof spFVector */
extern spFVector xfvextractchannel(spFVector x, int channel, int num_channel);
/** @~english Extracts data associated with \p channel from the vector containing multi-channel data with the interleaved format which the number of the channel is \p num_channel .
 *  @public @memberof spDVector */
extern spDVector xdvextractchannel(spDVector x, int channel, int num_channel);

/** @~english @} */

/** @~english @name Vector Dump (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDVector::dvdump
 *  @public @memberof spSVector */
extern void svdump(spSVector vec);
/** @copydoc spDVector::dvdump
 *  @public @memberof spLVector */
extern void lvdump(spLVector vec);
/** @copydoc spDVector::dvdump
 *  @public @memberof spFVector */
extern void fvdump(spFVector vec);
/** @~english Displays the contents of the vector \p vec to the standard output.
 *  @public @memberof spDVector */
extern void dvdump(spDVector vec);

/** @copydoc spDVector::dvfdump
 *  @public @memberof spSVector */
extern void svfdump(spSVector vec, FILE *fp);
/** @copydoc spDVector::dvfdump
 *  @public @memberof spLVector */
extern void lvfdump(spLVector vec, FILE *fp);
/** @copydoc spDVector::dvfdump
 *  @public @memberof spFVector */
extern void fvfdump(spFVector vec, FILE *fp);
/** @~english Outputs the contents of the vector \p vec to the file associated with the file pointer \p fp .
 *  @public @memberof spDVector */
extern void dvfdump(spDVector vec, FILE *fp);

/* real part only. must be NULL end */
/** @~english Outputs the real part of the multiple vectors of spSVector type to the file associated with the file pointer \p fp .
 *  Each column of the output corresponds to the contents of a vector. The final argument of this function must be NULL.
 *  @public @memberof spSVector */
extern void svnfdump(FILE *fp, ...);
/** @~english Outputs the real part of the multiple vectors of spLVector type to the file associated with the file pointer \p fp .
 *  Each column of the output corresponds to the contents of a vector. The final argument of this function must be NULL.
 *  @public @memberof spLVector */
extern void lvnfdump(FILE *fp, ...);
/** @~english Outputs the real part of the multiple vectors of spFVector type to the file associated with the file pointer \p fp .
 *  Each column of the output corresponds to the contents of a vector. The final argument of this function must be NULL.
 *  @public @memberof spFVector */
extern void fvnfdump(FILE *fp, ...);
/** @~english Outputs the real part of the multiple vectors of spDVector type to the file associated with the file pointer \p fp .
 *  Each column of the output corresponds to the contents of a vector. The final argument of this function must be NULL.
 *  @public @memberof spDVector */
extern void dvnfdump(FILE *fp, ...);

/** @~english @} */
/** @internal */

extern void svsfdump(spSVectors vecs, FILE *fp);
extern void lvsfdump(spLVectors vecs, FILE *fp);
extern void fvsfdump(spFVectors vecs, FILE *fp);
extern void dvsfdump(spDVectors vecs, FILE *fp);

/** @~english @name Matrix Dump (<sp/fileio.h>)
 *  @{ */

/** @copydoc spDMatrix::dmfdump
 *  @public @memberof spSMatrix */
extern void smfdump(spSMatrix mat, FILE *fp);
/** @copydoc spDMatrix::dmfdump
 *  @public @memberof spLMatrix */
extern void lmfdump(spLMatrix mat, FILE *fp);
/** @copydoc spDMatrix::dmfdump
 *  @public @memberof spFMatrix */
extern void fmfdump(spFMatrix mat, FILE *fp);
/** @~english Outputs the contents of the matrix \p mat to the file associated with the file pointer \p fp .
 *  @public @memberof spDMatrix */
extern void dmfdump(spDMatrix mat, FILE *fp);

/** @copydoc spDMatrix::dmdump
 *  @public @memberof spSMatrix */
#define smdump(mat) smfdump(mat, (stdout))
/** @copydoc spDMatrix::dmdump
 *  @public @memberof spLMatrix */
#define lmdump(mat) lmfdump(mat, (stdout))
/** @copydoc spDMatrix::dmdump
 *  @public @memberof spFMatrix */
#define fmdump(mat) fmfdump(mat, (stdout))
/** @~english Displays the contents of the matrix \p mat to the standard output.
 *  @public @memberof spDMatrix */
#define dmdump(mat) dmfdump(mat, (stdout))

/** @~english @} */
/** @internal */

extern spBool dmcsvwrite(char *filename, spDMatrix A, int decimal_point, spBool transpose);

/* spAudio library is required to use the following functions */
extern spBool dvsupportaudiofile(const char *filename, const char *plugin_name);
extern spDVector xdvreadaudiofileex(const char *filename, const char *plugin_name,
                                  double *samp_freq, int *samp_bit, int *num_channel, double weight,
                                  spDVectorCallbackFunc func, void *data); /* Version 0.9.1+ */
extern spDVector xdvreadaudiofile(const char *filename, const char *plugin_name,
				double *samp_freq, int *samp_bit, int *num_channel, double weight);
extern long dvwriteaudiofileex(spDVector x, const char *filename, const char *plugin_name,
                               double samp_freq, int samp_bit, int num_channel, double weight,
                               spDVectorCallbackFunc func, void *data); /* Version 0.9.1+ */
extern long dvwriteaudiofile(spDVector x, const char *filename, const char *plugin_name,
			     double samp_freq, int samp_bit, int num_channel, double weight);
    
/** @} */  /*----@addtogroup fileioGroup----*/

#if defined(MACOS)
#pragma import off
#endif

#if !defined(SP_DISABLE_ABBREVIATION)
/* for backwards compatibility */
#define xreadssignal xsvreadssignal
#define xreaddsignal xdvreaddsignal
#define writessignal svwritessignal
#define writedsignal dvwritedsignal

#define xreadsvector xsvreadsvector
#define xreaddvector xdvreaddvector
#define writesvector svwritesvector
#define writedvector dvwritedvector

#define xreadlmatrix xlmreadlmatrix
#define xreaddmatrix xdmreaddmatrix
#define writelmatrix lmwritelmatrix
#define writedmatrix dmwritedmatrix

#define readdvector_txt dvreaddvector_txt
#define xreaddvector_txt xdvreaddvector_txt
#define writedvector_txt dvwritedvector_txt
#endif /* !defined(SP_DISABLE_ABBREVIATION) */

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_FILEIO_H */
