/*
 *	lpc.h
 */

#ifndef SPLIB_LPC_H
#define SPLIB_LPC_H

#include <sp/spDefs.h>
#include <sp/fft.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Basic parameters:
 *   cor[]: Autocorrelation from 0th to order-th. The size must be larger than (order+1).
 *   ncor[]: Normalized autocorrelation from 1st to order-th. The size must be larger than order.
 *   alpha[]: LPC coefficients from 1st to order-th. The size must be larger than order.
 *          alpha(0) is assumed to be 1.0.
 *   parcor[]: PARCOR coefficients (reflection coefficients) from 1st to order-th.
 *           The size must be larger than order.
 *   psd[]: Power spectrum density from 0 to pi radian (not normalized by 2PI). The size must be larger than (N/2+1).
 *           (N is total spectrum points from 0 to 2pi)
 *   vta[]: Vocal tract area ratio function (vta[0]: case of lips). The size must be larger than order.
 *   lsp[]: LSP frequencies ranging between 0.0 and 1.0,
 *          increasingly ordered, i.e.,lsp(0)<lsp(1)<..<lsp(lp-1).
 *          The size must be larger than order.
 */

/* Autocorrelation */
extern long spSigToCor(double *sig, long siglen, double *cor, long corlen);
extern long spSigToNCor(double *sig, long siglen, double *ncor, long ncorlen, double *cor0 /* can be NULL*/);
extern long spSigToCorFFT(spFFTRec fftrec, double *sig, long siglen, double *cor, long corlen);
extern long spSigToNCorFFT(spFFTRec fftrec, double *sig, long siglen, double *ncor, long ncorlen, double *cor0 /* can be NULL*/);
/* preemph_db: preemphasis value in dB/oct */    
extern  long spSpecToCorFFT(spFFTRec fftrec, double *spec, long speclen, double *cor, long corlen, double preemph_db);
extern long spSpecToNCorFFT(spFFTRec fftrec, double *spec, long speclen, double *ncor, long ncorlen, double preemph_db, double *cor0 /* can be NULL */);
/* deemph_db: deemphasis value in dB/oct */    
extern long spCorToSpecFFT(spFFTRec fftrec, double *cor, long corlen, double *spec, long speclen, double deemph_db);
extern long spNCorToSpecFFT(spFFTRec fftrec, double *ncor, long ncorlen, double cor0,
                            double *spec, long speclen, double deemph_db);

/* LPC computation from autocorrelation. */
extern long spCorToParcor(double *cor, 		/* input. */
			  double *alpha,	/* output. */
			  double *parcor, 	/* output. can be NULL. */
			  long order,		/* input. */
			  double *resid); 	/* output. can be NULL. */
extern long spNCorToParcor(double *ncor, 	/* input. */
                           double *alpha,	/* output. */
                           double *parcor, 	/* output. can be NULL. */
                           long order,		/* input. */
                           double *resid); 	/* output. can be NULL. */
extern long spCorToAlpha(double *cor,		/* input. */
			 double *alpha, 	/* output. */
			 long order,		/* input. */
			 double *resid);	/* output. can be NULL. */
extern long spNCorToAlpha(double *ncor,		/* input. */
                          double *alpha, 	/* output. */
                          long order,		/* input. */
                          double *resid);	/* output. can be NULL. */

/* Convertion between parameters. */
extern long spParcorToAlpha(double *parcor, double *alpha, long order);
extern long spAlphaToParcor(double *alpha, double *parcor, long order, double *resid, spBool *is_stable);
extern long spAlphaToLSP(double *alpha, double *lsp, long order);
extern long spLSPToAlpha(double *alpha, double *lsp, long order);
extern long spParcorToCor(double *parcor, double *alpha, long order, double cor0, double *cor, long n, double *resid);
extern long spParcorToNCor(double *parcor, double *alpha, long order, double *ncor, long ncorlen, double *resid);

extern long spAlphaToPSD(double *alpha, long order, double resid, double *psd, long Nhalf);
extern long spAlphaToPSDFFT(double *alpha, long order, double resid, double *psd, long fftl);
extern long spParcorToVTAreaRatio(double *parcor, double *vtar, long order, spBool linear);
extern long spVTAreaRatioToParcor(double *vtar, double *parcor, long order, spBool linear);

/* vtar and vta can be identical (can be same address) */
extern long spVTRatioToArea(double *vtar, double *vta, long order, double ref_area, spBool linear);
extern long spVTAreaToRatio(double *vta, double *vtar, long order, double ref_area, spBool linear);

/* vta and parcor can be identical (can be same address) */
extern long spParcorToVTArea(double *parcor, double *vta, long order, double ref_area);
extern long spVTAreaToParcor(double *vta, double *parcor, long order, double ref_area);

/* LPC-based filtering for synthesis.
 *   Output: response from 
 *   state (input/output): inner status of the synthesis filter.
 *                         The size must be larger than order.
 *   state2, state2plus1:  inner status of the synthesis filter.
 *                         The size must be larger than (2*order) or (2*order+1).
 */
extern double spAlphaSynFiltering(double *alpha, double *state, long order, double insig);
extern double spParcorSynFiltering(double *parcor, double *state, long order, double insig);

/* signal & alpha --> residual signal */
extern long spSigAlphaToRessig(double *sig, long siglen, double *alpha, long order, double *ressig /* output */);
/* residual signal & alpha --> signal */
extern long spRessigAlphaToSig(double *ressig, long siglen, double *alpha, long order, double *sig /* output */);

/*
 * lag window (pascal) data generation.
 *   hb      : input.        real.
 *              0.0 < hb < 1.0 .    If hb=0, wdata(i)=1.0 for all i.
 *             window half value band width to sampling frequency ratio.
 *             example: lag window half value band width = 100 hz, and
 *             sampling frequency = 8 khz, then hb = 100/8k = 1/80 =0.0125
 */
extern long spLagWindow(double *wdata, long length, double hb);

/* alpha <--> LPC cepstrum */
extern long spAlphaToCep(double *alpha, long order, double *cep, long ceplen);
extern long spCepToAlpha(double *cep, long ceplen, double *alpha, long order);

/* alpha <--> pole <--> frequency & band width */
extern long spAlphaToPole(double *alpha, double *polere, double *poleim, long order, double eps);
extern long spPoleToFreq(double *polere, double *poleim, double *freq /* output */, double *bw /* output, can be NULL */, long order, double fs, spBool sort_flag);
extern long spFreqToPole(double *freq, double *bw, double *polere, double *poleim, long order, double fs);
extern long spPoleToAlpha(double *polere, double *poleim, double *alpha, long order);

extern double spCepSynFiltering(double *cep, double *state2, long order, double insig);
extern double spLSPSynFiltering(double *lsp, double *state2plus1, long order, double insig);

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_LPC_H */
