/*
 *	sfconv.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/vector.h>
#include <sp/voperate.h>
#include <sp/filter.h>
#include <sp/fileio.h>
#include <sp/fft.h>
#include <sp/sfconv.h>
#include <sp/kaiser.h>

static long sfc_fftl = 512;		/* fft point */
static double sfc_framem = 30.0;	/* frame length [ms] */
static double sfc_cutoff = 0.95;	/* normalized cut off frequency */
static double sfc_sidelobe = 60.0; 	/* sidelobe [dB] */
static double sfc_transition = 0.05;	/* normalized transition width */
static double sfc_tolerance = 2;	/* tolerance [%] */

/*
 *	set/get parameter for sampling frequency converting
 */
void setsfcframe(double framem)
{
    sfc_framem = MAX(framem, 1.0);
    return;
}

void getsfcframe(double *framem)
{
    if (framem != NULL)
	*framem = sfc_framem;
    return;
}

void setsfcparam(double cutoff, double sidelobe, double transition, double tolerance, long fftl)
{
    sfc_cutoff = MAX(cutoff, 0.0);
    sfc_sidelobe = MAX(sidelobe, 0.0);
    sfc_transition = MAX(transition, 0.0);
    sfc_tolerance = MAX(tolerance, 0.0);
    sfc_fftl = MAX(fftl, 128);
    return;
}

void getsfcparam(double *cutoff, double *sidelobe, double *transition, double *tolerance, long *fftl)
{
    if (cutoff != NULL) *cutoff = sfc_cutoff;
    if (sidelobe != NULL) *sidelobe = sfc_sidelobe;
    if (transition != NULL) *transition = sfc_transition;
    if (tolerance != NULL) *tolerance = sfc_tolerance;
    if (fftl != NULL) *fftl = sfc_fftl;
    return;
}

/*
 *	get ratio of sampling frequency converting
 */
double getsfcratio(double i_samp_freq, double o_samp_freq, double tolerance, 
		   long *upratio, long *downratio)
{
    int flag;
    long i, j;
    long n;
    double x;
    long div;
    long up, down;
    double new_samp_freq;
    
    spDebug(80, "getsfcratio", "i_samp_freq = %f, o_samp_freq = %f, tolerance = %f\n",
	    i_samp_freq, o_samp_freq, tolerance);
    
    up =  (long)round(o_samp_freq * 100.0);
    down = (long)round(i_samp_freq * 100.0);
    div = gcd(up, down);
    up /= div;
    down /= div;
    /*tolerance *= (double)up;*/ /* this is bug */
    tolerance *= (double)down;

    spDebug(80, "getsfcratio", "div = %ld, up = %ld, down = %ld, tolerance = %f\n", 
	    div, up, down, tolerance);
    
    flag = ((tolerance == 0.0 || (up < 10 && down < 10)) ? 0 : 1);
    for (i = 1; i <= up && flag == 1; i++) {
	for (j = 1; j <= down && flag == 1; j++) {
            n = down * i - up * j;
	    x = ABS(n);
	    if (x <= tolerance) {
		up = i;
		down = j;
		flag = 0;
	    }
	}
    }

    spDebug(80, "getsfcratio", "updated: up = %ld, down = %ld\n", up, down);
    
    *upratio = up;
    *downratio = down;
    new_samp_freq = i_samp_freq * (double)up / (double)down;

    return new_samp_freq;
}

spDVector xdvupsample(spDVector sig, long upratio)
{
    long k;
    spDVector osig;

    /* memory allocation */
    osig = xdvzeros(sig->length * upratio);

    /* up sampling */
    for (k = 0; k < sig->length; k++) {
	osig->data[k * upratio] = sig->data[k];
    }

    return osig;
}

spDVector xdvdownsample(spDVector sig, long downratio, long offset, long length)
{
    long k;
    long pos;
    spDVector osig;

    /* memory allocation */
    if (length <= 0) {
	length = (long)ceil((double)(sig->length - offset) / (double)downratio);
    }
    osig = xdvalloc(length);

    /* down sampling */
    for (k = 0; k < osig->length; k++) {
	pos = k * downratio + offset;
	if (pos < sig->length && pos >= 0) {
	    osig->data[k] = sig->data[pos];
	} else {
	    osig->data[k] = 0.0;
	}
    }

    return osig;
}

/*
 *	convert sampling frequency from file
 */
static int resamplefile_main(const char *i_filename, int i_swap, const char *o_filename, int o_swap,
			     spDVector filter, long upratio, long downratio, int double_flag)
{
    long k;
    long pos;
    long length;
    long shift;
    long syn_pos;
    long syn_offset;
    int nread;
    double value;
    spDVector isig, osig;
    spDVector upsig;
    spDVector filsig;
    FILE *fp, *fp2;

    /* open file */
    if (NULL == (fp = spOpenFile(i_filename, "rb"))) {
	return SP_FAILURE;
    }

    /* open file */
    if (NULL == (fp2 = spOpenFile(o_filename, "wb"))) {
	return SP_FAILURE;
    }
	
    /* get signal length */
    length = MAX(2048, (filter->length / 2) * 2);
    spDebug(10, "resamplefile", "filter length = %ld, frame length = %ld\n",
	    filter->length, length);
    
    /* allocate memory */
    isig = xdvzeros(length);
    osig = xdvzeros(isig->length * upratio + filter->length - 1);
    
    nread = 0;
    pos = (-filter->length / 2);
    syn_offset = 0;
    for (k = 0;; k++) {
	/* read frame */
	if (k == 0) {
	    if ((nread = dvreadfirst(isig, 0, isig->length, i_swap, double_flag, fp)) <= 0) {
		break;
	    }
	} else {
	    if ((nread = dvreadframe(isig, isig->length, isig->length, i_swap, double_flag, fp)) <= 0) {
		break;
	    }
	}
	
	/* up sampling */
	upsig = xdvupsample(isig, upratio);

	/* lowpass filtering */
	filsig = xdvfftfilt(filter, upsig, sfc_fftl);

	/* overlap add */
	dvadd(osig, pos, filsig, 0, filsig->length, 1);
	shift = (long)nread * upratio;
	pos += shift;

	if (pos > 0) {
	    for (syn_pos = syn_offset; syn_pos < pos; syn_pos += downratio) {
		if (syn_pos >= 0 && syn_pos < osig->length) {
		    value = osig->data[syn_pos];
		} else {
		    value = 0.0;
		}
		if (double_flag) {
		    fwritedouble(&value, 1, o_swap, fp2);
		} else {
		    fwritedoubletos(&value, 1, o_swap, fp2);
		}
	    }
	    syn_offset = syn_pos - pos;
	    
	    dvdatashift(osig, -pos);
	    pos = 0;
	}

	spDebug(10, "resamplefile", "shift = %ld, pos = %ld\n", shift, pos);
	
	/* free memory */
	xdvfree(upsig);
	xdvfree(filsig);
    }

    /* write last frame */
    pos = filter->length / 2 - 1;
    for (syn_pos = syn_offset; syn_pos < pos; syn_pos += downratio) {
	if (syn_pos >= 0 && syn_pos < osig->length) {
	    value = osig->data[syn_pos];
	} else {
	    value = 0.0;
	}
	if (double_flag) {
	    fwritedouble(&value, 1, o_swap, fp2);
	} else {
	    fwritedoubletos(&value, 1, o_swap, fp2);
	}
    }

    /* close file */
    spCloseFile(fp);
    spCloseFile(fp2);
    
    /* free memory */
    xdvfree(filter);
    xdvfree(isig);
    xdvfree(osig);
    
    return SP_SUCCESS;
}

int resamplefile(const char *i_filename, int i_swap, const char *o_filename, int o_swap,
		 spDVector filter, long upratio, long downratio)
{
    return resamplefile_main(i_filename, i_swap, o_filename, o_swap,
			     filter, upratio, downratio, 0);
}

int resampledfile(const char *i_filename, int i_swap, const char *o_filename, int o_swap,
		  spDVector filter, long upratio, long downratio)
{
    return resamplefile_main(i_filename, i_swap, o_filename, o_swap,
			     filter, upratio, downratio, 1);
}

static int sfconvlongfile_main(const char *i_filename, int i_swap, double i_samp_freq,
			       const char *o_filename, int o_swap, double o_samp_freq, int double_flag)
{
    long upratio, downratio;
    double new_samp_freq;
    double ratio;
    double cutoff;
    double transition;
    spDVector filter;

    /* get convert ratio */
    new_samp_freq = getsfcratio(i_samp_freq, o_samp_freq, 
				sfc_tolerance / 100.0, &upratio, &downratio);
    spDebug(10, "sfconvlongfile_main", "upratio = %ld, downratio = %ld\n", upratio, downratio);
    spMessage("New Sampling Frequency: %f\n", new_samp_freq);
    
    /* get lowpass filter */
    ratio = (double)MAX(upratio, downratio);
    cutoff = sfc_cutoff / ratio;
    transition = sfc_transition / ratio;
    filter = xdvlowpass(cutoff, sfc_sidelobe, transition, (double)upratio);

    return resamplefile_main(i_filename, i_swap, o_filename, o_swap, filter,
			     upratio, downratio, double_flag);
}

int sfconvlongfile(const char *i_filename, int i_swap, double i_samp_freq,
		   const char *o_filename, int o_swap, double o_samp_freq)
{
    return sfconvlongfile_main(i_filename, i_swap, i_samp_freq,
			       o_filename, o_swap, o_samp_freq, 0);
}

int sfconvlongdfile(const char *i_filename, int i_swap, double i_samp_freq,
		   const char *o_filename, int o_swap, double o_samp_freq)
{
    return sfconvlongfile_main(i_filename, i_swap, i_samp_freq,
			       o_filename, o_swap, o_samp_freq, 1);
}

static spDVector xdvsfconvfile_main(const char *i_filename, int swap, double i_samp_freq, double o_samp_freq,
				  int double_flag)
{
    long k;
    long pos;
    long shift;
    long sig_len;
    long length;
    long upratio, downratio;
    double new_samp_freq;
    double ratio;
    double cutoff;
    double transition;
    spDVector isig, osig;
    spDVector downsig, upsig;
    spDVector filsig;
    spDVector filter;
    FILE *fp;

    /* open file */
    if (NULL == (fp = spOpenFile(i_filename, "rb"))) {
	return NODATA;
    }
    if (double_flag) {
	sig_len = (long)getsiglen(i_filename, 0, double);
    } else {
	sig_len = (long)getsiglen(i_filename, 0, short);
    }
    length = (long)(sfc_framem * i_samp_freq / 1000.0);
    spMessage("Frame Length: %ld\n", length);
    spDebug(10, "xdvsfconvfile", "input signal length = %ld, frame length = %ld\n", sig_len, length);
	
    /* get convert ratio */
    new_samp_freq = getsfcratio(i_samp_freq, o_samp_freq, 
				sfc_tolerance / 100.0, &upratio, &downratio);
    spDebug(10, "xdvsfconvfile_main", "upratio = %ld, downratio = %ld\n", upratio, downratio);
    spMessage("New Sampling Frequency: %f\n", new_samp_freq);
    
    /* get lowpass filter */
    ratio = (double)MAX(upratio, downratio);
    cutoff = sfc_cutoff / ratio;
    transition = sfc_transition / ratio;
    filter = xdvlowpass(cutoff, sfc_sidelobe, transition, (double)upratio);
    
    /* allocate memory */
    isig = xdvzeros(length);
    osig = xdvzeros((sig_len * upratio) / downratio);
    
    spDebug(10, "xdvsfconvfile", "up = %ld, down = %ld, cutoff = %f\n", upratio, downratio, cutoff);

    pos = -filter->length / 2 / downratio;
    shift = (isig->length * upratio) / downratio;
    spMessage("shift = %ld\n", shift);
    for (k = 0;; k++) {
	/* read frame */
	if (k == 0) {
	    if (dvreadfirst(isig, 0, length, swap, double_flag, fp) <= 0) {
		break;
	    }
	} else if (dvreadframe(isig, length, length, swap, double_flag, fp) <= 0) {
	    break;
	}
	
	/* up sampling */
	upsig = xdvupsample(isig, upratio);

	/* lowpass filtering */
	filsig = xdvfftfilt(filter, upsig, sfc_fftl);

	/* down sampling */
	downsig = xdvdownsample(filsig, downratio, 0, 0);

	/* overlap add */
	dvadd(osig, pos, downsig, 0, downsig->length, 1);
	pos += shift;

	/* free memory */
	xdvfree(upsig);
	xdvfree(filsig);
	xdvfree(downsig);
    }

    /* close file */
    spCloseFile(fp);
    
    /* free memory */
    xdvfree(filter);
    xdvfree(isig);
    
    return osig;
}

spDVector xdvsfconvfile(const char *i_filename, int swap, double i_samp_freq, double o_samp_freq)
{
    return xdvsfconvfile_main(i_filename, swap, i_samp_freq, o_samp_freq, 0);
}

spDVector xdvsfconvdfile(const char *i_filename, int swap, double i_samp_freq, double o_samp_freq)
{
    return xdvsfconvfile_main(i_filename, swap, i_samp_freq, o_samp_freq, 1);
}

spDVector xdvfirresample(long upratio, long downratio, double n)
{
    long maxpq;
    long order;
    double fc;
    spDVector win;
    spDVector filter;
        
    maxpq = MAX(upratio, downratio);
    fc = 1.0 / (double)maxpq;
    order = (long)spRound(2.0 * n * (double)maxpq);
    win = xdvkaiser(order + 1, 5.0);
        
    filter = xdvfir1win(win, fc);
    dvscoper(filter, "*", (double)upratio / dvsum(filter));
    
    xdvfree(win);

    return filter;
}

spDVector xdvresample(spDVector isig, spDVector filter, long upratio, long downratio)
{	
    spDVector osig;
    spDVector upsig;
    spDVector filsig;
    long filter_delay;
    spBool alloc_filter = SP_FALSE;

    filter_delay = 0;
    
    if (filter == NODATA) {
        filter = xdvfirresample(upratio, downratio, 10.0);
        //dvdump(filter); spExit(0);
        alloc_filter = SP_TRUE;
        filter_delay = filter->length / 2;
    } else {
        dvmax(filter, &filter_delay);
    }

    /* up sampling */
    upsig = xdvupsample(isig, upratio);

    /* lowpass filtering */
    filsig = xdvfftfilt(filter, upsig, sfc_fftl);
    xdvfree(upsig);

    /* down sampling */
    osig = xdvdownsample(filsig, downratio, filter_delay,
                         (long)ceil((double)(isig->length * upratio) / (double)downratio));
    xdvfree(filsig);

    if (alloc_filter == SP_TRUE) {
        xdvfree(filter);
    }
    
    return osig;
}

spDVector xdvsfconv(spDVector isig, double i_samp_freq, double o_samp_freq, double *new_o_samp_freq)
{	
    double new_samp_freq;
    long upratio, downratio;
    double ratio;
    double cutoff;
    double transition;
    spDVector osig;
    spDVector filter;

    /* get ratio */
    new_samp_freq = getsfcratio(i_samp_freq, o_samp_freq, 
				sfc_tolerance / 100.0, &upratio, &downratio);
    spDebug(10, "xdvsfconv", "upratio = %ld, downratio = %ld\n", upratio, downratio);
    spMessage("New Sampling Frequency: %f\n", new_samp_freq);

    /* get lowpass filter */
    ratio = (double)MAX(upratio, downratio);
    cutoff = sfc_cutoff / ratio;
    transition = sfc_transition / ratio;
    filter = xdvlowpass(cutoff, sfc_sidelobe, transition, (double)upratio);

    spDebug(10, "xdvsfconv", "up = %ld, down = %ld, new_samp_freq = %f, cutoff = %f\n",
	    upratio, downratio, new_samp_freq, cutoff);

    /* resampling */
    osig = xdvresample(isig, filter, upratio, downratio);

    /* memory free */
    xdvfree(filter);

    if (new_o_samp_freq != NULL) {
	*new_o_samp_freq = new_samp_freq;
    }
    
    return osig;
}

int sfconvfile(const char *i_filename, int i_swap, double i_samp_freq,
	       const char *o_filename, int o_swap, double o_samp_freq)
{
    spDVector isig;
    spDVector osig;
    
    if ((isig = xdvreadssignal(i_filename, 0, i_swap)) == NODATA) {
	return SP_FAILURE;
    }
    
    osig = xdvsfconv(isig, i_samp_freq, o_samp_freq, NULL);
    dvwritessignal(o_filename, osig, o_swap);
    
    return SP_SUCCESS;
}

int sfconvdfile(const char *i_filename, int i_swap, double i_samp_freq,
		const char *o_filename, int o_swap, double o_samp_freq)
{
    spDVector isig;
    spDVector osig;
    
    if ((isig = xdvreaddsignal(i_filename, 0, i_swap)) == NODATA) {
	return SP_FAILURE;
    }
    
    osig = xdvsfconv(isig, i_samp_freq, o_samp_freq, NULL);
    dvwritedsignal(o_filename, osig, o_swap);
    
    return SP_SUCCESS;
}

int sfconvfile2(const char *i_filename, int i_swap, double i_samp_freq,
		const char *o_filename, int o_swap, double o_samp_freq)
{
    spDVector osig;
    
    if ((osig = xdvsfconvfile(i_filename, i_swap, i_samp_freq, o_samp_freq)) == NODATA) {
	return SP_FAILURE;
    }
    dvwritessignal(o_filename, osig, o_swap);
    
    return SP_SUCCESS;
}

#if 1
struct _spResampleRec {
    spResampleConfig *config;

    long fftl;
    long upratio;
    long downratio;
    
    long offset;
    spULong total_pos;
    spULong frame_count;

    long filter_delay;
    
    spFFTRec fftrec;
    spDVector filter;
    spDVector filterspec;
    spDVector upsig;
    spDVector buffer;
    
    spDVector input_buffer;
    spDVector output_buffer;
};
    
spBool _spResampleInitConfig(spResampleConfig *config, unsigned long version_id)
{
    if (config == NULL || version_id > SP_RESAMPLE_VERSION_ID) return SP_FALSE;
    
    memset(config, 0, sizeof(spResampleConfig));
    config->version_id = version_id;

    config->num_channel = 1;
    config->input_samp_bit = 64;
    config->output_samp_bit = 64;
    config->max_input_buf_length = 256;
    config->cutoff = 0.95;
    config->sidelobe = 60.0;
    config->transition = 0.05;
    config->tolerance = 2.0;
    
    return SP_TRUE;
}
    
spResampleRec spResampleOpen(spResampleConfig *config)
{
    spResampleRec resample;

    if (config == NULL || config->version_id > SP_RESAMPLE_VERSION_ID) return NULL;

    resample = xalloc(1, struct _spResampleRec);
    memset(resample, 0, sizeof(struct _spResampleRec));
    resample->config = xalloc(1, struct _spResampleConfig);
    memcpy(resample->config, config, sizeof(struct _spResampleConfig));
    resample->config->num_channel = MAX(resample->config->num_channel, 1);
    resample->config->input_samp_bit = MAX(resample->config->input_samp_bit, 16);
    resample->config->output_samp_bit = MAX(resample->config->output_samp_bit, 16);
    resample->config->cutoff = MAX(resample->config->cutoff, 0.0);
    resample->config->sidelobe = MAX(resample->config->sidelobe, 0.0);
    resample->config->transition = MAX(resample->config->transition, 0.0);
    resample->config->tolerance = MAX(resample->config->tolerance, 0.0);

    return resample;
}

spBool spResampleClose(spResampleRec resample)
{
    if (resample != NULL) {
	if (resample->input_buffer != NODATA) xdvfree(resample->input_buffer);
	if (resample->output_buffer != NODATA) xdvfree(resample->output_buffer);
	
	if (resample->filter != NODATA) xdvfree(resample->filter);
	if (resample->filterspec != NODATA) xdvfree(resample->filterspec);
	if (resample->upsig != NODATA) xdvfree(resample->upsig);
	if (resample->buffer != NODATA) xdvfree(resample->buffer);

	if (resample->fftrec != NULL) spFreeFFT(resample->fftrec);

        xfree(resample->config);
	xfree(resample);

	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spResampleSetRatioWithFilter(spResampleRec resample, spDVector filter, long upratio, long downratio,
                                    long *max_output_buf_length, long *default_delay)
{
    int c;
    long upsig_length;
    long fftorder;
    long fftl;
    double ratio;
    double cutoff;
    double transition;
    spFFTRec fftrec;

    spDebug(80, "spResampleSetRatio", "in\n");
    
    if (resample->input_buffer != NODATA) xdvfree(resample->input_buffer);
    if (resample->output_buffer != NODATA) xdvfree(resample->output_buffer);
    
    if (resample->filter != NODATA) xdvfree(resample->filter);
    if (resample->filterspec != NODATA) xdvfree(resample->filterspec);
    if (resample->upsig != NODATA) xdvfree(resample->upsig);
    if (resample->buffer != NODATA) xdvfree(resample->buffer);
    
    if (resample->fftrec != NULL) {
	spFreeFFT(resample->fftrec);
	resample->fftrec = NULL;
    }

    if (filter == NODATA) {
        /* get lowpass filter */
        ratio = (double)MAX(upratio, downratio);
        cutoff = resample->config->cutoff / ratio;
        transition = resample->config->transition / ratio;
        resample->filter = xdvlowpass(cutoff, resample->config->sidelobe, transition, (double)upratio);
        resample->filter_delay = resample->filter->length / 2;
    } else {
        resample->filter = xdvclone(filter);
        dvmax(resample->filter, &resample->filter_delay);
    }

    if (resample->filter == NODATA) {
	spDebug(80, "spResampleSetRatio", "xdvlowpass failed\n");
	return SP_FALSE;
    }

    upsig_length = (resample->config->max_input_buf_length / resample->config->num_channel) * upratio;
    fftorder = spNextPow2(upsig_length + resample->filter->length - 1);
    spDebug(80, "spResampleSetRatio", "upsig length = %ld, fftorder = %ld, num_channel = %d\n",
	    upsig_length, fftorder, resample->config->num_channel);

    if ((fftrec = spInitBatchedFFT(fftorder, resample->config->num_channel, SP_FFT_DEFAULT_PRECISION)) == NULL) {
	spDebug(80, "spResampleSetRatio", "spInitBatchedFFT failed\n");
	return SP_FALSE;
    }

    resample->fftrec = fftrec;
    fftl = spGetFFTLength(resample->fftrec);
    resample->upsig = xdvalloc(fftl * resample->config->num_channel);
    spDebug(80, "spResampleSetRatio", "filter length = %ld, fftl = %ld\n", resample->filter->length, fftl);

    resample->upratio = upratio;
    resample->downratio = downratio;

    resample->filterspec = xdvzeros(fftl * resample->config->num_channel);
    if (resample->config->num_channel >= 2) {
	for (c = 0; c < resample->config->num_channel; c++) {
	    dvadd(resample->filterspec, c * fftl, resample->filter, 0, resample->filter->length, 0);
	}
    } else {
	dvcopy(resample->filterspec, resample->filter);
    }
#if 0
    dvdump(resample->filterspec);
#endif
    spExecRealFFT(resample->fftrec, resample->filterspec->data, 0);

    resample->buffer = xdvzeros(fftl * resample->config->num_channel);

    if (resample->config->options & SP_RESAMPLE_OPTION_TRUNCATE_EDGE) {
	resample->offset = resample->filter_delay;
    } else {
        spDebug(80, "spResampleSetRatio", "resample->downratio = %ld, resample->filter_delay = %ld\n",
                resample->downratio, resample->filter_delay);
        if (resample->downratio >= resample->filter_delay) {
            resample->offset = 0;
        } else {
            resample->offset = resample->filter_delay % resample->downratio;
        }
    }
    spDebug(80, "spResampleSetRatio", "resample->offset = %ld\n", resample->offset);
    resample->total_pos = 0;
    resample->frame_count = 0;

    resample->input_buffer = xdvzeros(resample->config->max_input_buf_length);
    resample->output_buffer = xdvzeros((long)ceil((double)MAX(upsig_length, resample->filter->length - 1)  / (double)downratio)
				       * resample->config->num_channel);
    
    if (max_output_buf_length != NULL) {
	*max_output_buf_length = resample->output_buffer->length;
    }
    if (default_delay != NULL) {
	*default_delay = (long)ceil((double)resample->filter_delay / (double)downratio);
    }
    
    spDebug(80, "spResampleSetRatio", "done\n");
    
    return SP_TRUE;
}

spBool spResampleSetRatio(spResampleRec resample, long upratio, long downratio,
			  long *max_output_buf_length, long *default_delay)
{
    return spResampleSetRatioWithFilter(resample, NODATA, upratio, downratio,
                                        max_output_buf_length, default_delay);
}

double spResampleCalcRatioFromFrequency(double tolerance, double i_samp_freq, double o_samp_freq,
					long *upratio, long *downratio)
{
    double new_samp_freq;

    spDebug(80, "spResampleCalcRatioFromFrequency", "i_samp_freq = %f, o_samp_freq = %f, tolerance = %f\n",
	    i_samp_freq, o_samp_freq, tolerance);
    
    new_samp_freq = getsfcratio(i_samp_freq, o_samp_freq, tolerance / 100.0, 
				upratio, downratio);

    return new_samp_freq;
}

double spResampleSetFrequencyWithFilter(spResampleRec resample, spDVector filter, double i_samp_freq, double o_samp_freq,
                                        long *max_output_buf_length, long *default_delay)
{
    long upratio;
    long downratio;
    double new_samp_freq;

    new_samp_freq = spResampleCalcRatioFromFrequency(resample->config->tolerance, i_samp_freq, o_samp_freq, &upratio, &downratio);
    
    spDebug(80, "spResampleSetFrequency", "upratio = %ld, downratio = %ld, new_samp_freq = %f\n",
	    upratio, downratio, new_samp_freq);
    
    if (new_samp_freq > 0.0 && spResampleSetRatioWithFilter(resample, filter, upratio, downratio, max_output_buf_length, default_delay)) {
	return new_samp_freq;
    } else {
	return 0.0;
    }
}

double spResampleSetFrequency(spResampleRec resample, double i_samp_freq, double o_samp_freq,
			      long *max_output_buf_length, long *default_delay)
{
    return spResampleSetFrequencyWithFilter(resample, NODATA, i_samp_freq, o_samp_freq,
                                            max_output_buf_length, default_delay);
}

long spResampleGetNextMaxInputBufferLength(spResampleRec resample, long output_buf_length)
{
    long length;
    long output_length;

    output_length = MAX(output_buf_length, 0) / resample->config->num_channel;
    length = (output_length * resample->downratio + resample->offset) / resample->upratio;
    spDebug(50, "spResampleGetNextMaxInputBufferLength",
	    "upratio = %ld, downratio = %ld, offset = %ld, output_length = %ld, length = %ld\n",
	    resample->upratio, resample->downratio, resample->offset, output_length, length);

    return MIN(resample->config->num_channel * length, resample->config->max_input_buf_length);
}

/* current_delay: can be NULL */
long _spResampleProcess(spResampleRec resample, char *input_buf, long input_buf_length, char *output_buf, long *current_delay)
{
    int c;
    long k;
    long pos;
    long fftl;
    long up_length;
    long offset;
    long input_length;
    long output_length;
    long output_buf_length;

    if (resample == NULL ||
	resample->upsig == NODATA || resample->filterspec == NODATA || resample->buffer == NODATA) return -1;

    spDebug(100, "spResampleProcess", "input_buf_length = %ld, num_channel = %d, resample->frame_count = %ld\n",
	    input_buf_length, resample->config->num_channel, resample->frame_count);
    
#if 0
    if (resample->upratio == 2 && resample->config->output_samp_bit == 64 && resample->frame_count == 7) {
        double *dbuf = (double *)input_buf;
        for (k = 0; k < input_buf_length; k++) {
            printf("%f\n", dbuf[k]);
        }
        fflush(stdout);
        exit(0);
    }
#endif
    
    spConvertBitToDouble(input_buf, input_buf_length, resample->input_buffer->data,
			 resample->config->input_samp_bit, 1.0);
#if 0
    if (resample->upratio == 2 && resample->frame_count == 6) {
        dvdump(resample->input_buffer); fflush(stdout); spSleep(5); spExit(0);
    }
#endif
    
    fftl = spGetFFTLength(resample->fftrec);
    spDebug(100, "spResampleProcess", "frame_count = %ld, fftl = %ld, upratio = %ld, downratio = %ld\n",
	    (long)resample->frame_count, fftl, resample->upratio, resample->downratio);
    
    dvzeros(resample->upsig, resample->upsig->length);
    
    input_length = input_buf_length / resample->config->num_channel;
    spDebug(100, "spResampleProcess", "input_buf_length = %ld, num_channel = %d, input_length = %ld\n",
	    input_buf_length, resample->config->num_channel, input_length);
    
    if (resample->config->num_channel >= 2 && (resample->config->options & SP_RESAMPLE_OPTION_INPUT_INTERLEAVED)) {
	for (k = 0; k < input_length; k++) {
	    for (c = 0; c < resample->config->num_channel; c++) {
		resample->upsig->data[c * fftl + k * resample->upratio] = resample->input_buffer->data[k * resample->config->num_channel + c];
	    }
	}
    } else {
	for (c = 0; c < resample->config->num_channel; c++) {
	    for (k = 0; k < input_length; k++) {
		resample->upsig->data[c * fftl + k * resample->upratio] = resample->input_buffer->data[c * input_length + k];
	    }
	}
    }
#if 0
    if (resample->upratio == 2 && resample->frame_count == 7) {
        dvdump(resample->upsig); fflush(stdout); spSleep(5); spExit(0);
    }
#endif

    spExecRealFFT(resample->fftrec, resample->upsig->data, 0);
    spMultiplySpectrumOfRealFFT(resample->fftrec, resample->upsig->data, resample->filterspec->data);
    spExecRealFFT(resample->fftrec, resample->upsig->data, 1);
#if 0
    if (resample->upratio == 2 && resample->frame_count == 5) {
        dvdump(resample->upsig); fflush(stdout); spSleep(5); spExit(0);
    }
#endif

    up_length = input_length * resample->upratio;
    offset = resample->offset;
    output_length = (long)ceil((double)(up_length - offset) / (double)resample->downratio);
    spDebug(80, "spResampleProcess", "up_length = %ld, offset = %ld, output_length = %ld\n", up_length, offset, output_length);

    if (current_delay != NULL) {
	*current_delay = (long)ceil((double)(resample->filter_delay - offset) / (double)resample->downratio);
    }

    for (c = 0; c < resample->config->num_channel; c++) {
	dvadd(resample->buffer, c * fftl, resample->upsig, c * fftl, fftl, 1);
    }
#if 0
    if (resample->upratio == 2 && resample->frame_count == 5) {
        dvdump(resample->buffer); fflush(stdout); spSleep(5); spExit(0);
    }
#endif

    if (output_length > 0) {
	if (resample->config->num_channel >= 2 && (resample->config->options & SP_RESAMPLE_OPTION_OUTPUT_INTERLEAVED)) {
	    for (k = 0; k < output_length; k++) {
		pos = k * resample->downratio + offset;
		for (c = 0; c < resample->config->num_channel; c++) {
		    if (pos >= 0) {
			resample->output_buffer->data[k * resample->config->num_channel + c] = resample->buffer->data[c * fftl + pos];
		    } else {
			resample->output_buffer->data[k * resample->config->num_channel + c] = 0.0;
		    }
		}
	    }
	} else {
	    for (c = 0; c < resample->config->num_channel; c++) {
		for (k = 0; k < output_length; k++) {
		    pos = k * resample->downratio + offset;
                    //spDebug(-100, "spResampleProcess", "pos = %ld / %ld\n", pos, resample->buffer->length);
		    if (pos >= 0) {
			resample->output_buffer->data[c * output_length + k] = resample->buffer->data[c * fftl + pos];
		    } else {
			resample->output_buffer->data[c * output_length + k] = 0.0;
		    }
		}
	    }
	}
    }

    output_buf_length = MAX(output_length, 0) * resample->config->num_channel;
    spDebug(80, "spResampleProcess", "output_length = %ld, output_buf_length = %ld, output_samp_bit = %d\n",
            output_length, output_buf_length, resample->config->output_samp_bit);
#if 0
    if (resample->upratio == 2 && resample->frame_count == 5) {
        dvdump(resample->output_buffer); fflush(stdout); spSleep(5); spExit(0);
    }
#endif
    
    if (output_buf_length > 0) {
	spConvertDoubleToBit(resample->output_buffer->data, output_buf_length, output_buf, 
			     resample->config->output_samp_bit, 1.0);
#if 0
        if (resample->upratio == 2 && resample->config->output_samp_bit == 64 && resample->frame_count == 6) {
            double *dbuf = (double *)output_buf;
            for (k = 0; k < output_buf_length; k++) {
                printf("%f\n", dbuf[k]);
            }
            fflush(stdout);
            exit(0);
        }
#endif
    }
    
    resample->offset = MAX(output_length, 0) * resample->downratio + offset;
    resample->total_pos += (spULong)up_length;
    resample->offset -= up_length;
    spDebug(80, "spResampleProcess", "updated offset = %ld\n", resample->offset);

    spExecDataShift(resample->fftrec, resample->buffer->data, NULL, -up_length);

    resample->frame_count++;

    return output_buf_length;
}

spBool spResampleReset(spResampleRec resample)
{
    if (resample == NULL) return SP_FALSE;
    
    if (resample->filter != NODATA && resample->config->options & SP_RESAMPLE_OPTION_TRUNCATE_EDGE) {
	resample->offset = resample->filter_delay;
    } else {
	resample->offset = 0;
    }
    resample->total_pos = 0;
    resample->frame_count = 0;

    if (resample->buffer != NODATA) {
	dvzeros(resample->buffer, resample->buffer->length);
    }
    if (resample->input_buffer != NODATA) {
	dvzeros(resample->input_buffer, resample->input_buffer->length);
    }
    if (resample->output_buffer != NODATA) {
	dvzeros(resample->output_buffer, resample->output_buffer->length);
    }

    return SP_TRUE;
}

long _spResampleFlush(spResampleRec resample, char *output_buf)
{
    int c;
    long k;
    long fftl;
    long offset;
    long length;
    long pos;
    long output_length;
    long output_buf_length;
    
    fftl = spGetFFTLength(resample->fftrec);
    
    offset = resample->offset;
    if (resample->config->options & SP_RESAMPLE_OPTION_TRUNCATE_EDGE) {
	/*length = (long)ceil((double)resample->filter->length / 2.0);*/
	length = resample->filter->length - resample->filter_delay;
    } else {
	length = resample->filter->length - 1;
    }
    output_length = (long)ceil((double)(length/* - offset*/) / (double)resample->downratio);
    spDebug(80, "spResampleFlush", "offset = %ld, filter->length = %ld, length = %ld, output_length = %ld\n",
	    resample->offset, resample->filter->length, length, output_length);
    
    output_buf_length = MAX(output_length, 0) * resample->config->num_channel;

    if (output_length > 0 && output_buf != NULL) {
	if (resample->config->num_channel >= 2 && (resample->config->options & SP_RESAMPLE_OPTION_OUTPUT_INTERLEAVED)) {
	    for (k = 0; k < output_length; k++) {
		pos = k * resample->downratio + offset;
		for (c = 0; c < resample->config->num_channel; c++) {
		    resample->output_buffer->data[k * resample->config->num_channel + c] = resample->buffer->data[c * fftl + pos];
		}
	    }
	} else {
	    for (c = 0; c < resample->config->num_channel; c++) {
		for (k = 0; k < output_length; k++) {
		    pos = k * resample->downratio + offset;
		    resample->output_buffer->data[c * output_length + k] = resample->buffer->data[c * fftl + pos];
		}
	    }
	}
	
	spConvertDoubleToBit(resample->output_buffer->data, output_buf_length, output_buf, 
			     resample->config->output_samp_bit, 1.0);
    }

    spResampleReset(resample);

    return output_buf_length;
}
#endif
