/*
 *	main.c : sampling frequency converter main function
 *
 *	Last modified: <2024-11-13 17:19:05 hideki>
 */

#include <stdio.h>

#include <sp/spBaseLib.h>
#include <sp/spInputPlugin.h>

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/sfconv.h>

#include <sp/spMain.h>

#define SFCONV_VERSION_STRING "1.2.1"
#define SFCONV_HELP_MESSAGE "sampling frequency converter version %s"

static spResampleConfig resample_config;

static double o_samp_freq;

static double gain;
static char *plugin_search_path;
static char o_plugin_name[SP_MAX_LINE] = "";

static spBool quiet_flag;
static spBool supported_only;

static spBool help_flag;
static char i_param_file[SP_MAX_PATHNAME] = "";
static char o_param_file[SP_MAX_PATHNAME] = "";

static int debug_level = -1;

static spOptions options;
static spOption option[] = {
    {"-osf", NULL, "output sampling frequency [Hz]", "o_samp_freq",
	 SP_TYPE_DOUBLE, &o_samp_freq, "12000.0"},
    {"-fftl", NULL, "minimum FFT length [points]", "fft_length", 
	 SP_TYPE_LONG, &resample_config.max_input_buf_length, "512"},
    {"-co", "-cutoff", "normalized cutoff frequency (nyquist = 1)", "cutoff", 
	 SP_TYPE_DOUBLE, &resample_config.cutoff, "0.95"},
    {"-sl", "-sidelobe", "height of sidelobe [dB]", "sidelobe", 
	 SP_TYPE_DOUBLE, &resample_config.sidelobe, "80.0"},
    {"-trans", "-transition", "normalized transition width", "transition", 
	 SP_TYPE_DOUBLE, &resample_config.transition, "0.05"},
    {"-tol", "-tolerance", "tolerance [%]", "tolerance", 
	 SP_TYPE_DOUBLE, &resample_config.tolerance, "2.5"},
    {"-g", "-gain", "gain for output data", "gain",
	 SP_TYPE_DOUBLE, &gain, "1.0"},
    {"-op", "-oplugin", "plugin name for output", "output_plugin",
	 SP_TYPE_STRING_A, o_plugin_name, NULL},
    {"-path", NULL, "plugin search path", "plugin_path",
	 SP_TYPE_STRING, &plugin_search_path, NULL},
    {"-so", NULL, "supported file only", "supported_only",
	 SP_TYPE_BOOLEAN, &supported_only, SP_FALSE_STRING},
    {"-q", "-quiet", "quiet mode", NULL,
	 SP_TYPE_BOOLEAN, &quiet_flag, SP_FALSE_STRING},
    {"-iparam", NULL, "input parameter file", NULL,
	 SP_TYPE_STRING_A, i_param_file, NULL},
    {"-oparam", NULL, "output parameter file", NULL,
	 SP_TYPE_STRING_A, o_param_file, NULL},
    {"-debug", NULL, "debug level", NULL,
	 SP_TYPE_INT, &debug_level, NULL},
    {"-h", "-help", "display this message", NULL,
	 SP_TYPE_BOOLEAN, &help_flag, SP_FALSE_STRING},
    /* options for input_raw */
    SpNullOption("-f", "-freq", SP_TYPE_NONE),
    SpNullOption("-c", "-channel", SP_TYPE_NONE),
    SpNullOption("-b", "-bit", SP_TYPE_NONE),
    SpNullOption("-H", "-head", SP_TYPE_NONE),
    SpNullOption("-F", "-format", SP_TYPE_NONE),
};

static char *filelabel[] = {
    "<input file>",
    "[output file]",
};

static void usage(spPlugin *i_plugin, spPlugin *o_plugin)
{
    spPrintHelpHeader(SFCONV_HELP_MESSAGE, SFCONV_VERSION_STRING);
    spPrintUsageHeader();
    
    if (i_plugin != NULL) {
	spPrintPluginOptions(i_plugin);
	spCloseFilePlugin(i_plugin);
    } else {
	spPluginUsage("input_raw");
    }
    
    if (o_plugin != NULL) {
	spPrintPluginOptions(o_plugin);
	spCloseFilePlugin(o_plugin);
    }
    
    spPrintUsage();
    
    return;
}

int spMain(int argc, char *argv[])
{
    char *input_file;
    char *output_file;
    spPlugin *i_plugin;
    spPlugin *o_plugin;
    spWaveInfo i_wave_info;
    spWaveInfo o_wave_info;
    spPluginDeviceType device_type;
    spResampleRec resample;
    long max_output_buf_length;
    long orig_input_buf_length;
    long read_length;
    long write_length;
    long output_buf_length;
    long delay;
    unsigned long total_read;
    unsigned long total_write;
    double new_samp_freq;
    double *input_buf, *output_buf;

    spResampleInitConfig(&resample_config);
    
    /*spSetDebugLevel(20);*/
    spSetWarningFlag(1);
    
    /* get command line options */
    /*spSetHelpMessage(&help_flag, SFCONV_HELP_MESSAGE, SFCONV_VERSION_STRING);*/
    options = spGetOptions(argc, argv, option, filelabel);
    spIgnoreUnknownOption(options, SP_TRUE);
    spGetOptionsValue(argc, argv, options);
    spSetDebugLevel(debug_level);

    /* read parameter file */
    if (!strnone(i_param_file)) {
        if (quiet_flag == SP_FALSE) {
            spMessage("Input Parameter File: %s\n", i_param_file);
        }
	spReadSetup(i_param_file, options);
    }
    
    if ((input_file = spGetFile(options)) == NULL) {
	spPrintError("Not enough files.");
    }
    output_file = spGetFile(options);
    /*spCheckNumFile(options);*/

    spInitWaveInfo(&i_wave_info);

    if ((i_plugin = spOpenFilePluginArg(NULL, input_file, "r",
				      SP_PLUGIN_DEVICE_FILE,
                                        &i_wave_info, NULL, argc, argv, NULL)) == NULL) {
	if (supported_only == SP_TRUE) {
            spError(1, "This file is not supported: %s\n", input_file);
	}
	    
	spDebug(10, "spMain", "can't find suitable plugin\n");
	if ((i_plugin = spOpenFilePluginArg("input_raw", input_file, "r",
					    SP_PLUGIN_DEVICE_FILE,
					    &i_wave_info, NULL, argc, argv, NULL)) == NULL) {
	    spError(1, "Can't find plugins for input.\n");
	}
    }
    
    if (quiet_flag == SP_FALSE) {
        spMessage("\nInput Filename: %s\n", input_file);
        if (!strnone(i_wave_info.file_type)) {
            spMessage("Input File Type: %s\n", i_wave_info.file_type);
        }
        spMessage("Input Sampling Rate: %f\n", i_wave_info.samp_rate);
    }
    
    if (strnone(output_file)) {
        device_type = SP_PLUGIN_DEVICE_AUDIO;
        strcpy(o_plugin_name, "output_audio");
    } else {
        device_type = SP_PLUGIN_DEVICE_FILE;
        spDebug(1, "main", "output_file = %s\n", output_file);
    }
    spDebug(1, "main", "o_plugin_name = %s\n", o_plugin_name);

    spCopyWaveInfo(&o_wave_info, &i_wave_info);
    strcpy(o_wave_info.file_type, "");
    o_wave_info.samp_rate = o_samp_freq;
	    
    if ((o_plugin = spOpenFilePluginArg(o_plugin_name, output_file, "w",
                                        device_type,
                                        &o_wave_info, NULL, 0, NULL, NULL)) == NULL) {
        if (strnone(output_file)) {
            spError(1, "Cannot open audio engine\n");
        } else {
            spError(1, "Can't open output file: %s (plugin: %s)\n", output_file, o_plugin_name);
        }
    }
    
    if (i_wave_info.num_channel != o_wave_info.num_channel) {
	spError(1, "Number of channels of input file (%d) and output file (%d) must be same.\n",
		i_wave_info.num_channel, o_wave_info.num_channel);
    }

    if (help_flag == SP_TRUE) {
	usage(i_plugin, o_plugin);
    }
    
    resample_config.num_channel = i_wave_info.num_channel;
    resample_config.options |= SP_RESAMPLE_OPTION_INPUT_INTERLEAVED;
    resample_config.options |= SP_RESAMPLE_OPTION_OUTPUT_INTERLEAVED;
    resample_config.options |= SP_RESAMPLE_OPTION_TRUNCATE_EDGE;
	    
    if ((resample = spResampleOpen(&resample_config)) != NULL) {
        orig_input_buf_length = resample_config.max_input_buf_length;
		
        if ((new_samp_freq = spResampleSetFrequency(resample, i_wave_info.samp_rate, o_wave_info.samp_rate,
                                                    &max_output_buf_length, &delay)) > 0.0) {
	    if (quiet_flag == SP_FALSE) {
                spMessage("New Sampling Frequency: %f\n", new_samp_freq);
            }
            spDebug(1, "main", "orig_input_buf_length = %ld, max_output_buf_length = %ld, delay = %ld\n",
                    orig_input_buf_length, max_output_buf_length, delay);

            input_buf = xalloc(orig_input_buf_length, double);
            output_buf = xalloc(max_output_buf_length, double);

            total_read = total_write = 0;

            while (1) {
                read_length = spReadPluginDouble(i_plugin, input_buf, orig_input_buf_length);
                spDebug(1, "main", "read_length = %ld\n", read_length);
			
                if (read_length <= 0) {
                    output_buf_length = spResampleFlush(resample, output_buf);
                    spDebug(1, "main", "resample flushed: output_buf_length = %ld\n", output_buf_length);
                } else {
                    total_read += read_length;
                    output_buf_length = spResampleProcess(resample, input_buf, read_length, output_buf, &delay);
                    spDebug(1, "main", "read_length = %ld, output_buf_length = %ld, delay = %ld\n",
                            read_length, output_buf_length, delay);
                }

                if (output_buf_length > 0) {
                    write_length = spWritePluginDoubleWeighted(o_plugin, output_buf, output_buf_length, gain);
                    if (write_length > 0) {
                        total_write += write_length;
                    }
                }

                if (read_length <= 0) {
                    break;
                }
            }

            spDebug(1, "main", "total_read = %ld, total_write = %ld\n", total_read, total_write);
		    
            xfree(input_buf);
            xfree(output_buf);
        } else {
            spWarning("Cannot set sampling frequency: %f\n", o_wave_info.samp_rate);
        }

        spResampleClose(resample);
        spDebug(1, "main", "resample finished\n");
    } else {
        spWarning("Cannot open resample engine\n");
    }
	    
    if (quiet_flag == SP_FALSE && device_type == SP_PLUGIN_DEVICE_FILE) {
	spMessage("\nOutput Filename: %s\n", output_file);
        if (!strnone(o_wave_info.file_type)) {
            spMessage("File Type: %s\n", o_wave_info.file_type);
        }
	spMessage("Sampling Rate: %f\n", o_wave_info.samp_rate);
	spMessage("Number of Channels: %d\n", o_wave_info.num_channel);
	spMessage("Bits/Sample: %d\n", o_wave_info.samp_bit);
    }
    
    /* close plugin */
    spCloseFilePlugin(o_plugin);
    spCloseFilePlugin(i_plugin);

    /* write parameter file */
    if (!strnone(o_param_file)) {
        if (quiet_flag == SP_FALSE) {
            spMessage("Output Parameter File: %s\n", o_param_file);
        }
	spWriteSetup(o_param_file, options);
    }
    
    return 0;
}
