/*
 * 	spPicola.c: PICOLA library version
 *
 *	Last modified: <2019-12-02 00:07:03 hideki>
 *
 *------------------------------------------------------------------- 
 * This library version by Hideki BANNO
 *
 * Time domain harmonic scaling by
 * Pointer Inteval Controled OverLap and ADD (PICOLA) Method
 *		original C version by IKEDA Mikio
 *		original argolithm is developed by MORITA Naotaka
 *		about detail, see original paper.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <sp/spLib.h>
#include <sp/spPicola.h>

struct _spPicola {
    spPicolaConfig *config;

    long max_input_buf_length;
    long min_period;
    long max_period;
    
    long pitch_calc_length;

    double rate;
    double rcomp;
    double fraction;

    double *buf;
    long buf_length;

    long buf_offset;
    long buf_filled;
    
    long flush_need_length;

    long total_input_length;
    long total_output_length;
    double total_required_length;
};

/*
 *------------ PICOLA OverLap and Add (picOLA) stage ------------------
 */
static void ola(long pitch, double *is1, long is1_offset, double *is2, long is2_offset, long num_channel)
{
    long i, j;
    double s, w, step;
    double *ist1, *ist2;

    ist1 = is1 + num_channel * is1_offset;
    ist2 = is2 + num_channel * is2_offset;

    if (pitch < 1) {
        step = 0.0;
    } else {
        step = 1.0 / (double)pitch;
        /*step = 1.0 / (double)(pitch - 1);*/
    }
    
    for (i = 0; i < pitch; ++i) {
	w = (double)i * step;
	
	for (j = 0; j < num_channel; ++j) {
	    s = ist1[i * num_channel + j] * (1.0 - w) + ist2[i * num_channel + j] * w;
	    ist2[i * num_channel + j] = s;
	}
    }
    
    return;
}

static long amdfpitch(long pitmin, long pitmax, double *is, long is_offset, long length, long num_channel)
{
    long i, j, k;
    long pitch;
    long acc_length;
    double diff, acc, accmin;
    double *ist;

    ist = is + num_channel * is_offset;

    /*acc_length = length - pitmax;*/
    acc_length = length;
    
    pitch = pitmin;
    accmin = 0.0;
    for (j = 0; j < acc_length; ++j) {
	for (k = 0, diff = 0.0; k < num_channel; k++) {
	    if (j + pitmin >= length) {
		diff += FABS(ist[j * num_channel + k]);
	    } else {
		diff += FABS(ist[(j + pitmin) * num_channel + k] - ist[j * num_channel + k]);
	    }
	}
	accmin += diff;
    }
    
    for (i = pitmin + 1; i <= pitmax; ++i) {
	acc = 0.0;
	for (j = 0; j < acc_length; ++j) {
	    for (k = 0, diff = 0.0; k < num_channel; k++) {
		if (i + j >= length) {
		    diff += FABS(ist[j * num_channel + k]);
		} else {
		    diff += FABS(ist[(i + j) * num_channel + k] - ist[j * num_channel + k]);
		}
	    }
	    acc += diff;
	}
	if (acc < accmin) {
	    accmin = acc;
	    pitch = i;
	}
    }
    
    return pitch;
}

spBool _spPicolaInitConfig(spPicolaConfig *config, unsigned long version_id)
{
    if (config == NULL || version_id > SPPICOLA_VERSION_ID) return SP_FALSE;
    
    memset(config, 0, sizeof(spPicolaConfig));
    config->version_id = version_id;

    config->samp_freq = 8000.0;
    config->num_channel = 1;
    config->max_input_buf_length_ms = 20.0;
    config->min_period_ms = 4.0; 	/* 250Hz */
    config->max_period_ms = 15.0; 	/* 20ms=50Hz, 15ms=66.6Hz; original version: 60Hz */
    
    return SP_TRUE;
}

spPicola spPicolaOpen(spPicolaConfig *config)
{
    spPicola picola;
    
    picola = xalloc(1, struct _spPicola);
    memset(picola, 0, sizeof(struct _spPicola));
    picola->config = config;

    picola->config->num_channel = MAX(picola->config->num_channel, 1);

    picola->config->max_period_ms = MAX(picola->config->max_period_ms, picola->config->min_period_ms * 2.0);
    
    picola->rate = 1.0;
    picola->rcomp = 1.0;
    
    picola->fraction = 0.0;

    picola->max_input_buf_length = (long)spRound(config->samp_freq * config->max_input_buf_length_ms / 1000.0);
    picola->min_period = (long)spRound(config->samp_freq * config->min_period_ms / 1000.0);
    picola->max_period = (long)spRound(config->samp_freq * config->max_period_ms / 1000.0);
    picola->pitch_calc_length = picola->max_period * 2;
    spDebug(30, "spPicolaOpen", "max_input_buf_length = %ld, min_period = %ld, max_period = %ld, pitch_calc_length = %ld\n",
	    picola->max_input_buf_length, picola->min_period, picola->max_period, picola->pitch_calc_length);

    picola->buf_length = 2 * picola->pitch_calc_length + picola->max_period + picola->max_input_buf_length;
    picola->buf = xalloc(picola->buf_length * picola->config->num_channel, double);
    spDebug(30, "spPicolaOpen", "buf_length = %ld\n", picola->buf_length);
    
    picola->total_input_length = 0;
    picola->total_output_length = 0;
    picola->total_required_length = 0.0;
    
    return picola;
}

spBool spPicolaSetRate(spPicola picola, double rate, long *max_input_buf_length, long *max_output_buf_length)
{
    if (picola == NULL || max_output_buf_length == NULL) return SP_FALSE;
    
    if (rate == 1.0) {
	picola->rcomp = 0.0;
    } else if (rate > 1.0) {
	picola->rcomp = 1.0  / (rate - 1.0);
    } else if (rate > 0) {
	picola->rcomp = rate / (1.0 - rate);
    } else {
	return SP_FALSE;
    }
    picola->rate = rate;

    if (max_input_buf_length != NULL) *max_input_buf_length = picola->config->num_channel * picola->max_input_buf_length;

    *max_output_buf_length = picola->config->num_channel
	* (picola->buf_length + 2 + (long)spRound(2.0 * rate * picola->max_input_buf_length) / 2);
    
    spDebug(30, "spPicolaSetRate", "rate = %f, rcomp = %f, max_output_buf_length = %ld\n",
	    picola->rate, picola->rcomp, *max_output_buf_length);
    
    return SP_TRUE;
}

static long flushBuffer(spPicola picola, double *output_buf, long output_buf_length2, long num_channel, long output_offset)
{
    long k, l;
    long shift;
    
    if (picola->flush_need_length > 0) {
	for (k = 0; k < MIN(picola->flush_need_length, picola->buf_filled)
		 && output_offset < output_buf_length2; k++) {
	    for (l = 0; l < num_channel; l++) {
		output_buf[output_offset * num_channel + l] = picola->buf[k * num_channel + l];
	    }
	    output_offset++;
	    picola->total_output_length++;
	}
	shift = k;
	spDebug(80, "flushBuffer", "flushed: flush_need_length = %ld, output_offset = %ld, shift = %ld, buf_filled = %ld\n",
		picola->flush_need_length, output_offset, shift, picola->buf_filled);
	picola->flush_need_length -= shift;
	picola->buf_filled -= shift;
	for (k = 0; k < picola->buf_filled; k++) {
	    for (l = 0; l < num_channel; l++) {
		picola->buf[k * num_channel + l] = picola->buf[(k + shift) * num_channel + l];
	    }
	}
	picola->buf_offset = 0;
    }

    spDebug(80, "flushBuffer", "done: flush_need_length = %ld, buf_filled = %ld\n",
	    picola->flush_need_length, picola->buf_filled);
    
    return output_offset;
}

static void picolaMain(spPicola picola, long min_period, long max_period, long pitch_calc_length, long num_channel)
{
    long k, l;
    long lcopy;
    long pitch, pitch2;
    double skip_len;
    
    spDebug(80, "picolaMain", "pitch_calc_length = %ld\n", pitch_calc_length);
    
    /*---- pitch extraction ----*/
    pitch = amdfpitch(min_period, max_period,
		      picola->buf, picola->buf_offset, pitch_calc_length, num_channel);
    spDebug(80, "picolaMain", "pitch extracted: pitch = %ld\n", pitch);
		
    /*---- compensate compansion rate ----*/
    skip_len = (double)pitch * picola->rcomp;
    lcopy = (long)skip_len;
    picola->fraction += (double)lcopy - skip_len;
    if (picola->fraction >= 0.5) {	
	--lcopy;
	picola->fraction -= 1.0;
    } else if (picola->fraction <= -0.5) {
	++lcopy;
	picola->fraction += 1.0;
    }
    picola->flush_need_length += lcopy + picola->buf_offset;
    spDebug(80, "picolaMain", "skip_len = %f, lcopy = %ld, fraction = %f\n", skip_len, lcopy, picola->fraction);

    /*---- PICOLA OverLap and ADD stage ----*/
    if (picola->rate < 1.0) {
	ola(pitch, picola->buf, picola->buf_offset,
	    picola->buf, picola->buf_offset + pitch, num_channel);
	picola->buf_filled -= pitch;
	for (k = picola->buf_offset; k < picola->buf_filled; k++) {  /* remove Tp length wavelet */ 
	    for (l = 0; l < num_channel; l++) {
		picola->buf[k * num_channel + l] = picola->buf[(k + pitch) * num_channel + l];
	    }
	}
    } else {
	picola->buf_filled += pitch;
	for (k = picola->buf_filled-1; k >= pitch + picola->buf_offset; k--) {  /* insert Tp length wavelet */ 
	    for (l = 0; l < num_channel; l++) {
		picola->buf[k * num_channel + l] = picola->buf[(k - pitch) * num_channel + l];
	    }
	}
	picola->flush_need_length += pitch;

#if 1
	pitch2 = MIN(picola->buf_filled - picola->buf_offset - 2 * pitch, pitch);
	pitch2 = MAX(pitch2, 0);
#else
	pitch2 = pitch;
#endif
	spDebug(80, "picolaMain", "pitch = %ld, pitch2 = %ld\n", pitch, pitch2);
	
	if (pitch2 > 0) {
	    ola(pitch2, picola->buf, picola->buf_offset + 2 * pitch,
		picola->buf, picola->buf_offset + pitch, num_channel);
	}
    }

    return;
}

static long copyInput(spPicola picola, double *input_buf, long input_offset, long num_channel)
{
    long l;
    
    for (l = 0; l < num_channel; l++) {
	picola->buf[picola->buf_filled * num_channel + l] = input_buf[input_offset * num_channel + l];
    }
    picola->total_input_length++;
    picola->buf_filled++;
    input_offset++;

    return input_offset;
}

spBool spPicolaProcess(spPicola picola, 
		       double *input_buf, long *input_buf_length, double *output_buf, long *output_buf_length,
		       long *input_delay)
{
    long num_channel;
    long input_offset, output_offset;
    long input_buf_length2, output_buf_length2;

    if (picola == NULL || input_buf == NULL || input_buf_length == NULL
	|| output_buf == NULL || output_buf_length == NULL
	|| *input_buf_length <= 0 || *output_buf_length <= 0) {
	return SP_FALSE;
    }

    spDebug(30, "spPicolaProcess", "input_buf_length = %ld, output_buf_length = %ld\n",
	    *input_buf_length, *output_buf_length);

    num_channel = picola->config->num_channel;
    
    input_offset = 0;
    output_offset = 0;
    input_buf_length2 = *input_buf_length / num_channel;
    output_buf_length2 = *output_buf_length / num_channel;

    while (input_offset < input_buf_length2 && output_offset < output_buf_length2) {
	spDebug(100, "spPicolaProcess",
		"input_offset = %ld / %ld, output_offset = %ld / %ld, flush_need_length = %ld, buf_filled = %ld\n",
		input_offset, input_buf_length2, output_offset, output_buf_length2,
		picola->flush_need_length, picola->buf_filled);
	
	if (picola->flush_need_length > 0) {
	    output_offset = flushBuffer(picola, output_buf, output_buf_length2, num_channel, output_offset);
		
	    while (input_offset < input_buf_length2 && picola->buf_filled < picola->buf_length
		   && picola->buf_filled < picola->flush_need_length) {
		input_offset = copyInput(picola, input_buf, input_offset, num_channel);
	    }
	    
	    if (picola->flush_need_length > picola->buf_filled) {
		/* buffer contents are not enough */
		spDebug(50, "spPicolaProcess", "buffer contents are not enough: flush_need_length = %ld, buf_filled = %ld\n",
			picola->flush_need_length, picola->buf_filled);
		break;
	    }
	}

	spDebug(100, "spPicolaProcess", "buf_offset = %ld, buf_filled = %ld\n", picola->buf_offset, picola->buf_filled);
	
	if (picola->flush_need_length <= 0) {
	    while (input_offset < input_buf_length2
		   && ((picola->rate <= 1.0 && picola->buf_filled < picola->buf_length)
		       || (picola->rate > 1.0 && picola->buf_filled + picola->max_period < picola->buf_length))) {
		input_offset = copyInput(picola, input_buf, input_offset, num_channel);

		if (picola->rate == 1.0) {
		    picola->flush_need_length = picola->buf_filled;
		} else {
		    if (picola->buf_filled - picola->buf_offset >= picola->pitch_calc_length) {
			picolaMain(picola, picola->min_period, picola->max_period,
				   picola->pitch_calc_length, num_channel);
			break;
		    }
		}
	    }

	    if (picola->flush_need_length <= 0) {
		break;
	    }
	}
    }

    if (input_offset < input_buf_length2) {
	spDebug(30, "spPicolaProcess",
		"Warning: input buffer is not processed: input_offset = %ld, input_buf_length = %ld\n",
		input_offset, input_buf_length2);
    }
    
    *input_buf_length = input_offset * num_channel;
    *output_buf_length = output_offset * num_channel;
    picola->total_required_length += (picola->rate * (double)input_offset);
    spDebug(30, "spPicolaProcess", "total_output_length = %ld, total_required_length = %f\n",
	    picola->total_output_length, picola->total_required_length);

    if (input_delay != NULL) *input_delay = picola->buf_filled;

    if (picola->buf_filled >= picola->buf_length) {
	spDebug(30, "spPicolaProcess",
		"Warning: buffer too short!: buf_filled = %ld, buf_length = %ld\n", picola->buf_filled, picola->buf_length);
    }

    spDebug(30, "spPicolaProcess",
	    "finished: input_offset = %ld, output_offset = %ld, flush_need_length = %ld, buf_filled = %ld, buf_length = %ld, total_required_length = %f\n",
	    input_offset, output_offset, picola->flush_need_length, picola->buf_filled,
	    picola->buf_length, picola->total_required_length);
    
    return SP_TRUE;
}

spBool spPicolaFlush(spPicola picola, double *output_buf, long *output_buf_length)
{
    long k, l;
    long num_channel;
    long output_offset;
    long output_buf_length2;
    long pitch_calc_length, min_period, max_period;
    
    if (picola == NULL 
	|| output_buf == NULL || output_buf_length == NULL
	|| *output_buf_length <= 0) {
	return SP_FALSE;
    }

    spDebug(30, "spPicolaFlush", "total_input_length = %ld, total_output_length = %ld, total_required_length = %f\n",
	    picola->total_input_length, picola->total_output_length, picola->total_required_length);
    
    num_channel = picola->config->num_channel;
    output_offset = 0;
    output_buf_length2 = *output_buf_length / num_channel;

    spDebug(30, "spPicolaFlush", "flush_need_length = %ld, buf_filled = %ld, output_buf_length = %ld, buf_length = %ld, total_required_length = %f\n",
	    picola->flush_need_length, picola->buf_filled, output_buf_length2, picola->buf_length, picola->total_required_length);

#if 1
    /*max_period = MIN(picola->max_period, picola->buf_filled / 4);*/
    max_period = MIN(picola->max_period, picola->total_input_length / 2);
    min_period = MIN(picola->min_period, max_period / 2);
#else
    max_period = picola->max_period;
    min_period = picola->min_period;
#endif
    
    spDebug(30, "spPicolaFlush", "min_period = %ld, max_period = %ld\n",
	    min_period, max_period);
    
    /*while (output_offset < output_buf_length2 && picola->buf_filled - picola->buf_offset > 0) {*/
    /*while (output_offset < output_buf_length2 && picola->buf_filled - picola->buf_offset > 2 * picola->max_period) {*/
    while (output_offset < output_buf_length2 && picola->buf_filled - picola->buf_offset > picola->max_period) {
	if (picola->flush_need_length > 0) {
	    output_offset = flushBuffer(picola, output_buf, output_buf_length2, num_channel, output_offset);
	}
	
	if (picola->flush_need_length <= 0) {
	    if (picola->rate == 1.0) {
		picola->flush_need_length = picola->buf_filled;
	    } else {
		if (((picola->rate <= 1.0 && picola->buf_filled < picola->buf_length)
		     || (picola->rate > 1.0 && picola->buf_filled + picola->max_period < picola->buf_length))) {
		    pitch_calc_length = MIN(picola->pitch_calc_length, picola->buf_filled - picola->buf_offset);
		    if (pitch_calc_length < picola->max_period) {
			break;
		    }
		    picolaMain(picola, min_period, max_period, pitch_calc_length, num_channel);
		} else {
		    spDebug(30, "spPicolaFlush",
			    "Warning: buffer too short!: buf_filled = %ld, buf_length = %ld\n", picola->buf_filled, picola->buf_length);
		    break;
		}
	    }
	}
    }
    spDebug(30, "spPicolaFlush", "loop end: flush_need_length = %ld, buf_filled = %ld, output_buf_length = %ld\n",
	    picola->flush_need_length, picola->buf_filled, output_buf_length2);

    if (picola->buf_filled > 0) {
	for (k = 0; k < picola->buf_filled; k++) {
	    if (output_offset >= output_buf_length2) {
		spDebug(30, "spPicolaFlush",
			"buffer length is not enough: output_offset = %ld, output_buf_length = %ld\n",
			output_offset, output_buf_length2);
		break;
	    }
	    for (l = 0; l < num_channel; l++) {
		output_buf[output_offset * num_channel + l] = picola->buf[k * num_channel + l];
	    }
	    output_offset++;
	    picola->total_output_length++;
	}
	picola->buf_filled -= k;
    }
    
    *output_buf_length = output_offset * num_channel;
    
    spDebug(30, "spPicolaFlush",
	    "done: output_offset = %ld, buf_filled = %ld, total_output_length = %ld, total_required_length = %f\n",
	    output_offset, picola->buf_filled, picola->total_output_length, picola->total_required_length);
    
    picola->total_input_length = 0;
    picola->total_output_length = 0;
    picola->total_required_length = 0.0;
    
    return SP_TRUE;
}
     
spBool spPicolaClose(spPicola picola)
{
    if (picola == NULL) return SP_FALSE;
    
    spDebug(30, "spPicolaClose", "buf_length = %ld\n", picola->buf_length);
    
    if (picola->buf != NULL) xfree(picola->buf);
    xfree(picola);
    
    return SP_TRUE;
}
