/** @file sp/vector.h
 */

#ifndef SPLIB_VECTOR_H
#define SPLIB_VECTOR_H

#include <stdlib.h>
#include <sp/spPlugin.h>

#ifdef __cplusplus
extern "C" {
#endif

/** @~english @defgroup vectorGroup <sp/vector.h>: Vector
@code
#include <sp/vector.h>
@endcode

Introduction
------------

The vector type is a type which is used as an argument or a return value of many signal processing functions of this library.
Types of spSVector for short type, spLVector for long type, spFVector for float type and spDVector for double type are provided.

The memory of spSVector , spLVector , spFVector or spDVector can be allocated by the functions which have a prefix xsv, xlv, xfv or xdv respectively.
If you don't use the vector anymore,
you have to deallocate the memory by calling xsvfree() , xlvfree() , xfvfree() or xdvfree() .


Lock and Unlock
---------------

Lock and unlock functions for the vector type, which are introduced from version 0.9,
are important for fast computation.
When you access the data array of the vector such as `->data` and `->imag`,
you must lock the vector, i.e. must call lock function, before access.
Since the unlocked vector has correct data only in the internal data structure which enables fast computation internally,
fast computation can be realized by applying processing directly to the internal data structure
which can be different from the data structure of the vector type.
So, you have to unlock the vector before calling the function
when you need fast computation (the initial allocated vector is normally locked).
In the case of a function with multiple argument vectors, fast computation cannot be 
performed unless all the argument vectors are locked.
Note that the gain of fast computation will be small when you call the unlock/lock functions many times,
because the unlock/lock functions include big overhead on some environments.
Therefore, calling multiple signal processing functions after unlocking the vector, then locking the vector is desirable.
*/

/** @addtogroup vectorGroup
 *  @{ */  /*----@addtogroup vectorGroup----*/

/** @~english Indicates precision of computation. */
typedef enum {
    SP_HIGHEST_PRECISION = 128,         /**< @~english Highest precision. */
    SP_HIGH_DOUBLE_PRECISION = 72,      /**< @~english Double type precision higher than normal double type. */
    SP_DOUBLE_PRECISION = 64,           /**< @~english double type precision.  */
    SP_LOW_DOUBLE_PRECISION = 48,       /**< @~english double type precision lower than normal double type  */
    SP_HIGH_FLOAT_PRECISION = 34,       /**< @~english float type precision higher than normal double type.  */
    SP_FLOAT_PRECISION = 33,            /**< @~english float type precision.  */
    SP_32BIT_PRECISION = 32,            /**< @~english 32-bit integer precision.  */
    SP_24BIT_PRECISION = 24,            /**< @~english 24-bit integer precision.  */
    SP_16BIT_PRECISION = 16,            /**< @~english 16-bit integer precision.  */
    SP_8BIT_PRECISION = 8,              /**< @~english 8-bit integer precision.  */
    SP_IGNORE_PRECISION = 0,            /**< @~english Indicates that precision is ignored.  */
    SP_UNKNOWN_PRECISION = 255,         /**< @~english Unknown precision.  */
} spPrecision;

/** @~english Indicates speed of computation. */
typedef enum {
    SP_COMPUTE_SPEED_EXTREMELY_FAST = 35,       /**< @~english Extremely fast speed. */
    SP_COMPUTE_SPEED_VERY_FAST = 30,            /**< @~english Very fast speed.  */
    SP_COMPUTE_SPEED_FASTER = 25,               /**< @~english Faster speed.  */
    SP_COMPUTE_SPEED_FAST = 20,                 /**< @~english Fast speed.  */
    SP_COMPUTE_SPEED_NORMAL = 10,               /**< @~english Normal speed.  */
    SP_COMPUTE_SPEED_SLOW = 5,                  /**< @~english Slow speed.  */
    SP_COMPUTE_SPEED_SLOWER = 4,                /**< @~english Slower speed.  */
    SP_COMPUTE_SPEED_VERY_SLOW = 3,             /**< @~english Very slow speed.  */
    SP_COMPUTE_SPEED_EXTREMELY_SLOW = 2,        /**< @~english Extremely slow speed.  */
    SP_COMPUTE_SPEED_UNKNOWN = -1,              /**< @~english Unknown speed.  */
} spComputeSpeed;
    
#define SP_LOW_FLOAT_PRECISION SP_32BIT_PRECISION /**< @~english float type precision lower than normal float type  */
#define SP_LONG_PRECISION SP_32BIT_PRECISION /**< @~english long type precision.  */
#define SP_SHORT_PRECISION SP_16BIT_PRECISION /**< @~english short type precision.  */
#define SP_CHAR_PRECISION SP_8BIT_PRECISION /**< @~english char type precision.  */

#ifdef SP_USE_VECTOR_ENGINE

#define SP_VECTOR_LOCK_SWITCH_NORMAL 0
#define SP_VECTOR_LOCK_SWITCH_SYNC_ONLY 1
#define SP_VECTOR_LOCK_SWITCH_NOSYNC 2

/* currently unofficial plugin */
#define SP_PLUGIN_VECTOR "vector 1.2"

typedef struct _spVectorPluginRec spVectorPluginRec;
    
#ifdef SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE
#define SP_PLUGIN_BLAS "blas 0.1"
#define SP_PLUGIN_LAPACK "lapack 0.1"

#define SP_PLUGIN_CAPS_SUPPORT_BLAS_FUNCTIONS (1<<13)
#define SP_PLUGIN_CAPS_SUPPORT_LAPACK_FUNCTIONS (1<<14)
#define SP_PLUGIN_CAPS_SUPPORT_LINEAR_ALGEBRA_FUNCTIONS (SP_PLUGIN_CAPS_SUPPORT_BLAS_FUNCTIONS|SP_PLUGIN_CAPS_SUPPORT_LAPACK_FUNCTIONS)

#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_CHILD_PLUGIN (1<<1)
#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_UNLOCKED (1<<2)
#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_SYNCED (1<<3)
#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_DATA_IN_REAL (1<<4)
#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_DATA_IN_IMAG (1<<5)
#define SP_LINEAR_ALGEBRA_INSTANCE_STATUS_DATA_IN_CPLX (1<<6)

typedef enum {
    SP_MATRIX_MEMORY_LAYOUT_COLUMN_MAJOR = 0,
    SP_MATRIX_MEMORY_LAYOUT_ROW_MAJOR = 1,
} spMatrixMemoryLayout;

typedef struct _spLinearAlgebraPluginRec spLinearAlgebraPluginRec;
typedef struct _spLinearAlgebraPluginList *spLinearAlgebraPluginList;

struct _spLinearAlgebraPluginList {
    spPlugin *plugin;
    void *instance;
    unsigned long status_mask;
    struct _spLinearAlgebraPluginList *prev;
    struct _spLinearAlgebraPluginList *next;
};

#define SpVectorLinearAlgebraPluginRec(vec) ((spLinearAlgebraPluginRec *)((vec)->la_current_list->plugin->rec))
#define SpMatrixLinearAlgebraPluginRec(mat) ((spLinearAlgebraPluginRec *)((mat)->la_current_list->plugin->rec))

#endif /* SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE */
#endif /* SP_USE_VECTOR_ENGINE */
    
typedef struct _spSVector *spSVector;
typedef struct _spLVector *spLVector;
typedef struct _spFVector *spFVector;
typedef struct _spDVector *spDVector;

/** @class spSVector vector.h <sp/vector.h>
 *  @~english
 *  @brief Vector type that contains the elements of short type.
 *  @details Actually, This is a typedefed type by `typedef struct _spSVector *spSVector;`.
 *  To allocate memory for spSVector, call a function with prefix xsv such as xsvalloc() , xsvinit() , etc.
 *  To deallocate memory, call xsvfree() . */
struct _spSVector {
    /** @~english The number of elements of spSVector.
     *  You can access the number of elements as \p x->length for the memory allocated \p x of spSVector. */
    long length;
    
    /** @~english Data of the real part of spSVector.
     *  You can access the 0-th element of the data as \p x->data[0]
     *  for the memory allocated \p x of spSVector. */
    short *data;
    
    /** @~english Data of the imaginary part of spSVector.
     *  You can access the 0-th element of the data as \p x->imag[0]
     *  for the memory allocated \p x of spSVector.
     *  If no imaginary part exists, this member can be NULL. */
    short *imag;
    
#ifdef SP_USE_VECTOR_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spLVector vector.h <sp/vector.h>
 *  @~english
 *  @brief Vector type that contains the elements of long type.
 *  @details Actually, This is a typedefed type by `typedef struct _spLVector *spLVector;`.
 *  To allocate memory for spLVector, call a function with prefix xlv such as xlvalloc() , xlvinit() , etc.
 *  To deallocate memory, call xlvfree() . */
struct _spLVector {
    /** @~english The number of elements of spLVector.
     *  You can access the number of elements as \p x->length for the memory allocated \p x of spLVector. */
    long length;
    
    /** @~english Data of the real part of spLVector.
     *  You can access the 0-th element of the data as \p x->data[0]
     *  for the memory allocated \p x of spLVector. */
    long *data;
    
    /** @~english Data of the imaginary part of spLVector.
     *  You can access the 0-th element of the data as \p x->imag[0]
     *  for the memory allocated \p x of spLVector.
     *  If no imaginary part exists, this member can be NULL. */
    long *imag;
    
#ifdef SP_USE_VECTOR_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spFVector vector.h <sp/vector.h>
 *  @~english
 *  @brief Vector type that contains the elements of float type.
 *  @details Actually, This is a typedefed type by `typedef struct _spFVector *spFVector;`.
 *  To allocate memory for spFVector, call a function with prefix xfv such as xfvalloc() , xfvinit() , etc.
 *  To deallocate memory, call xfvfree() . */
struct _spFVector {
    /** @~english The number of elements of spFVector.
     *  You can access the number of elements as \p x->length for the memory allocated \p x of spFVector. */
    long length;
    
    /** @~english Data of the real part of spFVector.
     *  You can access the 0-th element of the data as \p x->data[0]
     *  for the memory allocated \p x of spFVector. */
    float *data;
    
    /** @~english Data of the imaginary part of spFVector.
     *  You can access the 0-th element of the data as \p x->imag[0]
     *  for the memory allocated \p x of spFVector.
     *  If no imaginary part exists, this member can be NULL. */
    float *imag;
    
#ifdef SP_USE_VECTOR_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

/** @class spDVector vector.h <sp/vector.h>
 *  @~english
 *  @brief Vector type that contains the elements of double type.
 *  @details Actually, This is a typedefed type by `typedef struct _spDVector *spDVector;`.
 *  To allocate memory for spDVector, call a function with prefix xdv such as xdvalloc() , xdvinit() , etc.
 *  To deallocate memory, call xdvfree() . */
struct _spDVector {
    /** @~english The number of elements of spDVector.
     *  You can access the number of elements as \p x->length for the memory allocated \p x of spDVector. */
    long length;
    
    /** @~english Data of the real part of spDVector.
     *  You can access the 0-th element of the data as \p x->data[0]
     *  for the memory allocated \p x of spDVector. */
    double *data;

    /** @~english Data of the imaginary part of spDVector.
     *  You can access the 0-th element of the data as \p x->imag[0]
     *  for the memory allocated \p x of spDVector.
     *  If no imaginary part exists, this member can be NULL. */
    double *imag;
    
#ifdef SP_USE_VECTOR_ENGINE
    spPlugin *plugin;
    void *instance;
    spBool locked;
#ifdef SP_USE_VECTOR_LINEAR_ALGEBRA_ENGINE
    spLinearAlgebraPluginList la_plugin_list;
    spLinearAlgebraPluginList la_current_list;
    unsigned long la_status_mask;
#endif
#endif
};

typedef struct _spSVectors *spSVectors;
typedef struct _spLVectors *spLVectors;
typedef struct _spFVectors *spFVectors;
typedef struct _spDVectors *spDVectors;

struct _spSVectors {
    long num_vector;
    spSVector *vector;
};

struct _spLVectors {
    long num_vector;
    spLVector *vector;
};

struct _spFVectors {
    long num_vector;
    spFVector *vector;
};

struct _spDVectors {
    long num_vector;
    spDVector *vector;
};

/** @} */  /*----@addtogroup vectorGroup----*/

#define spvlength(x) ((x)->length)
#define spvdata(x) ((x)->data)
#define spvreal(x) ((x)->data)
#define spvimag(x) ((x)->imag)

#define SVector spSVector
#define LVector spLVector
#define FVector spFVector
#define DVector spDVector
#define _SVector _spSVector
#define _LVector _spLVector
#define _FVector _spFVector
#define _DVector _spDVector

#define SVectors spSVectors
#define LVectors spLVectors
#define FVectors spFVectors
#define DVectors spDVectors
#define _SVectors _spSVectors
#define _LVectors _spLVectors
#define _FVectors _spFVectors
#define _DVectors _spDVectors

#define vlength spvlength
#define vdata spvdata
#define vreal spvreal
#define vimag spvimag

/* Version 0.9.1+ */
#define SP_DVECTOR_NO_CALLBACK 0L
#define SP_DVECTOR_READ_STARTED_CALLBACK (1L<<0) /* callback_data: 0 */
#define SP_DVECTOR_READ_PROGRESS_CALLBACK (1L<<1) /* callback_data: (long)current_read_length */
#define SP_DVECTOR_READ_FINISHED_CALLBACK (1L<<2) /* callback_data: (long)total_read_length */
#define SP_DVECTOR_WRITE_STARTED_CALLBACK (1L<<3) /* callback_data: 0 */
#define SP_DVECTOR_WRITE_PROGRESS_CALLBACK (1L<<4) /* callback_data: (long)current_write_length */
#define SP_DVECTOR_WRITE_FINISHED_CALLBACK (1L<<5) /* callback_data: (long)total_write_length */
typedef spBool (*spDVectorCallbackFunc)(spDVector x, unsigned long callback_type, void *callback_data, void *user_data);

#if defined(MACOS)
#pragma import on
#endif

#ifdef SP_USE_VECTOR_ENGINE
extern int spGetNumVectorPlugin(void);
extern char *xspSearchVectorPluginFile(int *index);
extern spPlugin *spSetDefaultVectorPlugin(const char *name);
extern spBool spIsVectorPlugin(spPlugin *plugin);
extern spPlugin *spLoadVectorPlugin(const char *name);
extern spBool spFreeVectorPlugin(spPlugin *plugin);
extern spPlugin *spGetDefaultVectorPlugin(void);
extern spBool spFreeDefaultVectorPlugin(void);

extern spBool spGetVectorPluginNumUnit(spPlugin *plugin, int *num_unit);
extern const char *spGetVectorPluginUnitName(spPlugin *plugin, int index);
extern spBool spSelectVectorPluginUnit(spPlugin *plugin, int index);
    
extern spBool spGetDefaultVectorPluginNumUnit(int *num_unit);
extern const char *spGetDefaultVectorPluginUnitName(int index);
extern spBool spSelectDefaultVectorPluginUnit(int index);

extern spBool spIsVectorPluginPrecisionSupported(spPlugin *plugin, spPrecision precision, spComputeSpeed *speed);
extern spBool spSetVectorPluginPrecision(spPlugin *plugin, spPrecision precision);
extern spBool spGetVectorPluginPrecision(spPlugin *plugin, spPrecision *precision);
    
extern spBool spIsVectorPluginFastModeSupported(spPlugin *plugin, spBool fast_flag);
extern spBool spSetVectorPluginFastMode(spPlugin *plugin, spBool fast_flag);
extern spBool spGetVectorPluginFastMode(spPlugin *plugin, spBool *fast_flag);
extern spBool spSetDefaultVectorPluginFastMode(spBool fast_flag);
extern spBool spGetDefaultVectorPluginFastMode(spBool *fast_flag);
#endif /* SP_USE_VECTOR_ENGINE */
    
extern long spsizeof(char type);
    
extern spPlugin *svgetplugin(spSVector x);
extern spPlugin *lvgetplugin(spLVector x);
extern spPlugin *fvgetplugin(spFVector x);
extern spPlugin *dvgetplugin(spDVector x);

extern spBool svsetplugin(spSVector x, spPlugin *plugin);
extern spBool lvsetplugin(spLVector x, spPlugin *plugin);
extern spBool fvsetplugin(spFVector x, spPlugin *plugin);
extern spBool dvsetplugin(spDVector x, spPlugin *plugin);
    
/** @~english @name Memory Allocation and Deallocation
 *  @{ */
 
/** @~english Allocates memory for spSVector. @public @memberof spSVector */
extern spSVector xsvalloc(long length);
/** @~english Allocates memory for spLVector. @public @memberof spLVector */
extern spLVector xlvalloc(long length);
/** @~english Allocates memory for spFVector. @public @memberof spFVector */
extern spFVector xfvalloc(long length);
/** @~english Allocates memory for spDVector. @public @memberof spDVector */
extern spDVector xdvalloc(long length);
/** @~english Deallocates memory for spSVector. @public @memberof spSVector */
extern void xsvfree(spSVector vector);
/** @~english Deallocates memory for spLVector. @public @memberof spLVector */
extern void xlvfree(spLVector vector);
/** @~english Deallocates memory for spFVector. @public @memberof spFVector */
extern void xfvfree(spFVector vector);
/** @~english Deallocates memory for spDVector. @public @memberof spDVector */
extern void xdvfree(spDVector vector);

extern spSVector xsvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag);
extern spLVector xlvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag);
extern spFVector xfvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag);
extern spDVector xdvallocfrominstance(spPlugin *plugin, void *instance, long length, spBool imag_flag);
    
/** @~english Allocates memory for spSVector and sets it in the lock/unlock state. @public @memberof spSVector */
extern spSVector xsvallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory for spLVector and sets it in the lock/unlock state. @public @memberof spLVector */
extern spLVector xlvallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory for spFVector and sets it in the lock/unlock state. @public @memberof spFVector */
extern spFVector xfvallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory for spDVector and sets it in the lock/unlock state. @public @memberof spDVector */
extern spDVector xdvallocul(spPlugin *plugin, long length, spBool unlock_flag);

/** @~english Allocates memory for spSVector and updates its lock/unlock state by using the status of the input vector. @public @memberof spSVector */
extern spSVector xsvalloculfromv(spSVector x, long length);
/** @~english Allocates memory for spLVector and updates its lock/unlock state by using the status of the input vector. @public @memberof spLVector */
extern spLVector xlvalloculfromv(spLVector x, long length);
/** @~english Allocates memory for spFVector and updates its lock/unlock state by using the status of the input vector. @public @memberof spFVector */
extern spFVector xfvalloculfromv(spFVector x, long length);
/** @~english Allocates memory for spDVector and updates its lock/unlock state by using the status of the input vector. @public @memberof spDVector */
extern spDVector xdvalloculfromv(spDVector x, long length);
    
/** @~english Deallocates memory of the real part for spSVector. @public @memberof spSVector */
extern void svrfree(spSVector x);
/** @~english Deallocates memory of the real part for spLVector. @public @memberof spLVector */
extern void lvrfree(spLVector x);
/** @~english Deallocates memory of the real part for spFVector. @public @memberof spFVector */
extern void fvrfree(spFVector x);
/** @~english Deallocates memory of the real part for spDVector. @public @memberof spDVector */
extern void dvrfree(spDVector x);

/** @~english Allocates memory of the imaginary part for spSVector. @public @memberof spSVector */
extern void svialloc(spSVector x);
/** @~english Allocates memory of the imaginary part for spLVector. @public @memberof spLVector */
extern void lvialloc(spLVector x);
/** @~english Allocates memory of the imaginary part for spFVector. @public @memberof spFVector */
extern void fvialloc(spFVector x);
/** @~english Allocates memory of the imaginary part for spDVector. @public @memberof spDVector */
extern void dvialloc(spDVector x);

/** @~english Deallocates memory of the imaginary part for spSVector. @public @memberof spSVector */
extern void svifree(spSVector x);
/** @~english Deallocates memory of the imaginary part for spLVector. @public @memberof spLVector */
extern void lvifree(spLVector x);
/** @~english Deallocates memory of the imaginary part for spFVector. @public @memberof spFVector */
extern void fvifree(spFVector x);
/** @~english Deallocates memory of the imaginary part for spDVector. @public @memberof spDVector */
extern void dvifree(spDVector x);

/** @~english Allocates memory of both the real and imaginary parts for spSVector and sets it in the lock/unlock state. @public @memberof spSVector */
extern spSVector xsvriallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory of both the real and imaginary parts for spLVector and sets it in the lock/unlock state. @public @memberof spLVector */
extern spLVector xlvriallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory of both the real and imaginary parts for spFVector and sets it in the lock/unlock state. @public @memberof spFVector */
extern spFVector xfvriallocul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english Allocates memory of both the real and imaginary parts for spDVector and sets it in the lock/unlock state. @public @memberof spDVector */
extern spDVector xdvriallocul(spPlugin *plugin, long length, spBool unlock_flag);

/** @~english Allocates memory of both the real and imaginary parts for spSVector. @public @memberof spSVector */
extern spSVector xsvrialloc(long length);
/** @~english Allocates memory of both the real and imaginary parts for spLVector. @public @memberof spLVector */
extern spLVector xlvrialloc(long length);
/** @~english Allocates memory of both the real and imaginary parts for spFVector. @public @memberof spFVector */
extern spFVector xfvrialloc(long length);
/** @~english Allocates memory of both the real and imaginary parts for spDVector. @public @memberof spDVector */
extern spDVector xdvrialloc(long length);

/** @copydoc spDVector::xdvrealloc()
 *  @public @memberof spSVector */
extern spSVector xsvrealloc(spSVector x, long length);
/** @copydoc spDVector::xdvrealloc()
 *  @public @memberof spLVector */
extern spLVector xlvrealloc(spLVector x, long length);
/** @copydoc spDVector::xdvrealloc()
 *  @public @memberof spFVector */
extern spFVector xfvrealloc(spFVector x, long length);
/** @~english Expands the size of the vector and resets the expanded elements to 0.
 *  @return The address of \p x for success, #SP_NODATA for failure.
 *  @public @memberof spDVector */
extern spDVector xdvrealloc(spDVector x, long length);

/** @~english @} */

/** @~english @name Lock and Unlock
 *  @{ */
 
/** @copydoc spDVector::dvisplugincomputable()
 *  @public @memberof spSVector */
extern spBool svisplugincomputable(spSVector x);
/** @copydoc spDVector::dvisplugincomputable()
 *  @public @memberof spLVector */
extern spBool lvisplugincomputable(spLVector x);
/** @copydoc spDVector::dvisplugincomputable()
 *  @public @memberof spFVector */
extern spBool fvisplugincomputable(spFVector x);
/** @~english Checks whether the vector is ready for plugin-based computation.
 *  @public @memberof spDVector */
extern spBool dvisplugincomputable(spDVector x);

/** @copydoc spDVector::dvislocked()
 *  @public @memberof spSVector */
extern spBool svislocked(spSVector x);
/** @copydoc spDVector::dvislocked()
 *  @public @memberof spLVector */
extern spBool lvislocked(spLVector x);
/** @copydoc spDVector::dvislocked()
 *  @public @memberof spFVector */
extern spBool fvislocked(spFVector x);
/** @~english Checks whether the vector is in the locked state.
 *  @public @memberof spDVector */
extern spBool dvislocked(spDVector x);

/** @copydoc spDVector::dvlock()
 *  @public @memberof spSVector */
extern spBool svlock(spSVector x);
/** @copydoc spDVector::dvunlock()
 *  @public @memberof spSVector */
extern spBool svunlock(spSVector x);
/** @copydoc spDVector::dvlock()
 *  @public @memberof spLVector */
extern spBool lvlock(spLVector x);
/** @copydoc spDVector::dvunlock()
 *  @public @memberof spLVector */
extern spBool lvunlock(spLVector x);
/** @copydoc spDVector::dvlock()
 *  @public @memberof spFVector */
extern spBool fvlock(spFVector x);
/** @copydoc spDVector::dvunlock()
 *  @public @memberof spFVector */
extern spBool fvunlock(spFVector x);
/** @~english Sets the vector in the locked state.
 *  @public @memberof spDVector */
extern spBool dvlock(spDVector x);
/** @~english Sets the vector in the unlocked state.
 *  @public @memberof spDVector */
extern spBool dvunlock(spDVector x);

/** @copydoc spDVector::dvsetlock()
 *  @public @memberof spSVector */
extern spBool svsetlock(spSVector x, spBool lock);
/** @copydoc spDVector::dvsetlock()
 *  @public @memberof spLVector */
extern spBool lvsetlock(spLVector x, spBool lock);
/** @copydoc spDVector::dvsetlock()
 *  @public @memberof spFVector */
extern spBool fvsetlock(spFVector x, spBool lock);
/** @~english Sets the vector in the locked/unlocked state without data synchronization.
 *  @public @memberof spDVector */
extern spBool dvsetlock(spDVector x, spBool lock);

/** @copydoc spDVector::dvsync()
 *  @public @memberof spSVector */
extern spBool svsync(spSVector x);
/** @copydoc spDVector::dvsync()
 *  @public @memberof spLVector */
extern spBool lvsync(spLVector x);
/** @copydoc spDVector::dvsync()
 *  @public @memberof spFVector */
extern spBool fvsync(spFVector x);
/** @~english Synchronizes the vector data (`->data` and `->imag`) with the internal data
 *  without changing the locked/unlocked state.
 *  @public @memberof spDVector */
extern spBool dvsync(spDVector x);

/** @~english @} */

/** @internal */
extern spSVectors xsvsalloc(long num);
extern spLVectors xlvsalloc(long num);
extern spFVectors xfvsalloc(long num);
extern spDVectors xdvsalloc(long num);
extern spSVectors xsvsrealloc(spSVectors xs, long num);
extern spLVectors xlvsrealloc(spLVectors xs, long num);
extern spFVectors xfvsrealloc(spFVectors xs, long num);
extern spDVectors xdvsrealloc(spDVectors xs, long num);
extern void xsvsfree(spSVectors xs);
extern void xlvsfree(spLVectors xs);
extern void xfvsfree(spFVectors xs);
extern void xdvsfree(spDVectors xs);

extern void svscopy(spSVectors ys, spSVectors xs);
extern void lvscopy(spLVectors ys, spLVectors xs);
extern void fvscopy(spFVectors ys, spFVectors xs);
extern void dvscopy(spDVectors ys, spDVectors xs);
extern spSVectors xsvsclone(spSVectors xs);
extern spLVectors xlvsclone(spLVectors xs);
extern spFVectors xfvsclone(spFVectors xs);
extern spDVectors xdvsclone(spDVectors xs);

extern void lvsshift(spLVectors xs, long shift);
extern void svsshift(spSVectors xs, long shift);
extern void fvsshift(spFVectors xs, long shift);
extern void dvsshift(spDVectors xs, long shift);
    
extern void svsmemshift(spSVectors xs, long shift);
extern void lvsmemshift(spLVectors xs, long shift);
extern void fvsmemshift(spFVectors xs, long shift);
extern void dvsmemshift(spDVectors xs, long shift);
    
/** @~english @name Complex Number Handling
 *  @{ */

/** @copydoc spDVector::xdvcplx()
 *  @public @memberof spSVector */
extern spSVector xsvcplx(spSVector xr, spSVector xi);
/** @copydoc spDVector::xdvcplx()
 *  @public @memberof spLVector */
extern spLVector xlvcplx(spLVector xr, spLVector xi);
/** @copydoc spDVector::xdvcplx()
 *  @public @memberof spFVector */
extern spFVector xfvcplx(spFVector xr, spFVector xi);
/** @~english Creates a new vector whose real and imaginary parts are \p xr and \p xi , respectively.
 *  @public @memberof spDVector */
extern spDVector xdvcplx(spDVector xr, spDVector xi);

/** @copydoc spDVector::dvreal()
 *  @public @memberof spSVector */
extern void svreal(spSVector x);
/** @copydoc spDVector::dvreal()
 *  @public @memberof spLVector */
extern void lvreal(spLVector x);
/** @copydoc spDVector::dvreal()
 *  @public @memberof spFVector */
extern void fvreal(spFVector x);
/** @~english Deallocates the imaginary part of \p x if it exists.
 *  @public @memberof spDVector */
extern void dvreal(spDVector x);

/** @copydoc spDVector::dvimag()
 *  @public @memberof spSVector */
extern void svimag(spSVector x);
/** @copydoc spDVector::dvimag()
 *  @public @memberof spLVector */
extern void lvimag(spLVector x);
/** @copydoc spDVector::dvimag()
 *  @public @memberof spFVector */
extern void fvimag(spFVector x);
/** @~english Sets the imaginary part of \p x as the real part.
 *  @public @memberof spDVector */
extern void dvimag(spDVector x);

/** @copydoc spDVector::xdvreal()
 *  @public @memberof spSVector */
extern spSVector xsvreal(spSVector x);
/** @copydoc spDVector::xdvreal()
 *  @public @memberof spLVector */
extern spLVector xlvreal(spLVector x);
/** @copydoc spDVector::xdvreal()
 *  @public @memberof spFVector */
extern spFVector xfvreal(spFVector x);
/** @~english Copies the real part of \p x to that of a new vector.
 *  @public @memberof spDVector */
extern spDVector xdvreal(spDVector x);

/** @copydoc spDVector::xdvimag()
 *  @public @memberof spSVector */
extern spSVector xsvimag(spSVector x);
/** @copydoc spDVector::xdvimag()
 *  @public @memberof spLVector */
extern spLVector xlvimag(spLVector x);
/** @copydoc spDVector::xdvimag()
 *  @public @memberof spFVector */
extern spFVector xfvimag(spFVector x);
/** @~english Copies the imaginary part of \p x to the real part of a new vector.
 *  @public @memberof spDVector */
extern spDVector xdvimag(spDVector x);

/** @copydoc spDVector::dvconj()
 *  @public @memberof spSVector */
extern void svconj(spSVector x);
/** @copydoc spDVector::dvconj()
 *  @public @memberof spLVector */
extern void lvconj(spLVector x);
/** @copydoc spDVector::dvconj()
 *  @public @memberof spFVector */
extern void fvconj(spFVector x);
/** @~english Takes the complex conjugate of \p x.
 *  @public @memberof spDVector */
extern void dvconj(spDVector x);
/** @copydoc spDVector::xdvconj()
 *  @public @memberof spSVector */
extern spSVector xsvconj(spSVector x);
/** @copydoc spDVector::xdvconj()
 *  @public @memberof spLVector */
extern spLVector xlvconj(spLVector x);
/** @copydoc spDVector::xdvconj()
 *  @public @memberof spFVector */
extern spFVector xfvconj(spFVector x);
/** @~english Creates a new vector which is the complex conjugate of \p x.
 *  @public @memberof spDVector */
extern spDVector xdvconj(spDVector x);

/** @copydoc spDVector::dvriswap()
 *  @public @memberof spSVector */
extern void svriswap(spSVector x);
/** @copydoc spDVector::dvriswap()
 *  @public @memberof spLVector */
extern void lvriswap(spLVector x);
/** @copydoc spDVector::dvriswap()
 *  @public @memberof spFVector */
extern void fvriswap(spFVector x);
/** @~english Swaps the real part of \p x for the imaginary part.
 *  @public @memberof spDVector */
extern void dvriswap(spDVector x);
/** @copydoc spDVector::xdvriswap()
 *  @public @memberof spSVector */
extern spSVector xsvriswap(spSVector x);
/** @copydoc spDVector::xdvriswap()
 *  @public @memberof spLVector */
extern spLVector xlvriswap(spLVector x);
/** @copydoc spDVector::xdvriswap()
 *  @public @memberof spFVector */
extern spFVector xfvriswap(spFVector x);
/** @~english Creates a new vector whose real and imaginary parts are the imaginary and real parts of \p x respectively.
 *  @public @memberof spDVector */
extern spDVector xdvriswap(spDVector x);

/** @~english @} */

/** @~english @name Data Copy
 *  @{ */
 
/** @copydoc spDVector::dvcopy()
 *  @public @memberof spSVector */
extern void svcopy(spSVector y, spSVector x);
/** @copydoc spDVector::dvcopy()
 *  @public @memberof spLVector */
extern void lvcopy(spLVector y, spLVector x);
/** @copydoc spDVector::dvcopy()
 *  @public @memberof spFVector */
extern void fvcopy(spFVector y, spFVector x);
/** @~english Copies data of \p x to \p y . The imaginary part of \p x is not copied unless `y->imag != NULL`.
 *  @public @memberof spDVector */
extern void dvcopy(spDVector y, spDVector x);

/** @copydoc spDVector::dvrcopyi()
 *  @public @memberof spSVector */
extern void svrcopyi(spSVector y, spSVector x);
/** @copydoc spDVector::dvrcopyi()
 *  @public @memberof spLVector */
extern void lvrcopyi(spLVector y, spLVector x);
/** @copydoc spDVector::dvrcopyi()
 *  @public @memberof spFVector */
extern void fvrcopyi(spFVector y, spFVector x);
/** @~english Copies data of the real part of \p x to the imaginary part of \p y .
 *  @public @memberof spDVector */
extern void dvrcopyi(spDVector y, spDVector x);
    
/** @copydoc spDVector::dvicopyr()
 *  @public @memberof spSVector */
extern void svicopyr(spSVector y, spSVector x);
/** @copydoc spDVector::dvicopyr()
 *  @public @memberof spLVector */
extern void lvicopyr(spLVector y, spLVector x);
/** @copydoc spDVector::dvicopyr()
 *  @public @memberof spFVector */
extern void fvicopyr(spFVector y, spFVector x);
/** @~english Copies data of the imaginary part of \p x to the real part of \p y .
 *  @public @memberof spDVector */
extern void dvicopyr(spDVector y, spDVector x);

/** @copydoc spDVector::dvrcopyr()
 *  @public @memberof spSVector */
extern void svrcopyr(spSVector y, spSVector x);
/** @copydoc spDVector::dvrcopyr()
 *  @public @memberof spLVector */
extern void lvrcopyr(spLVector y, spLVector x);
/** @copydoc spDVector::dvrcopyr()
 *  @public @memberof spFVector */
extern void fvrcopyr(spFVector y, spFVector x);
/** @~english Copies data of the real part of \p x to the real part of \p y .
 *  @public @memberof spDVector */
extern void dvrcopyr(spDVector y, spDVector x);

/** @copydoc spDVector::dvicopyi()
 *  @public @memberof spSVector */
extern void svicopyi(spSVector y, spSVector x);
/** @copydoc spDVector::dvicopyi()
 *  @public @memberof spLVector */
extern void lvicopyi(spLVector y, spLVector x);
/** @copydoc spDVector::dvicopyi()
 *  @public @memberof spFVector */
extern void fvicopyi(spFVector y, spFVector x);
/** @~english Copies data of the imaginary part of \p x to the imaginary part of \p y .
 *  @public @memberof spDVector */
extern void dvicopyi(spDVector y, spDVector x);

/** @copydoc spDVector::dvcopyfromarray()
 *  @public @memberof spSVector */
extern long svcopyfromarray(spSVector y, long y_offset, short *data, long length);
/** @copydoc spDVector::dvcopyfromarray()
 *  @public @memberof spLVector */
extern long lvcopyfromarray(spLVector y, long y_offset, long *data, long length);
/** @copydoc spDVector::dvcopyfromarray()
 *  @public @memberof spFVector */
extern long fvcopyfromarray(spFVector y, long y_offset, float *data, long length);
/** @~english Copies data of the array \p data to the target location \p y_offset of \p y .
 *  @public @memberof spDVector */
extern long dvcopyfromarray(spDVector y, long y_offset, double *data, long length);

/** @copydoc spDVector::dvcopyfromriarray()
 *  @public @memberof spSVector */
extern long svcopyfromriarray(spSVector y, long y_offset, short *data, short *imag, long length);
/** @copydoc spDVector::dvcopyfromriarray()
 *  @public @memberof spLVector */
extern long lvcopyfromriarray(spLVector y, long y_offset, long *data, long *imag, long length);
/** @copydoc spDVector::dvcopyfromriarray()
 *  @public @memberof spFVector */
extern long fvcopyfromriarray(spFVector y, long y_offset, float *data, float *imag, long length);
/** @~english Copies the complex array of the real part \p data and the imaginary part \p imag to the target location \p y_offset of \p y .
 *  @public @memberof spDVector */
extern long dvcopyfromriarray(spDVector y, long y_offset, double *data, double *imag, long length);

/** @copydoc spDVector::dvcopytoarray()
 *  @public @memberof spSVector */
extern long svcopytoarray(spSVector x, long x_offset, short *data, long length);
/** @copydoc spDVector::dvcopytoarray()
 *  @public @memberof spLVector */
extern long lvcopytoarray(spLVector x, long x_offset, long *data, long length);
/** @copydoc spDVector::dvcopytoarray()
 *  @public @memberof spFVector */
extern long fvcopytoarray(spFVector x, long x_offset, float *data, long length);
/** @~english Copies data extracted from \p x_offset of \p x to the array \p data .
 *  @public @memberof spDVector */
extern long dvcopytoarray(spDVector x, long x_offset, double *data, long length);

/** @copydoc spDVector::dvcopytoriarray()
 *  @public @memberof spSVector */
extern long svcopytoriarray(spSVector x, long x_offset, short *data, short *imag, long length);
/** @copydoc spDVector::dvcopytoriarray()
 *  @public @memberof spLVector */
extern long lvcopytoriarray(spLVector x, long x_offset, long *data, long *imag, long length);
/** @copydoc spDVector::dvcopytoriarray()
 *  @public @memberof spFVector */
extern long fvcopytoriarray(spFVector x, long x_offset, float *data, float *imag, long length);
/** @~english Copies data extracted from \p x_offset of \p x to the complex array of the real part \p data and the imaginary part \p imag .
 *  @public @memberof spDVector */
extern long dvcopytoriarray(spDVector x, long x_offset, double *data, double *imag, long length);

/** @copydoc spDVector::xdvtof()
 *  @public @memberof spSVector */
extern spLVector xsvtol(spSVector x);
/** @copydoc spDVector::xdvtof()
 *  @public @memberof spSVector */
extern spFVector xsvtof(spSVector x);
/** @copydoc spDVector::xdvtof()
 *  @public @memberof spSVector */
extern spDVector xsvtod(spSVector x);
/** @copydoc spDVector::xdvtof()
 *  @public @memberof spDVector */
extern spSVector xdvtos(spDVector x);
/** @copydoc spDVector::xdvtof()
 *  @public @memberof spDVector */
extern spLVector xdvtol(spDVector x);
/** @~english Copies data of \p x to a new vector with a different type.
 *  @public @memberof spDVector */
extern spFVector xdvtof(spDVector x);

/** @copydoc spDVector::xdvclone()
 *  @public @memberof spSVector */
extern spSVector xsvclone(spSVector x);
/** @copydoc spDVector::xdvclone()
 *  @public @memberof spLVector */
extern spLVector xlvclone(spLVector x);
/** @copydoc spDVector::xdvclone()
 *  @public @memberof spFVector */
extern spFVector xfvclone(spFVector x);
/** @~english Copies data of \p x to a new vector.
 *  @public @memberof spDVector */
extern spDVector xdvclone(spDVector x);

/** @copydoc spDVector::xdvcat()
 *  @public @memberof spSVector */
extern spSVector xsvcat(spSVector x, spSVector y);
/** @copydoc spDVector::xdvcat()
 *  @public @memberof spLVector */
extern spLVector xlvcat(spLVector x, spLVector y);
/** @copydoc spDVector::xdvcat()
 *  @public @memberof spFVector */
extern spFVector xfvcat(spFVector x, spFVector y);
/** @~english Appends data of \p y to a new vector which is a clone of \p x.
 *  @public @memberof spDVector */
extern spDVector xdvcat(spDVector x, spDVector y);
/** @copydoc spDVector::dvcat()
 *  @public @memberof spSVector */
extern void svcat(spSVector x, spSVector y);
/** @copydoc spDVector::dvcat()
 *  @public @memberof spLVector */
extern void lvcat(spLVector x, spLVector y);
/** @copydoc spDVector::dvcat()
 *  @public @memberof spFVector */
extern void fvcat(spFVector x, spFVector y);
/** @~english Appends data of \p y to \p x.
 *  @public @memberof spDVector */
extern void dvcat(spDVector x, spDVector y);

/** @~english @} */

/** @~english @name Data Initialization
 *  @{ */
 
extern long lvgetinitcount(long x_length, long m, long incr, long n);
extern long fvgetinitcount(long x_length, float m, float incr, float n);
extern long dvgetinitcount(long x_length, double m, double incr, double n);

/** @copydoc spDVector::dvinit()
 *  @public @memberof spSVector */
extern spBool svinit(spSVector x, long m, long incr, long n);
/** @copydoc spDVector::dvinit()
 *  @public @memberof spLVector */
extern spBool lvinit(spLVector x, long m, long incr, long n);
/** @copydoc spDVector::dvinit()
 *  @public @memberof spFVector */
extern spBool fvinit(spFVector x, float m, float incr, float n);
/** @~english Initialize the vector \p x so that the elements range from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spBool dvinit(spDVector x, double m, double incr, double n);

/** @copydoc spDVector::xdvinit()
 *  @public @memberof spSVector */
extern spSVector xsvinit(long m, long incr, long n);
/** @copydoc spDVector::xdvinit()
 *  @public @memberof spLVector */
extern spLVector xlvinit(long m, long incr, long n);
/** @copydoc spDVector::xdvinit()
 *  @public @memberof spFVector */
extern spFVector xfvinit(float m, float incr, float n);
/** @~english Creates a new vector ranging from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spDVector xdvinit(double m, double incr, double n);

/** @copydoc spDVector::xdvinitul()
 *  @public @memberof spSVector */
extern spSVector xsvinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag);
/** @copydoc spDVector::xdvinitul()
 *  @public @memberof spLVector */
extern spLVector xlvinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag);
/** @copydoc spDVector::xdvinitul()
 *  @public @memberof spFVector */
extern spFVector xfvinitul(spPlugin *plugin, float m, float incr, float n, spBool unlock_flag);
/** @~english Creates a new plugin-based vector ranging from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spDVector xdvinitul(spPlugin *plugin, double m, double incr, double n, spBool unlock_flag);
    
/** @copydoc spDVector::dviinit()
 *  @public @memberof spSVector */
extern spBool sviinit(spSVector x, long m, long incr, long n);
/** @copydoc spDVector::dviinit()
 *  @public @memberof spLVector */
extern spBool lviinit(spLVector x, long m, long incr, long n);
/** @copydoc spDVector::dviinit()
 *  @public @memberof spFVector */
extern spBool fviinit(spFVector x, float m, float incr, float n);
/** @~english Initialize the imaginary part of the vector \p x so that the elements range from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spBool dviinit(spDVector x, double m, double incr, double n);

/** @copydoc spDVector::dvriinit()
 *  @public @memberof spSVector */
extern spBool svriinit(spSVector x, long m, long incr, long n);
/** @copydoc spDVector::dvriinit()
 *  @public @memberof spLVector */
extern spBool lvriinit(spLVector x, long m, long incr, long n);
/** @copydoc spDVector::dvriinit()
 *  @public @memberof spFVector */
extern spBool fvriinit(spFVector x, float m, float incr, float n);
/** @~english Initialize the real and imaginary parts of the vector \p x so that the elements of both parts range from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spBool dvriinit(spDVector x, double m, double incr, double n);
    
/** @copydoc spDVector::xdvriinit()
 *  @public @memberof spSVector */
extern spSVector xsvriinit(long m, long incr, long n);
/** @copydoc spDVector::xdvriinit()
 *  @public @memberof spLVector */
extern spLVector xlvriinit(long m, long incr, long n);
/** @copydoc spDVector::xdvriinit()
 *  @public @memberof spFVector */
extern spFVector xfvriinit(float m, float incr, float n);
/** @~english Creates a new vector whose real and imaginary parts range from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spDVector xdvriinit(double m, double incr, double n);

/** @copydoc spDVector::xdvriinitul()
 *  @public @memberof spSVector */
extern spSVector xsvriinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag);
/** @copydoc spDVector::xdvriinitul()
 *  @public @memberof spLVector */
extern spLVector xlvriinitul(spPlugin *plugin, long m, long incr, long n, spBool unlock_flag);
/** @copydoc spDVector::xdvriinitul()
 *  @public @memberof spFVector */
extern spFVector xfvriinitul(spPlugin *plugin, float m, float incr, float n, spBool unlock_flag);
/** @~english Creates a new plugin-based vector whose real and imaginary parts range from \p m to \p n using \p incr as the increment between elements.
 *  @public @memberof spDVector */
extern spDVector xdvriinitul(spPlugin *plugin, double m, double incr, double n, spBool unlock_flag);
    
extern spBool svidxnums(spSVector x, spLVector idx, short value);
extern spBool sviidxnums(spSVector x, spLVector idx, short value);
extern spBool svriidxnums(spSVector x, spLVector idx, short r, short i);
extern spSVector xsvidxnums(spSVector x, spLVector idx, short value);
extern spSVector xsviidxnums(spSVector x, spLVector idx, short value);
extern spSVector xsvriidxnums(spSVector x, spLVector idx, short r, short i);
extern spBool lvidxnums(spLVector x, spLVector idx, long value);
extern spBool lviidxnums(spLVector x, spLVector idx, long value);
extern spBool lvriidxnums(spLVector x, spLVector idx, long r, long i);
extern spLVector xlvidxnums(spLVector x, spLVector idx, long value);
extern spLVector xlviidxnums(spLVector x, spLVector idx, long value);
extern spLVector xlvriidxnums(spLVector x, spLVector idx, long r, long i);
extern spBool fvidxnums(spFVector x, spLVector idx, float value);
extern spBool fviidxnums(spFVector x, spLVector idx, float value);
extern spBool fvriidxnums(spFVector x, spLVector idx, float r, float i);
extern spFVector xfvidxnums(spFVector x, spLVector idx, float value);
extern spFVector xfviidxnums(spFVector x, spLVector idx, float value);
extern spFVector xfvriidxnums(spFVector x, spLVector idx, float r, float i);
extern spBool dvidxnums(spDVector x, spLVector idx, double value);
extern spBool dviidxnums(spDVector x, spLVector idx, double value);
extern spBool dvriidxnums(spDVector x, spLVector idx, double r, double i);
extern spDVector xdvidxnums(spDVector x, spLVector idx, double value);
extern spDVector xdviidxnums(spDVector x, spLVector idx, double value);
extern spDVector xdvriidxnums(spDVector x, spLVector idx, double r, double i);
    
/** @~english @} */

/** @copydoc spDVector::dvnums()
 *  @relatesalso spSVector */
#define svnums(x, length, value) svinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvnums()
 *  @relatesalso spLVector */
#define lvnums(x, length, value) lvinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvnums()
 *  @relatesalso spFVector */
#define fvnums(x, length, value) fvinit(x, (float)(value), 0.0, (float)(length))
/** @~english Initialize the vector \p x so that the elements within \p length are all \p value .
 *  @relatesalso spDVector */
#define dvnums(x, length, value) dvinit(x, (double)(value), 0.0, (double)(length))

/** @copydoc spDVector::dvzeros()
 *  @relatesalso spSVector */
#define svzeros(x, length) svnums(x, length, 0)
/** @copydoc spDVector::dvzeros()
 *  @relatesalso spLVector */
#define lvzeros(x, length) lvnums(x, length, 0)
/** @copydoc spDVector::dvzeros()
 *  @relatesalso spFVector */
#define fvzeros(x, length) fvnums(x, length, 0.0)
/** @~english Initialize the vector \p x so that the elements within \p length are all zeros.
 *  @relatesalso spDVector */
#define dvzeros(x, length) dvnums(x, length, 0.0)

/** @copydoc spDVector::dvones()
 *  @relatesalso spSVector */
#define svones(x, length) svnums(x, length, 1)
/** @copydoc spDVector::dvones()
 *  @relatesalso spLVector */
#define lvones(x, length) lvnums(x, length, 1)
/** @copydoc spDVector::dvones()
 *  @relatesalso spFVector */
#define fvones(x, length) fvnums(x, length, 1.0)
/** @~english Initialize the vector \p x so that the elements within \p length are all ones.
 *  @relatesalso spDVector */
#define dvones(x, length) dvnums(x, length, 1.0)

/** @copydoc spDVector::dvinums()
 *  @relatesalso spSVector */
#define svinums(x, length, value) sviinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvinums()
 *  @relatesalso spLVector */
#define lvinums(x, length, value) lviinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvinums()
 *  @relatesalso spFVector */
#define fvinums(x, length, value) fviinit(x, (float)(value), 0.0, (float)(length))
/** @~english Initialize the imaginary part of the vector \p x so that the elements within \p length are all \p value .
 *  @relatesalso spDVector */
#define dvinums(x, length, value) dviinit(x, (double)(value), 0.0, (double)(length))

/** @copydoc spDVector::dvizeros()
 *  @relatesalso spSVector */
#define svizeros(x, length) svinums(x, length, 0.0)
/** @copydoc spDVector::dvizeros()
 *  @relatesalso spLVector */
#define lvizeros(x, length) lvinums(x, length, 0.0)
/** @copydoc spDVector::dvizeros()
 *  @relatesalso spFVector */
#define fvizeros(x, length) fvinums(x, length, 0.0)
/** @~english Initialize the imaginary part of the vector \p x so that the elements within \p length are all zeros.
 *  @relatesalso spDVector */
#define dvizeros(x, length) dvinums(x, length, 0.0)

/** @copydoc spDVector::dviones()
 *  @relatesalso spSVector */
#define sviones(x, length) svinums(x, length, 1.0)
/** @copydoc spDVector::dviones()
 *  @relatesalso spLVector */
#define lviones(x, length) lvinums(x, length, 1.0)
/** @copydoc spDVector::dviones()
 *  @relatesalso spFVector */
#define fviones(x, length) fvinums(x, length, 1.0)
/** @~english Initialize the imaginary part of the vector \p x so that the elements within \p length are all ones.
 *  @relatesalso spDVector */
#define dviones(x, length) dvinums(x, length, 1.0)
    
/** @copydoc spDVector::dvrinums()
 *  @relatesalso spSVector */
#define svrinums(x, length, value) svriinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvrinums()
 *  @relatesalso spLVector */
#define lvrinums(x, length, value) lvriinit(x, (long)(value), 0, (long)(length))
/** @copydoc spDVector::dvrinums()
 *  @relatesalso spFVector */
#define fvrinums(x, length, value) fvriinit(x, (float)(value), 0.0, (float)(length))
/** @~english Initialize the real and imaginary parts of the vector \p x so that the elements of both parts are all \p value .
 *  @relatesalso spDVector */
#define dvrinums(x, length, value) dvriinit(x, (double)(value), 0.0, (double)(length))

/** @copydoc spDVector::dvrizeros()
 *  @relatesalso spSVector */
#define svrizeros(x, length) svrinums(x, length, 0.0)
/** @copydoc spDVector::dvrizeros()
 *  @relatesalso spLVector */
#define lvrizeros(x, length) lvrinums(x, length, 0.0)
/** @copydoc spDVector::dvrizeros()
 *  @relatesalso spFVector */
#define fvrizeros(x, length) fvrinums(x, length, 0.0)
/** @~english Initialize the real and imaginary parts of the vector \p x so that the elements of both parts are all zeros.
 *  @relatesalso spDVector */
#define dvrizeros(x, length) dvrinums(x, length, 0.0)

/** @copydoc spDVector::dvriones()
 *  @relatesalso spSVector */
#define svriones(x, length) svrinums(x, length, 1.0)
/** @copydoc spDVector::dvriones()
 *  @relatesalso spLVector */
#define lvriones(x, length) lvrinums(x, length, 1.0)
/** @copydoc spDVector::dvriones()
 *  @relatesalso spFVector */
#define fvriones(x, length) fvrinums(x, length, 1.0)
/** @~english Initialize the real and imaginary parts of the vector \p x so that the elements of both parts are all ones.
 *  @relatesalso spDVector */
#define dvriones(x, length) dvrinums(x, length, 1.0)

/** @copydoc spDVector::xdvnums()
 *  @relatesalso spSVector */
#define xsvnums(length, value) xsvinit((long)(value), 0, (long)(length))
/** @copydoc spDVector::xdvnums()
 *  @relatesalso spLVector */
#define xlvnums(length, value) xlvinit((long)(value), 0, (long)(length))
/** @copydoc spDVector::xdvnums()
 *  @relatesalso spFVector */
#define xfvnums(length, value) xfvinit((float)(value), 0.0, (float)(length))
/** @~english Creates a new vector whose elements within \p length are all \p value .
 *  @relatesalso spDVector */
#define xdvnums(length, value) xdvinit((double)(value), 0.0, (double)(length))

/** @copydoc spDVector::xdvnumsul()
 *  @relatesalso spSVector */
#define xsvnumsul(plugin, length, value, unlock_flag) xsvinitul(plugin, (long)(value), 0, (long)(length), unlock_flag)
/** @copydoc spDVector::xdvnumsul()
 *  @relatesalso spLVector */
#define xlvnumsul(plugin, length, value, unlock_flag) xlvinitul(plugin, (long)(value), 0, (long)(length), unlock_flag)
/** @copydoc spDVector::xdvnumsul()
 *  @relatesalso spFVector */
#define xfvnumsul(plugin, length, value, unlock_flag) xfvinitul(plugin, (float)(value), 0.0, (float)(length), unlock_flag)
/** @~english Creates a new plugin-based vector whose elements within \p length are all \p value .
 *  @relatesalso spDVector */
#define xdvnumsul(plugin, length, value, unlock_flag) xdvinitul(plugin, (double)(value), 0.0, (double)(length), unlock_flag)

/** @copydoc spDVector::xdvzeros()
 *  @relatesalso spSVector */
#define xsvzeros(length) xsvnums(length, 0)
/** @copydoc spDVector::xdvzeros()
 *  @relatesalso spLVector */
#define xlvzeros(length) xlvnums(length, 0)
/** @copydoc spDVector::xdvzeros()
 *  @relatesalso spFVector */
#define xfvzeros(length) xfvnums(length, 0.0)
/** @~english Creates a new vector whose elements within \p length are all zeros.
 *  @relatesalso spDVector */
#define xdvzeros(length) xdvnums(length, 0.0)

/** @copydoc spDVector::xdvzerosul()
 *  @relatesalso spSVector */
#define xsvzerosul(plugin, length, unlock_flag) xsvnumsul(plugin, length, 0, unlock_flag)
/** @copydoc spDVector::xdvzerosul()
 *  @relatesalso spLVector */
#define xlvzerosul(plugin, length, unlock_flag) xlvnumsul(plugin, length, 0, unlock_flag)
/** @copydoc spDVector::xdvzerosul()
 *  @relatesalso spFVector */
#define xfvzerosul(plugin, length, unlock_flag) xfvnumsul(plugin, length, 0.0, unlock_flag)
/** @~english Creates a new plugin-based vector whose elements within \p length are all zeros.
 *  @relatesalso spDVector */
#define xdvzerosul(plugin, length, unlock_flag) xdvnumsul(plugin, length, 0.0, unlock_flag)
    
/** @copydoc spDVector::xdvones()
 *  @relatesalso spSVector */
#define xsvones(length) xsvnums(length, 1)
/** @copydoc spDVector::xdvones()
 *  @relatesalso spLVector */
#define xlvones(length) xlvnums(length, 1)
/** @copydoc spDVector::xdvones()
 *  @relatesalso spFVector */
#define xfvones(length) xfvnums(length, 1.0)
/** @~english Creates a new vector whose elements within \p length are all ones.
 *  @relatesalso spDVector */
#define xdvones(length) xdvnums(length, 1.0)

/** @copydoc spDVector::xdvonesul()
 *  @relatesalso spSVector */
#define xsvonesul(plugin, length, unlock_flag) xsvnumsul(plugin, length, 1, unlock_flag)
/** @copydoc spDVector::xdvonesul()
 *  @relatesalso spLVector */
#define xlvonesul(plugin, length, unlock_flag) xlvnumsul(plugin, length, 1, unlock_flag)
/** @copydoc spDVector::xdvonesul()
 *  @relatesalso spFVector */
#define xfvonesul(plugin, length, unlock_flag) xfvnumsul(plugin, length, 1.0, unlock_flag)
/** @~english Creates a new plugin-based vector whose elements within \p length are all ones.
 *  @relatesalso spDVector */
#define xdvonesul(plugin, length, unlock_flag) xdvnumsul(plugin, length, 1.0, unlock_flag)
    
/** @copydoc spDVector::xdvnull()
 *  @relatesalso spSVector */
#define xsvnull() xsvalloc(0)
/** @copydoc spDVector::xdvnull()
 *  @relatesalso spLVector */
#define xlvnull() xlvalloc(0)
/** @copydoc spDVector::xdvnull()
 *  @relatesalso spFVector */
#define xfvnull() xfvalloc(0)
/** @~english Creates a new null vector.
 *  @relatesalso spDVector */
#define xdvnull() xdvalloc(0)

/** @copydoc spDVector::xdvnullul()
 *  @relatesalso spSVector */
#define xsvnullul(plugin, unlock_flag) xsvallocul(plugin, 0, unlock_flag)
/** @copydoc spDVector::xdvnullul()
 *  @relatesalso spLVector */
#define xlvnullul(plugin, unlock_flag) xlvallocul(plugin, 0, unlock_flag)
/** @copydoc spDVector::xdvnullul()
 *  @relatesalso spFVector */
#define xfvnullul(plugin, unlock_flag) xfvallocul(plugin, 0, unlock_flag)
/** @~english Creates a new null plugin-based vector.
 *  @relatesalso spDVector */
#define xdvnullul(plugin, unlock_flag) xdvallocul(plugin, 0, unlock_flag)
    
/** @copydoc spDVector::xdvrinums()
 *  @relatesalso spSVector */
#define xsvrinums(length, value) xsvriinit((long)(value), 0, (long)(length))
/** @copydoc spDVector::xdvrinums()
 *  @relatesalso spLVector */
#define xlvrinums(length, value) xlvriinit((long)(value), 0, (long)(length))
/** @copydoc spDVector::xdvrinums()
 *  @relatesalso spFVector */
#define xfvrinums(length, value) xfvriinit((float)(value), 0.0, (float)(length))
/** @~english Creates a new vector whose real and imaginary parts are all \p value .
 *  @relatesalso spDVector */
#define xdvrinums(length, value) xdvriinit((double)(value), 0.0, (double)(length))

/** @copydoc spDVector::xdvrinumsul()
 *  @relatesalso spSVector */
#define xsvrinumsul(plugin, length, value, unlock_flag) xsvriinitul(plugin, (long)(value), 0, (long)(length), unlock_flag)
/** @copydoc spDVector::xdvrinumsul()
 *  @relatesalso spLVector */
#define xlvrinumsul(plugin, length, value, unlock_flag) xlvriinitul(plugin, (long)(value), 0, (long)(length), unlock_flag)
/** @copydoc spDVector::xdvrinumsul()
 *  @relatesalso spFVector */
#define xfvrinumsul(plugin, length, value, unlock_flag) xfvriinitul(plugin, (float)(value), 0.0, (float)(length), unlock_flag)
/** @~english Creates a new plugin-based vector whose real and imaginary parts are all \p value .
 *  @relatesalso spDVector */
#define xdvrinumsul(plugin, length, value, unlock_flag) xdvriinitul(plugin, (double)(value), 0.0, (double)(length), unlock_flag)
    
/** @copydoc spDVector::xdvrizeros()
 *  @relatesalso spSVector */
#define xsvrizeros(length) xsvrinums(length, 0)
/** @copydoc spDVector::xdvrizeros()
 *  @relatesalso spLVector */
#define xlvrizeros(length) xlvrinums(length, 0)
/** @copydoc spDVector::xdvrizeros()
 *  @relatesalso spFVector */
#define xfvrizeros(length) xfvrinums(length, 0.0)
/** @~english Creates a new vector whose real and imaginary parts are all zeros.
 *  @relatesalso spDVector */
#define xdvrizeros(length) xdvrinums(length, 0.0)

/** @copydoc spDVector::xdvrizerosul()
 *  @relatesalso spSVector */
#define xsvrizerosul(plugin, length, unlock_flag) xsvrinumsul(plugin, length, 0, unlock_flag)
/** @copydoc spDVector::xdvrizerosul()
 *  @relatesalso spLVector */
#define xlvrizerosul(plugin, length, unlock_flag) xlvrinumsul(plugin, length, 0, unlock_flag)
/** @copydoc spDVector::xdvrizerosul()
 *  @relatesalso spFVector */
#define xfvrizerosul(plugin, length, unlock_flag) xfvrinumsul(plugin, length, 0.0, unlock_flag)
/** @~english Creates a new plugin-based vector whose real and imaginary parts are all zeros.
 *  @relatesalso spDVector */
#define xdvrizerosul(plugin, length, unlock_flag) xdvrinumsul(plugin, length, 0.0, unlock_flag)
    
/** @copydoc spDVector::xdvriones()
 *  @relatesalso spSVector */
#define xsvriones(length) xsvrinums(length, 1)
/** @copydoc spDVector::xdvriones()
 *  @relatesalso spLVector */
#define xlvriones(length) xlvrinums(length, 1)
/** @copydoc spDVector::xdvriones()
 *  @relatesalso spFVector */
#define xfvriones(length) xfvrinums(length, 1.0)
/** @~english Creates a new vector whose real and imaginary parts are all ones.
 *  @relatesalso spDVector */
#define xdvriones(length) xdvrinums(length, 1.0)

/** @copydoc spDVector::xdvrionesul()
 *  @relatesalso spSVector */
#define xsvrionesul(plugin, length, unlock_flag) xsvrinumsul(plugin, length, 1, unlock_flag)
/** @copydoc spDVector::xdvrionesul()
 *  @relatesalso spLVector */
#define xlvrionesul(plugin, length, unlock_flag) xlvrinumsul(plugin, length, 1, unlock_flag)
/** @copydoc spDVector::xdvrionesul()
 *  @relatesalso spFVector */
#define xfvrionesul(plugin, length, unlock_flag) xfvrinumsul(plugin, length, 1.0, unlock_flag)
/** @~english Creates a new plugin-based vector whose real and imaginary parts are all ones.
 *  @relatesalso spDVector */
#define xdvrionesul(plugin, length, unlock_flag) xdvrinumsul(plugin, length, 1.0, unlock_flag)

/** @~english @name Data Editing
 *  @{ */
 
/** @copydoc spDVector::xdvcut()
 *  @public @memberof spSVector */
extern spSVector xsvcut(spSVector x, long offset, long length);
/** @copydoc spDVector::xdvcut()
 *  @public @memberof spLVector */
extern spLVector xlvcut(spLVector x, long offset, long length);
/** @copydoc spDVector::xdvcut()
 *  @public @memberof spFVector */
extern spFVector xfvcut(spFVector x, long offset, long length);
/** @~english Extracts data of \p length from \p offset of \p x .
 *  @public @memberof spDVector */
extern spDVector xdvcut(spDVector x, long offset, long length);

/** @copydoc spDVector::dvpaste()
 *  @public @memberof spSVector */
extern long svpaste(spSVector y, spSVector x, long y_offset, long x_length, spBool overlap);
/** @copydoc spDVector::dvpaste()
 *  @public @memberof spLVector */
extern long lvpaste(spLVector y, spLVector x, long y_offset, long x_length, spBool overlap);
/** @copydoc spDVector::dvpaste()
 *  @public @memberof spFVector */
extern long fvpaste(spFVector y, spFVector x, long y_offset, long x_length, spBool overlap);
/** @~english Pastes data of \p x cropped with \p x_length to the target location \p y_offset of \p y .
 *  The argument \p overlap determines whether the overlap addition is performed in pasting data.
 *  @public @memberof spDVector */
extern long dvpaste(spDVector y, spDVector x, long y_offset, long x_length, spBool overlap);

/** @copydoc spDVector::dvadd()
 *  @public @memberof spSVector */
extern long svadd(spSVector y, long y_offset, spSVector x, long x_offset, long x_length, spBool overlap);
/** @copydoc spDVector::dvadd()
 *  @public @memberof spLVector */
extern long lvadd(spLVector y, long y_offset, spLVector x, long x_offset, long x_length, spBool overlap);
/** @copydoc spDVector::dvadd()
 *  @public @memberof spFVector */
extern long fvadd(spFVector y, long y_offset, spFVector x, long x_offset, long x_length, spBool overlap);
/** @~english Pastes data of \p x_length extracted from \p x_offset of \p x to the target location \p y_offset of \p y .
 *  The argument \p overlap determines whether the overlap addition is performed in pasting data.
 *  @public @memberof spDVector */
extern long dvadd(spDVector y, long y_offset, spDVector x, long x_offset, long x_length, spBool overlap);
     
/** @copydoc spDVector::dverase()
 *  @public @memberof spSVector */
extern void sverase(spSVector x, long offset, long length, spBool inv);
/** @copydoc spDVector::dverase()
 *  @public @memberof spLVector */
extern void lverase(spLVector x, long offset, long length, spBool inv);
/** @copydoc spDVector::dverase()
 *  @public @memberof spFVector */
extern void fverase(spFVector x, long offset, long length, spBool inv);
/** @~english Erases data of \p length from \p offset .
 *  The argument \p inv determines whether the erasing location is inverted.
 *  @public @memberof spDVector */
extern void dverase(spDVector x, long offset, long length, spBool inv);
    
/** @~english @} */

/** @~english @name Data Modification
 *  @{ */
 
/** @copydoc spDVector::dvset()
 *  @public @memberof spSVector */
extern void svset(spSVector x, short *data, long length);
/** @copydoc spDVector::dvset()
 *  @public @memberof spLVector */
extern void lvset(spLVector x, long *data, long length);
/** @copydoc spDVector::dvset()
 *  @public @memberof spFVector */
extern void fvset(spFVector x, float *data, long length);
/** @~english Sets the address of \p data to \p x->data .
 *  @public @memberof spDVector */
extern void dvset(spDVector x, double *data, long length);

/** @copydoc spDVector::dviset()
 *  @public @memberof spSVector */
extern void sviset(spSVector x, short *imag, long length);
/** @copydoc spDVector::dviset()
 *  @public @memberof spLVector */
extern void lviset(spLVector x, long *imag, long length);
/** @copydoc spDVector::dviset()
 *  @public @memberof spFVector */
extern void fviset(spFVector x, float *imag, long length);
/** @~english Sets the address of \p imag to \p x->imag .
 *  @public @memberof spDVector */
extern void dviset(spDVector x, double *imag, long length);

/** @copydoc spDVector::xdvsetve()
 *  @public @memberof spSVector */
extern spSVector xsvsetve(spPlugin *plugin, short *data, long length);
/** @copydoc spDVector::xdvsetve()
 *  @public @memberof spLVector */
extern spLVector xlvsetve(spPlugin *plugin, long *data, long length);
/** @copydoc spDVector::xdvsetve()
 *  @public @memberof spFVector */
extern spFVector xfvsetve(spPlugin *plugin, float *data, long length);
/** @~english Sets the address of \p data to \p data of a plugin-based new vector.
 *  @public @memberof spDVector */
extern spDVector xdvsetve(spPlugin *plugin, double *data, long length);

/** @copydoc spDVector::xdvset()
 *  @public @memberof spSVector */
extern spSVector xsvset(short *data, long length);
/** @copydoc spDVector::xdvset()
 *  @public @memberof spLVector */
extern spLVector xlvset(long *data, long length);
/** @copydoc spDVector::xdvset()
 *  @public @memberof spFVector */
extern spFVector xfvset(float *data, long length);
/** @~english Sets the address of \p data to \p data of a new vector.
 *  @public @memberof spDVector */
extern spDVector xdvset(double *data, long length);

/** @copydoc spDVector::xdvsetnew()
 *  @public @memberof spSVector */
extern spSVector xsvsetnew(short *data, long length);
/** @copydoc spDVector::xdvsetnew()
 *  @public @memberof spLVector */
extern spLVector xlvsetnew(long *data, long length);
/** @copydoc spDVector::xdvsetnew()
 *  @public @memberof spFVector */
extern spFVector xfvsetnew(float *data, long length);
/** @~english Creates a new vector whose real part is a clone of \p data .
 *  @public @memberof spDVector */
extern spDVector xdvsetnew(double *data, long length);

/** @copydoc spDVector::dvsetval()
 *  @public @memberof spSVector */
extern spBool svsetval(spSVector x, long index, short value);
/** @copydoc spDVector::dvsetval()
 *  @public @memberof spLVector */
extern spBool lvsetval(spLVector x, long index, long value);
/** @copydoc spDVector::dvsetval()
 *  @public @memberof spFVector */
extern spBool fvsetval(spFVector x, long index, float value);
/** @~english Copies a value to the location of \p index of the real part of \p x .
 *  @public @memberof spDVector */
extern spBool dvsetval(spDVector x, long index, double value);
    
/** @copydoc spDVector::dvsetival()
 *  @public @memberof spSVector */
extern spBool svsetival(spSVector x, long index, short value);
/** @copydoc spDVector::dvsetival()
 *  @public @memberof spLVector */
extern spBool lvsetival(spLVector x, long index, long value);
/** @copydoc spDVector::dvsetival()
 *  @public @memberof spFVector */
extern spBool fvsetival(spFVector x, long index, float value);
/** @~english Copies a value to the location of \p index of the imaginary part of \p x .
 *  @public @memberof spDVector */
extern spBool dvsetival(spDVector x, long index, double value);
    
/** @~english @} */

/** @copydoc spDVector::dvunset()
 *  @relatesalso spSVector */
#define svunset(x) svset(x, NULL, 0)
/** @copydoc spDVector::dvunset()
 *  @relatesalso spLVector */
#define lvunset(x) lvset(x, NULL, 0)
/** @copydoc spDVector::dvunset()
 *  @relatesalso spFVector */
#define fvunset(x) fvset(x, NULL, 0)
/** @~english Sets \p x->data to NULL.
 *  @relatesalso spDVector */
#define dvunset(x) dvset(x, NULL, 0)

/** @copydoc spDVector::dviunset()
 *  @relatesalso spSVector */
#define sviunset(x) sviset(x, NULL, 0)
/** @copydoc spDVector::dviunset()
 *  @relatesalso spLVector */
#define lviunset(x) lviset(x, NULL, 0)
/** @copydoc spDVector::dviunset()
 *  @relatesalso spFVector */
#define fviunset(x) fviset(x, NULL, 0)
/** @~english Sets \p x->imag to NULL.
 *  @relatesalso spDVector */
#define dviunset(x) dviset(x, NULL, 0)

#if defined(MACOS)
#pragma import off
#endif

#define spvsset(xs, index, x) {(xs)->vector[index]=(x);}
#define svsset spvsset
#define lvsset spvsset
#define fvsset spvsset
#define dvsset spvsset

/* for backwards compatibility */
typedef spSVector SVECTOR;
typedef spLVector LVECTOR;
typedef spFVector FVECTOR;
typedef spDVector DVECTOR;

typedef spSVectors SVECTORS;
typedef spLVectors LVECTORS;
typedef spFVectors FVECTORS;
typedef spDVectors DVECTORS;

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_VECTOR_H */
