/*
 *	voperate.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifdef USE_SPL

#define nsp_UsesVector
#include <nsp.h>
#endif

#include <sp/sp.h>
#include <sp/base.h>
#include <sp/memory.h>
#include <sp/vector.h>
#include <sp/voperate.h>

#ifdef SP_USE_VECTOR_ENGINE
#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>
#endif

void svplus(spSVector x, spSVector y)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	svizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->plus != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->plus(x->instance, y->instance, MIN(x->length, y->length));
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    x->data[k] = x->data[k] + y->data[k];
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    x->imag[k] = x->imag[k] + y->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svminus(spSVector x, spSVector y, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->minus != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->minus(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = y->data[k] - x->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = y->imag[k] - x->imag[k];
		    } else {
			x->imag[k] = -x->imag[k];
		    }
		}
	    } else {
		x->data[k] = -x->data[k];
		if (x->imag != NULL) {
		    x->imag[k] = -x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = x->data[k] - y->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = x->imag[k] - y->imag[k];
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svtimes(spSVector x, spSVector y)
{
    long k;
    long xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->times != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->times(x->instance, y->instance, MIN(x->length, y->length));
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    xr = x->data[k] * y->data[k] - x->imag[k] * y->imag[k];
		    xi = x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
		    x->data[k] = (short)xr;
		    x->imag[k] = (short)xi;
		} else {
		    x->data[k] = x->data[k] * y->data[k];
		    x->imag[k] = x->imag[k] * y->data[k];
		}
	    } else {
		x->data[k] = x->data[k] * y->data[k];
	    }
	} else {
	    x->data[k] = 0;
	    if (x->imag != NULL) {
		x->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svdivide(spSVector x, spSVector y, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->divide != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->divide(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL) {
		    if (x->data[k] == 0 && x->imag[k] == 0) {
			spwarning("warning: svoper: divide by zero\n");
			    
			if (y->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = y->data[k] / x->data[k];
			}
			if (y->imag != NULL) {
			    if (y->imag[k] == 0) {
				x->imag[k] = 0;
			    } else {
				x->imag[k] = y->imag[k] / x->imag[k];
			    }
			} else {
			    x->imag[k] = 0;
			}
		    } else {
			a = CSQUARE((double)x->data[k], (double)x->imag[k]);
			if (y->imag != NULL) {
			    xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			    xi = x->data[k] * y->imag[k] - x->imag[k] * y->data[k];
			    x->data[k] = (short)spRound(xr / a);
			    x->imag[k] = (short)spRound(xi / a);
			} else {
			    x->data[k] = (short)spRound((double)x->data[k] * (double)y->data[k] / a);
			    x->imag[k] = (short)spRound((double)-x->imag[k] * (double)y->data[k] / a);
			}
		    }
		} else {
		    if (x->data[k] != 0) {
			x->data[k] = y->data[k] / x->data[k];
		    } else {
			spwarning("warning: svoper: divide by zero\n");

			if (y->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = y->data[k] / x->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0;
		if (x->imag != NULL) {
		    x->imag[k] = 0;
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL && y->imag != NULL) {
		    if (y->data[k] == 0 && y->imag[k] == 0) {
			spwarning("warning: svoper: divide by zero\n");

			if (x->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = x->data[k] / y->data[k];
			}
			if (x->imag[k] == 0) {
			    x->imag[k] = 0;
			} else {
			    x->imag[k] = x->imag[k] / y->imag[k];
			}
		    } else {
			a = CSQUARE((double)y->data[k], (double)y->imag[k]);
			xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			xi = -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
			x->data[k] = (short)spRound(xr / a);
			x->imag[k] = (short)spRound(xi / a);
		    }
		} else {
		    if (y->data[k] == 0) {
			spwarning("warning: svoper: divide by zero\n");

			if (x->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = x->data[k] / y->data[k];
			}
			if (x->imag != NULL) {
			    if (x->imag[k] == 0) {
				x->imag[k] = 0;
			    } else {
				x->imag[k] = x->imag[k] / y->imag[k];
			    }
			}
		    } else {
			x->data[k] = x->data[k] / y->data[k];
			if (x->imag != NULL) {
			    x->imag[k] = x->imag[k] / y->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0;
		if (x->imag != NULL) {
		    x->imag[k] = 0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svpower(spSVector x, spSVector y, spBool reverse)
{
    long k;
    double xr, xi;
    double xr1, xi1;
    double yr, yi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->power != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->power(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    if (reverse) {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (y->imag == NULL) {
			yr = y->data[k];
			yi = 0.0;
		    } else {
			yr = y->data[k];
			yi = y->imag[k];
		    }
		    if (yr == 0.0 && yi == 0.0) {
			x->data[k] = 0;
			x->imag[k] = 0;
		    } else if (x->imag[k] == 0 && yi == 0.0) {
			x->data[k] = (short)spRound(pow((double)y->data[k], 
							(double)x->data[k]));
		    } else {
			clog(&yr, &yi);
			xr = x->data[k] * yr - x->imag[k] * yi;
			xi = x->data[k] * yi + x->imag[k] * yr;
			cexp(&xr, &xi);
			x->data[k] = (short)spRound(xr);
			x->imag[k] = (short)spRound(xi);
		    }
		} else {
		    x->data[k] = 0;
		    x->imag[k] = 0;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = (short)spRound(pow((double)y->data[k], 
						    (double)x->data[k]));
		} else {
		    x->data[k] = 0;
		}
	    }
	}
    } else {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (x->data[k] == 0 && x->imag[k] == 0) {
			x->data[k] = 0;
			x->imag[k] = 0;
		    } else {
			if (y->imag == NULL) {
			    yr = y->data[k];
			    yi = 0.0;
			} else {
			    yr = y->data[k];
			    yi = y->imag[k];
			}
			if (x->imag[k] == 0 && yi == 0.0) {
			    x->data[k] = (short)spRound(pow((double)x->data[k],
							    (double)y->data[k]));
			} else {
			    xr1 = x->data[k];
			    xi1 = x->imag[k];
			    clog(&xr1, &xi1);
			    xr = xr1 * yr - xi1 * yi;
			    xi = xr1 * yi + xi1 * yr;
			    cexp(&xr, &xi);
			    x->data[k] = (short)spRound(xr);
			    x->imag[k] = (short)spRound(xi);
			}
		    }
		} else {
		    x->data[k] = 1;
		    x->imag[k] = 1;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = (short)spRound(pow((double)x->data[k], 
						    (double)y->data[k]));
		} else {
		    x->data[k] = 1;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svoper(spSVector x, const char *op, spSVector y)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	svplus(x, y);
    } else if (strveq(op2, "-")) {
	svminus(x, y, reverse);
    } else if (strveq(op2, "*")) {
	svtimes(x, y);
    } else if (strveq(op2, "/")) {
	svdivide(x, y, reverse);
    } else if (strveq(op2, "^")) {
	svpower(x, y, reverse);
    } else {
	fprintf(stderr, "svoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spSVector xsvoper(spSVector a, const char *op, spSVector b)
{
    spSVector c;

    c = xsvclone(a);
    svoper(c, op, b);

    return c;
}


void svscplus(spSVector x, double t)
{
    long k;
    short ts;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scplus != NULL) {
	flist->scplus(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    ts = (short)spRound(t);
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + ts;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svscminus(spSVector x, double t, spBool reverse)
{
    long k;
    short ts;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scminus != NULL) {
	flist->scminus(x->instance, x->length, t, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    ts = (short)spRound(t);
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = ts - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - ts;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svsctimes(spSVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->sctimes != NULL) {
	flist->sctimes(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = (short)spRound((double)x->data[k] * t);
	if (x->imag != NULL) {
	    x->imag[k] = (short)spRound((double)x->imag[k] * t);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svscdivide(spSVector x, double t, spBool reverse)
{
    long k;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scdivide != NULL) {
	flist->scdivide(x->instance, x->length, t, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0 && x->imag[k] == 0) {
		    spwarning("warning: svscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = (short)t / x->data[k];
		    }
		    x->imag[k] = 0;
		} else {
		    a = CSQUARE((double)x->data[k], (double)x->imag[k]);
		    x->data[k] = (short)spRound((double)x->data[k] * t / a);
		    x->imag[k] = (short)spRound((double)-x->imag[k] * t / a);
		}
	    } else {
		if (x->data[k] != 0) {
		    x->data[k] = (short)spRound(t / (double)x->data[k]);
		} else {
		    spwarning("warning: svscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = (short)t / x->data[k];
		    }
		}
	    }
	} else {
	    if (t != 0.0) {
		x->data[k] = (short)spRound((double)x->data[k] / t);
		if (x->imag != NULL) {
		    x->imag[k] = (short)spRound((double)x->imag[k] / t);
		}
	    } else {
		spwarning("warning: svscoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / (short)t;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0) {
			x->imag[k] = 0;
		    } else {
			x->imag[k] = x->imag[k] / (short)t;
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svscpower(spSVector x, double t, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scpower != NULL) {
	flist->scpower(x->instance, x->length, t, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0) {
		a = log(t);
		xr = (double)x->data[k] * a;
		xi = (double)x->imag[k] * a;
		cexp(&xr, &xi);
		x->data[k] = (short)spRound(xr);
		x->imag[k] = (short)spRound(xi);
	    } else {
		x->data[k] = (short)spRound(pow(t, (double)x->data[k]));
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0) {
		xr = x->data[k];
		xi = x->imag[k];
		clog(&xr, &xi);
		xr *= t;
		xi *= t;
		cexp(&xr, &xi);
		x->data[k] = (short)spRound(xr);
		x->imag[k] = (short)spRound(xi);
	    } else {
		x->data[k] = (short)spRound(pow((double)x->data[k], t));
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void svscoper(spSVector x, const char *op, double t)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	svscplus(x, t);
    } else if (strveq(op2, "-")) {
	svscminus(x, t, reverse);
    } else if (strveq(op2, "*")) {
	svsctimes(x, t);
    } else if (strveq(op2, "/")) {
	svscdivide(x, t, reverse);
    } else if (strveq(op2, "^")) {
	svscpower(x, t, reverse);
    } else {
	fprintf(stderr, "svscoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spSVector xsvscoper(spSVector a, const char *op, double t)
{
    spSVector c;

    c = xsvclone(a);
    svscoper(c, op, t);

    return c;
}

void svcplxplus(spSVector x, double tr, double ti)
{
    long k;
    short trs, tis;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxplus != NULL) {
	flist->cplxplus(x->instance, x->length, tr, ti);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    trs = (short)spRound(tr);
    tis = (short)spRound(ti);
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + trs;
	x->imag[k] = x->imag[k] + tis;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svcplxminus(spSVector x, double tr, double ti, spBool reverse)
{
    long k;
    short trs, tis;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxminus != NULL) {
	flist->cplxminus(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    trs = (short)spRound(tr);
    tis = (short)spRound(ti);
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = trs - x->data[k];
            x->imag[k] = tis - x->imag[k];
	} else {
	    x->data[k] = x->data[k] - trs;
	    x->imag[k] = x->imag[k] - tis;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svcplxtimes(spSVector x, double tr, double ti)
{
    long k;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxtimes != NULL) {
	flist->cplxtimes(x->instance, x->length, tr, ti);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	xr = (double)x->data[k];
        xi = (double)x->imag[k];
	x->data[k] = (short)spRound(xr * tr - xi * ti);
        x->imag[k] = (short)spRound(xr * ti + xi * tr);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svcplxdivide(spSVector x, double tr, double ti, spBool reverse)
{
    long k;
    double xr, xi;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxdivide != NULL) {
	flist->cplxdivide(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (x->data[k] == 0 && x->imag[k] == 0) {
                spwarning("warning: svcplxoper: divide by zero\n");

                if (tr == 0.0) {
                    x->data[k] = 0;
                } else {
                    x->data[k] = (short)tr / x->data[k];
                }
                if (ti == 0.0) {
                    x->imag[k] = 0;
                } else {
                    x->imag[k] = (short)ti / x->imag[k];
                }
            } else {
                a = CSQUARE((double)x->data[k], (double)x->imag[k]);
                xr = (double)x->data[k];
                xi = (double)x->imag[k];
                x->data[k] = (short)spRound((xr * tr + xi * ti) / a);
                x->imag[k] = (short)spRound((xr * ti - xi * tr) / a);
            }
	} else {
	    if (tr == 0.0 && ti == 0.0) {
		spwarning("warning: svcplxoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / (short)tr;
		}
                if (x->imag[k] == 0) {
                    x->imag[k] = 0;
                } else {
                    x->imag[k] = x->imag[k] / (short)tr;
                }
	    } else {
                a = CSQUARE(tr, ti);
                xr = (double)x->data[k];
                xi = (double)x->imag[k];
		x->data[k] = (short)spRound((tr * xr + ti * xi) / a);
                x->imag[k] = (short)spRound((tr * xi - ti * xr) / a);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void svcplxpower(spSVector x, double tr, double ti, spBool reverse)
{
    long k;
    double a;
    double xr0, xi0;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	svizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->cplxpower != NULL) {
	flist->cplxpower(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (tr == 0.0 && ti == 0.0) {
                x->data[k] = 0;
                x->imag[k] = 0;
            } else if (x->imag[k] == 0 && ti == 0.0) {
                x->data[k] = (short)spRound(pow(tr, (double)x->data[k]));
            } else {
                clog(&tr, &ti);
                xr = (double)x->data[k] * tr - (double)x->imag[k] * ti;
                xi = (double)x->data[k] * ti + (double)x->imag[k] * tr;
                cexp(&xr, &xi);
                x->data[k] = (short)spRound(xr);
                x->imag[k] = (short)spRound(xi);
            }
	} else {
            if (x->data[k] == 0 && x->imag[k] == 0) {
                x->data[k] = 0;
                x->imag[k] = 0;
            } else {
                if (x->imag[k] == 0 && ti == 0.0) {
                    x->data[k] = (short)spRound(pow((double)x->data[k], tr));
                } else {
                    xr0 = (double)x->data[k];
                    xi0 = (double)x->imag[k];
                    clog(&xr0, &xi0);
                    xr = xr0 * tr - xi0 * ti;
                    xi = xr0 * ti + xi0 * tr;
                    cexp(&xr, &xi);
                    x->data[k] = (short)spRound(xr);
                    x->imag[k] = (short)spRound(xi);
                }
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svcplxoper(spSVector x, const char *op, double tr, double ti)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	svcplxplus(x, tr, ti);
    } else if (strveq(op2, "-")) {
	svcplxminus(x, tr, ti, reverse);
    } else if (strveq(op2, "*")) {
	svcplxtimes(x, tr, ti);
    } else if (strveq(op2, "/")) {
	svcplxdivide(x, tr, ti, reverse);
    } else if (strveq(op2, "^")) {
	svcplxpower(x, tr, ti, reverse);
    } else {
	fprintf(stderr, "svcplxoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spSVector xsvcplxoper(spSVector a, const char *op, double tr, double ti)
{
    spSVector c;

    c = xsvclone(a);
    svcplxoper(c, op, tr, ti);

    return c;
}

void lvplus(spLVector x, spLVector y)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	lvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->plus != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->plus(x->instance, y->instance, MIN(x->length, y->length));
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    x->data[k] = x->data[k] + y->data[k];
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    x->imag[k] = x->imag[k] + y->imag[k];
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvminus(spLVector x, spLVector y, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->minus != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->minus(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = y->data[k] - x->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = y->imag[k] - x->imag[k];
		    } else {
			x->imag[k] = -x->imag[k];
		    }
		}
	    } else {
		x->data[k] = -x->data[k];
		if (x->imag != NULL) {
		    x->imag[k] = -x->imag[k];
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = x->data[k] - y->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = x->imag[k] - y->imag[k];
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvtimes(spLVector x, spLVector y)
{
    long k;
    long xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->times != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->times(x->instance, y->instance, MIN(x->length, y->length));
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    xr = x->data[k] * y->data[k] - x->imag[k] * y->imag[k];
		    xi = x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
		    x->data[k] = (long)xr;
		    x->imag[k] = (long)xi;
		} else {
		    x->data[k] = x->data[k] * y->data[k];
		    x->imag[k] = x->imag[k] * y->data[k];
		}
	    } else {
		x->data[k] = x->data[k] * y->data[k];
	    }
	} else {
	    x->data[k] = 0;
	    if (x->imag != NULL) {
		x->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvdivide(spLVector x, spLVector y, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->divide != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->divide(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL) {
		    if (x->data[k] == 0 && x->imag[k] == 0) {
			spwarning("warning: lvoper: divide by zero\n");
			    
			if (y->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = y->data[k] / x->data[k];
			}
			if (y->imag != NULL) {
			    if (y->imag[k] == 0) {
				x->imag[k] = 0;
			    } else {
				x->imag[k] = y->imag[k] / x->imag[k];
			    }
			} else {
			    x->imag[k] = 0;
			}
		    } else {
			a = (double)CSQUARE(x->data[k], x->imag[k]);
			if (y->imag != NULL) {
			    xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			    xi = x->data[k] * y->imag[k] - x->imag[k] * y->data[k];
			    x->data[k] = (long)spRound(xr / a);
			    x->imag[k] = (long)spRound(xi / a);
			} else {
			    x->data[k] = (long)spRound((double)x->data[k] * (double)y->data[k] / a);
			    x->imag[k] = (long)spRound((double)-x->imag[k] * (double)y->data[k] / a);
			}
		    }
		} else {
		    if (x->data[k] != 0) {
			x->data[k] = y->data[k] / x->data[k];
		    } else {
			spwarning("warning: lvoper: divide by zero\n");

			if (y->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = y->data[k] / x->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0;
		if (x->imag != NULL) {
		    x->imag[k] = 0;
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL && y->imag != NULL) {
		    if (y->data[k] == 0 && y->imag[k] == 0) {
			spwarning("warning: lvoper: divide by zero\n");

			if (x->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = x->data[k] / y->data[k];
			}
			if (x->imag[k] == 0) {
			    x->imag[k] = 0;
			} else {
			    x->imag[k] = x->imag[k] / y->imag[k];
			}
		    } else {
			a = (double)CSQUARE(y->data[k], y->imag[k]);
			xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			xi = -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
			x->data[k] = (long)spRound(xr / a);
			x->imag[k] = (long)spRound(xi / a);
		    }
		} else {
		    if (y->data[k] == 0) {
			spwarning("warning: lvoper: divide by zero\n");

			if (x->data[k] == 0) {
			    x->data[k] = 0;
			} else {
			    x->data[k] = x->data[k] / y->data[k];
			}
			if (x->imag != NULL) {
			    if (x->imag[k] == 0) {
				x->imag[k] = 0;
			    } else {
				x->imag[k] = x->imag[k] / y->imag[k];
			    }
			}
		    } else {
			x->data[k] = x->data[k] / y->data[k];
			if (x->imag != NULL) {
			    x->imag[k] = x->imag[k] / y->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0;
		if (x->imag != NULL) {
		    x->imag[k] = 0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvpower(spLVector x, spLVector y, spBool reverse)
{
    long k;
    double xr, xi;
    double xr1, xi1;
    double yr, yi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->power != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->power(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    if (reverse) {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (y->imag == NULL) {
			yr = y->data[k];
			yi = 0.0;
		    } else {
			yr = y->data[k];
			yi = y->imag[k];
		    }
		    if (yr == 0.0 && yi == 0.0) {
			x->data[k] = 0;
			x->imag[k] = 0;
		    } else if (x->imag[k] == 0 && yi == 0.0) {
			x->data[k] = (long)spRound(pow((double)y->data[k], 
							(double)x->data[k]));
		    } else {
			clog(&yr, &yi);
			xr = x->data[k] * yr - x->imag[k] * yi;
			xi = x->data[k] * yi + x->imag[k] * yr;
			cexp(&xr, &xi);
			x->data[k] = (long)spRound(xr);
			x->imag[k] = (long)spRound(xi);
		    }
		} else {
		    x->data[k] = 0;
		    x->imag[k] = 0;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = (long)spRound(pow((double)y->data[k], 
						    (double)x->data[k]));
		} else {
		    x->data[k] = 0;
		}
	    }
	}
    } else {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (x->data[k] == 0 && x->imag[k] == 0) {
			x->data[k] = 0;
			x->imag[k] = 0;
		    } else {
			if (y->imag == NULL) {
			    yr = y->data[k];
			    yi = 0.0;
			} else {
			    yr = y->data[k];
			    yi = y->imag[k];
			}
			if (x->imag[k] == 0 && yi == 0.0) {
			    x->data[k] = (long)spRound(pow((double)x->data[k],
							    (double)y->data[k]));
			} else {
			    xr1 = x->data[k];
			    xi1 = x->imag[k];
			    clog(&xr1, &xi1);
			    xr = xr1 * yr - xi1 * yi;
			    xi = xr1 * yi + xi1 * yr;
			    cexp(&xr, &xi);
			    x->data[k] = (long)spRound(xr);
			    x->imag[k] = (long)spRound(xi);
			}
		    }
		} else {
		    x->data[k] = 1;
		    x->imag[k] = 1;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = (long)spRound(pow((double)x->data[k], 
						    (double)y->data[k]));
		} else {
		    x->data[k] = 1;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvoper(spLVector x, const char *op, spLVector y)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	lvplus(x, y);
    } else if (strveq(op2, "-")) {
	lvminus(x, y, reverse);
    } else if (strveq(op2, "*")) {
	lvtimes(x, y);
    } else if (strveq(op2, "/")) {
	lvdivide(x, y, reverse);
    } else if (strveq(op2, "^")) {
	lvpower(x, y, reverse);
    } else {
	fprintf(stderr, "lvoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spLVector xlvoper(spLVector a, const char *op, spLVector b)
{
    spLVector c;

    c = xlvclone(a);
    lvoper(c, op, b);

    return c;
}


void lvscplus(spLVector x, double t)
{
    long k;
    long ts;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scplus != NULL) {
	flist->scplus(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    ts = (long)spRound(t);
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + ts;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvscminus(spLVector x, double t, spBool reverse)
{
    long k;
    long ts;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scminus != NULL) {
	flist->scminus(x->instance, x->length, t, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    ts = (long)spRound(t);
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = ts - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - ts;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvsctimes(spLVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->sctimes != NULL) {
	flist->sctimes(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	x->data[k] = (long)spRound((double)x->data[k] * t);
	if (x->imag != NULL) {
	    x->imag[k] = (long)spRound((double)x->imag[k] * t);
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvscdivide(spLVector x, double t, spBool reverse)
{
    long k;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scdivide != NULL) {
	flist->scdivide(x->instance, x->length, t, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0 && x->imag[k] == 0) {
		    spwarning("warning: lvscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = (long)t / x->data[k];
		    }
		    x->imag[k] = 0;
		} else {
		    a = (double)CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = (long)spRound((double)x->data[k] * t / a);
		    x->imag[k] = (long)spRound((double)-x->imag[k] * t / a);
		}
	    } else {
		if (x->data[k] != 0) {
		    x->data[k] = (long)spRound(t / (double)x->data[k]);
		} else {
		    spwarning("warning: lvscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = (long)t / x->data[k];
		    }
		}
	    }
	} else {
	    if (t != 0.0) {
		x->data[k] = (long)spRound((double)x->data[k] / t);
		if (x->imag != NULL) {
		    x->imag[k] = (long)spRound((double)x->imag[k] / t);
		}
	    } else {
		spwarning("warning: lvscoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / (long)t;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0) {
			x->imag[k] = 0;
		    } else {
			x->imag[k] = x->imag[k] / (long)t;
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvscpower(spLVector x, double t, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scpower != NULL) {
	flist->scpower(x->instance, x->length, t, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0) {
		a = log(t);
		xr = (double)x->data[k] * a;
		xi = (double)x->imag[k] * a;
		cexp(&xr, &xi);
		x->data[k] = (long)spRound(xr);
		x->imag[k] = (long)spRound(xi);
	    } else {
		x->data[k] = (long)spRound(pow(t, (double)x->data[k]));
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0) {
		xr = x->data[k];
		xi = x->imag[k];
		clog(&xr, &xi);
		xr *= t;
		xi *= t;
		cexp(&xr, &xi);
		x->data[k] = (long)spRound(xr);
		x->imag[k] = (long)spRound(xi);
	    } else {
		x->data[k] = (long)spRound(pow((double)x->data[k], t));
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void lvscoper(spLVector x, const char *op, double t)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	lvscplus(x, t);
    } else if (strveq(op2, "-")) {
	lvscminus(x, t, reverse);
    } else if (strveq(op2, "*")) {
	lvsctimes(x, t);
    } else if (strveq(op2, "/")) {
	lvscdivide(x, t, reverse);
    } else if (strveq(op2, "^")) {
	lvscpower(x, t, reverse);
    } else {
	fprintf(stderr, "lvscoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spLVector xlvscoper(spLVector a, const char *op, double t)
{
    spLVector c;

    c = xlvclone(a);
    lvscoper(c, op, t);

    return c;
}

void lvcplxplus(spLVector x, double tr, double ti)
{
    long k;
    long trs, tis;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxplus != NULL) {
	flist->cplxplus(x->instance, x->length, tr, ti);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    trs = (long)spRound(tr);
    tis = (long)spRound(ti);
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + trs;
	x->imag[k] = x->imag[k] + tis;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvcplxminus(spLVector x, double tr, double ti, spBool reverse)
{
    long k;
    long trs, tis;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxminus != NULL) {
	flist->cplxminus(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    trs = (long)spRound(tr);
    tis = (long)spRound(ti);
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = trs - x->data[k];
            x->imag[k] = tis - x->imag[k];
	} else {
	    x->data[k] = x->data[k] - trs;
	    x->imag[k] = x->imag[k] - tis;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvcplxtimes(spLVector x, double tr, double ti)
{
    long k;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxtimes != NULL) {
	flist->cplxtimes(x->instance, x->length, tr, ti);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	xr = (double)x->data[k];
        xi = (double)x->imag[k];
	x->data[k] = (long)spRound(xr * tr - xi * ti);
        x->imag[k] = (long)spRound(xr * ti + xi * tr);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvcplxdivide(spLVector x, double tr, double ti, spBool reverse)
{
    long k;
    double xr, xi;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxdivide != NULL) {
	flist->cplxdivide(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (x->data[k] == 0 && x->imag[k] == 0) {
                spwarning("warning: lvcplxoper: divide by zero\n");

                if (tr == 0.0) {
                    x->data[k] = 0;
                } else {
                    x->data[k] = (long)tr / x->data[k];
                }
                if (ti == 0.0) {
                    x->imag[k] = 0;
                } else {
                    x->imag[k] = (long)ti / x->imag[k];
                }
            } else {
                a = (double)CSQUARE(x->data[k], x->imag[k]);
                xr = (double)x->data[k];
                xi = (double)x->imag[k];
                x->data[k] = (long)spRound((xr * tr + xi * ti) / a);
                x->imag[k] = (long)spRound((xr * ti - xi * tr) / a);
            }
	} else {
	    if (tr == 0.0 && ti == 0.0) {
		spwarning("warning: lvcplxoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / (long)tr;
		}
                if (x->imag[k] == 0) {
                    x->imag[k] = 0;
                } else {
                    x->imag[k] = x->imag[k] / (long)tr;
                }
	    } else {
                a = CSQUARE(tr, ti);
                xr = (double)x->data[k];
                xi = (double)x->imag[k];
		x->data[k] = (long)spRound((tr * xr + ti * xi) / a);
                x->imag[k] = (long)spRound((tr * xi - ti * xr) / a);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void lvcplxpower(spLVector x, double tr, double ti, spBool reverse)
{
    long k;
    double a;
    double xr0, xi0;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	lvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->cplxpower != NULL) {
	flist->cplxpower(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (tr == 0.0 && ti == 0.0) {
                x->data[k] = 0;
                x->imag[k] = 0;
            } else if (x->imag[k] == 0 && ti == 0.0) {
                x->data[k] = (long)spRound(pow(tr, (double)x->data[k]));
            } else {
                clog(&tr, &ti);
                xr = (double)x->data[k] * tr - (double)x->imag[k] * ti;
                xi = (double)x->data[k] * ti + (double)x->imag[k] * tr;
                cexp(&xr, &xi);
                x->data[k] = (long)spRound(xr);
                x->imag[k] = (long)spRound(xi);
            }
	} else {
            if (x->data[k] == 0 && x->imag[k] == 0) {
                x->data[k] = 0;
                x->imag[k] = 0;
            } else {
                if (x->imag[k] == 0 && ti == 0.0) {
                    x->data[k] = (long)spRound(pow((double)x->data[k], tr));
                } else {
                    xr0 = (double)x->data[k];
                    xi0 = (double)x->imag[k];
                    clog(&xr0, &xi0);
                    xr = xr0 * tr - xi0 * ti;
                    xi = xr0 * ti + xi0 * tr;
                    cexp(&xr, &xi);
                    x->data[k] = (long)spRound(xr);
                    x->imag[k] = (long)spRound(xi);
                }
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvcplxoper(spLVector x, const char *op, double tr, double ti)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	lvcplxplus(x, tr, ti);
    } else if (strveq(op2, "-")) {
	lvcplxminus(x, tr, ti, reverse);
    } else if (strveq(op2, "*")) {
	lvcplxtimes(x, tr, ti);
    } else if (strveq(op2, "/")) {
	lvcplxdivide(x, tr, ti, reverse);
    } else if (strveq(op2, "^")) {
	lvcplxpower(x, tr, ti, reverse);
    } else {
	fprintf(stderr, "lvcplxoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spLVector xlvcplxoper(spLVector a, const char *op, double tr, double ti)
{
    spLVector c;

    c = xlvclone(a);
    lvcplxoper(c, op, tr, ti);

    return c;
}

void fvplus(spFVector x, spFVector y)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	fvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->plus != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->plus(x->instance, y->instance, MIN(x->length, y->length));
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
#ifdef USE_SPL
    nspsbAdd2(y->data, x->data, MIN(x->length, y->length));
    if (x->imag != NULL && y->imag != NULL) {
	nspsbAdd2(y->imag, x->imag, MIN(x->length, y->length));
    }
#else
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    x->data[k] = x->data[k] + y->data[k];
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    x->imag[k] = x->imag[k] + y->imag[k];
		}
	    }
	}
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvminus(spFVector x, spFVector y, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->minus != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->minus(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = y->data[k] - x->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = y->imag[k] - x->imag[k];
		    } else {
			x->imag[k] = -x->imag[k];
		    }
		}
	    } else {
		x->data[k] = -x->data[k];
		if (x->imag != NULL) {
		    x->imag[k] = -x->imag[k];
		}
	    }
	}
    } else {
#ifdef USE_SPL
	nspsbSub2(y->data, x->data, MIN(x->length, y->length));
	if (x->imag != NULL && y->imag != NULL) {
	    nspsbSub2(y->imag, x->imag, MIN(x->length, y->length));
	}
#else
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = x->data[k] - y->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = x->imag[k] - y->imag[k];
		    }
		}
	    }
	}
#endif	
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvtimes(spFVector x, spFVector y)
{
    long k;
    float xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->times != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->times(x->instance, y->instance, MIN(x->length, y->length));
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    xr = x->data[k] * y->data[k] - x->imag[k] * y->imag[k];
		    xi = x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
		    x->data[k] = xr;
		    x->imag[k] = xi;
		} else {
		    x->data[k] = x->data[k] * y->data[k];
		    x->imag[k] = x->imag[k] * y->data[k];
		}
	    } else {
		x->data[k] = x->data[k] * y->data[k];
	    }
	} else {
	    x->data[k] = 0.0f;
	    if (x->imag != NULL) {
		x->imag[k] = 0.0f;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvdivide(spFVector x, spFVector y, spBool reverse)
{
    long k;
    float a;
    float xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->divide != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->divide(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL) {
		    if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
			spwarning("warning: fvoper: divide by zero\n");
			    
			if (y->data[k] == 0.0f) {
			    x->data[k] = 0.0f;
			} else {
			    x->data[k] = y->data[k] / SP_TINY_NUMBERF;
			}
			if (y->imag != NULL) {
			    if (y->imag[k] == 0.0f) {
				x->imag[k] = 0.0f;
			    } else {
				x->imag[k] = y->imag[k] / SP_TINY_NUMBERF;
			    }
			} else {
			    x->imag[k] = 0.0f;
			}
		    } else {
			a = CSQUARE(x->data[k], x->imag[k]);
			if (y->imag != NULL) {
			    xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			    xi = x->data[k] * y->imag[k] - x->imag[k] * y->data[k];
			    x->data[k] = xr / a;
			    x->imag[k] = xi / a;
			} else {
			    x->data[k] =  x->data[k] * y->data[k] / a;
			    x->imag[k] = -x->imag[k] * y->data[k] / a;
			}
		    }
		} else {
		    if (x->data[k] != 0.0f) {
			x->data[k] = y->data[k] / x->data[k];
		    } else {
			spwarning("warning: fvoper: divide by zero\n");

			if (y->data[k] == 0.0f) {
			    x->data[k] = 0.0f;
			} else {
			    x->data[k] = y->data[k] / SP_TINY_NUMBERF;
			}
		    }
		}
	    } else {
		x->data[k] = 0.0f;
		if (x->imag != NULL) {
		    x->imag[k] = 0.0f;
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL && y->imag != NULL) {
		    if (y->data[k] == 0.0f && y->imag[k] == 0.0f) {
			spwarning("warning: fvoper: divide by zero\n");

			if (x->data[k] == 0.0f) {
			    x->data[k] = 0.0f;
			} else {
			    x->data[k] = x->data[k] / SP_TINY_NUMBERF;
			}
			if (x->imag[k] == 0.0f) {
			    x->imag[k] = 0.0f;
			} else {
			    x->imag[k] = x->imag[k] / SP_TINY_NUMBERF;
			}
		    } else {
			a = CSQUARE(y->data[k], y->imag[k]);
			xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			xi = -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
			x->data[k] = xr / a;
			x->imag[k] = xi / a;
		    }
		} else {
		    if (y->data[k] == 0.0f) {
			spwarning("warning: fvoper: divide by zero\n");

			if (x->data[k] == 0.0f) {
			    x->data[k] = 0.0f;
			} else {
			    x->data[k] = x->data[k] / SP_TINY_NUMBERF;
			}
			if (x->imag != NULL) {
			    if (x->imag[k] == 0.0f) {
				x->imag[k] = 0.0f;
			    } else {
				x->imag[k] = x->imag[k] / SP_TINY_NUMBERF;
			    }
			}
		    } else {
			x->data[k] = x->data[k] / y->data[k];
			if (x->imag != NULL) {
			    x->imag[k] = x->imag[k] / y->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0.0f;
		if (x->imag != NULL) {
		    x->imag[k] = 0.0f;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvpower(spFVector x, spFVector y, spBool reverse)
{
    long k;
    float xr, xi;
    float yr, yi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (y->imag != NULL && x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->power != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->power(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    if (reverse) {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (y->imag == NULL) {
			yr = y->data[k];
			yi = 0.0f;
		    } else {
			yr = y->data[k];
			yi = y->imag[k];
		    }
		    if (yr == 0.0f && yi == 0.0f) {
			x->data[k] = 0.0f;
			x->imag[k] = 0.0f;
		    } else if (x->imag[k] == 0.0f && yi == 0.0f) {
			x->data[k] = powf(y->data[k], x->data[k]);
		    } else {
			clogf(&yr, &yi);
			xr = x->data[k] * yr - x->imag[k] * yi;
			xi = x->data[k] * yi + x->imag[k] * yr;
			cexpf(&xr, &xi);
			x->data[k] = xr;
			x->imag[k] = xi;
		    }
		} else {
		    x->data[k] = 0.0f;
		    x->imag[k] = 0.0f;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = powf(y->data[k], x->data[k]);
		} else {
		    x->data[k] = 0.0f;
		}
	    }
	}
    } else {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
			x->data[k] = 0.0f;
			x->imag[k] = 0.0f;
		    } else {
			if (y->imag == NULL) {
			    yr = y->data[k];
			    yi = 0.0f;
			} else {
			    yr = y->data[k];
			    yi = y->imag[k];
			}
			if (x->imag[k] == 0.0f && yi == 0.0f) {
			    x->data[k] = powf(x->data[k], y->data[k]);
			} else {
			    clogf(&x->data[k], &x->imag[k]);
			    xr = x->data[k] * yr - x->imag[k] * yi;
			    xi = x->data[k] * yi + x->imag[k] * yr;
			    cexpf(&xr, &xi);
			    x->data[k] = xr;
			    x->imag[k] = xi;
			}
		    }
		} else {
		    x->data[k] = 1.0;
		    x->imag[k] = 1.0;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = powf(x->data[k], y->data[k]);
		} else {
		    x->data[k] = 1.0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvoper(spFVector x, const char *op, spFVector y)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	fvplus(x, y);
    } else if (strveq(op2, "-")) {
	fvminus(x, y, reverse);
    } else if (strveq(op2, "*")) {
	fvtimes(x, y);
    } else if (strveq(op2, "/")) {
	fvdivide(x, y, reverse);
    } else if (strveq(op2, "^")) {
	fvpower(x, y, reverse);
    } else {
	fprintf(stderr, "fvoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spFVector xfvoper(spFVector a, const char *op, spFVector b)
{
    spFVector c;

    c = xfvclone(a);
    fvoper(c, op, b);

    return c;
}

void fvscplus(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scplus != NULL) {
	flist->scplus(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspsbAdd1(t, x->data, x->length);
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + t;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvscminus(spFVector x, float t, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scminus != NULL) {
	flist->scminus(x->instance, x->length, t, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = t - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - t;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvsctimes(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->sctimes != NULL) {
	flist->sctimes(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspsbMpy1(t, x->data, x->length);
    if (x->imag != NULL) {
	nspsbMpy1(t, x->imag, x->length);
    }
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * t;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * t;
	}
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvscdivide(spFVector x, float t, spBool reverse)
{
    long k;
    float a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scdivide != NULL) {
	flist->scdivide(x->instance, x->length, t, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
		    spwarning("warning: fvscoper: divide by zero\n");

		    if (t == 0.0f) {
			x->data[k] = 0.0f;
		    } else {
			x->data[k] = t / SP_TINY_NUMBERF;
		    }
		    x->imag[k] = 0.0f;
		} else {
		    a = CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = x->data[k] * t / a;
		    x->imag[k] = -x->imag[k] * t / a;
		}
	    } else {
		if (x->data[k] != 0.0f) {
		    x->data[k] = t / x->data[k];
		} else {
		    spwarning("warning: fvscoper: divide by zero\n");

		    if (t == 0.0f) {
			x->data[k] = 0.0f;
		    } else {
			x->data[k] = t / SP_TINY_NUMBERF;
		    }
		}
	    }
	} else {
	    if (t != 0.0f) {
		x->data[k] = x->data[k] / t;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / t;
		}
	    } else {
		spwarning("warning: fvscoper: divide by zero\n");
		if (x->data[k] == 0.0f) {
		    x->data[k] = 0.0f;
		} else {
		    x->data[k] = x->data[k] / SP_TINY_NUMBERF;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0.0f) {
			x->imag[k] = 0.0f;
		    } else {
			x->imag[k] = x->imag[k] / SP_TINY_NUMBERF;
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvscpower(spFVector x, float t, spBool reverse)
{
    long k;
    float a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scpower != NULL) {
	flist->scpower(x->instance, x->length, t, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0.0f) {
		a = (float)log(t);
		x->data[k] *= a;
		x->imag[k] *= a;
		cexpf(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = powf(t, x->data[k]);
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0.0f) {
		clogf(&x->data[k], &x->imag[k]);
		x->data[k] *= t;
		x->imag[k] *= t;
		cexpf(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = powf(x->data[k], t);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void fvscoper(spFVector x, const char *op, float t)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	fvscplus(x, t);
    } else if (strveq(op2, "-")) {
	fvscminus(x, t, reverse);
    } else if (strveq(op2, "*")) {
	fvsctimes(x, t);
    } else if (strveq(op2, "/")) {
	fvscdivide(x, t, reverse);
    } else if (strveq(op2, "^")) {
	fvscpower(x, t, reverse);
    } else {
	fprintf(stderr, "fvscoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spFVector xfvscoper(spFVector a, const char *op, float t)
{
    spFVector c;

    c = xfvclone(a);
    fvscoper(c, op, t);

    return c;
}

void fvcplxplus(spFVector x, float tr, float ti)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxplus != NULL) {
	flist->cplxplus(x->instance, x->length, tr, ti);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspsbAdd1(tr, x->data, x->length);
    nspsbAdd1(ti, x->imag, x->length);
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + tr;
	x->imag[k] = x->imag[k] + ti;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvcplxminus(spFVector x, float tr, float ti, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxminus != NULL) {
	flist->cplxminus(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = tr - x->data[k];
            x->imag[k] = ti - x->imag[k];
	} else {
	    x->data[k] = x->data[k] - tr;
	    x->imag[k] = x->imag[k] - ti;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvcplxtimes(spFVector x, float tr, float ti)
{
    long k;
    float xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxtimes != NULL) {
	flist->cplxtimes(x->instance, x->length, tr, ti);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	xr = x->data[k];
        xi = x->imag[k];
	x->data[k] = xr * tr - xi * ti;
        x->imag[k] = xr * ti + xi * tr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvcplxdivide(spFVector x, float tr, float ti, spBool reverse)
{
    long k;
    float xr, xi;
    float a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxdivide != NULL) {
	flist->cplxdivide(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
                spwarning("warning: fvcplxoper: divide by zero\n");

                if (tr == 0.0f) {
                    x->data[k] = 0.0f;
                } else {
                    x->data[k] = tr / SP_TINY_NUMBERF;
                }
                if (ti == 0.0f) {
                    x->imag[k] = 0.0f;
                } else {
                    x->imag[k] = ti / SP_TINY_NUMBERF;
                }
            } else {
                a = CSQUARE(x->data[k], x->imag[k]);
                xr = x->data[k];
                xi = x->imag[k];
                x->data[k] = (xr * tr + xi * ti) / a;
                x->imag[k] = (xr * ti - xi * tr) / a;
            }
	} else {
	    if (tr == 0.0f && ti == 0.0f) {
		spwarning("warning: fvcplxoper: divide by zero\n");
		if (x->data[k] == 0.0f) {
		    x->data[k] = 0.0f;
		} else {
		    x->data[k] = x->data[k] / SP_TINY_NUMBERF;
		}
                if (x->imag[k] == 0.0f) {
                    x->imag[k] = 0.0f;
                } else {
                    x->imag[k] = x->imag[k] / SP_TINY_NUMBERF;
                }
	    } else {
                a = CSQUARE(tr, ti);
                xr = x->data[k];
                xi = x->imag[k];
		x->data[k] = (tr * xr + ti * xi) / a;
                x->imag[k] = (tr * xi - ti * xr) / a;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void fvcplxpower(spFVector x, float tr, float ti, spBool reverse)
{
    long k;
    float a;
    float xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	fvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->cplxpower != NULL) {
	flist->cplxpower(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (tr == 0.0f && ti == 0.0f) {
                x->data[k] = 0.0f;
                x->imag[k] = 0.0f;
            } else if (x->imag[k] == 0.0f && ti == 0.0f) {
                x->data[k] = powf(tr, x->data[k]);
            } else {
                clogf(&tr, &ti);
                xr = x->data[k] * tr - x->imag[k] * ti;
                xi = x->data[k] * ti + x->imag[k] * tr;
                cexpf(&xr, &xi);
                x->data[k] = xr;
                x->imag[k] = xi;
            }
	} else {
            if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
                x->data[k] = 0.0f;
                x->imag[k] = 0.0f;
            } else {
                if (x->imag[k] == 0.0f && ti == 0.0f) {
                    x->data[k] = powf(x->data[k], tr);
                } else {
                    clogf(&x->data[k], &x->imag[k]);
                    xr = x->data[k] * tr - x->imag[k] * ti;
                    xi = x->data[k] * ti + x->imag[k] * tr;
                    cexpf(&xr, &xi);
                    x->data[k] = xr;
                    x->imag[k] = xi;
                }
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvcplxoper(spFVector x, const char *op, float tr, float ti)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	fvcplxplus(x, tr, ti);
    } else if (strveq(op2, "-")) {
	fvcplxminus(x, tr, ti, reverse);
    } else if (strveq(op2, "*")) {
	fvcplxtimes(x, tr, ti);
    } else if (strveq(op2, "/")) {
	fvcplxdivide(x, tr, ti, reverse);
    } else if (strveq(op2, "^")) {
	fvcplxpower(x, tr, ti, reverse);
    } else {
	fprintf(stderr, "fvcplxoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spFVector xfvcplxoper(spFVector a, const char *op, float tr, float ti)
{
    spFVector c;

    c = xfvclone(a);
    fvcplxoper(c, op, tr, ti);

    return c;
}

void dvplus(spDVector x, spDVector y)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->plus != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->plus(x->instance, y->instance, MIN(x->length, y->length));
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
#ifdef USE_SPL
    nspdbAdd2(y->data, x->data, MIN(x->length, y->length));
    if (x->imag != NULL && y->imag != NULL) {
	nspdbAdd2(y->imag, x->imag, MIN(x->length, y->length));
    }
#else
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    x->data[k] = x->data[k] + y->data[k];
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    x->imag[k] = x->imag[k] + y->imag[k];
		}
	    }
	}
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvminus(spDVector x, spDVector y, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->minus != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->minus(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = y->data[k] - x->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = y->imag[k] - x->imag[k];
		    } else {
			x->imag[k] = -x->imag[k];
		    }
		}
	    } else {
		x->data[k] = -x->data[k];
		if (x->imag != NULL) {
		    x->imag[k] = -x->imag[k];
		}
	    }
	}
    } else {
#ifdef USE_SPL
	nspdbSub2(y->data, x->data, MIN(x->length, y->length));
	if (x->imag != NULL && y->imag != NULL) {
	    nspdbSub2(y->imag, x->imag, MIN(x->length, y->length));
	}
#else
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		x->data[k] = x->data[k] - y->data[k];
		if (x->imag != NULL) {
		    if (y->imag != NULL) {
			x->imag[k] = x->imag[k] - y->imag[k];
		    }
		}
	    }
	}
#endif
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvtimes(spDVector x, spDVector y)
{
    long k;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->times != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->times(x->instance, y->instance, MIN(x->length, y->length));
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
	    if (x->imag != NULL) {
		if (y->imag != NULL) {
		    xr = x->data[k] * y->data[k] - x->imag[k] * y->imag[k];
		    xi = x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
		    x->data[k] = xr;
		    x->imag[k] = xi;
		} else {
		    x->data[k] = x->data[k] * y->data[k];
		    x->imag[k] = x->imag[k] * y->data[k];
		}
	    } else {
		x->data[k] = x->data[k] * y->data[k];
	    }
	} else {
	    x->data[k] = 0.0;
	    if (x->imag != NULL) {
		x->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvdivide(spDVector x, spDVector y, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->divide != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->divide(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    if (reverse) {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL) {
		    if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
			spwarning("warning: dvoper: divide by zero\n");

			if (y->data[k] == 0.0) {
			    x->data[k] = 0.0;
			} else {
			    x->data[k] = y->data[k] / SP_TINY_NUMBER;
			}
			if (y->imag != NULL) {
			    if (y->imag[k] == 0.0) {
				x->imag[k] = 0.0;
			    } else {
				x->imag[k] = y->imag[k] / SP_TINY_NUMBER;
			    }
			} else {
			    x->imag[k] = 0.0;
			}
		    } else {
			a = CSQUARE(x->data[k], x->imag[k]);
			if (y->imag != NULL) {
			    xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			    xi = x->data[k] * y->imag[k] - x->imag[k] * y->data[k];
			    x->data[k] = xr / a;
			    x->imag[k] = xi / a;
			} else {
			    x->data[k] =  x->data[k] * y->data[k] / a;
			    x->imag[k] = -x->imag[k] * y->data[k] / a;
			}
		    }
		} else {
		    if (x->data[k] != 0.0) {
			x->data[k] = y->data[k] / x->data[k];
		    } else {
			spwarning("warning: dvoper: divide by zero\n");

			if (y->data[k] == 0.0) {
			    x->data[k] = 0.0;
			} else {
			    x->data[k] = y->data[k] / SP_TINY_NUMBER;
			}
		    }
		}
	    } else {
		x->data[k] = 0.0;
		if (x->imag != NULL) {
		    x->imag[k] = 0.0;
		}
	    }
	}
    } else {
	for (k = 0; k < x->length; k++) {
	    if (k < y->length) {
		if (x->imag != NULL && y->imag != NULL) {
		    if (y->data[k] == 0.0 && y->imag[k] == 0.0) {
			spwarning("warning: dvoper: divide by zero\n");

			if (x->data[k] == 0.0) {
			    x->data[k] = 0.0;
			} else {
			    x->data[k] = x->data[k] / SP_TINY_NUMBER;
			}
			if (x->imag[k] == 0.0) {
			    x->imag[k] = 0.0;
			} else {
			    x->imag[k] = x->imag[k] / SP_TINY_NUMBER;
			}
		    } else {
			a = CSQUARE(y->data[k], y->imag[k]);
			xr = x->data[k] * y->data[k] + x->imag[k] * y->imag[k];
			xi = -x->data[k] * y->imag[k] + x->imag[k] * y->data[k];
			x->data[k] = xr / a;
			x->imag[k] = xi / a;
		    }
		} else {
		    if (y->data[k] == 0.0) {
			spwarning("warning: dvoper: divide by zero\n");

			if (x->data[k] == 0.0) {
			    x->data[k] = 0.0;
			} else {
			    x->data[k] = x->data[k] / SP_TINY_NUMBER;
			}
			if (x->imag != NULL) {
			    if (x->imag[k] == 0.0) {
				x->imag[k] = 0.0;
			    } else {
				x->imag[k] = x->imag[k] / SP_TINY_NUMBER;
			    }
			}
		    } else {
			x->data[k] = x->data[k] / y->data[k];
			if (x->imag != NULL) {
			    x->imag[k] = x->imag[k] / y->data[k];
			}
		    }
		}
	    } else {
		x->data[k] = 0.0;
		if (x->imag != NULL) {
		    x->imag[k] = 0.0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvpower(spDVector x, spDVector y, spBool reverse)
{
    long k;
    double xr, xi;
    double yr, yi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->power != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->power(x->instance, y->instance, MIN(x->length, y->length), reverse);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    if (reverse) {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (y->imag == NULL) {
			yr = y->data[k];
			yi = 0.0;
		    } else {
			yr = y->data[k];
			yi = y->imag[k];
		    }
		    if (yr == 0.0 && yi == 0.0) {
			x->data[k] = 0.0;
			x->imag[k] = 0.0;
		    } else if (x->imag[k] == 0.0 && yi == 0.0) {
			x->data[k] = pow(y->data[k], x->data[k]);
		    } else {
			clog(&yr, &yi);
			xr = x->data[k] * yr - x->imag[k] * yi;
			xi = x->data[k] * yi + x->imag[k] * yr;
			cexp(&xr, &xi);
			x->data[k] = xr;
			x->imag[k] = xi;
		    }
		} else {
		    x->data[k] = 0.0;
		    x->imag[k] = 0.0;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = pow(y->data[k], x->data[k]);
		} else {
		    x->data[k] = 0.0;
		}
	    }
	}
    } else {
	if (x->imag != NULL) {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
			x->data[k] = 0.0;
			x->imag[k] = 0.0;
		    } else {
			if (y->imag == NULL) {
			    yr = y->data[k];
			    yi = 0.0;
			} else {
			    yr = y->data[k];
			    yi = y->imag[k];
			}
			if (x->imag[k] == 0.0 && yi == 0.0) {
			    x->data[k] = pow(x->data[k], y->data[k]);
			} else {
			    clog(&x->data[k], &x->imag[k]);
			    xr = x->data[k] * yr - x->imag[k] * yi;
			    xi = x->data[k] * yi + x->imag[k] * yr;
			    cexp(&xr, &xi);
			    x->data[k] = xr;
			    x->imag[k] = xi;
			}
		    }
		} else {
		    x->data[k] = 1.0;
		    x->imag[k] = 1.0;
		}
	    }
	} else {
	    for (k = 0; k < x->length; k++) {
		if (k < y->length) {
		    x->data[k] = pow(x->data[k], y->data[k]);
		} else {
		    x->data[k] = 1.0;
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvoper(spDVector x, const char *op, spDVector y)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (y->imag != NULL && x->imag == NULL) {
	dvizeros(x, x->length);
    }

    if (strveq(op2, "+")) {
	dvplus(x, y);
    } else if (strveq(op2, "-")) {
	dvminus(x, y, reverse);
    } else if (strveq(op2, "*")) {
	dvtimes(x, y);
    } else if (strveq(op2, "/")) {
	dvdivide(x, y, reverse);
    } else if (strveq(op2, "^")) {
	dvpower(x, y, reverse);
    } else {
	fprintf(stderr, "dvoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spDVector xdvoper(spDVector a, const char *op, spDVector b)
{
    spDVector c;

    c = xdvclone(a);
    dvoper(c, op, b);

    return c;
}

void dvscplus(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scplus != NULL) {
	flist->scplus(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspdbAdd1(t, x->data, x->length);
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + t;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvscminus(spDVector x, double t, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scminus != NULL) {
	flist->scminus(x->instance, x->length, t, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = t - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - t;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvsctimes(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->sctimes != NULL) {
	flist->sctimes(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspdbMpy1(t, x->data, x->length);
    if (x->imag != NULL) {
	nspdbMpy1(t, x->imag, x->length);
    }
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * t;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * t;
	}
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvscdivide(spDVector x, double t, spBool reverse)
{
    long k;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scdivide != NULL) {
	flist->scdivide(x->instance, x->length, t, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
		    spwarning("warning: dvscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0.0;
		    } else {
			x->data[k] = t / SP_TINY_NUMBER;
		    }
		    x->imag[k] = 0.0;
		} else {
		    a = CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = x->data[k] * t / a;
		    x->imag[k] = -x->imag[k] * t / a;
		}
	    } else {
		if (x->data[k] != 0.0) {
		    x->data[k] = t / x->data[k];
		} else {
		    spwarning("warning: dvscoper: divide by zero\n");

		    if (t == 0.0) {
			x->data[k] = 0.0;
		    } else {
			x->data[k] = t / SP_TINY_NUMBER;
		    }
		}
	    }
	} else {
	    if (t != 0.0) {
		x->data[k] = x->data[k] / t;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / t;
		}
	    } else {
		spwarning("warning: dvscoper: divide by zero\n");
		if (x->data[k] == 0.0) {
		    x->data[k] = 0.0;
		} else {
		    x->data[k] = x->data[k] / SP_TINY_NUMBER;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0.0) {
			x->imag[k] = 0.0;
		    } else {
			x->imag[k] = x->imag[k] / SP_TINY_NUMBER;
		    }
		}
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void dvscpower(spDVector x, double t, spBool reverse)
{
    long k;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scpower != NULL) {
	flist->scpower(x->instance, x->length, t, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0.0) {
		a = log(t);
		x->data[k] *= a;
		x->imag[k] *= a;
		cexp(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = pow(t, x->data[k]);
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0.0) {
		clog(&x->data[k], &x->imag[k]);
		x->data[k] *= t;
		x->imag[k] *= t;
		cexp(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = pow(x->data[k], t);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscoper(spDVector x, const char *op, double t)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	dvscplus(x, t);
    } else if (strveq(op2, "-")) {
	dvscminus(x, t, reverse);
    } else if (strveq(op2, "*")) {
	dvsctimes(x, t);
    } else if (strveq(op2, "/")) {
	dvscdivide(x, t, reverse);
    } else if (strveq(op2, "^")) {
	dvscpower(x, t, reverse);
    } else {
	fprintf(stderr, "dvscoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spDVector xdvscoper(spDVector a, const char *op, double t)
{
    spDVector c;

    c = xdvclone(a);
    dvscoper(c, op, t);

    return c;
}

void dvcplxplus(spDVector x, double tr, double ti)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxplus != NULL) {
	flist->cplxplus(x->instance, x->length, tr, ti);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
#ifdef USE_SPL
    nspdbAdd1(tr, x->data, x->length);
    nspdbAdd1(ti, x->imag, x->length);
#else
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + tr;
	x->imag[k] = x->imag[k] + ti;
    }
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvcplxminus(spDVector x, double tr, double ti, spBool reverse)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxminus != NULL) {
	flist->cplxminus(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = tr - x->data[k];
            x->imag[k] = ti - x->imag[k];
	} else {
	    x->data[k] = x->data[k] - tr;
	    x->imag[k] = x->imag[k] - ti;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvcplxtimes(spDVector x, double tr, double ti)
{
    long k;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxtimes != NULL) {
	flist->cplxtimes(x->instance, x->length, tr, ti);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	xr = x->data[k];
        xi = x->imag[k];
	x->data[k] = xr * tr - xi * ti;
        x->imag[k] = xr * ti + xi * tr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvcplxdivide(spDVector x, double tr, double ti, spBool reverse)
{
    long k;
    double xr, xi;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxdivide != NULL) {
	flist->cplxdivide(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
                spwarning("warning: dvcplxoper: divide by zero\n");

                if (tr == 0.0) {
                    x->data[k] = 0.0;
                } else {
                    x->data[k] = tr / SP_TINY_NUMBER;
                }
                if (ti == 0.0) {
                    x->imag[k] = 0.0;
                } else {
                    x->imag[k] = ti / SP_TINY_NUMBER;
                }
            } else {
                a = CSQUARE(x->data[k], x->imag[k]);
                xr = x->data[k];
                xi = x->imag[k];
                x->data[k] = (xr * tr + xi * ti) / a;
                x->imag[k] = (xr * ti - xi * tr) / a;
            }
	} else {
	    if (tr == 0.0 && ti == 0.0) {
		spwarning("warning: dvcplxoper: divide by zero\n");
		if (x->data[k] == 0.0) {
		    x->data[k] = 0.0;
		} else {
		    x->data[k] = x->data[k] / SP_TINY_NUMBER;
		}
                if (x->imag[k] == 0.0) {
                    x->imag[k] = 0.0;
                } else {
                    x->imag[k] = x->imag[k] / SP_TINY_NUMBER;
                }
	    } else {
                a = CSQUARE(tr, ti);
                xr = x->data[k];
                xi = x->imag[k];
		x->data[k] = (tr * xr + ti * xi) / a;
                x->imag[k] = (tr * xi - ti * xr) / a;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
	
    return;
}

void dvcplxpower(spDVector x, double tr, double ti, spBool reverse)
{
    long k;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag == NULL) {
	dvizeros(x, x->length);
    }
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->cplxpower != NULL) {
	flist->cplxpower(x->instance, x->length, tr, ti, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (reverse) {
            if (tr == 0.0 && ti == 0.0) {
                x->data[k] = 0.0;
                x->imag[k] = 0.0;
            } else if (x->imag[k] == 0.0 && ti == 0.0) {
                x->data[k] = pow(tr, x->data[k]);
            } else {
                clog(&tr, &ti);
                xr = x->data[k] * tr - x->imag[k] * ti;
                xi = x->data[k] * ti + x->imag[k] * tr;
                cexp(&xr, &xi);
                x->data[k] = xr;
                x->imag[k] = xi;
            }
	} else {
            if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
                x->data[k] = 0.0;
                x->imag[k] = 0.0;
            } else {
                if (x->imag[k] == 0.0 && ti == 0.0) {
                    x->data[k] = pow(x->data[k], tr);
                } else {
                    clog(&x->data[k], &x->imag[k]);
                    xr = x->data[k] * tr - x->imag[k] * ti;
                    xi = x->data[k] * ti + x->imag[k] * tr;
                    cexp(&xr, &xi);
                    x->data[k] = xr;
                    x->imag[k] = xi;
                }
            }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvcplxoper(spDVector x, const char *op, double tr, double ti)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	dvcplxplus(x, tr, ti);
    } else if (strveq(op2, "-")) {
	dvcplxminus(x, tr, ti, reverse);
    } else if (strveq(op2, "*")) {
	dvcplxtimes(x, tr, ti);
    } else if (strveq(op2, "/")) {
	dvcplxdivide(x, tr, ti, reverse);
    } else if (strveq(op2, "^")) {
	dvcplxpower(x, tr, ti, reverse);
    } else {
	fprintf(stderr, "dvcplxoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spDVector xdvcplxoper(spDVector a, const char *op, double tr, double ti)
{
    spDVector c;

    c = xdvclone(a);
    dvcplxoper(c, op, tr, ti);

    return c;
}

void svlinplus(spSVector x, short m, short incr)
{
    long k;
    short value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->linplus != NULL) {
	flist->linplus(x->instance, x->length, m, incr);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + value;
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlinminus(spSVector x, short m, short incr, spBool reverse)
{
    long k;
    short value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->linminus != NULL) {
	flist->linminus(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = value - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlintimes(spSVector x, short m, short incr)
{
    long k;
    short value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->lintimes != NULL) {
	flist->lintimes(x->instance, x->length, m, incr);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * value;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlindivide(spSVector x, short m, short incr, spBool reverse)
{
    long k;
    short value;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->lindivide != NULL) {
	flist->lindivide(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0 && x->imag[k] == 0) {
		    spwarning("warning: svlinoper: divide by zero\n");

		    if (value == 0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		    x->imag[k] = 0;
		} else {
		    a = CSQUARE((double)x->data[k], (double)x->imag[k]);
		    x->data[k] = (short)spRound((double)(x->data[k] * value) / a);
		    x->imag[k] = (short)spRound((double)(-x->imag[k] * value) / a);
		}
	    } else {
		if (x->data[k] != 0) {
		    x->data[k] = value / x->data[k];
		} else {
		    spwarning("warning: svlinoper: divide by zero\n");

		    if (value == 0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		}
	    }
	} else {
	    if (value != 0) {
		x->data[k] = x->data[k] / value;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / value;
		}
	    } else {
		spwarning("warning: svlinoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / value;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0) {
			x->imag[k] = 0;
		    } else {
			x->imag[k] = x->imag[k] / value;
		    }
		}
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlinpower(spSVector x, short m, short incr, spBool reverse)
{
    long k;
    short value;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->linpower != NULL) {
	flist->linpower(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0) {
		a = log((double)value);
		xr = (double)x->data[k] * a;
		xi = (double)x->imag[k] * a;
		cexp(&xr, &xi);
		x->data[k] = (short)spRound(xr);
		x->imag[k] = (short)spRound(xi);
	    } else {
		x->data[k] = (short)spRound(pow((double)value, (double)x->data[k]));
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0) {
		xr = (double)x->data[k];
		xi = (double)x->imag[k];
		clog(&xr, &xi);
		xr *= (double)value;
		xi *= (double)value;
		cexp(&xr, &xi);
		x->data[k] = (short)spRound(xr);
		x->imag[k] = (short)spRound(xi);
	    } else {
		x->data[k] = (short)spRound(pow((double)x->data[k], (double)value));
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlinoper(spSVector x, const char *op, short m, short incr)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	svlinplus(x, m, incr);
    } else if (strveq(op2, "-")) {
	svlinminus(x, m, incr, reverse);
    } else if (strveq(op2, "*")) {
	svlintimes(x, m, incr);
    } else if (strveq(op2, "/")) {
	svlindivide(x, m, incr, reverse);
    } else if (strveq(op2, "^")) {
	svlinpower(x, m, incr, reverse);
    } else {
	fprintf(stderr, "svlinoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spSVector xsvlinoper(spSVector a, const char *op, short m, short incr)
{
    spSVector c;

    c = xsvclone(a);
    svlinoper(c, op, m, incr);

    return c;
}

void lvlinplus(spLVector x, long m, long incr)
{
    long k;
    long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->linplus != NULL) {
	flist->linplus(x->instance, x->length, m, incr);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + value;
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlinminus(spLVector x, long m, long incr, spBool reverse)
{
    long k;
    long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->linminus != NULL) {
	flist->linminus(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = value - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlintimes(spLVector x, long m, long incr)
{
    long k;
    long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->lintimes != NULL) {
	flist->lintimes(x->instance, x->length, m, incr);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * value;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlindivide(spLVector x, long m, long incr, spBool reverse)
{
    long k;
    long value;
    double a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->lindivide != NULL) {
	flist->lindivide(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0 && x->imag[k] == 0) {
		    spwarning("warning: lvlinoper: divide by zero\n");

		    if (value == 0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		    x->imag[k] = 0;
		} else {
		    a = (double)CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = (long)spRound((double)(x->data[k] * value) / a);
		    x->imag[k] = (long)spRound((double)(-x->imag[k] * value) / a);
		}
	    } else {
		if (x->data[k] != 0) {
		    x->data[k] = value / x->data[k];
		} else {
		    spwarning("warning: lvlinoper: divide by zero\n");

		    if (value == 0) {
			x->data[k] = 0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		}
	    }
	} else {
	    if (value != 0) {
		x->data[k] = x->data[k] / value;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / value;
		}
	    } else {
		spwarning("warning: lvlinoper: divide by zero\n");
		if (x->data[k] == 0) {
		    x->data[k] = 0;
		} else {
		    x->data[k] = x->data[k] / value;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0) {
			x->imag[k] = 0;
		    } else {
			x->imag[k] = x->imag[k] / value;
		    }
		}
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlinpower(spLVector x, long m, long incr, spBool reverse)
{
    long k;
    long value;
    double a;
    double xr, xi;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->linpower != NULL) {
	flist->linpower(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0) {
		a = log((double)value);
		xr = (double)x->data[k] * a;
		xi = (double)x->imag[k] * a;
		cexp(&xr, &xi);
		x->data[k] = (long)spRound(xr);
		x->imag[k] = (long)spRound(xi);
	    } else {
		x->data[k] = (long)spRound(pow((double)value, (double)x->data[k]));
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0) {
		xr = (double)x->data[k];
		xi = (double)x->imag[k];
		clog(&xr, &xi);
		xr *= (double)value;
		xi *= (double)value;
		cexp(&xr, &xi);
		x->data[k] = (long)spRound(xr);
		x->imag[k] = (long)spRound(xi);
	    } else {
		x->data[k] = (long)spRound(pow((double)x->data[k], (double)value));
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlinoper(spLVector x, const char *op, long m, long incr)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	lvlinplus(x, m, incr);
    } else if (strveq(op2, "-")) {
	lvlinminus(x, m, incr, reverse);
    } else if (strveq(op2, "*")) {
	lvlintimes(x, m, incr);
    } else if (strveq(op2, "/")) {
	lvlindivide(x, m, incr, reverse);
    } else if (strveq(op2, "^")) {
	lvlinpower(x, m, incr, reverse);
    } else {
	fprintf(stderr, "lvlinoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spLVector xlvlinoper(spLVector a, const char *op, long m, long incr)
{
    spLVector c;

    c = xlvclone(a);
    lvlinoper(c, op, m, incr);

    return c;
}

void fvlinplus(spFVector x, float m, float incr)
{
    long k;
    float value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->linplus != NULL) {
	flist->linplus(x->instance, x->length, m, incr);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + value;
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlinminus(spFVector x, float m, float incr, spBool reverse)
{
    long k;
    float value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->linminus != NULL) {
	flist->linminus(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = value - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlintimes(spFVector x, float m, float incr)
{
    long k;
    float value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->lintimes != NULL) {
	flist->lintimes(x->instance, x->length, m, incr);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * value;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlindivide(spFVector x, float m, float incr, spBool reverse)
{
    long k;
    float value, a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->lindivide != NULL) {
	flist->lindivide(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0.0f && x->imag[k] == 0.0f) {
		    spwarning("warning: fvlinoper: divide by zero\n");

		    if (value == 0.0f) {
			x->data[k] = 0.0f;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		    x->imag[k] = 0.0f;
		} else {
		    a = CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = x->data[k] * value / a;
		    x->imag[k] = -x->imag[k] * value / a;
		}
	    } else {
		if (x->data[k] != 0.0f) {
		    x->data[k] = value / x->data[k];
		} else {
		    spwarning("warning: fvlinoper: divide by zero\n");

		    if (value == 0.0f) {
			x->data[k] = 0.0f;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		}
	    }
	} else {
	    if (value != 0.0f) {
		x->data[k] = x->data[k] / value;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / value;
		}
	    } else {
		spwarning("warning: fvlinoper: divide by zero\n");
		if (x->data[k] == 0.0f) {
		    x->data[k] = 0.0f;
		} else {
		    x->data[k] = x->data[k] / value;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0.0f) {
			x->imag[k] = 0.0f;
		    } else {
			x->imag[k] = x->imag[k] / value;
		    }
		}
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlinpower(spFVector x, float m, float incr, spBool reverse)
{
    long k;
    float value, a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->linpower != NULL) {
	flist->linpower(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0.0f) {
		a = logf(value);
		x->data[k] *= a;
		x->imag[k] *= a;
		cexpf(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = powf(value, x->data[k]);
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0.0f) {
		clogf(&x->data[k], &x->imag[k]);
		x->data[k] *= value;
		x->imag[k] *= value;
		cexpf(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = powf(x->data[k], value);
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlinoper(spFVector x, const char *op, float m, float incr)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	fvlinplus(x, m, incr);
    } else if (strveq(op2, "-")) {
	fvlinminus(x, m, incr, reverse);
    } else if (strveq(op2, "*")) {
	fvlintimes(x, m, incr);
    } else if (strveq(op2, "/")) {
	fvlindivide(x, m, incr, reverse);
    } else if (strveq(op2, "^")) {
	fvlinpower(x, m, incr, reverse);
    } else {
	fprintf(stderr, "fvlinoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spFVector xfvlinoper(spFVector a, const char *op, float m, float incr)
{
    spFVector c;

    c = xfvclone(a);
    fvlinoper(c, op, m, incr);

    return c;
}

void dvlinplus(spDVector x, double m, double incr)
{
    long k;
    double value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->linplus != NULL) {
	flist->linplus(x->instance, x->length, m, incr);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] + value;
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlinminus(spDVector x, double m, double incr, spBool reverse)
{
    long k;
    double value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->linminus != NULL) {
	flist->linminus(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    x->data[k] = value - x->data[k];
	    if (x->imag != NULL) {
		x->imag[k] = -x->imag[k];
	    }
	} else {
	    x->data[k] = x->data[k] - value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlintimes(spDVector x, double m, double incr)
{
    long k;
    double value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->lintimes != NULL) {
	flist->lintimes(x->instance, x->length, m, incr);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	x->data[k] = x->data[k] * value;
	if (x->imag != NULL) {
	    x->imag[k] = x->imag[k] * value;
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlindivide(spDVector x, double m, double incr, spBool reverse)
{
    long k;
    double value, a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->lindivide != NULL) {
	flist->lindivide(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL) {
		if (x->data[k] == 0.0 && x->imag[k] == 0.0) {
		    spwarning("warning: dvlinoper: divide by zero\n");

		    if (value == 0.0) {
			x->data[k] = 0.0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		    x->imag[k] = 0.0;
		} else {
		    a = CSQUARE(x->data[k], x->imag[k]);
		    x->data[k] = x->data[k] * value / a;
		    x->imag[k] = -x->imag[k] * value / a;
		}
	    } else {
		if (x->data[k] != 0.0) {
		    x->data[k] = value / x->data[k];
		} else {
		    spwarning("warning: dvlinoper: divide by zero\n");

		    if (value == 0.0) {
			x->data[k] = 0.0;
		    } else {
			x->data[k] = value / x->data[k];
		    }
		}
	    }
	} else {
	    if (value != 0.0) {
		x->data[k] = x->data[k] / value;
		if (x->imag != NULL) {
		    x->imag[k] = x->imag[k] / value;
		}
	    } else {
		spwarning("warning: dvlinoper: divide by zero\n");
		if (x->data[k] == 0.0) {
		    x->data[k] = 0.0;
		} else {
		    x->data[k] = x->data[k] / value;
		}
		if (x->imag != NULL) {
		    if (x->imag[k] == 0.0) {
			x->imag[k] = 0.0;
		    } else {
			x->imag[k] = x->imag[k] / value;
		    }
		}
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlinpower(spDVector x, double m, double incr, spBool reverse)
{
    long k;
    double value, a;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->linpower != NULL) {
	flist->linpower(x->instance, x->length, m, incr, reverse);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    value = m;
    for (k = 0; k < x->length; k++) {
	if (reverse) {
	    if (x->imag != NULL && x->imag[k] != 0.0) {
		a = log(value);
		x->data[k] *= a;
		x->imag[k] *= a;
		cexp(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = pow(value, x->data[k]);
	    }
	} else {
	    if (x->imag != NULL && x->imag[k] != 0.0) {
		clog(&x->data[k], &x->imag[k]);
		x->data[k] *= value;
		x->imag[k] *= value;
		cexp(&x->data[k], &x->imag[k]);
	    } else {
		x->data[k] = pow(x->data[k], value);
	    }
	}
	value += incr;
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlinoper(spDVector x, const char *op, double m, double incr)
{
    spBool reverse = 0;
    const char *op2 = op;

    if (x->length <= 0) return;

    if (strveq(op2, "!")) {
	reverse = 1;
	op2++;
    }

    if (strveq(op2, "+")) {
	dvlinplus(x, m, incr);
    } else if (strveq(op2, "-")) {
	dvlinminus(x, m, incr, reverse);
    } else if (strveq(op2, "*")) {
	dvlintimes(x, m, incr);
    } else if (strveq(op2, "/")) {
	dvlindivide(x, m, incr, reverse);
    } else if (strveq(op2, "^")) {
	dvlinpower(x, m, incr, reverse);
    } else {
	fprintf(stderr, "dvlinoper: unknouwn operation: %s\n", op2);
	exit(1);
    }

    return;
}

spDVector xdvlinoper(spDVector a, const char *op, double m, double incr)
{
    spDVector c;

    c = xdvclone(a);
    dvlinoper(c, op, m, incr);

    return c;
}

void svlogiand(spSVector x, spSVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	svreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->logiand != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->logiand(x->instance, y->instance, MIN(x->length, y->length));
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) & (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = (short)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlogior(spSVector x, spSVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	svreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->logior != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->logior(x->instance, y->instance, MIN(x->length, y->length));
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) | (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = (short)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void svlogixor(spSVector x, spSVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	svreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->logixor != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->logixor(x->instance, y->instance, MIN(x->length, y->length));
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) ^ (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = (short)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spSVector xsvlogiand(spSVector x, spSVector y)
{
    spSVector z;

    z = xsvclone(x);
    svlogiand(z, y);

    return z;
}

spSVector xsvlogior(spSVector x, spSVector y)
{
    spSVector z;

    z = xsvclone(x);
    svlogior(z, y);

    return z;
}

spSVector xsvlogixor(spSVector x, spSVector y)
{
    spSVector z;

    z = xsvclone(x);
    svlogixor(z, y);

    return z;
}

void lvlogiand(spLVector x, spLVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	lvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->logiand != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->logiand(x->instance, y->instance, MIN(x->length, y->length));
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) & (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlogior(spLVector x, spLVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	lvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->logior != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->logior(x->instance, y->instance, MIN(x->length, y->length));
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) | (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvlogixor(spLVector x, spLVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	lvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->logixor != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->logixor(x->instance, y->instance, MIN(x->length, y->length));
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0 ? 1 : 0) ^ (y->data[k] != 0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spLVector xlvlogiand(spLVector x, spLVector y)
{
    spLVector z;

    z = xlvclone(x);
    lvlogiand(z, y);

    return z;
}

spLVector xlvlogior(spLVector x, spLVector y)
{
    spLVector z;

    z = xlvclone(x);
    lvlogior(z, y);

    return z;
}

spLVector xlvlogixor(spLVector x, spLVector y)
{
    spLVector z;

    z = xlvclone(x);
    lvlogixor(z, y);

    return z;
}

void fvlogiand(spFVector x, spFVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	fvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->logiand != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->logiand(x->instance, y->instance, MIN(x->length, y->length));
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0f ? 1 : 0) & (y->data[k] != 0.0f ? 1 : 0);
	    x->data[k] = (float)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlogior(spFVector x, spFVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	fvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->logior != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->logior(x->instance, y->instance, MIN(x->length, y->length));
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0f ? 1 : 0) | (y->data[k] != 0.0f ? 1 : 0);
	    x->data[k] = (float)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvlogixor(spFVector x, spFVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	fvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->logixor != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->logixor(x->instance, y->instance, MIN(x->length, y->length));
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0f ? 1 : 0) ^ (y->data[k] != 0.0f ? 1 : 0);
	    x->data[k] = (float)value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spFVector xfvlogiand(spFVector x, spFVector y)
{
    spFVector z;

    z = xfvclone(x);
    fvlogiand(z, y);

    return z;
}

spFVector xfvlogior(spFVector x, spFVector y)
{
    spFVector z;

    z = xfvclone(x);
    fvlogior(z, y);

    return z;
}

spFVector xfvlogixor(spFVector x, spFVector y)
{
    spFVector z;

    z = xfvclone(x);
    fvlogixor(z, y);

    return z;
}

void dvlogiand(spDVector x, spDVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	dvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->logiand != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->logiand(x->instance, y->instance, MIN(x->length, y->length));
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0 ? 1 : 0) & (y->data[k] != 0.0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlogior(spDVector x, spDVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	dvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->logior != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->logior(x->instance, y->instance, MIN(x->length, y->length));
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0 ? 1 : 0) | (y->data[k] != 0.0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvlogixor(spDVector x, spDVector y)
{
    long k;
    unsigned long value;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
    if (x->imag != NULL) {
	dvreal(x);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->logixor != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->logixor(x->instance, y->instance, MIN(x->length, y->length));
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (k < y->length) {
            value = (x->data[k] != 0.0 ? 1 : 0) ^ (y->data[k] != 0.0 ? 1 : 0);
	    x->data[k] = value;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spDVector xdvlogiand(spDVector x, spDVector y)
{
    spDVector z;

    z = xdvclone(x);
    dvlogiand(z, y);

    return z;
}

spDVector xdvlogior(spDVector x, spDVector y)
{
    spDVector z;

    z = xdvclone(x);
    dvlogior(z, y);

    return z;
}

spDVector xdvlogixor(spDVector x, spDVector y)
{
    spDVector z;

    z = xdvclone(x);
    dvlogixor(z, y);

    return z;
}

void svmorph(spSVector x, spSVector y, double rate)
{
    long k;
    double a, b;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA)
	return;
    
    b = rate;
    a = 1.0 - b;

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->morph != NULL) {
	if (svislocked(x) == SP_FALSE && svislocked(y) == SP_TRUE) {
	    svsync(y);
	}

	flist->morph(x->instance, y->instance, MIN(x->length, y->length), a, b);
	
	if (svisplugincomputable(x) == SP_TRUE && svisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (svislocked(y) == SP_FALSE) {
        svsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (y != NODATA && k < y->length) {
	    x->data[k] = (short)round(a * x->data[k] + b * y->data[k]);
	} else {
	    x->data[k] = (short)round(a * x->data[k]);
	}
    }
    
    if (x->imag != NULL) {
	for (k = 0; k < x->length; k++) {
	    if (y != NODATA && (y->imag != NULL && k < y->length)) {
		x->imag[k] = (short)round(a * x->imag[k] + b * y->imag[k]);
	    } else {
		x->imag[k] = (short)round(a * x->imag[k]);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void lvmorph(spLVector x, spLVector y, double rate)
{
    long k;
    double a, b;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA)
	return;
    
    b = rate;
    a = 1.0 - b;

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->morph != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(y) == SP_TRUE) {
	    lvsync(y);
	}

	flist->morph(x->instance, y->instance, MIN(x->length, y->length), a, b);
	
	if (lvisplugincomputable(x) == SP_TRUE && lvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(y) == SP_FALSE) {
        lvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (y != NODATA && k < y->length) {
	    x->data[k] = (long)round(a * x->data[k] + b * y->data[k]);
	} else {
	    x->data[k] = (long)round(a * x->data[k]);
	}
    }
    
    if (x->imag != NULL) {
	for (k = 0; k < x->length; k++) {
	    if (y != NODATA && (y->imag != NULL && k < y->length)) {
		x->imag[k] = (long)round(a * x->imag[k] + b * y->imag[k]);
	    } else {
		x->imag[k] = (long)round(a * x->imag[k]);
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void fvmorph(spFVector x, spFVector y, float rate)
{
    long k;
    float a, b;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA)
	return;
    
    b = rate;
    a = 1.0f - b;

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->morph != NULL) {
	if (fvislocked(x) == SP_FALSE && fvislocked(y) == SP_TRUE) {
	    fvsync(y);
	}

	flist->morph(x->instance, y->instance, MIN(x->length, y->length), a, b);
	
	if (fvisplugincomputable(x) == SP_TRUE && fvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (fvislocked(y) == SP_FALSE) {
        fvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (y != NODATA && k < y->length) {
	    x->data[k] = a * x->data[k] + b * y->data[k];
	} else {
	    x->data[k] = a * x->data[k];
	}
    }
    
    if (x->imag != NULL) {
	for (k = 0; k < x->length; k++) {
	    if (y != NODATA && (y->imag != NULL && k < y->length)) {
		x->imag[k] = a * x->imag[k] + b * y->imag[k];
	    } else {
		x->imag[k] = a * x->imag[k];
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

void dvmorph(spDVector x, spDVector y, double rate)
{
    long k;
    double a, b;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    if (x == NODATA)
	return;
    
    b = rate;
    a = 1.0 - b;

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && y->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->morph != NULL) {
	if (dvislocked(x) == SP_FALSE && dvislocked(y) == SP_TRUE) {
	    dvsync(y);
	}

	flist->morph(x->instance, y->instance, MIN(x->length, y->length), a, b);
	
	if (dvisplugincomputable(x) == SP_TRUE && dvisplugincomputable(y) == SP_TRUE) {
	    return;
	}
    }

    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (dvislocked(y) == SP_FALSE) {
        dvsync(y);
    }
#endif
    
    for (k = 0; k < x->length; k++) {
	if (y != NODATA && k < y->length) {
	    x->data[k] = a * x->data[k] + b * y->data[k];
	} else {
	    x->data[k] = a * x->data[k];
	}
    }
    
    if (x->imag != NULL) {
	for (k = 0; k < x->length; k++) {
	    if (y != NODATA && (y->imag != NULL && k < y->length)) {
		x->imag[k] = a * x->imag[k] + b * y->imag[k];
	    } else {
		x->imag[k] = a * x->imag[k];
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif
    
    return;
}

spSVector xsvmorph(spSVector x, spSVector y, double rate)
{
    spSVector z;

    if (x == NODATA && y == NODATA) {
	z = NODATA;
    } else if (x == NODATA) {
	z = xsvzeros(y->length);
    } else {
	z = xsvclone(x);
    }
    svmorph(z, y, rate);
    
    return z;
}

spLVector xlvmorph(spLVector x, spLVector y, double rate)
{
    spLVector z;

    if (x == NODATA && y == NODATA) {
	z = NODATA;
    } else if (x == NODATA) {
	z = xlvzeros(y->length);
    } else {
	z = xlvclone(x);
    }
    lvmorph(z, y, rate);
    
    return z;
}

spFVector xfvmorph(spFVector x, spFVector y, float rate)
{
    spFVector z;

    if (x == NODATA && y == NODATA) {
	z = NODATA;
    } else if (x == NODATA) {
	z = xfvzeros(y->length);
    } else {
	z = xfvclone(x);
    }
    fvmorph(z, y, rate);
    
    return z;
}

spDVector xdvmorph(spDVector x, spDVector y, double rate)
{
    spDVector z;

    if (x == NODATA && y == NODATA) {
	z = NODATA;
    } else if (x == NODATA) {
	z = xdvzeros(y->length);
    } else {
	z = xdvclone(x);
    }
    dvmorph(z, y, rate);
    
    return z;
}

spSVector xsvremap(spSVector x, spLVector map)
{
    long k;
    spSVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    y = xsvallocul(svgetplugin(x), map->length, SP_FALSE);
    if (x->imag != NULL) {
	svialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE && map->instance != NULL
        && (flist = SpGetSVectorPluginInternalFuncList(y)) != NULL && flist->remap != NULL) {
	if (svislocked(x) == SP_FALSE && lvislocked(map) == SP_TRUE) {
	    lvsync(map);
	}
	svunlock(y);

	flist->remap(y->instance, map->instance, y->length, x->instance, x->length);
	
	return y;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
    if (lvislocked(map) == SP_FALSE) {
        lvsync(map);
    }
#endif
    
    for (k = 0; k < y->length; k++) {
	if (map->data[k] >= 0 && map->data[k] < x->length) {
	    y->data[k] = x->data[map->data[k]];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[map->data[k]];
	    }
	} else {
	    y->data[k] = 0;
	    if (y->imag != NULL) {
		y->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlock(y);
    }
#endif

    return y;
}

spLVector xlvremap(spLVector x, spLVector map)
{
    long k;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    y = xlvallocul(lvgetplugin(x), map->length, SP_FALSE);
    if (x->imag != NULL) {
	lvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE && map->instance != NULL
        && (flist = SpGetLVectorPluginInternalFuncList(y)) != NULL && flist->remap != NULL) {
	if (lvislocked(x) == SP_FALSE && lvislocked(map) == SP_TRUE) {
	    lvsync(map);
	}
	lvunlock(y);

	flist->remap(y->instance, map->instance, y->length, x->instance, x->length);
	
	return y;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
    if (lvislocked(map) == SP_FALSE) {
        lvsync(map);
    }
#endif
    
    for (k = 0; k < y->length; k++) {
	if (map->data[k] >= 0 && map->data[k] < x->length) {
	    y->data[k] = x->data[map->data[k]];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[map->data[k]];
	    }
	} else {
	    y->data[k] = 0;
	    if (y->imag != NULL) {
		y->imag[k] = 0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spFVector xfvremap(spFVector x, spLVector map)
{
    long k;
    spFVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    y = xfvallocul(fvgetplugin(x), map->length, SP_FALSE);
    if (x->imag != NULL) {
	fvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE && map->instance != NULL
        && (flist = SpGetFVectorPluginInternalFuncList(y)) != NULL && flist->remap != NULL) {
	if (fvislocked(x) == SP_FALSE && lvislocked(map) == SP_TRUE) {
	    lvsync(map);
	}
	fvunlock(y);

	flist->remap(y->instance, map->instance, y->length, x->instance, x->length);
	
	return y;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
    if (lvislocked(map) == SP_FALSE) {
        lvsync(map);
    }
#endif
    
    for (k = 0; k < y->length; k++) {
	if (map->data[k] >= 0 && map->data[k] < x->length) {
	    y->data[k] = x->data[map->data[k]];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[map->data[k]];
	    }
	} else {
	    y->data[k] = 0.0f;
	    if (y->imag != NULL) {
		y->imag[k] = 0.0f;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlock(y);
    }
#endif

    return y;
}

spDVector xdvremap(spDVector x, spLVector map)
{
    long k;
    spDVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

    y = xdvallocul(dvgetplugin(x), map->length, SP_FALSE);
    if (x->imag != NULL) {
	dvialloc(y);
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE && map->instance != NULL
        && (flist = SpGetDVectorPluginInternalFuncList(y)) != NULL && flist->remap != NULL) {
	if (dvislocked(x) == SP_FALSE && lvislocked(map) == SP_TRUE) {
	    lvsync(map);
	}
	dvunlock(y);

	flist->remap(y->instance, map->instance, y->length, x->instance, x->length);
	
	return y;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
    if (lvislocked(map) == SP_FALSE) {
        lvsync(map);
    }
#endif
    
    for (k = 0; k < y->length; k++) {
	if (map->data[k] >= 0 && map->data[k] < x->length) {
	    y->data[k] = x->data[map->data[k]];
	    if (y->imag != NULL) {
		y->imag[k] = x->imag[map->data[k]];
	    }
	} else {
	    y->data[k] = 0.0;
	    if (y->imag != NULL) {
		y->imag[k] = 0.0;
	    }
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlock(y);
    }
#endif

    return y;
}

spLVector xsvfind(spSVector x)
{
    long k, l;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	
	instance = flist->find(x->instance, x->length, &l);
	
	return xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    l++;
	}
    }

    y = xlvallocul(svgetplugin(x), l, SP_FALSE);

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    y->data[l] = k;
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    y->data[l] = k;
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spLVector xlvfind(spLVector x)
{
    long k, l;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	
	instance = flist->find(x->instance, x->length, &l);
	
	return xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    l++;
	}
    }

    y = xlvallocul(lvgetplugin(x), l, SP_FALSE);

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    y->data[l] = k;
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    y->data[l] = k;
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spLVector xfvfind(spFVector x)
{
    long k, l;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	
	instance = flist->find(x->instance, x->length, &l);
	
	return xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0f) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0f) {
	    l++;
	}
    }

    y = xlvallocul(fvgetplugin(x), l, SP_FALSE);

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0f) {
	    y->data[l] = k;
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0f) {
	    y->data[l] = k;
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spLVector xdvfind(spDVector x)
{
    long k, l;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	
	instance = flist->find(x->instance, x->length, &l);
	
	return xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0) {
	    l++;
	}
    }

    y = xlvallocul(dvgetplugin(x), l, SP_FALSE);

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0) {
	    y->data[l] = k;
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0) {
	    y->data[l] = k;
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spSVector xsvfindv(spSVector x)
{
    long k, l;
    spSVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	spLVector idxv;
	
	instance = flist->find(x->instance, x->length, &l);
	idxv = xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);

	y = xsvremap(x, idxv);

	xlvfree(idxv);

	return y;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    l++;
	}
    }

    if (x->imag != NULL) {
	y = xsvriallocul(svgetplugin(x), l, SP_FALSE);
    } else {
	y = xsvallocul(svgetplugin(x), l, SP_FALSE);
    }

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    y->data[l] = x->data[k];
	    if (x->imag != NULL) {
		y->imag[l] = x->imag[k];
	    }
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    y->data[l] = x->data[k];
	    y->imag[l] = x->imag[k];
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlock(y);
    }
#endif

    return y;
}

spLVector xlvfindv(spLVector x)
{
    long k, l;
    spLVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	spLVector idxv;
	
	instance = flist->find(x->instance, x->length, &l);
	idxv = xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);

	y = xlvremap(x, idxv);

	xlvfree(idxv);

	return y;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    l++;
	}
    }

    if (x->imag != NULL) {
	y = xlvriallocul(lvgetplugin(x), l, SP_FALSE);
    } else {
	y = xlvallocul(lvgetplugin(x), l, SP_FALSE);
    }

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0) {
	    y->data[l] = x->data[k];
	    if (x->imag != NULL) {
		y->imag[l] = x->imag[k];
	    }
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0) {
	    y->data[l] = x->data[k];
	    y->imag[l] = x->imag[k];
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlock(y);
    }
#endif

    return y;
}

spFVector xfvfindv(spFVector x)
{
    long k, l;
    spFVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	spLVector idxv;
	
	instance = flist->find(x->instance, x->length, &l);
	idxv = xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);

	y = xfvremap(x, idxv);

	xlvfree(idxv);

	return y;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0f) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0f) {
	    l++;
	}
    }

    if (x->imag != NULL) {
	y = xfvriallocul(fvgetplugin(x), l, SP_FALSE);
    } else {
	y = xfvallocul(fvgetplugin(x), l, SP_FALSE);
    }

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0f) {
	    y->data[l] = x->data[k];
	    if (x->imag != NULL) {
		y->imag[l] = x->imag[k];
	    }
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0f) {
	    y->data[l] = x->data[k];
	    y->imag[l] = x->imag[k];
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlock(y);
    }
#endif

    return y;
}

spDVector xdvfindv(spDVector x)
{
    long k, l;
    spDVector y;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif

#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->find != NULL) {
	void *instance;
	spLVector idxv;
	
	instance = flist->find(x->instance, x->length, &l);
	idxv = xlvallocfrominstance(x->plugin, instance, l, SP_FALSE);

	y = xdvremap(x, idxv);

	xlvfree(idxv);

	return y;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0) {
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0) {
	    l++;
	}
    }

    if (x->imag != NULL) {
	y = xdvriallocul(dvgetplugin(x), l, SP_FALSE);
    } else {
	y = xdvallocul(dvgetplugin(x), l, SP_FALSE);
    }

    for (k = 0, l = 0; k < x->length; k++) {
	if (x->data[k] != 0.0) {
	    y->data[l] = x->data[k];
	    if (x->imag != NULL) {
		y->imag[l] = x->imag[k];
	    }
	    l++;
	} else if (x->imag != NULL && x->imag[k] != 0.0) {
	    y->data[l] = x->data[k];
	    y->imag[l] = x->imag[k];
	    l++;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlock(y);
    }
#endif

    return y;
}

void svscevalle(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevalle != NULL) {
	flist->scevalle(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscevallt(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevallt != NULL) {
	flist->scevallt(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] < t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscevalge(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevalge != NULL) {
	flist->scevalge(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] >= t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscevalgt(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevalgt != NULL) {
	flist->scevalgt(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] > t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscevaleq(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevaleq != NULL) {
	flist->scevaleq(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] == t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svscevalne(spSVector x, short t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (svisplugincomputable(x) == SP_TRUE
        && (flist = SpGetSVectorPluginInternalFuncList(x)) != NULL && flist->scevalne != NULL) {
	flist->scevalne(x->instance, x->length, t);
	return;
    }
    
    if (svislocked(x) == SP_FALSE) {
        svsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] != t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (svislocked(x) == SP_FALSE) {
        svunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void svsceval(spSVector x, const char *op, short t)
{
    if (strveq(op, "<=")) {
	svscevalle(x, t);
    } else if (strveq(op, "<")) {
	svscevallt(x, t);
    } else if (strveq(op, ">=")) {
	svscevalge(x, t);
    } else if (strveq(op, ">")) {
	svscevalgt(x, t);
    } else if (strveq(op, "==")) {
	svscevaleq(x, t);
    } else if (strveq(op, "!=")) {
	svscevalne(x, t);
    } else {
	fprintf(stderr, "xsvsceval: unknouwn operation: %s\n", op);
	exit(1);
    }

    return;
}

spSVector xsvsceval(spSVector x, const char *op, short t)
{
    spSVector y;

    y = xsvclone(x);
    svsceval(y, op, t);

    return y;
}

void lvscevalle(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevalle != NULL) {
	flist->scevalle(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscevallt(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevallt != NULL) {
	flist->scevallt(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] < t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscevalge(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevalge != NULL) {
	flist->scevalge(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] >= t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscevalgt(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevalgt != NULL) {
	flist->scevalgt(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] > t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscevaleq(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevaleq != NULL) {
	flist->scevaleq(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] == t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvscevalne(spLVector x, long t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (lvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetLVectorPluginInternalFuncList(x)) != NULL && flist->scevalne != NULL) {
	flist->scevalne(x->instance, x->length, t);
	return;
    }
    
    if (lvislocked(x) == SP_FALSE) {
        lvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] != t) {
	    x->data[k] = 1;
	} else {
	    x->data[k] = 0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (lvislocked(x) == SP_FALSE) {
        lvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void lvsceval(spLVector x, const char *op, long t)
{
    if (strveq(op, "<=")) {
	lvscevalle(x, t);
    } else if (strveq(op, "<")) {
	lvscevallt(x, t);
    } else if (strveq(op, ">=")) {
	lvscevalge(x, t);
    } else if (strveq(op, ">")) {
	lvscevalgt(x, t);
    } else if (strveq(op, "==")) {
	lvscevaleq(x, t);
    } else if (strveq(op, "!=")) {
	lvscevalne(x, t);
    } else {
	fprintf(stderr, "xlvsceval: unknouwn operation: %s\n", op);
	exit(1);
    }

    return;
}

spLVector xlvsceval(spLVector x, const char *op, long t)
{
    spLVector y;

    y = xlvclone(x);
    lvsceval(y, op, t);

    return y;
}

void fvscevalle(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevalle != NULL) {
	flist->scevalle(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscevallt(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevallt != NULL) {
	flist->scevallt(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] < t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscevalge(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevalge != NULL) {
	flist->scevalge(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] >= t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscevalgt(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevalgt != NULL) {
	flist->scevalgt(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] > t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscevaleq(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevaleq != NULL) {
	flist->scevaleq(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] == t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvscevalne(spFVector x, float t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(x)) != NULL && flist->scevalne != NULL) {
	flist->scevalne(x->instance, x->length, t);
	return;
    }
    
    if (fvislocked(x) == SP_FALSE) {
        fvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] != t) {
	    x->data[k] = 1.0f;
	} else {
	    x->data[k] = 0.0f;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(x) == SP_FALSE) {
        fvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void fvsceval(spFVector x, const char *op, float t)
{
    if (strveq(op, "<=")) {
	fvscevalle(x, t);
    } else if (strveq(op, "<")) {
	fvscevallt(x, t);
    } else if (strveq(op, ">=")) {
	fvscevalge(x, t);
    } else if (strveq(op, ">")) {
	fvscevalgt(x, t);
    } else if (strveq(op, "==")) {
	fvscevaleq(x, t);
    } else if (strveq(op, "!=")) {
	fvscevalne(x, t);
    } else {
	fprintf(stderr, "xfvsceval: unknouwn operation: %s\n", op);
	exit(1);
    }

    return;
}

spFVector xfvsceval(spFVector x, const char *op, float t)
{
    spFVector y;

    y = xfvclone(x);
    fvsceval(y, op, t);

    return y;
}

void dvscevalle(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevalle != NULL) {
	flist->scevalle(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscevallt(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevallt != NULL) {
	flist->scevallt(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] < t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscevalge(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevalge != NULL) {
	flist->scevalge(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] >= t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscevalgt(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevalgt != NULL) {
	flist->scevalgt(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] > t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscevaleq(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevaleq != NULL) {
	flist->scevaleq(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] == t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvscevalne(spDVector x, double t)
{
    long k;
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
#endif
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvisplugincomputable(x) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(x)) != NULL && flist->scevalne != NULL) {
	flist->scevalne(x->instance, x->length, t);
	return;
    }
    
    if (dvislocked(x) == SP_FALSE) {
        dvsync(x);
    }
#endif

    for (k = 0; k < x->length; k++) {
	if (x->data[k] != t) {
	    x->data[k] = 1.0;
	} else {
	    x->data[k] = 0.0;
	}
    }

#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(x) == SP_FALSE) {
        dvunlockcore(x, SP_VECTOR_LOCK_SWITCH_SYNC_ONLY);
    }
#endif

    return;
}

void dvsceval(spDVector x, const char *op, double t)
{
    if (strveq(op, "<=")) {
	dvscevalle(x, t);
    } else if (strveq(op, "<")) {
	dvscevallt(x, t);
    } else if (strveq(op, ">=")) {
	dvscevalge(x, t);
    } else if (strveq(op, ">")) {
	dvscevalgt(x, t);
    } else if (strveq(op, "==")) {
	dvscevaleq(x, t);
    } else if (strveq(op, "!=")) {
	dvscevalne(x, t);
    } else {
	fprintf(stderr, "xdvsceval: unknouwn operation: %s\n", op);
	exit(1);
    }

    return;
}

spDVector xdvsceval(spDVector x, const char *op, double t)
{
    spDVector y;

    y = xdvclone(x);
    dvsceval(y, op, t);

    return y;
}

spLVector xsvscfind(spSVector x, const char *op, short t)
{
    spSVector y;
    spLVector z;

    y = xsvsceval(x, op, t);
    z = xsvfind(y);

    xsvfree(y);

    return z;
}

spLVector xlvscfind(spLVector x, const char *op, long t)
{
    spLVector y;
    spLVector z;

    y = xlvsceval(x, op, t);
    z = xlvfind(y);

    xlvfree(y);

    return z;
}

spLVector xfvscfind(spFVector x, const char *op, float t)
{
    spFVector y;
    spLVector z;

    y = xfvsceval(x, op, t);
    z = xfvfind(y);

    xfvfree(y);

    return z;
}

spLVector xdvscfind(spDVector x, const char *op, double t)
{
    spDVector y;
    spLVector z;

    y = xdvsceval(x, op, t);
    z = xdvfind(y);

    xdvfree(y);

    return z;
}

spSVector xsvscfindv(spSVector x, const char *op, short t)
{
    spLVector idx;
    spSVector xd;

    idx = xsvscfind(x, op, t);
    xd = xsvremap(x, idx);

    xlvfree(idx);

    return xd;
}

spLVector xlvscfindv(spLVector x, const char *op, long t)
{
    spLVector idx;
    spLVector xd;

    idx = xlvscfind(x, op, t);
    xd = xlvremap(x, idx);

    xlvfree(idx);

    return xd;
}

spFVector xfvscfindv(spFVector x, const char *op, float t)
{
    spLVector idx;
    spFVector xd;

    idx = xfvscfind(x, op, t);
    xd = xfvremap(x, idx);

    xlvfree(idx);

    return xd;
}

spDVector xdvscfindv(spDVector x, const char *op, double t)
{
    spLVector idx;
    spDVector xd;

    idx = xdvscfind(x, op, t);
    xd = xdvremap(x, idx);

    xlvfree(idx);

    return xd;
}

void svidxtoeval(spSVector x, spLVector idx)
{
    svzeros(x, x->length);
    svidxnums(x, idx, 1);

    return;
}

spSVector xsvidxtoeval(spLVector idx, long length)
{
    spSVector x;

    x = xsvalloc(length);
    svidxtoeval(x, idx);

    return x;
}

void lvidxtoeval(spLVector x, spLVector idx)
{
    lvzeros(x, x->length);
    lvidxnums(x, idx, 1);

    return;
}

spLVector xlvidxtoeval(spLVector idx, long length)
{
    spLVector x;

    x = xlvalloc(length);
    lvidxtoeval(x, idx);

    return x;
}

void fvidxtoeval(spFVector x, spLVector idx)
{
    fvzeros(x, x->length);
    fvidxnums(x, idx, 1.0f);
    
    return;
}

spFVector xfvidxtoeval(spLVector idx, long length)
{
    spFVector x;

    x = xfvalloc(length);
    fvidxtoeval(x, idx);

    return x;
}

void dvidxtoeval(spDVector x, spLVector idx)
{
    dvzeros(x, x->length);
    dvidxnums(x, idx, 1.0);
    
    return;
}

spDVector xdvidxtoeval(spLVector idx, long length)
{
    spDVector x;

    x = xdvalloc(length);
    dvidxtoeval(x, idx);

    return x;
}

spLVector xfvhistc(spFVector x, spFVector edges, spBool outside_flag)
{
    long k, l;
    long num_edge;
    spLVector count;

    num_edge = edges->length;
    if (outside_flag) num_edge++;

    count = xlvzeros(num_edge);

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= edges->data[0]) {
	    if (outside_flag) {
		++count->data[0];
	    }
	} else if (x->data[k] > edges->data[edges->length - 1]) {
	    if (outside_flag) {
		++count->data[count->length - 1];
	    }
	} else if (x->data[k] == edges->data[edges->length - 1]) {
	    ++count->data[count->length - 1];
	}
	for (l = 1; l < edges->length; l++) {
	    if (x->data[k] >= edges->data[l - 1] && x->data[k] < edges->data[l]) {
		if (outside_flag) {
		    ++count->data[l];
		} else {
		    ++count->data[l - 1];
		}
	    }
	}
    }

    return count;
}

spLVector xdvhistc(spDVector x, spDVector edges, spBool outside_flag)
{
    long k, l;
    long num_edge;
    spLVector count;

    num_edge = edges->length;
    if (outside_flag) num_edge++;

    count = xlvzeros(num_edge);

    for (k = 0; k < x->length; k++) {
	if (x->data[k] <= edges->data[0]) {
	    if (outside_flag) {
		++count->data[0];
	    }
	} else if (x->data[k] > edges->data[edges->length - 1]) {
	    if (outside_flag) {
		++count->data[count->length - 1];
	    }
	} else if (x->data[k] == edges->data[edges->length - 1]) {
	    ++count->data[count->length - 1];
	}
	for (l = 1; l < edges->length; l++) {
	    if (x->data[k] >= edges->data[l - 1] && x->data[k] < edges->data[l]) {
		if (outside_flag) {
		    ++count->data[l];
		} else {
		    ++count->data[l - 1];
		}
	    }
	}
    }

    return count;
}

spLVector xfvhist(spFVector x, spFVector centers, spFVector *xoedges)
{
    long k;
    float minx, maxx;
    float binwidth;
    float edge;
    spFVector edges;
    spLVector count;

    if (centers == NODATA || centers->length < 1) return NODATA;
    
    if (centers->length == 1) {
	minx = fvmin(x, NULL);
	maxx = fvmax(x, NULL);
    
	if (minx == maxx) {
	    minx -= floorf(centers->data[0]) - 0.5f;
	    minx += ceilf(centers->data[0]) - 0.5f;
	}
	binwidth = (maxx - minx) / centers->data[0];

	edges = xfvalloc(((long)centers->data[0]) - 1);
	edge = minx;
	for (k = 0; k < edges->length; k++) {
	    edge += binwidth;
	    edges->data[k] = edge + (float)SP_EPSILON;
	}
    } else {
	edges = xfvalloc(centers->length - 1);
	for (k = 0; k < edges->length; k++) {
	    edge = (centers->data[k] + centers->data[k + 1]) / 2.0f;
	    edges->data[k] = edge + (float)SP_EPSILON;
	}
    }

    count = xfvhistc(x, edges, SP_TRUE);

    if (xoedges != NULL) {
	*xoedges = edges;
    } else {
	xfvfree(edges);
    }

    return count;
}

spLVector xdvhist(spDVector x, spDVector centers, spDVector *xoedges)
{
    long k;
    double minx, maxx;
    double binwidth;
    double edge;
    spDVector edges;
    spLVector count;

    if (centers == NODATA || centers->length < 1) return NODATA;
    
    if (centers->length == 1) {
	minx = dvmin(x, NULL);
	maxx = dvmax(x, NULL);
    
	if (minx == maxx) {
	    minx -= floor(centers->data[0]) - 0.5;
	    minx += ceil(centers->data[0]) - 0.5;
	}
	binwidth = (maxx - minx) / centers->data[0];

	edges = xdvalloc(((long)centers->data[0]) - 1);
	edge = minx;
	for (k = 0; k < edges->length; k++) {
	    edge += binwidth;
	    edges->data[k] = edge + SP_EPSILON;
	}
    } else {
	edges = xdvalloc(centers->length - 1);
	for (k = 0; k < edges->length; k++) {
	    edge = (centers->data[k] + centers->data[k + 1]) / 2.0;
	    edges->data[k] = edge + SP_EPSILON;
	}
    }

    count = xdvhistc(x, edges, SP_TRUE);

    if (xoedges != NULL) {
	*xoedges = edges;
    } else {
	xdvfree(edges);
    }

    return count;
}

#if 1
static char *getnexttoken(const char *current, char *token)
{
    char c;
    char *p;

    p = (char *)current;
    while (1) {
	c = *p++;
	if (c == NUL) {
	    return NULL;
	} else if (c == '%') {
	    token[0] = c;
	    c = *p++;
	    token[1] = c;
	    if (c == 'l') {
		c = *p++;
		token[2] = c;
		token[3] = NUL;
	    } else {
		token[2] = NUL;
	    }
	    return p;
	} else if (c == '+' || c == '-' || c == '*' || c == '/' || c == '^'
		   || c == '(' || c == ')') {
	    token[0] = c;
	    token[1] = NUL;
	    return p;
	} else if (c == '\t' || c == '\\' || c == ' ') {
	    /* white space: do nothing */
	} else {
	    spwarning("warning: dvevalf: unknown token: %c\n", c);
	    return NULL;
	}
    }

    return p;
}

typedef struct _spEvalStack *spEvalStack;
struct _spEvalStack {
    spEvalStack prev;
    spEvalStack next;
    char token[4];
    spBool allocated;
    union {
	void *ptr;
	double value;
	int depth;
    } u;
};

spEvalStack spEvalStackCreate(const char *token)
{
    spEvalStack stack;

    stack = xalloc(1, struct _spEvalStack);
    memset(stack, 0, sizeof(struct _spEvalStack));
    spStrCopy(stack->token, sizeof(stack->token), token);

    return stack;
}

void spEvalStackDestroy(spEvalStack stack, spBool dv_flag)
{
    if (stack == NULL) return;

    if (stack->allocated == SP_TRUE) {
	if (dv_flag == SP_TRUE) {
	    xdvfree(stack->u.ptr);
	} else {
	    xfvfree(stack->u.ptr);
	}
    }
    
    if (stack->prev != NULL) {
	stack->prev->next = stack->next;
    }
    if (stack->next != NULL) {
	stack->next->prev = stack->prev;
    }
    xfree(stack);
    
    return;
}

spBool spEvalStackEqToken(spEvalStack stack, const char *token)
{
    if (stack == NULL) return SP_FALSE;
    
    if (streq(stack->token, token)) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spEvalStackIsOperator(spEvalStack stack)
{
    if (stack == NULL) return SP_FALSE;
    
    if (stack->token[0] == '+' || stack->token[0] == '-'
	|| stack->token[0] == '*' || stack->token[0] == '/' || stack->token[0] == '^'
	|| stack->token[0] == '(' || stack->token[0] == ')') {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spEvalStackIsParenthesis(spEvalStack stack)
{
    if (stack == NULL) return SP_FALSE;
    
    if (stack->token[0] == '(' || stack->token[0] == ')') {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spEvalStackIsWeakOperator(spEvalStack stack)
{
    if (stack == NULL) return SP_FALSE;
    
    if (stack->token[0] == '+' || stack->token[0] == '-') {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spEvalStackIsStrongOperator(spEvalStack stack)
{
    if (stack == NULL) return SP_FALSE;
    
    if (stack->token[0] == '*' || stack->token[0] == '/' || stack->token[0] == '^') {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

void spEvalStackSetToken(spEvalStack stack, const char *token)
{
    spStrCopy(stack->token, sizeof(stack->token), token);
    return;
}

void spEvalStackSetPointer(spEvalStack stack, void *ptr)
{
    stack->u.ptr = ptr;
    return;
}

spEvalStack spEvalStackCreatePointerStack(const char *token, void *ptr)
{
    spEvalStack stack;

    stack = spEvalStackCreate(token);
    spEvalStackSetPointer(stack, ptr);
    
    return stack;
}

void spEvalStackSetValue(spEvalStack stack, double value)
{
    stack->u.value = value;
    return;
}

spEvalStack spEvalStackCreateValueStack(const char *token, double value)
{
    spEvalStack stack;

    stack = spEvalStackCreate(token);
    spEvalStackSetValue(stack, value);
    
    return stack;
}

void spEvalStackSetPrev(spEvalStack stack, spEvalStack prev_stack)
{
    if (stack->prev != NULL) {
	stack->prev->next = prev_stack;
    }
    if (prev_stack != NULL) {
	prev_stack->prev = stack->prev;
	prev_stack->next = stack;
    }
    stack->prev = prev_stack;
    
    return;
}

void spEvalStackSetNext(spEvalStack stack, spEvalStack next_stack)
{
    if (stack->next != NULL) {
	stack->next->prev = next_stack;
    }
    if (next_stack != NULL) {
	next_stack->prev = stack;
	next_stack->next = stack->next;
    }
    stack->next = next_stack;
    
    return;
}

void spEvalStackPush(spEvalStack *first_stack, spEvalStack *last_stack, spEvalStack stack)
{
    if (*first_stack == NULL) *first_stack = stack;
    
    if (*last_stack != NULL) {
	spEvalStackSetNext(*last_stack, stack);
    }
    *last_stack = stack;
    
    return;
}

spEvalStack spEvalStackPop(spEvalStack *first_stack, spEvalStack *last_stack)
{
    spEvalStack stack;

    if (*first_stack == NULL || *last_stack == NULL) return NULL;

    stack = *last_stack;
    *last_stack = stack->prev;
    if (*last_stack == NULL) *first_stack = NULL;
    
    spEvalStackSetPrev(stack, NULL);
    
    return stack;
}

spBool spDVEvalStackCalc(spEvalStack *first_stack, spEvalStack *last_stack, spDVector x)
{
    spEvalStack stack, next_stack;
    spEvalStack operand1, operand2;
    spEvalStack result;
    spDVector v1, v2;
    double t1, t2;
    double t;
    char operstr[8];

    result = NULL;
    stack = *first_stack;
    while (1) {
	spDebug(100, "spDVEvalStackCalc", "token = %s\n", stack->token);
	next_stack = stack->next;

	if (spEvalStackIsOperator(stack) == SP_TRUE) {
	    operand2 = stack->prev;
	    if (operand2 == NULL) {
		spDebug(10, "spDVEvalStackCalc", "error: operand2 == NULL\n");
		return SP_FALSE;
	    }
	    operand1 = operand2->prev;
	    if (operand1 == NULL) {
		spDebug(10, "spDVEvalStackCalc", "error: operand1 == NULL\n");
		return SP_FALSE;
	    }

	    if (streq(operand1->token, "%v")) {
		v1 = (spDVector)operand1->u.ptr;
		t1 = 0.0;
	    } else {
		t1 = operand1->u.value;
		v1 = NODATA;
		spDebug(100, "spDVEvalStackCalc", "t1 = %f\n", t1);
	    }
	    if (streq(operand2->token, "%v")) {
		v2 = (spDVector)operand2->u.ptr;
		t2 = 0.0;
	    } else {
		t2 = operand2->u.value;
		v2 = NODATA;
		spDebug(100, "spDVEvalStackCalc", "t2 = %f\n", t2);
	    }

	    if (v1 == NODATA && v2 == NODATA) {
		t = 0.0;
		switch (stack->token[0]) {
		  case '+':
		    t = t1 + t2;
		    break;
		  case '-':
		    t = t1 - t2;
		    break;
		  case '*':
		    t = t1 * t2;
		    break;
		  case '/':
		    t = t1 / t2;
		    break;
		  case '^':
		    t = pow(t1, t2);
		    break;
		  default:
		    spDebug(10, "spDVEvalStackCalc", "error: unknown operator %s\n", stack->token);
		    return SP_FALSE;
		}
		spEvalStackSetValue(operand1, t);
		if (stack == *last_stack) *last_stack = operand1;
		spEvalStackDestroy(stack, SP_TRUE);
		spEvalStackDestroy(operand2, SP_TRUE);
	    } else if (v1 != NODATA && v2 != NODATA) {
		if (result == operand2) {
#if defined(HAVE_SPRINTF_S)
		    sprintf_s(operstr, sizeof(operstr), "!%s", stack->token);
#elif defined(HAVE_SNPRINTF)
		    snprintf(operstr, sizeof(operstr), "!%s", stack->token);
#else
		    sprintf(operstr, "!%s", stack->token);
#endif
		    
		    dvoper(v2, operstr, v1);
		    
		    if (stack == *last_stack) *last_stack = operand2;
		    spEvalStackDestroy(stack, SP_TRUE);
		    spEvalStackDestroy(operand1, SP_TRUE);
		} else {
		    spStrCopy(operstr, sizeof(operstr), stack->token);
		    
		    if (result == NULL) {
			result = operand1;
			dvcopy(x, v1);
			spEvalStackSetPointer(result, x);
			v1 = x;
		    } else if (result == operand1) {
			/* do nothing */
		    } else {
			operand1->u.ptr = xdvclone(v1);
			operand1->allocated = SP_TRUE;
			v1 = operand1->u.ptr;
			spDebug(10, "spDVEvalStackCalc", "**** allocate vector in operand1 ****\n");
		    }
		    
		    dvoper(v1, operstr, v2);
		    
		    if (stack == *last_stack) *last_stack = operand1;
		    spEvalStackDestroy(stack, SP_TRUE);
		    spEvalStackDestroy(operand2, SP_TRUE);
		}
	    } else {
		if (v1 == NODATA) {
#if defined(HAVE_SPRINTF_S)
		    sprintf_s(operstr, sizeof(operstr), "!%s", stack->token);
#elif defined(HAVE_SNPRINTF)
		    snprintf(operstr, sizeof(operstr), "!%s", stack->token);
#else
		    sprintf(operstr, "!%s", stack->token);
#endif
		    if (result == NULL) {
			result = operand2;
			dvcopy(x, v2);
			spEvalStackSetPointer(result, x);
			v2 = x;
		    } else if (result != operand2) {
			spDebug(100, "spDVEvalStackCalc", "original v2 = %lx\n", (long)v2);
			operand2->u.ptr = xdvclone(v2);
			operand2->allocated = SP_TRUE;
			/*v2 = operand1->u.ptr;*/
			v2 = operand2->u.ptr;
			spDebug(100, "spDVEvalStackCalc", "**** allocate vector in operand2 ****: %lx\n", (long)operand2->u.ptr);
		    }
		    dvscoper(v2, operstr, t1);
		    
		    if (stack == *last_stack) *last_stack = operand2;
		    spEvalStackDestroy(stack, SP_TRUE);
		    spEvalStackDestroy(operand1, SP_TRUE);
		} else {
		    spStrCopy(operstr, sizeof(operstr), stack->token);
		    if (result == NULL) {
			result = operand1;
			dvcopy(x, v1);
			spEvalStackSetPointer(result, x);
			v1 = x;
		    } else if (result != operand1) {
			operand1->u.ptr = xdvclone(v1);
			operand1->allocated = SP_TRUE;
			v1 = operand1->u.ptr;
			spDebug(10, "spDVEvalStackCalc", "**** allocate vector in operand1 ****\n");
		    }
		    dvscoper(v1, operstr, t2);

		    if (stack == *last_stack) *last_stack = operand1;
		    spEvalStackDestroy(stack, SP_TRUE);
		    spEvalStackDestroy(operand2, SP_TRUE);
		}
	    }
	}
	
	if (next_stack == NULL) {
	    break;
	}
	
	stack = next_stack;
    }

    if (*last_stack == NULL) return SP_FALSE;

    spDebug(80, "spDVEvalStackCalc", "last_stack->token = %s\n", (*last_stack)->token);
    
    if (result == NULL) {
	/* no vector */
	spDebug(80, "spDVEvalStackCalc", "result is NULL: value = %f\n", (*last_stack)->u.value);
    
	dvnums(x, x->length, (*last_stack)->u.value);
    }
    
    spEvalStackDestroy(*last_stack, SP_TRUE);
    *first_stack = NULL;
    *last_stack = NULL;
    
    return SP_TRUE;
}

static int spOperTokenGetPrecedence(const char *token)
{
    if (token[0] == '^') {
	return 3;
    } else if (token[0] == '*' || token[0] == '/') {
	return 4;
    } else if (token[0] == '+' || token[0] == '-') {
	return 5;
    } else {
	return -1;
    }
}

spBool spOperTokenComparePrecedence(const char *token1, const char *token2, int *prec_diff)
{
    int prec1, prec2;

    if ((prec1 = spOperTokenGetPrecedence(token1)) < 0) return SP_FALSE;
    if ((prec2 = spOperTokenGetPrecedence(token2)) < 0) return SP_FALSE;

    *prec_diff = prec2 - prec1;
    
    return SP_TRUE;
}

/*
 * based on Shunting-yard algorithm 
 * %d: int (short)
 * %ld: long
 * %f: double (float)
 * %v: Vector
 */
spBool dvevalf(spDVector x, const char *format, ...)
{
    va_list argp;
    char *p, *next_p;
    int i;
    long l;
    double t;
    spDVector v;
    char token[4];
    spEvalStack first_op_stack, last_op_stack;
    spEvalStack first_output_stack, last_output_stack;
    spEvalStack stack;
    int prec_diff;
    spBool calc_status;

    dvzeros(x, x->length);
    token[0] = NUL;
    
    va_start(argp, format);

    first_op_stack = last_op_stack = NULL;
    first_output_stack = last_output_stack = NULL;
    
    p = (char *)format;
    for (i = 0;; i++) {
	if ((next_p = getnexttoken(p, token)) == NULL) {
	    break;
	}
	spDebug(100, "dvevalf", "i = %d, token = %s\n", i, token);
	    
	if (token[0] == '%') {
	    if (token[1] == 'l') {
		if (token[2] == 'd') {
		    l = va_arg(argp, long);
		    stack = spEvalStackCreateValueStack(token, (double)l);
		} else if (token[2] == 'f') {
		    t = va_arg(argp, double);
		    stack = spEvalStackCreateValueStack(token, t);
		} else {
		    spwarning("warning: dvevalf: unknown type: %c\n", token[2]);
		    return SP_FALSE;
		}
	    } else if (token[1] == 'd') {
		i = va_arg(argp, int);
		stack = spEvalStackCreateValueStack(token, (double)i);
	    } else if (token[1] == 'f') {
		t = va_arg(argp, double);
		stack = spEvalStackCreateValueStack(token, t);
	    } else if (token[1] == 'v') {
		v = va_arg(argp, spDVector);
		stack = spEvalStackCreatePointerStack(token, v);
	    } else {
		spwarning("warning: dvevalf: unknown type: %c\n", token[1]);
		return SP_FALSE;
	    }

	    spEvalStackPush(&first_output_stack, &last_output_stack, stack);
	} else if (token[0] == '(') {
	    stack = spEvalStackCreate(token);
	    spEvalStackPush(&first_op_stack, &last_op_stack, stack);
	} else if (token[0] == ')') {
	    while ((stack = spEvalStackPop(&first_op_stack, &last_op_stack)) != NULL) {
		if (spEvalStackEqToken(stack, "(") == SP_TRUE) {
		    break;
		}
		spEvalStackPush(&first_output_stack, &last_output_stack, stack);
	    }
	    if (stack == NULL) {
		spwarning("warning: dvevalf: wrong parenthesis\n");
		return SP_FALSE;
	    }
	    spEvalStackDestroy(stack, SP_TRUE);
	} else if (token[0] == '*' || token[0] == '/' || token[0] == '^'
		   || token[0] == '+' || token[0] == '-') {
	    prec_diff = 10000;
	    if (last_op_stack != NULL && last_op_stack->token[0] != '(') {
		spDebug(100, "dvevalf", "token = %s, last_op_stack->token = %s\n", token, last_op_stack->token);
		if (spOperTokenComparePrecedence(token, last_op_stack->token, &prec_diff) == SP_FALSE) {
		    spwarning("warning: dvevalf: wrong expression\n");
		    return SP_FALSE;
		}
	    }
	    spDebug(100, "dvevalf", "prec_diff = %d\n", prec_diff);

	    /*if ((token[0] != '^' && prec_diff <= 0) || prec_diff < 0) {*/
	    if (prec_diff <= 0) { /* If you wish '^' is right-associative, use above expression */
		if ((stack = spEvalStackPop(&first_op_stack, &last_op_stack)) != NULL) {
		    spDebug(100, "dvevalf", "pop and push operator %s \n", stack->token);
		    spEvalStackPush(&first_output_stack, &last_output_stack, stack);
		}
	    }
	    stack = spEvalStackCreate(token);
	    spDebug(100, "dvevalf", "push token %s to operator stack\n", token);
	    spEvalStackPush(&first_op_stack, &last_op_stack, stack);
	}

	p = next_p;
    }

    while (last_op_stack != NULL) {
	if ((stack = spEvalStackPop(&first_op_stack, &last_op_stack)) == NULL) {
	    break;
	}
	spDebug(100, "dvevalf", "pop and push operator %s \n", stack->token);
	spEvalStackPush(&first_output_stack, &last_output_stack, stack);
    }
    
    va_end(argp);

    calc_status = spDVEvalStackCalc(&first_output_stack, &last_output_stack, x);

    return calc_status;
}
#endif
