/** @file sp/voperate.h
 */

#ifndef SPLIB_VOPERATE_H
#define SPLIB_VOPERATE_H

#include <stdarg.h>

#include <sp/vector.h>
#include <sp/vmath.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(MACOS)
#pragma import on
#endif

/** @~english @defgroup voperateGroup <sp/voperate.h>: Vector Operation
@code
#include <sp/voperate.h>
@endcode
 */

/** @addtogroup voperateGroup
 *  @{ */  /*----@addtogroup voperateGroup----*/

/** @~english @name Vector Operation (<sp/voperate.h>)
 *  @{ */

extern void svplus(spSVector x, spSVector y);
extern void svminus(spSVector x, spSVector y, spBool reverse);
extern void svtimes(spSVector x, spSVector y);
extern void svdivide(spSVector x, spSVector y, spBool reverse);
extern void svpower(spSVector x, spSVector y, spBool reverse);
/** @copydoc spDVector::dvoper()
 *  @public @memberof spSVector */
extern void svoper(spSVector a, const char *op, spSVector b);
/** @copydoc spDVector::xdvoper()
 *  @public @memberof spSVector */
extern spSVector xsvoper(spSVector a, const char *op, spSVector b);
    
extern void svscplus(spSVector x, double t);
extern void svscminus(spSVector x, double t, spBool reverse);
extern void svsctimes(spSVector x, double t);
extern void svscdivide(spSVector x, double t, spBool reverse);
extern void svscpower(spSVector x, double t, spBool reverse);
/** @copydoc spDVector::dvscoper()
 *  @public @memberof spSVector */
extern void svscoper(spSVector a, const char *op, double t);
/** @copydoc spDVector::xdvscoper()
 *  @public @memberof spSVector */
extern spSVector xsvscoper(spSVector a, const char *op, double t);

extern void svcplxplus(spSVector x, double tr, double ti);
extern void svcplxminus(spSVector x, double tr, double ti, spBool reverse);
extern void svcplxtimes(spSVector x, double tr, double ti);
extern void svcplxdivide(spSVector x, double tr, double ti, spBool reverse);
extern void svcplxpower(spSVector x, double tr, double ti, spBool reverse);
/** @copydoc spDVector::dvcplxoper()
 *  @public @memberof spSVector */
extern void svcplxoper(spSVector a, const char *op, double tr, double ti);
/** @copydoc spDVector::xdvcplxoper()
 *  @public @memberof spSVector */
extern spSVector xsvcplxoper(spSVector a, const char *op, double tr, double ti);

extern void lvplus(spLVector x, spLVector y);
extern void lvminus(spLVector x, spLVector y, spBool reverse);
extern void lvtimes(spLVector x, spLVector y);
extern void lvdivide(spLVector x, spLVector y, spBool reverse);
extern void lvpower(spLVector x, spLVector y, spBool reverse);
/** @copydoc spDVector::dvoper()
 *  @public @memberof spLVector */
extern void lvoper(spLVector a, const char *op, spLVector b);
/** @copydoc spDVector::xdvoper()
 *  @public @memberof spLVector */
extern spLVector xlvoper(spLVector a, const char *op, spLVector b);

extern void lvscplus(spLVector x, double t);
extern void lvscminus(spLVector x, double t, spBool reverse);
extern void lvsctimes(spLVector x, double t);
extern void lvscdivide(spLVector x, double t, spBool reverse);
extern void lvscpower(spLVector x, double t, spBool reverse);
/** @copydoc spDVector::dvscoper()
 *  @public @memberof spLVector */
extern void lvscoper(spLVector a, const char *op, double t);
/** @copydoc spDVector::xdvscoper()
 *  @public @memberof spLVector */
extern spLVector xlvscoper(spLVector a, const char *op, double t);

extern void lvcplxplus(spLVector x, double tr, double ti);
extern void lvcplxminus(spLVector x, double tr, double ti, spBool reverse);
extern void lvcplxtimes(spLVector x, double tr, double ti);
extern void lvcplxdivide(spLVector x, double tr, double ti, spBool reverse);
extern void lvcplxpower(spLVector x, double tr, double ti, spBool reverse);
/** @copydoc spDVector::dvcplxoper()
 *  @public @memberof spLVector */
extern void lvcplxoper(spLVector a, const char *op, double tr, double ti);
/** @copydoc spDVector::xdvcplxoper()
 *  @public @memberof spLVector */
extern spLVector xlvcplxoper(spLVector a, const char *op, double tr, double ti);

extern void fvplus(spFVector x, spFVector y);
extern void fvminus(spFVector x, spFVector y, spBool reverse);
extern void fvtimes(spFVector x, spFVector y);
extern void fvdivide(spFVector x, spFVector y, spBool reverse);
extern void fvpower(spFVector x, spFVector y, spBool reverse);
/** @copydoc spDVector::dvoper()
 *  @public @memberof spFVector */
extern void fvoper(spFVector a, const char *op, spFVector b);
/** @copydoc spDVector::xdvoper()
 *  @public @memberof spFVector */
extern spFVector xfvoper(spFVector a, const char *op, spFVector b);
    
extern void fvscplus(spFVector x, float t);
extern void fvscminus(spFVector x, float t, spBool reverse);
extern void fvsctimes(spFVector x, float t);
extern void fvscdivide(spFVector x, float t, spBool reverse);
extern void fvscpower(spFVector x, float t, spBool reverse);
/** @copydoc spDVector::dvscoper()
 *  @public @memberof spFVector */
extern void fvscoper(spFVector a, const char *op, float t);
/** @copydoc spDVector::xdvscoper()
 *  @public @memberof spFVector */
extern spFVector xfvscoper(spFVector a, const char *op, float t);

extern void fvcplxplus(spFVector x, float tr, float ti);
extern void fvcplxminus(spFVector x, float tr, float ti, spBool reverse);
extern void fvcplxtimes(spFVector x, float tr, float ti);
extern void fvcplxdivide(spFVector x, float tr, float ti, spBool reverse);
extern void fvcplxpower(spFVector x, float tr, float ti, spBool reverse);
/** @copydoc spDVector::dvcplxoper()
 *  @public @memberof spFVector */
extern void fvcplxoper(spFVector a, const char *op, float tr, float ti);
/** @copydoc spDVector::xdvcplxoper()
 *  @public @memberof spFVector */
extern spFVector xfvcplxoper(spFVector a, const char *op, float tr, float ti);

extern void dvplus(spDVector x, spDVector y);
extern void dvminus(spDVector x, spDVector y, spBool reverse);
extern void dvtimes(spDVector x, spDVector y);
extern void dvdivide(spDVector x, spDVector y, spBool reverse);
extern void dvpower(spDVector x, spDVector y, spBool reverse);
/** @~english Executes an element-wise arithmetic operation of two input vectors.
 *  @param[in,out] a The first input vector. The computed result for each element overwrites the input element of \p a.
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the input vectors will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides each element of \p b by the corresponding element of \p a .
 *  @param[in] b The second input vector.
 *  @public @memberof spDVector */
extern void dvoper(spDVector a, const char *op, spDVector b);
/** @~english Executes an element-wise arithmetic operation of two input vectors.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] a The first input vector. 
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the input vectors will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides each element of \p b by the corresponding element of \p a .
 *  @param[in] b The second input vector.
 *  @return The memory-allocated vector containing the operation results.
 *  @public @memberof spDVector */
extern spDVector xdvoper(spDVector a, const char *op, spDVector b);

extern void dvscplus(spDVector x, double t);
extern void dvscminus(spDVector x, double t, spBool reverse);
extern void dvsctimes(spDVector x, double t);
extern void dvscdivide(spDVector x, double t, spBool reverse);
extern void dvscpower(spDVector x, double t, spBool reverse);
/** @~english Executes an arithmetic operation between each element of the input vector and the input scalar.
 *  @param[in,out] a The input vector. The computed result for each element overwrites the input element of \p a.
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the inputs will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides the scalar \p t by each element of \p a .
 *  @param[in] t The input scalar.
 *  @public @memberof spDVector */
extern void dvscoper(spDVector a, const char *op, double t);
/** @~english Executes an arithmetic operation between each element of the input vector and the input scalar.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] a The input vector. 
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the inputs will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides the scalar \p t by each element of \p a .
 *  @param[in] t The input scalar.
 *  @return The memory-allocated vector containing the operation results.
 *  @public @memberof spDVector */
extern spDVector xdvscoper(spDVector a, const char *op, double t);

extern void dvcplxplus(spDVector x, double tr, double ti);
extern void dvcplxminus(spDVector x, double tr, double ti, spBool reverse);
extern void dvcplxtimes(spDVector x, double tr, double ti);
extern void dvcplxdivide(spDVector x, double tr, double ti, spBool reverse);
extern void dvcplxpower(spDVector x, double tr, double ti, spBool reverse);
/** @~english Executes an arithmetic operation between each element of the input vector and the input complex scalar.
 *  @param[in,out] a The input vector. The computed result for each element overwrites the input element of \p a.
 *  If this vector doesn't have the imaginary part, the memory of the imaginary part will be allocated automatically.
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the inputs will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides the scalar `tr + j * ti` by each element of \p a .
 *  @param[in] tr The real part of the input scalar.
 *  @param[in] ti The imaginary part of the input scalar.
 *  @public @memberof spDVector */
extern void dvcplxoper(spDVector a, const char *op, double tr, double ti);
/** @~english Executes an arithmetic operation between each element of the input vector and the input complex scalar.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] a The input vector.
 *  @param[in] op A null-terminated string indicating the type of the operator.
 *  "+", "-", "*", "/" and "^" correspond to addition, subtraction, multiplication, division and power respectively.
 *  Moreover, the order of the inputs will be reversed if "!" is specified before the operator.
 *  For example, "!/" divides the scalar `tr + j * ti` by each element of \p a .
 *  @param[in] tr The real part of the input scalar.
 *  @param[in] ti The imaginary part of the input scalar.
 *  @return The memory-allocated vector containing the operation results.
 *  @public @memberof spDVector */
extern spDVector xdvcplxoper(spDVector a, const char *op, double tr, double ti);

/** @~english @} */

extern void svlinplus(spSVector x, short m, short incr);
extern void svlinminus(spSVector x, short m, short incr, spBool reverse);
extern void svlintimes(spSVector x, short m, short incr);
extern void svlindivide(spSVector x, short m, short incr, spBool reverse);
extern void svlinpower(spSVector x, short m, short incr, spBool reverse);
extern void svlinoper(spSVector a, const char *op, short m, short incr);
extern spSVector xsvlinoper(spSVector a, const char *op, short m, short incr);

extern void lvlinplus(spLVector x, long m, long incr);
extern void lvlinminus(spLVector x, long m, long incr, spBool reverse);
extern void lvlintimes(spLVector x, long m, long incr);
extern void lvlindivide(spLVector x, long m, long incr, spBool reverse);
extern void lvlinpower(spLVector x, long m, long incr, spBool reverse);
extern void lvlinoper(spLVector a, const char *op, long m, long incr);
extern spLVector xlvlinoper(spLVector a, const char *op, long m, long incr);

extern void fvlinplus(spFVector x, float m, float incr);
extern void fvlinminus(spFVector x, float m, float incr, spBool reverse);
extern void fvlintimes(spFVector x, float m, float incr);
extern void fvlindivide(spFVector x, float m, float incr, spBool reverse);
extern void fvlinpower(spFVector x, float m, float incr, spBool reverse);
extern void fvlinoper(spFVector a, const char *op, float m, float incr);
extern spFVector xfvlinoper(spFVector a, const char *op, float m, float incr);

extern void dvlinplus(spDVector x, double m, double incr);
extern void dvlinminus(spDVector x, double m, double incr, spBool reverse);
extern void dvlintimes(spDVector x, double m, double incr);
extern void dvlindivide(spDVector x, double m, double incr, spBool reverse);
extern void dvlinpower(spDVector x, double m, double incr, spBool reverse);
extern void dvlinoper(spDVector a, const char *op, double m, double incr);
extern spDVector xdvlinoper(spDVector a, const char *op, double m, double incr);

extern void svlogiand(spSVector x, spSVector y);
extern void svlogior(spSVector x, spSVector y);
extern void svlogixor(spSVector x, spSVector y);
extern spSVector xsvlogiand(spSVector x, spSVector y);
extern spSVector xsvlogior(spSVector x, spSVector y);
extern spSVector xsvlogixor(spSVector x, spSVector y);
extern void lvlogiand(spLVector x, spLVector y);
extern void lvlogior(spLVector x, spLVector y);
extern void lvlogixor(spLVector x, spLVector y);
extern spLVector xlvlogiand(spLVector x, spLVector y);
extern spLVector xlvlogior(spLVector x, spLVector y);
extern spLVector xlvlogixor(spLVector x, spLVector y);
extern void fvlogiand(spFVector x, spFVector y);
extern void fvlogior(spFVector x, spFVector y);
extern void fvlogixor(spFVector x, spFVector y);
extern spFVector xfvlogiand(spFVector x, spFVector y);
extern spFVector xfvlogior(spFVector x, spFVector y);
extern spFVector xfvlogixor(spFVector x, spFVector y);
extern void dvlogiand(spDVector x, spDVector y);
extern void dvlogior(spDVector x, spDVector y);
extern void dvlogixor(spDVector x, spDVector y);
extern spDVector xdvlogiand(spDVector x, spDVector y);
extern spDVector xdvlogior(spDVector x, spDVector y);
extern spDVector xdvlogixor(spDVector x, spDVector y);

extern void svmorph(spSVector x, spSVector y, double rate);
extern void lvmorph(spLVector x, spLVector y, double rate);
extern void fvmorph(spFVector x, spFVector y, float rate);
extern void dvmorph(spDVector x, spDVector y, double rate);
extern spSVector xsvmorph(spSVector x, spSVector y, double rate);
extern spLVector xlvmorph(spLVector x, spLVector y, double rate);
extern spFVector xfvmorph(spFVector x, spFVector y, float rate);
extern spDVector xdvmorph(spDVector x, spDVector y, double rate);

/** @~english @name Element Reorder (<sp/voperate.h>)
 *  @{ */

/** @copydoc spDVector::xdvremap
 *  @public @memberof spSVector */
extern spSVector xsvremap(spSVector x, spLVector map);
/** @copydoc spDVector::xdvremap
 *  @public @memberof spLVector */
extern spLVector xlvremap(spLVector x, spLVector map);
/** @copydoc spDVector::xdvremap
 *  @public @memberof spFVector */
extern spFVector xfvremap(spFVector x, spLVector map);
/** @~english Reorders the elements of the input vector \p x according to \p map .
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector.
 *  @param[in] map The 0-based index vector indicating mapping from the index of the input vector to that of the generated vector.
 *  @return The memory-allocated vector after reordering.
 *  @public @memberof spDVector */
extern spDVector xdvremap(spDVector x, spLVector map);

/** @~english @} */

/** @~english @name Element Extraction (<sp/voperate.h>)
 *  @{ */

/** @copydoc spDVector::xdvfind
 *  @public @memberof spSVector */
extern spLVector xsvfind(spSVector x);
/** @copydoc spDVector::xdvfind
 *  @public @memberof spLVector */
extern spLVector xlvfind(spLVector x);
/** @copydoc spDVector::xdvfind
 *  @public @memberof spFVector */
extern spLVector xfvfind(spFVector x);
/** @~english Extracts the indices of nonzero elements of \p x .
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector.
 *  @return The memory-allocated vector containing the indices.
 *  @public @memberof spDVector */
extern spLVector xdvfind(spDVector x);

/** @copydoc spDVector::xdvfindv
 *  @public @memberof spSVector */
extern spSVector xsvfindv(spSVector x);
/** @copydoc spDVector::xdvfindv
 *  @public @memberof spLVector */
extern spLVector xlvfindv(spLVector x);
/** @copydoc spDVector::xdvfindv
 *  @public @memberof spFVector */
extern spFVector xfvfindv(spFVector x);
/** @~english Extracts the elements of \p x which have a nonzero value.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector.
 *  @return The memory-allocated vector containing elements with a nonzero value.
 *  @public @memberof spDVector */
extern spDVector xdvfindv(spDVector x);
    
/** @~english @} */

/** @~english @name Element Comparison (<sp/voperate.h>)
 *  @{ */

extern void svscevalle(spSVector x, short t);
extern void svscevallt(spSVector x, short t);
extern void svscevalge(spSVector x, short t);
extern void svscevalgt(spSVector x, short t);
extern void svscevaleq(spSVector x, short t);
extern void svscevalne(spSVector x, short t);
/** @copydoc spDVector::dvsceval
 *  @public @memberof spSVector */
extern void svsceval(spSVector x, const char *op, short t);
/** @copydoc spDVector::xdvsceval
 *  @public @memberof spSVector */
extern spSVector xsvsceval(spSVector x, const char *op, short t);
    
extern void lvscevalle(spLVector x, long t);
extern void lvscevallt(spLVector x, long t);
extern void lvscevalge(spLVector x, long t);
extern void lvscevalgt(spLVector x, long t);
extern void lvscevaleq(spLVector x, long t);
extern void lvscevalne(spLVector x, long t);
/** @copydoc spDVector::dvsceval
 *  @public @memberof spLVector */
extern void lvsceval(spLVector x, const char *op, long t);
/** @copydoc spDVector::xdvsceval
 *  @public @memberof spLVector */
extern spLVector xlvsceval(spLVector x, const char *op, long t);
    
extern void fvscevalle(spFVector x, float t);
extern void fvscevallt(spFVector x, float t);
extern void fvscevalge(spFVector x, float t);
extern void fvscevalgt(spFVector x, float t);
extern void fvscevaleq(spFVector x, float t);
extern void fvscevalne(spFVector x, float t);
/** @copydoc spDVector::dvsceval
 *  @public @memberof spFVector */
extern void fvsceval(spFVector x, const char *op, float t);
/** @copydoc spDVector::xdvsceval
 *  @public @memberof spFVector */
extern spFVector xfvsceval(spFVector x, const char *op, float t);
    
extern void dvscevalle(spDVector x, double t);
extern void dvscevallt(spDVector x, double t);
extern void dvscevalge(spDVector x, double t);
extern void dvscevalgt(spDVector x, double t);
extern void dvscevaleq(spDVector x, double t);
extern void dvscevalne(spDVector x, double t);
/** @~english Executes a relational operation between each element of the input vector and the input scalar,
 *  and resets the value of the element to 1 if the relation is true, or 0 otherwise.
 *  @param[in,out] x The input vector. The computed result for each element overwrites the input element of \p x.
 *  @param[in] op A null-terminated string indicating the type of the relational operator.
 *  "<=", "<", ">=", ">", "==" and "!=" correspond to "less than", "less than or equal to",
 *  "greater than", "greater than or equal to", "equal to"  and "not equal to" power respectively.
 *  @param[in] t The input scalar.
 *  @public @memberof spDVector */
extern void dvsceval(spDVector x, const char *op, double t);
/** @~english Executes a relational operation between each element of the input vector and the input scalar,
 *  and resets the value of the element to 1 if the relation is true, or 0 otherwise.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector. 
 *  @param[in] op A null-terminated string indicating the type of the relational operator.
 *  "<=", "<", ">=", ">", "==" and "!=" correspond to "less than", "less than or equal to",
 *  "greater than", "greater than or equal to", "equal to"  and "not equal to" power respectively.
 *  @param[in] t The input scalar.
 *  @return The memory-allocated vector.
 *  @public @memberof spDVector */
extern spDVector xdvsceval(spDVector x, const char *op, double t);
    
extern void svidxtoeval(spSVector x, spLVector idx);
extern spSVector xsvidxtoeval(spLVector idx, long length);
extern void lvidxtoeval(spLVector x, spLVector idx);
extern spLVector xlvidxtoeval(spLVector idx, long length);
extern void fvidxtoeval(spFVector x, spLVector idx);
extern spFVector xfvidxtoeval(spLVector idx, long length);
extern void dvidxtoeval(spDVector x, spLVector idx);
extern spDVector xdvidxtoeval(spLVector idx, long length);

/** @copydoc spDVector::xdvscfind
 *  @public @memberof spSVector */
extern spLVector xsvscfind(spSVector x, const char *op, short t);
/** @copydoc spDVector::xdvscfind
 *  @public @memberof spLVector */
extern spLVector xlvscfind(spLVector x, const char *op, long t);
/** @copydoc spDVector::xdvscfind
 *  @public @memberof spFVector */
extern spLVector xfvscfind(spFVector x, const char *op, float t);
/** @~english Executes a relational operation between each element of the input vector and the input scalar,
 *  and extracts the indices where the relation is true.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector. 
 *  @param[in] op A null-terminated string indicating the type of the relational operator.
 *  "<=", "<", ">=", ">", "==" and "!=" correspond to "less than", "less than or equal to",
 *  "greater than", "greater than or equal to", "equal to"  and "not equal to" power respectively.
 *  @param[in] t The input scalar.
 *  @return The memory-allocated index vector.
 *  @public @memberof spDVector */
extern spLVector xdvscfind(spDVector x, const char *op, double t);
    
/** @copydoc spDVector::xdvscfindv
 *  @public @memberof spSVector */
extern spSVector xsvscfindv(spSVector x, const char *op, short t);
/** @copydoc spDVector::xdvscfindv
 *  @public @memberof spLVector */
extern spLVector xlvscfindv(spLVector x, const char *op, long t);
/** @copydoc spDVector::xdvscfindv
 *  @public @memberof spFVector */
extern spFVector xfvscfindv(spFVector x, const char *op, float t);
/** @~english Executes a relational operation between each element of the input vector and the input scalar,
 *  and extracts the elements where the relation is true.
 *  The memory of the returned vector must be deallocated.
 *  @param[in] x The input vector. 
 *  @param[in] op A null-terminated string indicating the type of the relational operator.
 *  "<=", "<", ">=", ">", "==" and "!=" correspond to "less than", "less than or equal to",
 *  "greater than", "greater than or equal to", "equal to"  and "not equal to" power respectively.
 *  @param[in] t The input scalar.
 *  @return The memory-allocated vector containing elements where the relation is true.
 *  @public @memberof spDVector */
extern spDVector xdvscfindv(spDVector x, const char *op, double t);

/** @~english @} */

/** @} */  /*----@addtogroup voperateGroup----*/

/* output vector length: edges->length+1 (outside_flag == SP_TRUE),
 *                       edges->length (outside_flag == SP_FALSE)  */
/** @internal */
extern spLVector xfvhistc(spFVector x, spFVector edges, spBool outside_flag);
extern spLVector xdvhistc(spDVector x, spDVector edges, spBool outside_flag);
extern spLVector xfvhist(spFVector x, spFVector centers, spFVector *xoedges /* can be NULL, must be freed */);
extern spLVector xdvhist(spDVector x, spDVector centers, spDVector *xoedges /* can be NULL, must be freed */);
    
/*
 * based on Shunting-yard algorithm 
 * %d: int (short)
 * %ld: long
 * %f: double (float)
 * %v: Vector
 */
extern spBool dvevalf(spDVector x, const char *format, ...);
    
#if defined(MACOS)
#pragma import off
#endif

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_VOPERATE_H */
