/*
 *	window.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <sp/sp.h>
#include <sp/memory.h>
#include <sp/vector.h>
#include <sp/window.h>

#ifdef SP_USE_VECTOR_ENGINE
#include <sp/spPluginP.h>
#include <sp/spVectorPluginP.h>
#endif

/*
 *	blackman window
 */
void blackmanf(float *window, long length)
{
    long k;
    double a, b;
    double value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);
    b = 2.0 * a;

    for (k = 0; k < length; k++) {
	value = 0.42 - 0.5 * cos(a * (double)k) + 0.08 * cos(b * (double)k);
	window[k] = (float)value;
    }

    return;
}

void blackman(double *window, long length)
{
    long k;
    double a, b;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);
    b = 2.0 * a;

    for (k = 0; k < length; k++) {
	window[k] = 0.42 - 0.5 * cos(a * (double)k) + 0.08 * cos(b * (double)k);
    }

    return;
}

/*
 *	normalized blackman window
 */
void nblackmanf(float *window, long length)
{
    long k;
    double a, b;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);
    b = 2.0 * a;

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.42 - 0.5 * cos(a * (double)k) + 0.08 * cos(b * (double)k);
	power += SQUARE(value);
	window[k] = (float)value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= (float)rms;
    }

    return;
}

void nblackman(double *window, long length)
{
    long k;
    double a, b;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);
    b = 2.0 * a;

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.42 - 0.5 * cos(a * (double)k) + 0.08 * cos(b * (double)k);
	power += SQUARE(value);
	window[k] = value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= rms;
    }

    return;
}

/*
 *	hamming window
 */
void hammingf(float *window, long length)
{
    long k;
    double a;
    double value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0; k < length; k++) {
	value = 0.54 - 0.46 * cos(a * (double)k);
	window[k] = (float)value;
    }

    return;
}

void hamming(double *window, long length)
{
    long k;
    double a;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0; k < length; k++) {
	window[k] = 0.54 - 0.46 * cos(a * (double)k);
    }

    return;
}

/*
 *	normalized hamming window
 */
void nhammingf(float *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.54 - 0.46 * cos(a * (double)k);
	power += SQUARE(value);
	window[k] = (float)value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= (float)rms;
    }

    return;
}

void nhamming(double *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.54 - 0.46 * cos(a * (double)k);
	power += SQUARE(value);
	window[k] = value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= rms;
    }

    return;
}

/*
 *	hanning window
 */
void hanningf(float *window, long length)
{
    long k;
    double a;
    double value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length + 1);

    for (k = 0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)(k + 1));
	window[k] = (float)value;
    }

    return;
}

void hanning(double *window, long length)
{
    long k;
    double a;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length + 1);

    for (k = 0; k < length; k++) {
	window[k] = 0.5 - 0.5 * cos(a * (double)(k + 1));
    }

    return;
}

/*
 *	normalized hanning window
 */
void nhanningf(float *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length + 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)(k + 1));
	power += SQUARE(value);
	window[k] = (float)value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= (float)rms;
    }

    return;
}

void nhanning(double *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length + 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)(k + 1));
	power += SQUARE(value);
	window[k] = value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= rms;
    }

    return;
}

/*
 *	hann window
 */
void hannf(float *window, long length)
{
    long k;
    double a;
    double value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)k);
	window[k] = (float)value;
    }

    return;
}

void hann(double *window, long length)
{
    long k;
    double a;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0; k < length; k++) {
        window[k] = 0.5 - 0.5 * cos(a * (double)k);
    }

    return;
}

/*
 *	normalized hann window
 */
void nhannf(float *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)k);
	power += SQUARE(value);
	window[k] = (float)value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= (float)rms;
    }

    return;
}

void nhann(double *window, long length)
{
    long k;
    double a;
    double value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }
    a = 2.0 * PI / (double)(length - 1);

    for (k = 0, power = 0.0; k < length; k++) {
	value = 0.5 - 0.5 * cos(a * (double)k);
	power += SQUARE(value);
	window[k] = value;
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= rms;
    }

    return;
}

/*
 *	gauss window
 */
void gausswinf(float *window, long length, float alpha)
{
    long k;
    double t0, value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    t0 = (double)length / 2.0;
    
    for (k = 0; k < length; k++) {
	value = ((double)k - t0) / t0;
	window[k] = (float)exp(-0.5 * SQUARE((double)alpha * value));
    }

    return;
}

void gausswin(double *window, long length, double alpha)
{
    long k;
    double t0, value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    t0 = (double)(length - 1) / 2.0;
    
    for (k = 0; k < length; k++) {
	value = ((double)k - t0) / t0;
	window[k] = exp(-0.5 * SQUARE(alpha * value));
    }

    return;
}

/*
 *	normalized gauss window
 */
void ngausswinf(float *window, long length, float alpha)
{
    long k;
    double t0, value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    t0 = (double)length / 2.0;
    
    for (k = 0, power = 0.0; k < length; k++) {
	value = ((double)k - t0) / t0;
	window[k] = (float)exp(-0.5 * SQUARE((double)alpha * value));
	power += SQUARE(window[k]);
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
	window[k] /= (float)rms;
    }

    return;
}

void ngausswin(double *window, long length, double alpha)
{
    long k;
    double t0, value;
    double power, rms;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    t0 = (double)(length - 1) / 2.0;
    
    for (k = 0, power = 0.0; k < length; k++) {
	value = ((double)k - t0) / t0;
	window[k] = exp(-0.5 * SQUARE(alpha * value));
	power += SQUARE(window[k]);
    }
    rms = sqrt(power / (double)length);

    for (k = 0; k < length; k++) {
        window[k] /= (float)rms;
    }

    return;
}

/*
 *	normalized rectangular window
 */
void nboxcarf(float *window, long length)
{
    long k;
    float value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    value = 1.0f / (float)sqrt((double)length);
    for (k = 0; k < length; k++) {
	window[k] = value;
    }

    return;
}

void nboxcar(double *window, long length)
{
    long k;
    double value;

    if (length <= 1) {
	if (length == 1) {
	    window[0] = 1.0;
	}
	return;
    }

    value = 1.0 / sqrt((double)length);
    for (k = 0; k < length; k++) {
	window[k] = value;
    }

    return;
}

/*
 *	blackman window for vector
 */
void fvblackman(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->blackman != NULL) {
	flist->blackman(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    blackmanf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvblackman(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->blackman != NULL) {
	flist->blackman(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    blackman(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif

    return;
}

spFVector xfvblackmanul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvblackman(vec);

    return vec;
}

spDVector xdvblackmanul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvblackman(vec);

    return vec;
}

spFVector xfvblackman(long length)
{
    return xfvblackmanul(NULL, length, SP_FALSE);
}

spDVector xdvblackman(long length)
{
    return xdvblackmanul(NULL, length, SP_FALSE);
}

/*
 *	normalized blackman window for vector
 */
void fvnblackman(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nblackman != NULL) {
	flist->nblackman(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nblackmanf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnblackman(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nblackman != NULL) {
	flist->nblackman(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nblackman(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnblackmanul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvnblackman(vec);

    return vec;
}

spDVector xdvnblackmanul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnblackman(vec);

    return vec;
}

spFVector xfvnblackman(long length)
{
    return xfvnblackmanul(NULL, length, SP_FALSE);
}

spDVector xdvnblackman(long length)
{
    return xdvnblackmanul(NULL, length, SP_FALSE);
}

/*
 *	hamming window for vector
 */
void fvhamming(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->hamming != NULL) {
	flist->hamming(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    hammingf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvhamming(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->hamming != NULL) {
	flist->hamming(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    hamming(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvhammingul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvhamming(vec);

    return vec;
}

spDVector xdvhammingul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvhamming(vec);

    return vec;
}

spFVector xfvhamming(long length)
{
    return xfvhammingul(NULL, length, SP_FALSE);
}

spDVector xdvhamming(long length)
{
    return xdvhammingul(NULL, length, SP_FALSE);
}

/*
 *	normalized hamming window for vector
 */
void fvnhamming(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nhamming != NULL) {
	flist->nhamming(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nhammingf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnhamming(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nhamming != NULL) {
	flist->nhamming(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nhamming(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnhammingul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvnhamming(vec);

    return vec;
}

spDVector xdvnhammingul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnhamming(vec);

    return vec;
}

spFVector xfvnhamming(long length)
{
    return xfvnhammingul(NULL, length, SP_FALSE);
}

spDVector xdvnhamming(long length)
{
    return xdvnhammingul(NULL, length, SP_FALSE);
}

/*
 *	hanning window for vector
 */
void fvhanning(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->hanning != NULL) {
	flist->hanning(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    hanningf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvhanning(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->hanning != NULL) {
	flist->hanning(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    hanning(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvhanningul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvhanning(vec);

    return vec;
}

spDVector xdvhanningul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvhanning(vec);

    return vec;
}

spFVector xfvhanning(long length)
{
    return xfvhanningul(NULL, length, SP_FALSE);
}

spDVector xdvhanning(long length)
{
    return xdvhanningul(NULL, length, SP_FALSE);
}

/*
 *	normalized hanning window for vector
 */
void fvnhanning(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nhanning != NULL) {
	flist->nhanning(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nhanningf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnhanning(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nhanning != NULL) {
	flist->nhanning(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nhanning(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnhanningul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvnhanning(vec);

    return vec;
}

spDVector xdvnhanningul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnhanning(vec);

    return vec;
}

spFVector xfvnhanning(long length)
{
    return xfvnhanningul(NULL, length, SP_FALSE);
}

spDVector xdvnhanning(long length)
{
    return xdvnhanningul(NULL, length, SP_FALSE);
}

/*
 *	hann window for vector
 */
void fvhann(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->hann != NULL) {
	flist->hann(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    hannf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvhann(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->hann != NULL) {
	flist->hann(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    hann(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvhannul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvhann(vec);

    return vec;
}

spDVector xdvhannul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvhann(vec);

    return vec;
}

spFVector xfvhann(long length)
{
    return xfvhannul(NULL, length, SP_FALSE);
}

spDVector xdvhann(long length)
{
    return xdvhannul(NULL, length, SP_FALSE);
}

/*
 *	normalized hann window for vector
 */
void fvnhann(spFVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nhann != NULL) {
	flist->nhann(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nhannf(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnhann(spDVector vec)
{
#ifdef SP_USE_VECTOR_ENGINE
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nhann != NULL) {
	flist->nhann(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nhann(vec->data, vec->length);
    
#ifdef SP_USE_VECTOR_ENGINE
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnhannul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvnhann(vec);

    return vec;
}

spDVector xdvnhannul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnhann(vec);

    return vec;
}

spFVector xfvnhann(long length)
{
    return xfvnhannul(NULL, length, SP_FALSE);
}

spDVector xdvnhann(long length)
{
    return xdvnhannul(NULL, length, SP_FALSE);
}

/*
 *	gauss window for vector
 */
void fvgausswinex(spFVector vec, float alpha)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->gausswin != NULL) {
	flist->gausswin(vec->instance, vec->length, alpha);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    gausswinf(vec->data, vec->length, alpha);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvgausswinex(spDVector vec, double alpha)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->gausswin != NULL) {
	flist->gausswin(vec->instance, vec->length, alpha);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    gausswin(vec->data, vec->length, alpha);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif

    return;
}

spFVector xfvgausswinexul(spPlugin *plugin, long length, float alpha, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvgausswinex(vec, alpha);

    return vec;
}

spDVector xdvgausswinexul(spPlugin *plugin, long length, double alpha, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvgausswinex(vec, alpha);

    return vec;
}

spFVector xfvgausswinex(long length, float alpha)
{
    return xfvgausswinexul(NULL, length, alpha, SP_FALSE);
}

spDVector xdvgausswinex(long length, double alpha)
{
    return xdvgausswinexul(NULL, length, alpha, SP_FALSE);
}

void fvgausswin(spFVector vec)
{
    fvgausswinex(vec, 2.50f);
    return;
}

void dvgausswin(spDVector vec)
{
    dvgausswinex(vec, 2.50);
    return;
}

spFVector xfvgausswinul(spPlugin *plugin, long length, spBool unlock_flag)
{
    return xfvgausswinexul(plugin, length, 2.50f, unlock_flag);
}

spDVector xdvgausswinul(spPlugin *plugin, long length, spBool unlock_flag)
{
    return xdvgausswinexul(plugin, length, 2.50, unlock_flag);
}

spFVector xfvgausswin(long length)
{
    return xfvgausswinul(NULL, length, SP_FALSE);
}

spDVector xdvgausswin(long length)
{
    return xdvgausswinul(NULL, length, SP_FALSE);
}

/*
 *	normalized gauss window for vector
 */
void fvngausswinex(spFVector vec, float alpha)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->ngausswin != NULL) {
	flist->ngausswin(vec->instance, vec->length, alpha);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    ngausswinf(vec->data, vec->length, alpha);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvngausswinex(spDVector vec, double alpha)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->ngausswin != NULL) {
	flist->ngausswin(vec->instance, vec->length, alpha);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    ngausswin(vec->data, vec->length, alpha);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif

    return;
}

spFVector xfvngausswinexul(spPlugin *plugin, long length, float alpha, spBool unlock_flag)
{
    spFVector vec;

    vec = xfvallocul(plugin, length, unlock_flag);
    fvngausswinex(vec, alpha);

    return vec;
}

spDVector xdvngausswinexul(spPlugin *plugin, long length, double alpha, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvngausswinex(vec, alpha);

    return vec;
}

spFVector xfvngausswinex(long length, float alpha)
{
    return xfvngausswinexul(NULL, length, alpha, SP_FALSE);
}

spDVector xdvngausswinex(long length, double alpha)
{
    return xdvngausswinexul(NULL, length, alpha, SP_FALSE);
}

void fvngausswin(spFVector vec)
{
    fvngausswinex(vec, 2.50f);
    return;
}

void dvngausswin(spDVector vec)
{
    dvngausswinex(vec, 2.50);
    return;
}

spFVector xfvngausswinul(spPlugin *plugin, long length, spBool unlock_flag)
{
    return xfvngausswinexul(plugin, length, 2.50f, unlock_flag);
}

spDVector xdvngausswinul(spPlugin *plugin, long length, spBool unlock_flag)
{
    return xdvngausswinexul(plugin, length, 2.50, unlock_flag);
}

spFVector xfvngausswin(long length)
{
    return xfvngausswinul(NULL, length, SP_FALSE);
}

spDVector xdvngausswin(long length)
{
    return xdvngausswinul(NULL, length, SP_FALSE);
}

/*
 *	normalized rectangular window for vector
 */
void fvnboxcar(spFVector vec)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (fvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetFVectorPluginInternalFuncList(vec)) != NULL && flist->nboxcar != NULL) {
	flist->nboxcar(vec->instance, vec->length);
	return;
    }

    if (fvislocked(vec) == SP_FALSE) {
        fvsync(vec);
    }
#endif
    
    nboxcarf(vec->data, vec->length);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (fvislocked(vec) == SP_FALSE) {
        fvunlockcore(vec, 1);
    }
#endif
    
    return;
}

void dvnboxcar(spDVector vec)
{
#if defined(SP_USE_VECTOR_ENGINE)
    spVectorPluginInternalFuncList *flist;
    
    if (dvisplugincomputable(vec) == SP_TRUE
        && (flist = SpGetDVectorPluginInternalFuncList(vec)) != NULL && flist->nboxcar != NULL) {
	flist->nboxcar(vec->instance, vec->length);
	return;
    }

    if (dvislocked(vec) == SP_FALSE) {
        dvsync(vec);
    }
#endif
    
    nboxcar(vec->data, vec->length);
    
#if defined(SP_USE_VECTOR_ENGINE)
    if (dvislocked(vec) == SP_FALSE) {
        dvunlockcore(vec, 1);
    }
#endif
    
    return;
}

spFVector xfvnboxcarul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spFVector vec;
    
    vec = xfvallocul(plugin, length, unlock_flag);
    fvnboxcar(vec);

    return vec;
}

spDVector xdvnboxcarul(spPlugin *plugin, long length, spBool unlock_flag)
{
    spDVector vec;

    vec = xdvallocul(plugin, length, unlock_flag);
    dvnboxcar(vec);

    return vec;
}

spFVector xfvnboxcar(long length)
{
    return xfvnboxcarul(NULL, length, SP_FALSE);
}

spDVector xdvnboxcar(long length)
{
    return xdvnboxcarul(NULL, length, SP_FALSE);
}
