/** @file sp/window.h
 */

#ifndef SPLIB_WINDOW_H
#define SPLIB_WINDOW_H

#include <sp/vector.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(MACOS)
#pragma import on
#endif

/** @~english @defgroup windowGroup <sp/window.h>: Window Functions
@code
#include <sp/window.h>
@endcode
 */

/** @addtogroup windowGroup
 *  @{ */  /*----@addtogroup windowGroup----*/

/** @~english @name Blackman Window (<sp/window.h>)
 *  @{ */

/** @copydoc blackman() */
extern void blackmanf(float *window, long length);
/** @~english
 *  Create a Blackman window of \p length for the output buffer \p window .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void blackman(double *window, long length);

/** @copydoc nblackman() */
extern void nblackmanf(float *window, long length);
/** @~english
 *  Create a Blackman window of \p length normalized by the RMS for the output buffer \p window .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void nblackman(double *window, long length);

/** @~english @} */

/** @~english @name Hamming Window (<sp/window.h>)
 *  @{ */

/** @copydoc hamming() */
extern void hammingf(float *window, long length);
/** @~english
 *  Create a Hamming window of \p length for the output buffer \p window .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void hamming(double *window, long length);

/** @copydoc nhamming() */
extern void nhammingf(float *window, long length);
/** @~english
 *  Create a Hamming window of \p length normalized by the RMS for the output buffer \p window .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void nhamming(double *window, long length);

/** @~english @} */

/** @~english @name Hanning Window (<sp/window.h>)
 *  @{ */

/** @copydoc hanning() */
extern void hanningf(float *window, long length);
/** @~english
 *  Create a Hanning window of \p length for the output buffer \p window .
 *  The generated window doesn't include left and right edges with 0 in contrast to hann() or hannf() .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void hanning(double *window, long length);

/** @copydoc nhanning() */
extern void nhanningf(float *window, long length);
/** @~english
 *  Create a Hanning window of \p length normalized by the RMS for the output buffer \p window .
 *  The generated window doesn't include left and right edges with 0 in contrast to nhann() or nhannf() .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void nhanning(double *window, long length);

/** @~english @} */

/** @~english @name Hann Window (<sp/window.h>)
 *  @{ */

/** @copydoc hann() */
extern void hannf(float *window, long length);
/** @~english
 *  Create a Hann window of \p length for the output buffer \p window .
 *  The generated window includes left and right edges with 0 in contrast to hanning() or hanningf() .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void hann(double *window, long length);

/** @copydoc nhann() */
extern void nhannf(float *window, long length);
/** @~english
 *  Create a Hann window of \p length normalized by the RMS for the output buffer \p window .
 *  The generated window includes left and right edges with 0 in contrast to nhanning() or nhanningf() .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void nhann(double *window, long length);

/** @~english @} */

/** @~english @name Gaussian Window (<sp/window.h>)
 *  @{ */

/** @copydoc gausswin() */
extern void gausswinf(float *window, long length, float alpha);
/** @~english
 *  Create a Gaussian window of \p length for the output buffer \p window .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void gausswin(double *window, long length, double alpha);

/** @copydoc ngausswin() */
extern void ngausswinf(float *window, long length, float alpha);
/** @~english
 *  Create a Gaussian window of \p length normalized by the RMS for the output buffer \p window .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void ngausswin(double *window, long length, double alpha);

/** @~english @} */

/** @~english @name Rectangular Window (<sp/window.h>)
 *  @{ */

/** @copydoc nboxcar() */
extern void nboxcarf(float *window, long length);
/** @~english
 *  Create a rectangular window of \p length normalized by the RMS for the output buffer \p window .
 *  The output buffer \p window must be an array or a memory-allocated buffer. */
extern void nboxcar(double *window, long length);

/** @~english @} */

/** @~english @name Blackman Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvblackman()
 *  @public @memberof spFVector */
extern void fvblackman(spFVector vec);
/** @~english
 *  Create a Blackman window for the vector \p vec .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvblackman(spDVector vec);
/** @copydoc spDVector::xdvblackmanul()
 *  @public @memberof spFVector */
extern spFVector xfvblackmanul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Blackman window of \p length .
 *  @public @memberof spDVector */
extern spDVector xdvblackmanul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvblackman()
 *  @public @memberof spFVector */
extern spFVector xfvblackman(long length);
/** @~english
 *  Create a new vector whose contents are a Blackman window of \p length .
 *  @public @memberof spDVector */
extern spDVector xdvblackman(long length);

/** @copydoc spDVector::dvnblackman()
 *  @public @memberof spFVector */
extern void fvnblackman(spFVector vec);
/** @~english
 *  Create a Blackman window normalized by the RMS for the vector \p vec .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvnblackman(spDVector vec);
/** @copydoc spDVector::xdvnblackmanul()
 *  @public @memberof spFVector */
extern spFVector xfvnblackmanul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Blackman window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnblackmanul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvnblackman()
 *  @public @memberof spFVector */
extern spFVector xfvnblackman(long length);
/** @~english
 *  Create a new vector whose contents are a Blackman window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnblackman(long length);

/** @~english @} */

/** @~english @name Hamming Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvhamming()
 *  @public @memberof spFVector */
extern void fvhamming(spFVector vec);
/** @~english
 *  Create a Hamming window for the vector \p vec .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvhamming(spDVector vec);
/** @copydoc spDVector::xdvhammingul()
 *  @public @memberof spFVector */
extern spFVector xfvhammingul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hamming window of \p length .
 *  @public @memberof spDVector */
extern spDVector xdvhammingul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvhamming()
 *  @public @memberof spFVector */
extern spFVector xfvhamming(long length);
/** @~english
 *  Create a new vector whose contents are a Hamming window of \p length .
 *  @public @memberof spDVector */
extern spDVector xdvhamming(long length);

/** @copydoc spDVector::dvnhamming()
 *  @public @memberof spFVector */
extern void fvnhamming(spFVector vec);
/** @~english
 *  Create a Hamming window normalized by the RMS for the vector \p vec .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvnhamming(spDVector vec);
/** @copydoc spDVector::xdvnhammingul()
 *  @public @memberof spFVector */
extern spFVector xfvnhammingul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hamming window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnhammingul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvnhamming()
 *  @public @memberof spFVector */
extern spFVector xfvnhamming(long length);
/** @~english
 *  Create a new vector whose contents are a Hamming window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnhamming(long length);

/** @~english @} */

/** @~english @name Hanning Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvhanning()
 *  @public @memberof spFVector */
extern void fvhanning(spFVector vec);
/** @~english
 *  Create a Hanning window for the vector \p vec .
 *  The generated window doesn't include left and right edges with 0 in contrast to dvhann() or fvhann() .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvhanning(spDVector vec);
/** @copydoc spDVector::xdvhanningul()
 *  @public @memberof spFVector */
extern spFVector xfvhanningul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hanning window of \p length .
 *  The window doesn't include left and right edges with 0 in contrast to xdvhannul() or xfvhannul() .
 *  @public @memberof spDVector */
extern spDVector xdvhanningul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvhanning()
 *  @public @memberof spFVector */
extern spFVector xfvhanning(long length);
/** @~english
 *  Create a new vector whose contents are a Hanning window of \p length .
 *  The window doesn't include left and right edges with 0 in contrast to xdvhann() or xfvhann() .
 *  @public @memberof spDVector */
extern spDVector xdvhanning(long length);

/** @copydoc spDVector::dvnhanning()
 *  @public @memberof spFVector */
extern void fvnhanning(spFVector vec);
/** @~english
 *  Create a Hanning window normalized by the RMS for the vector \p vec .
 *  The generated window doesn't include left and right edges with 0 in contrast to dvnhann() or fvnhann() .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvnhanning(spDVector vec);
/** @copydoc spDVector::xdvnhanningul()
 *  @public @memberof spFVector */
extern spFVector xfvnhanningul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hanning window of \p length normalized by the RMS.
 *  The window doesn't include left and right edges with 0 in contrast to xdvnhannul() or xfvnhannul() .
 *  @public @memberof spDVector */
extern spDVector xdvnhanningul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvnhanning()
 *  @public @memberof spFVector */
extern spFVector xfvnhanning(long length);
/** @~english
 *  Create a new vector whose contents are a Hanning window of \p length normalized by the RMS.
 *  The window doesn't include left and right edges with 0 in contrast to xdvnhann() or xfvnhann() .
 *  @public @memberof spDVector */
extern spDVector xdvnhanning(long length);

/** @~english @} */

/** @~english @name Hann Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvhann()
 *  @public @memberof spFVector */
extern void fvhann(spFVector vec);
/** @~english
 *  Create a Hann window for the vector \p vec .
 *  The generated window includes left and right edges with 0 in contrast to dvhanning() or fvhanning() .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvhann(spDVector vec);
/** @copydoc spDVector::xdvhannul()
 *  @public @memberof spFVector */
extern spFVector xfvhannul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hann window of \p length .
 *  The window includes left and right edges with 0 in contrast to xdvhanningul() or xfvhanningul() .
 *  @public @memberof spDVector */
extern spDVector xdvhannul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvhann()
 *  @public @memberof spFVector */
extern spFVector xfvhann(long length);
/** @~english
 *  Create a new vector whose contents are a Hann window of \p length .
 *  The window includes left and right edges with 0 in contrast to xdvhanning() or xfvhanning() .
 *  @public @memberof spDVector */
extern spDVector xdvhann(long length);

/** @copydoc spDVector::dvnhann()
 *  @public @memberof spFVector */
extern void fvnhann(spFVector vec);
/** @~english
 *  Create a Hann window normalized by the RMS for the vector \p vec .
 *  The generated window includes left and right edges with 0 in contrast to dvnhanning() or fvnhanning() .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvnhann(spDVector vec);
/** @copydoc spDVector::xdvnhannul()
 *  @public @memberof spFVector */
extern spFVector xfvnhannul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Hann window of \p length normalized by the RMS.
 *  The window includes left and right edges with 0 in contrast to xdvnhanningul() or xfvnhanningul() .
 *  @public @memberof spDVector */
extern spDVector xdvnhannul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvnhann()
 *  @public @memberof spFVector */
extern spFVector xfvnhann(long length);
/** @~english
 *  Create a new vector whose contents are a Hann window of \p length normalized by the RMS.
 *  The window includes left and right edges with 0 in contrast to xdvnhanning() or xfvnhanning() .
 *  @public @memberof spDVector */
extern spDVector xdvnhann(long length);

/** @~english @} */

/** @~english @name Gaussian Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvgausswinex()
 *  @public @memberof spFVector */
extern void fvgausswinex(spFVector vec, float alpha);
/** @~english
 *  Create a Gaussian window for the vector \p vec .
 *  The standard deviation of the Gaussian is determined by `(vec->length - 1) / (2 * alpha)` .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvgausswinex(spDVector vec, double alpha);
/** @copydoc spDVector::xdvgausswinexul()
 *  @public @memberof spFVector */
extern spFVector xfvgausswinexul(spPlugin *plugin, long length, float alpha, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Gaussian window of \p length .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  @public @memberof spDVector */
extern spDVector xdvgausswinexul(spPlugin *plugin, long length, double alpha, spBool unlock_flag);
/** @copydoc spDVector::xdvgausswinex()
 *  @public @memberof spFVector */
extern spFVector xfvgausswinex(long length, float alpha);
/** @~english
 *  Create a new vector whose contents are a Gaussian window of \p length .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  @public @memberof spDVector */
extern spDVector xdvgausswinex(long length, double alpha);
/** @copydoc spDVector::dvgausswin()
 *  @public @memberof spFVector */
extern void fvgausswin(spFVector vec);
/** @~english
 *  Create a Gaussian window for the vector \p vec .
 *  The standard deviation of the Gaussian is determined by `(vec->length - 1) / 5` .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvgausswin(spDVector vec);
/** @copydoc spDVector::xdvgausswinul()
 *  @public @memberof spFVector */
extern spFVector xfvgausswinul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Gaussian window of \p length .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / 5` .
 *  @public @memberof spDVector */
extern spDVector xdvgausswinul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvgausswin()
 *  @public @memberof spFVector */
extern spFVector xfvgausswin(long length);
/** @~english
 *  Create a new vector whose contents are a Gaussian window of \p length .
 *  The standard deviation of the Gaussian is determined by `(length - 1) / 5` .
 *  @public @memberof spDVector */
extern spDVector xdvgausswin(long length);
    
/** @copydoc spDVector::dvngausswinex()
 *  @public @memberof spFVector */
extern void fvngausswinex(spFVector vec, float alpha);
/** @~english
 *  Create a Gaussian window normalized by the RMS for the vector \p vec .
 *  The standard deviation of the Gaussian is determined by `(vec->length - 1) / (2 * alpha)` .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvngausswinex(spDVector vec, double alpha);
/** @copydoc spDVector::xdvngausswinexul()
 *  @public @memberof spFVector */
extern spFVector xfvngausswinexul(spPlugin *plugin, long length, float alpha, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Gaussian window of \p length normalized by the RMS.
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  @public @memberof spDVector */
extern spDVector xdvngausswinexul(spPlugin *plugin, long length, double alpha, spBool unlock_flag);
/** @copydoc spDVector::xdvngausswinex()
 *  @public @memberof spFVector */
extern spFVector xfvngausswinex(long length, float alpha);
/** @~english
 *  Create a new vector whose contents are a Gaussian window of \p length normalized by the RMS.
 *  The standard deviation of the Gaussian is determined by `(length - 1) / (2 * alpha)` .
 *  @public @memberof spDVector */
extern spDVector xdvngausswinex(long length, double alpha);
/** @copydoc spDVector::dvngausswin()
 *  @public @memberof spFVector */
extern void fvngausswin(spFVector vec);
/** @~english
 *  Create a Gaussian window normalized by the RMS for the vector \p vec .
 *  The standard deviation of the Gaussian is determined by `(vec->length - 1) / 5` .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvngausswin(spDVector vec);
/** @copydoc spDVector::xdvngausswinul()
 *  @public @memberof spFVector */
extern spFVector xfvngausswinul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a Gaussian window of \p length normalized by the RMS.
 *  The standard deviation of the Gaussian is determined by `(length - 1) / 5` .
 *  @public @memberof spDVector */
extern spDVector xdvngausswinul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvngausswin()
 *  @public @memberof spFVector */
extern spFVector xfvngausswin(long length);
/** @~english
 *  Create a new vector whose contents are a Gaussian window of \p length normalized by the RMS.
 *  The standard deviation of the Gaussian is determined by `(length - 1) / 5` .
 *  @public @memberof spDVector */
extern spDVector xdvngausswin(long length);
    
/** @~english @} */

/** @~english @name Rectangular Window for Vector (<sp/window.h>)
 *  @{ */

/** @copydoc spDVector::dvnboxcar()
 *  @public @memberof spFVector */
extern void fvnboxcar(spFVector vec);
/** @~english
 *  Create a rectangular window normalized by the RMS for the vector \p vec .
 *  The vector \p vec must be a memory-allocated vector. */
extern void dvnboxcar(spDVector vec);
/** @copydoc spDVector::xdvnboxcarul()
 *  @public @memberof spFVector */
extern spFVector xfvnboxcarul(spPlugin *plugin, long length, spBool unlock_flag);
/** @~english
 *  Create a new plugin-based vector whose contents are a rectangular window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnboxcarul(spPlugin *plugin, long length, spBool unlock_flag);
/** @copydoc spDVector::xdvnboxcar()
 *  @public @memberof spFVector */
extern spFVector xfvnboxcar(long length);
/** @~english
 *  Create a new vector whose contents are a rectangular window of \p length normalized by the RMS.
 *  @public @memberof spDVector */
extern spDVector xdvnboxcar(long length);

/** @~english @} */

/** @} */  /*----@addtogroup windowGroup----*/

#if defined(MACOS)
#pragma import off
#endif

#ifdef __cplusplus
}  /* Close scope of 'extern "C"' declaration */
#endif

#endif /* SPLIB_WINDOW_H */
