/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>
#include <sp/spGL.h>
#include <sp/spComponentMain.h>

#if defined(SP_GL_ES)
#undef USE_DISPLAY_LIST
#define GL_FRONT GL_FRONT_AND_BACK
#else
#define USE_DISPLAY_LIST
#endif

#if 0
#define CAGE_LOOK_AT_Z (2.0 * (CAGE_RADIUS))
#else
#define CAGE_LOOK_AT_Z 0.0
#endif
#define CAGE_ALPHA 0.2

#define CAGE_RADIUS 5.0
#define CAGE_AZIMUTH_NDIV 36
#define CAGE_ELEVATION_NDIV 14
#define CAGE_ELEVATION_MIN -40.0
#define CAGE_FOVY 90.0
#define CAGE_FLOOR_SIZE_FACTOR 25.0
#define CAGE_FLOOR_INTERVAL 2.0
#define CAGE_AZIMUTH_INCREMENT 10.0
#define CAGE_ELEVATION_INCREMENT 10.0

#if 0
static GLfloat light0pos[] = { -2.0 * CAGE_RADIUS, 2.0 * CAGE_RADIUS, -2.0 * CAGE_RADIUS, 1.0 };
static GLfloat light1pos[] = { 2.0 * CAGE_RADIUS, 2.0 * CAGE_RADIUS, -2.0 * CAGE_RADIUS, 1.0 };
#else
static GLfloat light0pos[] = { 0.0, 2.0 * CAGE_RADIUS, 0.0, 1.0 };
static GLfloat light1pos[] = { 0.0, 2.0 * CAGE_RADIUS, -2.0 * CAGE_RADIUS, 1.0 };
#endif

/*static GLfloat spot_pos[] = { 2.0, 3.0, -2.0, 1.0 };*/
static GLfloat spot_pos[] = { 0.0, CAGE_RADIUS, -1.5 * CAGE_RADIUS, 1.0 };
static GLfloat spot_dir[] = { 0.0, -1.0, 0.0 };
static GLfloat spot_exp = /*2.0*/2.0;
static GLfloat spot_cutoff = /*30.0*/40.0;


static GLfloat white[] = { 1.0, 1.0, 1.0, 1.0 };

static spGLContext context = NULL;
static spComponent g_canvas = NULL;

static double g_x_rotation = 0.0;
static double g_y_rotation = 0.0;
static double g_cube_azimuth = 0.0;
static double g_cube_elevation = 0.0;

GLdouble g_look_at_z = CAGE_LOOK_AT_Z;

static GLuint g_cage_id = 0;
static GLuint g_cube_id = 0;

static double createCage(double radius, int azimuth_ndiv, int elevation_ndiv, double elevation_min,
                         double wall_alpha, spBool line_flag, spBool draw_bottom);
void createCube(void);

void displayCube(double azimuth, double elevation)
{
    glPushMatrix();
    glRotatef(azimuth, 0.0, 1.0, 0.0);
    glRotatef(elevation, 1.0, 0.0, 0.0);
    
#if defined(USE_DISPLAY_LIST)
    glCallList(g_cube_id);
#else
    createCube();    
#endif
    
    glPopMatrix();
}

void display(spComponent component)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glPushMatrix();

    /* rotation */
    glRotatef(g_x_rotation, 1.0, 0.0, 0.0);
    glRotatef(g_y_rotation, 0.0, 1.0, 0.0);

    glLightfv(GL_LIGHT0, GL_POSITION, light0pos);
    glLightfv(GL_LIGHT1, GL_POSITION, light1pos);
    glLightfv(GL_LIGHT2, GL_POSITION, spot_pos);
    glLightfv(GL_LIGHT2, GL_SPOT_DIRECTION, spot_dir);

    displayCube(g_cube_azimuth, g_cube_elevation);

#if defined(USE_DISPLAY_LIST)
    glCallList(g_cage_id);
#else
    createCage(CAGE_RADIUS, CAGE_AZIMUTH_NDIV, CAGE_ELEVATION_NDIV, CAGE_ELEVATION_MIN, CAGE_ALPHA,
               SP_TRUE, SP_FALSE);
#endif

    glPopMatrix();

    spGLSwapBuffers(component);
}

void resize(int w, int h)
{
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(CAGE_FOVY, (double)w / (double)h, 1.0, 1000.0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(0.0, 0.0, g_look_at_z,
              0.0, 0.0, -CAGE_RADIUS,
              0.0, 1.0, 0.0);

    return;
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    spDebug(80, "mouse", "reason = %d, (x, y) = (%d, %d)\n", reason, x, y);
    
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            if (fabs(g_look_at_z) < CAGE_RADIUS) {
                g_y_rotation -= (double)(x - px);
                g_x_rotation -= (double)(y - py);
            } else {
                g_y_rotation += (double)(x - px);
                g_x_rotation += (double)(y - py);
            }
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE) {
        if (code == SPK_Escape) {
            spQuit(0);
        } else if (code == SPK_Left) {
            g_cube_azimuth += CAGE_AZIMUTH_INCREMENT;
            g_cube_azimuth = MIN(180.0, g_cube_azimuth);
        } else if (code == SPK_Right) {
            g_cube_azimuth -= CAGE_AZIMUTH_INCREMENT;
            g_cube_azimuth = MAX(-180.0, g_cube_azimuth);
        } else if (code == SPK_Up) {
            g_cube_elevation += CAGE_ELEVATION_INCREMENT;
            g_cube_elevation = MIN(90.0, g_cube_elevation);
        } else if (code == SPK_Down) {
            g_cube_elevation -= CAGE_ELEVATION_INCREMENT;
            g_cube_elevation = MAX(CAGE_ELEVATION_MIN, g_cube_elevation);
        }
        spRedrawCanvas(component);
    }
}

static GLfloat material_colors[][4][4] =
{
    /* 0: ambient, 1: diffuse, 2: specular, 3: shininess */
    {/* default */
        /*{1.0F,     1.0F,      1.0F,     1.0F},*/
        {0.55F,    0.55F,     0.55F,    1.0F},
        {0.8F,     0.8F,      0.8F,     1.0F},
        {0.4F,     0.4F,      0.4F,     1.0F},
        {50.0F,    0.0F,      0.0F,     0.0F},
    },
    {/* white */
        {0.2F,     0.2F,      0.2F,     1.0F},
        {1.0F,     1.0F,      1.0F,     1.0F},
        {0.0F,     0.0F,      0.0F,     1.0F},
        {0.0F,     0.0F,      0.0F,     0.0F},
    },
    {/* ruby */
        {0.1745F,   0.01175F,  0.01175F,   1.0F},
        {0.61424F,  0.04136F,  0.04136F,   1.0F},
        {0.727811F, 0.626959F, 0.626959F,  1.0F},
        {76.8F,     0.0F,      0.0F,     0.0F},
    },
};

enum {
    MATERIAL_DEFAULT_INDEX = 0,
    MATERIAL_WHITE_INDEX,
    MATERIAL_RUBY_INDEX,
    MATERIAL_NUM_INDEX,
};

static void setMaterialColor(int index)
{
    spDebug(10, "setMaterialColor", "index = %d\n", index);
    
    if (index >= MATERIAL_NUM_INDEX) {
        /*setMaterialColor(MATERIAL_WHITE_INDEX);*/
        setMaterialColor(MATERIAL_DEFAULT_INDEX);
        return;
    }
    
    glMaterialfv(GL_FRONT, GL_AMBIENT, material_colors[index][0]);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, material_colors[index][1]);
    glMaterialfv(GL_FRONT, GL_SPECULAR, material_colors[index][2]);
    glMaterialfv(GL_FRONT, GL_SHININESS, material_colors[index][3]);
    
    return;
}

static void createFloor(double x_length, double z_length, double x_interval, double z_interval, double y_pos)
{
    GLfloat floor_color[][4] = {
        /*{ 0.6F, 0.7F, 0.6F, 1.0F },*/
        { 0.7F, 0.7F, 0.6F, 1.0F },
        /*{ 0.3F, 0.3F, 0.3F, 1.0F },*/
        { 0.6F, 0.6F, 0.2F, 1.0F },
        /*{ 0.2F, 0.5F, 0.2F, 1.0F },*/
    };
    int i, j;
    int x_ndiv, z_ndiv;
    double x_pos, z_pos;
    int color_index;
    GLdouble v1[3], v2[3], v1p[3], v2p[3];

    setMaterialColor(MATERIAL_WHITE_INDEX);

    glBegin(GL_QUADS);

    glNormal3d(0.0, 1.0, 0.0);

    x_ndiv = (int)(x_length / x_interval);
    z_ndiv = (int)(z_length / z_interval);
    x_interval = x_length / (double)x_ndiv;
    z_interval = z_length / (double)z_ndiv;

    v1[1] = v1p[1] = v2[1] = v2p[1] = y_pos;
    
    x_pos = -x_length / 2.0;
    for (i = 0; i < x_ndiv; i++) {
        color_index = i & 1;
        
        v1[0] = v1p[0] = x_pos;
        v2[0] = v2p[0] = x_pos + x_interval;
        
        z_pos = -z_length / 2.0;
        for (j = 0; j < z_ndiv; j++) {
            
            glMaterialfv(GL_FRONT, GL_DIFFUSE, floor_color[color_index & 1]);

            v1[2] = z_pos;
            v2[2] = z_pos;

            if (j > 0) {
                glVertex3dv(v1p);
                glVertex3dv(v2p);
                glVertex3dv(v2);
                glVertex3dv(v1);
            }

            ++color_index;

            v1p[2] = v1[2];
            v2p[2] = v2[2];
            
            z_pos += z_interval;
        }
        x_pos += x_interval;
    }
    
    glEnd();
    
    return;
}

static double createCage(double radius, int azimuth_ndiv, int elevation_ndiv, double elevation_min,
                         double wall_alpha, spBool line_flag, spBool draw_bottom)
{
    GLfloat wall_color[][4] = {
        /*{ 0.6F, 0.6F, 0.6F, 0.4F },*/
        { 0.7F, 0.9F, 0.7F, 0.4F },
        /*{ 0.6F, 0.6F, 0.6F, 0.4F },*/
        /*{ 0.3F, 0.3F, 0.3F, 0.4F },*/
        { 0.2F, 0.6F, 0.2F, 0.4F },
        /*{ 0.3F, 0.3F, 0.3F, 0.4F },*/
    };
    int i, j, k;
    int color_index;
    double n_direction;
    double elevation_min_radian;
    double az_theta, az_dtheta, az_theta_n;
    double el_theta, el_dtheta, el_theta_n;
    double cos_el_theta, sin_el_theta;
    double cos_el_theta_n, sin_el_theta_n;
    double cos_el_theta_prev, sin_el_theta_prev;
    double cos_az_theta, sin_az_theta;
    double min_y_pos;
    GLdouble n[3];
    GLdouble v1[3], v2[3], v1p[3], v2p[3];
    
    if (wall_alpha >= 0.0 && wall_alpha <= 1.0) {
        wall_color[0][3] = (GLfloat)wall_alpha;
        wall_color[1][3] = (GLfloat)wall_alpha;
    }

    setMaterialColor(MATERIAL_WHITE_INDEX);

    az_dtheta = 2.0 * PI / (double)azimuth_ndiv;
    elevation_min_radian = PI * elevation_min / 180.0;
    el_dtheta = (PI / 2.0 - elevation_min_radian) / (double)elevation_ndiv;
    min_y_pos = 0.0;
    n_direction = /*-1.0*/1.0;

    for (k = 0; k <= 1; k++) {
        el_theta = elevation_min_radian;
        for (j = 0; j <= elevation_ndiv; j++) {
            cos_el_theta = cos(el_theta);
            sin_el_theta = sin(el_theta);
            if (k == 0 && j == 0) {
                min_y_pos = radius * sin_el_theta;
            }

            if (j == 0) {
                if (k == 0) {
                    createFloor(radius * CAGE_FLOOR_SIZE_FACTOR, radius * CAGE_FLOOR_SIZE_FACTOR,
                                CAGE_FLOOR_INTERVAL, CAGE_FLOOR_INTERVAL, min_y_pos);
                    
                    glEnable(GL_BLEND);
                    glDepthMask(GL_FALSE);
                    /*glBlendFunc(GL_SRC_ALPHA, GL_ONE);*/
                    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
                }

                if (draw_bottom == SP_TRUE) {
                    n[0] = n[2] = 0.0;
                    n[1] = 1.0;
            
                    v2[0] = 0.0;
                    v2[2] = 0.0;
                    v2[1] = radius * sin_el_theta;

                    if (k == 0) {
                        glBegin(GL_TRIANGLES);
                    } else {
                        glBegin(GL_LINES);
                    }
                    glNormal3dv(n);
                    v1[1] = v1p[1] = v2[1];

                    color_index = 1;
                    az_theta = 0.0;
                    for (i = 0; i <= azimuth_ndiv; i++) {
                        cos_az_theta = cos(az_theta);
                        sin_az_theta = sin(az_theta);
                
                        v1[0] = radius * cos_az_theta * cos_el_theta;
                        v1[2] = radius * sin_az_theta * cos_el_theta;

                        if (i > 0) {
                            if (k == 0) {
                                glMaterialfv(GL_FRONT, GL_DIFFUSE, wall_color[color_index & 1]);
                                glVertex3dv(v2);
                                glVertex3dv(v1p);
                                glVertex3dv(v1);
                            } else {
                                glVertex3dv(v2);
                                glVertex3dv(v1p);
                                glVertex3dv(v1p);
                                glVertex3dv(v1);
                            }
                        }

                        v1p[0] = v1[0];
                        v1p[2] = v1[2];
                
                        az_theta += az_dtheta;
                        ++color_index;
                    }
                    glEnd();
                }
            
                if (k == 0) {
                    glBegin(GL_QUADS);
                } else {
                    glBegin(GL_LINES);
                }
            } else if (j > 0) {
                color_index = j & 1;
        
                el_theta_n = el_theta - el_dtheta / 2.0;
                cos_el_theta_n = cos(el_theta_n);
                sin_el_theta_n = sin(el_theta_n);
                n[1] = n_direction * sin_el_theta_n;
                v1[1] = v1p[1] = radius * sin_el_theta_prev;
                if (j == elevation_ndiv) {
                    v2[1] = v2p[1] = radius;
                } else {
                    v2[1] = v2p[1] = radius * sin_el_theta;
                }
        
                az_theta = 0.0;
                for (i = 0; i <= azimuth_ndiv; i++) {
                    cos_az_theta = cos(az_theta);
                    sin_az_theta = sin(az_theta);

                    v1[0] = radius * cos_az_theta * cos_el_theta_prev;
                    v1[2] = radius * sin_az_theta * cos_el_theta_prev;
                    if (j == elevation_ndiv) {
                        v2[0] = v2[2] = 0.0;
                    } else {
                        v2[0] = radius * cos_az_theta * cos_el_theta;
                        v2[2] = radius * sin_az_theta * cos_el_theta;
                    }
                
                    if (i > 0) {
                        az_theta_n = az_theta - az_dtheta / 2.0;

                        n[0] = n_direction * cos(az_theta_n) * cos_el_theta_n;
                        n[2] = n_direction * sin(az_theta_n) * cos_el_theta_n;
                        glNormal3dv(n);

                        if (k == 0) {
                            glMaterialfv(GL_FRONT, GL_DIFFUSE, wall_color[color_index & 1]);
                        } else {
                            glVertex3dv(v1p);
                            glVertex3dv(v1);
                        }
                        glVertex3dv(v1p);
                        glVertex3dv(v2p);
                        glVertex3dv(v2);
                        glVertex3dv(v1);
                    }
            
                    v1p[0] = v1[0];
                    v1p[2] = v1[2];
                    v2p[0] = v2[0];
                    v2p[2] = v2[2];
                
                    az_theta += az_dtheta;
                    ++color_index;
                }
            }

            cos_el_theta_prev = cos_el_theta;
            sin_el_theta_prev = sin_el_theta;

            el_theta += el_dtheta;
        }
    
        glEnd();

        if (k == 0) {
            glDepthMask(GL_TRUE);
            glDisable(GL_BLEND);
        }
        if (line_flag == SP_FALSE) {
            break;
        }
        setMaterialColor(MATERIAL_WHITE_INDEX);
    }
    
    return min_y_pos;
}

#if defined(USE_DISPLAY_LIST)
static GLuint newCage(double radius, int azimuth_ndiv, int elevation_ndiv, double elevation_min, double alpha)
{
    GLuint new_id;

    new_id = glGenLists(1);
    
    glNewList(new_id, GL_COMPILE);

    createCage(radius, azimuth_ndiv, elevation_ndiv, elevation_min, alpha, SP_TRUE, SP_FALSE);

    glEndList();
    
    return new_id;
}
#endif

void createCube(void)
{
    int i, j;
    static GLdouble vertex[][3] = {
        { -0.5, -0.5, -0.5 },
        { 0.5, -0.5, -0.5 },
        { 0.5, 0.5, -0.5 },
        { -0.5, 0.5, -0.5 },
        { -0.5, -0.5, 0.5 },
        { 0.5, -0.5, 0.5 },
        { 0.5, 0.5, 0.5 },
        { -0.5, 0.5, 0.5 }
    };
    static int face[][4] = {
        { 0, 1, 2, 3 },
        { 1, 5, 6, 2 },
        { 5, 4, 7, 6 },
        { 4, 0, 3, 7 },
        { 4, 5, 1, 0 },
        { 3, 2, 6, 7 }
    };
    static GLdouble normal[][3] = {
        { 0.0, 0.0,-1.0 },
        { 1.0, 0.0, 0.0 },
        { 0.0, 0.0, 1.0 },
        {-1.0, 0.0, 0.0 },
        { 0.0,-1.0, 0.0 },
        { 0.0, 1.0, 0.0 }
    };

    glPushMatrix();
    glTranslatef(0.0, 0.0, -CAGE_RADIUS);
    
    setMaterialColor(MATERIAL_RUBY_INDEX);
    glBegin(GL_QUADS);
    for (j = 0; j < 6; ++j) {
        glNormal3dv(normal[j]);
        for (i = 0; i < 4; ++i) {
            glVertex3dv(vertex[face[j][i]]);
        }
    }
    glEnd();

    glPopMatrix();
    
    return;
}

#if defined(USE_DISPLAY_LIST)
static GLuint newCube()
{
    GLuint new_id;

    new_id = glGenLists(1);
    
    glNewList(new_id, GL_COMPILE);

    createCube();

    glEndList();
    
    return new_id;
}
#endif

void init(void)
{
#if 1
    /*glClearColor(1.0F, 1.0F, 1.0F, 1.0F);*/
    glClearColor(0.6F, 0.6F, 1.0F, 1.0F);
#else
    /* black background */
    glClearColor(0.0F, 0.0F, 0.0F, 1.0F);
#endif

    glEnable(GL_DEPTH_TEST);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glEnable(GL_LIGHT2);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT0, GL_SPECULAR, white);
    glLightfv(GL_LIGHT1, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT1, GL_SPECULAR, white);
    
    glLightfv(GL_LIGHT2, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT2, GL_SPECULAR, white);
    glLightf(GL_LIGHT2, GL_SPOT_EXPONENT, spot_exp);
    glLightf(GL_LIGHT2, GL_SPOT_CUTOFF, spot_cutoff);
    glLightfv(GL_LIGHT2, GL_SPOT_DIRECTION, spot_dir);

#if defined(USE_DISPLAY_LIST)
    g_cube_id = newCube();
    g_cage_id = newCage(CAGE_RADIUS, CAGE_AZIMUTH_NDIV, CAGE_ELEVATION_NDIV, CAGE_ELEVATION_MIN, CAGE_ALPHA);
#endif
    
    spDebug(10, "init", "done\n");
}

void canvasCB(spComponent component, void *data)
{
    static int w = 0, h = 0;

    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || w == 0) {
        if (w == 0) {
          init();
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent window, canvas;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_DEPTH_SIZE, 1,
                                SP_GL_NONE };
  
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Cage", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);
  
    context = spCreateGLContext(canvas, NULL);

    spPopupWindow(window);
  
    spSetGLContext(canvas, context);
    g_canvas = canvas;
  
    return spMainLoop(toplevel);
}
