/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>

#include <sp/spGL.h>
#include <sp/spGLString.h>
#include <sp/spGLPhongModel.h>

#include <sp/spComponentMain.h>

#define USE_SHADER
#define DISABLE_TIMER

//#define USE_LEGACY_FUNCTION
//#define USE_TRIANGLES_VERTICES
//#define USE_SPOT_LIGHT
#define USE_PHONG_SHADING
//#define USE_FLAT_SHADING
//#define USE_HALFWAY_VECTOR
//#define USE_CLIP_PLANE

#define MULTISAMPLE_SAMPLES /*4*/0

#if MULTISAMPLE_SAMPLES >= 2
#define USE_MULTISAMPLE
#else
#undef USE_MULTISAMPLE
#endif

#define FREQUENCY_STRING_COLOR /*"white"*/"blue"
#define MAGNITUDE_STRING_COLOR /*"white"*/"red"
#define TIME_STRING_COLOR /*"white"*/"green"

#if defined(USE_SHADER)
#if defined(ANDROID) || defined(IPHONE)
#define SPECGRAM_GLSL_VERSION (SP_GL_PHONG_MODEL_SHADER_VERSION_ES|100L)
#define CONTEXT_MAJOR_VERSION 2
#elif defined(linux)
#define SPECGRAM_GLSL_VERSION 120L
#define CONTEXT_MAJOR_VERSION 2
#else
#define SPECGRAM_GLSL_VERSION /*150L*/140L
#define CONTEXT_MAJOR_VERSION 3
#endif
#elif !defined(USE_LEGACY_FUNCTION)
#define CONTEXT_MAJOR_VERSION 2
#else
#define CONTEXT_MAJOR_VERSION 1
#endif

#if defined(GTK3)
#if !defined(USE_SHADER)
#define USE_SHADER
#endif
#if CONTEXT_MAJOR_VERSION <= 2
#undef CONTEXT_MAJOR_VERSION
#define CONTEXT_MAJOR_VERSION 3
#endif
#if SPECGRAM_GLSL_VERSION < 130L
#undef SPECGRAM_GLSL_VERSION
#define SPECGRAM_GLSL_VERSION 130L
#endif
#endif /* defined(GTK3) */

#if (CONTEXT_MAJOR_VERSION >= 3)
#define USE_VAO
#define USE_VBO
#endif

#if defined(COCOA)
#if (CONTEXT_MAJOR_VERSION >= 3)
#if !defined(USE_VAO)
#define USE_VAO /* VAO must be used for Cocoa of context version 3+. */
#endif
#else /* (CONTEXT_MAJOR_VERSION >= 3) */
#undef USE_VAO
#endif /* !(CONTEXT_MAJOR_VERSION >= 3) */
#endif /* defined(COCOA) */

#if defined(ANDROID) || defined(IPHONE)
#if !defined(USE_LEGACY_FUNCTION) && (!defined(IPHONE) || CONTEXT_MAJOR_VERSION < 2)
#undef USE_CLIP_PLANE /* ES shader doesn't support clip plane */
#endif
#endif

#if defined(USE_SHADER)
#define USE_SHADER_FLAG SP_TRUE
#else
#define USE_SHADER_FLAG SP_FALSE
#endif

static spGLContext context = NULL;
static spComponent g_canvas = NULL;
static spGLPhongModel glpm = NULL;
static spGLString freq_string = NULL;
static spGLString amp_string = NULL;
static spGLString time_string = NULL;

static spBool init_done = SP_FALSE;

//static GLfloat light0pos[] = { 3.0f, 3.0f, -5.0f, 1.0f };
//static GLfloat light0pos[] = { 0.375f, 1.2f, -0.5f, 1.0f };
//static GLfloat light0pos[] = { 0.375f, 2.0f, -1.0f, 1.0f };
static GLfloat light0pos[] = { -0.125f, 0.7f, 0.0f, 1.0f };
static GLfloat light0_spot_direction[] = { 0.0f, -1.0f, 0.0f };
static GLfloat light0_attenuation[] = { 1.0f, 0.02f, 0.01f };
static GLfloat light1pos[] = { -2.0f, 7.0f, 3.0f, 1.0f };
static GLfloat light1_spot_direction[] = { 2.0f, -7.0f, -3.0f };
static GLfloat light1_attenuation[] = { 1.0f, 0.02f, 0.01f };

static GLfloat spot_cutoff = 30.0f;
static GLfloat spot_exponent = 15.0f;

static GLfloat black[] = { 0.0f, 0.0f, 0.0f, 1.0f };
static GLfloat white[] = { 1.0f, 1.0f, 1.0f, 1.0f };

#if defined(USE_CLIP_PLANE)
static GLfloat clip_plane_x[] = { -1.0f, 0.0f, 0.0f, /*0.5f*/0.3f };
static GLfloat clip_plane_y[] = { 0.0f, 1.0f, 0.0f, /*-0.5f*/-0.2f };
static GLfloat clip_plane_z[] = { 0.0f, 0.0f, 1.0f, 0.5f/*0.1f*/ };
#endif

GLfloat **g_specgram = NULL;
GLfloat ***g_normal = NULL;

typedef struct _SpecgramVertex {
    GLfloat position[3];
    GLfloat normal[3];
} SpecgramVertex;

SpecgramVertex *g_specgram_vertices = NULL;
#if !defined(USE_TRIANGLES_VERTICES)
GLushort *g_vertex_indices = NULL;
GLsizei g_vertex_indices_length = 0;
#endif
#if defined(USE_VAO)
static GLuint g_axes_vao = 0;
static GLuint g_specgram_vao = 0;
#endif
#if defined(USE_VBO) || defined(USE_VAO)
static GLuint g_axes_position_vbo = 0;
static GLuint g_axes_normal_vbo = 0;
static GLuint g_specgram_vbo = 0;
static GLuint g_vertex_indices_vbo = 0;
#endif

static float g_specgram_range = 100.0f;

static float g_x_rotation = 0.0f;
static float g_y_rotation = 0.0f;

static float g_arrow_size = 0.055f;

#if 0
static long g_hfftl = 513;
static float g_total_ms = 500.0f;
static float g_shift_ms = 8.0f;
static float g_samp_freq = 16000.0f;
#else
static long g_hfftl = 127;
static float g_total_ms = 160.0f;
static float g_shift_ms = 16.0f;
static float g_samp_freq = 8000.0f;
#endif

static long g_nframes = /*1000*//*100*/0;
static float g_spec_freq = 2000.0f;
static float g_amp_mod_freq = 3.0f;
static float g_amp_mod_amp = 2.0f;
static float g_specgram_min = -40.0f;
static float g_specgram_max = 20.0f;

#if !defined(DISABLE_TIMER)
static spTimerId g_timer_id = SP_TIMER_ID_NONE;
static long g_timer_interval = /*16*/200;
#endif

static spGLMat4 g_P;
static spGLMat4 g_V;
static spGLMat4 g_Rx;
static spGLMat4 g_Ry;
static spGLMat4 g_T;
static spGLMat4 g_M;
static spGLMat4 g_MV;

GLfloat g_axes_position_array[][3] =
{
    {0.0f, 0.0f, 0.0f},
    {1.2f, 0.0f, 0.0f},
        
    {1.2f, 0.0f, 0.0f},
    {0.0f, 0.0f, 0.0f},

    {1.2f, 0.0f, 0.0f},
    {0.0f, 0.0f, 0.0f},
        

    {0.0f, 0.0f, 0.0f},
    {0.0f, 1.2f, 0.0f},
        
    {0.0f, 1.2f, 0.0f},
    {0.0f, 0.0f, 0.0f},

    {0.0f, 1.2f, 0.0f},
    {0.0f, 0.0f, 0.0f},

        
    {0.0f, 0.0f, 0.0f},
    {0.0f, 0.0f, -1.2f},
        
    {0.0f, 0.0f, -1.2f},
    {0.0f, 0.0f, 0.0f},
        
    {0.0f, 0.0f, -1.2f},
    {0.0f, 0.0f, 0.0f},
};

GLfloat g_axes_normal_array[][3] =
{
    {1.0f, 0.0f, 0.0f},
    {1.0f, 0.0f, 0.0f},
        
    {1.0f, 0.0f, 0.0f},
    {1.0f, 0.0f, 0.0f},
        
    {1.0f, 0.0f, 0.0f},
    {1.0f, 0.0f, 0.0f},


    {0.0f, 1.0f, 0.0f},
    {0.0f, 1.0f, 0.0f},

    {0.0f, 1.0f, 0.0f},
    {0.0f, 1.0f, 0.0f},
        
    {0.0f, 1.0f, 0.0f},
    {0.0f, 1.0f, 0.0f},

        
    {0.0f, 0.0f, -1.0f},
    {0.0f, 0.0f, -1.0f},
        
    {0.0f, 0.0f, -1.0f},
    {0.0f, 0.0f, -1.0f},
        
    {0.0f, 0.0f, -1.0f},
    {0.0f, 0.0f, -1.0f},
};

spGLMat4 material_colors[] =
{
    /* 0: ambient, 1: diffuse, 2: specular, 3: shininess */
    {{/* default */
        {0.2f,     0.2f,      0.2f,     1.0f},
        {0.8f,     0.8f,      0.8f,     1.0f},
        {0.0f,     0.0f,      0.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* white */
        {0.2f,     0.2f,      0.2f,     1.0f},
        {1.0f,     1.0f,      1.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* emerald */
        {0.0215f,  0.1745f,   0.0215f,  1.0f},
        {0.07568f, 0.61424f,  0.07568f, 1.0f},
        {0.633f,   0.727811f, 0.633f,   1.0f},
        {76.8f,    0.0f,      0.0f,     0.0f},
    }},
    {{/* jade */
        {0.135f,     0.2225f,   0.1575f,   1.0f},
        {0.54f,      0.89f,     0.63f,     1.0f},
        {0.316228f,  0.316228f, 0.316228f, 1.0f},
        {12.8f,      0.0f,      0.0f,     0.0f},
    }},
    {{/* obsidian */
        {0.05375f, 0.05f,    0.06625f, 1.0f},
        {0.18275f, 0.17f,    0.22525f, 1.0f},
        {0.332741f,0.328634f,0.346435f,1.0f},
        {38.4f,    0.0f,     0.0f,     0.0f},
    }},
    {{/* pearl */
        {0.25f,     0.20725f,  0.20725f,  1.0f},
        {1.0f,      0.829f,    0.829f,    1.0f},
        {0.296648f, 0.296648f, 0.296648f, 1.0f},
        {10.24f,    0.0f,      0.0f,     0.0f},
    }},
    {{/* ruby */
        {0.1745f,   0.01175f,  0.01175f,   1.0f},
        {0.61424f,  0.04136f,  0.04136f,   1.0f},
        {0.727811f, 0.626959f, 0.626959f,  1.0f},
        {76.8f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* turquoise */
        {0.1f,     0.18725f, 0.1745f,  1.0f},
        {0.396f,   0.74151f, 0.69102f, 1.0f},
        {0.297254f,0.30829f, 0.306678f,1.0f},
        {12.8f,    0.0f,     0.0f,     0.0f},
    }},
    {{/* brass */
        {0.329412f,  0.223529f, 0.027451f, 1.0f},
        {0.780392f,  0.568627f, 0.113725f, 1.0f},
        {0.992157f,  0.941176f, 0.807843f, 1.0f},
        {25.6f,      0.0f,      0.0f,     0.0f},
    }},
    {{/* bronze */
        {0.2125f,   0.1275f,   0.054f,   1.0f},
        {0.714f,    0.4284f,   0.18144f, 1.0f},
        {0.393548f, 0.271906f, 0.166721f,1.0f},
        {25.6f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* chrome */
        {0.25f,    0.25f,     0.25f,     1.0f},
        {0.4f,     0.4f,      0.4f,      1.0f},
        {0.774597f,0.774597f, 0.774597f, 1.0f},
        {76.8f,    0.0f,      0.0f,     0.0f},
    }},
    {{/* copper */
        {0.19125f,  0.0735f,   0.0225f,  1.0f},
        {0.7038f,   0.27048f,  0.0828f,  1.0f},
        {0.256777f, 0.137622f, 0.086014f,1.0f},
        {12.8f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* gold */
        {0.24725f,  0.1995f,   0.0745f,    1.0f},
        {0.75164f,  0.60648f,  0.22648f,   1.0f},
        {0.628281f, 0.555802f, 0.366065f,  1.0f},
        {51.2f,     0.0f,      0.0f,     0.0f},
    }},
    {{/* silver */
        {0.19225f,  0.19225f,  0.19225f, 1.0f},
        {0.50754f,  0.50754f,  0.50754f, 1.0f},
        {0.508273f, 0.508273f, 0.508273f,1.0f},
        {51.2f,     0.0f,      0.0f,     0.0f},
    }},
};

enum {
    MATERIAL_DEFAULT_INDEX = 0,
    MATERIAL_WHITE_INDEX,
    MATERIAL_EMERALD_INDEX,
    MATERIAL_JADE_INDEX,
    MATERIAL_OBSIDIAN_INDEX,
    MATERIAL_PEARL_INDEX,
    MATERIAL_RUBY_INDEX,
    MATERIAL_TURQUOISE_INDEX,
    MATERIAL_BRASS_INDEX,
    MATERIAL_BRONZE_INDEX,
    MATERIAL_CHROME_INDEX,
    MATERIAL_COPPER_INDEX,
    MATERIAL_GOLD_INDEX,
    MATERIAL_SILVER_INDEX,
};

static int g_material_index = MATERIAL_GOLD_INDEX;

static GLfloat **allocPseudoSpecgram(long nframes, long hfftl)
{
    return xspFMatAlloc(nframes, hfftl);
}

static void freePseudoSpecgram(GLfloat **specgram, long nframes)
{
    xspFMatFree(specgram, nframes);
}

static GLfloat ***allocSpecgramNormal(long nframes, long hfftl)
{
    long k, l;
    long hfftl2;
    long nnframes;
    GLfloat ***normal;

    hfftl2 = hfftl * 2;

    nnframes = nframes - 1;
    normal = xspAlloc(nnframes, GLfloat **);
    for (k = 0; k < nnframes; k++) {
        normal[k] = xspAlloc(hfftl2, GLfloat *);
        normal[k][0] = xspAlloc(hfftl2 * 3, GLfloat);
        for (l = 1; l < hfftl2; l++) {
            normal[k][l] = normal[k][0] + l * 3;
        }
    }

    return normal;
}

static void freeSpecgramNormal(GLfloat ***normal, long nframes, long hfftl)
{
    long k;
    
    for (k = 0; k < nframes - 1; k++) {
        xspFree(normal[k][0]);
        xspFree(normal[k]);
    }
    xspFree(normal);
}

static SpecgramVertex *allocSpecgramVertices(long nframes, long hfftl)
{
    long length;
    SpecgramVertex *vertices;

#if defined(USE_TRIANGLES_VERTICES)
    length = 3 * 2 * (hfftl - 1) * (nframes - 1);
#else
    length = nframes * hfftl;
#endif
    spDebug(80, "allocSpecgramVertices", "nframes = %ld, hfftl = %ld, length = %ld\n", nframes, hfftl, length);

    vertices = xspAlloc(length, struct _SpecgramVertex);

    return vertices;
}

#if !defined(USE_TRIANGLES_VERTICES)
static GLushort *createVertexIndices(long nframes, long hfftl, GLsizei *olength)
{
    long k, l;
    long length;
    long pos;
    long offset, prev_offset;
    GLushort *indices;

    length = 2 * hfftl + (2 + 2 * hfftl) * (nframes - 2);

    indices = xspAlloc(length, GLushort);

    pos = 0;
    for (k = 1;; k++) {
        if (pos > 0) {
            indices[pos++] = (GLushort)((k - 1) * hfftl);
        }
        for (l = 0; l < hfftl; l++) {
            offset = k * hfftl + l;
            prev_offset = (offset - hfftl);
            indices[pos++] = (GLushort)prev_offset;
            indices[pos++] = (GLushort)offset;
        }

        if (k == nframes - 1) {
            break;
        }
        indices[pos] = indices[pos - 1];
        pos++;
    }

    spDebug(80, "createVertexIndices", "length = %ld, pos = %ld, indices[%ld] = %d\n", length, pos, pos-1, indices[pos-1]);

    *olength = (GLsizei)pos;
    
    return indices;
}
#endif

static void calculatePseudoSpecgram(GLfloat **specgram, long nframes, long hfftl, 
                                    float fs, float shift_ms,
                                    float min, float max,
                                    float spec_freq,
                                    float amp_mod_amp,
                                    float amp_mod_freq,
                                    long offset)
{
    long k, l;
    float hrange;
    float amp_bias;
    float fweight, aweight;
    static long amp_mod_index = 0;

    hrange = (max - min) / 2.0f;

    fweight = ((float)PI * fs) / (spec_freq * (float)(hfftl - 1));
    aweight = 2.0f * (float)PI * (shift_ms * amp_mod_freq / 1000.0f);
    
    for (k = offset; k < nframes; k++) {
        if (amp_mod_amp != 0.0) {
            amp_bias = amp_mod_amp * sinf(aweight * (float)amp_mod_index);
            amp_mod_index++;
        } else {
            amp_bias = 0.0;
        }
        for (l = 0; l < hfftl; l++) {
            specgram[k][l] = hrange * sinf(fweight * (float)l) + amp_bias;
        }
    }
    
    return;
}

static void updatePseudoSpecgram(GLfloat **specgram, long nframes, long hfftl, 
                                 float fs, float shift_ms,
                                 float min, float max,
                                 float spec_freq,
                                 float amp_mod_amp,
                                 float amp_mod_freq,
                                 long shift)
{
    long offset;

    offset = nframes - shift;

    if (shift > 0) {
        /* shift current spectrogram */
        memmove(specgram[0], specgram[shift], sizeof(GLfloat) * hfftl * offset);
    }
    
    /* calculate new spectrogram */
    calculatePseudoSpecgram(specgram, nframes, hfftl,
                            fs, shift_ms,
                            min, max,
                            spec_freq,
                            amp_mod_amp, amp_mod_freq,
                            offset);
    
    return;
}

static void calculateNormalizedProduct(float ax, float ay, float az,
                                       float bx, float by, float bz,
                                       float *ppx, float *ppy, float *ppz)
{
    double denom;
    float normrecip;
    float px, py, pz;
    
    px = ay * bz - az * by;
    py = az * bx - ax * bz;
    pz = ax * by - ay * bx;

    denom = sqrt((double)(px * px + py * py + pz * pz));
    if (denom == 0.0) {
        normrecip = 0.0f;
    } else {
        normrecip = (float)(1.0 / denom);
    }
    
    *ppx = px * normrecip;
    *ppy = py * normrecip;
    *ppz = pz * normrecip;
}

static void calculateNormal(GLfloat *normal1, GLfloat *normal2, GLfloat **specgram, long nframes, long hfftl,
                            long frame, long fn)
{
    float fx, fy, fz;
    float tx, ty, tz;
    float px, py, pz;

    fx = 1.0;
    fy = specgram[frame][fn + 1] - specgram[frame][fn];
    fz = 0.0;
            
    tx = 0.0;
    ty = specgram[frame + 1][fn] - specgram[frame][fn];
    tz = -1.0;
            
    calculateNormalizedProduct(fx, fy, fz, tx, ty, tz, &px, &py, &pz);
    /*spDebug(-100, "calculateNormal", "1: px = %f, py = %f, pz = %f\n", px, py, pz);*/
    normal1[0] = px;
    normal1[1] = py;
    normal1[2] = pz;
#if 0
    spDebug(-100, "calculateNormal", "1: norm = %f\n", sqrt(px * px + py * py + pz * pz));
#endif

    fx = -fx;
    fy = specgram[frame + 1][fn] - specgram[frame + 1][fn + 1];
            
    ty = specgram[frame][fn + 1] - specgram[frame + 1][fn + 1];
    tz = -tz;
    calculateNormalizedProduct(fx, fy, fz, tx, ty, tz, &px, &py, &pz);
    /*spDebug(-100, "calculateNormal", "2: px = %f, py = %f, pz = %f\n", px, py, pz);*/
    normal2[0] = px;
    normal2[1] = py;
    normal2[2] = pz;
#if 0
    spDebug(-100, "calculateNormal", "2: norm = %f\n", sqrt(px * px + py * py + pz * pz));
#endif

    return;
}

static void calculateSpecgramNormal(GLfloat ***normal, GLfloat **specgram, long nframes, long hfftl,
                                    long offset)
{
    long k, l;

    for (k = offset + 1; k < nframes; k++) {
        for (l = 1; l < hfftl; l++) {
            calculateNormal(normal[k - 1][(l - 1) * 2], normal[k - 1][(l - 1) * 2 + 1],
                            specgram, nframes, hfftl, k - 1, l - 1);
        }
    }

    return;
}

static void updateSpecgramNormal(GLfloat ***normal, GLfloat **specgram, long nframes, long hfftl,
                                 long shift)
{
    long offset;
    long k;
    long hfftl2;

    hfftl2 = hfftl * 2;
    offset = nframes - 1 - shift;

    /* shift current normal vectors */
    for (k = shift; k < nframes - 1; k++) {
        memmove(normal[k - shift][0], normal[k][0], sizeof(GLfloat) * hfftl2 * 3);
    }
    
    /* calculate new normal vectors */
    calculateSpecgramNormal(normal, specgram, nframes, hfftl, offset);

    return;
}

#if !defined(USE_TRIANGLES_VERTICES)
void calcGouraudVertexNormal(GLfloat vnormal[3], 
                             GLfloat ***normal, long nframes, long hfftl,
                             long frame, long frame_diff, long nfn)
{
    long k, l;
    long nfn2;
    double denom;
    GLfloat normrecip;
    
    vnormal[0] = vnormal[1] = vnormal[2] = 0.0;
    
    for (k = frame_diff + 1; k >= frame_diff; k--) {
        if (frame - k >= 0 && frame - k < nframes - 1) {
            for (l = -1; l <= 1; l++) {
                nfn2 = nfn + l - (frame_diff + 1) + k;
                if (nfn2 >= 0 && nfn2 < (hfftl - 1) * 2) {
                    vnormal[0] += normal[frame - k][nfn2][0];
                    vnormal[1] += normal[frame - k][nfn2][1];
                    vnormal[2] += normal[frame - k][nfn2][2];
                }
            }
        }
    }

    denom = sqrt((double)(SQUARE(vnormal[0]) + SQUARE(vnormal[1]) + SQUARE(vnormal[2])));
    if (denom == 0.0) {
        normrecip = 0.0f;
    } else {
        normrecip = (GLfloat)(1.0 / denom);
    }
    vnormal[0] *= normrecip; vnormal[1] *= normrecip; vnormal[2] *= normrecip;
    
    return;
}
#endif

static void getCurrentSpecVertex(GLfloat *v, GLfloat **specgram, long nframes, long hfftl,
                                 float specgram_max, float specgram_range,
                                 long frame, long fn)
{
    v[0] = (GLfloat)fn / (GLfloat)/*hfftl*/(hfftl - 1);
    v[1] = (specgram[frame][fn] - (specgram_max - specgram_range)) / specgram_range;
    v[2] = -((GLfloat)frame / (GLfloat)/*nframes*/(nframes - 1));
    
    return;
}

static void calculateSpecgramVertices(SpecgramVertex *vertices, GLfloat **specgram, GLfloat ***normal, 
                                      long nframes, long hfftl,
                                      float specgram_max, float specgram_range, long offset)
{
    long k, l;
    long voffset = 0;

#if defined(USE_TRIANGLES_VERTICES)
    for (k = offset; k < nframes - 1; k++) {
        for (l = 0; l < hfftl - 1; l++) {
            voffset = 3 * 2 * ((hfftl - 1) * k + l);
            getCurrentSpecVertex(vertices[voffset].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k, l);
            getCurrentSpecVertex(vertices[voffset+1].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k + 1, l);
            getCurrentSpecVertex(vertices[voffset+2].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k, l + 1);
            memcpy(vertices[voffset+0].normal, normal[k][2 * l], 3 * sizeof(GLfloat));
            memcpy(vertices[voffset+1].normal, normal[k][2 * l], 3 * sizeof(GLfloat));
            memcpy(vertices[voffset+2].normal, normal[k][2 * l], 3 * sizeof(GLfloat));
            
            getCurrentSpecVertex(vertices[voffset+3].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k + 1, l);
            getCurrentSpecVertex(vertices[voffset+4].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k + 1, l + 1);
            getCurrentSpecVertex(vertices[voffset+5].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k, l + 1);
            memcpy(vertices[voffset+3].normal, normal[k][2 * l + 1], 3 * sizeof(GLfloat));
            memcpy(vertices[voffset+4].normal, normal[k][2 * l + 1], 3 * sizeof(GLfloat));
            memcpy(vertices[voffset+5].normal, normal[k][2 * l + 1], 3 * sizeof(GLfloat));
        }
    }
    spDebug(10, "calculateSpecgramVertices", "voffset = %ld, k = %ld\n", voffset, k);
#else
    for (k = offset; k < nframes; k++) {
        for (l = 0; l < hfftl; l++) {
            voffset = k * hfftl + l;
            getCurrentSpecVertex(vertices[voffset].position, specgram, nframes, hfftl,
                                 specgram_max, specgram_range, k, l);

            calcGouraudVertexNormal(vertices[voffset].normal, normal, nframes, hfftl, k, 0, l * 2);
        }
    }
#endif
    
    return;
}

static void updateSpecgramVertices(SpecgramVertex *vertices, GLfloat **specgram, GLfloat ***normal, 
                                   long nframes, long hfftl,
                                   float specgram_max, float specgram_range, long shift)
{
    calculateSpecgramVertices(vertices, specgram, normal, nframes, hfftl,
                              specgram_max, specgram_range, 0);
    
    return;
}

static void setMaterialColor(int index)
{
    spGLPhongModelSetMaterialMatrix(glpm, &material_colors[index]);
    spGLPhongModelUpdateMaterialMatrix(glpm);
    
    return;
}

static void updateSpecgram(float **specgram, GLfloat ***normal, long nframes, long hfftl,
                           float fs, float shift_ms,
                           float min, float max, float specgram_range,
                           float spec_freq,
                           float amp_mod_amp,
                           float amp_mod_freq,
                           int material_index,
                           long shift)
{
    updatePseudoSpecgram(specgram, nframes, hfftl,
                         fs, shift_ms,
                         min, max,
                         spec_freq,
                         amp_mod_amp, amp_mod_freq,
                         shift);

    if (shift > 0) {
        updateSpecgramNormal(normal, specgram, nframes, hfftl, shift);
    }
    
    return;
}

static void createAxesArray(void)
{
    float arrow_size2;
    
    arrow_size2 = g_arrow_size / 2.0f;
    
    g_axes_position_array[3][0] = 1.2f - g_arrow_size;
    g_axes_position_array[3][1] = arrow_size2;
    g_axes_position_array[5][0] = 1.2f - g_arrow_size;
    g_axes_position_array[5][1] = -arrow_size2;
    
    g_axes_position_array[9][0] = arrow_size2;
    g_axes_position_array[9][1] = 1.2f - g_arrow_size;
    g_axes_position_array[11][0] = -arrow_size2;
    g_axes_position_array[11][1] = 1.2f - g_arrow_size;

    g_axes_position_array[15][1] = arrow_size2;
    g_axes_position_array[15][2] = -1.2f + g_arrow_size;
    g_axes_position_array[17][1] = -arrow_size2;
    g_axes_position_array[17][2] = -1.2f + g_arrow_size;

#if defined(USE_VBO) || defined(USE_VAO)
    spGLPhongModelGenBuffers(glpm, 1, &g_axes_position_vbo);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_position_vbo);
    spGLPhongModelVertexOrNormalBufferData(glpm, sizeof(GLfloat) * 18 * 3, g_axes_position_array, SP_GL_STATIC_DRAW);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);

    spGLPhongModelGenBuffers(glpm, 1, &g_axes_normal_vbo);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_normal_vbo);
    spGLPhongModelVertexOrNormalBufferData(glpm, sizeof(GLfloat) * 18 * 3, g_axes_normal_array, SP_GL_STATIC_DRAW);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif
    
#if defined(USE_VAO)
    spGLPhongModelGenVertexArrays(glpm, 1, &g_axes_vao);
    spGLPhongModelBindVertexArray(glpm, g_axes_vao);

    spGLPhongModelEnableVertexArray(glpm);
    spGLPhongModelEnableNormalArray(glpm);

    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_position_vbo);
    spGLPhongModelVertexPointer(glpm, GL_FLOAT, 0, NULL);
    
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_normal_vbo);
    spGLPhongModelNormalPointer(glpm, GL_FLOAT, 0, NULL);
    
    spGLPhongModelBindVertexArray(glpm, 0);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif
    
    return;
}

static void drawAxes(spComponent component)
{
    GLvoid *vertex_ptr = NULL;
    GLvoid *normal_ptr = NULL;
    
    setMaterialColor(MATERIAL_WHITE_INDEX);

#if defined(USE_VAO)
    spGLPhongModelBindVertexArray(glpm, g_axes_vao);
#else
#if defined(USE_VBO)
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_position_vbo);
#else
    vertex_ptr = g_axes_position_array;
    normal_ptr = g_axes_normal_array;
#endif
    spGLPhongModelVertexPointer(glpm, GL_FLOAT, 0, vertex_ptr);
    
#if defined(USE_VBO)
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_axes_normal_vbo);
#endif
    spGLPhongModelNormalPointer(glpm, GL_FLOAT, 0, normal_ptr);
#endif /* !defined(USE_VAO) */
    
    spGLPhongModelDrawArrays(glpm, GL_LINES, 0, 18);

#if defined(USE_VAO)
    spGLPhongModelBindVertexArray(glpm, 0);
#else
#if defined(USE_VBO)
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif
#endif
    
    return;
}

static void drawAxesStrings(spComponent component)
{
#if 1
    spSetGLStringParams(freq_string,
                        SppGLStringProjectionMatrix, g_P.data,
                        SppGLStringModelViewMatrix, g_MV.data,
                        NULL);
    spSetGLStringParams(amp_string,
                        SppGLStringProjectionMatrix, g_P.data,
                        SppGLStringModelViewMatrix, g_MV.data,
                        NULL);
    spSetGLStringParams(time_string,
                        SppGLStringProjectionMatrix, g_P.data,
                        SppGLStringModelViewMatrix, g_MV.data,
                        NULL);
    
    spDrawGLString(component, freq_string);
    spDrawGLString(component, amp_string);
    spDrawGLString(component, time_string);
#endif

    return;
}

static void drawSpecgram(spComponent component)
{
    GLvoid *vertex_ptr = NULL;
    GLvoid *normal_ptr = NULL;
        
    setMaterialColor(g_material_index);

#if defined(USE_VAO)
    spGLPhongModelBindVertexArray(glpm, g_specgram_vao);
#else
#if !defined(USE_TRIANGLES_VERTICES) && defined(USE_VBO)
    spGLPhongModelBindIndexBuffer(glpm, g_vertex_indices_vbo);
#endif
    
#if defined(USE_VBO)
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_specgram_vbo);
    vertex_ptr = NULL;
    normal_ptr = (GLvoid *)spOffsetOf(SpecgramVertex, normal);
#else
    vertex_ptr = (GLvoid *)g_specgram_vertices;
    normal_ptr = (GLvoid *)((char *)g_specgram_vertices + spOffsetOf(SpecgramVertex, normal));
#endif
    
    spGLPhongModelVertexPointer(glpm, GL_FLOAT, sizeof(struct _SpecgramVertex), vertex_ptr);
    spGLPhongModelNormalPointer(glpm, GL_FLOAT, sizeof(struct _SpecgramVertex), normal_ptr);
#endif  /* !defined(USE_VAO) */
    
#if defined(USE_TRIANGLES_VERTICES)
    spGLPhongModelDrawArrays(glpm, GL_TRIANGLES, 0, 3 * 2 * (g_hfftl - 1) * (g_nframes - 1));
#else
    spDebug(10, "drawSpecgram", "g_vertex_indices_length = %d, g_vertex_indices = %ld\n",
            (int)g_vertex_indices_length, (long)g_vertex_indices);
#if defined(USE_VBO) || defined(USE_VAO)    
    spGLPhongModelDrawElements(glpm, GL_TRIANGLE_STRIP, g_vertex_indices_length, GL_UNSIGNED_SHORT, NULL);
#else
    spGLPhongModelDrawElements(glpm, GL_TRIANGLE_STRIP, g_vertex_indices_length, GL_UNSIGNED_SHORT, g_vertex_indices);
#endif
#endif

#if defined(USE_VAO)
    spGLPhongModelBindVertexArray(glpm, 0);
#else
#if defined(USE_VBO)
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif
    
#if !defined(USE_TRIANGLES_VERTICES) && defined(USE_VBO)
    spGLPhongModelBindIndexBuffer(glpm, 0);
#endif
#endif  /* !defined(USE_VAO) */
    
    return;
}

void display(spComponent component)
{
    spGLMat4 R;
    spGLPhongModelDrawMode draw_mode = SP_GL_PHONG_MODEL_DRAW_MODE_DIRECT;
    
    spDebug(10, "display", "in\n");
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

#if defined(USE_CLIP_PLANE)
    spGLPhongModelSetModelViewMatrix(glpm, &g_V);
    
    spGLPhongModelSetClipPlane(glpm, 0, clip_plane_x);
    spGLPhongModelSetClipPlane(glpm, 1, clip_plane_y);
    spGLPhongModelSetClipPlane(glpm, 2, clip_plane_z);
#endif
    
    /* rotation */
    spDebug(10, "display", "g_x_rotation = %f, g_y_rotation = %f\n", g_x_rotation, g_y_rotation);
    g_Rx = spGLMat4CreateRotation(g_x_rotation, 1.0f, 0.0f, 0.0f, SP_FALSE);
    g_Ry = spGLMat4CreateRotation(g_y_rotation, 0.0f, 1.0f, 0.0f, SP_FALSE);
    R = spGLMat4MultMatrix(&g_Rx, &g_Ry, SP_FALSE);

    g_MV = spGLMat4MultMatrix(&g_V, &R, SP_FALSE);
    spGLPhongModelSetModelViewMatrix(glpm, &g_MV);

#if defined(USE_CLIP_PLANE)
    spGLPhongModelSetClipPlane(glpm, 0, clip_plane_x);
    spGLPhongModelSetClipPlane(glpm, 1, clip_plane_y);
    spGLPhongModelSetClipPlane(glpm, 2, clip_plane_z);
#endif
    
    spGLPhongModelSetLightAmbient(glpm, 0, black);
    spGLPhongModelSetLightDiffuse(glpm, 0, white);
    spGLPhongModelSetLightSpecular(glpm, 0, white);
    spGLPhongModelSetLightPosition(glpm, 0, light0pos);
    spGLPhongModelSetLightAttenuation(glpm, 0, light0_attenuation);
    
    spGLPhongModelSetLightAmbient(glpm, 1, black);
    spGLPhongModelSetLightDiffuse(glpm, 1, white);
    spGLPhongModelSetLightSpecular(glpm, 1, white);
    spGLPhongModelSetLightPosition(glpm, 1, light1pos);
    spGLPhongModelSetLightAttenuation(glpm, 1, light1_attenuation);

#if defined(USE_SPOT_LIGHT)
    spGLPhongModelSetLightSpotCutoff(glpm, 0, spot_cutoff);
    spGLPhongModelSetLightSpotExponent(glpm, 0, spot_exponent);
    spGLPhongModelSetLightSpotDirection(glpm, 0, light0_spot_direction);
    spGLPhongModelSetLightSpotCutoff(glpm, 1, spot_cutoff);
    spGLPhongModelSetLightSpotExponent(glpm, 1, spot_exponent);
    spGLPhongModelSetLightSpotDirection(glpm, 1, light1_spot_direction);
#else
    spGLPhongModelSetLightSpotCutoff(glpm, 0, 180.0f);
    spGLPhongModelSetLightSpotCutoff(glpm, 1, 180.0f);
#endif
    
    g_M = spGLMat4MultMatrix(&R, &g_T, SP_FALSE);
    g_MV = spGLMat4MultMatrix(&g_V, &g_M, SP_FALSE);
    spGLPhongModelSetModelViewMatrix(glpm, &g_MV);
    
#if defined(USE_CLIP_PLANE)
    spGLPhongModelSetClipPlane(glpm, 0, clip_plane_x);
    spGLPhongModelSetClipPlane(glpm, 1, clip_plane_y);
    spGLPhongModelSetClipPlane(glpm, 2, clip_plane_z);
#endif
    
#if defined(USE_VAO)
    draw_mode = SP_GL_PHONG_MODEL_DRAW_MODE_USE_VAO;
#elif defined(USE_VBO)
    draw_mode = SP_GL_PHONG_MODEL_DRAW_MODE_USE_VBO;
#endif
    
    /* draw */
    spGLPhongModelBegin(glpm, draw_mode);
    drawSpecgram(component);
    drawAxes(component);
    spGLPhongModelEnd(glpm);

    drawAxesStrings(component);

    spGLSwapBuffers(component);

    spGLWaitGLDrawing(component);
    spGLWaitSystemDrawing(component);
    
    spDebug(10, "display", "done\n");
}

void createPVMat(GLfloat look_at_z)
{
    g_P = spGLMat4CreateOrtho(-0.8f, 0.8f, -0.8f, 0.8f,
                              look_at_z - 2.0f, look_at_z + 2.0f, SP_FALSE);
    spGLPhongModelSetProjectionMatrix(glpm, &g_P);

    g_V = spGLMat4CreateLookAt(1.8f, 1.2f, look_at_z,
                               0.0f, 0.0f, 0.0f,
                               0.0f, 1.0f, 0.0f, SP_FALSE);
}

void resize(int w, int h)
{
    GLfloat look_at_z;
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);

    look_at_z = 5.0;
    createPVMat(look_at_z);

    spDebug(10, "resize", "done\n");
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            g_y_rotation += (float)(x - px);
            g_x_rotation += (float)(y - py);
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE
        && code == SPK_Escape) {
        spQuit(0);
    }
}

#if !defined(DISABLE_TIMER)
spBool timerCB(spComponent frame, void *data)
{
    spComponent canvas = (spComponent)data;

    if (init_done == SP_TRUE) {
        updateSpecgram(g_specgram, g_normal, g_nframes, g_hfftl,
                       g_samp_freq, g_shift_ms,
                       g_specgram_min, g_specgram_max, g_specgram_range,
                       g_spec_freq, g_amp_mod_amp, g_amp_mod_freq,
                       g_material_index,
                       1);
        updateSpecgramVertices(g_specgram_vertices, g_specgram, g_normal, 
                               g_nframes, g_hfftl, g_specgram_max, g_specgram_range, 1);
#if defined(USE_VBO) || defined(USE_VAO)
        spGLPhongModelBindVertexOrNormalBuffer(glpm, g_specgram_vbo);
        {
            long length;
        
#if defined(USE_TRIANGLES_VERTICES)
            length = 3 * 2 * (g_hfftl - 1) * (g_nframes - 1);
#else
            length = g_nframes * g_hfftl;
#endif
            spGLPhongModelVertexOrNormalBufferData(glpm, sizeof(struct _SpecgramVertex) * length,
                                                   g_specgram_vertices, SP_GL_STATIC_DRAW);
        }
        spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif
        
        spRefreshCanvas(canvas);
    }
    spDebug(10, "timerCB", "done\n");
    
    return SP_TRUE;
}
#endif

void changeMaterialCB(spComponent component, int material_index)
{
    g_material_index = material_index;
    return;
}

void init(void)
{
#if defined(USE_SHADER)
    unsigned long options = SPECGRAM_GLSL_VERSION;
#else
    unsigned long options = SP_GL_PHONG_MODEL_OPTION_USE_LEGACY_FUNCTION;
#endif

#if defined(USE_VAO)
    options |= SP_GL_STRING_OPTION_USE_VAO;
#endif
    
#if defined(USE_PHONG_SHADING)
    options |= SP_GL_PHONG_MODEL_OPTION_USE_PHONG_SHADING;
#elif defined(USE_LEGACY_FUNCTION)
    options |= SP_GL_PHONG_MODEL_OPTION_USE_LEGACY_FUNCTION;
#endif
    
#if defined(USE_FLAT_SHADING)
    options |= SP_GL_PHONG_MODEL_OPTION_USE_FLAT_SHADING;
#endif
#if defined(USE_HALFWAY_VECTOR)
    options |= SP_GL_PHONG_MODEL_OPTION_USE_HALFWAY_VECTOR;
#endif
#if defined(USE_CLIP_PLANE)
    options |= SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE;
#endif
    
    glpm = spGLPhongModelInit(2, options);

    {
        GLint buffers, samples;
 
        glGetIntegerv(GL_SAMPLE_BUFFERS, &buffers);
        glGetIntegerv(GL_SAMPLES, &samples);
        spDebug(10, "init", "buffers = %d, samples = %d\n", buffers, samples);
        if (buffers >= 1 && samples >= 2) {
            glEnable(GL_MULTISAMPLE);
        }
    }
    
#if 1
    freq_string = spCreateGLString(g_canvas, "Frequency (Hz)",
                                   SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                   SppForeground, FREQUENCY_STRING_COLOR,
                                   SppGLStringUseShader, USE_SHADER_FLAG,
                                   SppGLStringOptions, options,
                                   NULL);
    amp_string = spCreateGLString(g_canvas, "Magnitude (dB)",
                                  SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                  SppForeground, MAGNITUDE_STRING_COLOR,
                                  SppGLStringUseShader, USE_SHADER_FLAG,
                                  SppGLStringOptions, options,
                                  NULL);
    time_string = spCreateGLString(g_canvas, "Time (sec)",
                                   SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                   SppForeground, TIME_STRING_COLOR,
                                   SppGLStringUseShader, USE_SHADER_FLAG,
                                   SppGLStringOptions, options,
                                   NULL);
    
    spSetGLStringPos3f(freq_string, 1.0f, -0.1f, 0.0f);
    spSetGLStringPos3f(amp_string, -0.1f, 1.3f, 0.0f);
    spSetGLStringPos3f(time_string, 0.0f, -0.1f, -1.0f);
#endif
    createAxesArray();
    
#if 1
    /* yellow background */
    glClearColor(1.0, 1.0, 0.0, 1.0);
#elif 0
    /* white background */
    glClearColor(1.0, 1.0, 1.0, 1.0);
#else
    /* black background */
    glClearColor(0.0, 0.0, 0.0, 1.0);
#endif

    glEnable(GL_DEPTH_TEST);
    glFrontFace(GL_CW);

    g_T = spGLMat4CreateTranslation(-0.5f, -0.5f, 0.5f, SP_FALSE);

    g_nframes = (long)spRound(g_total_ms / g_shift_ms);
    g_specgram = allocPseudoSpecgram(g_nframes, g_hfftl);
    calculatePseudoSpecgram(g_specgram, g_nframes, g_hfftl,
                            g_samp_freq, g_shift_ms,
                            g_specgram_min, g_specgram_max,
                            g_spec_freq,
                            g_amp_mod_amp, g_amp_mod_freq, 0);
    
    g_normal = allocSpecgramNormal(g_nframes, g_hfftl);
    calculateSpecgramNormal(g_normal, g_specgram, g_nframes, g_hfftl, 0);

    g_specgram_vertices = allocSpecgramVertices(g_nframes, g_hfftl);
    calculateSpecgramVertices(g_specgram_vertices, g_specgram, g_normal, g_nframes, g_hfftl,
                              g_specgram_max, g_specgram_range, 0);
#if defined(USE_VBO) || defined(USE_VAO)
    g_specgram_vbo = 0;
    spGLPhongModelGenBuffers(glpm, 1, &g_specgram_vbo);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_specgram_vbo);
    {
        long length;
        
#if defined(USE_TRIANGLES_VERTICES)
        length = 3 * 2 * (g_hfftl - 1) * (g_nframes - 1);
#else
        length = g_nframes * g_hfftl;
#endif
        spGLPhongModelVertexOrNormalBufferData(glpm, sizeof(struct _SpecgramVertex) * length,
                                               g_specgram_vertices, SP_GL_STATIC_DRAW);
    }
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#endif

#if !defined(USE_TRIANGLES_VERTICES)
    g_vertex_indices = createVertexIndices(g_nframes, g_hfftl, &g_vertex_indices_length);
#if defined(USE_VBO) || defined(USE_VAO)
    g_vertex_indices_vbo = 0;
    spGLPhongModelGenBuffers(glpm, 1, &g_vertex_indices_vbo);
    spGLPhongModelBindIndexBuffer(glpm, g_vertex_indices_vbo);
    spGLPhongModelIndexBufferData(glpm, sizeof(GLushort) * g_vertex_indices_length,
                                  g_vertex_indices, SP_GL_STATIC_DRAW);
    spGLPhongModelBindIndexBuffer(glpm, 0);
#endif
    
#if defined(USE_VAO)
    spGLPhongModelGenVertexArrays(glpm, 1, &g_specgram_vao);
    spGLPhongModelBindVertexArray(glpm, g_specgram_vao);

    spGLPhongModelEnableVertexArray(glpm);
    spGLPhongModelEnableNormalArray(glpm);
    
#if !defined(USE_TRIANGLES_VERTICES)
    spGLPhongModelBindIndexBuffer(glpm, g_vertex_indices_vbo);
#endif
    
    spGLPhongModelBindVertexOrNormalBuffer(glpm, g_specgram_vbo);
    
    spGLPhongModelVertexPointer(glpm, GL_FLOAT, sizeof(struct _SpecgramVertex), NULL);
    spGLPhongModelNormalPointer(glpm, GL_FLOAT, sizeof(struct _SpecgramVertex), (GLvoid *)spOffsetOf(SpecgramVertex, normal));
    
    spGLPhongModelBindVertexArray(glpm, 0);
    spGLPhongModelBindVertexOrNormalBuffer(glpm, 0);
#if !defined(USE_TRIANGLES_VERTICES)
    spGLPhongModelBindIndexBuffer(glpm, 0);
#endif
#endif
    
#endif

    init_done = SP_TRUE;
    spDebug(10, "init", "done\n");
}

void canvasCB(spComponent component, void *data)
{
    int w = 0, h = 0;
    double scaling_factor = 1.0;
    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || init_done == SP_FALSE) {
        if (init_done == SP_FALSE) {
            init();
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;

        if ((scaling_factor = spGetGLCanvasScalingFactor(component)) != 1.0 && scaling_factor != 0.0) {
            w = (int)spRound(w * scaling_factor);
            h = (int)spRound(h * scaling_factor);
        }
        spDebug(10, "canvasCB", "scaling_factor = %f\n", scaling_factor);
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent window, canvas, menu, submenu;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_DEPTH_SIZE, 8,
#if defined(USE_MULTISAMPLE)
                                SP_GL_SAMPLE_BUFFERS, 1,
                                SP_GL_SAMPLES, MULTISAMPLE_SAMPLES,
#endif
                                SP_GL_CONTEXT_MAJOR_VERSION, CONTEXT_MAJOR_VERSION,
                                SP_GL_NONE };
  
    init_done = SP_FALSE;
    
    spDebug(10, "spMain", "call spInitialize\n");
    
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Spectrogram Test", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              SppGLAutoScaling, SP_FALSE,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);

    menu = spCreatePopupMenu(canvas, "popupMenu",
                             SppPopupButton, SP_RBUTTON,
                             NULL);
    submenu = spAddSubMenu(menu, "materialSubMenu",
                           SppTitle, "Change Spectrogram Material",
                           NULL);
    spAddRadioButtonMenuItem(submenu, "Emerald",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_EMERALD_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Jade",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_JADE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Obsidian",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_OBSIDIAN_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Pearl",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_PEARL_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Ruby",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_RUBY_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Turquoise",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_TURQUOISE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Bronze",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_BRONZE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Copper",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_COPPER_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Gold",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_GOLD_INDEX,
                             SppSet, SP_TRUE,
                             NULL);
  
    context = spCreateGLContext(canvas, NULL);

    spPopupWindow(window);
  
    spSetGLContext(canvas, context);
    g_canvas = canvas;
  
#if !defined(DISABLE_TIMER)
    g_timer_id = spSetTimer(window, g_timer_interval, timerCB, canvas);
#endif
    
    return spMainLoop(toplevel);
}


