#include <math.h>
#include <sp/spComponentLib.h>

#if defined(ANDROID) || defined(IPHONE)
#define USE_OPENGL_ES2 1
#elif defined(MACOSX) && defined(COCOA)
#define USE_OPENGL3 1
#endif

#if (defined(_WIN32) && !defined(__CYGWIN32__)) /*|| defined(USE_MOTIF) || defined(GTK)*/ || defined(USE_GLEW)
#define GLEW_STATIC
#include <gl/glew.h>
#if defined(_MSC_VER)
#pragma comment ( lib, "glew32s" )
#endif
#if !defined(USE_GLEW)
#define USE_GLEW
#endif
#endif

#if defined(GTK3) || (defined(MACOSX64) && !defined(IPHONE))
#if !defined(USE_VAO)
#define USE_VAO
#endif
#endif

#include <sp/spGL.h>
#include <sp/spComponentMain.h>

const GLchar vertex_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "in vec4 attr_position;\n"
#else
    "attribute vec4 attr_position;\n"
#endif
    "void main() {\n"
    "    gl_Position = attr_position;\n"
    "}\n";

const char fragment_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "#define gl_FragColor FragColor\n"
    "out vec4 FragColor;\n"
#endif
#if defined(SP_GL_ES)
    "precision mediump float;\n"
#endif
    "void main() {\n"
    "    gl_FragColor = vec4(0.0, 1.0, 0.0, 1.0);\n"
    "}\n";

static spBool init_done = SP_FALSE;
static spComponent canvas = NULL;
static spGLContext context = NULL;

static GLuint shader_program;
static GLuint vertex_shader;
static GLuint fragment_shader;
static GLint attr_position_location;
#if defined(USE_VAO)
static GLuint vbo = 0;
static GLuint vao = 0;
#endif

static GLint checkCompileError(GLuint shader)
{
    GLint status;

    glGetShaderiv(shader, GL_COMPILE_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetShaderInfoLog(shader, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLint checkLinkError(GLuint program)
{
    GLint status;

    glGetProgramiv(program, GL_LINK_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetProgramiv(program, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetProgramInfoLog(program, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLuint loadShader(GLenum shader_type, const GLchar *source)
{
    GLuint shader;

    if ((shader = glCreateShader(shader_type)) != 0) {
        glShaderSource(shader, 1, &source, NULL);
        glCompileShader(shader);

        if (checkCompileError(shader) == GL_FALSE) {
            glDeleteShader(shader);
            shader = 0;
        }
    }
    
    return shader;
}

static GLuint loadProgram(GLuint vshader, GLuint fshader)
{
    GLuint program;

    if ((program = glCreateProgram()) != 0) {
        glAttachShader(program, vshader);
        glAttachShader(program, fshader);
        glLinkProgram(program);

        if (checkLinkError(program) == GL_FALSE) {
            glDeleteProgram(program);
            program = 0;
        }
    }

    return program;
}

const GLfloat vertices[] = {
    0.0f,  0.5f,
    -0.5f, -0.5f,
    0.5f, -0.5f
};
        
static spBool init(void)
{
#if defined(USE_GLEW)
    glewExperimental = GL_TRUE;
    if (glewInit() != GLEW_OK) {
        spDebug(10, "init", "glewInit failed\n");
        return SP_FALSE;
    }
#endif

    vertex_shader = loadShader(GL_VERTEX_SHADER, vertex_shader_source);
    fragment_shader = loadShader(GL_FRAGMENT_SHADER, fragment_shader_source);

    if ((shader_program = loadProgram(vertex_shader, fragment_shader)) == 0) {
        glDeleteShader(vertex_shader);
        glDeleteShader(fragment_shader);
        return SP_FALSE;
    }

    attr_position_location = glGetAttribLocation(shader_program, "attr_position");
    
#if defined(USE_VAO)
    glGenBuffers(1, &vbo);
    spDebug(10, "init", "vbo = %d\n", vbo);
    glBindBuffer(GL_ARRAY_BUFFER, vbo);
    glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 3 * 2, vertices, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    
    glGenVertexArrays(1, &vao);
    spDebug(10, "init", "vao = %d\n", vao);
    glBindVertexArray(vao);
    
    glEnableVertexAttribArray(attr_position_location);
    glBindBuffer(GL_ARRAY_BUFFER, vbo);
    glVertexAttribPointer(attr_position_location, 2, GL_FLOAT, GL_FALSE, 0, NULL);
    
    glBindVertexArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
#endif
    
    glClearColor(1.0, 1.0, 1.0, 1.0);
    
    return SP_TRUE;
}

void resize(int w, int h)
{
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
    
    glViewport(0, 0, w, h);

    return;
}

void display(spComponent component)
{
    spDebug(10, "display", "in\n");
    
    glClear(GL_COLOR_BUFFER_BIT);
    
    spDebug(10, "display", "shader_program = %d\n", shader_program);
    glUseProgram(shader_program);
    
#if defined(USE_VAO)
    glBindVertexArray(vao);
    glDrawArrays(GL_TRIANGLES, 0, 3);
    glBindVertexArray(0);
#else
    glVertexAttribPointer(attr_position_location, 2, GL_FLOAT, GL_FALSE, 0, vertices);
    glEnableVertexAttribArray(attr_position_location);
    glDrawArrays(GL_TRIANGLES, 0, 3);
#endif
    
    glUseProgram(0);
    
    spGLSwapBuffers(component);
    
    spDebug(10, "display", "done\n");
    
    return;
}

void canvasCB(spComponent component, void *data)
{
    int w = 0, h = 0;

    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || init_done == SP_FALSE) {
        if (init_done == SP_FALSE) {
            init();
            init_done = SP_TRUE;
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
    
    spDebug(10, "canvasCB", "done\n");
    
    return;
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent window;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_CONTEXT_MAJOR_VERSION, 2,
                                SP_GL_NONE };

    /*spSetDebugLevel(50);*/
    
    init_done = SP_FALSE;
    
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("GLSL Test", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppCallbackFunc, canvasCB,
                              NULL);

    context = spCreateGLContext(canvas, NULL);

    spPopupWindow(window);
  
    spSetGLContext(canvas, context);
  
    return spMainLoop(toplevel);
}
    
