/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>

#if defined(ANDROID) || defined(IPHONE)
#define USE_OPENGL_ES2 1
#endif

#if (defined(_WIN32) && !defined(__CYGWIN32__)) /*|| defined(USE_MOTIF) || defined(GTK)*/ || defined(USE_GLEW)
#define GLEW_STATIC
#include <gl/glew.h>
#if defined(_MSC_VER)
#pragma comment ( lib, "glew32s" )
#endif
#if !defined(USE_GLEW)
#define USE_GLEW
#endif
#endif

#if defined(GTK3) || (defined(MACOSX64) && !defined(IPHONE))
#if !defined(USE_VAO)
#define USE_VAO
#endif
#endif

#include <sp/spGL.h>
#include <sp/spComponentMain.h>

#if 0
#define USE_MULTISAMPLE
#else
#undef USE_MULTISAMPLE
#endif
#define MULTISAMPLE_SAMPLES 4

const GLchar vertex_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "#define attribute in\n"
    "#define varying out\n";
#endif
    "attribute vec3 attr_position;\n"
    "attribute vec2 attr_uv;\n"
    "varying vec2 vary_uv;\n"
    "\n"
    "uniform mat4 unif_MVP;\n"
    "\n"
    "void main() {\n"
    "    gl_Position = unif_MVP * vec4(attr_position, 1.0);\n"
    "    vary_uv = attr_uv;\n"
    "}\n";

const char fragment_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "#define varying in\n"
    "#define texture2D texture\n"
    "#define gl_FragColor FragColor\nout vec4 FragColor;\n";
#endif
#if defined(SP_GL_ES)
    "precision mediump float;\n"
#endif
    "\n"
    "varying vec2 vary_uv;\n"
    "\n"
    "uniform sampler2D unif_texture;\n"
    "\n"
    "void main() {\n"
    "    gl_FragColor = texture2D(unif_texture, vary_uv);\n"
    "}\n";

static spBool init_done = SP_FALSE;

static spComponent window = NULL;

static spGLContext context = NULL;
static spDIBitmap dibitmap = NULL;

static GLuint tex_name = 0;

static spGLMat4 Projection;
static GLfloat g_x_rotation = 0.0f;
static GLfloat g_y_rotation = 0.0f;

static GLenum texture_unit = GL_TEXTURE0;
static GLuint shader_program;
static GLuint vertex_shader;
static GLuint fragment_shader;
static GLint attr_position_location;
static GLint attr_uv_location;
static GLint unif_MVP_location = 0;
static GLint unif_texture_location = 0;
#if defined(USE_VAO)
static GLuint uv_vbo = 0;
static GLuint pos_vbo = 0;
static GLuint vao = 0;
#endif

static GLint checkCompileError(GLuint shader)
{
    GLint status;

    glGetShaderiv(shader, GL_COMPILE_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetShaderInfoLog(shader, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLint checkLinkError(GLuint program)
{
    GLint status;

    glGetProgramiv(program, GL_LINK_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetProgramiv(program, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetProgramInfoLog(program, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLuint loadShader(GLenum shader_type, const GLchar *source)
{
    GLuint shader;

    if ((shader = glCreateShader(shader_type)) != 0) {
        glShaderSource(shader, 1, &source, NULL);
        glCompileShader(shader);

        if (checkCompileError(shader) == GL_FALSE) {
            glDeleteShader(shader);
            shader = 0;
        }
    }
    
    return shader;
}

static GLuint loadProgram(GLuint vshader, GLuint fshader)
{
    GLuint program;

    if ((program = glCreateProgram()) != 0) {
        glAttachShader(program, vshader);
        glAttachShader(program, fshader);
        glLinkProgram(program);

        if (checkLinkError(program) == GL_FALSE) {
            glDeleteProgram(program);
            program = 0;
        }
    }

    return program;
}


void resize(int w, int h)
{
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);
    Projection = spGLMat4CreateOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0, SP_FALSE);

    return;
}

const GLfloat vertices_upsidedown[] = {
    -0.75f, -0.75f, 0.0f,
    -0.75f, 0.75f, 0.0f,
    0.75f, -0.75f, 0.0f,
    0.75f, 0.75f, 0.0f,
};
        
const GLfloat vertices_upsideup[] = {
    -0.75f, 0.75f, 0.0f,
    -0.75f, -0.75f, 0.0f,
    0.75f, 0.75f, 0.0f,
    0.75f, -0.75f, 0.0f,
};
        
const GLfloat uv_vertices[] = {
    0.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 0.0f,
    1.0f, 1.0f,
};
        
spBool init(void)
{
    GLint internal_format;
    GLenum format;
    GLenum type;

#if defined(USE_GLEW)
    glewExperimental = GL_TRUE;
    if (glewInit() != GLEW_OK) {
        spDebug(10, "init", "glewInit failed\n");
        return SP_FALSE;
    }
#endif

    vertex_shader = loadShader(GL_VERTEX_SHADER, vertex_shader_source);
    fragment_shader = loadShader(GL_FRAGMENT_SHADER, fragment_shader_source);

    if ((shader_program = loadProgram(vertex_shader, fragment_shader)) == 0) {
        glDeleteShader(vertex_shader);
        glDeleteShader(fragment_shader);
        return SP_FALSE;
    }

    attr_position_location = glGetAttribLocation(shader_program, "attr_position");
    attr_uv_location = glGetAttribLocation(shader_program, "attr_uv");
    
    unif_MVP_location = glGetUniformLocation(shader_program, "unif_MVP");
    unif_texture_location = glGetUniformLocation(shader_program, "unif_texture");

#if defined(USE_VAO)
    glGenBuffers(1, &pos_vbo);
    spDebug(10, "init", "pos_vbo = %d\n", pos_vbo);
    glBindBuffer(GL_ARRAY_BUFFER, pos_vbo);
    {
        const GLfloat *vertices;
        
        if (dibitmap->info.upside_down) {
            vertices = vertices_upsidedown;
        } else {
            vertices = vertices_upsideup;
        }
        glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 3 * 4, vertices, GL_STATIC_DRAW);
    }
    glBindBuffer(GL_ARRAY_BUFFER, 0);

    glGenBuffers(1, &uv_vbo);
    glBindBuffer(GL_ARRAY_BUFFER, uv_vbo);
    glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * 4, uv_vertices, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    
    glGenVertexArrays(1, &vao);
    spDebug(10, "init", "vao = %d\n", vao);
    glBindVertexArray(vao);
    
    glEnableVertexAttribArray(attr_position_location);
    glEnableVertexAttribArray(attr_uv_location);
    
    glBindBuffer(GL_ARRAY_BUFFER, pos_vbo);
    glVertexAttribPointer(attr_position_location, 3, GL_FLOAT, GL_FALSE, 0, NULL);
    
    glBindBuffer(GL_ARRAY_BUFFER, uv_vbo);
    glVertexAttribPointer(attr_uv_location, 2, GL_FLOAT, GL_FALSE, 0, NULL);
    
    glBindVertexArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
#endif
    
#if defined(USE_MULTISAMPLE)
    {
        GLint buffers, samples;
 
        glGetIntegerv(GL_SAMPLE_BUFFERS, &buffers);
        glGetIntegerv(GL_SAMPLES, &samples);
        spDebug(10, "init", "buffers = %d, samples = %d\n", buffers, samples);
        if (buffers >= 1 && samples >= 2) {
            glEnable(GL_MULTISAMPLE);
        }
    }
#endif
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);

    if (dibitmap->info.pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK) {
        GLenum sfactor = GL_SRC_ALPHA, dfactor = GL_ONE_MINUS_SRC_ALPHA;
        spDebug(10, "init", "with alpha\n");
        glEnable(GL_BLEND);
        spGetGLDIPixelFormatAlphaBlendFactor(dibitmap->info.pixel_format, &sfactor, &dfactor);
        spDebug(10, "init", "alpha sfactor = %x (GL_SRC_ALPHA: %x), dfactor = %x (GL_ONE_MINUS_SRC_ALPHA: %x)\n",
                sfactor, GL_SRC_ALPHA, dfactor, GL_ONE_MINUS_SRC_ALPHA);
        glBlendFunc(sfactor, dfactor);
    }

    glActiveTexture(texture_unit);
    
    glGenTextures(1, &tex_name);
    glBindTexture(GL_TEXTURE_2D, tex_name);

    if (spGetGLDIBitmapFormat(dibitmap, &internal_format, &format, &type)) {
        spDebug(10, "init", "dibitmap->info.pixel_format = %lx, internal_format = %x, format = %x, type = %x\n",
                dibitmap->info.pixel_format, internal_format, format, type);

        glPixelStorei(GL_UNPACK_ALIGNMENT, spGetGLDIBitmapUnpackAlignment(dibitmap));
        glPixelStorei(GL_UNPACK_ROW_LENGTH, spGetGLDIBitmapUnpackRowLength(dibitmap));
    
        spLockDIBitmap(dibitmap);
        spDebug(10, "init", "dibitmap->info.data = %ld\n", (long)dibitmap->info.data);
        glTexImage2D(GL_TEXTURE_2D, 0, internal_format, dibitmap->info.width, dibitmap->info.height,
                     0, format, type, dibitmap->info.data);    
        spUnlockDIBitmap(dibitmap);
        
        glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    } else {
        spDisplayWarning(window, NULL, "This bitmap file isn't suitable for OpenGL texture.\n");
        spDebug(1, "init", "spGetGLDIBitmapFormat failed\n");
    }
    
#if 0
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
#else
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
#endif
    
    glBindTexture(GL_TEXTURE_2D, 0);
    
    glClearColor(0.6f, 0.6f, 1.0f, 1.0f);
    
    return SP_TRUE;
}

void display(spComponent component)
{
    spGLMat4 Rx, Ry, MV, MVP;
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glUseProgram(shader_program);
    
    /* rotation */
    Rx = spGLMat4CreateRotation(g_x_rotation, 1.0f, 0.0f, 0.0f, SP_FALSE);
    Ry = spGLMat4CreateRotation(g_y_rotation, 0.0f, 1.0f, 0.0f, SP_FALSE);
    MV = spGLMat4MultMatrix(&Rx, &Ry, SP_FALSE);

    MVP = spGLMat4MultMatrix(&Projection, &MV, SP_FALSE);
    glUniformMatrix4fv(unif_MVP_location, 1, GL_FALSE, (const GLfloat *)MVP.data);

    glActiveTexture(texture_unit);
    
    glBindTexture(GL_TEXTURE_2D, tex_name);
    glUniform1i(unif_texture_location, texture_unit - GL_TEXTURE0);
    
#if defined(USE_VAO)
    glBindVertexArray(vao);
#else
    glEnableVertexAttribArray(attr_position_location);
    glEnableVertexAttribArray(attr_uv_location);
    {
        const GLfloat *vertices;
        
        spDebug(80, "display", "upside_down = %d\n", dibitmap->info.upside_down);
        if (dibitmap->info.upside_down) {
            vertices = vertices_upsidedown;
        } else {
            vertices = vertices_upsideup;
        }
        glVertexAttribPointer(attr_position_location, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid *)vertices);
    }
    glVertexAttribPointer(attr_uv_location, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid *)uv_vertices);
#endif
    
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    spDebug(80, "display", "glDrawArrays error = %d\n", (int)glGetError());
    
#if defined(USE_VAO)
    glBindVertexArray(0);
#else
    glDisableVertexAttribArray(attr_uv_location);
    glDisableVertexAttribArray(attr_position_location);
#endif
    
    glBindTexture(GL_TEXTURE_2D, 0);
    
    glUseProgram(0);
    
    spGLSwapBuffers(component);
    
    return;
}

void quit(void)
{
    glBindTexture(GL_TEXTURE_2D, 0);
    glDeleteTextures(1, &tex_name);
    
    return;
}

void canvasCB(spComponent component, void *data)
{
    int w = 0, h = 0;

    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || init_done == SP_FALSE) {
        if (init_done == SP_FALSE) {
            init();
            init_done = SP_TRUE;
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;
  
        resize(w, h);
        spGLSwapBuffers(component);
    }
    display(component);
    
    return;
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            g_y_rotation += (GLfloat)(x - px);
            g_x_rotation += (GLfloat)(y - py);
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE) {
        if (code == SPK_Escape) {
            spQuit(0);
        }
    }

    return;
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent canvas;
    char *filename = NULL;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_CONTEXT_MAJOR_VERSION, 2,
                                SP_GL_DEPTH_SIZE, 1,
#if defined(USE_MULTISAMPLE)
                                SP_GL_SAMPLE_BUFFERS, 1,
                                SP_GL_SAMPLES, MULTISAMPLE_SAMPLES,
#endif
                                SP_GL_NONE };

    /*spSetDebugLevel(100);*/
    
    init_done = SP_FALSE;
    
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Texture", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);

    if (argc >= 2) {
        filename = argv[1];
    } else {
        filename = xspGetOpenFileName(window, NULL,
                                      SppPathMustExist, SP_TRUE,
                                      SppFileMustExist, SP_TRUE,
                                      NULL);

        if (filename == NULL) {
            spDisplayError(window, NULL, "File name must be specified.\n");
            spQuit(0);
        }
    }
    spDebug(10, "spMain", "filename = %s\n", filename);
    
    /* load bitmap file */
    if ((dibitmap = spCreateDIBitmapFromFileNonnative(filename)) == NULL) {
        spDisplayError(window, NULL, "Cannot open bitmap file: %s", filename);
        spQuit(0);
    }
    spDebug(10, "spMain", "pixel_format = %lx, size = (%ld, %ld), bit_size = %ld, pixel_stride = %ld, line_stride = %ld\n",
            dibitmap->info.pixel_format, dibitmap->info.width, dibitmap->info.height,
            dibitmap->info.bit_size, dibitmap->info.pixel_stride, dibitmap->info.line_stride);
    
    context = spCreateGLContext(canvas, NULL);

    spDebug(100, "spMain", "call spPopupWindow\n");
    spPopupWindow(window);
  
    spDebug(100, "spMain", "call spSetGLContext\n");
    spSetGLContext(canvas, context);
  
    spDebug(100, "spMain", "call spMainLoop\n");
    return spMainLoop(toplevel);
}
