/*
 * Version for spComponentEx (GLSL version) by Hideki Banno
 */
   
/*
 * Mesa 3-D graphics library
 * 
 * Copyright (C) 1999  Brian Paul   All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/*
 * Example of using glXUseXFont().
 * 5 November 1999
 * Brian Paul
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sp/spComponentLib.h>

#if defined(ANDROID) || defined(IPHONE)
#define USE_OPENGL_ES2 1
#elif defined(MACOSX) && defined(COCOA)
#define USE_OPENGL3 1
#endif

#if (defined(_WIN32) && !defined(__CYGWIN32__)) /*|| defined(USE_MOTIF) || defined(GTK)*/ || defined(USE_GLEW)
#define GLEW_STATIC
#include <GL/glew.h>
#if defined(_MSC_VER)
#pragma comment ( lib, "glew32s" )
#endif
#if !defined(USE_GLEW)
#define USE_GLEW
#endif
#endif

#if defined(GTK3) || (defined(MACOSX64) && !defined(IPHONE))
#if !defined(USE_VAO)
#define USE_VAO
#endif
#endif

#include <sp/spGL.h>
#include <sp/spGLString.h>
#include <sp/spComponentMain.h>

#define JAPANESE_TEST

#if defined(JAPANESE_TEST)
#if defined(COCOA) || (defined(SP_SUPPORT_UTF8_STRING) && !defined(_WIN32) && !defined(TARGET_API_MAC_CARBON))
#include "strings_utf8jp.h"
#elif (defined(_WIN32) && !defined(__CYGWIN32__)) || defined(MACOS)
#include "strings_sjis.h"
#else
#include "strings_eucjp.h"
#endif
#else
#define TEST_STRING "This is a test string."
#endif

#if defined(MACOSX64) && !defined(IPHONE)
#define CONTEXT_MAJOR_VERSION 3
#define USE_GLSL150
#else
#define CONTEXT_MAJOR_VERSION 2
#endif

static char *ProgramName = "xfont";

static char *FontName = "-*-*-*-*-*--48-*-*-*-*-*-*-*";
/*static char *FontName = "-*-*-medium-*-normal--64-*-*-*-*-*-*-*";*/
/*static char *FontName = "-*-Times-bold-i-normal--32-*-*-*-*-*-*-*";*/

#include <sp/spGLString.h>

static spGLMat4 g_P;
static spGLMat4 g_MV;

static spGLString glstring = NULL;

static void setup_font( spComponent component );

const GLchar vertex_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "out vec3 vary_color;\n"
    "in vec4 attr_position;\n"
#else
    "varying vec3 vary_color;\n"
    "attribute vec4 attr_position;\n"
#endif
    "\n"
    "void main() {\n"
    "    gl_Position = attr_position;\n"
    "}\n";

const char fragment_shader_source[] =
#if defined(USE_GLSL150)
    "#version 150\n"
    "#define gl_FragColor FragColor\n"
    "out vec4 FragColor;\n"
#endif
#if defined(SP_GL_ES)
    "precision mediump float;"
#endif
    "uniform vec4 unif_color;\n"
    "\n"
    "void main() {\n"
    "    gl_FragColor = unif_color;\n"
    "}\n";

static GLuint shader_program;
static GLuint vertex_shader;
static GLuint fragment_shader;
static GLint attr_position_location;
static GLint unif_color_location;
#if defined(USE_VAO)
static GLuint vbo = 0;
static GLuint vao = 0;
#endif

static GLint checkCompileError(GLuint shader)
{
    GLint status;

    glGetShaderiv(shader, GL_COMPILE_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetShaderInfoLog(shader, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLint checkLinkError(GLuint program)
{
    GLint status;

    glGetProgramiv(program, GL_LINK_STATUS, &status);

    if (status == GL_FALSE) {
        GLsizei buf_size;
        
        glGetProgramiv(program, GL_INFO_LOG_LENGTH , &buf_size);

        if (buf_size > 1) {
            GLchar *info_log;
            GLsizei length;

            if ((info_log = malloc(buf_size)) != NULL) {
                glGetProgramInfoLog(program, buf_size, &length, info_log);
                fprintf(stderr, "%s\n", info_log);
                free(info_log);
            }
        }
    }

    return status;
}

static GLuint loadShader(GLenum shader_type, const GLchar *source)
{
    GLuint shader;

    if ((shader = glCreateShader(shader_type)) != 0) {
        glShaderSource(shader, 1, &source, NULL);
        glCompileShader(shader);

        if (checkCompileError(shader) == GL_FALSE) {
            glDeleteShader(shader);
            shader = 0;
        }
    }
    
    return shader;
}

static GLuint loadProgram(GLuint vshader, GLuint fshader)
{
    GLuint program;

    if ((program = glCreateProgram()) != 0) {
        glAttachShader(program, vshader);
        glAttachShader(program, fshader);
        glLinkProgram(program);

        if (checkLinkError(program) == GL_FALSE) {
            glDeleteProgram(program);
            program = 0;
        }
    }

    return program;
}

const GLfloat vertices[] = {
   0.0f, 0.8f,
   -0.8f, -0.7f,
   0.8f, -0.7f,
};

static spBool init(void)
{
#if defined(USE_GLEW)
   glewExperimental = GL_TRUE;
   if (glewInit() != GLEW_OK) {
      spDebug(10, "init", "glewInit failed\n");
      return SP_FALSE;
   }
#endif

   vertex_shader = loadShader(GL_VERTEX_SHADER, vertex_shader_source);
   fragment_shader = loadShader(GL_FRAGMENT_SHADER, fragment_shader_source);

   if ((shader_program = loadProgram(vertex_shader, fragment_shader)) == 0) {
      glDeleteShader(vertex_shader);
      glDeleteShader(fragment_shader);
      return SP_FALSE;
   }

   attr_position_location = glGetAttribLocation(shader_program, "attr_position");
   unif_color_location = glGetUniformLocation(shader_program, "unif_color");

   g_P = spGLMat4CreateOrtho( -1.0, 1.0, -1.0, 1.0, -1.0, 1.0, SP_FALSE );
   g_MV = spGLMat4CreateIdentity();

#if defined(USE_VAO)
   glGenBuffers( 1, &vbo );
   spDebug(10, "init", "vbo = %d\n", vbo);
   glBindBuffer( GL_ARRAY_BUFFER, vbo );
   glBufferData( GL_ARRAY_BUFFER, sizeof(GLfloat) * 3 * 2, vertices, GL_STATIC_DRAW );
   glBindBuffer( GL_ARRAY_BUFFER, 0 );
    
   glGenVertexArrays( 1, &vao );
   spDebug(10, "init", "vao = %d\n", vao);
   glBindVertexArray( vao );
    
   glEnableVertexAttribArray( attr_position_location );
   glBindBuffer( GL_ARRAY_BUFFER, vbo );
   glVertexAttribPointer( attr_position_location, 2, GL_FLOAT, GL_FALSE, 0, NULL );
    
   glBindVertexArray( 0 );
   glBindBuffer( GL_ARRAY_BUFFER, 0 );
#endif
   
   glClearColor( 0.5, 0.5, 1.0, 1.0 );

   return SP_TRUE;
}

static void resize( spComponent component, void *data )
{
   int width, height;

   spDebug(10, "resize", "in\n");
   
   if (spGetSize( component, &width, &height ) == SP_TRUE) {
      spDebug(10, "resize", "width = %d, height = %d\n", width, height);
      glViewport( 0, 0, width, height );
   }
   
   spDebug(10, "resize", "done\n");
}


static void redraw( spComponent component, void *data )
{
   spBool *init_done = (spBool *)data;

   spDebug(10, "redraw", "in\n");

   if (spGetCallbackReason( component ) == SP_CR_RESIZE || *init_done == SP_FALSE) {
      if (*init_done == SP_FALSE) {
         init();
         setup_font( component );

         *init_done = SP_TRUE;
      }
      resize( component, NULL );
      /*spGLSwapBuffers( component );*/
   }

   glClear( GL_COLOR_BUFFER_BIT );

   glUseProgram( shader_program );
    
   /* triangle */
   glUniform4f( unif_color_location, 0.2f, 0.2f, 1.0f, 1.0f ); 

#if defined(USE_VAO)
   glBindVertexArray( vao );
   glDrawArrays( GL_TRIANGLES, 0, 3 );
   glBindVertexArray( 0 );
#else
   glVertexAttribPointer( attr_position_location, 2, GL_FLOAT, GL_FALSE, 0, vertices );
   glEnableVertexAttribArray( attr_position_location );
   glDrawArrays( GL_TRIANGLES, 0, 3 );
#endif
    
   glUseProgram( 0 );
   
   /* text */
   spDebug(10, "redraw", "draw text\n");
   if (glstring != NULL) {
      spSetGLStringPos3f( glstring, -0.8f, -0.7f, 0.0f );
      spSetGLStringParams( glstring,
                           SppGLStringProjectionMatrix, g_P.data,
                           SppGLStringModelViewMatrix, g_MV.data,
                           NULL);
      spDrawGLString( component, glstring );
   }

   spGLSwapBuffers( component );
   
   spDebug(10, "redraw", "done\n");
}


static void keyPress( spComponent component, void *data )
{
   spKeySym code;
    
   if (spGetCallbackKeySym( component, &code ) == SP_TRUE) {
      spDebug(10, "keyPress", "code = %ld\n", (long)code);
      if (code == SPK_Escape) {
         /* escape */
         spQuit(0);
      }
   }
}


static void setup_font( spComponent component )
{
   unsigned long options = 0L;

#if defined(USE_GLSL150)
   options = 150L;
#elif !defined(USE_OPENGL_ES2)
   options = 120L;
#endif
   
   if (glstring != NULL) {
      spDestroyGLString( glstring );
   }
#if defined(USE_VAO)
   options |= SP_GL_STRING_OPTION_USE_VAO;
#endif
   glstring = spCreateGLString( component, TEST_STRING,
                                SppFontName, FontName,
                                SppForeground, /*"white"*/"red",
                                SppGLStringUseShader, SP_TRUE,
                                SppGLStringOptions, options,
                                NULL);
   if (!glstring) {
      printf("Error: font %s not found\n", FontName);
      exit(0);
   }
}

static spBool init_done = SP_FALSE;

static spComponent make_rgb_db_window( spTopLevel toplevel, int width, int height )
{
   spGLAttribute attrib[] = { SP_GL_RGBA,
                              SP_GL_RED_SIZE, 1,
                              SP_GL_GREEN_SIZE, 1,
                              SP_GL_BLUE_SIZE, 1,
                              SP_GL_DOUBLEBUFFER,
                              SP_GL_CONTEXT_MAJOR_VERSION, CONTEXT_MAJOR_VERSION,
                              SP_GL_NONE };
   spComponent win, canvas;
   spGLVisual visinfo;
   spGLContext ctx;

   init_done = SP_FALSE;
   
   visinfo = spCreateGLVisual( toplevel, attrib );
   if (!visinfo) {
      printf("Error: couldn't get an RGB, Double-buffered visual\n");
      exit(1);
   }

   win = spCreateMainFrame( ProgramName, NULL );
   canvas = spCreateGLCanvas( win, "canvas", visinfo, width, height,
                              SppCallbackFunc, redraw,
                              SppCallbackData, &init_done,
                              NULL );
   spAddCallback( canvas, SP_KEY_PRESS_CALLBACK, keyPress, NULL );
   
   ctx = spCreateGLContext( canvas, NULL );

   spPopupWindow( win );
   spDebug(10, "make_rgb_db_window", "spPopupWindow done\n");
   
   spSetGLContext( canvas, ctx );
   spDebug(10, "make_rgb_db_window", "spSetGLContext done\n");

   return canvas;
}


int spMain( int argc, char *argv[] )
{
   spTopLevel toplevel;
   spComponent canvas;

   /*spSetDebugLevel(10);*/

   /* initialize toolkit */
   toplevel = spInitialize(&argc, &argv, NULL);

   canvas = make_rgb_db_window( toplevel, 300, 300 );

   return spMainLoop(toplevel);
}

