/*
 * Version for spComponentEx  by Hideki Banno
 *
 * ----
 *
 * Create N GLX windows/contexts and render to them in round-robin order.
 * Also, have the contexts share all texture objects.
 * Press 'd' to delete a texture, 'u' to unbind it.
 *
 * Copyright (C) 2000  Brian Paul   All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


#include <sp/spComponentLib.h>
#include <sp/spGL.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sp/spComponentMain.h>

/*
 * Each display/window/context:
 */
struct head {
   char DisplayName[1000];
   spComponent Win;
   spComponent Canvas;
   spGLContext Context;
   float Angle;
   char Renderer[1000];
   char Vendor[1000];
   char Version[1000];
};

static void ExposeCB(spComponent canvas, struct head *h);
static void KeyPressCB(spComponent canvas, struct head *h);
static void ResizeCB(spComponent canvas, struct head *h);

#define MAX_HEADS 200
static struct head Heads[MAX_HEADS];
static int NumHeads = 0;
static GLboolean SwapSeparate = GL_TRUE;
static GLuint TexObj = 0;


static void
Error(spComponent Win, const char *msg)
{
   spDisplayError(Win, NULL, "Error - %s", msg);
}


static struct head *
AddHead(spTopLevel toplevel, const char *name)
{
   spComponent win, canvas;
   spGLContext ctx;
   spGLAttribute attrib[] = { SP_GL_RGBA,
                    SP_GL_RED_SIZE, 1,
                    SP_GL_GREEN_SIZE, 1,
                    SP_GL_BLUE_SIZE, 1,
                    SP_GL_DOUBLEBUFFER,
                    SP_GL_NONE };
   spGLVisual visinfo;
   /*int width = 90, height = 90;*/
   int width = 120, height = 120;
   int xpos = 0, ypos = 0;

   if (NumHeads >= MAX_HEADS)
      return NULL;

   /* window attributes */
   xpos = (NumHeads % 10) * (width + 10);
   ypos = (NumHeads / 10) * (height + 10) + 50;
   printf("%d, %d\n", xpos, ypos);

   win = spCreateFrame((char *)name,
                       SppInitialX, xpos,
                       SppInitialY, ypos,
                       NULL);

   visinfo = spCreateGLVisual(toplevel, attrib);
   if (!visinfo) {
      Error(win, "Unable to find RGB, double-buffered visual");
      return NULL;
   }

   canvas = spCreateGLCanvas(win, "canvas", visinfo, width, height,
                             NULL);
   
   if (NumHeads == 0) {
      ctx = spCreateGLContext(canvas, NULL);
   }
   else {
      /* share textures & dlists with 0th context */
      printf("sharing\n");
      ctx = spCreateGLContext(canvas, Heads[0].Context);
   }
   if (!ctx) {
      Error(win, "Couldn't create context");
      return NULL;
   }

   spPopupWindow(win);

   spDebug(100, "AddHead", "spPopupWindow done\n");
   
   if (!spSetGLContext(canvas, ctx)) {
      Error(win, "spSetGLContext failed");
      printf("spSetGLContext failed in Redraw()\n");
      return NULL;
   }

   spDebug(100, "AddHead", "NumHeads = %d\n", NumHeads);
   
   if (NumHeads == 0) {
      /* create texture object now */
      static const GLubyte checker[2][2][4] = {
         { {255, 255, 255, 255}, {  0,   0,   0, 255} },
         { {  0,   0,   0,   0}, {255, 255, 255, 255} }
      };
      glGenTextures(1, &TexObj);
      assert(TexObj);
      glBindTexture(GL_TEXTURE_2D, TexObj);
      glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 2, 2, 0, GL_RGB,
                   GL_UNSIGNED_BYTE, checker);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
   }
   else {
      /* bind 0th context's texture in this context too */
      assert(TexObj);
      glBindTexture(GL_TEXTURE_2D, TexObj);
   }
   glEnable(GL_TEXTURE_2D);

   /* save the info for this head */
   {
      struct head *h = &Heads[NumHeads];
      strcpy(h->DisplayName, name);
      h->Win = win;
      h->Canvas = canvas;
      h->Context = ctx;
      h->Angle = 0.0;
      strcpy(h->Version, (char *) glGetString(GL_VERSION));
      strcpy(h->Vendor, (char *) glGetString(GL_VENDOR));
      strcpy(h->Renderer, (char *) glGetString(GL_RENDERER));
      NumHeads++;
      
      spAddCallback(canvas, SP_EXPOSE_CALLBACK, (spCallbackFunc)ExposeCB, h);
      spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, (spCallbackFunc)KeyPressCB, h);
      spAddCallback(canvas, SP_RESIZE_CALLBACK, (spCallbackFunc)ResizeCB, h);

      /* required for Windows */
      ResizeCB(canvas, h);

      return &Heads[NumHeads-1];
   }

}


static void
DestroyHeads(void)
{
   int i;
   for (i = 0; i < NumHeads; i++) {
      spDestroyGLContext(Heads[i].Context);
      spDestroyWindow(Heads[i].Win);
   }
}


static void
Redraw(struct head *h)
{
   spDebug(100, "Redraw", "in\n");
   
   if (!spSetGLContext(h->Canvas, h->Context)) {
      Error(h->Win, "spSetGLContext failed");
      printf("spSetGLContext failed in Redraw()\n");
      return;
   }

   h->Angle += 1.0;

   glShadeModel(GL_FLAT);
   glClearColor(0.5, 0.5, 0.5, 1.0);
   glClear(GL_COLOR_BUFFER_BIT);

   /* draw green triangle */
   glColor3f(0.0, 1.0, 0.0);
   glPushMatrix();
   glRotatef(h->Angle, 0, 0, 1);
   glBegin(GL_TRIANGLES);
   glTexCoord2f(0.5, 1.0);   glVertex2f(0, 0.8);
   glTexCoord2f(0.0, 0.0);   glVertex2f(-0.8, -0.7);
   glTexCoord2f(1.0, 0.0);   glVertex2f(0.8, -0.7);
   glEnd();
   glPopMatrix();

   if (!SwapSeparate)
      spGLSwapBuffers(h->Canvas);
   
   spDebug(100, "Redraw", "done\n");
}


static void
Swap(struct head *h)
{
   spGLSwapBuffers(h->Canvas);
}


static void
Resize(const struct head *h, unsigned int width, unsigned int height)
{
    spDebug(50, "Resize", "in: width = %d, height = %d\n", width, height);
   if (!spSetGLContext(h->Canvas, h->Context)) {
      Error(h->Win, "spSetGLContext failed in Resize()");
      return;
   }
   glFlush();
   glViewport(0, 0, width, height);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);
   spDebug(50, "Resize", "done\n");
}

static void
ExposeCB(spComponent canvas, struct head *h)
{
   spDebug(10, "ExposeCB", "in\n");
   if (h == NULL) return;
   Redraw(h);
   if (SwapSeparate)
      Swap(h);
   spDebug(10, "ExposeCB", "done\n");
}

static void
KeyPressCB(spComponent canvas, struct head *h)
{
    int i;
    char buf[100];
    spKeySym keySym;

    for (i = 0; i < NumHeads; i++) {
        if (h == &Heads[i]) {
            break;
        }
    }
    if (spGetCallbackKeyString(canvas, buf, sizeof(buf), NULL) > 0) {
        if ((buf[0] == 'd' || buf[0] == 'D') && TexObj != 0) {
            printf("Delete Texture in window %d\n", i);
            spSetGLContext(h->Canvas, h->Context);
            spDebug(10, "KyePressCB", "before glDeleteTextures: TexObj = %d\n", TexObj);
            glDeleteTextures(1, &TexObj);
            spDebug(10, "KyePressCB", "after glDeleteTextures: TexObj = %d\n", TexObj);
            TexObj = 0;
        } else if ((buf[0] == 'b' || buf[0] == 'B') && TexObj != 0) {
            printf("Bind Texture in window %d\n", i);
            spSetGLContext(h->Canvas, h->Context);
            glBindTexture(GL_TEXTURE_2D, TexObj);
        } else if (buf[0] == 'u' || buf[0] == 'U') {
            printf("Unbind Texture in window %d\n", i);
            spSetGLContext(h->Canvas, h->Context);
            glBindTexture(GL_TEXTURE_2D, 0);
        }
        spDebug(10, "KyePressCB", "error code = %d\n", glGetError());
    } else if (spGetCallbackKeySym(canvas, &keySym) == SP_TRUE) {
        if (keySym == SPK_Escape) {
            spQuit(0);
        }
    }
}

static void
ResizeCB(spComponent canvas, struct head *h)
{
   int width, height;
   if (h == NULL) return;
   if (spGetSize(canvas, &width, &height) == SP_TRUE) {
      Resize(h, width, height);
   }
}

static void
EventLoop(spTopLevel toplevel)
{
   int i;
    
   while (1) {
      spDebug(100, "EventLoop", "before spDispatchEvent: NumHeads = %d\n", NumHeads);
      spDispatchEvent(toplevel);

      /* redraw all windows */
      for (i = 0; i < NumHeads; i++) {
         Redraw(&Heads[i]);
      }
      /* swapbuffers on all windows, if not already done */
      if (SwapSeparate) {
         for (i = 0; i < NumHeads; i++) {
            Swap(&Heads[i]);
         }
      }
      spMSleep(1);
   }
}



static void
PrintInfo(const struct head *h)
{
   printf("Name: %s\n", h->DisplayName);
   printf("  Window:      0x%x\n", (int) h->Win);
   printf("  Canvas:      0x%x\n", (int) h->Canvas);
   printf("  Context:     0x%lx\n", (long) h->Context);
   printf("  GL_VERSION:  %s\n", h->Version);
   printf("  GL_VENDOR:   %s\n", h->Vendor);
   printf("  GL_RENDERER: %s\n", h->Renderer);
}

spComponent dialog, text, checkBox;

static void
DialogCB(spComponent component, int *numWindows)
{
    char *string;
    spBool flag;
    spCallbackReason reason;
    
    reason = spGetCallbackReason(component);
    
    if (reason == SP_CR_OK) {
        if (spGetToggleState(checkBox, &flag) == SP_TRUE
            && flag == SP_FALSE) {
            SwapSeparate = GL_FALSE;
        }
        if ((string = xspGetTextString(text)) != NULL) {
            *numWindows = atoi(string);
            xspFree(string);
        }
    }
    if (*numWindows <= 0) {
        spDisplayError(component, NULL, "Number of windows must be positive value.");
    } else {
        spPopdownWindow(component);
    }
}

static int
CreateNumWindowsDialog(void)
{
    int numWindows = 0;

    dialog = spCreateDialogBox("Options",
                               SppDialogBoxButtonType, SP_DB_OK_CANCEL,
                               SppCallbackFunc, DialogCB,
                               SppCallbackData, &numWindows,
                               NULL);
    text = spCreateParamField(dialog, "numWindowsField", 0,
                              SppTitle, "Number of Windows", 
                              SppFieldOffset, 120,
                              SppFieldSize, 120,
                              SppEditable, SP_TRUE,
                              SppFieldString, "3",
                              NULL);
    checkBox = spCreateCheckBox(dialog, "swapCheckBox",
                                SppTitle, "Swap immediately after drawing",
                                SppSet, SwapSeparate,
                                NULL);
    spPopupWindow(dialog);

    printf("Number of windows = %d\n", numWindows);

    return numWindows;
}

int
spMain(int argc, char *argv[])
{
   char *dpyName = NULL;
   int i;
   int n = 3;
   spTopLevel toplevel;
   
   /* initialize toolkit */
   toplevel = spInitialize(&argc, &argv, NULL);

   if (argc == 1) {
#if 0
      printf("manywin: open N simultaneous glx windows\n");
      printf("Usage:\n");
      printf("  manywin [-s] numWindows\n");
      printf("Options:\n");
      printf("  -s = swap immediately after drawing (see src code)\n");
      printf("Example:\n");
      printf("  manywin 10\n");
      return 0;
#else
      if ((n = CreateNumWindowsDialog()) <= 0) {
          return 0;
      }
#endif
   }
   
   for (i = 1; i < argc; i++) {
      if (strcmp(argv[i], "-s") == 0) {
         SwapSeparate = GL_FALSE;
      }
      else if (strcmp(argv[i], "-display") == 0 && i < argc) {
         dpyName = argv[i+1];
         i++;
      }
      else {
         n = atoi(argv[i]);
      }
   }
   if (n < 1)
      n = 1;

   printf("%d windows\n", n);
   for (i = 0; i < n; i++) {
      char name[100];
      struct head *h;
      sprintf(name, "%d", i);
      h = AddHead(toplevel, name);
      if (h) {
         PrintInfo(h);
      }
   }

   EventLoop(toplevel);
   DestroyHeads();
   return 0;
}
