/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>
#include <sp/spGL.h>
#include <sp/spComponentMain.h>

#if defined(SP_GL_ES)
#undef USE_DISPLAY_LIST
#define GL_FRONT GL_FRONT_AND_BACK
#else
#define USE_DISPLAY_LIST
#endif

#define SPEAKER_BOX_WIDTH 2.8
#define SPEAKER_BOX_HEIGHT 3.6
#define SPEAKER_BOX_DEPTH 2.3
#define SPEAKER_RADIUS 1.0
#define SPEAKER_DEPTH 0.5
#define SPEAKER_NDIV /*10*/50
#define SPEAKER_DUSTCAP_RADIUS 0.3
#define SPEAKER_DUSTCAP_SPHERE_RADIUS 0.5
#define SPEAKER_DUSTCAP_NDIV 3
#define SPEAKER_EDGE_SIZE /*0.12*/0.13
#define SPEAKER_EDGE_NDIV 4
#define SPEAKER_ANIMATION_DEPTH_CHANGE 0.2
#define SPEAKER_ANIMATION_TOTAL_COUNT 30

#if defined(USE_DISPLAY_LIST)
static GLuint newSpeaker(double radius, double depth, int ndiv,
                         double dustcap_radius, double dustcap_sphere_radius, int dustcap_ndiv,
                         double edge_size, int edge_ndiv);
#endif
static void createSpeaker(double radius, double depth, int ndiv,
                          double dustcap_radius, double dustcap_sphere_radius, int dustcap_ndiv,
                          double edge_size, int edge_ndiv);
static void createSpeakerBox(double width, double height, double depth, double radius, int ndiv, double edge_size);

static spGLContext context = NULL;
static spComponent g_canvas = NULL;

#if 0
static GLfloat light0pos[] = { 3.0, 3.0, -5.0, 1.0 };
static GLfloat light1pos[] = { -2.0, 7.0, 3.0, 1.0 };
#else
/*static GLfloat light0pos[] = { 0.0, 3.0, 5.0, 1.0 };*/
static GLfloat light0pos[] = { -3.0, 0.0, 2.0, 1.0 };
/*static GLfloat light1pos[] = { 5.0, 3.0, 0.0, 1.0 };*/
static GLfloat light1pos[] = { 3.0, 2.0, 2.0, 1.0 };
#endif
static GLfloat light2pos[] = { 2.0, 3.0, -2.0, 1.0 };

static GLfloat white[] = { 1.0, 1.0, 1.0, 1.0 };

static GLfloat g_x_rotation = 0.0;
static GLfloat g_y_rotation = 0.0;

static GLuint g_speaker_id = 0;
static GLuint g_speaker_box_id = 0;

static spTimerId g_timer_id = SP_TIMER_ID_NONE;
static long g_timer_interval = /*16*/1;

static double g_speaker_depth = SPEAKER_DEPTH;

void display(spComponent component)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLightfv(GL_LIGHT0, GL_POSITION, light0pos);
    glLightfv(GL_LIGHT1, GL_POSITION, light1pos);
    glLightfv(GL_LIGHT2, GL_POSITION, light2pos);

    glPushMatrix();

    /* rotation */
    glRotatef(g_x_rotation, 1.0, 0.0, 0.0);
    glRotatef(g_y_rotation, 0.0, 1.0, 0.0);

#if defined(USE_DISPLAY_LIST)
    glCallList(g_speaker_box_id);
    glCallList(g_speaker_id);
#else
    createSpeakerBox(SPEAKER_BOX_WIDTH, SPEAKER_BOX_HEIGHT, SPEAKER_BOX_DEPTH,
                     SPEAKER_RADIUS, SPEAKER_NDIV, SPEAKER_EDGE_SIZE);
    
    createSpeaker(SPEAKER_RADIUS, g_speaker_depth, SPEAKER_NDIV,
                  SPEAKER_DUSTCAP_RADIUS, SPEAKER_DUSTCAP_SPHERE_RADIUS, SPEAKER_DUSTCAP_NDIV,
                  SPEAKER_EDGE_SIZE, SPEAKER_EDGE_NDIV);
#endif

    glPopMatrix();

    spGLSwapBuffers(component);
}

void resize(int w, int h)
{
    GLdouble look_at_z;
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);

    look_at_z = /*5.0*/12.0;

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
#if 0
    glOrtho(-1.0, 1.0, -1.0, 1.0,
            look_at_z - 2.0, look_at_z + 2.0);
#else
    gluPerspective(30.0, (double)w / (double)h, 1.0, 100.0);
#endif

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(/*1.8*//*1.0*/3.0, /*1.2*/3.0, look_at_z,
              0.0, 0.0, 0.0,
              0.0, 1.0, 0.0);
}

double speakerAnimationFunc(double count, double total_count)
{
    double depth;
    double refcount;
    double move_rate;

    refcount = total_count * 0.85;
    if (count < refcount) {
        move_rate = count / refcount;
    } else {
        move_rate = (total_count - count) / (total_count - refcount);
    }
    depth = SPEAKER_DEPTH - SPEAKER_ANIMATION_DEPTH_CHANGE * move_rate;

    return depth;
}

spBool timerCB(spComponent frame, void *data)
{
    GLuint old_id;
    int *count = (int *)data;

    old_id = g_speaker_id;

    g_speaker_depth = speakerAnimationFunc((double)(*count), SPEAKER_ANIMATION_TOTAL_COUNT);

#if defined(USE_DISPLAY_LIST)
    g_speaker_id = newSpeaker(SPEAKER_RADIUS, g_speaker_depth, SPEAKER_NDIV,
                              SPEAKER_DUSTCAP_RADIUS, SPEAKER_DUSTCAP_SPHERE_RADIUS, SPEAKER_DUSTCAP_NDIV,
                              SPEAKER_EDGE_SIZE, SPEAKER_EDGE_NDIV);
    glDeleteLists(old_id, 1);
#endif

    spRefreshCanvas(g_canvas);
    spDebug(10, "timerCB", "done\n");

    (*count)--;

    if (*count <= 0) {
        return SP_FALSE;
    } else {
        return SP_TRUE;
    }
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int count = SPEAKER_ANIMATION_TOTAL_COUNT;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            g_y_rotation += (GLfloat)(x - px);
            g_x_rotation += (GLfloat)(y - py);
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        if (g_timer_id != SP_TIMER_ID_NONE) {
            spKillTimer(spGetWindow(component), g_timer_id);
        }
        count = SPEAKER_ANIMATION_TOTAL_COUNT;
        g_timer_id = spSetTimer(spGetWindow(component), g_timer_interval, timerCB, (void *)&count);
        
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE
        && code == SPK_Escape) {
        spQuit(0);
    }
}

static void setBlackRubberColor(void)
{
    GLfloat black_rubber_ambient[]  = {0.02F, 0.02F, 0.02F, 1.0F};
    GLfloat black_rubber_diffuse[] = {0.01F, 0.01F, 0.01F, 1.0F};
    GLfloat black_rubber_specular[] = {0.4F, 0.4F, 0.4F, 1.0F};
    GLfloat black_rubber_shininess[] = {10.0F};

    glMaterialfv(GL_FRONT, GL_AMBIENT, black_rubber_ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, black_rubber_diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, black_rubber_specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, black_rubber_shininess);

    return;
}

static void setBlackPlasticColor(void)
{
    GLfloat black_plastic_ambient[]  = {0.0F, 0.0F, 0.0F, 1.0F};
    GLfloat black_plastic_diffuse[]  = {0.01F, 0.01F, 0.01F, 1.0F};
    GLfloat black_plastic_specular[] = {0.50F, 0.50F, 0.50F, 1.0F};
    GLfloat black_plastic_shininess[]= {32.0F};

    glMaterialfv(GL_FRONT, GL_AMBIENT, black_plastic_ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, black_plastic_diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, black_plastic_specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, black_plastic_shininess);

    return;
}

static void createEdge(double radius, double depth, int ndiv, double edge_size, int edge_ndiv)
{
    int i, j;
    double theta;
    double dtheta;
    double xy_length;
    double xy_theta, xy_dtheta;
    double edge_radius;
    double size_weight;
    double cos_theta, sin_theta;
    double cos_xy_theta, sin_xy_theta;
    GLdouble v1[3], v2[3], n1[3], n2[3];

    dtheta = PI / (double)edge_ndiv;
    xy_dtheta = 2.0 * PI / (double)ndiv;
    edge_radius = edge_size / 2.0;
    size_weight = 1.0 / cos(xy_dtheta / 2);
    
    xy_theta = 0.0;
    for (j = 0; j < ndiv; j++) {
        glBegin(GL_QUAD_STRIP);

        theta = 0.0;
        for (i = 0; i <= edge_ndiv; i++) {
            cos_theta = cos(theta);
            sin_theta = sin(theta);
            xy_length = radius + edge_radius * size_weight * (1 + cos_theta);
            
            cos_xy_theta = cos(xy_theta);
            sin_xy_theta = sin(xy_theta);
            n1[0] = cos_theta * cos_xy_theta;
            n1[1] = cos_theta * sin_xy_theta;
            n1[2] = sin_theta;
            v1[0] = xy_length * cos_xy_theta;
            v1[1] = xy_length * sin_xy_theta;

            cos_xy_theta = cos(xy_theta + xy_dtheta);
            sin_xy_theta = sin(xy_theta + xy_dtheta);
            n2[0] = cos_theta * cos_xy_theta;
            n2[1] = cos_theta * sin_xy_theta;
            n2[2] = sin_theta;
            v2[0] = xy_length * cos_xy_theta;
            v2[1] = xy_length * sin_xy_theta;
            
            v1[2] = v2[2] = edge_size * sin_theta + depth;

            glNormal3dv(n1);
            glVertex3dv(v1);
            glNormal3dv(n2);
            glVertex3dv(v2);

            theta += dtheta;
        }
        
        xy_theta += xy_dtheta;
        
        glEnd();
    }
    
    
    return;
}

static void createDustcap(int ndiv, double depth_to_center, double dustcap_radius,
                          double dustcap_sphere_radius, int dustcap_ndiv)
{
    int i, j;
    double theta_pie;
    double theta;
    double dtheta;
    double xy_length;
    double xy_theta, xy_dtheta;
    GLdouble v1[3], v2[3], n1[3], n2[3];

    spDebug(50, "createDustcap", "dustcap_sphere_radius = %f, dustcap_radius = %f, depth_to_center = %f\n",
            dustcap_sphere_radius, dustcap_radius, depth_to_center);

    if (depth_to_center <= 0.0) {
        theta_pie = 0.0;
    } else {
        theta_pie = atan2(dustcap_radius, depth_to_center);
    }
    dtheta = theta_pie / (double)dustcap_ndiv;
    spDebug(50, "createDustcap", "theta_pie = %f, dtheta = %f\n", theta_pie, dtheta);

    xy_dtheta = 2.0 * PI / (double)ndiv;
    spDebug(50, "createDustcap", "xy_dtheta = %f\n", xy_dtheta);

    xy_theta = 0.0;
    for (j = 0; j < ndiv; j++) {
        glBegin(GL_QUAD_STRIP);
    
        theta = PI / 2.0 - theta_pie;
        for (i = 0; i < dustcap_ndiv; i++) {
            xy_length = cos(theta);
            n1[0] = xy_length * cos(xy_theta);
            n1[1] = xy_length * sin(xy_theta);
            v1[0] = n1[0] * dustcap_sphere_radius;
            v1[1] = n1[1] * dustcap_sphere_radius;
            
            n2[0] = xy_length * cos(xy_theta + xy_dtheta);
            n2[1] = xy_length * sin(xy_theta + xy_dtheta);
            v2[0] = n2[0] * dustcap_sphere_radius;
            v2[1] = n2[1] * dustcap_sphere_radius;
            
            n1[2] = n2[2] = sin(theta);
            v1[2] = v2[2] = dustcap_sphere_radius * n1[2];
            
            glNormal3dv(n1);
            glVertex3dv(v1);
            glNormal3dv(n2);
            glVertex3dv(v2);
        
            theta += dtheta;
        }
        
        n1[0] = 0.0;
        n1[1] = 0.0;
        n1[2] = 1.0;
        glNormal3dv(n1);

        v1[0] = v2[0] = 0.0;
        v1[1] = v2[1] = 0.0;
        v1[2] = v2[2] = dustcap_sphere_radius;
        glVertex3dv(v1);
        glVertex3dv(v2);

        xy_theta += xy_dtheta;
        
        glEnd();
    }
    
    return;
}

static void createCone(double radius, double depth, int ndiv, double dustcap_radius, double depth_to_center)
{
    int i;
    double theta_zn;
    double xy_n_projection;
    double theta, dtheta;
    double cos_theta, sin_theta;
    GLdouble n[3];
    GLdouble v1[3], v2[3];

    glBegin(GL_QUAD_STRIP);

    theta_zn = PI / 2.0 - atan2(depth, radius - dustcap_radius);
    xy_n_projection = -cos(theta_zn);
    n[2] = sin(theta_zn);

    dtheta = 2.0 * PI / (double)ndiv;

    v1[2] = depth_to_center;
    v2[2] = depth + depth_to_center;

    theta = dtheta;
    for (i = 0; i <= ndiv; i++) {
        cos_theta = cos(theta);
        sin_theta = sin(theta);
        v1[0] = dustcap_radius * cos_theta;
        v1[1] = dustcap_radius * sin_theta;
        v2[0] = radius * cos_theta;
        v2[1] = radius * sin_theta;

        n[0] = xy_n_projection * cos_theta;
        n[1] = xy_n_projection * sin_theta;
        
        glNormal3dv(n);
        glVertex3dv(v1);
        glVertex3dv(v2);
        
        theta += dtheta;
    }

    glEnd();
    
    return;
}

static void createSpeaker(double radius, double depth, int ndiv,
                          double dustcap_radius, double dustcap_sphere_radius, int dustcap_ndiv,
                          double edge_size, int edge_ndiv)
{
    double depth_total;
    double depth_to_center = 0.0;

    depth_to_center = sqrt(MAX(dustcap_sphere_radius * dustcap_sphere_radius
                               - dustcap_radius * dustcap_radius, 0.01));

    depth_total = depth_to_center + depth;
    glTranslatef(0.0, 0.0, -depth_total);

    setBlackRubberColor();
    createEdge(radius, depth_total, ndiv, edge_size, edge_ndiv);

    setBlackPlasticColor();

    createCone(radius, depth, ndiv, dustcap_radius, depth_to_center);
    createDustcap(ndiv, depth_to_center, dustcap_radius, dustcap_sphere_radius, dustcap_ndiv);
    
    return;
}

#if defined(USE_DISPLAY_LIST)
static GLuint newSpeaker(double radius, double depth, int ndiv,
                         double dustcap_radius, double dustcap_sphere_radius, int dustcap_ndiv,
                         double edge_size, int edge_ndiv)
{
    GLuint new_id;

    new_id = glGenLists(1);
    
    glNewList(new_id, GL_COMPILE);
    createSpeaker(radius, depth, ndiv,
                  dustcap_radius, dustcap_sphere_radius, dustcap_ndiv,
                  edge_size, edge_ndiv);
    glEndList();
    
    return new_id;
}
#endif

static void setSpeakerBoxColor(void)
{
    /*GLfloat ambient[]  = {0.15F, 0.06F, 0.05F, 1.0F};*/
    /*GLfloat ambient[]  = {0.15F, 0.08F, 0.05F, 1.0F};*/
    GLfloat ambient[]  = {0.18F, 0.08F, 0.05F, 1.0F};
    /*GLfloat ambient[]  = {0.2F, 0.08F, 0.05F, 1.0F};*/
    GLfloat diffuse[]  = {0.09F, 0.06F, 0.04F, 1.0F};
    /*GLfloat specular[] = {0.13F, 0.06F, 0.04F, 1.0F};*/
    /*GLfloat specular[] = {0.09F, 0.04F, 0.02F, 1.0F};*/
    GLfloat specular[] = {0.11F, 0.06F, 0.04F, 1.0F};
    GLfloat shininess[]= {/*32.0F*/8.0F};

    glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, shininess);

    return;
}

static void createSpeakerBoxFront(double width, double height, double depth,
                                  double radius, int ndiv, double edge_size)
{
    int j;
    double radius2;
    double w2, h2;
    double tan_thresh;
    double tan_theta;
    double cos_theta, sin_theta;
    double xy_theta, xy_dtheta;
    double value;
    GLdouble n[3], v1[3], v2[3];

    radius2 = radius + edge_size;
    xy_dtheta = 2.0 * PI / (double)ndiv;
    w2 = width / 2.0;
    h2 = height / 2.0;
    tan_thresh = height / width;

    glBegin(GL_QUAD_STRIP);
    
    n[0] = n[1] = 0.0;
    n[2] = 1.0;
    glNormal3dv(n);

    v1[2] = v2[2] = depth;
    v2[0] = v2[1] = 0.0;
    
    xy_theta = 0.0;
    for (j = 0; j <= ndiv; j++) {
        tan_theta = tan(xy_theta);
        cos_theta = cos(xy_theta);
        sin_theta = sin(xy_theta);

        v1[0] = radius2 * cos_theta;
        v1[1] = radius2 * sin_theta;
        
        if (tan_theta <= tan_thresh && tan_theta > -tan_thresh) {
            value = w2 * (cos_theta >= 0.0 ? 1.0 : -1.0);
            if (j != 0 && v2[0] != value) {
                v2[0] = value;
                v2[1] = h2 * (sin_theta >= 0.0 ? 1.0 : -1.0);
                glVertex3dv(v1);
                glVertex3dv(v2);
#if 0
                spDebug(-100, "createBoxFront", "edge w -> h: theta = %.0f, v1 = (%.2f, %.2f), v2 = (%.2f, %.2f)\n",
                        180.0 * xy_theta / PI, v1[0], v1[1], v2[0], v2[1]);
#endif
            }
            v2[0] = value;
            v2[1] = v2[0] * tan_theta;
        } else {
            value = h2 * (sin_theta >= 0.0 ? 1.0 : -1.0);
            if (j != 0 && v2[1] != value) {
                v2[1] = value;
                v2[0] = w2 * (cos_theta >= 0.0 ? 1.0 : -1.0);
                glVertex3dv(v1);
                glVertex3dv(v2);
#if 0
                spDebug(-100, "createBoxFront", "edge h -> w: theta = %.0f, v1 = (%.2f, %.2f), v2 = (%.2f, %.2f)\n",
                        180.0 * xy_theta / PI, v1[0], v1[1], v2[0], v2[1]);
#endif
            }
            v2[1] = value;
            v2[0] = cos_theta * v2[1] / sin_theta;
        }
#if 0
        spDebug(-100, "createBoxFront", "theta = %.0f, v1 = (%.2f, %.2f), v2 = (%.2f, %.2f)\n",
                180.0 * xy_theta / PI, v1[0], v1[1], v2[0], v2[1]);
#endif

        glVertex3dv(v1);
        glVertex3dv(v2);
        
        xy_theta += xy_dtheta;
    }

    glEnd();
    
    return;
}

static void createSpeakerBoxBack(double width, double height, double depth)
{
    int i;
    double w2, h2;
    GLdouble n[3], v1[3], v2[3], pv1[3], pv2[3];

    glBegin(GL_QUADS);
    
    w2 = width / 2.0;
    h2 = height / 2.0;
    
    n[2] = 0.0;
    v1[2] = pv1[2] = 0.0;
    v2[2] = pv2[2] = -depth;

    pv1[0] = pv2[0] = -w2;
    pv1[1] = pv2[1] = -h2;
    
    for (i = 0; i < 4; i++) {
        if (i == 0) {
            n[0] = 1.0; n[1] = 0.0;
            v1[0] = v2[0] = w2;
            v1[1] = v2[1] = -h2;
        } else if (i == 1) {
            n[0] = 0.0; n[1] = 1.0;
            v1[0] = v2[0] = w2;
            v1[1] = v2[1] = h2;
        } else if (i == 2) {
            n[0] = -1.0; n[1] = 0.0;
            v1[0] = v2[0] = -w2;
            v1[1] = v2[1] = h2;
        } else if (i == 3) {
            n[0] = 0.0; n[1] = -1.0;
            v1[0] = v2[0] = -w2;
            v1[1] = v2[1] = -h2;
        }
        glNormal3dv(n);
        glVertex3dv(pv1);
        glVertex3dv(pv2);
        glVertex3dv(v2);
        glVertex3dv(v1);
        
        pv1[0] = pv2[0] = v1[0];
        pv1[1] = pv2[1] = v2[1];
    }

    n[0] = n[1] = 0.0;
    n[2] = -1.0;
    glNormal3dv(n);
    
    pv2[0] = w2;
    pv2[1] = -h2;
    v2[0] = w2;
    v2[1] = h2;
    glVertex3dv(pv2);
    glVertex3dv(v2);

    pv2[0] = -w2;
    pv2[1] = h2;
    v2[0] = -w2;
    v2[1] = -h2;
    glVertex3dv(pv2);
    glVertex3dv(v2);
    
    glEnd();
    
    return;
}

static void createSpeakerBox(double width, double height, double depth, double radius, int ndiv, double edge_size)
{
    setSpeakerBoxColor();
    createSpeakerBoxBack(width, height, depth);
    createSpeakerBoxFront(width, height, 0.0, radius, ndiv, edge_size);
    
    return;
}
#if defined(USE_DISPLAY_LIST)
static GLuint newSpeakerBox(double width, double height, double depth, double radius, int ndiv, double edge_size)
{
    GLuint new_id;

    new_id = glGenLists(1);
    
    glNewList(new_id, GL_COMPILE);
    createSpeakerBox(width, height, depth, radius, ndiv, edge_size);
    glEndList();
    
    return new_id;
}
#endif

void init(void)
{
#if 1
    /* white background */
    glClearColor(1.0, 1.0, 1.0, 1.0);
#else
    /* black background */
    glClearColor(0.0, 0.0, 0.0, 1.0);
#endif

    glEnable(GL_DEPTH_TEST);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glEnable(GL_LIGHT2);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT0, GL_SPECULAR, white);
    glLightfv(GL_LIGHT1, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT1, GL_SPECULAR, white);
    glLightfv(GL_LIGHT2, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT2, GL_SPECULAR, white);

#if defined(USE_DISPLAY_LIST)
    g_speaker_box_id = newSpeakerBox(SPEAKER_BOX_WIDTH, SPEAKER_BOX_HEIGHT, SPEAKER_BOX_DEPTH,
                                     SPEAKER_RADIUS, SPEAKER_NDIV, SPEAKER_EDGE_SIZE);
    
    g_speaker_id = newSpeaker(SPEAKER_RADIUS, SPEAKER_DEPTH, SPEAKER_NDIV,
                              SPEAKER_DUSTCAP_RADIUS, SPEAKER_DUSTCAP_SPHERE_RADIUS, SPEAKER_DUSTCAP_NDIV,
                              SPEAKER_EDGE_SIZE, SPEAKER_EDGE_NDIV);
#endif
    
    spDebug(10, "init", "done\n");
}

void canvasCB(spComponent component, void *data)
{
    static int w = 0, h = 0;

    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || w == 0) {
        if (w == 0) {
            init();
        }
        
        if (spGetSize(component, &w, &h) == SP_FALSE) return;
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent window, canvas;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_DEPTH_SIZE, 1,
                                SP_GL_NONE };
  
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Speaker", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);
  
    context = spCreateGLContext(canvas, NULL);

    spPopupWindow(window);
  
    spSetGLContext(canvas, context);
    g_canvas = canvas;
  
    return spMainLoop(toplevel);
}
