/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>
#include <sp/spGL.h>
#include <sp/spGLString.h>

#include <sp/spComponentMain.h>


#define FREQUNCY_STRING_COLOR /*"white"*/"blue"
#define MAGNITUDE_STRING_COLOR /*"white"*/"red"
#define TIME_STRING_COLOR /*"white"*/"green"

//#define USE_SPOT_LIGHT
//#define USE_CLIP_PLANE

#define USE_TEXTURE_FOR_GLSTRING
//#define DISABLE_TIMER

#if defined(SP_GL_ES)
#undef GL_FRONT                        /* GL_FRONT doesn't work in glMaterialfv */
#define GL_FRONT GL_FRONT_AND_BACK
#undef USE_DISPLAY_LIST
#else
/*#define USE_DISPLAY_LIST*/
#undef USE_DISPLAY_LIST
#endif

#if defined(MACOS) && !defined(COCOA)
/* non-Cocoa MacOS doesn't support texture-based spGLString */
#undef USE_TEXTURE_FOR_GLSTRING
#endif

static spGLContext context = NULL;
static spComponent g_canvas = NULL;
static spGLString freq_string = NULL;
static spGLString amp_string = NULL;
static spGLString time_string = NULL;

static spBool init_done = SP_FALSE;

//static GLfloat light0pos[] = { 3.0f, 3.0f, -5.0f, 1.0f };
//static GLfloat light0pos[] = { 0.5f, 1.0f, -0.5f, 1.0f };
//static GLfloat light0pos[] = { 0.375f, 1.2f, -0.5f, 1.0f };
//static GLfloat light0pos[] = { 0.0f, 2.0f, 0.0f, 1.0f };
static GLfloat light0pos[] = { -0.125f, 0.7f, 0.0f, 1.0f };
static GLfloat light0_spot_direction[] = { 0.0f, -1.0f, 0.0f };
static GLfloat light1pos[] = { -2.0f, 7.0f, 3.0f, 1.0f };
static GLfloat light1_spot_direction[] = { 2.0f, -7.0f, -3.0f };

static GLfloat spot_cutoff = 30.0f;
static GLfloat spot_exponent = 15.0f;

static GLfloat white[] = { 1.0f, 1.0f, 1.0f, 1.0f };

#if defined(USE_CLIP_PLANE)
#if defined(SP_GL_ES)
#define glClipPlane glClipPlanef
#endif
static GLdouble clip_plane_x[] = { -1.0f, 0.0f, 0.0f, /*0.125f*/0.0f };
static GLdouble clip_plane_y[] = { 0.0f, 1.0f, 0.0f, -0.25f };
static GLdouble clip_plane_z[] = { 0.0f, 0.0f, 1.0f, /*0.5f*/0.0f };
#endif

double **g_specgram = NULL;
GLdouble ***g_normal = NULL;

static double g_specgram_range = 100.0;

static double g_x_rotation = 0.0;
static double g_y_rotation = 0.0;

static double g_arrow_size = 0.05;

#if 0
static long g_hfftl = 513;
static double g_total_ms = 500.0;
static double g_shift_ms = 8.0;
static double g_samp_freq = 16000.0;
#else
static long g_hfftl = 127;
static double g_total_ms = 160.0;
static double g_shift_ms = 16.0;
static double g_samp_freq = 8000.0;
#endif

static long g_nframes = /*1000*//*100*/0;
#if defined(USE_DISPLAY_LIST)
static GLuint g_specgram_id = 0;
#endif
static double g_spec_freq = 2000.0;
static double g_amp_mod_freq = 3.0;
static double g_amp_mod_amp = 2.0;
static double g_specgram_min = -40.0;
static double g_specgram_max = 20.0;

#if !defined(DISABLE_TIMER)
static spTimerId g_timer_id = SP_TIMER_ID_NONE;
static long g_timer_interval = /*16*/200;
#endif

GLfloat material_colors[][4][4] =
{
    /* 0: ambient, 1: diffuse, 2: specular, 3: shininess */
    {/* default */
        {0.2f,     0.2f,      0.2f,     1.0f},
        {0.8f,     0.8f,      0.8f,     1.0f},
        {0.0f,     0.0f,      0.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     0.0f},
    },
    {/* white */
        {0.2f,     0.2f,      0.2f,     1.0f},
        {1.0f,     1.0f,      1.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     1.0f},
        {0.0f,     0.0f,      0.0f,     0.0f},
    },
    {/* emerald */
        {0.0215f,  0.1745f,   0.0215f,  1.0f},
        {0.07568f, 0.61424f,  0.07568f, 1.0f},
        {0.633f,   0.727811f, 0.633f,   1.0f},
        {76.8f,    0.0f,      0.0f,     0.0f},
    },
    {/* jade */
        {0.135f,     0.2225f,   0.1575f,   1.0f},
        {0.54f,      0.89f,     0.63f,     1.0f},
        {0.316228f,  0.316228f, 0.316228f, 1.0f},
        {12.8f,      0.0f,      0.0f,     0.0f},
    },
    {/* obsidian */
        {0.05375f, 0.05f,    0.06625f, 1.0f},
        {0.18275f, 0.17f,    0.22525f, 1.0f},
        {0.332741f,0.328634f,0.346435f,1.0f},
        {38.4f,    0.0f,     0.0f,     0.0f},
    },
    {/* pearl */
        {0.25f,     0.20725f,  0.20725f,  1.0f},
        {1.0f,      0.829f,    0.829f,    1.0f},
        {0.296648f, 0.296648f, 0.296648f, 1.0f},
        {10.24f,    0.0f,      0.0f,     0.0f},
    },
    {/* ruby */
        {0.1745f,   0.01175f,  0.01175f,   1.0f},
        {0.61424f,  0.04136f,  0.04136f,   1.0f},
        {0.727811f, 0.626959f, 0.626959f,  1.0f},
        {76.8f,     0.0f,      0.0f,     0.0f},
    },
    {/* turquoise */
        {0.1f,     0.18725f, 0.1745f,  1.0f},
        {0.396f,   0.74151f, 0.69102f, 1.0f},
        {0.297254f,0.30829f, 0.306678f,1.0f},
        {12.8f,    0.0f,     0.0f,     0.0f},
    },
    {/* brass */
        {0.329412f,  0.223529f, 0.027451f, 1.0f},
        {0.780392f,  0.568627f, 0.113725f, 1.0f},
        {0.992157f,  0.941176f, 0.807843f, 1.0f},
        {25.6f,      0.0f,      0.0f,     0.0f},
    },
    {/* bronze */
        {0.2125f,   0.1275f,   0.054f,   1.0f},
        {0.714f,    0.4284f,   0.18144f, 1.0f},
        {0.393548f, 0.271906f, 0.166721f,1.0f},
        {25.6f,     0.0f,      0.0f,     0.0f},
    },
    {/* chrome */
        {0.25f,    0.25f,     0.25f,     1.0f},
        {0.4f,     0.4f,      0.4f,      1.0f},
        {0.774597f,0.774597f, 0.774597f, 1.0f},
        {76.8f,    0.0f,      0.0f,     0.0f},
    },
    {/* copper */
        {0.19125f,  0.0735f,   0.0225f,  1.0f},
        {0.7038f,   0.27048f,  0.0828f,  1.0f},
        {0.256777f, 0.137622f, 0.086014f,1.0f},
        {12.8f,     0.0f,      0.0f,     0.0f},
    },
    {/* gold */
        {0.24725f,  0.1995f,   0.0745f,    1.0f},
        {0.75164f,  0.60648f,  0.22648f,   1.0f},
        {0.628281f, 0.555802f, 0.366065f,  1.0f},
        {51.2f,     0.0f,      0.0f,     0.0f},
    },
    {/* silver */
        {0.19225f,  0.19225f,  0.19225f, 1.0f},
        {0.50754f,  0.50754f,  0.50754f, 1.0f},
        {0.508273f, 0.508273f, 0.508273f,1.0f},
        {51.2f,     0.0f,      0.0f,     0.0f},
    },
};

enum {
    MATERIAL_DEFAULT_INDEX = 0,
    MATERIAL_WHITE_INDEX,
    MATERIAL_EMERALD_INDEX,
    MATERIAL_JADE_INDEX,
    MATERIAL_OBSIDIAN_INDEX,
    MATERIAL_PEARL_INDEX,
    MATERIAL_RUBY_INDEX,
    MATERIAL_TURQUOISE_INDEX,
    MATERIAL_BRASS_INDEX,
    MATERIAL_BRONZE_INDEX,
    MATERIAL_CHROME_INDEX,
    MATERIAL_COPPER_INDEX,
    MATERIAL_GOLD_INDEX,
    MATERIAL_SILVER_INDEX,
};

static int g_material_index = MATERIAL_GOLD_INDEX;

static double **allocPseudoSpecgram(long nframes, long hfftl)
{
    return xspDMatAlloc(nframes, hfftl);
}

static void freePseudoSpecgram(double **specgram, long nframes)
{
    xspDMatFree(specgram, nframes);
}

static GLdouble ***allocSpecgramNormal(long nframes, long hfftl)
{
    long k, l;
    long hfftl2;
    long nnframes;
    GLdouble ***normal;

    hfftl2 = hfftl * 2;

    nnframes = nframes - 1;
    normal = xspAlloc(nnframes, GLdouble **);
    for (k = 0; k < nnframes; k++) {
        normal[k] = xspAlloc(hfftl2, GLdouble *);
        normal[k][0] = xspAlloc(hfftl2 * 3, GLdouble);
        for (l = 1; l < hfftl2; l++) {
            normal[k][l] = normal[k][0] + l * 3;
        }
    }

    return normal;
}

static void freeSpecgramNormal(GLdouble ***normal, long nframes, long hfftl)
{
    long k;
    
    for (k = 0; k < nframes - 1; k++) {
        xspFree(normal[k][0]);
        xspFree(normal[k]);
    }
    xspFree(normal);
}

static void calculatePseudoSpecgram(double **specgram, long nframes, long hfftl, 
                                    double fs, double shift_ms,
                                    double min, double max,
                                    double spec_freq,
                                    double amp_mod_amp,
                                    double amp_mod_freq,
                                    long offset)
{
    long k, l;
    double hrange;
    double amp_bias;
    double fweight, aweight;
    static long amp_mod_index = 0;

    hrange = (max - min) / 2.0;

    fweight = (PI * fs) / (spec_freq * (double)(hfftl - 1));
    aweight = 2.0 * PI * (shift_ms * amp_mod_freq / 1000.0);
    
    for (k = offset; k < nframes; k++) {
        if (amp_mod_amp != 0.0) {
            amp_bias = amp_mod_amp * sin(aweight * (double)amp_mod_index);
            amp_mod_index++;
        } else {
            amp_bias = 0.0;
        }
        for (l = 0; l < hfftl; l++) {
            specgram[k][l] = hrange * sin(fweight * (double)l) + amp_bias;
        }
    }
    
    return;
}

static void updatePseudoSpecgram(double **specgram, long nframes, long hfftl, 
                                 double fs, double shift_ms,
                                 double min, double max,
                                 double spec_freq,
                                 double amp_mod_amp,
                                 double amp_mod_freq,
                                 long shift)
{
    long offset;

    offset = nframes - shift;
    
    if (shift > 0) {
        /* shift current spectrogram */
        memmove(specgram[0], specgram[shift], sizeof(double) * hfftl * offset);
    }
    
    /* calculate new spectrogram */
    calculatePseudoSpecgram(specgram, nframes, hfftl,
                            fs, shift_ms,
                            min, max,
                            spec_freq,
                            amp_mod_amp, amp_mod_freq,
                            offset);
    
    return;
}

static void calculateNormalizedProduct(double ax, double ay, double az,
                                       double bx, double by, double bz,
                                       double *ppx, double *ppy, double *ppz)
{
    double norm;
    double px, py, pz;
    
    px = ay * bz - az * by;
    py = az * bx - ax * bz;
    pz = ax * by - ay * bx;

    norm = sqrt(px * px + py * py + pz * pz);
    
    *ppx = px / norm;
    *ppy = py / norm;
    *ppz = pz / norm;
}

static void calculateSpecgramNormal(GLdouble ***normal, double **specgram, long nframes, long hfftl,
                                    long offset)
{
    long k, l;
    double fx, fy, fz;
    double tx, ty, tz;
    double px, py, pz;

    for (k = offset + 1; k < nframes; k++) {
        for (l = 1; l < hfftl; l++) {
            fx = 1.0;
            fy = specgram[k - 1][l] - specgram[k - 1][l - 1];
            fz = 0.0;
            
            tx = 0.0;
            ty = specgram[k][l - 1] - specgram[k - 1][l - 1];
            tz = -1.0;
            
            calculateNormalizedProduct(fx, fy, fz, tx, ty, tz, &px, &py, &pz);
            /*spDebug(-100, "calculateSpecgramNormal", "1: px = %f, py = %f, pz = %f\n", px, py, pz);*/
            normal[k - 1][(l - 1) * 2][0] = px;
            normal[k - 1][(l - 1) * 2][1] = py;
            normal[k - 1][(l - 1) * 2][2] = pz;

            fx = -fx;
            fy = specgram[k][l - 1] - specgram[k][l];
            
            ty = specgram[k - 1][l] - specgram[k][l];
            tz = -tz;
            calculateNormalizedProduct(fx, fy, fz, tx, ty, tz, &px, &py, &pz);
            /*spDebug(-100, "calculateSpecgramNormal", "2: px = %f, py = %f, pz = %f\n", px, py, pz);*/
            normal[k - 1][(l - 1) * 2 + 1][0] = px;
            normal[k - 1][(l - 1) * 2 + 1][1] = py;
            normal[k - 1][(l - 1) * 2 + 1][2] = pz;
        }
    }

    return;
}

static void updateSpecgramNormal(GLdouble ***normal, double **specgram, long nframes, long hfftl,
                                 long shift)
{
    long offset;
    long k;
    long hfftl2;

    hfftl2 = hfftl * 2;
    offset = nframes -1 - shift;

    /* shift current normal vectors */
    for (k = shift; k < nframes - 1; k++) {
        memmove(normal[k - shift][0], normal[k][0], sizeof(GLdouble) * hfftl2 * 3);
    }
    
    /* calculate new normal vectors */
    calculateSpecgramNormal(normal, specgram, nframes, hfftl, offset);

    return;
}


static void getCurrentSpecVertex(GLdouble *v, double **specgram, long nframes, long hfftl,
                                 double specgram_max, double specgram_range,
                                 long frame, long fn)
{
    v[0] = (GLfloat)fn / (GLfloat)/*hfftl*/(hfftl - 1);
    v[1] = (specgram[frame][fn] - (specgram_max - specgram_range)) / specgram_range;
    v[2] = -((GLfloat)frame / (GLfloat)/*nframes*/(nframes - 1));
    
    return;
}

static void setMaterialColor(int index)
{
    glMaterialfv(GL_FRONT, GL_AMBIENT, material_colors[index][0]);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, material_colors[index][1]);
    glMaterialfv(GL_FRONT, GL_SPECULAR, material_colors[index][2]);
    glMaterialfv(GL_FRONT, GL_SHININESS, material_colors[index][3]);
    return;
}

static void createSpecgram(double **specgram, GLdouble ***normal, long nframes, long hfftl,
                           double specgram_max, double specgram_range, int material_index)
{
    long k, l;
    GLdouble v[3];
    int triangles = 1;

    glTranslatef(-0.5, -0.5, 0.5);
    
    if (triangles) {
        glBegin(GL_TRIANGLES);
    }

    setMaterialColor(material_index);

    for (k = 1; k < nframes; k++) {
        for (l = 1; l < hfftl; l++) {
            if (triangles) {
                glNormal3dv(normal[k - 1][(l - 1) * 2]);
            } else {
                glBegin(GL_LINE_LOOP);
            }

            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k - 1, l - 1);
            glVertex3dv(v);
            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k - 1, l);
            glVertex3dv(v);
            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k, l - 1);
            glVertex3dv(v);
            
            if (triangles) {
                glNormal3dv(normal[k - 1][(l - 1) * 2 + 1]);
            } else {
                glEnd();
                glBegin(GL_LINE_LOOP);
            }
            
            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k, l - 1);
            glVertex3dv(v);
            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k - 1, l);
            glVertex3dv(v);
            getCurrentSpecVertex(v, specgram, nframes, hfftl, specgram_max, specgram_range, k, l);
            glVertex3dv(v);
            
            if (!triangles) {
                glEnd();
            }
        }
    }

    if (triangles) {
        glEnd();
    }

    return;
}

static void createAxes(spComponent component)
{
    double arrow_size2;
    
    setMaterialColor(MATERIAL_WHITE_INDEX);
    
    glBegin(GL_LINES);
    
    /* X axis */
    glNormal3d(1.0, 0.0, 0.0);
    glVertex3d(0.0, 0.0, 0.0);
    glVertex3d(1.2, 0.0, 0.0);

    arrow_size2 = g_arrow_size / 2.0;
    glVertex3d(1.2, 0.0, 0.0);
    glVertex3d(1.2 - g_arrow_size, arrow_size2, 0.0);
    glVertex3d(1.2, 0.0, 0.0);
    glVertex3d(1.2 - g_arrow_size, -arrow_size2, 0.0);

    
    /* Y axis */
    glNormal3d(0.0, 1.0, 0.0);
    glVertex3d(0.0, 0.0, 0.0);
    glVertex3d(0.0, 1.2, 0.0);

    glVertex3d(0.0, 1.2, 0.0);
    glVertex3d(arrow_size2, 1.2 - g_arrow_size, 0.0);
    glVertex3d(0.0, 1.2, 0.0);
    glVertex3d(-arrow_size2, 1.2 - g_arrow_size, 0.0);

    
    /* Z axis */
    glNormal3d(0.0, 0.0, -1.0);
    glVertex3d(0.0, 0.0, 0.0);
    glVertex3d(0.0, 0.0, -1.2);

    glVertex3d(0.0, 0.0, -1.2);
    glVertex3d(0.0, arrow_size2, -1.2 + g_arrow_size);
    glVertex3d(0.0, 0.0, -1.2);
    glVertex3d(0.0, -arrow_size2, -1.2 + g_arrow_size);
    
    glEnd();

    glPushMatrix();
    spSetGLStringPos3f(freq_string, 1.0f, -0.1f, 0.0f);
    spDrawGLString(component, freq_string);
    spSetGLStringPos3f(amp_string, -0.1f, 1.3f, 0.0f);
    spDrawGLString(component, amp_string);
    spSetGLStringPos3f(time_string, 0.0f, -0.1f, -1.0f);
    spDrawGLString(component, time_string);
    glPopMatrix();

    return;
}

#if defined(USE_DISPLAY_LIST)
static GLuint newSpecgram(double **specgram, GLdouble ***normal, long nframes, long hfftl,
                          double max, double specgram_range, int material_index)
{
    GLuint new_id;
    
    new_id = glGenLists(1);
    
    glNewList(new_id, GL_COMPILE);
    createSpecgram(specgram, normal, nframes, hfftl,
                   max, specgram_range, material_index);
    glEndList();

    return new_id;
}
#endif

static void updateSpecgram(double **specgram, GLdouble ***normal, long nframes, long hfftl,
                           double fs, double shift_ms,
                           double min, double max, double specgram_range,
                           double spec_freq,
                           double amp_mod_amp,
                           double amp_mod_freq,
                           int material_index,
                           long shift)
{
    updatePseudoSpecgram(specgram, nframes, hfftl,
                         fs, shift_ms,
                         min, max,
                         spec_freq,
                         amp_mod_amp, amp_mod_freq,
                         shift);

    updateSpecgramNormal(normal, specgram, nframes, hfftl, shift);
    
#if defined(USE_DISPLAY_LIST)
    {
        GLuint old_id, new_id;
    
        new_id = newSpecgram(specgram, normal, nframes, hfftl,
                             max, specgram_range, material_index);

        /* update g_specgram_id */
        old_id = g_specgram_id;
        g_specgram_id = new_id;
    
        /* delete old g_specgram_id */
        glDeleteLists(old_id, 1);
    }
#endif
    
    return;
}

void display(spComponent component)
{
    spDebug(10, "display", "in\n");
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

#if 1 && defined(USE_CLIP_PLANE)
    glClipPlane(GL_CLIP_PLANE0 + 0, clip_plane_x);
    glEnable(GL_CLIP_PLANE0 + 0);
    glClipPlane(GL_CLIP_PLANE0 + 1, clip_plane_y);
    glEnable(GL_CLIP_PLANE0 + 1);
    glClipPlane(GL_CLIP_PLANE0 + 2, clip_plane_z);
    glEnable(GL_CLIP_PLANE0 + 2);
#endif
    
    glPushMatrix();

    /* rotation */
    glRotatef((GLfloat)g_x_rotation, 1.0f, 0.0f, 0.0f);
    glRotatef((GLfloat)g_y_rotation, 0.0f, 1.0f, 0.0f);

    glLightfv(GL_LIGHT0, GL_POSITION, light0pos);
    glLightfv(GL_LIGHT1, GL_POSITION, light1pos);
#if defined(USE_SPOT_LIGHT)
    glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, light0_spot_direction);
    glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, light1_spot_direction);
#endif

#if 0 && defined(USE_CLIP_PLANE)
    glClipPlane(GL_CLIP_PLANE0 + 0, clip_plane_x);
    glEnable(GL_CLIP_PLANE0 + 0);
    glClipPlane(GL_CLIP_PLANE0 + 1, clip_plane_y);
    glEnable(GL_CLIP_PLANE0 + 1);
    glClipPlane(GL_CLIP_PLANE0 + 2, clip_plane_z);
    glEnable(GL_CLIP_PLANE0 + 2);
#endif
    
#if defined(USE_DISPLAY_LIST)
    glCallList(g_specgram_id);
#else
    createSpecgram(g_specgram, g_normal, g_nframes, g_hfftl,
                   g_specgram_max, g_specgram_range, g_material_index);
#endif
    createAxes(component);

    glPopMatrix();

#if defined(USE_CLIP_PLANE)
    glDisable(GL_CLIP_PLANE0 + 0);
    glDisable(GL_CLIP_PLANE0 + 1);
    glDisable(GL_CLIP_PLANE0 + 2);
#endif
    
    spGLSwapBuffers(component);

    spGLWaitGLDrawing(component);
    spGLWaitSystemDrawing(component);
    
    spDebug(10, "display", "done\n");
}

void resize(int w, int h)
{
    GLdouble look_at_z;
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);

    look_at_z = 5.0;

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-0.8, 0.8, -0.8, 0.8,
             look_at_z - 2.0, look_at_z + 2.0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(1.8, 1.2, look_at_z,
              0.0, 0.0, 0.0,
              0.0, 1.0, 0.0);

    spDebug(10, "resize", "done\n");
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            g_y_rotation += (double)(x - px);
            g_x_rotation += (double)(y - py);
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE
        && code == SPK_Escape) {
        spQuit(0);
    }
}

#if !defined(DISABLE_TIMER)
spBool timerCB(spComponent frame, void *data)
{
    spComponent canvas = (spComponent)data;

    if (init_done == SP_TRUE) {
        updateSpecgram(g_specgram, g_normal, g_nframes, g_hfftl,
                       g_samp_freq, g_shift_ms,
                       g_specgram_min, g_specgram_max, g_specgram_range,
                       g_spec_freq, g_amp_mod_amp, g_amp_mod_freq,
                       g_material_index,
                       1);
        spRefreshCanvas(canvas);
    }
    spDebug(10, "timerCB", "done\n");
    
    return SP_TRUE;
}
#endif

void changeMaterialCB(spComponent component, int material_index)
{
    g_material_index = material_index;
    return;
}

void init(void)
{
#if 0
    /* white background */
    glClearColor(1.0, 1.0, 1.0, 1.0);
#else
    /* black background */
    glClearColor(0.0, 0.0, 0.0, 1.0);
#endif

    glEnable(GL_DEPTH_TEST);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glLightfv(GL_LIGHT1, GL_DIFFUSE, white);
    glLightfv(GL_LIGHT1, GL_SPECULAR, white);
#if defined(USE_SPOT_LIGHT)
    glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, spot_cutoff);
    glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, spot_exponent);
    glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, spot_cutoff);
    glLightf(GL_LIGHT1, GL_SPOT_EXPONENT, spot_exponent);
#endif

    g_nframes = (long)spRound(g_total_ms / g_shift_ms);
    g_specgram = allocPseudoSpecgram(g_nframes, g_hfftl);
    calculatePseudoSpecgram(g_specgram, g_nframes, g_hfftl,
                            g_samp_freq, g_shift_ms,
                            g_specgram_min, g_specgram_max,
                            g_spec_freq,
                            g_amp_mod_amp, g_amp_mod_freq, 0);

    g_normal = allocSpecgramNormal(g_nframes, g_hfftl);
    calculateSpecgramNormal(g_normal, g_specgram, g_nframes, g_hfftl, 0);
    
#if defined(USE_DISPLAY_LIST)
    g_specgram_id = newSpecgram(g_specgram, g_normal, g_nframes, g_hfftl,
                                g_specgram_max, g_specgram_range, g_material_index);
    spDebug(10, "init", "g_specgram_id = %d\n", g_specgram_id);
#endif

    freq_string = spCreateGLString(g_canvas, "Frequency (Hz)",
                                   SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                   SppForeground, FREQUNCY_STRING_COLOR,
#if defined(USE_TEXTURE_FOR_GLSTRING)
                                   SppGLStringUseTexture, SP_TRUE,
#endif
                                   NULL);
    amp_string = spCreateGLString(g_canvas, "Magnitude (dB)",
                                  SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                  SppForeground, MAGNITUDE_STRING_COLOR,
#if defined(USE_TEXTURE_FOR_GLSTRING)
                                  SppGLStringUseTexture, SP_TRUE,
#endif
                                  NULL);
    time_string = spCreateGLString(g_canvas, "Time (sec)",
                                   SppFontName, "-*-*-medium-*-normal--16-*-*-*-*-*-*-*",
                                   SppForeground, TIME_STRING_COLOR,
#if defined(USE_TEXTURE_FOR_GLSTRING)
                                   SppGLStringUseTexture, SP_TRUE,
#endif
                                   NULL);

    init_done = SP_TRUE;
    spDebug(10, "init", "done\n");
}

void canvasCB(spComponent component, void *data)
{
    int w = 0, h = 0;
    double scaling_factor = 1.0;
    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || init_done == SP_FALSE) {
        if (init_done == SP_FALSE) {
            init();
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;

        if ((scaling_factor = spGetGLCanvasScalingFactor(component)) != 1.0 && scaling_factor != 0.0) {
            w = (int)spRound(w * scaling_factor);
            h = (int)spRound(h * scaling_factor);
        }
        spDebug(10, "canvasCB", "scaling_factor = %f\n", scaling_factor);
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent window, canvas, menu, submenu;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_DEPTH_SIZE, 1,
                                SP_GL_NONE };
  
    init_done = SP_FALSE;

    /*spSetDebugLevel(100);*/
    
    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Spectrogram Test", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              SppGLAutoScaling, SP_FALSE,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);

    menu = spCreatePopupMenu(canvas, "popupMenu",
                             SppPopupButton, SP_RBUTTON,
                             NULL);
    submenu = spAddSubMenu(menu, "materialSubMenu",
                           SppTitle, "Change Spectrogram Material",
                           NULL);
    spAddRadioButtonMenuItem(submenu, "Emerald",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_EMERALD_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Jade",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_JADE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Obsidian",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_OBSIDIAN_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Pearl",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_PEARL_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Ruby",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_RUBY_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Turquoise",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_TURQUOISE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Bronze",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_BRONZE_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Copper",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_COPPER_INDEX,
                             NULL);
    spAddRadioButtonMenuItem(submenu, "Gold",
                             SppCallbackFunc, changeMaterialCB,
                             SppCallbackData, MATERIAL_GOLD_INDEX,
                             SppSet, SP_TRUE,
                             NULL);
  
    context = spCreateGLContext(canvas, NULL);

    spPopupWindow(window);
  
    spSetGLContext(canvas, context);
    g_canvas = canvas;
  
#if !defined(DISABLE_TIMER)
    g_timer_id = spSetTimer(window, g_timer_interval, timerCB, canvas);
#endif
    
    return spMainLoop(toplevel);
}
