/*
 * spComponentEx  OpenGL sample code  by Hideki Banno
 */

#include <math.h>
#include <sp/spComponentLib.h>
#include <sp/spGL.h>
#include <sp/spComponentMain.h>

#if 0
#define USE_MULTISAMPLE
#else
#undef USE_MULTISAMPLE
#endif
#define MULTISAMPLE_SAMPLES 4

static spComponent window = NULL;

static spGLContext context = NULL;
static spDIBitmap dibitmap = NULL;

static GLuint tex_name = 0;

static GLfloat g_x_rotation = 0.0f;
static GLfloat g_y_rotation = 0.0f;

void resize(int w, int h)
{
    spDebug(10, "resize", "w = %d, h = %d\n", w, h);
  
    glViewport(0, 0, w, h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);

    return;
}

void init(void)
{
    GLint internal_format;
    GLenum format;
    GLenum type;
    
#if defined(USE_MULTISAMPLE)
    GLint buffers, samples;
 
    glGetIntegerv(GL_SAMPLE_BUFFERS, &buffers);
    glGetIntegerv(GL_SAMPLES, &samples);
    spDebug(10, "init", "buffers = %d, samples = %d\n", buffers, samples);
    if (buffers >= 1 && samples >= 2) {
        glEnable(GL_MULTISAMPLE);
    }
#endif
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);

    if (dibitmap->info.pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK) {
        GLenum sfactor = GL_SRC_ALPHA, dfactor = GL_ONE_MINUS_SRC_ALPHA;
        spDebug(10, "init", "with alpha\n");
        glEnable(GL_BLEND);
        spGetGLDIPixelFormatAlphaBlendFactor(dibitmap->info.pixel_format, &sfactor, &dfactor);
        glBlendFunc(sfactor, dfactor);
    }
    
    glGenTextures(1, &tex_name);
    glBindTexture(GL_TEXTURE_2D, tex_name);

    if (spGetGLDIBitmapFormat(dibitmap, &internal_format, &format, &type)) {
        spDebug(10, "init", "dibitmap->info.pixel_format = %lx, internal_format = %x, format = %x, type = %x\n",
                dibitmap->info.pixel_format, internal_format, format, type);

        glPixelStorei(GL_UNPACK_ALIGNMENT, spGetGLDIBitmapUnpackAlignment(dibitmap));
        glPixelStorei(GL_UNPACK_ROW_LENGTH, spGetGLDIBitmapUnpackRowLength(dibitmap));
    
        spLockDIBitmap(dibitmap);
        spDebug(10, "init", "dibitmap->info.data = %ld\n", (long)dibitmap->info.data);
        glTexImage2D(GL_TEXTURE_2D, 0, internal_format, dibitmap->info.width, dibitmap->info.height,
                     0, format, type, dibitmap->info.data);    
        spUnlockDIBitmap(dibitmap);
        
        glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
    } else {
        spDisplayWarning(window, NULL, "This bitmap file isn't suitable for OpenGL texture.\n");
        spDebug(1, "init", "spGetGLDIBitmapFormat failed\n");
    }
    
#if 0
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
#else
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
#endif
    
    glClearColor(0.6f, 0.6f, 1.0f, 1.0f);
    
    return;
}

void display(spComponent component)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glPushMatrix();
    
    /* rotation */
    glRotatef(g_x_rotation, 1.0f, 0.0f, 0.0f);
    glRotatef(g_y_rotation, 0.0f, 1.0f, 0.0f);
    
    glBindTexture(GL_TEXTURE_2D, tex_name);

    glBegin(GL_TRIANGLE_STRIP);
    spDebug(80, "display", "upside_down = %d\n", dibitmap->info.upside_down);
    if (dibitmap->info.upside_down) {
        glTexCoord2f(0.0f, 0.0f); glVertex3f(-0.75f, -0.75f, 0.0f);
        glTexCoord2f(0.0f, 1.0f); glVertex3f(-0.75f, 0.75f, 0.0f);
        glTexCoord2f(1.0f, 0.0f); glVertex3f(0.75f, -0.75f, 0.0f);
        glTexCoord2f(1.0f, 1.0f); glVertex3f(0.75f, 0.75f, 0.0f);
    } else {
        glTexCoord2f(0.0f, 0.0f); glVertex3f(-0.75f, 0.75f, 0.0f);
        glTexCoord2f(0.0f, 1.0f); glVertex3f(-0.75f, -0.75f, 0.0f);
        glTexCoord2f(1.0f, 0.0f); glVertex3f(0.75f, 0.75f, 0.0f);
        glTexCoord2f(1.0f, 1.0f); glVertex3f(0.75f, -0.75f, 0.0f);
    }
    glEnd();    
    
    glPopMatrix();
    
    spGLSwapBuffers(component);
    
    return;
}

void quit(void)
{
    glBindTexture(GL_TEXTURE_2D, 0);
    glDeleteTextures(1, &tex_name);
    
    return;
}

void canvasCB(spComponent component, void *data)
{
    static int w = 0, h = 0;

    spCallbackReason reason;
  
    reason = spGetCallbackReason(component);

    spDebug(10, "canvasCB", "reason = %d\n", reason);
  
    if (reason == SP_CR_RESIZE || w == 0) {
        if (w == 0) {
            init();
        }
        if (spGetSize(component, &w, &h) == SP_FALSE) return;
  
        resize(w, h);
        /*spGLSwapBuffers(component);*/
    }
    display(component);
    
    return;
}

void mouse(spComponent component, void *data)
{
    int x, y;
    spCallbackReason reason;
    static int moving = 0;
    static int px = 0, py = 0;
    
    if (spGetCallbackMousePosition(component, &x, &y) == SP_FALSE) return;

    reason = spGetCallbackReason(component);
    switch (reason) {
      case SP_CR_LBUTTON_PRESS:
        moving = 1;
        px = x;
        py = y;
        break;
      case SP_CR_LBUTTON_RELEASE:
        moving = 0;
        break;
      case SP_CR_LBUTTON_MOTION:
        if (moving) {
            g_y_rotation += (GLfloat)(x - px);
            g_x_rotation += (GLfloat)(y - py);
            px = x;
            py = y;
            spRedrawCanvas(component);
        }
        break;
        
      case SP_CR_RBUTTON_PRESS:
        break;

      default:
        break;
    }
}

void keyboard(spComponent component, void *data)
{
    spKeySym code;
  
    if (spGetCallbackKeySym(component, &code) == SP_TRUE) {
        if (code == SPK_Escape) {
            spQuit(0);
        }
    }

    return;
}

int spMain(int argc, char *argv[])
{
    spTopLevel toplevel;
    spComponent canvas;
    char *filename = NULL;
    spGLVisual visual;
    spGLAttribute attribs[] = { SP_GL_RGBA,
                                SP_GL_DOUBLEBUFFER,
                                SP_GL_DEPTH_SIZE, 1,
#if defined(USE_MULTISAMPLE)
                                SP_GL_SAMPLE_BUFFERS, 1,
                                SP_GL_SAMPLES, MULTISAMPLE_SAMPLES,
#endif
                                SP_GL_NONE };

    toplevel = spInitialize(&argc, &argv, NULL);

    visual = spCreateGLVisual(toplevel, attribs);

    window = spCreateMainFrame("Texture", NULL);
    canvas = spCreateGLCanvas(window, "canvas", visual, 500, 500,
                              SppTitle, argv[0],
                              SppCallbackFunc, canvasCB,
                              NULL);
    spAddCallback(canvas, SP_BUTTON_PRESS_CALLBACK | SP_BUTTON_RELEASE_CALLBACK | SP_BUTTON_MOTION_CALLBACK,
                  mouse, NULL);
    spAddCallback(canvas, SP_KEY_PRESS_CALLBACK, keyboard, NULL);

    if (argc >= 2) {
        filename = argv[1];
    } else {
        filename = xspGetOpenFileName(window, NULL,
                                      SppPathMustExist, SP_TRUE,
                                      SppFileMustExist, SP_TRUE,
                                      NULL);

        if (filename == NULL) {
            spDisplayError(window, NULL, "File name must be specified.\n");
            spQuit(0);
        }
    }
    spDebug(10, "spMain", "filename = %s\n", filename);
    
    /* load bitmap file */
    if ((dibitmap = spCreateDIBitmapFromFileNonnative(filename)) == NULL) {
        spDisplayError(window, NULL, "Cannot open bitmap file: %s", filename);
        spQuit(0);
    }
    spDebug(10, "spMain", "pixel_format = %lx, size = (%ld, %ld), bit_size = %ld, pixel_stride = %ld, line_stride = %ld\n",
            dibitmap->info.pixel_format, dibitmap->info.width, dibitmap->info.height,
            dibitmap->info.bit_size, dibitmap->info.pixel_stride, dibitmap->info.line_stride);
    
    context = spCreateGLContext(canvas, NULL);

    spDebug(100, "spMain", "call spPopupWindow\n");
    spPopupWindow(window);
  
    spDebug(100, "spMain", "call spSetGLContext\n");
    spSetGLContext(canvas, context);
  
    spDebug(100, "spMain", "call spMainLoop\n");
    return spMainLoop(toplevel);
}
