#include <sp/spComponent.h>
#include <sp/spContainer.h>

#include <sp/spGLP.h>

#if defined(USE_OPENGL3)
#pragma message "USE_OPENGL3"
#endif

static spParamTable sp_glcanvas_param_tables[] = {
    {SppGLVisual, SP_CREATE_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLCanvas, glcanvas.glvisual), NULL},
    {SppGLAutoScaling, SP_CREATE_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLCanvas, glcanvas.auto_scaling), NULL},
    {SppGLScalingFactor, SP_GET_ACCESS,
	 spOffset(spGLCanvas, glcanvas.scaling_factor), NULL},
};

spGLCanvasClassRec SpGLCanvasClassRec = {
    /* spObjectClassPart */
    {
	SpGLCanvas,
	(spObjectClass)&SpCanvasClassRec,
	sizeof(spGLCanvasRec),
	spArraySize(sp_glcanvas_param_tables),
	sp_glcanvas_param_tables,
	spGLCanvasPartInit,
	spGLCanvasPartFree,
	SP_FALSE,
	NULL,
	NULL,
	spGLCanvasCreate,
	spGLCanvasDestroy,
	spGLCanvasSetParams,
	NULL,
    },
    /* spComponentClassPart */
    {
	SP_TRUE,
	SP_FALSE,
	SP_TRUE,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,

	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
    },
    /* spPrimitiveClassPart */
    {
	0,
    },
    /* spCanvasClassPart */
    {
	spGLCanvasRefresh,
	spGLCanvasRedraw,
    },
    /* spGLCanvasClassPart */
    {
	0,
    },
};

spComponentClass SpGLCanvasClass = (spComponentClass)&SpGLCanvasClassRec;

#if defined(COCOA) && !defined(IPHONE)
uint32_t sp_gl_cocoa_profile_version = 0;
#endif

void spGLCanvasPartInit(spObject object)
{
    spComponent component = (spComponent)object;
    
    SpGLCanvasPart(component).glvisual = NULL;
    SpGLCanvasPart(component).current_context = NULL;
    
    SpGLCanvasPart(component).auto_scaling = SP_TRUE;
    SpGLCanvasPart(component).scaling_factor = 1.0;
    
    return;
}

void spGLCanvasPartFree(spObject object)
{
    return;
}

spBool spGLCanvasCreate(spObject object)
{
    spComponent component = (spComponent)object;
    
    if (component == NULL) return SP_FALSE;

    spCanvasSetDefaultSize(component);
    spPrimitiveScaleDefaultSizeArch(component);
    
    if (spGLCanvasCreateArch(component) == SP_TRUE) {
	if (SpComponentPart(component).call_func != NULL) {
	    SpPrimitivePart(component).draw_func = SpComponentPart(component).call_func;
	    SpPrimitivePart(component).draw_data = SpComponentPart(component).call_data;
	}

	spAddCallback(component, SP_EXPOSE_CALLBACK, spExposeCanvasCB, NULL);
	
	spDebug(50, "spGLCanvasCreate", "done\n");
	
	return SP_TRUE;
    } else {
	spDebug(10, "spGLCanvasCreate", "spGLCanvasCreate failed\n");
	return SP_FALSE;
    }
}

spBool spGLCanvasSetParams(spObject object)
{
    spComponent component = (spComponent)object;
    
    if (component == NULL) return SP_FALSE;
	 
    return spGLCanvasSetParamsArch(component);
}

spBool spGLCanvasDestroy(spObject object)
{
    spComponent component = (spComponent)object;
    
    if (component == NULL) return SP_FALSE;

    return spGLCanvasDestroyArch(component);
}

spBool spIsGLCanvas(spComponent component)
{
    return spIsSubClass(component, SpGLCanvas);
}

spBool spIsGLDrawableComponent(spComponent component)
{
    if (spIsSubClass(component, SpGLCanvas) == SP_TRUE
	|| spIsSubClass(component, SpGLPixmap) == SP_TRUE) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spIsGLDrawable(spComponent component)
{
    if (spIsSubClass(component, SpGLCanvas) == SP_TRUE
	&& SpGLCanvasPart(component).current_context != NULL) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spComponent spCreateGLCanvas(spComponent parent, char *name, spGLVisual visual,
			     int canvas_width, int canvas_height, ...)
{
    int num_arg = 0;
    spArg args[SP_MAX_NUM_ARG];
    va_list argp;
    
    if (spIsCreated(parent) == SP_FALSE || spIsContainer(parent) == SP_FALSE
	|| visual == NULL) {
	spDebug(50, "spCreateGLCanvas", "input parameters error\n");
	return NULL;
    }
    
    va_start(argp, canvas_height);
    spGetArgs(argp, args, num_arg);
    va_end(argp);
    
    spSetArg(args[num_arg], SppInitialWidth, canvas_width); num_arg++;
    spSetArg(args[num_arg], SppInitialHeight, canvas_height); num_arg++;
    spSetArg(args[num_arg], SppGLVisual, visual); num_arg++;

    return spCreateComponentArg(SpGLCanvasClass, NULL, name, parent, args, num_arg);
}

spBool spGLCanvasRefresh(spComponent component)
{
    return spGLCanvasRedraw(component);
}

spBool spGLCanvasRedraw(spComponent component)
{
    spDebug(50, "spGLCanvasRedraw", "in\n");
    
    if (spIsGLCanvas(component) == SP_FALSE) return SP_FALSE;

    if (spIsCanvasExposed(component) == SP_FALSE) {
	spDebug(50, "spGLCanvasRedraw", "done (not exposed yet)\n");
	return SP_FALSE;
    }

    if (spGLCanvasRedrawArch(component) == SP_TRUE) {
	if (SpPrimitivePart(component).draw_func != NULL) {
	    spDebug(50, "spGLCanvasRedraw", "call draw_func\n");
	    SpPrimitivePart(component).draw_func(component, SpPrimitivePart(component).draw_data);
	}
        spGLCanvasPostRedrawArch(component);
	spDebug(50, "spGLCanvasRedraw", "successfully done\n");
	return SP_TRUE;
    }

    spDebug(50, "spGLCanvasRedraw", "done (without redrawing)\n");
    
    return SP_FALSE;
}

double spGetGLCanvasScalingFactor(spComponent component)
{
    if (spIsSubClass(component, SpGLCanvas) == SP_FALSE) {
	return 0.0;
    }

#if defined(COCOA)
    if (spGetGLCanvasScalingFactorArch(component) == SP_FALSE) {
	return 0.0;
    }
#endif
    
    return SpGLCanvasPart(component).scaling_factor;
}

spGLVisual spCreateGLVisual(spTopLevel toplevel, spGLAttribute *attributes)
{
    spGLVisual visual;
    
    if (toplevel == NULL) return NULL;

    visual = xalloc(1, struct _spGLVisual);

    if (spCreateGLVisualArch(visual, toplevel, attributes) == SP_FALSE) {
	xfree(visual);
	return NULL;
    }

    return visual;
}

spBool spDestroyGLVisual(spGLVisual visual)
{
    spBool flag;

    if (visual == NULL) return SP_FALSE;
    
    flag = spDestroyGLVisualArch(visual);

    xfree(visual);

    return flag;
}

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
spGLAttribute sp_wgl_attrib_mapping[][2] = {
    {SP_GL_USE_GL, 0x2010/*WGL_SUPPORT_OPENGL_ARB*/},
    {SP_GL_DOUBLEBUFFER, 0x2011/*WGL_DOUBLE_BUFFER_ARB*/},
    {SP_GL_STEREO, 0x2012/*WGL_STEREO_ARB*/},
    {SP_GL_AUX_BUFFERS, 0x2024/*WGL_AUX_BUFFERS_ARB*/},
    {SP_GL_RED_SIZE, 0x2015/*WGL_RED_BITS_ARB*/},
    {SP_GL_GREEN_SIZE,0x2017/*WGL_GREEN_BITS_ARB*/},
    {SP_GL_BLUE_SIZE, 0x2019/*WGL_BLUE_BITS_ARB*/},
    {SP_GL_ALPHA_SIZE, 0x201B/*WGL_ALPHA_BITS_ARB*/},
    {SP_GL_DEPTH_SIZE, 0x2022/*WGL_DEPTH_BITS_ARB*/},
    {SP_GL_STENCIL_SIZE, 0x2023/*WGL_STENCIL_BITS_ARB*/},
    {SP_GL_ACCUM_RED_SIZE, 0x201E/*WGL_ACCUM_RED_BITS_ARB*/},
    {SP_GL_ACCUM_GREEN_SIZE, 0x201F/*WGL_ACCUM_GREEN_BITS_ARB*/},
    {SP_GL_ACCUM_BLUE_SIZE, 0x2020/*WGL_ACCUM_BLUE_BITS_ARB*/},
    {SP_GL_ACCUM_ALPHA_SIZE, 0x2021/*WGL_ACCUM_ALPHA_BITS_ARB*/},
    {0, 0},
};

spGLAttribute spMapToWGLAttribute(spGLAttribute iattr)
{
    int j;
    spGLAttribute oattr = 0;

    for (j = 0; sp_wgl_attrib_mapping[j][0] != 0; j++) {
	if (sp_wgl_attrib_mapping[j][0] == iattr) {
	    return sp_wgl_attrib_mapping[j][1];
	}
    }
    
    return 0;
}
#endif

spGLAttributeSupportInfo sp_gl_attribute_support_infos[] = {
    /* {attribute, min_version_id, max_version_id, legacy_num_param, legacy_only, type_mask}, */
    {SP_GL_USE_GL, 0, 1001, 0, SP_TRUE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#if !defined(COCOA)
    {SP_GL_BUFFER_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_LEVEL, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#if !defined(_WIN32) || defined(USE_MOTIF) || defined(USE_GTK)
    {SP_GL_RGBA, 0, 1002, 0, SP_TRUE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#endif
#else /* defined(COCOA) */
    {SP_GL_RGBA, 0, 1002, 0, SP_TRUE, 0},
#endif /* defined(COCOA) */
    {SP_GL_DOUBLEBUFFER, 0, 0, 0, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_STEREO, 0, 0, 0, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_AUX_BUFFERS, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_RED_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#if !defined(COCOA)
    {SP_GL_GREEN_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_BLUE_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#endif
    {SP_GL_ALPHA_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_DEPTH_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_STENCIL_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_ACCUM_RED_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#if !defined(COCOA)	
    {SP_GL_ACCUM_GREEN_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_ACCUM_BLUE_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_ACCUM_ALPHA_SIZE, 0, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_PIXEL_TYPE, 1003, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_CONFIG_CAVEAT, 1003, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#endif
	
#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    {SP_GL_ACCELERATION, 1002, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
#endif
	
    {SP_GL_SAMPLE_BUFFERS, 1003, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
    {SP_GL_SAMPLES, 1003, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CORE_MASK},
	
#if !defined(COCOA) || defined(IPHONE)
    {SP_GL_CONTEXT_MAJOR_VERSION, 1004, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CONTEXT_MASK},
    {SP_GL_CONTEXT_MINOR_VERSION, 1004, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CONTEXT_MASK},
    {SP_GL_CONTEXT_FLAGS, 1004, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CONTEXT_MASK},
    {SP_GL_CONTEXT_PROFILE_MASK, 1004, 0, 1, SP_FALSE, SP_GL_ATTRIBUTE_TYPE_CONTEXT_MASK},
#endif
	
    {0, 0, 0, 0, 0},
};

int spNumGLAttributeParams(spGLAttribute attribute)
{
    int j;
    
    for (j = 0; sp_gl_attribute_support_infos[j].attribute != 0; j++) {
	if (attribute == sp_gl_attribute_support_infos[j].attribute) {
	    return sp_gl_attribute_support_infos[j].legacy_num_param;
	}
    }

    return 1;
}

static int extractGLAttributes(spGLAttribute *attributes, int version_id, spBool legacy_flag, unsigned long type_mask,
			       spGLAttribute *o_attributes)
{
    int i, j;
    int incr;
    int count = 0;
    spGLAttribute oattr;
    spBool found_flag;
#if defined(COCOA) && !defined(IPHONE)
    spBool multisample_setting_added = SP_FALSE;
#endif
	
    spDebug(80, "extractGLAttributes", "version_id = %d, legacy_flag = %d, type_mask = %lx\n", version_id, legacy_flag, type_mask);
    
    for (i = 0;; i++) {
	found_flag = SP_FALSE;
	
	spDebug(100, "extractGLAttributes", "attributes[%d] = %d (0x%x)\n", i, attributes[i], attributes[i]);
	
	if (attributes[i] == 0) {
	    spDebug(80, "extractGLAttributes", "end: count = %d\n", count);
	    if (count > 0) {
		if (o_attributes != NULL) o_attributes[count] = attributes[i];
		++count;
	    }
	    break;
#if defined(XmVersion) || defined(_WIN32)
	} else if (attributes[i] == SP_GL_RGBA && legacy_flag == SP_FALSE) {
	    if (type_mask & SP_GL_ATTRIBUTE_TYPE_CORE_MASK) {
	        if (o_attributes != NULL) o_attributes[count] = SP_GL_PIXEL_TYPE;
                ++count;
	        if (o_attributes != NULL) o_attributes[count] = SP_GL_PIXEL_TYPE_RGBA;
	        ++count;
            }
	    found_flag = SP_TRUE;
#endif
#if defined(COCOA) && !defined(IPHONE)
	} else if (attributes[i] == SP_GL_CONTEXT_MAJOR_VERSION) {
	    if (type_mask & SP_GL_ATTRIBUTE_TYPE_CORE_MASK) {
	        if (o_attributes != NULL) o_attributes[count] = 99/*kCGLPFAOpenGLProfile*/;
                ++count;
		++i;
	        if (o_attributes != NULL) {
		    if (attributes[i] >= 4) {
			o_attributes[count] = 0x4100;
		    } else if (attributes[i] >= 3) {
			o_attributes[count] = 0x3200;
		    } else {
			o_attributes[count] = 0x1000;
		    }
                    sp_gl_cocoa_profile_version = o_attributes[count];
		}
	        ++count;
	    }
	    found_flag = SP_TRUE;
#endif
	} else {
	    for (j = 0; sp_gl_attribute_support_infos[j].attribute != 0; j++) {
		if (attributes[i] == sp_gl_attribute_support_infos[j].attribute) {
		    found_flag = SP_TRUE;
		    spDebug(100, "extractGLAttributes", "found: attributes[%d] = %d (0x%x)\n", i, attributes[i], attributes[i]);
		    
		    if ((sp_gl_attribute_support_infos[j].legacy_only == SP_FALSE || legacy_flag == SP_TRUE)
			&& (version_id == 0
			    || (version_id >= sp_gl_attribute_support_infos[j].min_version_id
				&& (sp_gl_attribute_support_infos[j].max_version_id <= 0
				    || version_id <= sp_gl_attribute_support_infos[j].max_version_id)))
			&& (type_mask == 0 || (type_mask & sp_gl_attribute_support_infos[j].type_mask))) {
			spDebug(100, "extractGLAttributes", "valid: attributes[%d] = %d (0x%x)\n", i, attributes[i], attributes[i]);

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
			if ((oattr = spMapToWGLAttribute(attributes[i])) <= 0) {
			    spDebug(100, "extractGLAttributes", "spMapToWGLAttribute failed: attributes[%d] = %d (0x%x)\n",
				    i, attributes[i], attributes[i]);
			    oattr = attributes[i];
			}
#else
			oattr = attributes[i];
#endif
			if (o_attributes != NULL) {
			    o_attributes[count] = oattr;
			}
			++count;
			
			i += sp_gl_attribute_support_infos[j].legacy_num_param;
			
			if (legacy_flag) {
			    incr = sp_gl_attribute_support_infos[j].legacy_num_param;
			} else {
			    incr = 1;
			}

			if (incr > 0) {
	                    if (o_attributes != NULL) {
				if (sp_gl_attribute_support_infos[j].legacy_num_param > 0) {
				    o_attributes[count] = attributes[i];
				} else {
				    o_attributes[count] = 1;
				}
			    }
			    
			    count += incr;
			    
#if defined(COCOA) && !defined(IPHONE)
			    if (multisample_setting_added == SP_FALSE
				&& ((oattr == SP_GL_SAMPLES && attributes[i] >= 2)
				    || (oattr == SP_GL_SAMPLE_BUFFERS && attributes[i] >= 1))) {
				if (o_attributes != NULL) {
				    o_attributes[count] = NSOpenGLPFAMultisample;
				}
				++count;
			    }
#endif
			}
		    } else {
			spDebug(100, "extractGLAttributes", "skip: attributes[%d] = %d (0x%x)\n", i, attributes[i], attributes[i]);
			i += sp_gl_attribute_support_infos[j].legacy_num_param;
		    }
		    break;
		}
	    }
	}

	if (found_flag == SP_FALSE) {
	    ++i;
	}
    }

    
#if 0
    if (o_attributes != NULL) {
	spDebug(80, "extractGLAttributes", "count = %d\n", count);
	
	for (i = 0; i < count; i++) {
	    spDebug(100, "extractGLAttributes", "o_attributes[%d] = %d (0x%x)\n", i, o_attributes[i], o_attributes[i]);
	}
    }
#endif
	
    return count;
}

spGLAttribute *xspExtractGLAttributes(spGLAttribute *attributes, int version_id, spBool legacy_flag, unsigned long type_mask, int *o_count)
{
    int count;
    spGLAttribute *o_attributes = NULL;
    
    if (attributes == NULL) return NULL;
    
    count = extractGLAttributes(attributes, version_id, legacy_flag, type_mask, NULL);
    if (count > 0) {
	o_attributes = xalloc(count, spGLAttribute);
	extractGLAttributes(attributes, version_id, legacy_flag, type_mask, o_attributes);
    }

    if (o_count != NULL) *o_count = count;

    return o_attributes;
}

spBool spGetVersionFromGLAttributes(spGLAttribute *attributes, int *major, int *minor)
{
    int i, j;
    spBool found_flag;
    spBool major_found = SP_FALSE;

    if (major != NULL) *major = 0;
    if (minor != NULL) *minor = 0;
	
    if (attributes == NULL) return SP_FALSE;

    for (i = 0;; i++) {
	found_flag = SP_FALSE;
	
	if (attributes[i] == 0) {
	    break;
	} else if (attributes[i] == SP_GL_CONTEXT_MAJOR_VERSION) {
	    ++i;
	    if (major != NULL) *major = attributes[i];
	    major_found = SP_TRUE;
	} else if (attributes[i] == SP_GL_CONTEXT_MINOR_VERSION) {
	    ++i;
	    if (minor != NULL) *minor = attributes[i];
	} else {
	    for (j = 0; sp_gl_attribute_support_infos[j].attribute != 0; j++) {
		if (attributes[i] == sp_gl_attribute_support_infos[j].attribute) {
		    found_flag = SP_TRUE;
		    i += sp_gl_attribute_support_infos[j].legacy_num_param;
		    break;
		}
	    }
	    
	    if (found_flag == SP_FALSE) {
		++i;
	    }
	}
    }

    return major_found;
}

spGLContext spCreateGLContext(spComponent gldrawable, spGLContext share)
{
    spGLContext context;

    if (spIsGLDrawableComponent(gldrawable) == SP_FALSE
	|| SpGLCanvasPart(gldrawable).glvisual == NULL) {
	spDebug(50, "spCreateGLContext", "not drawable component\n");
	return NULL;
    }
    
    context = xalloc(1, struct _spGLContext);

    if (spCreateGLContextArch(context, gldrawable, SpGLCanvasPart(gldrawable).glvisual, share) == SP_FALSE) {
	xfree(context);
	return NULL;
    }

    return context;
}

spBool spDestroyGLContext(spGLContext context)
{
    spBool flag;

    if (context == NULL) return SP_FALSE;

    flag = spDestroyGLContextArch(context);

    xfree(context);

    return flag;
}

spBool spSetGLContext(spComponent gldrawable, spGLContext context)
{
    if (spIsGLDrawableComponent(gldrawable) == SP_FALSE) return SP_FALSE;
    
    SpGLCanvasPart(gldrawable).current_context = context;
    
    return spSetGLContextArch(gldrawable, context);
}

spGLContext spGetGLContext(spComponent gldrawable)
{
    if (spIsGLDrawableComponent(gldrawable) == SP_FALSE) return NULL;

    return SpGLCanvasPart(gldrawable).current_context;
}

spBool spGLSwapBuffers(spComponent gldrawable)
{
    if (spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;

    return spGLSwapBuffersArch(gldrawable);
}

spBool spGLFlush(spComponent gldrawable)
{
    if (spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;
    
    if (spGLSwapBuffersArch(gldrawable) == SP_FALSE) {
	glFlush();
    }

    return SP_TRUE;
}

spBool spGLWaitGLDrawing(spComponent gldrawable)
{
    if (spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;

    return spGLWaitGLDrawingArch(gldrawable);
}

spBool spGLWaitSystemDrawing(spComponent gldrawable)
{
    if (spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;

    return spGLWaitSystemDrawingArch(gldrawable);
}

GLint spGetGLDIBitmapUnpackAlignment(spDIBitmap dibitmap)
{
    if (dibitmap == NULL) return 0;

    if (dibitmap->info.pixel_stride == 3) {
	return 1;
    } else if (dibitmap->info.pixel_stride == 2 || dibitmap->info.pixel_stride == 4 || dibitmap->info.pixel_stride == 8) {
	return dibitmap->info.pixel_stride;
    } else {
	return 0;
    }
}

GLint spGetGLDIBitmapUnpackRowLength(spDIBitmap dibitmap)
{
    long row_length = 0;
    long ideal_line_stride;
    spBool locked;
        
    if (dibitmap == NULL) return 0;

    ideal_line_stride = dibitmap->info.width * dibitmap->info.pixel_stride;

    locked = dibitmap->info.locked;
    if (locked == SP_FALSE) {
	spLockDIBitmap(dibitmap);
    }

    spDebug(80, "spGetGLDIBitmapUnpackRowLength", "dibitmap->info.line_stride = %ld, ideal_line_stride = %ld\n",
            dibitmap->info.line_stride, ideal_line_stride);
    
    if (dibitmap->info.line_stride != ideal_line_stride) {
        row_length = dibitmap->info.line_stride / dibitmap->info.pixel_stride;
    }
    
    if (locked == SP_FALSE) {
	spUnlockDIBitmap(dibitmap);
    }

    return row_length;
}

#if defined(_WIN32)
#if !defined(GL_UNSIGNED_BYTE_3_3_2)
#define GL_UNSIGNED_BYTE_3_3_2            0x8032
#define GL_UNSIGNED_SHORT_5_6_5           0x8363
#define GL_UNSIGNED_SHORT_4_4_4_4         0x8033
#define GL_UNSIGNED_SHORT_5_5_5_1         0x8034
#define GL_UNSIGNED_INT_8_8_8_8           0x8035
#define GL_UNSIGNED_INT_10_10_10_2        0x8036
#endif

#if !defined(GL_UNSIGNED_BYTE_2_3_3_REV)
#define GL_UNSIGNED_BYTE_2_3_3_REV        0x8362
#define GL_UNSIGNED_SHORT_5_6_5_REV       0x8364
#define GL_UNSIGNED_SHORT_4_4_4_4_REV     0x8365
#define GL_UNSIGNED_SHORT_1_5_5_5_REV     0x8366
#define GL_UNSIGNED_INT_8_8_8_8_REV       0x8367
#define GL_UNSIGNED_INT_2_10_10_10_REV    0x8368
#endif
#endif

spBool spGetGLDIBitmapFormat(spDIBitmap dibitmap, GLint *pinternal_format, GLenum *pformat, GLenum *ptype)
{
    GLenum internal_format = GL_RGB;
    GLenum format = 0;
    GLenum type = 0;
    spDIPixelFormat pixel_format = 0L;
    
    if (dibitmap == NULL) return SP_FALSE;

    pixel_format = (SP_DI_PIXEL_FORMAT_MASK & ~SP_DI_PIXEL_FORMAT_ALPHA_PREMULTIPLIED_MASK) & dibitmap->info.pixel_format;
    spDebug(100, "spGetGLDIBitmapFormat", "pixel_format = %lx, bit_size = %ld, pixel_stride = %ld\n",
	    pixel_format, dibitmap->info.bit_size, dibitmap->info.pixel_stride);

    if (pixel_format & SP_DI_PIXEL_FORMAT_RGB_MASK) {
	spDebug(100, "spGetGLDIBitmapFormat", "RGB\n");
	if (dibitmap->info.pixel_stride == 1) {
	    if (dibitmap->info.bit_size == 32) {
		if (pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK) {
		    internal_format = GL_RGBA;
		    
		    if ((pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK)
			&& (pixel_format & SP_DI_PIXEL_FORMAT_FIRST_ALPHA_MASK)) {
			if (pixel_format & SP_DI_PIXEL_FORMAT_SINGLE_RGB_ALPHA_RATIO51) {
#if defined(GL_UNSIGNED_INT_2_10_10_10_REV)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_INT_2_10_10_10_REV;
#endif
			} else {
#if defined(GL_UNSIGNED_INT_8_8_8_8_REV)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_INT_8_8_8_8_REV;
#endif
			}
		    } else if (!(pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK)
			       && (pixel_format & SP_DI_PIXEL_FORMAT_LAST_ALPHA_MASK)) {
			if (pixel_format & SP_DI_PIXEL_FORMAT_SINGLE_RGB_ALPHA_RATIO51) {
#if defined(GL_UNSIGNED_INT_10_10_10_2)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_INT_10_10_10_2;
#endif
			} else {
#if defined(GL_UNSIGNED_INT_8_8_8_8)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_INT_8_8_8_8;
#endif
			}
		    } else if (!(pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK)
                               && (pixel_format & SP_DI_PIXEL_FORMAT_FIRST_ALPHA_MASK)) {
			if (!(pixel_format & SP_DI_PIXEL_FORMAT_SINGLE_RGB_ALPHA_RATIO51)) {
#if defined(GL_UNSIGNED_INT_8_8_8_8_REV) && defined(GL_BGRA)
                            format = GL_BGRA;
                            type = GL_UNSIGNED_INT_8_8_8_8_REV;
#endif
                        }
		    }
		}
	    } else if (dibitmap->info.bit_size == 16) {
		if (pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK) {
		    if ((pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK)
			&& (pixel_format & SP_DI_PIXEL_FORMAT_FIRST_ALPHA_MASK)) {
			internal_format = GL_RGBA;
			
			if (pixel_format & SP_DI_PIXEL_FORMAT_SINGLE_RGB_ALPHA_RATIO51) {
#if defined(GL_UNSIGNED_SHORT_1_5_5_5_REV)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_SHORT_1_5_5_5_REV;
#endif
			} else {
#if defined(GL_UNSIGNED_SHORT_4_4_4_4_REV)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_SHORT_4_4_4_4_REV;
#endif
			}
		    } else if (!(pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK)
			       && (pixel_format & SP_DI_PIXEL_FORMAT_LAST_ALPHA_MASK)) {
			if (pixel_format & SP_DI_PIXEL_FORMAT_SINGLE_RGB_ALPHA_RATIO51) {
#if defined(GL_UNSIGNED_SHORT_5_5_5_1)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_SHORT_5_5_5_1;
#endif
			} else {
#if defined(GL_UNSIGNED_SHORT_4_4_4_4)
			    format = GL_RGBA;
			    type = GL_UNSIGNED_SHORT_4_4_4_4;
#endif
			}
		    }
		} else {
		    if (pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK) {
#if defined(GL_UNSIGNED_SHORT_5_6_5_REV)
			format = GL_RGB;
			type = GL_UNSIGNED_SHORT_5_6_5_REV;
#elif defined(GL_BGR) && defined(GL_UNSIGNED_SHORT_5_6_5)
			format = GL_BGR;
			type = GL_UNSIGNED_SHORT_5_6_5;
#elif defined(GL_BGR_EXT) && defined(GL_UNSIGNED_SHORT_5_6_5)
			format = GL_BGR_EXT;
			type = GL_UNSIGNED_SHORT_5_6_5;
#endif
		    } else {
#if defined(GL_UNSIGNED_SHORT_5_6_5)
			format = GL_RGB;
			type = GL_UNSIGNED_SHORT_5_6_5;
#endif
		    }
		}
	    } else if (dibitmap->info.bit_size == 8 && !(pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK)) {
		if (pixel_format & SP_DI_PIXEL_FORMAT_INVERSE_MASK) {
#if defined(GL_UNSIGNED_BYTE_2_3_3_REV)
		    format = GL_RGB;
		    type = GL_UNSIGNED_BYTE_2_3_3_REV;
#elif defined(GL_BGR) && defined(GL_UNSIGNED_BYTE_3_3_2)
		    format = GL_BGR;
		    type = GL_UNSIGNED_BYTE_3_3_2;
#elif defined(GL_BGR_EXT) && defined(GL_UNSIGNED_BYTE_3_3_2)
		    format = GL_BGR_EXT;
		    type = GL_UNSIGNED_BYTE_3_3_2;
#endif
		} else {
#if defined(GL_UNSIGNED_BYTE_3_3_2)
		    type = GL_UNSIGNED_BYTE_3_3_2;
#endif
		}
	    }
	} else if (dibitmap->info.pixel_stride == 4 && (pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK)) {
	    spDebug(100, "spGetGLDIBitmapFormat", "4byte RGBA\n");
	    internal_format = GL_RGBA;
	    
	    if (pixel_format == SP_DI_PIXEL_FORMAT_BGRA) {
                spDebug(100, "spGetGLDIBitmapFormat", "BGRA\n");
#if defined(GL_BGRA)
		format = GL_BGRA;
#elif defined(GL_BGRA_EXT)
		format = GL_BGRA_EXT;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_ABGR) {
                spDebug(100, "spGetGLDIBitmapFormat", "ABGR\n");
#if defined(GL_ABGR)
		format = GL_ABGR;
#elif defined(GL_ABGR_EXT)
		format = GL_ABGR_EXT;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_ARGB) {
                spDebug(100, "spGetGLDIBitmapFormat", "ARGB\n");
#if defined(GL_ARGB)
		format = GL_ARGB;
#elif defined(GL_ARGB_EXT)
		format = GL_ARGB_EXT;
#elif defined(GL_UNSIGNED_INT_8_8_8_8_REV) && defined(GL_BGRA)
                format = GL_BGRA;
                type = GL_UNSIGNED_INT_8_8_8_8_REV;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_RGBA) {
                spDebug(100, "spGetGLDIBitmapFormat", "RGBA\n");
		format = GL_RGBA;
	    } else {
                spDebug(10, "spGetGLDIBitmapFormat", "unknown RGBA: pixel_format = %lx\n", pixel_format);
            }
	} else if (dibitmap->info.pixel_stride == 4 && !(pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK)) {
	    spDebug(100, "spGetGLDIBitmapFormat", "4byte RGB\n");
	    if (pixel_format == SP_DI_PIXEL_FORMAT_BGRN) {
#if defined(GL_BGRA)
		format = GL_BGRA;
#elif defined(GL_BGRA_EXT)
		format = GL_BGRA_EXT;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_NBGR) {
#if defined(GL_ABGR)
		format = GL_ABGR;
#elif defined(GL_ABGR_EXT)
		format = GL_ABGR_EXT;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_NRGB) {
#if defined(GL_ARGB)
		format = GL_ARGB;
#elif defined(GL_ARGB_EXT)
		format = GL_ARGB_EXT;
#elif defined(GL_UNSIGNED_INT_8_8_8_8_REV) && defined(GL_BGRA)
                format = GL_BGRA;
                type = GL_UNSIGNED_INT_8_8_8_8_REV;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_RGBN) {
		format = GL_RGBA;
	    }
	} else if (dibitmap->info.pixel_stride == 3 && !(pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK)) {
	    spDebug(100, "spGetGLDIBitmapFormat", "3byte\n");
	    if (dibitmap->info.line_stride % 3 != 0) {
                spDebug(100, "spGetGLDIBitmapFormat", "line_stride (%ld) must be a multiple of 3\n",
                        dibitmap->info.line_stride);
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_BGR) {
#if defined(GL_BGR)
		format = GL_BGR;
#elif defined(GL_BGR_EXT)
		format = GL_BGR_EXT;
#endif
	    } else if (pixel_format == SP_DI_PIXEL_FORMAT_RGB) {
		format = GL_RGB;
	    } 
	}
    } else if (pixel_format & SP_DI_PIXEL_FORMAT_GRAYSCALE_MASK) {
        if (dibitmap->info.pixel_stride == 2 && pixel_format == SP_DI_PIXEL_FORMAT_GRAY_ALPHA) {
            format = GL_LUMINANCE_ALPHA;
            internal_format = GL_LUMINANCE_ALPHA;
        } else if (dibitmap->info.pixel_stride == 1) {
            format = GL_LUMINANCE;
            internal_format = GL_LUMINANCE;
        }
    } else if (pixel_format == SP_DI_PIXEL_FORMAT_ALPHA_ONLY
	       && dibitmap->info.pixel_stride == 1) {
	format = GL_ALPHA;
	internal_format = GL_ALPHA;
    }

    spDebug(80, "spGetGLDIBitmapFormat", "internal_format = %x, format = %x, type = %x\n",
	    internal_format, format, type);
    
    if (format != 0) {
	if (pinternal_format != NULL) {
	    *pinternal_format = internal_format;
	}
	if (pformat != NULL) {
	    *pformat = format;
	}
	
	if (ptype != NULL) {
	    if (format != 0 && type == 0) {
		if (dibitmap->info.bit_size == 32) {
		    if (pixel_format & SP_DI_PIXEL_FORMAT_FLOAT_MASK) {
			type = GL_FLOAT;
		    } else {
#if defined(GL_UNSIGNED_INT)
			type = GL_UNSIGNED_INT;
#else
			type = 0; format = 0;
#endif
		    }
		} else if (dibitmap->info.bit_size == 16) {
		    if (pixel_format & SP_DI_PIXEL_FORMAT_FLOAT_MASK) {
#if defined(GL_HALF_FLOAT)
			type = GL_HALF_FLOAT;
#else
			type = 0; format = 0;
#endif
		    } else {
			type = GL_UNSIGNED_SHORT;
		    }
		} else if (dibitmap->info.bit_size == 8 && !(pixel_format & SP_DI_PIXEL_FORMAT_FLOAT_MASK)) {
		    type = GL_UNSIGNED_BYTE;
		} else {
		    type = 0; format = 0;
		}
	    }
	
	    *ptype = type;
	}
	
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spGetGLDIPixelFormatAlphaBlendFactor(spDIPixelFormat pixel_format, GLenum *psfactor, GLenum *pdfactor)
{
    if (!(pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_MASK)) return SP_FALSE;

    if (pixel_format & SP_DI_PIXEL_FORMAT_ALPHA_PREMULTIPLIED_MASK) {
        if (psfactor != NULL) *psfactor = GL_ONE;
        if (pdfactor != NULL) *pdfactor = GL_ONE_MINUS_SRC_ALPHA;
    } else {
        if (psfactor != NULL) *psfactor = GL_SRC_ALPHA;
        if (pdfactor != NULL) *pdfactor = GL_ONE_MINUS_SRC_ALPHA;
    }

    return SP_TRUE;
}

spBool spGetGLDIBitmapAlphaBlendFactor(spDIBitmap dibitmap, GLenum *psfactor, GLenum *pdfactor)
{
    if (dibitmap == NULL) return SP_FALSE;

    return spGetGLDIPixelFormatAlphaBlendFactor(dibitmap->info.pixel_format, psfactor, pdfactor);
}

/*
 * Reference: http://www.opengl.org/wiki/Tutorial:_OpenGL_3.0_Context_Creation_%28GLX%29
 */
/* Helper to check for extension string presence.  Adapted from:
 *   http://www.opengl.org/resources/features/OGLextensions/
 */
spBool spIsGLExtensionIncludedInList(char *extList, char *extension)
{
    char *start;
    char *where, *terminator;

    if (extList == NULL || extension == NULL) return SP_FALSE;

    /* Extension names should not have spaces. */
    where = strchr(extension, ' ');
    if (where || *extension == '\0')
	return SP_FALSE;
 
    /* It takes a bit of care to be fool-proof about parsing the
       OpenGL extensions string. Don't be fooled by sub-strings,
       etc. */
    for (start = extList;;) {
	where = strstr(start, extension);
 
	if (!where)
	    break;
 
	terminator = where + strlen(extension);
 
	if (where == start || *(where - 1) == ' ')
	    if (*terminator == ' ' || *terminator == '\0') {
		spDebug(50, "spIsGLExtensionIncludedInList", "supported: %s\n", extension);
		return SP_TRUE;
	    }
 
	start = terminator;
    }
 
    spDebug(50, "spIsGLExtensionIncludedInList", "not supported: %s\n", extension);
    
    return SP_FALSE;
}

spBool spIsGLExtensionSupported(GLenum name, char *extension)
{
    char *extList;
    
    if ((extList = (char *)glGetString(name)) == NULL) return SP_FALSE;

    return spIsGLExtensionIncludedInList(extList, extension);
}

char *spGetGLSystemExtensionList(spGLContext context)
{
    if (context == NULL) return NULL;

#if defined(_WIN32) || defined(XmVersion) || defined(GTK)
    return SpGLContextArch(context).ext_list;
#else
    return spGetGLSystemExtensionListArch(context);
#endif
}

void *spGetGLProcAddress(char *proc)
{
#if defined(GTK) && !defined(SP_GL_USE_GTK_GL_AREA)
    return gdk_gl_get_proc_address((const char *)proc);
#elif defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    return wglGetProcAddress(proc);
#elif defined(XmVersion) || defined(SP_GL_USE_GTK_GL_AREA)
#if defined(GLX_VERSION_1_4) || defined(SP_GL_USE_GTK_GL_AREA)
    return glXGetProcAddress((GLubyte *)proc);
#else
    return glXGetProcAddressARB(proc);
#endif
#else
    return spGetGLProcAddressArch(proc);
#endif
}

spBool spGLUseFont(spComponent gldrawable, char *font_name, int first, int count, int list_base)
{
#if defined(SP_GL_ES)
    return SP_FALSE;
#else
    if (strnone(font_name) || first < 0 || count <= 0 || list_base < 0
	|| spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;

    return spGLUseFontArch(gldrawable, font_name, first, count, list_base);
#endif
}

/*---- matrix function for data exchange between host and GLSL; 0.5.0+ ----*/
spGLMat4 spGLMat4CreateIdentity(void)
{
    GLint i, j;
    spGLMat4 E;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    if (i == j) {
		E.data[i][j] = 1.0;
	    } else {
		E.data[i][j] = 0.0;
	    }
	}
    }

    return E;
}

spGLMat4 spGLMat4CreateZero(void)
{
    GLint i, j;
    spGLMat4 O;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    O.data[i][j] = 0.0;
	}
    }

    return O;
}

spGLMat4 spGLMat4CreateRotation(GLfloat angleDegree, GLfloat x, GLfloat y, GLfloat z, spBool rowMajor)
{
    GLfloat c, s, a;
    GLfloat xs, ys, zs;
    spGLMat4 R;

    R = spGLMat4CreateIdentity();

    c = (GLfloat)cos(2.0 * PI * (double)angleDegree / 360.0);
    s = (GLfloat)sin(2.0 * PI * (double)angleDegree / 360.0);
    a = 1.0f - c;

    R.data[0][0] = x * x * a + c;
    R.data[1][1] = y * y * a + c;
    R.data[2][2] = z * z * a + c;
    R.data[0][1] = R.data[1][0] = x * y * a;
    R.data[0][2] = R.data[2][0] = x * z * a;
    R.data[1][2] = R.data[2][1] = y * z * a;

    xs = x * s;
    ys = y * s;
    zs = z * s;
    
    if (rowMajor) {
	R.data[0][1] -= zs;
	R.data[1][0] += zs;
	R.data[0][2] += ys;
	R.data[2][0] -= ys;
	R.data[1][2] -= xs;
	R.data[2][1] += xs;
    } else {
	R.data[0][1] += zs;
	R.data[1][0] -= zs;
	R.data[0][2] -= ys;
	R.data[2][0] += ys;
	R.data[1][2] += xs;
	R.data[2][1] -= xs;
    }

    return R;
}

spGLMat4 spGLMat4CreateScaling(GLfloat x, GLfloat y, GLfloat z)
{
    spGLMat4 S;

    S = spGLMat4CreateIdentity();

    S.data[0][0] = x;
    S.data[1][1] = y;
    S.data[2][2] = z;

    return S;
}

spGLMat4 spGLMat4CreateTranslation(GLfloat x, GLfloat y, GLfloat z, spBool rowMajor)
{
    spGLMat4 T;

    T = spGLMat4CreateIdentity();

    if (rowMajor) {
	T.data[0][3] = x;
	T.data[1][3] = y;
	T.data[2][3] = z;
    } else {
	T.data[3][0] = x;
	T.data[3][1] = y;
	T.data[3][2] = z;
    }
    
    return T;
}

spGLMat4 spGLMat4CreateOrtho(GLfloat left, GLfloat right, GLfloat bottom, GLfloat top,
			     GLfloat nearVal, GLfloat farVal, spBool rowMajor)
{
    spGLMat4 A;
    GLfloat rml, tmb, fmn;
    GLfloat tx, ty, tz;

    rml = right - left;
    tmb = top - bottom;
    fmn = farVal - nearVal;

    tx = -(right + left) / rml;
    ty = -(top + bottom) / tmb;
    tz = -(farVal + nearVal) / fmn;

    A = spGLMat4CreateIdentity();
    A.data[0][0] = 2.0f / rml;
    A.data[1][1] = 2.0f / tmb;
    A.data[2][2] = -2.0f / fmn;

    if (rowMajor) {
	A.data[0][3] = tx;
	A.data[1][3] = ty;
	A.data[2][3] = tz;
    } else {
	A.data[3][0] = tx;
	A.data[3][1] = ty;
	A.data[3][2] = tz;
    }

    return A;
}

spGLMat4 spGLMat4CreateFrustum(GLfloat left, GLfloat right, GLfloat bottom, GLfloat top,
			       GLfloat nearVal, GLfloat farVal, spBool rowMajor)
{
    spGLMat4 A;
    GLfloat rml, tmb, fmn;
    GLfloat a, b, d;
    
    rml = right - left;
    tmb = top - bottom;
    fmn = farVal - nearVal;

    a = (right + left) / rml;
    b = (top + bottom) / tmb;
    d = -(2.0f * farVal * nearVal) / fmn;

    A = spGLMat4CreateZero();
    A.data[0][0] = 2.0f * nearVal / rml;
    A.data[1][1] = 2.0f * nearVal / tmb;
    A.data[2][2] = -(farVal + nearVal) / fmn;

    if (rowMajor) {
	A.data[0][2] = a;
	A.data[1][2] = b;
	A.data[2][3] = d;
	A.data[3][2] = -1.0f;
    } else {
	A.data[2][0] = a;
	A.data[2][1] = b;
	A.data[3][2] = d;
	A.data[2][3] = -1.0f;
    }
    
    return A;
}

spGLMat4 spGLMat4CreatePerspective(GLfloat fovyDegree, GLfloat aspect, GLfloat zNear, GLfloat zFar, spBool rowMajor)
{
    spGLMat4 A;
    GLfloat f, znmzf;

    f = (GLfloat)(1.0 / tan((double)fovyDegree / 2.0));
    znmzf = zNear - zFar;
    
    A = spGLMat4CreateZero();
    A.data[0][0] = f / aspect;
    A.data[1][1] = f;
    A.data[2][2] = (zFar + zNear) / znmzf;

    if (rowMajor) {
	A.data[2][3] = 2.0f * zFar * zNear / znmzf;
	A.data[3][2] = -1.0f;
    } else {
	A.data[3][2] = 2.0f * zFar * zNear / znmzf;
	A.data[2][3] = -1.0f;
    }
    
    return A;
}

spGLMat4 spGLMat4CreateLookAt(GLfloat eyeX, GLfloat eyeY, GLfloat eyeZ,
			      GLfloat centerX, GLfloat centerY, GLfloat centerZ,
			      GLfloat upX, GLfloat upY, GLfloat upZ, spBool rowMajor)
{
    spGLMat4 A;
    GLfloat zX, zY, zZ;
    GLfloat zss, zWeight;
    GLfloat xX, xY, xZ;
    GLfloat xss, xWeight;
    GLfloat yX, yY, yZ;
    GLfloat tX, tY, tZ;

    /* z = (eye - center) */
    zX = eyeX - centerX;
    zY = eyeY - centerY;
    zZ = eyeZ - centerZ;
    zss = zX * zX + zY * zY + zZ * zZ;
    if (zss == 0.0f) {
	zWeight = 1.0f;
    } else {
	zWeight = (GLfloat)(1.0 / sqrt((double)zss));
    }
    zX *= zWeight;
    zY *= zWeight;
    zZ *= zWeight;

    /* x = up * z */
    xX = upY * zZ - upZ * zY;
    xY = upZ * zX - upX * zZ;
    xZ = upX * zY - upY * zX;
    xss = xX * xX + xY * xY + xZ * xZ;
    if (xss == 0.0f) {
	xWeight = 1.0f;
    } else {
	xWeight = (GLfloat)(1.0 / sqrt((double)xss));
    }
    xX *= xWeight;
    xY *= xWeight;
    xZ *= xWeight;
    
    /* y = z * x */
    yX = zY * xZ - zZ * xY;
    yY = zZ * xX - zX * xZ;
    yZ = zX * xY - zY * xX;

    /* translate */
    tX = -(eyeX * xX + eyeY * xY + eyeZ * xZ);
    tY = -(eyeX * yX + eyeY * yY + eyeZ * yZ);
    tZ = -(eyeX * zX + eyeY * zY + eyeZ * zZ);
    
    A.data[0][0] = xX;
    A.data[1][1] = yY;
    A.data[2][2] = zZ;
    A.data[3][3] = 1.0;
    
    if (rowMajor) {
	A.data[0][1] = xY;
	A.data[0][2] = xZ;
	A.data[1][0] = yX;
	A.data[1][2] = yZ;
	A.data[2][0] = zX;
	A.data[2][1] = zY;
	
	A.data[0][3] = tX;
	A.data[1][3] = tY;
	A.data[2][3] = tZ;

	A.data[3][0] = A.data[3][1] = A.data[3][2] = 0.0f;
    } else {
	A.data[1][0] = xY;
	A.data[2][0] = xZ;
	A.data[0][1] = yX;
	A.data[2][1] = yZ;
	A.data[0][2] = zX;
	A.data[1][2] = zY;
	
	A.data[3][0] = tX;
	A.data[3][1] = tY;
	A.data[3][2] = tZ;

	A.data[0][3] = A.data[1][3] = A.data[2][3] = 0.0f;
    }
    
    return A;
}

/*
 * lu
 *   reference: ``C gengo ni yoru saishin algorithm jiten'' [in Japanese]
 *   see http://oku.edu.mie-u.ac.jp/~okumura/algo/
 */
#define SP_GL_LU_EPSILON 1.0e-30
static GLfloat lu(GLfloat a[4][4], GLint *ip)
{
    GLint n;
    GLint i, j, k, ii, ik;
    GLfloat t, u, det;
    GLfloat weight[4];

    n = 4;

    det = 0;
    for (k = 0; k < n; k++) {
	ip[k] = k;
        u = 0.0;
        for (j = 0; j < n; j++) {
            t = FABS(a[k][j]);  if (t > u) u = t;
        }
	spDebug(100, "lu", "weight calculation: k = %ld / %ld, u = %f\n", k, n, u);
	
        if (FABS(u) < SP_GL_LU_EPSILON) goto EXIT;
	weight[k] = 1 / u;
	spDebug(100, "lu", "weight calculation: weight[%ld] = %f\n", k, weight[k]);
    }
    det = 1;
    for (k = 0; k < n; k++) {
        u = -1.0;
        for (i = k; i < n; i++) {
            ii = ip[i];
            t = FABS(a[ii][k]) * weight[ii];
            if (t > u) {  u = t;  j = i;  }
        }
        ik = ip[j];
        if (j != k) {
            ip[j] = ip[k];
	    ip[k] = ik;
            det = -det;
        }
        u = a[ik][k];
	det *= u;
	spDebug(100, "lu", "det calculation: k = %ld / %ld, u = %f, det = %f\n", k, n, u, det);
	
        if (FABS(u) < SP_GL_LU_EPSILON) goto EXIT;
        for (i = k + 1; i < n; i++) {
            ii = ip[i];
            t = (a[ii][k] /= u);
            for (j = k + 1; j < n; j++)
                a[ii][j] -= t * a[ik][j];
        }
    }
  EXIT:
    
    spDebug(80, "lu", "det = %f\n", det);
    return det;
}

GLfloat spGLMat4LUDecomposition(spGLMat4 *A, GLint *index /* index[4] */)
{
    return lu(A->data, index);
}

spGLMat4 spGLMat4CreateInverse(spGLMat4 *A, spBool transpose)
{
    GLint i, ip, j, k;
    GLint N;
    GLfloat t;
    GLint index[4];
    spGLMat4 LU;
    spGLMat4 Ainv;

    spGLMat4Copy(&LU, A, transpose);
    if (spGLMat4LUDecomposition(&LU, index) == 0.0f) {
	return spGLMat4CreateIdentity();
    }

    N = 4;

    for (k = 0; k < N; k++) {
	for (i = 0; i < N; i++) {
	    ip = index[i];
	    
	    t = (ip == k ? 1.0f : 0.0f);
	    for (j = 0; j < i; j++) {
		t -= LU.data[ip][j] * Ainv.data[j][k];
	    }
	    Ainv.data[i][k] = t;
	}
	for (i = N - 1; i >= 0; i--) {
	    ip = index[i];
	    t = Ainv.data[i][k];
	    for (j = i + 1; j < N; j++) {
		t -= LU.data[ip][j] * Ainv.data[j][k];
	    }
	    Ainv.data[i][k] = t / LU.data[ip][i];
	}
    }
    
    return Ainv;
}

spGLMat4 spGLMat4CreateNormal(spGLMat4 *A, spBool transpose)
{
    spGLMat4 B;

    spGLMat4Copy(&B, A, SP_FALSE);
    B.data[0][3] = B.data[1][3] = B.data[2][3] = B.data[3][0] = B.data[3][1] = B.data[3][2] = 0.0f;
    B.data[3][3] = 1.0f;

    return spGLMat4CreateInverse(&B, spIsFalse(transpose));
}

spGLMat4 spGLMat4Createfv(GLfloat *data, spBool transpose)
{
    spGLMat4 A;

    spGLMat4Setfv(&A, data, transpose);
    
    return A;
}
	
void spGLMat4Setfv(spGLMat4 *A, GLfloat *data, spBool transpose)
{
    if (transpose == SP_FALSE) {
	memcpy(A->data, data, sizeof(GLfloat) * 16);
    } else {
	GLint i, j;

        for (i = 0; i < 4; i++) {
	    for (j = 0; j < 4; j++) {
		A->data[i][j] = data[j * 4 + i];
	    }
	}
    }

    return;
}
	
void spGLMat4Getfv(spGLMat4 *A, GLfloat *data, spBool transpose)
{
    if (transpose == SP_FALSE) {
	memcpy(data, A->data, sizeof(GLfloat) * 16);
    } else {
	GLint i, j;

        for (i = 0; i < 4; i++) {
	    for (j = 0; j < 4; j++) {
		data[j * 4 + i] = A->data[i][j];
	    }
	}
    }

    return;
}

void spGLMat4SetRow4fv(spGLMat4 *A, GLint row, GLfloat *vec, spBool rowMajor)
{
    if (rowMajor) {
	memcpy(A->data[row], vec, sizeof(GLfloat) * 4);
    } else {
	GLint j;

	for (j = 0; j < 4; j++) {
	    A->data[j][row] = vec[j];
	}
    }
}

void spGLMat4SetColumn4fv(spGLMat4 *A, GLint column, GLfloat *vec, spBool rowMajor)
{
    if (rowMajor == SP_FALSE) {
	memcpy(A->data[column], vec, sizeof(GLfloat) * 4);
    } else {
	GLint i;

	for (i = 0; i < 4; i++) {
	    A->data[i][column] = vec[i];
	}
    }
}

void spGLMat4GetRow4fv(spGLMat4 *A, GLint row, GLfloat *vec, spBool rowMajor)
{
    if (rowMajor) {
	memcpy(vec, A->data[row], sizeof(GLfloat) * 4);
    } else {
	GLint j;

	for (j = 0; j < 4; j++) {
	    vec[j] = A->data[j][row];
	}
    }
}

void spGLMat4GetColumn4fv(spGLMat4 *A, GLint column, GLfloat *vec, spBool rowMajor)
{
    if (rowMajor == SP_FALSE) {
	memcpy(vec, A->data[column], sizeof(GLfloat) * 4);
    } else {
	GLint i;

	for (i = 0; i < 4; i++) {
	    vec[i] = A->data[i][column];
	}
    }
}

spGLMat4 spGLMat4Clone(spGLMat4 *A, spBool transpose)
{
    spGLMat4 B;
    
    if (transpose == SP_FALSE) {
	memcpy(B.data, A->data, sizeof(GLfloat) * 16);
    } else {
	GLint i, j;

        for (i = 0; i < 4; i++) {
	    for (j = 0; j < 4; j++) {
		B.data[i][j] = A->data[j][i];
	    }
	}
    }

    return B;
}

void spGLMat4Copy(spGLMat4 *D, spGLMat4 *S, spBool transpose)
{
    if (transpose == SP_FALSE) {
	memcpy(D->data, S->data, sizeof(GLfloat) * 16);
    } else {
	GLint i, j;

        for (i = 0; i < 4; i++) {
	    for (j = 0; j < 4; j++) {
		D->data[i][j] = S->data[j][i];
	    }
	}
    }

    return;
}

void spGLMat4ArrayToVec(GLfloat *dest, spGLMat4 *Ss, GLint n, spBool transpose)
{
    GLint k;
    GLint pos;
    
    if (transpose == SP_FALSE) {
        for (k = 0, pos = 0; k < n; k++) {
            memcpy(dest + pos, Ss[k].data, sizeof(GLfloat) * 16);
            pos += 16;
        }
    } else {
	GLint i, j;

        for (k = 0, pos = 0; k < n; k++) {
            for (i = 0; i < 4; i++) {
                for (j = 0; j < 4; j++) {
                    dest[pos] = Ss[k].data[j][i];
                    ++pos;
                }
            }
        }
    }

    return;
}

void spGLMat4ArrayFromVec(spGLMat4 *Ds, GLfloat *src, GLint n, spBool transpose)
{
    GLint k;
    GLint pos;
    
    if (transpose == SP_FALSE) {
        for (k = 0, pos = 0; k < n; k++) {
            memcpy(Ds[k].data, src + pos, sizeof(GLfloat) * 16);
            pos += 16;
        }
    } else {
	GLint i, j;

        for (k = 0, pos = 0; k < n; k++) {
            for (i = 0; i < 4; i++) {
                for (j = 0; j < 4; j++) {
                    Ds[k].data[j][i] = src[pos];
                    ++pos;
                }
            }
        }
    }

    return;
}

void spGLMat4Transpose(spGLMat4 *A)
{
    GLint i, j;
    GLfloat value;
    
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    value = A->data[i][j];
	    A->data[i][j] = A->data[j][i];
	    A->data[j][i] = value;
	}
    }

    return;
}

spGLMat4 spGLMat4MultMatrix(spGLMat4 *A, spGLMat4 *B, spBool rowMajor)
{
    GLint i, j, k;
    spGLMat4 C;

    if (rowMajor) {
	for (i = 0; i < 4; i++) {
	    for (k = 0; k < 4; k++) {
		for (j = 0; j < 4; j++) {
		    if (k == 0) {
			C.data[i][j] = 0.0;
		    }
		    C.data[i][j] += A->data[i][k] * B->data[k][j];
		}
	    }
	}
    } else {
	for (j = 0; j < 4; j++) {
	    for (k = 0; k < 4; k++) {
		for (i = 0; i < 4; i++) {
		    if (k == 0) {
			C.data[j][i] = 0.0;
		    }
		    C.data[j][i] += A->data[k][i] * B->data[j][k];
		}
	    }
	}
    }
    
    return C;
}

spGLMat4 spGLMat4MultTransposeMatrix(spGLMat4 *A, spGLMat4 *Bt, spBool rowMajor)
{
    GLint i, j, k;
    spGLMat4 C;

    if (rowMajor) {
	for (i = 0; i < 4; i++) {
	    for (k = 0; k < 4; k++) {
		for (j = 0; j < 4; j++) {
		    if (k == 0) {
			C.data[i][j] = 0.0;
		    }
		    C.data[i][j] += A->data[i][k] * Bt->data[j][k];
		}
	    }
	}
    } else {
	for (j = 0; j < 4; j++) {
	    for (k = 0; k < 4; k++) {
		for (i = 0; i < 4; i++) {
		    if (k == 0) {
			C.data[j][i] = 0.0;
		    }
		    C.data[j][i] += A->data[k][i] * Bt->data[k][j];
		}
	    }
	}
    }
    
    return C;
}

void spGLMat4MultVector3fv(GLfloat *y, spGLMat4 *A, GLfloat *x, GLfloat xw, spBool rowMajor)
{
    GLint i;

    if (rowMajor) {
	for (i = 0; i < 3; i++) {
	    y[i] = A->data[i][0] * x[0] + A->data[i][1] * x[1] + A->data[i][2] * x[2] + A->data[i][3] * xw;
	}
    } else {
	for (i = 0; i < 3; i++) {
	    y[i] = A->data[0][i] * x[0] + A->data[1][i] * x[1] + A->data[2][i] * x[2] + A->data[3][i] * xw;
	}
    }

    return;
}

void spGLMat4MultVector4fv(GLfloat *y, spGLMat4 *A, GLfloat *x, spBool rowMajor)
{
    GLint i;

    if (rowMajor) {
	for (i = 0; i < 4; i++) {
	    y[i] = A->data[i][0] * x[0] + A->data[i][1] * x[1] + A->data[i][2] * x[2] + A->data[i][3] * x[3];
	}
    } else {
	for (i = 0; i < 4; i++) {
	    y[i] = A->data[0][i] * x[0] + A->data[1][i] * x[1] + A->data[2][i] * x[2] + A->data[3][i] * x[3];
	}
    }
    
    return;
}

void spGLMat4FDump(spGLMat4 *A, spBool transpose, FILE *fp)
{
    long i, j;

    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    if (transpose) {
		fprintf(fp, "%.12f  ", A->data[j][i]);
	    } else {
		fprintf(fp, "%.12f  ", A->data[i][j]);
	    }
	}
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return;
}

void spGLMat4Dump(spGLMat4 *A, spBool transpose)
{
    spGLMat4FDump(A, transpose, stdout);
    return;
}

void spGLVecValuesfv(GLint n, GLfloat *a, GLfloat value)
{
    GLint i;
    
    for (i = 0; i < n; i++) {
	a[i]  = value;
    }

    return;
}

void spGLVecZerofv(GLint n, GLfloat *a)
{
    spGLVecValuesfv(n, a, 0.0f);
    return;
}

void spGLVecMultScalarfv(GLint n, GLfloat *a, GLfloat s)
{
    GLint i;
    
    for (i = 0; i < n; i++) {
	a[i]  *= s;
    }

    return;
}

void spGLVecAddfv(GLint n, GLfloat *c, GLfloat *a, GLfloat *b) /* c and a can have same address */
{
    GLint i;
    
    for (i = 0; i < n; i++) {
	c[i] = a[i] + b[i];
    }
}

void spGLVecSubtractfv(GLint n, GLfloat *c, GLfloat *a, GLfloat *b) /* c and a can have same address */
{
    GLint i;
    
    for (i = 0; i < n; i++) {
	c[i] = a[i] - b[i];
    }
}

GLfloat spGLVecDotfv(GLint n, GLfloat *a, GLfloat *b) /* a and b can have same address */
{
    GLint i;
    GLfloat value = 0.0f;
    
    for (i = 0; i < n; i++) {
	value += a[i] * b[i];
    }

    return value;
}

void spGLVecCross3fv(GLfloat *c, GLfloat *a, GLfloat *b)
{
    GLfloat cx, cy, cz;
    
    cx = a[1] * b[2] - a[2] * b[1];
    cy = a[2] * b[0] - a[0] * b[2];
    cz = a[0] * b[1] - a[1] * b[0];
    
    c[0] = cx; c[1] = cy; c[2] = cz;
    
    return;
}

GLfloat spGLVecLengthfv(GLint n, GLfloat *a)
{
    return (GLfloat)sqrt(spGLVecDotfv(n, a, a));
}

GLfloat spGLVecNormalizefv(GLint n, GLfloat *a)
{
    double den;

    den = sqrt(spGLVecDotfv(n, a, a));
    
    if (den != 0.0) {
	spGLVecMultScalarfv(n, a, (GLfloat)(1.0 / den));
    }

    return (GLfloat)den;
}

#if defined(SP_GL_ES)
static int sp_gles_mode;
static int sp_gles_num_normal = 0;
static GLfloat *sp_gles_normals = NULL;
static int sp_gles_num_vertex = 0;
static GLfloat *sp_gles_vertices = NULL;

#ifdef glNormal3f
#undef glNormal3f
#endif

static GLfloat *allocVectors(void)
{
    GLfloat *vectors;

    vectors = xalloc(3, GLfloat);
    
    return vectors;
}

static void freeVectors(GLfloat *vectors)
{
    if (vectors != NULL) {
	xfree(vectors);
    }
}

static void freeAllVectors(void)
{
    if (sp_gles_normals != NULL) {
	freeVectors(sp_gles_normals);
    }
    sp_gles_num_normal = 0;
    sp_gles_normals = NULL;
    
    if (sp_gles_vertices != NULL) {
	freeVectors(sp_gles_vertices);
    }
    sp_gles_num_vertex = 0;
    sp_gles_vertices = NULL;
    spDebug(80, "freeAllVectors", "done\n");
}

static GLfloat *reallocVectors(GLfloat *vectors, int num)
{
    vectors = xrealloc(vectors, num, GLfloat);
    spDebug(80, "reallocVectors", "done\n");

    return vectors;
}

static GLfloat *setVector(GLfloat *vectors, int *pindex, GLfloat x, GLfloat y, GLfloat z)
{
    int index;
    int vindex;
    
    spDebug(80, "setVector", "in: x,y,z = %f,%f,%f\n", x, y, z);
    
    index = *pindex;

    vectors = reallocVectors(vectors, index + 3);

    vectors[index++] = x;
    vectors[index++] = y;
    vectors[index++] = z;

    vindex = index / 3;
    spDebug(80, "setVector", "vindex = %d\n", vindex);
    
    if (sp_gles_mode == GL_QUADS && vindex % 6 == 3) {
	spDebug(80, "setVector", "quad to triangle for GL_QUADS, index = %d, vindex = %d\n", index, vindex);
	vindex -= 3;
	/* 1st point in next triangle */
	vectors = setVector(vectors, &index,
			    vectors[3 * vindex], vectors[3 * vindex + 1], vectors[3 * vindex + 2]);
	spDebug(80, "setVector", "set 1st point in next triangle for GL_QUADS done: index = %d\n", index);

	vindex += 2;
	/* 2nd point in next triangle */
	vectors = setVector(vectors, &index,
			    vectors[3 * vindex], vectors[3 * vindex + 1], vectors[3 * vindex + 2]);
	spDebug(80, "setVector", "set 2nd point in next triangle for GL_QUADS done: index = %d\n", index);
    }
    
    *pindex = index;

    spDebug(80, "setVector", "done: index = %d\n", index);
    
    return vectors;
}

static GLfloat *extendVector(GLfloat *vectors, int *pindex)
{
    int index;
    GLfloat x, y, z;

    if (*pindex < 3) {
	spDebug(80, "extendVector", "extend not required\n");
	return vectors;
    }
    
    index = *pindex;
    spDebug(80, "extendVector", "index = %d\n", index);
    x = vectors[index - 3];
    y = vectors[index - 2];
    z = vectors[index - 1];
    
    vectors = setVector(vectors, pindex, x, y, z);
    
    spDebug(80, "extendVector", "done: index = %d\n", *pindex);
    
    return vectors;
}

void glBegin(GLenum mode)
{
    freeAllVectors();
    sp_gles_mode = mode;
    spDebug(80, "glBegin", "mode = %d\n", sp_gles_mode);
}

void glVertex3f(GLfloat x, GLfloat y, GLfloat z)
{
    if (sp_gles_mode >= 0) {
	spDebug(80, "glVertex3f", "in: sp_gles_num_vertex = %d, sp_gles_num_normal = %d\n",
		sp_gles_num_vertex, sp_gles_num_normal);
	sp_gles_vertices = setVector(sp_gles_vertices, &sp_gles_num_vertex, x, y, z);
	spDebug(80, "glVertex3f", "done: sp_gles_num_vertex = %d, sp_gles_num_normal = %d\n",
		sp_gles_num_vertex, sp_gles_num_normal);
    }
}

void glVertex3fv(GLfloat *v)
{
    glVertex3f(v[0], v[1], v[2]);
}

void glVertex2f(GLfloat x, GLfloat y)
{
    glVertex3f(x, y, 0.0f);
}

void glVertex2fv(GLfloat *v)
{
    glVertex2f(v[0], v[1]);
}

void spGLNormal3f(GLfloat nx, GLfloat ny, GLfloat nz)
{
    if (sp_gles_mode >= 0) {
	spDebug(80, "spGLNormal3f", "in: sp_gles_num_normal = %d, sp_gles_num_vertex = %d\n",
		sp_gles_num_normal, sp_gles_num_vertex);
	if (sp_gles_num_normal > 3) { /* we don't extend vector if single vector only */
	    while (sp_gles_num_normal < sp_gles_num_vertex) {
		sp_gles_normals = extendVector(sp_gles_normals, &sp_gles_num_normal);
	    }
	}
	spDebug(80, "spGLNormal3f", "after extend: sp_gles_num_normal = %d, sp_gles_num_vertex = %d\n",
		sp_gles_num_normal, sp_gles_num_vertex);
	
	sp_gles_normals = setVector(sp_gles_normals, &sp_gles_num_normal, nx, ny, nz);
	spDebug(80, "spGLNormal3f", "done for mode %d: sp_gles_num_normal = %d\n",
		sp_gles_mode, sp_gles_num_normal);
    } else {
	glNormal3f(nx, ny, nz);
	spDebug(80, "spGLNormal3f", "glNormal3f done\n");
    }
}

void spGLNormal3fv(GLfloat *v)
{
    spGLNormal3f(v[0], v[1], v[2]);
}

void glEnd(void)
{
    int mode;

    spDebug(80, "glEnd", "in: sp_gles_mode = %d\n", sp_gles_mode);
    
    if (sp_gles_mode < 0 || sp_gles_num_vertex <= 0) {
	spDebug(10, "glEnd", "failed: glStart has not been called\n");
	return;
    }

    if (sp_gles_mode == GL_QUAD_STRIP) {
	mode = GL_TRIANGLE_STRIP;
    } else if (sp_gles_mode == GL_QUADS) {
	mode = GL_TRIANGLES;
    } else if (sp_gles_mode == GL_POLYGON) {
	mode = GL_TRIANGLE_FAN;
    } else {
	mode = sp_gles_mode;
    }
    
    spDebug(80, "glEnd", "sp_gles_num_vertex = %d, sp_gles_num_normal = %d\n",
	    sp_gles_num_vertex, sp_gles_num_normal);
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, sp_gles_vertices);
    if (sp_gles_num_normal > 3) {
	while (sp_gles_num_normal < sp_gles_num_vertex) {
	    sp_gles_normals = extendVector(sp_gles_normals, &sp_gles_num_normal);
	}
	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 0, sp_gles_normals);
    } else if (sp_gles_num_normal == 3) {
	glNormal3f(sp_gles_normals[sp_gles_num_normal - 3],
		   sp_gles_normals[sp_gles_num_normal - 2],
		   sp_gles_normals[sp_gles_num_normal - 1]);
    }
    glDrawArrays(mode, 0, sp_gles_num_vertex / 3);
    glDisableClientState(GL_VERTEX_ARRAY);
    
    if (sp_gles_num_normal > 0) {
	if (sp_gles_num_normal > 3) {
	    glDisableClientState(GL_NORMAL_ARRAY);
	    glNormalPointer(GL_FLOAT, 0, NULL);
	}
	glNormal3f(sp_gles_normals[sp_gles_num_normal - 3],
		   sp_gles_normals[sp_gles_num_normal - 2],
		   sp_gles_normals[sp_gles_num_normal - 1]);
    }

    freeAllVectors();
    sp_gles_mode = -1;

    spDebug(80, "glEnd", "done\n");
}

void glColor4fv(GLfloat *v)
{
    glColor4f(v[0], v[1], v[2], v[3]);
    return;
}

/*
 * gluPerspective and gluLookAt are from Android's `san-angeles' sample.
 */
void gluPerspective(GLfloat fovyDegree, GLfloat aspect, GLfloat zNear, GLfloat zFar)
{
    GLfloat xmin, xmax, ymin, ymax;

    ymax = zNear * (GLfloat)tan(fovyDegree * PI / 360);
    ymin = -ymax;
    xmin = ymin * aspect;
    xmax = ymax * aspect;

    glFrustumx((GLfixed)(xmin * 65536), (GLfixed)(xmax * 65536),
               (GLfixed)(ymin * 65536), (GLfixed)(ymax * 65536),
               (GLfixed)(zNear * 65536), (GLfixed)(zFar * 65536));
}

/* Following gluLookAt implementation is adapted from the
 * Mesa 3D Graphics library. http://www.mesa3d.org
 */
void gluLookAt(GLfloat eyex, GLfloat eyey, GLfloat eyez,
	       GLfloat centerx, GLfloat centery, GLfloat centerz,
	       GLfloat upx, GLfloat upy, GLfloat upz)
{
    GLfloat m[16];
    GLfloat x[3], y[3], z[3];
    GLfloat mag;

    /* Make rotation matrix */

    /* Z vector */
    z[0] = eyex - centerx;
    z[1] = eyey - centery;
    z[2] = eyez - centerz;
    mag = (float)sqrt(z[0] * z[0] + z[1] * z[1] + z[2] * z[2]);
    if (mag) {			/* mpichler, 19950515 */
        z[0] /= mag;
        z[1] /= mag;
        z[2] /= mag;
    }

    /* Y vector */
    y[0] = upx;
    y[1] = upy;
    y[2] = upz;

    /* X vector = Y cross Z */
    x[0] = y[1] * z[2] - y[2] * z[1];
    x[1] = -y[0] * z[2] + y[2] * z[0];
    x[2] = y[0] * z[1] - y[1] * z[0];

    /* Recompute Y = Z cross X */
    y[0] = z[1] * x[2] - z[2] * x[1];
    y[1] = -z[0] * x[2] + z[2] * x[0];
    y[2] = z[0] * x[1] - z[1] * x[0];

    /* mpichler, 19950515 */
    /* cross product gives area of parallelogram, which is < 1.0 for
     * non-perpendicular unit-length vectors; so normalize x, y here
     */

    mag = (float)sqrt(x[0] * x[0] + x[1] * x[1] + x[2] * x[2]);
    if (mag) {
        x[0] /= mag;
        x[1] /= mag;
        x[2] /= mag;
    }

    mag = (float)sqrt(y[0] * y[0] + y[1] * y[1] + y[2] * y[2]);
    if (mag) {
        y[0] /= mag;
        y[1] /= mag;
        y[2] /= mag;
    }

#define M(row,col)  m[col*4+row]
    M(0, 0) = x[0];
    M(0, 1) = x[1];
    M(0, 2) = x[2];
    M(0, 3) = 0.0;
    M(1, 0) = y[0];
    M(1, 1) = y[1];
    M(1, 2) = y[2];
    M(1, 3) = 0.0;
    M(2, 0) = z[0];
    M(2, 1) = z[1];
    M(2, 2) = z[2];
    M(2, 3) = 0.0;
    M(3, 0) = 0.0;
    M(3, 1) = 0.0;
    M(3, 2) = 0.0;
    M(3, 3) = 1.0;
#undef M
    {
        int a;
        GLfixed fixedM[16];
        for (a = 0; a < 16; ++a)
            fixedM[a] = (GLfixed)(m[a] * 65536);
        glMultMatrixx(fixedM);
    }

    /* Translate Eye to Origin */
    glTranslatex((GLfixed)(-eyex * 65536),
                 (GLfixed)(-eyey * 65536),
                 (GLfixed)(-eyez * 65536));
}
#endif
