#include <math.h>
#include <sp/spComponentLib.h>

#if defined(ANDROID) || defined(IPHONE)
#define USE_OPENGL_ES2 1
#elif defined(MACOSX)
#include <AvailabilityMacros.h>
#if defined(MAC_OS_X_VERSION_10_15) && MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_15
#define USE_OPENGL3 1
#endif
#endif

#include <sp/spGLP.h>
#include <sp/spGLShaderP.h>

#define USE_HALFWAY_VECTOR
//#define USE_DIFFUSE_ONLY

#if defined(SP_GL_ES)
#undef GL_FRONT			/* GL_FRONT doesn't work in glMaterialfv */
#define GL_FRONT GL_FRONT_AND_BACK
#endif

#include <sp/spGLPhongModel.h>

#define SP_GL_ARRAY_BUFFER 0x8892
#define SP_GL_ELEMENT_ARRAY_BUFFER 0x8893

typedef struct _spGLSLLightParams
{
    GLboolean last_enabled;

#if defined(SP_GL_SHADER_SUPPORTED)
    GLint position_location;
    GLint ambient_location;
    GLint diffuse_location;
    GLint specular_location;
    
    GLint spot_cutoff_location;
    GLint spot_exponent_location;
    GLint spot_direction_location;
    GLint attenuation_location;
#endif
    
    GLfloat position[4];        /* defaut: 0, 0, 1, 0 */
    GLfloat converted_position[4];
    GLfloat ambient[4];         /* defaut: 0, 0, 0, 1 */
    GLfloat diffuse[4];         /* default: 1, 1, 1, 1 (light 0), 0, 0, 0, 1 (other lights) */
    GLfloat specular[4];        /* default: 1, 1, 1, 1 (light 0), 0, 0, 0, 1 (other lights) */
    
    GLfloat spot_cutoff;        /* default: 180 (non-spot) */
    GLfloat spot_exponent;      /* default: 0 */
    GLfloat spot_direction[3];  /* default: 0, 0, -1 */
    GLfloat spot_converted_direction[3];
    
    GLfloat attenuation[3];     /* constant att., linear att., quadratic att. (default: 1, 0, 0) */
} spGLSLLightParams;

struct _spGLPhongModel {
    unsigned long options;
    spBool use_shader;
    
    GLuint shader_program;
    GLuint vertex_shader;
    GLuint fragment_shader;

    GLboolean last_lighting;
    GLuint old_program;
    spBool begin_flag;
    spGLPhongModelDrawMode draw_mode;

#if defined(SP_GL_SHADER_SUPPORTED)
    GLint attr_position_location;
    GLint attr_normal_location;

    GLint unif_material_ambient_location;
    GLint unif_material_diffuse_location;
    GLint unif_material_specular_location;
    GLint unif_material_shininess_location;

    GLint unif_front_mix_rate_location;
    GLint unif_front_mix_color_location;
    GLint unif_back_mix_rate_location;
    GLint unif_back_mix_color_location;
    
    GLint unif_N_location;
    GLint unif_MV_location;
    GLint unif_MVP_location;
    
    GLint unif_clip_plane_location;
#endif

    spGLMat4 P;
    spGLMat4 N;
    spGLMat4 MV;
    spGLMat4 MVP;

    int light_count;
    spGLSLLightParams *lights;

    spGLMat4 material;

    GLfloat front_mix_rate;
    GLfloat front_mix_color[3];
    GLfloat back_mix_rate;
    GLfloat back_mix_color[3];

    GLint max_clip_plane;
    GLboolean *plane_enabled;
    GLfloat *plane_equation;
    GLfloat *plane_converted_equation;
    spGLMat4 *clip_plane_MV;
};

#if defined(SP_GL_SHADER_SUPPORTED)
static const GLchar sp_glpm_shader_version_line_format[] = "#version %ld\n";
static const GLchar sp_glpm_shader_version_line_format_es[] = "#version %ld es\n";

static const GLchar sp_glpm_shader_extensions[] =
#if defined(IPHONE)
    "#extension GL_APPLE_clip_distance : require\n"
#else
    ""
#endif
    ;

static const GLchar sp_glpm_vertex_shader_macros_for_version130[] =
    "#define attribute in\n"
    "#define varying out\n"
    ;

static const GLchar sp_glpm_fragment_shader_macros_for_version130[] =
    "#define varying in\n"
    "#define gl_FragColor FragColor\nout vec4 FragColor;\n"
    ;

static const GLchar sp_glpm_vertex_shader_clip_plane_vars_source_for_version130[] =
    //"out float gl_ClipDistance[gl_MaxClipDistances];\n"
    "varying float gl_ClipDistance[gl_MaxClipDistances];\n"
    "uniform vec4 unif_clipPlane[gl_MaxClipDistances];\n"
    ;

static const GLchar sp_glpm_vertex_shader_clip_plane_substitution_for_version130[] =
    "    for (int i = 0; i < gl_MaxClipDistances; i++) {\n"
    "        gl_ClipDistance[i] = dot(unif_MV * vec4(attr_position, 1.0), unif_clipPlane[i]);\n"
    "    }\n"
    ;

static const GLchar sp_glpm_vertex_shader_clip_plane_substitution_for_version120[] =
    "gl_ClipVertex = unif_MV * vec4(attr_position, 1.0);\n"
    ;

static const GLchar sp_glpm_vertex_shader_prefix_source[] =
    "\n"
    "attribute vec3 attr_position;\n"
    "attribute vec3 attr_normal;\n"
    "\n"
    "uniform mat4 unif_N;\n"
    "uniform mat4 unif_MV;\n"
    "uniform mat4 unif_MVP;\n"
    "\n"
    "void getEyeSpace(out vec3 eyeNorm, out vec4 eyePosition)\n" /* L9 */
    "{\n"
    "    eyeNorm = normalize(vec3(unif_N * vec4(attr_normal, 0.0)));\n"
    "    eyePosition = unif_MV * vec4(attr_position, 1.0);\n"
    "}\n"
    "\n" /* L14 */
    ;

static const GLchar sp_glpm_gouraud_shading_vertex_shader_flat_shading_vary_vars_source[] =
    "flat out vec4 vary_frontColor;\n"
    "flat out vec4 vary_backColor;\n"
    ;
static const GLchar sp_glpm_gouraud_shading_vertex_shader_normal_vary_vars_source[] =
    "varying vec4 vary_frontColor;\n"
    "varying vec4 vary_backColor;\n"
    ;

static const GLchar sp_glpm_gouraud_shading_vertex_shader_main_source[] =
    "void main()\n"
    "{\n"
    "    vec3 eyeNorm;\n"
    "    vec4 eyePosition;\n"
    "    \n"
    "    getEyeSpace(eyeNorm, eyePosition);\n"
    "    vary_frontColor = vec4(0.0);\n"
    "    vary_backColor = vec4(0.0);\n"
    "    for (int i = 0; i < NUM_LIGHT; i++) {\n" /* L9 */
    "        vary_frontColor += phongModel(i, eyePosition, eyeNorm);\n"
    "        vary_backColor += phongModel(i, eyePosition, -eyeNorm);\n"
    "    }\n"
    "    gl_Position = unif_MVP * vec4(attr_position, 1.0);\n" /* L13 */
    ;

static const GLchar sp_glpm_phong_shading_vertex_shader_flat_shading_vary_vars_source[] =
    "flat out vec3 vary_normal;\n"
    "out vec4 vary_position;\n"
    ;
static const GLchar sp_glpm_phong_shading_vertex_shader_normal_vary_vars_source[] =
    "varying vec3 vary_normal;\n"
    "varying vec4 vary_position;\n"
    ;

static const GLchar sp_glpm_phong_shading_vertex_shader_main_source[] =
    "void main()\n"
    "{\n"
    "    getEyeSpace(vary_normal, vary_position);\n"
    "    gl_Position = unif_MVP * vec4(attr_position, 1.0);\n" /* L4 */
    ;

static const GLchar sp_glpm_phong_model_source_num_light_format[] =
    "const int NUM_LIGHT = %d;\n";
    
static const GLchar sp_glpm_phong_model_halfway_vector_rdotv_source[] =
    "            vec3 h = normalize(v + s);\n"
    "            float rDotV = dot(h, eyeNorm);\n"
    ;
static const GLchar sp_glpm_phong_model_reflect_rdotv_source[] =
    "            vec3 r = reflect(-s, eyeNorm);\n"
    "            float rDotV = dot(r, v);\n"
    ;

static const GLchar sp_glpm_phong_model_source[] =
    "\n"
    "struct LightInfo {\n"
    "    vec4 position;\n"
    "    vec4 ambient;\n"
    "    vec4 diffuse;\n"
    "    vec4 specular;\n"
    "    vec3 attenuation;\n"
    "    float spot_cutoff;\n"
    "    float spot_exponent;\n"
    "    vec3 spot_direction;\n"
    "};\n"
    "uniform LightInfo unif_lights[NUM_LIGHT];\n"
    "\n"
    "struct MaterialInfo {\n"
    "    vec4 ambient;\n"
    "    vec4 diffuse;\n"
    "    vec4 specular;\n"
    "    float shininess;\n"
    "};\n"
    "uniform MaterialInfo unif_material;\n" /* L19 */
    "\n"
    "vec4 phongModel(int lightIndex, vec4 eyePosition, vec3 eyeNorm)\n" /* L21 */
    "{\n"
    "    vec4 lightPosition;\n"
    "    vec3 s;\n"
    "    vec3 spotDirection = vec3(0.0);\n"
    "    \n"
#if defined(MODEL_RELATIVE_LIGHT)
    "    lightPosition = unif_MV * unif_lights[lightIndex].position;\n"
#else
    "    lightPosition = unif_lights[lightIndex].position;\n"
#endif
    "    float attenuation = 1.0;\n"
    "    if (unif_lights[lightIndex].position.w == 0.0) {\n" /* L29 */
    "        s = normalize(vec3(lightPosition));\n"
    "    } else {\n"
    "        float d;\n"
    "        vec3 dv;\n"
    "        vec3 lightDirection = vec3(lightPosition - eyePosition);\n"
    "        d = length(lightDirection);\n"
    "        dv = vec3(1.0, d, d * d);\n"
    "        attenuation = 1.0 / dot(dv, unif_lights[lightIndex].attenuation);\n"
    "        s = normalize(lightDirection);\n"
    "    }\n"
    "    float cutoff = 10.0;\n"
    "    float angle = 0.0;\n"
    "    if (unif_lights[lightIndex].position.w != 0.0 && unif_lights[lightIndex].spot_cutoff != 180.0) {\n"
    "        cutoff = radians(clamp(unif_lights[lightIndex].spot_cutoff, 0.0, 90.0));\n"
    "        spotDirection = normalize(unif_lights[lightIndex].spot_direction);\n"
    "        angle = acos(dot(-s, spotDirection));\n"
    "    }\n"
    "    vec4 ambient = unif_lights[lightIndex].ambient * unif_material.ambient;\n"
    "    if (angle < cutoff) {\n" /* L48 */
    "        float spotFactor = 1.0;\n"
    "        if (unif_lights[lightIndex].position.w != 0.0 && unif_lights[lightIndex].spot_cutoff != 180.0) {\n"
    "            spotFactor = pow(dot(-s, spotDirection), unif_lights[lightIndex].spot_exponent);\n"
    "        }\n"
    "        float sDotN = max(dot(s, eyeNorm), 0.0);\n"
    "        vec4 diffuse = unif_lights[lightIndex].diffuse * unif_material.diffuse * sDotN;\n"
    "        vec4 spec = vec4(0.0);\n"
    "        if (sDotN > 0.0) {\n" /* L56 */
    "            vec3 v = normalize(-eyePosition.xyz);\n"
    "%s"
    "            float shininessWeight = pow(max(rDotV, 0.0), unif_material.shininess);\n"
    "            spec = unif_lights[lightIndex].specular * unif_material.specular * shininessWeight;\n"
    "        }\n"
#if defined(USE_DIFFUSE_ONLY)
    "        return attenuation * spotFactor * diffuse;\n"
#else
    "        return attenuation * (ambient + spotFactor * (diffuse + spec));\n"
#endif
    "    } else {\n"
    "        return attenuation * ambient;\n"
    "    }\n"
    "}\n" /* L67 */
    ;

static const GLchar sp_glpm_fragment_shader_prefix_source[] =
#if defined(SP_GL_ES)
    "precision mediump float;"
#endif
    "\n"
    ;    

static const GLchar sp_glpm_gouraud_shading_fragment_shader_flat_shading_vary_vars_source[] =
    "flat in vec4 vary_frontColor;\n"
    "flat in vec4 vary_backColor;\n"
    ;
static const GLchar sp_glpm_gouraud_shading_fragment_shader_normal_vary_vars_source[] =
    "varying vec4 vary_frontColor;\n"
    "varying vec4 vary_backColor;\n"
    ;

static const GLchar sp_glpm_gouraud_shading_fragment_shader_vars_source[] =
    "uniform float unif_frontMixRate;\n"
    "uniform vec3 unif_frontMixColor;\n"
    "uniform float unif_backMixRate;\n"
    "uniform vec3 unif_backMixColor;\n"
    ;

static const GLchar sp_glpm_gouraud_shading_fragment_shader_main_source[] =
    "void main() {\n"
    "    if (gl_FrontFacing) {\n"
    "        if (unif_frontMixRate > 0.0) {\n"
    "            gl_FragColor = mix(vary_frontColor, vec4(unif_frontMixColor, vary_frontColor.a), unif_frontMixRate);\n"
    "        } else {\n"
    "            gl_FragColor = vary_frontColor;\n"
    "        }\n"
    "    } else {\n"
    "        if (unif_backMixRate > 0.0) {\n"
    "            gl_FragColor = mix(vary_backColor, vec4(unif_backMixColor, vary_backColor.a), unif_backMixRate);\n"
    "        } else {\n"
    "            gl_FragColor = vary_backColor;\n"
    "        }\n"
    "    }\n"
    "}\n" /* L7 */
    ;

static const GLchar sp_glpm_phong_shading_fragment_shader_flat_shading_vary_vars_source[] =
    "flat in vec3 vary_normal;\n"
    "in vec4 vary_position;\n"
    ;
static const GLchar sp_glpm_phong_shading_fragment_shader_normal_vary_vars_source[] =
    "varying vec3 vary_normal;\n"
    "varying vec4 vary_position;\n"
    ;

static const GLchar sp_glpm_phong_shading_fragment_shader_vars_source[] =
    "uniform float unif_frontMixRate;\n"
    "uniform vec3 unif_frontMixColor;\n"
    "uniform float unif_backMixRate;\n"
    "uniform vec3 unif_backMixColor;\n"
    ;

static const GLchar sp_glpm_phong_shading_fragment_shader_main_source[] =
    "void main() {\n"
    "    vec4 result;\n"
    "    gl_FragColor = vec4(0.0);\n"
    "    if (gl_FrontFacing) {\n"
    "        for (int i = 0; i < NUM_LIGHT; i++) {\n" 
    "            result = phongModel(i, vary_position, vary_normal);\n"
    "            if (unif_frontMixRate > 0.0) {\n"
    "                gl_FragColor += mix(result, vec4(unif_frontMixColor, result.a), unif_frontMixRate);\n"
    "            } else {\n"
    "                gl_FragColor += result;\n"
    "            }\n"
    "        }\n"
    "    } else {\n"
    "        for (int i = 0; i < NUM_LIGHT; i++) {\n" 
    "            result = phongModel(i, vary_position, -vary_normal);\n"
    "            if (unif_backMixRate > 0.0) {\n"
    "                gl_FragColor += mix(result, vec4(unif_backMixColor, result.a), unif_backMixRate);\n"
    "            } else {\n"
    "                gl_FragColor += result;\n"
    "            }\n"
    "        }\n"
    "    }\n"
    "}\n" /* L12 */
    ;


#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
extern spBool spInitShaderGLFunctionsWin(HGLRC *global_hglrc);

HGLRC sp_glpm_shader_functions_hglrc = NULL;

SP_WGL_DECLARE_EXTERN_FUNCTION(glGetProgramiv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetProgramInfoLog)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderiv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderInfoLog)
SP_WGL_DECLARE_EXTERN_FUNCTION(glAttachShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDetachShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCreateProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCreateShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCompileShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glIsProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glIsShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glLinkProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUseProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glShaderSource)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderSource)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetAttribLocation)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetUniformLocation)
SP_WGL_DECLARE_EXTERN_FUNCTION(glEnableVertexAttribArray)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDisableVertexAttribArray)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniform1i)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniformMatrix4fv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniform1f)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniform3fv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniform4fv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glVertexAttribPointer)

SP_WGL_DECLARE_EXTERN_FUNCTION(glGenBuffers)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteBuffers)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBindBuffer)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBufferData)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBufferSubData)

SP_WGL_DECLARE_EXTERN_FUNCTION(glGenVertexArrays)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteVertexArrays)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBindVertexArray)
#endif

static GLuint loadProgram(GLuint vshader, GLuint fshader)
{
    GLuint program;

    if ((program = glCreateProgram()) != 0) {
	glAttachShader(program, vshader);
	glAttachShader(program, fshader);
	glLinkProgram(program);

	if (spGLShaderCheckLinkError(program) == GL_FALSE) {
	    glDeleteProgram(program);
	    program = 0;
	}
    }

    return program;
}
#endif

spGLPhongModel spGLPhongModelInit(int light_count, unsigned long options)
{
    int i;
    GLint vertex_shader = 0;
    GLint fragment_shader = 0;
    GLint shader_program = 0;
    spGLPhongModel glpm;
    char buf[512];
    unsigned long version;
#if defined(MACOS9)
    spBool use_shader = SP_FALSE;
#else
    spBool use_shader = SP_TRUE;
#endif
    spBool use_phong_shading = SP_FALSE;
    spBool use_flat_shading = SP_FALSE;

    if (light_count <= 0) return NULL;

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spInitShaderGLFunctionsWin(&sp_glpm_shader_functions_hglrc) == SP_FALSE) {
        spDebug(10, "spGLPhongModelInit", "spInitShaderGLFunctionsWin failed\n");
        return NULL;
    }
#endif

    version = options & SP_GL_PHONG_MODEL_SHADER_VERSION_MASK;
    
    if (options & SP_GL_PHONG_MODEL_OPTION_USE_SHADER) {
        use_shader = SP_TRUE;
    } else if (options & SP_GL_PHONG_MODEL_OPTION_USE_LEGACY_FUNCTION) {
        use_shader = SP_FALSE;
    }
    
    spDebug(100, "spGLPhongModelInit", "version = %ld, use_shader = %d, light_count = %d\n",
            version, use_shader, light_count);
    
    if (use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        GLchar *version_line_format;
        GLchar phong_model_source[2560];
        GLchar source[4096];

        if (version >= 300 && (options & SP_GL_PHONG_MODEL_SHADER_VERSION_ES)) {
            version_line_format = (GLchar *)sp_glpm_shader_version_line_format_es;
        } else {
            version_line_format = (GLchar *)sp_glpm_shader_version_line_format;
        }
        
        if (options & SP_GL_PHONG_MODEL_OPTION_USE_PHONG_SHADING) {
            use_phong_shading = SP_TRUE;
        }
        if (options & SP_GL_PHONG_MODEL_OPTION_USE_FLAT_SHADING) {
            use_flat_shading = SP_TRUE;
            if (version < 150) {
                version = 150;
            }
        }
        
        sprintf(phong_model_source, sp_glpm_phong_model_source,
                ((options & SP_GL_PHONG_MODEL_OPTION_USE_HALFWAY_VECTOR)
                 ? sp_glpm_phong_model_halfway_vector_rdotv_source
                 : sp_glpm_phong_model_reflect_rdotv_source));
#if 0
        fprintf(spgetstderr(), "phong model source (length = %d):\n%s\n", strlen(phong_model_source), phong_model_source);
#endif
        sprintf(buf, sp_glpm_phong_model_source_num_light_format, light_count);
        
        if (version > 0) {
            sprintf(source, version_line_format, version);
        } else {
            source[0] = '\n'; source[1] = NUL;
        }

#if defined(IPHONE)
        if (options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE) {
            spStrCat(source, sizeof(source), (char *)sp_glpm_shader_extensions);
        }
#endif        

        if (version >= 130) {
            spStrCat(source, sizeof(source), (char *)sp_glpm_vertex_shader_macros_for_version130);
        }
        spDebug(100, "spGLPhongModelInit", "vertex shader top source = %s\n", source);
        
        if (options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE) {
            if (version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_vertex_shader_clip_plane_vars_source_for_version130);
            }
        }
        spStrCat(source, sizeof(source), (char *)sp_glpm_vertex_shader_prefix_source);
        if (use_phong_shading) {
            if (use_flat_shading) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_vertex_shader_flat_shading_vary_vars_source);
            } else {
                spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_vertex_shader_normal_vary_vars_source);
            }
            spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_vertex_shader_main_source);
        } else {
            if (use_flat_shading) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_vertex_shader_flat_shading_vary_vars_source);
            } else {
                spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_vertex_shader_normal_vary_vars_source);
            }
            spStrCat(source, sizeof(source), (char *)buf);
            spStrCat(source, sizeof(source), (char *)phong_model_source);
            spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_vertex_shader_main_source);
        }
        if (options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE) {
            if (version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_vertex_shader_clip_plane_substitution_for_version130);
#if !defined(SP_GL_ES)
            } else if (version > 100) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_vertex_shader_clip_plane_substitution_for_version120);
#endif
            }
        }
        spStrCat(source, sizeof(source), "}\n");
#if 0
        fprintf(spgetstderr(), "vertex shader source (length = %d):\n%s\n", strlen(source), source);
#endif
        vertex_shader = spGLShaderCompile(GL_VERTEX_SHADER, source);
    
        if (version > 0) {
            sprintf(source, version_line_format, version);
            if (version >= 130) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_fragment_shader_macros_for_version130);
            }
            spDebug(100, "spGLPhongModelInit", "fragment shader top source = %s\n", source);
        } else {
            source[0] = '\n'; source[1] = NUL;
        }
        spStrCat(source, sizeof(source), (char *)sp_glpm_fragment_shader_prefix_source);
        if (use_phong_shading) {
            if (use_flat_shading) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_fragment_shader_flat_shading_vary_vars_source);
            } else {
                spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_fragment_shader_normal_vary_vars_source);
            }
            spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_fragment_shader_vars_source);
            spStrCat(source, sizeof(source), (char *)buf);
            spStrCat(source, sizeof(source), (char *)phong_model_source);
            spStrCat(source, sizeof(source), (char *)sp_glpm_phong_shading_fragment_shader_main_source);
        } else {
            if (use_flat_shading) {
                spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_fragment_shader_flat_shading_vary_vars_source);
            } else {
                spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_fragment_shader_normal_vary_vars_source);
            }
            spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_fragment_shader_vars_source);
            spStrCat(source, sizeof(source), (char *)sp_glpm_gouraud_shading_fragment_shader_main_source);
        }
#if 0
        fprintf(spgetstderr(), "fragment shader source:\n%s\n", source);
#endif
        fragment_shader = spGLShaderCompile(GL_FRAGMENT_SHADER, source);

        spDebug(100, "spGLPhongModelInit", "vertex_shader = %d, fragment_shader = %d\n",
                vertex_shader, fragment_shader);
        
        if ((shader_program = loadProgram(vertex_shader, fragment_shader)) == 0) {
            glDeleteShader(vertex_shader);
            glDeleteShader(fragment_shader);
            return NULL;
        }
#else
        return NULL;
#endif
    }

    glpm = xalloc(1, struct _spGLPhongModel);
    memset(glpm, 0, sizeof(struct _spGLPhongModel));
    
    glpm->options = options;
    glpm->use_shader = use_shader;
    
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm->use_shader) {
        glpm->vertex_shader = vertex_shader;
        glpm->fragment_shader = fragment_shader;
        glpm->shader_program = shader_program;

        glpm->attr_position_location = glGetAttribLocation(glpm->shader_program, "attr_position");
        glpm->attr_normal_location = glGetAttribLocation(glpm->shader_program, "attr_normal");
        spDebug(10, "spGLPhongModelInit", "attr_position_location = %d, attr_normal_location = %d\n",
                glpm->attr_position_location, glpm->attr_normal_location);
        
        glpm->unif_material_ambient_location = glGetUniformLocation(glpm->shader_program, "unif_material.ambient");
        glpm->unif_material_diffuse_location = glGetUniformLocation(glpm->shader_program, "unif_material.diffuse");
        glpm->unif_material_specular_location = glGetUniformLocation(glpm->shader_program, "unif_material.specular");
        glpm->unif_material_shininess_location = glGetUniformLocation(glpm->shader_program, "unif_material.shininess");
        spDebug(10, "spGLPhongModelInit", "material location: %d, %d, %d, %d\n",
                glpm->unif_material_ambient_location, glpm->unif_material_diffuse_location,
                glpm->unif_material_specular_location, glpm->unif_material_shininess_location);

        glpm->unif_front_mix_rate_location = glGetUniformLocation(glpm->shader_program, "unif_frontMixRate");
        glpm->unif_front_mix_color_location = glGetUniformLocation(glpm->shader_program, "unif_frontMixColor");
        glpm->unif_back_mix_rate_location = glGetUniformLocation(glpm->shader_program, "unif_backMixRate");
        glpm->unif_back_mix_color_location = glGetUniformLocation(glpm->shader_program, "unif_backMixColor");
        
        glpm->unif_N_location = glGetUniformLocation(glpm->shader_program, "unif_N");
        glpm->unif_MV_location = glGetUniformLocation(glpm->shader_program, "unif_MV");
        glpm->unif_MVP_location = glGetUniformLocation(glpm->shader_program, "unif_MVP");
        spDebug(10, "spGLPhongModelInit", "matrix location: %d, %d, %d\n",
                glpm->unif_N_location, glpm->unif_MV_location, glpm->unif_MVP_location);
    }
#endif

    glpm->front_mix_rate = glpm->back_mix_rate = 0.0f;

    glpm->light_count = light_count;
    glpm->lights = xalloc(glpm->light_count, struct _spGLSLLightParams);
    for (i = 0; i < glpm->light_count; i++) {
        memset(&glpm->lights[i], 0, sizeof(struct _spGLSLLightParams));
        
#if defined(SP_GL_SHADER_SUPPORTED)
        if (glpm->use_shader) {
            sprintf(buf, "unif_lights[%d].position", i);
            glpm->lights[i].position_location = glGetUniformLocation(glpm->shader_program, buf);
            sprintf(buf, "unif_lights[%d].ambient", i);
            glpm->lights[i].ambient_location = glGetUniformLocation(glpm->shader_program, buf);
            sprintf(buf, "unif_lights[%d].diffuse", i);
            glpm->lights[i].diffuse_location = glGetUniformLocation(glpm->shader_program, buf);
            sprintf(buf, "unif_lights[%d].specular", i);
            glpm->lights[i].specular_location = glGetUniformLocation(glpm->shader_program, buf);

            sprintf(buf, "unif_lights[%d].attenuation", i);
            glpm->lights[i].attenuation_location = glGetUniformLocation(glpm->shader_program, buf);
            
            sprintf(buf, "unif_lights[%d].spot_cutoff", i);
            glpm->lights[i].spot_cutoff_location = glGetUniformLocation(glpm->shader_program, buf);
            sprintf(buf, "unif_lights[%d].spot_exponent", i);
            glpm->lights[i].spot_exponent_location = glGetUniformLocation(glpm->shader_program, buf);
            sprintf(buf, "unif_lights[%d].spot_direction", i);
            glpm->lights[i].spot_direction_location = glGetUniformLocation(glpm->shader_program, buf);
            
            spDebug(10, "spGLPhongModelInit", "light%d location: %d, %d, %d, %d\n",
                    i, glpm->lights[i].position_location, glpm->lights[i].ambient_location,
                    glpm->lights[i].diffuse_location, glpm->lights[i].specular_location);
        }
#endif

        glpm->lights[i].position[0] = 0.0f; glpm->lights[i].position[1] = 0.0f;
        glpm->lights[i].position[2] = 1.0f; glpm->lights[i].position[3] = 0.0f;

        glpm->lights[i].ambient[0] = 0.0f; glpm->lights[i].ambient[1] = 0.0f;
        glpm->lights[i].ambient[2] = 0.0f; glpm->lights[i].ambient[3] = 1.0f;

        if (i == 0) {
            glpm->lights[i].diffuse[0] = 1.0f; glpm->lights[i].diffuse[1] = 1.0f;
            glpm->lights[i].diffuse[2] = 1.0f; glpm->lights[i].diffuse[3] = 1.0f;
            glpm->lights[i].specular[0] = 1.0f; glpm->lights[i].specular[1] = 1.0f;
            glpm->lights[i].specular[2] = 1.0f; glpm->lights[i].specular[3] = 1.0f;
        } else {
            glpm->lights[i].diffuse[0] = 0.0f; glpm->lights[i].diffuse[1] = 0.0f;
            glpm->lights[i].diffuse[2] = 0.0f; glpm->lights[i].diffuse[3] = 1.0f;
            glpm->lights[i].specular[0] = 0.0f; glpm->lights[i].specular[1] = 0.0f;
            glpm->lights[i].specular[2] = 0.0f; glpm->lights[i].specular[3] = 1.0f;
        }
        
        glpm->lights[i].spot_cutoff = 180.0f;
        glpm->lights[i].spot_exponent = 0.0f;
        
        glpm->lights[i].spot_direction[0] = 0.0f; glpm->lights[i].spot_direction[1] = -1.0f;
        glpm->lights[i].spot_direction[2] = 0.0f;
        
        glpm->lights[i].attenuation[0] = 1.0f; glpm->lights[i].attenuation[1] = 0.0f;
        glpm->lights[i].attenuation[2] = 0.0f;
    }

    if (glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE) {
#if defined(SP_GL_SHADER_SUPPORTED)
        if (glpm->use_shader) {
            if (version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
                glpm->unif_clip_plane_location = glGetUniformLocation(glpm->shader_program, "unif_clipPlane");
                spDebug(100, "spGLPhongModelInit", "glpm->unif_clip_plane_location = %d\n", glpm->unif_clip_plane_location);
            }
        }
#endif
        
        if (glpm->use_shader && version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
            glGetIntegerv(0x0D32/*GL_MAX_CLIP_DISTANCES*/, &glpm->max_clip_plane);
        } else {
            glGetIntegerv(GL_MAX_CLIP_PLANES, &glpm->max_clip_plane);
        }
        spDebug(100, "spGLPhongModelInit", "version = %ld, max_clip_plane = %d\n", version, glpm->max_clip_plane);
        
        if (glpm->max_clip_plane > 0) {
            glpm->plane_enabled = xalloc(glpm->max_clip_plane, GLboolean);
            memset(glpm->plane_enabled, 0, glpm->max_clip_plane * sizeof(GLboolean));
            
            glpm->plane_equation = xalloc(glpm->max_clip_plane * 4, GLfloat);
            memset(glpm->plane_equation, 0, 4 * glpm->max_clip_plane * sizeof(GLfloat));
            
            if (glpm->use_shader && version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
                glpm->plane_converted_equation = xalloc(glpm->max_clip_plane * 4, GLfloat);
                memset(glpm->plane_converted_equation, 0, 4 * glpm->max_clip_plane * sizeof(GLfloat));
            } else {
                glpm->clip_plane_MV = xalloc(glpm->max_clip_plane, spGLMat4);
                for (i = 0; i < glpm->max_clip_plane; i++) {
                    glpm->clip_plane_MV[i] = spGLMat4CreateIdentity();
                }
            }
        }
    }

    glpm->P = spGLMat4CreateIdentity();
    glpm->MV = spGLMat4CreateIdentity();
    
    return glpm;
}

spBool spGLPhongModelFree(spGLPhongModel glpm)
{
    if (glpm == NULL) return SP_FALSE;

#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm->use_shader) {
        glDeleteProgram(glpm->shader_program);
    
        glDeleteShader(glpm->vertex_shader);
        glDeleteShader(glpm->fragment_shader);
    }
#endif

    xfree(glpm);
    
    return SP_TRUE;
}

spBool spGLPhongModelSetProjectionMatrix(spGLPhongModel glpm, spGLMat4 *P)
{
    if (glpm == NULL || P == NULL) return SP_FALSE;

    glpm->P = *P;

    return SP_TRUE;
}

spBool spGLPhongModelGetProjectionMatrix(spGLPhongModel glpm, spGLMat4 *P)
{
    if (glpm == NULL || P == NULL) return SP_FALSE;

    *P = glpm->P;
    
    return SP_TRUE;
}

spBool spGLPhongModelSetModelViewMatrix(spGLPhongModel glpm, spGLMat4 *MV)
{
    if (glpm == NULL || MV == NULL) return SP_FALSE;

    glpm->MV = *MV;

    return SP_TRUE;
}

spBool spGLPhongModelGetModelViewMatrix(spGLPhongModel glpm, spGLMat4 *MV)
{
    if (glpm == NULL || MV == NULL) return SP_FALSE;

    *MV = glpm->MV;
    
    return SP_TRUE;
}

spBool spGLPhongModelUpdateTransformationMatrix(spGLPhongModel glpm)
{
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glpm->N = spGLMat4CreateNormal(&glpm->MV, SP_FALSE);
        glpm->MVP = spGLMat4MultMatrix(&glpm->P, &glpm->MV, SP_FALSE);
        
        glUniformMatrix4fv(glpm->unif_MV_location, 1, GL_FALSE, (const GLfloat *)glpm->MV.data);
        glUniformMatrix4fv(glpm->unif_N_location, 1, GL_FALSE, (const GLfloat *)glpm->N.data);
        glUniformMatrix4fv(glpm->unif_MVP_location, 1, GL_FALSE, (const GLfloat *)glpm->MVP.data);
#else
        return SP_FALSE;
#endif
    } else {
        glMatrixMode(GL_PROJECTION);
        glLoadMatrixf((const GLfloat *)glpm->P.data);
        
        glMatrixMode(GL_MODELVIEW);
        glLoadMatrixf((const GLfloat *)glpm->MV.data);
    }

    return SP_TRUE;
}

/*
 * material[0][0-4]: ambient
 * material[1][0-4]: diffuse
 * material[2][0-4]: specular
 * material[3][0]: shininess
 */
spBool spGLPhongModelSetMaterialMatrix(spGLPhongModel glpm, spGLMat4 *material)
{
    if (glpm == NULL || material == NULL) return SP_FALSE;

    glpm->material = *material;

    return SP_TRUE;
}

spBool spGLPhongModelGetMaterialMatrix(spGLPhongModel glpm, spGLMat4 *material)
{
    if (glpm == NULL || material == NULL) return SP_FALSE;

    *material = glpm->material;

    return SP_TRUE;
}

spBool spGLPhongModelUpdateMaterialMatrix(spGLPhongModel glpm)
{
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glUniform4fv(glpm->unif_material_ambient_location, 1, glpm->material.data[0]);
        glUniform4fv(glpm->unif_material_diffuse_location, 1, glpm->material.data[1]);
        glUniform4fv(glpm->unif_material_specular_location, 1, glpm->material.data[2]);
        glUniform1f(glpm->unif_material_shininess_location, glpm->material.data[3][0]);
#else
        return SP_FALSE;
#endif
    } else {
        glMaterialfv(GL_FRONT, GL_AMBIENT, glpm->material.data[0]);
        glMaterialfv(GL_FRONT, GL_DIFFUSE, glpm->material.data[1]);
        glMaterialfv(GL_FRONT, GL_SPECULAR, glpm->material.data[2]);
        glMaterialfv(GL_FRONT, GL_SHININESS, glpm->material.data[3]);
    }
    
    return SP_TRUE;
}

/* shader version only */
spBool spGLPhongModelSetFrontMixColor(spGLPhongModel glpm, GLfloat rate, GLfloat *color)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL || rate < 0.0f || rate > 1.0f) return SP_FALSE;

    glpm->front_mix_rate = rate;
    memcpy(glpm->front_mix_color, color, 3 * sizeof(GLfloat));

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelGetFrontMixColor(spGLPhongModel glpm, GLfloat *rate, GLfloat *color)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (rate != NULL) *rate = glpm->front_mix_rate;
    if (color != NULL) memcpy(color, glpm->front_mix_color, 3 * sizeof(GLfloat));

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelSetBackMixColor(spGLPhongModel glpm, GLfloat rate, GLfloat *color)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL || rate < 0.0f || rate > 1.0f) return SP_FALSE;

    glpm->back_mix_rate = rate;
    memcpy(glpm->back_mix_color, color, 3 * sizeof(GLfloat));

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelGetBackMixColor(spGLPhongModel glpm, GLfloat *rate, GLfloat *color)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (rate != NULL) *rate = glpm->back_mix_rate;
    if (color != NULL) memcpy(color, glpm->back_mix_color, 3 * sizeof(GLfloat));

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelUpdateMixColor(spGLPhongModel glpm)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    if (glpm->use_shader) {
        glUniform1f(glpm->unif_front_mix_rate_location, glpm->front_mix_rate);
        glUniform3fv(glpm->unif_front_mix_color_location, 1, glpm->front_mix_color);
        
        glUniform1f(glpm->unif_back_mix_rate_location, glpm->back_mix_rate);
        glUniform3fv(glpm->unif_back_mix_color_location, 1, glpm->back_mix_color);
        
        return SP_TRUE;
    } else {
        return SP_FALSE;
    }
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelSetLightPosition(spGLPhongModel glpm, int index, GLfloat *position)
{
    if (glpm == NULL || position == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].position, position, 4 * sizeof(GLfloat));
    spGLMat4MultVector4fv(glpm->lights[index].converted_position, &glpm->MV,
                          glpm->lights[index].position, SP_FALSE);

    return SP_TRUE;
}

spBool spGLPhongModelGetLightPosition(spGLPhongModel glpm, int index, GLfloat *position)
{
    if (glpm == NULL || position == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(position, glpm->lights[index].position, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightAmbient(spGLPhongModel glpm, int index, GLfloat *ambient)
{
    if (glpm == NULL || ambient == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].ambient, ambient, 4 * sizeof(GLfloat));

    return SP_TRUE;
}

spBool spGLPhongModelGetLightAmbient(spGLPhongModel glpm, int index, GLfloat *ambient)
{
    if (glpm == NULL || ambient == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(ambient, glpm->lights[index].ambient, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightDiffuse(spGLPhongModel glpm, int index, GLfloat *diffuse)
{
    if (glpm == NULL || diffuse == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].diffuse, diffuse, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelGetLightDiffuse(spGLPhongModel glpm, int index, GLfloat *diffuse)
{
    if (glpm == NULL || diffuse == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(diffuse, glpm->lights[index].diffuse, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightSpecular(spGLPhongModel glpm, int index, GLfloat *specular)
{
    if (glpm == NULL || specular == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].specular, specular, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelGetLightSpecular(spGLPhongModel glpm, int index, GLfloat *specular)
{
    if (glpm == NULL || specular == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(specular, glpm->lights[index].specular, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightAttenuation(spGLPhongModel glpm, int index, GLfloat *attenuation)
{
    if (glpm == NULL || attenuation == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].attenuation, attenuation, 3 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelGetLightAttenuation(spGLPhongModel glpm, int index, GLfloat *attenuation)
{
    if (glpm == NULL || attenuation == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(attenuation, glpm->lights[index].attenuation, 3 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightSpotCutoff(spGLPhongModel glpm, int index, GLfloat cutoff)
{
    if (glpm == NULL || index >= glpm->light_count) return SP_FALSE;

    glpm->lights[index].spot_cutoff = cutoff;

    return SP_TRUE;
}

spBool spGLPhongModelGetLightSpotCutoff(spGLPhongModel glpm, int index, GLfloat *cutoff)
{
    if (glpm == NULL || cutoff == NULL || index >= glpm->light_count) return SP_FALSE;

    *cutoff = glpm->lights[index].spot_cutoff;
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightSpotExponent(spGLPhongModel glpm, int index, GLfloat exponent)
{
    if (glpm == NULL || index >= glpm->light_count) return SP_FALSE;

    glpm->lights[index].spot_exponent = exponent;

    return SP_TRUE;
}

spBool spGLPhongModelGetLightSpotExponent(spGLPhongModel glpm, int index, GLfloat *exponent)
{
    if (glpm == NULL || exponent == NULL || index >= glpm->light_count) return SP_FALSE;

    *exponent = glpm->lights[index].spot_exponent;
    
    return SP_TRUE;
}

spBool spGLPhongModelSetLightSpotDirection(spGLPhongModel glpm, int index, GLfloat *direction)
{
    spGLMat4 N;
    
    if (glpm == NULL || direction == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(glpm->lights[index].spot_direction, direction, 3 * sizeof(GLfloat));
    
    N = spGLMat4CreateNormal(&glpm->MV, SP_FALSE);
    spGLMat4MultVector3fv(glpm->lights[index].spot_converted_direction, &N,
                          glpm->lights[index].spot_direction, 1.0f, SP_FALSE);

    return SP_TRUE;
}

spBool spGLPhongModelGetLightSpotDirection(spGLPhongModel glpm, int index, GLfloat *direction)
{
    if (glpm == NULL || direction == NULL || index >= glpm->light_count) return SP_FALSE;

    memcpy(direction, glpm->lights[index].spot_direction, 3 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelUpdateLight(spGLPhongModel glpm, int index)
{
    if (glpm == NULL || glpm->begin_flag == SP_FALSE || index >= glpm->light_count) return SP_FALSE;
    
    spDebug(100, "spGLPhongModelUpdateLight", "index = %d\n", index);
    
    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glUniform4fv(glpm->lights[index].position_location, 1, glpm->lights[index].converted_position);
        glUniform4fv(glpm->lights[index].ambient_location, 1, glpm->lights[index].ambient);
        glUniform4fv(glpm->lights[index].diffuse_location, 1, glpm->lights[index].diffuse);
        glUniform4fv(glpm->lights[index].specular_location, 1, glpm->lights[index].specular);

        glUniform3fv(glpm->lights[index].attenuation_location, 1, glpm->lights[index].attenuation);
        
        glUniform1f(glpm->lights[index].spot_cutoff_location, glpm->lights[index].spot_cutoff);
        if (glpm->lights[index].spot_cutoff != 180.0) {
            glUniform1f(glpm->lights[index].spot_exponent_location, glpm->lights[index].spot_exponent);
            glUniform3fv(glpm->lights[index].spot_direction_location, 1, glpm->lights[index].spot_converted_direction);
        }
#else
        return SP_FALSE;
#endif
    } else {
        GLfloat position[4];
        spGLMat4 MVinv;

        /* The inverse of MV is applied to converted_position since MV is multiplied by glLightfv */
        MVinv = spGLMat4CreateInverse(&glpm->MV, SP_FALSE);
        spGLMat4MultVector4fv(position, &MVinv, glpm->lights[index].converted_position, SP_FALSE);
        glLightfv(GL_LIGHT0 + index, GL_POSITION, position);
        glLightfv(GL_LIGHT0 + index, GL_AMBIENT, glpm->lights[index].ambient);
        glLightfv(GL_LIGHT0 + index, GL_DIFFUSE, glpm->lights[index].diffuse);
        glLightfv(GL_LIGHT0 + index, GL_SPECULAR, glpm->lights[index].specular);
        
        glLightf(GL_LIGHT0 + index, GL_CONSTANT_ATTENUATION, glpm->lights[index].attenuation[0]);
        glLightf(GL_LIGHT0 + index, GL_LINEAR_ATTENUATION, glpm->lights[index].attenuation[1]);
        glLightf(GL_LIGHT0 + index, GL_QUADRATIC_ATTENUATION, glpm->lights[index].attenuation[2]);

        glLightf(GL_LIGHT0 + index, GL_SPOT_CUTOFF, glpm->lights[index].spot_cutoff);
        
        if (glpm->lights[index].spot_cutoff != 180.0) {
            spGLMat4 N, Ninv;
            
            glLightf(GL_LIGHT0 + index, GL_SPOT_EXPONENT, glpm->lights[index].spot_exponent);
            
            N = spGLMat4CreateNormal(&glpm->MV, SP_FALSE);
            Ninv = spGLMat4CreateInverse(&N, SP_FALSE);
            spGLMat4MultVector3fv(position, &Ninv, glpm->lights[index].spot_converted_direction,
                                  1.0, SP_FALSE);
            glLightfv(GL_LIGHT0 + index, GL_SPOT_DIRECTION, position);
        }
    }

    return SP_TRUE;
}

#if 1
spBool spGLPhongModelSetClipPlane(spGLPhongModel glpm, int index, GLfloat *equation)
{
    unsigned long version;
    
    if (glpm == NULL || !(glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE)
        || index < 0 || index >= glpm->max_clip_plane) return SP_FALSE;
    
    glpm->plane_enabled[index] = SP_TRUE;
    memcpy(&glpm->plane_equation[4 * index], equation, 4 * sizeof(GLfloat));
    
    version = glpm->options & SP_GL_PHONG_MODEL_SHADER_VERSION_MASK;
    
    if (glpm->use_shader && version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
#if defined(SP_GL_SHADER_SUPPORTED)
        double den, w;
        GLfloat vec[3];
        GLfloat vec2[3];
        GLfloat normal[3];
        GLfloat d;
        spGLMat4 N;

        den = (double)(equation[0] * equation[0] + equation[1] * equation[1] + equation[2] * equation[2]);
        spDebug(100, "spGLPhongModelSetClipPlane", "den = %f\n", den);
        
        if (den == 0.0) {
            memcpy(&glpm->plane_converted_equation[4 * index], equation, 4 * sizeof(GLfloat));
        } else {
            N = spGLMat4CreateNormal(&glpm->MV, SP_FALSE);
            spGLMat4MultVector3fv(normal, &N, equation, 1.0, SP_FALSE);
            
            w = -(double)equation[3] / den;
            vec[0] = (GLfloat)((double)equation[0] * w);
            vec[1] = (GLfloat)((double)equation[1] * w);
            vec[2] = (GLfloat)((double)equation[2] * w);
            spGLMat4MultVector3fv(vec2, &glpm->MV, vec, 1.0, SP_FALSE);

            d = -(normal[0] * vec2[0] + normal[1] * vec2[1] + normal[2] * vec2[2]);
            spDebug(100, "spGLPhongModelSetClipPlane", "new equation = (%f, %f, %f, %f)\n",
                    vec2[0], vec2[1], vec2[2], d);
            
            glpm->plane_converted_equation[4 * index + 0] = normal[0];
            glpm->plane_converted_equation[4 * index + 1] = normal[1];
            glpm->plane_converted_equation[4 * index + 2] = normal[2];
            glpm->plane_converted_equation[4 * index + 3] = d;
        }
#else
        return SP_FALSE;
#endif
    } else {
        spGLMat4Copy(&glpm->clip_plane_MV[index], &glpm->MV, SP_FALSE);
    }
    
    return SP_TRUE;
}

spBool spGLPhongModelUnsetClipPlane(spGLPhongModel glpm, int index)
{
    if (glpm == NULL || !(glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE)
        || index < 0 || index >= glpm->max_clip_plane) return SP_FALSE;
    
    glpm->plane_enabled[index] = SP_FALSE;
    memset(&glpm->plane_equation[4 * index], 0, 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelGetClipPlane(spGLPhongModel glpm, int index, GLfloat *equation)
{
    if (glpm == NULL || !(glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE)
        || index < 0 || index >= glpm->max_clip_plane
        || glpm->plane_enabled[index] == SP_FALSE) return SP_FALSE;
    
    memcpy(equation, &glpm->plane_equation[4 * index], 4 * sizeof(GLfloat));
    
    return SP_TRUE;
}

spBool spGLPhongModelUpdateClipPlane(spGLPhongModel glpm)
{
    int i;
    unsigned long version;
    
    if (glpm == NULL || !(glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE)
        || glpm->max_clip_plane <= 0 || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    version = glpm->options & SP_GL_PHONG_MODEL_SHADER_VERSION_MASK;
        
    if (glpm->use_shader && version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glUniform4fv(glpm->unif_clip_plane_location, glpm->max_clip_plane, glpm->plane_converted_equation);
        for (i = 0; i < glpm->max_clip_plane; i++) {
            if (glpm->plane_enabled[i]) {
                glEnable(0x3000/*GL_CLIP_DISTANCE0*/ + i);
            }
        }
#else
        return SP_FALSE;
#endif
    } else {
#if !defined(SP_GL_ES)
        GLdouble equation[4];
        int j;
#endif
        for (i = 0; i < glpm->max_clip_plane; i++) {
            if (glpm->plane_enabled[i]) {
                glPushMatrix();
                glMatrixMode(GL_MODELVIEW);
                glLoadMatrixf((const GLfloat *)glpm->clip_plane_MV[i].data);
#if !defined(SP_GL_ES)
                for (j = 0; j < 4; j++) {
                    equation[j] = (GLdouble)glpm->plane_equation[4 * i + j];
                }
                glClipPlane(GL_CLIP_PLANE0 + i, equation);
#else
                glClipPlanef(GL_CLIP_PLANE0 + i, &glpm->plane_equation[4 * i]);
#endif
                glEnable(GL_CLIP_PLANE0 + i);
                glPopMatrix();
            }
        }
    }
    
    return SP_TRUE;
}

#endif

spBool spGLPhongModelBegin(spGLPhongModel glpm, spGLPhongModelDrawMode draw_mode)
{
    int i;
    
    if (glpm == NULL) return SP_FALSE;
    
    glpm->begin_flag = SP_TRUE;
    glpm->draw_mode = draw_mode;
    
    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glGetIntegerv(GL_CURRENT_PROGRAM, (GLint *)&glpm->old_program);
        spDebug(100, "spGLPhongModelBegin", "old_program = %d, shader_program = %d\n",
                glpm->old_program, glpm->shader_program);
        glUseProgram(glpm->shader_program);
#else
        return SP_FALSE;
#endif
    } else {
        glPushMatrix();
        
        glpm->last_lighting = glIsEnabled(GL_LIGHTING);
        spDebug(100, "spGLPhongModelBegin", "last_lighting = %d\n", glpm->last_lighting);
        glEnable(GL_LIGHTING);
        
        for (i = 0; i < glpm->light_count; i++) {
            glpm->lights[i].last_enabled = glIsEnabled(GL_LIGHT0 + i);
            spDebug(100, "spGLPhongModelBegin", "light %d: last_enabled = %d\n",
                    i, glpm->lights[i].last_enabled);
            glEnable(GL_LIGHT0 + i);
        }
    }

    spGLPhongModelUpdateTransformationMatrix(glpm);

    for (i = 0; i < glpm->light_count; i++) {
        spGLPhongModelUpdateLight(glpm, i);
    }

    spGLPhongModelUpdateMaterialMatrix(glpm);
    spGLPhongModelUpdateMixColor(glpm);

    spGLPhongModelUpdateClipPlane(glpm);

#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm->use_shader) {
        if (glpm->draw_mode != SP_GL_PHONG_MODEL_DRAW_MODE_USE_VAO) {
            glEnableVertexAttribArray(glpm->attr_position_location);
            glEnableVertexAttribArray(glpm->attr_normal_location);
        }
    } else
#endif
    {
        if (glpm->draw_mode != SP_GL_PHONG_MODEL_DRAW_MODE_USE_VAO) {
            glEnableClientState(GL_VERTEX_ARRAY);
            glEnableClientState(GL_NORMAL_ARRAY);
        }
    }

    return SP_TRUE;
}

spBool spGLPhongModelEnd(spGLPhongModel glpm)
{
    int i;
    unsigned long version;
    
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    version = glpm->options & SP_GL_PHONG_MODEL_SHADER_VERSION_MASK;
    
    if (glpm->options & SP_GL_PHONG_MODEL_OPTION_USE_CLIP_PLANE) {
        for (i = 0; i < glpm->max_clip_plane; i++) {
            if (glpm->plane_enabled[i]) {
                if (glpm->use_shader && version >= SP_GL_CLIP_PLANE_SUPPORTED_SHADER_VERSION) {
                    glDisable(0x3000/*GL_CLIP_DISTANCE0*/ + i);
                } else {
                    glDisable(GL_CLIP_PLANE0 + i);
                }
            }
        }
    }
    
    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        if (glpm->draw_mode != SP_GL_PHONG_MODEL_DRAW_MODE_USE_VAO) {
            glDisableVertexAttribArray(glpm->attr_position_location);
            glDisableVertexAttribArray(glpm->attr_normal_location);
        }
        
        glUseProgram(glpm->old_program);
        glpm->old_program = 0;
#else
        return SP_FALSE;
#endif
    } else {
        if (glpm->draw_mode != SP_GL_PHONG_MODEL_DRAW_MODE_USE_VAO) {
            glDisableClientState(GL_VERTEX_ARRAY);
            glDisableClientState(GL_NORMAL_ARRAY);
        }

        for (i = 0; i < glpm->light_count; i++) {
            if (glpm->lights[i].last_enabled == GL_FALSE) {
                glDisable(GL_LIGHT0 + i);
            }
        }

        if (glpm->last_lighting == GL_FALSE) {
            glDisable(GL_LIGHTING);
        }
        
        glPopMatrix();
    }
    
    glpm->begin_flag = SP_FALSE;
    
    return SP_TRUE;
}

spBool spGLPhongModelVertexPointer(spGLPhongModel glpm, GLenum type, GLsizei stride, const GLfloat *pointer)
{
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glVertexAttribPointer(glpm->attr_position_location, 3, type, GL_FALSE, stride, pointer);
#else
        return SP_FALSE;
#endif
    } else {
        glVertexPointer(3, type, stride, pointer);
    }
    
    /*spDebug(100, "spGLPhongModelVertexPointer", "error = %d\n", (int)glGetError());*/
    
    return SP_TRUE;
}

spBool spGLPhongModelNormalPointer(spGLPhongModel glpm, GLenum type, GLsizei stride, const GLfloat *pointer)
{
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
#if defined(SP_GL_SHADER_SUPPORTED)
        glVertexAttribPointer(glpm->attr_normal_location, 3, type, GL_FALSE, stride, pointer);
#else
        return SP_FALSE;
#endif
    } else {
        glNormalPointer(type, stride, pointer);
    }
    
    /*spDebug(100, "spGLPhongModelNormalPointer", "error = %d\n", (int)glGetError());*/
    
    return SP_TRUE;
}

spBool spGLPhongModelDrawArrays(spGLPhongModel glpm, GLenum mode, GLint first, GLsizei count)
{
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    spDebug(100, "spGLPhongModelDrawArrays", "call glDrawArrays\n");
    glDrawArrays(mode, first, count);
    
    /*spDebug(100, "spGLPhongModelDrawArrays", "error = %d\n", (int)glGetError());*/
    
    return SP_TRUE;
}

spBool spGLPhongModelDrawElements(spGLPhongModel glpm, GLenum mode, GLsizei count, GLenum type, const GLvoid *indices)
{
    if (glpm == NULL || glpm->begin_flag == SP_FALSE) return SP_FALSE;

    spDebug(100, "spGLPhongModelDrawElements", "call glDrawElements\n");
    glDrawElements(mode, count, type, indices);
    
    /*spDebug(100, "spGLPhongModelDrawElements", "error = %d\n", (int)glGetError());*/
    
    return SP_TRUE;
}

spBool spGLPhongModelGenBuffers(spGLPhongModel glpm, GLsizei n, GLuint *buffers)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glGenBuffers(n, buffers);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelDeleteBuffers(spGLPhongModel glpm, GLsizei n, GLuint *buffers)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glDeleteBuffers(n, buffers);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

static spBool spGLPhongModelBufferData(spGLPhongModel glpm, GLenum target, /*GLsizeiptr*/size_t size,
                                       const GLvoid * data, GLenum usage)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glBufferData(target, size, data, usage);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

static spBool spGLPhongModelBufferSubData(spGLPhongModel glpm, GLenum target, /*GLintptr*/size_t offset,
                                          /*GLsizeiptr*/size_t size, const GLvoid * data)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glBufferSubData(target, offset, size, data);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelBindVertexOrNormalBuffer(spGLPhongModel glpm, GLuint buffer)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glBindBuffer(SP_GL_ARRAY_BUFFER, buffer);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelVertexOrNormalBufferData(spGLPhongModel glpm, /*GLsizeiptr*/size_t size,
                                              const GLvoid * data, GLenum usage)
{
    return spGLPhongModelBufferData(glpm, SP_GL_ARRAY_BUFFER, size, data, usage);
}

spBool spGLPhongModelVertexOrNormalBufferSubData(spGLPhongModel glpm, /*GLintptr*/size_t offset,
                                                 /*GLsizeiptr*/size_t size, const GLvoid * data)
{
    return spGLPhongModelBufferSubData(glpm, SP_GL_ARRAY_BUFFER, offset, size, data);
}

spBool spGLPhongModelBindIndexBuffer(spGLPhongModel glpm, GLuint buffer)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    glBindBuffer(SP_GL_ELEMENT_ARRAY_BUFFER, buffer);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelIndexBufferData(spGLPhongModel glpm, /*GLsizeiptr*/size_t size,
                                     const GLvoid * data, GLenum usage)
{
    return spGLPhongModelBufferData(glpm, SP_GL_ELEMENT_ARRAY_BUFFER, size, data, usage);
}

spBool spGLPhongModelIndexBufferSubData(spGLPhongModel glpm, /*GLintptr*/size_t offset,
                                        /*GLsizeiptr*/size_t size, const GLvoid * data)
{
    return spGLPhongModelBufferSubData(glpm, SP_GL_ELEMENT_ARRAY_BUFFER, offset, size, data);
}

spBool spGLPhongModelGenVertexArrays(spGLPhongModel glpm, GLsizei n, GLuint *arrays)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spIsWGLFunctionLoaded(glGenVertexArrays) == SP_FALSE) {
        return SP_FALSE;
    }
#endif
    
    glGenVertexArrays(n, arrays);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelDeleteVertexArrays(spGLPhongModel glpm, GLsizei n, GLuint *arrays)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spIsWGLFunctionLoaded(glDeleteVertexArrays) == SP_FALSE) {
        return SP_FALSE;
    }
#endif
    
    glDeleteVertexArrays(n, arrays);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelBindVertexArray(spGLPhongModel glpm, GLuint array)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spIsWGLFunctionLoaded(glBindVertexArray) == SP_FALSE) {
        return SP_FALSE;
    }
#endif
    
    glBindVertexArray(array);

    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelEnableVertexArray(spGLPhongModel glpm)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
        glEnableVertexAttribArray(glpm->attr_position_location);
    } else {
        glEnableClientState(GL_VERTEX_ARRAY);
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelDisableVertexArray(spGLPhongModel glpm)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
        glDisableVertexAttribArray(glpm->attr_position_location);
    } else {
        glDisableClientState(GL_VERTEX_ARRAY);
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelEnableNormalArray(spGLPhongModel glpm)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
        glEnableVertexAttribArray(glpm->attr_normal_location);
    } else {
        glEnableClientState(GL_NORMAL_ARRAY);
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}

spBool spGLPhongModelDisableNormalArray(spGLPhongModel glpm)
{
#if defined(SP_GL_SHADER_SUPPORTED)
    if (glpm == NULL) return SP_FALSE;

    if (glpm->use_shader) {
        glDisableVertexAttribArray(glpm->attr_normal_location);
    } else {
        glDisableClientState(GL_NORMAL_ARRAY);
    }
    
    return SP_TRUE;
#else
    return SP_FALSE;
#endif
}
