#include <sp/spComponent.h>
#include <sp/spContainer.h>

#if defined(ANDROID) || defined(IPHONE)
#define USE_OPENGL_ES2 1
#elif defined(MACOSX)
#include <AvailabilityMacros.h>
#if defined(MAC_OS_X_VERSION_10_15) && MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_15
#pragma message "USE_OPENGL3"
#define USE_OPENGL3 1
#endif
#endif

#include <sp/spGLP.h>
#include <sp/spGLStringP.h>

static spParamTable sp_glstring_param_tables[] = {
    {SppFontName, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.font_name), NULL},
    {SppForeground, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.foreground), NULL},
    {SppForegroundPixel, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.fg_pixel), NULL},
    {SppStringAlignment, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.string_alignment), NULL},
    {SppUseAntiAliasFont, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.use_anti_alias_font), NULL},
    {SppGLStringUseTexture, SP_CREATE_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.use_texture), NULL},
    {SppGLStringUseShader, SP_CREATE_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.use_shader), NULL},
    {SppGLStringProjectionMatrix, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.projection_matrix), NULL},
    {SppGLStringModelViewMatrix, SP_CREATE_ACCESS | SP_SET_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.modelview_matrix), NULL},
    {SppGLStringOptions, SP_CREATE_ACCESS | SP_GET_ACCESS,
	 spOffset(spGLString, glstring.options), NULL},
};

spGLStringClassRec SpGLStringClassRec = {
    /* spObjectClassPart */
    {
	SpGLString,
	NULL,
	sizeof(spGLStringRec),
	spArraySize(sp_glstring_param_tables),
	sp_glstring_param_tables,
	spGLStringPartInit,
	spGLStringPartFree,
	SP_FALSE,
	NULL,
	NULL,
	spGLStringCreate,
	spGLStringDestroy,
	NULL,
	NULL,
    },
    /* spGLStringClassPart */
    {
	0,
    },
};

spObjectClass SpGLStringClass = (spObjectClass)&SpGLStringClassRec;

void spGLStringPartInit(spObject object)
{
    spGLString glstring = (spGLString)object;
    
    SpGLStringPart(glstring).font_name = NULL;
    SpGLStringPart(glstring).foreground = NULL;
    SpGLStringPart(glstring).fg_pixel = spGetColorPixel("white");
    
    SpGLStringPart(glstring).string_alignment = SP_ALIGNMENT_BEGINNING;
    SpGLStringPart(glstring).use_anti_alias_font = SP_TRUE;

    SpGLStringPart(glstring).options = 0L;
    SpGLStringPart(glstring).use_texture = SP_FALSE;
    SpGLStringPart(glstring).use_shader = SP_FALSE;
    SpGLStringPart(glstring).projection_matrix = NULL;
    SpGLStringPart(glstring).modelview_matrix = NULL;
    
    SpGLStringPart(glstring).x = 0;
    SpGLStringPart(glstring).y = 0;
    SpGLStringPart(glstring).width = 0;
    SpGLStringPart(glstring).height = 0;
    SpGLStringPart(glstring).next_x = 0;

    SpGLStringPart(glstring).pos_x = 0.0f;
    SpGLStringPart(glstring).pos_y = 0.0f;
    SpGLStringPart(glstring).pos_z = 0.0f;
    
#if defined(SP_GL_STRING_LIST_BASE)
    SpGLStringPart(glstring).count = 0;
    SpGLStringPart(glstring).list_base = 0;
#endif
    
#if defined(SP_GL_STRING_BY_TEXTURE)
    SpGLStringArch(glstring).inside_margin_width = 0;
    SpGLStringArch(glstring).inside_margin_height = 0;
    SpGLStringArch(glstring).dibitmap = NULL;
    SpGLStringArch(glstring).region = NULL;
#endif

    spGLStringPartInitArch(glstring);
    
    return;
}

void spGLStringPartFree(spObject object)
{
    spGLString glstring = (spGLString)object;

    spGLStringPartFreeArch(glstring);

    if (SpGLStringPart(glstring).font_name != NULL)
	xfree(SpGLStringPart(glstring).font_name);
    
    if (SpGLStringPart(glstring).foreground != NULL)
	xfree(SpGLStringPart(glstring).foreground);
    
    return;
}

spBool spGLStringCreate(spObject object)
{
    spGLString glstring = (spGLString)object;

    spDebug(50, "spGLStringCreate", "initial options = %lx\n", SpGLStringPart(glstring).options);
    
#if defined(SP_GL_ES)
    SpGLStringPart(glstring).options |= SP_GL_STRING_SHADER_VERSION_ES;
#endif

    if ((SpGLStringPart(glstring).options & SP_GL_STRING_OPTION_USE_SHADER)
        || (SpGLStringPart(glstring).options & SP_GL_STRING_SHADER_VERSION_MASK) > 0) {
        SpGLStringPart(glstring).use_shader = SP_TRUE;
    }
    if (SpGLStringPart(glstring).options & SP_GL_STRING_OPTION_USE_TEXTURE) {
        SpGLStringPart(glstring).use_texture = SP_TRUE;
    }
    
    spDebug(50, "spGLStringCreate", "use_shader = %d\n", SpGLStringPart(glstring).use_shader);
    
    if (SpGLStringPart(glstring).use_shader) {
#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
	spDebug(50, "spGLStringCreate", "call spInitGLStringShader\n");
	if (spInitGLStringShader(SpGLStringPart(glstring).options) == SP_FALSE) {
	    spDebug(10, "spGLStringCreate", "spInitGLStringShader failed\n");
	    return SP_FALSE;
	}
#else
	spDebug(50, "spGLStringCreate", "shader not supported\n");
	return SP_FALSE;
#endif
    }
    
    if (!strnone(SpGLStringPart(glstring).foreground)) {
	SpGLStringPart(glstring).fg_pixel = spGetColorPixel(SpGLStringPart(glstring).foreground);
    }

#if defined(SP_GL_STRING_LIST_BASE)
    if (SpGLStringPart(glstring).use_texture == SP_FALSE && SpGLStringPart(glstring).use_shader == SP_FALSE) {
	SpGLStringPart(glstring).count = 0;
	SpGLStringPart(glstring).list_base = glGenLists(strlen(SpGetName(glstring)));
	spDebug(50, "spGLStringCreate", "list_base = %d\n", SpGLStringPart(glstring).list_base);
    }
#endif

    return spGLStringCreateArch(glstring, SpGetName(glstring));
}

spBool spGLStringDestroy(spObject object)
{
    spBool flag;
    spGLString glstring = (spGLString)object;
    
    flag = spGLStringDestroyArch(glstring);
    
#if defined(SP_GL_STRING_BY_TEXTURE)
    spDestroyGLStringByTexture(glstring);
#endif
    
#if defined(SP_GL_STRING_LIST_BASE)
    glDeleteLists(SpGLStringPart(glstring).list_base, SpGLStringPart(glstring).count);
#endif
    
    return flag;
}

spGLString spCreateGLStringArg(spComponent gldrawable, char *string, spArg *args, int num_arg)
{
    spGLString glstring;
    
    if (spIsGLDrawable(gldrawable) == SP_FALSE || strnone(string)) return NULL;

    if ((glstring = (spGLString)spAllocObject((spObjectClass)SpGLStringClass, NULL, string)) == NULL) {
	return NULL;
    }

    SpGLStringPart(glstring).gldrawable = gldrawable;
    spDebug(60, "spCreateGLStringArg", "spAllocObject done\n");

    if (spSetObjectParamsArg((spObject)glstring, args, num_arg) == SP_FALSE) {
	spFreeObject((spObject)glstring);
	glstring = NULL;
    }
    
    return glstring;
}

spGLString spCreateGLString(spComponent gldrawable, char *string, ...)
{
    int num_arg = 0;
    spArg args[SP_MAX_NUM_ARG];
    va_list argp;

    va_start(argp, string);
    spGetArgs(argp, args, num_arg);
    va_end(argp);
    spDebug(60, "spCreateGLString", "num_arg = %d\n", num_arg);

    return spCreateGLStringArg(gldrawable, string, args, num_arg);
}

spBool spDestroyGLString(spGLString glstring)
{
    if (glstring == NULL) return SP_FALSE;
    
    return spDestroyObject((spObject)glstring);
}

void spSetGLStringParams(spGLString glstring, ...)
{
    int num_arg = 0;
    spArg args[SP_MAX_NUM_ARG];
    va_list argp;
    
    if (glstring == NULL) return;
    
    va_start(argp, glstring);
    spGetArgs(argp, args, num_arg);
    va_end(argp);

    spSetObjectParamsArg((spObject)glstring, args, num_arg);

    return;
}
    
spBool spSetGLStringPos3f(spGLString glstring, GLfloat x, GLfloat y, GLfloat z)
{
    if (glstring == NULL) return SP_FALSE;

    SpGLStringPart(glstring).pos_x = x;
    SpGLStringPart(glstring).pos_y = y;
    SpGLStringPart(glstring).pos_z = z;
    
#if defined(SP_GL_STRING_LIST_BASE) || defined(SP_GL_STRING_BY_TEXTURE)
    return SP_TRUE;
#else
    return spSetGLStringPos3fArch(glstring, x, y, z);
#endif
}

spBool spGetGLStringExtent(spComponent gldrawable, spGLString glstring, int *x, int *y, int *width, int *height, int *next_x)
{
    if (glstring == NULL || spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;
    
    if (spGetGLStringExtentArch(gldrawable, glstring) == SP_TRUE) {
	if (x != NULL) *x = SpGLStringPart(glstring).x;
	if (y != NULL) *y = SpGLStringPart(glstring).y;
	if (width != NULL) *width = SpGLStringPart(glstring).width;
	if (height != NULL) *height = SpGLStringPart(glstring).height;
	if (next_x != NULL) *next_x = SpGLStringPart(glstring).next_x;
	
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

#if defined(SP_GL_STRING_BY_TEXTURE)
#include <sp/spGLShaderP.h>

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
SP_WGL_DECLARE_EXTERN_FUNCTION(glActiveTexture)
SP_WGL_DECLARE_EXTERN_FUNCTION(glClientActiveTexture)

SP_WGL_DECLARE_EXTERN_FUNCTION(glEnableVertexAttribArray)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDisableVertexAttribArray)
SP_WGL_DECLARE_EXTERN_FUNCTION(glVertexAttribPointer)

SP_WGL_DECLARE_EXTERN_FUNCTION(glIsBuffer)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGenBuffers)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteBuffers)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBindBuffer)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBufferData)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBufferSubData)

SP_WGL_DECLARE_EXTERN_FUNCTION(glIsVertexArray)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGenVertexArrays)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteVertexArrays)
SP_WGL_DECLARE_EXTERN_FUNCTION(glBindVertexArray)
#elif defined(ANDROID)
SP_EGL_DECLARE_EXTERN_FUNCTION(glIsVertexArrayOES)
SP_EGL_DECLARE_EXTERN_FUNCTION(glGenVertexArraysOES)
SP_EGL_DECLARE_EXTERN_FUNCTION(glDeleteVertexArraysOES)
SP_EGL_DECLARE_EXTERN_FUNCTION(glBindVertexArrayOES)
#endif

#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
static GLuint sp_glstring_shader_program = 0;
static GLuint sp_glstring_vertex_shader = 0;
static GLuint sp_glstring_fragment_shader = 0;

static GLint sp_glstring_attr_position_location = 0;
static GLint sp_glstring_attr_uv_location = 0;

static GLint sp_glstring_unif_MVP_location = 0;
static GLint sp_glstring_unif_texture_location = 0;
#endif

static spGLTextureInfo sp_first_gl_string_texture_info = NULL;
static spGLTextureInfo sp_last_gl_string_texture_info = NULL;

void spGetGLTextureRegionCoord(spGLTextureRegion region,
			       GLfloat *left, GLfloat *top, GLfloat *right, GLfloat *bottom)
{
    *left = (GLfloat)(region->x + region->line->info->inside_margin_width) / (GLfloat)region->line->info->texture_width;
    *right = (GLfloat)(region->x + region->width - region->line->info->inside_margin_width) / (GLfloat)region->line->info->texture_width;
    *top = (GLfloat)(region->y + region->height - region->line->info->inside_margin_height) / (GLfloat)region->line->info->texture_height;
    *bottom = (GLfloat)(region->y + region->line->info->inside_margin_height) / (GLfloat)region->line->info->texture_height;
    
    spDebug(80, "spGetGLTextureRegionCoord", "left = %f, top = %f, right = %f, bottom = %f\n",
	    *left, *top, *right, *bottom);

    return;
}

spGLTextureRegion spAllocGLTextureRegion(spGLTextureLine line, spGLTextureRegion next, int width, int height)
{
    spGLTextureRegion region;

    spDebug(80, "spAllocGLTextureRegion", "in: next = %ld\n", (long)next);
    
    region = xalloc(1, struct _spGLTextureRegion);
    memset(region, 0, sizeof(struct _spGLTextureRegion));

    region->line = line;
    region->y = line->ypos;
    region->width = width;
    region->height = height;

    region->alpha_sfactor = GL_SRC_ALPHA;
    region->alpha_dfactor = GL_ONE_MINUS_SRC_ALPHA;
    
    if (next != NULL) {
	if (next->prev != NULL) {
	    region->x = next->prev->x + next->prev->width;
	    region->prev = next->prev;
	    next->prev->next = region;
	} else {
	    region->x = 0;
	    line->first_region = region;
	}
	next->prev = region;
	region->next = next;
    } else {
	if (line->first_region == NULL) {
	    line->first_region = region;
	    region->x = 0;
	} else {
	    region->x = line->last_region->x + line->last_region->width;
	    line->last_region->next = region;
	}
	region->prev = line->last_region;
	line->last_region = region;
    }
    line->height = MAX(line->height, height);

    ++region->line->info->num_region;
    spDebug(80, "spAllocGLTextureRegion", "num_region = %d\n", region->line->info->num_region);
    
#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
    if (region->line->info->use_vao == SP_TRUE) {
        GLfloat uvBuffer[8];
        
        spGetGLTextureRegionCoord(region, &uvBuffer[2], &uvBuffer[3], &uvBuffer[4], &uvBuffer[5]);
        uvBuffer[0] = uvBuffer[2];
        uvBuffer[1] = uvBuffer[5];
        uvBuffer[6] = uvBuffer[4];
        uvBuffer[7] = uvBuffer[3];
        
        glGenBuffers(1, &region->uv_vbo);
        glBindBuffer(GL_ARRAY_BUFFER, region->uv_vbo);
        glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 8, uvBuffer, GL_STATIC_DRAW);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
            
        glGenVertexArrays(1, &region->vao);
        glBindVertexArray(region->vao);
        
        glEnableVertexAttribArray(sp_glstring_attr_position_location);
        glEnableVertexAttribArray(sp_glstring_attr_uv_location);
        
        glBindBuffer(GL_ARRAY_BUFFER, region->line->info->position_vbo);
        glVertexAttribPointer(sp_glstring_attr_position_location, 3, GL_FLOAT, GL_FALSE, 0, NULL);
        /*spDebug(80, "spAllocGLTextureRegion", "glVertexAttribPointer for position error = %d\n", (int)glGetError());*/

        glBindBuffer(GL_ARRAY_BUFFER, region->uv_vbo);
        glVertexAttribPointer(sp_glstring_attr_uv_location, 2, GL_FLOAT, GL_FALSE, 0, NULL);
        /*spDebug(80, "spAllocGLTextureRegion", "glVertexAttribPointer for uv error = %d\n", (int)glGetError());*/
        
        glBindVertexArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
    }
#endif
    
    return region;
}

void spFreeGLTextureRegion(spGLTextureRegion region)
{
    spDebug(80, "spFreeGLTextureRegion", "in\n");
    
#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
    if (region->vao != 0) {
        glBindVertexArray(region->vao);
        
        glDisableVertexAttribArray(sp_glstring_attr_position_location);
        glDisableVertexAttribArray(sp_glstring_attr_uv_location);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        
        glBindVertexArray(0);
        
        glDeleteVertexArrays(1, &region->vao);
    }
    if (region->uv_vbo != 0) {
        glDeleteBuffers(1, &region->uv_vbo);
    }
#endif
    
    --region->line->info->num_region;
    
    if (region->prev != NULL) {
	region->prev->next = region->next;
    }
    if (region->next != NULL) {
	region->next->prev = region->prev;
    }
    if (region == region->line->first_region) {
	region->line->first_region = region->next;
    }
    if (region == region->line->last_region) {
	region->line->last_region = region->prev;
    }

    xfree(region);
    
    spDebug(80, "spFreeGLTextureRegion", "done\n");
    
    return;
}

spGLTextureLine spAllocGLTextureLine(spGLTextureInfo info, int height)
{
    spGLTextureLine line;

    spDebug(80, "spAllocGLTextureLine", "in\n");
    
    line = xalloc(1, struct _spGLTextureLine);
    memset(line, 0, sizeof(struct _spGLTextureLine));

    line->info = info;
    line->height = height;
    
    if (info->first_line == NULL) {
	info->first_line = line;
	line->ypos = 0;
    } else {
	info->last_line->next = line;
	line->ypos = info->last_line->ypos + info->last_line->height;
    }
    line->prev = info->last_line;
    info->last_line = line;

    spDebug(80, "spAllocGLTextureLine", "done\n");
    
    return line;
}

GLenum spCreateNewGLTexture(GLenum texture_unit, int texture_width, int texture_height, spBool use_shader, GLuint *texture_name)
{
    GLboolean gl_texture_2d = GL_FALSE;
    GLenum err;
    GLenum format = SP_GL_STRING_BY_TEXTURE_BITMAP_PIXEL_FORMAT;
    GLenum type = SP_GL_STRING_BY_TEXTURE_BITMAP_DATA_TYPE;

#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spIsWGLFunctionLoaded(glActiveTexture) == SP_FALSE) {
        SP_WGL_LOAD_EXT_FUNCTION(glActiveTexture);
        SP_WGL_LOAD_EXT_FUNCTION(glClientActiveTexture);

        spDebug(80, "spCreateNewGLTexture", "sp_glActiveTexture = %ld, sp_glClientActiveTexture = %ld\n",
                (long)sp_glActiveTexture, (long)sp_glClientActiveTexture);

        if (spIsWGLFunctionLoaded(glActiveTexture) == SP_FALSE || spIsWGLFunctionLoaded(glClientActiveTexture) == SP_FALSE) {
            spDebug(80, "spCreateNewGLTexture", "load ext functions failed\n");
            return GL_INVALID_OPERATION;
        }
    }
#endif
    
    glActiveTexture(texture_unit);

    if (use_shader == SP_FALSE) {
	gl_texture_2d = glIsEnabled(GL_TEXTURE_2D);
	spDebug(80, "spCreateNewGLTexture", "gl_texture_2d = %d\n", gl_texture_2d);
	if (gl_texture_2d == GL_FALSE) glEnable(GL_TEXTURE_2D);
    }
    
    glGenTextures(1, texture_name);
    
    spDebug(80, "spCreateNewGLTexture", "texture_unit = %d, texture_name = %d\n", texture_unit, *texture_name);
    
    glBindTexture(GL_TEXTURE_2D, *texture_name);
    
    spDebug(80, "spCreateNewGLTexture", "texture_width = %d, texture_height = %d\n", texture_width, texture_height);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texture_width, texture_height, 0, format, type, NULL);

    if ((err = glGetError()) != GL_NO_ERROR) {
	spDebug(10, "spCreateNewGLTexture", "glTexImage2D failed: error code = %x\n", err);
	glDeleteTextures(1, texture_name);
    } else {
#if 0
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
#else
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
#endif
    }
    
    glBindTexture(GL_TEXTURE_2D, 0);

    if (use_shader == SP_FALSE) {
	if (gl_texture_2d == GL_FALSE) glDisable(GL_TEXTURE_2D);
    }

    spDebug(80, "spCreateNewGLTexture", "done: err = %x, GL_NO_ERROR = %d\n", err, GL_NO_ERROR);
    
    return err;
}

void spFreeAllGLTextureInfos(void *data);

spGLTextureInfo spAllocGLTextureInfo(int min_width, int min_height, int inside_margin_width, int inside_margin_height,
				     int width, int height, spBool use_shader, spBool use_vao)
{
    GLint num_texture_unit = 0;
    GLenum texture_unit;
    GLuint texture_name;
    spGLTextureInfo info;
    GLenum err;
    int texture_size = 0;
    int texture_width, texture_height;

    texture_width = POW2(spNextPow2(MAX(min_width, width)));
    texture_height = POW2(spNextPow2(MAX(min_height, height)));

    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &texture_size);
    /*spDebug(10, "spAllocGLTextureInfo", "glGetIntegerv of GL_MAX_TEXTURE_SIZE: error code = %x\n", glGetError());*/
    spDebug(80, "spAllocGLTextureInfo", "texture_size = %d\n", texture_size);

    texture_width = MAX(texture_width, texture_height);
    texture_width = MIN(texture_width, texture_size);
    texture_height = texture_width;
    
    spDebug(80, "spAllocGLTextureInfo", "min_width = %d, min_height = %d, width = %d, height = %d, texture_width = %d, texture_height = %d\n",
	    min_width, min_height, width, height, texture_width, texture_height);

    if (sp_last_gl_string_texture_info != NULL) {
	texture_unit = sp_last_gl_string_texture_info->texture_unit - 1;
    } else {
#if 0
	spDebug(80, "spAllocGLTextureInfo", "GL_VERSION = %s\n", glGetString(GL_VERSION));
	spDebug(10, "spAllocGLTextureInfo", "glGetString: error code = %x\n", glGetError());
#endif

	if (use_shader) {
	    spDebug(80, "spAllocGLTextureInfo", "call glGetIntegerv of GL_MAX_TEXTURE_IMAGE_UNITS\n");
	    glGetIntegerv(/*GL_MAX_TEXTURE_IMAGE_UNITS*/0x8872, &num_texture_unit);
	    /*spDebug(10, "spAllocGLTextureInfo", "glGetIntegerv of GL_MAX_TEXTURE_IMAGE_UNITS: error code = %x\n", glGetError());*/
	} else {
	    spDebug(80, "spAllocGLTextureInfo", "call glGetIntegerv of GL_MAX_TEXTURE_UNIT\n");
	    glGetIntegerv(GL_MAX_TEXTURE_UNITS, &num_texture_unit);
	}
	spDebug(80, "spAllocGLTextureInfo", "num_texture_unit = %d\n", num_texture_unit);

	if ((err = glGetError()) != GL_NO_ERROR || num_texture_unit == 0) {
	    spDebug(80, "spAllocGLTextureInfo", "cannot get the number of texture unit: error code = %x\n", err);
	    return NULL;
	}
	
	num_texture_unit = MIN(num_texture_unit, 16);
	texture_unit = GL_TEXTURE0 + num_texture_unit - 1;
	spDebug(80, "spAllocGLTextureInfo", "texture_unit = %d, GL_TEXTURE0 = %d\n", texture_unit, GL_TEXTURE0);
    }

    if ((err = spCreateNewGLTexture(texture_unit, texture_width, texture_height, use_shader, &texture_name)) != GL_NO_ERROR) {
	spDebug(80, "spAllocGLTextureInfo", "spCreateNewGLTexture failed: error code = %x\n", err);
	return NULL;
    }

    info = xalloc(1, struct _spGLTextureInfo);
    memset(info, 0, sizeof(struct _spGLTextureInfo));

    info->inside_margin_width = inside_margin_width;
    info->inside_margin_height = inside_margin_height;
    
    info->texture_unit = texture_unit;
    info->texture_name = texture_name;
    
    info->texture_width = texture_width;
    info->texture_height = texture_height;

#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
    info->use_shader = use_shader;
    if (info->use_shader == SP_FALSE) {
        info->use_vao = SP_FALSE;
    } else {
#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
        if (use_vao == SP_TRUE) {
            if (spIsWGLFunctionLoaded(glBindVertexArray) == SP_FALSE) {
                use_vao = SP_FALSE;
            }
        }
#elif defined(ANDROID)
        if (use_vao == SP_TRUE) {
            if (spIsEGLFunctionLoaded(glBindVertexArrayOES) == SP_FALSE) {
                use_vao = SP_FALSE;
            }
        }
#endif
    
        info->use_vao = use_vao;
        
        if (info->use_vao == SP_TRUE) {
            static GLfloat positionBuffer[12] = {
                0.0f, 0.0f, 0.0f,
                0.0f, -1.0f, 0.0f,
                1.0f, 0.0f, 0.0f,
                1.0f, -1.0f, 0.0f,
            };
            
            glGenBuffers(1, &info->position_vbo);
            glBindBuffer(GL_ARRAY_BUFFER, info->position_vbo);
            glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 12, positionBuffer, GL_STATIC_DRAW);
            glBindBuffer(GL_ARRAY_BUFFER, 0);
        }
    }
#else
    info->use_shader = SP_FALSE;
    info->use_vao = SP_FALSE;
#endif
    
    spDebug(80, "spAllocGLTextureInfo", "use_shader = %d, use_vao = %d\n", info->use_shader, info->use_vao);
    
    if (sp_last_gl_string_texture_info == NULL) {
	spAddExitCallback(spFreeAllGLTextureInfos, NULL);
	sp_first_gl_string_texture_info = info;
    } else {
	sp_last_gl_string_texture_info->next = info;
	info->prev = sp_last_gl_string_texture_info;
    }
    sp_last_gl_string_texture_info = info;

    spDebug(80, "spAllocGLTextureInfo", "done\n");
    
    return info;
}

void spFreeGLTextureInfo(spGLTextureInfo info)
{
    spGLTextureLine line, next_line;
    spGLTextureRegion region, next_region;

    spDebug(80, "spFreeGLTextureInfo", "in\n");
    
    line = info->first_line;
    
    while (line != NULL) {
	next_line = line->next;

	region = line->first_region;
	while (region != NULL) {
	    next_region = region->next;
	    spFreeGLTextureRegion(region);
	    region = next_region;
	}
	xfree(line);
	
	line = next_line;
    }

    glActiveTexture(info->texture_unit);
    
    glDeleteTextures(1, &info->texture_name);

    if (info->use_shader == SP_FALSE) {
	glDisable(GL_TEXTURE_2D);
    } else {
#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
        if (info->position_vbo != 0) {
            glDeleteBuffers(1, &info->position_vbo);
        }
#endif
    }

    if (sp_first_gl_string_texture_info == info) {
	sp_first_gl_string_texture_info = info->next;
    }
    if (sp_last_gl_string_texture_info == info) {
	sp_last_gl_string_texture_info = info->prev;
    }

    xfree(info);
	
    spDebug(80, "spFreeGLTextureInfo", "done\n");
    
    return;
}

void spFreeAllGLTextureInfos(void *data)
{
    spGLTextureInfo info, prev_info;
    
    spDebug(80, "spFreeAllGLTextureInfos", "in\n");
    
    info = sp_last_gl_string_texture_info;

    while (info != NULL) {
	prev_info = info->prev;
	spFreeGLTextureInfo(info);
	info = prev_info;
    }

    sp_first_gl_string_texture_info = NULL;
    sp_last_gl_string_texture_info = NULL;
    
    spDebug(80, "spFreeAllGLTextureInfos", "done\n");
    
    return;
}

spGLTextureRegion spFindGLTextureVacantRegionFromLine(spGLTextureLine line, int width, int height)
{
    spGLTextureRegion region;
    int prev_x;

    prev_x = 0;
    region = line->first_region;
    if (region == NULL) {
	spDebug(80, "spFindGLTextureVacantRegionFromLine", "call spAllocGLTextureRegion, because no region is found\n");
	return spAllocGLTextureRegion(line, NULL, width, height);
    }

    while (region != NULL) {
	if (region->x - prev_x >= width) {
	    spDebug(80, "spFindGLTextureVacantRegionFromLine", "call spAllocGLTextureRegion for insert, because vacant region is found\n");
	    return spAllocGLTextureRegion(line, region, width, height);
	} else if (region->next == NULL && region->x + region->width + width <= line->info->texture_width) {
	    spDebug(80, "spFindGLTextureVacantRegionFromLine", "call spAllocGLTextureRegion for next to current last region\n");
	    return spAllocGLTextureRegion(line, NULL, width, height);
	}

	prev_x = region->x + region->width;
	region = region->next;
    }
    
    spDebug(80, "spFindGLTextureVacantRegionFromLine", "not found\n");

    return NULL;
}
	
spGLTextureRegion spGetGLTextureVacantRegion(spGLTextureInfo info, int width, int height)
{
    int y;
    spGLTextureLine line;
    spGLTextureRegion region;

    if (info == NULL) return NULL;

    line = info->first_line;
    spDebug(80, "spGetGLTextureVacantRegion", "width = %d, height = %d, first line = %ld\n", width, height, (long)line);
    y = 0;

    while (line != NULL) {
	if (height <= line->height
	    || (line->next == NULL && line->ypos + height <= info->texture_height)) {
	    region = spFindGLTextureVacantRegionFromLine(line, width, height);
	    spDebug(80, "spGetGLTextureVacantRegion", "spFindGLTextureVacantRegionFromLine result: %ld\n", (long)region);
	    if (region != NULL) {
		return region;
	    }
	}

	y += line->height;
	line = line->next;
    }

    spDebug(80, "spGetGLTextureVacantRegion", "y = %d, height = %d, texture_height = %d\n",
	    y, height, info->texture_height);
    
    if (y + height <= info->texture_height) {
	line = spAllocGLTextureLine(info, height);
	return spAllocGLTextureRegion(line, NULL, width, height);
    }

    spDebug(80, "spGetGLTextureVacantRegion", "not found\n");
    
    return NULL;
}
		 
spGLTextureRegion spFindGLTextureRegion(int inside_margin_width, int inside_margin_height, int width, int height,
                                        spBool use_shader, spBool use_vao)
{
    spGLTextureInfo info = NULL;
    spGLTextureRegion region = NULL;
    
    info = sp_last_gl_string_texture_info;
	
    spDebug(80, "spFindGLTextureRegion", "width = %d, height = %d, info = %ld\n", width, height, (long)info);

    if (info == NULL
	|| (region = spGetGLTextureVacantRegion(info, width, height)) == NULL) {
	if ((info = spAllocGLTextureInfo(SP_GL_STRING_BY_TEXTURE_MIN_TEXTURE_WIDTH,
					 SP_GL_STRING_BY_TEXTURE_MIN_TEXTURE_HEIGHT,
					 inside_margin_width, inside_margin_height,
					 width, height, use_shader, use_vao)) == NULL) {
	    return NULL;
	}
	region = spGetGLTextureVacantRegion(info, width, height);
    }

    return region;
}

GLenum spSubImage2DForGLTextureRegion(spGLTextureRegion region, void *pixels)
{
    GLboolean gl_texture_2d = GL_FALSE;
    GLenum err;
    GLenum format = SP_GL_STRING_BY_TEXTURE_BITMAP_PIXEL_FORMAT;
    GLenum type = SP_GL_STRING_BY_TEXTURE_BITMAP_DATA_TYPE;
    
    glActiveTexture(region->line->info->texture_unit);

    if (region->line->info->use_shader == SP_FALSE) {
	gl_texture_2d = glIsEnabled(GL_TEXTURE_2D);
	spDebug(80, "spSubImage2DForGLTextureRegion", "gl_texture_2d = %d\n", gl_texture_2d);
	if (gl_texture_2d == GL_FALSE) glEnable(GL_TEXTURE_2D);
    }
    
    spDebug(80, "spSubImage2DForGLTextureRegion", "texture_name = %d\n", region->line->info->texture_name);
    glBindTexture(GL_TEXTURE_2D, region->line->info->texture_name);
    
    spDebug(80, "spSubImage2DForGLTextureRegion", "x = %d, y = %d, width = %d, height = %d\n",
	    region->x, region->y, region->width, region->height);
    glTexSubImage2D(GL_TEXTURE_2D, 0, region->x, region->y, region->width, region->height,
		    format, type, pixels);
    
    if ((err = glGetError()) != GL_NO_ERROR) {
	spDebug(10, "spSubImage2DForGLTextureRegion", "glTexSubImage2D failed: error code = %x\n", err);
    }
    
    glBindTexture(GL_TEXTURE_2D, 0);

    if (region->line->info->use_shader == SP_FALSE) {
	if (gl_texture_2d == GL_FALSE) glDisable(GL_TEXTURE_2D);
    }

    return err;
}

void spDrawGLTextureRegionString(spGLTextureRegion region, GLfloat x, GLfloat y, GLfloat z, GLfloat xfactor, GLfloat yfactor,
				 GLfloat *projection_matrix, GLfloat *modelview_matrix)
{
    static GLfloat vertexBuffer[8] = {
	0.0f, 0.0f,
	0.0f, -1.0f,
	1.0f, 0.0f,
	1.0f, -1.0f,
    };
    GLfloat textureBuffer[8];
    
    GLboolean gl_texture_2d;
    GLboolean gl_vertex_array;
    GLboolean gl_texture_coord_array;
    GLboolean gl_color_array;
    GLboolean gl_normal_array;
    GLboolean gl_blend;
    GLboolean gl_lighting;
    GLfloat current[4];
    GLfloat mvm[16];
    GLint mode = GL_MODELVIEW;
    
    spDebug(80, "spDrawGLTextureRegionString", "x = %f, y = %f, z = %f\n", x, y, z);

    glGetIntegerv(GL_MATRIX_MODE, &mode);
    spDebug(80, "spDrawGLTextureRegionString", "mode = %d\n", mode);

    gl_blend = glIsEnabled(GL_BLEND); /* glGetBooleanv sometimes doesn't work on Android */
    if (gl_blend == GL_FALSE) glEnable(GL_BLEND);

    glBlendFunc(region->alpha_sfactor, region->alpha_dfactor);
    
    glActiveTexture(region->line->info->texture_unit);
    glClientActiveTexture(region->line->info->texture_unit);
    
    gl_texture_2d = glIsEnabled(GL_TEXTURE_2D);
    if (gl_texture_2d == GL_FALSE) glEnable(GL_TEXTURE_2D);
    
    spDebug(80, "spDrawGLTextureRegionString", "gl_blend = %d, gl_texture_2d = %d\n",
	    gl_blend, gl_texture_2d);
    
    glBindTexture(GL_TEXTURE_2D, region->line->info->texture_name);
    
    gl_vertex_array = glIsEnabled(GL_VERTEX_ARRAY);
    if (gl_vertex_array == GL_FALSE) glEnableClientState(GL_VERTEX_ARRAY);
    
    gl_texture_coord_array = glIsEnabled(GL_TEXTURE_COORD_ARRAY);
    if (gl_texture_coord_array == GL_FALSE) glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    
    gl_color_array = glIsEnabled(GL_COLOR_ARRAY);
    if (gl_color_array == GL_TRUE) glDisableClientState(GL_COLOR_ARRAY);

    gl_normal_array = glIsEnabled(GL_NORMAL_ARRAY);
    if (gl_normal_array == GL_TRUE) glDisableClientState(GL_NORMAL_ARRAY);

    gl_lighting = glIsEnabled(GL_LIGHTING);
    if (gl_lighting == GL_TRUE) glDisable(GL_LIGHTING);

    spGetGLTextureRegionCoord(region, &textureBuffer[2], &textureBuffer[3], &textureBuffer[4], &textureBuffer[5]);
    textureBuffer[0] = textureBuffer[2];
    textureBuffer[1] = textureBuffer[5];
    textureBuffer[6] = textureBuffer[4];
    textureBuffer[7] = textureBuffer[3];
	    
    spDebug(80, "spDrawGLTextureRegionString", "call glVertexPointer\n");
    glVertexPointer(2, GL_FLOAT, 0, vertexBuffer);
    glTexCoordPointer(2, GL_FLOAT, 0, textureBuffer);
    
    glGetFloatv(GL_CURRENT_COLOR, current);
    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

    if (projection_matrix != NULL) {
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf(projection_matrix);
    }
    
    spDebug(80, "spDrawGLTextureRegionString", "call glPushMatrix\n");
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    {
	if (modelview_matrix != NULL) {
	    glLoadMatrixf(modelview_matrix);
	}
	
	glTranslatef(x, y, z);

	glGetFloatv(GL_MODELVIEW_MATRIX, mvm);

	mvm[0] = mvm[5] = mvm[10] = 1.0f;
	mvm[1] = mvm[2] = mvm[4] = mvm[6] = mvm[8] = mvm[9] = 0.0f;
	
	glLoadMatrixf(mvm);
	
	glScalef(xfactor, yfactor, 1.0f);
	spDebug(80, "spDrawGLTextureRegionString", "call glDrawArrays\n");
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    }

    spDebug(80, "spDrawGLTextureRegionString", "call glPopMatrix\n");
    glPopMatrix();    
    
    if (projection_matrix != NULL) {
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();    
    }
    
    glColor4f(current[0], current[1], current[2], current[3]);

    if (gl_lighting == GL_TRUE) glEnable(GL_LIGHTING);

    if (gl_normal_array == GL_TRUE) glEnableClientState(GL_NORMAL_ARRAY);
    if (gl_color_array == GL_TRUE) glEnableClientState(GL_COLOR_ARRAY);
    
    if (gl_texture_coord_array == GL_FALSE) glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    if (gl_vertex_array == GL_FALSE) glDisableClientState(GL_VERTEX_ARRAY);
    
    glBindTexture(GL_TEXTURE_2D, 0);

    if (gl_texture_2d == GL_FALSE) glDisable(GL_TEXTURE_2D);

    if (gl_blend == GL_FALSE) glDisable(GL_BLEND);
    
    glMatrixMode(mode);
    
    spDebug(80, "spDrawGLTextureRegionString", "done\n");
    
    return;
}

spBool spCreateGLTextureRegionString(spGLString glstring)
{
    spBool flag = SP_TRUE;
    
    if (SpGLStringArch(glstring).region == NULL) {
	if (spLockDIBitmap(SpGLStringArch(glstring).dibitmap) == SP_TRUE) {
	    SpGLStringArch(glstring).region = spFindGLTextureRegion(SpGLStringArch(glstring).inside_margin_width,
								    SpGLStringArch(glstring).inside_margin_height,
								    SpGLStringArch(glstring).dibitmap->info.width,
								    SpGLStringArch(glstring).dibitmap->info.height,
								    SpGLStringPart(glstring).use_shader,
                                                                    (SpGLStringPart(glstring).options & SP_GL_STRING_OPTION_USE_VAO) ? SP_TRUE : SP_FALSE);
	    if (SpGLStringArch(glstring).region == NULL) {
		spDebug(10, "spCreateGLTextureRegionString", "region == NULL\n");
		flag = SP_FALSE;
	    } else {
		if (spSubImage2DForGLTextureRegion(SpGLStringArch(glstring).region, SpGLStringArch(glstring).dibitmap->info.data) != GL_NO_ERROR) {
		    spDebug(10, "spCreateGLTextureRegionString", "spSubImage2DForGLTextureRegion failed\n");
		    spFreeGLTextureRegion(SpGLStringArch(glstring).region);
		    SpGLStringArch(glstring).region = NULL;
		    flag = SP_FALSE;
		} else {
                    spGetGLDIBitmapAlphaBlendFactor(SpGLStringArch(glstring).dibitmap,
                                                    &SpGLStringArch(glstring).region->alpha_sfactor,
                                                    &SpGLStringArch(glstring).region->alpha_dfactor);
                }
	    }
			
	    spUnlockDIBitmap(SpGLStringArch(glstring).dibitmap);
	} else {
	    spDebug(10, "spCreateGLTextureRegionString", "spLockDIBitmap failed\n");
	}
    }

    return flag;
}

spBool spGLStringByTextureCreate(spGLString glstring, char *string)
{
    if ((SpGLStringArch(glstring).dibitmap = spCreateDIBitmapStringArch(glstring, string)) != NULL) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spDestroyGLStringByTexture(spGLString glstring)
{
    spDebug(80, "spDestroyGLStringByTexture", "region = %ld\n", (long)SpGLStringArch(glstring).region);
    
    if (SpGLStringArch(glstring).region != NULL) {
	spGLTextureInfo info;
	
	info = SpGLStringArch(glstring).region->line->info;
	spFreeGLTextureRegion(SpGLStringArch(glstring).region);
	if (info->num_region <= 0) {
	    spFreeGLTextureInfo(info);
	}

	SpGLStringArch(glstring).region = NULL;
    }
	
    if (SpGLStringArch(glstring).dibitmap != NULL) {
	spDestroyDIBitmap(SpGLStringArch(glstring).dibitmap);
	SpGLStringArch(glstring).dibitmap = NULL;
    }
    
    spDebug(80, "spDestroyGLStringByTexture", "done\n");
    
    return SP_TRUE;
}

#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
HGLRC sp_gl_string_shader_functions_hglrc = NULL;

SP_WGL_DECLARE_EXTERN_FUNCTION(glGetProgramiv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetProgramInfoLog)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderiv)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderInfoLog)
SP_WGL_DECLARE_EXTERN_FUNCTION(glAttachShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDetachShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCreateProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCreateShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glDeleteShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glCompileShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glIsProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glIsShader)
SP_WGL_DECLARE_EXTERN_FUNCTION(glLinkProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUseProgram)
SP_WGL_DECLARE_EXTERN_FUNCTION(glShaderSource)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetShaderSource)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetAttribLocation)
SP_WGL_DECLARE_EXTERN_FUNCTION(glGetUniformLocation)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniform1i)
SP_WGL_DECLARE_EXTERN_FUNCTION(glUniformMatrix4fv)
#endif

static GLuint loadProgram(GLuint vshader, GLuint fshader)
{
    GLuint program;

    if ((program = glCreateProgram()) != 0) {
	glAttachShader(program, vshader);
	glAttachShader(program, fshader);
	glLinkProgram(program);

	if (spGLShaderCheckLinkError(program) == GL_FALSE) {
	    glDeleteProgram(program);
	    program = 0;
	}
    }

    return program;
}

static GLchar sp_glstring_shader_version_line_format[] = "#version %ld\n";

static GLchar sp_glstring_vertex_shader_macros_for_version130[] =
    "#define attribute in\n"
    "#define varying out\n";

static GLchar sp_glstring_vertex_shader_source[] =
    "\n"
    "attribute vec3 attr_position;\n"
    "attribute vec2 attr_uv;\n"
    "varying vec2 vary_uv;\n"
    "\n"
    "uniform mat4 unif_MVP;\n"
    "\n"
    "void main() {\n" /* L9 */
    "    gl_Position = unif_MVP * vec4(attr_position, 1.0);\n"
    "    vary_uv = attr_uv;\n"
    "}\n";

static GLchar sp_glstring_fragment_shader_macros_for_version130[] =
    "#define varying in\n"
    "#define texture2D texture\n"
    "#define gl_FragColor FragColor\nout vec4 FragColor;\n";

static GLchar sp_glstring_fragment_shader_source[] =
    "\n"
    "varying vec2 vary_uv;\n"
    "\n"
    "uniform sampler2D unif_texture;\n"
    "\n"
    "void main() {\n" /* L7 */
    "    gl_FragColor = texture2D(unif_texture, vary_uv);\n"
    "}\n";

static void freeShader(void *data)
{
    if (sp_glstring_shader_program != 0) {
	glUseProgram(0);
	
	glDetachShader(sp_glstring_shader_program, sp_glstring_vertex_shader);
	glDetachShader(sp_glstring_shader_program, sp_glstring_fragment_shader);
	glDeleteProgram(sp_glstring_shader_program);
	sp_glstring_shader_program = 0;
    }
    
    if (sp_glstring_vertex_shader != 0) {
	glDeleteShader(sp_glstring_vertex_shader);
	sp_glstring_vertex_shader = 0;
    }
    if (sp_glstring_fragment_shader != 0) {
	glDeleteShader(sp_glstring_fragment_shader);
	sp_glstring_fragment_shader = 0;
    }

    sp_glstring_attr_position_location = 0;
    sp_glstring_attr_uv_location = 0;

    sp_glstring_unif_MVP_location = 0;
    sp_glstring_unif_texture_location = 0;
    
    spDebug(80, "freeShader", "done\n");
    
    return;
}

static spBool initShader(unsigned long options)
{
    unsigned long version;
    GLchar source[4096];
    
#if defined(_WIN32) && !(defined(USE_MOTIF) || defined(GTK))
    if (spInitShaderGLFunctionsWin(&sp_gl_string_shader_functions_hglrc) == SP_FALSE) {
	return SP_FALSE;
    }
#elif defined(ANDROID)
    if (spInitShaderGLFunctionsAndroid() == SP_FALSE) {
	return SP_FALSE;
    }
#endif

    version = options & SP_GL_STRING_SHADER_VERSION_MASK;
    spDebug(80, "initShader", "version = %ld\n", version);
    
    if (version > 0) {
        sprintf(source, sp_glstring_shader_version_line_format, version);
        if (version >= 130) {
            spStrCat(source, sizeof(source), sp_glstring_vertex_shader_macros_for_version130);
        }
    } else {
        source[0] = '\n'; source[1] = NUL;
    }
    spStrCat(source, sizeof(source), sp_glstring_vertex_shader_source);
    spDebug(80, "initShader", "vertex shader source:\n%s\n", source);
    sp_glstring_vertex_shader = spGLShaderCompile(GL_VERTEX_SHADER, source);
    
    if (version > 0) {
        sprintf(source, sp_glstring_shader_version_line_format, version);
    } else {
        source[0] = '\n'; source[1] = NUL;
    }
    if (options & SP_GL_STRING_SHADER_VERSION_ES) {
        spStrCat(source, sizeof(source), "precision mediump float;");
    }
    if (version >= 130) {
        spStrCat(source, sizeof(source), sp_glstring_fragment_shader_macros_for_version130);
    }
    spDebug(80, "initShader", "fragment shader source header:\n%s\n", source);
    spStrCat(source, sizeof(source), sp_glstring_fragment_shader_source);
    spDebug(80, "initShader", "fragment shader source:\n%s\n", source);
    sp_glstring_fragment_shader = spGLShaderCompile(GL_FRAGMENT_SHADER, source);

    if ((sp_glstring_shader_program = loadProgram(sp_glstring_vertex_shader, sp_glstring_fragment_shader)) == 0) {
	glDeleteShader(sp_glstring_vertex_shader);
	glDeleteShader(sp_glstring_fragment_shader);
	spDebug(10, "initShader", "loadProgram failed\n");
	return SP_FALSE;
    }
    
    spAddExitCallback(freeShader, NULL);
    
    sp_glstring_attr_position_location = glGetAttribLocation(sp_glstring_shader_program, "attr_position");
    sp_glstring_attr_uv_location = glGetAttribLocation(sp_glstring_shader_program, "attr_uv");
    
    sp_glstring_unif_MVP_location = glGetUniformLocation(sp_glstring_shader_program, "unif_MVP");
    sp_glstring_unif_texture_location = glGetUniformLocation(sp_glstring_shader_program, "unif_texture");

    spDebug(80, "initShader", "done\n");
    
    return SP_TRUE;
}

spBool spInitGLStringShader(unsigned long options)
{
    if (sp_glstring_shader_program == 0) {
	if (initShader(options) == SP_FALSE) {
	    spDebug(50, "spDrawGLTextureRegionStringShader", "initShader failed\n");
	    return SP_FALSE;
	}
    }

    return SP_TRUE;
}

static void spSetUVVertexGLTextureRegionStringShader(spGLTextureRegion region, GLfloat *uvBuffer)
{
    spGetGLTextureRegionCoord(region, &uvBuffer[2], &uvBuffer[3], &uvBuffer[4], &uvBuffer[5]);
    uvBuffer[0] = uvBuffer[2];
    uvBuffer[1] = uvBuffer[5];
    uvBuffer[6] = uvBuffer[4];
    uvBuffer[7] = uvBuffer[3];
	    
    glVertexAttribPointer(sp_glstring_attr_uv_location, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid *)uvBuffer);
    /*spDebug(80, "spSetUVVertexGLTextureRegionStringShader", "glVertexAttribPointer for uv error = %d\n", (int)glGetError());*/
    
    return;
}

void spDrawGLTextureRegionStringShader(spGLTextureRegion region, GLfloat x, GLfloat y, GLfloat z, GLfloat xfactor, GLfloat yfactor,
				       GLfloat *projection_matrix, GLfloat *modelview_matrix)
{
    static GLfloat positionBuffer[12] = {
	0.0f, 0.0f, 0.0f,
	0.0f, -1.0f, 0.0f,
	1.0f, 0.0f, 0.0f,
	1.0f, -1.0f, 0.0f,
    };
    GLfloat uvBuffer[8];
    
    GLboolean gl_blend;
    GLuint currentProgram = 0;
    spGLMat4 P;
    spGLMat4 MV, MV2, MV3;
    spGLMat4 T, S;
    spGLMat4 MVP;
    
    spDebug(80, "spDrawGLTextureRegionStringShader", "x = %f, y = %f, z = %f\n", x, y, z);

    glGetIntegerv(GL_CURRENT_PROGRAM, (GLint *)&currentProgram);
    spDebug(80, "spDrawGLTextureRegionStringShader", "currentProgram = %d, sp_glstring_shader_program = %d\n",
	    currentProgram, sp_glstring_shader_program);
    glUseProgram(sp_glstring_shader_program);

    gl_blend = glIsEnabled(GL_BLEND); /* glGetBooleanv sometimes doesn't work on Android */
    if (gl_blend == GL_FALSE) glEnable(GL_BLEND);

    glBlendFunc(region->alpha_sfactor, region->alpha_dfactor);
    
    glActiveTexture(region->line->info->texture_unit);
    
    spDebug(80, "spDrawGLTextureRegionStringShader", "gl_blend = %d\n", gl_blend);
    
    glBindTexture(GL_TEXTURE_2D, region->line->info->texture_name);
    spDebug(80, "spDrawGLTextureRegionStringShader", "glBindTexture error = %d\n", (int)glGetError());

    glUniform1i(sp_glstring_unif_texture_location, region->line->info->texture_unit - GL_TEXTURE0);

    if (projection_matrix != NULL) {
	P = spGLMat4Createfv(projection_matrix, SP_FALSE);
    } else {
	P = spGLMat4CreateIdentity();
    }
    if (modelview_matrix != NULL) {
	MV = spGLMat4Createfv(modelview_matrix, SP_FALSE);
    } else {
	MV = spGLMat4CreateIdentity();
    }
    
    T = spGLMat4CreateTranslation(x, y, z, SP_FALSE);
    MV2 = spGLMat4MultMatrix(&MV, &T, SP_FALSE);
    MV2.data[0][0] = MV2.data[1][1] = MV2.data[2][2] = 1.0f;
    MV2.data[0][1] = MV2.data[0][2] = MV2.data[1][0] = MV2.data[1][2] = MV2.data[2][0] = MV2.data[2][1] = 0.0f;
    
    S = spGLMat4CreateScaling(xfactor, yfactor, 1.0f);
    MV3 = spGLMat4MultMatrix(&MV2, &S, SP_FALSE);
    
    MVP = spGLMat4MultMatrix(&P, &MV3, SP_FALSE);
    glUniformMatrix4fv(sp_glstring_unif_MVP_location, 1, GL_FALSE, (const GLfloat *)MVP.data);

    if (region->line->info->use_vao == SP_TRUE) {
        glBindVertexArray(region->vao);
    } else {
        glEnableVertexAttribArray(sp_glstring_attr_position_location);
        /*spDebug(80, "spDrawGLTextureRegionStringShader", "glEnableVertexAttribArray for position error = %d\n", (int)glGetError());*/
        glEnableVertexAttribArray(sp_glstring_attr_uv_location);
        /*spDebug(80, "spDrawGLTextureRegionStringShader", "glEnableVertexAttribArray for uv error = %d\n", (int)glGetError());*/

        spDebug(80, "spDrawGLTextureRegionStringShader", "call glVertexAttribPointer\n");
        glVertexAttribPointer(sp_glstring_attr_position_location, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid *)positionBuffer);
        /*spDebug(80, "spDrawGLTextureRegionStringShader", "glVertexAttribPointer for position error = %d\n", (int)glGetError());*/
    
        spSetUVVertexGLTextureRegionStringShader(region, uvBuffer);
    }
    
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    spDebug(80, "spDrawGLTextureRegionStringShader", "glDrawArrays error = %d\n", (int)glGetError());
    
    if (region->line->info->use_vao == SP_TRUE) {
        glBindVertexArray(0);
    } else {
        glDisableVertexAttribArray(sp_glstring_attr_uv_location);
        glDisableVertexAttribArray(sp_glstring_attr_position_location);
    }
    
    glBindTexture(GL_TEXTURE_2D, 0);

    if (gl_blend == GL_FALSE) glDisable(GL_BLEND);
    
    glUseProgram(currentProgram);
    
    spDebug(80, "spDrawGLTextureRegionStringShader", "done\n");
    
    return;
}
#endif

spBool spDrawGLStringByTexture(spComponent gldrawable, spGLString glstring, GLfloat factor)
{
    int vpiv[4];
    int width, height;
    int content_width, content_height;
    GLfloat xfactor, yfactor, yshift;
    GLfloat xlen, ylen;

    spDebug(10, "spDrawGLStringByTexture", "in\n");
    
    if (spCreateGLTextureRegionString(glstring) == SP_FALSE) {
	spDebug(10, "spDrawGLStringByTexture", "spCreateGLTextureRegionString failed\n");
	return SP_FALSE;
    }

    glGetIntegerv(GL_VIEWPORT, vpiv);
    spDebug(80, "spDrawGLStringByTexture", "viewport = %d, %d, %d, %d\n", vpiv[0], vpiv[1], vpiv[2], vpiv[3]);
    width = vpiv[2];
    height = vpiv[3];

    if (SpGLStringPart(glstring).projection_matrix != NULL) {
	xlen = spGLVecLength4fv(&SpGLStringPart(glstring).projection_matrix[0]);
	ylen = spGLVecLength4fv(&SpGLStringPart(glstring).projection_matrix[4]);
    } else if (SpGLStringPart(glstring).use_shader == SP_FALSE) {
	GLfloat pm[16];
	glGetFloatv(GL_PROJECTION_MATRIX, pm);
	xlen = spGLVecLength4fv(&pm[0]);
	ylen = spGLVecLength4fv(&pm[4]);
    } else {
	xlen = ylen = 1.0f;
    }
    spDebug(80, "spDrawGLStringByTexture", "xlen = %f, ylen = %f\n", xlen, ylen);

    content_width = SpGLStringArch(glstring).dibitmap->info.width - 2 * SpGLStringArch(glstring).inside_margin_width;
    content_height = SpGLStringArch(glstring).dibitmap->info.height - 2 * SpGLStringArch(glstring).inside_margin_height;
    spDebug(80, "spDrawGLStringByTexture", "content_width = %d, content_height = %d\n", content_width, content_height);
    
    xfactor = ((GLfloat)content_width * 2.0f * factor / xlen) / (GLfloat)width;
    yfactor = ((GLfloat)content_height * 2.0f * factor / ylen) / (GLfloat)height;
    yshift = ((GLfloat)SpGLStringPart(glstring).y * 2.0f / ylen) / (GLfloat)height;
    spDebug(80, "spDrawGLStringByTexture", "width = %d, height = %d, xfactor = %f, yfactor = %f, yshift = %f\n",
	    width, height, xfactor, yfactor, yshift);

#if defined(SP_GL_STRING_BY_TEXTURE_SUPPORTS_SHADER)
    if (SpGLStringPart(glstring).use_shader) {
	spDebug(80, "spDrawGLStringByTexture", "call spDrawGLTextureRegionStringShader\n");
        spDrawGLTextureRegionStringShader(SpGLStringArch(glstring).region,
				          SpGLStringPart(glstring).pos_x,
				          SpGLStringPart(glstring).pos_y - yshift,
				          SpGLStringPart(glstring).pos_z,
					  xfactor, yfactor,
	                                  SpGLStringPart(glstring).projection_matrix,
	                                  SpGLStringPart(glstring).modelview_matrix);
    } else
#endif
    spDrawGLTextureRegionString(SpGLStringArch(glstring).region,
				SpGLStringPart(glstring).pos_x,
				SpGLStringPart(glstring).pos_y - yshift,
				SpGLStringPart(glstring).pos_z,
				xfactor, yfactor,
				SpGLStringPart(glstring).projection_matrix,
				SpGLStringPart(glstring).modelview_matrix);

    spDebug(80, "spDrawGLStringByTexture", "done\n");
    
    return SP_TRUE;
}
#endif

#if defined(SP_GL_STRING_LIST_BASE)
spBool spDrawGLStringListBase(spComponent gldrawable, spGLString glstring)
{
    GLuint i;
    GLfloat current[4];
    GLfloat new[4];
    unsigned char r, g, b;
    GLboolean gl_lighting;
    GLint mode = GL_MODELVIEW;

    gl_lighting = glIsEnabled(GL_LIGHTING);
    if (gl_lighting == GL_TRUE) glDisable(GL_LIGHTING);
    spDebug(80, "spDrawGLStringListBase", "gl_lighting = %d\n", gl_lighting);
	
    glGetFloatv(GL_CURRENT_COLOR, current);
    spDebug(80, "spDrawGLStringListBase", "current: r = %f, g = %f, b = %f, a = %f\n",
	    current[0], current[1], current[2], current[3]);
	
    spGetRGB(SpGLStringPart(glstring).fg_pixel, &r, &g, &b);
    new[0] = (GLfloat)r / 255.0f;
    new[1] = (GLfloat)g / 255.0f;
    new[2] = (GLfloat)b / 255.0f;
    new[3] = 1.0f;
    spDebug(80, "spDrawGLStringListBase", "new: r = %f, g = %f, b = %f, a = %f\n",
	    new[0], new[1], new[2], new[3]);

    glColor4fv(new);

    if (SpGLStringPart(glstring).projection_matrix != NULL || SpGLStringPart(glstring).modelview_matrix != NULL) {
	glGetIntegerv(GL_MATRIX_MODE, &mode);
	spDebug(80, "spDrawGLStringListBase", "current matrix mode: mode = %d\n", mode);
    }

    if (SpGLStringPart(glstring).projection_matrix != NULL) {
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf(SpGLStringPart(glstring).projection_matrix);
    }
	
    if (SpGLStringPart(glstring).modelview_matrix != NULL) {
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(SpGLStringPart(glstring).modelview_matrix);
    }
	    
    glRasterPos3f(SpGLStringPart(glstring).pos_x, SpGLStringPart(glstring).pos_y, SpGLStringPart(glstring).pos_z);
    
    for (i = 0; i < SpGLStringPart(glstring).count; i++) {
	spDebug(100, "spDrawGLStringListBase", "i = %d / %d\n", i, SpGLStringPart(glstring).count);
	glCallList(SpGLStringPart(glstring).list_base + i);
    }

    if (SpGLStringPart(glstring).modelview_matrix != NULL) {
	glPopMatrix();    
    }
    if (SpGLStringPart(glstring).projection_matrix != NULL) {
	if (SpGLStringPart(glstring).modelview_matrix != NULL) {
	    glMatrixMode(GL_PROJECTION);
	}
	glPopMatrix();    
    }
	
    if (SpGLStringPart(glstring).projection_matrix != NULL || SpGLStringPart(glstring).modelview_matrix != NULL) {
	glMatrixMode(mode);
    }
	
    glColor4fv(current);
	
    if (gl_lighting == GL_TRUE) glEnable(GL_LIGHTING);

    spDebug(80, "spDrawGLStringListBase", "done\n");
    
    return SP_TRUE;
}
#endif

spBool spDrawGLString(spComponent gldrawable, spGLString glstring)
{
    if (glstring == NULL || spIsGLDrawable(gldrawable) == SP_FALSE) return SP_FALSE;

#if defined(SP_GL_STRING_LIST_BASE)
    if (SpGLStringPart(glstring).use_texture == SP_FALSE && SpGLStringPart(glstring).use_shader == SP_FALSE) {
	return spDrawGLStringListBase(gldrawable, glstring);
    }
#endif
    
#if !defined(SP_GL_STRING_LIST_BASE) || defined(SP_GL_STRING_BY_TEXTURE)
    return spDrawGLStringArch(gldrawable, glstring);
#else
    return SP_FALSE;
#endif
}

#if (defined(GTK) && !defined(SP_GL_STRING_USE_PANGO)) || defined(XmVersion)
#include <sp/spString.h>
#include <sp/spKanji.h>
#include <sp/spLocale.h>

#include <X11/Xlib.h>
#include <GL/glx.h> 

spBool initGLStringX11(spGLString glstring)
{
    Display *display;
    char **miss, *def;
    int n_miss;
    
    if ((display = glXGetCurrentDisplay()) == NULL) return SP_FALSE;

    if (!strnone(SpGLStringPart(glstring).font_name)) {
	spDebug(10, "initGLStringX11", "font_name = %s\n", SpGLStringPart(glstring).font_name);
	SpGLStringArch(glstring).font_set =
	    XCreateFontSet(display, SpGLStringPart(glstring).font_name, &miss, &n_miss, &def);
    } else {
	SpGLStringArch(glstring).font_set =
	    XCreateFontSet(display,
			   /*"-*-fixed-medium-r-normal--14-*"*/"-*-*-medium-r-normal--14-*",
			   &miss, &n_miss, &def);
    }

    if (SpGLStringArch(glstring).font_set == NULL) {
	spDebug(10, "initGLStringX11", "error of XCreateFontSet for specified font\n");
	SpGLStringArch(glstring).font_set =
	    XCreateFontSet(display, "-*-*-*-*-*--*-*", &miss, &n_miss, &def);
    }
    if (SpGLStringArch(glstring).font_set == NULL) {
	spDebug(10, "initGLStringX11", "Cannot find font.\n");
	return SP_FALSE;
    }
    
    SpGLStringArch(glstring).num_font = XFontsOfFontSet(SpGLStringArch(glstring).font_set,
							&SpGLStringArch(glstring).font_struct_list,
							&SpGLStringArch(glstring).font_name_list);
    spDebug(50, "initGLStringX11", "num_font = %d\n", SpGLStringArch(glstring).num_font);
    if (SpGLStringArch(glstring).num_font <= 0) {
	XFreeFontSet(display, SpGLStringArch(glstring).font_set);
	SpGLStringArch(glstring).font_set = NULL;
	return SP_FALSE;
    }

    return SP_TRUE;
}

static Font findSuitableFont(spGLString glstring, int c)
{
    int i;
    int first, last;
    int firstrow, lastrow;
    Font fid = None;
    XFontStruct *font_struct;

    spDebug(80, "findSuitableFont", "c = %d, num_font = %d\n", c, SpGLStringArch(glstring).num_font);
    
    for (i = 0; i < SpGLStringArch(glstring).num_font; i++) {
	font_struct = SpGLStringArch(glstring).font_struct_list[i];
	first = (int)font_struct->min_char_or_byte2;
	last = (int)font_struct->max_char_or_byte2;
	firstrow = (int)font_struct->min_byte1;
	lastrow = (int)font_struct->max_byte1;
    
	spDebug(80, "findSuitableFont", "%d %s: first = %d, last = %d, firstrow = %d, lastrow = %d, fid = %ld\n",
		i, SpGLStringArch(glstring).font_name_list[i], first, last, firstrow, lastrow, (long)font_struct->fid);

	if (c >= firstrow * 256 + first && c <= lastrow * 256 + last) {
	    if (font_struct->fid != None) {
		fid = font_struct->fid;
	    } else {
		fid = XLoadFont(glXGetCurrentDisplay(), SpGLStringArch(glstring).font_name_list[i]);
	    }
	    spDebug(50, "findSuitableFont", "**** found. ****: fid = %ld\n", (long)fid);
	    break;
	}
    }

    return fid;
}

spBool spGLStringCreateX11(spGLString glstring, char *string)
{
    int i;
    int id;
    char c, pc;
    int mbc;
    Font fid;
    spBool flag;
    const char *lang;
    unsigned char *buf = NULL;

    spDebug(100, "spGLStringCreateX11", "in\n");
    
    if (initGLStringX11(glstring) == SP_FALSE) {
	spDebug(10, "spGLStringCreateX11", "initGLStringX11 failed\n");
	return SP_FALSE;
    }
    
    if ((lang = spGetLanguage()) != NULL) {
	spDebug(60, "spGLStringCreateX11", "lang = %s\n", lang); 
	if (spIsJapaneseLang(lang) == SP_TRUE) { /* UTF-8 doesn't work */
	    int size;
	    size = strlen(string) * 2; /* has margin */
	    spDebug(60, "spGLStringCreateX11", "size = %d\n", size); 
	    buf = xalloc(size, unsigned char);
	    spStrCopy((char *)buf, size, string);
	    spConvertKanjiFromLocaleCode(buf, size, SP_KANJI_CODE_EUC);
	    string = (char *)buf;
	}
    }
    
    pc = string[0];
    id = SpGLStringPart(glstring).list_base;
    flag = SP_TRUE;
	
    for (i = 1; pc != '\0'; i++) {
	spDebug(100, "spGLStringCreateX11", "i = %d\n", i); 
	c = string[i];

	if (c != '\0' && (pc & 0x80) != 0) {
	    int cu, cl;

	    cu = pc & 0x7f;
	    cl = c & 0x7f;
	    mbc = (cu * 256) + cl;
	    spDebug(100, "spGLStringCreateX11", "cu = %d, cl = %d, mbc = %d\n", cu, cl, mbc);

	    if ((fid = findSuitableFont(glstring, mbc)) == None) {
		spDebug(100, "spGLStringCreateX11", "Cannot find suitable font for MB char %d\n", mbc);
		flag = SP_FALSE;
		break;
	    }
	    
	    glXUseXFont(fid, mbc, 1, id);
	    
	    i++;
	    c = string[i];
	} else {
	    if ((fid = findSuitableFont(glstring, pc)) == None) {
		spDebug(100, "spGLStringCreateX11", "Cannot find suitable font for char %d\n", pc);
		flag = SP_FALSE;
		break;
	    }
	    
	    glXUseXFont(fid, pc, 1, id);
	}
	
	pc = c;
	    
	id++;
    }

    if (flag == SP_TRUE) {
	XRectangle ink, log;
	
	SpGLStringPart(glstring).count = id - SpGLStringPart(glstring).list_base;

	XmbTextExtents(SpGLStringArch(glstring).font_set, string, strlen(string), &ink, &log);
	SpGLStringPart(glstring).x = log.x;
	SpGLStringPart(glstring).y = log.y;
	SpGLStringPart(glstring).width = log.width;
	SpGLStringPart(glstring).height = log.height;

	SpGLStringPart(glstring).next_x = XmbTextEscapement(SpGLStringArch(glstring).font_set, string, strlen(string));
    }

    if (buf != NULL) {
	xfree(buf);
    }
    
    return flag;
}

spBool spGLStringDestroyX11(spGLString glstring)
{
    Display *display;
    
    if (SpGLStringArch(glstring).font_set != NULL) {
	if ((display = glXGetCurrentDisplay()) != NULL) {
	    XFreeFontSet(display, SpGLStringArch(glstring).font_set);
	}
	SpGLStringArch(glstring).font_set = NULL;
    }
    
    return SP_TRUE;
}

#if defined(SP_GL_STRING_BY_TEXTURE)
static spBool pixelToXColor(Display *display, Colormap colormap, spPixel pixel, XColor *color)
{
    color->red = 256 * (unsigned short)spGetRValue(pixel);
    color->green = 256 * (unsigned short)spGetGValue(pixel);
    color->blue = 256 * (unsigned short)spGetBValue(pixel);
    color->flags = DoRed|DoGreen|DoBlue;
    if (!XAllocColor(display, colormap, color)) {
	return SP_FALSE;
    }

    return SP_TRUE;
}

static void convertXImageDataToRGBA(int width, int height, XImage *ximage, unsigned char *data)
{
    int x, y;
    unsigned char r, g, b, a;
    unsigned long pixel = 0L, bg_pixel = 0L;
    unsigned char *ptr;

    ptr = data;
    for (y = 0; y < height; y++) {
	for (x = 0; x < width; x++) {
	    pixel = XGetPixel(ximage, x, y);
	    if (x == 0 && y == 0) {
		bg_pixel = pixel;
	    }

	    if (pixel == bg_pixel) {
		a = 0;
	    } else {
		a = 0xff;
	    }

	    r = (unsigned char)(0xFF & (pixel >> 16));
	    g = (unsigned char)(0xFF & (pixel >> 8));
	    b = (unsigned char)(0xFF & (pixel));
	    
	    *ptr++ = r;
	    *ptr++ = g;
	    *ptr++ = b;
	    *ptr++ = a;
	}
    }
    
    return;
}

spDIBitmap spCreateDIBitmapStringX11(spGLString glstring, char *string)
{
    Display *display;
    Screen *screen;
    int x, y;
    int depth;
    int bitmap_pad;
    int bitmap_width, bitmap_height;
    XRectangle ink, log;
    XColor color;
    spPixel bg_pixel = 0L;
    spDIBitmap dibitmap = NULL;
    spDIBitmapInfo dibinfo;

    spDebug(100, "spCreateDIBitmapStringArch", "in\n");
    
    SpGLStringArch(glstring).pixmap = None;
    SpGLStringArch(glstring).gc = NULL;
    
    if (initGLStringX11(glstring) == SP_FALSE) {
	spDebug(50, "spCreateDIBitmapStringArch", "initGLStringX11 failed\n");
	return NULL;
    }

    display = glXGetCurrentDisplay();
    screen = DefaultScreenOfDisplay(display);
    spDebug(100, "spCreateDIBitmapStringArch", "display = %ld, screen = %ld\n", (long)display, (long)screen);
    
    depth = DefaultDepthOfScreen(screen);
    if (depth <= 16) {
	bitmap_pad = 16;
    } else {
	bitmap_pad = 32;
    }
    spDebug(100, "spCreateDIBitmapStringArch", "depth = %d, bitmap_pad = %d\n", depth, bitmap_pad);
    
    XmbTextExtents(SpGLStringArch(glstring).font_set, string, strlen(string), &ink, &log);
    
    SpGLStringPart(glstring).x = log.x;
    SpGLStringPart(glstring).y = log.y;
    SpGLStringPart(glstring).width = log.width;
    SpGLStringPart(glstring).height = log.height;
    SpGLStringPart(glstring).next_x = XmbTextEscapement(SpGLStringArch(glstring).font_set, string, strlen(string));
    
    SpGLStringArch(glstring).inside_margin_width = SP_GL_STRING_BY_TEXTURE_INSIDE_MARGIN_WIDTH;
    SpGLStringArch(glstring).inside_margin_height = SP_GL_STRING_BY_TEXTURE_INSIDE_MARGIN_HEIGHT;
    
    bitmap_width = SpGLStringPart(glstring).width + 2 * SpGLStringArch(glstring).inside_margin_width;
    bitmap_height = SpGLStringPart(glstring).height + 2 * SpGLStringArch(glstring).inside_margin_height;
    spDebug(50, "spCreateDIBitmapStringArch", "bitmap size = (%d, %d)\n", bitmap_width, bitmap_height);

    SpGLStringArch(glstring).pixmap = XCreatePixmap(display, RootWindowOfScreen(screen),
						    bitmap_width, bitmap_height, depth);
    if (SpGLStringArch(glstring).pixmap == None) {
	spDebug(50, "spCreateDIBitmapStringArch", "XCreatePixmap failed\n");
	return NULL;
    }
    
    SpGLStringArch(glstring).gc = XCreateGC(display, RootWindowOfScreen(screen), 0, NULL);
    if (SpGLStringPart(glstring).fg_pixel == 0L) {
	bg_pixel = 0xffffff;
    }

    /* set background color */
    pixelToXColor(display, DefaultColormapOfScreen(screen), bg_pixel, &color);
    XSetForeground(display, SpGLStringArch(glstring).gc, color.pixel);

    /* draw background */
    XFillRectangle(display, SpGLStringArch(glstring).pixmap, SpGLStringArch(glstring).gc,
		   0, 0, bitmap_width, bitmap_height);
    
    /* set string color */
    pixelToXColor(display, DefaultColormapOfScreen(screen), SpGLStringPart(glstring).fg_pixel, &color);
    XSetForeground(display, SpGLStringArch(glstring).gc, color.pixel);

    /* draw string to pixmap */
    x = SpGLStringArch(glstring).inside_margin_width - SpGLStringPart(glstring).x;
    y = SpGLStringArch(glstring).inside_margin_height - SpGLStringPart(glstring).y;
    XmbDrawString(display, SpGLStringArch(glstring).pixmap, SpGLStringArch(glstring).font_set,
		  SpGLStringArch(glstring).gc, x, y, string, strlen(string));
    
    dibinfo.pixel_format = SP_DI_PIXEL_FORMAT_RGBA;
    dibinfo.width = bitmap_width;
    dibinfo.height = bitmap_height;
    dibinfo.upside_down = SP_FALSE;
    dibinfo.bit_size = 8;
    
    if ((dibitmap = spCreateDIBitmapWithOptions(&dibinfo, SP_DI_BITMAP_OPTION_FORCE_NONNATIVE)) != NULL) {
	dibitmap->info.pixel_format = SP_DI_PIXEL_FORMAT_RGBA;
	if (spLockDIBitmap(dibitmap) == SP_TRUE) {
	    char *buf;
	    XImage *ximage;

	    buf = malloc((bitmap_pad / 8) * bitmap_width * bitmap_height);
	    ximage = XCreateImage(display, DefaultVisualOfScreen(screen),
				  depth, ZPixmap, 0, buf,
				  bitmap_width, bitmap_height, bitmap_pad, 0);
	    
	    XGetSubImage(display, SpGLStringArch(glstring).pixmap,
			 0, 0, bitmap_width, bitmap_height,
			 AllPlanes, ZPixmap, ximage, 0, 0);

	    convertXImageDataToRGBA(bitmap_width, bitmap_height, ximage, dibitmap->info.data);
	    
	    XDestroyImage(ximage);
	    
	    spUnlockDIBitmap(dibitmap);
	}
    } else {
	spDebug(10, "spCreateDIBitmapStringArch", "spCreateDIBitmap failed\n");
    }
    
    return dibitmap;
}

spBool spGLStringByTextureDestroyX11(spGLString glstring)
{
    Display *display;
    
    display = glXGetCurrentDisplay();
    
    if (SpGLStringArch(glstring).pixmap != None) {
	XFreePixmap(display, SpGLStringArch(glstring).pixmap);
	SpGLStringArch(glstring).pixmap = None;
    }
    if (SpGLStringArch(glstring).gc != NULL) {
	XFreeGC(display, SpGLStringArch(glstring).gc);
    }
    
    return SP_TRUE;
}
#endif
#endif
