#include <sp/spComponent.h>
#include <sp/spCanvasP.h>
#include <sp/spGraphicsP.h>

#include <sp/spGLP.h>

#include <gdk/gdkx.h>

static gboolean setGLContext(GtkWidget *widget, spGLContext context)
{
    gboolean flag = FALSE;
#if defined(SP_GL_USE_GTK_GL_AREA)
    GtkGLArea *glarea;
    
    glarea = GTK_GL_AREA(widget);
    spDebug(80, "setGLContext", "glarea = %lx\n", (long)glarea);

    flag = TRUE;

    gtk_gl_area_make_current(glarea);
    if (gtk_gl_area_get_error(glarea) != NULL) {
        spDebug(80, "setGLContext", "gtk_gl_area_make_current failed\n");
        return FALSE;
    }
    
    gtk_gl_area_attach_buffers(glarea);
    if (gtk_gl_area_get_error(glarea) != NULL) {
        spDebug(80, "setGLContext", "gtk_gl_area_attach_buffers failed\n");
        return FALSE;
    }
    
#else
    GdkGLDrawable *gdkgldrawable;
    
    spDebug(80, "setGLContext", "in\n");
    
    if (widget->window == NULL
	|| (gdkgldrawable = gtk_widget_get_gl_drawable(widget)) == NULL) {
	spDebug(80, "setGLContext", "widget is currently not drawable\n");
	return FALSE;
    }
    
    if (context == NULL) {
	gdk_gl_drawable_gl_end(gdkgldrawable);
	spDebug(80, "setGLContext", "gdk_gl_drawable_gl_end done\n");
	flag = TRUE;
    } else {
	flag = gdk_gl_drawable_gl_begin(gdkgldrawable, SpGLContextArch(context).glcontext);
	spDebug(80, "setGLContext", "gdk_gl_drawable_gl_begin done\n");
    }
#endif
    
    spDebug(80, "setGLContext", "done: flag = %d\n", flag);
    
    return flag;
}

#if defined(SP_GL_USE_GTK_GL_AREA)
static void realizeCB(GtkGLArea *glarea, gpointer user_data)
{
    const GLubyte *renderer;
    const GLubyte *version;
    
    spDebug(10, "realizeCB", "initial error = %d\n", (int)glGetError());
    
    gtk_gl_area_make_current(glarea);
    renderer = glGetString(GL_RENDERER);
    version = glGetString(GL_VERSION);
    spDebug(10, "realizeCB", "renderer = %s, version = %s\n", renderer, version);
    
    spDebug(80, "realizeCB", "done\n");
    
    return;
}
#else
static gboolean configureEventCB(GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
    spComponent component = (spComponent)data;
    
    spDebug(80, "configureEventCB", "in\n");
    
    if (SpGLCanvasPart(component).current_context == NULL) {
	spDebug(80, "configureEventCB", "current_context == NULL\n");
	return TRUE;
    }
    
    return FALSE;
}
#endif

spBool spGLCanvasCreateArch(spComponent component)
{
    spCanvasCreateArch(component);

#if defined(SP_GL_USE_GTK_GL_AREA)
    g_signal_connect(SpPrimitiveArch(component).widget,
                     "realize", G_CALLBACK(realizeCB), component);
    
    spDebug(80, "spGLCanvasCreateArch", "call gtk_gl_area_set_required_version: %d.%d\n",
            SpGLVisualArch(SpGLCanvasPart(component).glvisual).major_version, SpGLVisualArch(SpGLCanvasPart(component).glvisual).minor_version);
    gtk_gl_area_set_required_version(GTK_GL_AREA(SpPrimitiveArch(component).widget),
                                     SpGLVisualArch(SpGLCanvasPart(component).glvisual).major_version,
                                     SpGLVisualArch(SpGLCanvasPart(component).glvisual).minor_version);
    
    spDebug(80, "spGLCanvasCreateArch", "has_alpha = %d, has_depth_buffer = %d, has_stencil_buffer = %d\n",
            SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_alpha, SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_depth_buffer,
            SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_stencil_buffer);
    gtk_gl_area_set_has_alpha(GTK_GL_AREA(SpPrimitiveArch(component).widget),
                              SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_alpha);
    gtk_gl_area_set_has_depth_buffer(GTK_GL_AREA(SpPrimitiveArch(component).widget),
                                     SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_depth_buffer);
    gtk_gl_area_set_has_stencil_buffer(GTK_GL_AREA(SpPrimitiveArch(component).widget),
                                       SpGLVisualArch(SpGLCanvasPart(component).glvisual).has_stencil_buffer);
    gtk_gl_area_set_auto_render(GTK_GL_AREA(SpPrimitiveArch(component).widget), FALSE);
    
#else
    gtk_widget_set_gl_capability(SpPrimitiveArch(component).widget,
				 SpGLVisualArch(SpGLCanvasPart(component).glvisual).glconfig,
				 NULL,
				 TRUE,
				 GDK_GL_RGBA_TYPE);

#if GTK_CHECK_VERSION(3,0,0)
    g_signal_connect(SpPrimitiveArch(component).widget,
		     "configure_event", G_CALLBACK(configureEventCB),
		     component);
#else
    gtk_signal_connect(GTK_OBJECT(SpPrimitiveArch(component).widget),
		       "configure_event", GTK_SIGNAL_FUNC(configureEventCB),
		       component);
#endif
#endif
    
    return SP_TRUE;
}

spBool spGLCanvasSetParamsArch(spComponent component)
{
    return SP_TRUE;
}
    
spBool spGLCanvasDestroyArch(spComponent component)
{
    spCanvasDestroyArch(component);
    spPrimitiveDestroyArch(component);
    return SP_TRUE;
}

spBool spGLCanvasRedrawArch(spComponent component)
{
    spDebug(80, "spGLCanvasRedrawArch", "in\n");
   
#if defined(SP_GL_USE_GTK_GL_AREA)
    if (SpCanvasArch(component).in_draw_func == SP_TRUE) {
        spDebug(80, "spGLCanvasRedrawArch", "in draw func, return\n");
        return SP_TRUE;
    }
#else
    spGLWaitSystemDrawingArch(component);
#endif
    
    if (SpGLCanvasPart(component).current_context != NULL) {
        spGLContext context;

	context = SpGLCanvasPart(component).current_context;
	/* Resizing window causes truncated drawing and occasional crash without following code.
	 * OpenGL for X11 or Mesa problem? */
	if (!setGLContext(SpPrimitiveArch(component).widget, context)) {
	    spDebug(10, "spGLCanvasRedrawArch", "setGLContext error\n");
	    return SP_FALSE;
	}
    }
    
   spDebug(80, "spGLCanvasRedrawArch", "done\n");
   
    return SP_TRUE;
}

spBool spGLCanvasPostRedrawArch(spComponent component)
{
    return SP_TRUE;
}

#if !defined(SP_GL_USE_GTK_GL_AREA)
static GdkGLConfig *createGdkGLConfigByMode(spGLAttribute *attributes)
{
    int i;
    GdkGLConfigMode configmode = GDK_GL_MODE_DEPTH;
    
    for (i = 0; attributes[i] != SP_GL_NONE; i++) {
	switch (attributes[i]) {
	  case SP_GL_USE_GL:
	  case SP_GL_RGBA:
	    break;
	  case SP_GL_DOUBLEBUFFER:
	    configmode |= GDK_GL_MODE_DOUBLE;
	    break;
	  case SP_GL_STEREO:
	    configmode |= GDK_GL_MODE_STEREO;
	    break;

	  case SP_GL_ALPHA_SIZE:
	    configmode |= GDK_GL_MODE_ALPHA;
	    i++;
	    break;
	  case SP_GL_ACCUM_RED_SIZE:
	  case SP_GL_ACCUM_GREEN_SIZE:
	  case SP_GL_ACCUM_BLUE_SIZE:
	  case SP_GL_ACCUM_ALPHA_SIZE:
	    configmode |= GDK_GL_MODE_ACCUM;
	    i++;
	    break;
	  case SP_GL_STENCIL_SIZE:
	    configmode |= GDK_GL_MODE_STENCIL;
	    i++;
	    break;
	    
	  case SP_GL_SAMPLE_BUFFERS:
	    if (attributes[i + 1] >= 1) {
		spDebug(80, "createGdkGLConfigByMode", "multisample mode enabled\n");
		configmode |= GDK_GL_MODE_MULTISAMPLE;
	    }
	    i++;
	    break;
	    
	  case SP_GL_SAMPLES:
	    if (attributes[i + 1] >= 2) {
		spDebug(80, "createGdkGLConfigByMode", "multisample mode enabled\n");
		configmode |= GDK_GL_MODE_MULTISAMPLE;
	    }
	    i++;
	    break;
	    
	  case SP_GL_LEVEL:
	  case SP_GL_BUFFER_SIZE:
	  case SP_GL_AUX_BUFFERS:
	  case SP_GL_RED_SIZE:
	  case SP_GL_GREEN_SIZE:
	  case SP_GL_BLUE_SIZE:
	  case SP_GL_DEPTH_SIZE:
	    i++;
	    break;
	}
    }

    return gdk_gl_config_new_by_mode(configmode);
}
#endif

spBool spCreateGLVisualArch(spGLVisual visual, spTopLevel toplevel, spGLAttribute *attributes)
{
#if defined(SP_GL_USE_GTK_GL_AREA)
    int i;

    if (spGetVersionFromGLAttributes(attributes, &SpGLVisualArch(visual).major_version, &SpGLVisualArch(visual).minor_version)
        == SP_FALSE
        || (SpGLVisualArch(visual).major_version*1000+SpGLVisualArch(visual).minor_version) < 3200) {
        SpGLVisualArch(visual).major_version = 3;
        SpGLVisualArch(visual).minor_version = 2;
    }

    SpGLVisualArch(visual).has_alpha = FALSE;
    SpGLVisualArch(visual).has_depth_buffer = FALSE;
    SpGLVisualArch(visual).has_stencil_buffer = FALSE;
    
    for (i = 0; attributes[i] != SP_GL_NONE; i++) {
	switch (attributes[i]) {
	  case SP_GL_RGBA:
	  case SP_GL_ALPHA_SIZE:
            SpGLVisualArch(visual).has_alpha = TRUE;
            break;
	  case SP_GL_DEPTH_SIZE:
            SpGLVisualArch(visual).has_depth_buffer = TRUE;
            break;
	  case SP_GL_STENCIL_SIZE:
            SpGLVisualArch(visual).has_stencil_buffer = TRUE;
            break;
        }
    }
#else    
    int i;
    int count;
    int version_id;
    
    SpGLVisualArch(visual).core_attributes = NULL;
	
    gtk_gl_init(SpTopLevelPart(toplevel).argcp, SpTopLevelPart(toplevel).argvp);

    gdk_gl_query_version(&SpGLVisualArch(visual).major_version, &SpGLVisualArch(visual).minor_version);
    spDebug(80, "spCreateGLVisualArch", "major_version = %d, minor_version = %d\n",
	    SpGLVisualArch(visual).major_version, SpGLVisualArch(visual).minor_version);

    version_id = SpGLVisualArch(visual).major_version * 1000 + SpGLVisualArch(visual).minor_version;

    for (i = 0; i <= 1; i++) {
	count = 0;
	if (i == 0) {
	    SpGLVisualArch(visual).core_attributes = xspExtractGLAttributes(attributes, version_id, SP_TRUE,
                                                                            SP_GL_ATTRIBUTE_TYPE_CORE_MASK, &count);
	} else {
	    SpGLVisualArch(visual).core_attributes = xspExtractGLAttributes(attributes, 1000, SP_TRUE,
                                                                            SP_GL_ATTRIBUTE_TYPE_CORE_MASK, &count);
	}
	spDebug(80, "spCreateGLVisualArch", "i = %d, count = %d\n", i, count);
	
#if 0
	SpGLVisualArch(visual).glconfig = createGdkGLConfigByMode(SpGLVisualArch(visual).core_attributes);
#else
	SpGLVisualArch(visual).glconfig = gdk_gl_config_new(SpGLVisualArch(visual).core_attributes);
#endif

	if (SpGLVisualArch(visual).glconfig != NULL) {
	    spDebug(80, "spCreateGLVisualArch", "glconfig created\n");
	    break;
	}
	
	if (SpGLVisualArch(visual).core_attributes != NULL) {
	    xfree(SpGLVisualArch(visual).core_attributes);
	}
    }
    
    if (SpGLVisualArch(visual).glconfig == NULL) {
	spDebug(10, "spCreateGLVisualArch", "can't create glconfig\n");
	if (SpGLVisualArch(visual).core_attributes != NULL) {
	    xfree(SpGLVisualArch(visual).core_attributes);
	    SpGLVisualArch(visual).core_attributes = NULL;
	}
	return SP_FALSE;
    }
#endif
    
    return SP_TRUE;
}

spBool spDestroyGLVisualArch(spGLVisual visual)
{
#if !defined(SP_GL_USE_GTK_GL_AREA)
    if (SpGLVisualArch(visual).core_attributes != NULL) {
	xfree(SpGLVisualArch(visual).core_attributes);
	SpGLVisualArch(visual).core_attributes = NULL;
    }
#endif
    
    return SP_TRUE;
}

#if !defined(SP_GL_USE_GTK_GL_AREA)
static void createGLContextCB(GtkWidget *widget, spGLContext context)
{
    GdkGLContext *gdkglcshare = NULL;

    spDebug(80, "createGLContextCB", "in\n");
   
    if (SpGLContextArch(context).share != NULL) {
	spDebug(80, "createGLContextCB", "SpGLContextArch(context).share != NULL\n");
	gdkglcshare = SpGLContextArch(SpGLContextArch(context).share).glcontext;
    }
    
    SpGLContextArch(context).glcontext
	= gtk_widget_create_gl_context(widget,
				       gdkglcshare,
				       TRUE, GDK_GL_RGBA_TYPE);

    SpGLContextArch(context).share = NULL;
    
    spDebug(80, "createGLContextCB", "done\n");
   
    return;
}
#endif

spBool spCreateGLContextArch(spGLContext context, spComponent gldrawable, spGLVisual visual, spGLContext share)
{
    spDebug(80, "spCreateGLContextArch", "in\n");
    
    SpGLContextArch(context).glcontext = NULL;
    SpGLContextArch(context).ext_list = NULL;

#if defined(SP_GL_USE_GTK_GL_AREA)
    /* currently, not supported */
    SpGLContextArch(context).share = share;
#else
    SpGLContextArch(context).share = NULL;
    if (SpPrimitiveArch(gldrawable).widget->window == NULL) {
	spDebug(80, "spCreateGLContextArch", "not drawable, using signal\n");
	
	SpGLContextArch(context).share = share;
#if GTK_CHECK_VERSION(3,0,0)
	g_signal_connect(SpPrimitiveArch(gldrawable).widget,
			 "realize", G_CALLBACK(createGLContextCB), context);
#elif GTK_CHECK_VERSION(2,0,0)
	g_signal_connect(GTK_OBJECT(SpPrimitiveArch(gldrawable).widget),
			 "realize", G_CALLBACK(createGLContextCB), context);
#else
	gtk_signal_connect(GTK_OBJECT(SpPrimitiveArch(gldrawable).widget),
			   "realize", GTK_SIGNAL_FUNC(createGLContextCB), context);
#endif
    } else {
        GdkGLContext *gdkglcshare = NULL;
    
	spDebug(80, "spCreateGLContextArch", "canvas is drawable\n");
	
	if (share != NULL) gdkglcshare = SpGLContextArch(share).glcontext;
	
	if ((SpGLContextArch(context).glcontext
	     = gtk_widget_create_gl_context(SpPrimitiveArch(gldrawable).widget,
					    gdkglcshare,
					    TRUE, GDK_GL_RGBA_TYPE)) == NULL) {
	    return SP_FALSE;
	}
    }
#endif

    {
	Display *display = NULL;
#if GTK_CHECK_VERSION(2,2,0)
        display = gdk_x11_display_get_xdisplay(gdk_display_get_default());
#elif GTK_CHECK_VERSION(2,0,0)
	display = GDK_DISPLAY_XDISPLAY(gdk_display_get_default());
#else
	display = GDK_DISPLAY();
#endif
	SpGLContextArch(context).ext_list = (char *)glXQueryExtensionsString(display, DefaultScreen(display));
    }
    
    spDebug(80, "spCreateGLContextArch", "done\n");
    
    return SP_TRUE;
}

spBool spDestroyGLContextArch(spGLContext context)
{
    if (SpGLContextArch(context).glcontext != NULL) {
#if !defined(SP_GL_USE_GTK_GL_AREA)
	gdk_gl_context_destroy(SpGLContextArch(context).glcontext);
#endif
	SpGLContextArch(context).glcontext = NULL;
    }
    
    return SP_TRUE;
}

#if !defined(SP_GL_USE_GTK_GL_AREA)
static void setGLContextCB(GtkWidget *widget, spGLContext context)
{
    setGLContext(widget, context);
    spDebug(100, "setGLContextCB", "done\n");
    return;
}
#endif

spBool spSetGLContextArch(spComponent gldrawable, spGLContext context)
{
    gboolean flag;

    if (
#if GTK_CHECK_VERSION(3,0,0)
	gtk_widget_get_window(SpPrimitiveArch(gldrawable).widget) == NULL
#else
        SpPrimitiveArch(gldrawable).widget->window == NULL
#endif
        ) {
#if !defined(SP_GL_USE_GTK_GL_AREA)
#if GTK_CHECK_VERSION(3,0,0)
	g_signal_connect(SpPrimitiveArch(gldrawable).widget,
			 "realize", G_CALLBACK(setGLContextCB), context);
#elif GTK_CHECK_VERSION(2,0,0)
	g_signal_connect(GTK_OBJECT(SpPrimitiveArch(gldrawable).widget),
			 "realize", G_CALLBACK(setGLContextCB), context);
#else
	gtk_signal_connect(GTK_OBJECT(SpPrimitiveArch(gldrawable).widget),
			   "realize", GTK_SIGNAL_FUNC(setGLContextCB), context);
#endif
#endif
	flag = TRUE;
    } else {
#if defined(SP_GL_USE_GTK_GL_AREA)
        if (SpCanvasArch(gldrawable).in_draw_func == SP_TRUE) {
            spDebug(50, "spSetGLContextArch", "called in render func\n");
            return TRUE;
        }
#endif
	flag = setGLContext(SpPrimitiveArch(gldrawable).widget, context);
    }

    spDebug(100, "spSetGLContextArch", "done: flag = %d\n", flag);
    
    if (flag) {
	return SP_TRUE;
    } else {
	return SP_FALSE;
    }
}

spBool spGLSwapBuffersArch(spComponent gldrawable)
{
#if defined(SP_GL_USE_GTK_GL_AREA)
    spDebug(100, "spGLSwapBuffersArch", "in_draw_func = %d\n", SpCanvasArch(gldrawable).in_draw_func);
    if (SpCanvasArch(gldrawable).in_draw_func == SP_FALSE) {
        gtk_widget_queue_draw(SpPrimitiveArch(gldrawable).widget);
    }
    spDebug(100, "spGLSwapBuffersArch", "done\n");
    return SP_TRUE;
#else
    GdkGLDrawable *gdkgldrawable;

    gdkgldrawable = gtk_widget_get_gl_drawable(SpPrimitiveArch(gldrawable).widget);
    
    if (gdk_gl_drawable_is_double_buffered(gdkgldrawable)) {
	gdk_gl_drawable_swap_buffers(gdkgldrawable);
	return SP_TRUE;
    }
    
    return SP_FALSE;
#endif
}

spBool spGLWaitGLDrawingArch(spComponent gldrawable)
{
#if defined(SP_GL_USE_GTK_GL_AREA)
#if defined(GDK_WINDOWING_X11)
    glXWaitGL();
#else
    glFinish();
#endif
    return SP_TRUE;
#else
    GdkGLDrawable *gdkgldrawable;

    gdkgldrawable = gtk_widget_get_gl_drawable(SpPrimitiveArch(gldrawable).widget);
    
    gdk_gl_drawable_wait_gl(gdkgldrawable);
    
    return SP_TRUE;
#endif
}

spBool spGLWaitSystemDrawingArch(spComponent gldrawable)
{
#if defined(SP_GL_USE_GTK_GL_AREA)
#if defined(GDK_WINDOWING_X11)
    glXWaitX();
#elif defined(_WIN32)
    GdiFlush();
#endif
    return SP_TRUE;
#else
    GdkGLDrawable *gdkgldrawable;

    gdkgldrawable = gtk_widget_get_gl_drawable(SpPrimitiveArch(gldrawable).widget);
    
    gdk_gl_drawable_wait_gdk(gdkgldrawable);
    
    return SP_TRUE;
#endif
}

spBool spGLUseFontArch(spComponent gldrawable, char *font_name, int first, int count, int list_base)
{
#if defined(SP_GL_USE_GTK_GL_AREA)
    return SP_FALSE;
#else
    spBool flag;
    PangoFontDescription *font_desc;

    font_desc = pango_font_description_from_string(SP_GTK_DEFAULT_FONT_NAME);
    spFontNameToFontDescriptionGtk(font_name, font_desc);

    if (gdk_gl_font_use_pango_font(font_desc, first, count, list_base) == NULL) {
	spDebug(50, "spGLUseFontArch", "gdk_gl_font_use_pango_font == NULL\n");
	flag = SP_FALSE;
    } else {
	spDebug(50, "spGLUseFontArch", "gdk_gl_font_use_pango_font OK\n");
	flag = SP_TRUE;
    }

    pango_font_description_free(font_desc);
    
    spDebug(50, "spGLUseFontArch", "flag = %d\n", flag);
    
    return flag;
#endif
}
